// Revision: 76 1.8.2.5 source/ui/cnr/icnrcol.cpp, container, ioc.v400, 001006 
/******************************************************************************
* FILE NAME: icnrcol.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in icnrcol.hpp, e.g., IContainerColumn. This class provides the ability    *
*   to add, delete, hide, and show columns in a container. This class only     *
*   needs to be used if a details view will be provided in the container.      *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

#define INCL_WINSTDDRAG
#define INCL_WINSTDCNR

extern "C"
{
   #include <string.h>
}

#include <icnrrec.hpp>  // Must be first for OS flags

#ifdef IC_PMWIN
  #include <icnrctlw.hpp>
#endif
#include <icnrcol.hpp>
#include <icnrobj.hpp>
#include <icnrctl.hpp>
#include <itrace.hpp>
#include <iexcept.hpp>
#include <ireslib.hpp>
#include <idate.hpp>
#include <itime.hpp>


// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRCOL_CPP_
  #include <ipagetun.h>
#endif

const unsigned long ulAllDataTypes = CFA_DATE | CFA_TIME | CFA_STRING
                            | CFA_BITMAPORICON;

const IContainerColumn::Style IContainerColumn::string(CFA_STRING);
const IContainerColumn::Style IContainerColumn::icon(CFA_BITMAPORICON);
const IContainerColumn::Style IContainerColumn::alignTop(CFA_TOP);
const IContainerColumn::Style IContainerColumn::alignBottom(CFA_BOTTOM);
const IContainerColumn::Style IContainerColumn::alignVerticallyCentered(CFA_VCENTER);
const IContainerColumn::Style IContainerColumn::alignLeft(CFA_LEFT);
const IContainerColumn::Style IContainerColumn::alignRight(CFA_RIGHT);
const IContainerColumn::Style IContainerColumn::alignCentered(CFA_CENTER);

const IContainerColumn::DataStyle IContainerColumn::readOnly(CFA_FIREADONLY);
const IContainerColumn::DataStyle IContainerColumn::date(CFA_DATE);
const IContainerColumn::DataStyle IContainerColumn::time(CFA_TIME);
const IContainerColumn::DataStyle IContainerColumn::number(CFA_ULONG);
const IContainerColumn::DataStyle IContainerColumn::handleDrawItem(CFA_OWNER);
const IContainerColumn::DataStyle IContainerColumn::invisible(CFA_INVISIBLE);
const IContainerColumn::DataStyle IContainerColumn::horizontalSeparator(CFA_HORZSEPARATOR);
const IContainerColumn::DataStyle IContainerColumn::verticalSeparator(CFA_SEPARATOR);

const IContainerColumn::HeadingStyle IContainerColumn::readOnlyHeading(CFA_FITITLEREADONLY);

const IContainerColumn::HeadingStyle IContainerColumn::classDefaultHeadingStyle(
           CFA_STRING | CFA_VCENTER | CFA_LEFT | CFA_FITITLEREADONLY);

const IContainerColumn::DataStyle IContainerColumn::classDefaultDataStyle(
           CFA_STRING | CFA_VCENTER | CFA_LEFT | CFA_FIREADONLY);

IContainerColumn::DataStyle IContainerColumn::currentDefaultDataStyle(
            CFA_STRING | CFA_VCENTER | CFA_LEFT | CFA_FIREADONLY);

IContainerColumn::HeadingStyle IContainerColumn::currentDefaultHeadingStyle(
            CFA_STRING | CFA_VCENTER | CFA_LEFT | CFA_FITITLEREADONLY);



/*-----------------------------------------------------------------------------
 IContainerColumn::IContainerColumn

 Construct a container column on a container window.
------------------------------------------------------------------------------*/
IContainerColumn::IContainerColumn(unsigned long ulDataOffset,
                           const HeadingStyle& titleAttributes,
                           const DataStyle& fieldAttributes)
                     :  ptrhCl(0), pcnrctlCl(0), ulHelpId(0)
{
   IMODTRACE_DEVELOP("CnrCol:CnrCol");

#ifdef IC_PMWIN
   pfieldinfoCl = (PFIELDINFO)(char*)
                   IContainerControl::hwndAllocation().sendEvent(
                           CM_ALLOCDETAILFIELDINFO,
                           IEventParameter1(1ul),
                           IEventParameter2(0ul));
#endif
#ifdef IC_MOTIF
   pfieldinfoCl = (PFIELDINFO) new FIELDINFO;
#endif

   if(pfieldinfoCl==0)
   {
      ITHROWGUIERROR("CM_ALLOCDETAILFIELDINFO");
   }

   /* size of the record  */
   pfieldinfoCl->cb = sizeof(FIELDINFO);
   pfieldinfoCl->offStruct = ulDataOffset;
   pfieldinfoCl->pNextFieldInfo = 0;

   /* Attributes of Title */
   pfieldinfoCl->flTitle = titleAttributes;
   pfieldinfoCl->flData = fieldAttributes;
   pfieldinfoCl->pTitleData = 0;
   pfieldinfoCl->cxWidth = 0;

   /* Save ourselves into the columns User Data */
   pfieldinfoCl->pUserData = this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::IContainerColumn

 Construct a container column on a container window.
------------------------------------------------------------------------------*/
IContainerColumn::IContainerColumn(DataSource ObjectDataType,
                           const HeadingStyle& titleAttributes,
                           const DataStyle& fieldAttributes)
                     :  ptrhCl(0), pcnrctlCl(0), ulHelpId(0)
{
   IMODTRACE_DEVELOP("CnrCol:CnrCol");

#ifdef IC_PMWIN
   pfieldinfoCl = (PFIELDINFO)(char*)
                  IContainerControl::hwndAllocation().sendEvent(
                        CM_ALLOCDETAILFIELDINFO,
                        IEventParameter1(1ul),
                        IEventParameter2(0ul));
#endif
#ifdef IC_MOTIF
   pfieldinfoCl = (PFIELDINFO) new FIELDINFO;
#endif

   if(pfieldinfoCl==0)
   {
      ITHROWGUIERROR("CM_ALLOCDETAILFIELDINFO");
   }

   /* size of the record  */
   pfieldinfoCl->cb = sizeof(FIELDINFO);

   /* Attributes of Title */
   pfieldinfoCl->flTitle = titleAttributes;
   pfieldinfoCl->flData = fieldAttributes;
   pfieldinfoCl->pTitleData = 0;
   pfieldinfoCl->pNextFieldInfo = 0;
   pfieldinfoCl->cxWidth = 0;

   /* Save ourselves into the column data */
   pfieldinfoCl->pUserData = this;

   /* Offset of data in record */
   pfieldinfoCl->offStruct = IFIELDINBASE;

   switch(ObjectDataType)
   {
    case isIcon:
      pfieldinfoCl->offStruct +=(unsigned long)&(((ICnrRecord *)0)->hptrIcon);
      pfieldinfoCl->flData &= ~((unsigned long)CFA_STRING);
      pfieldinfoCl->flData |=  CFA_BITMAPORICON;
      break;
    case isIconViewText:
      pfieldinfoCl->offStruct +=(unsigned long)&(((ICnrRecord *)0)->strIcon);
      pfieldinfoCl->flData &=  ~((unsigned long)CFA_BITMAPORICON);
      pfieldinfoCl->flData |=  CFA_STRING;
      break;
   }


}

/*-----------------------------------------------------------------------------
 IContainerColumn::IContainerColumn

 Construct a container column from an existing
    column (Copy constructor).
------------------------------------------------------------------------------*/
IContainerColumn::IContainerColumn(const IContainerColumn& cnrcolIn)
                             : pcnrctlCl(0)
{
   IMODTRACE_DEVELOP("CnrCol:CnrCol(&)");

#ifdef IC_PMWIN
   pfieldinfoCl = (PFIELDINFO)(char*)
                  IContainerControl::hwndAllocation().sendEvent(
                              CM_ALLOCDETAILFIELDINFO,
                              IEventParameter1(1ul),
                              IEventParameter2(0ul));
#endif
#ifdef IC_MOTIF
   pfieldinfoCl = (PFIELDINFO) new FIELDINFO;
#endif


   if(pfieldinfoCl==0)
   {
      ITHROWGUIERROR("CM_ALLOCDETAILFIELDINFO");
   }

   *pfieldinfoCl = *(cnrcolIn.pfieldinfoCl);
   if (cnrcolIn.headingTextRef()!=0)
   {
      long lLen = strlen(cnrcolIn.headingTextRef())+1;
      pfieldinfoCl->pTitleData = new char[lLen];
      strcpy((char*)(pfieldinfoCl->pTitleData), cnrcolIn.headingTextRef());
   }
   pfieldinfoCl->pUserData = this;

}

/*-----------------------------------------------------------------------------
 IContainerColumn::~IContainerColumn

 Delete a container column.
------------------------------------------------------------------------------*/
IContainerColumn::~IContainerColumn()
{
   IMODTRACE_DEVELOP("CnrCol:~CnrCol");


   /* if string added, remove it  */
   if ((pfieldinfoCl->flTitle & CFA_STRING) && pfieldinfoCl->pTitleData !=0)
     delete (char*)(pfieldinfoCl->pTitleData);

      /* If in a container, then remove the column */
   if((pcnrctlCl!=0) && (pcnrctlCl->flClState & IContainerControl::ok))
   {
     pcnrctlCl->sendEvent( CM_REMOVEDETAILFIELDINFO,
                         &pfieldinfoCl,
                         IEventData(1,CMA_INVALIDATE));
   }

#ifdef IC_PMWIN
   IContainerControl::hwndAllocation().sendEvent(
           CM_FREEDETAILFIELDINFO, &pfieldinfoCl, 1);
#endif
#ifdef IC_MOTIF
    delete pfieldinfoCl;
#endif
}


/*-----------------------------------------------------------------------------
 IContainerColumn::setHeadingText

 Update the heading text in a container column.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setHeadingText(const char*  pszText)
{

   /* if string added, remove it  */
   if (pfieldinfoCl->pTitleData != 0 && pfieldinfoCl->flTitle & CFA_STRING)
     delete (char *)pfieldinfoCl->pTitleData;

   long lLen = strlen(pszText)+1;
   pfieldinfoCl->pTitleData = new char[lLen];
   strcpy((char*)(pfieldinfoCl->pTitleData), pszText);
   pfieldinfoCl->flTitle &= ~((unsigned long)CFA_BITMAPORICON);
   pfieldinfoCl->flTitle |=  CFA_STRING;
#ifdef IC_WIN
   // If this column has been added to a native windows container,
   // then we must force the update as the invalidate() is not enough.
   if ( pcnrctlCl && !pcnrctlCl->ppd->pmCompatible )
   {
     int        colId(0);
     LV_COLUMN  lvColumn;

     // In native Windows, only single line headings are supported, so
     // we will strip out any newline characters and replace them with
     // blanks.
     IString temp( (char*)pfieldinfoCl->pTitleData );
     temp.change( "\n", " " );
     strcpy( (char*)(pfieldinfoCl->pTitleData), temp );

     colId = pcnrctlCl->ppd->columnSet->columnIdFromContainerColumn( this );
     lvColumn.mask     = LVCF_TEXT;
     lvColumn.pszText  = (char*)pfieldinfoCl->pTitleData;
     ListView_SetColumn( pcnrctlCl->ppd->pListview->handle(),
                         colId, &lvColumn );
   }
#endif
   invalidate();
   return *this;
}



/*-----------------------------------------------------------------------------
 IContainerColumn::setHeadingText

 Update the heading text in a container column.
          from a resource file.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setHeadingText(const IResourceId& resid)
{
  IString strText = resid.resourceLibrary().loadString(resid);
  setHeadingText((char*)strText);
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::headingText

 Return an IString containing the heading text.
------------------------------------------------------------------------------*/
IString IContainerColumn::headingText() const
{

   const char* heading = headingTextRef();
   if(heading!=0)
      return IString(heading);
   else
      return IString("");
}

/*-----------------------------------------------------------------------------
 IContainerColumn::headingTextRef

 Return a constant pointer to the heading text.
------------------------------------------------------------------------------*/
const char* IContainerColumn::headingTextRef() const
{
  if(pfieldinfoCl->flTitle & CFA_STRING)
    return ((char*)(pfieldinfoCl->pTitleData));
  else
    return 0;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::headingIcon

 Return the Icon if being used.
------------------------------------------------------------------------------*/
IPointerHandle IContainerColumn::headingIcon() const
{
  return ptrhCl;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::setHeadingIcon

 Indicate the Icon handle to use for a column heading.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setHeadingIcon(const IPointerHandle& ptrhIcon)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   ptrhCl = ptrhIcon;

   /* if string added, remove it  */
   if ( (pfieldinfoCl->pTitleData != 0) &&
        (pfieldinfoCl->flTitle&CFA_STRING) )
   {
     delete (char *)pfieldinfoCl->pTitleData;
   }
   pfieldinfoCl->flTitle &= ~((unsigned long)CFA_STRING);
   pfieldinfoCl->flTitle |= CFA_BITMAPORICON;
#ifdef IC_WIN
   pfieldinfoCl->pTitleData = (void*)ptrhIcon;
#endif
#ifdef IC_PM
   pfieldinfoCl->pTitleData = (void*)(unsigned long)ptrhIcon;
#endif
   invalidate();
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}

/*-----------------------------------------------------------------------------
 IContainerColumn::`etHeadingIcon

 Indicate the Icon id from a resource file to use for the
   column heading.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setHeadingIcon(const IResourceId& resid)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  if(resid != 0)
    setHeadingIcon(resid.resourceLibrary().loadIcon(resid));
  else
   ptrhCl = 0;
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}

/*-----------------------------------------------------------------------------
 IContainerColumn::setHeadingIcon

 Indicate the Icon id from a resource file to use for the
   column heading.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setHeadingIcon(unsigned long ulResid)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   setHeadingIcon(IResourceId(ulResid));
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 IContainerColumn::enableHeadingUpdate

 Allow the heading to be modified.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::enableHeadingUpdate(bool fEnable)
{
   if(fEnable)
     pfieldinfoCl->flTitle &=  ~((unsigned long)CFA_FITITLEREADONLY);
   else
     pfieldinfoCl->flTitle |= CFA_FITITLEREADONLY;
   invalidate();
   return *this;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::disableHeadingUpdate

 Don't allow the heading to be modified.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::disableHeadingUpdate()
{
   return enableHeadingUpdate(false);
}

/*-----------------------------------------------------------------------------
 IContainerColumn::justifyHeading

 Justify the heading based on the input flags.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::justifyHeading(VerticalAlignment  Vertical,
                                        HorizontalAlignment Horizontal)
{
// ITRACE_MOTIF_NOTYET();
#ifdef IC_PMWIN
   unsigned long ulAlignment = 0;
   pfieldinfoCl->flTitle &= ((unsigned long)
                             ~(CFA_LEFT | CFA_RIGHT | CFA_CENTER |
                               CFA_TOP | CFA_BOTTOM | CFA_VCENTER) );
   switch(Vertical)
   {
      case IContainerColumn::centeredVertically:
      {
         ulAlignment =  CFA_VCENTER;
         break;
      }
      case IContainerColumn::top:
      {
         ulAlignment =  CFA_TOP;
         break;
      }
      case IContainerColumn::bottom:
      {
         ulAlignment =  CFA_BOTTOM;
         break;
      }
   }
   switch(Horizontal)
   {
      case IContainerColumn::left:
      {
         ulAlignment |=  CFA_LEFT;
         break;
      }
      case IContainerColumn::right:
      {
         ulAlignment |=  CFA_RIGHT;
         break;
      }
      case IContainerColumn::centered:
      {
         ulAlignment |=  CFA_CENTER;
         break;
      }
   }
   pfieldinfoCl->flTitle |=  ulAlignment;
   invalidate();
#endif
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::show

 Show the column in details view.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::show(bool fVisible)
{
   if(fVisible)
     pfieldinfoCl->flData &= ~((unsigned long)CFA_INVISIBLE);
   else
     pfieldinfoCl->flData |= CFA_INVISIBLE;
   invalidate();
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::hide

 Hide the column in details view.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::hide()
{
   pfieldinfoCl->flData |= CFA_INVISIBLE;
   invalidate();
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::showSeparators

 Turn Vertical and Horizontal Separators on.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::showSeparators(const DataStyle& dataStyleIn)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   pfieldinfoCl->flData &=((unsigned long)~(CFA_SEPARATOR | CFA_HORZSEPARATOR));

   if(dataStyleIn & verticalSeparator)
      pfieldinfoCl->flData |=  CFA_SEPARATOR;
   if(dataStyleIn & horizontalSeparator)
      pfieldinfoCl->flData |=  CFA_HORZSEPARATOR;

   invalidate();
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}

/*-----------------------------------------------------------------------------
 IContainerColumn::hideSeparators

 Turn Vertical and Horizontal Separators off.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::hideSeparators(const DataStyle& dataStyleIn)
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   if(dataStyleIn & verticalSeparator)
      pfieldinfoCl->flData &= ~((unsigned long)CFA_SEPARATOR);
   if(dataStyleIn & horizontalSeparator)
      pfieldinfoCl->flData &=  ~((unsigned long)CFA_HORZSEPARATOR);
   invalidate();
   return *this;
#endif
#ifdef IC_MOTIF
  return *this;
#endif
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isVisible

 Return true if the column is visible.(!CFA_INVISIBLE)
------------------------------------------------------------------------------*/
bool IContainerColumn::isVisible() const
{
   if (pfieldinfoCl->flData &  CFA_INVISIBLE)
     return(false);
   else
     return(true);
}


/*-----------------------------------------------------------------------------
 IContainerColumn::enableDataUpdate

 Enable the editing of column data.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::enableDataUpdate(bool fEnable)
{
   if(fEnable)
     pfieldinfoCl->flData &=  ~((unsigned long)CFA_FIREADONLY);
   else
     pfieldinfoCl->flData |= CFA_FIREADONLY;
   invalidate();
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::disableDataUpdate

 Disable the editing of column data.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::disableDataUpdate()
{
   pfieldinfoCl->flData |= CFA_FIREADONLY;
   invalidate();
   return *this;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::justifyData

 Justify the data vertical and horizontally in the column.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::justifyData(VerticalAlignment   Vertical,
                                        HorizontalAlignment Horizontal)
{
// ITRACE_MOTIF_NOTYET();
#ifdef IC_PMWIN
   unsigned long ulAlignment = 0;
   pfieldinfoCl->flData &=  ~((unsigned long)
                              (CFA_LEFT | CFA_RIGHT | CFA_CENTER |
                               CFA_TOP | CFA_BOTTOM | CFA_VCENTER));
   switch(Vertical)
   {
      case IContainerColumn::top:
      {
         ulAlignment =  CFA_TOP;
         break;
      }
      case IContainerColumn::bottom:
      {
         ulAlignment =  CFA_BOTTOM;
         break;
      }
      case IContainerColumn::centeredVertically:
      {
         ulAlignment =  CFA_VCENTER;
         break;
      }
   }
   switch(Horizontal)
   {
      case IContainerColumn::left:
      {
         ulAlignment |=  CFA_LEFT;
         break;
      }
      case IContainerColumn::right:
      {
         ulAlignment |=  CFA_RIGHT;
         break;
      }
      case IContainerColumn::centered:
      {
         ulAlignment |=  CFA_CENTER;
         break;
      }
   }
   pfieldinfoCl->flData |=  ulAlignment;

#ifdef IC_WIN
   // If this column has been added to a native windows container,
   // then we must force the update as the invalidate() is not enough.
   if ( pcnrctlCl && !pcnrctlCl->ppd->pmCompatible )
   {
     int        colId(0);
     LV_COLUMN  lvColumn;

     colId = pcnrctlCl->ppd->columnSet->columnIdFromContainerColumn( this );
     lvColumn.mask  = LVCF_FMT;

     if ( pfieldinfoCl->flData & CFA_CENTER )
       lvColumn.fmt      = LVCFMT_CENTER;
     else if ( pfieldinfoCl->flData & CFA_RIGHT )
       lvColumn.fmt      = LVCFMT_RIGHT;
     else    //default is LEFT
       lvColumn.fmt      = LVCFMT_LEFT;

     ListView_SetColumn( pcnrctlCl->ppd->pListview->handle(),
                         colId, &lvColumn );
   }
#endif

   invalidate();
#endif
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::setDisplayWidth

 Indicate the number of PELS used in the column.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setDisplayWidth(unsigned long ulPelWidth)
{
// ITRACE_MOTIF_NOTYET();
#ifdef IC_PMWIN
   pfieldinfoCl->cxWidth = ulPelWidth;
   invalidate();
#endif
#ifdef IC_WIN
   // If this column has been added to a native windows container,
   // then we must force the update as the invalidate() is not enough.
   if ( pcnrctlCl && !pcnrctlCl->ppd->pmCompatible )
   {
     int        colId(0);
     LV_COLUMN  lvColumn;

     colId = pcnrctlCl->ppd->columnSet->columnIdFromContainerColumn( this );
     lvColumn.mask  = LVCF_WIDTH;
     lvColumn.cx    = pfieldinfoCl->cxWidth;
     ListView_SetColumn( pcnrctlCl->ppd->pListview->handle(),
                         colId, &lvColumn );
   }
#endif
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::displayWidth

 Return the number of PELS used in the column.
------------------------------------------------------------------------------*/
unsigned long IContainerColumn::displayWidth()
{
  return (unsigned long)(pcnrctlCl->sendEvent( CM_QUERYDETAILFIELDINFO,
                        this->columnInfo(), CMA_DATAWIDTH));
}


/*-----------------------------------------------------------------------------
 IContainerColumn::setDataOffset

 Indicate the offset of the data in IContainerObject.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setDataOffset(unsigned long ulDataOffset)
{
  pfieldinfoCl->offStruct = ulDataOffset;
   invalidate();
   return *this;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::setHelpId

 Stores a help panel identifier.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setHelpId (unsigned long helpId)
{
   ulHelpId = helpId;
   return *this;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::dataAsNumber

 Returns the column data as an unsigned long integer no matter what the
 type is.
------------------------------------------------------------------------------*/
unsigned long IContainerColumn::dataAsNumber( const IContainerObject* object) const
{
   IASSERTPARM(container() != 0 && object !=0 );
   IASSERTPARM(container()->containsObject(object));
   IMiniCnrRecord* pcnrrec = IRecFromObj(object);
   return *(unsigned long*)(((char*)pcnrrec) + columnInfo()->offStruct);

}

/*-----------------------------------------------------------------------------
 IContainerColumn::dataAsString

 Returns the column data as an IString if the type is string.
------------------------------------------------------------------------------*/
IString IContainerColumn::dataAsString( const IContainerObject* object) const
{
   IASSERTPARM(container() != 0 && object !=0 && (dataAttributes() & string));
   IASSERTPARM(container()->containsObject(object));
   IMiniCnrRecord* pcnrrec = IRecFromObj(object);
   char** ppData = (char**)( ((char*)pcnrrec) + columnInfo()->offStruct);
   return IString(*ppData);
}

/*-----------------------------------------------------------------------------
 IContainerColumn::dataAsDate

 Returns the column data as an IDate if the type is date.
------------------------------------------------------------------------------*/
IDate IContainerColumn::dataAsDate( const IContainerObject* object) const
{
   IASSERTPARM(container() != 0 && object !=0 && (dataAttributes() & date));
   IASSERTPARM(container()->containsObject(object));
   IMiniCnrRecord* pcnrrec = IRecFromObj(object);
   char* pDate = ((char*)pcnrrec) + columnInfo()->offStruct;
   IDate date;
   switch(*(pDate+1))
   {
     case 1:
       date = IDate(IDate::January, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 2:
       date = IDate(IDate::February, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 3:
       date = IDate(IDate::March, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 4:
       date = IDate(IDate::April, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 5:
       date = IDate(IDate::May, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 6:
       date = IDate(IDate::June, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 7:
       date = IDate(IDate::July, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 8:
       date = IDate(IDate::August, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 9:
       date = IDate(IDate::September, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 10:
       date = IDate(IDate::October, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 11:
       date = IDate(IDate::November, *pDate,*(unsigned short*)(pDate+2));
       break;
     case 12:
       date = IDate(IDate::December, *pDate,*(unsigned short*)(pDate+2));
       break;
   }
   return date;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::dataAsTime

 Returns the column data as an ITime if the type is time.
------------------------------------------------------------------------------*/
ITime IContainerColumn::dataAsTime( const IContainerObject* object) const
{
   IASSERTPARM(container() != 0 && object !=0 && (dataAttributes() & time));
   IASSERTPARM(container()->containsObject(object));
   IMiniCnrRecord* pcnrrec = IRecFromObj(object);
   char* pTime = ((char*)pcnrrec) + columnInfo()->offStruct;
   return ITime(*pTime, *(pTime+1), *(pTime+2));

}


/*-----------------------------------------------------------------------------
 IContainerColumn::dataAsIcon

 Returns the column data as an IPointerHandle if the type is icon.
------------------------------------------------------------------------------*/
IPointerHandle IContainerColumn::dataAsIcon( const IContainerObject* object) const
{
   IASSERTPARM(container() != 0 && object !=0 && (dataAttributes() & icon));
   IASSERTPARM(container()->containsObject(object));
   IMiniCnrRecord* pcnrrec = IRecFromObj(object);
   return IPointerHandle( (IPointerHandle::Value)( *(unsigned long*)
                            ((char*)pcnrrec) + columnInfo()->offStruct ) );
}




/*-----------------------------------------------------------------------------
 IContainerColumn::isDate

 Returns true if the column data is a date in CDATE format.
------------------------------------------------------------------------------*/
bool IContainerColumn::isDate ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_DATE))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isTime

 Returns true if the column data is a time in CTIME format.
------------------------------------------------------------------------------*/
bool IContainerColumn::isTime ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_TIME))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isNumber

 Returns true if the column data is an unsigned long integer.
------------------------------------------------------------------------------*/
bool IContainerColumn::isNumber ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_ULONG))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isString

 Returns true if the column data is an IString or a character pointer.
------------------------------------------------------------------------------*/
bool IContainerColumn::isString ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_STRING))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isIconHandle

 Returns true if the column data is an Icon.
------------------------------------------------------------------------------*/
bool IContainerColumn::isIconHandle ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_BITMAPORICON))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isWriteable

 Returns true if the column data can be edited.
------------------------------------------------------------------------------*/
bool IContainerColumn::isWriteable ( ) const
{
  if(pfieldinfoCl->flData & CFA_FIREADONLY)
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isHeadingWriteable

 Returns true if the column heading can be edited.
------------------------------------------------------------------------------*/
bool IContainerColumn::isHeadingWriteable ( ) const
{
  if(pfieldinfoCl->flTitle & CFA_FITITLEREADONLY)
    return false;
  else
    return true;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::isHeadingString

 Returns true if the column heading data is a character pointer.
------------------------------------------------------------------------------*/
bool IContainerColumn::isHeadingString ( ) const
{
#ifdef IC_PMWIN
  if(!(pfieldinfoCl->flTitle & CFA_STRING))
    return false;
  else
    return true;
#endif
#ifdef IC_MOTIF
  return true;
#endif
}


/*-----------------------------------------------------------------------------
 IContainerColumn::isHeadingIconHandle

 Returns true if the column heading data is a pointer handle
------------------------------------------------------------------------------*/
bool IContainerColumn::isHeadingIconHandle ( ) const
{
#ifdef IC_PMWIN
  if(!(pfieldinfoCl->flTitle & CFA_BITMAPORICON))
    return false;
  else
    return true;
#endif
#ifdef IC_MOTIF
  return false;
#endif
}

/*-----------------------------------------------------------------------------
 IContainerColumn::hasHorizontalSeparator

 Returns true if the column has a horizontal separator line under the heading
 visible.
------------------------------------------------------------------------------*/
bool IContainerColumn::hasHorizontalSeparator ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_HORZSEPARATOR))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::hasVerticalSeparator

 Returns true if the column has a verticle separator line visible.
------------------------------------------------------------------------------*/
bool IContainerColumn::hasVerticalSeparator ( ) const
{
  if(!(pfieldinfoCl->flData & CFA_SEPARATOR))
    return false;
  else
    return true;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::verticalDataAlignment

 Returns an enum identifying the vertical data alignment.
------------------------------------------------------------------------------*/
IContainerColumn::VerticalAlignment IContainerColumn::verticalDataAlignment( ) const
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  VerticalAlignment vert = IContainerColumn::centeredVertically;

  if (pfieldinfoCl->flData & CFA_TOP)
    vert = IContainerColumn::top;
  else if (pfieldinfoCl->flData & CFA_BOTTOM)
    vert = IContainerColumn::bottom;

  return vert;
#endif
#ifdef IC_MOTIF
  return IContainerColumn::top;
#endif
}

/*-----------------------------------------------------------------------------
 IContainerColumn::verticalHeadingAlignment

 Returns an enum identifying the vertical heading alignment.
------------------------------------------------------------------------------*/
IContainerColumn::VerticalAlignment IContainerColumn::verticalHeadingAlignment( ) const
{
ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  VerticalAlignment vert = IContainerColumn::centeredVertically;

  if (pfieldinfoCl->flTitle & CFA_TOP)
    vert = IContainerColumn::top;
  else if (pfieldinfoCl->flTitle & CFA_BOTTOM)
    vert = IContainerColumn::bottom;

  return vert;
#endif
#ifdef IC_MOTIF
  return IContainerColumn::top;
#endif
}

/*-----------------------------------------------------------------------------
 IContainerColumn::horizontalDataAlignment

 Returns an enum identifying the horizontal data alignment.
------------------------------------------------------------------------------*/
IContainerColumn::HorizontalAlignment IContainerColumn::horizontalDataAlignment( ) const
{
  HorizontalAlignment horz = IContainerColumn::left;
#ifdef IC_PMWIN
  if (pfieldinfoCl->flData & CFA_RIGHT)
    horz = IContainerColumn::right;
  else if (pfieldinfoCl->flData & CFA_CENTER)
    horz = IContainerColumn::centered;
#endif
  return horz;

}

/*-----------------------------------------------------------------------------
 IContainerColumn::horizontalHeadingAlignment

 Returns an enum identifying the horizontal heading alignment.
------------------------------------------------------------------------------*/
IContainerColumn::HorizontalAlignment IContainerColumn::horizontalHeadingAlignment( ) const
{
  HorizontalAlignment horz = IContainerColumn::left;
#ifdef IC_PMWIN
  if (pfieldinfoCl->flTitle & CFA_RIGHT)
    horz = IContainerColumn::right;
  else if (pfieldinfoCl->flTitle & CFA_CENTER)
    horz = IContainerColumn::centered;
#endif
  return horz;
}






/*-----------------------------------------------------------------------------
 IContainerColumn::setTitleAttributes

 Update the attributes for the column title.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setTitleAttributes(unsigned long ulTitleAttributes)
{
  // ITRACE_MOTIF_NOTYET();
  pfieldinfoCl->flTitle = ulTitleAttributes;

#ifdef IC_WIN
   // If this column has been added to a native windows container,
   // then we must force the update as the invalidate() is not enough.
   // The only thing we need to worry about here is that the title is
   // being changed from a string to an icon.
   if ( pcnrctlCl && !pcnrctlCl->ppd->pmCompatible &&
        (pfieldinfoCl->flTitle & CFA_BITMAPORICON) )
   {
     int        colId(0);
     LV_COLUMN  lvColumn;

     colId = pcnrctlCl->ppd->columnSet->columnIdFromContainerColumn( this );
     lvColumn.mask     = LVCF_TEXT;
     lvColumn.pszText  = "";
     ListView_SetColumn( pcnrctlCl->ppd->pListview->handle(),
                         colId, &lvColumn );
   }
#endif

  invalidate();
  return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::setDataAttributes

 Update the attributes for the column data..
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::setDataAttributes(unsigned long ulDataAttributes)
{
  // ITRACE_MOTIF_NOTYET();
  pfieldinfoCl->flData = ulDataAttributes;

#ifdef IC_WIN
   // If this column has been added to a native windows container,
   // then we must force the update as the invalidate() is not enough.
   // The only thing we need to worry about here is the justification.
   if ( pcnrctlCl && !pcnrctlCl->ppd->pmCompatible )
   {
     int        colId(0);
     LV_COLUMN  lvColumn;

     colId = pcnrctlCl->ppd->columnSet->columnIdFromContainerColumn( this );
     lvColumn.mask  = LVCF_FMT;

     if ( pfieldinfoCl->flData & CFA_CENTER )
       lvColumn.fmt      = LVCFMT_CENTER;
     else if ( pfieldinfoCl->flData & CFA_RIGHT )
       lvColumn.fmt      = LVCFMT_RIGHT;
     else    //default is LEFT
       lvColumn.fmt      = LVCFMT_LEFT;

     ListView_SetColumn( pcnrctlCl->ppd->pListview->handle(),
                         colId, &lvColumn );
   }
#endif

  invalidate();
  return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::titleAttributes

 Return the attributes for the column title.
------------------------------------------------------------------------------*/
unsigned long IContainerColumn::titleAttributes() const
{
  // ITRACE_MOTIF_NOTYET();
  return(pfieldinfoCl->flTitle);
}

/*-----------------------------------------------------------------------------
 IContainerColumn::dataAttributes

 Return the attributes for the column data.
------------------------------------------------------------------------------*/
unsigned long IContainerColumn::dataAttributes() const
{
  // ITRACE_MOTIF_NOTYET();
  return(pfieldinfoCl->flData);
}

/*-----------------------------------------------------------------------------
 IContainerColumn::invalidate

 Invalidate the column if container exists.
------------------------------------------------------------------------------*/
IContainerColumn& IContainerColumn::invalidate()
{
   IMODTRACE_DEVELOP("CnrCol:invalidate");
   if(pcnrctlCl!=0)
   {
      unsigned fSuccess = pcnrctlCl->sendEvent( CM_INVALIDATEDETAILFIELDINFO);
      if(!fSuccess)
      {
         ITHROWGUIERROR("CM_INVALIDATEDETAILFIELDINFO");
      }
   }
   return *this;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::defaultHeadingStyle

 Return the default style for new columns.
------------------------------------------------------------------------------*/
IContainerColumn::Style  IContainerColumn::defaultHeadingStyle()
{
  return currentDefaultHeadingStyle;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::defaultHeadingStyle

 Replace the default style for new columns.
------------------------------------------------------------------------------*/
void  IContainerColumn::setDefaultHeadingStyle
                                    (const IContainerColumn::Style& hStyle)
{
  currentDefaultHeadingStyle = hStyle;
}

/*-----------------------------------------------------------------------------
 IContainerColumn::defaultHeadingStyle

 Return the default style for new columns.
------------------------------------------------------------------------------*/
IContainerColumn::Style  IContainerColumn::defaultDataStyle()
{
  return currentDefaultDataStyle;
}


/*-----------------------------------------------------------------------------
 IContainerColumn::defaultHeadingStyle

 Replace the default style for new columns.
------------------------------------------------------------------------------*/
void  IContainerColumn::setDefaultDataStyle
                                      (const IContainerColumn::Style& dStyle)
{
  currentDefaultDataStyle = dStyle;
}
