/* Revision: 61 1.5 source/ui/cclnt/slider/sliderp.c, cclslid, ioc.v400, 001006  */
/**********************************************************************/
/*                                                                    */
/*                         IBM CONFIDENTIAL                           */
/*                                                                    */
/*        (IBM Confidential-Restricted when combined with the         */
/*         aggregated OCO source modules for this program)            */
/*                                                                    */
/*                       OCO SOURCE MATERIALS                         */
/*                                                                    */
/*         20G0400 (C) Copyright IBM Corp. 1992, 1997 (Unpublished)   */
/*                                                                    */
/**********************************************************************/
/**********************************************************************/
/*                                                                    */
/*                           S L I D E R                              */
/*                                                                    */
/*                       Filename: SLIDERP.C                          */
/*                                                                    */
/*              ( Windows NT Slider Control Component )               */
/*                                                                    */
/*                         IBM CONFIDENTIAL                           */
/*                                                                    */
/*                (C) Copyright IBM Corporation 1991                  */
/*                       All Rights Reserved                          */
/*            Licensed Material - Program Property of IBM             */
/*                                                                    */
/**********************************************************************/
/**********************************************************************/
/* Product: SLIDER.DLL                                                */
/* Filename: SLIDERP.C - Paint/Display routines for the slider control*/
/*                                                                    */
/* Flag    Date    By:  Description                                   */
/* ----  --------  ---  --------------------------------------------- */
/* JEH1  06/21/93  JEH  Initial modifications for NT execution        */
/* DAB2  10/11/95  DAB  Fixed problem with ownerdraw cnr id in mp1    */
/**********************************************************************/
#include <string.h>

#include <windows.h>
#include "iclsldw.h"               /* Slider control public includes  */
#include "sliderp.h"               /* Slider control private includes */

/**********************************************************************/
/* Global handles for Arrow Bitmaps (declared in SLIDERM.C)           */
/**********************************************************************/
extern HBITMAP hbmVIncr;
extern HBITMAP hbmVDecr;
extern HBITMAP hbmHIncr;
extern HBITMAP hbmHDecr;
extern HBITMAP hbmVIncrS;
extern HBITMAP hbmVDecrS;
extern HBITMAP hbmHIncrS;
extern HBITMAP hbmHDecrS;
extern HBITMAP hbmVIncrD;
extern HBITMAP hbmVDecrD;
extern HBITMAP hbmHIncrD;
extern HBITMAP hbmHDecrD;
extern HBITMAP hbmVIncrSD;
extern HBITMAP hbmVDecrSD;
extern HBITMAP hbmHIncrSD;
extern HBITMAP hbmHDecrSD;

/**********************************************************************/
/* Procedure: DrawSlider - This procedure will draw the slider        */
/*            control on the screen.  The drawing will be done based  */
/*            on the style set for the slider control.  It will draw  */
/*            in the following order (if needed):                     */
/*             1) Draw button area (if any)                           */
/*             2) Draw slider shaft border                            */
/*             3) Fill in slider shaft area (including ribbon strip)  */
/*             4) Fill background quadrants (area around shaft)       */
/*             5) Paint tick marks                                    */
/*             6) Paint detents                                       */
/*             7) Draw the slider arm in position                     */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID DrawSlider( HANDLE hwndSld, PISLDDATA pIntData )
{
  ULONG    ulStyle;                                                     /*JEH1*/
  HDC      hps;
  RECT    rclButton;
  RECT    rclBorder;
  RECT    rclIntersect;
  RECT    rclWindow;
  RECT    rclQuad1;
  RECT    rclQuad2;
  RECT    rclQuad3;
  RECT    rclQuad4;
  RECT    rclQuad5;
  long     lEdge;
  int    sEdge1;
  int    sEdge2;
  int    sEdge3;
  int    sEdge4;
  BOOL     bPaintScale=FALSE;
  USHORT   usButtonLength;                                              /*JEH1*/
  PAINTSTRUCT paint;
  ULONG   rasterOp = SRCCOPY;                                           /*JEH1*/
#define PATTERNORSOURCE   0x00FC008A
  HBRUSH     hHalftoneBrush=NULL;
  HBRUSH     hOldBrush=NULL;
  HBRUSH     background=NULL;
  HBITMAP    hHalftonePatternBitmap=NULL;
  BYTE       HalftonePattern [] =
                 { 0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
                   0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55 };

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Get the length of a button based on the current shaft breadth    */
  /********************************************************************/
  usButtonLength = CalcButtonLength( pIntData );

  /********************************************************************/
  /* Get presentation space to paint in, an anchor block for the      */
  /* window and get the rectangle for the slider window               */
  /********************************************************************/
  hps = BeginPaint( hwndSld, &paint );

  /********************************************************************/
  /* Send a message to the slider requesting the colors to use to     */
  /* paint the slider                                                 */
  /********************************************************************/
  background = (HBRUSH)SendMessage( hwndSld, WM_CTLCOLORSLIDER, hps, hwndSld );
  if (background)
  {
    pIntData->ulColorVal[CLR_SLDBACKGROUND] = GetBkColor( hps );
    pIntData->ulColorVal[CLR_SLDFOREGROUND] = GetTextColor( hps );
  }
  else
  {
    pIntData->ulColorVal[CLR_SLDBACKGROUND] = GetSysColor(COLOR_WINDOW);
    pIntData->ulColorVal[CLR_SLDFOREGROUND] = GetSysColor(COLOR_WINDOWTEXT);
  }

  GetClientRect( hwndSld, &rclWindow );

  /********************************************************************/
  /* Check if buttons are to be drawn                                 */
  /********************************************************************/
  if ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT))
  {
    /******************************************************************/
    /* Calculate button rectangle from internal data                  */
    /******************************************************************/
    CalcRectangle( hwndSld, pIntData, pIntData->xButton,
         pIntData->yButton, (USHORT)(usButtonLength << 1), &rclButton );/*JEH1*/

    /******************************************************************/
    /* Check if the area to paint intersects the button area          */
    /******************************************************************/

    if ( IntersectRect( &rclIntersect, &paint.rcPaint, &rclButton))
      PaintButtons( hwndSld, pIntData, hps );

  }

  /********************************************************************/
  /* Calculate border rectangle from internal data                    */
  /********************************************************************/
  CalcRectangle( hwndSld, pIntData, pIntData->xShaft, pIntData->yShaft,
                 pIntData->usShaftLength, &rclBorder );
  /********************************************************************/
  /* Check if the area to paint intersects the border area            */
  /********************************************************************/

  if ( IntersectRect( &rclIntersect, &paint.rcPaint, &rclBorder))
  {

    PaintBorder( hwndSld, pIntData, hps, &rclBorder, &rclButton,
                 &(pIntData->rclSliderArm) );
    PaintShaft( hwndSld, hps, &rclBorder, &(pIntData->rclSliderArm) );
  }

  /********************************************************************/
  /* Get the edges of the border and button area                      */
  /********************************************************************/
  lEdge = CalcShaftnButtonEdges( hwndSld, &rclBorder, &rclButton, TRUE);
  sEdge1 = LOWORD( lEdge );
  sEdge2 = HIWORD( lEdge );

  lEdge = CalcShaftnButtonEdges( hwndSld, &rclBorder, &rclButton, FALSE);
  sEdge3 = LOWORD( lEdge );
  sEdge4 = HIWORD( lEdge );

  /********************************************************************/
  /* Calculate the offset of the four background quadrants            */
  /* If vertical, quadrants labeled 1-4 starting at right going ccw   */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* Calculate the rectangle quadrants surrounding the slider       */
    /******************************************************************/             
    SetRect( &rclQuad1, 0, 0, (int)pIntData->xShaft,
             (int)rclWindow.bottom);
    SetRect( &rclQuad2, (int)pIntData->rclSliderArm.left, sEdge1,
             (int)pIntData->rclSliderArm.right, rclWindow.bottom );
    SetRect( &rclQuad3, (int)pIntData->xShaft + (int)pIntData->usShaftBreadth, 0,
             (int)rclWindow.right, (int)rclWindow.bottom );
    SetRect( &rclQuad4, (int)pIntData->rclSliderArm.left, 0,    
             (int)pIntData->rclSliderArm.right, sEdge2 );
    SetRect( &rclQuad5,(int)pIntData->xShaft, sEdge4,
    				(int)pIntData->xShaft + (int)pIntData->usShaftBreadth, sEdge3);
  }
  /********************************************************************/
  /* Calculate the offset of the four background quadrants            */
  /* If horizontal, quadrants labeled 1-4 starting at top going ccw   */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Calculate the rectangle quadrants surrounding the slider       */
    /******************************************************************/
    SetRect( &rclQuad1, 0, (int)rclWindow.top,
            (int)rclWindow.right, (int)pIntData->yShaft);
    SetRect( &rclQuad2, 0, (int)pIntData->rclSliderArm.top,
             sEdge1, (int)pIntData->rclSliderArm.bottom );
    SetRect( &rclQuad3, 0, (int)pIntData->yShaft + (int)pIntData->usShaftBreadth,
            (int)rclWindow.right, rclWindow.bottom);
    SetRect( &rclQuad4, sEdge2, (int)pIntData->
               rclSliderArm.top, (int)rclWindow.right,
               (int)pIntData->rclSliderArm.bottom );
    SetRect( &rclQuad5, sEdge3, (int)pIntData->rclSliderArm.top, 
    				sEdge4, (int)pIntData->rclSliderArm.bottom );
  }


  /********************************************************************/
  /* For each quadrant to be painted, check for ownerdraw and paint   */
  /*  If quadrants on sides of shaft are painted, also paint scale    */
  /********************************************************************/
  if ( PaintBackground( hwndSld, pIntData, hps, &paint.rcPaint,
                        &rclQuad1, FALSE ))
    bPaintScale = TRUE;
  if ( PaintBackground( hwndSld, pIntData, hps, &paint.rcPaint,
                        &rclQuad3, FALSE ))
    bPaintScale = TRUE;
  PaintBackground( hwndSld, pIntData, hps, &paint.rcPaint, &rclQuad2,
                   FALSE );
  PaintBackground( hwndSld, pIntData, hps, &paint.rcPaint, &rclQuad4,
                   FALSE );
  PaintBackground( hwndSld, pIntData, hps, &paint.rcPaint, &rclQuad5,
                   FALSE );

  /********************************************************************/
  /* Paint the ticks along the shaft and any text if needed           */
  /********************************************************************/
  if (bPaintScale)
  {
    PaintTicks( hwndSld, pIntData, hps );

    /******************************************************************/
    /* If not readonly, paint detents as well                         */
    /******************************************************************/
    if (!(ulStyle & SLS_READONLY))
      PaintDetents( hwndSld, pIntData, hps );
  }

  /******************************************************************/
  /* If the arm should be drawn disabled, set up the raster flag and*/
  /* create the halftoning brush pattern; however, if either the    */
  /* bitmap or the brush can't be created, don't select the brush   */
  /******************************************************************/
  if (ulStyle & WS_DISABLED)
  {
    if ((hHalftonePatternBitmap = CreateBitmap (16, 16, 1, 1,
        (LPSTR) HalftonePattern)) &&
        (hHalftoneBrush = CreatePatternBrush (hHalftonePatternBitmap)))
    {
      hOldBrush = SelectObject (hps, hHalftoneBrush);
      rasterOp = PATTERNORSOURCE;
    }
  }

  /********************************************************************/
  /* Finally draw arm out to the screen                               */
  /********************************************************************/
  BltCopy( pIntData->hbmArm, hps, 0, 0,
          (WORD)(pIntData->rclSliderArm.right -
          pIntData->rclSliderArm.left),
          (WORD)(pIntData-> rclSliderArm.bottom -
          pIntData->rclSliderArm.top),
           (int)pIntData->rclSliderArm.left,
           (int)pIntData-> rclSliderArm.top, rasterOp, SRC_BITMAP );

  /******************************************************************/
  /* If the halftoning bitmaps were loaded, free their resources    */
  /******************************************************************/
  if (hOldBrush)
  {
    SelectObject (hps, hOldBrush);
    DeleteObject (hHalftoneBrush);
    DeleteObject (hHalftonePatternBitmap);
  }

  EndPaint( hwndSld, &paint);
  return;
}

/**********************************************************************/
/* Procedure: PaintButtonOutline - This procedure will paint the      */
/*            outline of the button, leaving the center blank for     */
/*            placing the arrow bitmap on.                            */
/*                                                                    */
/* Inputs: pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*         hwndSld  - slider window handle                            */
/*         prclBtn  - Pointer to the button rectangle to be painted   */
/*         bDown    - If TRUE, button is pressed down                 */
/*                    If FALSE, button is not pressed                 */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID PaintButtonOutline( PISLDDATA pIntData, HDC hps, HANDLE hwndSld,
                         LPRECT prclBtn, BOOL bDown )
{
  POINT  ptl[4];        /* Array of points used for drawing button    */
  long   lButtonMd;     /* Color to use for middle of button          */
  long   lOutlineClr;   /* Color to use for outlining button          */
  USHORT idx;           /* Index count for # of times to draw shading *//*JEH1*/
  HBRUSH hBrushOld, hBrushCurrent;
  HPEN   hPenOld, hPenCurrent;
  COLORREF lShadClr, lReflClr;


  lButtonMd = GetSysColor(COLOR_BTNFACE);

  if (bDown)
  {
    lReflClr = GetSysColor(COLOR_BTNSHADOW);
    lShadClr = lButtonMd;
  }
  else
  {
    lReflClr = RGB (255, 255, 255);
    lShadClr = GetSysColor(COLOR_BTNSHADOW);
  }

  /********************************************************************/
  /* If regular shaft size, outline button with black                 */
  /********************************************************************/
  if (pIntData->usShaftBreadth > 8)
    lOutlineClr = RGB(0x0, 0x0, 0x0);

  /********************************************************************/
  /* otherwise shaft is very small, so make some visual changes       */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Set outline to fill color (so no outline appears)              */
    /******************************************************************/
    lOutlineClr = lButtonMd;

    /******************************************************************/
    /* If button up, then change reflected color to interior color    */
    /******************************************************************/
    if (bDown)
      lShadClr = lButtonMd;

    /******************************************************************/
    /* otherwise down, so change shaded color to interior color       */
    /******************************************************************/
    else
      lReflClr = lButtonMd;
  }

  /********************************************************************/
  /* First draw outline of button filling the interior if requested   */
  /* Don't draw the outline if the brush or pen can't be created      */
  /********************************************************************/
  if ((hBrushCurrent = CreateSolidBrush(lButtonMd)) &&
      (hPenCurrent = CreatePen (PS_SOLID, 1, lOutlineClr)))
  {
    hBrushOld = SelectObject (hps, hBrushCurrent);
    hPenOld = SelectObject (hps, hPenCurrent);

    Rectangle(hps,prclBtn->left,prclBtn->top,prclBtn->right,
              prclBtn->bottom);

    DeleteObject (SelectObject (hps, hPenOld));
    DeleteObject (SelectObject (hps, hBrushOld));
  }

  /********************************************************************/
  /* Perform all of the outline drawing twice if breadth > 44 pixels  */
  /********************************************************************/
  for ( idx=0; idx<(WORD)(1+(pIntData->usShaftBreadth > 44)); idx++)
  {
    /******************************************************************/
    /* Now adjust rectangle to just inside what was drawn if border   */
    /******************************************************************/
    if (pIntData->usShaftBreadth > 8)
      InflateRect( prclBtn , -1, -1);                /* change y sign */

    /******************************************************************/
    /* Set up the points for the four corners of the button, adjusting*/
    /* the points for the way rectangles don't include top/right point*/
    /******************************************************************/
    ptl[0].x = ptl[3].x = prclBtn->left;      /* ptl[0] - upper left  */
    ptl[1].x = ptl[2].x = prclBtn->right-1;   /* ptl[1] - upper right */
    ptl[2].y = ptl[3].y = prclBtn->bottom-1;  /* ptl[2] - lower right */
    ptl[0].y = ptl[1].y = prclBtn->top;       /* ptl[3] - lower left  */

    /******************************************************************/
    /* Now draw the top and left borders of the         0  1    */
    /*  button in reflected light color                              */
    /*                                                    3           */
    /* If either the brush or the pen can't be created, don't draw    */
    /******************************************************************/
    if ((hBrushCurrent = CreateSolidBrush(lButtonMd)) &&
        (hPenCurrent = CreatePen (PS_SOLID, 1, lReflClr)))
    {
      hBrushOld = SelectObject (hps, hBrushCurrent);
      hPenOld = SelectObject (hps, hPenCurrent);

      MoveToEx( hps, ptl[3].x, ptl[3].y-1, NULL );                      /*JEH1*/
      LineTo( hps, ptl[0].x, ptl[0].y-1 );

      Polyline( hps, &ptl[0], 2);

      DeleteObject (SelectObject (hps, hPenOld));
      DeleteObject (SelectObject (hps, hBrushOld));
    }

    /******************************************************************/
    /* Now draw right and bottom borders of the                  1    */
    /*  button in shaded light color.  Adjust start                  */
    /*  and end points just inside light points.          3  2  */
    /******************************************************************/
    ptl[1].y += 1;
//  ptl[3].x += 1;

    /******************************************************************/
    /* Only draw if both the brush and pen can be created             */
    /******************************************************************/
    if ((hBrushCurrent = CreateSolidBrush(lButtonMd)) &&
        (hPenCurrent = CreatePen (PS_SOLID, 1, lShadClr)))
    {
      hBrushOld = SelectObject (hps, hBrushCurrent);
      hPenOld = SelectObject (hps, hPenCurrent);

      Polyline( hps, &ptl[1], 3 );

      DeleteObject (SelectObject (hps, hPenOld));
      DeleteObject (SelectObject (hps, hBrushOld));
    }
  }
  return;
}
/**********************************************************************/
/* Procedure: PaintButtons - This procedure paints the buttons next   */
/*            to the shaft, positioning them relative to it based     */
/*            on the style requested.                                 */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID PaintButtons( HANDLE hwndSld, PISLDDATA pIntData, HDC hps )
{
  ULONG       ulStyle;                                                  /*JEH1*/
  HBITMAP     hbmArrowDecr;
  HBITMAP     hbmArrowIncr;
  USHORT      usArrowSizeX;                                             /*JEH1*/
  USHORT      usArrowSizeY;                                             /*JEH1*/
  POINT      ptlDecr;
  POINT      ptlIncr;
  RECT       rclArrowDecr;
  RECT       rclArrowIncr;
  HDC         hMemDC;
  HBITMAP     hOldBitmap;
  BITMAP      bitmap;

  /********************************************************************/
  /* Query for the style of the slider control and default arrow size */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );
  usArrowSizeX = ARROW_SIZE_STD;
  usArrowSizeY = ARROW_SIZE_STD;

  /********************************************************************/
  /* Check style to determine which arrow bitmaps to use for painting */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* If shaft breadth is less than 11 (smaller than standard arrow) */
    /******************************************************************/
    if (pIntData->usShaftBreadth < 11)
    {
      /****************************************************************/
      /* Set bitmap handles to use the tiny up and down vertical      */
      /* bitmaps and change the arrow size for centering calculations */
      /****************************************************************/
//    hbmArrowDecr = hbmVDecrS;
//    hbmArrowIncr = hbmVIncrS;
      usArrowSizeX = ARROW_SIZE_SMALL_Y;
      usArrowSizeY = ARROW_SIZE_SMALL_X;

      /****************************************************************/
      /* Check if low button is to be disabled                        */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATLOWEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowDecr = hbmVDecrSD;
      else
        hbmArrowDecr = hbmVDecrS;

      /****************************************************************/
      /* Then check if the high end is to be disabled                 */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATHIGHEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowIncr = hbmVIncrSD;
      else
        hbmArrowIncr = hbmVIncrS;

    }
    /******************************************************************/
    /* Otherwise, use regular arrow bitmaps (standard size)           */
    /******************************************************************/
    else
    {
      /****************************************************************/
      /* Check if low button is to be disabled                        */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATLOWEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowDecr = hbmVDecrD;
      else
        hbmArrowDecr = hbmVDecr;

      /****************************************************************/
      /* Then check if the high end is to be disabled                 */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATHIGHEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowIncr = hbmVIncrD;
      else
        hbmArrowIncr = hbmVIncr;
    }
  }

  /********************************************************************/
  /* otherwise style is horizontal so use horizontal arrow bitmaps    */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* If shaft breadth is less than 11 (smaller than standard arrow) */
    /******************************************************************/
    if (pIntData->usShaftBreadth < 11)
    {
      /****************************************************************/
      /* Set bitmap handles to use the tiny up and down horizontal    */
      /* bitmaps and change the arrow size for centering calculations */
      /****************************************************************/
      usArrowSizeX = ARROW_SIZE_SMALL_X;
      usArrowSizeY = ARROW_SIZE_SMALL_Y;
//    hbmArrowDecr = hbmHDecrS;
//    hbmArrowIncr = hbmHIncrS;

      /****************************************************************/
      /* Check if low button is to be disabled                        */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATLOWEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowDecr = hbmHDecrSD;
      else
        hbmArrowDecr = hbmHDecrS;

      /****************************************************************/
      /* Then check if the high end is to be disabled                 */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATHIGHEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowIncr = hbmHIncrSD;
      else
        hbmArrowIncr = hbmHIncrS;

    }
    /******************************************************************/
    /* Otherwise, use regular arrow bitmaps (standard size)           */
    /******************************************************************/
    else
    {
      /****************************************************************/
      /* Check if low button is to be disabled                        */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATLOWEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowDecr = hbmHDecrD;
      else
        hbmArrowDecr = hbmHDecr;

      /****************************************************************/
      /* Then check if the high end is to be disabled                 */
      /****************************************************************/
      if ((pIntData->usStatusSld & FL_SLDARMATHIGHEND) ||
          (ulStyle & WS_DISABLED))
        hbmArrowIncr = hbmHIncrD;
      else
        hbmArrowIncr = hbmHIncr;
    }
  }

  /********************************************************************/
  /* Calculate the two separate button rectangles from the entire rect*/
  /********************************************************************/
  CalcArrowRectangles( hwndSld, pIntData, &rclArrowDecr, &rclArrowIncr);

  /********************************************************************/
  /* Find the x,y coordinates for each button which will center the   */
  /* arrow in the bitmap on the button                                */
  /********************************************************************/
  ptlDecr.x = rclArrowDecr.left + ((( rclArrowDecr.right -
              rclArrowDecr.left ) - usArrowSizeX) >> 1 );
  ptlDecr.y = rclArrowDecr.top + ((( rclArrowDecr.bottom -
              rclArrowDecr.top ) - usArrowSizeY) >> 1 );
  ptlIncr.x = rclArrowIncr.left + ((( rclArrowIncr.right -
              rclArrowIncr.left ) - usArrowSizeX) >> 1 );
  ptlIncr.y = rclArrowIncr.top + ((( rclArrowIncr.bottom -
              rclArrowIncr.top ) - usArrowSizeY) >> 1 );

  /********************************************************************/
  /* Adjust for vertical buttons                                      */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
    ptlIncr.y;

  /********************************************************************/
  /* otherwise adjust for horizontal                                  */
  /********************************************************************/
  else
  {
    ptlIncr.x++;

    /******************************************************************/
    /* Check if even shaft height, if so add one to arrow position    */
    /******************************************************************/
//  if (!(pIntData->usShaftBreadth & 1))
//  {
//    ptlDecr.y++;
//    ptlIncr.y++;
//  }
  }

  /********************************************************************/
  /* Now check the status of the buttons, check if the decrement      */
  /* button is depressed - this part only draws outline of the button */
  /********************************************************************/
  if ( pIntData->usStatusSld & FL_SLDDECREMENTDOWN )
  {
    /******************************************************************/
    /* If so, draw the button as depressed (shadow modified)          */
    /******************************************************************/
    PaintButtonOutline( pIntData, hps, hwndSld, &rclArrowDecr, TRUE );

    /******************************************************************/
    /* Arrow bitmap on depressed button is shifted down and right 1   */
    /* to give an appearance of button movement inward                */
    /******************************************************************/
    ptlDecr.x++;
    ptlDecr.y++;
  }
  /********************************************************************/
  /* Otherwise the button is not depressed                            */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* so draw the button normally (with regular shading)             */
    /******************************************************************/
    PaintButtonOutline( pIntData, hps, hwndSld, &rclArrowDecr, FALSE );
  }

  /********************************************************************/
  /* Now check if the increment button is depressed - this part only  */
  /* draws outline of the button                                      */
  /********************************************************************/
  if ( pIntData->usStatusSld & FL_SLDINCREMENTDOWN )
  {
    /******************************************************************/
    /* If so, draw the button as depressed (shadow modified)          */
    /******************************************************************/
    PaintButtonOutline( pIntData, hps, hwndSld, &rclArrowIncr, TRUE );

    /******************************************************************/
    /* Arrow bitmap on depressed button is shifted down and right 1   */
    /* to give an appearance of button movement inward                */
    /******************************************************************/
    ptlIncr.x++;
    ptlIncr.y++;
  }
  /********************************************************************/
  /* Otherwise the button is not depressed                            */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* so draw the button normally (with regular shading)             */
    /******************************************************************/
    PaintButtonOutline( pIntData, hps, hwndSld, &rclArrowIncr, FALSE );
  }

  /********************************************************************/
  /* Finally, draw the inside of the buttons by placing the arrow     */
  /* bitmaps at the calculated position                               */
  /*                                                                  */
  /* @XXX  -  Modify here to adjust bitmap width ...                  */
  /********************************************************************/

  /********************************************************************/
  /* First create a compatible device context.  If that succeeds, then*/
  /* select the decrement button bitmap into that device context.     */
  /********************************************************************/
  if (hMemDC = CreateCompatibleDC(hps))
    hOldBitmap = SelectObject(hMemDC,hbmArrowDecr);

  /********************************************************************/
  /* Fill the bitmap buffer with the logical bitmap data.  We can     */
  /* only draw the bitmap if this succeeds                            */
  /********************************************************************/
  if (GetObject(hbmArrowDecr, sizeof(BITMAP), (LPSTR)&bitmap))
    BitBlt( hps, ptlDecr.x, ptlDecr.y, (bitmap.bmWidth - 1),
        bitmap.bmHeight, hMemDC, 0, 0, SRCCOPY);

  /********************************************************************/
  /* If the compatible DC was created OK, select the increment button */
  /* bitmap into it                                                   */
  /********************************************************************/
  if (hMemDC)
    SelectObject(hMemDC,hbmArrowIncr);

  /********************************************************************/
  /* Fill the bitmap buffer with the logical bitmap data.  We can     */
  /* only draw the bitmap if this succeeds                            */
  /********************************************************************/
  if (GetObject(hbmArrowIncr, sizeof(BITMAP), (LPSTR)&bitmap))
    BitBlt( hps,ptlIncr.x, ptlIncr.y, (bitmap.bmWidth - 1),
        bitmap.bmHeight, hMemDC, 0, 0,SRCCOPY);

  if (hMemDC)
    SelectObject(hMemDC, hOldBitmap);
  DeleteDC (hMemDC);

  return;
}

/**********************************************************************/
/* Procedure: CalcShaftnButtonPos - This function is called to set or */
/*            change the position of the slider shaft within the      */
/*            slider window.  The shaft will be oriented based on the */
/*            control style bits determining position.                */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CalcShaftnButtonPos( HANDLE hwndSld, PISLDDATA pIntData )
{
  ULONG     ulStyle;                                                    /*JEH1*/
  int     usSliderLength;
  BOOL      bButtons=FALSE;
  RECT     rclWindow;
  int     sWindowTop;
  int     sWindowEdge;
  int     sTemp;
  USHORT    usButtonLength;                                             /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Default the overall slider length to the shaft length            */
  /********************************************************************/
  usSliderLength = pIntData->usShaftLength;

  /********************************************************************/
  /* Get the length of a button based on the current shaft breadth    */
  /********************************************************************/
  usButtonLength = CalcButtonLength( pIntData );

  /********************************************************************/
  /* Determine the overall length of the slider area by checking if   */
  /* buttons are to be included or not.                               */
  /********************************************************************/
  if ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT))
  {
    /******************************************************************/
    /* If buttons included, add 2 times a single button length plus a */
    /* two pel space between the shaft and buttons for slider length  */
    /******************************************************************/
    usSliderLength += ((usButtonLength << 1) + BUTTONSPACE);
    bButtons = TRUE;
  }

  /********************************************************************/
  /* Get the rectangle for the slider window                          */
  /********************************************************************/
  GetClientRect( hwndSld, &rclWindow );

  /********************************************************************/
  /* Set top of window based on orientation                           */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    sWindowTop = rclWindow.right;
    sWindowEdge = rclWindow.bottom;
  }
  else
  {
    sWindowTop = rclWindow.bottom;
    sWindowEdge = rclWindow.right;
  }

  /********************************************************************/
  /* If owner has positioned the shaft, don't recalculate position    */
  /* based on orientation of slider and alignment requested           */
  /********************************************************************/
  if ( !(pIntData->usStatusSld & FL_SLDOWNERPOSITIONED))
  {
    /******************************************************************/
    /* Calculate as if horizontal slider                              */
    /*   Check if requested shaft orientation is at bottom of window  */
    /******************************************************************/
    if (((!(ulStyle & SLS_VERTICAL)) && (ulStyle & SLS_TOP)) ||
         ((ulStyle & SLS_VERTICAL) && (ulStyle & SLS_BOTTOM)))
    {
      /****************************************************************/
      /* If orient at bottom, calculate y coordinate as half of the   */
      /* width that the arm is wider than the shaft                   */
      /****************************************************************/
      pIntData->yShaft = ( pIntData->usArmBreadth -
                                        pIntData->usShaftBreadth ) >> 1;
    }
    /******************************************************************/
    /* If not, check if requested shaft orientation is at window top  */
    /******************************************************************/
    else if (((!(ulStyle & SLS_VERTICAL)) && (ulStyle & SLS_BOTTOM)) ||
         ((ulStyle & SLS_VERTICAL) && (ulStyle & SLS_TOP)))
    {
      /****************************************************************/
      /* If orient at top, calculate y coordinate as the distance the */
      /* shaft + half of the width that the arm is wider than the     */
      /* shaft is from the top of the window.                         */
      /****************************************************************/
      pIntData->yShaft = sWindowTop - (int)pIntData->usShaftBreadth -
                         (int)(( pIntData->usArmBreadth - pIntData->
                         usShaftBreadth ) >> 1);
    }
    else
    {
      /****************************************************************/
      /* Default is to center so calculate the y coordinate:          */
      /*   Take the overall window height, subtract the shaft height  */
      /*   and halve the remaining pixels                             */
      /****************************************************************/
      pIntData->yShaft = ( sWindowTop - (int)pIntData->
                           usShaftBreadth) >> 1;
    }

    /******************************************************************/
    /* Now calculate the x coordinate:                                */
    /*   Take the overall window width, subtract the overall slider   */
    /*   width and halve the remaining pixels.                        */
    /******************************************************************/
    pIntData->xShaft = ( sWindowEdge - (int)usSliderLength ) >> 1;
  }

  /********************************************************************/
  /* If owner positioned, swap coordinates (if vertical) for button   */
  /* positioning calculation                                          */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* If vertical slider, just flip the starting coordinates         */
    /******************************************************************/
    if (ulStyle & SLS_VERTICAL)
    {
      /****************************************************************/
      /* Transpose the x and y shaft coordinates                      */
      /****************************************************************/
      sTemp = pIntData->xShaft;
      pIntData->xShaft = pIntData->yShaft;
      pIntData->yShaft = sTemp;
    }
  }

  /********************************************************************/
  /* Now check for buttons on the left                                */
  /********************************************************************/
  if (ulStyle & SLS_BUTTONSLEFT)
  {
    /******************************************************************/
    /* If so, first save this position as the button position         */
    /******************************************************************/
    pIntData->yButton = pIntData->yShaft;

    if (ulStyle & SLS_VERTICAL)
    {
      pIntData->xButton = pIntData->xShaft + pIntData->usShaftLength +
                          BUTTONSPACE;
    }
    else
    {
      pIntData->xButton = pIntData->xShaft;

    /******************************************************************/
    /* then adjust shaft position to start after buttons              */
    /******************************************************************/
      pIntData->xShaft += ((usButtonLength << 1) + BUTTONSPACE);
    }
  }
  /********************************************************************/
  /* if not on left then check for them on the right                  */
  /********************************************************************/
  else if (ulStyle & SLS_BUTTONSRIGHT)
  {
    /******************************************************************/
    /* If so, then calculate the button position from the shaft       */
    /******************************************************************/
    pIntData->yButton = pIntData->yShaft;

    if (ulStyle & SLS_VERTICAL)
    {
      pIntData->xButton = pIntData->xShaft;

      /****************************************************************/
      /* then adjust shaft position to start after buttons            */
      /****************************************************************/
      pIntData->xShaft += ((usButtonLength << 1) + BUTTONSPACE);
    }
    else
      pIntData->xButton = pIntData->xShaft + pIntData->usShaftLength +
                          BUTTONSPACE;
  }

  /********************************************************************/
  /* Now if vertical slider, just flip the starting coordinates       */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* Transpose the x and y shaft coordinates                        */
    /******************************************************************/
    sTemp = pIntData->xShaft;
    pIntData->xShaft = pIntData->yShaft;
    pIntData->yShaft = sTemp;

    /******************************************************************/
    /* Check for buttons - if none, don't waste time flipping NULL    */
    /******************************************************************/
    if (bButtons)
    {
      /****************************************************************/
      /* Transpose the x and y button coordinates                     */
      /****************************************************************/
      sTemp = pIntData->xButton;
      pIntData->xButton = pIntData->yButton;
      pIntData->yButton = sTemp;
    }
  }
  return;
}


/**********************************************************************/
/* Procedure: CalcRectangle - This function is called to calculate    */
/*            the rectangle for the slider shaft or buttons based on  */
/*            the internal data.  Calculation depends on slider       */
/*            orientation.                                            */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         xStart   - lower left x coordinate of rectangle            */
/*         yStart   - lower left y coordinate of rectangle            */
/*         usLength - Overall length of rectangle                     */
/*         prclCalc - Pointer to calculated rectangle to return in    */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CalcRectangle( HANDLE hwndSld, PISLDDATA pIntData, USHORT xStart,
                    USHORT yStart, USHORT usLength, LPRECT prclCalc )   /*JEH1*/
{
  ULONG  ulStyle;                                                       /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control and anchor block       */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Set rectangle lower left to starting coordinates                 */
  /********************************************************************/
  (*prclCalc).left = xStart;
  (*prclCalc).top  = yStart;

  /********************************************************************/
  /* If vertical add length to y and breadth to x                     */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    (*prclCalc).right = xStart + pIntData->usShaftBreadth;
    (*prclCalc).bottom= yStart + usLength;
  }
  /********************************************************************/
  /* otherwise horizontal so add length to x and breadth to y         */
  /********************************************************************/
  else
  {
    (*prclCalc).right = xStart + usLength;
    (*prclCalc).bottom= yStart + pIntData->usShaftBreadth;
  }
  return;
}


/**********************************************************************/
/* Procedure: CalcArmRectangle - This function is called to calculate */
/*            the rectangle for the slider arm based on the internal  */
/*            data.  Calculation depends on slider orientation.       */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         prclArm  - Pointer to where to place calculated rectangle  */
/*         usArmPos - Position to calculate rectangle at              */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CalcArmRectangle( HANDLE hwndSld, PISLDDATA pIntData,
                       LPRECT prclArm, USHORT usArmPos )                /*JEH1*/
{
  ULONG  ulStyle;                                                       /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* If vertical calculate arm position based off of home position    */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* Check which end is home                                        */
    /******************************************************************/
    if ( ulStyle & SLS_HOMETOP )
    {
      prclArm->bottom = pIntData->yShaft + (BORDERWIDTH + usArmPos +
                        pIntData->usArmLength);
    }
    else
    {
      prclArm->bottom = (pIntData->yShaft + pIntData->usShaftLength) -
                        (BORDERWIDTH + usArmPos);
    }
    /******************************************************************/
    /* X coordinate is same regardless of home                        */
    /******************************************************************/
    prclArm->left = pIntData->xShaft - (((int)(pIntData->
                    usArmBreadth - pIntData->usShaftBreadth )) >> 1);

    /******************************************************************/
    /* Now calculate rest of rectangle from starting point            */
    /******************************************************************/
    prclArm->top   = prclArm->bottom - pIntData->usArmLength;
    prclArm->right = prclArm->left + pIntData->usArmBreadth;
  }
  /********************************************************************/
  /* otherwise horizontal, so calculate based on home position        */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Check which end is home                                        */
    /******************************************************************/
    if ( ulStyle & SLS_HOMERIGHT )
    {
      prclArm->left = (pIntData->xShaft + pIntData->usShaftLength) -
            (BORDERWIDTH + pIntData->usArmLength + usArmPos );
    }
    else
    {
      prclArm->left = pIntData->xShaft + BORDERWIDTH + usArmPos;
    }
    /******************************************************************/
    /* Y coordinate is same regardless of home                        */
    /******************************************************************/
    prclArm->top    = pIntData->yShaft - (((int)(pIntData->
                      usArmBreadth - pIntData->usShaftBreadth )) >> 1);

    /******************************************************************/
    /* Now calculate rest of rectangle from starting point            */
    /******************************************************************/
    prclArm->right = prclArm->left + pIntData->usArmLength;
    prclArm->bottom= prclArm->top + pIntData->usArmBreadth;
  }
  return;
}


/**********************************************************************/
/* Procedure: CalcArrowRectangles - This function is called to        */
/*            calculate the rectangles for each individual button,    */
/*            both the increment and decrement buttons, based on the  */
/*            button starting point and the shaft breadth.            */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         prclDecr - Pointer to calculated rectangle to return in    */
/*         prclIncr - Pointer to calculated rectangle to return in    */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CalcArrowRectangles( HANDLE hwndSld, PISLDDATA pIntData,
                          LPRECT prclDecr, LPRECT prclIncr )
{
  ULONG  ulStyle;                                                       /*JEH1*/
  USHORT usButtonLength;                                                /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control and anchor block       */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Get the length of a button based on the current shaft breadth    */
  /********************************************************************/
  usButtonLength = CalcButtonLength( pIntData );

  /********************************************************************/
  /* Check slider orientation before calculating individual rectangles*/
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* If vertical orientation, decrement rectangle is bottom part    */
    /******************************************************************/
    SetRect( prclIncr, pIntData->xButton,
             pIntData->yButton ,
             pIntData->xButton + pIntData->usShaftBreadth,
             pIntData->yButton + usButtonLength);

    /******************************************************************/
    /* and increment rectangle is just above it                       */
    /******************************************************************/
    SetRect( prclDecr, pIntData->xButton,
             pIntData->yButton + usButtonLength,
             pIntData->xButton + pIntData->usShaftBreadth,
             pIntData->yButton + (usButtonLength << 1));
  }
  /********************************************************************/
  /* Otherwise orientation is horizontal                              */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* which means decrement rectangle is on the left                 */
    /******************************************************************/
    SetRect( prclDecr, pIntData->xButton,
             pIntData->yButton ,
             pIntData->xButton + usButtonLength,
             pIntData->yButton + pIntData->usShaftBreadth);

    /******************************************************************/
    /* and increment rectangle is just to the right of it             */
    /******************************************************************/
    SetRect( prclIncr, pIntData->xButton + usButtonLength,
             pIntData->yButton ,
             pIntData->xButton + (usButtonLength << 1),
             pIntData->yButton + pIntData->usShaftBreadth);
  }
  return;
}


/**********************************************************************/
/* Procedure: CalcButtonLength - This function is called to determine */
/*            the button length based on the breadth of the shaft.    */
/*            Length should be as follows:                            */
/*               7 for shaft widths of  6 - 8                         */
/*              10 for shaft widths of  9 - 10                        */
/*              12 for shaft widths of 11 - 19                        */
/*              14 for shaft widths of 20 - 24                        */
/*              16 for shaft widths of 25 - 29                        */
/*              18 for shaft widths of 30 - 34                        */
/*              .. ... ..... ...... .. .. . ..                        */
/*              28 for shaft widths of 55 - 59                        */
/*              30 for shaft widths over 60                           */
/*                                                                    */
/* Inputs: pIntData  - Pointer to the internal control block          */
/*                                                                    */
/* Output: usButtonLength - Length of button to use                   */
/**********************************************************************/
USHORT CalcButtonLength( PISLDDATA pIntData )                           /*JEH1*/
{
  USHORT usButtonLength=7;                                              /*JEH1*/

  /********************************************************************/
  /* Calculate length of button to handle lower end of shaft size     */
  /********************************************************************/
  usButtonLength += (3*(pIntData->usShaftBreadth > 8)) +
                    (2*(pIntData->usShaftBreadth > 10));

  /********************************************************************/
  /* Now calculate button length to handle mid to upper shaft sizes   */
  /********************************************************************/
  if (pIntData->usShaftBreadth > 19)
    usButtonLength += min( 18, (2*((pIntData->usShaftBreadth-15)/5)));

  /********************************************************************/
  /* Return the length calculated to the calling routine              */
  /********************************************************************/
  return (usButtonLength);
}


/**********************************************************************/
/* Procedure: BltCopy - Worker function to quickly copy from a bitmap */
/*            or PS to a bitmap without shrinkage or expansion.       */
/*                                                                    */
/*  Inputs: hbmSrc - handle of bitmap or PS to copy from              */
/*          hpsDst - handle of PS to copy to                          */
/*          xSrc   - horz offset in hbmSrc to copy from               */
/*          ySrc   - vert offset in hbmSrc to copy from               */
/*          cx     - horz amount to copy (# of pels wide)             */
/*          cy     - vert amount to copy (# of pels high)             */
/*          xDst   - horz offset in hpsDst to copy to                 */
/*          yDst   - vert offset in hpsDst to copy to                 */
/*          rop    - raster operation                                 */
/*          mode   - BLTMODE_SRC_BITMAP - hbmSrc is a bitmap handle   */
/*                   NULL               - hbmSrc is a PS handle       */
/*                                                                    */
/*  Output: None                                                      */
/**********************************************************************/
VOID BltCopy( HBITMAP hbmSrc, HDC hpsDst, LONG xSrc, LONG ySrc,         /*JEH1*/
              LONG cx, LONG cy, LONG xDst, LONG yDst, ULONG rop,        /*JEH1*/
              ULONG mode )                                              /*JEH1*/
{
  POINT aptl[3];
  HDC         hMemDC;
  HBITMAP     hOldBitmap;

  aptl[0].x = xDst;          aptl[0].y = yDst;
  aptl[1].x = xDst + cx;     aptl[1].y = yDst + cy;
  aptl[2].x = xSrc;          aptl[2].y = ySrc;

  if(mode == SRC_BITMAP)
  {
    if (hMemDC = CreateCompatibleDC(hpsDst))
      hOldBitmap = SelectObject(hMemDC,hbmSrc);
  }
  else
    hMemDC = (HDC)hbmSrc;

  if (hMemDC)
    BitBlt( hpsDst, xDst, yDst, cx, cy,
        hMemDC, aptl[2].x, aptl[2].y, rop);

  if(mode == SRC_BITMAP)
  {
    if (hMemDC)
      SelectObject(hMemDC, hOldBitmap);
    DeleteDC (hMemDC);
  }
  return;
}


/**********************************************************************/
/* Procedure: CalcDetentRectangle - This function is called to        */
/*            calculate the rectangle for a detent at a specific      */
/*            position on a specific scale.                           */
/*                                                                    */
/* Inputs: hwndSld   - handle of the slider window                    */
/*         pIntData  - Pointer to the internal control block          */
/*         pDetentId - Pointer to detent id to get rectangle of       */
/*         usDetentSize - Size of rectangle (based off shaft width)   */
/*         prclCalc  - Pointer to calculated rectangle to return in   */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CalcDetentRectangle( HANDLE hwndSld, PISLDDATA pIntData,
                          PDETENTDATA pDetentId, USHORT usDetentSize,   /*JEH1*/
                          LPRECT prclCalc )
{
  ULONG  ulStyle;                                                       /*JEH1*/
  USHORT usTickSpace;                                                   /*JEH1*/
  USHORT usTickOddSpace;                                                /*JEH1*/
  USHORT usTickROSpace=0;                                               /*JEH1*/
  USHORT xDetPos;                                                       /*JEH1*/
  USHORT yDetPos;                                                       /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control and anchor block       */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Calculate spacing of ticks from shaft (just outside of arm)      */
  /* and check if there is an odd number in the pel differential      */
  /********************************************************************/
  usTickSpace = (( pIntData->usArmBreadth - pIntData->usShaftBreadth )
                   >> 1 );
  usTickOddSpace = ((pIntData->usArmBreadth - pIntData->usShaftBreadth)
                   & 1 );

  /********************************************************************/
  /* Set spacing offset if a one pel read only arm is in use - This is*/
  /* to adjust tick alignment since normal arm center is offset on arm*/
  /* and if only one pel wide, cannot draw with offset.               */
  /********************************************************************/
  usTickROSpace = (pIntData->usArmLength == 1);

  /********************************************************************/
  /* Check if vertical or horizontal orientation of slider            */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* If vertical, check which end is home for calculate y coordinate*/
    /******************************************************************/
    if ( ulStyle & SLS_HOMETOP )

      /****************************************************************/
      /* If home top, calculate y position from top of shaft          */
      /****************************************************************/
      yDetPos = pIntData->yShaft +
                (BORDERWIDTH + (pIntData->usArmLength >> 1) +
                usTickROSpace + pDetentId->usDetentPos -
                (usDetentSize >> 1));
    else
      /****************************************************************/
      /* If home bottom, calculate y position from bottom of shaft    */
      /****************************************************************/
      yDetPos = (pIntData->yShaft + pIntData->usShaftLength) -
                (BORDERWIDTH + (pIntData->usArmLength >> 1) +
                pDetentId->usDetentPos + (usDetentSize >> 1));

    /******************************************************************/
    /* Now check if on scale 1 or 2                                   */
    /******************************************************************/
    if (pDetentId->usScale == SMA_SCALE2)

      /****************************************************************/
      /* If scale 2, calculate rectangle on left side of slider       */
      /****************************************************************/
      xDetPos = pIntData->xShaft - (usTickSpace + usDetentSize);

    else
      /****************************************************************/
      /* If scale 1, calculate rectangle on right side of slider      */
      /* If odd differential, extra pel will be on right so add it in */
      /****************************************************************/
      xDetPos = pIntData->xShaft + pIntData->usShaftBreadth +
                usTickSpace + usTickOddSpace;
  }
  /********************************************************************/
  /* otherwise must be horizontal                                     */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* If horizontal, check which end is home for calculate x coord   */
    /******************************************************************/
    if ( ulStyle & SLS_HOMERIGHT )

      /****************************************************************/
      /* If home right, calculate x position from right of shaft      */
      /****************************************************************/
      xDetPos = (pIntData->xShaft + pIntData->usShaftLength) -
                (BORDERWIDTH + (pIntData->usArmLength >> 1) + 1 +
                pDetentId->usDetentPos + (usDetentSize >> 1));

    else
      /****************************************************************/
      /* If horz, calculate x starting coordinate for detent rectangle*/
      /****************************************************************/
      xDetPos = pIntData->xShaft + BORDERWIDTH + (pIntData->usArmLength
               >> 1) - 1 + usTickROSpace + pDetentId->usDetentPos -
               (usDetentSize >> 1);

    /******************************************************************/
    /* Now check if on scale 1 or 2                                   */
    /******************************************************************/
    if (pDetentId->usScale == SMA_SCALE2)

      /****************************************************************/
      /* If scale 2, calculate rectangle on bottom side of slider     */
      /****************************************************************/
      yDetPos = pIntData->yShaft + pIntData->usShaftBreadth +
                                   (usTickSpace);

    else
      /****************************************************************/
      /* If scale 1, calculate rectangle on top side of slider        */
      /* If odd differential, extra pel will be on top so add it in   */
      /****************************************************************/
      yDetPos = pIntData->yShaft - ( usTickSpace + usTickOddSpace +
                                     usDetentSize);
  }

  /********************************************************************/
  /* Calculate rectangle off the above starting points                */
  /********************************************************************/
  SetRect( prclCalc, xDetPos, yDetPos,
           xDetPos + usDetentSize, yDetPos + usDetentSize );

  return;
}


/**********************************************************************/
/* Procedure: CalcRibbonRectangles - This function is called to       */
/*            calculate the rectangles on either side of the slider   */
/*            arm.  These are used for painting the shaft either the  */
/*            normal color or if a ribbon strip is set, the ribbon    */
/*            color.                                                  */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         prclShaft - Pointer to current shaft rectangle             */
/*         prclArm   - Pointer to current arm rectangle               */
/*         prclCalc1 - Pointer to calculated rectangle to return in   */
/*         prclCalc2 - Pointer to calculated rectangle to return in   */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CalcRibbonRectangles( HANDLE hwndSld, LPRECT prclShaft,
                           LPRECT prclArm, LPRECT prclCalc1,
                           LPRECT prclCalc2 )
{
  ULONG  ulStyle;

  /********************************************************************/
  /* Query for the style of the slider control and anchor block       */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* If vertical style calculate bottom shaft rectangle               */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)

    /******************************************************************/
    /* Rectangle is from bottom of shaft to bottom of arm             */
    /******************************************************************/
    SetRect( prclCalc1,
             (int)prclShaft->left, (int)prclArm->bottom,
             (int)prclShaft->right, (int)prclShaft->bottom );

  /********************************************************************/
  /* otherwise calculate left shaft rectangle                         */
  /********************************************************************/
  else

    /******************************************************************/
    /* Rectangle is from left end of shaft to left of arm             */
    /******************************************************************/
    SetRect( prclCalc1,
             (int)prclShaft->left, (int)prclShaft->top,
             (int)prclArm->left, (int)prclShaft->bottom );

  /********************************************************************/
  /* If vertical style calculate top shaft rectangle                  */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)

    /******************************************************************/
    /* Rectangle is from top of arm to top of shaft                   */
    /******************************************************************/
    SetRect( prclCalc2,
             (int)prclShaft->left, (int)prclShaft->top,
             (int)prclShaft->right, (int)prclArm->top );

  /********************************************************************/
  /* otherwise calculate right shaft rectangle                        */
  /********************************************************************/
  else

    /******************************************************************/
    /* Rectangle is from right of arm to right end of shaft           */
    /******************************************************************/
    SetRect( prclCalc2,
             (int)prclArm->right,   (int)prclShaft->top,
             (int)prclShaft->right, (int)prclShaft->bottom);
  return;
}

/**********************************************************************/
/* Procedure: CalcShaftnButtonEdges - This function is called to get  */
/*            the edges of the area bounded by the shaft and if any   */
/*            buttons, those as well.  It will return the left and    */
/*            right (or top and bottom) edges bounded by these.       */
/*                                                                    */
/* Inputs: hwndSld    - handle of the slider window                   */
/*         prclBorder - Pointer to border rectangle                   */
/*         prclButton - Pointer to button rectangle                   */
/*         bOuter     - If TRUE, return outer edges of shaft/buttons  */
/*                    - If FALSE, return inner edges of shaft/buttons */
/*                                                                    */
/* Output: sEdge1     - left/bottom edge of this area                 */
/*         sEdge2     - right/top edge of this area                   */
/**********************************************************************/
long CalcShaftnButtonEdges( HANDLE hwndSld, LPRECT prclBorder,
                            LPRECT prclButton, BOOL bOuter )
{
  ULONG ulStyle;
  int sEdge1;
  int sEdge2;
  int sEdge3=0;
  int sEdge4=0;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Check whether we need the top and bottom edges for vertical      */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* Set edges of overall slider area depending on button position. */
    /* First check the bottom edge for buttons                        */
    /******************************************************************/
    if (ulStyle & SLS_BUTTONSBOTTOM)
    {
      sEdge1 = (int)prclButton->bottom;
      sEdge3 = (int)prclButton->top;
      sEdge4 = (int)prclBorder->bottom;
    }
    else
      sEdge1 = (int)prclBorder->bottom;

    /******************************************************************/
    /* Then check the top edge for buttons                            */
    /******************************************************************/
    if (ulStyle & SLS_BUTTONSTOP)
    {
      sEdge2 = (int)prclButton->top;
      sEdge3 = (int)prclBorder->top;
      sEdge4 = (int)prclButton->bottom;
    }
    else
      sEdge2 = (int)prclBorder->top;
  }
  /********************************************************************/
  /* otherwise we need the left and right edges for horizontal        */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Set edges of overall slider area depending on button position. */
    /* First check the left edge for buttons                          */
    /******************************************************************/
    if (ulStyle & SLS_BUTTONSLEFT)
    {
      sEdge1 = (int)prclButton->left;
      sEdge3 = (int)prclButton->right;
      sEdge4 = (int)prclBorder->left;
    }
    else
      sEdge1 = (int)prclBorder->left;

    /******************************************************************/
    /* Then check the right edge for buttons                          */
    /******************************************************************/
    if (ulStyle & SLS_BUTTONSRIGHT)
    {
      sEdge2 = (int)prclButton->right;
      sEdge3 = (int)prclBorder->right;
      sEdge4 = (int)prclButton->left;
    }
    else
      sEdge2 = (int)prclBorder->right;
  }
  /********************************************************************/
  /* Return the two edges to the calling procedure                    */
  /********************************************************************/
  if (bOuter)
    /******************************************************************/
    /* If outer flag set, return outer edges of button/shaft area     */
    /******************************************************************/
    return ( MAKELONG( sEdge1, sEdge2 ));

  else
    /******************************************************************/
    /* otherwise, return inner edges of button/shaft (only if buttons)*/
    /******************************************************************/
    return ( MAKELONG( sEdge3, sEdge4 ));
}


/**********************************************************************/
/* Procedure: CalcTextRect - This routine gets the pixel length width */
/*            and height of a given text string.                      */
/*                                                                    */
/* Inputs: hps     - presentation space                               */
/*         usLen   - count of charcters in string                     */
/*         pszText - pointer to text string to return length of       */
/*                                                                    */
/* Output: sPxlLength, sPxlWidth - number of pixels in length and     */
/*                   width of string.                                 */
/**********************************************************************/
SIZE CalcTextRect( HDC hps, int usLen, LPSTR pszText )                  /*JEH1*/
{
  SIZE  dwSize;                                                         /*JEH1*/

  /********************************************************************/
  /* Query the text box for the string                                */
  /********************************************************************/
  GetTextExtentPoint(hps, pszText, (int)usLen, &dwSize);                /*JEH1*/
  /********************************************************************/
  /* Calculate length and width - Handle italic fonts                 */
  /********************************************************************/
//sPxlLength = (int)( max( aptl[TXTBOX_TOPRIGHT].x,
//             aptl[TXTBOX_BOTTOMRIGHT].x) - min(aptl[TXTBOX_TOPLEFT].x,
//             aptl[TXTBOX_BOTTOMLEFT].x ));
//sPxlWidth  = (int)( max( aptl[TXTBOX_TOPRIGHT].y,
//             aptl[TXTBOX_TOPLEFT].y) - min(aptl[TXTBOX_BOTTOMRIGHT].y,
//             aptl[TXTBOX_BOTTOMLEFT].y ));

  /********************************************************************/
  /* Return the calculate length and width of the text string         */
  /********************************************************************/
  return (dwSize);
}


/**********************************************************************/
/* Procedure: PaintBackground - This procedure will check if the area */
/*            needs painting.  If so it will then check if the        */
/*            ownerdraw style is set.  If so it will send a drawitem  */
/*            message to the owner giving him the opportunity to      */
/*            paint first.  If he doesn't paint it, then we will.     */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*         prclPaint  - pointer to rectangle to be painted            */
/*         prclQuad   - pointer to rectangle quadrant with background */
/*                      to paint                                      */
/*         bPaintIt   - If TRUE, override intersect check and paint   */
/*                      If FALSE, intersect check is paint criteria   */
/*                                                                    */
/* Output: If TRUE, intersected so paint was necessary                */
/**********************************************************************/
BOOL PaintBackground( HANDLE hwndSld, PISLDDATA pIntData, HDC hps,
                      LPRECT prclPaint, LPRECT prclQuad, BOOL bPaintIt )
{
  ULONG     ulStyle;                                                    /*JEH1*/
  RECT     rclIntersect;
  DRAWITEMSTRUCT ownerItem;
  HBRUSH     hBrushCurrent;
  HBRUSH     hBrushOld;
  int        lPSid = 0;
  HRGN       hrgnCell=NULL;
  HRGN       hrgnOld=NULL;
  ULONG      rc;                                                        /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control and the anchor block   */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Check if quadrant rectangle is in area to be painted             */
  /********************************************************************/
  if (( IntersectRect( &rclIntersect, prclPaint, prclQuad )) ||
      bPaintIt )
  {
    /******************************************************************/
    /* Now check if ownerdraw flag is set                             */
    /******************************************************************/
    if ( ulStyle & SLS_OWNERDRAW )
    {
      ownerItem.CtlType     = ODT_SLIDER;
      ownerItem.CtlID       = GetWindowLong(hwndSld, GWL_ID);           /*JEH1*/
      ownerItem.itemID      = SDA_BACKGROUND;
      ownerItem.itemAction  = ODA_DRAWENTIRE;
      ownerItem.itemState   = 0;                                        /*JEH1*/
      ownerItem.hwndItem    = hwndSld;
      ownerItem.hDC         = hps;
      ownerItem.rcItem      = *prclQuad;
      ownerItem.itemData    = 0;

      /****************************************************************/
      /* Save the current hps (to prevent owner from messing it up)   */
      /****************************************************************/
      lPSid = SaveDC (hps);

      /****************************************************************/
      /* Create a clipping region the size of the background rectangle*/
      /****************************************************************/
      hrgnCell = CreateRectRgn (prclQuad->left, prclQuad->top,
          prclQuad->right, prclQuad->bottom);
      SelectClipRgn (hps, hrgnCell);

      /****************************************************************/
      /* Now send the draw item message to the owner, passing the     */
      /* information to him for drawing in the structure              */
      /****************************************************************/
      rc = SendMessage( GetParent(hwndSld), WM_DRAWITEM,
                                (WPARAM)ownerItem.CtlID,                /*DAB2*/
                                (ULONG)((LPDRAWITEMSTRUCT) &ownerItem));

      /****************************************************************/
      /* If there was no error with saving the DC, restore it         */
      /****************************************************************/
      if (lPSid)
        RestoreDC (hps, lPSid);
      DeleteObject (hrgnCell);

      /****************************************************************/
      /* If the owner drew it, then no further action is needed       */
      /****************************************************************/
      if (rc)
        return (TRUE);

    }
    /******************************************************************/
    /* otherwise if not ownerdraw, or if he did not do anything, paint*/
    /* unless the brush can't be created                              */
    /******************************************************************/
    if (hBrushCurrent =
          CreateSolidBrush(pIntData->ulColorVal[CLR_SLDBACKGROUND]))
    {
      hBrushOld = SelectObject (hps, hBrushCurrent);
      FillRect( hps, prclQuad, hBrushCurrent );
      DeleteObject(SelectObject (hps, hBrushOld));
    }

    return (TRUE);
  }
  /********************************************************************/
  /* If no intersect, then no painting occurred so return FALSE       */
  /********************************************************************/
  return (FALSE);
}


/**********************************************************************/
/* Procedure: PaintBorder - This procedure paints the area around the */
/*            shaft using the internal coordinates for size and       */
/*            starting position.  This consists of a one pel          */
/*            rectangle around the outside dimensions of the shaft    */
/*            and one pel shading just inside this rectangle to show  */
/*            depth.                                                  */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*         prclBorder - pointer to border rectangle to use for paint  */
/*         prclButton - pointer to button rectangle, needed to calc   */
/*                      max extent for background rectangle near shaft*/
/*         prclArm    - pointer to arm rectangle, needed to calc      */
/*                      max extent for background rectangle near shaft*/
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID PaintBorder( HANDLE hwndSld, PISLDDATA pIntData, HDC hps,
                  LPRECT prclBorder, LPRECT prclButton, LPRECT prclArm )
{
  ULONG  ulStyle;
  POINT aptl[3];
  COLORREF crBorderClr;
  COLORREF crShadowClr;
  RECT  rclQuad1;            /* Paint background color in area next  */
  RECT  rclQuad2;            /* to shaft where arm overhangs it and  */
  RECT  rclQuad3;            /* between shaft and buttons (if any)   */
  long   lEdge;
  int  sEdge1;
  int  sEdge2;
  int  sEdge3;
  int  sEdge4;
  ULONG  ulRGB;                                                         /*JEH1*/
  HPEN   hPenCurrent=0;
  HPEN   hPenOld;
  HBRUSH hBrushOld;
  HPEN   hWhitePen;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Get the edges of the border and button area                      */
  /*   First get the outer edges                                      */
  /********************************************************************/
  lEdge = CalcShaftnButtonEdges( hwndSld, prclBorder, prclButton, TRUE);
  sEdge1 = LOWORD( lEdge );
  sEdge2 = HIWORD( lEdge );

  /********************************************************************/
  /* Then get the inner edges                                         */
  /********************************************************************/
  lEdge = CalcShaftnButtonEdges( hwndSld, prclBorder, prclButton,FALSE);
  sEdge3 = LOWORD( lEdge );
  sEdge4 = HIWORD( lEdge );

  /********************************************************************/
  /* Calculate the rectangle for where the arm overhangs the shaft    */
  /* This must be repainted with the background as the arm glides     */
  /* across this area.        ĿĿ */
  /* Also if buttons exist,   *************rclQuad1************** */
  /* calculate area between   ĴA ĿQuadĿ */
  /* the buttons and the       R       SHAFT           **3*BTNS */
  /* shaft as this also must  ĴM **** */
  /* be repainted with the    *************rclQuad2************** */
  /* background.               */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* Calculate the rectangles to the left and right of the shaft    */
    /******************************************************************/
    SetRect( &rclQuad1, (int)prclArm->left,
               sEdge2, (int)prclBorder->left, sEdge1 );
    SetRect( &rclQuad2, (int)prclBorder->right, sEdge2,
               (int)prclArm->right, sEdge1 );
    SetRect( &rclQuad3, (int)prclBorder->left, sEdge4,
               (int)prclBorder->right, sEdge3 );
  }
  /********************************************************************/
  /* otherwise horizontal so calculate above and below rectangles     */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Calculate the rectangles above and below the shaft for horz    */
    /******************************************************************/
    SetRect( &rclQuad1, sEdge1, (int)prclArm->top,
               sEdge2, (int)prclBorder->top );
    SetRect( &rclQuad2, sEdge1, (int)prclBorder->bottom,
               sEdge2, (int)prclArm->bottom );
    SetRect( &rclQuad3, sEdge3, (int)prclBorder->top,
               sEdge4, (int)prclBorder->bottom );
  }

  /********************************************************************/
  /* Finally fill those rectangles with the background color          */
  /********************************************************************/
  PaintBackground( hwndSld, pIntData, hps, &rclQuad1, &rclQuad1, TRUE );
  PaintBackground( hwndSld, pIntData, hps, &rclQuad2, &rclQuad2, TRUE );
  PaintBackground( hwndSld, pIntData, hps, &rclQuad3, &rclQuad3, TRUE );

  /********************************************************************/
  /* Get the system defined color for shadow                          */
  /********************************************************************/
  ulRGB = GetSysColor(COLOR_BTNSHADOW);
  crShadowClr = 0x02000000 | ulRGB;

  /********************************************************************/
  /* Check which color to use based on focus status                   */
  /********************************************************************/
  if ((pIntData->usStatusSld & FL_SLDHASFOCUS) &&
      !(ulStyle & WS_DISABLED))
    hPenCurrent = GetStockObject (BLACK_PEN);
  else
    hPenCurrent = CreatePen (PS_SOLID, 1, crShadowClr);

  if (hPenCurrent)
    hPenOld = SelectObject (hps, hPenCurrent);
  crBorderClr = crShadowClr;

  /********************************************************************/
  /* Draw a one pel border using color set based on focus; if brush   */
  /* or pen couldn't be created, then don't draw                      */
  /********************************************************************/
  if ((hBrushOld = SelectObject (hps, GetStockObject(NULL_BRUSH))) &&
      hPenCurrent)
  {
    Rectangle(hps,prclBorder->left,prclBorder->top,prclBorder->right,
              prclBorder->bottom);
    SelectObject ( hps, hBrushOld);
    DeleteObject (SelectObject (hps, hPenOld));
  }

  /********************************************************************/
  /* First calculate points for drawing shadow inside border shaft    */
  /*    1  2                         */
  /*                                                                 */
  /*      0                                                           */
  /********************************************************************/
  aptl[0].x = aptl[1].x = prclBorder->left + 1;
  aptl[0].y = prclBorder->bottom - 2;
  aptl[1].y = aptl[2].y = prclBorder->top + 1;
  aptl[2].x = prclBorder->right - 2;

  /********************************************************************/
  /* Move to the first point, set color to black, draw shadow lines;  */
  /* if pen can't be created, don't attempt to draw                   */
  /********************************************************************/
  if (hPenCurrent = CreatePen (PS_SOLID, 1, crShadowClr))
  {
    hPenOld = SelectObject (hps, hPenCurrent);
    Polyline(hps, aptl, 3);
    DeleteObject (SelectObject (hps, hPenOld));
  }

  /********************************************************************/
  /* Next calculate points for drawing reflected light inside shaft   */
  /*                                       0                          */
  /*                                                                 */
  /*    2  1                        */
  /********************************************************************/
  aptl[0].x = aptl[1].x = prclBorder->right - 2;
  aptl[0].y = prclBorder->top + 1;
  aptl[1].y = aptl[2].y = prclBorder->bottom - 2;
  aptl[2].x = prclBorder->left + 1;

  /********************************************************************/
  /* Already at first point, set color to white, draw light lines; if */
  /* the stock pen can't be created, then don't draw                  */
  /********************************************************************/
  if (hWhitePen = GetStockObject (WHITE_PEN))
  {
    hPenOld = SelectObject (hps, hWhitePen);
    Polyline(hps, aptl, 3);
    SelectObject ( hps, hPenOld);
  }

  return;
}


/**********************************************************************/
/* Procedure: PaintDetents - This procedure paints the detent marks   */
/*            along the shaft using the information stored for each   */
/*            detent.                                                 */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID PaintDetents( HANDLE hwndSld, PISLDDATA pIntData, HDC hps )
{
  ULONG       ulStyle;                                                  /*JEH1*/
  PDETENTDATA pDetentCurr=NULL;
  RECT       rclDetent;
//ULONG       ulShadowClr;       /* must remove */                      /*JEH1*/
  POINT      ptl[9];
  USHORT      usDetentSize=5;                                           /*JEH1*/
//ULONG       ulRGB;                                                    /*JEH1*/
//BYTE   cRed, cBlue, cGreen;
  HPEN   hPen, hPenOld;
  HBRUSH     hNullBrush;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* First check if there is detent data to be used for drawing       */
  /********************************************************************/
  if ( pDetentCurr = pIntData->pDetentData )
  {
    /******************************************************************/
    /* Calculate size of the detent rectangle based on shaft breadth  */
    /* Detent size should be as follows:                              */
    /*    5 x 5     for shaft widths of  6 - 24                       */
    /*    7 x 7     for shaft widths of 25 - 39                       */
    /*    9 x 9     for shaft widths of 40 - 54                       */
    /*    11 x 11   for shaft widths of 55 - 69                       */
    /*    13 x 13   for shaft widths over 70                          */
    /******************************************************************/
    if ( pIntData->usShaftBreadth > 24 )
      usDetentSize += min( 8, (2*((pIntData->usShaftBreadth-10)/15)));

    /******************************************************************/
    /* If we have detent information stored, then draw it             */
    /******************************************************************/
    do
    {
      /****************************************************************/
      /* Calculate the rectangle of the current detent                */
      /****************************************************************/
      CalcDetentRectangle( hwndSld, pIntData, pDetentCurr, usDetentSize,
                           &rclDetent );

      /****************************************************************/
      /* If this is a vertical detent then           **4**            */
      /*  points are set up as shown on right        **6**            */
      /*  ( 0 & 2 are swapped for scale 2)           01**2            */
      /* Draw from 2 to 0, 3 to 0, 4 to 0,           **5**            */
      /*  5 to 1, 6 to 1 for arrow (all sizes)       **3**            */
      /****************************************************************/
      if (ulStyle & SLS_VERTICAL)
      {
        /**************************************************************/
        /* Set up end y and first start point in middle of rectangle  */
        /**************************************************************/
        ptl[0].y = ptl[1].y = ptl[2].y = rclDetent.top +
                                                    (usDetentSize >> 1);

        /**************************************************************/
        /* Set up other start points x and y at middle outside pts    */
        /**************************************************************/
        ptl[3].y = ptl[7].y = rclDetent.top;
        ptl[4].y = ptl[8].y = rclDetent.bottom - 1;
        ptl[5].y = ptl[3].y - 1;
        ptl[6].y = ptl[4].y + 1;
        ptl[3].x = ptl[4].x = ptl[5].x = ptl[6].x = rclDetent.left
                                                  + (usDetentSize >> 1);

        /**************************************************************/
        /* If vertical and scale 2, draw detent arrow pointing right  */
        /**************************************************************/
        if (pDetentCurr->usScale == SMA_SCALE2)
        {
          ptl[0].x = rclDetent.right - 1;
          ptl[1].x = ptl[0].x - 1;
          ptl[2].x = rclDetent.left;
          ptl[7].x = ptl[3].x - 1;
          ptl[8].x = ptl[4].x - 1;
        }
        /**************************************************************/
        /* If vertical and scale 1, draw detent arrow pointing left   */
        /**************************************************************/
        else
        {
          ptl[0].x = rclDetent.left;
          ptl[1].x = ptl[0].x + 1;
          ptl[2].x = rclDetent.right - 1;
          ptl[7].x = ptl[3].x + 1;
          ptl[8].x = ptl[4].x + 1;
        }
      }
      /****************************************************************/
      /* Otherwise this is a horizontal detent       **2**            */
      /*  Points are set up as shown on right        **1**            */
      /*  ( 0 & 2 are swapped for scale 2)           35*64            */
      /* Draw from 2 to 0, 3 to 0, 4 to 0,           **1**            */
      /*  5 to 1, 6 to 1 for arrow (all sizes)       **0**            */
      /****************************************************************/
      else
      {
        /**************************************************************/
        /* Set up end x and first start point in middle of rectangle  */
        /**************************************************************/
        ptl[0].x = ptl[1].x = ptl[2].x = rclDetent.left +
                                                    (usDetentSize >> 1);

        /**************************************************************/
        /* Set up other start points x and y at middle outside pts    */
        /**************************************************************/
        ptl[3].x = ptl[7].x = rclDetent.left;
        ptl[4].x = ptl[8].x = rclDetent.right - 1;
        ptl[5].x = ptl[3].x + 1;
        ptl[6].x = ptl[4].x - 1;
        ptl[3].y = ptl[4].y = ptl[5].y = ptl[6].y = rclDetent.bottom
                                         - ((usDetentSize >> 1)+1);

        /**************************************************************/
        /* If horizontal and scale 2, draw detent arrow pointing up   */
        /**************************************************************/
        if (pDetentCurr->usScale == SMA_SCALE2)
        {
          ptl[0].y = rclDetent.top;
          ptl[1].y = ptl[0].y + 1;
          ptl[2].y = rclDetent.bottom - 1;
          ptl[7].y = ptl[3].y + 1;
          ptl[8].y = ptl[4].y + 1;
        }
        /**************************************************************/
        /* If horizontal and scale 1, draw detent arrow pointing down */
        /**************************************************************/
        else
        {
          ptl[0].y = rclDetent.bottom - 1;
          ptl[1].y = ptl[0].y - 1;
          ptl[2].y = rclDetent.top;
          ptl[7].y = ptl[3].y - 1;
          ptl[8].y = ptl[4].y - 1;
        }
      }

      /****************************************************************/
      /* Set the color of the detent to the text color                */
      /****************************************************************/
//    ulRGB = GetSysColor(COLOR_WINDOWTEXT);
//    ulShadowClr = 0x02000000 | ulRGB;
//    cRed   = LOBYTE(HIWORD(ulRGB));
//    cGreen = HIBYTE(LOWORD(ulRGB));
//    cBlue  = LOBYTE(LOWORD(ulRGB));

      /****************************************************************/
      /* Draw the detent using points set up above, but don't attempt */
      /* to draw if the brush and pen can't be created                */
      /****************************************************************/
      if ((hNullBrush = GetStockObject (NULL_BRUSH)) &&
          (hPen    = CreatePen (PS_SOLID, 1,
                                pIntData->ulColorVal[CLR_SLDFOREGROUND])))
      {
        SelectObject (hps, hNullBrush);
        hPenOld = SelectObject (hps, hPen );

        MoveToEx (hps, ptl[2].x, ptl[2].y, NULL);                       /*JEH1*/
        LineTo( hps, ptl[0].x, ptl[0].y);
        MoveToEx (hps, ptl[3].x, ptl[3].y, NULL);                       /*JEH1*/
        LineTo( hps, ptl[0].x, ptl[0].y);
        MoveToEx (hps, ptl[4].x, ptl[4].y, NULL);                       /*JEH1*/
        LineTo( hps, ptl[0].x, ptl[0].y);
        MoveToEx (hps, ptl[5].x, ptl[5].y, NULL);                       /*JEH1*/
        LineTo( hps, ptl[1].x, ptl[1].y);
        MoveToEx (hps, ptl[6].x, ptl[6].y, NULL);                       /*JEH1*/
        LineTo( hps, ptl[1].x, ptl[1].y);

        DeleteObject(SelectObject (hps, hPenOld));
      }

      SetPixel ( hps, ptl[0].x, ptl[0].y,
                 pIntData->ulColorVal[CLR_SLDFOREGROUND]);
      /****************************************************************/
      /* If size of 7 or 13, add corner points to detent              */
      /****************************************************************/
      if ( ( usDetentSize == 7 ) || ( usDetentSize == 13 ))
      {
        SetPixel ( hps, ptl[7].x, ptl[7].y,
                   pIntData->ulColorVal[CLR_SLDFOREGROUND]);
        SetPixel ( hps, ptl[8].x, ptl[8].y,
                   pIntData->ulColorVal[CLR_SLDFOREGROUND]);
      }

      /****************************************************************/
      /* Set the next detent as the current detent and repeat (if any)*/
      /****************************************************************/
    } while ( pDetentCurr = pDetentCurr->pNext );
  }
  return;
}


/**********************************************************************/
/* Procedure: PaintShaft - This procedure paints the shaft using the  */
/*            internal coordinates for size and starting position.    */
/*            The areas to be painted are on either side of the arm   */
/*            just inside the slider border w/shading.                */
/*                               Ŀ                                 */
/*          rclShaft1 -> Ĵ  Ŀ <- rclShaft2   */
/*                       Ĵ                  */
/*                                                                */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         hps      - presentation space to draw into                 */
/*         prclBorder - pointer to border rectangle to use for paint  */
/*         prclArm    - pointer to arm rectangle to use for paint     */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID PaintShaft( HANDLE hwndSld, HDC hps, LPRECT prclBorder,
                 LPRECT prclArm )
{
  ULONG  ulStyle;                                                       /*JEH1*/
  RECT  rclShaft1;
  RECT  rclShaft2;
  ULONG  ulColor1;                                                      /*JEH1*/
  ULONG  ulColor2;                                                      /*JEH1*/
  DRAWITEMSTRUCT ownerItem;
  RECT  rclBorder;
  ULONG  ulShadowClr;                                                   /*JEH1*/
  USHORT idx;                                                           /*JEH1*/
  USHORT usRibbon=0;                                                    /*JEH1*/
  ULONG  ulRGB;                                                         /*JEH1*/
  HBRUSH hBrushCurrent;
  HBRUSH hBrushOld;
  BOOL   bOwnerDrew1=FALSE;
  BOOL   bOwnerDrew2=FALSE;
  int    lPSid=0;
  HRGN   hrgnCell;
  ULONG  rc;                                                            /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* Get the system defined color for shadow                          */
  /********************************************************************/
  ulRGB = GetSysColor(COLOR_BTNSHADOW);
  ulShadowClr = 0x02000000 | ulRGB;

  /********************************************************************/
  /* Adjust the rectangle to just inside the border                   */
  /********************************************************************/
  rclBorder = *prclBorder;
  InflateRect( &rclBorder, -BORDERWIDTH, -BORDERWIDTH );

  /********************************************************************/
  /* Now calculate the rectangles on either side of the slider arm    */
  /********************************************************************/
  CalcRibbonRectangles( hwndSld, &rclBorder, prclArm, &rclShaft1,
                        &rclShaft2 );

  /********************************************************************/
  /* Set up default color for shaft drawing                           */
  /********************************************************************/
  ulRGB = GetSysColor(COLOR_BTNFACE);
  ulColor1 = 0x02000000 | ulRGB;
  ulColor2 = ulColor1;

  /********************************************************************/
  /* Now check if a ribbon strip is to be drawn                       */
  /********************************************************************/
  if ( ulStyle & SLS_RIBBONSTRIP )
  {
    /******************************************************************/
    /* If so check which side it is attached to                       */
    /******************************************************************/
    if ( ulStyle & SLS_HOMERIGHT )
    {
      /****************************************************************/
      /* If home on right, then set shaft color to dark and set flag  */
      /****************************************************************/
      ulColor2 = ulShadowClr;
      usRibbon = 2;
    }
    else
    {
      /****************************************************************/
      /* If home on left, then set shaft color to dark and set flag   */
      /****************************************************************/
      ulColor1 = ulShadowClr;
      usRibbon = 1;
    }
  }

  /********************************************************************/
  /* Now check if ownerdraw flag is set                               */
  /********************************************************************/

  if ( ulStyle & SLS_OWNERDRAW )
  {
    /******************************************************************/
    /* If so, for each side of the shaft, give the owner the          */
    /* opportunity to paint the shaft area.                           */
    /******************************************************************/
    for ( idx=1; idx <= 2; idx++)
    {
      /****************************************************************/
      /* Fill up the owner item structure with the correct information*/
      /****************************************************************/
      ownerItem.CtlType     = ODT_SLIDER;
      ownerItem.CtlID       = GetWindowLong(hwndSld, GWL_ID);           /*JEH1*/
      ownerItem.itemAction  = ODA_DRAWENTIRE;
      ownerItem.itemState   = 0;                                        /*JEH1*/
      ownerItem.hwndItem    = hwndSld;
      ownerItem.hDC         = hps;
      ownerItem.itemData    = 0;                                        /*JEH1*/

      /****************************************************************/
      /* Pass rectangle for the side of the shaft to be drawn         */
      /****************************************************************/
      if ( idx == 1 )
        ownerItem.rcItem = rclShaft1;
      else
        ownerItem.rcItem = rclShaft2;

      /****************************************************************/
      /* Check if a ribbon strip is attached to this side             */
      /****************************************************************/
      if ( idx == usRibbon )

        /**************************************************************/
        /* If ribbon strip is attached to this side, identify as such */
        /**************************************************************/
        ownerItem.itemID = SDA_RIBBONSTRIP;

      else
        /**************************************************************/
        /* otherwise no ribbon strip so identify as normal shaft draw */
        /**************************************************************/
        ownerItem.itemID = SDA_SLIDERSHAFT;

      /****************************************************************/
      /* Save the current hps (to prevent owner from messing it up)   */
      /****************************************************************/
      lPSid = SaveDC (hps);

      /****************************************************************/
      /* Create a clipping region the size of the background rect     */
      /****************************************************************/
      hrgnCell = CreateRectRgn (ownerItem.rcItem.left,
          ownerItem.rcItem.top,
          ownerItem.rcItem.right,
          ownerItem.rcItem.bottom);
      SelectClipRgn (hps, hrgnCell);

      /****************************************************************/
      /* Now send the draw item message to the owner, passing the     */
      /* information to him for drawing in the structure              */
      /****************************************************************/
      rc = SendMessage( GetParent( hwndSld), WM_DRAWITEM,               /*JEH1*/
                        (WPARAM)ownerItem.CtlID,                        /*DAB2*/
                        (ULONG)((LPDRAWITEMSTRUCT)&ownerItem ));        /*JEH1*/

      /****************************************************************/
      /* If there was no error with saving the DC, restore it         */
      /****************************************************************/
      if (lPSid)
        RestoreDC (hps, lPSid);
      DeleteObject (hrgnCell);

      /****************************************************************/
      /* If the owner drew it, then set flag so we don't paint it     */
      /****************************************************************/
      if (rc)
      {
        if (idx == 1)
          bOwnerDrew1 = TRUE;
        else
          bOwnerDrew2 = TRUE;
      }
    }
  }

  /********************************************************************/
  /* Draw the shafts using the set shaft colors, but don't attempt to */
  /* draw either if the associated brush can't be created             */
  /********************************************************************/
  if ((!bOwnerDrew1) && (hBrushCurrent = CreateSolidBrush(ulColor1)))
  {
    hBrushOld = SelectObject (hps, hBrushCurrent);
    FillRect( hps, &rclShaft1, hBrushCurrent );
    DeleteObject(SelectObject (hps, hBrushOld));
  }

  if ((!bOwnerDrew1) && (hBrushCurrent = CreateSolidBrush(ulColor2)))
  {
    hBrushOld = SelectObject (hps, hBrushCurrent);
    FillRect( hps, &rclShaft2, hBrushCurrent );
    DeleteObject(SelectObject (hps, hBrushOld));
  }

  return;
}


/**********************************************************************/
/* Procedure: PaintTicks - This procedure paints the tick marks along */
/*            the shaft using the information stored along with any   */
/*            associated text.  It will traverse the tick linked list,*/
/*            determine for each tick found whether it is to be drawn,*/
/*            how big it should be, where it is to be drawn and if    */
/*            there is any text to draw, where it should be drawn.    */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID PaintTicks( HANDLE hwndSld, PISLDDATA pIntData, HDC hps )
{
  ULONG     ulStyle;                                                    /*JEH1*/
  PTICKDATA pTickCurr=NULL;
  ULONG     ulTickPos;                                                  /*JEH1*/
  POINT    ptlTick;
  POINT    ptlText;
  int       usLen;                                                      /*JEH1*/
  SIZE      lTextSize;                                                  /*JEH1*/
  long      lTextClr;
  long      lTickClr;
  TEXTMETRIC fmText;
  ULONG     ulOldRGBValue;                                              /*JEH1*/
//ULONG     ulRGB;                                                      /*JEH1*/
  int       usOldBKMode;
  HPEN      hPen, hPenOld;
  HFONT     hOldFont = NULL;
  HBRUSH    hNullBrush;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = GetWindowLong( hwndSld, GWL_STYLE );

  /********************************************************************/
  /* If this slider has a special font, select it into the hps        */
  /********************************************************************/
  if (pIntData->hSliderFont)
  {
    hOldFont = SelectObject (hps, pIntData->hSliderFont);
  }

  /********************************************************************/
  /* Query for the current font metrics information (to get spacing)  */
  /********************************************************************/
  GetTextMetrics(hps, &fmText);

  /********************************************************************/
  /* Get the system defined color for shadow and output text          */
  /********************************************************************/
//ulRGB = GetSysColor(COLOR_WINDOWTEXT);
  lTickClr = pIntData->ulColorVal[CLR_SLDFOREGROUND];

  /********************************************************************/
  /* If the slider is disabled, use grayed text -- otherwise regular  */
  /* window text color                                                */
  /********************************************************************/
  if (!(ulStyle & WS_DISABLED))
    lTextClr = pIntData->ulColorVal[CLR_SLDFOREGROUND];
  else
    lTextClr =  0x02000000 | (GetSysColor (COLOR_GRAYTEXT));
//lTextClr = 0x02000000 | ulRGB;

  /********************************************************************/
  /* First check if there is tick data to be used for drawing         */
  /********************************************************************/
  if ( pTickCurr = pIntData->pTickData )
  {
    /******************************************************************/
    /* Create the pen and brush to use for drawing tick data; we're   */
    /* only going to attempt to draw tick stuff if they can be created*/
    /******************************************************************/
    if ((hNullBrush = GetStockObject (NULL_BRUSH)) &&
        (hPen = CreatePen (PS_SOLID, 1, lTickClr)))
    {
      SelectObject (hps, hNullBrush);
      hPenOld = SelectObject (hps, hPen);

      /****************************************************************/
      /* If we have tick information stored, then draw it             */
      /****************************************************************/
      do
      {
        /**************************************************************/
        /* First, get the x,y position of the tick specified and      */
        /* convert them to a point type                               */
        /**************************************************************/
        ulTickPos = QueryTickPos( hwndSld, pIntData, pTickCurr->usScale,
                                  pTickCurr->usTickNum );
        ptlTick.x = ptlText.x = LOWORD( ulTickPos );
        ptlTick.y = ptlText.y = HIWORD( ulTickPos );

        /**************************************************************/
        /* Next determine if there is a physical tick size for tick   */
        /**************************************************************/
        if ( pTickCurr->usTickSize )
        {
          /************************************************************/
          /* If so then move to that point on the slider              */
          /************************************************************/
          MoveToEx(hps, ptlTick.x, ptlTick.y, NULL);                    /*JEH1*/
          /************************************************************/
          /* Now check which direction to draw in - check which scale */
          /************************************************************/
          if ( pTickCurr->usScale == SMA_SCALE1 )
          {
            /**********************************************************/
            /* If Scale one, check orientation of slider              */
            /**********************************************************/
            if (ulStyle & SLS_VERTICAL)
            {
              /********************************************************/
              /* If vertical, then draw tick to the left              */
              /********************************************************/
              ptlTick.x += pTickCurr->usTickSize;
              ptlText.x = ptlTick.x + (fmText.tmAveCharWidth >> 1) - 1;
            }
            else
            {
              /********************************************************/
              /* otherwise horizontal so draw tick upwards            */
              /********************************************************/
              ptlTick.y -= pTickCurr->usTickSize;
              ptlText.y = ptlTick.y - (fmText.tmAveCharWidth >> 1) + 1;
            }
          }
          /************************************************************/
          /* otherwise scale 2 (below or right of shaft)              */
          /************************************************************/
          else
          {
            /**********************************************************/
            /* If scale two, check orientation of slider              */
            /**********************************************************/
            if (ulStyle & SLS_VERTICAL)
            {
              /********************************************************/
              /* If vertical, then draw tick to the right             */
              /********************************************************/
              ptlTick.x -= pTickCurr->usTickSize;
              ptlText.x = ptlTick.x - (fmText.tmAveCharWidth >> 1) + 1;
            }
            else
            {
              /********************************************************/
              /* otherwise horizontal so draw tick downwards          */
              /********************************************************/
              ptlTick.y += pTickCurr->usTickSize;
              ptlText.y = ptlTick.y + (fmText.tmAveCharWidth >> 1) - 1;
            }
          }
          /************************************************************/
          /* Now that we have the new point, draw to that point       */
          /************************************************************/
          LineTo(hps, ptlTick.x, ptlTick.y);
        }

        /**************************************************************/
        /* Now check if there is text to be drawn with this tick      */
        /**************************************************************/
        if ( pTickCurr->pszText )
        {
          /************************************************************/
          /* Calculate pixel size of string to paint (for centering)  */
          /************************************************************/
          usLen = lstrlen( pTickCurr->pszText );
          lTextSize = CalcTextRect( hps, usLen, pTickCurr->pszText );

          /************************************************************/
          /* Check orientation of slider for centering text on tick   */
          /************************************************************/
          if (ulStyle & SLS_VERTICAL)
          {
            /**********************************************************/
            /* If vertical, shift y coordinate down half text height  */
            /**********************************************************/
            ptlText.y += (lTextSize.cy >> 1);                           /*JEH1*/

            /**********************************************************/
            /* If on left scale, shift it left full string width      */
            /**********************************************************/
            if ( pTickCurr->usScale == SMA_SCALE2 )
              ptlText.x -= lTextSize.cx;                                /*JEH1*/
          }
          /************************************************************/
          /* otherwise horizontal                                     */
          /************************************************************/
          else
          {
            /**********************************************************/
            /* so shift x coordinate left half text width             */
            /**********************************************************/
            ptlText.x -= (lTextSize.cx >> 1);                           /*JEH1*/

            /**********************************************************/
            /* If on bottom scale, shift it down full character height*/
            /**********************************************************/
            if ( pTickCurr->usScale == SMA_SCALE2 )
              ptlText.y += lTextSize.cy;                                /*JEH1*/
          }
          /************************************************************/
          /* Adjust y coordinate by max descender value to get proper */
          /* baseline positioning to start drawing text from          */
          /************************************************************/
          ptlText.y -= fmText.tmDescent;

          /************************************************************/
          /* If valid string, draw the text in the box                */
          /************************************************************/
          SetTextAlign(hps,TA_LEFT | TA_BASELINE | TA_NOUPDATECP);
          ulOldRGBValue = SetTextColor(hps, lTextClr);
          usOldBKMode = SetBkMode(hps, TRANSPARENT);
          TextOut( hps, ptlText.x, ptlText.y,
              (LPSTR)(pTickCurr->pszText), usLen);
          SetTextColor(hps, ulOldRGBValue);
          SetBkMode(hps, usOldBKMode);

        }

        /**************************************************************/
        /* Set the next tick as the current tick and repeat (if any)  */
        /**************************************************************/
      } while ( pTickCurr = pTickCurr->pNext );
      DeleteObject (SelectObject (hps, hPenOld));
    }
  }
  /********************************************************************/
  /* Reset the old font if it was changed                             */
  /********************************************************************/
  if (hOldFont)
    SelectObject (hps, hOldFont);

  return;
}

/**********************************************************************/
/* Procedure: UpdateArmPosition - This function is called to update   */
/*            the position of the slider arm in units for the scale.  */
/*            It verifies the position to insure it does not exceed   */
/*            the maximum range for the slider.                       */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         usArmPos - New position to try to set the arm to           */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
SHORT UpdateArmPosition( HANDLE hwndSld, PISLDDATA pIntData,            /*JEH1*/
                         USHORT usArmPos )                              /*JEH1*/
{
  USHORT usMaxRange;                                                    /*JEH1*/

  /********************************************************************/
  /* Query for the maximum range for the current scale                */
  /********************************************************************/
  usMaxRange = QueryMaxRange( hwndSld, pIntData );

  /********************************************************************/
  /* Check to insure not above maximum range, otherwise set to max    */
  /********************************************************************/
  if ( usArmPos > usMaxRange )

    /******************************************************************/
    /* If greater than max, set to max position                       */
    /******************************************************************/
    usArmPos = usMaxRange;

  /********************************************************************/
  /* Return new position to the calling routine                       */
  /********************************************************************/
  return (usArmPos);
}
