/* Revision: 57 1.6 source/ui/cclnt/slider/slidera.c, cclslid, ioc.v400, 001006  */
/**********************************************************************/
/*                                                                    */
/*                         IBM CONFIDENTIAL                           */
/*                                                                    */
/*        (IBM Confidential-Restricted when combined with the         */
/*         aggregated OCO source modules for this program)            */
/*                                                                    */
/*                       OCO SOURCE MATERIALS                         */
/*                                                                    */
/*         20G0400 (C) Copyright IBM Corp. 1992, 1997 (Unpublished)   */
/*                                                                    */
/**********************************************************************/
/**********************************************************************/
/*                                                                    */
/*                           S L I D E R                              */
/*                                                                    */
/*                       Filename: SLIDERA.C                          */
/*                                                                    */
/*              ( Windows NT Slider Control Component )               */
/*                                                                    */
/*                         IBM CONFIDENTIAL                           */
/*                                                                    */
/*                (C) Copyright IBM Corporation 1997                  */
/*                       All Rights Reserved                          */
/*            Licensed Material - Program Property of IBM             */
/*                                                                    */
/**********************************************************************/
/**********************************************************************/
/* Product: SLIDER.DLL                                                */
/* Filename: SLIDERA.C - Slider arm paint/control routines for slider */
/*                                                                    */
/* Flag    Date    By:  Description                                   */
/* ----  --------  ---  --------------------------------------------- */
/* JEH1  06/21/93  JEH  Initial modifications for NT execution        */
/* DAB1  11/12/93  DAB  Added logic to prime blit rectangle w/ backgnd*/
/* DAB2  10/11/95  DAB  Fixed problem with ownerdraw cnr id in mp1    */
/**********************************************************************/
#include <string.h>

#include <windows.h>
#include "iclsldw.h"               /* Slider control public includes  */
#include "sliderp.h"               /* Slider control private includes */


/**********************************************************************/
/* Global handles for Arrow Bitmaps (declared in SLIDERM.C)           */
/**********************************************************************/
extern HDC hpsMem;
extern HDC hpsDragArm;
extern HBITMAP hbmHDecr;
extern HBITMAP hbmScrSave;
extern HBITMAP hbmIcon;

/**********************************************************************/
/* Procedure: CreateSliderArm - This procedure will create a bitmap   */
/*            in memory and draw the arm into it using the dimensions */
/*            of the arm that are specified in the internal data.     */
/*            The bitmap will be drawn based on the orientation of    */
/*            the slider and will display focus attributes.           */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID CreateSliderArm( HANDLE hwndSld, PISLDDATA pIntData )
{
  ULONG            ulStyle;   /* Current style bit flags              *//*JEH1*/
  BITMAP           bmp;       /* Bitmap information used for create   */
  HBITMAP          hOldBitmap;
  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Create a bitmap information header block for the arm bitmap by   */
  /* querying the data for a button bitmap and changing the dimensions*/
  /* If we can't get the bitmap data, don't attempt to draw it        */
  /********************************************************************/
  if (GetObject( hbmHDecr, sizeof(BITMAP), (LPSTR)&bmp ))
  {

    /******************************************************************/
    /* Change the bitmap dimensions to that of the slider arm         */
    /******************************************************************/
    if (ulStyle & SLS_VERTICAL)
    {
      /****************************************************************/
      /* If vertical slider, breadth is in x direction, length in y   */
      /****************************************************************/
      bmp.bmWidth  = pIntData->usArmBreadth;
      bmp.bmHeight = pIntData->usArmLength;
    }
    else
    {
      /****************************************************************/
      /* If horizontal slider, length is in x direction, breadth in y */
      /****************************************************************/
      bmp.bmWidth  = pIntData->usArmLength;
      bmp.bmHeight = pIntData->usArmBreadth;
    }

    /******************************************************************/
    /* If an arm bitmap exists, delete it as we are creating new one  */
    /******************************************************************/
    if (pIntData->hbmArm)
      DeleteObject( pIntData->hbmArm );

    /******************************************************************/
    /* Create the arm bitmap in the memory presentation space using   */
    /* bitmap info header set earlier and set bitmap into dev context;*/
    /* if the bitmap can't be created, don't attempt to draw it       */
    /******************************************************************/
    if (pIntData->hbmArm = CreateBitmap(bmp.bmWidth, bmp.bmHeight,
                       bmp.bmPlanes, bmp.bmBitsPixel, (LPSTR)0L))
    {

      hOldBitmap = SelectObject( hpsMem, pIntData->hbmArm );
      /****************************************************************/
      /* Now draw the details of the arm                              */
      /****************************************************************/
      DrawSliderArm( hwndSld, pIntData );

      /****************************************************************/
      /* Finally disassociate the arm bitmap from the dev context     */
      /****************************************************************/
      SelectObject( hpsMem, hOldBitmap );
    }

  }
  return;
}



/**********************************************************************/
/* Procedure: DrawSliderArm - This procedure will paint the contents  */
/*            of the slider arm in memory based on focus and style    */
/*            of the slider.  This is done for fast drawing of the    */
/*            arm on the screen.  The arm will be either a readonly   */
/*            style arm or a regular arm, with or without focus       */
/*            attributes.                                             */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID DrawSliderArm( HANDLE hwndSld, PISLDDATA pIntData )
{
  ULONG  ulStyle;             /* Current style bit flags              *//*JEH1*/
  POINT ptl[4];
  POINT ptlRefl[4];
  RECT  rclFocus;
  RECT  rclSliderArm;
  SHORT  sCx1;                                                          /*JEH1*/
  SHORT  sCy1;                                                          /*JEH1*/
  SHORT  sCx2;                                                          /*JEH1*/
  SHORT  sCy2;
  long   lBorderClr;
  long   lShadowClr;
  DRAWITEMSTRUCT ownerItem;
  ULONG  ulRGB;                                                         /*JEH1*/
  HBRUSH     hBrushCurrent;
  HBRUSH     hBrushOld;
  HPEN       hPenCurrent;
  HPEN       hPenOld;
  SHORT    lPSid=0;                                                     /*JEH1*/
  ULONG  rc;                                                            /*JEH1*/


  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Set up points to the four outside corners of the slider arm      */
  /*  0Ŀ1    Use upper left as starting point since difference    */
  /*  32    between horizontal and vertical is pivot on this pt  */
  /********************************************************************/
  ptl[0].x = ptl[3].x = 0;
  ptl[0].y = ptl[1].y = 0;                              /* swap for Win*/

  /********************************************************************/
  /* If vertical, then breadth is in x direction, length is in y      */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    ptl[1].x = ptl[2].x = pIntData->usArmBreadth-1;
    ptl[2].y = ptl[3].y = pIntData->usArmLength-1;      /*swap for Win*/
  }
  /********************************************************************/
  /* otherwise horizontal so length is in x direction, breadth is in y*/
  /********************************************************************/
  else
  {
    ptl[1].x = ptl[2].x = pIntData->usArmLength-1;
    ptl[2].y = ptl[3].y = pIntData->usArmBreadth-1;     /*swap for Win*/
  }

  /********************************************************************/
  /* Create a rectangle governing this bitmap area for the arm        */
  /********************************************************************/
  SetRect(&rclSliderArm, (int)ptl[0].x, (int)ptl[0].y,     /* win */
            (int)ptl[2].x+1, (int)ptl[2].y+1);

  /********************************************************************/
  /* Now check if ownerdraw flag is set                               */
  /********************************************************************/
  if ( ulStyle & SLS_OWNERDRAW )
  {
    ownerItem.CtlType     = ODT_SLIDER;
    ownerItem.CtlID       = IDOF(hwndSld);                              /*JEH1*/
    ownerItem.itemID      = SDA_SLIDERARM;
    ownerItem.itemAction  = ODA_DRAWENTIRE;
    ownerItem.itemState   = 0;
    ownerItem.hwndItem    = hwndSld;
    ownerItem.hDC         = hpsMem;
    ownerItem.rcItem      = rclSliderArm;
    ownerItem.itemData    = 0;                                          /*JEH1*/

    if (pIntData->usStatusSld & FL_SLDFOCUSCHANGE)
      ownerItem.itemAction |= ODA_FOCUS;
    if (pIntData->usStatusSld & FL_SLDHASFOCUS)
      ownerItem.itemState |= ODS_FOCUS;
    if (ulStyle & WS_DISABLED)
      ownerItem.itemState |= ODS_DISABLED;

    /******************************************************************/
    /* Save the current hps to prevent the owner from messing it up   */
    /******************************************************************/
    lPSid = SaveDC (hpsMem);

    /******************************************************************/
    /* Now send the draw item message to the owner, passing the       */
    /* information to him for drawing in the structure                */
    /******************************************************************/
    rc = SendMessage( PARENTOF(hwndSld), WM_DRAWITEM,                   /*JEH1*/
                      (WPARAM)ownerItem.CtlID,                          /*DAB2*/
                      (ULONG)((LPDRAWITEMSTRUCT)&ownerItem ));          /*JEH1*/

    /******************************************************************/
    /* If there was no problem with saving the DC, restore it         */
    /******************************************************************/
    if (lPSid)
      RestoreDC (hpsMem, lPSid);

    /******************************************************************/
    /* If owner drew in rectangle, then return since no more to do    */
    /******************************************************************/
    if (rc)
      return;
  }

  /********************************************************************/
  /* Get the system defined color for shadow                          */
  /********************************************************************/
  ulRGB = GetSysColor(COLOR_BTNSHADOW);
  lShadowClr = 0x02000000 | ulRGB;

  /********************************************************************/
  /* Check if regular arm bitmap or readonly arm bitmap to be drawn   */
  /********************************************************************/
  if (ulStyle & SLS_READONLY)
  {
    /******************************************************************/
    /* Fill the readonly arm with shaft color, as long as brush can   */
    /* be created                                                     */
    /******************************************************************/
    if (hBrushCurrent = GetStockObject(LTGRAY_BRUSH))
    {
      hBrushOld = SelectObject (hpsMem, hBrushCurrent);
      FillRect( hpsMem, &rclSliderArm, hBrushCurrent );
      SelectObject (hpsMem, hBrushOld);
    }

    /******************************************************************/
    /* If focus readonly arm, use a different color for drawing       */
    /******************************************************************/
    if (pIntData->usStatusSld & FL_SLDHASFOCUS)
      lBorderClr = lShadowClr;
    else
      lBorderClr = lShadowClr;

    /******************************************************************/
    /* Set color and move to starting point, as long as pen can be    */
    /* obtained                                                       */
    /******************************************************************/
    if (hPenCurrent = GetStockObject(BLACK_PEN))
    {
      hPenOld = SelectObject (hpsMem, hPenCurrent);

      MoveToEx ( hpsMem, ptl[0].x, ptl[0].y, NULL);                     /*JEH1*/
      /****************************************************************/
      /* Check if vertical orientation                                */
      /****************************************************************/
      if (ulStyle & SLS_VERTICAL)
      {
        /**************************************************************/
        /* If vertical, start at left and draw right                  */
        /**************************************************************/
        LineTo ( hpsMem, ptl[1].x+1, ptl[1].y);

        /**************************************************************/
        /* Calculate light and shadow points on vertical line         */
        /**************************************************************/
        ptl[0].x++;
        ptl[2].x--;
      }
      else
      {
        /**************************************************************/
        /* otherwise horizontal, start at top and draw down           */
        /**************************************************************/
        LineTo ( hpsMem, ptl[3].x, ptl[3].y+1);

        /**************************************************************/
        /* Calculate light and shadow points on horizontal line       */
        /**************************************************************/
        ptl[0].y++;
        ptl[2].y--;
      }
      /****************************************************************/
      /* Set color to reflected light color and paint reflected light */
      /****************************************************************/
      SetPixel ( hpsMem, ptl[2].x, ptl[2].y,
          PALETTERGB(0xFF, 0x0, 0x62));
      /****************************************************************/
      /* Set color to shadow color and paint shadow on arm            */
      /****************************************************************/
      SetPixel ( hpsMem, ptl[0].x, ptl[0].y, lShadowClr);
      SelectObject (hpsMem, hPenOld);
    }
  }

  /********************************************************************/
  /* Otherwise this is a regular slider arm to be drawn               */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Paint the slider arm rectangle with the arm inside color       */
    /* and the border color around it; don't attempt to paint if the  */
    /* pen and brush can't be created                                 */
    /******************************************************************/
    if ((hBrushCurrent = GetStockObject (LTGRAY_BRUSH)) &&
        (hPenCurrent = GetStockObject (BLACK_PEN)))
    {
      hBrushOld = SelectObject (hpsMem, hBrushCurrent);
      hPenOld = SelectObject (hpsMem, hPenCurrent);
      Rectangle(hpsMem,rclSliderArm.left,rclSliderArm.bottom,
                rclSliderArm.right, rclSliderArm.top);    /* win */
      SelectObject (hpsMem, hBrushOld);
      SelectObject (hpsMem, hPenOld);
    }

    /******************************************************************/
    /* Now paint the four corner points in background color           */
    /******************************************************************/
    SetPixel ( hpsMem, ptl[0].x, ptl[0].y,
               pIntData->ulColorVal[CLR_SLDBACKGROUND]);
    SetPixel ( hpsMem, ptl[1].x, ptl[1].y,
               pIntData->ulColorVal[CLR_SLDBACKGROUND]);
    SetPixel ( hpsMem, ptl[2].x, ptl[2].y,
               pIntData->ulColorVal[CLR_SLDBACKGROUND]);
    SetPixel ( hpsMem, ptl[3].x, ptl[3].y,
               pIntData->ulColorVal[CLR_SLDBACKGROUND]);
    /******************************************************************/
    /* Setup points for drawing reflected light L on arm              */
    /******************************************************************/
    ptlRefl[1].x = ptl[0].x + 1;
    ptlRefl[1].y = ptl[0].y + 1;                               /* win */

    /******************************************************************/
    /* Flip starting point on axis if vertical                        */
    /******************************************************************/
    if (ulStyle & SLS_VERTICAL)
    {
      ptlRefl[0].x = ptl[1].x - 1;
      ptlRefl[0].y = ptl[1].y + 1;
      ptlRefl[2].x = ptl[3].x + 1;
      ptlRefl[2].y = ptl[3].y - 1;
      MoveToEx(hpsMem, ptlRefl[2].x+1, ptlRefl[2].y-1, NULL);           /*JEH1*/
    }
    else
    {
      ptlRefl[0].x = ptl[3].x + 1;
      ptlRefl[0].y = ptl[3].y - 1;
      ptlRefl[2].x = ptl[1].x - 1;
      ptlRefl[2].y = ptl[1].y + 1;
      MoveToEx(hpsMem, ptlRefl[2].x-1, ptlRefl[2].y+1, NULL);           /*JEH1*/
    }

    /******************************************************************/
    /* Now draw the arm reflected light shading on left top, unless   */
    /* the pen can't be created                                       */
    /******************************************************************/
    if (hPenCurrent = GetStockObject(WHITE_PEN))
    {
      hPenOld = SelectObject (hpsMem, hPenCurrent);
      Polyline(hpsMem, &ptlRefl[0], 3);
      SelectObject (hpsMem, hPenOld);
    }

    /******************************************************************/
    /* Setup points for drawing dark shading on arm                   */
    /******************************************************************/
    ptlRefl[2].x = ptl[2].x - 1;
    ptlRefl[2].y = ptl[2].y - 1;

    /******************************************************************/
    /* Flip starting point on axis if vertical                        */
    /******************************************************************/
    if (ulStyle & SLS_VERTICAL)
    {
      ptlRefl[0].x = ptl[2].x - 1;
      ptlRefl[0].y = ptl[2].y - 2;
      ptlRefl[1].x = ptl[1].x - 1;
      ptlRefl[1].y = ptl[1].y + 1;
      ptlRefl[3].x = ptl[3].x + 1;
      ptlRefl[3].y = ptl[3].y - 1;
    }
    else
    {
      ptlRefl[0].x = ptl[2].x - 2;
      ptlRefl[0].y = ptl[2].y - 1;
      ptlRefl[1].x = ptl[3].x + 1;
      ptlRefl[1].y = ptl[3].y - 1;
      ptlRefl[3].x = ptl[1].x - 1;
      ptlRefl[3].y = ptl[1].y;
    }

    /******************************************************************/
    /* Now draw shaded arm on right bottom (already at starting pt),  */
    /* unless the pen can't be created                                */
    /******************************************************************/
    if (hPenCurrent = CreatePen(PS_SOLID, 1, lShadowClr))
    {
      hPenOld = SelectObject (hpsMem, hPenCurrent);
      LineTo ( hpsMem, ptlRefl[0].x, ptlRefl[0].y);
      Polyline(hpsMem, &ptlRefl[0], 4);
      DeleteObject(SelectObject (hpsMem, hPenOld));
    }

    /******************************************************************/
    /* Now calculate midpoint for shading and light line next to it   */
    /******************************************************************/
    if (ulStyle & SLS_VERTICAL)
    {
      ptlRefl[0].y = ptlRefl[1].y = ptl[1].y + (pIntData->usArmLength/2);
      ptlRefl[0].x = ptlRefl[2].x = ptl[1].x - 2;
      ptlRefl[1].x = ptlRefl[3].x = ptl[0].x;  /* LineTo leaves out end pt */
      ptlRefl[2].y = ptlRefl[3].y = ptlRefl[0].y - 1;
    }
    else
    {
      ptlRefl[0].x = ptlRefl[1].x = ptl[3].x + (pIntData->usArmLength/2);
      ptlRefl[0].y = ptlRefl[2].y = ptl[3].y - 2;
      ptlRefl[1].y = ptlRefl[3].y = ptl[0].y;  /* LineTo leaves out end pt */
      ptlRefl[2].x = ptlRefl[3].x = ptlRefl[0].x - 1;
    }

    /******************************************************************/
    /* Move to point 0 and draw line to 1 in light color; check for   */
    /* successful creation of pen first
    /******************************************************************/
    if (hPenCurrent = GetStockObject(WHITE_PEN))
    {
      hPenOld = SelectObject (hpsMem, hPenCurrent);
      MoveToEx( hpsMem, ptlRefl[0].x, ptlRefl[0].y, NULL);              /*JEH1*/
      LineTo( hpsMem, ptlRefl[1].x, ptlRefl[1].y);
      SelectObject (hpsMem, hPenOld);
    }

    /******************************************************************/
    /* Move to point 2 and draw line to 3 in dark color; check for    */
    /* successful creation of pen first                               */
    /******************************************************************/
    if (hPenCurrent = CreatePen(PS_SOLID, 1, lShadowClr))
    {
      hPenOld = SelectObject (hpsMem, hPenCurrent);
      MoveToEx( hpsMem, ptlRefl[2].x, ptlRefl[2].y, NULL);              /*JEH1*/
      LineTo( hpsMem, ptlRefl[3].x, ptlRefl[3].y);
      DeleteObject(SelectObject (hpsMem, hPenOld));
    }

    /******************************************************************/
    /* Add focus shading to arm if this slider has the focus          */
    /******************************************************************/
    if (pIntData->usStatusSld & FL_SLDHASFOCUS)
    {
      /****************************************************************/
      /* Check if vertical or horizontal orientation for arm calc     */
      /****************************************************************/
      if (ulStyle & SLS_VERTICAL)
      {
        /**************************************************************/
        /* If vertical then calculate off of lower right              */
        /**************************************************************/
        sCx1 = (int)ptl[1].x - (pIntData->usArmBreadth/2);
        sCy1 = (int)ptl[1].y + (pIntData->usArmLength/2) + 1;
        sCx2 = sCx1 + 2;
        sCy2 = sCy1 - 3;
      }
      else
      {
        /**************************************************************/
        /* otherwise horizontal so calculate off of upper left        */
        /**************************************************************/
        sCx1 = (int)ptl[3].x + (pIntData->usArmLength/2) - 2;
        sCy1 = (int)ptl[3].y - (pIntData->usArmBreadth/2) + 2;
        sCx2 = sCx1 + 3;
        sCy2 = sCy1 - 2;
      }

      /****************************************************************/
      /* Now use boundary of rectangle to create a rectangle          */
      /****************************************************************/
      SetRect(&rclFocus, sCx1, sCy2, sCx2, sCy1 );       /* win */

      /****************************************************************/
      /* Now fill the rectangle just calculated; check for creation   */
      /* of the brush first                                           */
      /****************************************************************/
      if (hBrushCurrent = GetStockObject(BLACK_BRUSH))
      {
        hBrushOld = SelectObject (hpsMem, hBrushCurrent);
        FillRect( hpsMem, &rclFocus, hBrushCurrent );
        SelectObject (hpsMem, hBrushOld);
      }
    }
  }

  return;
}




/**********************************************************************/
/* Procedure: CalcArmPosFromPt - This procedure will determine the    */
/*            location of the arm within the slider based upon the    */
/*            x,y coordinates of the mouse which are passed in.       */
/*            The location is dependent on the orientation of the     */
/*            slider, the home position and the current primary scale */
/*            in effect for the slider.  The position is range        */
/*            checked and a valid position is returned.               */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         sCx      - x position of mouse in window                   */
/*         sCy      - y position of mouse in window                   */
/*                                                                    */
/* Output: usArmPos - New position of the arm                         */
/**********************************************************************/
USHORT CalcArmPosFromPt( HANDLE hwndSld, PISLDDATA pIntData, LONG sCx,  /*JEH1*/
                         LONG sCy )                                     /*JEH1*/
{
  ULONG  ulStyle;             /* Current style bit flags              *//*JEH1*/
  USHORT usArmPos = 0;                                                  /*JEH1*/
  SHORT  sOffset;                                                       /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* If vertical calculate arm position based off of home position    */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
  {
    /******************************************************************/
    /* Check which end is home                                        */
    /******************************************************************/
    if ( ulStyle & SLS_HOMETOP )
    {
      /****************************************************************/
      /* If home is from top, then get y coordinate for home          */
      /****************************************************************/
      sOffset = pIntData->yShaft  +
                (BORDERWIDTH + (pIntData->usArmLength >> 1));

      /****************************************************************/
      /* Calculate arm position from this offset                      */
      /****************************************************************/
      if ( sCy > sOffset )
        usArmPos = sCy - sOffset;
    }
    /******************************************************************/
    /* otherwise bottom is home                                       */
    /******************************************************************/
    else
    {
      /****************************************************************/
      /* If home is from bottom, then get y coordinate for home       */
      /****************************************************************/
      sOffset = (pIntData->yShaft + pIntData->usShaftLength) -
                (BORDERWIDTH + (pIntData->usArmLength >> 1)) - 1;

      /****************************************************************/
      /* Calculate arm position from this offset                      */
      /****************************************************************/
      if ( sCy < sOffset )
        usArmPos = sOffset - (sCy * (sCy > 0));
    }
  }
  /********************************************************************/
  /* otherwise horizontal, so calculate based on home position        */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Check which end is home                                        */
    /******************************************************************/
    if ( ulStyle & SLS_HOMERIGHT )
    {
      /****************************************************************/
      /* If home is from right, then get x coordinate for home        */
      /****************************************************************/
      sOffset = (pIntData->xShaft + pIntData->usShaftLength) -
                  (BORDERWIDTH + (pIntData->usArmLength >> 1));

      /****************************************************************/
      /* Calculate arm position from this offset                      */
      /****************************************************************/
      if ( sCx < sOffset )
        usArmPos = sOffset - (sCx * (sCx > 0));
    }
    /******************************************************************/
    /* otherwise left is home                                         */
    /******************************************************************/
    else
    {
      /****************************************************************/
      /* If home is from left, then get x coordinate for home         */
      /****************************************************************/
      sOffset = pIntData->xShaft + BORDERWIDTH +
                                       (pIntData->usArmLength >> 1) - 1;

      /****************************************************************/
      /* Calculate arm position from this offset                      */
      /****************************************************************/
      if ( sCx > sOffset )
        usArmPos = sCx - sOffset;
    }
  }
  /********************************************************************/
  /* Update the position to insure not beyond maximum range to slide  */
  /********************************************************************/
  usArmPos = UpdateArmPosition( hwndSld, pIntData, usArmPos );

  /********************************************************************/
  /* Return the calculated arm position to move to                    */
  /********************************************************************/
  return ( usArmPos );
}


/**********************************************************************/
/* Procedure: DrawDraggedArm - This routine manages the slider arm    */
/*            while the arm is being dragged by the user.  The arm    */
/*            is stored as a bitmap and bitmaps are created for the   */
/*            screen background.  When the user moves the arm, if     */
/*            the arm moves less than the width of the slider, a      */
/*            combined delta bitmap is created and blitted to the     */
/*            screen, otherwise, the old arm is erased and the new    */
/*            arm is blitted to the screen.                           */
/*                                                                    */
/* Inputs: pIntData - Pointer to the internal control block           */
/*         hps      - Presentation space to draw into                 */
/*         fs       - flags indicating the operation:                 */
/*                                                                    */
/*                   DI_START - we're drawing for the first time, so  */
/*                              save the screen contents and draw our */
/*                              bitmaps                               */
/*                   DI_END   - we're ending our drag, so restore the */
/*                              original screen contents              */
/*                   DI_SLIDE - we are dragging:  restore the screen  */
/*                              at our former position, save the      */
/*                              screen at the current position, and   */
/*                              and draw out bitmaps                  */
/* Output: None                                                       */
/**********************************************************************/
VOID DrawDraggedArm( PISLDDATA pIntData, HDC hps, USHORT fs )           /*JEH1*/
{
  RECT   wrcUnion, wrcTo, wrcFrom;
  LONG x, y, xTo, yTo, xFrom, yFrom;                                    /*JEH1*/
  LONG xMin, xMax, yMin, yMax;                                          /*JEH1*/
  LONG  cx, cy, cxTotal, cyTotal;                                       /*JEH1*/
  static LONG xOld, yOld, cxOldTotal, cyOldTotal;                       /*JEH1*/
  BITMAP  bitmap;
  HBITMAP hOldBitmap;


  /********************************************************************/
  /* Check if this is end then terminate the drag                     */
  /********************************************************************/
  if (fs & DI_END)
  {
    /******************************************************************/
    /* End of drag so redraw the shaft as it was saved originally; if */
    /* the screen save bitmap was not created, don't attempt to draw  */
    /******************************************************************/
    if (hbmScrSave)
      BltCopy( hbmScrSave, hps, 0, 0, cxOldTotal, cyOldTotal,
          xOld, yOld, SRCCOPY, SRC_BITMAP );
    return;
  }

  /********************************************************************/
  /* Get the x,y position to draw the slider arm at                   */
  /********************************************************************/
  x = pIntData->rclSliderArm.left;                                      /*JEH1*/
  y = pIntData->rclSliderArm.top;         /* win */                     /*JEH1*/

  /********************************************************************/
  /* If not end (start or slide), then query bitmap information of    */
  /* bitmap passed in; if we can't get the bitmap data, don't proceed */
  /********************************************************************/
  if (!(GetObject(pIntData->hbmArm, sizeof(BITMAP), (LPSTR)&bitmap)))
    return;

  /********************************************************************/
  /* Set the values of xMin, yMin to 0, xMax, yMax to top of bitmap   */
  /* and cx, cy and cxTotal, cyTotal to extent of bitmap              */
  /********************************************************************/
  xMin = 0;
  yMin = 0;
  xMax = cx = cxTotal = bitmap.bmWidth;
  yMax = cy = cyTotal = bitmap.bmHeight;

  /********************************************************************/
  /* Check if this is start drag                                      */
  /********************************************************************/
  if (fs & DI_START)
  {
    /******************************************************************/
    /* Save the current screen image (shaft only) to a bitmap         */
    /* covering the area we are dragging at; don't attempt it if the  */
    /* screen save bitmap wasn't created                              */
    /******************************************************************/
    if (hbmScrSave)
    {
      hOldBitmap = SelectObject( hpsMem, hbmScrSave );
      BltCopy( hps, hpsMem, x, y, cxTotal, cyTotal, 0, 0, SRCCOPY, 0L );
      SelectObject( hpsMem, hOldBitmap );
    }
  }

  /********************************************************************/
  /* If there's an icon bitmap, associate it with the device context  */
  /********************************************************************/
  if (hbmIcon)
    hOldBitmap = SelectObject( hpsMem, hbmIcon );

  /********************************************************************/
  /* Check if this is drag slide (from mouse move, not button action) */
  /********************************************************************/
  if (fs & DI_SLIDE)
  {
    /******************************************************************/
    /* If dragging, must calculate positions of old and new           */
    /******************************************************************/
    wrcFrom.left   = xOld;
    wrcFrom.bottom = yOld + cyOldTotal;   /* win */
    wrcFrom.right  = xOld + cxOldTotal;
    wrcFrom.top    = yOld;                /* win */

    wrcTo.left   = xMin + x;
    wrcTo.bottom = yMax + y;              /* win */
    wrcTo.right  = xMax + x;
    wrcTo.top    = yMin + y;              /* win */
    /******************************************************************/
    /* If the from and to positions overlap, perform special update   */
    /******************************************************************/
    if (IntersectRect((LPRECT) &wrcUnion, (LPRECT) &wrcFrom,(LPRECT) &wrcTo))
    {
      /****************************************************************/
      /* Calculate the union of the from rectangle and to rectangles  */
      /****************************************************************/
      UnionRect((LPRECT)&wrcUnion, (LPRECT)&wrcFrom, (LPRECT)&wrcTo);

      /****************************************************************/
      /* Find the offset position within the union rectangle          */
      /****************************************************************/
      xFrom = wrcFrom.left   - wrcUnion.left;
      yFrom = wrcFrom.top - wrcUnion.top;   /* win */
      xTo   = x - wrcUnion.left;
      yTo   = y - wrcUnion.top;             /* win */

      /****************************************************************/
      /* Copy the screen around the from and to icon positions        */
      /****************************************************************/
      BltCopy( hps, hpsMem, wrcUnion.left,
               wrcUnion.top,
               wrcUnion.right - wrcUnion.left,                  /* win */
               wrcUnion.bottom - wrcUnion.top, 0, 0, SRCCOPY, 0L );

      /****************************************************************/
      /* Restore the saved from part of the screen, if the screen save*/
      /* bitmap was created                                           */
      /****************************************************************/
      if (hbmScrSave)
        BltCopy( hbmScrSave, hpsMem, 0, 0, cxOldTotal, cyOldTotal,
                 xFrom, yFrom, SRCCOPY, SRC_BITMAP );

      /****************************************************************/
      /* Save the 'to' portion of the screen, unless the screen save  */
      /* bitmap or the icon bitmap wasn't created                     */
      /****************************************************************/
      if (hbmScrSave && hbmIcon)
      {
        SelectObject( hpsMem, hbmScrSave );
        BltCopy( hbmIcon, hpsMem, wrcTo.left - wrcUnion.left,
                 wrcTo.top - wrcUnion.top, cxTotal, cyTotal,       /* win */
                 0, 0, SRCCOPY, SRC_BITMAP);
      }

      /****************************************************************/
      /* Copy the arm into the bitmap screen image, unless the icon   */
      /* bitmap couldn't be created                                   */
      /****************************************************************/
      if (hbmIcon)
      {
        SelectObject( hpsMem, hbmIcon );
        BltCopy( pIntData->hbmArm, hpsMem, 0, 0, cx, cy, xTo, yTo,
                 SRCCOPY, SRC_BITMAP );
      }

      /****************************************************************/
      /* Copy the new image to the screen                             */
      /****************************************************************/
      BltCopy( hpsMem, hps, 0, 0, wrcUnion.right - wrcUnion.left,  /* win */
               wrcUnion.bottom - wrcUnion.top, wrcUnion.left,
               wrcUnion.top, SRCCOPY, 0L);
    }

    /******************************************************************/
    /* Otherwise, the source and destination don't overlap.           */
    /******************************************************************/
    else
    {
      /****************************************************************/
      /* Restore the screen to its original look (shaft only), unless */
      /* there was a problem creating the screen save bitmap          */
      /****************************************************************/
      if (hbmScrSave)
        BltCopy( hbmScrSave, hps, 0, 0, cxOldTotal, cyOldTotal, xOld,
                 yOld, SRCCOPY, SRC_BITMAP );

      /****************************************************************/
      /* Save the area about to be overwritten, as long as the screen */
      /* save bitmap is available                                     */
      /****************************************************************/
      if (hbmScrSave)
      {
        SelectObject( hpsMem, hbmScrSave );
        BltCopy( hps, hpsMem, x + xMin, y + yMin, cxTotal, cyTotal,
            0, 0, SRCCOPY, 0L );
      }

      /****************************************************************/
      /* Associate the device context with the icon bitmap (if one    */
      /* exists) and reset the slide flag (to overwrite screen with   */
      /* normal drawing)                                              */
      /****************************************************************/
      if (hbmIcon)
        SelectObject( hpsMem, hbmIcon );
      fs &= ~DI_SLIDE;
    }
  }

  /********************************************************************/
  /* If not slide (either start or non overlapped drag)               */
  /********************************************************************/
  if ( !(fs & DI_SLIDE) )
  {
    /******************************************************************/
    /* Copy the area we are about to overwrite with the screen        */
    /******************************************************************/
    BltCopy( hps, hpsMem, x + xMin, y + yMin, cxTotal, cyTotal, 0, 0,
             SRCCOPY, 0L);

    /******************************************************************/
    /* Copy the arm image into the memory image                       */
    /******************************************************************/

    BltCopy( pIntData->hbmArm, hpsMem, 0, 0, cx, cy, -xMin, -yMin,
             SRCCOPY, SRC_BITMAP );

    /******************************************************************/
    /* Copy the composite result back to the screen again             */
    /******************************************************************/
    BltCopy( hpsMem, hps, 0, 0, cxTotal, cyTotal, x + xMin, y + yMin,
             SRCCOPY, 0L );
  }

  /********************************************************************/
  /* Disassociate all bitmaps with this device context                */
  /********************************************************************/
  SelectObject( hpsMem, hOldBitmap );

  /********************************************************************/
  /* Update the static variables with their new settings              */
  /********************************************************************/
  xOld = x + xMin;
  yOld = y + yMin;
  cxOldTotal = cxTotal;
  cyOldTotal = cyTotal;

  return;
}


/**********************************************************************/
/* Procedure: IncrementArmPosition - This routine will increment or   */
/*            decrement the arm position based on the flag sent and   */
/*            the current home position.                              */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         bBump    - If TRUE, increment(Home left), decrement(right) */
/*                      FALSE, decrement(Home left), increment(right) */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID IncrementArmPosition( HANDLE hwndSld, PISLDDATA pIntData,
                           BOOL bBump )
{
  ULONG    ulStyle;           /* Current style bit flags              *//*JEH1*/
  USHORT   usSpacing;                                                   /*JEH1*/
  USHORT   usArmPos;                                                    /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Get the spacing between increments for the current scale         */
  /********************************************************************/
  usSpacing = QueryScaleSpacing( hwndSld, pIntData );

  /********************************************************************/
  /* If decrement flag passed and home is on right or if increment    */
  /*  flag passed and home is on left then increment arm position     */
  /********************************************************************/
  if ((!bBump && ( ulStyle & SLS_HOMERIGHT )) || (bBump &&
       !( ulStyle & SLS_HOMERIGHT )))
  {
    /******************************************************************/
    /* If home on other end, then swap button function, i.e. this     */
    /* decrement button is now doing increment function               */
    /******************************************************************/
    usArmPos = pIntData->usArmPos + usSpacing;
  }
  /********************************************************************/
  /* otherwise regular decrement or increment with opposite home      */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Calculate new arm position if decrementing                     */
    /******************************************************************/
    if ( pIntData->usArmPos < usSpacing )

      /****************************************************************/
      /* If less than minimum decrement space, move to end            */
      /****************************************************************/
      usArmPos = 0;

    /******************************************************************/
    /* otherwise just subtract decrement from current arm position    */
    /******************************************************************/
    else
      usArmPos = pIntData->usArmPos - usSpacing;
  }
  /********************************************************************/
  /* Move to this new position                                        */
  /********************************************************************/
  MoveSliderArm( hwndSld, pIntData, usArmPos, TRUE, TRUE );

  return;
}


/**********************************************************************/
/* Procedure: MoveSliderArm - This procedure will move the slider     */
/*            arm to the new position passed in.  It will update the  */
/*            screen with arm at its new location and it will update  */
/*            internal data with the new arm position and rectangle.  */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         usArmPos - position to move the arm to                     */
/*         bDrawArm - If TRUE, draw arm at new position               */
/*                    If FALSE, don't draw the arm here               */
/*         bNotify  - If TRUE, notify owner that change occurred      */
/*                    If FALSE, no notification should be sent        */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID MoveSliderArm( HANDLE hwndSld, PISLDDATA pIntData, USHORT usArmPos,/*JEH1*/
                    BOOL bDrawArm, BOOL bNotify )
{
  ULONG   ulStyle;            /* Current style bit flags              *//*JEH1*/
  USHORT  usButtonLength;                                               /*JEH1*/
  HDC     hps;
  HDC     hpsTmp;
  RECT   rclNewArm;
  RECT   rclNewArm1;
  RECT   rclToUpdate;
  RECT   rclToUpdate1;
  RECT   rclBorder1;
  RECT   rclButton1;
  HBITMAP hbmUpdate;
  HBITMAP hOldBitmap;
  BITMAP  bmp;                /* Bitmap information used for create   */
  ULONG  ulQueryVal; /* Temp variable for storing query variable */     /*JEH1*/
  int    lPSId;
  HBRUSH  hBrushCurrent;                                                /*DAB1*/
  HBRUSH  hBrushOld;                                                    /*DAB1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Check if snap is on                                              */
  /********************************************************************/
  if (ulStyle & SLS_SNAPTOINCREMENT)
  {
    /******************************************************************/
    /* Query the closest tick to this position to be set to and get   */
    /* the position at that tick                                      */
    /******************************************************************/
    ulQueryVal = QueryClosestTick (hwndSld, pIntData, usArmPos);
    usArmPos = HIWORD (ulQueryVal);
  }

  /********************************************************************/
  /* Change the slider arm position to the one passed in              */
  /********************************************************************/
  if ((( usArmPos = (USHORT)UpdateArmPosition( hwndSld, pIntData,       /*JEH1*/
     usArmPos )) == pIntData->usArmPos ) && bDrawArm )
  {
    /******************************************************************/
    /* If arm already at this position, don't move, just check scale  */
    /* Check for end of scale reached with the slider arm             */
    /*  and if a flag has been modified, update button area           */
    /******************************************************************/
    if (CheckForEndOfScale( hwndSld, pIntData, LOWORD( ulQueryVal ) ))
      if ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT))
      {
        if (hps = GetDC( hwndSld ))
        {
          PaintButtons( hwndSld, pIntData, hps );
          ReleaseDC( hwndSld, hps );
        }
      }
    if (bNotify)
      SendMessage( PARENTOF(hwndSld), WM_COMMAND,                       /*DAB2*/
                   (WPARAM)MAKELONG( IDOF(hwndSld), SLN_CHANGE ),       /*DAB2*/
                   (LPARAM)hwndSld );
    return;
  }

  /********************************************************************/
  /* Set the slider arm position to this new position                 */
  /********************************************************************/
  pIntData->usArmPos = usArmPos;

  /********************************************************************/
  /* Get a presentation space to draw into                            */
  /********************************************************************/
  if (hps = GetDC (hwndSld))
  {

    /******************************************************************/
    /* If the end of scale has been reached and if a flag has been    */
    /* modified, update the button area.                              */
    /******************************************************************/
    if (CheckForEndOfScale (hwndSld, pIntData, LOWORD (ulQueryVal)))
      if ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT))
        PaintButtons (hwndSld, pIntData, hps);

    /******************************************************************/
    /* Get the length of a button based on the current shaft breadth  */
    /******************************************************************/
    usButtonLength = CalcButtonLength( pIntData );

    /******************************************************************/
    /* Calculate the new rectangle for the arm at this position       */
    /******************************************************************/
    CalcArmRectangle( hwndSld, pIntData, &rclNewArm, pIntData->usArmPos );

    /******************************************************************/
    /* Calculate the rectangle bounded between the old and the new    */
    /******************************************************************/
    UnionRect( &rclToUpdate, &(pIntData->rclSliderArm), &rclNewArm );

    /******************************************************************/
    /* Set the internal data arm rectangle to the new value           */
    /******************************************************************/
    pIntData->rclSliderArm = rclNewArm;

    /******************************************************************/
    /* Make a copy of the bounded rectangle and the new arm rectangle */
    /* and offset them to a new coordinate place where lower left for */
    /* the bounded rectangle is 0,0                                   */
    /******************************************************************/
    rclToUpdate1 = rclToUpdate;
    rclNewArm1 = rclNewArm;
    OffsetRect( &rclToUpdate1, (int)-rclToUpdate.left,     /* win */
                (int)-rclToUpdate.top );
    OffsetRect( &rclNewArm1, (int)-rclToUpdate.left,       /* win */
                (int)-rclToUpdate.top );

    /******************************************************************/
    /* Create a bitmap information header block for the arm bitmap by */
    /* querying the data for a button bitmap and changing dimensions; */
    /* if the button bitmap can't be obtained, don't try to draw      */
    /******************************************************************/
    if (GetObject(hbmHDecr, sizeof(BITMAP), (LPSTR)&bmp))
    {

      /****************************************************************/
      /* Set the bitmap size to that of the update rectangle          */
      /****************************************************************/
      bmp.bmWidth = (WORD)rclToUpdate1.right;
      bmp.bmHeight = (WORD)rclToUpdate1.bottom;       /* win */

      /****************************************************************/
      /* Create the update bitmap in memory presentation space using  */
      /* bitmap info header set earlier and set bitmap into DC; if    */
      /* the bitmap can't be created, don't attempt the draw          */
      /****************************************************************/
      if (hbmUpdate  = CreateBitmap(bmp.bmWidth, bmp.bmHeight,
                   bmp.bmPlanes, bmp.bmBitsPixel, (LPSTR)0L))
      {
        hOldBitmap = SelectObject( hpsMem, hbmUpdate);

        /**************************************************************/
        /* Calculate border rectangle from internal data              */
        /**************************************************************/
        CalcRectangle( hwndSld, pIntData, pIntData->xShaft,
            pIntData->yShaft, pIntData->usShaftLength, &rclBorder1 );

        /**************************************************************/
        /* Also calculate button rectangle from internal data (needed */
        /* for background area painting within PaintBorder) if any    */
        /**************************************************************/
        if ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT))
        {
          /************************************************************/
          /* Get button rectangle to pass to Paint Border             */
          /************************************************************/
          CalcRectangle( hwndSld, pIntData, pIntData->xButton,
             pIntData->yButton, (USHORT)(usButtonLength << 1),          /*JEH1*/
             &rclButton1 );

          /************************************************************/
          /* Change button rectangle to our new coordinate system     */
          /************************************************************/
          OffsetRect( &rclButton1, (int)-rclToUpdate.left,     /* win */
                      (int)-rclToUpdate.top );
        }
        /**************************************************************/
        /* Change border rectangle to our new coordinate system       */
        /**************************************************************/
        OffsetRect( &rclBorder1, (int)-rclToUpdate.left,       /* win */
                    (int)-rclToUpdate.top );

        /**************************************************************/
        /* Set entire presentation space to background color first    */
        /*  before any drawing occurs                                 */
        /**************************************************************/
        if (hBrushCurrent =                                             /*DAB1*/
              CreateSolidBrush(pIntData->ulColorVal[CLR_SLDBACKGROUND]))/*DAB1*/
        {                                                               /*DAB1*/
          hBrushOld = SelectObject (hpsMem, hBrushCurrent);             /*DAB1*/
          FillRect( hpsMem, &rclToUpdate1, hBrushCurrent );             /*DAB1*/
          DeleteObject(SelectObject (hpsMem, hBrushOld));               /*DAB1*/
        }                                                               /*DAB1*/

        /**************************************************************/
        /* Using this new border rectangle, paint border and shaft    */
        /* into the memory bitmap                                     */
        /**************************************************************/
        PaintBorder( hwndSld, pIntData, hpsMem, &rclBorder1,
            &rclButton1, &rclNewArm1 );
        PaintShaft( hwndSld, hpsMem, &rclBorder1, &rclNewArm1 );

        /**************************************************************/
        /* Save the PS (because BitBlt messes up the raster operation,*/
        /* causing problems in future DrawBorder calls to hpsMem)     */
        /**************************************************************/
        lPSId = SaveDC (hpsMem);

        /**************************************************************/
        /* Now draw the arm into the memory bitmap                    */
        /**************************************************************/
        if ( bDrawArm )
        {
          HBITMAP hbmOld;
          ULONG   rasterOp = SRCCOPY;                                   /*JEH1*/
#define PATTERNORSOURCE   0x00FC008A
          HBRUSH     hHalftoneBrush=0;                                  /*JEH1*/
          HBRUSH     hOldBrush=0;                                       /*JEH1*/
          HBITMAP    hHalftonePatternBitmap=0;                          /*JEH1*/
          BYTE       HalftonePattern [] =
                     { 0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
                       0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55 };

          /************************************************************/
          /* Get the logical data for the bitmap and create a DC for  */
          /* it, but if either operation fails, don't draw            */
          /************************************************************/
          if ((GetObject(pIntData->hbmArm,
              sizeof(BITMAP), (LPSTR)&bmp)) &&
              (hpsTmp = CreateCompatibleDC(NULL)))
          {
            hbmOld = SelectObject(hpsTmp, pIntData->hbmArm);

            /**********************************************************/
            /* If the arm should be drawn disabled, set up raster flag*/
            /* and create the halftoning brush pattern                */
            /**********************************************************/
            if (ulStyle & WS_DISABLED)
            {
              if ((hHalftonePatternBitmap = CreateBitmap (16, 16, 1, 1,
                  (LPSTR) HalftonePattern)) &&
                  (hHalftoneBrush =
                  CreatePatternBrush (hHalftonePatternBitmap)))
              {
                rasterOp = PATTERNORSOURCE;
                hOldBrush = SelectObject (hps, hHalftoneBrush);
              }
            }

            BitBlt ( hpsMem, rclNewArm1.left, rclNewArm1.top,         /* win */
                     bmp.bmWidth, bmp.bmHeight, hpsTmp, 0, 0, rasterOp);

            /**********************************************************/
            /* If the halftoning bitmaps were loaded, free their      */
            /* resources                                              */
            /**********************************************************/
            if (hOldBrush)
            {
              DeleteObject (SelectObject (hps, hOldBrush));
              DeleteObject (hHalftonePatternBitmap);
            }

            SelectObject(hpsTmp, hbmOld);
            DeleteDC(hpsTmp);
          }
        }

        /**************************************************************/
        /* If there was no problem with saving the DC, restore it     */
        /**************************************************************/
        if (lPSId)
          RestoreDC (hpsMem, lPSId);

        /**************************************************************/
        /* Disassociate the arm bitmap from dev context for drawing   */
        /**************************************************************/
        SelectObject( hpsMem, hOldBitmap);

        /**************************************************************/
        /* Finally copy information to the screen at union rectangle  */
        /**************************************************************/
        BltCopy( hbmUpdate, hps, 0, 0, rclToUpdate1.right,              /*JEH1*/
                 rclToUpdate1.bottom, rclToUpdate.left,                 /*JEH1*/
                 rclToUpdate.top, SRCCOPY, SRC_BITMAP );

        /**************************************************************/
        /* Now that it is drawn, get rid of the bitmap                */
        /**************************************************************/
        DeleteObject(hbmUpdate);

      }
    }
    /******************************************************************/
    /* When update drawing complete, release the hps obtained         */
    /******************************************************************/
    ReleaseDC( hwndSld, hps );
  }

  /********************************************************************/
  /* Lastly, notify the owner that the arm position has changed if    */
  /*  this is a move with arm drawing flag set                        */
  /********************************************************************/
  if ( bDrawArm && bNotify)
    SendMessage( PARENTOF(hwndSld), WM_COMMAND,                         /*DAB2*/
                 (WPARAM)MAKELONG( IDOF(hwndSld), SLN_CHANGE ),         /*DAB2*/
                 (LPARAM)hwndSld );
  return;
}


/**********************************************************************/
/* Procedure: MoveToNextDetent - This routine will move the slider    */
/*            arm position to the next detent in the direction        */
/*            specified.                                              */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         bDir     - If TRUE, move away from home                    */
/*                      FALSE, move towards home                      */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID MoveToNextDetent( HANDLE hwndSld, PISLDDATA pIntData, BOOL bDir )
{
  ULONG    ulStyle;           /* Current style bit flags              *//*JEH1*/
  USHORT   usArmPos = 0xFFFF;                                           /*JEH1*/
  USHORT   usScale;                                                     /*JEH1*/
  PDETENTDATA pDetentCurr;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Save the scale equate for the primary scale                      */
  /********************************************************************/
  if (ulStyle & SLS_PRIMARYSCALE2)
    usScale = SMA_SCALE2;
  else
    usScale = SMA_SCALE1;

  /********************************************************************/
  /* First check if the list is non-null and a valid Id was passed    */
  /********************************************************************/
  if ( pDetentCurr = pIntData->pDetentData )
  {
    /******************************************************************/
    /* Search through detent list to compare positions                */
    /******************************************************************/
    while ( pDetentCurr )
    {
      /****************************************************************/
      /* If left flag passed and home is on right or if right         */
      /*  flag passed and home is on left then move away from home    */
      /****************************************************************/
      if ((!bDir && ( ulStyle & SLS_HOMERIGHT )) || (bDir &&
           !( ulStyle & SLS_HOMERIGHT )))
      {
        /**************************************************************/
        /* Compare detent position to current position                */
        /**************************************************************/
        if ( pDetentCurr->usDetentPos > (pIntData->usArmPos) )          /*JEH1*/
        {
          /************************************************************/
          /* If found, save that position and break since sorted      */
          /************************************************************/
          usArmPos = pDetentCurr->usDetentPos;
          break;
        }
      }
      /****************************************************************/
      /* otherwise move towards home so look for value just lower     */
      /****************************************************************/
      else
      {
        /**************************************************************/
        /* Compare detent position to current position                */
        /**************************************************************/
        if ( pDetentCurr->usDetentPos < pIntData->usArmPos )
        {
          /************************************************************/
          /* If found, save that position.  Don't break since we      */
          /* want the highest one just below arm position (because    */
          /* detents are stored in ascending position order)          */
          /************************************************************/
          usArmPos = pDetentCurr->usDetentPos;
        }
      }
      /****************************************************************/
      /* Get next node (if any)                                       */
      /****************************************************************/
      pDetentCurr = pDetentCurr->pNext;
    }
  }
  /********************************************************************/
  /* Move to this new position                                        */
  /********************************************************************/
  if ( usArmPos != 0xFFFF )                                             /*JEH1*/
    MoveSliderArm( hwndSld, pIntData, usArmPos, TRUE, TRUE);

  return;
}


/**********************************************************************/
/* Procedure: SldProcessArmPress - This routine will start a drag     */
/*            if the arm is clicked upon for dragging around.  It     */
/*            will create some memory bitmaps used for fast drawing   */
/*            to the screen and will remove the arm from the screen   */
/*            so that the screen image can be saved in these bitmaps. */
/*            Finally it calls the drag start routine to initiate the */
/*            movement of the arm.                                    */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         hps      - presentation space to draw into                 */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID SldProcessArmPress( HANDLE hwndSld, PISLDDATA pIntData, HDC hps )
{
  ULONG  ulStyle;   /* Current style bit flags              */          /*JEH1*/
  BITMAP bmp;       /* Bitmap information used for create   */

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Create a bitmap information header block for the new bitmap by   */
  /* querying the data for a button bitmap and changing the dimensions*/
  /********************************************************************/
  GetObject ( pIntData->hbmArm, sizeof(BITMAP), (LPSTR)&bmp);

  /********************************************************************/
  /* Update the bitmap size to be twice the size of the arm in the    */
  /* direction of sliding                                             */
  /********************************************************************/
  if (ulStyle & SLS_VERTICAL)
    bmp.bmHeight = (pIntData->usArmLength << 1);
  else
    bmp.bmWidth = (pIntData->usArmLength << 1);

  /********************************************************************/
  /* Check if bitmaps exist already for the screensave and icon and   */
  /* if so delete them.  (We'll check their validity at drawing time) */
  /********************************************************************/
  if ( hbmScrSave )
    DeleteObject( hbmScrSave );
  if ( hbmIcon )
    DeleteObject( hbmIcon);

  hbmScrSave = CreateBitmap(bmp.bmWidth, bmp.bmHeight,
               bmp.bmPlanes, bmp.bmBitsPixel, (LPSTR)0L);
  hbmIcon    = CreateBitmap(bmp.bmWidth, bmp.bmHeight,
               bmp.bmPlanes, bmp.bmBitsPixel, (LPSTR)0L);

  /********************************************************************/
  /* Save the current arm position (in case drag is cancelled)        */
  /********************************************************************/
  pIntData->usPreviousArmPos = pIntData->usArmPos;

  /********************************************************************/
  /* Repaint the area the arm occupies by calling move with same      */
  /* position and set flag to FALSE to not draw the arm               */
  /********************************************************************/
  MoveSliderArm( hwndSld, pIntData, pIntData->usArmPos, FALSE, FALSE);

  /********************************************************************/
  /* Start the drag of the arm by calling the drag blitting routine   */
  /********************************************************************/
  DrawDraggedArm( pIntData, hps, DI_START );

  return;
}


/**********************************************************************/
/* Procedure: SldProcessButtonPress - This routine will determine     */
/*            which button was pressed by the mouse (left or right)   */
/*            and process accordingly.  Each button will move the     */
/*            slider arm one increment in the appropriate direction.  */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         sCx      - x position of mouse                             */
/*         sCy      - y position of mouse                             */
/*         bInitial - If TRUE, this is the initial button down        */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID SldProcessButtonPress( HANDLE hwndSld, PISLDDATA pIntData,
                            LONG sCx, LONG sCy, BOOL bInitial )         /*JEH1*/
{
  RECT  rclArrowDecr;
  RECT  rclArrowIncr;
  POINT ptlCoords;
  BOOL   bResetButton=FALSE;

  /********************************************************************/
  /* Calculate the two separate button rectangles from the entire rect*/
  /********************************************************************/
  CalcArrowRectangles( hwndSld, pIntData, &rclArrowDecr, &rclArrowIncr);

  /********************************************************************/
  /* Map x, y coordinates to a point and get anchor block handle      */
  /********************************************************************/
  ptlCoords.x = sCx;
  ptlCoords.y = sCy;

  /********************************************************************/
  /* Check if point is within the decrement button rectangle          */
  /********************************************************************/
  if ( PtInRect( &rclArrowDecr, ptlCoords ))
  {
    /******************************************************************/
    /* If initial press, reset flag to show that decrement was initial*/
    /******************************************************************/
    if (bInitial)
      pIntData->usStatusSld &= ~FL_SLDINITIALINCR;

    /******************************************************************/
    /* If initial press was decrement and we are over button, depress */
    /******************************************************************/
    if (!(pIntData->usStatusSld & FL_SLDINITIALINCR ))
    {
      /****************************************************************/
      /* If button up, drop and paint it, else don't(to avoid flicker)*/
      /****************************************************************/
      if (!( pIntData->usStatusSld & FL_SLDDECREMENTDOWN ))
      {
        /**************************************************************/
        /* But only paint it if not already at end of scale           */
        /**************************************************************/
        if (!( pIntData->usStatusSld & FL_SLDARMATLOWEND ))
        {

          /************************************************************/
          /* If so, set the decrement down arrow flag                 */
          /************************************************************/
          pIntData->usStatusSld |= FL_SLDDECREMENTDOWN;

          /************************************************************/
          /* Invalidate button so that it gets repainted (if needed)  */
          /************************************************************/
          InvalidateRect( hwndSld, &rclArrowDecr, FALSE );
        }
        /**************************************************************/
        /* Otherwise, if at the end of scale and user tries to go     */
        /* lower, beep at user to let him know he can't do it         */
        /**************************************************************/
        else
          MessageBeep (0);
      }
    }
    else
      bResetButton = TRUE;

    /******************************************************************/
    /* If in rectangle over button and initial press was this button  */
    /* then move the slider arm in this direction                     */
    /******************************************************************/
    if ( pIntData->usStatusSld & FL_SLDDECREMENTDOWN )
    {
      /****************************************************************/
      /* Decrement the arm position based on home location            */
      /****************************************************************/
      IncrementArmPosition( hwndSld, pIntData, FALSE );
    }
  }
  /********************************************************************/
  /* If not, check if point is within the increment button rectangle  */
  /********************************************************************/
  else if ( PtInRect( &rclArrowIncr, ptlCoords ))
  {
    /******************************************************************/
    /* If initial press, set flag to show that increment was initial  */
    /******************************************************************/
    if (bInitial)
      pIntData->usStatusSld |= FL_SLDINITIALINCR;

    /******************************************************************/
    /* If initial press was increment and we are over button, depress */
    /******************************************************************/
    if (pIntData->usStatusSld & FL_SLDINITIALINCR )
    {
      /****************************************************************/
      /* If button up, drop and paint it, else don't(to avoid flicker)*/
      /****************************************************************/
      if (!( pIntData->usStatusSld & FL_SLDINCREMENTDOWN ))
      {

        /**************************************************************/
        /* But only paint it if not already at end of scale           */
        /**************************************************************/
        if (!( pIntData->usStatusSld & FL_SLDARMATHIGHEND ))
        {
          /************************************************************/
          /* If so, set the increment down arrow flag                 */
          /************************************************************/
          pIntData->usStatusSld |= FL_SLDINCREMENTDOWN;

          /************************************************************/
          /* Invalidate button so that it gets repainted (if needed)  */
          /************************************************************/
          InvalidateRect( hwndSld, &rclArrowIncr, FALSE );
        }
        /**************************************************************/
        /* Otherwise, if at the end of the scale and user tries to go */
        /* higher, beep at the user to let him know he can't do it    */
        /**************************************************************/
        else
          MessageBeep (0);
      }
    }
    else
      bResetButton = TRUE;

    /******************************************************************/
    /* If in rectangle over button and initial press was this button  */
    /* then move the slider arm in this direction                     */
    /******************************************************************/
    if ( pIntData->usStatusSld & FL_SLDINCREMENTDOWN )
    {
      /****************************************************************/
      /* Increment the arm position based on home location            */
      /****************************************************************/
      IncrementArmPosition( hwndSld, pIntData, TRUE );
    }
  }
  else
    bResetButton = TRUE;

  /********************************************************************/
  /* Check if a button is to be reset (because mouse is down and not  */
  /*  over the initial button pressed)                                */
  /********************************************************************/
  if (bResetButton)
  {
    /******************************************************************/
    /* Reset flag for button which was down since mouse is not over it*/
    /******************************************************************/
    if (pIntData->usStatusSld & FL_SLDINITIALINCR )
    {
      /****************************************************************/
      /* If button already up, don't paint it (to avoid flicker)      */
      /****************************************************************/
      if ( pIntData->usStatusSld & FL_SLDINCREMENTDOWN )
      {
        pIntData->usStatusSld &= ~FL_SLDINCREMENTDOWN;
        InvalidateRect( hwndSld, &rclArrowIncr, FALSE);
      }
    }
    else
    {
      /****************************************************************/
      /* If button already up, don't paint it (to avoid flicker)      */
      /****************************************************************/
      if ( pIntData->usStatusSld & FL_SLDDECREMENTDOWN )
      {
        pIntData->usStatusSld &= ~FL_SLDDECREMENTDOWN;
        InvalidateRect( hwndSld, &rclArrowDecr, FALSE);
      }
    }
  }

  return;
}


/**********************************************************************/
/* Procedure: SldProcessChar - This procedure will process keystrokes */
/*            from the user in relation to the slider control.        */
/*            Keys which perform specific action in the slider are:   */
/*                                                                    */
/*             home, c-home  - Move slider to the home position       */
/*             end, c-end    - Move slider to the far end position    */
/*             right         - Move slider one increment right (horz) */
/*             left          - Move slider one increment left (horz)  */
/*             up            - Move slider one increment up (vert)    */
/*             down          - Move slider one increment down (vert)  */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         usVrtlKey  - virtual key codes                             */
/*                                                                    */
/* Output: bProcessed - If TRUE, keystroke was processed              */
/*                      If FALSE, keystroke ignored                   */
/**********************************************************************/
BOOL SldProcessVirtKey ( HANDLE hwndSld,
                         PISLDDATA pIntData,
                         UINT   usVrtlKey,                              /*JEH1*/
                         ULONG  ulKeyInfo)                              /*JEH1*/
{
  USHORT usMaxRange;                                                    /*JEH1*/
  ULONG            ulStyle;   /* Current style bit flags              *//*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Check if readonly and if so, do not process keystroke            */
  /********************************************************************/
  if (ulStyle & SLS_READONLY)
    return (FALSE);

  /********************************************************************/
  /* Check for virtual key transitions on a key down                  */
  /********************************************************************/
  if (!(ulKeyInfo & ALT_KEY_PRESSED))
  {
    /******************************************************************/
    /* Check for the home key                                         */
    /******************************************************************/
    if (( usVrtlKey == VK_HOME ) && (GetKeyState(VK_SHIFT)>=0))
    {
      /****************************************************************/
      /* If home or Ctrl home key, then move to slider home           */
      /****************************************************************/
      MoveSliderArm( hwndSld, pIntData, 0, TRUE, TRUE);
    }
    /******************************************************************/
    /* Check for the end key                                          */
    /******************************************************************/
    else if (( usVrtlKey == VK_END ) && (GetKeyState(VK_SHIFT)>=0))
    {
      /****************************************************************/
      /* Query for the maximum range for the current scale            */
      /****************************************************************/
      usMaxRange = QueryMaxRange( hwndSld, pIntData );

      /****************************************************************/
      /* If end or Ctrl end key, move to slider end (last increment)  */
      /****************************************************************/
      MoveSliderArm( hwndSld, pIntData, usMaxRange, TRUE, TRUE);
    }
    /******************************************************************/
    /* If control pressed, skip keys which don't have special meaning */
    /******************************************************************/
    else if (GetKeyState(VK_CONTROL)>=0)
    {
      /****************************************************************/
      /* Check for the up or right key                                */
      /****************************************************************/
      if ((( usVrtlKey == VK_UP ) && (ulStyle & SLS_VERTICAL))
        || (( usVrtlKey == VK_RIGHT ) && !(ulStyle & SLS_VERTICAL)))
      {
        /**************************************************************/
        /* If shift key pressed, move in direction to next detent     */
        /**************************************************************/
        if (GetKeyState(VK_SHIFT)<0)

          /************************************************************/
          /* If up or right key, then increment to next detent        */
          /************************************************************/
          MoveToNextDetent( hwndSld, pIntData, TRUE );

        /**************************************************************/
        /* otherwise no shift so if up or right key, then increment   */
        /* slider based on home                                       */
        /**************************************************************/
        else
          IncrementArmPosition( hwndSld, pIntData, TRUE );
      }
      /****************************************************************/
      /* Check for the down or left key                               */
      /****************************************************************/
      else if ((( usVrtlKey == VK_DOWN ) && (ulStyle & SLS_VERTICAL))
        || (( usVrtlKey == VK_LEFT ) && !(ulStyle & SLS_VERTICAL)))
      {
        /**************************************************************/
        /* If shift key pressed, move in direction to next detent     */
        /**************************************************************/
        if (GetKeyState(VK_SHIFT)<0)

          /************************************************************/
          /* If up or right key, then increment to next detent        */
          /************************************************************/
          MoveToNextDetent( hwndSld, pIntData, FALSE );

        /**************************************************************/
        /* otherwise no shift so if down or left key, then decrement  */
        /* slider based on home                                       */
        /**************************************************************/
        else
          IncrementArmPosition( hwndSld, pIntData, FALSE );
      }
      /****************************************************************/
      /* If none of above virtual keys, return FALSE to show not used */
      /****************************************************************/
      else
        return (FALSE);
    }
    else
      return (FALSE);
  }
  else
    return (FALSE);

  /********************************************************************/
  /* If processed, then will come here so return TRUE to show this    */
  /********************************************************************/
  return (TRUE);
}


/**********************************************************************/
/* Procedure: SldProcessFocusChg - This routine is called whenever    */
/*            a focus change message is received by the slider        */
/*            control.  It will update the slider status and cause    */
/*            the control to repaint certain areas to reflect this    */
/*            change to the end user.                                 */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         bFocus   - Flag which tells whether we are gaining or      */
/*                    losing focus                                    */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID SldProcessFocusChg( HANDLE hwndSld, PISLDDATA pIntData, BOOL bFocus )
{
  RECT rclBorder;

  /********************************************************************/
  /* If focus is TRUE, then window is gaining focus so set flag       */
  /********************************************************************/
  if (bFocus)
  {
    /******************************************************************/
    /* Set slider has focus flag in the internal data area            */
    /******************************************************************/
    pIntData->usStatusSld |= FL_SLDHASFOCUS;

    /******************************************************************/
    /* Send notification message to owner to let him know focus gained*/
    /******************************************************************/
    SendMessage( PARENTOF(hwndSld), WM_COMMAND,                         /*DAB2*/
                 (WPARAM)MAKELONG( IDOF(hwndSld), SLN_SETFOCUS ),       /*DAB2*/
                 (LPARAM)hwndSld );                                     /*DAB2*/
  }
  /********************************************************************/
  /* Otherwise window is losing focus                                 */
  /********************************************************************/
  else
  {
    /******************************************************************/
    /* Reset the slider has focus flag in the internal data area      */
    /******************************************************************/
    pIntData->usStatusSld &= ~FL_SLDHASFOCUS;

    /******************************************************************/
    /* Send notification message to owner to let him know focus lost  */
    /******************************************************************/
    SendMessage( PARENTOF(hwndSld), WM_COMMAND,                         /*DAB2*/
                 (WPARAM)MAKELONG( IDOF(hwndSld), SLN_KILLFOCUS ),      /*DAB2*/
                 (LPARAM)hwndSld );                                     /*DAB2*/
  }

  /********************************************************************/
  /* Calculate border rectangle from internal data                    */
  /********************************************************************/
  CalcRectangle( hwndSld, pIntData, pIntData->xShaft,pIntData->yShaft,
                 pIntData->usShaftLength, &rclBorder );

  /********************************************************************/
  /* Redraw arm bitmap in memory (since focus has changed)            */
  /********************************************************************/
  pIntData->usStatusSld |= FL_SLDFOCUSCHANGE;
  CreateSliderArm( hwndSld, pIntData );

  /********************************************************************/
  /* Now redraw the slider with all of the focus attributes set and   */
  /* force paint to occur (if dragging, painting later will cause an  */
  /* invalid saved area without focus)                                */
  /********************************************************************/
  InvalidateRect( hwndSld, &rclBorder, FALSE);
  UpdateWindow (hwndSld);

  return;
}

/**********************************************************************/
/* Procedure: SldProcessMouseDown - This routine will determine over  */
/*            what part of the slider the mouse was depressed.  It    */
/*            will compare the x,y coordinates of the mouse against   */
/*            the key slider components, beginning with the slider    */
/*            arm, then the shaft, and if buttons exist, the button   */
/*            area.  Finally it will check for detents, and if any    */
/*            it will check if one of them was clicked upon.          */
/*            The appropriate routine will be called if a mouse       */
/*            click was of the correct type and on the proper target. */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         sCx      - x position of mouse                             */
/*         sCy      - y position of mouse                             */
/*         bSelectBtn - If TRUE, button one pressed                   */
/*                      If FALSE, button two pressed                  */
/*                                                                    */
/* Output: bProcessed - If TRUE, button click was processed           */
/*                      If FALSE, button click was ignored            */
/**********************************************************************/
BOOL SldProcessMouseDown( HANDLE hwndSld, PISLDDATA pIntData,
                          LONG sCx, LONG sCy, BOOL bSelectBtn )         /*JEH1*/
{
  ULONG  ulStyle;             /* Current style bit flags              *//*JEH1*/
  POINT ptlCoords;
  RECT  rclBorder;
  RECT  rclButton;
  PDETENTDATA pDetentCurr;
  RECT  rclDetent;
  RECT  rclWindow;      /* rectl for client and setviewportorg */
  USHORT usScrollRate;                                                  /*JEH1*/
  USHORT usDetentSize=5;                                                /*JEH1*/
  USHORT usButtonLength;                                                /*JEH1*/
  BOOL   bMBDChk = FALSE;

  /********************************************************************/
  /* If button action over window, set focus to this window           */
  /********************************************************************/
  if (!( pIntData->usStatusSld & FL_SLDHASFOCUS ))
  {
    /******************************************************************/
    /* Set active window to this window and set the focus here also   */
    /******************************************************************/
    SetActiveWindow( hwndSld);
    SetFocus( hwndSld);
  }

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Check if readonly and if so, do not process button click         */
  /********************************************************************/
  if (ulStyle & SLS_READONLY)
    return (FALSE);

  /********************************************************************/
  /* Capture the mouse if over the window and mouse button is pressed */
  /********************************************************************/
  SetCapture( hwndSld );

  /********************************************************************/
  /* Map x, y coordinates to a point and get anchor block handle      */
  /********************************************************************/
  ptlCoords.x = sCx;
  ptlCoords.y = sCy;

  /********************************************************************/
  /* Check if point is within arm rectangle (dragging arm).  Both     */
  /* mouse buttons can drag the arm so no need to check which button  */
  /********************************************************************/
  if ( PtInRect( &(pIntData->rclSliderArm), ptlCoords ))
  {
    /******************************************************************/
    /* Get a presentation space to draw into                          */
    /******************************************************************/
    GetClientRect( hwndSld, &rclWindow );
    hpsDragArm = GetDC( hwndSld );

    /******************************************************************/
    /* If the DC could be obtained, then process clicking on the      */
    /* slider arm by starting to drag it and set flag to show drag is */
    /* occurring (and HDC was gotten)                                 */
    /******************************************************************/
    if (hpsDragArm)
    {
      pIntData->usStatusSld |= FL_SLDDRAGGINGARM;
      SldProcessArmPress( hwndSld, pIntData, hpsDragArm );
    }

    /******************************************************************/
    /* if The DC could not be obtained, then we can't be drawing any  */
    /* of the arm drag; release the mouse capture and return FALSE to */
    /* indicate that we are basically ignoring the mouse click        */
    /******************************************************************/
    else
    {
      ReleaseCapture();
      return (FALSE);
    }
  }
  /********************************************************************/
  /* Check for clicks within the shaft                                */
  /********************************************************************/
  else
  {

    /******************************************************************/
    /* Calculate border rectangle from internal data                  */
    /******************************************************************/
    CalcRectangle( hwndSld, pIntData, pIntData->xShaft,pIntData->yShaft,
                   pIntData->usShaftLength, &rclBorder );

    /******************************************************************/
    /* Check if point is within border rectangle (clicking on shaft)  */
    /* Pass button flag to shaft routine since different action is    */
    /*  taken depending on mouse button used                          */
    /******************************************************************/
    if ( PtInRect( &rclBorder, ptlCoords ))
    {
      SldProcessShaftPress( hwndSld, pIntData, sCx, sCy, bSelectBtn );
    }
    else
    {
      /****************************************************************/
      /* Check if buttons exist on the slider                         */
      /****************************************************************/
      if ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT))
      {
        /**************************************************************/
        /* Get the length of a button based on current shaft breadth  */
        /**************************************************************/
        usButtonLength = CalcButtonLength( pIntData );

        /**************************************************************/
        /* Calculate button rectangle from internal data              */
        /**************************************************************/
        CalcRectangle( hwndSld, pIntData, pIntData->xButton, pIntData->
                   yButton, (USHORT)(usButtonLength << 1), &rclButton );/*JEH1*/

        /**************************************************************/
        /* Check if point is within button rectangle (pressing a      */
        /* slider button) and if mouse button 1 was used              */
        /**************************************************************/
        if (( PtInRect( &rclButton, ptlCoords )) && bSelectBtn)
        {
          /************************************************************/
          /* Process the button press by updating position of arm     */
          /* based on which button was pressed and set button flag    */
          /************************************************************/
          bMBDChk = TRUE;
          pIntData->usStatusSld |= FL_SLDBUTTONPRESS;
          SldProcessButtonPress( hwndSld, pIntData, sCx, sCy, TRUE );

          /************************************************************/
          /* Set initial timer flag since starting at slower rate     */
          /************************************************************/
          pIntData->usStatusSld |= FL_SLDINITIALTIMER;

          /************************************************************/
          /* Query system for current scrolling rate and use on slider*/
          /************************************************************/
          usScrollRate = 200;

          /************************************************************/
          /* Start a timer (for case where user holds button down)    */
          /************************************************************/
          SetTimer( hwndSld, TID_SLIDERBUTTONS, usScrollRate, NULL);
        }
      }

      /****************************************************************/
      /* Check flag for either no buttons or buttons but not clickedon*/
      /****************************************************************/
      if ( !bMBDChk )
      {
        /**************************************************************/
        /* If FALSE, mouse click still not claimed.  Check for detents*/
        /* to see if one of them was clicked on.  First check if there*/
        /* is detent data to be used for clicking on and MB1 used     */
        /**************************************************************/
        if (( pDetentCurr = pIntData->pDetentData ) && bSelectBtn )
        {
          /************************************************************/
          /* Get detent size of the rectangle based on shaft width    */
          /************************************************************/
          if ( pIntData->usShaftBreadth > 24 )
            usDetentSize += min( 8, ( 2 * ((pIntData->usShaftBreadth -
                                 10) / 15 )));

          /************************************************************/
          /* If we have detent information stored, then check it      */
          /************************************************************/
          do
          {
            /**********************************************************/
            /* Calculate the rectangle of the current detent          */
            /**********************************************************/
            CalcDetentRectangle( hwndSld, pIntData, pDetentCurr,
                                 usDetentSize, &rclDetent );

            /**********************************************************/
            /* Check if point is within detent rectangle              */
            /**********************************************************/
            if ( PtInRect( &rclDetent, ptlCoords ))
            {
              /********************************************************/
              /* If so, set that as position to move to and go there  */
              /********************************************************/
              MoveSliderArm( hwndSld, pIntData,
                             pDetentCurr->usDetentPos, TRUE, TRUE);
              break;
            }

          /************************************************************/
          /* Set next detent as current detent and repeat if no match */
          /************************************************************/
          } while ( pDetentCurr = pDetentCurr->pNext );
        }
        /**************************************************************/
        /* If mouse button two outside of shaft, send context menu    */
        /**************************************************************/
        else if (!bSelectBtn)
        {
          ptlCoords.x = sCx;
          ptlCoords.y = sCy;
          ClientToScreen( hwndSld, &ptlCoords );
          PostMessage( hwndSld, WM_CONTEXTMENU, (WPARAM)hwndSld,
                       (LPARAM)MAKELONG( ptlCoords.x, ptlCoords.y ));
        }
        else
          return (FALSE);
      }
    }
  }
  return (TRUE);
}


/**********************************************************************/
/* Procedure: SldProcessMouseMove - This routine will determine over  */
/*            what part of the slider the mouse is moving.  It will   */
/*            perform custom pointer processing initially.  If a drag */
/*            of the slider arm is in progress, it will update the    */
/*            arm to the appropriate position based on the mouse ptr. */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         sCx      - x position of mouse                             */
/*         sCy      - y position of mouse                             */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID SldProcessMouseMove( HANDLE hwndSld, PISLDDATA pIntData,
                          LONG sCx, LONG sCy )                          /*JEH1*/
{
  ULONG    ulStyle;           /* Current style bit flags              *//*JEH1*/
  POINT   ptlCoords;
  HICON hptCur=0;                                                       /*JEH1*/
  HICON hptRet=0;                                                       /*JEH1*/
  RECT    rclBorder;
  RECT    rclWindow;
  RECT    rclArmArea;
  SHORT    sMouseOffset;                                                /*JEH1*/

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Map x, y coordinates to a point and get anchor block handle      */
  /********************************************************************/
  ptlCoords.x = sCx;
  ptlCoords.y = sCy;

  /********************************************************************/
  /* Get the rectangle for the slider window                          */
  /********************************************************************/
  GetClientRect( hwndSld, &rclWindow );

  /********************************************************************/
  /* If the mouse is within valid slider window coordinates and is    */
  /* not captured, send a WM_SETCURSOR message to the slider's owner  */
  /* to allow the owner to specify a custom pointer.                  */
  /********************************************************************/
  if ( PtInRect( &rclWindow, ptlCoords ) && !GetCapture())
    SendMessage ( PARENTOF(hwndSld), WM_SETCURSOR, (WORD)hwndSld,
                  MAKELONG( HTCLIENT, WM_MOUSEMOVE ));

  /********************************************************************/
  /* Check if this mouse move is part of dragging the slider          */
  /********************************************************************/
  if (( pIntData->usStatusSld & FL_SLDDRAGGINGARM ) &&
      ( hwndSld == GetCapture() ))
  {
    /******************************************************************/
    /* Get distance one can travel from mouse before losing tracking  */
    /******************************************************************/
    sMouseOffset = (int)GetDefaultSize( SLD_DEFARMBREADTH, ulStyle );

    /******************************************************************/
    /* Calculate border rectangle from internal data                  */
    /******************************************************************/
    CalcRectangle( hwndSld, pIntData, pIntData->xShaft, pIntData->
                   yShaft, pIntData->usShaftLength, &rclBorder );

    /******************************************************************/
    /* Check orientation of the slider                                */
    /******************************************************************/
    if (ulStyle & SLS_VERTICAL)

      /****************************************************************/
      /* Set arm area rectangle to be inside the border edges and     */
      /* right and left of the slider arm by the default arm breadth  */
      /****************************************************************/
      SetRect( &rclArmArea,
               (int)pIntData->rclSliderArm.left - sMouseOffset,
               (int)rclBorder.top - sMouseOffset,     /* win */
               (int)pIntData->rclSliderArm.right + sMouseOffset,
               (int)rclBorder.bottom + sMouseOffset);
    else
      /****************************************************************/
      /* Set arm area rectangle to be inside the border edges and     */
      /* right and left of the slider arm by the default arm breadth  */
      /****************************************************************/
      SetRect( &rclArmArea,
               (int)rclBorder.left - sMouseOffset,
               (int)pIntData->rclSliderArm.top - sMouseOffset,    /* win */
               (int)rclBorder.right + sMouseOffset,
               (int)pIntData->rclSliderArm.bottom + sMouseOffset);

    /******************************************************************/
    /* Check if point is within the surrounding arm rectangle         */
    /******************************************************************/
    if ( PtInRect( &rclArmArea, ptlCoords ))

      /****************************************************************/
      /* If drag, calculate where the arm would be at this point      */
      /* and set the slider arm position to this new position         */
      /****************************************************************/
      pIntData->usArmPos = CalcArmPosFromPt( hwndSld, pIntData, sCx,
                                             sCy );

    /******************************************************************/
    /* If outside rectangle, then reset position to previous spot     */
    /******************************************************************/
    else
      pIntData->usArmPos = pIntData->usPreviousArmPos;

    /******************************************************************/
    /* Calculate the arm rectangle if it was at this point            */
    /******************************************************************/
    CalcArmRectangle( hwndSld, pIntData, &(pIntData->rclSliderArm),
                      pIntData->usArmPos );

    /******************************************************************/
    /* Drag the slider arm to this new position                       */
    /******************************************************************/
    DrawDraggedArm( pIntData, hpsDragArm, DI_SLIDE );

    /******************************************************************/
    /* Repaint the shaft (done here because of ribbon strip)          */
    /******************************************************************/
    PaintShaft( hwndSld, hpsDragArm, &rclBorder,
        &(pIntData->rclSliderArm) );

    /******************************************************************/
    /* Send a slider tracking notification message to the owner,      */
    /* passing the current arm position in hi mp2 position            */
    /******************************************************************/
    SendMessage( PARENTOF(hwndSld), WM_COMMAND,                         /*DAB2*/
                 (WPARAM)MAKELONG( IDOF(hwndSld), SLN_SLIDERTRACK ),    /*DAB2*/
                 (LPARAM)hwndSld );
  }
  else
    pIntData->usStatusSld &= ~FL_SLDDRAGGINGARM;
  return;
}


/**********************************************************************/
/* Procedure: SldProcessMouseUp - This routine will reset actions in  */
/*            progress due to a mouse down on the slider.  If the     */
/*            user had pressed Mouse button 1 (MB1) on a button,      */
/*            this routine will stop the timer (which was moving the  */
/*            arm) and will reset the button visually to an non-      */
/*            depressed state.  If the user was dragging the slider   */
/*            arm (with either mouse button), this will stop the      */
/*            sliding and update the screen accordingly.              */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*                                                                    */
/* Output: bProcessed - If TRUE, button click was processed           */
/*                      If FALSE, button click was ignored            */
/**********************************************************************/
BOOL SldProcessMouseUp( HANDLE hwndSld, PISLDDATA pIntData )
{
  ULONG  ulStyle;             /* Current style bit flags              *//*JEH1*/
  HDC    hps;
  RECT   rclArrowDecr;
  RECT   rclArrowIncr;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Check if readonly and if so, do not process button click         */
  /********************************************************************/
  if (ulStyle & SLS_READONLY)
    return (FALSE);

  /********************************************************************/
  /* Release the mouse                                                */
  /********************************************************************/
  ReleaseCapture();

  /********************************************************************/
  /* Check if this up is from mouse click on a slider button          */
  /********************************************************************/
  if ( pIntData->usStatusSld & FL_SLDBUTTONPRESS )
  {
    /******************************************************************/
    /* Stop the button timer                                          */
    /******************************************************************/
    KillTimer( hwndSld, TID_SLIDERBUTTONS );

    /******************************************************************/
    /* Calculate two separate button rectangles from the entire rectl */
    /******************************************************************/
    CalcArrowRectangles (hwndSld, pIntData, &rclArrowDecr,
        &rclArrowIncr);

    /******************************************************************/
    /* Check if the decrement button is still depressed               */
    /******************************************************************/
    if (pIntData->usStatusSld & FL_SLDDECREMENTDOWN)
    {

      /****************************************************************/
      /* Reset flags for button since it must be up now               */
      /****************************************************************/
      pIntData->usStatusSld &= ~FL_SLDDECREMENTDOWN;

      /****************************************************************/
      /* Invalidate button so that it gets repainted                  */
      /****************************************************************/
      InvalidateRect (hwndSld, &rclArrowDecr, FALSE);
    }

    /******************************************************************/
    /* Check if the decrement button is still depressed               */
    /******************************************************************/
    if (pIntData->usStatusSld & FL_SLDINCREMENTDOWN)
    {

      /****************************************************************/
      /* Reset flag for button since it must be up now                */
      /****************************************************************/
      pIntData->usStatusSld &= ~FL_SLDINCREMENTDOWN;

      /****************************************************************/
      /* Invalidate button so that it gets repainted                  */
      /****************************************************************/
      InvalidateRect (hwndSld, &rclArrowIncr, FALSE);
    }

    /******************************************************************/
    /* Reset button press flag since button is no longer pressed      */
    /******************************************************************/
    pIntData->usStatusSld &= ~FL_SLDBUTTONPRESS;
  }

  /********************************************************************/
  /* Check if this up is from a mouse click on an arm rectangle       */
  /********************************************************************/
  else if ( pIntData->usStatusSld & FL_SLDDRAGGINGARM )
  {
    /******************************************************************/
    /* Terminate the drag of the slider arm at this new position      */
    /******************************************************************/
    DrawDraggedArm( pIntData, hpsDragArm, DI_END );

    /******************************************************************/
    /* Drop the arm here by blitting it out to screen                 */
    /******************************************************************/
    BltCopy( pIntData->hbmArm, hpsDragArm, 0, 0, (pIntData->            /*JEH1*/
             rclSliderArm.right - pIntData->rclSliderArm.left),
             (pIntData->rclSliderArm.bottom - pIntData->                /*JEH1*/
             rclSliderArm.top), pIntData->rclSliderArm.left,            /*JEH1*/
             pIntData->rclSliderArm.top, SRCCOPY,                       /*JEH1*/
             SRC_BITMAP );

    /******************************************************************/
    /* Release presentation space we got for the drag and reset flag  */
    /******************************************************************/
    ReleaseDC( hwndSld, hpsDragArm );
    pIntData->usStatusSld &= ~FL_SLDDRAGGINGARM;

    /******************************************************************/
    /* Check if snap is on                                            */
    /******************************************************************/
    if (ulStyle & SLS_SNAPTOINCREMENT)

      /****************************************************************/
      /* Move the slider arm to this new position                     */
      /****************************************************************/
      MoveSliderArm( hwndSld, pIntData, pIntData->usArmPos, TRUE,
          (pIntData->usArmPos != pIntData->usPreviousArmPos));

    /******************************************************************/
    /* If snap is off, check if buttons are to be painted             */
    /******************************************************************/
    else
    {

      /****************************************************************/
      /* Check for end of scale reached with the slider arm and if a  */
      /* flag has been modified, update button area                   */
      /****************************************************************/
      if ((CheckForEndOfScale (hwndSld, pIntData, 0)) &&                /*JEH1*/
          ((ulStyle & SLS_BUTTONSLEFT) || (ulStyle & SLS_BUTTONSRIGHT)))
      {

        /**************************************************************/
        /* Get a presentation space to draw into and update buttons   */
        /**************************************************************/
        if (hps = GetDC (hwndSld))
        {
          PaintButtons (hwndSld, pIntData, hps);

          /************************************************************/
          /* When update drawing complete, release the obtained hps   */
          /************************************************************/
          ReleaseDC (hwndSld, hps);
        }
      }

      /****************************************************************/
      /* If snap is off, send notification message (if on, it was sent*/
      /* at end of MoveSliderArm) - only send if position changed     */
      /****************************************************************/
      if (pIntData->usArmPos != pIntData->usPreviousArmPos)
        SendMessage( PARENTOF(hwndSld), WM_COMMAND,                     /*DAB2*/
                     (WPARAM)MAKELONG( IDOF(hwndSld), SLN_CHANGE ),     /*DAB2*/
                     (LPARAM)hwndSld );
    }
  }

  return (TRUE);
}


/**********************************************************************/
/* Procedure: SldProcessShaftPress - This routine will determine over */
/*            what part of the slider shaft the mouse button was      */
/*            pressed.  If mouse button 1 was used, it will move one  */
/*            increment in that direction.  If mouse button 2 was     */
/*            used, it will move the arm to that position.            */
/*                                                                    */
/* Inputs: hwndSld  - handle of the slider window                     */
/*         pIntData - Pointer to the internal control block           */
/*         sCx      - x position of mouse                             */
/*         sCy      - y position of mouse                             */
/*         bSelectBtn - If TRUE, button one pressed                   */
/*                      If FALSE, button two pressed                  */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID SldProcessShaftPress( HANDLE hwndSld, PISLDDATA pIntData,
                           LONG sCx, LONG sCy, BOOL bSelectBtn )        /*JEH1*/
{
  USHORT usArmPos;                                                      /*JEH1*/
  ULONG  ulStyle;             /* Current style bit flags              *//*JEH1*/
  BOOL   bDecrement=FALSE;
  BOOL   bIncrement=TRUE;

  /********************************************************************/
  /* Query for the style of the slider control                        */
  /********************************************************************/
  ulStyle = STYLEOF( hwndSld );

  /********************************************************************/
  /* Reset flag definitions based on home - done here to override     */
  /* home check in IncrementArmPosition routine which switches dir.   */
  /********************************************************************/
  if (ulStyle & SLS_HOMERIGHT)
  {
    bDecrement = !bDecrement;
    bIncrement = !bIncrement;
  }

  /********************************************************************/
  /* If drag, calculate where the arm would be at this point          */
  /* and set the slider arm position according to the type of button  */
  /********************************************************************/
  usArmPos = CalcArmPosFromPt( hwndSld, pIntData, sCx, sCy);

  /********************************************************************/
  /* Check if mouse button 1 was used                                 */
  /********************************************************************/
  if (bSelectBtn)
  {
    /******************************************************************/
    /* Check which direction to move the arm one increment in         */
    /******************************************************************/
    if (usArmPos < pIntData->usArmPos)

      /****************************************************************/
      /* If new position is less than old, decrement in that direction*/
      /****************************************************************/
      IncrementArmPosition( hwndSld, pIntData, bDecrement );

    /******************************************************************/
    /* Otherwise position is greater                                  */
    /******************************************************************/
    else if (usArmPos > pIntData->usArmPos)

      /****************************************************************/
      /* so increment in that direction                               */
      /****************************************************************/
      IncrementArmPosition( hwndSld, pIntData, bIncrement );
  }

  /********************************************************************/
  /* otherwise mouse button two used so go directly to this position  */
  /********************************************************************/
  else

    /******************************************************************/
    /* Move the slider arm to this new position                       */
    /******************************************************************/
    MoveSliderArm( hwndSld, pIntData, usArmPos, TRUE, TRUE);

  return;
}


/**********************************************************************/
/* Procedure: SldProcessTimer - This routine will process the timer   */
/*            message (if it is slider's timer) and move the arm      */
/*            accordingly.  The timer is started when MB1 is pressed  */
/*            down on a button.  If the button is held down, then we  */
/*            need to reflect a typamatic simulation of holding a key */
/*            down.  The slider arm will move for each timer message  */
/*            received.  If the button is held down long, a faster    */
/*            rate will be used.                                      */
/*                                                                    */
/* Inputs: hwndSld   - handle of the slider window                    */
/*         pIntData  - Pointer to the internal control block          */
/*         usTimerId - Id of the timer message                        */
/*                                                                    */
/* Output: none                                                       */
/**********************************************************************/
VOID SldProcessTimer( HANDLE hwndSld, PISLDDATA pIntData,
                      UINT   usTimerId )                                /*JEH1*/
{
  POINT ptlMousePos;
  USHORT usScrollRate;                                                  /*JEH1*/

  /********************************************************************/
  /* Check if this timer message is ours (compare timer id)           */
  /********************************************************************/
  if ( usTimerId == TID_SLIDERBUTTONS )
  {
    /******************************************************************/
    /* Query current mouse position and map it to window coordinates  */
    /******************************************************************/
    GetCursorPos( &ptlMousePos );
    ScreenToClient ( hwndSld, &ptlMousePos);
    /******************************************************************/
    /* Process it as another button depression                        */
    /******************************************************************/
    SldProcessButtonPress( hwndSld, pIntData, ptlMousePos.x,            /*JEH1*/
                           ptlMousePos.y, FALSE );                      /*JEH1*/

    /******************************************************************/
    /* If initial timer setup, then change from first scroll rate     */
    /******************************************************************/
    if ( pIntData->usStatusSld & FL_SLDINITIALTIMER )
    {
      /****************************************************************/
      /* Reset initial timer flag                                     */
      /****************************************************************/
      pIntData->usStatusSld &= ~FL_SLDINITIALTIMER;

      /****************************************************************/
      /* Query system for current scrolling rate and use on slider    */
      /****************************************************************/
      usScrollRate = 250;
      /****************************************************************/
      /* Change timer rate (for case where user holds button down)    */
      /****************************************************************/
      SetTimer( hwndSld, TID_SLIDERBUTTONS, usScrollRate, NULL);
    }
  }
  return;
}

