/* Revision: 42 1.6 source/ui/cclnt/notebook/book.c, cclbook, ioc.v400, 001006  */
/**********************************************************************/
/*                                                                    */
/*                         IBM CONFIDENTIAL                           */
/*                                                                    */
/*        (IBM Confidential-Restricted when combined with the         */
/*         aggregated OCO source modules for this program)            */
/*                                                                    */
/*                       OCO SOURCE MATERIALS                         */
/*                                                                    */
/*         20G0400 (C) Copyright IBM Corp. 1992 (Unpublished)         */
/*                                                                    */
/* ================================================================== */
/*                                                                    */
/*                          N O T E B O O K                           */
/*                                                                    */
/*                         Filename: BOOK.C                           */
/*                                                                    */
/*             ( Windows NT Notebook Control Component )              */
/*                                                                    */
/*                         IBM CONFIDENTIAL                           */
/*                                                                    */
/*                (C) Copyright IBM Corporation 1997                  */
/*                       All Rights Reserved                          */
/*            Licensed Material - Program Property of IBM             */
/*                                                                    */
/**********************************************************************/
/**********************************************************************/
/* The internal structure of the book:                                */
/* -----------------------------------                                */
/* The book is set up as a series of pages.  Internally each of the   */
/* pages is identical (all are PAGE) the difference depends on the    */
/* attributes set for the particular page.  A page with major tab     */
/* attributes and a page with minor tab attributes will from here on  */
/* out be named a "major tab" and "minor tab" respectively.           */
/*                                                                    */
/* A Major tab is defined by itemType = BKA_MAJOR                     */
/* A Minor tab is defined by itemType = BKA_MINOR                     */
/*                                                                    */
/* All the pages are connected via a doubly linked list.              */
/**********************************************************************/
/******************************************************************************/
/* Product: ICLUINT.DLL                                                       */
/* Filename: BOOK.C - Main source for the Windows Notebook control            */
/*                                                                            */
/* Flag    Date    Defect  Description                                        */
/* ----  --------  -----  --------------------------------------------------- */
/* JEH1  06/22/93         Initial modifications for NT execution              */
/* DAB1  12/01/93         Additional modifications for NT execution           */
/* DAB2  10/04/94         Updated notifications to return information         */
/*                        supported in PM which was left out of Win3.1        */
/* DAB3  10/08/95         Corrected logic error with not checking ptr         */
/*                        before using to dereference                         */
/* MAB1  12/21/95  6249   Add support for BKN_PAGESELECTEDPENDING             */
/*                        notification                                        */
/* MAB2  01/08/96  50612  Disabled scroll arrow visible when it should be     */
/*                        invisible.                                          */
/* MAB3  01/13/96  6468   <Alt> UpArrow not working - cannot use keyboard to  */
/*                        move from page window to tab.                       */
/******************************************************************************/
#include <windows.h>
#include "iclnbw.h"
#include "bookp.h"

extern HANDLE vhBookResource[CRESOURCES];
extern USHORT vsDevIndX;
extern USHORT vsDevIndY;
extern PPAGE  vppgDeleted;
extern PPAGE  vppgPrevSelected;

/*---------------------------------------------------------------------------*
 * AdjustBinding
 *
 * Adjust the book rectangle by the binding that appears outside the top
 * page area.  Also add the top page border width.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

void AdjustBinding(
  PBOOK  pbk,                               /* book instance data (locked)   */
  LPRECT lprc,
  BOOL   fShrink)
{
  SHORT cxBinding, dx;

  cxBinding = (pbk->flStyle & BKS_SPIRALBIND) ? BKD_SPIRALBINDWIDTH/2 :
                                                BKD_SOLIDBINDWIDTH;

  dx = ((fShrink) ? 1 : -1) * cxBinding * vsDevIndX;

  if      (pbk->flStyle & BKS_MAJORTABRIGHT)   lprc->left   += dx;
  else if (pbk->flStyle & BKS_MAJORTABLEFT)    lprc->right  -= dx;
  else if (pbk->flStyle & BKS_MAJORTABTOP)     lprc->bottom -= dx;
  else if (pbk->flStyle & BKS_MAJORTABBOTTOM)  lprc->top    += dx;
}


/*---------------------------------------------------------------------------*
 * CalcBinding
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID CalcBinding(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  SHORT cxBinding, dxOpposite, dxSpiral;
  BOOL  fSpiral = (pbk->flStyle & BKS_SPIRALBIND) ? TRUE : FALSE;

  pbk->rcBinding = pbk->rcTopPage;
  cxBinding      = (fSpiral) ? BKD_SPIRALBINDWIDTH : BKD_SOLIDBINDWIDTH;
  dxOpposite     = cxBinding   * vsDevIndX;
  dxSpiral       = cxBinding/2 * vsDevIndX;

#ifdef BKS_SPIRALBIND
  switch (pbk->flStyle & BKS_MAJORTABMASK)
  {
    case BKS_MAJORTABBOTTOM:
      if (fSpiral)            pbk->rcBinding.top -= dxSpiral;
      pbk->rcBinding.bottom = pbk->rcBinding.top +  dxOpposite;
      break;

    case BKS_MAJORTABTOP:
      if (fSpiral)         pbk->rcBinding.bottom += dxSpiral;
      pbk->rcBinding.top = pbk->rcBinding.bottom -  dxOpposite;
      break;

    case BKS_MAJORTABRIGHT:
      if (fSpiral)           pbk->rcBinding.left -= dxSpiral;
      pbk->rcBinding.right = pbk->rcBinding.left +  dxOpposite;
      break;

    case BKS_MAJORTABLEFT:
      if (fSpiral)          pbk->rcBinding.right += dxSpiral;
      pbk->rcBinding.left = pbk->rcBinding.right -  dxOpposite;
      break;
  }

#else
  if (pbk->flStyle & BKS_MAJORTABBOTTOM)
    pbk->rcBinding.bottom = pbk->rcBinding.top + dxOpposite;

  else if (pbk->flStyle & BKS_MAJORTABTOP)
    pbk->rcBinding.top = pbk->rcBinding.bottom - dxOpposite;

  else if (pbk->flStyle & BKS_MAJORTABRIGHT)
    pbk->rcBinding.right = pbk->rcBinding.left + dxOpposite;

  else if (pbk->flStyle & BKS_MAJORTABLEFT)
    pbk->rcBinding.left = pbk->rcBinding.right - dxOpposite;
#endif
}


/*---------------------------------------------------------------------------*
 * CalcMaxVisibleTabs
 *
 * Calculates the maximum number of visible tabs for the major and minor tab
 * sides of the notebook.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID CalcMaxVisTabs(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  /*-------------------------------------*/
  /* major tabs are on the left or right */
  /*-------------------------------------*/
  if (pbk->flStyle & (BKS_MAJORTABRIGHT | BKS_MAJORTABLEFT))
  {
    pbk->cMaxVisMajors = CountDisplayableTabs (pbk, pbk->rcTopPage.top,
                            pbk->rcTopPage.bottom, pbk->cyMajorTab, 0);

    pbk->cMaxVisMinors = CountDisplayableTabs (pbk, pbk->rcTopPage.left,
                            pbk->rcTopPage.right, pbk->cxMinorTab, 0);

    pbk->cMaxVisMajorsScroll = CountDisplayableTabs (pbk,pbk->rcTopPage.top,
                                  pbk->rcTopPage.bottom, pbk->cyMajorTab,
                                  pbk->cxScrollArrow);

    pbk->cMaxVisMinorsScroll = CountDisplayableTabs (pbk, pbk->rcTopPage.left,
                                  pbk->rcTopPage.right, pbk->cxMinorTab,
                                  pbk->cxScrollArrow);
  }

  /*-------------------------------------*/
  /* major tabs are on the top or bottom */
  /*-------------------------------------*/
  else
  {
    pbk->cMaxVisMajors = CountDisplayableTabs (pbk, pbk->rcTopPage.left,
                            pbk->rcTopPage.right, pbk->cxMajorTab, 0);

    pbk->cMaxVisMinors = CountDisplayableTabs (pbk, pbk->rcTopPage.top,
                            pbk->rcTopPage.bottom, pbk->cyMinorTab, 0);

    pbk->cMaxVisMajorsScroll = CountDisplayableTabs (pbk, pbk->rcTopPage.left,
                                  pbk->rcTopPage.right, pbk->cxMajorTab,
                                  pbk->cxScrollArrow);

    pbk->cMaxVisMinorsScroll = CountDisplayableTabs (pbk,pbk->rcTopPage.top,
                                  pbk->rcTopPage.bottom, pbk->cyMinorTab,
                                  pbk->cxScrollArrow);
  }
}


/*---------------------------------------------------------------------------*
 * CalcPageButton
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID CalcPageButton(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  SHORT xWndLeft, xWndRight, yWnd;
  HWND  hwndLeft, hwndRight;

  switch (LOWORD (pbk->flStyle & FC_BACKPAGES))
  {
    case BKS_BACKPAGESBR:
      xWndRight = pbk->rcTopPage.right - pbk->cxPageButton - 1;
      xWndLeft  = xWndRight - (pbk->cxPageButton - 1);
      yWnd      = pbk->rcTopPage.bottom - (pbk->cyPageButton + 1);
      break;

    case BKS_BACKPAGESBL:
      xWndLeft  = pbk->rcTopPage.left + 2;
      xWndRight = xWndLeft + (pbk->cxPageButton - 1);
      yWnd      = pbk->rcTopPage.bottom - (pbk->cyPageButton + 1);
      break;

    case BKS_BACKPAGESTR:
      xWndRight = pbk->rcTopPage.right - pbk->cxPageButton - 1;
      xWndLeft  = xWndRight - (pbk->cxPageButton - 1);
      yWnd      = pbk->rcTopPage.top + 2;
      break;

    case BKS_BACKPAGESTL:
      xWndLeft  = pbk->rcTopPage.left + 2;
      xWndRight = xWndLeft + (pbk->cxPageButton - 1);
      yWnd      = pbk->rcTopPage.top + 2;
      break;
  }

  hwndLeft  = GetDlgItem (pbk->hwndBook, ID_PAGEBUTTONLEFT);
  hwndRight = GetDlgItem (pbk->hwndBook, ID_PAGEBUTTONRIGHT);

  SetWindowPos (hwndLeft, NULL, xWndLeft, yWnd,
                pbk->cxPageButton, pbk->cyPageButton, 0);               /*JEH1*/

  SetWindowPos (hwndRight, NULL, xWndRight, yWnd,
                pbk->cxPageButton, pbk->cyPageButton, 0);               /*JEH1*/

  BookShowWindow (hwndLeft,  TRUE);
  BookShowWindow (hwndRight, TRUE);
  InvalidateRect (hwndLeft,  NULL, FALSE);
  InvalidateRect (hwndRight, NULL, FALSE);
}


/*---------------------------------------------------------------------------*
 * CalcPageRect
 *
 * Calculate the application page rectangle from a Notebook rectangle or
 * calculate a Notebook rectangle from a page rectangle
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

BOOL CalcPageRect(
  PBOOK  pbk,                               /* book instance data (locked)   */
  LPRECT lprc,
  BOOL   fPage)
{
  /*---------------------------------------------------*/
  /* If an invalid rectangle is passed, return failure */
  /*---------------------------------------------------*/
  if (!lprc || (lprc->left > lprc->right) || (lprc->top > lprc->bottom))
    return FALSE;

  /*---------------------------------------------------------------*/
  /* Adjust the Notebook rectangle to compsensate for the backpage */
  /* spacing and the tabs.                                         */
  /*---------------------------------------------------------------*/
  CalcTopPage (pbk, lprc, fPage);

  /*-------------------------------------------------------------*/
  /* Adjust the Notebook rectangle to compsensate for the inside */
  /* binding and the page button/status area space.              */
  /*-------------------------------------------------------------*/
  CalcPageWnd (pbk, lprc, fPage);

  /*--------------------------------------------------------------------------*/
  /* if the calculation turned the rectangle inside out, return an empty rect */
  /*--------------------------------------------------------------------------*/
  if ((lprc->left >= lprc->right) || (lprc->top >= lprc->bottom))
    SetRectEmpty (lprc);

  return TRUE;
}


/*---------------------------------------------------------------------------*
 * CalcPages
 *
 * Calculate the rectangle of the top page including the backpage spacing.
 * This is used for painting the backpage space area the same color as the
 * top page.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

void CalcPages(
  PBOOK  pbk,                               /* book instance data (locked)   */
  LPRECT lprc)
{
  USHORT cxPageSpacing;

  /*--------------------------*/
  /* Calculate backpage space */
  /*--------------------------*/
  cxPageSpacing = BKD_NUMVISPAGEEDGES * BKD_PAGESPACING * vsDevIndX;

  /*-------------------------------------*/
  /* Backpages bottom right or top right */
  /*-------------------------------------*/
  if (pbk->flStyle & (BKS_BACKPAGESBR | BKS_BACKPAGESTR))
  {
    lprc->right += cxPageSpacing;
    if (pbk->flStyle & BKS_BACKPAGESBR)  lprc->bottom += cxPageSpacing;
    else                                 lprc->top    -= cxPageSpacing;
  }

  /*-----------------------------------*/
  /* Backpages bottom left or top left */
  /*-----------------------------------*/
  else if (pbk->flStyle & (BKS_BACKPAGESBL | BKS_BACKPAGESTL))
  {
    lprc->left -= cxPageSpacing;
    if (pbk->flStyle & BKS_BACKPAGESBL)  lprc->bottom += cxPageSpacing;
    else                                 lprc->top    -= cxPageSpacing;
  }
}


/*---------------------------------------------------------------------------*
 * CalcPageWnd
 *
 * Calculate the application page window rectangle.  This is the book top
 * page area less the inside binding and pageButton/status line spacing.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID CalcPageWnd(
  PBOOK  pbk,                               /* book instance data (locked)   */
  LPRECT lprc,
  BOOL   fShrink)
{
  TEXTMETRIC tm;
  HDC        hdc;
  SHORT      cyFont;
  BOOL       fSelectFont;
  SHORT      dx = (fShrink) ? 1 : -1;

  /*--------------------------------------------------*/
  /* If a null rectangle pointer is passed in, return */
  /*--------------------------------------------------*/
  if (!lprc)   return;

  /*------------------------------------------------------------*/
  /* Adjust the rectangle to compsensate for the border and the */
  /* inside binding                                             */
  /*------------------------------------------------------------*/
  AdjustBinding (pbk, lprc, fShrink);

  /*-----------------------------------------*/
  /* Get the maximum height of any character */
  /*-----------------------------------------*/
  hdc = GetDC (pbk->hwndBook);
  if (fSelectFont = (BOOL)pbk->hFont)                                   /*JEH1*/
    pbk->hFont = SelectObject (hdc, pbk->hFont);
  GetTextMetrics (hdc, &tm);
  if (fSelectFont)
    pbk->hFont = SelectObject (hdc, pbk->hFont);
  ReleaseDC (pbk->hwndBook, hdc);

  /*----------------------------------------------------------------*/
  /* calculate the height of a line of text including some spacing, */
  /* making sure there is at least enough room for the page buttons */
  /*----------------------------------------------------------------*/
  cyFont = max (tm.tmHeight + tm.tmExternalLeading + 4, pbk->cyPageButton);

  /*-------------------------------------------------------------------*/
  /* inset the rectangle so it doesn't overwrite anything else we draw */
  /*-------------------------------------------------------------------*/
  InflateRect (lprc, -dx * 2, -dx * 2);

  /*------------------------------------------------*/
  /* make room for the status line and page buttons */
  /*------------------------------------------------*/
  if (pbk->flStyle & (BKS_BACKPAGESBR | BKS_BACKPAGESBL))
  {
    lprc->bottom -= dx * cyFont;
  }
  else
  {
    lprc->top += dx * cyFont;
  }
}


/*---------------------------------------------------------------------------*
 * CalcStatusLine
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

void CalcStatusLine(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  /*-----------------------------------------------*/
  /* sets the page coordinates to book coordinates */
  /*-----------------------------------------------*/
  pbk->rcStatus = pbk->rcAppPageWnd;

  if (pbk->flStyle & (BKS_BACKPAGESBR | BKS_BACKPAGESTR))
  {
    pbk->rcStatus.right -= 2 * pbk->cxPageButton;

    if (pbk->flStyle & BKS_BACKPAGESBR)
    {
      pbk->rcStatus.top    = pbk->rcAppPageWnd.bottom + 1;
      pbk->rcStatus.bottom = pbk->rcTopPage.bottom    - 1;
    }
    else
    {
      pbk->rcStatus.top    = pbk->rcTopPage.top    + 1;
      pbk->rcStatus.bottom = pbk->rcAppPageWnd.top - 1;
    }
  }

  else if (pbk->flStyle & (BKS_BACKPAGESBL | BKS_BACKPAGESTL))
  {
    pbk->rcStatus.left += 2 * pbk->cxPageButton;

    if (pbk->flStyle & BKS_BACKPAGESBL)
    {
      pbk->rcStatus.top    = pbk->rcAppPageWnd.bottom + 1;
      pbk->rcStatus.bottom = pbk->rcTopPage.bottom    - 1;
    }

    else
    {
      pbk->rcStatus.bottom = pbk->rcAppPageWnd.top - 1;
      pbk->rcStatus.top    = pbk->rcTopPage.top    + 1;
    }
  }
}


/*---------------------------------------------------------------------------*
 * CalcTopPage
 *
 * Calculate the rectangle of the top page.  It is assumed that the
 * rectangle has been adjusted by the binding.  This will adjust the
 * rectangle by the space taken up by the backpages and the tabs.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID CalcTopPage(
  PBOOK  pbk,                               /* book instance data (locked)   */
  LPRECT lprc,
  BOOL   fShrink)
{
  SHORT dx = (fShrink) ? 1 : -1;

  InflateRect (lprc, -dx * BKD_BOOKBORDER, -dx * BKD_BOOKBORDER);

//FIXME:  This may not be necessary with the way CalcPageRect was rewritten
  if (pbk->flStyle & BKS_SPIRALBIND)
  {
    AdjustBinding (pbk, lprc, fShrink);
  }

  /*-------------------------------*/
  /* backpages bottom or top right */
  /*-------------------------------*/
  if (pbk->flStyle & (BKS_BACKPAGESBR | BKS_BACKPAGESTR))
  {
    lprc->right -= dx * pbk->cxPageSpacing;

    if (pbk->flStyle & BKS_BACKPAGESBR)  lprc->bottom -= dx * pbk->cyPageSpacing;
    else                                 lprc->top    += dx * pbk->cyPageSpacing;
  }

  /*------------------------------*/
  /* backpages bottom or top left */
  /*------------------------------*/
  else
  {
    lprc->left += dx * pbk->cxPageSpacing;

    if (pbk->flStyle & BKS_BACKPAGESBL)  lprc->bottom -= dx * pbk->cyPageSpacing;
    else                                 lprc->top    += dx * pbk->cyPageSpacing;
  }
}


/*---------------------------------------------------------------------------*
 * CalcVertHorzSpace
 *
 * Set the vertical and horizontal spacing.  The spacing includes the
 * backpage size and the tab width and height and scroll arrow size.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID CalcVertHorzSpace(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  SHORT cxPageSpacing;

  /*-------------------------------------*/
  /* Calculate space between back pages. */
  /*-------------------------------------*/
  cxPageSpacing = BKD_NUMVISPAGEEDGES * BKD_PAGESPACING * vsDevIndX + 1;

  /*---------------------------------------------------------------*/
  /* If the major tabs are on the top or bottom, use the major tab */
  /* width for the horizontal spacing and the minor tab height for */
  /* the vertical spacing.                                         */
  /*---------------------------------------------------------------*/
  if (pbk->flStyle & (BKS_MAJORTABRIGHT | BKS_MAJORTABLEFT))
  {
    pbk->cxPageSpacing = cxPageSpacing + max (pbk->cxScrollArrow, pbk->cxMajorTab);
    pbk->cyPageSpacing = cxPageSpacing + max (pbk->cxScrollArrow, pbk->cyMinorTab);
  }

  /*---------------------------------------------------------------*/
  /* If the major tabs are on the top or bottom, use the minor tab */
  /* width for the horizontal spacing and the major tab height for */
  /* the vertical spacing.                                         */
  /*---------------------------------------------------------------*/
  else
  {
    pbk->cxPageSpacing = cxPageSpacing + max (pbk->cxScrollArrow, pbk->cxMinorTab);
    pbk->cyPageSpacing = cxPageSpacing + max (pbk->cxScrollArrow, pbk->cyMajorTab);
  }
}


/*---------------------------------------------------------------------------*
 * DeletePage
 *
 * Delete the specified page from the list.  If the page is a major or minor
 * tab, all child pages will be deleted.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

BOOL DeletePage (
  PBOOK  pbk,                               /* book instance data (locked)   */
  PPAGE  ppgDelete,
  USHORT fsDelete,
  BOOL   fNotify)
{
  PPAGE  ppgDeleteTemp;
  USHORT fsAttr;

  /*-------------------------------------------------------------*/
  /* If BKA_TAB is specified, save the page type to know when to */
  /* stop deleting.                                              */
  /*-------------------------------------------------------------*/
  if (ppgDelete && (fsDelete & BKA_TAB))   fsAttr = ppgDelete->fsAttr;

  /*-----------------------------------------------------------------*/
  /* If all pages are to be deleted, initialize the pointer to start */
  /* with the first major tab.                                       */
  /*-----------------------------------------------------------------*/
  if (fsDelete & BKA_ALL)   ppgDelete = pbk->ppgFirst;

  /*-----------------------*/
  /* Delete a SINGLE page. */
  /*-----------------------*/
  if ((VerifyPageId (pbk, ppgDelete)) && (fsDelete & BKA_SINGLE))
  {
    /*------------------------------------------------------------*/
    /* Destroy any windows associated with the page being deleted */
    /*------------------------------------------------------------*/
    DestroyPageWindows (pbk, ppgDelete);

    /*--------------------------------*/
    /* Reset the page/global pointers */
    /*--------------------------------*/
    ResetPagePtrs (pbk, ppgDelete);

    /*--------------------------------------------------------*/
    /* Free all memory/windows allocated for the deleted page */
    /*--------------------------------------------------------*/
    FreePageData (pbk, ppgDelete, fNotify);
  }

  /*------------------*/
  /* Delete ALL pages */
  /*------------------*/
  else if (fsDelete & BKA_ALL)
  {
    /*-----------------------------------------------------------*/
    /* reset all of the global pointers to their original values */
    /*-----------------------------------------------------------*/
    ResetGlobalPagePtrs (pbk, NULL);

    while (ppgDelete)
    {
      /*--------------------------------------------------------*/
      /* Save a pointer to the next record before the memory is */
      /*--------------------------------------------------------*/
      ppgDeleteTemp = ppgDelete->ppgNext;

      /*------------------------------------------------------------*/
      /* Destroy any windows associated with the page being deleted */
      /*------------------------------------------------------------*/
      DestroyPageWindows (pbk, ppgDelete);

      /*--------------------------------------------------------*/
      /* Free all memory/windows allocated for the deleted page */
      /*--------------------------------------------------------*/
      FreePageData (pbk, ppgDelete, fNotify);

      /*-----------------------------*/
      /* get next page to be deleted */
      /*-----------------------------*/
      ppgDelete = ppgDeleteTemp;
    }

    /*--------------------------------*/
    /* hide any visible scroll arrows */
    /*--------------------------------*/
    HideScrollArrows (pbk, ID_HORIZONTALLEFT, ID_HORIZONTALRIGHT);
    HideScrollArrows (pbk, ID_VERTICALTOP,    ID_VERTICALBOTTOM);

    /*-----------------------------------------------------*/
    /* invalidate the status text so it'll get cleared out */
    /*-----------------------------------------------------*/
    InvalidateRect (pbk->hwndBook, &pbk->rcStatus, TRUE);
  }

  /*-----------------------*/
  /* Delete a TAB section. */
  /*-----------------------*/
  else if ((VerifyPageId (pbk, ppgDelete)) &&
           (fsDelete == BKA_TAB) && (fsAttr & (BKA_MAJOR | BKA_MINOR)))
  {
    /*----------------------------------------------------------------*/
    /* Determine tab type at which to end delete. If the page type    */
    /* is a major, all pages up to the next major page will be        */
    /* deleted. If the page type is a minor, all pages up to the next */
    /* major or minor will be deleted.                                */
    /*----------------------------------------------------------------*/
    fsAttr = (fsAttr & BKA_MINOR) ? BKA_MAJOR | BKA_MINOR : BKA_MAJOR;

    /*---------------------------------------------------------------*/
    /* Save a pointer to the next record before the memory is freed. */
    /*---------------------------------------------------------------*/
    ppgDeleteTemp = ppgDelete->ppgNext;

    /*------------------------------------------------------------*/
    /* Destroy any windows associated with the page being deleted */
    /*------------------------------------------------------------*/
    DestroyPageWindows (pbk, ppgDelete);

    /*--------------------------------*/
    /* Reset the page/global pointers */
    /*--------------------------------*/
    ResetPagePtrs (pbk, ppgDelete);

    /*--------------------------------------------------------*/
    /* Free all memory/windows allocated for the deleted page */
    /*--------------------------------------------------------*/
    FreePageData (pbk, ppgDelete, fNotify);

    /*-----------------------------*/
    /* get next page to be deleted */
    /*-----------------------------*/
    ppgDelete = ppgDeleteTemp;

    while (ppgDelete && !(fsAttr & ppgDelete->fsAttr))
    {
      /*------------------------------------------------------------*/
      /* Destroy any windows associated with the page being deleted */
      /*------------------------------------------------------------*/
      DestroyPageWindows(pbk,ppgDelete);

      /*--------------------------------*/
      /* Reset the page/global pointers */
      /*--------------------------------*/
      ResetPagePtrs(pbk,ppgDelete);

      /*--------------------------------------------------------*/
      /* Save a pointer to the next record before the memory is */
      /*--------------------------------------------------------*/
      ppgDeleteTemp = ppgDelete->ppgNext;

      /*--------------------------------------------------------*/
      /* Free all memory/windows allocated for the deleted page */
      /*--------------------------------------------------------*/
      FreePageData (pbk, ppgDelete, fNotify);

      /*-----------------------------*/
      /* get next page to be deleted */
      /*-----------------------------*/
      ppgDelete = ppgDeleteTemp;
    }
  }

  /*----------------------------------------------------------------*/
  /* if we get here, we had an invalid parameter, so return failure */
  /*----------------------------------------------------------------*/
  else
  {
    return FALSE;
  }

  /*----------------------------------------------------------------*/
  /* If after reseting the global pointers, there are no top pages, */
  /* set to the first top page                                      */
  /*----------------------------------------------------------------*/
  if (pbk->ppgFirst && !QueryTopPage (pbk))
  {
    if (pbk->ppgFirst->fsAttr & BKA_MAJOR)
      pbk->ppgTopMajor = pbk->ppgFirst;
    else if (pbk->ppgFirst->fsAttr & BKA_MINOR)
      pbk->ppgTopMinor = pbk->ppgFirst;
    else
      pbk->ppgTop = pbk->ppgFirst;
  }

  UpdatePageButtonStates (pbk);
  UpdateScrollStates (pbk, BKA_MAJOR);
  UpdateScrollStates (pbk, BKA_MINOR);

  return TRUE;
}


/*---------------------------------------------------------------------------*
 * DestroyBookResources
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID DestroyBookResources()
{
  USHORT i;

  /*-----------------------------------------------------------*/
  /* Destroy the page button bitmaps if they have been loaded. */
  /*-----------------------------------------------------------*/
  for (i = ID_PAGEBUTTONRUP; i < CRESOURCES; i++)
  {
    if (vhBookResource[i])
    {
      DeleteObject (vhBookResource[i]);
      vhBookResource[i] = NULL;
    }
  }
}


/*---------------------------------------------------------------------------*
 * DestroyPageWindows
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID DestroyPageWindows(
  PBOOK pbk,                                /* book instance data (locked)   */
  PPAGE ppgDestroy)
{
  /*------------------------------------------------------------------*/
  /* Hide the Notebook page window used for clipping application page */
  /*------------------------------------------------------------------*/
  if (ppgDestroy == QueryTopPage (pbk))
  {
    HideTopPage (pbk,ppgDestroy);
  }
}


/*---------------------------------------------------------------------------*
 * FreePageData
 *
 * Frees any data associated with a page.  Destroys any windows associated
 * with a page.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID FreePageData(
  PBOOK pbk,                                /* book instance data (locked)   */
  PPAGE ppgFree,
  BOOL  fNotify)
{
  PPAGE        ppgT;
  DELETENOTIFY deleteNotify;                                            /*DAB2*/

  /*------------------------------------------------------------------*/
  /* Check if notification is supposed to occur                       */
  /*------------------------------------------------------------------*/
  if (fNotify)
  {
    /*----------------------------------------------------------------*/
    /* If so, initialize a delete notification structure              */
    /*----------------------------------------------------------------*/
    deleteNotify.hwndBook = pbk->hwndBook;                              /*DAB2*/
    deleteNotify.hwndPage = ppgFree->hwndPage;                          /*DAB2*/
    deleteNotify.ulAppPageData = ppgFree->ulAppData;                    /*DAB2*/
    deleteNotify.hbmTab = NULL;                                         /*DAB2*/
    if (ppgFree->fsAttr & (BKA_MAJOR | BKA_MINOR))                      /*DAB2*/
      deleteNotify.hbmTab = ppgFree->ptab->hbmTab;                      /*DAB2*/

    vppgDeleted = ppgFree;
    NotifyParent( pbk->hwndBook, BKN_PAGEDELETED, TRUE,                 /*DAB2*/
                  (LPARAM)&deleteNotify );                              /*DAB2*/
    vppgDeleted = NULL;
  }

  /*----------------------------------------------------------*/
  /* Reset the parent of the app's page window.  We only want */
  /* to do this if this app page isn't associated with any    */
  /* other notebook page.                                     */
  /*----------------------------------------------------------*/
  for (ppgT = pbk->ppgFirst;
       ppgT && (ppgT != ppgFree) && (ppgT->hwndPage != ppgFree->hwndPage);
       ppgT = ppgT->ppgNext);

  /*-----------------------------------------------------------*/
  /* if we didn't find a page (other than the one being freed) */
  /* that has the same page window as the one being freed,     */
  /* reset the page window's parent to the original parent     */
  /*-----------------------------------------------------------*/
  if (!ppgT && IsWindow (ppgFree->hwndPage))
  {
    SetParent (ppgFree->hwndPage, ppgFree->hwndPageParent);
    BookShowWindow (ppgFree->hwndPage, FALSE);
  }

  /*-----------------------*/
  /* free tab and tab text */
  /*-----------------------*/
  if (ppgFree->ptab)
  {
    if (ppgFree->ptab->npszText)  BookFreeMem (ppgFree->ptab->npszText);
    BookFreeMem (ppgFree->ptab);
  }

  /*-----------------------*/
  /* free status line text */
  /*-----------------------*/
  if (ppgFree->npszStatus)  BookFreeMem (ppgFree->npszStatus);

  /*-----------*/
  /* free page */
  /*-----------*/
  BookFreeMem (ppgFree);
}


/*---------------------------------------------------------------------------*
 * GetTabPtr
 *
 * Gets a pointer to the next page with a major or minor tab.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

PPAGE GetTabPtr( PPAGE ppg, USHORT usTabType, UINT usTabOrder )         /*JEH1*/
{
  /*-----------------------------------------------------------------------*/
  /* usTabType may contain other page attributes, so make sure they are 0. */
  /*-----------------------------------------------------------------------*/
  usTabType &= (BKA_MAJOR | BKA_MINOR);

  /*----------------------------------------------------------------*/
  /* Search until the next matching tab page is found or the end of */
  /* the list is reached.                                           */
  /*----------------------------------------------------------------*/
  while (ppg && !(ppg->fsAttr & usTabType))
  {
    if (!(usTabType & BKA_MAJOR) && (ppg->fsAttr & BKA_MAJOR))  ppg = NULL;
    else if (usTabOrder & BKA_NEXT)    ppg = ppg->ppgNext;
    else                               ppg = ppg->ppgPrev;
  }

  return (ppg);
}


/*---------------------------------------------------------------------------*
 * GotoPage
 *
 * Brings a new page to the top of the notebook.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

BOOL GotoPage (
  PBOOK  pbk,                               /* book instance data (locked)   */
  PPAGE  ppgGoto,
  ULONG  usOrder,                                                       /*JEH1*/
  BOOL   fDrawImmediately)
{
  PPAGE ppgMajor, ppgMinor, ppgTopTab, ppgCurrentTop, ppgNextTab;
  PPAGE ppgCursorMajor, ppgCursorMinor, ppgCursor;
  RECT  rcInvalid, rcInvalid1, rcUpdate;
  PPAGE ppgNewTop;
  PAGESELECTNOTIFY pageSelectNotify;                                    /*DAB2*/

  /****************************************************************/   //->@MAB1
  /* Initialize rectangles used for painting                      */
  /****************************************************************/
  SetRectEmpty( &rcInvalid );
  SetRectEmpty( &rcInvalid1 );
  SetRectEmpty( &rcUpdate );                                           //@MAB1<-

  /*--------------------------------------------*/
  /* Save current major and minor page pointers */
  /*--------------------------------------------*/
  ppgMajor       = pbk->ppgTopMajor;
  ppgMinor       = pbk->ppgTopMinor;
  ppgCursorMajor = pbk->ppgCursorMajor;
  ppgCursorMinor = pbk->ppgCursorMinor;
  ppgCursor      = pbk->ppgCursor;
  ppgCurrentTop  = QueryTopPage(pbk);

  /****************************************************************/   //->@MAB1
  /* Determine what the new top page will be. This will allow     */
  /* us to send the BKN_PAGESELECTEDPENDING without actually      */
  /* turning the page.                                            */
  /****************************************************************/
  if ( (ppgGoto) && VerifyPageId( pbk, ppgGoto ) )
    ppgNewTop = ppgGoto;
  else if (usOrder & BKA_NEXT)
  {
    ppgNewTop = ppgCurrentTop;
    if (ppgNewTop)
      ppgNewTop = ppgNewTop->ppgNext;
  }
  else if (usOrder & BKA_PREV)
  {
    ppgNewTop = ppgCurrentTop;
    if (ppgNewTop)
      ppgNewTop = ppgNewTop->ppgPrev;
  }
  else
    return FALSE;

  /****************************************************************/
  /* Send a notification to the application that a new page has   */
  /* been selected.                                               */
  /****************************************************************/
  pageSelectNotify.hwndBook = pbk->hwndBook;
  pageSelectNotify.ulPageIdCur = (DWORD)ppgCurrentTop;
  pageSelectNotify.ulPageIdNew = (DWORD)ppgNewTop;

  vppgPrevSelected = ppgCurrentTop;
  NotifyParent( pbk->hwndBook, BKN_PAGESELECTEDPENDING, TRUE,
                (LPARAM)&pageSelectNotify );
  vppgPrevSelected = NULL;

  /****************************************************************/
  /* If the application sets the ulPageIdNew field to 0, the page */
  /* is not turned, as long as we're not inserting the first page.*/
  /****************************************************************/
  if ( pageSelectNotify.ulPageIdNew == 0L )
  {
    if (ppgCurrentTop)            //Ensure 1st page is always inserted...
    {
      return( FALSE );
    }
  }                                                                    //@MAB1<-

  /********************************************************************/
  /* Set new top page pointer                                         */
  /********************************************************************/
  if (ppgGoto)                  ResetTopPagePtrs (pbk, ppgGoto, 0);     /*JEH1*/
  else if (usOrder & BKA_NEXT)  ResetTopPagePtrs (pbk, NULL, BKA_NEXT);
  else if (usOrder & BKA_PREV)  ResetTopPagePtrs (pbk, NULL, BKA_PREV);
  else                          return (FALSE);

  ppgNewTop = QueryTopPage (pbk);                                       /*DAB2*/

  pageSelectNotify.hwndBook = pbk->hwndBook;                            /*DAB2*/
  pageSelectNotify.ulPageIdCur = (DWORD)ppgCurrentTop;                  /*DAB2*/
  pageSelectNotify.ulPageIdNew = (DWORD)ppgNewTop;                      /*DAB2*/

  vppgPrevSelected = ppgCurrentTop;
  NotifyParent( pbk->hwndBook, BKN_PAGESELECTED, TRUE,                  /*DAB2*/
                (LPARAM)&pageSelectNotify );                            /*DAB2*/
  vppgPrevSelected = NULL;

  /*--------------------------------*/
  /* if the top page has changed... */
  /*--------------------------------*/
  if (ppgCurrentTop != ppgNewTop)
  {
//#ifdef FOCUS_TO_TOP_PAGE                                               //@MAB3
    BOOL fChangeFocus = FALSE;
//#endif                                                                 //@MAB3

    /*---------------------------*/
    /* Hide the current top page */
    /*---------------------------*/
    if (ppgCurrentTop && ppgCurrentTop->hwndPage)
    {
//#ifdef FOCUS_TO_TOP_PAGE                                               //@MAB3
      /*------------------------------------*/
      /* if the old top page has the focus, */
      /* move the focus to the new top page */
      /*------------------------------------*/
      fChangeFocus = ((ppgCurrentTop->hwndPage == GetFocus()) &&
                       ppgNewTop->hwndPage);
//#endif                                                                 //@MAB3

      /*-----------------------------------------------------*/
      /* don't use BookShowWindow here because it won't hide */
      /* the window if the notebook is invisible,            */
      /*-----------------------------------------------------*/
      ShowWindow (ppgCurrentTop->hwndPage, SW_HIDE);
    }

    /*-----------------------------*/
    /* Invalidate the new top page */
    /*-----------------------------*/
    ShowTopPage (pbk, NULL);
    UpdatePageButtonStates (pbk);

//#ifdef FOCUS_TO_TOP_PAGE                                               //@MAB3
    /*-----------------------------------------------*/
    /* change the focus after ShowTopPage because we */
    /* can't set the focus to an invisible window    */
    /*-----------------------------------------------*/
    if (fChangeFocus)
    {
      SetFocus (ppgNewTop->hwndPage);
    }
//#endif                                                                 //@MAB3
  }

  /*---------------------------------------*/
  /* Get a pointer to the topmost tab page */
  /* for drawing the new cursor emphasis   */
  /*---------------------------------------*/
  ppgTopTab  = (pbk->ppgTopMinor) ? pbk->ppgTopMinor : pbk->ppgTopMajor;
  ppgNextTab = (ppgTopTab) ? ppgTopTab->ptab->ppgNextTab : NULL;
  pbk->ppgCursor = ppgTopTab;

  /*------------------------------------*/
  /* Reset global cursor record pointer */
  /*------------------------------------*/
  if (pbk->ppgCursor)
  {
    /*--------------------------------*/
    /* New cursored record is a major */
    /*--------------------------------*/
    if (ppgTopTab->fsAttr & BKA_MAJOR)
    {
      pbk->ppgCursorMajor = ppgTopTab;
      pbk->ppgCursorMinor = ((ppgTopTab->ptab->ppgNextTab) &&
                             (ppgTopTab->ptab->ppgNextTab->fsAttr & BKA_MINOR))
                          ? ppgTopTab->ptab->ppgNextTab : NULL;
    }

    /*--------------------------------*/
    /* New cursored record is a minor */
    /*--------------------------------*/
    else
    {
      pbk->ppgCursorMajor = (pbk->ppgTopMajor) ? pbk->ppgTopMajor : NULL;
      pbk->ppgCursorMinor = ppgTopTab;
    }
  }

  /*---------------------------------*/
  /* if the top tab isn't visible... */
  /*---------------------------------*/
  if (!ppgTopTab ||
      (ppgTopTab &&
       ((ppgTopTab->fsAttr & (BKA_MAJOR | BKA_MINOR)) &&
        (!IsTabVisible (pbk, pbk->ppgTopMajor)        ||
         (ppgMajor != pbk->ppgTopMajor)               ||
         (ppgCurrentTop == ppgCursorMajor)            ||
         (ppgTopTab == pbk->ppgCursorMajor)))))
  {
    /*----------------------------------------------------------------------*/
    /* Determine the invalidate rectangle for the currently displayed tabs. */
    /*----------------------------------------------------------------------*/
    CalcInvalidateRect (pbk, pbk->ppgFirstVisMajor, pbk->ppgLastVisMajor,
                        &rcInvalid);

    /*-----------------------------------------------------*/
    /* Reset the major visible page start and end pointers */
    /*-----------------------------------------------------*/
    if (ppgTopTab && (ppgTopTab->fsAttr & BKA_MAJOR))
    {
      if (!IsTabVisible (pbk, ppgTopTab))
        ResetVisPagePtrs (pbk, ppgTopTab, BKA_MAJOR, 0);                /*JEH1*/
      ResetTabData (pbk, ppgTopTab, 0);                                 /*JEH1*/
    }
    else
    {
      PPAGE ppgT = (!pbk->ppgTopMajor && ppgMajor) ? ppgMajor :
                    (pbk->ppgTopMajor)             ? pbk->ppgTopMajor : NULL;

      if (!IsTabVisible (pbk, pbk->ppgTopMajor) && ppgT)
        ResetVisPagePtrs (pbk, ppgT, BKA_MAJOR, 0);                     /*JEH1*/

      if (ppgT)   ResetTabData (pbk, ppgT, 0);                          /*JEH1*/
    }

    /*----------------------------------------------------------------------*/
    /* Determine the invalidate rectangle for the new tabs to be displayed. */
    /*----------------------------------------------------------------------*/
    CalcInvalidateRect (pbk, pbk->ppgFirstVisMajor,
                        pbk->ppgLastVisMajor, &rcInvalid1);

    /*-------------------------------------------------------*/
    /* Determine union of the current and new tab rectangles */
    /*-------------------------------------------------------*/
    if (UnionRect (&rcUpdate, &rcInvalid, &rcInvalid1))
    {
      if (fDrawImmediately)
      {
        DrawUpdateRect (pbk, pbk->ppgFirstVisMajor,
                        pbk->ppgLastVisMajor, &rcUpdate, NULL);
      }
      else
      {
        InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
      }
    }
  }

  /*------------------------------------------------------------*/
  /* if the top page is a minor, or the top page is a major and */
  /* the next tab is a minor, or a minor tab is cursored...     */
  /*------------------------------------------------------------*/
#if 0   /*-------------------------< JDR >--------------------------------*/
  if (!ppgTopTab ||
      (ppgTopTab &&
       ((ppgTopTab->fsAttr & BKA_MINOR) ||
        ((ppgTopTab->fsAttr & BKA_MAJOR) &&
         (ppgNextTab && (ppgNextTab->fsAttr & BKA_MINOR)) ||
          ppgCursorMinor))))
#endif     /*-------------------------< JDR >--------------------------------*/
  if (TRUE)
  {
    /*----------------------------------------------------------------------*/
    /* Determine the invalidate rectangle for the currently displayed tabs. */
    /*----------------------------------------------------------------------*/
    CalcInvalidateRect(pbk, pbk->ppgFirstVisMinor,
                       pbk->ppgLastVisMinor, &rcInvalid);

    /*-----------------------------------------------------*/
    /* Reset the major visible page start and end pointers */
    /*-----------------------------------------------------*/
    if (ppgTopTab && (ppgTopTab->fsAttr & BKA_MAJOR))
    {
      if (ppgNextTab && (ppgNextTab->fsAttr & BKA_MINOR))
      {
        if (!IsTabVisible (pbk, ppgNextTab))
          ResetVisPagePtrs (pbk, ppgNextTab, BKA_MINOR, 0);             /*JEH1*/

        ResetTabData (pbk, ppgNextTab, 0);                              /*JEH1*/
      }
      else
      {
        ResetVisPagePtrs (pbk, NULL, BKA_MINOR, 0);                     /*JEH1*/
      }
    }

    else
    {
      if (!IsTabVisible (pbk, pbk->ppgTopMinor))
        ResetVisPagePtrs (pbk, pbk->ppgTopMinor, BKA_MINOR, 0);         /*JEH1*/
      ResetTabData (pbk, pbk->ppgTopMinor, 0);                          /*JEH1*/
    }

    /*----------------------------------------------------------------------*/
    /* Determine the invalidate rectangle for the new tabs to be displayed. */
    /*----------------------------------------------------------------------*/
    CalcInvalidateRect(pbk, pbk->ppgFirstVisMinor,
                       pbk->ppgLastVisMinor, &rcInvalid1);

    /*-------------------------------------------------------*/
    /* Determine union of the current and new tab rectangles */
    /*-------------------------------------------------------*/
    if (UnionRect (&rcUpdate, &rcInvalid, &rcInvalid1))
    {
      if (fDrawImmediately)
      {
        DrawUpdateRect(pbk, pbk->ppgFirstVisMinor,
                       pbk->ppgLastVisMinor, &rcUpdate, NULL);
      }
      else
      {
        InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
      }
    }

    if (ppgTopTab && (ppgTopTab->fsAttr & BKA_MINOR) &&
        (CountTabs (pbk, ppgTopTab->fsAttr, pbk->cMaxVisMinors) >
                                            pbk->cMaxVisMinors))
    {
      DrawScrollArrows(pbk,ppgTopTab,&pbk->rcBookPages);
    }
  }

  UpdateScrollStates(pbk, BKA_MAJOR);
  UpdateScrollStates(pbk, BKA_MINOR);

  return (TRUE);
}


/*---------------------------------------------------------------------------*
 * HideAllWindows
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID HideAllWindows(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  /*-------------------------------------------------------*/
  /* Hide the vertical and horizontal scroll arrow windows */
  /*-------------------------------------------------------*/
  HideScrollArrows(pbk, ID_HORIZONTALLEFT, ID_HORIZONTALRIGHT);
  HideScrollArrows(pbk, ID_VERTICALTOP, ID_VERTICALBOTTOM);

#ifdef HIDE_EVERYTHING
  /*-----------------------------*/
  /* Hide the page button window */
  /*-----------------------------*/
  BookShowWindow (GetDlgItem (pbk->hwndBook, ID_PAGEBUTTONLEFT), FALSE);
  BookShowWindow (GetDlgItem (pbk->hwndBook, ID_PAGEBUTTONRIGHT), FALSE);

  /*--------------------------*/
  /* Hide the top page window */
  /*--------------------------*/
  if (pbk->ppgFirst)  HideTopPage (pbk, NULL);
#endif
}


/*---------------------------------------------------------------------------*
 * HideScrollArrows
 *
 * Hides scroll arrows with given IDs.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

void HideScrollArrows(
  PBOOK pbk,                                /* book instance data (locked)   */
  SHORT id1,
  SHORT id2)
{
  /*-----------------------------------------------------*/            //->@MAB2
  /* don't use BookShowWindow here because it won't hide */
  /* the window if the notebook is invisible,            */
  /*-----------------------------------------------------*/
  ShowWindow( GetDlgItem( pbk->hwndBook,id1 ), SW_HIDE );
  ShowWindow( GetDlgItem( pbk->hwndBook,id2 ), SW_HIDE );              //@MAB2<-
}


/*---------------------------------------------------------------------------*
 * HideTopPage
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID HideTopPage(
  PBOOK pbk,                                /* book instance data (locked)   */
  PPAGE ppg)
{
  PPAGE  ppgTop;
  HDC    hdc;

  /*--------------------------*/
  /* Get topmost page pointer */
  /*--------------------------*/
  ppgTop = (ppg) ? ppg : QueryTopPage (pbk);

  /*------------------------------------*/
  /* Get presentation space for drawing */
  /*------------------------------------*/
  if (!(hdc = GetDC (pbk->hwndBook)))  return;

  if (ppgTop && ppgTop->hwndPage)
  {
    BookShowWindow (ppgTop->hwndPage, FALSE);
  }
  else
  {
    BookFillRect (hdc, &pbk->rcAppPageWnd, pbk->crColor[CLR_PAGEBG]);
  }

  DrawStatusLineText (pbk, ppgTop, hdc, FALSE);
  ReleaseDC (pbk->hwndBook, hdc);
}


/*---------------------------------------------------------------------------*
 * InsertPage
 *
 * Inserts a page into the notebook.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

BOOL InsertPage(
  PBOOK  pbk,                               /* book instance data (locked)   */
  PPAGE  ppgInsert,
  PPAGE  ppgOrder,
  USHORT usPageOrder)
{
  /*-----------------------------------------------------------------*/
  /* if we're inserting relative to a previously inserted page and   */
  /* the page isn't valid, or we couldn't insert the new page in our */
  /* page list, return failure                                       */
  /*-----------------------------------------------------------------*/
  if ((((usPageOrder == BKA_PREV) || (usPageOrder == BKA_NEXT)) &&
       !VerifyPageId (pbk, ppgOrder))  ||
      !InsertPageList (pbk, ppgInsert, ppgOrder, usPageOrder))
  {
    return (FALSE);
  }

  if (ppgInsert->fsAttr & (BKA_MAJOR | BKA_MINOR))
  {
    InsertTabPageList  (pbk, ppgInsert, ppgOrder, usPageOrder);
    UpdateScrollStates (pbk, BKA_MAJOR);
    UpdateScrollStates (pbk, BKA_MINOR);
  }

  UpdatePageButtonStates(pbk);
  return (TRUE);
}

/*---------------------------------------------------------------------------*
 * InsertPageList
 *
 * Inserts a page into a page list.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

BOOL InsertPageList(
  PBOOK  pbk,                               /* book instance data (locked)   */
  PPAGE  ppgInsert,
  PPAGE  ppgOrder,
  USHORT usPageOrder)
{
  /*---------------------------------------------*/
  /* if this isn't the first page in the book... */
  /*---------------------------------------------*/
  if (pbk->ppgFirst)
  {
    /*-----------------------------------------------------------------*/
    /* if FIRST or PREV before first, insert at the front of the list. */
    /*-----------------------------------------------------------------*/
    if ((usPageOrder & BKA_FIRST) ||
        ((usPageOrder & BKA_PREV) && (ppgOrder == pbk->ppgFirst)))
    {
      pbk->ppgFirst->ppgPrev = ppgInsert;
      ppgInsert->ppgNext = pbk->ppgFirst;
      pbk->ppgFirst = ppgInsert;
    }

    /*------------------------------------------------------------*/
    /* if LAST or NEXT after last, insert at the end of the list. */
    /*------------------------------------------------------------*/
    else if ((usPageOrder & BKA_LAST) ||
             ((usPageOrder & BKA_NEXT) && (ppgOrder == pbk->ppgLast)))
    {
      pbk->ppgLast->ppgNext = ppgInsert;
      ppgInsert->ppgPrev = pbk->ppgLast;
      pbk->ppgLast = ppgInsert;
    }

    /*----------------------------------------------------*/
    /* if NEXT specified, insert after the specified page */
    /*----------------------------------------------------*/
    else if ((usPageOrder & BKA_NEXT) && ppgOrder)
    {
      ppgInsert->ppgNext = ppgOrder->ppgNext;
      ppgInsert->ppgPrev = ppgOrder;
      ppgOrder->ppgNext->ppgPrev = ppgInsert;
      ppgOrder->ppgNext = ppgInsert;
    }

    /*-----------------------------------------------------*/
    /* if PREV specified, insert before the specified page */
    /*-----------------------------------------------------*/
    else if ((usPageOrder & BKA_PREV) && ppgOrder)
    {
      ppgInsert->ppgNext = ppgOrder;
      ppgInsert->ppgPrev = ppgOrder->ppgPrev;
      ppgOrder->ppgPrev->ppgNext = ppgInsert;
      ppgOrder->ppgPrev = ppgInsert;
    }

    /*-----------------------------------------------------*/
    /* otherwise the page order is invalid, return failure */
    /*-----------------------------------------------------*/
    else  return (FALSE);
  }

  /*----------------------------------------------------*/
  /* otherwise, this is the first page; if it is        */
  /* designated at the first or last page (the only     */
  /* legal option now), initialize global page pointers */
  /*----------------------------------------------------*/
  else if ((usPageOrder == BKA_FIRST) || (usPageOrder == BKA_LAST))
  {
    pbk->ppgFirst = ppgInsert;
    pbk->ppgLast  = ppgInsert;

    if (!(ppgInsert->fsAttr & (BKA_MAJOR | BKA_MINOR)))
      GotoPage(pbk, ppgInsert, 0, FALSE);                               /*JEH1*/
  }

  /*--------------------*/
  /* otherwise, we fail */
  /*--------------------*/
  else  return (FALSE);

  return (TRUE);
}


/*---------------------------------------------------------------------------*
 * InsertTabPageList
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

void InsertTabPageList(
  PBOOK  pbk,                               /* book instance data (locked)   */
  PPAGE  ppgInsert,
  PPAGE  ppgOrder,
  USHORT usPageOrder)
{
  PPAGE  ppgTab;
  USHORT cMaxTabs;                                                      /*JEH1*/
  RECT   rcInvalid1;
  RECT   rcInvalid;
  RECT   rcUpdate;
  BOOL   fResetMinors = FALSE;

  SetRectEmpty (&rcInvalid);
  SetRectEmpty (&rcInvalid1);

  /*------------------------------------------*/
  /* if there are tabs in the book already... */
  /*------------------------------------------*/
  if (pbk->ppgFirstTab)
  {
    /*-----------------------------------------------------------------*/
    /* if FIRST or PREV before first, insert at the front of the list. */
    /*-----------------------------------------------------------------*/
    if ( (usPageOrder & BKA_FIRST) ||
        ((usPageOrder & BKA_PREV) && (ppgOrder == pbk->ppgFirstTab)))
    {
      ppgInsert->ptab->ppgNextTab = pbk->ppgFirstTab;
      pbk->ppgFirstTab->ptab->ppgPrevTab = ppgInsert;
      pbk->ppgFirstTab = ppgInsert;
    }

    /*------------------------------------------------------------*/
    /* if LAST or NEXT after last, insert at the end of the list. */
    /*------------------------------------------------------------*/
    else if ( (usPageOrder & BKA_LAST) ||
             ((usPageOrder & BKA_NEXT) && (ppgOrder == pbk->ppgLastTab)))
    {
      ppgInsert->ptab->ppgPrevTab = pbk->ppgLastTab;
      pbk->ppgLastTab->ptab->ppgNextTab = ppgInsert;
      pbk->ppgLastTab = ppgInsert;
    }

    /*----------------------------------------------------*/
    /* if NEXT specified, insert after the specified page */
    /*----------------------------------------------------*/
    else if ((usPageOrder & BKA_NEXT) && ppgOrder)
    {
      if (ppgTab = GetTabPtr (ppgOrder, BKA_MAJOR | BKA_MINOR, BKA_PREV))
      {
        if (ppgTab->ptab->ppgNextTab)
            ppgTab->ptab->ppgNextTab->ptab->ppgPrevTab = ppgInsert;
        ppgInsert->ptab->ppgNextTab = ppgTab->ptab->ppgNextTab;
        ppgInsert->ptab->ppgPrevTab = ppgTab;
        ppgTab->ptab->ppgNextTab    = ppgInsert;
        if (ppgTab == pbk->ppgLastTab)  pbk->ppgLastTab = ppgInsert;
      }

      else if (ppgTab = GetTabPtr (ppgOrder, BKA_MAJOR | BKA_MINOR, BKA_NEXT))
      {
        if (ppgTab->ptab->ppgPrevTab)
            ppgTab->ptab->ppgPrevTab->ptab->ppgNextTab = ppgInsert;
        ppgInsert->ptab->ppgPrevTab = ppgTab->ptab->ppgPrevTab;
        ppgInsert->ptab->ppgNextTab = ppgTab;
        ppgTab->ptab->ppgPrevTab    = ppgInsert;
        if (ppgTab == pbk->ppgFirstTab)  pbk->ppgFirstTab = ppgInsert;
      }

      /*----------------------------------------------------------*/
      /* if this is a major tab, and it's being inserted before a */
      /* minor tab that's currently visible, we'll need to hide   */
      /* all of the visible minors after this new major           */
      /*----------------------------------------------------------*/
      if (ppgInsert->fsAttr & BKA_MAJOR)
      {
        if (ppgTab = GetTabPtr (ppgInsert->ppgNext, BKA_MINOR, BKA_NEXT))
        {
          CalcInvalidateRect (pbk, pbk->ppgFirstVisMinor, pbk->ppgLastVisMinor,
                              &rcInvalid);
          InvalidateRect (pbk->hwndBook, &rcInvalid, FALSE);
          fResetMinors = TRUE;
        }
      }
    }

    /*-----------------------------------------------------*/
    /* if PREV specified, insert before the specified page */
    /*-----------------------------------------------------*/
    else if ((usPageOrder & BKA_PREV) && ppgOrder)
    {
      /*----------------------------------------------------------*/
      /* if this is a major tab, and it's being inserted before a */
      /* minor tab that's currently visible, we'll need to hide   */
      /* all of the visible minors after this new major           */
      /*----------------------------------------------------------*/
      if ((ppgInsert->fsAttr & BKA_MAJOR) && (ppgOrder->fsAttr & BKA_MINOR))
      {
        CalcInvalidateRect (pbk, pbk->ppgFirstVisMinor, pbk->ppgLastVisMinor,
                            &rcInvalid);
        InvalidateRect (pbk->hwndBook, &rcInvalid, FALSE);
        fResetMinors = TRUE;
      }

      if (ppgTab = GetTabPtr(ppgOrder,(BKA_MAJOR | BKA_MINOR), BKA_NEXT))
      {
        if (ppgTab->ptab->ppgPrevTab)
            ppgTab->ptab->ppgPrevTab->ptab->ppgNextTab = ppgInsert;
        ppgInsert->ptab->ppgPrevTab = ppgTab->ptab->ppgPrevTab;
        ppgInsert->ptab->ppgNextTab = ppgTab;
        ppgTab->ptab->ppgPrevTab    = ppgInsert;
        if (ppgTab == pbk->ppgFirstTab)  pbk->ppgFirstTab = ppgInsert;
      }

      else if (ppgTab = GetTabPtr(ppgOrder,(BKA_MAJOR | BKA_MINOR), BKA_PREV))
      {
        if (ppgTab->ptab->ppgNextTab)
            ppgTab->ptab->ppgNextTab->ptab->ppgPrevTab = ppgInsert;
        ppgInsert->ptab->ppgNextTab = ppgTab->ptab->ppgNextTab;
        ppgInsert->ptab->ppgPrevTab = ppgTab;
        ppgTab->ptab->ppgNextTab    = ppgInsert;
        if (ppgTab == pbk->ppgLastTab)  pbk->ppgLastTab = ppgInsert;
      }
    }
  }

  /*-----------------------------------------------------*/
  /* otherwise, this is the first tab in the book so far */
  /*-----------------------------------------------------*/
  else
  {
    pbk->ppgFirstTab = ppgInsert;
    pbk->ppgLastTab = ppgInsert;
    ResetVisPagePtrs(pbk,ppgInsert,ppgInsert->fsAttr, 0);               /*JEH1*/
    GotoPage(pbk, ppgInsert, 0, FALSE);                                 /*JEH1*/
  }

  /*-----------------------------------------*/
  /* if the inserted page has a MINOR tab... */
  /*-----------------------------------------*/
  if (ppgInsert->fsAttr & BKA_MINOR)
  {
    if ((ppgInsert->ptab->ppgPrevTab == pbk->ppgTopMajor) ||
        ((ppgInsert == pbk->ppgFirstTab) && !(pbk->ppgTopMajor)))
    {
      pbk->ppgCursorMinor = ppgInsert;
    }

    cMaxTabs = CountTabs(pbk,ppgInsert->fsAttr, pbk->cMaxVisMinors);

    /*----------------------------------------------------------*/
    /* if this tab is visible, or it belongs to the current top */
    /* major tab and there's room left on the minor edge        */
    /*----------------------------------------------------------*/
    if (IsTabVisible (pbk, ppgInsert) ||
        ((ppgInsert->ptab->ppgPrevTab == pbk->ppgTopMajor) &&
            (cMaxTabs < pbk->cMaxVisMinors)))
    {
      ppgInsert->ptab->usLevel = BKD_MAXMINORTABLEVEL;

      /*------------------------------------*/
      /* Determine the invalidate rectangle */
      /* for the currently displayed tabs.  */
      /*------------------------------------*/
      CalcInvalidateRect(pbk, pbk->ppgFirstVisMinor,
                         pbk->ppgLastVisMinor, &rcInvalid);

      if (cMaxTabs > pbk->cMaxVisMinors)
      {
        ResetVisPagePtrs(pbk,ppgInsert,ppgInsert->fsAttr, 0);           /*JEH1*/

        // defect: 29052  aac
        // Removed next line, because ResetVisPagePtrs sets up all
        // pointers correctly.  This call just makes them off by 1.
//      pbk->ppgLastVisMinor = GetTabPtr(
//                                   pbk->ppgLastVisMinor->ptab->ppgPrevTab,
//                                   BKA_MINOR, BKA_PREV);
      }
      else
      {
        ResetVisPagePtrs(pbk,ppgInsert,ppgInsert->fsAttr, 0);           /*JEH1*/
      }
      ResetTabData(pbk, pbk->ppgCursorMinor,
                   QueryStartTabLevel(pbk, BKA_MINOR));
    }

    else if (cMaxTabs > pbk->cMaxVisMinors)
    {
      CalcInvalidateRect (pbk, pbk->ppgFirstVisMinor,
                          pbk->ppgLastVisMinor, &rcInvalid);

      if (pbk->ppgCursorMinor)
        ResetVisPagePtrs (pbk, pbk->ppgCursorMinor, BKA_MINOR, 0);      /*JEH1*/
      else
        ResetVisPagePtrs (pbk, ppgInsert, ppgInsert->fsAttr, 0);        /*JEH1*/

      ResetTabData (pbk, pbk->ppgFirstVisMinor,
                    QueryStartTabLevel (pbk, BKA_MINOR));
    }

    /*----------------------------------------------------------------------*/
    /* Determine the invalidate rectangle for the new tabs to be displayed. */
    /*----------------------------------------------------------------------*/
    CalcInvalidateRect(pbk, pbk->ppgFirstVisMinor,
                       pbk->ppgLastVisMinor, &rcInvalid1);

    /*-------------------------------------------------------*/
    /* Determine union of the current and new tab rectangles */
    /*-------------------------------------------------------*/
    if (UnionRect (&rcUpdate,&rcInvalid,&rcInvalid1))
    {
      InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
//    DrawUpdateRect(pbk, pbk->ppgFirstVisMinor,
//                   pbk->ppgLastVisMinor, &rcUpdate, NULL);
    }
  }

  /*-------------------------------------------------*/
  /* otherwise, the inserted page has a MAJOR tab... */
  /*-------------------------------------------------*/
  else
  {
    if (!(pbk->ppgCursorMajor))
      pbk->ppgCursorMajor = ppgInsert;

    cMaxTabs = CountTabs(pbk,ppgInsert->fsAttr, pbk->cMaxVisMajors);

    if ((IsTabVisible(pbk,ppgInsert)) || (cMaxTabs < pbk->cMaxVisMajors))
    {
      ppgInsert->ptab->usLevel = BKD_MAXMAJORTABLEVEL;

      /*------------------------------------*/
      /* Determine the invalidate rectangle */
      /* for the currently displayed tabs.  */
      /*------------------------------------*/
      CalcInvalidateRect(pbk, pbk->ppgFirstVisMajor,
                         pbk->ppgLastVisMajor, &rcInvalid);

      if (cMaxTabs > pbk->cMaxVisMajors)
      {
        ResetVisPagePtrs (pbk, ppgInsert, ppgInsert->fsAttr, 0);        /*JEH1*/
        // defect: 29052  aac
        // Removed next line, because ResetVisPagePtrs sets up all
        // pointers correctly.  This call just makes them off by 1.
//      pbk->ppgLastVisMajor =
//           GetTabPtr (pbk->ppgLastVisMajor->ptab->ppgPrevTab, BKA_MAJOR,
//                      BKA_PREV);
      }

#if 0      /*-------------------------< JDR >--------------------------------*/
      /*------------------------------------------*/
      /* if the new minor's major is the top one, */
      /* add it to the list of visible minor tabs */
      /*------------------------------------------*/
      else if (pbk->ppgTopMinor == GetTabPtr (ppgInsert, BKA_MAJOR, BKA_PREV))
#endif     /*-------------------------< JDR >--------------------------------*/
      else
      {
        ResetVisPagePtrs (pbk, ppgInsert, ppgInsert->fsAttr, 0);        /*JEH1*/
      }

      ResetTabData(pbk, pbk->ppgCursorMajor,
                   QueryStartTabLevel(pbk, BKA_MAJOR));
    }

    else if (cMaxTabs > pbk->cMaxVisMajors)
    {
      CalcInvalidateRect (pbk, pbk->ppgFirstVisMajor,
                          pbk->ppgLastVisMajor, &rcInvalid);

      if (pbk->ppgCursorMajor)
        ResetVisPagePtrs (pbk, pbk->ppgCursorMajor, BKA_MAJOR, 0);      /*JEH1*/
      else
        ResetVisPagePtrs (pbk, ppgInsert, ppgInsert->fsAttr, 0);        /*JEH1*/

      ResetTabData (pbk, pbk->ppgFirstVisMajor,
                    QueryStartTabLevel (pbk, BKA_MAJOR));
    }

    /*----------------------------------------------------------------------*/
    /* Determine the invalidate rectangle for the new tabs to be displayed. */
    /*----------------------------------------------------------------------*/
    CalcInvalidateRect(pbk, pbk->ppgFirstVisMajor,
                       pbk->ppgLastVisMajor, &rcInvalid1);

    /*-------------------------------------------------------*/
    /* Determine union of the current and new tab rectangles */
    /*-------------------------------------------------------*/
    if (UnionRect (&rcUpdate, &rcInvalid, &rcInvalid1))
    {
      InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
//    DrawUpdateRect(pbk, pbk->ppgFirstVisMajor,
//                   pbk->ppgLastVisMajor, &rcUpdate, NULL);
    }

    /*--------------------------------------------------------*/
    /* if this is a major inserted before a minor (determined */
    /* above), reset the vis page pointers for the minor edge */
    /*--------------------------------------------------------*/
    if (fResetMinors)
    {
      ResetVisPagePtrs (pbk, pbk->ppgFirstVisMinor, BKA_MINOR, 0);      /*JEH1*/
    }
  }
}


/*---------------------------------------------------------------------------*
 * ResetCursorPagePtr
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID ResetCursorPagePtr(
  PBOOK  pbk,                               /* book instance data (locked)   */
  UINT   usOrder,                                                       /*JEH1*/
  USHORT usTabType)
{
  PPAGE  ppg, ppgCursor, ppgFirstVis, ppgLastVis;
  RECT   rcUpdate, rcInvalid1, rcInvalid2;
  USHORT cTabs;
  SHORT  sMaxVisTabs = (usTabType & BKA_MINOR) ?                        /*JEH1*/
                        pbk->cMaxVisMinors : pbk->cMaxVisMajors;        /*JEH1*/
  cTabs = CountTabs (pbk, usTabType, sMaxVisTabs);                      /*JEH1*/

  /*----------------------------------------------------*/
  /* If there is no currently outlined tab page, return */
  /*----------------------------------------------------*/
  if (pbk->ppgCursor)
  {
    /*-----------------------------------------------------*/
    /* Get tab page that is previous to currently outlined */
    /*-----------------------------------------------------*/
    ppg = (usOrder & BKA_PREV) ?
            GetTabPtr (pbk->ppgCursor->ptab->ppgPrevTab, usTabType, usOrder) :
            GetTabPtr (pbk->ppgCursor->ptab->ppgNextTab, usTabType, usOrder);

    /*------------------------------------------------------*/
    /* If a new record was found, toggle the emphasis to it */
    /*------------------------------------------------------*/
    if (ppg && (ppg != pbk->ppgCursor))
    {
      /*------------------------------------*/
      /* Reset global cursor record pointer */
      /*------------------------------------*/
      ppgCursor = pbk->ppgCursor;
      pbk->ppgCursor = ppg;

      if (ppg->fsAttr & BKA_MAJOR)
        pbk->ppgCursorMajor = ppg;
      else
        pbk->ppgCursorMinor = ppg;

      CalcInvalidateRect (pbk, ppgCursor, ppgCursor, &rcInvalid1);

      if (usTabType & BKA_MINOR)
      {
        ppgFirstVis = pbk->ppgFirstVisMinor;
        ppgLastVis  = pbk->ppgLastVisMinor;
      }
      else
      {
        ppgFirstVis = pbk->ppgFirstVisMajor;
        ppgLastVis  = pbk->ppgLastVisMajor;
      }

      if (IsTabVisible (pbk, ppg))
      {
        CalcInvalidateRect (pbk, pbk->ppgCursor, pbk->ppgCursor, &rcInvalid2);
      }

      else
      {
        ResetVisPagePtrs (pbk, NULL, usTabType, usOrder);

        if (!IsTabVisible (pbk, ppg))
          ResetVisPagePtrs (pbk, ppg, usTabType, 0);                    /*JEH1*/

        ResetTabData (pbk, pbk->ppgCursor, QueryStartTabLevel(pbk, usTabType));
        CalcInvalidateRect (pbk, ppgFirstVis, ppgLastVis, &rcInvalid2);
      }

      /*--------------------------------------------------------*/
      /* Remove cursor from the current cursored record and set */
      /* it on new cursored record                              */
      /*--------------------------------------------------------*/
      if (UnionRect (&rcUpdate, &rcInvalid1, &rcInvalid2))
      {
//      if (usTabType & BKA_MAJOR)
//        DrawUpdateRect (pbk, pbk->ppgFirstVisMajor, pbk->ppgLastVisMajor,
//                        &rcUpdate, NULL);
//      else
//        DrawUpdateRect (pbk, pbk->ppgFirstVisMinor, pbk->ppgLastVisMinor,
//                        &rcUpdate, NULL);

        InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
      }
    }
  }

  UpdateScrollStates (pbk, BKA_MAJOR);
  UpdateScrollStates (pbk, BKA_MINOR);
}


/*---------------------------------------------------------------------------*
 * ResetGlobalPagePtrs
 *
 * Resets any global page variables that may need to be updated due to a
 * page deletion.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID ResetGlobalPagePtrs(
  PBOOK pbk,                                /* book instance data (locked)   */
  PPAGE ppgDelete)
{
  PPAGE ppgTopMajor, ppgOldTop, ppgNewTop;
  RECT  rcInvalid;
  PAGESELECTNOTIFY pageSelectNotify;                                    /*DAB2*/

  /*---------------------------------------------------------*/
  /* If the page pointer is NULL, initialize all global page */
  /*---------------------------------------------------------*/
  if (!ppgDelete)
  {
    pbk->ppgTopMajor    = NULL;
    pbk->ppgTopMinor    = NULL;
    pbk->ppgTop         = NULL;
    pbk->ppgFirst       = NULL;
    pbk->ppgLast        = NULL;
    pbk->ppgFirstTab    = NULL;
    pbk->ppgLastTab     = NULL;
    pbk->ppgCursor      = NULL;
    pbk->ppgCursorMajor = NULL;
    pbk->ppgCursorMinor = NULL;

    CalcInvalidateRect(pbk, pbk->ppgFirstVisMinor,
                       pbk->ppgLastVisMinor, &rcInvalid);

    /*-------------------------------------------------------*/
    /* Determine union of the current and new tab rectangles */
    /*-------------------------------------------------------*/
    pbk->ppgFirstVisMinor=NULL;
    DrawUpdateRect(pbk, pbk->ppgFirstVisMinor,
                   pbk->ppgLastVisMinor, &rcInvalid, NULL);

    CalcInvalidateRect(pbk, pbk->ppgFirstVisMajor,
                       pbk->ppgLastVisMajor, &rcInvalid);

    /*-------------------------------------------------------*/
    /* Determine union of the current and new tab rectangles */
    /*-------------------------------------------------------*/
    pbk->ppgFirstVisMajor=NULL;
    DrawUpdateRect(pbk, pbk->ppgFirstVisMajor,
                   pbk->ppgLastVisMajor, &rcInvalid, NULL);

    return;
  }

  ppgOldTop = QueryTopPage (pbk);

  /*---------------------------------------------------------------*/
  /* If the page is a top major pointer reset to the next major if */
  /* it exists. Otherwise set it to the previous major page.       */
  /*---------------------------------------------------------------*/
  if (ppgDelete == pbk->ppgTopMajor)
  {
    ppgTopMajor      = pbk->ppgTopMajor;
    pbk->ppgTopMajor = GetTabPtr (ppgDelete->ptab->ppgNextTab, BKA_MAJOR,
                                  BKA_NEXT);

    if (!pbk->ppgTopMajor)
      pbk->ppgTopMajor = GetTabPtr (ppgDelete->ptab->ppgPrevTab, BKA_MAJOR,
                                    BKA_PREV);
  }

  /*---------------------------------------------------------------*/
  /* If the page is a top minor pointer reset to the next major if */
  /* it exists. Otherwise set it to the previous minor page.       */
  /*---------------------------------------------------------------*/
  if (ppgDelete == pbk->ppgTopMinor)
  {
    if ( ppgDelete->ptab->ppgNextTab &&
        (ppgDelete->ptab->ppgNextTab->fsAttr & BKA_MINOR))
      pbk->ppgTopMinor = ppgDelete->ptab->ppgNextTab;

    else if ( ppgDelete->ptab->ppgPrevTab &&
             (ppgDelete->ptab->ppgPrevTab->fsAttr & BKA_MINOR))
      pbk->ppgTopMinor = ppgDelete->ptab->ppgPrevTab;

    else  pbk->ppgTopMinor = NULL;
  }

  /*--------------------------------------------------------------*/
  /* If the page is a top minor pointer reset to the next page if */
  /* it exists. Otherwise set it to the previous page.            */
  /*--------------------------------------------------------------*/
  if (ppgDelete == pbk->ppgTop)
  {
    if (  ppgDelete->ppgNext &&
        !(ppgDelete->ppgNext->fsAttr & (BKA_MAJOR | BKA_MINOR)))
      pbk->ppgTop = ppgDelete->ppgNext;

    else if (  ppgDelete->ppgPrev &&
             !(ppgDelete->ppgPrev->fsAttr & (BKA_MAJOR | BKA_MINOR)))
      pbk->ppgTop = ppgDelete->ppgPrev;

    else  pbk->ppgTop = NULL;
  }

  /*-----------------------------------------------------*/
  /* If the delete page is the first page, reset to next */
  /*-----------------------------------------------------*/
  if (ppgDelete == pbk->ppgFirst)
    pbk->ppgFirst = ppgDelete->ppgNext;

  /*--------------------------------------------------------*/
  /* If the delete page is the last page, reset to previous */
  /*--------------------------------------------------------*/
  if (ppgDelete == pbk->ppgLast)
    pbk->ppgLast = ppgDelete->ppgPrev;

  /*-------------------------------------------------------------*/
  /* If the delete page is the first tab page, reset to next tab */
  /*-------------------------------------------------------------*/
  if (ppgDelete == pbk->ppgFirstTab)
    pbk->ppgFirstTab = ppgDelete->ptab->ppgNextTab;

  /*----------------------------------------------------------------*/
  /* If the delete page is the last tab page, reset to previous tab */
  /*----------------------------------------------------------------*/
  if (ppgDelete == pbk->ppgLastTab)
    pbk->ppgLastTab = ppgDelete->ptab->ppgPrevTab;

  /*------------------------------------------------------------------*/
  /* If the page being deleted is the cursored page, reset the global */
  /* cursor page pointer.                                             */
  /*------------------------------------------------------------------*/
  if (ppgDelete == pbk->ppgCursor)
  {
    /*--------------------------------------------------*/
    /* move the cursor to the next tab of the same type */
    /*--------------------------------------------------*/
    ResetCursorPagePtr (pbk, BKA_NEXT, ppgDelete->fsAttr);

    /*------------------------------------------------------*/
    /* if there isn't a succeeding tab of the same type,    */
    /* move the cursor to the previous tab of the same type */
    /*------------------------------------------------------*/
    if (ppgDelete == pbk->ppgCursor)
      ResetCursorPagePtr (pbk, BKA_PREV, ppgDelete->fsAttr);

    /*----------------------------------------------------------*/
    /* if the cursor moved from major to major, set the minor   */
    /* cursor to the first minor after the newly cursored major */
    /*----------------------------------------------------------*/
    if (ppgDelete != pbk->ppgCursor)
    {
      if (pbk->ppgCursor->fsAttr & BKA_MAJOR)
      {
        pbk->ppgCursorMinor = ((pbk->ppgCursor->ptab->ppgNextTab) &&    /*DAB3*/
            (pbk->ppgCursor->ptab->ppgNextTab->fsAttr & BKA_MINOR)) ?   /*DAB3*/
            pbk->ppgCursor->ptab->ppgNextTab : NULL;
      }
    }

    /*-----------------------------------------*/
    /* otherwise, we're deleting the last page */
    /* in the book (or section) of this type   */
    /*-----------------------------------------*/
    else
    {
      if (ppgDelete->fsAttr & BKA_MAJOR)
      {
#if 0      /*-------------------------< JDR >--------------------------------*/
        pbk->ppgCursor =
                 (pbk->ppgCursorMinor)         ? pbk->ppgCursorMinor :
                 (ppgDelete->ptab->ppgNextTab) ? ppgDelete->ptab->ppgNextTab :
                                                 ppgDelete->ptab->ppgPrevTab;
#endif     /*-------------------------< JDR >--------------------------------*/

        pbk->ppgCursor      = pbk->ppgFirstTab;
        pbk->ppgCursorMajor = NULL;
        pbk->ppgCursorMinor = pbk->ppgCursor;
      }
      else
      {
        pbk->ppgCursor =
                 (pbk->ppgCursorMajor)         ? pbk->ppgCursorMajor :
                 (ppgDelete->ptab->ppgNextTab) ? ppgDelete->ptab->ppgNextTab :
                                                 ppgDelete->ptab->ppgPrevTab;
        pbk->ppgCursorMajor = pbk->ppgCursor;
        pbk->ppgCursorMinor = NULL;
      }
    }
  }

  /*-----------------------------------------------------------------*/
  /* if there is no top page, reset it to the first page in the book */
  /*-----------------------------------------------------------------*/
  if (!QueryTopPage (pbk))
  {
    if ((pbk->ppgFirst) && (pbk->ppgFirst->fsAttr & BKA_MAJOR))         /*DAB3*/
      pbk->ppgTopMajor = pbk->ppgFirst;
    else if ((pbk->ppgFirst) && (pbk->ppgFirst->fsAttr & BKA_MINOR))    /*DAB3*/
      pbk->ppgTopMinor = pbk->ppgFirst;
    else
      pbk->ppgTop      = pbk->ppgFirst;
  }

  /*----------------------------------------------------------------------*/
  /* if the top page changes, notify the owner of the newly selected page */
  /*----------------------------------------------------------------------*/
  if (ppgOldTop != (ppgNewTop = QueryTopPage (pbk)))
  {
    HWND hwndNewTop;

    if (ppgNewTop)  hwndNewTop = ppgNewTop->hwndPage;

    pageSelectNotify.hwndBook = pbk->hwndBook;                          /*DAB2*/
    pageSelectNotify.ulPageIdCur = (DWORD)ppgOldTop;                    /*DAB2*/
    pageSelectNotify.ulPageIdNew = (DWORD)ppgNewTop;                    /*DAB2*/

    vppgPrevSelected = ppgOldTop;
    NotifyParent( pbk->hwndBook, BKN_PAGESELECTED, TRUE,                /*DAB2*/
                (LPARAM)&pageSelectNotify );                            /*DAB2*/
    vppgPrevSelected = NULL;

    /*-----------------------*/
    /* hide the old top page */
    /*-----------------------*/
    if (ppgOldTop && IsWindow (ppgOldTop->hwndPage))
      BookShowWindow (ppgOldTop->hwndPage, FALSE);

    /*-------------------------------------------------------------------*/
    /* Show the new top page.  Check to make sure the app has not sent a */
    /* BKM_SETPAGEWINDOWHWND when it received the BKN_PAGESELECTED.  If  */
    /* so, SetPageWindowHwnd will handle showing the top page, so we     */
    /* don't have to worry about it here.                                */
    /*-------------------------------------------------------------------*/
    if (ppgNewTop && ppgNewTop->hwndPage == hwndNewTop)
      ShowTopPage (pbk, NULL);
  }
}


/*---------------------------------------------------------------------------*
 * ResetPagePtrs
 *
 * Resets any page variables that may need to be updated due to a page
 * deletion.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID ResetPagePtrs (
  PBOOK pbk,                                /* book instance data (locked)   */
  PPAGE ppgDelete)
{
  PPAGE ppgVisMajor = NULL, ppgVisMinor = NULL;
  RECT  rcInv1, rcInv2, rcInv3, rcInv4, rcUpdate;
  BOOL  fResetMinorTabs       = FALSE;
  BOOL  fInvalidateEverything = FALSE;
  USHORT cMaxTabs;                                                      /*JEH1*/
  WORD  wSide;

  /*-----------------------------------------*/
  /* Initialize rectangles used for painting */
  /*-----------------------------------------*/
  SetRectEmpty (&rcInv1);
  SetRectEmpty (&rcInv2);
  SetRectEmpty (&rcInv3);
  SetRectEmpty (&rcInv4);
  SetRectEmpty (&rcUpdate);

  if ((ppgDelete->fsAttr & (BKA_MAJOR | BKA_MINOR)) &&
       (IsTabVisible (pbk, ppgDelete) || (ppgDelete == pbk->ppgTopMajor)))
  {
    if ((ppgDelete->fsAttr & BKA_MINOR) || (ppgDelete == pbk->ppgTopMajor) ||
        ( ppgDelete->ptab->ppgNextTab &&
         (ppgDelete->ptab->ppgNextTab->fsAttr & BKA_MINOR)))
    {
      /*------------------------------------*/
      /* Determine the invalidate rectangle */
      /* for the new tabs to be displayed.  */
      /*------------------------------------*/
      CalcInvalidateRect(pbk, pbk->ppgFirstVisMinor,
                         pbk->ppgLastVisMinor, &rcInv3);

      if (ppgDelete->fsAttr & BKA_MINOR)
      {
        cMaxTabs    = CountTabs (pbk, BKA_MINOR, pbk->cMaxVisMinors);
        ppgVisMinor = pbk->ppgFirstVisMinor;
        pbk->ppgFirstVisMinor =
          GetTabPtr (pbk->ppgFirstVisMinor->ptab->ppgPrevTab,
                     BKA_MINOR, BKA_PREV);

        if (!pbk->ppgFirstVisMinor)
        {
          pbk->ppgFirstVisMinor = ppgVisMinor;
          ppgVisMinor = pbk->ppgLastVisMinor;
          pbk->ppgLastVisMinor =
            GetTabPtr (pbk->ppgLastVisMinor->ptab->ppgNextTab,
                       BKA_MINOR, BKA_NEXT);

          if (!pbk->ppgLastVisMinor)
          {
            cMaxTabs = CountTabs (pbk, BKA_MINOR, pbk->cMaxVisMinors);

            if (cMaxTabs > pbk->cMaxVisMinors)
            {
              pbk->ppgLastVisMinor = NULL;
              pbk->ppgFirstVisMinor = NULL;
            }
            else
            {
              pbk->ppgLastVisMinor = ppgVisMinor;

              if (ppgDelete == pbk->ppgFirstVisMinor)
                pbk->ppgFirstVisMinor =
                  GetTabPtr(pbk->ppgFirstVisMinor->ptab->ppgNextTab,
                            BKA_MINOR, BKA_NEXT);

              if (ppgDelete == pbk->ppgLastVisMinor)
                pbk->ppgLastVisMinor =
                  GetTabPtr(pbk->ppgLastVisMinor->ptab->ppgPrevTab,
                            BKA_MINOR, BKA_PREV);
            }
          }

          else
            if (ppgDelete == pbk->ppgFirstVisMinor)
              pbk->ppgFirstVisMinor =
                GetTabPtr(pbk->ppgFirstVisMinor->ptab->ppgNextTab,
                          BKA_MINOR, BKA_NEXT);
        }
        else
        {
          if (ppgDelete == pbk->ppgLastVisMinor)
            pbk->ppgLastVisMinor =
              GetTabPtr(pbk->ppgLastVisMinor->ptab->ppgPrevTab,
                        BKA_MINOR, BKA_PREV);
        }
      }
      else
        fResetMinorTabs = TRUE;
    }

    if ((ppgDelete->fsAttr & BKA_MAJOR) && (IsTabVisible (pbk, ppgDelete)))
    {
      /*------------------------------------*/
      /* Determine the invalidate rectangle */
      /* for the new tabs to be displayed.  */
      /*------------------------------------*/
      CalcInvalidateRect(pbk, pbk->ppgFirstVisMajor,
                         pbk->ppgLastVisMajor, &rcInv1);
      ppgVisMajor = pbk->ppgFirstVisMajor;
      pbk->ppgFirstVisMajor =
        GetTabPtr(pbk->ppgFirstVisMajor->ptab->ppgPrevTab,
                  BKA_MAJOR, BKA_PREV);

      if (!pbk->ppgFirstVisMajor)
      {
        pbk->ppgFirstVisMajor = ppgVisMajor;
        ppgVisMajor = pbk->ppgLastVisMajor;
        pbk->ppgLastVisMajor =
          GetTabPtr(pbk->ppgLastVisMajor->ptab->ppgNextTab,
                    BKA_MAJOR, BKA_NEXT);

        if (!pbk->ppgLastVisMajor)
        {
          cMaxTabs = CountTabs (pbk, BKA_MAJOR, pbk->cMaxVisMajors);

          if (cMaxTabs > pbk->cMaxVisMajors)
          {
            pbk->ppgLastVisMajor  = NULL;
            pbk->ppgFirstVisMajor = NULL;
          }
          else
          {
            pbk->ppgLastVisMajor = ppgVisMajor;

            if (ppgDelete == pbk->ppgFirstVisMajor)
              pbk->ppgFirstVisMajor =
                GetTabPtr(pbk->ppgFirstVisMajor->ptab->ppgNextTab,
                          BKA_MAJOR, BKA_NEXT);

            if (ppgDelete == pbk->ppgLastVisMajor)
              pbk->ppgLastVisMajor =
                GetTabPtr(pbk->ppgLastVisMajor->ptab->ppgPrevTab,
                          BKA_MAJOR, BKA_PREV);
          }
        }

        else if (ppgDelete == pbk->ppgFirstVisMajor)
        {
          pbk->ppgFirstVisMajor =
            GetTabPtr(pbk->ppgFirstVisMajor->ptab->ppgNextTab,
                      BKA_MAJOR, BKA_NEXT);
        }
      }

      else if (ppgDelete == pbk->ppgLastVisMajor)
      {
        pbk->ppgLastVisMajor =
          GetTabPtr(pbk->ppgLastVisMajor->ptab->ppgPrevTab,
                    BKA_MAJOR, BKA_PREV);
      }
    }
  }

  /*-----------------------------------------------------------------*/
  /* Otherwise, we might have to redraw all of the tabs.  We'll have */
  /* to do this if:                                                  */
  /*                                                                 */
  /*   1.  This is not a major or minor tab, and                     */
  /*   2.  This is the last page in this major or minor section.     */
  /*                                                                 */
  /* If these conditions are met, we'll be making the first page in  */
  /* the book the top page later in ResetGlobalPagePtrs, so we'll    */
  /* need to redraw all of the tabs.                                 */
  /*-----------------------------------------------------------------*/
  else if ( (ppgDelete == pbk->ppgTop)                              &&
           !(ppgDelete->fsAttr & (BKA_MAJOR | BKA_MINOR))           &&
           (!ppgDelete->ppgNext ||
            (ppgDelete->ppgNext->fsAttr & (BKA_MAJOR | BKA_MINOR))) &&
           (!ppgDelete->ppgPrev ||
            (ppgDelete->ppgPrev->fsAttr & (BKA_MAJOR | BKA_MINOR))))
  {
    fInvalidateEverything = TRUE;
  }

  /*--------------------------------------*/
  /* remove this page from the page chain */
  /*--------------------------------------*/
  if (ppgDelete->ppgPrev)  ppgDelete->ppgPrev->ppgNext = ppgDelete->ppgNext;
  if (ppgDelete->ppgNext)  ppgDelete->ppgNext->ppgPrev = ppgDelete->ppgPrev;
  if (ppgDelete->ptab)
  {
    if (ppgDelete->ptab->ppgPrevTab)
      ppgDelete->ptab->ppgPrevTab->ptab->ppgNextTab =
                           ppgDelete->ptab->ppgNextTab;

    if (ppgDelete->ptab->ppgNextTab)
      ppgDelete->ptab->ppgNextTab->ptab->ppgPrevTab =
                           ppgDelete->ptab->ppgPrevTab;
  }

  ResetGlobalPagePtrs (pbk, ppgDelete);

  /*-------------------------------------------------------*/
  /* Determine union of the current and new tab rectangles */
  /*-------------------------------------------------------*/
  if (ppgDelete->fsAttr & (BKA_MAJOR | BKA_MINOR))
  {
    /*-------------------------------------------*/
    /* if this is a minor tab, redraw the minors */
    /*-------------------------------------------*/
    if ((ppgDelete->fsAttr & BKA_MINOR) || fResetMinorTabs)
    {
      if (fResetMinorTabs)
      {
        if (pbk->ppgTopMajor)
        {
          if ( pbk->ppgTopMajor->ptab->ppgNextTab &&
              (pbk->ppgTopMajor->ptab->ppgNextTab->fsAttr & BKA_MINOR))
            ResetVisPagePtrs (pbk, pbk->ppgTopMajor->ptab->ppgNextTab,
                              BKA_MINOR, 0);                            /*JEH1*/
          else
            ResetVisPagePtrs (pbk, NULL, BKA_MINOR, 0);                 /*JEH1*/
        }
        else
        {
          ResetVisPagePtrs (pbk, pbk->ppgFirstTab, BKA_MINOR, 0);       /*JEH1*/
        }
      }

      cMaxTabs = CountTabs (pbk, BKA_MINOR, pbk->cMaxVisMinors);

      if (cMaxTabs <= pbk->cMaxVisMinors)
      {
        wSide = DetermineSide (pbk, pbk->ppgFirstVisMinor, 0);          /*JEH1*/

        if (((wSide == BKV_TOP) || (wSide == BKV_BOTTOM)) &&
            IsWindowVisible (GetDlgItem (pbk->hwndBook, ID_HORIZONTALLEFT)))
        {
          HideScrollArrows (pbk, ID_HORIZONTALLEFT, ID_HORIZONTALRIGHT);
          ResetVisPagePtrs (pbk, pbk->ppgFirstVisMinor, BKA_MINOR, 0);  /*JEH1*/
        }

        else if (((wSide == BKV_RIGHT) || (wSide == BKV_LEFT)) &&
                  IsWindowVisible (GetDlgItem (pbk->hwndBook, ID_VERTICALTOP)))
        {
          HideScrollArrows (pbk, ID_VERTICALTOP, ID_VERTICALBOTTOM);
          ResetVisPagePtrs (pbk, pbk->ppgFirstVisMinor, BKA_MINOR, 0);  /*JEH1*/
        }
      }

      ResetTabData (pbk, pbk->ppgFirstVisMinor,
                    QueryStartTabLevel (pbk, BKA_MINOR));
      CalcInvalidateRect (pbk, pbk->ppgFirstVisMinor,
                          pbk->ppgLastVisMinor, &rcInv4);
      if (UnionRect (&rcUpdate, &rcInv3, &rcInv4))
      {
        InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
//      DrawUpdateRect (pbk, pbk->ppgFirstVisMinor, pbk->ppgLastVisMinor,
//                      &rcUpdate, NULL);
      }
    }

    /*-----------------------------------------------------*/
    /* if this is a major tab, or we deleted the last      */
    /* minor tab for this major section, redraw the majors */
    /*-----------------------------------------------------*/
    if ( (ppgDelete->fsAttr & BKA_MAJOR) ||
        ((ppgDelete->fsAttr & BKA_MINOR) && !pbk->ppgFirstVisMinor))

    {
      cMaxTabs = CountTabs (pbk, BKA_MAJOR, pbk->cMaxVisMajors);

      if (cMaxTabs <= pbk->cMaxVisMajors)
      {
        wSide = DetermineSide (pbk, ppgDelete, 0);                      /*JEH1*/

        if (((wSide == BKV_TOP) || (wSide == BKV_BOTTOM)) &&
            IsWindowVisible (GetDlgItem (pbk->hwndBook, ID_HORIZONTALLEFT)))
        {
          HideScrollArrows (pbk, ID_HORIZONTALLEFT, ID_HORIZONTALRIGHT);
          ResetVisPagePtrs (pbk, pbk->ppgFirstVisMajor, BKA_MAJOR, 0);  /*JEH1*/
        }

        else if (((wSide == BKV_RIGHT) || (wSide == BKV_LEFT)) &&
                  IsWindowVisible (GetDlgItem (pbk->hwndBook, ID_VERTICALTOP)))
        {
          HideScrollArrows (pbk, ID_VERTICALTOP, ID_VERTICALBOTTOM);
          ResetVisPagePtrs (pbk, pbk->ppgFirstVisMajor, BKA_MAJOR, 0);  /*JEH1*/
        }
      }

      ResetTabData (pbk, ppgDelete,
                    QueryStartTabLevel(pbk, ppgDelete->fsAttr));
      CalcInvalidateRect (pbk, pbk->ppgFirstVisMajor,
                          pbk->ppgLastVisMajor, &rcInv2);

      if (UnionRect (&rcUpdate, &rcInv1, &rcInv2))
      {
        InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
//      DrawUpdateRect (pbk, pbk->ppgFirstVisMajor, pbk->ppgLastVisMajor,
//                      &rcUpdate, NULL);
      }
    }
  }

  /*---------------------------------------------------------*/
  /* if we need to invalidate everything, go ahead and do it */
  /*---------------------------------------------------------*/
  else if (fInvalidateEverything)
  {
    PPAGE ppgFirstMajor, ppgFirstMinor;

    CalcInvalidateRect (pbk, pbk->ppgFirstVisMajor,
                        pbk->ppgLastVisMajor, &rcInv1);
    CalcInvalidateRect (pbk, pbk->ppgFirstVisMinor,
                        pbk->ppgLastVisMinor, &rcInv3);

    ppgFirstMajor = GetTabPtr (pbk->ppgFirstTab, BKA_MAJOR, BKA_NEXT);
    if (pbk->ppgFirstTab)
    {
      ppgFirstMinor = (pbk->ppgFirstTab->fsAttr & BKA_MINOR) ?
                          pbk->ppgFirstTab :
                          GetTabPtr( pbk->ppgFirstTab->ptab->ppgNextTab,
                                     BKA_MINOR, BKA_NEXT );
    }
    else
      ppgFirstMinor = NULL;

    ResetVisPagePtrs (pbk, ppgFirstMajor, BKA_MAJOR, 0);                /*JEH1*/
    ResetVisPagePtrs (pbk, ppgFirstMinor, BKA_MINOR, 0);                /*JEH1*/
    ResetTopPagePtrs (pbk, pbk->ppgFirst, 0);                           /*JEH1*/

    CalcInvalidateRect (pbk, pbk->ppgFirstVisMajor,
                        pbk->ppgLastVisMajor, &rcInv2);
    CalcInvalidateRect (pbk, pbk->ppgFirstVisMinor,
                        pbk->ppgLastVisMinor, &rcInv4);

    UnionRect (&rcUpdate, &rcInv1, &rcInv2);
    InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);

    UnionRect (&rcUpdate, &rcInv3, &rcInv4);
    InvalidateRect (pbk->hwndBook, &rcUpdate, FALSE);
  }
}


/*---------------------------------------------------------------------------*
 * ResetTopPagePtrs
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

void ResetTopPagePtrs(
  PBOOK  pbk,                               /* book instance data (locked)   */
  PPAGE  ppgNewTop,
  USHORT usOrder)
{
  PPAGE ppgCurrentTop, ppgT;

  /*------------------------------------------------------------------*/
  /* if a new top page isn't specified, we must be moving relative to */
  /* the current top page so figure out what the current top page is  */
  /*------------------------------------------------------------------*/
  if (!ppgNewTop)
  {
    if      (pbk->ppgTop)       ppgCurrentTop = pbk->ppgTop;
    else if (pbk->ppgTopMinor)  ppgCurrentTop = pbk->ppgTopMinor;
    else if (pbk->ppgTopMajor)  ppgCurrentTop = pbk->ppgTopMajor;
    else                        ppgCurrentTop = NULL;

    if (ppgCurrentTop)
    {
      ppgNewTop = (usOrder & BKA_PREV) ? ppgCurrentTop->ppgPrev :
                                         ppgCurrentTop->ppgNext;
    }
  }

  /*----------------------------------------------------*/
  /* If a next/prev page is found reset global pointers */
  /* to make this the current top page                  */
  /*----------------------------------------------------*/
  if (ppgNewTop)
  {
    /*-------------------------*/
    /* new top page is a major */
    /*-------------------------*/
    if (ppgNewTop->fsAttr & BKA_MAJOR)
    {
      pbk->ppgTopMajor = ppgNewTop;
      pbk->ppgTopMinor = NULL;
      pbk->ppgTop      = NULL;
    }

    /*-------------------------*/
    /* new top page is a minor */
    /*-------------------------*/
    else if (ppgNewTop->fsAttr & BKA_MINOR)
    {
      /*----------------------------------------*/
      /* if this minor is the first page in the */
      /* book, we know it doesn't have a major  */
      /*----------------------------------------*/
      if (ppgNewTop == pbk->ppgFirst)
        pbk->ppgTopMajor = NULL;

      /*-----------------------------------------------------------------*/
      /* otherwise we need to find the associated major; we can optimize */
      /* in the case where we did a simple page forward, because in that */
      /* case we know the associated major tab cannot have changed       */
      /*-----------------------------------------------------------------*/
      else if (usOrder != BKA_NEXT)
        pbk->ppgTopMajor = GetTabPtr (ppgNewTop, BKA_MAJOR, BKA_PREV);

      pbk->ppgTopMinor = ppgNewTop;
      pbk->ppgTop      = NULL;
    }

    /*-------------------------------*/
    /* new top page is a simple page */
    /*-------------------------------*/
    else
    {
      pbk->ppgTop = ppgNewTop;

      /*------------------------------------------------*/
      /* if this page is the first in the book, we know */
      /* it can't have any associated majors or minors  */
      /*------------------------------------------------*/
      if (ppgNewTop == pbk->ppgFirst)
      {
        pbk->ppgTopMinor = NULL;
        pbk->ppgTopMajor = NULL;
      }

      /*-----------------------------------------------------------------*/
      /* otherwise we need to find the associated tabs; we can optimize  */
      /* in the case where we did a simple page forward, because in that */
      /* case we know the associated tabs cannot have changed            */
      /*-----------------------------------------------------------------*/
      else if (usOrder != BKA_NEXT)
      {
        if (ppgT = GetTabPtr (ppgNewTop, BKA_MAJOR | BKA_MINOR, BKA_PREV))
        {
          if (ppgT->fsAttr & BKA_MAJOR)
          {
            pbk->ppgTopMajor = ppgT;
            pbk->ppgTopMinor = NULL;
          }
          else
          {
            pbk->ppgTopMajor = GetTabPtr (ppgT, BKA_MAJOR, BKA_PREV);
            pbk->ppgTopMinor = ppgT;
          }
        }
        else
        {
          pbk->ppgTopMinor = NULL;
          pbk->ppgTopMajor = NULL;
        }
      }
    }
  }
}


/*---------------------------------------------------------------------------*
 * ResetVisCursorPage
 *
 * Makes sure the cursored page is visible.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID ResetVisCursorPage(
  PBOOK  pbk,                               /* book instance data (locked)   */
  USHORT usOrder,
  PPAGE  ppg)
{
  SHORT cMaxTabs;
  USHORT cTabs;                                                         /*JEH1*/
  SHORT i;
  PPAGE ppgT;

  /*-------------------------------*/
  /* Currently outlined is a major */
  /*-------------------------------*/
  if (ppg && (ppg->fsAttr & BKA_MAJOR))
  {
    /*----------------------------------------------*/
    /* calculate the number of tabs to be displayed */
    /*----------------------------------------------*/
    cTabs    = CountTabs (pbk, ppg->fsAttr, cMaxTabs);
    cMaxTabs = (cTabs > pbk->cMaxVisMajors) ? pbk->cMaxVisMajorsScroll : cTabs;

    if (!(usOrder & BKA_PREV))
    {
      ppgT = ppg;
      for (i = 0; (ppgT && (i < cMaxTabs)); i++)
      {
        pbk->ppgFirstVisMajor = ppgT;
        ppgT = GetTabPtr (ppgT->ptab->ppgPrevTab, BKA_MAJOR, BKA_PREV);
      }
    }
    else
    {
      pbk->ppgFirstVisMajor = ppg;
    }
  }

  /*-------------------------------*/
  /* Currently outlined is a minor */
  /*-------------------------------*/
  if (ppg && (ppg->fsAttr & BKA_MINOR))
  {
    if (!(usOrder & BKA_PREV))
    {
      /*----------------------------------------------*/
      /* calculate the number of tabs to be displayed */
      /*----------------------------------------------*/
      cTabs    = CountTabs (pbk, ppg->fsAttr, cMaxTabs);
      cMaxTabs = (cTabs > pbk->cMaxVisMinors) ? pbk->cMaxVisMinorsScroll : cTabs;

      ppgT = ppg;
      for (i = 0; (ppgT && (i < cMaxTabs)); i++)
      {
        pbk->ppgFirstVisMinor = ppgT;
        ppgT = GetTabPtr (ppgT->ptab->ppgPrevTab, BKA_MINOR, BKA_PREV);
      }
    }
    else
    {
      pbk->ppgFirstVisMinor = ppg;
    }
  }
}

/*---------------------------------------------------------------------------*
 * ShowTopPage
 *
 * Sets the window of the current top tab to the top.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 * 28-Jul-92 JeffR    Fixed local heap corruption.
 *---------------------------------------------------------------------------*/

void ShowTopPage(
  PBOOK pbk,                                /* book instance data (locked)   */
  HDC   hdc)
{
  BOOL   fReleaseDC;
  PPAGE  ppgTop;

  /*---------------------------------------*/
  /* Get a presentation space to draw into */
  /*---------------------------------------*/
  if ((fReleaseDC = (hdc == NULL)) && !(hdc = GetDC (pbk->hwndBook)))  return;

  /*----------------------------------*/
  /* unsubclass the previous top page */
  /*----------------------------------*/
  SubclassTopPage (pbk, NULL, FALSE);

  /*--------------------*/
  /* Get top pagerecord */
  /*--------------------*/
  if (ppgTop = QueryTopPage (pbk))
  {
    /*------------------------------------------------*/
    /* Update the page windows so they paint together */
    /*------------------------------------------------*/
//  UpdateWindow (pbk->hwndBookPage);
    BookShowWindow (pbk->hwndBookPage, TRUE);

    /*------------------------------------------*/
    /* Display the top page window if it exists */
    /*------------------------------------------*/
    if (ppgTop->hwndPage)
    {
      if (ppgTop->fsAttr & BKA_AUTOPAGESIZE)
      {
        SetWindowPos (ppgTop->hwndPage, NULL, 0, 0,
                      pbk->rcAppPageWnd.right - pbk->rcAppPageWnd.left,
                      pbk->rcAppPageWnd.bottom - pbk->rcAppPageWnd.top,
                      0);                                               /*JEH1*/
      }

      BookShowWindow (ppgTop->hwndPage, TRUE);
//    UpdateWindow (ppgTop->hwndPage);
      SubclassTopPage (pbk, ppgTop, TRUE);
    }

    /*----------------------------------------------------*/
    /* otherwise make sure the book's page is invalidated */
    /* so it'll erase anything that was there before      */
    /*----------------------------------------------------*/
    else
    {
      InvalidateRect (pbk->hwndBookPage, NULL, TRUE);
    }

    /*-----------------------------------------------*/
    /* Draw the status text for the current top page */
    /*-----------------------------------------------*/
    DrawStatusLineText (pbk, ppgTop, hdc, TRUE);
  }
  else
  {
    BookShowWindow (pbk->hwndBookPage, FALSE);
  }

  /*--------------------------------------------------*/
  /* If a presentation space was obtained, release it */
  /*--------------------------------------------------*/
  if (fReleaseDC)  ReleaseDC (pbk->hwndBook, hdc);
}


/*---------------------------------------------------------------------------*
 * SubclassTopPage
 *
 * Subclasses or unsubclasses the top page window so Alt+PageUp, Alt+PageDn
 * and Alt+Up work properly without any work on the app's part.
 *
 * History:
 * 28-Jul-92 JeffR    Created to solve local heap corruption problem.
 *---------------------------------------------------------------------------*/

VOID SubclassTopPage (
  PBOOK pbk,
  PPAGE ppgTop,
  BOOL  fSubclass)
{
  /*-------------------*/
  /* we're subclassing */
  /*-------------------*/
  if (fSubclass)
  {
    pbk->hwndTopPage        = ppgTop->hwndPage;
    pbk->lpfnTopPageWndProc = (FARPROC) SetWindowLong (ppgTop->hwndPage,
                                   GWL_WNDPROC, (LONG) TopPageSubclassProc);

    /*-----------------------------------------------------------------*/
    /* stick the notebook's instance data address in a window property */
    /* so the subclass proc can get to it when it needs it             */
    /*-----------------------------------------------------------------*/
    SetProp (ppgTop->hwndPage, ICL_NOTEBOOK, (HANDLE) pbk);
  }

  /*---------------------*/
  /* we're unsubclassing */
  /*---------------------*/
  else
  {
    if (pbk->hwndTopPage)
    {
      SetWindowLong (pbk->hwndTopPage, GWL_WNDPROC,
                     (LONG) pbk->lpfnTopPageWndProc);
      RemoveProp (pbk->hwndTopPage, ICL_NOTEBOOK);
      pbk->hwndTopPage        = NULL;
      pbk->lpfnTopPageWndProc = NULL;
    }
  }
}


/*---------------------------------------------------------------------------*
 * UpdatePageButtonStates
 *
 * Updates the page turning flags for disabling visuals.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID UpdatePageButtonStates(
  PBOOK pbk)                                /* book instance data (locked)   */
{
  PPAGE ppgTop = QueryTopPage(pbk);

  /*---------------------------------------------------------------------*/
  /* If the topmost page is the first page disable backward page turning */
  /*---------------------------------------------------------------------*/
  BookEnableWindow (GetDlgItem (pbk->hwndBook, ID_PAGEBUTTONLEFT),
                    (ppgTop != pbk->ppgFirst));

  /*-------------------------------------------------------------------*/
  /* if the topmost page is the last page disable forward page turning */
  /*-------------------------------------------------------------------*/
  BookEnableWindow (GetDlgItem (pbk->hwndBook, ID_PAGEBUTTONRIGHT),
                    (ppgTop != pbk->ppgLast));
}


/*---------------------------------------------------------------------------*
 * UpdateScrollStates
 *
 * Updates the scroll arrow flags for disabling visuals.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

VOID UpdateScrollStates(
  PBOOK pbk,                                /* book instance data (locked)   */
  USHORT usTabType)
{
  PPAGE  ppgFirstVisTab,  ppgLastVisTab;
  HWND   hwndBackwardTab, hwndForwardTab;

  if (usTabType & BKA_MAJOR)
  {
    ppgFirstVisTab = pbk->ppgFirstVisMajor;
    ppgLastVisTab  = pbk->ppgLastVisMajor;
  }
  else
  {
    ppgFirstVisTab = pbk->ppgFirstVisMinor;
    ppgLastVisTab  = pbk->ppgLastVisMinor;
  }

  switch (DetermineSide (pbk, NULL, usTabType))
  {
    case BKV_LEFT:
    case BKV_RIGHT:
      hwndBackwardTab = GetDlgItem (pbk->hwndBook, ID_VERTICALTOP);
      hwndForwardTab  = GetDlgItem (pbk->hwndBook, ID_VERTICALBOTTOM);
      break;

    case BKV_TOP:
    case BKV_BOTTOM:
      hwndBackwardTab = GetDlgItem (pbk->hwndBook, ID_HORIZONTALLEFT);
      hwndForwardTab  = GetDlgItem (pbk->hwndBook, ID_HORIZONTALRIGHT);
      break;
  }

  if (ppgFirstVisTab)
  {
    /*---------------------------------------------------------------*/
    /* if there are tabs preceding the first, enable backward scroll */
    /*---------------------------------------------------------------*/
    BookEnableWindow (hwndBackwardTab,
                      (GetTabPtr (ppgFirstVisTab->ptab->ppgPrevTab,
                                  usTabType, BKA_PREV)) ? TRUE : FALSE);

    /*-------------------------------------------------------------*/
    /* if there are tabs following the last, enable forward scroll */
    /*-------------------------------------------------------------*/
    BookEnableWindow (hwndForwardTab,
                      (GetTabPtr (ppgLastVisTab->ptab->ppgNextTab,
                                  usTabType, BKA_NEXT)) ? TRUE : FALSE);
  }

  /*------------------------------------------------------------------------*/
  /* otherwise, no tabs are visible on this edge, so disable scroll buttons */
  /*------------------------------------------------------------------------*/
  else
  {
    BookEnableWindow (hwndForwardTab,  FALSE);
    BookEnableWindow (hwndBackwardTab, FALSE);
  }
}


/*---------------------------------------------------------------------------*
 * VerifyPageId
 *
 * Verifies that a given page is inserted in the notebook.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

BOOL VerifyPageId(
  PBOOK pbk,                                /* book instance data (locked)   */
  PPAGE ppgVerify)                          /* page id to verify             */
{
  PPAGE ppgT;

  /*----------------------------------------------------------------------*/
  /* Traverse page record list to determine if passed record is included. */
  /*----------------------------------------------------------------------*/
  for (ppgT=pbk->ppgFirst; ppgT && (ppgT != ppgVerify); ppgT=ppgT->ppgNext);

  /*-----------------------------------------------*/
  /* If the pageid is in the list, return success. */
  /*-----------------------------------------------*/
  return ((ppgT) ? TRUE : FALSE);
}


/*---------------------------------------------------------------------------*
 * NotifyParent
 *
 * Sends a WM_COMMAND to a window's parent with a given notification code.
 *
 * History:
 * 14-Nov-91 JeffR    Created.
 *---------------------------------------------------------------------------*/
VOID NotifyParent ( HWND hwnd, WORD wNotifyCode, BOOL bExtra, DWORD lp2)/*DAB2*/
{
  /********************************************************************/
  /* Check if an extra data parameter is to be passed on the          */
  /* notification message to the application                          */
  /********************************************************************/
  if (!bExtra)                                                          /*DAB2*/
  {                                                                     /*DAB2*/
    /******************************************************************/
    /* If not, then pass the window handle in the LPARAM value        */
    /******************************************************************/
    SendMessage( PARENTOF(hwnd), WM_COMMAND,
                 (WPARAM)MAKELONG( IDOF(hwnd), wNotifyCode ),           /*DAB1*/
                 (LPARAM)hwnd );                                        /*DAB1*/
  }                                                                     /*DAB2*/
  else                                                                  /*DAB2*/
  {                                                                     /*DAB2*/
    /******************************************************************/
    /* otherwise, pass the pointer to the notification block in LPARAM*/
    /******************************************************************/
    SendMessage( PARENTOF(hwnd), WM_COMMAND,                            /*DAB2*/
                 (WPARAM)MAKELONG( IDOF(hwnd), wNotifyCode ),           /*DAB2*/
                 (LPARAM)lp2 );                                         /*DAB2*/
  }                                                                     /*DAB2*/
  return;
}


/*---------------------------------------------------------------------------*
 * BookAllocMem
 *
 * This function allocates a memory block from notebook's heap.  The
 * allocated memory is zero-initilized, and the rest of the code in this
 * DLL depends on that fact.
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

NPVOID BookAllocMem (
  UINT cb)                               /* size of block to allocate *//*JEH1*/
{
  return ((NPVOID) LocalAlloc (LPTR, cb));
}


/*---------------------------------------------------------------------------*
 * BookFreeMem
 *
 * Frees memory allocated by BookAllocMem
 *
 * History:
 * 11-Sep-91 JeffR    Initial port from PM.
 *---------------------------------------------------------------------------*/

NPVOID BookFreeMem (
  NPVOID hmemFree)                          /* block to free                 */
{
  return ((NPVOID) LocalFree ((HMEM) hmemFree));
}
