// Revision: 96 1.30.1.3 source/ui/basectl/ivport.cpp, canvas, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ivport.cpp                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ivport.hpp.                                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

extern "C" {
  #define INCL_WININPUT                // WinQueryFocus for iwindefs.h
  #include <iwindefs.h>
}

#include <ivport.hpp>
#include <icconst.h>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <inotifev.hpp>
#include <irecohdr.hpp>
#include <iscrlevt.hpp>
#include <iscroll.hpp>
#include <itrace.hpp>
#include <ivporhdr.hpp>
#include <ivrect.hpp>
#include <iwposbuf.hpp>
#include <ibidiset.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IVPORT_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Public view port styles.                                                     |
------------------------------------------------------------------------------*/
const IViewPort::Style
  IViewPort::asNeededHorizontalScrollBar ( 0, ICNV_ASNEEDEDHORZSB ),
  IViewPort::alwaysHorizontalScrollBar   ( 0, ICNV_ALWAYSHORZSB ),
  IViewPort::noHorizontalScrollBar       ( 0, ICNV_NOHORZSB ),
  IViewPort::asNeededVerticalScrollBar   ( 0, ICNV_ASNEEDEDVERTSB ),
  IViewPort::alwaysVerticalScrollBar     ( 0, ICNV_ALWAYSVERTSB ),
  IViewPort::noVerticalScrollBar         ( 0, ICNV_NOVERTSB ),
  IViewPort::noViewWindowFill            ( 0, ICNV_NOVIEWWINFILL ),
  IViewPort::expandableViewWindow        ( 0, ICNV_EXPANDVIEWWIN ),
  IViewPort::classDefaultStyle           ( WS_VISIBLE, ICNV_ASNEEDEDHORZSB |
                                                       ICNV_ASNEEDEDVERTSB );


/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
IViewPort::Style
           IViewPort::currentDefaultStyle ( WS_VISIBLE, ICNV_ASNEEDEDHORZSB |
                                                        ICNV_ASNEEDEDVERTSB );


/*------------------------------------------------------------------------------
| Class:   IViewPortData                                                       |
|                                                                              |
| Store private data used by the class.                                        |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IViewPortData
{
public:
  IViewPortData ( );
 ~IViewPortData ( );
IViewPortResizeHandler
  fResizeHandler;
IBidiSettings::BidiLayout
  bidiLayout;
IViewPortScrollHandler
  fScrollHandler;
IViewPortKeyboardHandler
  fKeyboardHandler;
#ifdef IC_PM
IViewPortPaintHandler
  fPaintHandler;
IViewRectanglePaintHandler
  fRectPaintHandler;
#endif
unsigned long
  rightJustifyIndex;
unsigned long
  insideLayout;
IWindowHandle
  hwndClViewWindow;
private:
  IViewPortData ( const IViewPortData& );
IViewPortData
 &operator=     ( const IViewPortData& );
}; // IViewPortData

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IViewPortData::IViewPortData                                                 |
------------------------------------------------------------------------------*/
IViewPortData::IViewPortData ( )
  : rightJustifyIndex( 0 ),
    fResizeHandler ( ),
    bidiLayout( IBidiSettings::layoutLeftToRight ),
    fScrollHandler ( ),
    fKeyboardHandler ( ),
#ifdef IC_PM
    fPaintHandler ( ),
    fRectPaintHandler( ),
#endif
    insideLayout( 0 ),
    hwndClViewWindow( 0 )
{ }

/*------------------------------------------------------------------------------
| IViewPortData::~IViewPortData                                                |
------------------------------------------------------------------------------*/
IViewPortData::~IViewPortData ( )
{ }


/*------------------------------------------------------------------------------
| IViewRectangle::IViewRectangle                                               |
|                                                                              |
| Constructor here for page tuning.                                            |
------------------------------------------------------------------------------*/
IViewRectangle::IViewRectangle ( unsigned long windowId,
                                 IWindow* parent,
                                 IWindow* owner,
                                 const IRectangle& initial,
                                 const Inherited::Style& style )
  : IViewRectangle::Inherited( )
{
   this->initialize( windowId,
                     parent,
                     owner,
                     initial,
                     ICanvas::convertToGUIStyle( style ),
                     ICanvas::convertToGUIStyle( style, true ) );
}

/*------------------------------------------------------------------------------
| IViewRectangle::~IViewRectangle                                              |
|                                                                              |
| Destructor here for page tuning.                                             |
------------------------------------------------------------------------------*/
IViewRectangle::~IViewRectangle ( )
{ }

/*------------------------------------------------------------------------------
| IViewRectangle::setLayoutDistorted                                           |
------------------------------------------------------------------------------*/
IViewRectangle& IViewRectangle::setLayoutDistorted (
                                            unsigned long layoutAttributeOn,
                                            unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = (layoutAttributeOn &
                               (unsigned long)
                                  (~(IWindow::childMinimumSizeChanged |
                                     IWindow::childWindowCreated |
                                     IWindow::childWindowDestroyed)));

  // Notify view port if view window's minimum size is changing.
  if ( layoutAttributeOn & IWindow::childMinimumSizeChanged )
  {                               // Change to child window size.
     ulFlagsOn |= IWindow::minimumSizeChanged;
  }                               // Reset flag to pass up to view port.

  // Notify view port if new or destroyed view window.
  if ( layoutAttributeOn & IWindow::childWindowCreated )
  {                               // New view window.
     ulFlagsOn |= IWindow::windowCreated;
  }                               // Reset flag to pass up to view port.
  if ( layoutAttributeOn & IWindow::childWindowDestroyed )
  {                               // Removed view window.
     ulFlagsOn |= IWindow::windowDestroyed;
  }                               // Reset flag to pass up to view port.

  Inherited::setLayoutDistorted( ulFlagsOn, layoutAttributeOff );

  return *this;
}


#define MAX_SCROLL_LIMIT          0x8000


/*------------------------------------------------------------------------------
| IViewPort::IViewPort                                                         |
|                                                                              |
| Constructor for IViewPort.                                                   |
------------------------------------------------------------------------------*/
IViewPort::IViewPort ( unsigned long windowIdentifier,
                       IWindow* parent,
                       IWindow* owner,
                       const IRectangle& initialSize,
                       const Style& style )
  : IViewPort::Inherited( )
  , ptClTopLeftView( 0, 0 )
  , psbClVert( 0 )
  , psbClHorz( 0 )
  , pcvClViewRect( 0 )
  , sizClViewWindow( -1, -1 )
  , ulClFlag( 0 )
  , fViewPortData( new IViewPortData )
{
  IMODTRACE_DEVELOP( "IViewPort::ctor" );

  // Check for valid style combination.
  if ( ((style & asNeededHorizontalScrollBar)  &&
        (style & alwaysHorizontalScrollBar))  ||
       ((style & asNeededHorizontalScrollBar)  &&
        (style & noHorizontalScrollBar))  ||
       ((style & alwaysHorizontalScrollBar)  &&
        (style & noHorizontalScrollBar))  ||
       ((style & asNeededVerticalScrollBar)  &&
        (style & alwaysVerticalScrollBar))  ||
       ((style & asNeededVerticalScrollBar)  &&
        (style & noVerticalScrollBar))  ||
       ((style & alwaysVerticalScrollBar)  &&
        (style & noVerticalScrollBar)) )
  {                               // Mutually exclusive styles.
     ITHROWLIBRARYERROR( IC_INVALIDSTYLE,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable );
  } /* endif */

  Style createStyle = style;
#ifdef IC_MOTIFWIN
  if ( style & noViewWindowFill )
  {           // noViewWindowFill is equivalent to clipChildren.
     createStyle |= IWindow::clipChildren;
  }
#endif

  // Save the extended style to make sure we have a copy of it stored.
  setExtendedStyle( extendedStyle() | createStyle.asExtendedUnsignedLong() );

  // Create the presentation system window.
  this->initialize( windowIdentifier,
                    parent,
                    owner,
                    initialSize,
                    this->convertToGUIStyle( createStyle ),
                    this->convertToGUIStyle( createStyle, true ) );
  /*****************************************************************/
  /* Create the "standard" child windows, omitting the visible     */
  /* style and size/position for now.                              */
  /*****************************************************************/
  psbClHorz =
    new IScrollBar( IC_VIEWPORT_HORZSCROLLBAR,
                    this, this,
                    IRectangle( 0, 0,
                                0, IScrollBar::systemScrollBarWidth(false) ),
                    IScrollBar::horizontal | IWindow::group );
  psbClVert =
    new IScrollBar( IC_VIEWPORT_VERTSCROLLBAR,
                    this, this,
                    IRectangle( 0, 0,
                                IScrollBar::systemScrollBarWidth( true ), 0 ),
                    IScrollBar::vertical | IWindow::group );
#ifdef IC_MOTIF
  XtVaSetValues( (Widget) psbClHorz->handle(), XmNhighlightThickness, 0, 0);
  XtVaSetValues( (Widget) psbClVert->handle(), XmNhighlightThickness, 0, 0);
#endif // IC_MOTIF

  pcvClViewRect =
    new IViewRectangle( IC_VIEWPORT_VIEWRECTANGLE,
                        this, this, IRectangle(),
#ifdef IC_MOTIFWIN
                        (createStyle & noViewWindowFill ) ?
                          IWindow::visible | IWindow::clipChildren :
                          IWindow::visible );
#endif
#ifdef IC_PM
                        IWindow::visible );
#endif
  pcvClViewRect->positionOnSiblings();  // For proper input focus.

  /*****************************************************************/
  /* If bidirectional support is set,                              */
  /* Create private data class and initialize BIDI setting         */
  /* and be sure that our vertical                                 */
  /* scroll bar works left-to-right.  This is needed due to the    */
  /* logic for scrolling the view window.                          */
  /*****************************************************************/
  if ( IBidiSettings::isBidiSupported() )
  /* Create private data class and initialize BIDI setting         */
  {
     fViewPortData->bidiLayout = IBidiSettings(*this).windowLayout();
     IBidiSettings settings( *psbClHorz );
     settings.setWindowLayout( IBidiSettings::layoutLeftToRight );
     settings.apply( *psbClHorz );
  }

  /*****************************************************************/
  /* Set the scroll bar increments based off of the current font   */
  /* (use half of a character).                                    */
  /*****************************************************************/
  ISize sizLineScroll = characterSize() / IPair(2);
  psbClHorz->setMinScrollIncrement( sizLineScroll.width() );
  psbClVert->setMinScrollIncrement( sizLineScroll.height() );

  /*****************************************************************/
  /* Add the private resize, scroll, and keyboard handlers.        */
  /*****************************************************************/
  fViewPortData->fResizeHandler.handleEventsFor( this );
  ulClFlag |= IViewPort::resizeHandlerAdded;

  fViewPortData->fScrollHandler.handleEventsFor( this );
  ulClFlag |= IViewPort::scrollHandlerAdded;

  fViewPortData->fKeyboardHandler.handleEventsFor( this );
  ulClFlag |= IViewPort::keyboardHandlerAdded;

#ifdef IC_PM
  if ( createStyle & noViewWindowFill )
  {                     // Add special paint handlers.
     fViewPortData->fPaintHandler.handleEventsFor( this );
     ulClFlag |= IViewPort::paintHandlerAdded;

     fViewPortData->fRectPaintHandler.handleEventsFor( pcvClViewRect );
     ulClFlag |= IViewPort::rectanglePaintHandlerAdded;
  }
#endif

  /************************************************************/
  /* The view window is to be sized to fill the view port,    */
  /* but no smaller than its minimum size.                    */
  /************************************************************/
  if ( this->extendedStyle() &
         IViewPort::expandableViewWindow.asExtendedUnsignedLong() )
  {           // Flag to size the view window based on its min size.
     ulClFlag |= IViewPort::useMinimumSizeView;
  }

  ulClFlag |= IViewPort::initialized;
}

/*------------------------------------------------------------------------------
| IViewPort::~IViewPort                                                        |
|                                                                              |
| Destructor for clean-up.                                                     |
------------------------------------------------------------------------------*/
IViewPort::~IViewPort ( )
{
  IMODTRACE_DEVELOP( "IViewPort::dtor" );
  if ( psbClVert )
  {
     delete psbClVert;
  }
  if ( psbClHorz )
  {
     delete psbClHorz;
  }
  if ( pcvClViewRect )
  {
#ifdef IC_PM
     if ( ulClFlag & IViewPort::rectanglePaintHandlerAdded )
     {                            // Remove the private paint handler.
        fViewPortData->fRectPaintHandler
         .stopHandlingEventsFor( pcvClViewRect );
        ulClFlag &= (unsigned long)(~IViewPort::rectanglePaintHandlerAdded);
     }
#endif
     delete pcvClViewRect;
  }

  if ( ulClFlag & IViewPort::resizeHandlerAdded )
  {                              // Remove the private resize handler.
     fViewPortData->fResizeHandler.stopHandlingEventsFor( this );
     ulClFlag &= (unsigned long)(~IViewPort::resizeHandlerAdded);
  }

  if ( ulClFlag & IViewPort::scrollHandlerAdded )
  {                              // Remove the private scroll handler.
     fViewPortData->fScrollHandler.stopHandlingEventsFor( this );
     ulClFlag &= (unsigned long)(~IViewPort::scrollHandlerAdded);
  }

  if ( ulClFlag & IViewPort::keyboardHandlerAdded )
  {                            // Remove the private keyboard handler.
     fViewPortData->fKeyboardHandler.stopHandlingEventsFor( this );
     ulClFlag &= (unsigned long)(~IViewPort::keyboardHandlerAdded);
  }

#ifdef IC_PM
  if ( ulClFlag & IViewPort::paintHandlerAdded )
  {                            // Remove the private paint handler.
     fViewPortData->fPaintHandler.stopHandlingEventsFor( this );
     ulClFlag &= (unsigned long)(~IViewPort::paintHandlerAdded);
  }
#endif
  if ( fViewPortData )
  {
     delete fViewPortData;
  }
}

/*------------------------------------------------------------------------------
| IViewPort::defaultStyle                                                      |
------------------------------------------------------------------------------*/
IViewPort::Style IViewPort::defaultStyle ( )
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IViewPort::convertToGUIStyle                                                 |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IViewPort::convertToGUIStyle ( const IBitFlag& guiStyle,
                                             bool            bExtOnly ) const
{
  // Obtain the style from the class (ICanvas) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if ( bExtOnly )
  {
     // Use mask to only return extended styles in the user defined range.
     ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
     // Replace canvas window style with viewport window style.
#ifdef IC_PMWIN
     ulStyle |= ( guiStyle.asUnsignedLong()
                   & (unsigned long)~IC_CVS_CANVAS )
                   | IC_CVS_VIEWPORT;
#else
     // No styles are defined for the base canvas, so we do not mask.
     ulStyle |= guiStyle.asUnsignedLong();
#endif

     // No additional processing required since ICanvas::convertToGUIStyle is
     // doing all the required work.
  }

  return ulStyle;
}

/*------------------------------------------------------------------------------
| IViewPort::viewWindow                                                        |
|                                                                              |
| Enumerate child windows to find the view window (it's the one that isn't one |
| of the scroll bars or the canvas view rectangle).  Or just return the last   |
| view window set via setViewWindow.                                           |
------------------------------------------------------------------------------*/
IWindowHandle IViewPort::viewWindow ( )
{
	bool bChangeParent = false;
  IWindowHandle hwndView = 0;

  /*****************************************************************/
  /* If child windows added via setViewWindow interface, no need   */
  /* to enumerate child windows to determine view window; just     */
  /* return the view window handle stored via setViewWindow.       */
  /*****************************************************************/
  if ( fViewPortData->hwndClViewWindow )
  {
     hwndView = fViewPortData->hwndClViewWindow;
  }
  else
  {
     /*****************************************************************/
     /* First enumerate all child windows of the view port itself.    */
     /* If this function has been called before, the view window has  */
     /* already been moved to be a child of the view area canvas.     */
     /*****************************************************************/
     IWindowHandle hwndChild = 0;
     IWindow::ChildCursor cursor( *this );
     for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
     {                               // Enumerate all child controls.
        hwndChild = this->childAt( cursor );
        if ( hwndChild != psbClHorz->handle()  &&
             hwndChild != psbClVert->handle()  &&
             hwndChild != pcvClViewRect->handle() )
        {                            // Not a standard child.
           if ( ! hwndView )
           {
              hwndView = hwndChild;
              bChangeParent = true;
           }
        }
     }

     /*****************************************************************/
     /* Now enumerate all child windows of the canvas view area.      */
     /* If this function has been called before, we've changed the    */
     /* parent of the view window to the canvas.                      */
     /*****************************************************************/
     IWindow::ChildCursor viewRectCursor( *pcvClViewRect );
     for ( viewRectCursor.setToFirst();
           viewRectCursor.isValid();
           viewRectCursor.setToNext() )
     {                               // Enumerate all child controls.
        hwndChild = pcvClViewRect->childAt( viewRectCursor );
        if ( ! hwndView )
        {
           hwndView = hwndChild;
           bChangeParent = false;
        }
     }

#ifdef IC_PMWIN
     if ( bChangeParent )
     {
        bool bNew = false;
        IWindow* wndView = IWindow::windowWithHandle( hwndView );
        if ( ! wndView )
        {
           wndView = new IWindow( hwndView );
           bNew =true;
        }

        wndView->setParent( pcvClViewRect);
        if ( pcvClViewRect->handle() == 
                IQUERYFOCUS( IWindow::desktopWindow()->handle() ) )
        {
           wndView->setFocus();
        }
        if ( bNew )
        {
           delete wndView;
        }
     }  
#endif     
  }

  return hwndView;
}

#ifdef IC_MOTIF
#define ISETFOCUS(hwnd, newstate) XmProcessTraversal(hwnd, XmTRAVERSE_CURRENT)
#define IQUERYFOCUS(hwnd) XmGetFocusWidget(hwnd)
#define IISWINDOWVISIBLE(hwnd) false
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| IViewPort::setViewWindow                                                     |
|                                                                              |
| Replace the current view window (if there is one) with the specified view    |
| window (which must be created as a child of the view port).  Hide previous   |
| view window, Show the new view window, and reset focus to new view window.   |
------------------------------------------------------------------------------*/
IViewPort& IViewPort::setViewWindow (const IWindowHandle& hwndNewView )
{
  IWindowHandle hwndView( 0 );
  /*****************************************************************/
  /* Retrieve previously set or initial view window                */
  /*****************************************************************/
  if ( hwndNewView )
  {
#ifdef IC_MOTIF
    IWindow* newWin = IWindow::windowWithHandle( hwndNewView );
#endif
    if ( fViewPortData->hwndClViewWindow )
      hwndView = fViewPortData->hwndClViewWindow;
    else
      hwndView = this->viewWindow();

    fViewPortData->hwndClViewWindow = hwndNewView;
    if ( (hwndView) && (hwndView != hwndNewView) )
    {
      /***************************************************************/
      /* Hide previous view window                                   */
      /***************************************************************/
#ifdef IC_PMWIN
      ISHOWWINDOW ( hwndView, false );
#endif
#ifdef IC_MOTIF
      IWindow* oldWin = IWindow::windowWithHandle( hwndView );
      if ( oldWin )
      {
         oldWin->show( false );
      }
      else
      {
         XtVaSetValues( hwndView,
                        XmNmappedWhenManaged, false,
                        NULL );
         XtUnmapWidget( hwndView );
      }
#endif

      if ( pcvClViewRect->handle() ==
              IQUERYFOCUS ( IWindow::desktopWindow()->handle() ))
                // View rectangle has focus.
                // Shift focus to new view window
      {
#ifdef IC_WIN
        // If the window to set the focus to is not created on the
        // current thread or the current focus window is in another
        // thread, SetForegroundWindow must be used to change focus.
        IWindowHandle fg = GetForegroundWindow();
        unsigned long winTID = GetWindowThreadProcessId( hwndNewView,
                                                       0 );
        if ( ( winTID != GetCurrentThreadId() ) ||
             ( winTID != GetWindowThreadProcessId( fg, 0 ) ) )
        {
          SetForegroundWindow ( hwndNewView );
        }
        else
        {
          ISETFOCUS ( IWindow::desktopWindow()->handle(),
                      hwndNewView );
        }
#endif // IC_WIN
#ifdef IC_PM
        ISETFOCUS ( IWindow::desktopWindow()->handle(), hwndNewView );
#endif
#ifdef IC_MOTIF
        ISETFOCUS ( hwndNewView, XmTRAVERSE_CURRENT );
#endif
      }
    }
    /***************************************************************/
    /* Always make the new view window visible                     */
    /***************************************************************/
#ifdef IC_PMWIN
    if ( !( IISWINDOWVISIBLE ( hwndNewView ) ) )
    {
      ISHOWWINDOW( hwndNewView, true );
    }
#endif
#ifdef IC_MOTIF
    if ( newWin )
    {
      newWin->show( true );
    }
    else
    {
      XtVaSetValues( hwndNewView,
                     XmNmappedWhenManaged, true,
                     NULL );
      if ( XtIsManaged( hwndNewView) )
        XtMapWidget(hwndNewView);
    }
#endif

    ptClTopLeftView.setX ( 0 );
    ptClTopLeftView.setY ( 0 );
    this->setLayoutDistorted ( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::scrollViewVerticallyTo                                            |
|                                                                              |
| Scroll the view window by moving it relative to the view area.  The          |
| specified zero-based value is used as the offset of the top-most visible row |
| from the top edge of the view window.                                        |
------------------------------------------------------------------------------*/
IViewPort& IViewPort::scrollViewVerticallyTo ( unsigned long ulScrollTo )
{
  unsigned long ulViewWindowHeight = viewWindowSize().height();
  unsigned long ulViewRectHeight = pcvClViewRect->size().height();
  if ( ulViewWindowHeight > pcvClViewRect->size().height() )
  {                         // View window bigger than view rectangle.
     /**************************************************************/
     /* Note that some function is duplicated between here and     */
     /* IViewPortHandler, since the user could call this directly. */
     /**************************************************************/
     unsigned long ulMaxScrollTo =
                      ulViewWindowHeight - ulViewRectHeight;
     unsigned long ul0BasedScrollTo =
                      (ulScrollTo > ulMaxScrollTo) ?
                                 ulMaxScrollTo : ulScrollTo;
                        // Adjust scroll request if scrolling too far.
     ptClTopLeftView.setY( ul0BasedScrollTo );
     IWindowHandle hwndView = this->viewWindow();

     if ( hwndView )
     {                  // Scroll the view window.
        this->positionViewWindow( hwndView, pcvClViewRect->rect() );
     }

     // Adjust for huge logical height.
     if ( ulViewWindowHeight >= MAX_SCROLL_LIMIT )
     {             // Scroll bar units are ratio of view port units.
        unsigned long yRatio =
                        (ulViewWindowHeight / MAX_SCROLL_LIMIT) + 1;
        unsigned long ulSave = ul0BasedScrollTo;
        ul0BasedScrollTo /= yRatio;
        if ( ulSave != 0  &&  ul0BasedScrollTo == 0 )
        {               // Adjust for rounding to zero.
           ul0BasedScrollTo++;
        }
        else if ( ulSave != (ulViewWindowHeight - ulViewRectHeight)  &&
                  ul0BasedScrollTo ==
                    (ulViewWindowHeight - ulViewRectHeight) / yRatio )
        {               // Adjust for rounding to max.
           ul0BasedScrollTo--;
        }
     }
     psbClVert->moveScrollBoxTo( ul0BasedScrollTo + 1 );
                                  // Adjust scroll bar thumb
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::scrollViewHorizontallyTo                                          |
|                                                                              |
| Scroll the view window by moving it relative to the view area.  The          |
| specified zero-based value is used as the offset of the left-most visible    |
| column from the left edge of the view window.                                |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: scrollViewHorizontallyTo ( unsigned long ulScrollTo )
{
  unsigned long ulViewWindowWidth = viewWindowSize().width();
  unsigned long ulViewRectWidth = pcvClViewRect->size().width();
  if ( ulViewWindowWidth > pcvClViewRect->size().width() )
  {                         // View window bigger than view rectangle.
     /**************************************************************/
     /* Note that some function is duplicated between here and     */
     /* IViewPortHandler, since the user could call this directly. */
     /**************************************************************/
     unsigned long ulMaxScrollTo =
                        ulViewWindowWidth - ulViewRectWidth;
     unsigned long ul0BasedScrollTo =
                      (ulScrollTo > ulMaxScrollTo) ?
                                 ulMaxScrollTo : ulScrollTo;
                        // Adjust scroll request if scrolling too far.
     ptClTopLeftView.setX( ul0BasedScrollTo );
     IWindowHandle hwndView = this->viewWindow();

     if ( hwndView )
     {                  // Scroll the view window.
        this->positionViewWindow( hwndView, pcvClViewRect->rect() );
     }

     // Adjust for huge logical width.
     if ( ulViewWindowWidth >= MAX_SCROLL_LIMIT )
     {             // Scroll bar units are ratio of view port units.
        unsigned long xRatio =
                        (ulViewWindowWidth / MAX_SCROLL_LIMIT) + 1;
        unsigned long ulSave = ul0BasedScrollTo;
        ul0BasedScrollTo /= xRatio;
        if ( ulSave != 0  &&  ul0BasedScrollTo == 0 )
        {               // Adjust for rounding to zero.
           ul0BasedScrollTo++;
        }
        else if ( ulSave != (ulViewWindowWidth - ulViewRectWidth)  &&
                  ul0BasedScrollTo ==
                    (ulViewWindowWidth - ulViewRectWidth) / xRatio )
        {               // Adjust for rounding to max.
           ul0BasedScrollTo--;
        }
     }
     psbClHorz->moveScrollBoxTo( ul0BasedScrollTo + 1 );
                        // Adjust scroll bar thumb.
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::setViewWindowSize                                                 |
|                                                                              |
| Set the logical size of the view window.                                     |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: setViewWindowSize ( const ISize& siz )
{
  if ( sizClViewWindow != siz )
  {
     if ( siz == ISize( -1, -1 ) )
     {  // Disabling logical size support.
#ifdef IC_ORIGINLOWERLEFT
        // Remove the recoordination handler.
        IRecoordHandler::defaultHandler()->stopHandlingEventsFor( pcvClViewRect );
#endif
        // We could either use the actual (physical) size of the view
        // window or its minimum size. Unfortunately it's a toss up.
        ulClFlag |= (unsigned long) IViewPort::useMinimumSizeView;
     }
     else
     {  // The caller is specifying a logical size.
        if ( sizClViewWindow == ISize( -1, -1 ) )
        {  // Enabling logical size support (this is not a change to a
           // previously-set logical size).
#ifdef IC_ORIGINLOWERLEFT
           // Add a recoordination handler to keep the top of the view
           // window at the same place relative to the clipping window
           // for the case when the application and native coordinate
           // systems differ.
           IRecoordHandler::defaultHandler()->handleEventsFor( pcvClViewRect );
#endif
           ulClFlag &= (unsigned long)(~IViewPort::useMinimumSizeView);
        }
        this->setLayoutSize( siz );
     }

     // Store the new size and force a new layout.
     sizClViewWindow = siz;
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
     ISize vWinSize = siz;
     this->notifyObservers( INotificationEvent( IViewPort::viewWindowSizeId,
                                                *this, true, &vWinSize ) );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::viewWindowSize                                                    |
|                                                                              |
| Return the logical/actual size of the view window.                           |
------------------------------------------------------------------------------*/
ISize IViewPort :: viewWindowSize ( ) const
{
  ISize siz = sizClViewWindow;
  if ( siz == ISize( -1, -1 ))
  {                     // No logical size used.
     ((IViewPort*)this)->layout();   // Cast away const.
     siz = this->layoutSize();
  }
  return siz;
}

/*------------------------------------------------------------------------------
| IViewPort::viewWindowDrawRectangle                                           |
|                                                                              |
| Return the rectangle surrounding the visible portion of the view window.     |
------------------------------------------------------------------------------*/
IRectangle IViewPort :: viewWindowDrawRectangle ( ) const
{
   if (!hasChildrenToLayout())
      return IRectangle();
   ISize
     viewRectangleSize = pcvClViewRect->size();
   if ( ICoordinateSystem::applicationOrientation() ==
                       ICoordinateSystem::kOriginLowerLeft )
      return IRectangle( this->topLeftViewPoint() -
                           IPoint( 0, viewRectangleSize.height() ),
                         viewRectangleSize );
   else
      return IRectangle( IPoint( this->topLeftViewPoint().x(),
                                 ( this->layoutSize().height() -
                                   this->topLeftViewPoint().y() ) ),
                         viewRectangleSize );
}

/*------------------------------------------------------------------------------
| IViewPort::topLeftViewPoint                                                  |
|                                                                              |
| Return the view window point displayed in the top left corner of the view    |
| rectangle.                                                                   |
------------------------------------------------------------------------------*/
IPoint IViewPort :: topLeftViewPoint ( ) const
{
   return IPoint( ptClTopLeftView.x(),
                  this->layoutSize().height() - ptClTopLeftView.y() );
}

/*------------------------------------------------------------------------------
| IViewPort::setTopLeftViewPoint                                               |
|                                                                              |
| Set the view window point displayed in the top left corner of the view       |
| rectangle.                                                                   |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: setTopLeftViewPoint ( const IPoint& topLeft )
{
   ptClTopLeftView = IPoint( topLeft.x(),
                             this->layoutSize().height() - topLeft.y() );
   return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::hasChildrenToLayout                                               |
|                                                                              |
| Identify whether the view port contains a child window to manage (the view   |
| window).                                                                     |
------------------------------------------------------------------------------*/
bool IViewPort::hasChildrenToLayout ( ) const
{
  IViewPort
   *nonConstThis = (IViewPort*)this;
  return ( nonConstThis->viewWindow() ? true : false );
}

/*------------------------------------------------------------------------------
| IViewPort::setLayoutDistorted                                                |
|                                                                              |
| Treats a size change like a layout change.                                   |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: setLayoutDistorted( unsigned long layoutAttributeOn,
                                            unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = (layoutAttributeOn &
                               (unsigned long)
                                  (~(IWindow::sizeChanged |
                                     IWindow::childMinimumSizeChanged)));
  if ( ( (layoutAttributeOn & IWindow::sizeChanged)  ||
         ( (ulClFlag & IViewPort::useMinimumSizeView)  &&
           (layoutAttributeOn & IWindow::childMinimumSizeChanged) ) ) &&
       ( fViewPortData->insideLayout == 0 ) )
  {                               // Need to update scroll bars.
     ulFlagsOn |= IWindow::layoutChanged;  // Flag for ICanvas.
     if ( !(layoutAttributeOn & IWindow::fontPropogated)  &&
          !isLayoutDistorted(IWindow::fontPropogated) )
     {                            // Change buffered til now?
        ulFlagsOn |= IWindow::immediateUpdate;
     }
  }
  Inherited::setLayoutDistorted( ulFlagsOn, layoutAttributeOff );

  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::calcMinimumSize                                                   |
|                                                                              |
| Determine the minimum size of the view port based on the minimum needed to   |
| display scroll bars.                                                         |
------------------------------------------------------------------------------*/
ISize IViewPort :: calcMinimumSize ( ) const
{
  return ( psbClHorz->minimumSize() + psbClVert->minimumSize() +
           this->topLeftLayoutOffset() +
           this->bottomRightLayoutOffset() );
}

/*------------------------------------------------------------------------------
| IViewPort::layout                                                            |
|                                                                              |
| Position the view window and size and position the scroll bars around it in  |
| the view rectangle.                                                          |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: layout ( )
{
  if ( this->isLayoutDistorted( IWindow::layoutChanged )  &&
       (ulClFlag & IViewPort::initialized) )
  {                               // Something has changed.
     // Set layout flag for BIDI systems.
     fViewPortData->insideLayout++;
     if ( fViewPortData->insideLayout == 1 )
     {
        fViewPortData->rightJustifyIndex =
                         this->viewWindowDrawRectangle().right();
     }

     IWindowHandle hwndView = this->viewWindow();
     this->isTabStop();           // Substitute for fixupChildren().

     ISize sizView( 0, 0 );       // Assume no view window.
     if ( hwndView )
     {                            // Have a view window.
        /***********************************************************/
        /* Set the layout size (used by setupScrollBars()) equal   */
        /* to the size of the view window.  This is done here      */
        /* since fixupChildren() will include all children (i.e.   */
        /* the scroll bars) in the layout size and is sensitive to */
        /* the window position of the children, which is not what  */
        /* we want here.                                           */
        /***********************************************************/
        if ( sizClViewWindow == ISize( -1, -1 ) )
        {  // Logical scrolling is not being used, so scroll the
           // view window based on its minimum or physical size.
           IWindow* pwndView = IWindow::windowWithHandle( hwndView );
           if ( pwndView )
           {                 // Have an IWindow for the view window.
              sizView = pwndView->size();

              if ( (ulClFlag & IViewPort::useMinimumSizeView)  ||
                   sizView == ISize() )  // 0-sized child window.
              {                   // Assume this isn't right.
                 sizView = pwndView->minimumSize();

                 pwndView->sizeTo(sizView);  // Use min size instead.

                 ulClFlag |= IViewPort::useMinimumSizeView;
              }     // (Needed to allow child canvas windows to grow.)
              else
              {                   // Use real window size.
                 ulClFlag &= (unsigned long)(~IViewPort::useMinimumSizeView);
              } /* endif */
           }
           else                      // A non-wrappered control.
           {                         // Need to support these also.
#ifdef IC_PM
              SWP swp;
              WinQueryWindowPos ( hwndView, &swp );
              sizView = ISize ( swp.cx, swp.cy );
#endif // IC_PM
#ifdef IC_WIN
              RECT rect;
              IQUERYWINDOWRECT ( hwndView, &rect );
              IWindowHandle parentHwnd = IPARENTOF ( hwndView );
              MapWindowPoints ( HWND_DESKTOP,
                                parentHwnd,
                                (POINT*)&rect,
                                2 );
              IRectangle theRect ( rect );
              sizView = theRect.size();
#endif // IC_WIN
#ifdef IC_MOTIF
              Position x, y;
              Dimension height, width, borderWidth;
              XtVaGetValues ( hwndView,
                              XmNheight, &height,
                              XmNwidth, &width,
                              XmNborderWidth, &borderWidth,
                              NULL );
              sizView = (ISize ( ( width + 2*borderWidth ),
                                 ( height + 2*borderWidth ) ) );
#endif // IC_MOTIF
           }
        }
        else
        {  // Logical scrolling, so scroll the view window based on
           // its logical window size.
           sizView = sizClViewWindow;
        } /* endif */
     } /* endif */

     // Check if horizontal scroll was needed before this layout change
     // (this is needed to handle BIDI requirements for right-to-left systems).
     bool
       hScrollPrevEnabled = psbClHorz->isEnabled(),
       hScrollBidiChange = false;

     // Check if the view port window layout changed
     if ( IBidiSettings::isBidiSupported() )
     {
        IBidiSettings settings( *this );
        if ( settings.windowLayout() != fViewPortData->bidiLayout )
        {
           hScrollBidiChange = true;
           fViewPortData->bidiLayout = settings.windowLayout();
        }
     }

     // Check if scroll bar BIDI settings have been changed to right-to-left
     // and change it back to left-to-right (so our scrolling works).
     if ( IBidiSettings::isBidiSupported() )
     {
        IBidiSettings settings( *psbClHorz );
        if ( settings.windowLayout() != IBidiSettings::layoutLeftToRight )
        {
           settings.setWindowLayout( IBidiSettings::layoutLeftToRight );
           settings.apply( *psbClHorz );
        }
     }

     this->setLayoutSize( sizView );
     this->setupScrollBars();     // Add scroll bars if necessary.

     if ( hwndView )
     {                            // Have a view window.
                                  // Position client window.
        this->positionViewWindow( hwndView, pcvClViewRect->rect() );
     } /* endif */

     // Check for special BIDI horizontal scroll processing.
     if ( hScrollPrevEnabled && psbClHorz->isEnabled() &&
          hScrollBidiChange && (fViewPortData->insideLayout==1))
     {
        scrollViewHorizontallyTo( sizView.width() -
                                  this->viewWindowDrawRectangle().width() -
                                  this->viewWindowDrawRectangle().left() );
     }

     // Check for bidi horizontal scroll on right-to-left system.
     if ( IBidiSettings::isBidiSupported() && psbClHorz->isEnabled() &&
          (fViewPortData->insideLayout==1) )
     {
        // Check for right to left bidi support.
        IBidiSettings settings( *this );
        if ( settings.windowLayout() == IBidiSettings::layoutRightToLeft )
        {
          if ( hScrollPrevEnabled )
          {
            // The scroll bar was already enabled, so we just need to
            // maintain the right justification (unless this was a
            // layout change due to BIDI changing in which case it has
            // already been handled above).
            if ( !hScrollBidiChange )
            {
              scrollViewHorizontallyTo( fViewPortData->rightJustifyIndex -
                                        this->viewWindowDrawRectangle().width() );
            }
          }
          else
          {
            // The scroll bar was not enabled, so we need to scroll
            // all the way to the right of the view window.
            scrollViewHorizontallyTo( sizView.width() + 1 -
                                      this->viewWindowDrawRectangle().width() );
          }
        }
     }

     fViewPortData->insideLayout--;

     this->setLayoutDistorted( 0, IWindow::layoutChanged );
  }   /* end isDistorted */

  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::setupScrollBars                                                   |
|                                                                              |
| Position, size, and show/hide scroll bars.  Additionally, set the size and   |
| position of the scroll boxes.                                                |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: setupScrollBars ( )
{
  bool
    bNeedHScroll = false,
    bNeedVScroll = false,
    expandableViewWindow = false;
  unsigned long
    ulHScrollHeight = 0,
    ulVScrollWidth = 0,
    ulExtStyle = this->extendedStyle();
  ISize
    sizViewArea( this->size() ),
    viewWindowSize( this->layoutSize() ),
    viewWindowMinSize( viewWindowSize );

  // Account for an optional border and border text.
  IPoint
    upperLeftLayoutOffset( this->topLeftLayoutOffset() ),
    lowerRightLayoutOffset( this->bottomRightLayoutOffset() );
  ISize
    totalLayoutOffset( upperLeftLayoutOffset + lowerRightLayoutOffset );
  if ( ! ( sizViewArea > totalLayoutOffset ) )
  {  // No room for the view window or scroll bars within the border.
     sizViewArea
      .setWidth( 0 )
      .setHeight( 0 );
  }
  else
  {  // Still room for the view window within the border.
     sizViewArea -= totalLayoutOffset;

     /*******************************************************************/
     /* Check for the special case where the view window is to be sized */
     /* to fill the view port, except when that would shrink the view   */
     /* window below its minimum size.  In this case, leave the view    */
     /* window at its minimum size and let it be scrolled by the view   */
     /* port.  This behavior allows a multicell canvas with expandable  */
     /* rows and/or columns to work well in a view port.                */
     /*******************************************************************/
     if ( sizClViewWindow == ISize(-1, -1)  &&
          ( ulExtStyle &
              IViewPort::expandableViewWindow.asExtendedUnsignedLong() ) )
     {                // Special case for sizing the view window.
        expandableViewWindow = true;
        viewWindowMinSize = viewWindowSize;
        viewWindowSize = sizViewArea.maximum( viewWindowMinSize );
     }

     /*****************************************************************/
     /* Check if the scroll bars are needed (the size of the view     */
     /* window is bigger than the size of the view area).             */
     /*****************************************************************/
     if ( (ulExtStyle & alwaysHorizontalScrollBar.asExtendedUnsignedLong())  ||
          ((ulExtStyle & asNeededHorizontalScrollBar.asExtendedUnsignedLong())  &&
           (sizViewArea.width() < viewWindowSize.width())) )
     {                               // Need a horizontal scroll bar.
        bNeedHScroll = true;         // Flag it.
        ulHScrollHeight = psbClHorz->size().height();
        if ( sizViewArea.height() > ulHScrollHeight )
        {  // Still room for a view window.
           sizViewArea -= ISize( 0, ulHScrollHeight );
        }
        else
        {  // No room for the view window now.
           sizViewArea.setHeight( 0 );
        }
        if ( expandableViewWindow )
        {
           viewWindowSize = sizViewArea.maximum( viewWindowMinSize );
        }
     }

     if ( (ulExtStyle & alwaysVerticalScrollBar.asExtendedUnsignedLong())  ||
          ((ulExtStyle & asNeededVerticalScrollBar.asExtendedUnsignedLong())  &&
           (sizViewArea.height() < viewWindowSize.height())) )
     {                               // Need a vertical scroll bar.
        bNeedVScroll = true;         // Flag it.
        ulVScrollWidth = psbClVert->size().width();
        if ( sizViewArea.width() > ulVScrollWidth )
        {  // Still room for a view window.
           sizViewArea -= ISize( ulVScrollWidth, 0 );
        }
        else
        {  // No room for the view window now.
           sizViewArea.setWidth( 0 );
        }
        if ( expandableViewWindow )
        {
           viewWindowSize = sizViewArea.maximum( viewWindowMinSize );
        }

        if ( bNeedHScroll == false  &&
             (ulExtStyle &
                asNeededHorizontalScrollBar.asExtendedUnsignedLong())  &&
             (sizViewArea.width() < viewWindowSize.width()) )
        {                            // Need a horiz scroll bar now.
           bNeedHScroll = true;
           ulHScrollHeight = psbClHorz->size().height();
           if ( sizViewArea.height() > ulHScrollHeight )
           {  // Still room for a view window.
              sizViewArea -= ISize( 0, ulHScrollHeight );
           }
           else
           {  // No room for the view window now.
              sizViewArea.setHeight( 0 );
           }
           if ( expandableViewWindow )
           {
              viewWindowSize = sizViewArea.maximum( viewWindowMinSize );
           }
        }
     }
  }  // End room exists for the view window within the canvas border.

  /*****************************************************************/
  /* Size the rectangle where the child controls can be drawn (the */
  /* size of the view port window, minus the size needed by any    */
  /* scroll bars).  This is the area within which the user's       */
  /* window is scrolled.                                           */
  /*****************************************************************/
  IWindowPosBuffer wposbuf( this );  // To size/pos scroll bars.

#ifdef IC_ORIGINLOWERLEFT
  IRectangle
    rectViewArea( IPoint( upperLeftLayoutOffset.x(),
                          lowerRightLayoutOffset.y() + ulHScrollHeight ),
                  sizViewArea );
#else
  IRectangle
    rectViewArea( upperLeftLayoutOffset, sizViewArea );
#endif
  wposbuf.moveSizeTo( pcvClViewRect, rectViewArea );
                                  // Position/size scroll area.
  wposbuf.show( pcvClViewRect );

  /*****************************************************************/
  /* Special case where the view window is sized to the maximum of */
  /* its minimum size or what is necessary to fill the view        */
  /* rectangle.                                                    */
  /*****************************************************************/
  if ( expandableViewWindow  &&
       viewWindowSize != this->layoutSize() )
  {
     IWindow* child = IWindow::windowWithHandle( this->viewWindow() );
     if ( child )
     {
        child->sizeTo( viewWindowSize );
     }
     this->setLayoutSize( viewWindowSize );  // New size of view window.
  }

  /*****************************************************************/
  /* Size the horizontal scroll bar and set its scroll box.        */
  /*****************************************************************/
  // Adjust for huge logical width.
  unsigned long xRatio = 1;
  unsigned long cxView = viewWindowSize.width();
  if ( cxView >= MAX_SCROLL_LIMIT  &&
       sizClViewWindow != ISize( -1, -1 ))
  {                     // Too big for scroll bar to handle.
     xRatio = (cxView / MAX_SCROLL_LIMIT) + 1; // Adjusting ratio.
     cxView /= xRatio;  // New scroll bar value.
     long pageScroll = rectViewArea.width() -
                         psbClHorz->minScrollIncrement();
     if ( pageScroll < 0 )
     {                  // Don't use a negative value.
        pageScroll = psbClHorz->minScrollIncrement();
     }
     psbClHorz->setPageScrollIncrement( pageScroll );
                        // User value must be set afterwards.
  }

  // The scroll box size is fixed.  IScrollBar tracks it and viewport
  // uses it in positioning the view window.  The scroll range is set
  // to the full width of the view window.
  // IScrollBar allows the full range of the scroll box to be the full
  // range of the scroll bar.
  psbClHorz->setScrollBar( IRange( 1, cxView ),
                           rectViewArea.width() / xRatio );

                 // Total size is range, visible window is scroll bar.
  bool bEnableScrollBar = true;
  if ( ptClTopLeftView.x() + rectViewArea.width() >
                                       viewWindowSize.width() )
  {                               // Adjust for smaller window.
     if ( rectViewArea.width() >= viewWindowSize.width() )
     {                            // All of window is visible.
        bEnableScrollBar = false; // Disable scroll bar.
        ptClTopLeftView.setX( 0 );
     }
     else
     {                            // Fill all of view rectangle.
        ptClTopLeftView.setX( viewWindowSize.width() -
                                  rectViewArea.width() );
     }
  }
  psbClHorz->enable( bEnableScrollBar );  // En/disable scroll bar.
  if ( bEnableScrollBar )
  {  // Avoid the side effect of enabling the scroll bar on Windows.
     psbClHorz->moveScrollBoxTo( ptClTopLeftView.x() / xRatio + 1 );
  }

  if ( bNeedHScroll )
  {                               // View port shows horizontal scroll bar.
#ifdef IC_ORIGINLOWERLEFT
     unsigned long
       viewPortHeight = this->size().height();
     IPoint
       scrollBarPos( upperLeftLayoutOffset.x(),
                     lowerRightLayoutOffset.y() );
     if ( viewPortHeight < totalLayoutOffset.height() + ulHScrollHeight )
     {  // Entire scroll bar won't fit.
        // Position it to be clipped by the view port.
        scrollBarPos
         .setY( viewPortHeight - upperLeftLayoutOffset.y()
                               - ulHScrollHeight );
     }
     wposbuf.moveSizeTo( psbClHorz,  // Position/size horizontal scroll bar.
                         IRectangle( scrollBarPos,
                                     ISize( rectViewArea.width(),
                                            ulHScrollHeight ) ) );
#else
     wposbuf.moveSizeTo( psbClHorz,  // Position/size horizontal scroll bar.
                         IRectangle( upperLeftLayoutOffset +
                                       IPoint( 0, rectViewArea.height() ),
                                     ISize( rectViewArea.width(),
                                            ulHScrollHeight ) ) );
#endif
     wposbuf.show( psbClHorz );
  }
  else
  {                               // Hide the horizontal scroll bar.
#ifdef IC_MOTIF
     // Position it to be clipped out of view by the view port.
     wposbuf.moveTo( psbClHorz, IPoint( 0, this->size().height() + 1 ) );
     XtUnmanageChild( (Widget)psbClHorz->handle() );
#endif
     wposbuf.hide( psbClHorz );
  }

  /*****************************************************************/
  /* Size the vertical scroll bar and set its scroll box.          */
  /*****************************************************************/
  // Adjust for huge logical height.
  unsigned long yRatio = 1;
  unsigned long cyView = viewWindowSize.height();
  if ( cyView >= MAX_SCROLL_LIMIT  &&
       sizClViewWindow != ISize( -1, -1 ))
  {                     // Too big for scroll bar to handle.
     yRatio = (cyView / MAX_SCROLL_LIMIT) + 1; // Adjusting ratio.
     cyView /= yRatio;  // New scroll bar value.
     long pageScroll = rectViewArea.height() -
                         psbClVert->minScrollIncrement();
     if ( pageScroll < 0 )
     {                  // Don't use a negative value.
        pageScroll = psbClVert->minScrollIncrement();
     }
     psbClVert->setPageScrollIncrement( pageScroll );
                        // User value must be set afterwards.
  }

  // The scroll box size is fixed.  IScrollBar tracks it and viewport
  // uses it in positioning the view window.  The scroll range is set
  // to the full height of the view window.
  // IScrollBar allows the full range of the scroll box to be the full
  // range of the scroll bar.
  psbClVert->setScrollBar( IRange( 1, cyView ),
                           rectViewArea.height() / yRatio );

                 // Total size is range, visible window is scroll bar.
  bEnableScrollBar = true;
  if ( ptClTopLeftView.y() + rectViewArea.height() >
                                       viewWindowSize.height() )
  {                               // Adjust for smaller window.
     if ( rectViewArea.height() >= viewWindowSize.height() )
     {                            // All of window is visible.
        bEnableScrollBar = false; // Disable scroll bar.
        ptClTopLeftView.setY( 0 );
     }
     else
     {                            // Fill all of view rectangle.
        ptClTopLeftView.setY( viewWindowSize.height() -
                                 rectViewArea.height() );
     }
  }
  psbClVert->enable( bEnableScrollBar );  // En/disable scroll bar.
  if ( bEnableScrollBar )
  {  // Avoid the side effect of enabling the scroll bar on Windows.
     psbClVert->moveScrollBoxTo( ptClTopLeftView.y() / yRatio + 1 );
  }

  if ( bNeedVScroll )
  {                               // View port shows vertical scroll bar.
#ifdef IC_ORIGINLOWERLEFT
     wposbuf.moveSizeTo( psbClVert,
                         IRectangle( IPoint( upperLeftLayoutOffset.x() +
                                               rectViewArea.width(),
                                             lowerRightLayoutOffset.y() +
                                               ulHScrollHeight ),
                                     ISize( ulVScrollWidth,
                                            rectViewArea.height() ) ) );
#else
     wposbuf.moveSizeTo( psbClVert,
                         IRectangle( upperLeftLayoutOffset +
                                       IPoint( rectViewArea.width(), 0 ),
                                     ISize( ulVScrollWidth,
                                            rectViewArea.height() ) ) );
#endif
     wposbuf.show( psbClVert );
  }
  else
  {                               // Hide the vertical scroll bar.
#ifdef IC_MOTIF
     // Position it to be clipped out of view by the view port.
     wposbuf.moveTo( psbClVert, IPoint( this->size().width() + 1, 0 ) );
     XtUnmanageChild( (Widget)psbClVert->handle() );
#endif
     wposbuf.hide( psbClVert );
  }
  wposbuf.apply();                // Reposition/size scroll bars.
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::positionViewWindow                                                |
|                                                                              |
| Move and position the client view window.                                    |
------------------------------------------------------------------------------*/
IViewPort& IViewPort :: positionViewWindow ( const IWindowHandle& hwndView,
                                             const IRectangle& rectView )
{
  IMODTRACE_DEVELOP( "IViewPort::positionViewWindow()" );
#ifdef IC_ORIGINLOWERLEFT
  IPoint ptBottomLeft( 0 - ptClTopLeftView.x(), 0 );
#else
  IPoint ptTopLeft( IPoint( 0, 0 ) - ptClTopLeftView );
#endif

  // Check for BIDI right-to-left requirements so that the view window
  // is positioned on the right if the horizonal scroll bar is disabled.
  if (( ptClTopLeftView.x() == 0 ) && IBidiSettings::isBidiSupported() &&
        !psbClHorz->isEnabled() )
  {
     IBidiSettings settings( *this );
     if ( settings.windowLayout() == IBidiSettings::layoutRightToLeft )
     {
#ifdef IC_ORIGINLOWERLEFT
        ptBottomLeft.setX( rectView.width() - this->layoutSize().width() );
#else
        ptTopLeft.setX( rectView.width() - this->layoutSize().width() );
#endif
     }
  }

#ifdef IC_ORIGINLOWERLEFT
  if ( ptClTopLeftView.y() == 0 )
  {                               // Top align in view area.
     ptBottomLeft.setY( rectView.height() - this->layoutSize().height() );
  }
  else
  {                               // Position according to ptClTopLeftView.
     ptBottomLeft.setY( ptClTopLeftView.y() +
                          psbClVert->visibleCount() -
                          this->layoutSize().height() );
  }
#endif

  // (Assume the application will replace this function if using a
  // logical-sized view window, since we aren't looking at the view
  // window's physical size at all.)

#ifdef IC_TRACE_ALL
  ITRACE_DEVELOP( ">> view area height = " + IString( rectView.height() ));
  ITRACE_DEVELOP( ">> visible count = "
                       + IString( psbClVert->visibleCount() ));
  ITRACE_DEVELOP( ">> scroll position = "
                       + IString( psbClVert->scrollBoxPosition() ));
  ITRACE_DEVELOP( ">> ptClTopLeftView = " + ptClTopLeftView.asString() );
#ifdef IC_ORIGINLOWERLEFT
  ITRACE_DEVELOP( ">> position at: " + ptBottomLeft.asString() );
#else
  ITRACE_DEVELOP( ">> position at: " + ptTopLeft.asString() );
#endif
#endif

  IWindowPosBuffer wposbuf( pcvClViewRect );
#ifdef IC_ORIGINLOWERLEFT
  wposbuf.moveTo( hwndView, ptBottomLeft );
#else
  wposbuf.moveTo( hwndView, ptTopLeft );
#endif

  wposbuf.apply( false );              // Reposition/size view window.
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPort::setDefaultStyle                                                   |
|                                                                              |
| Sets the defaultStyle.                                                       |
|                                                                              |
| Note: This function is inlined on all platforms except Windows due to        |
|       variables they reference not being exported.                           |
------------------------------------------------------------------------------*/
#ifdef IC_WIN
void IViewPort :: setDefaultStyle ( const IViewPort::Style& style )
{
  currentDefaultStyle = style;
}
#endif // IC_WIN

/*------------------------------------------------------------------------------
| IViewPort::handleForChildCreation                                            |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowHandle IViewPort :: handleForChildCreation ( )  const
{
  /*****************************************************************/
  /* Until IViewRectangle is created, must use IViewPort handle.   */
  /*****************************************************************/
  if (pcvClViewRect)
    return (pcvClViewRect->handle());
  return (this->handle());
}
