// Revision: 94 1.12 source/ui/basectl/ivporhdr.cpp, canvas, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ivporhdr.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ivporhdr.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

#define INCL_WINSCROLLBARS        // SB_xxx
#define INCL_WINWINDOWMGR         // WinValidateRect
#include <iwindefs.h>
#ifdef IC_MOTIF
  #include <Xm/ScrollBar.h>
#endif

#include <ivporhdr.hpp>
#include <ivport.hpp>
#include <icconst.h>
#include <icoordsy.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ikeyevt.hpp>
#include <ipainevt.hpp>
#include <iscroll.hpp>
#include <itrace.hpp>
#include <ivrect.hpp>

#define MAX_SCROLL_LIMIT  0x8000

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IVPORHDR_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::IViewPortResizeHandler                               |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IViewPortResizeHandler :: IViewPortResizeHandler ( )
  : IViewPortResizeHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::~IViewPortResizeHandler                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IViewPortResizeHandler :: ~IViewPortResizeHandler ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::windowResize                                         |
|                                                                              |
| Resizes the view port window and adjust the scroll bars if necessary.        |
------------------------------------------------------------------------------*/
bool IViewPortResizeHandler :: windowResize ( IResizeEvent& event )
{
  IMODTRACE_DEVELOP( "IViewPortResizeHandler::windowResize()" );

  if ( event.oldSize() != event.newSize() )
  {                               // Size change occurred.
     /**************************************************************/
     /* Cause the viewport to size and position scroll bars and    */
     /* view window.  IViewPort::layout will be run as a result of */
     /* ICanvas processing the soon-to-follow paint/expose event.  */
     /**************************************************************/
     IViewPort *pvport = (IViewPort*)event.window();
     pvport->setLayoutDistorted( IWindow::sizeChanged, 0 );
  }
  return true;
}

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::handleEventsFor                                      |
|                                                                              |
| Attaches the resize handler to the specified view port object.               |
------------------------------------------------------------------------------*/
IViewPortResizeHandler&
  IViewPortResizeHandler :: handleEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::handleEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::stopHandlingEventsFor                                |
|                                                                              |
| Detach the resize handler from the specified view port object.               |
------------------------------------------------------------------------------*/
IViewPortResizeHandler&
  IViewPortResizeHandler :: stopHandlingEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::stopHandlingEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::handleEventsFor                                      |
------------------------------------------------------------------------------*/
IHandler& IViewPortResizeHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortResizeHandler::stopHandlingEventsFor                                |
------------------------------------------------------------------------------*/
IHandler& IViewPortResizeHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::IViewPortScrollHandler                               |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IViewPortScrollHandler :: IViewPortScrollHandler ( )
  : IViewPortScrollHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::~IViewPortScrollHandler                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IViewPortScrollHandler :: ~IViewPortScrollHandler ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::lineUp                                               |
|                                                                              |
| Scrolls the view window upward when the up scroll button of the vertical     |
| scroll bar is clicked.                                                       |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: lineUp ( IScrollEvent& event )
{
  return this->verticalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::lineDown                                             |
|                                                                              |
| Scrolls the view window downward when the down scroll button of the vertical |
| scroll bar is clicked.                                                       |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: lineDown ( IScrollEvent& event )
{
  return this->verticalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::lineLeft                                             |
|                                                                              |
| Scrolls the view window left when the left scroll button of the horizontal   |
| scroll bar is clicked.                                                       |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: lineLeft ( IScrollEvent& event )
{
  return this->horizontalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::lineRight                                            |
|                                                                              |
| Scrolls the view window right when the right scroll button of the horizontal |
| scroll bar is clicked.                                                       |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: lineRight ( IScrollEvent& event )
{
  return this->horizontalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::pageUp                                               |
|                                                                              |
| Scrolls the view window up a page when the scroll shaft above the scroll box |
| of the vertical scroll bar is clicked.                                       |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: pageUp ( IScrollEvent& event )
{
  return this->verticalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::pageDown                                             |
|                                                                              |
| Scrolls the view window down a page when the scroll shaft below the scroll   |
| box of the vertical scroll bar is clicked.                                   |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: pageDown ( IScrollEvent& event )
{
  return this->verticalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::pageLeft                                             |
|                                                                              |
| Scrolls the view window left a page when the scroll shaft to the left of the |
| scroll box of the horizontal scroll bar is clicked.                          |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: pageLeft ( IScrollEvent& event )
{
  return this->horizontalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::pageRight                                            |
|                                                                              |
| Scrolls the view window right a page when the scroll shaft to the right of   |
| the scroll box of the horizontal scroll bar is clicked.                      |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: pageRight ( IScrollEvent& event )
{
  return this->horizontalScroll( event );
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::verticalScroll                                       |
|                                                                              |
| Scroll the view window vertically and move the scroll box of the vertical    |
| scroll bar as appropriate for the scroll message.                            |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: verticalScroll ( IScrollEvent& event )
{
  bool bProcessed = false;
  IViewPort *pvportOwner = (IViewPort*)event.window();
  IScrollBar* scrollBar = pvportOwner->verticalScrollBar();
  if ( event.scrollBarWindow() == scrollBar )
  {                               // Viewport's vertical scroll bar.
     unsigned long ulPos;         // Position for scroll box.

     // Adjust for huge logical height.
     unsigned long viewWindowHeight =
                     pvportOwner->viewWindowSize().height();
     if ( viewWindowHeight < MAX_SCROLL_LIMIT )
     {                  // View port with normal <32K range.
#ifdef IC_MOTIF
        // Because this function is now being called as a result of
        // posting a IC_UM client message, parameter2 must be checked
        // to determine if the scrollBox has already been moved by the
        // system.  If the scrollBox has already been moved by virtue
        // of the XmCR_PAGE_* event being issued because of a mouse
        // button 1 click, just retrieve the new position by calling
        // prevScrollBoxPosition()
        if ( (event.eventId() == IC_UM_CRPAGE_DECREMENT ||
              event.eventId() == IC_UM_CRPAGE_INCREMENT) &&
              event.parameter2().asUnsignedLong() == true )
        {
           ulPos = scrollBar->prevScrollBoxPosition();
        }
        else
        {
           this->moveScrollBox( event );
           ulPos = scrollBar->scrollBoxPosition();
        }
#endif // IC_MOTIF
#ifdef IC_PMWIN
        this->moveScrollBox( event );
        ulPos = scrollBar->scrollBoxPosition();
#endif // IC_PMWIN
     }
     else
     {           // View port gave scaled down values to scroll bar.
        // Get the current scroll box position in view port units
        // directly from the view port (don't trust using the real
        // scroll box position and converting it to view port units
        // because of potential rounding errors).
        if ( ICoordinateSystem::applicationOrientation() ==
                            ICoordinateSystem::kOriginLowerLeft )
        {  // Lower left.
           ulPos = viewWindowHeight -
                     pvportOwner->viewWindowDrawRectangle().maxY() + 1;
        }
        else
        {  // Upper left.
           ulPos = pvportOwner->viewWindowDrawRectangle().minY() + 1;
        }
        long lDelta = event.scrollAmount();  // In view port units.

        // Re-adjust the scroll amount for the case of a partial
        // scroll to the top or bottom of the view window.  The
        // adjustments made by IScrollEvent::scrollAmount were using
        // scroll bar units.  The below adjustments use view port
        // units.
#ifdef IC_WIN
        unsigned long type = event.parameter1().lowNumber();
#endif // IC_WIN
#ifdef IC_PM
        unsigned long type = event.parameter2().highNumber();
#endif
#ifdef IC_MOTIF
        unsigned long type = event.eventId();
#endif
        unsigned long ulNormalScroll;
#ifdef IC_PMWIN
        if ( type == SB_PAGEUP  ||  type == SB_PAGEDOWN )
#endif
#ifdef IC_MOTIF
        if ( type == IC_UM_CRPAGE_INCREMENT || type == IC_UM_CRPAGE_DECREMENT )
#endif
        {
           ulNormalScroll = scrollBar->pageScrollIncrement();
        }
#ifdef IC_PMWIN
        else if ( type == SB_LINEUP  ||  type == SB_LINEDOWN )
#endif
#ifdef IC_MOTIF
        else if ( type == IC_UM_CR_INCREMENT || type == IC_UM_CR_DECREMENT )
#endif
        {
           ulNormalScroll = scrollBar->minScrollIncrement();
        }

        bool bAtMaxEnd = false;
#ifdef IC_PMWIN
        if ( type == SB_PAGEUP  ||  type == SB_LINEUP )
#endif
#ifdef IC_MOTIF
        if ( type == IC_UM_CRPAGE_DECREMENT || type == IC_UM_CR_DECREMENT )
#endif
        {
           if ( 0 - lDelta < ulNormalScroll )
           {            // Need to adjust the adjusted value.
              lDelta = 1 - ulPos;
              if ( ulNormalScroll < 0 - lDelta )
              {
                 lDelta = 0 - ulNormalScroll;
              }
           }
        }
#ifdef IC_PMWIN
        else if ( (type == SB_PAGEDOWN  ||  type == SB_LINEDOWN)  &&
#endif
#ifdef IC_MOTIF
        else if ( (type == IC_UM_CRPAGE_INCREMENT || type == IC_UM_CR_INCREMENT) &&
#endif
                  (lDelta < ulNormalScroll) )
        {               // Need to adjust the adjusted value.
           if ( ICoordinateSystem::applicationOrientation() ==
                               ICoordinateSystem::kOriginLowerLeft )
           {  // Lower left.
              lDelta = pvportOwner->viewWindowDrawRectangle().minY();
           }
           else
           {  // Upper left.
              lDelta = viewWindowHeight -
                         pvportOwner->viewWindowDrawRectangle().maxY() + 1;
           }
           if (ulNormalScroll < lDelta)
           {
              lDelta = ulNormalScroll;
           }
           else
           {
              bAtMaxEnd = true;
           }
        }

        ulPos += lDelta;   // New scroll box position.
        unsigned long ulNewPos;
        if ( ! bAtMaxEnd )
        {
           unsigned long yRatio =
                        (viewWindowHeight / MAX_SCROLL_LIMIT) + 1;
           ulNewPos = ulPos / yRatio;  // Remap to scroll bar units.
        }
        else
        {
           ulNewPos = scrollBar->scrollBoxRange().upperBound();
        }
        scrollBar->moveScrollBoxTo( ulNewPos );
     }                  // End huge logical height.

     pvportOwner->scrollViewVerticallyTo( ulPos - 1 );
     bProcessed = true;
  }
  return bProcessed;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::horizontalScroll                                     |
|                                                                              |
| Scroll the view window horizontally and move the scroll box of the horizontal|
| scroll bar as needed for the scroll message.                                 |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: horizontalScroll ( IScrollEvent& event )
{
  bool bProcessed = false;
  IViewPort *pvportOwner = (IViewPort*)event.window();
  IScrollBar* scrollBar = pvportOwner->horizontalScrollBar();
  if ( event.scrollBarWindow() == scrollBar )
  {                               // Viewport's horizontal scroll bar.
     unsigned long ulPos;         // Position for scroll box.

     // Adjust for huge logical width.
     unsigned long viewWindowWidth =
                     pvportOwner->viewWindowSize().width();
     if ( viewWindowWidth < MAX_SCROLL_LIMIT )
     {                  // View port with normal <32K range.
#ifdef IC_MOTIF
        // Because this function is now being called as a result of
        // posting a IC_UM client message, parameter2 must be checked
        // to determine if the scrollBox has already been moved by the
        // system.  If the scrollBox has already been moved by virtue
        // of the XmCR_PAGE_* event being issued because of a mouse
        // button 1 click, just retrieve the new position by calling
        // prevScrollBoxPosition()
        if ( (event.eventId() == IC_UM_CRPAGE_DECREMENT ||
              event.eventId() == IC_UM_CRPAGE_INCREMENT) &&
              event.parameter2().asUnsignedLong() == true )
        {
           ulPos = scrollBar->prevScrollBoxPosition();
        }
        else
        {
           this->moveScrollBox( event );
           ulPos = scrollBar->scrollBoxPosition();
        }
#endif // IC_MOTIF
#ifdef IC_PMWIN
        this->moveScrollBox( event );
        ulPos = scrollBar->scrollBoxPosition();
#endif // IC_PMWIN
     }
     else
     {           // View port gave scaled down values to scroll bar.
        // Get the current scroll box position in view port units
        // directly from the view port (don't trust using the real
        // scroll box position and converting it to view port units
        // because of potential rounding errors).
        ulPos = pvportOwner->viewWindowDrawRectangle().left() + 1;
        long lDelta = event.scrollAmount();  // In view port units.

        // Re-adjust the scroll amount for the case of a partial
        // scroll to the left or right edge of the view window.  The
        // adjustments made by IScrollEvent::scrollAmount were using
        // scroll bar units.  The below adjustments use view port
        // units.
#ifdef IC_WIN
        unsigned long type = event.parameter1().lowNumber();
#endif
#ifdef IC_PM
        unsigned long type = event.parameter2().highNumber();
#endif
#ifdef IC_MOTIF
        unsigned long type = event.eventId();
#endif
        unsigned long ulNormalScroll;
#ifdef IC_PMWIN
        if ( type == SB_PAGELEFT  ||  type == SB_PAGERIGHT )
#endif
#ifdef IC_MOTIF
        if ( type == IC_UM_CRPAGE_DECREMENT || type == IC_UM_CRPAGE_INCREMENT )
#endif
        {
           ulNormalScroll = scrollBar->pageScrollIncrement();
        }
#ifdef IC_PMWIN
        else if ( type == SB_LINELEFT  ||  type == SB_LINERIGHT )
#endif
#ifdef IC_MOTIF
        else if ( type == IC_UM_CR_DECREMENT || type == IC_UM_CR_INCREMENT)
#endif
        {
           ulNormalScroll = scrollBar->minScrollIncrement();
        }

        bool bAtMaxEnd = false;
#ifdef IC_PMWIN
        if ( type == SB_PAGELEFT  ||  type == SB_LINELEFT )
#endif
#ifdef IC_MOTIF
        if ( type == IC_UM_CRPAGE_DECREMENT || type == IC_UM_CR_DECREMENT )
#endif
        {
           if ( 0 - lDelta < ulNormalScroll )
           {            // Need to adjust the adjusted value.
              lDelta = 1 - ulPos;
              if ( ulNormalScroll < 0 - lDelta )
              {
                 lDelta = 0 - ulNormalScroll;
              }
           }
        }
#ifdef IC_PMWIN
        else if ( (type == SB_PAGERIGHT  ||  type == SB_LINERIGHT)  &&
#endif
#ifdef IC_MOTIF
        else if ( (type == IC_UM_CRPAGE_INCREMENT || type == IC_UM_CR_INCREMENT ) &&
#endif
                  (lDelta < ulNormalScroll) )
        {               // Need to adjust the adjusted value.
           lDelta = viewWindowWidth -
                      pvportOwner->viewWindowDrawRectangle().right();
           if ( ulNormalScroll < lDelta )
           {
              lDelta = ulNormalScroll;
           }
           else
           {
              bAtMaxEnd = true;
           }
        }

        ulPos += lDelta;   // New scroll box position.
        unsigned long ulNewPos;
        if ( ! bAtMaxEnd )
        {
           unsigned long xRatio =
                        (viewWindowWidth / MAX_SCROLL_LIMIT) + 1;
           ulNewPos = ulPos / xRatio;  // Remap to scroll bar units.
        }
        else
        {
           ulNewPos = scrollBar->scrollBoxRange().upperBound();
        }
        scrollBar->moveScrollBoxTo( ulNewPos );
     }                  // End huge logical height.

     pvportOwner->scrollViewHorizontallyTo( ulPos - 1 );
     bProcessed = true;
  }
  return bProcessed;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::scrollBoxTrack                                       |
|                                                                              |
| View window has been scrolled by dragging the scroll box of either the       |
| vertical or horizontal scroll bar.                                           |
------------------------------------------------------------------------------*/
bool IViewPortScrollHandler :: scrollBoxTrack ( IScrollEvent& event )
{
  bool bProcessed = false;
  IViewPort *pvportOwner = (IViewPort*)event.window();

  IScrollBar* scrollBar = event.scrollBarWindow();
  if ( scrollBar == pvportOwner->horizontalScrollBar() )
  {                               // Horizontal scroll bar.
     unsigned long ul0BasedScrollTo = event.newScrollBoxPosition() - 1;

     // Adjust for huge logical width.
     unsigned long viewWindowWidth =
                       pvportOwner->viewWindowSize().width();
     if ( viewWindowWidth >= MAX_SCROLL_LIMIT )
     {           // View port gave scaled down values to scroll bar.
        unsigned long scrollBoxMax = scrollBar->scrollBoxRange().upperBound();
        unsigned long effectiveScrollBoxMax =
                    viewWindowWidth + 1 -
                      pvportOwner->viewWindowDrawRectangle().width();
        if ( ul0BasedScrollTo + 1 == scrollBoxMax )
        {               // Scrolled to end.
           ul0BasedScrollTo = effectiveScrollBoxMax - 1;
        }
        else
        {               // Extrapolate where we are.
           unsigned long
             xRatio = ( viewWindowWidth / MAX_SCROLL_LIMIT ) + 1;
           ul0BasedScrollTo *= xRatio;
                        // Remap it back to view port's units.
        }
     }                  // End huge logical width.

     pvportOwner->scrollViewHorizontallyTo( ul0BasedScrollTo );

     bProcessed = true;
  }
  else if ( scrollBar == pvportOwner->verticalScrollBar() )
  {                               // Vertical scroll bar
     unsigned long ul0BasedScrollTo = event.newScrollBoxPosition() - 1;

     // Adjust for huge logical height.
     unsigned long viewWindowHeight =
                       pvportOwner->viewWindowSize().height();
     if ( viewWindowHeight >= MAX_SCROLL_LIMIT )
     {           // View port gave scaled down values to scroll bar.
        unsigned long scrollBoxMax = scrollBar->scrollBoxRange().upperBound();
        unsigned long effectiveScrollBoxMax =
                   viewWindowHeight + 1 -
                     pvportOwner->viewWindowDrawRectangle().height();
        if ( ul0BasedScrollTo + 1 == scrollBoxMax )
        {               // Scrolled to end.
           ul0BasedScrollTo = effectiveScrollBoxMax - 1;
        }
        else
        {               // Extrapolate where we are.
           unsigned long
             yRatio = ( viewWindowHeight / MAX_SCROLL_LIMIT ) + 1;
           ul0BasedScrollTo *= yRatio;
                        // Remap it back to view port's units.
        }
     }                  // End huge logical height.

     pvportOwner->scrollViewVerticallyTo( ul0BasedScrollTo );

     bProcessed = true;
  }                               // Else user's scroll bar.

  return bProcessed;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::handleEventsFor                                      |
|                                                                              |
| Attach the scroll handler to the specified view port object.                 |
------------------------------------------------------------------------------*/
IViewPortScrollHandler&
  IViewPortScrollHandler :: handleEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::handleEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::stopHandlingEventsFor                                |
|                                                                              |
| Detach the scroll handler from the specified view port object.               |
------------------------------------------------------------------------------*/
IViewPortScrollHandler&
  IViewPortScrollHandler :: stopHandlingEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::stopHandlingEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::handleEventsFor                                      |
------------------------------------------------------------------------------*/
IHandler& IViewPortScrollHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortScrollHandler::stopHandlingEventsFor                                |
------------------------------------------------------------------------------*/
IHandler& IViewPortScrollHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}


/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::IViewPortKeyboardHandler                           |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IViewPortKeyboardHandler :: IViewPortKeyboardHandler ( )
  : IViewPortKeyboardHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::~IViewPortKeyboardHandler                          |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IViewPortKeyboardHandler :: ~IViewPortKeyboardHandler ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::virtualKeyPress                                    |
|                                                                              |
| Process keys used to scroll the client window.                               |
------------------------------------------------------------------------------*/
bool IViewPortKeyboardHandler :: virtualKeyPress ( IKeyboardEvent& event )
{
  bool bProcessed = false;

  IKeyboardEvent::VirtualKey vkey = event.virtualKey();

  // Page scrolling.
  if (vkey == IKeyboardEvent::pageUp)
  {                               // PageUp/PgUp key.
     if ( event.isCtrlDown() )
     {                            // Ctrl+PageUp: scroll left.
        IViewPort *pvport = (IViewPort*)event.window();
#ifdef IC_PMWIN
        pvport->sendEvent( WM_HSCROLL,
#ifdef  IC_WIN
                           MPFROM2SHORT( SB_PAGELEFT, 0 ),
                           IEventData(
                              pvport->horizontalScrollBar()->handle()) );
#endif
#ifdef IC_PM
                           pvport->horizontalScrollBar()->id(),
                           MPFROM2SHORT( 0, SB_PAGELEFT ));
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
        IEvent newEvent((IWindow*)pvport->horizontalScrollBar(),
                        IC_UM_CRPAGE_DECREMENT,
                        event.parameter1(),
                        event.parameter2());
        newEvent.setControlHandle(pvport->horizontalScrollBar()->handle());
        pvport->horizontalScrollBar()->sendEvent(newEvent);
#endif
        bProcessed = true;
     }
     else if ( event.isShiftDown() == false  &&  event.isAltDown() == false )
     {                            // PageUp alone: scroll up.
        IViewPort *pvport = (IViewPort*)event.window();
#ifdef IC_PMWIN
        pvport->sendEvent( WM_VSCROLL,
#ifdef  IC_WIN
                           MPFROM2SHORT(SB_PAGEUP, 0 ) ,
                           IEventData(
                              pvport->verticalScrollBar()->handle()) );
#endif
#ifdef IC_PM
                           pvport->verticalScrollBar()->id(),
                           MPFROM2SHORT( 0, SB_PAGEUP ));
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
        IEvent newEvent((IWindow*)pvport->verticalScrollBar(),
                        IC_UM_CRPAGE_DECREMENT,
                        event.parameter1(),
                        event.parameter2());
        newEvent.setControlHandle(pvport->verticalScrollBar()->handle());
        pvport->verticalScrollBar()->sendEvent(newEvent);
#endif
        bProcessed = true;
     }
  }                               // End pageUp key.
  else if ( vkey == IKeyboardEvent::pageDown )
  {                               // PageDown/PgDn key.
     if ( event.isCtrlDown() )
     {                            // Ctrl+PageDown: scroll right.
        IViewPort *pvport = (IViewPort*)event.window();
#ifdef IC_PMWIN
        pvport->sendEvent( WM_HSCROLL,
#ifdef  IC_WIN
                           MPFROM2SHORT( SB_PAGERIGHT, 0 ),
                           IEventData(
                              pvport->horizontalScrollBar()->handle()) );
#endif
#ifdef IC_PM
                           pvport->horizontalScrollBar()->id(),
                           MPFROM2SHORT( 0, SB_PAGERIGHT ));
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
        IEvent newEvent((IWindow*)pvport->horizontalScrollBar(),
                        IC_UM_CRPAGE_INCREMENT,
                        event.parameter1(),
                        event.parameter2());
        newEvent.setControlHandle(pvport->horizontalScrollBar()->handle());
        pvport->horizontalScrollBar()->sendEvent(newEvent);
#endif
        bProcessed = true;
     }
     else if ( event.isShiftDown() == false  &&  event.isAltDown() == false )
     {                            // PageDown alone: scroll down.
        IViewPort *pvport = (IViewPort*)event.window();
#ifdef IC_PMWIN
        pvport->sendEvent( WM_VSCROLL,
#ifdef  IC_WIN
                           MPFROM2SHORT( SB_PAGEDOWN, 0 ),
                           IEventData(
                              pvport->verticalScrollBar()->handle()) );
#endif
#ifdef IC_PM
                           pvport->verticalScrollBar()->id(),
                           MPFROM2SHORT( 0, SB_PAGEDOWN ));
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
        IEvent newEvent((IWindow*)pvport->verticalScrollBar(),
                        IC_UM_CRPAGE_INCREMENT,
                        event.parameter1(),
                        event.parameter2());
        newEvent.setControlHandle(pvport->verticalScrollBar()->handle());
        pvport->verticalScrollBar()->sendEvent(newEvent);
#endif
        bProcessed = true;
     }
  }                               // End pageDown key.

  // Line scrolling.
  else if ( vkey == IKeyboardEvent::up  ||
            vkey == IKeyboardEvent::down )
  {                               // Up or down arrow key.
     if ( event.isCtrlDown() == false  &&
          event.isShiftDown() == false  &&  event.isAltDown() == false )
     {                            // Arrow key alone: line scroll.
        IViewPort *pvport = (IViewPort*)event.window();
#ifdef IC_PMWIN
        pvport->sendEvent( WM_VSCROLL,
#ifdef  IC_WIN
                           MPFROM2SHORT( (vkey == IKeyboardEvent::up) ?
                                            SB_LINEUP : SB_LINEDOWN ,
                                         0 ) ,
                           IEventData(
                              pvport->verticalScrollBar()->handle()) );
#endif
#ifdef IC_PM
                           pvport->verticalScrollBar()->id(),
                           MPFROM2SHORT( 0,
                                         (vkey == IKeyboardEvent::up) ?
                                            SB_LINEUP : SB_LINEDOWN) );
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
        IEvent newEvent((IWindow*)pvport->verticalScrollBar(),
                        (vkey == IKeyboardEvent::up) ?
                           IC_UM_CR_DECREMENT : IC_UM_CR_INCREMENT,
                        event.parameter1(),
                        event.parameter2());
        newEvent.setControlHandle(pvport->verticalScrollBar()->handle());
        pvport->verticalScrollBar()->sendEvent(newEvent);
#endif
        bProcessed = true;
     }
  }                               // End up or down arrow key.
  else if ( vkey == IKeyboardEvent::left  ||
            vkey == IKeyboardEvent::right )
  {                               // Left or right arrow key.
     if ( event.isCtrlDown() == false  &&
          event.isShiftDown() == false  &&  event.isAltDown() == false )
     {                            // Arrow key alone: line scroll.
        IViewPort *pvport = (IViewPort*)event.window();
#ifdef IC_PMWIN
        pvport->sendEvent( WM_HSCROLL,
#ifdef  IC_WIN
                           MPFROM2SHORT( (vkey == IKeyboardEvent::left) ?
                                            SB_LINELEFT : SB_LINERIGHT ,
                                         0 ) ,
                           IEventData(
                              pvport->horizontalScrollBar()->handle()) );
#endif
#ifdef IC_PM
                           pvport->horizontalScrollBar()->id(),
                           MPFROM2SHORT( 0,
                                         (vkey == IKeyboardEvent::left) ?
                                            SB_LINELEFT : SB_LINERIGHT) );
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
        IEvent newEvent((IWindow*)pvport->horizontalScrollBar(),
                        (vkey == IKeyboardEvent::left) ?
                           IC_UM_CR_DECREMENT : IC_UM_CR_INCREMENT,
                        event.parameter1(),
                        event.parameter2());
        newEvent.setControlHandle(pvport->horizontalScrollBar()->handle());
        pvport->horizontalScrollBar()->sendEvent(newEvent);
#endif
        bProcessed = true;
     }
  }                               // End left or right arrow key.

  /*****************************************************************/
  /* Also need to cause window scrolling to keep the cursor        */
  /* visible during tabbing/cursor movement.                       */
  /*****************************************************************/

  return bProcessed;
}

/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::handleEventsFor                                    |
|                                                                              |
| Attach the keyboard handler to the specified view port object.               |
------------------------------------------------------------------------------*/
IViewPortKeyboardHandler&
  IViewPortKeyboardHandler :: handleEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::handleEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::stopHandlingEventsFor                              |
|                                                                              |
| Detach the keyboard handler from the specified view port object.             |
------------------------------------------------------------------------------*/
IViewPortKeyboardHandler&
  IViewPortKeyboardHandler :: stopHandlingEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::stopHandlingEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::handleEventsFor                                    |
------------------------------------------------------------------------------*/
IHandler& IViewPortKeyboardHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortKeyboardHandler::stopHandlingEventsFor                              |
------------------------------------------------------------------------------*/
IHandler& IViewPortKeyboardHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IViewPortPaintHandler::IViewPortPaintHandler                                 |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IViewPortPaintHandler :: IViewPortPaintHandler ( )
  : IViewPortPaintHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortPaintHandler::~IViewPortPaintHandler                                |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IViewPortPaintHandler :: ~IViewPortPaintHandler ( )
{ }

/*------------------------------------------------------------------------------
| IViewPortPaintHandler::paintWindow                                           |
|                                                                              |
| Ignore paint events, since child windows will do painting.                   |
------------------------------------------------------------------------------*/
bool IViewPortPaintHandler :: paintWindow ( IPaintEvent& event )
{
  IViewPort* pvpViewPort = (IViewPort*)event.window();
  ISize dummy = pvpViewPort->viewWindowSize();
                  // Force view port to update child window positions.

  // Don't need to paint where horizontal scroll bar will.
  IScrollBar* horizontal = pvpViewPort->horizontalScrollBar();
  SWP swp;
  WinQueryWindowPos( horizontal->handle(), &swp );
  if ( swp.fl & SWP_SHOW )   // (IWindow::isShowing invalidates too much.)
  {                                    // Scroll bar being used.
     IRectangle rectChild = horizontal->nativeRect();
     RECTL rectlChild = rectChild.asRECTL();
     IVALIDATERECT( event.handle(), &rectlChild );
  }

  // Don't need to paint where vertical scroll bar will.
  IScrollBar* vertical = pvpViewPort->verticalScrollBar();
  WinQueryWindowPos( vertical->handle(), &swp );
  if ( swp.fl & SWP_SHOW )  // (IWindow::isShowing invalidates too much.)
  {                                    // Scroll bar being used.
     IRectangle rectChild = vertical->nativeRect();
     RECTL rectlChild = rectChild.asRECTL();
     IVALIDATERECT( event.handle(), &rectlChild );
  }

  // Don't need to paint where view rectangle will.
  IWindow* viewRect = IWindow::windowWithOwner( IC_VIEWPORT_VIEWRECTANGLE,
                                                pvpViewPort );
  IRectangle rectChild = viewRect->nativeRect();
  RECTL rectlChild = rectChild.asRECTL();
  IVALIDATERECT( event.handle(), &rectlChild );
  return false;                        // Always pass event on.
}

/*------------------------------------------------------------------------------
| IViewPortPaintHandler::handleEventsFor                                       |
|                                                                              |
| Attach the paint handler to the specified view port object.                  |
------------------------------------------------------------------------------*/
IViewPortPaintHandler&
  IViewPortPaintHandler :: handleEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::handleEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortPaintHandler::stopHandlingEventsFor                                 |
|                                                                              |
| Detach the paint handler from the specified view port object.                |
------------------------------------------------------------------------------*/
IViewPortPaintHandler&
  IViewPortPaintHandler :: stopHandlingEventsFor ( IViewPort* viewport )
{
  IASSERTPARM( viewport != 0 );
  Inherited::stopHandlingEventsFor( viewport );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortPaintHandler::handleEventsFor                                       |
------------------------------------------------------------------------------*/
IHandler& IViewPortPaintHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewPortPaintHandler::stopHandlingEventsFor                                 |
------------------------------------------------------------------------------*/
IHandler& IViewPortPaintHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}


/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::IViewRectanglePaintHandler                       |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IViewRectanglePaintHandler :: IViewRectanglePaintHandler ( )
  : IViewRectanglePaintHandler::Inherited ( )
{ }

/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::~IViewRectanglePaintHandler                      |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IViewRectanglePaintHandler :: ~IViewRectanglePaintHandler ( )
{ }

/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::paintWindow                                      |
|                                                                              |
| Ignore paint events, since child windows will do painting.                   |
------------------------------------------------------------------------------*/
bool IViewRectanglePaintHandler :: paintWindow ( IPaintEvent& event )
{
  IViewRectangle* pvrViewRect = (IViewRectangle*)event.window();
  IViewPort* pvpViewPort = (IViewPort*)pvrViewRect->parent();
  if ( pvpViewPort  &&  pvpViewPort->viewWindow() )
  {                                  // Validate where child will paint.
     IRectangle rectViewWindow( IPoint(), pvpViewPort->viewWindowSize() );
                                       // Size of the view window.
     IRectangle rectVisibleView =
         ICoordinateSystem::convertToNative(
            pvpViewPort->viewWindowDrawRectangle(),
            pvpViewPort->size() );
                                       // Portion in view rectangle.
     IPoint visibleOrigin = rectVisibleView.bottomLeft();
     long coord = visibleOrigin.x();
     visibleOrigin.setX( coord >= 0 ? 0 : 0 - coord );
                                    // (Should always end up being 0.)
     coord = visibleOrigin.y();
     visibleOrigin.setY( coord >= 0 ? 0 : 0 - coord );
                          // Adjust for white space below view window.
     IRectangle rectActualVisible = rectViewWindow & rectVisibleView;
     rectActualVisible.moveTo( visibleOrigin );
     RECTL rectlActualVisible = rectActualVisible.asRECTL();
     IVALIDATERECT( event.handle(), &rectlActualVisible );
  }
  return false;                        // Always pass event on.
}

/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::handleEventsFor                                  |
|                                                                              |
| Attach the paint handler to the specified view port object.                  |
------------------------------------------------------------------------------*/
IViewRectanglePaintHandler&
  IViewRectanglePaintHandler :: handleEventsFor ( IViewRectangle* viewRect )
{
  IASSERTPARM( viewRect != 0 );
  Inherited::handleEventsFor( viewRect );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::stopHandlingEventsFor                            |
|                                                                              |
| Detach the paint handler from the specified view port object.                |
------------------------------------------------------------------------------*/
IViewRectanglePaintHandler&
  IViewRectanglePaintHandler :: stopHandlingEventsFor
                                                 ( IViewRectangle* viewRect )
{
  IASSERTPARM( viewRect != 0 );
  Inherited::stopHandlingEventsFor( viewRect );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::handleEventsFor                                  |
------------------------------------------------------------------------------*/
IHandler& IViewRectanglePaintHandler :: handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IViewRectanglePaintHandler::stopHandlingEventsFor                            |
------------------------------------------------------------------------------*/
IHandler& IViewRectanglePaintHandler :: stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}
#endif // IC_PM
