// Revision: 70 1.19.1.1 source/ui/basectl/itabpriv.cpp, notebook, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: itabpriv.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the following classes:            *
*     ITabControlData                                                          *
*     ITabControlDefaultHandler                                                *
*     ITabControlButton                                                        *
*     ITabControlButtonSequence                                                *
*     ITabPageData                                                             *
*                                                                              *
*   This file also contains the implementation of the following classes which  *
*   are specific to the Windows environment:                                   *
*     ITabControlButtonCanvas                                                  *
*     ITabControlClipPaintHandler                                              *
*     ITabControlClipWindow                                                    *
*     ITabControlPageSequence                                                  *
*     ITabPageKeyboardHandler                                                  *
*                                                                              *
*   This file also contains the implementation of the following classes which  *
*   are specific to the OS/2 environment:                                      *
*     ITabPageResizeHandler                                                    *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   Licensed Material - Property of IBM                                        *
*   (C) Copyright IBM Corp.  1997  All Rights Reserved.                        *
*                                                                              *
*******************************************************************************/
extern "C" {
  #define INCL_WINSYS
  #define INCL_WINWINDOWMGR         // WinQueryWindowULong
  #define INCL_WINBUTTONS           // BS_NOTEBOOKBUTTON
  #define INCL_WINSTDBOOK
  #define INCL_WINFRAMEMGR
  #define INCL_WINDIALOGS           // Needed for PDLGTEMPLATE in BOOKPAGEINFO
  #define INCL_WININPUT
  #include <iwindefs.h>
#ifdef IC_WIN
  #include <commctrl.h>
#endif // IC_WIN
}

#include <itabctl.hpp>
#include <itabpage.hpp>
#include <itabpriv.hpp>
#include <icmdevt.hpp>
#include <ictlevt.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <iframe.hpp>
#include <ihelp.hpp>
#include <ikeyevt.hpp>
#include <inotify.hpp>
#include <iplatfrm.hpp>
#include <ipushbut.hpp>
#include <ireslib.hpp>
#include <iwcname.hpp>

#ifdef IC_WIN
#include <ibcolor.hpp>
#include <ibundles.hpp>
#include <ifont.hpp>
#include <igrport.hpp>
#include <igline2d.hpp>
#include <ipainevt.hpp>
#include <isetcv.hpp>
#include <iwposbuf.hpp>
#endif // IC_WIN

#ifdef IC_PM
#include <icanvas.hpp>
#endif // IC_PM

class ICanvas;

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ITABPRIV_CPP_
  #include <ipagetun.h>
#endif // IC_PAGETUNE

// Make private data pointer names less verbose.
#define tcd fTabControlData
#define tpd fTabPageData

#ifdef IC_PM
const unsigned long ITabPageData::autoColors[] = {0x0055DBFF, 0x0080DBAA, 0x008092FF,
						  0x00D5B6AA, 0x00FFFFAA, 0x00AA92AA,
						  0x00FF9255, 0x00FFDB55, 0x00FFB6AA,
						  0x00FFDBAA};
#endif // IC_PM

/*------------------------------------------------------------------------------
| ITabControlData::ITabControlData                                             |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlData :: ITabControlData ( ITabControl* tabControl )
  : fTabControl( tabControl )
  , fPageSequence( 0 )
  , fCommonButtons( 0 )
  , fDefaultButton( 0 )
  , fAutoDeletePages( false )
  , fPageBkgrndColorSet( false )
  , fPageBkgrndColor( IColor::notebookPageBgnd )
  , fValidate( 0 )
#ifdef IC_WIN
  , fButtonCanvas( 0 )
  , fImageList( 0 )
  , fInCreate( false )
#endif // IC_WIN
{
  for (unsigned long i=0; i<NUM_STANDARD_BUTTONS; i++)
  {
     fStandardButton[i] = 0;
  }
}


/*------------------------------------------------------------------------------
| ITabControlData::ITabControlData                                             |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlData :: ~ITabControlData ( )
{
  IMODTRACE_DEVELOP( "ITabControlData::dtor" );

  delete fDefaultHandler;

#ifdef IC_WIN
  delete fPageKeyboardHandler;
#endif

#ifdef IC_PM
  delete fTabPageResizeHandler;
#endif // IC_PM

#ifdef IC_WIN
  // Destroy the page clipping window.
  delete fClipWindow;

  // Destroy the image list if it exists
  if (fImageList)
  {
    if (!ImageList_Destroy( (HIMAGELIST)fImageList ))
      ITHROWGUIERROR("ImageList_Destroy");
  }
#endif // IC_WIN

  // Delete the button sequence.
  delete fCommonButtons;

  // Delete the page collection.
  delete fPageSequence;
}


/*------------------------------------------------------------------------------
| ITabControlData::createButton                                                |
|                                                                              |
| Create a standard button.                                                    |
------------------------------------------------------------------------------*/
IPushButton* ITabControlData::createButton ( ITabControl::EButton buttonType )
{
  IMODTRACE_DEVELOP( "ITabControlData::createButton" );

  // Map the enum to an array index.
  unsigned long index = buttonIndex( buttonType );

  // If we have not yet created a standard button of this type, create it.
  if (!fStandardButton[ index ])
  {
     IPushButton::Style buttonStyle = IPushButton::classDefaultStyle;
     if (buttonType == ITabControl::kHelp)
	buttonStyle |= IPushButton::help | IPushButton::noPointerFocus;

     fStandardButton[ index ] = new IPushButton( buttonType,
#ifdef IC_WIN
						 fButtonCanvas,
						 fButtonCanvas,
#elif IC_PM
						 fTabControl,
						 fTabControl,
#endif // IC_WIN
						 IRectangle(),
						 buttonStyle );

     // Use auto deletion for the standard buttons.
     fStandardButton[ index ]->setAutoDeleteObject();

#ifdef IC_PM
     // For OS/2, we always need the BS_NOTEBOOKBUTTON style.
     IWindowHandle hwndButton = fStandardButton[ index ]->handle();
     unsigned long ulButtonStyle = ISTYLEOF( hwndButton );
     ulButtonStyle |= BS_NOTEBOOKBUTTON;
     ISETWINDOWSTYLE( hwndButton, ulButtonStyle );
#endif
  }

  // Try to load the button text from the application resource file.
  IString buttonText;
  try
  {
     buttonText =
	IApplication::current().resourceLibrary().loadString( buttonType );
  }
  catch (...)
  {
     switch (buttonType)
     {
       case ITabControl::kApply:
	 buttonText = IString( "Apply" );
	 break;
       case ITabControl::kOk:
	 buttonText = IString( "Ok" );
	 break;
       case ITabControl::kCancel:
	 buttonText = IString( "Cancel" );
	 break;
       case ITabControl::kDefault:
	 buttonText = IString( "Reset" );
	 break;
       case ITabControl::kHelp:
	 buttonText = IString( "Help" );
	 break;
       default:
	 break;
     }
  }

  fStandardButton[ index ]->setText( buttonText );

  return fStandardButton[ index ];
}


/*------------------------------------------------------------------------------
| ITabControlData::addCommonButton                                             |
|                                                                              |
| Add a common button to the tab control.                                      |
------------------------------------------------------------------------------*/
void ITabControlData::addCommonButton( ITabControlButton *tcButton,
				       bool               isDefaultButton )
{
  // Get the actual button.
  IPushButton *button = tcButton->button( fTabControl );

  // Create a collection to contain the common buttons if there isn't already
  // one.
  if (!fCommonButtons)
  {
     fCommonButtons = new ITabControlButtonSequence();
  }
  else
  {
     // Throw an exception if this button has already been added.
     ITabControlButtonSequence::Cursor cursor( *fCommonButtons );
     forCursor (cursor)
     {
	ITabControlButton *tcCommonButton = fCommonButtons->elementAt( cursor );
	if (tcCommonButton->fApplicationButton == tcButton->fApplicationButton &&
	    tcCommonButton->fStandardButton == tcButton->fStandardButton)
	{
	   delete tcButton;
	   ITHROWLIBRARYERROR1( IC_BUTTON_ALREADY_ADDED,
				IBaseErrorInfo::invalidRequest,
				IException::recoverable,
				"ITabControl" );
	}
     }
  }
  // Add this button as the last one in the button sequence.
  fCommonButtons->addAsLast( tcButton );

  // If this is to be the default button, cache that information.  This
  // enables a common button to be the default button in one sequence
  // (common or page) and not in another.
  if (isDefaultButton)
     fDefaultButton = tcButton;

#ifdef IC_WIN
  // Common buttons are initially added to the button area unless the top page
  // has page buttons.
  ITabPage *currentPage = fTabControl->topPage();
  if (currentPage && !currentPage->tpd->fPageButtons)
  {
     fButtonCanvas->add( button );
     if (!fInCreate)
	showButtons( 0, currentPage );
  }
  else if (currentPage == 0){
     fButtonCanvas->add( button );
     fButtonCanvas->refresh();
  }
#endif // IC_WIN

#ifdef IC_PM
  // Make sure the BS_NOTEBOOKBUTTON style is set for the button.
  unsigned long ulButtonStyle = ISTYLEOF( button->handle() );
  if (~ulButtonStyle & BS_NOTEBOOKBUTTON)
  {
     ulButtonStyle |= BS_NOTEBOOKBUTTON;
     ISETWINDOWSTYLE( button->handle(), ulButtonStyle );
  }

  if (fTabControl->totalPages())
  {
     // We are not using the native common button structure to implement common
     // buttons on PM since the PM notebook creates the common buttons and our
     // interface allows client code to create the pushbuttons.  Therefore, all
     // tab control pages without page button sequences use the common button
     // sequence as their buttons.  Add this new common button to all such pages.
     ITabControl::Cursor pageCursor( *fTabControl );
     for (pageCursor.setToFirst();
	  pageCursor.isValid();
	  pageCursor.setToNext())
     {
	ITabPage *tabPage = pageCursor.current();
	if (!tabPage->tpd->fPageButtons &&
	     tabPage->tpd->fPageWindow &&
	     tabPage->tpd->fPageId)
	{
	   // Set the page's buttons.
	   tabPage->tpd->setButtons( fCommonButtons );
	}
     }
  }

  // If this button is to be a default button and we don't currently have default
  // emphasis on any button, go ahead and set this one as default.  This ensures
  // that the tab control will initially have a default button if the common buttons
  // are on the initial top page.
  if (isDefaultButton)
  {
     IPushButton *emphasisButton = (IPushButton*)IWindow::windowWithHandle(
				      fTabControl->defaultEmphasisButton() );
     if (emphasisButton == 0)
	button->enableDefault();
  }
#endif // IC_PM

  // Send a notification.  Include the button as the event data.
  fTabControl->notifyObservers( INotificationEvent( ITabControl::buttonAddId,
						    *fTabControl,
						    false,
						    (void*)button ));

  // Indicate that the tab control's minimum size may have changed.
#ifdef IC_WIN
  if (!fInCreate || !fTabControl->isVisible())
#else
  if (!fTabControl->isVisible())
#endif
     fTabControl->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
}


/*------------------------------------------------------------------------------
| ITabControlData::removeCommonButton                                          |
|                                                                              |
| Remove a common button from the tab control.                                 |
------------------------------------------------------------------------------*/
void ITabControlData::removeCommonButton( ITabControlButton *tcButton )
{
  // Get the actual push button.
  IPushButton *button = tcButton->button( fTabControl );

  // Remove this button from the button sequence.
  fCommonButtons->remove( tcButton );

  // If we are removing the default button, clear the cached pointer.
  if (fDefaultButton == tcButton)
     fDefaultButton = 0;

  // If there are no more buttons in the sequence, delete the sequence.
  if (fCommonButtons->isEmpty())
  {
     delete fCommonButtons;
     fCommonButtons = 0;
  }

#ifdef IC_WIN
  else
  {
     // We may have removed the first element in the sequence.  If so,
     // set the tabstop and group styles for the new first button.
     ITabControlButton *tcFirstButton = fCommonButtons->firstElement();
     IPushButton *firstButton = tcFirstButton->button( fTabControl );

     if (!firstButton->isGroup())
     {
	firstButton->enableTabStop();
	firstButton->enableGroup();
     }
  }

  // Only remove this button from the button area if it is visible for the
  // current top page.
  ITabPage *currentPage = fTabControl->topPage();
  if (currentPage && fTabControl->isVisible() && !currentPage->tpd->fPageButtons)
  {
     // Remove the button from the set canvas if it is in the layout.  It
     // is possible that it is not in the layout if it is being deleted.
     if (fButtonCanvas->isInLayout( button ))
	fButtonCanvas->remove( button );
     if (!fInCreate)
	showButtons( 0, currentPage );
  }
#endif // IC_WIN

#ifdef IC_PM
  // In order to remove this button from all of the pages utilizing the
  // common buttons, we need to reparent all of the common buttons to each
  // of these pages and reset the page windows.
  if (!fPageSequence->isEmpty())
  {
     ITabControlPageSequence::Cursor pageCursor( *fPageSequence );
     forCursor (pageCursor)
     {
	ITabPage *tabPage = fPageSequence->elementAt( pageCursor );
	if (!tabPage->tpd->fPageButtons &&
	     tabPage->tpd->fPageWindow &&
	     tabPage->tpd->fPageId)
	{
	   // Set the page's buttons.
	   tabPage->tpd->setButtons( fCommonButtons );
	}
     }
  }
#endif // IC_PM

  // Send a notification.  Include the button as the event data.
  fTabControl->notifyObservers( INotificationEvent( ITabControl::buttonRemoveId,
						    *fTabControl,
						    false,
						    (void*)button ));

  // Indicate that the tab control's minimum size may have changed.
#ifdef IC_WIN
  if (!fInCreate || !fTabControl->isVisible())
#else
  if (!fTabControl->isVisible())
#endif
     fTabControl->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
}


/*------------------------------------------------------------------------------
| ITabControlData::buttonIndex                                                 |
|                                                                              |
| Function to map from button id enum to button array index.                   |
------------------------------------------------------------------------------*/
unsigned long ITabControlData::buttonIndex(
				 const ITabControl::EButton buttonType ) const
{
  unsigned long index = 0;
  switch (buttonType)
  {
    case ITabControl::kApply:
      index = 0;
      break;
    case ITabControl::kOk:
      index = 1;
      break;
    case ITabControl::kCancel:
      index = 2;
      break;
    case ITabControl::kDefault:
      index = 3;
      break;
    case ITabControl::kHelp:
      index = 4;
      break;
    default:
      ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
			  IBaseErrorInfo::invalidParameter,
			  IException::recoverable);
      break;
  }
  return index;
}


/*------------------------------------------------------------------------------
| ITabControlData::buttonEnum                                                  |
|                                                                              |
| Function to map from pointer to an IPushButton to an enum.                   |
------------------------------------------------------------------------------*/
ITabControl::EButton ITabControlData::buttonEnum(
					 const IPushButton* button ) const
{
  if (button == fStandardButton[0])
     return ITabControl::kApply;
  else if (button == fStandardButton[1])
     return ITabControl::kOk;
  else if (button == fStandardButton[2])
     return ITabControl::kCancel;
  else if (button == fStandardButton[3])
     return ITabControl::kDefault;
  else if (button == fStandardButton[4])
     return ITabControl::kHelp;
  return (ITabControl::EButton)0;
}


/*------------------------------------------------------------------------------
| ITabControlData::addPage                                                     |
|                                                                              |
| Add a new page to the tab control.  A pointer to the tab page is stored in   |
| the TC_ITEM structure for this page.  The pointer to the tab page is also    |
| stored in the page sequence collection.  The position of the tab in this     |
| sequence indicates the actual index of the page in the tab control.  Note    |
| that the positions in the equality sequence are 1-based and the tab control  |
| page indices are 0-based.  The position passed to this function is 1-based.  |
------------------------------------------------------------------------------*/
void ITabControlData :: addPage ( unsigned long   pageIndex,
				  const ITabPage* tabPage )
{
  IMODTRACE_DEVELOP( "ITabControlData::addPage" );

  // Cast away constness.
  ITabPage     *newPage = (ITabPage*)tabPage;
  ITabPageData *newPageData = newPage->tpd;

  // Insert the tab page into the page sequence.
  if (pageIndex && pageIndex <= fPageSequence->numberOfElements()+1)
  {
     fPageSequence->addAtPosition( pageIndex, newPage );
  }
  else
  {
     ITHROWLIBRARYERROR( IC_INDEX_OUT_OF_RANGE,
			 IBaseErrorInfo::invalidRequest,
			 IException::recoverable);
  }

  // If we have page buttons on this newly added page, we need to verify
  // that the buttons style is set on the tab control.
  if (newPageData->fPageButtons &&
#ifdef IC_WIN
      !fButtonCanvas)
#elif IC_PM
      !(fTabControl->style() & BKS_BUTTONAREA))
#endif // IC_WIN
  {
     ITHROWLIBRARYERROR1( IC_INVALID_STYLE_FOR_FUNCTION,
			  IBaseErrorInfo::invalidRequest,
			  IException::recoverable,
			  "ITabControl::buttons" );
  }

  // If the page has page buttons added before the page was added to the
  // tab control, go ahead and "finish" adding them now.
  if (newPageData->fPageButtons)
  {
     ITabControlButtonSequence *pageButtons = newPageData->fPageButtons;
     ITabControlButtonSequence::Cursor cursor( *pageButtons );
     cursor.setToFirst();
     while (cursor.isValid())
     {
	ITabControlButton *tcButton = pageButtons->elementAt( cursor );
	IPushButton *button = tcButton->button( fTabControl );

	if (IWindow::isWindowValid( button ))
	{
	   // Verify that any application created page buttons have the set canvas
	   // as parent.  Although we could just reparent, enforce this constraint.
	   // This constraint exists to ensure compatibility with any future AIX
	   // release.
	   if (tcButton->fApplicationButton &&
	       tcButton->fApplicationButton->parent() != fTabControl->buttonParent())
	   {
	      ITHROWLIBRARYERROR( IC_INVALID_BUTTON_PARENT,
				  IBaseErrorInfo::invalidRequest,
				  IException::recoverable);
	   }

#ifdef IC_PM
	   // Make sure the BS_NOTEBOOKBUTTON style is set if this is an
	   // application created button.
	   if (tcButton->fApplicationButton)
	   {
	      unsigned long ulButtonStyle = ISTYLEOF( button->handle() );
	      if (~ulButtonStyle & BS_NOTEBOOKBUTTON)
	      {
		 ulButtonStyle |= BS_NOTEBOOKBUTTON;
		 ISETWINDOWSTYLE( button->handle(), ulButtonStyle );
	      }
	   }
#endif

	   // Send a notification.  Include the button as the event data.
	   fTabControl->notifyObservers( INotificationEvent( ITabControl::buttonAddId,
							     *fTabControl,
							     false,
							     (void*)button ));

	   cursor.setToNext();
	}
	else
	{
	   // We found a button that's not valid; just remove it from the
	   // sequence.  We checked for a valid window when the button was
	   // added to the tab page.  Maybe it was deleted since then.  (Ok,
	   // that's pretty unlikely; just being thorough.)
	   unsigned long position = pageButtons->positionAt( cursor );
	   pageButtons->removeAt( cursor );
	   if (!pageButtons->isEmpty())
	   {
	      if (position == 0)
		 cursor.setToFirst();
	      else
		 pageButtons->setToPosition( position, cursor );
	   }
	   else
	      delete pageButtons;
	}
     }
  }

#ifdef IC_WIN
  IWindow* pageWindow = newPage->pageWindow();
  if (pageWindow)
  {
     // If we have an application page window, reparent it to the clipping
     // window.  When the page is removed from the tab control, we'll reset its
     // parent.

     // Store the original parent of the page window.
     pageWindow->setParent( fClipWindow );
     newPageData->fPageWindowParent = pageWindow->parent();

     // We must check to see whether this window is the page window for another
     // tab page.  If so, store the original parent found in the other page window
     // as the original parent for this page window.
     if (!fPageSequence->isEmpty())
     {
	ITabControlPageSequence::Cursor pageCursor( *fPageSequence );
	forCursor (pageCursor)
	{
	   ITabPage* currentPage = fPageSequence->elementAt( pageCursor );
	   if (currentPage->pageWindow() == pageWindow)
	   {
	      newPageData->fPageWindowParent =
		    currentPage->tpd->fPageWindowParent;
	      pageCursor.setToLast();
	   }
	}
     }
  }

  // Insert the tab control page at the given index.  Store a pointer to the
  // ITabPage object as the application-defined data.
  TC_ITEM tabItem;
  tabItem.mask = TCIF_PARAM;
  tabItem.lParam = (unsigned long)(void*)newPage;
  if (newPage->tpd->fTabText != IText())
  {
     tabItem.mask |= TCIF_TEXT;
     tabItem.pszText = (char*)(const char*)newPage->tabText();
  }

  if (TabCtrl_InsertItem( fTabControl->handle(),
			  pageIndex-1,
			  &tabItem ) == -1)
  {
     // If insert was not successful, throw an exception
     ITHROWGUIERROR("TCM_INSERTITEM");
  }

  newPage->tpd->fTabControl = fTabControl;
#endif // IC_WIN

#ifdef IC_PM
  unsigned short pageOrder;
  unsigned long  pageId = 0;
  if (pageIndex == 1)
     pageOrder = BKA_FIRST;
  else if (pageIndex > fPageSequence->numberOfElements())
     pageOrder = BKA_LAST;
  else
  {
     pageOrder = BKA_NEXT;
     pageId = fPageSequence->elementAtPosition( pageIndex-1 )->tpd->fPageId;
  }

  // Store a pointer to this tab control in the tab page object.
  newPageData->fTabControl = fTabControl;

  newPageData->fPageId =
	fTabControl->sendEvent( BKM_INSERTPAGE,
				pageId,
				MPFROM2SHORT( newPage->tpd->fStyle, pageOrder ));
  if (!newPageData->fPageId)
  {
     ITHROWGUIERROR( "BKM_INSERTPAGE" );
  }

  // Store a pointer to the ITabPage object as the user data.
  if (!fTabControl->sendEvent( BKM_SETPAGEDATA,
			       newPageData->fPageId,
			       newPage ).asUnsignedLong())
  {
     ITHROWGUIERROR( "BKM_SETPAGEDATA" );
  }
#endif // IC_PM

  fValidate++;

#ifdef IC_PM
  // Set the tab text if it has already been set in the tab page.
  if (newPageData->fTabText != IText())
  {
     newPage->setTabText( newPageData->fTabText );
  }

  // Set the tab background color if it has already been set in the tab page.
  if (newPageData->fTabBkgrndColorSet)
  {
     if (!fTabControl->sendEvent( BKM_SETTABCOLOR,
				  (unsigned long)(newPageData->fPageId),
				  (unsigned long)(newPageData->fTabBkgrndColor.asRGBLong()) )
	 .asUnsignedLong())
     {
	ITHROWGUIERROR( "BKM_SETTABCOLOR" );
     }

     // Send a notification to all observers.
     fTabControl->notifyObservers(
			  INotificationEvent( ITabControl::tabBackgroundColorId,
					      *(fTabControl),
					      false,
					      (void*)newPage ));
  }
#endif

  // If the tab page had a stored bitmap, set it now.  We do it this way to
  // isolate the image list management.
  IBitmapHandle tabBitmap = newPage->tabBitmap();
  if (tabBitmap)
  {
     newPage->setTabBitmap( tabBitmap );
  }

#ifdef IC_PM
  // If there are any buttons, add them as notebook page buttons.
  if ((fCommonButtons || newPageData->fPageButtons)
      && newPageData->fPageWindow)
  {
     ITabControlButtonSequence
	*buttons = (newPageData->fPageButtons) ?
		      newPageData->fPageButtons : fCommonButtons;
     newPageData->setButtons( buttons );
  }

  if (newPageData->fPageWindow)
  {
      if (!fTabControl->sendEvent( BKM_SETPAGEWINDOWHWND,
				(unsigned long)(newPageData->fPageId),
				 newPageData->fPageWindow->handle().asUnsigned() ).asUnsignedLong())
      {
	 ITHROWGUIERROR( "BKM_SETPAGEWINDOWHWND" );
      }
     // Add the page window resize handler.  Only add it if this page
     // window is not already a page window for another page.
     ITabControl::Cursor cursor( *fTabControl );
     bool isHandled = false;
     for (cursor.setToFirst();
	  cursor.isValid() && !isHandled;
	  cursor.setToNext())
     {
	ITabPage *tabPage = cursor.current();
	if (tabPage != newPage &&
	    tabPage->tpd->fPageWindow == newPageData->fPageWindow)
	   isHandled = true;
     }
     if (!isHandled)
	fTabPageResizeHandler->handleEventsFor( newPageData->fPageWindow );
  }

#endif // IC_PM

  // Send a notification event for the page added.
  fTabControl->notifyObservers( INotificationEvent( ITabControl::pageAddId,
						    *fTabControl,
						    false,
						    (void*)newPage ));

#ifdef IC_WIN
  // Send a notification that the page has been selected if this is the
  // top page in the tab control.
  if (tabPage == fTabControl->topPage())
  {
     // Generate a page selection notification.
     NMHDR notifyInfo;
     notifyInfo.hwndFrom = fTabControl->handle();
     notifyInfo.idFrom = (UINT)fTabControl->id();
     notifyInfo.code = TCN_SELCHANGE;
     fTabControl->parent()->handle().sendEvent(
				      WM_NOTIFY,
				      IEventParameter1( fTabControl->id() ),
				      IEventParameter2( &notifyInfo ) );
  }
  else if (pageWindow)
     pageWindow->hide();
#endif // IC_WIN

  // IWindowPrivateData may have a cached minimum value size.  Need to indicate
  // that it may no longer be valid if this is the first page added or the page
  // has a page window.
  if (fTabControl->totalPages() == 1)
     fTabControl->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
}


/*------------------------------------------------------------------------------
| ITabControlData::tabPageIndex                                                |
|                                                                              |
| Returns the 1-based index of the specified tab page.                         |
------------------------------------------------------------------------------*/
unsigned long ITabControlData :: tabPageIndex( const ITabPage* tabPage ) const
{
  IMODTRACE_DEVELOP( "ITabControlData::tabPageIndex" );

  long pageIndex = 0;

  ITabControlPageSequence::Cursor pageCursor( *fPageSequence );
  if (fPageSequence->locate( (ITabPage*)tabPage, pageCursor ))
  {
     pageIndex = fPageSequence->position( pageCursor );
  }
  return( pageIndex );
}


/*------------------------------------------------------------------------------
| ITabControlData::deletePage                                                  |
------------------------------------------------------------------------------*/
void ITabControlData :: deletePage ( ITabPage* removePage,
				     bool      bRemoveAll )
{
  IMODTRACE_DEVELOP( "ITabControlData::deletePage" );

#ifdef IC_WIN
  // We cannot use the Windows' tab control's delete all function if we want to
  // send a page deleted notification.  Instead we have to loop thru
  // every page to simulate the PM behavior, since the tab control does
  // not have a TCN_ notification defined for page deletion.

  // Hide the page clipping window if we're deleting all of the pages.
  if (bRemoveAll)
     fClipWindow->hide();
#endif // IC_WIN

  IWindowHandle hwndTabControl = fTabControl->handle();

  unsigned long
     deleteCount = bRemoveAll ? fPageSequence->numberOfElements() : 1;

  ITabPage *tabPage = removePage;
  for (unsigned long i=1; i <= deleteCount; i++)
  {
     if (bRemoveAll)
	tabPage = fPageSequence->firstElement();
     ITabPageData *tabPageData = tabPage->tpd;

     unsigned long pageIndex = tabPageIndex( tabPage );

#ifdef IC_WIN
     if (hwndTabControl)
     {
	// If we have created an image list, then check to see if an image exists
	// for the specified page.  If an image exists, then remove it from the
	// image list.
	if (fImageList)
	{
	   TC_ITEM tabCtrlItem;
	   tabCtrlItem.mask = TCIF_IMAGE;
	
	   if (!TabCtrl_GetItem( hwndTabControl, pageIndex-1, &tabCtrlItem ))
	   {
	      ITHROWGUIERROR("TabCtrl_GetItem");
	   }

	   if (tabCtrlItem.iImage != -1 )
	   {
	      TabCtrl_RemoveImage( hwndTabControl, tabCtrlItem.iImage );
	   }
	}

	// If the delete request is for the current top page, we must select a
	// new top page, unless the tab control is empty.
	if (hwndTabControl &&
	    tabPage == fTabControl->topPage() &&
	    fTabControl->totalPages() > 1 &&
	    !bRemoveAll)
	{
	   // Reset the top page to the next page if it exists, or the previous
	   // page if it exists.  Since the tab control is 0-based and pageIndex
	   // is 1-based, the next page in the tab control has an index of pageIndex.
	   long newPageIndex = pageIndex;
	   long retVal = TabCtrl_SetCurSel( hwndTabControl, newPageIndex );
	   if ( retVal == -1 && pageIndex > 1 )
	   {
	      newPageIndex = pageIndex - 2;
	      retVal = TabCtrl_SetCurSel( hwndTabControl, newPageIndex );
	   }

	   if ( retVal != -1 )
	   {
	      processTabSelect( pageIndex, newPageIndex );
	   }
	}

	// Reset the tab control pointer in the tab page to reflect the page's
	// removal from the tab control.
	tabPageData->fTabControl = 0;

	// Delete the specified page.
	if (!TabCtrl_DeleteItem( hwndTabControl, pageIndex-1 ))
	{
	   ITHROWGUIERROR("TabCtrl_DeleteItem");
	}

	// Send the page deleted notification to simulate PM behavior.
	fTabControl->parent()->handle().sendEvent(
				WM_CONTROL,
				IEventParameter1( (unsigned short)fTabControl->id(),
						  (unsigned short)IC_UM_TCN_PAGEDELETED ),
				IEventParameter2( (void*)tabPage ));
     }

     // Remove the tab page from the page sequence.
     fPageSequence->removeAtPosition( pageIndex );

     // Reset the parent of the application page window for the tab page we
     // just deleted.  We only want to do this if this if the application page
     // isn't associated with any other tab page.
     IWindow* pageWindow = tabPage->pageWindow();
     if (pageWindow && pageWindow->isValid())
     {
	bool bResetParent = true;
	if (!fPageSequence->isEmpty())
	{
	   ITabControlPageSequence::Cursor pageCursor( *fPageSequence );
	   forCursor (pageCursor)
	   {
	      if (fPageSequence->elementAt( pageCursor )->pageWindow() == pageWindow)
	      {
		 bResetParent = false;
		 pageCursor.setToLast();
	      }
	   }
	}

	// If we didn't find a page (other than the one being freed) that has the
	// same application page window as the one being freed, reset the
	// application page window's parent to its original parent.
	if (bResetParent)
	{
      SetParent( pageWindow->handle(),
                 tabPageData->fPageWindowParent->handle() );
	   pageWindow->hide();
	}
     }

     if (fClipWindow->isShowing())
	fClipWindow->refresh();
     if (fButtonCanvas && fButtonCanvas->isShowing())
	fButtonCanvas->refresh();
#endif // IC_WIN

#ifdef IC_PM
     // If this page has a page window, detach the page resize handler from the
     // page window.  Only remove it if the page window is not also the page
     // window for another page.
//////////// Set Top Page to another page before delette
	if (hwndTabControl &&
	    tabPage == fTabControl->topPage() &&
	    fTabControl->totalPages() > 1 &&
	    !bRemoveAll)
	{
	   long newPageIndex = pageIndex;

	     ITabPage *previousPage = fTabControl->previousPage( fTabControl->topPage() );
	     if (previousPage)
		  fTabControl->setTopPage( previousPage );
	     else {
		  ITabPage *nextPage = fTabControl->nextPage( fTabControl->topPage() );
		  if (nextPage)
		  fTabControl->setTopPage( nextPage );
	     }
	}
///////////////////////////////////////////////////////////

     if (tabPageData->fPageWindow &&
	 tabPageData->fPageWindow->isValid())
     {
	ITabControl::Cursor cursor( *fTabControl );
	bool pageWindowFound = false;
	for (cursor.setToFirst();
	     cursor.isValid() && !pageWindowFound;
	     cursor.setToNext())
	{
	   ITabPage *currentPage = cursor.current();
	   if (currentPage != tabPage &&
	       currentPage->tpd->fPageWindow == tabPageData->fPageWindow)
	      pageWindowFound = true;
	}
	if (!pageWindowFound)
	{
	   fTabPageResizeHandler->stopHandlingEventsFor( tabPageData->fPageWindow );
	}
     }

     if (!bRemoveAll)
     {
	if (!fTabControl->sendEvent( BKM_DELETEPAGE,
				     tabPageData->fPageId,
				     BKA_SINGLE ).asUnsignedLong())
	{
	   ITHROWGUIERROR( "BKM_DELETEPAGE" );
	}
     }

     // Remove the tab page from our tab page sequence and reset the page id in
     // the tab page object.
     fPageSequence->removeAtPosition( pageIndex );
     tabPageData->fPageId = 0;

     // Reset the tab control pointer in the tab page to reflect the page's
     // removal from the tab control.
     tabPageData->fTabControl = 0;
#endif // IC_PM

     fValidate--;
  }

#ifdef IC_PM
  if (bRemoveAll)
  {
     if (!fTabControl->sendEvent( BKM_DELETEPAGE,
				  0,
				  BKA_ALL ).asUnsignedLong())
     {
	ITHROWGUIERROR( "BKM_DELETEPAGE" );
     }
  }
#endif // IC_PM
}


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ITabControlData::processTabSelect                                            |
|                                                                              |
| Generate the notification when a tab is selected and display the page window |
| if it is available.  The indices passed to this function are 1-based.        |
------------------------------------------------------------------------------*/
void ITabControlData :: processTabSelect ( unsigned long currentPageIndex,
					   unsigned long newPageIndex )
{
  IMODTRACE_DEVELOP( "ITabControlData::processTabSelect" );

  bool bChangeFocus = false;

  ITabPage
     *oldPage = currentPageIndex ?
		     fPageSequence->elementAtPosition( currentPageIndex ) : 0;
  ITabPage
     *newPage = fPageSequence->elementAtPosition( newPageIndex );

  // If the current page index equals 0, then we are processing the selection
  // notification that occurs when the first page is added to the tab control.
  // If this is the case, do not hide the application window for the current page.
  if (oldPage != 0)
  {
    // Hide the application page window for the current tab page if the
    // application page window exists.  Check to see if it has the focus
    // before hiding as we will use this indicator later.
    IWindow* oldPageWindow = oldPage->tpd->fPageWindow;
    if (oldPageWindow)
    {
      bChangeFocus = oldPageWindow->hasFocus();
      oldPageWindow->hide();
      fPageKeyboardHandler->stopHandlingEventsFor( oldPageWindow );
    }
  }

  // If the top page does not have an application page window set,
  // invoke the virtual function to allow derived class to set
  // it now.  Set a flag to prevent unnecessary calls to showButtons
  // which could occur if the application code adds buttons in
  // createPageWindow.
  IWindow* newPageWindow = newPage->tpd->fPageWindow;
  if (!newPageWindow)
  {
     fInCreate = true;
     // Remove the current default push button if there is one.  We need
     // to do this prior to calling createPageWindow to handle the case
     // where page buttons are added during createPageWindow and the
     // previous default button is added as a page button but isn't the
     // default for the new page.
     if (oldPage != 0)
     {
	ITabControlButton *prevDefault = oldPage->tpd->fDefaultButton;
	if (prevDefault)
	   prevDefault->button( fTabControl )->disableDefault();
	else if (fDefaultButton)
	   fDefaultButton->button( fTabControl )->disableDefault();
     }
     newPageWindow = newPage->createPageWindow();
     if (newPageWindow)
	newPage->setPageWindow( newPageWindow );
     fInCreate = false;
  }
  else
  {
     // Show the new top page window.
     newPage->updatePageWindow();
     showTopPage( newPageIndex );
  }

  // Manage the buttons in the set canvas.
  if (fButtonCanvas)
  {
     ITabPage *currentPage = fPageSequence->elementAtPosition( currentPageIndex );
     showButtons( currentPage, newPage );
  }

  // If the focus was on the old application page window, set it to the new
  // application page window if it exists.
  if (bChangeFocus && newPageWindow)
     newPageWindow->setFocus();
}


/*------------------------------------------------------------------------------
| ITabControlData::showTopPage                                                 |
|                                                                              |
| Show the top page. Appropriately size the clipping and application windows.  |
| The index passed to this function is 1-based.                                |
------------------------------------------------------------------------------*/
void ITabControlData::showTopPage( unsigned long pageIndex,
				   bool          bNotResizing )
{
  IMODTRACE_DEVELOP( "ITabControlData::showTopPage" );

  if (!fClipWindow->isVisible() && bNotResizing)
  {
     // Layout the clip window and button area.
     layoutChildren( fTabControl->size() );
  }

  // If the application page window exists, position and size it to fit the
  // tab control's display area, as long as the autoPageSize attribute is
  // specified.  Also, show it as well.
  ITabPage* tabPage = fPageSequence->elementAtPosition( pageIndex );
  IWindow* pageWindow = tabPage->pageWindow();
  if ( pageWindow )
  {
     if (tabPage->isAutoPageSize())
     {
	pageWindow->moveSizeTo( IRectangle( IPoint( 0, 0 ),
					   fClipWindow->size() ));
     }

     if (!pageWindow->isVisible())
	pageWindow->show();

     if (bNotResizing)
	fPageKeyboardHandler->handleEventsFor( pageWindow );
  }
  else
  {
     // Otherwise, invalidate the page clipping window so it is repainted.
     fClipWindow->refresh();
  }
}


/*------------------------------------------------------------------------------
| ITabControlData::tabControlResize                                            |
|                                                                              |
| Resize the tab control, page clipping, and application page windows.         |
------------------------------------------------------------------------------*/
void ITabControlData::tabControlResize ( const ISize& newSize )
{
  IMODTRACE_DEVELOP( "ITabControlData::tabControlResize" );

  // Layout the clip window and button area.
  layoutChildren( newSize );

  // We are finished if there are no pages in the notebook.
  if (!fTabControl->totalPages())
    return;

  // Get the index for the currently selected tab page.
  long pageIndex = TabCtrl_GetCurSel( fTabControl->handle() );
  if ( pageIndex == -1 )
  {
    ITHROWGUIERROR("TabCtrl_GetCurSel");
  }

  // Show the top page window.
  showTopPage( pageIndex+1, false );
}


/*------------------------------------------------------------------------------
| ITabControlData::layoutChildren                                              |
|                                                                              |
| Layout the page clipping, button area, and application page windows.         |
------------------------------------------------------------------------------*/
void ITabControlData::layoutChildren ( const ISize& size )
{
  IMODTRACE_DEVELOP( "ITabControlData::layoutChildren" );

  // If the tab control has no size, just return.
  if (size == ISize())
     return;

  // Calculate the display rectangle.
  RECT tabCtrlRect;
  IWindowPosBuffer wposbuf( fTabControl );

  SetRect( &tabCtrlRect, 0, 0,
	   (int)size.width(),
	   (int)size.height() );

  adjustRect( false, &tabCtrlRect );

  // Position and size the page clipping window, and show it if it is not
  // visible.
  IRectangle rectClip( IPoint( tabCtrlRect.left, tabCtrlRect.top ),
		       ISize( tabCtrlRect.right - tabCtrlRect.left,
			      tabCtrlRect.bottom - tabCtrlRect.top
				 - fButtonCanvasHeight ));
  wposbuf.moveSizeTo( fClipWindow, rectClip );
  if (!fClipWindow->isVisible())
     wposbuf.show( fClipWindow );

  // If this tab control has buttons, size and position the button canvas
  // as well.
  if (fButtonCanvas)
  {
     // Position the canvas at the bottom of the tab control.  We will overlay
     // the bottom shadow border of the tab control and draw one later above
     // the set canvas.
     IRectangle rectButtons( IPoint( 0,
				     size.height() - fButtonCanvasHeight),
			     ISize( size.width(), fButtonCanvasHeight ));
     wposbuf.moveSizeTo( fButtonCanvas, rectButtons );

     if (!fButtonCanvas->isVisible())
	wposbuf.show( fButtonCanvas );
  }
  wposbuf.apply();

  // Right justify the buttons within the set canvas similar to the Windows
  // property sheets.
  if (fButtonCanvas)
     alignButtons();
}


/*------------------------------------------------------------------------------
| ITabControlData::alignButtons                                                |
|                                                                              |
| Align the tab control buttons within the button canvas.                      |
------------------------------------------------------------------------------*/
void ITabControlData::alignButtons ( )
{
  IMODTRACE_DEVELOP( "ITabControlData::alignButtons" );

  if (!fButtonCanvas)
     return;

  ISize currentMargin, newMargin, minSizeButtons;
  minSizeButtons = fButtonCanvas->minimumSize();

  currentMargin = fButtonCanvas->margin();

  newMargin.setWidth( MAX( (long)(fTabControl->size().width()
				  - minSizeButtons.width()
				  + 2*currentMargin.width()
				  - fDefButtonMargin.width()),
			   fDefButtonMargin.width() ));

  // Account for normal tab control border.
  newMargin -= ISize( 4, 0 );
  unsigned long ulExtStyle = GetWindowLong( fTabControl->handle(), GWL_EXSTYLE );
  if (ulExtStyle & WS_EX_CLIENTEDGE)
  {
     newMargin -= ISize( 4, 0 );
  }

  newMargin.setHeight( MAX( (long)(fButtonCanvasHeight
				   - minSizeButtons.height()) / 2,
			    fDefButtonMargin.height() ));
  fButtonCanvas->setMargin( newMargin );
}


/*------------------------------------------------------------------------------
| ITabControlData::adjustRect                                                  |
|                                                                              |
| Adjust the display rectangle.                                                |
------------------------------------------------------------------------------*/
void ITabControlData::adjustRect ( bool  bTabCtrlSize,
				   void* rect ) const
{
  IMODTRACE_DEVELOP( "ITabControlData::adjustRect" );

  if ( IRectangle( *(LPRECTL)rect ) != IRectangle() )
  {
     TabCtrl_AdjustRect( fTabControl->handle(), bTabCtrlSize, (LPRECTL)rect );
  }
}


/*------------------------------------------------------------------------------
| ITabControlData::buttonCanvasHeight                                          |
|                                                                              |
| Calculate the minimum button canvas height.                                  |
------------------------------------------------------------------------------*/
unsigned long ITabControlData :: buttonCanvasHeight ( ) const
{
   IMODTRACE_DEVELOP( "ITabControlData::buttonCanvasHeight" );

   // If we don't have a button canvas, then just return 0.
   if (!fButtonCanvas)
      return 0;

   // We need to cursor through all of the common and page buttons
   // using the max minimum size to determine the canvas height.
   unsigned long canvasHeight = 0;

   if (fCommonButtons && !fCommonButtons->isEmpty())
   {
      ITabControlButtonSequence::Cursor cursor( *fCommonButtons );
      forCursor( cursor )
      {
	 ITabControlButton *tcButton = fCommonButtons->elementAt( cursor );
	 IPushButton *button = tcButton->button( fTabControl );
	 if ((button->minimumSize().height() + 2*fDefButtonMargin.height())
	     > canvasHeight)
	 {
	    canvasHeight = button->minimumSize().height()
			   + 2*fDefButtonMargin.height();
	 }
      }
   }

   // Now check all of the page buttons.
   if (!fPageSequence->isEmpty())
   {
      ITabControlPageSequence::Cursor pageCursor( *fPageSequence );
      forCursor( pageCursor )
      {
	 ITabPage *tabPage = fPageSequence->elementAt( pageCursor );
	 ITabControlButtonSequence
	    *pageButtons = tabPage->tpd->fPageButtons;
	 if (pageButtons)
	 {
	    ITabControlButtonSequence::Cursor buttonCursor( *pageButtons );
	    forCursor( buttonCursor )
	    {
	       ITabControlButton *tcButton = pageButtons->elementAt( buttonCursor );
	       IPushButton *button = tcButton->button( fTabControl );
	       if ((button->minimumSize().height() + 2*fDefButtonMargin.height())
		   > canvasHeight)
	       {
		  canvasHeight = button->minimumSize().height()
				 + 2*fDefButtonMargin.height();
	       }
	    }
	 }
      }
   }
				
   // If we found no buttons, then the canvasHeight will still be zero.  In this
   // case, create a dummy push button and use its height to calculate the canvas
   // height.
   if (!canvasHeight && !IWindow::windowWithParent( 1, fButtonCanvas ))
   {
      IPushButton dummy( 1,
			 fButtonCanvas,
			 fButtonCanvas );
      canvasHeight = dummy.minimumSize().height() + 2*fDefButtonMargin.height();
   }

   return canvasHeight;
}


/*------------------------------------------------------------------------------
| ITabControlData::showButtons                                                 |
|                                                                              |
| Function to manage the common and page buttons in the set canvas.            |
------------------------------------------------------------------------------*/
void ITabControlData :: showButtons ( const ITabPage *prevPage,
				      const ITabPage *newPage )
{
  IMODTRACE_DEVELOP( "ITabControlData::showButtons" );

  ITabPageData *prevPageData = prevPage ? prevPage->tpd : 0;
  ITabPageData *newPageData = newPage->tpd;
  ISize prevMinSizeButtons = fButtonCanvas->minimumSize();
  ITabControlButton
     *prevDefaultButton = 0,
     *newDefaultButton = 0;

  // Prevent flashing in the set canvas.
  fButtonCanvas->disableUpdate();

  // If there were page buttons for the previous page, remove them from the
  // set canvas.
  if (prevPage && prevPageData->fPageButtons)
  {
     ITabControlButtonSequence *prevPageButtons = prevPageData->fPageButtons;
     ITabControlButtonSequence::Cursor cursor( *prevPageButtons );
     forCursor (cursor)
     {
	ITabControlButton *tcPageButton = prevPageButtons->elementAt( cursor );
	IPushButton *pageButton = tcPageButton->button( fTabControl );
	if (fButtonCanvas->isInLayout( pageButton ))
	   fButtonCanvas->remove( pageButton );
	if (pageButton->isGroup())
	   pageButton->disableGroup();
	if (pageButton->isTabStop())
	   pageButton->disableTabStop();

	prevDefaultButton = prevPageData->fDefaultButton;
     }
  }

  // If the new page has page buttons, they need to be added to the set canvas.
  if (newPageData->fPageButtons)
  {
     // If the previous page used common buttons and this page has page buttons,
     // remove the common buttons from the set canvas.
     if (prevPage && !prevPageData->fPageButtons && fCommonButtons)
     {
	ITabControlButtonSequence::Cursor cursor( *fCommonButtons );
	forCursor (cursor)
	{
	   ITabControlButton *tcCommonButton = fCommonButtons->elementAt( cursor );
	   IPushButton *commonButton = tcCommonButton->button( fTabControl );
	   if (fButtonCanvas->isInLayout( commonButton ))
	   {
	      fButtonCanvas->remove( commonButton );
	      if (commonButton->isGroup())
		 commonButton->disableGroup();
	      if (commonButton->isTabStop())
		 commonButton->disableTabStop();
	   }
	}

	prevDefaultButton = fDefaultButton;
     }

     // Now add the page buttons.
     ITabControlButtonSequence *newPageButtons = newPageData->fPageButtons;
     ITabControlButtonSequence::Cursor cursor( *newPageButtons );
     forCursor (cursor)
     {
	ITabControlButton *tcPageButton = newPageButtons->elementAt( cursor );
	IPushButton *pageButton = tcPageButton->button( fTabControl );
	fButtonCanvas->add( pageButton );

	// In order to ensure that the order of the buttons in the set
	// canvas is the same as the order of the buttons in the sequence,
	// force each button to be at the bottom of the z-order as it is
	// added to the canvas.  We must do this here since a button can
	// be in more than one button sequence.
	pageButton->positionBehindSiblings();
     }
     // Give the first button the group and tabstop styles.
     newPageButtons->firstElement()->button( fTabControl )->enableTabStop();
     newPageButtons->firstElement()->button( fTabControl )->enableGroup();

     newDefaultButton = newPageData->fDefaultButton;
  }

  // The previous page had page buttons, but the new page doesn't.  Need to
  // add the common buttons back to the canvas.
  if (!newPageData->fPageButtons &&
      (!prevPage || (prevPage && prevPageData->fPageButtons)) &&
      fCommonButtons)
  {
     ITabControlButtonSequence::Cursor cursor( *fCommonButtons );
     forCursor (cursor)
     {
	ITabControlButton *tcCommonButton = fCommonButtons->elementAt( cursor );
	IPushButton *commonButton = tcCommonButton->button( fTabControl );
	fButtonCanvas->add( commonButton );

	// Force the z-order of the buttons to reflect their position
	// in the button sequence.
	commonButton->positionBehindSiblings();
	
	// Give the first button the group and tabstop styles.
	fCommonButtons->firstElement()->button( fTabControl )->enableTabStop();
	fCommonButtons->firstElement()->button( fTabControl )->enableGroup();
     }

     newDefaultButton = fDefaultButton;
  }

  // If the button with emphasis is not on the current page, remove the
  // emphasis from it.
  IPushButton *emphasisButton = (IPushButton*)IWindow::windowWithHandle(
				   fButtonCanvas->defaultEmphasisButton() );
  if (emphasisButton && !fButtonCanvas->isInLayout( emphasisButton ))
     emphasisButton->removeEmphasis();

  // Change the default button for this page, if necessary.
  if (newDefaultButton)
  {
     if (prevDefaultButton != newDefaultButton)
     {
	newDefaultButton->button( fTabControl )->enableDefault();
     }
  }
  else if (prevDefaultButton)
  {
     prevDefaultButton->button( fTabControl )->disableDefault();
  }

  // Right justify the buttons within the set canvas similar to the Windows
  // property sheets.
  if (prevPage == 0 ||
      prevMinSizeButtons.width() != fButtonCanvas->minimumSize().width())
     alignButtons();

  // Refresh the button canvas.
  fButtonCanvas->enableUpdate();
  fButtonCanvas->refresh();
}
#endif // IC_WIN


#ifdef IC_PM
/*------------------------------------------------------------------------------
| ITabControlData::queryPage                                                   |
|                                                                              |
| Return a pointer to ITabPage object corresponding to the requested page.     |
------------------------------------------------------------------------------*/
ITabPage* ITabControlData :: queryTabPage ( unsigned short  pageOrder,
					    const ITabPage* referencePage ) const
{
  IMODTRACE_DEVELOP( "ITabControlData::queryPage" );

  ITabPage *tabPage = 0;

  unsigned long refPageId = 0;
  if (pageOrder == BKA_NEXT || pageOrder == BKA_PREV)
  {
     if (referencePage)
	refPageId = referencePage->tpd->fPageId;

     if (!refPageId)
     {
	ITHROWLIBRARYERROR( IC_INVALIDHANDLE,
			    IBaseErrorInfo::invalidParameter,
			    IException::recoverable );
     }
  }

  unsigned long
     tabPageId = fTabControl->sendEvent( BKM_QUERYPAGEID,
					 refPageId,
					 IEventParameter2( pageOrder, 0 ))
			     .asUnsignedLong();
  if ((long)tabPageId == BOOKERR_INVALID_PARAMETERS)
  {
     ITHROWGUIERROR( "BKM_QUERYPAGEID:BOOKERR_INVALID_PARAMETERS" );
  }
  if (tabPageId)
  {
     // Since we store a pointer to the tab page object as application
     // data, retrieve the application data for the tab page.
     tabPage = (ITabPage*)(void*)(fTabControl->sendEvent(
					BKM_QUERYPAGEDATA,
					tabPageId,
					0 ));
     if ((long)tabPage == BOOKERR_INVALID_PARAMETERS)
     {
	ITHROWGUIERROR( "BKM_QUERYPAGEDATA:BOOKERR_INVALID_PARAMETERS" );
     }
  }

  return tabPage;
}
#endif // IC_PM


/*------------------------------------------------------------------------------
| ITabControlDefaultHandler::ITabControlDefaultHandler                         |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlDefaultHandler :: ITabControlDefaultHandler ( )
#ifdef IC_WIN
  : fFocusSet( false )
  , fPreviousPage( 0 )
#endif // IC_WIN
{ }


/*------------------------------------------------------------------------------
| ITabControlDefaultHandler::dispatchHandlerEvent                              |
|                                                                              |
| Dispatch command functions for this handler.                                 |
------------------------------------------------------------------------------*/
bool ITabControlDefaultHandler::dispatchHandlerEvent ( IEvent &event )
{
  IMODTRACE_DEVELOP( "ITabControlDefaultHandler::dispatchHandlerEvent" );

  bool stopProcessingEvent = false;

#ifdef IC_PMWIN
  switch (event.eventId())
  {
    case WM_HELP:
    {
#ifdef IC_WIN
      // Get a pointer to the tab control object.
      ITabControl *tabControl = dynamic_cast<ITabControl*>(event.controlWindow());
      IWindow* parent = event.window()->parent();
      HELPINFO helpInfo=*((LPHELPINFO)(void*)(event.parameter2()));

      // If focus is on a tab control's tab, then set the panel id for the help.
      if (tabControl && (event.handle() == helpInfo.hItemHandle))
      {
	 long pageIndex = TabCtrl_GetCurFocus( tabControl->handle() );
	 if (pageIndex != -1)
	 {
	    ITabPage *focusPage = tabControl->tcd->
		  fPageSequence->elementAtPosition( pageIndex + 1 );
	    unsigned long panelId = focusPage->helpId();
	    if (panelId)
	    {
	       IHelpWindow *helpWindow = IHelpWindow::helpWindow( tabControl );
	       if (helpWindow)
	       {
		  helpWindow->show( IResourceId( panelId ));
		  stopProcessingEvent = true;
	       }
	    }
	 }
      }
      else
      {
	 if (parent)
	 {
	    // The user must have requested contextual help from a
	    // page window.  We will avoid the tab control's processing
	    // of WM_HELP, which is geared for its tabs.  If the help
	    // request gets here, the page window must not be a frame
	    // (it is probably a canvas).  By routing the requests to
	    // the tab control's parent, these contextual helps can be
	    // resolved through the help subtable of the tab control's
	    // frame window.  Please note that we must send the help message
	    // rather than post it since parameter2 is a pointer to the
	    // HELPINFO structure.
	    parent->handle().sendEvent( event.eventId(),
					event.parameter1(),
					event.parameter2() );

	    stopProcessingEvent = true;
	 }
      }
#endif // IC_WIN

#ifdef IC_PM
      // Get a pointer to the tab control object.
      ITabControl *tabControl = dynamic_cast<ITabControl*>(event.dispatchingWindow());
      if (tabControl && tabControl->hasFocus())
      {
	 // Check to see whether the event was generated via a pushbutton.
	 if (event.parameter2().number1() == CMDSRC_PUSHBUTTON)
	 {
	    unsigned long
	       ulPageId = tabControl->sendEvent( BKM_QUERYPAGEID,
						 0,
						 IEventParameter2( BKA_TOP, 0 ))
				     .asUnsignedLong();
	    if (ulPageId == 0 || (long)ulPageId == BOOKERR_INVALID_PARAMETERS)
	    {
	       ITHROWGUIERROR( "BKM_QUERYPAGEID" );
	    }

	    // Generate an event for the tab control.
	    tabControl->parent()->postEvent( WM_CONTROL,
					     IEventParameter1( (unsigned short)tabControl->id(),
							       BKN_HELP ),
					     ulPageId );
	    stopProcessingEvent = true;
	 }
      }
      else
      {
	 IWindow* parent = event.window()->parent();
	 // Focus is not on the tab control (and hence, not a tab).
	 if (parent)
	 {
	    // The user must have requested contextual help from a
	    // page window.  We will avoid the notebook's processing
	    // of WM_HELP, which is geared for its tabs.  If the help
	    // request gets here, the page window must not be a frame
	    // (it is probably a canvas).  By routing the requests to
	    // the notebook's parent, these contextual helps can be
	    // resolved through the help subtable of the notebook's
	    // frame window.
	    parent->handle().postEvent( event.eventId(),
					event.parameter1(),
					event.parameter2() );
	    stopProcessingEvent = true;
	 }
      }
#endif //IC_PM
      break;
    } // WM_HELP

    case WM_COMMAND:
    {
      ITabControl *tabControl = dynamic_cast<ITabControl*>(event.dispatchingWindow());
      ICommandEvent cmdEvent( event );

      // Route any unprocessed WM_COMMAND messages to the tab control's owner.
      // This allows handling of button events at the client/frame level.
      // However, if the tab control is the client window, we must only route
      // the command event to the frame window once since it will route it right
      // back to us if no handler processes it at the frame window level.
      IWindow *pwndOwner = (tabControl == 0) ? 0 : tabControl->owner();
      if (pwndOwner &&
	  pwndOwner->handle() != IWindow::desktopWindow()->handle() &&
	  pwndOwner->handle() != IWindow::objectWindow()->handle())
      {
	 bool isClient = tabControl->id() == FID_CLIENT &&
			    pwndOwner->isFrameWindow();
	 if (!isClient || !cmdEvent.isFromFrame())
	 {
#ifdef IC_WIN
	    unsigned long parm1 = event.parameter1();
	    if (isClient)
	    {
	       // The tab control is the client, but the command event is not from
	       // the frame.  Send it on to the frame window, but make sure we know
	       // not to process it again.
	       parm1 |= 0x80000000ul;
	    }
	    event.setResult( pwndOwner->sendEvent( event.eventId(),
						   IEventParameter1( parm1 ),
						   event.parameter2() ));
#endif // IC_WIN

#ifdef IC_PM
	    unsigned long parm2 = event.parameter2();
	    if (isClient)
	    {
	       // The tab control is the client, but the command event is not from
	       // the frame.  Send it on to the frame window, but make sure we know
	       // not to process it again.
	       parm2 |= 0x8000ul;
	    }
	    event.setResult( pwndOwner->sendEvent( event.eventId(),
						   event.parameter1(),
						   parm2 ));
#endif // IC_PM
	    stopProcessingEvent = true;
	 }
      }
      break;
    }

#ifdef IC_WIN
    case WM_SIZE:
    {
      IWindowHandle hwndControl( event.controlHandle() );
      if (hwndControl)
      {
	 // Obtain pointer to tab control object.
	 ITabControl *tabControl =
	       dynamic_cast<ITabControl*>(IWindow::windowWithHandle( hwndControl ));

	 ISize newSize( event.parameter2().lowNumber(),
			event.parameter2().highNumber() );

	 tabControl->tcd->tabControlResize( newSize );
      }
      break;
    } // WM_SIZE

    case WM_CHAR:
    {
      // Handle tabbing for the tab control.

      IKeyboardEvent keyEvent( event );
      if (keyEvent.isVirtual())
      {
	 IKeyboardEvent::VirtualKey vkKey = keyEvent.virtualKey();

	 if ((vkKey == IKeyboardEvent::tab ||
	      vkKey == IKeyboardEvent::backTab) &&
	     !keyEvent.isUpTransition())
	    stopProcessingEvent = handleTab( keyEvent );
      }
      break;
    }

    case WM_SETFOCUS:
    {
      // If the tab control has received focus, check to determine whether
      // it was traversed to via a back tab.  If so, determine whether
      // the focus was previously on a tab control button.  If so, then
      // set the focus on the page window if possible.  If the focus was on
      // a control within the top page window, set the focus on the tab
      // control.  If focus was previously outside the tab control, set
      // the focus on the tab control buttons or page window.

      unsigned long ulKeyState =
	 IGETKEYSTATE( IWindow::desktopWindow()->handle(), VK_TAB ) &
	 IGETKEYSTATE( IWindow::desktopWindow()->handle(), VK_SHIFT );

      if (ulKeyState & 0x8000)
      {
	 if (!fFocusSet)
	    stopProcessingEvent = gotFocus( event );
	 else
	    fFocusSet = false;
      }

      break;
    }

    case WM_SYSKEYDOWN:
    {
      // Only process tab control keyboard events of interest.
      IWindowHandle hwndControl( event.controlHandle() );
      if (hwndControl)
      {
	 // Obtain pointer to tab control object.
	 ITabControl *tabControl =
	       dynamic_cast<ITabControl*>(IWindow::windowWithHandle( hwndControl ));

	 IKeyboardEvent keyevt( event );
	 if (!keyevt.isVirtual() || !keyevt.isAltDown() ||
	     keyevt.isShiftDown() || keyevt.isCtrlDown())
	    break;

	 // Alt+Down:  set the focus to the application's page window.
	 if (keyevt.virtualKey() == IKeyboardEvent::down)
	 {
	    // Get the top page if it exists.
	    ITabPage *topPage = tabControl->topPage();

	    if (topPage)
	    {
	       // Get the application page window if it exists and set the
	       // focus to it.
	       IWindow* pageWindow = topPage->pageWindow();
	       if (pageWindow)
	       {
		 pageWindow->setFocus();
		 stopProcessingEvent = true;
	       }
	    }
	 }

	 // Alt+PageUp:  bring the previous page to the top
	 else if (keyevt.virtualKey() == IKeyboardEvent::pageUp)
	 {
	    ITabPage *topPage = tabControl->topPage();
	    if (topPage)
	    {
	       ITabPage *previousPage = tabControl->previousPage( topPage );
	       if (previousPage)
	       {
		  tabControl->setTopPage( previousPage );
		  stopProcessingEvent = true;
	       }
	    }
	 }

	 // Alt+PageDown:  bring the next page to the top
	 else if (keyevt.virtualKey() == IKeyboardEvent::pageDown)
	 {
	    ITabPage *topPage = tabControl->topPage();
	    if (topPage)
	    {
	       ITabPage *nextPage = tabControl->nextPage( topPage );
	       if (nextPage)
	       {
		  tabControl->setTopPage( nextPage );
		  stopProcessingEvent = true;
	       }
	    }
	 }
      }
      break;
    } // WM_SYSKEYDOWN

    case WM_NOTIFY:
    {
      // Process the tab control's notifications for selection events.
      LPNMHDR lpnmhdr = (LPNMHDR)event.parameter2().asUnsignedLong();

      IWindowHandle hwndControl( lpnmhdr->hwndFrom );

      if (hwndControl)
      {
	 if (lpnmhdr->code == TCN_SELCHANGE ||
	     lpnmhdr->code == TCN_SELCHANGING)
	 {
	    ITabControl *tabControl =
		  dynamic_cast<ITabControl*>(IWindow::windowWithHandle( hwndControl ));
	    ITabControlData *tabControlData = tabControl->tcd;

	    if (lpnmhdr->code == TCN_SELCHANGE)
	    {
	       ITabPage *newPage = tabControl->topPage();

	       // The tab selection has already occurred and the new tab is
	       // now the top page in the tab control.  Do our required
	       // processing and issue the selection notification.
	       unsigned long newPageIndex,
			     previousPageIndex;
	       previousPageIndex = tabControlData->tabPageIndex( fPreviousPage );
	       newPageIndex = tabControlData->tabPageIndex( newPage );
	       tabControlData->processTabSelect( previousPageIndex,
						 newPageIndex );

	       tabControl->notifyObservers( INotificationEvent(
					       ITabControl::pageSelectId,
					       *tabControl ));
	    }
	    else
	    {
	       // The tab selection for a new tab is pending.
	       // Call the validate function in the tab page object for the top page.
	       // This handler will be called after any application defined handlers.
	       ITabPage *topPage = tabControl->topPage();
	       if (!topPage->validatePageWindow())
	       {
		  event.setResult( true );
		  stopProcessingEvent = true;
	       }
	       else
	       {
		  // Cache the current top page to be used as the previous page when
		  // the TCN_SELCHANGE event comes in.
		  fPreviousPage = topPage;
		  event.setResult( false );
	       }
	    }
	 }
      }
      break;
    } //WM_NOTIFY

    case WM_SYSCOMMAND:
    {
#ifdef IC_NOTYET
// Is this workaround still needed?
#endif
      // Process the Windows tab control's WM_SYSCOMMAND/SC_CLOSE event
      // for Windows95 to work around a Windows95 bug.
      IWindowHandle hwndControl( event.controlHandle() );
      if (!(hwndControl && IPlatform::isWin9x()))
	 break;

      if ((((unsigned long)event.parameter1()) & 0xFFF0) == SC_CLOSE)
      {
	 ITabControl *tabControl =
	       dynamic_cast<ITabControl*>(IWindow::windowWithHandle( hwndControl ));

	 // Return true to circumvent Windows95 bug.
	 if (tabControl)
	    return true;
      }
      break;
    } //WM_SYSCOMMAND

    case IC_UM_TCM_LAYOUT:
    {
      // The minimum size of the button canvas has changed.  Handle the
      // layout if necessary.
      ITabControl *tabControl = dynamic_cast<ITabControl*>(event.controlWindow());
      ITabControlData *tabControlData = tabControl->tcd;

      unsigned long newCanvasHeight = tabControlData->buttonCanvasHeight();
      if (newCanvasHeight != tabControlData->fButtonCanvasHeight)
      {
	 tabControlData->fButtonCanvasHeight = newCanvasHeight;
	 tabControlData->layoutChildren( tabControl->size() );
      }
      event.setResult( true );
      stopProcessingEvent = true;

      break;
    } // IC_UM_TCM_LAYOUT
#endif // IC_WIN

    case IC_UM_TCN_BTNDELETED:
    {
      // One of the tab control buttons was destroyed.  Remove it from
      // all of the button sequences.  We should also re-layout the
      // buttons for the top page.
      ITabControl *tabControl = dynamic_cast<ITabControl*>(event.controlWindow());
      ITabControlData *tabControlData = tabControl->tcd;
#ifdef IC_PM
	 bool resetCommonButtons = false;
#endif // IC_PM

      if (tabControlData->fCommonButtons)
      {
	 ITabControlButtonSequence *commonButtons = tabControlData->fCommonButtons;
	 ITabControlButtonSequence::Cursor cursor( *commonButtons );
	 cursor.setToFirst();
	 while (cursor.isValid())
	 {
	    ITabControlButton
	       *tcCommonButton = commonButtons->elementAt( cursor );
	    if (tcCommonButton->button( tabControl )->
		      isLayoutDistorted( IWindow::windowDestroyed ))
	    {
	       commonButtons->removeAt( cursor );
	       if (tabControlData->fDefaultButton == tcCommonButton)
		  tabControlData->fDefaultButton = 0;
#ifdef IC_WIN
	       if (!tabControl->topPage()->tpd->fPageButtons)
		  tabControlData->alignButtons();
#else
	       resetCommonButtons = true;
#endif // IC_WIN
	    }
	    else
	       cursor.setToNext();
	 }
      }

      if (!tabControlData->fPageSequence->isEmpty())
      {
	 ITabControlPageSequence *pageSequence = tabControlData->fPageSequence;
	 ITabControlPageSequence::Cursor pageCursor( *pageSequence );
	 forCursor( pageCursor )
	 {
	    ITabPage *tabPage = pageSequence->elementAt( pageCursor );
	    ITabControlButtonSequence
	     *pageButtons = tabPage->tpd->fPageButtons;
	    if (pageButtons)
	    {
	       bool resetPageButtons = false;
	       ITabControlButtonSequence::Cursor cursor( *pageButtons );
	       cursor.setToFirst();
	       while (cursor.isValid())
	       {
		  ITabControlButton
		     *tcPageButton = pageButtons->elementAt( cursor );
		  if (tcPageButton->button( tabControl )->
			    isLayoutDistorted( IWindow::windowDestroyed ))
		  {
		     pageButtons->removeAt( cursor );
		     if (tabPage->tpd->fDefaultButton == tcPageButton)
			tabPage->tpd->fDefaultButton = 0;
#ifdef IC_WIN
		     if (tabControl->topPage() == tabPage)
#endif // IC_WIN
			resetPageButtons = true;
		  }
		  else
		     cursor.setToNext();
	       }

	       // If we've removed any of the buttons for the page, reflect the
	       // change in the tab control.
	       if (resetPageButtons)
#ifdef IC_WIN
		  tabControlData->alignButtons();
#else
		  tabPage->tpd->setButtons( pageButtons );
#endif // IC_WIN
	    }
#ifdef IC_PM
	    else if (resetCommonButtons)
	       tabPage->tpd->setButtons( tabControlData->fCommonButtons );
#endif // IC_PM
	 }
      }

      event.setResult( true );
      stopProcessingEvent = true;

      break;
    } // IC_UM_TCN_BTNDELETED

#ifdef IC_PM
    case WM_CONTROL:
    {
      IControlEvent ctlEvent( event );
      unsigned long msgId  = ctlEvent.parameter1().highNumber();
      switch (msgId)
      {
	case BKN_PAGESELECTEDPENDING:
	{
	  ITabControl *tabControl =
		 dynamic_cast<ITabControl*>(ctlEvent.controlWindow());
	  if (tabControl)
	  {
	     PAGESELECTNOTIFY* pNotify = (PAGESELECTNOTIFY*)(void*)ctlEvent.parameter2();
	
	     // If the tab for the current page was selected, ignore this notification.
	     if (pNotify->ulPageIdNew == pNotify->ulPageIdCur)
	     {
		stopProcessingEvent = true;
		break;
	     }

	     // The tab selection for a new tab is pending.
	     // Call the validate function in the tab page object for the top page.
	     ITabPage *currentPage = 0;
	     if (pNotify->ulPageIdCur)
	     {
		currentPage = (ITabPage*)(void*)tabControl->sendEvent(
						      BKM_QUERYPAGEDATA,
						      pNotify->ulPageIdCur,
						      0 );
	     }

	     // Determine whether one of the controls on the previous page has the
	     // input focus.  We will use this later to determine where focus should be
	     // after the page turn.
	     fFocusPage = false;
	     if (currentPage && currentPage->tpd->fPageWindow)
	     {
		IWindowHandle hwndCurrentPage = currentPage->tpd->fPageWindow->handle();
		for (IWindowHandle hwnd = IQUERYFOCUS( IWindow::desktopWindow()->handle() );
		     !fFocusPage && hwnd;
		     hwnd = IPARENTOF( hwnd ))
		{
		   if (hwnd == hwndCurrentPage)
		      fFocusPage = true;
		}
	     }

	     if (currentPage && !currentPage->validatePageWindow())
	     {
		if (fFocusPage)
		   currentPage->tpd->fPageWindow->setFocus();
		pNotify->ulPageIdNew = 0;
		event.setResult( false );
		stopProcessingEvent = true;
		break;
	     }
	  }
	  break;
	} // BKN_PAGESELECTEDPENDING

	case BKN_PAGESELECTED:
	{
	  ITabControl *tabControl =
		 dynamic_cast<ITabControl*>(ctlEvent.controlWindow());
	  if (tabControl)
	  {
	     PAGESELECTNOTIFY* pNotify = (PAGESELECTNOTIFY*)(void*)ctlEvent.parameter2();
	
	     // Get the new page so we can create the page window if necessary.
	     ITabPage* newPage = 0;
	     if (pNotify->ulPageIdNew)
	     {
		newPage = (ITabPage*)(void*)tabControl->sendEvent( BKM_QUERYPAGEDATA,
								   pNotify->ulPageIdNew,
								   0 );
	     }

	     if (!newPage)
	     {
		// We have a page selection occurring prior to the page data being set.
		// This is the case when the page added is the top page in the control.
		// Cursor through all of the tab pages to find this one.
		ITabControlPageSequence *fPageSequence = tabControl->fTabControlData->fPageSequence;
		ITabControlPageSequence::Cursor cursor( *fPageSequence );
		for (cursor.setToFirst(); cursor.isValid() && !newPage; cursor.setToNext())
		{
		   ITabPage *currentPage = fPageSequence->elementAt( cursor );
		   if (currentPage->fTabPageData->fPageId == 0)
		   {
		      newPage = currentPage;
		      newPage->fTabPageData->fPageId = pNotify->ulPageIdNew;

		      // Store a pointer to the ITabPage object as the user data.
		      if (!tabControl->sendEvent( BKM_SETPAGEDATA,
						  pNotify->ulPageIdNew,
						  newPage ).asUnsignedLong())
		      {
			 ITHROWGUIERROR( "BKM_SETPAGEDATA" );
		      }
		   }
		}
	     }

	     if (newPage)
	     {
		IWindow* newPageWindow = newPage->tpd->fPageWindow;
		if (!newPageWindow)
		{
		   // If the new top page does not have an application page window set,
		   // invoke the virtual function to allow derived class to set
		   // it now.
		   newPageWindow = newPage->createPageWindow();
		   if (newPageWindow)
		   {
		      newPage->setPageWindow( newPageWindow );
		   }
		}
		else
		{
		   // If the new top page already had an application page window set,
		   // invoke the virtual function to allow derived class to update
		   // the page window contents in response to the selection.
		   newPage->updatePageWindow();
		}

		// Set the focus on the new page if it was on the previous page.
		if (newPageWindow && fFocusPage)
		   newPageWindow->setFocus();

		// If the button with emphasis is not on the current page, remove the
		// emphasis from it.
		IPushButton *emphasisButton = (IPushButton*)IWindow::windowWithHandle(
						 tabControl->defaultEmphasisButton() );
		if (emphasisButton)
		{
		   ITabControlButtonSequence *newPageButtons = newPage->tpd->fPageButtons;
		   if (!newPageButtons)
		      newPageButtons = tabControl->tcd->fCommonButtons;

		   // Check whether the buttons displayed for this new page will
		   // include the current emphasis button.
		   if (!newPageButtons)
		   {
		      emphasisButton->removeEmphasis();
		      emphasisButton = 0;
		   }
		   else
		   {
		      ITabControlButtonSequence::Cursor cursor( *newPageButtons );
		      bool emphasisButtonShown = false;
		      forCursor (cursor)
		      {
			 ITabControlButton *tcButton = newPageButtons->elementAt( cursor );
			 if (tcButton->button( tabControl ) == emphasisButton)
			 {
			    emphasisButtonShown = true;
			    cursor.setToLast();
			 }
		      }
		      if (!emphasisButtonShown)
		      {
			 emphasisButton->removeEmphasis();
			 emphasisButton = 0;
		      }
		   }
		}

		// Set the new default button if necessary.
		if (emphasisButton && !emphasisButton->isDefault())
		   emphasisButton->enableDefault();
		else if (!emphasisButton)
		{
		   ITabControlButton *prevDefaultButton=0,
				     *newDefaultButton=0;

		   // Get the previous page.
		   ITabPage* prevPage = 0;
		   if (pNotify->ulPageIdCur)
		   {
		      prevPage = (ITabPage*)(void*)tabControl->sendEvent( BKM_QUERYPAGEDATA,
									  pNotify->ulPageIdCur,
									  0 );

		      prevDefaultButton = (prevPage->tpd->fDefaultButton) ?
					     prevPage->tpd->fDefaultButton :
					     tabControl->tcd->fDefaultButton;
		   }

		   newDefaultButton = newPage->tpd->fDefaultButton;
		   if (!newDefaultButton && !newPage->tpd->fPageButtons)
		      newDefaultButton = tabControl->tcd->fDefaultButton;
		
		   if (newDefaultButton)
		   {
		      if (prevDefaultButton != newDefaultButton)
			 newDefaultButton->button( tabControl )->enableDefault();
		   }
		   else if (prevDefaultButton)
		   {
		      prevDefaultButton->button( tabControl )->disableDefault();
		   }
		}
	     }
	  }
	  break;
	} // BKN_PAGESELECTED

	case BKN_HELP:
	{
	  ITabControl *tabControl =
		 dynamic_cast<ITabControl*>(ctlEvent.dispatchingWindow());
	  if (tabControl)
	  {
	     // Since we store a pointer to the tab page object as application
	     // data, retrieve the application data for the tab page.
	     ITabPage *tabPage = (ITabPage*)(void*)(tabControl->sendEvent(
						    BKM_QUERYPAGEDATA,
						    event.parameter2(),
						    0 ));
	     if ((long)tabPage == BOOKERR_INVALID_PARAMETERS)
	     {
		ITHROWGUIERROR( "BKM_QUERYPAGEDATA:BOOKERR_INVALID_PARAMETERS" );
	     }

	     // Now check for a helpId stored in the tab page object and show the
	     // help panel.
	     unsigned long panelId = tabPage->helpId();
	     if (panelId)
	     {
		IHelpWindow *helpWindow = IHelpWindow::helpWindow( tabControl );
		if (helpWindow)
		{
		   helpWindow->show( IResourceId( panelId ));
		   stopProcessingEvent = true;
		}
	     }
	  }
	  break;
	} // BKN_HELP

	default:
	  break;
      } // WM_CONTROL
      break;
    }

    // Store the colors that are being set and then let this message
    // pass on to the notebook control.
    case BKM_SETNOTEBOOKCOLORS:
    {
      ITabControl *tabControl = dynamic_cast<ITabControl*>(event.controlWindow());
      if (tabControl)
      {
	 switch (event.parameter2())
	 {
	   case BKA_BACKGROUNDPAGECOLOR:
	   {
	      tabControl->tcd->fPageBkgrndColorSet = true;
	      unsigned long color = event.parameter1();
	      unsigned char blue  = (unsigned char)(color);
	      unsigned char red   = (unsigned char)(color >> 16);
	      unsigned char green = (unsigned char)(color >> 8);
	      tabControl->tcd->fPageBkgrndColor =
		    IColor( red, green, blue );
	      break;
	   }

	   default:
	      break;
	 }
      }
      break;
    } // BKM_SETNOTEBOOKCOLORS
#endif // IC_PM

    case IC_UM_IS_AGGREGATE_CTRL:
    {
      event.setResult(true);
      stopProcessingEvent = true;
      break;
    } // IC_UM_IS_AGGREGATE_CTRL

    default:
      break;
  }
#endif // IC_PMWIN

  return stopProcessingEvent;
}


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ITabControlDefaultHandler::handleTab                                         |
|                                                                              |
| Process tabbing for the tab control.                                         |
------------------------------------------------------------------------------*/
bool ITabControlDefaultHandler::handleTab ( IKeyboardEvent& keyEvent )
{
  bool stopProcessingEvent = false;

  IKeyboardEvent::VirtualKey vkKey = keyEvent.virtualKey();
  IWindowHandle hwndFocus =
     IQUERYFOCUS(IWindow::desktopWindow()->handle());
  ITabControl *tabControl = (ITabControl*)(keyEvent.dispatchingWindow());
  IWindowHandle hwndTabControl = tabControl->handle();
  ITabPage *topPage = tabControl->topPage();
  IWindow *pageWindow = (topPage) ? topPage->pageWindow() : 0;

  // If focus is on the tab control itself, then set focus to the
  // page window for a forward tab.
  if (hwndFocus == hwndTabControl)
  {
     if (vkKey == IKeyboardEvent::tab)
     {
	if (pageWindow)
	{
	   ICanvas *canvasPage = dynamic_cast<ICanvas*>(pageWindow);
	   if (canvasPage)
	   {
	      if (canvasPage->isTabStop())
	      {
		 canvasPage->setFocus();
		 stopProcessingEvent = true;
	      }
	   }
	   else
	   {
	      IFrameWindow *framePage = dynamic_cast<IFrameWindow*>(pageWindow);
	      if (framePage)
	      {
		 IWindowHandle
		    hwndNextControl =
			  GetNextDlgTabItem( framePage->handle(),
					     NULL,
					     vkKey == IKeyboardEvent::backTab );
		 if (hwndNextControl)
		 {
		    framePage->postEvent( WM_NEXTDLGCTL,
					  IEventParameter1( hwndNextControl ),
					  IEventParameter2( true ));
		    stopProcessingEvent = true;
		 }
	      }
	      else
	      {
		 if (pageWindow->isTabStop())
		 {
		    pageWindow->setFocus();
		    stopProcessingEvent = true;
		 }
	      }
	   }
	}

	if (!stopProcessingEvent)
	{
	   if (tabControl->buttonParent() &&
	       tabControl->buttonParent()->isTabStop())
	   {
	      tabControl->buttonParent()->setFocus();
	      stopProcessingEvent = true;
	   }
	}
     }
  }
  else if (IPARENTOF( hwndFocus ) == tabControl->buttonParent()->handle())
  {
     // The focus is on one of the tab control buttons.  Set the focus
     // on the top page window or the tab control.
     if (vkKey == IKeyboardEvent::backTab)
     {
	if (pageWindow)
	{
	   ICanvas *canvasPage = dynamic_cast<ICanvas*>(pageWindow);
	   if (canvasPage && canvasPage->isTabStop())
	   {
	      canvasPage->setFocus();
	      stopProcessingEvent = true;
	   }
	   else
	   {
	      IFrameWindow *framePage = dynamic_cast<IFrameWindow*>(pageWindow);
	      if (framePage)
	      {
		 IWindowHandle
		    hwndNextControl =
			  GetNextDlgTabItem( framePage->handle(),
					     NULL,
					     vkKey == IKeyboardEvent::backTab );
		 if (hwndNextControl)
		 {
		    framePage->postEvent( WM_NEXTDLGCTL,
					  IEventParameter1( hwndNextControl ),
					  IEventParameter2( true ));
		    stopProcessingEvent = true;
		 }
	      }
	      else if (pageWindow->isTabStop())
	      {
		 pageWindow->setFocus();
		 stopProcessingEvent = true;
	      }
	   }
	}
	if (!stopProcessingEvent)
	{
	   // We want to back tab to the tab control since the focus
	   // could not be set on the page window.  Indicate that
	   // we are the ones who set the focus on the tab control.
	   fFocusSet = true;
	   tabControl->setFocus();
	   stopProcessingEvent = true;
	}
     }
  }
  else
  {
     // The focus must be on a descendant of the page window.  Set the focus
     // to either the button canvas or the tab control.
     if (vkKey == IKeyboardEvent::tab &&
	 tabControl->buttonParent() &&
	 tabControl->buttonParent()->isTabStop())
     {
	tabControl->buttonParent()->setFocus();
	stopProcessingEvent = true;
     }
     else
     {
	if (vkKey == IKeyboardEvent::backTab)
	   fFocusSet = true;
	tabControl->setFocus();
	stopProcessingEvent = true;
     }
  }

  return stopProcessingEvent;
}


/*------------------------------------------------------------------------------
| ITabControlDefaultHandler::gotFocus                                          |
|                                                                              |
| Handle a focus request for the tab control.  We only handle this focus       |
| event during a back tab into the tab control.                                |
------------------------------------------------------------------------------*/
bool ITabControlDefaultHandler::gotFocus ( IEvent& event )
{
  bool stopProcessingEvent = false;
  ITabControl *tabControl = (ITabControl*)(event.dispatchingWindow());
  IWindowHandle hwndTabControl = tabControl->handle();
  ITabPage *topPage = tabControl->topPage();
  IWindow *pageWindow = (topPage) ? topPage->pageWindow() : 0;

  // If there is a tabbable button canvas, tab into that.
  if (tabControl->buttonParent() &&
      tabControl->buttonParent()->isTabStop())
  {
     tabControl->buttonParent()->postEvent( IC_UM_CANVAS_SETFOCUS, 0, 0 );
     stopProcessingEvent = true;
  }

  if (!stopProcessingEvent)
  {
     // If the top page window is either a tabbable control or contains a tabbable
     // control, traverse into the page window.
     if (pageWindow)
     {
	ICanvas *canvasPage = dynamic_cast<ICanvas*>(pageWindow);
	if (canvasPage)
	{
	   if (canvasPage->isTabStop())
	   {
	      canvasPage->postEvent( IC_UM_CANVAS_SETFOCUS, 0, 0 );
	      stopProcessingEvent = true;
	   }
	}
	else
	{
	   if (pageWindow->isFrameWindow())
	   {
	      // Note:  There is a problem with the GetNextDlgTabItem returning
	      //        NULL when the direction flag is TRUE (search previous)
	      //        and the starting handle is NULL.  So we just check for
	      //        the existence of any tabbable control in the dialog and
	      //        let the frame window determine which control gets focus.
	      IFrameWindow *framePage = dynamic_cast<IFrameWindow*>(pageWindow);
	      IWindowHandle
		 hwndNextControl = GetNextDlgTabItem( framePage->handle(),
						      NULL,
						      FALSE );
	      if (hwndNextControl)
	      {
		 framePage->postEvent( WM_NEXTDLGCTL,
				       IEventParameter1( hwndNextControl ),
				       IEventParameter2( true ));
		 stopProcessingEvent = true;
	      }
	   }
	   else
	   {
	      if (pageWindow->isTabStop())
	      {
		 pageWindow->setFocus();
		 stopProcessingEvent = true;
	      }
	   }
	}
     }
  }
  return stopProcessingEvent;
}
#endif // IC_WIN


/*------------------------------------------------------------------------------
| ITabControlButton::ITabControlButton                                         |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlButton :: ITabControlButton ( )
  : fApplicationButton( 0 )
  , fStandardButton( (ITabControl::EButton)0 )
{ }


/*------------------------------------------------------------------------------
| ITabControlButton::~ITabControlButton                                        |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlButton :: ~ITabControlButton ( )
{ }


/*------------------------------------------------------------------------------
| ITabControlButton::button                                                    |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
IPushButton* ITabControlButton :: button ( const ITabControl *tabControl ) const
{
  if (fApplicationButton)
     return fApplicationButton;

  return tabControl->button( fStandardButton );
}


/*------------------------------------------------------------------------------
| ITabControlButtonSequence::ITabControlButtonSequence                         |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlButtonSequence :: ITabControlButtonSequence ( )
  : IEqualitySequence< ITabControlButton* >( 10 )
{ }


/*------------------------------------------------------------------------------
| ITabControlButtonSequence::~ITabControlButtonSequence                        |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlButtonSequence :: ~ITabControlButtonSequence ( )
{ }


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ITabControlButtonCanvas::ITabControlButtonCanvas                             |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlButtonCanvas :: ITabControlButtonCanvas ( ITabControl* tabControl )
  : ISetCanvas( ID_TABBUTTONS,
		tabControl,
		tabControl,
		IRectangle(),
		ISetCanvas::packExpanded
		 | ISetCanvas::horizontalDecks
		 | ISetCanvas::explicitAddsNeeded
		 | IWindow::visible )

{
  // Set the background color of the set canvas to the dialog background.
  setBackgroundColor( IGUIColor::dialogBgnd );
}


/*------------------------------------------------------------------------------
| ITabControlButtonCanvas::~ITabControlButtonCanvas                            |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlButtonCanvas :: ~ITabControlButtonCanvas ( )
{ }


/*------------------------------------------------------------------------------
| ITabControlButtonCanvas::setLayoutDistorted                                  |
| Handle any layout changes.                                                   |
------------------------------------------------------------------------------*/
ITabControlButtonCanvas& ITabControlButtonCanvas :: setLayoutDistorted (
					      unsigned long layoutAttributesOn,
					      unsigned long layoutAttributesOff )
{
  if ((layoutAttributesOn & IWindow::childMinimumSizeChanged) &&
      (~layoutAttributesOn & IWindow::childWindowCreated))
  {
     // We need to relayout the button canvas and recalculate its height whenever
     // the minimum size of one of the tab control buttons changes (font change).
     // Note that we need to look for changes in buttons that are not currently
     // added to the button canvas; thus, it isn't sufficient to look for a
     // minimumSizeChanged on the set canvas.  Another reason we can't look for
     // minimumSizeChanged on the set canvas is that we will add/remove buttons
     // from the layout on page turns.  We don't want to recalculate the canvas
     // size on page turns.
     parent()->handle().sendEvent( IC_UM_TCM_LAYOUT,
				   IEventParameter1( 0 ),
				   IEventParameter2( 0 ));
  }
  else if (layoutAttributesOn & IWindow::childWindowDestroyed)
  {
     // One of our children may has been destroyed.  We need to remove any
     // deleted button from the button sequences.
     parent()->handle().sendEvent( IC_UM_TCN_BTNDELETED,
				   IEventParameter1( 0 ),
				   IEventParameter2( 0 ));
  }

  Inherited::setLayoutDistorted( layoutAttributesOn, layoutAttributesOff );

  return *this;
}
#endif // IC_WIN


/*------------------------------------------------------------------------------
| ITabPageData::ITabPageData                                                   |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabPageData :: ITabPageData ( ITabPage *tabPage )
  : fTabControl( 0 )
  , fTabPage( tabPage )
  , fPageWindow( 0 )
  , fHelpId( 0 )
  , fPageButtons( 0 )
  , fDefaultButton( 0 )
  , fStyle( 0 )
  , fTabBkgrndColorSet( false )
  , fTabBkgrndColor( IColor::notebookPageBgnd )
#ifdef IC_WIN
  , fAutoPageSize( false )
  , fPageWindowParent( 0 )
#endif // IC_WIN
#ifdef IC_PM
  , fPageId( 0 )
#endif // IC_PM
{ }


/*------------------------------------------------------------------------------
| ITabPageData::~ITabPageData                                                  |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabPageData :: ~ITabPageData ( )
{
  if (fPageButtons)
  {
     while (!fPageButtons->isEmpty())
     {
	delete fPageButtons->firstElement();
	fPageButtons->removeFirst();
     }
  }
}


/*------------------------------------------------------------------------------
| ITabControlPageSequence::ITabControlPageSequence                             |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlPageSequence :: ITabControlPageSequence ( )
  : IEqualitySequence< ITabPage* >( 20 )
{ }

/*------------------------------------------------------------------------------
| ITabControlPageSequence::~ITabControlPageSequence                            |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlPageSequence :: ~ITabControlPageSequence ( )
{ }


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ITabControlClipPaintHandler::ITabControlClipPaintHandler                     |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlClipPaintHandler :: ITabControlClipPaintHandler ( )
{}


/*------------------------------------------------------------------------------
| ITabControlClipPaintHandler::~ITabControlClipPaintHandler                    |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlClipPaintHandler :: ~ITabControlClipPaintHandler ( )
{}

/*------------------------------------------------------------------------------
| ITabControlClipPaintHandler::paintWindow                                     |
|                                                                              |
------------------------------------------------------------------------------*/
bool ITabControlClipPaintHandler :: paintWindow ( IPaintEvent& event )
{
  IMODTRACE_DEVELOP( "ITabControlClipPaintHandler::paintWindow" );

  ITabControl*
    tabControl = dynamic_cast<ITabControl*>(event.window()->parent());

  if (tabControl)
  {
     IPresSpaceHandle hps = event.presSpaceHandle();

     // Clear the page background only if the top page is not auto sized.
     if (tabControl->totalPages() && !tabControl->topPage()->isAutoPageSize())
	event.clearBackground( tabControl->pageBackgroundColor() );

     ITabControlButtonCanvas *buttonCanvas =
	(ITabControlButtonCanvas*)IWindow::windowWithOwner( ID_TABBUTTONS, tabControl );
     if (buttonCanvas)
     {
	// We want the rectangle that normally appears around the entire tab page
	// to be drawn only around the page window.  Since the tab control has
	// already been painted, we can go ahead and draw the bottom shadow of the
	// page outline.  Note that we shifted the size/position of the button canvas
	// to paint over the original border.

	IManagedPresSpaceHandle hpsTabControl( tabControl );
	IRootGrafPort shadowPort( hpsTabControl );
	IBasicColor clrDarkShadow( CharIntensity( GetSysColor( COLOR_3DDKSHADOW )));
	IFrameBundle shadowBundle( clrDarkShadow );
	IRectangle rectButtons = buttonCanvas->nativeRect();
	IGLine2D line;
	line.setStartPoint( IGPoint2D( rectButtons.left(),
				       rectButtons.minY() - 1 ));
	line.setEndPoint( IGPoint2D( rectButtons.right(),
				     rectButtons.minY() - 1 ));
	shadowPort.draw( line, shadowBundle );

	IBasicColor clrShadow( CharIntensity( GetSysColor( COLOR_3DSHADOW )));
	shadowBundle.setFrameColor( clrShadow );
	line.setStartPoint( IGPoint2D( rectButtons.left() + 1,
				       rectButtons.minY() - 2 ));
	line.setEndPoint( IGPoint2D( rectButtons.right() - 1,
				     rectButtons.minY() - 2 ));
	shadowPort.draw( line, shadowBundle );
     }
     return( true );
  }

  return false;
}

/*------------------------------------------------------------------------------
| ITabControlClipPaintHandler::dispatchHandlerEvent                            |
|                                                                              |
------------------------------------------------------------------------------*/
bool ITabControlClipPaintHandler :: dispatchHandlerEvent( IEvent& event )
{
  switch( event.eventId() )
  {
    case WM_ERASEBKGND:
      // Processes WM_ERASEBKGND to prevent default processing
      event.setResult( true );
      return( true );

    default:
      break;
  }

  return IPaintHandler::dispatchHandlerEvent( event );
}

/*------------------------------------------------------------------------------
| ITabControlClipWindow::ITabControlClipWindow                                 |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
ITabControlClipWindow :: ITabControlClipWindow( ITabControl* tabControl )
{
  HINSTANCE hinst = GetModuleHandle(0);
  WNDCLASS  wndClass;

  // Before registering, check to make sure it is not already registered.
  bool fRegistered = GetClassInfo( hinst, WC_TABPAGECLIP, &wndClass);

  if (!fRegistered)
  {
     wndClass.style = CS_HREDRAW | CS_VREDRAW | CS_GLOBALCLASS;
     wndClass.lpfnWndProc = DefWindowProc;
     wndClass.cbClsExtra = 0;
     wndClass.cbWndExtra = 0;
     wndClass.hInstance = GetModuleHandle( 0 );
     wndClass.hIcon = 0;
     wndClass.hCursor = LoadCursor( NULL, IDC_ARROW );
     wndClass.hbrBackground = (HBRUSH)(COLOR_3DFACE+1);
     wndClass.lpszMenuName = 0;
     wndClass.lpszClassName = WC_TABPAGECLIP;

     if (!RegisterClass( &wndClass ))
     {
	ITHROWGUIERROR( "RegisterClass" );
     }
  }

  IWindowHandle
    handlePage = create( ID_TABPAGECLIP,
			 0,
			 WS_CHILD | WS_CLIPCHILDREN,
			 WC_TABPAGECLIP,
			 tabControl->handle(),
			 IWindowHandle(0),
			 IRectangle(),
			 0,
			 0 );

  if (handlePage.isValid())
  {
     setAutoDestroyWindow( true );
     startHandlingEventsFor( handlePage );
  }
  else
  {
     ITHROWGUIERROR("CreateWindow");
  }

  fPaintHandler = new ITabControlClipPaintHandler();
  fPaintHandler->handleEventsFor( this );
}


/*------------------------------------------------------------------------------
| ITabControlClipWindow::ITabControlClipWindow                                 |
| Copy constructor                                                             |
------------------------------------------------------------------------------*/
ITabControlClipWindow :: ITabControlClipWindow(
			    const ITabControlClipWindow& tabPageClipWindow )
{ }


/*------------------------------------------------------------------------------
| ITabControlClipWindow::operator =                                            |
| Assignment                                                                   |
------------------------------------------------------------------------------*/
ITabControlClipWindow& ITabControlClipWindow ::
		      operator = ( const ITabControlClipWindow& tabPageClipWindow )
{
  return *this;
}


/*------------------------------------------------------------------------------
| ITabControlClipWindow::~ITabControlClipWindow                                |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITabControlClipWindow :: ~ITabControlClipWindow( )
{
  // Detach the tab control paint handler from this clip window.
  // It would only have been attached if the background color was explicitly
  // set for the tab control, but the stop handling is safe.
  fPaintHandler->stopHandlingEventsFor( this );
}


/*------------------------------------------------------------------------------
| ITabPageKeyboardHandler::ITabPageKeyboardHandler                             |
------------------------------------------------------------------------------*/
ITabPageKeyboardHandler :: ITabPageKeyboardHandler( )
{}


/*------------------------------------------------------------------------------
| ITabPageKeyboardHandler::~ITabPageKeyboardHandler                            |
------------------------------------------------------------------------------*/
ITabPageKeyboardHandler :: ~ITabPageKeyboardHandler( )
{}


/*------------------------------------------------------------------------------
| ITabPageKeyboardHandler::dispatchHandlerEvent                                |
|                                                                              |
------------------------------------------------------------------------------*/
bool ITabPageKeyboardHandler :: dispatchHandlerEvent( IEvent& event )
{
  bool stopProcessingEvent = false;

  // Only processes WM_SYSKEYDOWN to simulate CUA '91 notebook behavior.
  switch( event.eventId() )
  {
    case WM_SYSKEYDOWN:
      {
	// Get a pointer to the tab control object.  The parent of the top
	// application page window is always the page clipping window, and
	// the parent of the page clipping window is always the tab control.
	IWindowHandle
	  hwndControl( event.window()->parent()->parent()->handle() );
	if ( !hwndControl )
	  break;

	// Only process Windows tab control keyboard events of interest.
	if ( IWindowClassName( hwndControl ) != WC_TABCONTROL )
	  break;

	// Process key events for the Windows tab control.
	IKeyboardEvent keyevt( event );
	if ( !keyevt.isVirtual() || !keyevt.isAltDown() ||
	     keyevt.isShiftDown() || keyevt.isCtrlDown() )
	  break;

	// Obtain pointer to notebook object.
	ITabControl* tabControl =
	    dynamic_cast<ITabControl*>(IWindow::windowWithHandle( hwndControl ));

	// If Alt+PageUp or Alt+PageDown was pressed, send to the notebook
	// for processing.
	if ( (keyevt.virtualKey() == IKeyboardEvent::pageUp)  ||
	     (keyevt.virtualKey() == IKeyboardEvent::pageDown) )
	{
	  HWND hwndOldFocus = IQUERYFOCUS(HWND_DESKTOP);

	  tabControl->sendEvent( event );

	  // If the focus hasn't changed after the page-turn, the application
	  // left it on the old page, so put it on the tab control
	  if ( IQUERYFOCUS(HWND_DESKTOP) == hwndOldFocus )
	    tabControl->setFocus();

	  stopProcessingEvent = true;
	}
	// Alt+Up:  set the focus to the tab control
	else if ( keyevt.virtualKey() == IKeyboardEvent::up )
	{
	  tabControl->setFocus();
	  stopProcessingEvent = true;
	}
      } //WM_SYSKEYDOWN
      break;

    default:
      break;
  }

  return( stopProcessingEvent );
}
#endif // IC_WIN


#ifdef IC_PM
/*------------------------------------------------------------------------------
| ITabPageResizeHandler :: windowResize                                        |
------------------------------------------------------------------------------*/
bool ITabPageResizeHandler :: windowResize( IResizeEvent& event )
{
  // Get the tab control for this page window.
  IWindow   *pageWindow = event.controlWindow();
  ITabControl *tabControl = 0;
  if (pageWindow)
     tabControl = dynamic_cast<ITabControl*>(pageWindow->parent());
  if (tabControl)
  {
     // Get the top page.
     ITabPage *topPage = tabControl->topPage();

     // If the top page window is being resized, just return.
     if (!topPage || (topPage->pageWindow() == pageWindow))
	return false;

     // Reset the page window for the first page found that has this window as
     // its page window.
     ITabControl::Cursor cursor( *tabControl );
     for (cursor.setToFirst();
	  cursor.isValid();
	  cursor.setToNext())
     {
	ITabPage *tabPage = cursor.current();
	if (tabPage->pageWindow() == pageWindow)
	{
	   ITabControlButtonSequence
	      *buttons = (tabPage->tpd->fPageButtons) ?
			    tabPage->tpd->fPageButtons :
			    tabControl->tcd->fCommonButtons;
	    if (buttons)
	    {
	       // We have to reparent each button to this page's page window
	       // before resetting the page window so we don't lose the page
	       // buttons.
	       ITabControlButtonSequence::Cursor buttonCursor( *buttons );
	       forCursor (buttonCursor)
	       {
		  // Reparent each button to this page's page window.
		  ITabControlButton *tcButton = buttons->elementAt( buttonCursor );
		  IPushButton *button = tcButton->button( tabControl );
		  if (button->parent() != pageWindow)
		     button->setParent( pageWindow );
		  button->positionBehindSiblings();
	       }
	    }

	    if (!tabControl->sendEvent( BKM_SETPAGEWINDOWHWND,
					tabPage->tpd->fPageId,
					(unsigned long)pageWindow->handle() )
			     .asUnsignedLong())
	    {
	       ITHROWGUIERROR( "BKM_SETPAGEWINDOWHWND" );
	    }
	    cursor.setToLast();
	 }
      }
   }
   return false;
}
#endif // IC_PM


/*------------------------------------------------------------------------------
| ITabPageData::addPageButton                                                  |
|                                                                              |
| Add a page button.                                                           |
------------------------------------------------------------------------------*/
void ITabPageData :: addPageButton ( ITabControlButton *tcButton,
				     bool               isDefaultButton )
{
  IMODTRACE_DEVELOP( "ITabPageData::addPageButton" );

  // Create a page button sequence for this page if it doesn't already exist.
  if (!fPageButtons)
  {
     fPageButtons = new ITabControlButtonSequence();
  }
  else
  {
     // Throw an exception if this button has already been added for this page.
     ITabControlButtonSequence::Cursor cursor( *fPageButtons );
     forCursor (cursor)
     {
	ITabControlButton *tcPageButton = fPageButtons->elementAt( cursor );
	if (tcPageButton->fApplicationButton == tcButton->fApplicationButton &&
	    tcPageButton->fStandardButton == tcButton->fStandardButton)
	{
	   delete tcButton;
	   ITHROWLIBRARYERROR1( IC_BUTTON_ALREADY_ADDED,
				IBaseErrorInfo::invalidRequest,
				IException::recoverable,
				"ITabPage" );
	}
     }
  }

  // Add this button as the last one in the button sequence.
  fPageButtons->addAsLast( tcButton );

  // Set the default button for the page if this is a default button.
  // The last button added to the page with the default specification
  // will be the page's default button.
  if (isDefaultButton)
     fDefaultButton = tcButton;

#ifdef IC_PM
  // Make sure the BS_NOTEBOOKBUTTON style is set for the button.
  if (tcButton->fApplicationButton)
  {
     IPushButton *button = tcButton->fApplicationButton;
     unsigned long ulButtonStyle = ISTYLEOF( button->handle() );
     if (~ulButtonStyle & BS_NOTEBOOKBUTTON)
     {
	ulButtonStyle |= BS_NOTEBOOKBUTTON;
	ISETWINDOWSTYLE( button->handle(), ulButtonStyle );
     }
  }
#endif // IC_PM

  if (fTabControl)
  {
#ifdef IC_WIN
     // If this page is currently the top page then go ahead and show the new
     // button.
     if (fTabControl->isVisible() &&
	 fTabControl->topPage() == fTabPage &&
	 !fTabControl->tcd->fInCreate)
	fTabControl->tcd->showButtons( 0, fTabPage );
#endif // IC_WIN

#ifdef IC_PM
     if (fPageWindow)
     {
	// Set the page's buttons.
	setButtons( fPageButtons );
     }
#endif // IC_PM

     // Send a notification.  Include the button as the event data.
     IPushButton *button = tcButton->button( fTabControl );
     fTabControl->notifyObservers( INotificationEvent( ITabControl::buttonAddId,
						       *fTabControl,
						       false,
						       (void*)button ));

     // Indicate that the tab control's minimum size may have changed.
#ifdef IC_WIN
     if (!fTabControl->tcd->fInCreate || !fTabControl->isVisible())
#else
     if (!fTabControl->isVisible())
#endif
	fTabControl->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
  }
}


/*------------------------------------------------------------------------------
| ITabPageData::removePageButton                                               |
|                                                                              |
| Remove a page button.                                                        |
------------------------------------------------------------------------------*/
void ITabPageData :: removePageButton ( ITabControlButton *tcButton )
{
  IMODTRACE_DEVELOP( "ITabPageData::removePageButton" );

  ITabControlButtonSequence::Cursor cursor( *fPageButtons );
  if (fPageButtons->locate( tcButton, cursor ))
  {
     // Remove this button from the button sequence.
     fPageButtons->removeAt( cursor );

     // If there are no more buttons in the sequence, delete the sequence.
     if (fPageButtons->isEmpty())
     {
	delete fPageButtons;
	fPageButtons = 0;
     }

     if (fTabControl)
     {
	IPushButton *button = tcButton->button( fTabControl );

#ifdef IC_WIN
	// Only remove this button from the button area if it is visible for the
	// current top page.
	if (fTabControl->isVisible() &&
	    fTabControl->topPage() == fTabPage)
	{
	   // Remove the button from the set canvas if it is in the layout.  It
	   // is possible that it is not in the layout if it is being deleted.
	   if (fTabControl->tcd->fButtonCanvas->isInLayout( button ))
	      fTabControl->tcd->fButtonCanvas->remove( button );
	   if (!fTabControl->tcd->fInCreate)
	      fTabControl->tcd->showButtons( 0, fTabPage );

	   // We may have removed the first element in the sequence.  If so,
	   // set the group and tabstop styles for the new first button.
	   IPushButton
	      *firstButton = fPageButtons->firstElement()->button( fTabControl );
	   if (!firstButton->isGroup())
	   {
	      firstButton->enableTabStop();
	      firstButton->enableGroup();
	   }
	}
#endif // IC_WIN

#ifdef IC_PM
	// Reset the button's parent to the tab control.
	button->setParent( fTabControl );

	// Reset the page buttons.
	setButtons( fPageButtons );
#endif // IC_PM

	// Send a notification.  Include the button as the event data.
	fTabControl->notifyObservers( INotificationEvent( ITabControl::buttonRemoveId,
							  *fTabControl,
							  false,
							  (void*)button ));

	// Indicate that the tab control's minimum size may have changed.
#ifdef IC_WIN
	if (!fTabControl->tcd->fInCreate || !fTabControl->isVisible())
#else
	if (!fTabControl->isVisible())
#endif
	   fTabControl->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
     }
  }
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| ITabPageData::setButtons                                                     |
|                                                                              |
| Set the notebook page buttons for the tab page.                              |
------------------------------------------------------------------------------*/
void ITabPageData :: setButtons ( ITabControlButtonSequence *buttons )
{
  IMODTRACE_DEVELOP( "ITabPageData::setButtons" );

  if (fPageId && fPageWindow)
  {
     if (buttons)
     {
	ITabControlButtonSequence::Cursor buttonCursor( *buttons );
	forCursor (buttonCursor)
	{
	   // Reparent each button to this page's page window.
	   ITabControlButton *tcButton = buttons->elementAt( buttonCursor );
	   IPushButton *button = tcButton->button( fTabControl );
	   button->setParent( fPageWindow );
	   button->positionBehindSiblings();
	}

	if (!fTabControl->sendEvent( BKM_SETPAGEWINDOWHWND,
				     IEventParameter1( fPageId ),
				     IEventParameter2( fPageWindow->handle() ))
	    .asUnsignedLong())
	{
	   ITHROWGUIERROR( "BKM_SETPAGEWINDOWHWND" );
	}
     }
  }
}


/*------------------------------------------------------------------------------
| ITabPageData::autoTabColor                                                   |
|                                                                              |
| Return the auto color value for this tab page.                               |
------------------------------------------------------------------------------*/
unsigned long ITabPageData::autoTabColor ( ITabPage *tabPage )
{
  IMODTRACE_DEVELOP( "ITabPageData::autoTabColor" );

  unsigned long tabColor = 0;
  ITabControl *tabControl = tabPage->tabControl();
  if (tabControl)
  {
     // Determine the relative position of this page in the tab control.
     ITabControl::Cursor pageCursor( *tabControl );
     unsigned long pageIndex = 0;
     for (pageCursor.setToFirst();
	  pageCursor.isValid();
	  pageCursor.setToNext())
     {
	if (pageCursor.current() == tabPage)
	   pageCursor.invalidate();
	else
	   pageIndex++;
     }
     tabColor = autoColors[pageIndex % 10];
  }

  return tabColor;
}
#endif // IC_PM
