// Revision: 73 1.21.1.1 source/ui/basectl/isplitcv.cpp, canvas, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: isplitcv.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in isplitcv.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C" {
  #define INCL_GPILOGCOLORTABLE
  #define INCL_GPIPRIMITIVES
  #define INCL_WINFRAMEMGR
  #define INCL_WININPUT
  #define INCL_WINMESSAGEMGR
  #define INCL_WINPOINTERS
  #define INCL_WINSYS
  #define INCL_WINTRACKRECT
  #define INCL_WINWINDOWMGR
  #include <iwindefs.h>
#ifdef IC_MOTIF
  #include <Xm/SashP.h>
#endif
}

#include <isplitch.hpp>
#include <isplitpc.hpp>
#include <isplitcv.hpp>
#include <ibidiset.hpp>
#include <icconst.h>
#include <icolor.hpp>
#include <idefstyl.h>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ihandler.hpp>
#include <ilanglvl.hpp>
#include <inotifev.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <isizeevt.hpp>
#include <itrace.hpp>
#include <iwposbuf.hpp>
#include <iksset2.h>

#ifdef IC_MOTIF
  #include <ispltbar.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ISPLITCV_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Public canvas styles.                                                        |
|                                                                              |
| Notes: The below style bits must not conflict with those used by inherited   |
|          class.                                                              |
------------------------------------------------------------------------------*/
const ISplitCanvas::Style
  ISplitCanvas::vertical          ( 1, ISplitCanvas__vertical ),
  ISplitCanvas::horizontal        ( 1, ISplitCanvas__horizontal ),
  ISplitCanvas::noSplitBars       ( 1, ISplitCanvas__noSplitBars ),
  ISplitCanvas::classDefaultStyle ( 2, ISplitCanvas__classDefaultStyle );


/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
ISplitCanvas::Style
  ISplitCanvas::currentDefaultStyle ( 2, ISplitCanvas__classDefaultStyle );

/*------------------------------------------------------------------------------
| ISplitPercentageSet                                                          |
|                                                                              |
| Declare a key sorted set for the split percentage objects.                   |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class ISplitPercentageSet : public IVPtrKeySortedSet<ISplitPercentage*, IWindow*> {
public:
  ISplitPercentageSet ( );
 ~ISplitPercentageSet ( );
}; // ISplitPercentageSet

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| ISplitPercentageSet::ISplitPercentageSet                                     |
|                                                                              |
| Empty constructor here for page tuning.                                      |
------------------------------------------------------------------------------*/
ISplitPercentageSet::ISplitPercentageSet ( )
  : IVPtrKeySortedSet<ISplitPercentage*, IWindow*>()
{ }

/*------------------------------------------------------------------------------
| ISplitPercentageSet::~ISplitPercentageSet                                    |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ISplitPercentageSet::~ISplitPercentageSet ( )
{
  this->removeAll( &ISplitPercentageDeleter );
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| INumberSplitBarDeleter                                                       |
|                                                                              |
| This function is used as input to allElementsDo() and results in all members |
| of the collection being deleted.                                             |
------------------------------------------------------------------------------*/
bool INumberSplitBarDeleter ( INumberSplitBar* const& obj,
                              void*   anything )
{
  if ( obj )
  {
     delete obj->splitBar();       // Delete the ISplitBar.
     delete (INumberSplitBar*)obj; // Delete the INumberSplitBar.
  }
  return true;
}

/*------------------------------------------------------------------------------
| ISplitBarSet                                                                 |
|                                                                              |
| Declare a key sorted set for the split bar objects.                          |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class ISplitBarSet : public IVPtrKeySortedSet<INumberSplitBar*, unsigned long> {
public:
  ISplitBarSet ( );
 ~ISplitBarSet ( );
}; // ISplitBarSet

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| ISplitBarSet::ISplitBarSet                                                   |
|                                                                              |
| Empty constructor here for page tuning.                                      |
------------------------------------------------------------------------------*/
ISplitBarSet::ISplitBarSet ( )
  : IVPtrKeySortedSet<INumberSplitBar*, unsigned long>()
{ }

/*------------------------------------------------------------------------------
| ISplitBarSet::~ISplitBarSet                                                  |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ISplitBarSet::~ISplitBarSet ( )
{
   this->removeAll( &INumberSplitBarDeleter );
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| IPaneSeq                                                                     |
|                                                                              |
| Declare a linked list for the pane objects                                   |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IPaneSeq {
public:
  IPaneSeq ( );
 ~IPaneSeq ( );
bool
  isEmpty  ( );
void
  addFirst ( IPaneObj * ),
  addLast  ( IPaneObj * );

private:
IPaneObj
 *pFirstCl,
 *pLastCl;

friend class ISplitCanvas;
}; // IPaneSeq

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IPaneSeq::IPaneSeq                                                           |
|                                                                              |
| Initialize the linked list.                                                  |
------------------------------------------------------------------------------*/
IPaneSeq::IPaneSeq ( )
  : pFirstCl( 0 ), pLastCl( 0 )
{ }

/*------------------------------------------------------------------------------
| IPaneSeq::~IPaneSeq                                                          |
|                                                                              |
| Delete all objects in the linked list.                                       |
------------------------------------------------------------------------------*/
IPaneSeq::~IPaneSeq ( )
{
  if ( pFirstCl )
  {
     IPaneObj *tmpPaneObj;
     while ( pFirstCl )
     {
        tmpPaneObj = pFirstCl->pNext;
        delete pFirstCl;
        pFirstCl = tmpPaneObj;
     }
  }
}

/*------------------------------------------------------------------------------
| IPaneSeq::addFirst                                                           |
|                                                                              |
| Add an item to the beginning of the linked list.                             |
------------------------------------------------------------------------------*/
void IPaneSeq::addFirst ( IPaneObj* paneObj )
{
  paneObj->pNext = pFirstCl;
  pFirstCl = paneObj;
}

/*------------------------------------------------------------------------------
| IPaneSeq::addLast                                                            |
|                                                                              |
| Add an item to the end of the linked list.                                   |
------------------------------------------------------------------------------*/
void IPaneSeq::addLast ( IPaneObj* paneObj )
{
  if ( ! pFirstCl )
  {
     pFirstCl = paneObj;
     pLastCl  = paneObj;
  }
  else
  {
     pLastCl->pNext = paneObj;
     pLastCl = paneObj;
  }
}

/*------------------------------------------------------------------------------
| IPaneSeq::isEmpty                                                            |
|                                                                              |
| See if the linked list is empty.                                             |
------------------------------------------------------------------------------*/
bool IPaneSeq::isEmpty ( )
{
  return (pFirstCl ? false : true);
}


/*------------------------------------------------------------------------------
| ISplitCanvas::ISplitCanvas                                                   |
|                                                                              |
| Constructor.  Window is created in ICanvas class.                            |
------------------------------------------------------------------------------*/
ISplitCanvas::ISplitCanvas( unsigned long windowIdentifier,
                            IWindow* parent,
                            IWindow* owner,
                            const IRectangle& initialSize,
                            const Style& style )
  : ISplitCanvas::Inherited( )
#ifdef IC_PMWIN
  , pClSplitBarSet( 0 )
#endif
#ifdef IC_MOTIF
  , pClSplitBarSet( new ISplitBarSet() )
#endif
  , sizClCanvasSize( -1, -1 )
  , clrClMiddleColor( IColor::paleGray )
  , clrClEdgeColor( IColor::darkGray )
  , fSplitCanvasData( 0 )
{
  // Check for valid style combination.
  if ( (style & horizontal)  &&  (style & vertical) )
  {                                   // Mutually exclusive styles.
     ITHROWLIBRARYERROR( IC_INVALIDSTYLE,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable );
  }

  // If no split bar orientation was specified, assume horizontal
  // for backwards compatibility.
  fCurrentStyle = style;
  if ( !( style & horizontal )  &&  !( style & vertical ) )
  {                // No split bar orientation specified.
     fCurrentStyle |= ISplitCanvas::horizontal;
  }

  bClRatiosNotResolved = true;
#ifdef IC_MOTIF
  bClNoSplitBars = ( style & noSplitBars ) ? true : false;
#endif

  pClSplitPercentageSet = new ISplitPercentageSet();
  pClSplitCanvasHandler = new ISplitCanvasHandler();

#ifdef IC_PMWIN
  // Get the border widths used to draw the split bar.
  ulClEdgeThickness   = IQUERYSYSVALUE( SV_CXBORDER );
  ulClMiddleThickness = IQUERYSYSVALUE( SV_CXSIZEBORDER ) -
                          (2 * ulClEdgeThickness);
#endif

  pClSplitCanvasHandler->handleEventsFor( this );

  // Create the presentation system window.
  this->initialize( windowIdentifier,
                    parent,
                    owner,
                    initialSize,
                    this->convertToGUIStyle( fCurrentStyle ),
                    this->convertToGUIStyle( fCurrentStyle, true ) );
#ifdef IC_MOTIF
  (fCurrentStyle & ISplitCanvas::vertical) ? bClVertical = true
                                         : bClVertical = false;
  ulClEdgeThickness = 0;
  ulClMiddleThickness = 10;
#endif
}

#if IC_EXTENDED_RTTI_ENABLED
ISplitCanvas::ISplitCanvas ( const ISplitCanvas& )
  : clrClMiddleColor( IColor::paleGray )
  , clrClEdgeColor( IColor::darkGray )
{
}
#endif // IC_EXTENDED_RTTI_ENABLED

/*------------------------------------------------------------------------------
| ISplitCanvas::~ISplitCanvas                                                  |
|                                                                              |
| Destructor.  Remove and delete collection objects.                           |
------------------------------------------------------------------------------*/
ISplitCanvas::~ISplitCanvas ( )
{
  if ( pClSplitPercentageSet )
  {
     delete pClSplitPercentageSet;
  }

  if ( pClSplitCanvasHandler )
  {
     pClSplitCanvasHandler->stopHandlingEventsFor( this );
     delete pClSplitCanvasHandler;
  }
#ifdef IC_MOTIF
  if ( pClSplitBarSet)
    delete pClSplitBarSet;
#endif
  // delete fSplitCanvasData;
}

/*------------------------------------------------------------------------------
| ISplitPercentageDeleter                                                      |
|                                                                              |
| This function is used as input to allElementsDo() and results in all members |
| of the collection being deleted.                                             |
------------------------------------------------------------------------------*/
bool ISplitPercentageDeleter ( ISplitPercentage* const& splitPercentageObj,
                               void* )
{
  delete (ISplitPercentage*)splitPercentageObj;
  return true;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::layout                                                         |
|                                                                              |
| Layout management routine.                                                   |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::layout ( )
{
  IMODTRACE_DEVELOP( "ISplitCanvas::layout()" );

  if ( this->isLayoutDistorted( IWindow::layoutChanged ))
  {
     ITRACE_DEVELOP( "isLayoutDistorted(layoutChanged)==true" );
     /**************************************************************/
     /* The layout is distorted so calculate the position and size */
     /* of each of the child controls based on the percentage of   */
     /* the window they occupied at time of window resizing.       */
     /**************************************************************/

     /**************************************************************/
     /* Build a list of the canvas' child controls, as enumerated  */
     /* from PM.  This list will be used as a workspace for        */
     /* positioning and sizing the child controls.                 */
     /*                                                            */
     /* Note that child controls are enumerated in the z-order,    */
     /* which is the reverse order that they are layed out (and    */
     /* tabbed through).                                           */
     /**************************************************************/
     ITRACE_DEVELOP( "Start getting all child controls" );
     ISize szCanvas;
     unsigned long ulCanvasWidth;
     unsigned long ulCanvasHeight;
     unsigned long ulRemainingWindowPixels;
     unsigned long ulTotalPixelsUsed = 0;
     bool verticalSplitBars =
            ( fCurrentStyle & vertical ) ? true : false;

     /***************************************************************/
     /* if sizClCanvasSize.width == -1, the queried size of the     */
     /* canvas is valid, else we entered layout() from the WM_SIZE  */
     /* message and we should use the size passed into the WM_SIZE  */
     /* message which has been saved in the sizClCanvasSize object. */
     /***************************************************************/
     if ( sizClCanvasSize.width() == -1 )
     {
        szCanvas = this->size();
     }
     else
     {
        szCanvas = sizClCanvasSize;
        sizClCanvasSize.setWidth( -1 ).setHeight( -1 );
     }

     ulCanvasWidth  = szCanvas.width();
     ulCanvasHeight = szCanvas.height();

     /***************************************************************/
     /* Do NOT size controls if canvas is ZERO height or width      */
     /* since it is not really visible.                             */
     /***************************************************************/
     if ( (ulCanvasWidth == 0) || (ulCanvasHeight == 0) )
     {
        return *this;
     }

     IPoint
       topLeftOffset( this->topLeftLayoutOffset() ),
       bottomRightOffset( this->bottomRightLayoutOffset() );
     ISize
       totalBorder( topLeftOffset + bottomRightOffset );

     // Size all child windows to 0 if the border consumes all room.
     if ( ( totalBorder.width() > ulCanvasWidth )  ||
          ( totalBorder.height() > ulCanvasHeight ) )
     {
        ulCanvasWidth = 0;
        ulCanvasHeight = 0;
        // Now size all children to 0.  IC_NOTYET
        return *this;
     }

     ulCanvasWidth -= totalBorder.width();
     ulCanvasHeight -= totalBorder.height();

     /***************************************************************/
     /* Set the split bar width according to whether the canvas has */
     /* split-bars or not.                                          */
     /***************************************************************/
     unsigned long ulSplitBarWidth = 0;
     if ( !(fCurrentStyle & noSplitBars) )
        ulSplitBarWidth = (ulClEdgeThickness * 2) + ulClMiddleThickness;

     if ( bClRatiosNotResolved )
        this->resolveRatios();

     ulClChildCount = 0;                 // Reset child count to zero.
     unsigned long ulNonZeroSizeChildCount = 0;
     unsigned long ulRatioTotal = 0;
     IWindowHandle whChild;
     IPaneSeq* pPaneSeq = new IPaneSeq();

     IWindow::ChildCursor cwCursor( *this );
     for ( cwCursor.setToFirst(); cwCursor.isValid(); cwCursor.setToNext() )
     {
        whChild = this->childAt( cwCursor );
        IWindow* pwinChild = IWindow::windowWithHandle( whChild );
#ifdef IC_PMWIN
        if ( pwinChild )
#endif
#ifdef IC_MOTIF
        // Make use of addRelated.
        bool isSplitBar=false;
        if ( pwinChild  &&  XmIsForm( (Widget)whChild ) )
        {
           Cardinal numChildren;
           WidgetList children;
           XtVaGetValues( (Widget) pwinChild->handle(),
                          XmNnumChildren, &numChildren,
                          XmNchildren, &children,
                          NULL );
           for ( int i=0; i < numChildren; i++ )
           {
              if ( children  &&
                   XmIsSash( (Widget)children[i] ) )
              {
                 isSplitBar = true;
                 break;
              }
           }
        }
        if ( !isSplitBar )
#endif // IC_MOTIF
        {                         // Have child control object.
           IPaneObj* ppcvChild = new IPaneObj( pwinChild );

           /**********************************************************/
           /* If the control has a ratio other than the canvas       */
           /* default ratio, get the controls ratio.                 */
           /**********************************************************/
           if ( pClSplitPercentageSet->containsElementWithKey( pwinChild ))
           {
              ppcvChild->ulClPercentage = resolvedPercentage( pwinChild );
           }
           if ( ppcvChild->ulClPercentage != 0 )
           {  // Give this child window a non-zero size.
              ulRatioTotal += ppcvChild->ulClPercentage;
              ulNonZeroSizeChildCount++;
           }
#ifdef IC_MOTIF
           else
           {  // Size this child window to one pixel.
              ulRatioTotal++;
           }
#endif
           if ( verticalSplitBars )
           {
              if ( this->areChildrenReversed() )
                 pPaneSeq->addLast( ppcvChild );
              else
                 pPaneSeq->addFirst( ppcvChild );
           }
           else               // Horizontal splitbars.
           {
              if ( this->areChildrenReversed() )
                 pPaneSeq->addFirst( ppcvChild );
              else
                 pPaneSeq->addLast( ppcvChild );
           }

           ulClChildCount++;
        }
     }
     ITRACE_DEVELOP( "End getting all child controls" );

     if ( pPaneSeq->isEmpty() )  // No controls are in the canvas.
     {
        bClRatiosNotResolved = true;
        delete pPaneSeq;
        return *this;
     }
     /***************************************************************/
     /* Remove the split-bar width from the ratio unit calculation  */
     /* and remember there are no split bars to the left of the     */
     /* first control and to the right of the last control.         */
     /***************************************************************/
     unsigned long
       totalSplitBarThickness = ulSplitBarWidth * ( ulClChildCount - 1 );
     if ( verticalSplitBars )
     {
        if ( ulCanvasWidth > totalSplitBarThickness )
        {
           ulCanvasWidth -= totalSplitBarThickness;
        }
        else
        {  // Canvas is too narrow to even display all the split
           // bars.
           ulCanvasWidth = 0;
        }
     }
     else
     {  // Horizontal split bars.
        if ( ulCanvasHeight > totalSplitBarThickness )
        {
           ulCanvasHeight -= totalSplitBarThickness;
        }
        else
        {  // Canvas is too short to even display all the split
           // bars.
           ulCanvasHeight = 0;
        }
     }

     IPaneObj* ppcvWinnerOfPixelsObj;
     IPaneObj* ppcvObj = pPaneSeq->pFirstCl;

     while ( ppcvObj )
     {                            // Process all contents.
       if ( ppcvObj->ulClPercentage != 0 )
          ppcvWinnerOfPixelsObj = ppcvObj;

       if ( verticalSplitBars )
       {
          ppcvObj->ulControlSize = (ulCanvasWidth *
                                     (ppcvObj->ulClPercentage)) /
                                       ulRatioTotal;
       }
       else
       {
          ppcvObj->ulControlSize = (ulCanvasHeight *
                                     (ppcvObj->ulClPercentage)) /
                                       ulRatioTotal;
       }
#ifdef IC_MOTIF
       if ( ppcvObj->ulControlSize == 0 )
       {  // Size the child window to a minimum of one pixel.
          ppcvObj->ulControlSize = 1;
       }
#endif
       ulTotalPixelsUsed += ppcvObj->ulControlSize;

       ppcvObj = ppcvObj->pNext;
     }

     /*****************************************************************/
     /* Due to rounding errors, there is still the possibility that   */
     /* a window unit (or a few window units) is unused, this unit    */
     /* will be added to the size of the first non-zero size control. */
     /*****************************************************************/
     if ( verticalSplitBars )
     {
        ulRemainingWindowPixels =
           ( ulCanvasWidth > ulTotalPixelsUsed ) ?
                ulCanvasWidth - ulTotalPixelsUsed : 0;
     }
     else
     {
        ulRemainingWindowPixels =
           ( ulCanvasHeight > ulTotalPixelsUsed ) ?
                ulCanvasHeight - ulTotalPixelsUsed : 0;
     }

     ITRACE_DEVELOP( IString( "Pixels = " ) +
                       IString( ulRemainingWindowPixels ));
     if ( ulRemainingWindowPixels > 4 )
     {
        ppcvObj = pPaneSeq->pFirstCl;
        while ( ppcvObj )
        {                            // Process all contents.
           unsigned long ulNonZeroSizeParts = 0;

           if ( verticalSplitBars )
              ulNonZeroSizeParts = (ulRemainingWindowPixels *
                                     (ppcvObj->ulClPercentage)) /
                                       ulRatioTotal;
           else
              ulNonZeroSizeParts = (ulRemainingWindowPixels *
                                     (ppcvObj->ulClPercentage)) /
                                       ulRatioTotal;

           ppcvObj->ulControlSize += ulNonZeroSizeParts;
           ulTotalPixelsUsed += ulNonZeroSizeParts;

           ppcvObj = ppcvObj->pNext;
        }

        if ( verticalSplitBars )
           ulRemainingWindowPixels = ulCanvasWidth - ulTotalPixelsUsed;
        else
           ulRemainingWindowPixels = ulCanvasHeight - ulTotalPixelsUsed;
     }
     /***************************************************************/
     /* Add final pixels to last window that has a non-zero size.   */
     /***************************************************************/
     ppcvWinnerOfPixelsObj->ulControlSize += ulRemainingWindowPixels;

     IWindowPosBuffer wposbuf = this->fixupChildren();

#ifdef IC_MOTIF
     // Make sure we have enough split bars
     unsigned long ulClSplitBarNumber = pClSplitBarSet->numberOfElements();
     ISplitBar* aSplitBar;
     if  ( !bClNoSplitBars  &&
           ulClChildCount  &&
           ( ulClSplitBarNumber < ulClChildCount - 1 ) )
     {
        // We need more split bars.
        while ( ulClSplitBarNumber < ulClChildCount - 1 )
        {
           if ( bClVertical)
              aSplitBar = new ISplitBar( 3000 + ulClSplitBarNumber, this, this,
                                         IRectangle( IPoint(0, 0), ISize(0,0) ),
                                         IWindow::visible | ISplitBar::vertical );
           else
              aSplitBar = new ISplitBar( 3000 + ulClSplitBarNumber, this, this,
                                         IRectangle( IPoint(0, 0), ISize(0,0) ),
                                         IWindow::visible | ISplitBar::horizontal );
           pClSplitCanvasHandler->handleEventsFor( aSplitBar );
           pClSplitBarSet->add(
             new INumberSplitBar( aSplitBar, ulClSplitBarNumber ) );
           ulClSplitBarNumber++;
        }
     }
     if ( !bClNoSplitBars  &&  ulClSplitBarNumber )
     {
        // Make sure split bars are the right kind.
        ulClSplitBarNumber = 0;
        while ( ulClSplitBarNumber < ulClChildCount - 1 )
        {
           aSplitBar =
             pClSplitBarSet->elementWithKey( ulClSplitBarNumber )->splitBar();
           if ( bClVertical )
           {
              if ( aSplitBar->orientation() != ISplitBar::verticalSplit )
                 aSplitBar->setOrientation( ISplitBar::verticalSplit );
           }
           else if ( aSplitBar->orientation() != ISplitBar::horizontalSplit )
              aSplitBar->setOrientation( ISplitBar::horizontalSplit );
           aSplitBar->show();
           ulClSplitBarNumber++;
        }
     }
     // Hide the rest.
     // ulClSplitBarNumber should be 1 greater than the number of split bars
     // needed.
     if ( bClNoSplitBars )
        ulClSplitBarNumber = 0;   // if no split bars, hide them all,
     for ( int index = ulClSplitBarNumber;
           index < pClSplitBarSet->numberOfElements();
           index++ )
     {
        aSplitBar =
          pClSplitBarSet->elementWithKey( ulClSplitBarNumber )->splitBar();
        aSplitBar->hide();
     }
     ulClSplitBarNumber = 0;
#endif // IC_MOTIF

     // Size and position child windows in native coordinates, since
     // this is what IWindowPosBuffer expects.
     unsigned long x0 = topLeftOffset.x();
#ifdef IC_ORIGINLOWERLEFT
     unsigned long y0 = bottomRightOffset.y();
#else
     unsigned long y0 =
       verticalSplitBars ? topLeftOffset.y() : bottomRightOffset.y();
#endif
     unsigned long ulCoord = y0;
     if ( verticalSplitBars )
     {
        ulCoord = x0;
     }

     ppcvObj = pPaneSeq->pFirstCl;
     while ( ppcvObj )
     {
       if ( verticalSplitBars )
       {
          wposbuf.moveSizeTo( ppcvObj->pwinCl,
                              IRectangle( IPair( ulCoord, y0 ),
                                          ISize( ppcvObj->ulControlSize,
                                                 ulCanvasHeight ) ) );
#ifdef IC_MOTIF
          ulCoord += ppcvObj->ulControlSize;
#else
          ulCoord += (ppcvObj->ulControlSize + ulSplitBarWidth);
#endif
       }
       else
       {
#ifdef IC_ORIGINLOWERLEFT
          wposbuf.moveSizeTo( ppcvObj->pwinCl,
                              IRectangle( IPair( x0, ulCoord ),
                                          ISize( ulCanvasWidth,
                                                 ppcvObj->ulControlSize ) ) );
#ifdef IC_MOTIF
          ulCoord += ppcvObj->ulControlSize;
#else
          ulCoord += (ppcvObj->ulControlSize + ulSplitBarWidth);
#endif
#else // ! IC_ORIGINLOWERLEFT
          ulCoord += ppcvObj->ulControlSize;
          wposbuf.moveSizeTo( ppcvObj->pwinCl,
                              IRectangle( IPair( x0, szCanvas.height() - ulCoord ),
                                          ISize( ulCanvasWidth,
                                                 ppcvObj->ulControlSize ) ) );
#ifndef IC_MOTIF
          ulCoord += ulSplitBarWidth;
#endif
#endif // ! IC_ORIGINLOWERLEFT
       }

#ifdef IC_MOTIF
       if ( !bClNoSplitBars )
       {
          ISplitBar* aSplitBar;
          if ( ppcvObj->pNext )
          {
             aSplitBar =
               pClSplitBarSet->elementWithKey( ulClSplitBarNumber )->splitBar();
             if ( bClVertical )
             {  // Vertical split bars.
                wposbuf.moveSizeTo( aSplitBar,
                                    IRectangle( IPoint( ulCoord, y0 ),
                                                ISize( ulSplitBarWidth,
                                                       ulCanvasHeight ) ) );
                bool
                  leftToRight = this->areChildrenReversed();

                if ( IBidiSettings::isBidiSupported() )
                {  // Support reversing of child windows for bidi.
                   IBidiSettings
                     bidiSettings( *this );
                   if ( bidiSettings.windowLayout() ==
                                       IBidiSettings::layoutRightToLeft )
                   {
                      leftToRight = ! leftToRight;
                   }
                }

                if ( leftToRight )
                {  // Arrange children left to right.
                   aSplitBar->setPrevControl( ppcvObj->pwinCl );
                   aSplitBar->setNextControl( ppcvObj->pNext->pwinCl );
                }
                else
                {  // Arrange children right to left.
                   aSplitBar->setPrevControl( ppcvObj->pNext->pwinCl );
                   aSplitBar->setNextControl( ppcvObj->pwinCl );
                }
             }
             else
             {  // Horizontal split bars.
                wposbuf.moveSizeTo( aSplitBar,
                                    IRectangle( IPoint( x0,
                                                        (szCanvas.height() -
                                                         (ulCoord +
                                                          ulSplitBarWidth)) ),
                                                ISize( ulCanvasWidth,
                                                       ulSplitBarWidth ) ) );
                if ( areChildrenReversed() )
                {  // Arrange children bottom to top.
                   aSplitBar->setPrevControl( ppcvObj->pNext->pwinCl );
                   aSplitBar->setNextControl( ppcvObj->pwinCl );
                }
                else
                {  // Arrange children top to bottom.
                   aSplitBar->setPrevControl( ppcvObj->pwinCl );
                   aSplitBar->setNextControl( ppcvObj->pNext->pwinCl );
                }
             }
          }
          ulCoord += ulSplitBarWidth;;
          ulClSplitBarNumber++;
       }
#endif // IC_MOTIF
       ITRACE_DEVELOP( IString( "hwndCtrl = " ) +
                       IString( ppcvObj->pwinCl->handle().asUnsigned()) );
       ITRACE_DEVELOP( IString( "Ratio = " ) +
                       IString( ppcvObj->ulClPercentage ));
       ITRACE_DEVELOP( IString( "Size = " ) +
                       IString( (unsigned long)(ppcvObj->ulControlSize)) );
       ITRACE_DEVELOP( IString( "CVSize = " ) + szCanvas.asString() );

       ppcvObj = ppcvObj->pNext;
     }

     wposbuf.apply();           // Reposition/size child controls.

     if ( pPaneSeq )
     {
        delete pPaneSeq;
     }

    sizClCanvasSize.setWidth( -1 ).setHeight( -1 );

    this->setLayoutDistorted( 0, IWindow::layoutChanged );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setSplitWindowPercentage                                       |
|                                                                              |
| Set the default split bar ratio (percentage of the window).                  |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::setSplitWindowPercentage ( IWindow* pwin,
                                                  unsigned long ulPercentage )
{
  if ( pClSplitPercentageSet->containsElementWithKey( pwin ) )
  {
     ISplitPercentage* obj = pClSplitPercentageSet->elementWithKey( pwin );
     obj->ulClPercentage = ulPercentage;
     obj->ulClResolvedPercentage = ulPercentage;
  }
  else
  {
     ISplitPercentage* obj = new ISplitPercentage( pwin, ulPercentage );
     pClSplitPercentageSet->add( obj );
     obj->ulClResolvedPercentage = ulPercentage;
  }

  bClRatiosNotResolved = true;
  this->setLayoutDistorted( IWindow::layoutChanged, 0 );

  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::splitWindowPercentage                                          |
|                                                                              |
| Get the default split bar ratio (percentage of the window).                  |
------------------------------------------------------------------------------*/
unsigned long ISplitCanvas::splitWindowPercentage ( IWindow* pwin )
{
  if ( bClRatiosNotResolved )
  {
     this->resolveRatios();
  }

  if ( pClSplitPercentageSet->containsElementWithKey( pwin ) )
  {
     ISplitPercentage* obj = pClSplitPercentageSet->elementWithKey( pwin );
     return obj->ulClPercentage;
  }

  return 0;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setResolvedPercentage                                          |
|                                                                              |
| Set resolved percentage of the window.                                       |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::setResolvedPercentage ( IWindow* pwin,
                                                    unsigned long ulPercentage )
{
  if ( pClSplitPercentageSet->containsElementWithKey( pwin ) )
  {
     ISplitPercentage* obj = pClSplitPercentageSet->elementWithKey( pwin );
     obj->ulClResolvedPercentage = ulPercentage;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::resolvedPercentage                                             |
|                                                                              |
| Get the resolved percentage of the window.                                   |
------------------------------------------------------------------------------*/
unsigned long ISplitCanvas::resolvedPercentage ( IWindow* pwin )
{
  if ( pClSplitPercentageSet->containsElementWithKey( pwin ) )
  {
     ISplitPercentage* obj = pClSplitPercentageSet->elementWithKey( pwin );
     return obj->ulClResolvedPercentage;
  }
  return 0;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::resolveRatios                                                  |
|                                                                              |
| Resolve percentages to a total of 128 (which is used to divide by).          |
------------------------------------------------------------------------------*/
void ISplitCanvas::resolveRatios ( )
{
  IMODTRACE_DEVELOP( "ISplitCanvas::resolveRatios()" );

  IPaneSeq* pPaneSeq  = new IPaneSeq();
  IWindowHandle whChild;

  bool                 bNoRatioForWindow = false;
  unsigned long        ulWindowCount = 0;
  unsigned long        ulRatioTotal = 0;
  IWindow::ChildCursor cwCursor( *this );

  for ( cwCursor.setToFirst(); cwCursor.isValid(); cwCursor.setToNext() )
  {
     whChild = this->childAt( cwCursor );
     IWindow* pwinChild = IWindow::windowWithHandle( whChild );
#ifdef IC_PMWIN
     if ( pwinChild )
#endif // IC_PMWIN
#ifdef IC_MOTIF
     // Make use of addRelated.
     bool isSplitBar=false;
     if ( pwinChild  &&  XmIsForm( (Widget)whChild ) )
     {
        Cardinal numChildren;
        WidgetList children;
        XtVaGetValues( (Widget) pwinChild->handle(),
                       XmNnumChildren, &numChildren,
                       XmNchildren, &children,
                       NULL );
        for ( int i=0; i < numChildren; i++ )
        {
           if ( children  &&
                XmIsSash( (Widget)children[i] ) )
           {
              isSplitBar = true;
              break;
           }
        }
     }
     if ( !isSplitBar )
#endif
     {                         // Have child control object.
        IPaneObj* ppcvChild = new IPaneObj( pwinChild );

        /**********************************************************/
        /* If the control has a ratio other than the canvas       */
        /* default ratio, get the controls ratio.                 */
        /**********************************************************/
        if ( pClSplitPercentageSet->containsElementWithKey( pwinChild ) )
        {
           ISplitPercentage* pdsrObject =
                       pClSplitPercentageSet->elementWithKey( pwinChild );
           ppcvChild->ulClPercentage = pdsrObject->ulClResolvedPercentage;
           ulWindowCount++;
           ulRatioTotal += ppcvChild->ulClPercentage;
        }
        else
        {
           bNoRatioForWindow = true;
        }
        pPaneSeq->addFirst( ppcvChild );
     }
  }

  /****************************************************************/
  /* If a window was added without setting a ratio for it, we     */
  /* need to give it a ratio value proportional to the ratios of  */
  /* the other windows (that already have a value).               */
  /****************************************************************/

  IPaneObj* ppcvObj;
  if ( bNoRatioForWindow )
  {
     ppcvObj = pPaneSeq->pFirstCl;
     while ( ppcvObj )
     {
        if ( !pClSplitPercentageSet->containsElementWithKey( ppcvObj->pwinCl ) )
        {
           /*******************************************************/
           /* This window does not have a ratio, so create one.   */
           /*******************************************************/
           if ( ( ulWindowCount != 0 )  &&  ( ulRatioTotal != 0 ) )
              ppcvObj->ulClPercentage = ulRatioTotal / ulWindowCount;
           else
              ppcvObj->ulClPercentage = 10L;
           this->setSplitWindowPercentage( ppcvObj->pwinCl,
                                           ppcvObj->ulClPercentage );
           ulRatioTotal += ppcvObj->ulClPercentage;
           ulWindowCount++;
        }
        ppcvObj = ppcvObj->pNext;
     }
  }

  if ( pPaneSeq->isEmpty() )  // No controls are in the canvas.
  {                           // Nothing more to do.
     delete pPaneSeq;
     return;
  }

  unsigned long ul100Ratio;
  unsigned long ulTotal100PercentUsed = 0;
  IPaneObj* ppcvWinnerOfPercents;

  ppcvObj = pPaneSeq->pFirstCl;
  while ( ppcvObj )
  {
     ul100Ratio = (ppcvObj->ulClPercentage * 100) / ulRatioTotal;
     this->setSplitWindowPercentage( ppcvObj->pwinCl, ul100Ratio );
     this->setResolvedPercentage( ppcvObj->pwinCl, ppcvObj->ulClPercentage );

     if ( ul100Ratio != 0 )
        ppcvWinnerOfPercents = ppcvObj;

     ulTotal100PercentUsed += ul100Ratio;
     ppcvObj = ppcvObj->pNext;
  }

  bClRatiosNotResolved = false;        // Clear flag.

  /*******************************************************************/
  /* Add the left over percentage to the last window with a non-zero */
  /* ratio.                                                          */
  /*******************************************************************/
  if ( ppcvWinnerOfPercents )
  {
     ul100Ratio = this->splitWindowPercentage( ppcvWinnerOfPercents->pwinCl );
     this->setSplitWindowPercentage( ppcvWinnerOfPercents->pwinCl,
                                     (100 - ulTotal100PercentUsed +
                                                          ul100Ratio) );
     /****************************************************************/
     /* Since setSplitWindowPercentage also changes the resolved     */
     /* value, we need to reset it back to the correct user setting. */
     /****************************************************************/
     this->setResolvedPercentage( ppcvWinnerOfPercents->pwinCl,
                                  ppcvWinnerOfPercents->ulClPercentage );
  }

  if ( pPaneSeq )
  {
     delete pPaneSeq;
  }

  return;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setOrientation                                                 |
|                                                                              |
| Change the panes orientation.                                                |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::setOrientation ( Orientation value )
{
  bool attrChanged = false;

  if ( value == verticalSplit  &&
       !( fCurrentStyle & vertical ) )
  {
     fCurrentStyle |= vertical;
     fCurrentStyle &= ~horizontal;
     this->setLayoutDistorted( IWindow::layoutChanged |
                                 IWindow::immediateUpdate, 0 );
     attrChanged = true;
#ifdef IC_MOTIF
     bClVertical = true;
#endif
  }
  else if ( value == horizontalSplit  &&
            ( fCurrentStyle & vertical ) )
  {
     fCurrentStyle |= horizontal;
     fCurrentStyle &= ~vertical;
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
     attrChanged = true;
#ifdef IC_MOTIF
     bClVertical = false;
#endif
  }

  if ( attrChanged )
  {
     this->notifyObservers( INotificationEvent( ISplitCanvas::orientationId,
                                                *this, true, (void*)value ) );
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::defaultStyle                                                   |
------------------------------------------------------------------------------*/
ISplitCanvas::Style ISplitCanvas::defaultStyle ( )
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::convertToGUIStyle                                              |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long ISplitCanvas::convertToGUIStyle ( const IBitFlag& guiStyle,
                                                bool bExtOnly ) const
{
  // Obtain the style from the class (ICanvas) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if ( bExtOnly )
  {
     // Use mask to only return extended styles in the user defined range.
     ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
     // Replace canvas window style with split canvas window style.
#ifdef IC_PMWIN
     ulStyle |= ( guiStyle.asUnsignedLong() & (unsigned long)~IC_CVS_CANVAS ) |
                                               IC_CVS_SPLITCANVAS;
#else
     // No styles are defined for the base canvas, so we do not mask.
     ulStyle |= guiStyle.asUnsignedLong();
#endif

     // No additional processing required since ICanvas::convertToGUIStyle is
     // doing all the required work ...
  }

  return ulStyle;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setLayoutDistorted                                             |
|                                                                              |
| Treat a size change like a layout change.                                    |
------------------------------------------------------------------------------*/
ISplitCanvas&
  ISplitCanvas::setLayoutDistorted ( unsigned long layoutAttributeOn,
                                     unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn =
     (layoutAttributeOn & (unsigned long)(~IWindow::sizeChanged));

  if ( layoutAttributeOn & IWindow::sizeChanged )
  {                               // Need to resize/reposition panes.
     ulFlagsOn |= IWindow::layoutChanged;  // Flag for ICanvas.
  }
  Inherited::setLayoutDistorted( ulFlagsOn, layoutAttributeOff );

  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setSplitBarThickness                                           |
|                                                                              |
| Sets the thickness of the specified portion of the split bar.                |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::setSplitBarThickness ( SplitBarArea value,
                                                   unsigned long thickness )
{
  if ( this->splitBarThickness( value ) != thickness )
  {
    if ( value == splitBarEdge )
    {
       ulClEdgeThickness = thickness;
    }
    else
    {
       ulClMiddleThickness = thickness;
    }
    this->setLayoutDistorted( IWindow::layoutChanged |
                                IWindow::immediateUpdate, 0 );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::splitBarThickness                                              |
|                                                                              |
| Get the thickness of the specified portion of the split bar.                 |
------------------------------------------------------------------------------*/
unsigned long ISplitCanvas::splitBarThickness ( SplitBarArea value )
{
  return (value == splitBarEdge) ? ulClEdgeThickness : ulClMiddleThickness;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::splitBarEdgeColor                                              |
|                                                                              |
| Returns the color of the edge of the splitbar of an ISplitCanvas.            |
------------------------------------------------------------------------------*/
IColor ISplitCanvas::splitBarEdgeColor ( ) const
{
  ITRACE_MOTIF_NOP();
  return clrClEdgeColor;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::splitBarMiddleColor                                            |
|                                                                              |
| Returns the color of the middle of the splitbar of an ISplitCanvas.          |
------------------------------------------------------------------------------*/
IColor ISplitCanvas::splitBarMiddleColor ( ) const
{
  return clrClMiddleColor;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setSplitBarEdgeColor                                           |
|                                                                              |
| Sets the color of the edge of the splitbar of an ISplitCanvas.               |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::setSplitBarEdgeColor ( const IColor& color )
{
  ITRACE_MOTIF_NOP();
  clrClEdgeColor = color;
#ifdef IC_PMWIN
  this->refresh();
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setSplitBarMiddleColor                                         |
|                                                                              |
| Sets the color of the middle of the splitbar of an ISplitCanvas.             |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::setSplitBarMiddleColor ( const IColor &color )
{
  clrClMiddleColor = color;

#ifdef IC_PMWIN
  this->refresh();
#endif
#ifdef IC_MOTIF
  for ( int index = 0; index < pClSplitBarSet->numberOfElements(); index++ )
  {
     ISplitBar *aSplitBar = pClSplitBarSet->elementWithKey( index )->splitBar();
     aSplitBar->setSplitBarColor( color );
  }
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::resetSplitBarEdgeColor                                         |
|                                                                              |
| Resets the color of the edge of the splitbar of an ISplitCanvas.             |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::resetSplitBarEdgeColor ( )
{
  ITRACE_MOTIF_NOP();
  clrClEdgeColor = IColor( IColor::darkGray );
#ifdef IC_PMWIN
  this->refresh();
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::resetSplitBarMiddleColor                                       |
|                                                                              |
| Resets the color of the middle of the splitbar of an ISplitCanvas.           |
------------------------------------------------------------------------------*/
ISplitCanvas& ISplitCanvas::resetSplitBarMiddleColor ( )
{
  ITRACE_MOTIF_NOP();
  clrClMiddleColor = IColor( IColor::paleGray );
#ifdef IC_PMWIN
  this->refresh();
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::setDefaultStyle                                                |
|                                                                              |
| Sets the default style.                                                      |
|                                                                              |
------------------------------------------------------------------------------*/
void  ISplitCanvas::setDefaultStyle ( const ISplitCanvas::Style& style )
{
  currentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| ISplitCanvas::orientation                                                    |
|                                                                              |
| Returns the orientation.                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
ISplitCanvas::Orientation  ISplitCanvas::orientation ( ) const
{
  return (fCurrentStyle & vertical ?
         ISplitCanvas::verticalSplit : ISplitCanvas::horizontalSplit);
}
