// Revision: 71 1.25.1.2 source/ui/basectl/isplitch.cpp, canvas, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: isplitch.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in isplitch.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINFRAMEMGR
#define INCL_WININPUT
#define INCL_WINMESSAGEMGR
#define INCL_WINPOINTERS
#define INCL_WINSYS
#define INCL_WINTRACKRECT
#define INCL_WINWINDOWMGR
extern "C" {
  #include <iwindefs.h>
#ifdef IC_MOTIF
  #include <Xm/SashP.h>
#endif
}

#include <isplitch.hpp>

#include <icconst.h>
#include <icmnfun.hpp>
#include <icolor.hpp>
#include <icoordsy.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ibundles.hpp>
#include <iexgrprt.hpp>
#include <igbase2d.hpp>
#include <igrport.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <isizeevt.hpp>
#include <isplitcv.hpp>
#include <itrace.hpp>
#ifdef IC_MOTIF
  #include <ikeyevt.hpp>
  #include <isizrect.hpp>
  #include <ispltbar.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ISPLITCH_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_PMWIN
#include <iqu.h>

class ISplitBarInfo {
public:
  ISplitBarInfo ( )
    : before(0)
    , after(0)
    { }
  ISplitBarInfo ( const IRectangle&    rect,
                  IWindow*             winBefore,
                  IWindow*             winAfter )
    : splitBar(rect)
    , before(winBefore)
    , after(winAfter)
    { }
  ISplitBarInfo ( const ISplitBarInfo& right )
    : splitBar(right.splitBar)
    , before(right.before)
    , after(right.after)
    { }

 ~ISplitBarInfo ( )
    { }

ISplitBarInfo
 &operator= ( const ISplitBarInfo& right )
    {
      splitBar = right.splitBar;
      before = right.before;
      after = right.after;
      return *this;
    }

IRectangle
  splitBar;
IWindow
 *before;
IWindow
 *after;
}; // ISplitBarInfo

class ISplitCanvasHandlerData {
public:
  ISplitCanvasHandlerData ( )
    { }

IQueue< ISplitBarInfo >
  splitBarRects;
}; // ISplitCanvasHandlerData
#endif  // IC_PMWIN

#ifdef IC_MOTIF
#pragma enum(4)
#pragma pack(push,4)

class ISplitCanvasHandlerData {
public:
  ISplitCanvasHandlerData ( )
    { }
 ~ISplitCanvasHandlerData ( )
    { }
bool
  isSplitBar         ( const IWindowHandle& form );
ISplitCanvasHandlerData
 &repositionControls ( ISplitCanvas*     splitcv,
                       ISplitBar*        splitBar );
IRectangle
  splitBarPosition   ( ISplitBar*        splitBar,
                       const IRectangle& boundingRectangle,
                       IPair::Coord      deltaPosition,
                       IPair::Coord      offset,
                       IPair::Coord      length );
ISplitCanvasHandlerData&
  drawSplitIndicator ( const IRectangle& lineRect,
                       const GC&         gcSet,
                       Display*          display,
                       ISplitCanvas*     splitcv );

private:
  ISplitCanvasHandlerData ( const ISplitCanvasHandlerData& );
ISplitCanvasHandlerData
 &operator=               ( const ISplitCanvasHandlerData& );
}; // ISplitCanvasHandlerData

#pragma pack(pop)
#pragma enum(pop)

ISplitCanvasHandlerData&
  ISplitCanvasHandlerData::drawSplitIndicator( const IRectangle& lineRect,
                                               const GC& gcSet,
                                               Display* display,
                                               ISplitCanvas* splitcv )
{
  if ( lineRect != IRectangle() )
     XDrawLine( display,
                XtWindow( (Widget) splitcv->handle() ),
                gcSet,
                lineRect.left(),
                lineRect.bottom(),
                lineRect.right(),
                lineRect.top() );
  return *this;
}

IRectangle ISplitCanvasHandlerData::splitBarPosition( ISplitBar *splitBar,
                                      const IRectangle& boundingRectangle,
                                            IPair::Coord deltaPosition,
                                            IPair::Coord offset,
                                            IPair::Coord length )
{
  // All values are assumed to be in native coordinates.
  IPoint newOrigin;
  IPoint newCorner;
  if ( splitBar->orientation() == ISplitBar::verticalSplit )
  {
     newOrigin = IPoint( deltaPosition, offset );
     newCorner = IPoint( deltaPosition, length );
  }
  else
  {
    newOrigin = IPoint(offset , deltaPosition );
    newCorner = IPoint(length , deltaPosition );
  }

  if ( !boundingRectangle.contains( newOrigin ) )
  {
     if ( splitBar->orientation() == ISplitBar::verticalSplit )
     {
        // fix X
        if ( newOrigin.x() < boundingRectangle.left() )
        {
           newOrigin.setX( boundingRectangle.left() );
           newCorner.setX( boundingRectangle.left() );
        }
        else if ( newOrigin.x() >= boundingRectangle.right() )
        {
           newOrigin.setX( boundingRectangle.right() - 1 );
           newCorner.setX( boundingRectangle.right() - 1 );
        }
     }
     else
     {  // Horizontal split bars.
        // fix Y
        if ( newOrigin.y() >= boundingRectangle.maxY() )
        {
           newOrigin.setY( boundingRectangle.maxY() - 1);
           newCorner.setY( boundingRectangle.maxY() - 1);
        }
        else if ( newOrigin.y() < boundingRectangle.minY() )
        {
           newOrigin.setY( boundingRectangle.minY() );
           newCorner.setY( boundingRectangle.minY() );
        }
     }
  }
  return IRectangle( newOrigin, newCorner );
}

ISplitCanvasHandlerData&
  ISplitCanvasHandlerData::repositionControls( ISplitCanvas *splitcv,
                                               ISplitBar    *splitBar )
{
  IWindow* pwinChildPrev = 0;
  IWindow* pwinChild = 0;
  IWindow* pwinSizeChild = 0;
  IRectangle rclChildPrev;
  IRectangle rclChild;
  IRectangle rclSplitBar;
  ISize szParentSize;

  IWindowHandle whChild;
  IWindow::ChildCursor cwCursor(*splitcv);

  pwinChildPrev = splitBar->prevControl();
  rclChildPrev = pwinChildPrev->nativeRect();
  pwinChild = splitBar->nextControl();
  rclChild = pwinChild->nativeRect();

  rclSplitBar = splitBar->nativeRect();
  szParentSize = pwinChild->parent()->size();

  if ( splitBar->orientation() == ISplitBar::verticalSplit )
  {  // Vertical split bars.
     pwinChildPrev->sizeTo( ISize(splitBar->position().x() -
                                  rclChildPrev.left(),
                                  rclChildPrev.height()) );
     pwinChild->moveSizeTo(ICoordinateSystem::convertToApplication(
                IRectangle( rclSplitBar.bottomRight(),
                            rclChild.topRight() ),
                            szParentSize) );

     /********************************************************/
     /* If the split bar was moved, the control ratios are   */
     /* invalid and must be recalculated based on the current*/
     /* control window sizes.                                */
     /********************************************************/

     /********************************************************/
     /* Save new window sizes.                               */
     /********************************************************/
     for ( cwCursor.setToFirst(); cwCursor.isValid(); cwCursor.setToNext() )
     {  // Enumerate all child windows.
        whChild = splitcv->childAt( cwCursor );
        if ( !isSplitBar(whChild) )
        {
           pwinSizeChild = IWindow::windowWithHandle( whChild );
           splitcv->setResolvedPercentage( pwinSizeChild,
                                           pwinSizeChild->size().width() );
        }
     }
  }
  else
  {  // Horizontal split bars.
     pwinChild->moveSizeTo( ICoordinateSystem::convertToApplication(
                              IRectangle( rclSplitBar.topLeft(),
                                          rclChild.topRight() ),
                              szParentSize) );
     pwinChildPrev->moveSizeTo(ICoordinateSystem::convertToApplication(
                                IRectangle( rclChildPrev.bottomLeft(),
                                   ISize( rclChildPrev.width(),
                                          rclSplitBar.bottom() -
                                          rclChildPrev.bottom() ) ),
                                szParentSize) );

     /*********************************************************/
     /* If the split bar was moved, the control ratios are    */
     /* invalid and must be recalculated based on the         */
     /* current control window sizes.                         */
     /*********************************************************/

     /********************************************************/
     /* Save new window sizes.                               */
     /********************************************************/
     for ( cwCursor.setToFirst(); cwCursor.isValid(); cwCursor.setToNext() )
     {  // Enumerate all child windows.
        whChild = splitcv->childAt( cwCursor );
        if ( !isSplitBar(whChild) )
        {
           pwinSizeChild = IWindow::windowWithHandle( whChild );
           splitcv->setResolvedPercentage( pwinSizeChild,
                                           pwinSizeChild->size().height() );
        }
     }
  }

  /*********************************************************/
  /* Must update percentages based on 100% also.           */
  /*********************************************************/
  splitcv->resolveRatios();
  splitcv->setLayoutDistorted( 0, IWindow::layoutChanged );
                                       // Prevent re-layout of controls

  splitcv->bClRatiosNotResolved = false;
                                       // Flag all percentages match
  /*********************************************************/
  /* Finished updating percentages based on 100%.          */
  /*********************************************************/

  ITRACE_DEVELOP( IString( (unsigned long)(Widget)
                               ( splitBar->prevControl()->handle() ) ) +
                  IString( "Prev Control Ratio is  = " ) +
                  IString( splitcv->resolvedPercentage(
                           splitBar->prevControl()) ) );
  ITRACE_DEVELOP( IString( (unsigned long)(Widget)
                               ( splitBar->nextControl()->handle() ) ) +
                  IString( "Child Control Ratio is = " ) +
                  IString( splitcv->resolvedPercentage(
                           splitBar->nextControl()) ) );
  return *this;
}

bool
  ISplitCanvasHandlerData::isSplitBar ( const IWindowHandle& form )
{
  Cardinal numChildren;
  WidgetList children;
  if ( XmIsForm( (Widget) form ) )
  {
     XtVaGetValues( (Widget) form,
                    XmNnumChildren, &numChildren,
                    XmNchildren, &children,
                    NULL );
     for ( int i=0; i < numChildren; i++ )
     {
         if ( children  &&
              XmIsSash( (Widget)children[i] ) )
            return true;
     }
  }
  return false;
}

#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| ISplitCanvasHandler::ISplitCanvasHandler                                     |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
ISplitCanvasHandler::ISplitCanvasHandler ( )
  : ISplitCanvasHandler::Inherited ( )
  , fSplitCanvasHandlerData( new ISplitCanvasHandlerData )
{ }

/*------------------------------------------------------------------------------
| ISplitCanvasHandler::~ISplitCanvasHandler                                    |
|                                                                              |
| Empy destructor here for page tuning.                                        |
------------------------------------------------------------------------------*/
ISplitCanvasHandler::~ISplitCanvasHandler ( )
{
  if ( fSplitCanvasHandlerData )
     delete fSplitCanvasHandlerData;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ISplitCanvasHandler::dispatchHandlerEvent                                    |
|                                                                              |
| Processes events as appropriate for a split canvas.                          |
------------------------------------------------------------------------------*/
bool ISplitCanvasHandler::dispatchHandlerEvent ( IEvent& evt )
{
  switch ( evt.eventId() )
  {
     case WM_PAINT:
     /***************************************************************/
     /* Paint any invalidated split bars (let the default canvas    */
     /* handler paint the background of the canvas if it has no     */
     /* child windows).                                             */
     /***************************************************************/
     {
        HWND hwnd = evt.handle();
        ISplitCanvas* splitcv = (ISplitCanvas*)evt.window();
        IRectangle paintRect;
        bool
          painted = false,
          bordered = splitcv->hasBorder();

        // paint background
#ifdef IC_WIN
        RECT updateRect;
#else
        RECTL updateRect;
#endif
        // The below assigns paintRect a value in native coordinates.
        IQUERYUPDATERECT( splitcv->handle(), &updateRect );
        paintRect = IRectangle( updateRect );
        splitcv->defaultProcedure(evt);

        if ( bordered )
        {  // Need to let the default canvas painting draw the borders,
           // but we need to paint the split bars afterwards.
           painted = true;
        }
        else
        {
           IINVALIDATERECT( splitcv->handle(), &updateRect, false );
        }

        IWindow::ChildCursor cwCursor( *splitcv );
        if ( cwCursor.setToFirst() )
        {          // Have at least one child window.

           IPresSpaceHandle pshCanvas;

           splitcv->layout();
           IRectangle
             childArea( IPoint(), splitcv->size() );

#ifdef IC_PM
           RECTL rect;
#endif
#ifdef IC_WIN
           PAINTSTRUCT rect;
#endif
           if ( bordered )
           {
              // Set paintRect so it doesn't include the border (if any),
              // to avoid having the split bars paint over it.
              pshCanvas = splitcv->presSpace();
              // Need to hide the cursor to avoid painting over it.  IC_NOTYET
              childArea = splitcv->rectangleInsideBorder( childArea.size() );

              if ( ICoordinateSystem::isConversionNeeded() )
              {
                 childArea =
                   ICoordinateSystem::convertToNative
                              ( childArea, splitcv->size() );
              }
              paintRect &= childArea;
           }
           else
           {
              // The below assigns paintRect a value in native coordinates.
              pshCanvas = IPresSpaceHandle( IBEGINPAINT( hwnd, 0L, &rect ),
                                            splitcv->handle() );
#ifdef IC_PM
              paintRect = IRectangle( rect );
#endif
#ifdef IC_WIN
              paintRect = IRectangle( rect.rcPaint.left,
                                      rect.rcPaint.top,
                                      rect.rcPaint.right,
                                      rect.rcPaint.bottom );
#endif
           }
           if ( ! (splitcv->fCurrentStyle & ISplitCanvas::noSplitBars) )
           {
              IWindowHandle whChild;
              IWindow* pwinChild     = 0;
              IWindow* pwinChildPrev = 0;
              bool rightToLeft = false;
#ifdef IC_PMWIN
              if ( IBidiSettings::isBidiSupported() )
              {
                 IBidiSettings bidi(*splitcv);
                 if ( bidi.windowLayout() == IBidiSettings::layoutRightToLeft )
                    rightToLeft = true;
              }
#endif
              fSplitCanvasHandlerData->splitBarRects.removeAll();

              for ( ; cwCursor.isValid(); cwCursor.setToNext() )
              {                   // Enumerate all child windows.
                 whChild = splitcv->childAt( cwCursor );
                 pwinChild = IWindow::windowWithHandle( whChild );
                 if ( pwinChild )
                 {                // Have child window object.
                    if ( pwinChildPrev )
                    {
                       IRectangle rclSplit;
                       if (splitcv->orientation() == ISplitCanvas::verticalSplit)
                       {
                          IRectangle rclControl = pwinChildPrev->nativeRect();
                          if ( rightToLeft )
                            rclControl = pwinChild->nativeRect();
                          rclSplit =
                            IRectangle( rclControl.maxX(),
                                        childArea.minY(),
                                        rclControl.maxX() +
                                          (splitcv->ulClEdgeThickness * 2) +
                                          splitcv->ulClMiddleThickness,
                                        childArea.maxY() );
                       }
                       else
                       {
#ifdef IC_ORIGINUPPERLEFT
                          IRectangle rclControl = pwinChildPrev->nativeRect();
#else
                          IRectangle rclControl = pwinChild->nativeRect();
#endif
                          rclSplit =
                            IRectangle( childArea.minX(),
                                        rclControl.maxY(),
                                        childArea.maxX(),
                                        rclControl.maxY() +
                                          (splitcv->ulClEdgeThickness * 2) +
                                          splitcv->ulClMiddleThickness );
                       }

                       fSplitCanvasHandlerData->splitBarRects.addAsLast( ISplitBarInfo( rclSplit,
                                                                           pwinChildPrev,
                                                                           pwinChild) );
                       if ( rclSplit.intersects( paintRect ) )
                       {
                          ITRACE_DEVELOP( IString( "Split Rect : " ) +
                                            rclSplit.asString() );

                          this->paintSplitBar( splitcv, pshCanvas, rclSplit );
                       }
                    }
                    pwinChildPrev = pwinChild;
                 }
              }
           }

           if ( bordered )
           {
              splitcv->releasePresSpace( pshCanvas );
           }
           else
           {
              IENDPAINT( hwnd, pshCanvas, &rect );
           }
           evt.setResult( 0 );
        }               // End have at least one child window.

        return painted;
     } /* end WM_PAINT */

#ifdef IC_PM
     case WM_MOUSEMOVE:
#endif
#ifdef IC_WIN
     case WM_SETCURSOR:
#endif
     /***************************************************************/
     /* If the mouse pointer is over a split bar, set it to the     */
     /* sizeHorizontal/sizeVertical pointer.                        */
     /***************************************************************/
     {
        ISplitCanvas* splitcv = (ISplitCanvas*)evt.window();
        if ( splitcv->isEnabled() )
        {               // Canvas and its split bars are enabled.
#ifdef IC_WIN
           if ( splitcv->handle().asUnsigned() ==
                     evt.parameter1().asUnsignedLong() )
           {
#endif
           IWindow::ChildCursor cwCursor( *splitcv );
           if ( cwCursor.setToFirst() )
           {            // Has at least one child window.
              bool
                changeMousePointer = true;
              IPoint mousePoint;
              POINTL ptScreen;

              IQUERYMSGPOS( IQUERYANCHOR( splitcv->handle() ), &ptScreen );
              IMAPWINDOWPOINTS( IWindow::desktopWindow()->handle(),
                          splitcv->handle(), &ptScreen, 1 );
              mousePoint = IPoint( ptScreen.x,ptScreen.y );

              // Check that the mouse pointer is inside any borders.
              if ( splitcv->hasBorder() )
              {
                 IRectangle
                   childArea( splitcv->rectangleInsideBorder( splitcv->size() ) );
                 if ( ICoordinateSystem::isConversionNeeded() )
                 {
                    // Compensate for the affect of changing coordinate
                    // systems on IRectangle::contains (the maxY edge of the
                    // rectangle in native coordinates would not contain the
                    // point, but the same edge does when it becomes the minY
                    // edge in application coordinates).
                    childArea
                     .moveBy( IPair( 0, 1 ) );
                    childArea
                     .sizeTo( childArea.size() - ISize( 0, 1 ) );
                 }
                 if ( ! childArea.contains( mousePoint ) )
                 {  // Mouse pointer is over part of the border.
                    changeMousePointer = false;
                 }
              }

              ICursor& splitRectCursor = *fSplitCanvasHandlerData->splitBarRects.newCursor();
              ISplitBarInfo splitBarInfo;
              bool bOverSplitBar = false;

              for ( splitRectCursor.setToFirst(); splitRectCursor.isValid() && changeMousePointer;                                         splitRectCursor.setToNext() )
              {                         // Enumerate all child windows.
                 splitBarInfo = fSplitCanvasHandlerData->splitBarRects.elementAt( splitRectCursor );

                 if ( splitBarInfo.splitBar.contains( mousePoint ))
                 {
                    bOverSplitBar = true;
                    break;        // Leave enumeration loop.
                 }
              }
              delete &splitRectCursor;

              if ( changeMousePointer && bOverSplitBar )
              {
                 // Now determine the correct pointer to use.
                 ISystemPointerHandle
                   splitPtr( ISystemPointerHandle::sizeHorizontal );
                 if ( splitcv->orientation() == ISplitCanvas::horizontalSplit )
                 {  // Horizontal split bars.
                    splitPtr =
                      ISystemPointerHandle( ISystemPointerHandle::sizeVertical );
                 }

#ifdef IC_PM
                 // Give a window in the owner window chain the
                 // opportunity to change the mouse pointer.  Do this
                 // via the WM_CONTROLPOINTER message.
                 IWindowHandle wHandle = splitcv->owner() ?
                                           splitcv->owner()->handle() :
                                           splitcv->handle();
                 IEventResult newPtr =
                    wHandle.sendEvent( WM_CONTROLPOINTER,
                                       IEventParameter1( splitcv->id() ),
                                       IEventParameter2( splitPtr ) );
                 ISETPOINTER( newPtr );
#endif
#ifdef IC_WIN
                 // Give a window in the parent window chain the
                 // opportunity to set the mouse pointer.  Do this via
                 // the WM_SETCURSOR message.
                 IWindowHandle
                   parentHwnd = IPARENTOF( splitcv->handle() );
                 IEventResult
                   newPtrSet = parentHwnd.sendEvent( WM_SETCURSOR,
                                                     evt.parameter1(),
                                                     evt.parameter2() );
                 if ( ! newPtrSet.asUnsignedLong() )
                 {  // A window in the parent chain did not set the mouse
                    // pointer, so set it now.
                    ISETPOINTER( splitPtr );
                 }
#endif

                 evt.setResult( true );
                 return true;
              }         // End change the mouse pointer.
           }            // End have at least one child window.
#ifdef IC_WIN
           }
#endif
        }               // End canvas is enabled.
        return false;   // No mouse pointer processing needed.
#ifdef IC_PM
     } /* end WM_MOUSEMOVE */
#endif
#ifdef IC_WIN
     } /* end WM_SETCURSOR */
#endif

     case WM_BUTTON1DOWN:
     case WM_BUTTON2DOWN:
     case WM_BUTTON3DOWN:
     /***************************************************************/
     /* If mouse button down on a split bar, start tracking the     */
     /* mouse, and resize the child windows on both sides of split  */
     /* bar according to where the mouse button was released.       */
     /***************************************************************/
     {
        HWND hwnd = evt.handle();
        ISplitCanvas* splitcv = (ISplitCanvas*)evt.window();
        POINTL ptScreen;
        IWindow* pwinChild     = 0;
        IWindow* pwinChildPrev = 0;
        IWindow* pwinSizeChild = 0;
        unsigned long ulSplitBarWidth = (2 * splitcv->ulClEdgeThickness) +
                                           splitcv->ulClMiddleThickness;

        /**********************************************************/
        /* Compute mouse location within the split canvas window. */
        /**********************************************************/
        IQUERYMSGPOS( IQUERYANCHOR( hwnd ), &ptScreen );
        IMAPWINDOWPOINTS( IWindow::desktopWindow()->handle(),
                          hwnd, &ptScreen, 1 );
        IPoint ptMouse( ptScreen.x,ptScreen.y );
        ITRACE_ALL( IString("ptMouse=") + ptMouse.asString() );

        /*********************************************************/
        /* Used to flag if button down was over a split bar.     */
        /*********************************************************/
        bool bOverSplitBar = false;

#ifdef IC_PMWIN
        /*********************************************************/
        /* Check for BIDI right-to-left layout.                  */
        /*********************************************************/
        bool rightToLeft = false;
        if ( IBidiSettings::isBidiSupported() )
        {
           IBidiSettings bidi(*splitcv);
           if ( bidi.windowLayout() == IBidiSettings::layoutRightToLeft )
              rightToLeft = true;
        }
#endif
        bOverSplitBar = false;
        ICursor& splitRectCursor = *fSplitCanvasHandlerData->splitBarRects.newCursor();
        ISplitBarInfo splitBarInfo;

        for ( splitRectCursor.setToFirst(); splitRectCursor.isValid(); splitRectCursor.setToNext() )
        {                         // Enumerate all child windows.
           splitBarInfo = fSplitCanvasHandlerData->splitBarRects.elementAt( splitRectCursor );

           if ( splitBarInfo.splitBar.contains( ptMouse ))
           {
              bOverSplitBar = true;
              break;        // Leave enumeration loop.
           }
        }
        delete &splitRectCursor;

        if ( ! bOverSplitBar )  // Button down was not over a split bar.
           return false;

        IRectangle rclChild;
        IRectangle rclChildPrev;
        pwinChildPrev = splitBarInfo.before;
        pwinChild = splitBarInfo.after;
        rclChild = pwinChild->nativeRect();
        rclChildPrev = pwinChildPrev->nativeRect();

        IWindow::ChildCursor cwCursor( *splitcv );
        IWindowHandle whChild(pwinChild->handle());

        /*********************************/
        /* Clear WS_CLIPCHILDREN bit.    */
        /*********************************/
        long lStyle = ISTYLEOF( hwnd );
        ISETWINDOWSTYLE( hwnd, (lStyle & ~WS_CLIPCHILDREN) );

        // Start tracking of mouse.
        TRACKINFO trk;
        /************************************************************/
        /* The split bar should track below the titlebar, and not   */
        /* go any further right than the right edge of the child    */
        /* window to the right of the split bar, and not go any     */
        /* further left than the left edge of the child window to   */
        /* left of the split bar.                                   */
        /************************************************************/
        if ( splitcv->orientation() == ISplitCanvas::verticalSplit )
        {
           if (!rightToLeft)
           {
             trk.rclBoundary.MINX = rclChildPrev.minX();
             trk.rclBoundary.MINY = rclChild.minY();
             trk.rclBoundary.MAXX = rclChild.maxX();
             trk.rclBoundary.MAXY = rclChildPrev.maxY();
           }
           else
           {
             trk.rclBoundary.MINX = rclChild.minX();
             trk.rclBoundary.MINY = rclChildPrev.minY();
             trk.rclBoundary.MAXX = rclChildPrev.maxX();
             trk.rclBoundary.MAXY = rclChild.maxY();
           }

           /********************************************************/
           /* This specifies the width of the split bar, so        */
           /* WinTrackRect knows how wide of a thing to track. The */
           /* split bar is the width of a size border.             */
           /********************************************************/
           trk.cxBorder = (int)ulSplitBarWidth;

           /********************************************************/
           /* This specifies the height of the split bar, so       */
           /* WinTrackRect knows how tall of a thing to track.     */
           /********************************************************/
           trk.cyBorder = (int)( trk.rclBoundary.MAXY -
                                 trk.rclBoundary.MINY );

           /********************************************************/
           /* This specifies the tracking rectangle used by        */
           /* WinTrackRect.                                        */
           /********************************************************/
           if (!rightToLeft)
           {
             trk.rclTrack.MINX = rclChildPrev.maxX();
             trk.rclTrack.MINY = rclChild.minY();
             trk.rclTrack.MAXX = rclChild.minX();
             trk.rclTrack.MAXY = trk.rclBoundary.MAXY;
           }
           else
           {
             trk.rclTrack.MINX = rclChild.maxX();
             trk.rclTrack.MINY = rclChildPrev.minY();
             trk.rclTrack.MAXX = rclChildPrev.minX();
             trk.rclTrack.MAXY = trk.rclBoundary.MAXY;
           }
        }
        else
        {          // Horizontal split bars.
#ifdef  IC_PM   // originLowerLeft
           trk.rclBoundary.MINY = rclChild.minY();
           trk.rclBoundary.MAXY = rclChildPrev.maxY();
#endif
#ifdef IC_WIN   // originUpperLeft
           trk.rclBoundary.MINY = rclChildPrev.minY();
           trk.rclBoundary.MAXY = rclChild.maxY();
#endif

           trk.rclBoundary.MINX = rclChild.minX();
           trk.rclBoundary.MAXX = rclChild.maxX();

           /********************************************************/
           /* This specifies the length of the split bar, so       */
           /* WinTrackRect knows how wide of a thing to track.     */
           /* The split bar is the width of a size border.         */
           /********************************************************/
           trk.cxBorder = (int)( trk.rclBoundary.MAXX -
                                 trk.rclBoundary.MINX );

           /********************************************************/
           /* This specifies the height of the split bar, so       */
           /* WinTrackRect knows how tall of a thing to track.     */
           /* The split bar is the width of a size border.         */
           /********************************************************/
           trk.cyBorder = (int)ulSplitBarWidth;

           /********************************************************/
           /* This specifies the tracking rectangle used by        */
           /* WinTrackRect.                                        */
           /********************************************************/
           trk.rclTrack.MINX = rclChild.minX();
           trk.rclTrack.MAXX = rclChild.maxX();
#ifdef  IC_PM   // originLowerLeft
           trk.rclTrack.MINY = rclChild.maxY();
           trk.rclTrack.MAXY = rclChildPrev.minY();
#endif
#ifdef IC_WIN   // originUpperLeft
           trk.rclTrack.MINY = rclChildPrev.maxY();
           trk.rclTrack.MAXY = rclChild.minY();
#endif
        }

        trk.cxGrid = 0;
        trk.cyGrid = 0;
        trk.cxKeyboard = 0;
        trk.cyKeyboard = 0;

        /**********************************************************/
        /* Specify the minimum and maximum tracking size.         */
        /**********************************************************/
        trk.ptlMinTrackSize.x = trk.cxBorder;
        trk.ptlMinTrackSize.y = trk.cyBorder;
        trk.ptlMaxTrackSize.x = trk.cxBorder;
        trk.ptlMaxTrackSize.y = trk.cyBorder;

        /**********************************************************/
        /* These tracking options specify:  TF_MOVE - to center   */
        /* the pointer in the tracking rectangle. TF_ALLINBOUNDARY*/
        /* performs tracking such that no part of the tracking    */
        /* rectangle ever falls outside the bounding rectangle.   */
        /**********************************************************/
        trk.fs = TF_MOVE | TF_ALLINBOUNDARY;

        WinTrackRect( hwnd, 0, &trk );

        /**********************************************************/
        /* Restore WS_CLIPCHILDREN bit.                           */
        /**********************************************************/
        ISETWINDOWSTYLE( hwnd, lStyle );

        /************************************************************/
        /* Resize the two child windows affected by the movement of */
        /* the split bar.  The child to the right of the split bar  */
        /* split bar has to be moved and resized.  The child to the */
        /* left of the split bar only has to be resized.            */
        /* In the case of a horizontal split bar we have to move    */
        /* and size both controls to insure correct operation when  */
        /* the application and native coordinate systems are not    */
        /* the same.                                                */
        /************************************************************/
        if ( splitcv->orientation() == ISplitCanvas::verticalSplit )
        {
           ISize cvSize = splitcv->size();

           if ( !rightToLeft )
           {
              pwinChildPrev->sizeTo( ISize( trk.rclTrack.MINX -
                                              rclChildPrev.minX(),
                                            rclChildPrev.maxY() -
                                              rclChildPrev.minY() ) );

              pwinChild->moveSizeTo(
                 ICoordinateSystem::convertToApplication(
                    IRectangle( trk.rclTrack.MAXX,
                                rclChild.minY(),
                                rclChild.maxX(),
                                rclChild.maxY() ),
                    cvSize ) );
           }
           else
           {
              pwinChild->sizeTo( ISize( trk.rclTrack.MINX -
                                              rclChild.minX(),
                                            rclChild.maxY() -
                                              rclChild.minY() ) );

              pwinChildPrev->moveSizeTo(
                 ICoordinateSystem::convertToApplication(
                    IRectangle( trk.rclTrack.MAXX,
                                rclChildPrev.minY(),
                                rclChildPrev.maxX(),
                                rclChildPrev.maxY() ),
                    cvSize ) );
           }

           /*************************************************************/
           /* Save new window sizes.                                    */
           /*************************************************************/
           for ( cwCursor.setToFirst();
                 cwCursor.isValid();
                 cwCursor.setToNext() )
           {            // Enumerate all child windows.
              whChild = splitcv->childAt( cwCursor );
              pwinSizeChild = IWindow::windowWithHandle( whChild );
              splitcv->setResolvedPercentage( pwinSizeChild,
                                              pwinSizeChild->size().width() );
           }
        }
        else
        {
           ISize cvSize = splitcv->size();
#ifdef IC_ORIGINLOWERLEFT
           pwinChild->moveSizeTo(
              ICoordinateSystem::convertToApplication(
                 IRectangle( rclChild.minXMinY(),
                             ISize( rclChild.maxX() -
                                      rclChild.minX(),
                                    trk.rclTrack.MINY -
                                      rclChild.minY() ) ),
                 cvSize ) );

           ITRACE_ALL( IString("pwinChild new rect=") +
                       IRectangle( rclChild.minXMinY(),
                                   ISize( rclChild.maxX() -
                                            rclChild.minX(),
                                          trk.rclTrack.MINY -
                                            rclChild.minY() ) ).asString() );

           pwinChildPrev->moveSizeTo(
              ICoordinateSystem::convertToApplication(
                 IRectangle( rclChildPrev.minX(),
                             trk.rclTrack.MAXY,
                             rclChildPrev.maxX(),
                             rclChildPrev.maxY() ) ,
                 cvSize ) );

           ITRACE_ALL( IString("pwinChildPrev new rect=") +
                       IRectangle( rclChildPrev.minX(),
                                   trk.rclTrack.MAXY,
                                   rclChildPrev.maxX(),
                                   rclChildPrev.maxY() ).asString() );
#else // IC_ORIGINUPPERLEFT
           pwinChild->moveSizeTo(
              ICoordinateSystem::convertToApplication(
                 IRectangle( rclChild.minX(),
                             trk.rclTrack.MAXY,
                             rclChild.maxX(),
                             rclChild.maxY() ),
                 cvSize ) );

           ITRACE_ALL( IString("pwinChild new rect=") +
                       IRectangle( rclChild.minX(),
                                   trk.rclTrack.MAXY,
                                   rclChild.maxX(),
                                   rclChild.maxY() ).asString() );

           pwinChildPrev->moveSizeTo(
              ICoordinateSystem::convertToApplication(
                 IRectangle( rclChildPrev.minXMinY(),
                             ISize( rclChildPrev.width(),
                                    trk.rclTrack.MINY -
                                      rclChildPrev.minY() ) ),
                 cvSize ) );

           ITRACE_ALL( IString("pwinChildPrev new rect=") +
                       IRectangle( rclChildPrev.minXMinY(),
                                   ISize( rclChildPrev.width(),
                                          trk.rclTrack.MINY -
                                            rclChildPrev.minY() ) ).asString() );
#endif // IC_ORIGINUPPERLEFT

           /*************************************************************/
           /* Save new window sizes.                                    */
           /*************************************************************/
           for ( cwCursor.setToFirst();
                 cwCursor.isValid();
                 cwCursor.setToNext() )
           {            // Enumerate all child windows.
              whChild = splitcv->childAt( cwCursor );
              pwinSizeChild = IWindow::windowWithHandle( whChild );
              splitcv->setResolvedPercentage( pwinSizeChild,
                                              pwinSizeChild->size().height() );
           }
        }

        /*********************************************************/
        /* Resolve rations base on 100%.                         */
        /*********************************************************/
        splitcv->resolveRatios();
        splitcv->setLayoutDistorted( 0, IWindow::layoutChanged );
                             // Prevent re-layout of controls.

        /*********************************************************/
        /* Say we processed event.                               */
        /*********************************************************/
        IINVALIDATERECT( hwnd, &(trk.rclTrack), false );

        // Update the window so the split bar rects get updated
        IUPDATEWINDOW( hwnd );

        return false;
     } /* end WM_BUTTONxDOWN */

     case WM_SIZE:
     {
        /************************************************************/
        /* Resize the child controls based on the percentage of the */
        /* window they occupied when the window was resized, thus   */
        /* maintaining their ratios to each other.                  */
        /************************************************************/
        ISplitCanvas* splitcv = (ISplitCanvas*)evt.window();
        splitcv->sizClCanvasSize = ISize( evt.parameter2().number1(),
                                          evt.parameter2().number2() );
        splitcv->setLayoutDistorted( IWindow::sizeChanged, 0 );
        splitcv->layout();

        break;
     }
  } /* endswitch */

  return false;
}
#endif // IC_PMWIN

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ISplitCanvasHandler::dispatchHandlerEvent                                    |
|                                                                              |
| Processes events as appropriate for a split canvas.                          |
------------------------------------------------------------------------------*/
bool ISplitCanvasHandler::dispatchHandlerEvent ( IEvent& evt )
{
  bool brc = false;

  static GC gcSet;
  // Use these to calculate how far split bars have been dragged.
  static Dimension
    mouseDownX, mouseDownY, origSeparatorX, origSeparatorY;
  // Use these to minimize the number of queries during a drag.
  static Dimension
    separatorHeight, separatorWidth;
  static IRectangle prevLine, boundingRectangle;
  static bool moveInProgress(false);    // protect against mouse and keyboard at the same
                             // time, still need to add this protection

  switch ( evt.eventId() )
  {
    case xEvent( Expose ):
    {
       XEvent
         *xEvent = (XEvent*)(char*) evt.parameter2();
       XExposeEvent
         *exposeEvent = (XExposeEvent*)xEvent;
       unsigned long
          exposeCount = exposeEvent->count;
       if ( exposeCount != 0 )
       {  // More expose messages still to come.
          break;
       }
       else
       {
         if ( !fSplitCanvasHandlerData->isSplitBar(
                 (IWindowHandle) evt.controlHandle() ) )
         {
           ISplitCanvas* splitcv = (ISplitCanvas*)evt.controlWindow();

           if ( splitcv )
             splitcv->layout();

           evt.setResult((unsigned long)false);
           brc = true;                    // Do not pass on to next handler
         }
       }
       break;
    } /* end Expose   */
    case xEvent( MapNotify ):
    {
      if ( !fSplitCanvasHandlerData->isSplitBar(
              (IWindowHandle) evt.controlHandle() ) )
      {
        if (evt.parameter1().asUnsignedLong() )
        {
          ISplitCanvas* splitcv = (ISplitCanvas*)evt.controlWindow();

          if ( splitcv )
            splitcv->layout();

          evt.setResult((unsigned long)false);
          brc = true;                    // Do not pass on to next handler
        }
      }
    }
    break;
    case xEvent( MotionNotify ):
    {
      XMotionEvent *motionEvent =
                   (XMotionEvent *) evt.parameter2().asUnsignedLong();
      if ( fSplitCanvasHandlerData->isSplitBar(              // if splitbar
             (IWindowHandle) XtParent (
                             XtWindowToWidget( motionEvent->display,
                                         motionEvent->window)) ) )
      {
        ISplitBar *splitBar = (ISplitBar *) evt.controlWindow();
        ISplitCanvas *splitcv = (ISplitCanvas *) splitBar->parent();

        Display *display = motionEvent->display;
        fSplitCanvasHandlerData->drawSplitIndicator( prevLine, gcSet, display, splitcv );

        if ( splitBar->orientation() == ISplitBar::verticalSplit )
          prevLine = fSplitCanvasHandlerData->splitBarPosition( splitBar,
                                          boundingRectangle,
                                          origSeparatorX +
                                            ( motionEvent->x - mouseDownX ),
                                          origSeparatorY,
                                          separatorHeight);
        else
          prevLine = fSplitCanvasHandlerData->splitBarPosition( splitBar,
                                          boundingRectangle,
                                          origSeparatorY +
                                            ( motionEvent->y - mouseDownY ),
                                          origSeparatorX,
                                          separatorWidth );
        fSplitCanvasHandlerData->drawSplitIndicator( prevLine, gcSet, display, splitcv );
        evt.setResult(true);
        brc = true;
      }
      break;
    } /* end MotionNotify */
    case xEvent( KeyPress ):
    case xEvent( KeyRelease ):
    {
      if ( !moveInProgress)
      {
        if ( fSplitCanvasHandlerData->isSplitBar(
               (IWindowHandle) evt.controlHandle()) )
        {
          ISplitBar *splitBar = (ISplitBar *) evt.controlWindow();
          IKeyboardEvent aKey(evt);
          if (aKey.isVirtual() )
          {
            if ( splitBar->orientation() == ISplitBar::verticalSplit )
            {
              IPair::Coord delta = 0;
              if ( aKey.virtualKey() == IKeyboardEvent::left )
                delta = -1;
              else if ( aKey.virtualKey() == IKeyboardEvent::right )
                delta =  1;
              if ( delta )
              {
                IRectangle boundingRectangle =
                            splitBar->prevControl()->nativeRect() |
                            splitBar->nextControl()->nativeRect();
                // Add a work-around for IWindow::nativeRect incorrectly
                // handling 0-sized windows (defect 28693).
                unsigned long
                  prevChildWidth = splitBar->prevControl()->size().width(),
                  nextChildWidth = splitBar->nextControl()->size().width();
                if ( prevChildWidth == 0 )
                {
                   if ( nextChildWidth == 0 )
                   {
                      boundingRectangle
                       .sizeTo( ISize() );
                   }
                   else
                   {
                      boundingRectangle =
                        splitBar->nextControl()->nativeRect()
                         | splitBar->nativeRect();
                   }
                }
                else if ( nextChildWidth == 0 )
                {
                   boundingRectangle =
                     splitBar->prevControl()->nativeRect()
                      | splitBar->nativeRect();
                }

                IRectangle
                  splitBarRect = splitBar->nativeRect();
                if ( boundingRectangle.width() > splitBarRect.width() + 1 )
                {
                   // Ensure the child window to the left never gets
                   // smaller than one pixel in width. Also take the
                   // width of the split bar in consideration when
                   // calculating how far right it can move.
                   boundingRectangle
                    .moveTo( boundingRectangle.minXMinY() + IPoint( 1, 0 ) );
                   boundingRectangle
                    .sizeTo( boundingRectangle.size() -
                               ISize( splitBarRect.width() + 1, 0 ) );
                   IRectangle newPosition = fSplitCanvasHandlerData->splitBarPosition( splitBar,
                                              boundingRectangle,
                                              splitBarRect.minX() + delta,
                                              splitBarRect.minY(),
                                              splitBarRect.height() );
                   splitBar->moveTo( newPosition.bottomLeft() );
                   fSplitCanvasHandlerData->repositionControls(
                                              (ISplitCanvas *) splitBar->parent(),
                                              splitBar );
                }
                brc = true;
              }
            }
            else
            { // Horizontal split bar.
              // All calculations are initially done using native coordinates.
              IPair::Coord delta = 0;
              if ( aKey.virtualKey() == IKeyboardEvent::up )
                 delta = -1;
              else if ( aKey.virtualKey() == IKeyboardEvent::down )
                 delta = 1;
              if ( delta )
              {
                IRectangle boundingRectangle =
                            splitBar->prevControl()->nativeRect() |
                            splitBar->nextControl()->nativeRect();
                // Add a work-around for IWindow::nativeRect incorrectly
                // handling 0-sized windows (defect 28693).
                unsigned long
                  prevChildHeight = splitBar->prevControl()->size().height(),
                  nextChildHeight = splitBar->nextControl()->size().height();
                if ( prevChildHeight == 0 )
                {
                   if ( nextChildHeight == 0 )
                   {
                      boundingRectangle
                       .sizeTo( ISize() );
                   }
                   else
                   {
                      boundingRectangle =
                        splitBar->nextControl()->nativeRect()
                         | splitBar->nativeRect();
                   }
                }
                else if ( nextChildHeight == 0 )
                {
                   boundingRectangle =
                     splitBar->prevControl()->nativeRect()
                      | splitBar->nativeRect();
                }

                IRectangle
                  splitBarRect = splitBar->nativeRect();
                if ( boundingRectangle.height() > splitBarRect.height() + 1 )
                {
                   // Ensure the child window above never gets smaller
                   // than one pixel in height. Also take the height
                   // of the split bar in consideration when calculating
                   // how far down it can move.
                   boundingRectangle
                    .moveTo( boundingRectangle.minXMinY() + IPoint( 0, 1 ) );
                   boundingRectangle
                    .sizeTo( boundingRectangle.size() -
                               ISize( 0, splitBarRect.height() + 1 ) );
                   IRectangle newPosition = fSplitCanvasHandlerData->splitBarPosition(
                                              splitBar,
                                              boundingRectangle,
                                              splitBarRect.minY() + delta,
                                              splitBarRect.minX(),
                                              splitBarRect.width() );
                   IPoint
                     moveToPoint( newPosition.minXMinY() );
                   if ( ICoordinateSystem::isConversionNeeded() )
                   {  // Convert to lower-left coordinates.
                      moveToPoint += ISize( 0, splitBarRect.height( ) );
                      moveToPoint = ICoordinateSystem::convertToApplication(
                                        moveToPoint, splitBar->parent()->size() );
                   }
                   splitBar->moveTo( moveToPoint );
                   fSplitCanvasHandlerData->repositionControls(
                                              (ISplitCanvas *) splitBar->parent(),
                                              splitBar );
                }
                brc = true;
              }
            }
          }
        }
      }
      break;
    }
    case xEvent( ConfigureNotify ):
    {
      ISizeRectangle *sizeRect =
        (ISizeRectangle *) evt.parameter1().asUnsignedLong();

      if ( sizeRect->sizeChanged() )
      {
        /************************************************************/
        /* Resize the child controls based on the percentage of the */
        /* window they occupied when the window was resized, thus   */
        /* maintaining their ratios to each other.                  */
        /************************************************************/
        if ( !fSplitCanvasHandlerData->isSplitBar(
                (IWindowHandle) evt.controlHandle()) )
        {
          ISplitCanvas* splitcv = (ISplitCanvas*)evt.controlWindow();
          XConfigureEvent *xevent = (XConfigureEvent *)
              evt.parameter2().asUnsignedLong();
          splitcv->sizClCanvasSize = ISize( xevent->width, xevent->height );
          splitcv->setLayoutDistorted(IWindow::sizeChanged, 0);
          splitcv->layout();
          brc = true;
        }
      }
      break;
    } /* end case */

    case WM_BUTTON1DOWN:
    case WM_BUTTON2DOWN:
    {
      XButtonEvent *buttonEvent =
                 (XButtonEvent *) evt.parameter2().asUnsignedLong();

      if ( fSplitCanvasHandlerData->isSplitBar(
             (IWindowHandle) XtParent (
                             XtWindowToWidget(buttonEvent->display,
                                              buttonEvent->window ))))
      {
        // start the tracking
        mouseDownX = buttonEvent->x;
        mouseDownY = buttonEvent->y;
        moveInProgress = true; // Keep track of which split bar is being
                               // moved. This is used to detect trying to
                               // move one split bar with the keyboard and
                               // another with the mouse. Also used to detect
                               // a double click so the subsequent up event
                               // can be ignored.
        ISplitBar *splitBar = (ISplitBar *) evt.controlWindow();
        ITRACE_DEVELOP("BUTTON DOWN");
        IColor black( IColor::black);
        IColor white( IColor::white);

        XGCValues values;
        values.foreground = black.asPixel() ^ white.asPixel();
        values.function = GXxor;
        values.subwindow_mode = IncludeInferiors;
        gcSet = XCreateGC( buttonEvent->display,
                           RootWindow( buttonEvent->display, 0),
                          (  GCForeground |
                             GCFunction |
                             GCSubwindowMode ),
                          &values);
        // Get the two windows separated by the split bar. The bounding
        // rectangle for moving the split bar is the space they occupy.
        // MotionNotify will use this bounding rectangle for tracking a
        // separator line to the mouse as the user drags the split bar.
        // Use native coordinates to mix X calls in the calculations.
        boundingRectangle = splitBar->prevControl()->nativeRect() |
                            splitBar->nextControl()->nativeRect();

        // Add a work-around for IWindow::nativeRect incorrectly
        // handling 0-sized windows (defect 28693).
        if ( splitBar->orientation() == ISplitBar::verticalSplit )
        {
           unsigned long
             prevChildWidth = splitBar->prevControl()->size().width(),
             nextChildWidth = splitBar->nextControl()->size().width();
           if ( prevChildWidth == 0 )
           {
              if ( nextChildWidth == 0 )
              {
                 boundingRectangle
                  .sizeTo( ISize() );
              }
              else
              {
                 boundingRectangle =
                   splitBar->nextControl()->nativeRect()
                    | splitBar->nativeRect();
              }
           }
           else if ( nextChildWidth == 0 )
           {
              boundingRectangle =
                splitBar->prevControl()->nativeRect()
                 | splitBar->nativeRect();
           }
        }
        else
        {  // Horizontal split bars.
           unsigned long
             prevChildHeight = splitBar->prevControl()->size().height(),
             nextChildHeight = splitBar->nextControl()->size().height();
           if ( prevChildHeight == 0 )
           {
              if ( nextChildHeight == 0 )
              {
                 boundingRectangle
                  .sizeTo( ISize() );
              }
              else
              {
                 boundingRectangle =
                   splitBar->nextControl()->nativeRect()
                    | splitBar->nativeRect();
              }
           }
           else if ( nextChildHeight == 0 )
           {
              boundingRectangle =
                splitBar->prevControl()->nativeRect()
                 | splitBar->nativeRect();
           }
        }

        Region clipRegion = XCreateRegion();

        // Get the X origin of the bounded rectangle
        Dimension x, y, width, height;
        XtVaGetValues( splitBar->prevControl()->handle(),
          XmNx, &x,
          XmNy, &y,
          NULL);

        XRectangle clipRectangle;
        clipRectangle.x = x;
        clipRectangle.y = y;
        clipRectangle.width = boundingRectangle.width();
        clipRectangle.height = boundingRectangle.height();
        XUnionRectWithRegion( &clipRectangle, clipRegion, clipRegion );

        XSetRegion( buttonEvent->display, gcSet, clipRegion );
        XDestroyRegion( clipRegion );

        XtVaGetValues( splitBar->separator(),
                       XmNx,      &origSeparatorX,
                       XmNy,      &origSeparatorY,
                       XmNwidth,  &separatorWidth,
                       XmNheight, &separatorHeight,
                       NULL );
        // Make the separator coordinates relative to ISplitCanvas.
        IPoint
          splitBarOrigin = splitBar->nativeRect().minXMinY();
        origSeparatorX += splitBarOrigin.x();
        origSeparatorY += splitBarOrigin.y();

        prevLine = IRectangle();
        brc = true;
      }
      break;
    }/* end WM_BUTTONXDOWN */

    case WM_BUTTON1UP:
    case WM_BUTTON2UP:
    {
      if ( moveInProgress )
      {
        XButtonEvent *buttonEvent =
                    (XButtonEvent *) evt.parameter2().asUnsignedLong();
        if ( fSplitCanvasHandlerData->isSplitBar(
               (IWindowHandle) XtParent (
                               XtWindowToWidget( buttonEvent->display,
                                                 buttonEvent->window))))
        {
          ISplitBar *splitBar = (ISplitBar *) evt.controlWindow();
          ISplitCanvas *splitcv = (ISplitCanvas *) evt.controlWindow()->parent();
          moveInProgress = false;
          fSplitCanvasHandlerData->drawSplitIndicator( prevLine,
                                                       gcSet,
                                                       buttonEvent->display,
                                                       splitcv );
          IRectangle newPosition;
          if ( splitBar->orientation() == ISplitBar::verticalSplit )
            newPosition = fSplitCanvasHandlerData->splitBarPosition( splitBar,
                                           boundingRectangle,
                                           origSeparatorX
                                             - ( ( splitBar->size().width()
                                                     - separatorWidth ) / 2 )
                                             + ( buttonEvent->x - mouseDownX ),
                                           origSeparatorY,
                                           separatorHeight );
          else
            newPosition = fSplitCanvasHandlerData->splitBarPosition( splitBar,
                                           boundingRectangle,
                                           origSeparatorY
                                             - ( ( splitBar->size().height()
                                                     - separatorHeight ) / 2 )
                                             + ( buttonEvent->y - mouseDownY ),
                                           origSeparatorX,
                                           separatorWidth );

          // Make sure split bar doesn't overlap adjacent split bar.
          IPoint moveToPt;
          if (splitBar->orientation() == ISplitBar::verticalSplit)
          {
            int farRight = boundingRectangle.right() - 1 -
                           splitBar->size().width();
            int farLeft  = boundingRectangle.left() + 1;
            moveToPt = newPosition.bottomLeft();
            if (newPosition.right() > farRight)
              moveToPt.setX(farRight);
            if (newPosition.left() < farLeft)
              moveToPt.setX(farLeft);
            splitBar->moveTo(moveToPt);
          }
          else
          { // Horizontal split bars.
            moveToPt = newPosition.minXMinY();
            int absoluteMaxY = boundingRectangle.maxY() - 1 -
                                 splitBar->size().height();
            int absoluteMinY = boundingRectangle.minY() + 1;
            if ( moveToPt.y() > absoluteMaxY )
            {
               moveToPt.setY( absoluteMaxY );
            }
            else if ( moveToPt.y() < absoluteMinY )
            {
               moveToPt.setY( absoluteMinY );
            }
            // Currently, boundingRectangle, newPosition, and moveToPt
            // are all in native coordinates. Convert moveToPt back to
            // application coordinates if necessary.
            if ( ICoordinateSystem::isConversionNeeded() )
            {
               moveToPt += ISize( 0, splitBar->size().height() );
               moveToPt = ICoordinateSystem::convertToApplication(
                                 moveToPt, splitBar->parent()->size() );
            }
            splitBar->moveTo(moveToPt);
          }
          fSplitCanvasHandlerData->repositionControls(splitcv, splitBar);

          XFreeGC( buttonEvent->display, gcSet );
          /*********************************************************/
          /* Say we processed event                                */
          /*********************************************************/
          evt.setResult(true);
          brc = true;
        }
      }
      break;
    }
  }   /* endswitch */
  return brc;
}
#endif

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ISplitCanvasHandler::paintSplitBar                                           |
------------------------------------------------------------------------------*/
void ISplitCanvasHandler::paintSplitBar ( const ISplitCanvas* splitcv,
                                          const IPresSpaceHandle& pshCanvas,
                                          const IRectangle& rect )
{
  /********************************************************************/
  /* Get the width of a window border, and the width of a window size */
  /* border.  These widths will be used to make up the split bar. The */
  /* split bar is painted in 3 steps.  The left and right sides are   */
  /* the width of a window border, with the center being the width of */
  /* a window sizing border, minus the width of 2 window borders.     */
  /* A rough picture may help:                                        */
  /*                                                                  */
  /*           Width of a size border                                 */
  /*           minus the width of 2                                   */
  /*           window borders                                         */
  /*                   |                                              */
  /*   Width of a      |      Width of a                              */
  /*   Window     \    |    / Window                                  */
  /*   Border      ||     ||  Border                                  */
  /*               ||     ||                                          */
  /*               ||     ||                                          */
  /*               ||     ||                                          */
  /*                                                                  */
  /* Note that rect is in native coordinates.                         */
  /********************************************************************/

#ifdef IC_ORIGINUPPERLEFT
  // Coordinate system with an upper-left origin (e.g. Windows, Motif).
  IRootGrafPort
    gc( pshCanvas );
#else
  // Coordinate system with a lower-left origin (e.g. PM).
  IExtendedRootGrafPort
    gc( pshCanvas, kRightHand );
#endif

  // Use exclusive fill painting because the split bar rectangle is based
  // on a window rectangle, rather than a true graphics rectangle.
  IFillBundle
    clrClMiddle( IBasicColor( splitcv->splitBarMiddleColor() ) ),
    clrClEdge  ( IBasicColor( splitcv->splitBarEdgeColor() ) );

  ILinkedGrafPort thePort( &gc, &clrClEdge );
  if ( splitcv->orientation() == ISplitCanvas::verticalSplit )
  {
     IRectangle box = rect.sizedTo( ISize( splitcv->ulClEdgeThickness,
                                           rect.height() ) );

     /***************************************************************/
     /* Paint the left edge of the split bar.                       */
     /***************************************************************/
     thePort.draw( IGRect2D( box ) );

     /***************************************************************/
     /* Paint the middle of the split bar.                          */
     /***************************************************************/
     box.moveTo( box.maxXMinY() );
     box.sizeTo( ISize( splitcv->ulClMiddleThickness, box.height() ) );
     thePort.draw( IGRect2D( box ), clrClMiddle );

     /***************************************************************/
     /* Paint the right edge of the split bar.                      */
     /***************************************************************/
     box.moveTo( box.maxXMinY() );
     box.sizeTo( ISize( splitcv->ulClEdgeThickness, box.height() ) );
     thePort.draw( IGRect2D( box ) );
  }
  else
  {  // Must be a horizontal split bar.
     IRectangle box = rect.sizedTo( ISize( rect.width(),
                                           splitcv->ulClEdgeThickness ) );

     /***************************************************************/
     /* Paint the minimum edge of the split bar.                    */
     /***************************************************************/
     thePort.draw( IGRect2D( box ) );

     /***************************************************************/
     /* Paint the middle of the split bar.                          */
     /***************************************************************/
     box.moveTo( box.minXMaxY() );
     box.sizeTo( ISize( box.width(), splitcv->ulClMiddleThickness ) );
     thePort.draw( IGRect2D( box ), clrClMiddle );

     /***************************************************************/
     /* Paint the maximum edge of the split bar.                    */
     /***************************************************************/
     box.moveTo( box.minXMaxY() );
     box.sizeTo( ISize( box.width(), splitcv->ulClEdgeThickness ) );
     thePort.draw( IGRect2D( box ) );
  }
}
#endif // IC_PMWIN

/*------------------------------------------------------------------------------
| ISplitCanvasHandler::handleEventsFor                                         |
|                                                                              |
| Attach the handler to the canvas.                                            |
------------------------------------------------------------------------------*/
ISplitCanvasHandler& ISplitCanvasHandler::handleEventsFor
                                            ( ISplitCanvas *canvas )
{
  IASSERTPARM( canvas != 0 );
  Inherited::handleEventsFor( canvas );
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvasHandler::stopHandlingEventsFor                                   |
|                                                                              |
| Detach the handler from the canvas.                                          |
------------------------------------------------------------------------------*/
ISplitCanvasHandler& ISplitCanvasHandler::stopHandlingEventsFor
                                             ( ISplitCanvas *canvas )
{
  IASSERTPARM( canvas != 0 );
  Inherited::stopHandlingEventsFor( canvas );
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvasHandler::handleEventsFor                                         |
------------------------------------------------------------------------------*/
IHandler& ISplitCanvasHandler::handleEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| ISplitCanvasHandler::stopHandlingEventsFor                                   |
------------------------------------------------------------------------------*/
IHandler& ISplitCanvasHandler::stopHandlingEventsFor ( IWindow* )
{                              // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ISplitCanvasHandler::handleEventsFor                                         |
|                                                                              |
| Attach the handler to the canvas.                                            |
------------------------------------------------------------------------------*/
ISplitCanvasHandler& ISplitCanvasHandler::handleEventsFor
                                             ( ISplitBar *canvas )
{
  IASSERTPARM( canvas != 0 );
  Inherited::handleEventsFor( canvas );
  canvas->startHandling( IWindow::someMouseMoves );
  return *this;
}


/*------------------------------------------------------------------------------
| ISplitCanvasHandler::stopHandlingEventsFor                                   |
|                                                                              |
| Detach the handler from the canvas.                                          |
------------------------------------------------------------------------------*/
ISplitCanvasHandler& ISplitCanvasHandler::stopHandlingEventsFor
                                             ( ISplitBar *canvas )
{
  IASSERTPARM( canvas != 0 );
  canvas->stopHandling( IWindow::someMouseMoves );
  Inherited::stopHandlingEventsFor( canvas );
  return *this;
}

#endif
