// Revision: 65 1.7.2.1 source/ui/basectl/ispinszh.cpp, listctls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ispinszh.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ispinszh.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

#define INCL_WINMESSAGEMGR   // For WM_ADJUSTWINDOWPOS.
#define INCL_WINWINDOWMGR    // For QWS_ constants
#define INCL_WINSTDSPIN      // For SPBS_ constants
extern "C" {
  #include <iwindefs.h>
#ifdef IC_WIN
  #include <commctrl.h>
  #include <iclspbw.h>
#endif
}

#include <ispinszh.hpp>
#include <ispinbas.hpp>
#include <ispintxt.hpp>
#include <ispinnum.hpp>
#include <ibidiset.hpp>
#include <icconst.h>
#include <iexcept.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#ifdef IC_WIN
  #include <iplatfrm.hpp>
  #include <icolor.hpp>
#endif


// Segment definitions
#ifdef IC_PAGETUNE
  #define _ISPINSZH_CPP_
  #include <ipagetun.h>
#endif


#ifdef IC_WIN
UDACCEL*                ISpinButtonStatics :: fsfastSpinTable = 0;
UDACCEL*                ISpinButtonStatics :: fsnormalSpinTable = 0;
#endif

/*------------------------------------------------------------------------------
| ISpinButtonStatics::~ISpinButtonStatics                                      |
|                                                                              |
| Destructor to clean up.                                                      |
------------------------------------------------------------------------------*/
ISpinButtonStatics :: ~ISpinButtonStatics ( )
{
  IMODTRACE_DEVELOP("ISpinButtonStatics::~ISpinButtonStatics");
#if IC_STATIC_PRIORITY_SUPPORTED
  // It is only safe to delete when static init/term order can be controlled.
#ifdef IC_WIN
  if (fsfastSpinTable)
  {
     delete [] fsfastSpinTable;
     fsfastSpinTable = 0;
  }
  if (fsnormalSpinTable)
  {
     delete fsnormalSpinTable;
     fsnormalSpinTable = 0;
  }
#endif

#endif
}

/*------------------------------------------------------------------------------
| ISpinButtonStatics::buddyHandle                                              |
|                                                                              |
| Returns the handle of the spin button edit control.                          |
------------------------------------------------------------------------------*/
IWindowHandle ISpinButtonStatics::buddyHandle(
                                     const IWindowHandle& spinbutton )
{
   IWindowHandle ef = IWindow::handleWithParent( IIDOF( spinbutton ),
                                                 spinbutton );
#ifdef IC_PMWIN
   // Work-around for window id breakage in Warp GA for the child
   // entry field of a spin button control (it should be the window
   // id of the parent spin button, but is 1).
   if (!ef)
   {            // Can't find a child entry field of a spin button.
      ef = IWindow::handleWithParent( 1, spinbutton );
   }            // Try again with the one Warp GA uses.
#endif
  return ef;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ISpinButtonStatics::convertToBuddyStyle                                      |
|                                                                              |
| Returns the style flags needed for the buddy control.                        |
------------------------------------------------------------------------------*/
unsigned long ISpinButtonStatics :: convertToBuddyStyle(
                                      unsigned long extendedStyle )
{
  // Determine correct style needed for edit control.  We use ES_MULTILINE
  // to get alignment to work, and subclass the control to handle
  // the newlines (like IEntryField)
  unsigned long result = WS_CHILD | WS_VISIBLE | ES_MULTILINE | ES_AUTOHSCROLL;
  if ( extendedStyle & IBaseSpinButton::readOnly.asExtendedUnsignedLong()    )
     result |= ES_READONLY;
  if ( extendedStyle & IBaseSpinButton::centerAlign.asExtendedUnsignedLong() )
     result |= ES_CENTER;
  if ( extendedStyle & IBaseSpinButton::rightAlign.asExtendedUnsignedLong()  )
     result |= ES_RIGHT;
  return result;
}
#endif

/*------------------------------------------------------------------------------
| ISpinButtonStatics::convertToExtendedStyle                                   |
|                                                                              |
| Returns the extended style flags based on the styles set in the PM compatible|
| spin button control.  This function is used to get extended styles set       |
| properly on wrapper constructors.                                            |
------------------------------------------------------------------------------*/
unsigned long ISpinButtonStatics :: convertToExtendedStyle(
                                      unsigned long guiStyle )
{
  // Determine correct extended style
  unsigned long ulStyle = 0;
  if ( guiStyle & SPBS_MASTER )
     ulStyle |= ISPBS_MASTER;
  else
     ulStyle |= ISPBS_SERVANT;

  if ( guiStyle & SPBS_READONLY )
     ulStyle |= ISPBS_READONLY;

  if ( (guiStyle & (SPBS_JUSTLEFT | SPBS_JUSTRIGHT)) == SPBS_JUSTLEFT )
     ulStyle |= ISPBS_JUSTLEFT;
  else if ( (guiStyle & (SPBS_JUSTLEFT | SPBS_JUSTRIGHT)) == SPBS_JUSTRIGHT)
     ulStyle |= ISPBS_JUSTRIGHT;
  else
     ulStyle |= ISPBS_CENTERALIGN;

  if ( guiStyle & SPBS_NOBORDER )
     ulStyle |= ISPBS_NOBORDER;
  if ( guiStyle & SPBS_FASTSPIN )
     ulStyle |= ISPBS_FASTSPIN;
  if ( guiStyle & SPBS_PADWITHZEROS )
     ulStyle |= ISPBS_PADWITHZEROS;
  if ( guiStyle & SPBS_NUMERICONLY )
     ulStyle |= ISPBS_NUMERICONLY;

  return ulStyle;

}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ISpinButtonStatics::fastSpinTableSize                                        |
|                                                                              |
| Returns the number of entries in the fast spin table.  Currently hardwired   |
| to be 3 entries.                                                             |
------------------------------------------------------------------------------*/
unsigned long ISpinButtonStatics::fastSpinTableSize()
{
  return 3;
}

/*------------------------------------------------------------------------------
| ISpinButtonStatics::fastSpinTable                                            |
|                                                                              |
| Returns the fast spin table.   If the table is not allocated we allocate     |
| and initialize it.                                                           |
------------------------------------------------------------------------------*/
UDACCEL* ISpinButtonStatics::fastSpinTable()
{
  if (!fsfastSpinTable)
    {
    unsigned long tableSize = ISpinButtonStatics::fastSpinTableSize();
    fsfastSpinTable = new UDACCEL [ tableSize ];
    unsigned i;
    unsigned accelFactor = 1;
    for (i=0; i < tableSize; i++, accelFactor *= 2)
      {
      fsfastSpinTable[i].nSec = (i+1) * 2;      // 2 second intervals
      fsfastSpinTable[i].nInc = accelFactor;    // double the rate
      }
    }
  return fsfastSpinTable;
}
/*------------------------------------------------------------------------------
| ISpinButtonStatics::normalSpinTable                                          |
|                                                                              |
| Returns the normal spin table.   If the table is not allocated we allocate   |
| and initialize it.   Table is assumed to be 1 entry long.                    |
------------------------------------------------------------------------------*/
UDACCEL* ISpinButtonStatics::normalSpinTable()
{
  if (!fsnormalSpinTable)
    {
    fsnormalSpinTable = new UDACCEL;
    fsnormalSpinTable->nSec = 2;      // 2 second intervals
    fsnormalSpinTable->nInc = 1;      // no acceleration
    }
  return fsnormalSpinTable;
}
#endif

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::IBaseSpinDefaultHandler                               |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IBaseSpinDefaultHandler :: IBaseSpinDefaultHandler ( )
  : IBaseSpinDefaultHandler::Inherited ( )
#ifdef IC_WIN
  , bkgrndbrush( 0 )
#endif //IC_WIN
{ }

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::~IBaseSpinDefaultHandler                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IBaseSpinDefaultHandler :: ~IBaseSpinDefaultHandler ( )
{
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::dispatchHandlerEvent                                |
| This handler generates WM_COMMAND notification messages.   The notification  |
| codes are the same as the CCL spinbutton, and are:                           |
|       SPBN_UPARROW        up arrow button was pressed                        |
|       SPBN_DOWNARROW      down arrow button was pressed                      |
|       SPBN_ENDSPIN        mouse button was released                          |
|       SPBN_CHANGE         spinfield text has changed                         |
|       SPBN_SETFOCUS       spinfield received focus                           |
|       SPBN_KILLFOCUS      spinfield lost focus                               |
------------------------------------------------------------------------------*/
bool IBaseSpinDefaultHandler::dispatchHandlerEvent ( IEvent& event )
{
  bool bHandled = false;
  switch (event.eventId())
     {
     case WM_NOTIFY:
     {
        // Since the WM_NOTIFY message (UDN_DELTAPOS) notification is received prior
        // to the scroll event, we can't use it to generate the SPBN_DOWNARROW nor
        // the SPBN_UPARROW notifications.  Doing so would result in portability
        // problems with the PM compatible controls.  Instead, we will generate these
        // notifications in response to the WM_VSCROLL which is generated by Windows
        // after the WM_NOTIFY.
        NM_UPDOWN* nm = (NM_UPDOWN*)((void*)event.parameter2().asUnsignedLong());
        if (nm->hdr.code == UDN_DELTAPOS)
        {
           // Cache the spin direction for use when we generate the SPBN_ARROW...
           // notification.
           hwndDelta = event.handle();
           iDelta = nm->iDelta;
        }
        break;
     }

     case WM_VSCROLL:
     case IC_UM_VSCROLL:
        {
        if (event.parameter1().lowNumber() == SB_THUMBPOSITION)
           {
           int newValue = (short)event.parameter1().highNumber();

           // The edit field updated depends on whether the update is
           // coming as a result of user interaction or our member functions.
           // The IC_UM_VSCROLL only comes from our member functions.
           IWindowHandle ef(0);
           if (event.eventId() == IC_UM_VSCROLL)
              ef = ISpinButtonStatics::buddyHandle( event.handle() );
           else
              ef = IQUERYACTIVEBUDDY( event.handle() );

           IBaseSpinButton* spinbutton = (IBaseSpinButton*)
                IWindow::windowWithHandle( IPARENTOF(ef) );
           if (spinbutton)
              {
              // Store the current position in the spinbutton parent of
              // the active edit field
              ISETLASTPOS(spinbutton->handle(), newValue);

              unsigned long ulstyle = ISTYLEOF(spinbutton->handle());
              if (ulstyle & INSPBS_NUMERICONLY )
                 {
                 // numeric
                 IString newText( newValue );
                 if (ulstyle & INSPBS_PADWITHZEROS)
                    {
                    // Pad to limit or 11, whichever is less
                    unsigned width = (unsigned)spinbutton->limit();
                    if (width > 11)
                       width = 11;
                    if (newValue < 0)
                       {
                       newText.stripLeading('-');
                       newText.rightJustify(width-1, '0');
                       newText.insert("-");
                       }
                    else
                       newText.rightJustify(width, '0');

                    }
                 SetWindowText( ef, newText );
                 }
              else
                 {
                 // text
                 bool written = false;
                 if (newValue == 0)
                    {
                    // check for empty case
                    bool empty = IQUERYEMPTYFLAG( spinbutton->handle() );
                    if (empty)
                       {
                       // If empty ... null the entry field
                       SetWindowText( ef, "");
                       written = true;
                       }
                    } // if position 0
                 if (!written)
                    {
                    SetWindowText(
                       ef,
                       ((ITextSpinButton*)spinbutton)->elementAt( newValue ) );
                    }
                 }

              // Send the SPBN_UPARROW or SPBN_DOWNARROW notification.
              if (event.eventId() == WM_VSCROLL)
              {
                 if (!(spinbutton->isPMCompatible())
                     && event.handle() == hwndDelta)
                 {
                    if (event.parameter1().lowNumber() == SB_THUMBPOSITION)
                    {
                       // Generate the notifications expected by handlers
                       IWindowHandle parent = IPARENTOF( spinbutton->handle() );
                       parent.sendEvent( WM_COMMAND,
                                       IEventParameter1(
                                          (short)IIDOF( spinbutton->handle() ),
                                       (short)((iDelta < 0)
                                          ? SPBN_DOWNARROW : SPBN_UPARROW)),
                                       IEventParameter2(
                                          spinbutton->handle().asUnsigned()) );
                    }
                 }

                 // set the focus on the edit field
                 event.handle().postEvent(WM_SETFOCUS, 0, 0);
              }

              // send the edit field changed message.
              IWindowHandle parent = IPARENTOF( spinbutton->handle() );
              parent.sendEvent(
                 WM_COMMAND,
                 IEventParameter1( (short)IIDOF( event.handle() ), SPBN_CHANGE),
                 IEventParameter2( event.handle().asUnsigned()) );
              }   // if spinbutton
           }    // if delta pos
        else if (event.parameter1().lowNumber() == SB_ENDSCROLL)
           {
           // If focus is on the updown set it to the edit field.  We
           // use the canvas user message for this purpose.
           event.handle().postEvent(IC_UM_CANVAS_SETFOCUS, 0, 0);

           // send the scroll ended changed message.
           IWindowHandle ef = IQUERYACTIVEBUDDY( event.handle() );
           IBaseSpinButton* spinbutton = (IBaseSpinButton*)
                IWindow::windowWithHandle( IPARENTOF(ef) );
           if (spinbutton)
           {
              IWindowHandle parent = IPARENTOF( spinbutton->handle() );
              parent.sendEvent(
                 WM_COMMAND,
                 IEventParameter1( (short)IIDOF( spinbutton->handle() ), SPBN_ENDSPIN),
                 IEventParameter2( spinbutton->handle().asUnsigned()) );
           }
           }
        }
        break;

     case IC_UM_CANVAS_SETFOCUS:
        {
        // If focus is on the updown set it to the edit field.
        HWND updown = IQUERYUPDOWNHANDLE( event.handle() );
        HWND ef = IQUERYACTIVEBUDDY( event.handle() );
        if ( (ef) && (IQUERYFOCUS(0) == updown) )
           ISETFOCUS(0,ef);
        }
        break;

     case WM_SETFOCUS:
        {
        // Make sure focus is set to the edit field.
        HWND updown = IQUERYUPDOWNHANDLE( event.handle() );
        HWND ef = IQUERYACTIVEBUDDY( event.handle() );
        if (ef)
           ISETFOCUS( 0, ef );
        }
        break;

     case WM_WINDOWPOSCHANGED:
        {
        IBaseSpinButton* spinbutton = (IBaseSpinButton*)event.window();
        if (!spinbutton->isPMCompatible())
           {
           // Check if the spin button is left-to-right (the entry field
           // is left of the spin arrows) or right-to-left (the entry
           // field is right of the spin arrows).
           bool
             leftToRight = true;
           if ( IBidiSettings::isBidiSupported() )
           {
              IBidiSettings
                bidiSettings( *spinbutton );
              if ( bidiSettings.windowLayout() ==
                     IBidiSettings::layoutRightToLeft )
              {
                 leftToRight = false;
              }
           }

           WINDOWPOS* wpos = (WINDOWPOS*)(event.parameter2().asUnsignedLong());
           ISize newsz(wpos->cx, wpos->cy);
           if ( spinbutton->hasBorder() )
              {
              newsz -= ISize( GetSystemMetrics(SM_CXBORDER),
                              GetSystemMetrics(SM_CYBORDER) );
	      if ((IPlatform::isWin9x()) || (IPlatform::isNTNewShell()))
                 newsz -= IPoint(4,4);     //Account for 3d borders
              }
           int updownWidth = GetSystemMetrics( SM_CXVSCROLL );
           if (spinbutton->isMaster())
              {
              if (newsz.width() < updownWidth)
                 updownWidth = (int)newsz.width();
              IWindowHandle updown = IQUERYUPDOWNHANDLE(spinbutton->handle());
              SetWindowPos(updown, 0,
                           leftToRight ?                        // x
                             (int)newsz.width() - updownWidth : 0,
                           0,                                   // y
                           updownWidth,                         //width
                           (int)newsz.height(),                 //height
                           SWP_NOZORDER | SWP_NOACTIVATE );
              }
           else
              {
              // Servant.  Dont allow space for the updown
              updownWidth = 0;
              }
           IWindowHandle buddy =
              ISpinButtonStatics::buddyHandle( spinbutton->handle() );
           SetWindowPos( buddy, 0,
                         leftToRight ? 0 : updownWidth,         // x
                         0,                                     // y
                         (int)newsz.width() - updownWidth,      //width
                         (int)newsz.height(),                   //height
                         SWP_NOZORDER | SWP_NOACTIVATE );
           }

        }
        break;

     case WM_SETFONT:
     case WM_GETFONT:
        {
        IWindowHandle buddy = IWindowHandle( IQUERYACTIVEBUDDY(event.handle()) );
        event.setResult( buddy.sendEvent( event.eventId(),
                                          event.parameter1(),
                                          event.parameter2()) );
        bHandled = true;
        }
        break;

     case WM_CONTROL:
        if (event.parameter1().lowNumber() == IIDOF(event.handle()) )
           {
           // Event from buddy window.
           IWindowHandle spb    = event.handle();
           IWindowHandle parent = IPARENTOF( spb );
           switch ( event.parameter1().highNumber() )
              {
              case EN_CHANGE:
                 {
                 if (ISTYLEOF(spb) & INSPBS_NUMERICONLY)
                    {
                    INumericSpinButton* spinbutton =
                        (INumericSpinButton*)event.window();
                    // set current position to value if value is in range
                    if (spinbutton)
                       {
                       long newValue = spinbutton->value();
                       if (spinbutton->range().includes( newValue ))
                          {
                          IWindowHandle updown = IQUERYUPDOWNHANDLE( spb );
                          // If current spinbutton not active one don't mess
                          // with updown
                          if ( (updown.asUnsigned()) &&
                               (IQUERYACTIVEBUDDY( IPARENTOF( updown )) ==
                                (HWND)event.parameter2().asUnsignedLong() ) )
                             updown.sendEvent(
                                UDM_SETPOS,
                                IEventParameter1(0),
                                IEventParameter2((short)newValue, 0) );

                          ISETLASTPOS(spb, newValue);
                          }
                       }
                    }
                 parent.sendEvent(
                    WM_COMMAND,
                    IEventParameter1( (short)IIDOF( spb ), SPBN_CHANGE),
                    IEventParameter2( spb.asUnsigned()) );
                 }
                 break;
              case EN_KILLFOCUS:
                 parent.sendEvent(
                    WM_COMMAND,
                    IEventParameter1( (short)IIDOF( spb ), SPBN_KILLFOCUS),
                    IEventParameter2( spb.asUnsigned()) );
                 break;
              case EN_SETFOCUS:
                 {
                 // The edit field gained focus.  Check to see if we
                 // are dealing with a master/servant case
                 IWindowHandle updown = IQUERYUPDOWNHANDLE( spb );
                 if (updown.asUnsigned())
                    {
                    IWindowHandle masterspb = IPARENTOF( updown );
                    IWindowHandle active = IQUERYACTIVEBUDDY( masterspb );
                    if (active.asUnsigned() !=
                        event.parameter2().asUnsignedLong() )
                       {
                       // Set the active buddy window word in the master
                       // spinbutton wrapper control.
                       ISETACTIVEBUDDY( masterspb, event.parameter2().asLong());
                       // We need to set the parameters of the updown
                       // to match the position and range of the new buddy.
                       updown.sendEvent( UDM_SETRANGE,
                                         IEventParameter1(0),
                                         IEventParameter2(
                                            (unsigned long)IQUERYRANGE(spb)) );
                       short lastPos = (short)IQUERYLASTPOS(spb);
                       updown.sendEvent( UDM_SETPOS,
                                         IEventParameter1(0),
                                         IEventParameter2(
                                            lastPos,0) );
                       // have to set the text in the fake buddy
                       IWindowHandle fakeBuddy = (HWND)
                          updown.sendEvent(UDM_GETBUDDY);
                       SetWindowText( fakeBuddy, IString(lastPos) );
                       }
                    }
                 // Send notification message
                 parent.sendEvent(
                    WM_COMMAND,
                    IEventParameter1( (short)IIDOF( spb ), SPBN_SETFOCUS),
                    IEventParameter2( spb.asUnsigned()) );
                 }
                 break;
              }  // switch
           } // is buddy window
        break;


    case IC_UM_IS_AGGREGATE_CTRL:
        {
        event.setResult(true);
        bHandled = true;
        }
        break;

     }
  return bHandled;
}

/*------------------------------------------------------------------------------
| Window procedure for edit field in spinbutton.                               |
------------------------------------------------------------------------------*/
LRESULT CALLBACK fnwpSpinButtonEditProc(
                    HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam )
{
   // Get the spin button handle
   IWindowHandle spb   = IPARENTOF( hwnd );
 #ifdef IC_WIN_STRICT
   WNDPROC defaultProc = IQUERYBUDDYPROC( spb );
 #else
   FARPROC defaultProc = (FARPROC)IQUERYBUDDYPROC( spb );
 #endif
   if (defaultProc == 0)
      {
      ITRACE_DEVELOP("fnSpinButtonEditProc - no default procedure");
      return DefWindowProc( hwnd, msg, wparam, lparam);
      }

   LRESULT result = 0;
   switch (msg)
      {
      case WM_CHAR:
      case WM_KEYDOWN:
      case WM_KEYUP:
        if ( (msg == WM_CHAR ) &&
             ( (wparam == VK_RETURN ) ||
               (wparam == VK_ESCAPE ) ||
               (wparam == VK_TAB    )) )
           {
           // Send these on to the spinbutton and do not pass to the
           // edit control.
           result = spb.sendEvent(msg, (int)wparam, (int)lparam).asUnsignedLong();
           }
        else if ( (wparam == VK_UP ) || (wparam == VK_DOWN ) )
        {
           IBaseSpinButton* spinbutton =
              (IBaseSpinButton*)IWindow::windowWithHandle( spb );
           IWindowHandle parent = IPARENTOF( spinbutton->handle() );
           if (msg == WM_KEYDOWN)
           {
              if (spinbutton)
              {
                 // Spin the button - Note spin notification is
                 // done inside of the spinUp/spinDown functions
                 if (wparam == VK_UP)
                    spinbutton->spinUp();
                 else
                    spinbutton->spinDown();
              }
           }
           else if (msg == WM_KEYUP)
           {
              // Generate the spin ended event.
              parent.sendEvent( WM_COMMAND,
                                IEventParameter1(
                                   (short)IIDOF( spinbutton->handle() ),
                                   (short)SPBN_ENDSPIN ),
                                IEventParameter2(
                                   spinbutton->handle().asUnsigned()) );
           }
           result = 0;          // processed
        }
        else if (((msg == WM_CHAR) && (ISTYLEOF(spb) & INSPBS_NUMERICONLY)) &&
                 (!(( LOBYTE( wparam ) == VK_BACK ) ||
                      IString( LOBYTE( wparam )).isDigits())) )
        {
          // Accept only numeric characters (or a backspace) in the
          // entry field of a numeric spinbutton.
          result = 0;
        }
        else if ( (msg == WM_CHAR ) &&
                 ((wparam != VK_ENTER ) &&
                  (wparam != VK_ESC ) &&
                  (wparam != VK_BACKSPACE ) &&
                  (wparam != VK_NEWLINE ) &&
                  (wparam != VK_BREAK ) &&
                  (wparam != VK_TAB    )) )
        {
          // allow text change handler to process (if attached)
          result = spb.sendEvent((unsigned long) msg,
                                 (unsigned long)wparam,
                                 (unsigned long)lparam);
          // if text change handler doesn't process, allow default proc a chance
          if (!result)
          {
            // Allow default procedure a chance ... then pass on to
            // parent spinbutton if not handled.
            result = CallWindowProc( defaultProc, hwnd, msg, wparam, lparam );

            // If not processed, send event to the spinbutton.
            if (!result)
              result = spb.sendEvent(msg, (int)wparam, (int)lparam);

          }
        }

        else
        {
          // Allow default procedure a chance ... then pass on to
          // parent spinbutton if not handled.
          result = CallWindowProc( defaultProc, hwnd, msg, wparam, lparam );

          // If not processed, send event to the spinbutton.
          if (!result)
            result = spb.sendEvent(msg, (int)wparam, (int)lparam);
        }
        break;

      case WM_SYSKEYDOWN:
         {
            // If this is an ALT-Up, pass the event up to the spin button's
            // parent.  If the spin button is in a notebook, the focus will
            // go to the notebook's tab.
            if (wparam == VK_UP)
            {
               IBaseSpinButton* spinbutton =
                  (IBaseSpinButton*)IWindow::windowWithHandle( spb );
               IWindowHandle parent = IPARENTOF( spinbutton->handle() );
               result = parent.sendEvent( msg,
                                          IEventParameter1( (int)wparam ),
                                          IEventParameter2( (int)lparam ));
            }
            else // Any other ALT combination
            {
	      IWindowHandle parent = IPARENTOF( spb );
	      parent.sendEvent( (unsigned long)msg,
			        (unsigned long)wparam,
	    		        (unsigned long)lparam );
	      result = true;
            }
         }
         break;

      case WM_SYSCHAR:
        {
          IWindowHandle parent = IPARENTOF( spb );
          parent.sendEvent( (unsigned long)msg,
                            (unsigned long)wparam,
                            (unsigned long)lparam );
          result = true;
        }
        break;

      case WM_CONTEXTMENU:
        {
          result = spb.sendEvent(msg, (int)wparam, (int)lparam);
          break;
        }

      case EM_PASTE:
        {
          // send paste to the text change handler (if attached)
          result = spb.sendEvent((unsigned long)IC_UM_EM_PASTE,
                                 (unsigned long)wparam,
                                 (unsigned long)lparam);

          // if not handled by text change handler
          if (!result)
            result = CallWindowProc( defaultProc, hwnd, msg, wparam, lparam);

          break;
        }

      default:
        // Call the original window procedure
        result = CallWindowProc( defaultProc, hwnd, msg, wparam, lparam);
        break;
      }  // switch
   return result;
}

#else

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::windowResize                                         |
|                                                                              |
| Gives the spin button the opportunity to properly resize its child entry     |
| field and reposition its child spin arrows.                                  |
------------------------------------------------------------------------------*/
bool IBaseSpinDefaultHandler :: windowResize( IResizeEvent& event )
{
  PSWP pswp = (PSWP)(char*)(event.parameter1());
  if (pswp->fl & SWP_NOADJUST  ||  event.parameter2() == 0)
  {                // No WM_ADJUSTWINDOWPOS message yet.
     event.window()->sendEvent( WM_ADJUSTWINDOWPOS,
                                event.parameter1() );
  }
  return false;
}
#endif

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::handleEventsFor                                      |
|                                                                              |
| Attach the handler to a spin button.                                         |
------------------------------------------------------------------------------*/
IBaseSpinDefaultHandler& IBaseSpinDefaultHandler :: handleEventsFor
                                               ( IBaseSpinButton* spinButton )
{
  IASSERTPARM( spinButton != 0 );
  Inherited::handleEventsFor( spinButton );
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::stopHandlingEventsFor                                |
|                                                                              |
| Detach the handler from a spin button.                                       |
------------------------------------------------------------------------------*/
IBaseSpinDefaultHandler&
  IBaseSpinDefaultHandler :: stopHandlingEventsFor
                              ( IBaseSpinButton* spinButton )
{
  IASSERTPARM( spinButton != 0 );
  Inherited::stopHandlingEventsFor( spinButton );
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::handleEventsFor                                      |
------------------------------------------------------------------------------*/
IHandler& IBaseSpinDefaultHandler :: handleEventsFor ( IWindow* /*window*/ )
{                  // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinDefaultHandler::stopHandlingEventsFor                                |
------------------------------------------------------------------------------*/
IHandler& IBaseSpinDefaultHandler :: stopHandlingEventsFor
                                       ( IWindow* /*window*/ )
{                  // Private to hide version in IHandler.
  ITHROWLIBRARYERROR( IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable );
  return *this;
}

