// Revision: 51 1.23.1.1 source/ui/basectl/isliderw.cpp, slidectls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: isliderw.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of the native Windows slider         *
*   functions declared in islider.hpp.                                         *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

extern "C"
{
  #include <iwindefs.h>
  #include <commctrl.h>
}

#include <iclsldw.h>
#include <islider.hpp>
#include <isliderw.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <igimage.hpp>
#include <igrport.hpp>
#include <igbase2d.hpp>
#include <ihandle.hpp>
#include <iplatfrm.hpp>
#include <irect.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>
#include <icconst.h>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ISLIDERW_CPP_
  #include <ipagetun.h>
#endif

#define IC_SLIDER_WIDTH  34
#define IC_BUTTON_SIZE   20


/*------------------------------------------------------------------------------
| IProgressIndicatorData::IProgressIndicatorData                               |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IProgressIndicatorData :: IProgressIndicatorData () :
  trackbar( 0 ),
  isWrappered( false ),
  scaleFactor( 1 ),
  leftDown( 0 ),
  leftDownPressed( 0 ),
  leftDownDisabled( 0 ),
  rightUp( 0 ),
  rightUpPressed( 0 ),
  rightUpDisabled( 0 ),
  tickPos0( 0 ),
  textVisible( true ),
  timer( 0 ),
  fdefaultHandler(),
  needsTextMinCalc( false )
{
  numberTicks[0] = 0;
  numberTicks[1] = 0;
  tickSpacing[0] = 0;
  tickSpacing[1] = 0;
  longestTick[0] = 0;
  longestTick[1] = 0;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::~IProgressIndicatorData                               |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IProgressIndicatorData :: ~IProgressIndicatorData ()
{
  // Cancel any outstanding timer.
  if (timer)
     cancelTimer();

  // Essentially we had created some IGLists. We need to delete
  // all the IGraphic objects that were added to these lists.

  // When IHitGraphicGroup variables go out of scope, all the
  // elements inside the group will also be deleted. Nothing
  // to be done here.

  unsigned long  cnt = buttons.count ();
  while (cnt != 0)
  {
    (void) buttons.orphanFirst ();
    cnt = buttons.count ();
  }

  for (unsigned long scale = 0; scale < 2; scale++)
  {
    IKeyGraphicPair           dummy;
    IMGraphic *               gText;

    unsigned long  numElements = tickText[scale].numberOfValues ();
    for (unsigned long  i = 0; i < numElements; i ++)
    {
      dummy = tickText[scale].value (i);
      gText = (IGraphicText *) dummy.fObj;
      delete gText;

    } // for every graphic

  } // for each scale

  delete leftDown;
  delete leftDownPressed;
  delete leftDownDisabled;
  delete rightUp;
  delete rightUpPressed;
  delete rightUpDisabled;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::registerSlider                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::registerSlider( )
{
  HINSTANCE hinst = GetModuleHandle(0);
  WNDCLASS  wndClass;

  // Before registering, check to make sure it is not already registered.
  bool fRegistered = GetClassInfo( hinst, WC_NATIVESLIDER, &wndClass);

  if (!fRegistered)
  {
    // Make sure common controls dll is loaded.
    InitCommonControls();

    HBRUSH   defBkgrndBrush = CreateSolidBrush( IGUIColor( IGUIColor::buttonLight )
                                .asRGBLong() );

    wndClass.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
    wndClass.cbClsExtra    = 0;
    wndClass.cbWndExtra    = sizeof(LONG);
    wndClass.hCursor       = LoadCursor(0,IDC_ARROW);
    wndClass.hInstance     = GetModuleHandle(0);
    wndClass.lpfnWndProc   = (WNDPROC)IDEFWINDOWPROC;
    wndClass.hIcon         = 0;
    wndClass.hbrBackground = defBkgrndBrush;
    wndClass.lpszMenuName  = 0;
    wndClass.lpszClassName = WC_NATIVESLIDER;

    // Register the wrapper window class.
    if (!RegisterClass(&wndClass))
      ITHROWGUIERROR( "RegisterClass" );
  }
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::createNativeSlider                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::createNativeSlider(
                                   const IWindowHandle& parent,
                                   const IWindowHandle& owner,
                                   unsigned long        ulStyle,
                                   unsigned long        extStyle )
{
  IMODTRACE_ALL("IProgressIndicatorData::createNativeSlider");

  HINSTANCE hInst = GetModuleHandle(0);

  if (isWrappered)
  {
     // We have a wrappered trackbar, so just resize it.
     ISETWINDOWPOS( trackbar,
                    NULL, 0, 0, 100, 100, SWP_NOACTIVATE | SWP_NOZORDER );
  }
  else
  {
     // Create the native trackbar.  Use an initial size of 100 x 100
     // in order to obtain a valid left/topmost tick offset.
     trackbar = ICREATEWINDOW( parent,
                               TRACKBAR_CLASS,
                               0,
                               ulStyle | WS_VISIBLE,
                               0,
                               0,
                               100,
                               100,
                               owner,
                               HWND_BOTTOM,
                               IIDOF( parent ),
                               0,
                               0,
                               hInst );

     if (!trackbar)
       ITHROWGUIERROR(IString("WinCreateWindow: Id=")+IString(IIDOF( parent ))+
              IString(" Class=")+IString((unsigned short)(unsigned long)TRACKBAR_CLASS));
  }

  // If this is a slider with buttons, create the bitmaps for the buttons.
  if (extStyle & ISLS_BUTTONSLEFT || extStyle & ISLS_BUTTONSRIGHT)
  {
     if (ulStyle & TBS_VERT)
     {
       leftDown          = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarDownArrow));
       leftDownPressed   = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarDownArrowPressed));
        leftDownDisabled = new IImage (
          ISystemBitmapHandle (
            ISystemBitmapHandle::scrollBarDownArrowDisabled));
        rightUp          = new IImage (
          ISystemBitmapHandle (
            ISystemBitmapHandle::scrollBarUpArrow));
        rightUpPressed   = new IImage (
          ISystemBitmapHandle (
            ISystemBitmapHandle::scrollBarUpArrowPressed));
        rightUpDisabled  = new IImage (
          ISystemBitmapHandle (
            ISystemBitmapHandle::scrollBarUpArrowDisabled));
     }
     else
     {
       leftDown         = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarLeftArrow));
       leftDownPressed  = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarLeftArrowPressed));
       leftDownDisabled = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarLeftArrowDisabled));
       rightUp          = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarRightArrow));
       rightUpPressed   = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarRightArrowPressed));
       rightUpDisabled  = new IImage (
         ISystemBitmapHandle (
           ISystemBitmapHandle::scrollBarRightArrowDisabled));
     }

     leftDownDisabled->setHitEnabled (false);
     rightUpDisabled->setHitEnabled (false);

     // Size the button bitmaps.

     leftDown->sizeTo (IGPoint2D (IC_BUTTON_SIZE, IC_BUTTON_SIZE));
     leftDownPressed->sizeTo (IGPoint2D (IC_BUTTON_SIZE, IC_BUTTON_SIZE));
     leftDownDisabled->sizeTo (IGPoint2D (IC_BUTTON_SIZE, IC_BUTTON_SIZE));
     rightUp->sizeTo( IGPoint2D (IC_BUTTON_SIZE, IC_BUTTON_SIZE));
     rightUpPressed->sizeTo (IGPoint2D (IC_BUTTON_SIZE, IC_BUTTON_SIZE));
     rightUpDisabled->sizeTo (IGPoint2D (IC_BUTTON_SIZE, IC_BUTTON_SIZE));

     // Add bitmaps to the buttons IGList. The correct state will
     // be determined in setButtonState.

     buttons.adoptLast (leftDown);
     buttons.adoptLast (rightUp);
  }

  // Get the offset of the left/topmost tick and store it to be
  // used later in layout.  We want to get it before we set the
  // scale to avoid problems when using the homeRight or
  // homeBottom styles.  The offset of this tick within the
  // trackbar does not change.
  unsigned long ulResult = trackbar.sendEvent(
                                       TBM_SETTIC,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ));
  if (!ulResult)
  {
     ITHROWGUIERROR2("TBM_SETTIC",
                     IBaseErrorInfo::invalidParameter,
                     IException::recoverable);
  }
  tickPos0 = trackbar.sendEvent( TBM_GETTICPOS,
                                 IEventParameter1( 0 ),
                                 IEventParameter2( 0 ));

  // Subclass the trackbar's window procedure so we can send focus notifications
  // to our parent.  This is necessary to enable an IFocusHandler attached to the
  // slider.  This subclassed window procedure is needed for progress indicators
  // to prevent them from receiving input focus.
  fdefaultTrackbarProc = (WNDPROC)ISUBCLASSWINDOW( trackbar,
                                                   isliderTrackbarWinProc );
  if (!fdefaultTrackbarProc)
  {
     ITHROWGUIERROR( "ISUBCLASSWINDOW" );
  }
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::destroyNativeSlider                                  |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::destroyNativeSlider( )
{
   // Remove the subclass of the trackbar window proc
   if ( trackbar && IISWINDOW( 0, trackbar ) )
   {
      ISUBCLASSWINDOW( trackbar,
                       fdefaultTrackbarProc );
   }
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::setupNativeScale                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::setupNativeScale( IProgressIndicator::Scale scale )
{
  IMODTRACE_ALL("IProgressIndicatorData::setupNativeScale");

  unsigned long range,
                pageSize;

  IProgressIndicator *progressInd = (IProgressIndicator*)IWindow::windowWithHandle(
                                          IPARENTOF (trackbar) );
  if (numberTicks [scale] == 0)
  {
     range = 0;
     pageSize = 0;
  }
  else
  {
     if (progressInd->isSnapToTickEnabled())
     {
        // SnapToTick is implemented via the page and line size for
        // the trackbar.  These values represent the movement
        // increment for mouse and keyboard.
        range = numberTicks [scale] - 1;
        pageSize = 1;
     }
     else
     {
        // If this slider is not snapToTick, the range is set to the closest
        // multiple of the page size less that the maximum short value.
        // Set the page size to the proper proportion.
        pageSize = SHRT_MAX / (numberTicks [scale] - 1);
        range = pageSize * (numberTicks[scale] - 1);
     }
  }
  if (!progressInd->isVertical() &&
         progressInd->homePosition() == IProgressIndicator::homeBottomLeft ||
       progressInd->isVertical() &&
         progressInd->homePosition() == IProgressIndicator::homeTopRight)
  {
     trackbar.sendEvent( TBM_SETRANGEMAX,
                         IEventParameter1( true ),
                         IEventParameter2( range ));
     scaleFactor = 1;
  }
  else
  {
     trackbar.sendEvent( TBM_SETRANGEMIN,
                         IEventParameter1( true ),
                         IEventParameter1( -range ));
     scaleFactor = -1;
  }

  trackbar.sendEvent( TBM_SETPAGESIZE,
                      IEventParameter1( 0 ),
                      IEventParameter2( pageSize ));
  trackbar.sendEvent( TBM_SETLINESIZE,
                      IEventParameter1( 0 ),
                      IEventParameter2( pageSize ));

  // Now clear and reset ticks for this scale.
  trackbar.sendEvent( TBM_CLEARTICS,
                      IEventParameter1( true ),
                      IEventParameter2( 0 ));

  if (!visibleTicks[scale].isEmpty())
  {
     TickSet tempTicks = visibleTicks[scale];
     TickSet::Cursor cursor (tempTicks);
     unsigned long ulTickNum;
     visibleTicks[scale].removeAll();
     for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
     {
        ulTickNum = tempTicks.elementAt( cursor );
        if (ulTickNum < numberTicks[scale])
           visibleTicks[scale].add( ulTickNum );
     }

     if (!visibleTicks[scale].isEmpty())
     {
        unsigned long ulPosition, ulResult;
        TickSet::Cursor tickCursor (visibleTicks[scale]);
        forCursor (tickCursor)
        {
           ulTickNum = visibleTicks[scale].elementAt( tickCursor );

           ulPosition = ulTickNum * pageSize * scaleFactor;

           ulResult = trackbar.sendEvent( TBM_SETTIC,
                                    IEventParameter1( 0 ),
                                    IEventParameter2( ulPosition ));
           if (!ulResult)
              ITHROWGUIERROR2("TBM_SETTIC",
                              IBaseErrorInfo::invalidParameter,
                              IException::recoverable);
        }
     }
  }

  // Determine whether text will be visible given this alignment and
  // primary scale.  If not, we won't calculate text positions nor draw
  // text.
  IProgressIndicator::Alignment align = progressInd->alignment();
  if ((align == IProgressIndicator::topRight
            && scale == IProgressIndicator::scale1)
      || (align == IProgressIndicator::bottomLeft
            && scale == IProgressIndicator::scale2))
     textVisible = false;
  else
     textVisible = true;
}

/*------------------------------------------------------------------------------
| IProgressIndicatorData::layoutNativeSlider                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::layoutNativeSlider( )
{
  IMODTRACE_ALL("IProgressIndicatorData::layoutNativeSlider");

  IProgressIndicator *progressInd = (IProgressIndicator*)
        IWindow::windowWithHandle( IPARENTOF( trackbar ));

  long trackbarx, trackbary, trackbarcx, trackbarcy, spacing;
  IManagedPresSpaceHandle  presSpace (progressInd);
  IRootGrafPort  port (presSpace);
  ISize sliderSize( progressInd->size() );

  if (sliderSize == ISize())
     return;

  if (GetWindowLong( progressInd->handle(), GWL_EXSTYLE ) & WS_EX_CLIENTEDGE)
     sliderSize -= 4;

  IProgressIndicator::Alignment align = progressInd->alignment();
  IProgressIndicator::HomePosition home = progressInd->homePosition();
  IProgressIndicator::Scale scale = progressInd->primaryScale();
  bool isVertical = progressInd->isVertical();
  unsigned long numTicks = progressInd->numberOfTicks( scale );
  if (numTicks == 0)
     return;

  ISlider::ButtonsPosition buttonPos;
  if (~ISTYLEOF( trackbar ) & TBS_NOTHUMB)
  {
     // This is a slider so we need to check for button position.
     ISlider *slider = (ISlider*)progressInd;
     buttonPos = slider->buttonsPosition();
  }
  else
     buttonPos = ISlider::none;

  long buttonSize = 0;
  if (buttonPos != ISlider::none)
     buttonSize = leftDown->geometricBounds().width(); // Assume square bitmap.

  // Calculate size and position of trackbar within the slider static
  // control using top left origin.  Allow some additional room at the ends of
  // the slider to accomodate text if possible.

  IGRect2D  firstSize;
  IGRect2D  lastSize;
  IGraphicText *  firstString;
  IGraphicText *  lastString;
  IKeyGraphicPair dummy;

  if ((tickText[scale].numberOfValues () != 0) && textVisible)
  {
    dummy = tickText[scale].value (0);
    firstString = (IGraphicText *) dummy.fObj;

    if (dummy.fKey == 0)
    {
      firstSize = firstString->looseFitBounds (&port);
    }

    dummy = tickText[scale].value ( tickText[scale].numberOfValues () - 1 );
    lastString = (IGraphicText *) dummy.fObj;

    if (dummy.fKey == numTicks - 1)
    {
      lastSize = lastString->looseFitBounds (&port);
    }

  } // if

  if (isVertical)
  {
    // Vertical slider.
    if (tickSpacing[scale] != 0)
    {
      spacing = tickSpacing[scale];
    }
    else
    {
      // Calculate maximum spacing (assuming no text adjustment)
      spacing = (sliderSize.height() - 2*buttonSize - 2*tickPos0) / (numTicks-1);

      // Check if the top or bottom tick text could use a little room
      // in case a large font is being used.
      long bottomNeeds,
           topNeeds,
           extraToAdd;

      if (buttonPos == ISlider::bottom)
         bottomNeeds =
            MAX( 0, (long)((firstSize.height()+1)/2 - tickPos0 - 2*buttonSize));
      else
         bottomNeeds = MAX( 0, (long)((firstSize.height()+1)/2 - tickPos0));

      if (buttonPos == ISlider::top)
         topNeeds =
            MAX( 0, (long)((lastSize.height()+1)/2 - tickPos0 - 2*buttonSize));
      else
         topNeeds = MAX( 0, (long)((lastSize.height()+1)/2 - tickPos0));

      // Now see how much extra room is available at each end
      // using the spacing above.
      unsigned long
         extraSpace = (sliderSize.height() - (2*buttonSize +
                                2*tickPos0 + (spacing * (numTicks-1)))) / 2;

      // Determine the extra space to add.
      if (bottomNeeds > extraSpace || topNeeds > extraSpace)
      {
         extraToAdd = 2 * MAX( bottomNeeds, topNeeds );

         // Subject additional space to maximum since we do not want
         // major changes to the slider look (40 is arbitrary value)
         // and recalculate spacing based on adding this additional delta
         extraToAdd = MIN(extraToAdd, 40);
         if ( extraToAdd )
           spacing = MIN( spacing, (sliderSize.height() - 2*tickPos0 -
                             extraToAdd - 2*buttonSize) / (numTicks-1));
      }
    }
    // Make sure that the spacing is not zero.
    if (spacing < 1)
       spacing = 1;

    pixelRange = spacing * (numTicks-1);
    trackbarcx = IC_SLIDER_WIDTH;
    trackbarcy = pixelRange + 2*tickPos0;
    trackbary = MAX( 0, (long)((sliderSize.height() - trackbarcy)/2 - buttonSize) );
    if (buttonPos == ISlider::top)
      trackbary += 2*buttonSize;

    switch (align)
    {
      case IProgressIndicator::centered:
        trackbarx = (sliderSize.width() - IC_SLIDER_WIDTH) / 2;
        break;
      case IProgressIndicator::bottomLeft:
        trackbarx = 0;
        break;
      default:
        trackbarx = sliderSize.width() - IC_SLIDER_WIDTH;
    }
  }
  else
  {
    // Horizontal slider.
    if (tickSpacing[scale] != 0)
    {
      spacing = tickSpacing[scale];
    }
    else
    {
      // Calculate maximum spacing (assuming no text adjustment)
      spacing = (sliderSize.width() - 2*buttonSize - 2*tickPos0) / (numTicks-1);

      // Check if the first or last tick text could use a little room
      long leftNeeds,
           rightNeeds,
           extraToAdd;

      if (buttonPos == ISlider::left)
         leftNeeds =
            MAX( 0, (long)((firstSize.width()+1)/2 - tickPos0 - 2*buttonSize));
      else
         leftNeeds = MAX( 0, (long)((firstSize.width()+1)/2 - tickPos0));

      if (buttonPos == ISlider::right)
         rightNeeds =
            MAX( 0, (long)((lastSize.width()+1)/2 - tickPos0 - 2*buttonSize));
      else
         rightNeeds = MAX( 0, (long)((lastSize.width()+1)/2 - tickPos0));

      // Now see how much extra room is available at each end
      // using the spacing above.
      unsigned long
         extraSpace = (sliderSize.width() - (2*buttonSize +
                                2*tickPos0 + (spacing * (numTicks-1)))) / 2;

      // Determine the extra space to add.
      if (leftNeeds > extraSpace || rightNeeds > extraSpace)
      {
         extraToAdd = 2 * MAX( leftNeeds, rightNeeds );

         // Subject additional space to maximum since we do not want
         // major changes to the slider look (100 is arbitrary value)
         // and recalculate spacing based on adding this additional delta
         extraToAdd = MIN(extraToAdd, 100);
         if ( extraToAdd )
           spacing = MIN( spacing, (sliderSize.width() - 2*tickPos0 -
                             extraToAdd - 2*buttonSize) / (numTicks-1));
      }
    }
    // Make sure that the spacing is not zero.
    if (spacing < 1)
       spacing = 1;

    pixelRange = spacing * (numTicks-1);
    trackbarcx = pixelRange + 2*tickPos0;
    trackbarx = MAX( 0, (long)((sliderSize.width() - trackbarcx)/2 - buttonSize));
    if (buttonPos == ISlider::left)
       trackbarx += 2*buttonSize;
    trackbarcy = IC_SLIDER_WIDTH;

    switch (align)
    {
      case IProgressIndicator::centered:
        trackbary = (sliderSize.height() - IC_SLIDER_WIDTH) / 2;
        break;
      case IProgressIndicator::bottomLeft:
        trackbary = sliderSize.height() - IC_SLIDER_WIDTH;
        break;
      default:
        trackbary = 0;
    }
  }

  // Set the size/position of the trackbar.
  MoveWindow( trackbar,
              trackbarx,
              trackbary,
              trackbarcx,
              trackbarcy,
              true );

  // Now we need to set the clip rectangle and position of
  // the tick text strings.
  if ((tickText[scale].numberOfValues () != 0) && textVisible)
  {
    long                      tick0xy, tickxy, stringId;
    IGraphicText *            gText;
    IKeyGraphicPair           dummy;
    IMGraphic *               obj;
    IRectangle                clipRect;
    IGRect2D                  textSize;
    IGPoint2D                 textPosition;

    // Set the value of offset0.  This value is the position of the
    // left or top most tick mark in the static control.
    tick0xy = (isVertical) ? (tickPos0 + trackbary) : (tickPos0 + trackbarx);

    // Calculate the new clipping rectangle and text position.

    unsigned long  numElements = tickText[scale].numberOfValues ();
    for (unsigned long  i = 0; i < numElements; i ++)
    {
      dummy = tickText[scale].value (i);
      gText = (IGraphicText *) dummy.fObj;
      stringId = dummy.fKey;
      textSize = gText->looseFitBounds ();
      tickxy = tick0xy + stringId * spacing;

      if (isVertical)
      {
        if (scale == IProgressIndicator::scale1)
        {
          // Text to right of trackbar.
          clipRect.moveTo( IPoint( trackbarx + trackbarcx, 0 ));
          clipRect.sizeTo( ISize( sliderSize.width() - trackbarcx - trackbarx,
                                  sliderSize.height() ));

          textPosition = IGPoint2D (trackbarx + trackbarcx,
                                    tickxy - textSize.height () / 2);
        }
        else
        {
          // Text left of trackbar.  Clipping rectangle is entire area
          // to the left of the trackbar.
          clipRect.moveTo( IPoint( 0,0 ));
          clipRect.sizeTo( ISize( trackbarx, sliderSize.height() ));

          textPosition = IGPoint2D (trackbarx - textSize.width (),
                                    tickxy - textSize.height () / 2);
        }

      } // if laid out vertically

      else
      {
        // Horizontal slider.
        if (scale == IProgressIndicator::scale1)
        {
          // Text on top of trackbar.

          clipRect.moveTo( IPoint(0,0) );
          clipRect.sizeTo( ISize( sliderSize.width(), trackbary ));

          textPosition = IGPoint2D (
                           MAX (0, (long)(tickxy - textSize.width () / 2)),
                           MAX (0, (long)(trackbary - textSize.height ())) );
        }
        else
        {
          // Text below trackbar.
          clipRect.moveTo( IPoint( 0, trackbary + trackbarcy ));
          clipRect.sizeTo( ISize( sliderSize.width(),
                           sliderSize.height()-trackbary-trackbarcy ));

          textPosition = IGPoint2D (
                           MAX (0, (long)(tickxy - textSize.width () / 2)),
                           trackbary + trackbarcy);
        }
      }

      // Currently we don't have support for setting the clipping
      // rectangle of IGraphicText. So the following line is commented.
      // gText->setClippingRect (clipRect);

      // The y-coordinate of the graphic text location is the baseline coordinate.
      // In order to account for this, get the current font for the control and
      // add in the offset above the baseline.
      IFont font = progressInd->font();
      textPosition += IGPoint2D( 0, font.maxAscender() );
      gText->setLocation( textPosition );

    } // for every graphic

  } // if text scale has labels

  // Set the positions of the buttons.
  if (buttonPos != ISlider::none)
  {
     unsigned long leftDownPos, rightUpPos;
     switch (buttonPos)
     {
       case ISlider::left:
         leftDownPos = trackbarx - 2*buttonSize;
         rightUpPos = leftDownPos + buttonSize;
         break;
       case ISlider::right:
         leftDownPos = trackbarx + trackbarcx;
         rightUpPos = leftDownPos + buttonSize;
         break;
       case ISlider::bottom:
         rightUpPos = trackbary + trackbarcy;
         leftDownPos = rightUpPos + buttonSize;
         break;
       case ISlider::top:
         rightUpPos = trackbary - 2*buttonSize;
         leftDownPos = rightUpPos + buttonSize;
         break;
     }

     unsigned long buttonx = trackbarx + (trackbarcx - buttonSize)/2,
                   buttony = trackbary + (trackbarcy - buttonSize)/2;
     if (isVertical)
     {
        rightUp->moveTo (IGPoint2D ( buttonx, rightUpPos ));
        rightUpPressed->moveTo (IGPoint2D ( buttonx, rightUpPos ));
        rightUpDisabled->moveTo (IGPoint2D ( buttonx, rightUpPos ));
        leftDown->moveTo(IGPoint2D  ( buttonx, leftDownPos ));
        leftDownPressed->moveTo (IGPoint2D ( buttonx, leftDownPos ));
        leftDownDisabled->moveTo (IGPoint2D ( buttonx, leftDownPos ));
     }
     else
     {
        leftDown->moveTo (IGPoint2D ( leftDownPos, buttony ));
        leftDownPressed->moveTo (IGPoint2D ( leftDownPos, buttony ));
        leftDownDisabled->moveTo (IGPoint2D ( leftDownPos, buttony ));
        rightUp->moveTo(IGPoint2D  ( rightUpPos, buttony ));
        rightUpPressed->moveTo (IGPoint2D ( rightUpPos, buttony ));
        rightUpDisabled->moveTo (IGPoint2D ( rightUpPos, buttony ));
     }
  }
  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::calcLongestTick                                      |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::calcLongestTick( )
{
  IProgressIndicator *progressInd = (IProgressIndicator*)IWindow::windowWithHandle(
                                          IPARENTOF (trackbar) );
  bool isVertical = progressInd->isVertical();

  longestTick[0] = 0;
  longestTick[1] = 0;

  for (int scale = IProgressIndicator::scale1;
       scale <= IProgressIndicator::scale2;
       scale++)
  {
     IGraphicText    *gText;
     IKeyGraphicPair  dummy;
     IGRect2D         textSize;
     unsigned long    numElements = tickText[scale].numberOfValues(),
                      curTickLength;

     if (numElements != 0)
     {
        for (unsigned long i = 0; i < numElements; i ++)
        {
           dummy = tickText[scale].value( i );
           gText = (IGraphicText*)dummy.fObj;
           textSize = gText->looseFitBounds();

           if (isVertical)
              curTickLength = textSize.width() + 4;
           else
              curTickLength = textSize.height() + 4;

           if (curTickLength > longestTick[scale])
              longestTick[scale] = curTickLength;
        }
     }
  }
  needsTextMinCalc = false;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::trackbarRange                                        |
|                                                                              |
| Return the range extent of the trackbar.                                     |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicatorData::trackbarRange( )
{
  IMODTRACE_ALL("IProgressIndicatorData::trackbarRange");

  if (scaleFactor == 1)
     return (trackbar.sendEvent( TBM_GETRANGEMAX,
                                 IEventParameter1( 0 ),
                                 IEventParameter2( 0 )));
  else
     return (-(trackbar.sendEvent( TBM_GETRANGEMIN,
                                  IEventParameter1( 0 ),
                                  IEventParameter2( 0 ))));
}

/*------------------------------------------------------------------------------
| IProgressIndicatorData::armRangeOffset                                       |
|                                                                              |
| Return the position in the native trackbar.  This value will be scaled to a  |
| positive position since the actual position may be negative depending on the |
| control windows's home position.                                             |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicatorData::armRangeOffset( )
{
  IMODTRACE_ALL("IProgressIndicatorData::armRangeOffset");

  unsigned long ulStyle = ISTYLEOF( trackbar );
  long position;

  if (ulStyle & TBS_NOTHUMB)
     // This is a progress indicator (ie, read only).  In this case, only a
     // selection range is available.
     if (scaleFactor == 1)
        // We have a homeLeft horizontal or a homeTop vertical
        // trackbar.
        position = trackbar.sendEvent( TBM_GETSELEND,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ) );
     else
        // We have a homeRight horizontal or a homeBottom vertical
        // trackbar.  In this case we want the (negative) start of the
        // selection range.
        position = trackbar.sendEvent( TBM_GETSELSTART,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ) );
  else
     // This is a slider.  We can query the arm position.
     position = trackbar.sendEvent( TBM_GETPOS,
                                    IEventParameter1( 0 ),
                                    IEventParameter2( 0 ) );

  return ( position * scaleFactor );
}

/*------------------------------------------------------------------------------
| IProgressIndicatorData::setArmRangeOffset                                    |
|                                                                              |
| Set the position in the native trackbar.  The new position provided will be  |
| scaled appropriately to account for the control window's home position.  The |
| control will be redrawn.                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
IProgressIndicatorData& IProgressIndicatorData::setArmRangeOffset(
                              unsigned long newPosition )
{
  IMODTRACE_ALL("IProgressIndicatorData::setArmRangeOffset");

  unsigned long ulStyle = ISTYLEOF( trackbar );

  // Set the arm position.
  trackbar.sendEvent( TBM_SETPOS,
                      IEventParameter1( true ),
                      IEventParameter2( scaleFactor * newPosition ) );

  if (~ulStyle & TBS_NOTHUMB)
  {
     // Set the button state.
     setButtonState();
  }

  if (ulStyle & TBS_ENABLESELRANGE)
  {
     drawRibbon( newPosition );
  }

  // Since Windows won't send an event, send one here to enable
  // handler and notification.
  IWindowHandle hwndParent = IPARENTOF( trackbar );
  hwndParent.sendEvent( IC_UM_VSCROLL,
                        IEventParameter1( TB_ENDTRACK, 0 ),
                        IEventParameter2( trackbar ));

  return *this;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::drawRibbon                                           |
|                                                                              |
| Draw the ribbon in the native trackbar by adjusting the selection range.     |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::drawRibbon( unsigned long newPosition )
{
  IMODTRACE_ALL("IProgressIndicatorData::drawRibbon");

  unsigned long currentSelection;

  if (scaleFactor == 1)
  {
     // We have a homeLeft horizontal or a homeTop vertical
     // trackbar.
     currentSelection = trackbar.sendEvent( TBM_GETSELEND,
                                            IEventParameter1( 0 ),
                                            IEventParameter2( 0 ));
     if (currentSelection != newPosition)
        trackbar.sendEvent( TBM_SETSELEND,
                            IEventParameter1( true ),
                            IEventParameter2( newPosition ) );
  }
  else
  {
     // We have a homeRight horizontal or a homeBottom vertical
     // trackbar.  In this case we want to set the (negative) start of the
     // selection range.
     currentSelection = trackbar.sendEvent( TBM_GETSELSTART,
                                            IEventParameter1( 0 ),
                                            IEventParameter2( 0 ));
     if (currentSelection != -newPosition)
        trackbar.sendEvent( TBM_SETSELSTART,
                            IEventParameter1( true ),
                            IEventParameter2( -newPosition ) );
  }
  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::setButtonState                                       |
|                                                                              |
| Draw the slider buttons in their correct state.                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::setButtonState( )
{
  IMODTRACE_ALL("IProgressIndicatorData::setButtonState");

  ISlider *slider;
  ISlider::ButtonsPosition buttonPos;
  if (~ISTYLEOF( trackbar ) & TBS_NOTHUMB)
  {
     slider = (ISlider*)IWindow::windowWithHandle( IPARENTOF( trackbar ));
     buttonPos = slider->buttonsPosition();
  }
  else
     // This is a progress indicator.
     buttonPos = ISlider::none;

  if (buttonPos == ISlider::none)
     return;

  bool isVertical = slider->isVertical();
  IProgressIndicator::HomePosition homePosition = slider->homePosition();
  unsigned long offset, range, rangeMin, rangeMax;

  // Get the max offset value.
  rangeMin = trackbar.sendEvent( TBM_GETRANGEMIN,
                                 IEventParameter1( 0 ),
                                 IEventParameter2( 0 ));

  rangeMax = trackbar.sendEvent( TBM_GETRANGEMAX,
                                 IEventParameter1( 0 ),
                                 IEventParameter2( 0 ));

  range = rangeMax - rangeMin;

  // Calculate the arm offset from the left/top.
  if (homePosition == IProgressIndicator::homeBottomLeft)
     offset = armRangeOffset();
  else
     offset = range - armRangeOffset();

  // Determine the button states.  Don't check for hit selected here; we do that
  // in our handler, but disable the timer if needed.
  IImage *  prevLeftDown = (IImage *)
                         (IHitGraphicGroupIterator (&buttons)).first ();
  IImage *  prevRightUp  = (IImage *)
                         (IHitGraphicGroupIterator (&buttons)).last ();

  if (offset == 0)
  {
    if (prevLeftDown == leftDownPressed)
    {
      cancelTimer ();
    }
    (void) buttons.orphanFirst ();
    buttons.adoptFirst (leftDownDisabled);
  }
  else if (prevLeftDown != leftDownPressed)
  {
    (void) buttons.orphanFirst ();
    buttons.adoptFirst (leftDown);
  }

  if (offset == range)
  {
     if (prevRightUp == rightUpPressed)
     {
       cancelTimer();
     }
     (void) buttons.orphanLast ();
     buttons.adoptLast (rightUpDisabled);
  }
  else if (prevRightUp != rightUpPressed)
  {
    (void) buttons.orphanLast ();
    buttons.adoptLast (rightUp);
  }

  if (slider->isVisible ())
  {
    IManagedPresSpaceHandle  presSpace (slider);
    IRootGrafPort  port (presSpace);
    if (prevLeftDown != (IImage *)
                        (IHitGraphicGroupIterator (&buttons)).first ())
    {
      ((IImage * )
       (IHitGraphicGroupIterator (&buttons)).first ())->draw (port);
    }

    if (prevRightUp != (IImage *)
                        (IHitGraphicGroupIterator (&buttons)).last ())
    {
      ((IImage * )
       (IHitGraphicGroupIterator (&buttons)).last ())->draw (port);
    }
  }

  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::buttonPressed                                        |
|                                                                              |
| Handle a button press.                                                       |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::buttonPressed (IImage *  button)
{
  IMODTRACE_ALL("IProgressIndicatorData::buttonPressed");

  IManagedPresSpaceHandle  presSpace ((IWindowHandle) (IPARENTOF (trackbar)));
  IRootGrafPort  port (presSpace);

  // Determine which button was pressed and process it.
  if (button == (IImage *)
                (IHitGraphicGroupIterator (&buttons)).first ())
  {
    // The leftDown button was pressed.  Set the new button state.
    (void) buttons.orphanFirst ();
    buttons.adoptFirst (leftDownPressed);
    leftDownPressed->draw (port);
  }
  else if (button == (IImage *)
                     (IHitGraphicGroupIterator (&buttons)).last ())
  {
    // The rightUp button was pressed.  Set the new button state.
    (void) buttons.orphanLast ();
    buttons.adoptLast (rightUpPressed);
    rightUpPressed->draw (port);
  }

  // Move the slider.
  moveArm ();

  // Start the timer.
  if (!timer)
  {
    startTimer ();
  }

} // IProgressIndicatorData::buttonPressed


/*------------------------------------------------------------------------------
| IProgressIndicatorData::buttonReleased                                       |
|                                                                              |
| Handle a button release.                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::buttonReleased( )
{
  IMODTRACE_ALL("IProgressIndicatorData::buttonReleased");

  // Cancel any outstanding timer.
  if (timer)
     cancelTimer();

  // Restore the pressed button to normal state.
  IManagedPresSpaceHandle  presSpace ((IWindowHandle) (IPARENTOF (trackbar)));
  IRootGrafPort  port (presSpace);
  if ((IImage *)
      (IHitGraphicGroupIterator (&buttons)).first () == leftDownPressed)
  {
    (void) buttons.orphanFirst();
    buttons.adoptFirst (leftDown);
    leftDown->draw (port);
  }
  else if ((IImage *)
    (IHitGraphicGroupIterator (&buttons)).last () == rightUpPressed)
  {
    (void) buttons.orphanLast ();
    buttons.adoptLast (rightUp);
    rightUp->draw (port);
  }

  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::startTimer                                           |
|                                                                              |
| Start the timer for button repeat.                                           |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::startTimer( )
{
  IMODTRACE_ALL("IProgressIndicatorData::startTimer");

  if (!timer)
  {
     bool mouseCaptured( false );
     ISETCAPTURE( mouseCaptured, IPARENTOF( trackbar ));
     if (mouseCaptured)
     {
        timer = new ITimer( new ITimerMemberFn0<IProgressIndicatorData>(
                                   *this, &IProgressIndicatorData::moveArm ),
                            500);
        // The following setFocus is a workaround to prevent the frame
        // from losing focus when the IObjectWindow used by the ITimer
        // object is created.
        SetFocus( trackbar );
     }
  }
  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::cancelTimer                                          |
|                                                                              |
| Kill the button repeat timer.                                                |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::cancelTimer( )
{
  IMODTRACE_ALL("IProgressIndicatorData::cancelTimer");

  if (timer)
  {
     timer->stop();
     delete timer;
     timer = 0;
     IRELEASECAPTURE;
  }
  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::moveArm                                              |
|                                                                              |
| Move the slider in response to button pressed or timer elapse.               |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::moveArm( )
{
  IMODTRACE_ALL("IProgressIndicatorData::moveArm");

  ISlider *slider =
        (ISlider*)IWindow::windowWithHandle( IPARENTOF( trackbar ));

  // Get the home position so we know which way to move the slider.
  IProgressIndicator::HomePosition homePosition =
        slider->homePosition();

  // Get the current slider position.
  unsigned long armOffset = armRangeOffset();

  // Determine the amount to move the slider arm.
  unsigned long ulPageSize = trackbar.sendEvent( TBM_GETPAGESIZE,
                                                 IEventParameter1( 0 ),
                                                 IEventParameter2( 0 ));

  // Determine which button is pressed and move the slider arm.
  if ((IImage *)
      (IHitGraphicGroupIterator (&buttons)).first () == leftDownPressed)
  {
     if (homePosition == IProgressIndicator::homeBottomLeft)
        setArmRangeOffset( armOffset - ulPageSize );
     else
        setArmRangeOffset( armOffset + ulPageSize );
  }
  else
  {
     if (homePosition == IProgressIndicator::homeBottomLeft)
        setArmRangeOffset( armOffset + ulPageSize );
     else
        setArmRangeOffset( armOffset - ulPageSize );
  }

  return;
}


/*------------------------------------------------------------------------------
| INativeSliderHandler::INativeSliderHandler                                   |
|                                                                              |
| A default handler is needed when using the Windows native control.  Since    |
| the Windows trackbar does not support arrow buttons, a default handler is    |
| used for positioning/sizing the trackbar and drawing its arrow bitmaps.      |
------------------------------------------------------------------------------*/
INativeSliderHandler::INativeSliderHandler( )
  : INativeSliderHandler::Inherited( )
{ }

/*------------------------------------------------------------------------------
| INativeSliderHandler::~INativeSliderHandler                            |
|                                                                              |
------------------------------------------------------------------------------*/
INativeSliderHandler::~INativeSliderHandler( )
{ }


/*------------------------------------------------------------------------------
| INativeSliderHandler::dispatchHandlerEvent                                   |
|                                                                              |
------------------------------------------------------------------------------*/
bool INativeSliderHandler::dispatchHandlerEvent( IEvent& event )
{
  IMODTRACE_ALL("INativeSliderHandler::dispatchHandlerEvent");

  bool fStopHandling( false );

  IWindowHandle hwndControl(event.controlHandle());

  if ( hwndControl )
  {
     IWindowClassName className( hwndControl );

     if (className == WC_NATIVESLIDER)
     {
        switch (event.eventId())
        {
          case WM_HSCROLL:
          case WM_VSCROLL:
          case IC_UM_VSCROLL:
          {
            IMODTRACE_DEVELOP("INativeSliderHandler::dispatchHandlerEvent(WM_SCROLL)");

            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();
            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;

               if (~ISTYLEOF( ppd->trackbar ) & TBS_NOTHUMB)
               {
                  if (progressInd->isRibbonStripEnabled())
                     // This event is for a slider with a ribbon strip, so
                     // the selection range may need to be updated.
                     ppd->drawRibbon( ppd->armRangeOffset() );
                  if (((ISlider*)progressInd)->buttonsPosition() != ISlider::none)
                     // This slider has buttons, so check the button state.
                     ppd->setButtonState();
               }

               // Send an SLN_ notification so handlers will work.
               unsigned long subMsg = event.parameter1().number1();
               IWindowHandle hwndParent = IPARENTOF( progressInd->handle() );
               if (subMsg == TB_THUMBTRACK)
                  hwndParent.sendEvent( WM_COMMAND,
                        IEventParameter1( (short)IIDOF( event.handle() ), SLN_SLIDERTRACK ),
                        IEventParameter2( event.parameter2() )); //ekim
               else if (subMsg == TB_ENDTRACK)
                  hwndParent.sendEvent( WM_COMMAND,
                        IEventParameter1( (short)IIDOF( event.handle() ), SLN_CHANGE ),
                        IEventParameter2( event.parameter2() )); //ekim
            }
            fStopHandling = true;
            break;
          }

          case WM_SETFOCUS:
          {
            IMODTRACE_DEVELOP(
                  "INativeSliderHandler::dispatchHandlerEvent(WM_SETFOCUS)");

            // If this is a progress indicator, prevent it from getting the input
            // focus.
            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();
            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;
               unsigned long ulStyle = ISTYLEOF( ppd->trackbar );
               if (~ulStyle & TBS_NOTHUMB)
                  SetFocus( ppd->trackbar );
            }
          }
          break;

          case IC_UM_CANVAS_SETFOCUS:
          {
            IMODTRACE_DEVELOP(
                  "INativeSliderHandler::dispatchHandlerEvent(IC_UM_CANVAS_SETFOCUS)");

            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();
            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;
               SetFocus( ppd->trackbar );
            }
          }
          break;

          case WM_ENABLE:
          {
            IMODTRACE_DEVELOP(
                  "INativeSliderHandler::dispatchHandlerEvent(WM_ENABLE)");

            IProgressIndicator *progressInd =
               dynamic_cast<IProgressIndicator*>(event.controlWindow());
            if (progressInd)
            {
               EnableWindow( progressInd->fProgressIndicatorData->trackbar,
                             (bool)event.parameter1() );
            }
          }
          break;

          case WM_WINDOWPOSCHANGED:
          {
            IMODTRACE_DEVELOP(
                  "INativeSliderHandler::dispatchHandlerEvent(WM_WINDOWPOSCHANGED)");

            // Move or size event.
            PWINDOWPOS pswp = (PWINDOWPOS)(void*)event.parameter2();
            if (!(pswp->flags & SWP_NOSIZE))
            {
               // Size event found.  The native trackbar needs to be positioned
               // within the static control.
               IProgressIndicator *progressInd =
                     (IProgressIndicator*)event.controlWindow();
               if (progressInd)
                  progressInd->fProgressIndicatorData->layoutNativeSlider();
            }
            break;
          }

          case WM_PAINT:
          {
            IMODTRACE_DEVELOP("INativeSliderHandler::dispatchHandlerEvent(WM_PAINT)");

            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();

            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;
               IWindowHandle hwnd = progressInd->handle();
               PAINTSTRUCT rectlInvalid;
               IPresSpaceHandle
                 hps( IBEGINPAINT( hwnd, 0, &rectlInvalid ), hwnd );
               {
                 IRootGrafPort  port (hps);

                 // Color the static control background with the background color.
                 IColor      penColor (progressInd->foregroundColor()),
                             backgroundColor (progressInd->backgroundColor());
                 IRectangle  rectPaint (rectlInvalid.rcPaint);
                 IGRect2D    colorRect (rectPaint);
                 IGArea      clipRegion (colorRect);

                 IBasicColor          bgndColor (backgroundColor);
                 IFillBundle          bgndFill (bgndColor);
                 ILinkedGrafPort      thePort (&port, &bgndFill);

                 thePort.draw (colorRect);

                 bgndFill.setFillColor (IBasicColor (penColor));
                 if (ppd->textVisible)
                 {
                   unsigned long  numElements =
                     ppd->tickText[progressInd->primaryScale()].numberOfValues ();
                   IKeyGraphicPair  dummy;
                   for (unsigned long  i = 0; i < numElements; i ++)
                   {
                     dummy = ppd->tickText[progressInd->primaryScale()].value (i);
                     ((IGraphicText *) (dummy.fObj))->draw (thePort);
                   } // for i
                 } // if text is visible

                 ppd->buttons.draw (thePort);

               }  /* brackets to ensure ~IGraphicContext called before IENDPAINT */
               IENDPAINT( hwnd, hps, &rectlInvalid );

               event.setResult( true );
               fStopHandling = true;
            }
            break;
          }

          case WM_LBUTTONDOWN:
          case WM_LBUTTONDBLCLK:
          {
            IMODTRACE_DEVELOP("INativeSliderHandler::dispatchHandlerEvent(WM_LBUTTONDOWN)");

            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();
            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;

               if (GetFocus() != ppd->trackbar)
                  SetFocus( ppd->trackbar );

               // If progress indicator or slider with no buttons, do nothing.
               if ((ISTYLEOF( ppd->trackbar ) & TBS_NOTHUMB) ||
                   ((ISlider*)progressInd)->buttonsPosition() == ISlider::none)
                  break;

               // Set a reasonable aperture size for button hit testing.
               // Not supported.
               // gc.setHitApertureSize( ISize( 5, 5 ) );

               // Check for button pressed.
               IGPoint2D  pointer (event.parameter2 ().number1 (),
                                   event.parameter2 ().number2 ());

               IImage *  button = (IImage *)
                 ppd->buttons.topGraphicUnderPoint (pointer);

               if (button)
               {
                 ppd->buttonPressed (button);
                 event.setResult (true);
               }

            }
            break;
          }

          case WM_LBUTTONUP:
          {
            IMODTRACE_DEVELOP("INativeSliderHandler::dispatchHandlerEvent(WM_LBUTTONUP)");

            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();
            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;
               if ((ISTYLEOF( ppd->trackbar ) & TBS_NOTHUMB) ||
                   ((ISlider*)progressInd)->buttonsPosition() == ISlider::none)
                  // Either this is a progress indicator, or a slider with no buttons.
                  break;

               // Cancel any outstanding timer and reset the button state.
               if (ppd->timer)
               {
                  ppd->buttonReleased();
                  event.setResult( true );
               }
            }
            break;
          }

          case IC_UM_IS_AGGREGATE_CTRL:
          {
            event.setResult(true);
            fStopHandling = true;
            break;
          }

          case TBM_CLEARSEL:          // Clear selection range.
          case TBM_CLEARTICS:         // Remove tick marks.
          case TBM_GETCHANNELRECT:    // Get channel bounding rectangle.
          case TBM_GETLINESIZE:       // Get Up/Down/Left/Right key increment.
          case TBM_GETNUMTICS:        // Get number of tick marks.
          case TBM_GETPAGESIZE:       // Get PgUp/PgDown key increment.
          case TBM_GETPOS:            // Get current position of slider.
          case TBM_GETPTICS:          // Get tick array.
          case TBM_GETRANGEMIN:       // Get the current range of
          case TBM_GETRANGEMAX:       //    the slider.
          case TBM_GETSELSTART:       // Get the current selection range
          case TBM_GETSELEND:         //    of the slider.
          case TBM_GETTHUMBLENGTH:    // Get the slider length.
          case TBM_GETTHUMBRECT:      // Get the slider bounding rectangle.
          case TBM_GETTIC:            // Get tick mark position.
          case TBM_GETTICPOS:         // Get tick mark position in client coord.
          case TBM_SETLINESIZE:       // Set Up/Down/Left/Right key increment.
          case TBM_SETPAGESIZE:       // Set PgUp/PgDown key increment.
          case TBM_SETPOS:            // Set current position of slider.
          case TBM_SETRANGE:          // Set the min/max positions for slider.
          case TBM_SETRANGEMIN:       // Set the min position for slider.
          case TBM_SETRANGEMAX:       // Set the max position for slider.
          case TBM_SETSEL:            // Set the selection range in trackbar.
          case TBM_SETSELSTART:       // Set the start pos. of the selection range.
          case TBM_SETSELEND:         // Set the end position of the selection range.
          case TBM_SETTHUMBLENGTH:    // Set the slider length.
          case TBM_SETTIC:            // Set tick mark position.
          case TBM_SETTICFREQ:        // Set the tick frequency.
          {
            // Forward all trackbar messages to the native trackbar window.
            // This handles the case where a message is sent directly to the control
            // window whose handle is returned from handle() function for the
            // IProgressIndicator window.
            IProgressIndicator *progressInd =
                  (IProgressIndicator*)event.controlWindow();
            if (progressInd)
            {
               IProgressIndicatorData *ppd = progressInd->fProgressIndicatorData;
               IEventResult result =
                     ppd->trackbar.sendEvent( event.eventId(),
                                              event.parameter1(),
                                              event.parameter2() );
               event.setResult( result );
               fStopHandling = result.asUnsignedLong();
            }
            break;
          }
        }
     }
  }   // if control

  return fStopHandling;
}


/*------------------------------------------------------------------------------
| isliderTrackbarWinProc                                                       |
| Window procedure for the trackbar part of a native slider.  This procedure   |
| intercepts the focus notifications for the trackbar and sends them to        |
| the trackbar owner.  Other messages are sent on to the original procedure    |
| for the slider's trackbar.                                                   |
------------------------------------------------------------------------------*/
long  __stdcall isliderTrackbarWinProc( HWND          hwnd,
                                        unsigned int  msg,
                                        unsigned int  mp1,
                                        long          mp2)
{
   unsigned long ulStyle = ISTYLEOF( hwnd );
   IWindowHandle hwndSlider = IPARENTOF( hwnd );
   IWindowHandle hwndParent = IPARENTOF( hwndSlider );
   IProgressIndicator *progInd =
         dynamic_cast<IProgressIndicator*>
            (IWindow::windowWithHandle( hwndSlider, false ));
   if (progInd)
   {
      switch( msg )
      {
       case WM_KEYDOWN:
         {
            if (mp1 == VK_UP || mp1 == VK_DOWN || mp1 == VK_LEFT || mp1 == VK_RIGHT)
            {
               // The Windows trackbar has a non-intuitive handling of
               // cursor up/down for a horizontal slider or cursor
               // right/left for a vertical trackbar.  In order to
               // prevent this, we'll eat the up/down events for horizontal
               // trackbars and the right/left for vertical trackbars.
               bool isVertical = ISTYLEOF( hwnd ) & TBS_VERT ? true : false;
               if ((isVertical && (mp1 == VK_LEFT || mp1 == VK_RIGHT)) ||
                   (!isVertical && (mp1 == VK_UP || mp1 == VK_DOWN)))
                  return true;
            }
         }
         break;

       case WM_SYSCHAR:
       case WM_CHAR:
         if ((msg == WM_SYSCHAR) || (mp1 == VK_TAB))
         {
            // Send these to the trackbar controls parent.  We need
            // VK_TAB to handle keyboard navigation.  We need to pass on
            // WM_SYSCHAR for correct mnemonic behavior.
            hwndParent.sendEvent((unsigned long)msg,
                                 (unsigned long)mp1,
                                 (unsigned long)mp2);
            return true;
         }
         break;

       case WM_SETFOCUS:
         if (ulStyle & TBS_NOTHUMB)
         {
            // This is a progress indicator.  We want to prevent it from
            // receiving input focus.
            SetFocus( (HWND)mp1 );
            return true;
         }
         else
         {
            // This is a slider so send a focus notification to the slider's
            // parent.
            hwndParent.sendEvent(
                  WM_COMMAND,
                  IEventParameter1( (short)IIDOF( hwndSlider ), SLN_SETFOCUS ),
                  IEventParameter2( hwndSlider.asUnsigned() ));
         }
         break;

       case WM_KILLFOCUS:
         hwndParent.sendEvent(
               WM_COMMAND,
               IEventParameter1( (short)IIDOF( hwndSlider ), SLN_KILLFOCUS ),
               IEventParameter2( hwndSlider.asUnsigned() ));
         break;

       default:
         break;
      }

      return CallWindowProc(
#ifdef IC_WIN_STRICT
                progInd->fProgressIndicatorData->fdefaultTrackbarProc,
#else
                (FARPROC)progInd->fProgressIndicatorData->fdefaultTrackbarProc,
#endif
                hwnd, msg, mp1, mp2 );
   }
   else
   {
      // We are basically busted if we can't find our parent IWindow.  We do
      // what is probably the most reasonable thing and call the
      // default procedure.
      return IDEFWINDOWPROC( hwnd, msg, mp1, mp2 );
   }
}

