// Revision: 47 1.45.1.4 source/ui/basectl/islider.cpp, slidectls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: islider.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in islider.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

extern "C"
{
  #define INCL_WINERRORS
  #define INCL_WINSTDSLIDER
  #define INCL_DOSMODULEMGR
  #define INCL_WINMESSAGEMGR
  #define INCL_WINSYS
  #include <iwindefs.h>

#ifdef IC_WIN
   #include <commctrl.h>
#endif
   
#ifdef IC_MOTIF
  #include <X11/Xlib.h>
  #include <X11/keysym.h>
  #include <Xm/Scale.h>
  #include <Xm/ScrollBar.h>
  #include <Xm/ArrowB.h>
  #include <Xm/Form.h>
  #include <Xm/Label.h>
  #include <Xm/Separator.h>
  #include <Xm/DrawingA.h>
  #include <X11/IntrinsicP.h>
  #include <X11/CoreP.h>
#endif
}

#ifdef IC_WIN
  #ifndef _ICLSLIDERW_
  #include <iclsldw.h>
  #endif
  #include <icctlsta.hpp>         // statics to load dll
  #include <isliderw.hpp>         // native slider class
  #include <igrport.hpp>
  #include <igraftxt.hpp>
  #include <ibrushdr.hpp>
#endif
#include <islider.hpp>
#include <icconst.h>
#include <icmnfun.hpp>
#include <icolor.hpp>
#include <icoordsy.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <ihandle.hpp>
#include <ihandler.hpp>
#include <inotifev.hpp>
#include <iplatfrm.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <istring.hpp>
#include <istyles.hpp>
#include <istylset.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>
#include <iwinpriv.hpp>
#include <ilanglvl.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ISLIDER_CPP_
  #include <ipagetun.h>
#endif


#ifdef IC_MOTIF

// Define the slider arm size.  Necessary since calculating value results in
// inconsistent results.
#define ARM_SIZE 30

void _System islArmCallback (         Widget  w,
                                      void    *client_data,
                                      void    *call_data );
void _System islDisarmCallback (      Widget  w,
                                      void    *client_data,
                                      void    *call_data );
void _System islValueChangedCallback (Widget  w,
                                      void    *client_data,
                                      void    *call_data );
void _System islDragCallback (        Widget  w,
                                      void    *client_data,
                                      void    *call_data );
void _System islResizeCallback (      Widget  w,
                                      void    *client_data,
                                      void    *call_data );
void _System islExposeEventCallback ( Widget  w,
                                      void    *client_data,
                                      _XEvent *event,
                                      ::Boolean *continueToDispatch );
void _System islKeyPressCallback(     Widget  w,
                                      void    *client_data,
                                      _XEvent *event,
                                      ::Boolean *continueToDispatch );
void _System islMoveSlider ( IProgressIndicatorData *fProgressIndicatorData,
                             XtIntervalId *id );

/*------------------------------------------------------------------------------
| IProgressIndicatorData class                                                 |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IProgressIndicatorData {
public:
  IProgressIndicatorData( IProgressIndicator* progressInd ) :
        pControl (progressInd),
        primaryScale (IProgressIndicator::scale1),
        timer (0),
        activeButton (0),
        buttonsPosition (ISlider::none)
  { };

  IProgressIndicator* pControl;
  Widget        slider,
                tickForm[2],
                arrowLeftDown,
                arrowRightUp,
                activeButton;
  XtIntervalId  timer;
  unsigned long numberTicks[2],
                tickSpacing[2],
                longestTick[2];
  ISlider::ButtonsPosition
                buttonsPosition;
  IProgressIndicator::Scale
                primaryScale;
  IProgressIndicator::Alignment
                align;
  bool          isRibbonStrip,
                isSnapToTick,
                alreadyVisible;

  void setSensitivity ();
  void fillWithRibbon ();
  void layoutControl ();
  void registerCallbacks ();
  void unregisterCallbacks ();

private:
#if IC_EXTENDED_RTTI_ENABLED
  IProgressIndicatorData() { };
  IProgressIndicatorData(const IProgressIndicatorData&) { };
#else
  IProgressIndicatorData();
  IProgressIndicatorData(const IProgressIndicatorData&);
#endif // IC_EXTENDED_RTTI_ENABLED
  IProgressIndicatorData& operator= (const IProgressIndicatorData&);
};    // IProgressIndicatorData

/*------------------------------------------------------------------------------
| ISliderData class                                                            |
------------------------------------------------------------------------------*/
class ISliderData {
public:
  ISliderData() : buttonsPosition (ISlider::none)
  { };

  ISlider::ButtonsPosition
                buttonsPosition;

private:
#if IC_EXTENDED_RTTI_ENABLED
  ISliderData (const ISliderData&) { };
#else
  ISliderData (const ISliderData&);
#endif // IC_EXTENDED_RTTI_ENABLED
  ISliderData& operator= (const ISliderData&);

};    // ISliderData

/*------------------------------------------------------------------------------
| Slider control data structure                                                |
------------------------------------------------------------------------------*/
typedef struct _SLDCDATA
{
  unsigned long increments[2];
  unsigned long spacing[2];
} sldcdata;

#pragma pack(pop)
#pragma enum(pop)

#endif //IC_MOTIF

#ifdef IC_PMWIN
class IPMProgressIndicatorData;
/*------------------------------------------------------------------------------
| IDefaultSliderHandler class                                                  |
|                                                                              |
| The IDefaultSliderHandler class implements the default handler needed for    |
| the PM compatible progress indicator / slider.                               |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IDefaultSliderHandler : public IHandler
{
typedef IHandler
  Inherited;

public:
IDefaultSliderHandler( IPMProgressIndicatorData *progressData );

protected:
virtual bool
  dispatchHandlerEvent  ( IEvent& event );

private:
IPMProgressIndicatorData *progressData;
};

/*------------------------------------------------------------------------------
| IPMProgressIndicatorData class                                               |
|                                                                              |
| The IPMProgressIndicatorData class is the private data class for the PM      |
| compatible progress indicator / slider.                                      |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IPMProgressIndicatorData {
public:
  IPMProgressIndicatorData( IProgressIndicator *progressInd );

IDefaultSliderHandler
  fDefaultHandler;

#ifdef IC_WIN
IBrushHandler
  fBrushHandler;
#endif

void
  calcLongestTick( );

IProgressIndicator
 *progressInd;

unsigned long
  longestTick[2];

bool
  variableSpacing[2],
  needsTextMinCalc;

unsigned long
  initialArmOffset;

private:
#if IC_EXTENDED_RTTI_ENABLED
  IPMProgressIndicatorData(const IPMProgressIndicatorData&)
  : fDefaultHandler( 0 )
#ifdef IC_WIN
  , fBrushHandler( )
#endif
  { };
#else
  IPMProgressIndicatorData(const IPMProgressIndicatorData&)
  : fDefaultHandler( 0 )
#ifdef IC_WIN
  , fBrushHandler( )
#endif
  { };
#endif // IC_EXTENDED_RTTI_ENABLED
  IPMProgressIndicatorData& operator= (const IPMProgressIndicatorData&);
};    // IPMProgressIndicatorData
#endif // IC_PMWIN

/*------------------------------------------------------------------------------
| Public progress indicator styles.                                            |
------------------------------------------------------------------------------*/
const IProgressIndicator::Style
  IProgressIndicator::vertical          = SLS_VERTICAL,
  IProgressIndicator::alignBottom       = SLS_BOTTOM,
  IProgressIndicator::alignTop          = SLS_TOP,
  IProgressIndicator::alignLeft         = SLS_LEFT,
  IProgressIndicator::alignRight        = SLS_RIGHT,
  IProgressIndicator::snapToTickMark    = SLS_SNAPTOINCREMENT,
  IProgressIndicator::handleDrawItem    = SLS_OWNERDRAW,
  IProgressIndicator::ribbonStrip       = SLS_RIBBONSTRIP,
  IProgressIndicator::homeTop           = SLS_HOMETOP,
  IProgressIndicator::homeRight         = SLS_HOMERIGHT,
  IProgressIndicator::primaryScale2     = SLS_PRIMARYSCALE2,
  IProgressIndicator::pmCompatible      ( 0, ISLS_PMCOMPATIBILITY);

#ifdef IC_PMWIN
const IProgressIndicator::Style
  IProgressIndicator::horizontal        ( 0, ISLS_HORIZONTAL ),
  IProgressIndicator::alignCentered     ( 0, ISLS_CENTER ),
  IProgressIndicator::homeBottom        ( 0, ISLS_HOMEBOTTOM ),
  IProgressIndicator::homeLeft          ( 0, ISLS_HOMELEFT ),
  IProgressIndicator::primaryScale1     ( 0, ISLS_PRIMARYSCALE1 ),
  IProgressIndicator::border3D          ( 0, IWS_BORDER3D ),
  IProgressIndicator::classDefaultStyle ( (WS_VISIBLE      |
                                           SLS_RIBBONSTRIP |
                                           SLS_READONLY),
                                                       (ISLS_HORIZONTAL |
                                                        ISLS_CENTER     |
                                                        ISLS_HOMELEFT   |
                                                        ISLS_PRIMARYSCALE1 |
                                                        IWS_BORDER3D) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
const IProgressIndicator::Style
  IProgressIndicator::horizontal        (IC_SLS_HORIZONTAL),
  IProgressIndicator::alignCentered     (IC_SLS_CENTER),
  IProgressIndicator::homeLeft          (IC_SLS_HOMELEFT),
  IProgressIndicator::homeBottom        (IC_SLS_HOMEBOTTOM),
  IProgressIndicator::primaryScale1     (IC_SLS_PRIMARYSCALE1),
  IProgressIndicator::border3D          ( 0 ),
  IProgressIndicator::classDefaultStyle =
                                   ( WS_VISIBLE           |
                                     IC_SLS_HORIZONTAL    |
                                     IC_SLS_CENTER        |
                                     IC_SLS_HOMELEFT      |
                                     SLS_RIBBONSTRIP      |
                                     IC_SLS_PRIMARYSCALE1 |
                                     SLS_READONLY );
#endif //IC_MOTIF


/*------------------------------------------------------------------------------
| Default style for new progress indicator objects (initial value).            |
------------------------------------------------------------------------------*/
#ifdef IC_PMWIN
IProgressIndicator::Style
  IProgressIndicator::currentDefaultStyle ( (WS_VISIBLE      |
                                             SLS_RIBBONSTRIP |
                                             SLS_READONLY),
                                                        (ISLS_HORIZONTAL |
                                                         ISLS_CENTER     |
                                                         ISLS_HOMELEFT   |
                                                         ISLS_PRIMARYSCALE1 |
                                                         IWS_BORDER3D) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
IProgressIndicator::Style IProgressIndicator::currentDefaultStyle =
                                IProgressIndicator::classDefaultStyle;
#endif //IC_MOTIF


/*------------------------------------------------------------------------------
| IProgressIndicator::IProgressIndicator                                       |
------------------------------------------------------------------------------*/
IProgressIndicator :: IProgressIndicator ( unsigned long     windowId,
                                           IWindow*          parent,
                                           IWindow*          owner,
                                           const IRectangle& rectInit,
                                           unsigned long     numberOfTicks,
                                           unsigned long     tickSpacing,
                                           const Style&      style )
  : Inherited( )
#ifdef IC_PMWIN
  , fPMProgressIndicatorData( 0 )
#endif
  , fProgressIndicatorData( 0 )
{
   IMODTRACE_DEVELOP("IProgressIndicator::ctor1");
   IASSERTPARM(parent!=0)

   Style readonlyStyle(style);
   readonlyStyle |= SLS_READONLY;

   // Save the extended style to make sure we have a copy of it stored
   setExtendedStyle( extendedStyle() |
                     style.asExtendedUnsignedLong() |
                     readonlyStyle.asExtendedUnsignedLong() );
#ifdef IC_PMWIN
#ifdef IC_WIN
   // If this is a native Windows slider, we have to save the snap to tick
   // style as an extended style.  This style is implemented with the native
   // trackbar by setting the max range to SHRT_MAX and interpreting tick
   // positions accordingly.  We also have to save the alignment and home
   // position as extended styles.
   if (!isPMCompatible())
   {
      unsigned long nativeExtended = 0;

      nativeExtended |= (style & snapToTickMark) ? ISLS_SNAPTOTICK : 0;

      nativeExtended |= (style & alignTop || style & alignRight)
                        ? ISLS_TOP
                      : ((style & alignBottom || style & alignLeft)
                        ? ISLS_BOTTOM : 0);

      nativeExtended |= (style & homeTop || style & homeRight)
                        ? ISLS_HOMETOP : 0;

      setExtendedStyle( extendedStyle() | nativeExtended );
   }
#endif

   SLDCDATA sldcData;                  //Slider control data
   sldcData.cbSize = sizeof(SLDCDATA);
   if (style & primaryScale2)
   {
      sldcData.usScale2Increments = (unsigned short)numberOfTicks;
      sldcData.usScale2Spacing = (unsigned short)tickSpacing;
      sldcData.usScale1Increments = (unsigned short)0;
      sldcData.usScale1Spacing = (unsigned short)0;
   }
   else
   {
      sldcData.usScale1Increments = (unsigned short)numberOfTicks;
      sldcData.usScale1Spacing = (unsigned short)tickSpacing;
      sldcData.usScale2Increments = (unsigned short)0;
      sldcData.usScale2Spacing = (unsigned short)0;
   }
   IWindowHandle::Value hwndOwner = 0;
   if (owner != 0)
      hwndOwner = owner->handle();

	 initialize(windowId, parent->handleForChildCreation(), hwndOwner,
              convertToGUIStyle( readonlyStyle ), rectInit,
              (void*)&sldcData);


#endif //IC_PMWIN

#ifdef IC_MOTIF
  IWindowHandle ownerHandle = 0;
  if (owner != 0)
     ownerHandle = owner->handle();

  sldcdata sliderData;
  sliderData.increments[scale1] = numberOfTicks;
  sliderData.spacing[scale1] = tickSpacing;
  sliderData.increments[scale2] = 0;
  sliderData.spacing[scale2] = 0;

  fProgressIndicatorData = new IProgressIndicatorData( this );
  initialize(windowId,
             parent->handleForChildCreation(),
             ownerHandle,
             readonlyStyle.asUnsignedLong(),
             rectInit,
             (void*)&sliderData);
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IProgressIndicator::IProgressIndicator                                       |
------------------------------------------------------------------------------*/
IProgressIndicator :: IProgressIndicator ( unsigned long   windowId,
                                           IWindow*        parent,
                                           IWindow*        owner,
                                           const IRectangle& rectInit,
                                           unsigned long   scale1NumberOfTicks,
                                           unsigned long   scale1TickSpacing,
                                           unsigned long   scale2NumberOfTicks,
                                           unsigned long   scale2TickSpacing,
                                           const Style&    style )
  : Inherited( )
#ifdef IC_PMWIN
  , fPMProgressIndicatorData( 0 )
#endif
  , fProgressIndicatorData( 0 )
{
   IMODTRACE_DEVELOP("IProgressIndicator::ctor2");
   IASSERTPARM(parent!=0)

   Style readonlyStyle(style);
   readonlyStyle |= SLS_READONLY;

#ifdef IC_PMWIN
  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() |
                    style.asExtendedUnsignedLong() |
                    readonlyStyle.asExtendedUnsignedLong() );

#ifdef IC_WIN
   // If this is a native Windows slider, we have to save the snap to tick
   // style as an extended style.  This style is implemented with the native
   // trackbar by setting the max range to SHRT_MAX and interpreting tick
   // positions accordingly.  We also have to save the alignment and home
   // position as extended styles.
   if (!isPMCompatible())
   {
      unsigned long nativeExtended = 0;

      nativeExtended |= (style & snapToTickMark) ? ISLS_SNAPTOTICK : 0;

      nativeExtended |= (style & alignTop || style & alignRight)
                        ? ISLS_TOP
                      : ((style & alignBottom || style & alignLeft)
                        ? ISLS_BOTTOM : 0);

      nativeExtended |= (style & homeTop || style & homeRight)
                        ? ISLS_HOMETOP : 0;

      setExtendedStyle( extendedStyle() | nativeExtended );
   }
#endif

   SLDCDATA sldcData;                  //Slider control data
   sldcData.cbSize = sizeof(SLDCDATA);
   sldcData.usScale1Increments = (unsigned short)scale1NumberOfTicks;
   sldcData.usScale2Increments = (unsigned short)scale2NumberOfTicks;
   sldcData.usScale1Spacing = (unsigned short)scale1TickSpacing;
   sldcData.usScale2Spacing = (unsigned short)scale2TickSpacing;
   IWindowHandle::Value hwndOwner = 0;
   if (owner != 0)
      hwndOwner = owner->handle();
   initialize(windowId, parent->handleForChildCreation(), hwndOwner,
              convertToGUIStyle( readonlyStyle ), rectInit,
              (void*)&sldcData);

#endif //IC_PMWIN

#ifdef IC_MOTIF
  IWindowHandle ownerHandle = 0;
  if (owner != 0)
     ownerHandle = owner->handle();

  sldcdata sliderData;
  sliderData.increments[scale1] = scale1NumberOfTicks;
  sliderData.spacing[scale1] = scale1TickSpacing;
  sliderData.increments[scale2] = scale2NumberOfTicks;
  sliderData.spacing[scale2] = scale2TickSpacing;

  fProgressIndicatorData = new IProgressIndicatorData( this );
  initialize(windowId,
             parent->handleForChildCreation(),
             ownerHandle,
             readonlyStyle.asUnsignedLong(),
             rectInit,
             (void*)&sliderData);
#endif //IC_MOTIF
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IProgressIndicator::IProgressIndicator                                       |
------------------------------------------------------------------------------*/
IProgressIndicator :: IProgressIndicator( unsigned long ulId,
                                          IWindow* pwndParent )
  : Inherited( )
  , fPMProgressIndicatorData( 0 )
  , fProgressIndicatorData( 0 )
{
  IMODTRACE_DEVELOP("IProgressIndicator::ctor3");
  IASSERTPARM(pwndParent!=0)

   fPMProgressIndicatorData = new IPMProgressIndicatorData( this );

#ifdef IC_PM
  this->setAutoDestroyWindow( false );
  this->reserveUserWindowWord( false );
  this->startHandlingEventsFor( ulId, pwndParent );

  // We should not be altering the existing control by changing its
  // color, but this code has been here for a long time. So long
  // that it is safer at this point to leave it until somebody
  // complains about it (since removing it could break existing code
  // that is relying on this behavior).
  this->setBackgroundColor( defaultbackgroundcolor( handle() ) );
#endif //IC_PM

#ifdef IC_WIN
  IWindowHandle hwndParent = pwndParent->handle();
  IWindowHandle hwndControl = IWindow::handleWithParent( ulId,
                                                         hwndParent );
  IWindowClassName className( hwndControl );
  if (className == WC_SLIDER)
  {
     // Identify that this is a PM-compatible control.
     this->setExtendedStyle( this->extendedStyle()
                              | pmCompatible.asExtendedUnsignedLong() );

     this->setAutoDestroyWindow( false );
     this->reserveUserWindowWord( false );
     this->startHandlingEventsFor( ulId, pwndParent );

     // Set other styles as necessary.
     if ( this->style() & WS_EX_CLIENTEDGE )
     {
        this->setExtendedStyle( this->extendedStyle()
                                 | border3D.asExtendedUnsignedLong() );
     }
  }
  else if (className == TRACKBAR_CLASS)
  {
     initialize( hwndControl );
  }
  else
  {
     // Invalid window class.
     ITHROWLIBRARYERROR( IC_WRONG_CONTROL_TYPE,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
  }
#endif
}

/*------------------------------------------------------------------------------
| IProgressIndicator::IProgressIndicator                                       |
|                                                                              |
| Constructor from an existing progress indicator.                             |
------------------------------------------------------------------------------*/
IProgressIndicator :: IProgressIndicator( const IWindowHandle& wh)
  : Inherited( )
  , fPMProgressIndicatorData( 0 )
  , fProgressIndicatorData( 0 )
{
  fPMProgressIndicatorData = new IPMProgressIndicatorData( this );

#ifdef IC_PM
  this->setAutoDestroyWindow( false );
  this->reserveUserWindowWord( false );
  this->startHandlingEventsFor( wh );
#endif

#ifdef IC_WIN
  IWindowClassName className( wh );
  if (className == WC_SLIDER)
  {
     // Identify that this is a PM-compatible control.
     this->setExtendedStyle( this->extendedStyle()
                              | pmCompatible.asExtendedUnsignedLong() );

     this->setAutoDestroyWindow( false );
     this->reserveUserWindowWord( false );
     this->startHandlingEventsFor( wh );

     // Set other styles as necessary.
     if ( this->style() & WS_EX_CLIENTEDGE )
     {
        this->setExtendedStyle( this->extendedStyle()
                                 | border3D.asExtendedUnsignedLong() );
     }
  }
  else if (className == TRACKBAR_CLASS)
  {
     initialize( wh );
  }
  else
  {
     ITHROWLIBRARYERROR( IC_WRONG_CONTROL_TYPE,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
  }
#endif
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IProgressIndicator :: ProgressIndicator                                      |
------------------------------------------------------------------------------*/
IProgressIndicator :: IProgressIndicator ( )
  : Inherited( )
#ifdef IC_PMWIN
  , fPMProgressIndicatorData( 0 )
#endif
  , fProgressIndicatorData( 0 )
{
#ifdef IC_PMWIN
  fPMProgressIndicatorData = new IPMProgressIndicatorData( this );
#endif
#ifdef IC_MOTIF
  fProgressIndicatorData = new IProgressIndicatorData( this );
#endif
}


/*------------------------------------------------------------------------------
| IProgressIndicator::~IProgressIndicator                                      |
------------------------------------------------------------------------------*/
IProgressIndicator :: ~IProgressIndicator()
{
  IMODTRACE_DEVELOP("IProgressIndicator::dtor");

#ifdef IC_PMWIN
#ifdef IC_WIN
  if (!isPMCompatible())
  {
     fProgressIndicatorData->fdefaultHandler.stopHandlingEventsFor( this );

     if (isValid())
     {
        if (fProgressIndicatorData->isWrappered)
        {
           // If this is a wrappered trackbar, we want to reparent and resize the
           // trackbar to use the parent and size of our WC_NATIVESLIDER window.  We
           // also need to remove our subclassed window procedure and destroy the
           // WC_NATIVESLIDER window.

           // Move the visible, tabStop, and group styles back to the trackbar if
           // they are present.
           unsigned long wrapperStyle = ISTYLEOF( handle() );
           unsigned long wrapperExtStyle = GetWindowLong( handle(),
                                                          GWL_EXSTYLE );
           unsigned long trackbarStyle = ISTYLEOF( fProgressIndicatorData->trackbar );
           unsigned long trackbarExtStyle = GetWindowLong(
                                                  fProgressIndicatorData->trackbar,
                                                  GWL_EXSTYLE );
           trackbarStyle |= (wrapperStyle & (WS_TABSTOP | WS_GROUP ));
           if (!(wrapperStyle & WS_VISIBLE))
              trackbarStyle &= ~WS_VISIBLE;
           ISETWINDOWSTYLE( fProgressIndicatorData->trackbar, trackbarStyle );

           trackbarExtStyle |= (wrapperExtStyle & WS_EX_CLIENTEDGE);
           SetWindowLong( fProgressIndicatorData->trackbar,
                          GWL_EXSTYLE,
                          trackbarExtStyle );

           // Reparent the trackbar to the wrapper window's parent.
           ISETPARENT( fProgressIndicatorData->trackbar,
                        IPARENTOF( handle() ),
                        false );

           // Get the wrapper window size and set the trackbar to this size.
           // Set the z-order of the trackbar to be behind the wrapper window.
           // After the wrapper window is destroyed, the trackbar's z-order will
           // be correct.
           IRectangle rectWrapper = nativeRect();
           SetWindowPos( fProgressIndicatorData->trackbar,
                         handle(),
                         rectWrapper.minX(),
                         rectWrapper.minY(),
                         rectWrapper.width(),
                         rectWrapper.height(),
                         SWP_NOACTIVATE );

           // Restore the trackbar's window procedure.
           fProgressIndicatorData->destroyNativeSlider();

           // Set autoDestroyWindow on the wrapper window to allow IWindow to
           // finish the clean up.
           setAutoDestroyWindow( true );
        }
        else
           fProgressIndicatorData->destroyNativeSlider();
     }

     delete fProgressIndicatorData;
  }
  else
#endif // IC_WIN
  {
     fPMProgressIndicatorData->fDefaultHandler.stopHandlingEventsFor( this );
#ifdef IC_WIN
     fPMProgressIndicatorData->fBrushHandler.stopHandlingEventsFor( this );
#endif
  delete fPMProgressIndicatorData;
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  // Cancel any outstanding timer.
  if (fProgressIndicatorData->timer)
     XtRemoveTimeOut (fProgressIndicatorData->timer);

  if( isValid() )
  	fProgressIndicatorData->unregisterCallbacks();
  
  delete fProgressIndicatorData;
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IProgressIndicator::defaultStyle                                             |
|                                                                              |
| Return the default style for new progress indicator objects.                 |
------------------------------------------------------------------------------*/
IProgressIndicator::Style  IProgressIndicator :: defaultStyle ( )
{
   IMODTRACE_DEVELOP("IProgressIndicator::defaultStyle");

   return currentDefaultStyle;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IProgressIndicator::setDefaultStyle                                          |
|                                                                              |
| Sets the default style for new progress indicator objects.                   |
------------------------------------------------------------------------------*/
void IProgressIndicator::setDefaultStyle( const IProgressIndicator::Style& style )
{
   currentDefaultStyle = style;
}
#endif

/*------------------------------------------------------------------------------
| IProgressIndicator::convertToGUIStyle                                        |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: convertToGUIStyle(const IBitFlag& guiStyle,
                                                      bool bExtOnly) const
{
  IMODTRACE_DEVELOP("IProgressIndicator::convertToGUIStyle");

  // Obtain the style from the class (IControl) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    ulStyle |= guiStyle.asExtendedUnsignedLong() & ISLS_EXTGUIMASK;
  }
  else
  {
#ifdef IC_WIN
    if (extendedStyle() & ISLS_PMCOMPATIBILITY)
    {
#endif
    // SLS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word. Five of the styles
    // were denoted as 0 (OS/2 PM), due to the lack of available bits.
    // We are representing those as extended bits so we or the developer
    // can accurately test for them later.  Since their native values are 0,
    // we do not have to do any additional processing here.  Note that
    // since this class only deals with shaft related styles, we only
    // mask those out here.
    ulStyle |= guiStyle.asUnsignedLong() & ISLS_SHAFTMASK;
#ifdef IC_WIN
    }
    else
    {
       // Since we are using the CCL control SLS_ values, we need to translate
       // for the Windows native trackbar.

       unsigned long PMStyle = guiStyle.asUnsignedLong() & ISLS_SHAFTMASK;

       ulStyle |= (PMStyle & SLS_VERTICAL) ?
             ((PMStyle & SLS_PRIMARYSCALE2) ?
                   ( (TBS_VERT | TBS_LEFT) ) :
                   ( (TBS_VERT | TBS_RIGHT) )) :
             ((PMStyle & SLS_PRIMARYSCALE2) ?
                   ( (TBS_HORZ | TBS_BOTTOM) ) :
                   ( (TBS_HORZ | TBS_TOP) ));

       ulStyle |= (PMStyle & SLS_READONLY) ? TBS_NOTHUMB : 0;

       ulStyle |= (PMStyle & SLS_RIBBONSTRIP) ? TBS_ENABLESELRANGE : 0;
    }

    ulStyle |= WS_CHILD;
#endif
  }

  return( ulStyle );
}


/*------------------------------------------------------------------------------
| IProgressIndicator::tickPosition                                             |
|                                                                              |
| Return the pixel position of the tick at the specified index.  Ticks are     |
| numbered starting with zero.                                                 |
------------------------------------------------------------------------------*/
IPoint IProgressIndicator :: tickPosition ( unsigned long ulTickNum ) const
{
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("IProgressIndicator::tickPosition");

#ifdef IC_PMWIN
   IEventResult evtTickData;
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      return IPoint();
   }
   else
   {
#endif
   evtTickData = handle().sendEvent(SLM_QUERYTICKPOS,
                                 IEventParameter1(ulTickNum),
                                 IEventParameter2((unsigned long)0));

   if (evtTickData.number1() == 0 || evtTickData.number2() == 0)
      ITHROWGUIERROR2("SLM_QUERYTICKPOS",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
#ifdef IC_WIN
   }
#endif

   IPoint pt((unsigned short)evtTickData.number1(),
             (unsigned short)evtTickData.number2());

   // Always returns end of tick closest to shaft
   return ICoordinateSystem::convertToApplication( pt, this->size() );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (ulTickNum > numberOfTicks (primaryScale()))
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_NUMBER,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);

  Position      tickFormx, tickFormy;
  unsigned long tickx, ticky, spacing;
  Dimension     tickFormHeight, tickFormWidth, daHeight, scaleHeight;

  XtVaGetValues (handle(),
                 XmNheight, &daHeight,
                 NULL);

  if (daHeight <= 1)
     // Scale probably is not mapped.
     return IPoint (0, 0);

  XtVaGetValues (handle(),
                 XmNheight, &scaleHeight,
                 NULL);

  XtVaGetValues (fProgressIndicatorData->tickForm[primaryScale()],
                 XmNx,      &tickFormx,
                 XmNy,      &tickFormy,
                 XmNheight, &tickFormHeight,
                 XmNwidth,  &tickFormWidth,
                 NULL);

  spacing = tickSpacing (primaryScale());

  if (isVertical())
  {
     if (primaryScale() == scale1)
        tickx = tickFormx;
     else
        tickx = tickFormx + tickFormWidth;
     ticky = daHeight - tickFormHeight - tickFormy;
     if (homePosition() == homeBottomLeft)
        // May have roundoff error here.
        ticky += ulTickNum * spacing + spacing / 2;
     else
        ticky += (numberOfTicks (primaryScale()) - ulTickNum - 1) *
                 spacing + spacing / 2;
  }
  else
  {
     if (homePosition() == homeBottomLeft)
        tickx = tickFormx + ulTickNum * spacing + spacing / 2;
     ticky = daHeight - tickFormHeight - tickFormy;
     if (primaryScale() == scale2)
        ticky -= scaleHeight;
  }
  return IPoint (tickx, ticky);
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IProgressIndicator::tickLength                                               |
|                                                                              |
| Return the length of the tick at the specified index, in pixels.             |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: tickLength ( unsigned long ulTickNum ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::tickLength");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      Scale primary = primaryScale();
      IASSERTPARM( ulTickNum <
            fProgressIndicatorData->numberTicks[primary]);

      if (fProgressIndicatorData->visibleTicks[primary].contains( ulTickNum )
          || ulTickNum == 0
          || ulTickNum == fProgressIndicatorData->numberTicks[primary]-1)
         return (1);
      else
         return (0);
   }
   else
   {
#endif
   IEventResult evtTickData =
           handle().sendEvent(SLM_QUERYTICKSIZE,
                              IEventParameter1(ulTickNum),
                              IEventParameter2((unsigned long)0));

   if (evtTickData.number1() == (USHORT)SLDERR_INVALID_PARAMETERS)
      ITHROWGUIERROR2("SLM_QUERYTICKSIZE",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);

   return evtTickData.number1();
#ifdef IC_WIN
   }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (ulTickNum > numberOfTicks (primaryScale()))
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_NUMBER,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);

  unsigned char tickOrientation;
  Dimension     tickHeight, tickWidth;
  Widget tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                      IString (ulTickNum));

  if (!XtIsManaged (tickWidget))
     return 0;

  XtVaGetValues (tickWidget,
                 XmNorientation, &tickOrientation,
                 XmNheight,      &tickHeight,
                 XmNwidth,       &tickWidth,
                 NULL);
  if (tickOrientation == XmVERTICAL)
     return (unsigned long)tickHeight;
  return (unsigned long)tickWidth;
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| IProgressIndicator::tickSpacing                                              |
|                                                                              |
| Returns the number of pixels between ticks for the specified scale.          |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: tickSpacing  ( Scale scale ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::tickSpacing");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      if (primaryScale() == scale
          && fProgressIndicatorData->tickSpacing[ scale ] == 0)
         return ((this->armRange()-1) / (this->numberOfTicks( scale )-1));
      else
         return fProgressIndicatorData->tickSpacing[ scale ];
   }
   else
   {
      // Query Slider control for the tick spacing information
      unsigned long ulTickSpacing =
            handle().sendEvent( SLM_QUERYSLIDERINFO,
                                IEventParameter1(SMA_SLIDERSCALEINFO,
                                                 SMA_SLIDERTICKSPACING),
                                IEventParameter2((unsigned long)0));

      // Check if it return an error (or the correct information
      if (ulTickSpacing == (ULONG)SLDERR_INVALID_PARAMETERS)
        ITHROWLIBRARYERROR( IC_SLDR_GET_CTLDATA,
                            IBaseErrorInfo::accessError,
                            IException::recoverable);

      // Return correct information based on scale requested
      if (scale == scale1)
        return ( LOWORD(ulTickSpacing) );
      else
        return ( HIWORD(ulTickSpacing) );
   }
#else
   WNDPARAMS wndParams;
   SLDCDATA sldcData;

   // Set up buffer with query information
   wndParams.fsStatus = WPM_CTLDATA | WPM_CBCTLDATA;
   wndParams.cbCtlData = sizeof( SLDCDATA );
   wndParams.pCtlData = (PVOID)&sldcData;

   unsigned long ulSuccess =
           handle().sendEvent(WM_QUERYWINDOWPARAMS,
                              IEventParameter1(&wndParams),
                              IEventParameter2((unsigned long)0));

   if (!ulSuccess)
      ITHROWLIBRARYERROR(IC_SLDR_GET_CTLDATA,
                         IBaseErrorInfo::accessError,
                         IException::recoverable);

   if (scale == scale1)
      return ((PSLDCDATA)(wndParams.pCtlData))->usScale1Spacing;
   else
      return ((PSLDCDATA)(wndParams.pCtlData))->usScale2Spacing;
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (!numberOfTicks (scale))
     return 0;

  if (primaryScale() == scale
      && fProgressIndicatorData->tickSpacing[ scale ] == 0)
  {
     int maxValue;
     XtVaGetValues (fProgressIndicatorData->slider,
                    XmNmaximum, &maxValue,
                    NULL);
     return (maxValue / (fProgressIndicatorData->numberTicks[primaryScale()] - 1));
  }
  else
     return fProgressIndicatorData->tickSpacing[ scale ];
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator::numberOfTicks                                            |
|                                                                              |
| Returns the number of ticks for the specified scale.                         |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: numberOfTicks ( Scale scale ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::numberOfTicks");

#ifdef IC_WIN
   if (!isPMCompatible())
   {
      // Since native Windows trackbar only supports one scale, private data
      // is used to store the number of ticks per scale.
      return fProgressIndicatorData->numberTicks [scale];
   }
   else
   {
      // Query Slider control for the tick spacing information
      unsigned long ulTickCount =
           handle().sendEvent( SLM_QUERYSLIDERINFO,
                               IEventParameter1(SMA_SLIDERSCALEINFO,
                                                SMA_SLIDERTICKCOUNT),
                               IEventParameter2((unsigned long)0));

      // Check if it return an error (or the correct information
      if (ulTickCount == (ULONG)SLDERR_INVALID_PARAMETERS)
        ITHROWLIBRARYERROR( IC_SLDR_GET_CTLDATA,
                            IBaseErrorInfo::accessError,
                            IException::recoverable);

      // Return correct information based on scale requested
      if (scale == scale1)
        return ( LOWORD(ulTickCount) );
      else
        return ( HIWORD(ulTickCount) );
   }
#endif //IC_WIN

#ifdef IC_PM
   WNDPARAMS wndParams;
   SLDCDATA sldcData;

   // Set up buffer with query information
   wndParams.fsStatus = WPM_CTLDATA | WPM_CBCTLDATA;
   wndParams.cbCtlData = sizeof( SLDCDATA );
   wndParams.pCtlData = (PVOID)&sldcData;

   unsigned long ulSuccess =
           handle().sendEvent(WM_QUERYWINDOWPARAMS,
                              IEventParameter1(&wndParams),
                              IEventParameter2((unsigned long)0));

   if (!ulSuccess)
      ITHROWLIBRARYERROR(IC_SLDR_GET_CTLDATA,
                         IBaseErrorInfo::accessError,
                         IException::recoverable);
   if (scale == scale1)
      return ((PSLDCDATA)(wndParams.pCtlData))->usScale1Increments;
   else
      return ((PSLDCDATA)(wndParams.pCtlData))->usScale2Increments;
#endif //IC_PM

#ifdef IC_MOTIF
  return fProgressIndicatorData->numberTicks[scale];
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setTicks                                                 |
|                                                                              |
| Sets the number of ticks and number of pixels between ticks.                 |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setTicks (
                                             unsigned long scale1NumberOfTicks,
                                             unsigned long scale2NumberOfTicks,
                                             unsigned long scale1TickSpacing,
                                             unsigned long scale2TickSpacing )
{
   IMODTRACE_DEVELOP("IProgressIndicator::setTicks");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      setTicks( scale1,
                scale1NumberOfTicks,
                scale1TickSpacing );
      setTicks( scale2,
                scale2NumberOfTicks,
                scale2TickSpacing );
   }
   else
   {
#endif
   SLDCDATA sldcData;

   sldcData.cbSize = sizeof(SLDCDATA);
   sldcData.usScale1Increments = (unsigned short)scale1NumberOfTicks;
   sldcData.usScale1Spacing =  (unsigned short)scale1TickSpacing;
   sldcData.usScale2Increments = (unsigned short)scale2NumberOfTicks;
   sldcData.usScale2Spacing = (unsigned short)scale2TickSpacing;

   // We need to know whether the scales are using variable spacing for minimum
   // size calculation.
   if (scale1TickSpacing == 0)
      fPMProgressIndicatorData->variableSpacing[0] = true;
   else
      fPMProgressIndicatorData->variableSpacing[0] = false;
   if (scale2TickSpacing == 0)
      fPMProgressIndicatorData->variableSpacing[1] = true;
   else
      fPMProgressIndicatorData->variableSpacing[1] = false;

#ifdef IC_WIN
      unsigned long ulResult =
            handle().sendEvent( SLM_SETSLIDERINFO,
                                IEventParameter1( SMA_SLIDERSCALEINFO,
                                                  SMA_SLIDERTICKCOUNT),
                                IEventParameter2( &sldcData ));
      if (ulResult == SLDERR_INVALID_PARAMETERS)
         ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable);

      ulResult =
            handle().sendEvent( SLM_SETSLIDERINFO,
                                IEventParameter1( SMA_SLIDERSCALEINFO,
                                                  SMA_SLIDERTICKSPACING),
                                IEventParameter2( &sldcData ));
      if (ulResult == SLDERR_INVALID_PARAMETERS)
         ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable);
   }
#else
   WNDPARAMS wndParams;
   wndParams.fsStatus = WPM_CTLDATA;
   wndParams.cbCtlData = sizeof(SLDCDATA);
   wndParams.pCtlData = (void*)(&sldcData);

   unsigned long ulSuccess =
           handle().sendEvent(WM_SETWINDOWPARAMS,
                              IEventParameter1(&wndParams),
                              IEventParameter2((unsigned long)0));
   if (!ulSuccess)
      ITHROWLIBRARYERROR(IC_SLDR_SET_CTLDATA,
                         IBaseErrorInfo::accessError,
                         IException::recoverable);
   refresh();
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  setTicks (scale1, scale1NumberOfTicks, scale1TickSpacing);
  setTicks (scale2, scale2NumberOfTicks, scale2TickSpacing);
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setTicks                                                 |
|                                                                              |
| Sets the number of ticks and number of pixels between ticks.                 |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setTicks (Scale scale,
                                                    unsigned long numberOfTicks,
                                                    unsigned long tickSpacing )
{
   IMODTRACE_DEVELOP("IProgressIndicator::setTicks");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      // Make sure at least two ticks have been specified for the scale if it
      // is the primary scale.
      Scale primary = primaryScale();
      if (scale == primary && numberOfTicks < 2)
         ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable);

      bool primaryChanged (false);

      unsigned long prevNumberTicks = fProgressIndicatorData->numberTicks[scale];
      if (scale == primary &&
          (prevNumberTicks != numberOfTicks
           || fProgressIndicatorData->tickSpacing[scale] != tickSpacing))
         primaryChanged = true;

      fProgressIndicatorData->numberTicks [scale] = numberOfTicks;
      fProgressIndicatorData->tickSpacing [scale] = tickSpacing;

      // Adjust trackbar for the primary scale.
      if (primaryChanged)
      {
        // If the number of ticks has changed, need to adjust the tick text
        // set.  Text for non-existent ticks will be removed.
        if (((fProgressIndicatorData->tickText[scale]).
                                         numberOfValues () != 0) &&
            ((fProgressIndicatorData->scaleFactor == 1 &&
              prevNumberTicks > numberOfTicks) ||
             (fProgressIndicatorData->scaleFactor == -1
                    && prevNumberTicks != numberOfTicks)))
        {
          unsigned long     numElements =
              (fProgressIndicatorData->tickText[scale]).numberOfValues ();

          IRawArray<IKeyGraphicPair>  tempStore (numElements);

          IKeyGraphicPair   dummy;
          IMGraphic *       obj;
          IGraphicText *    gText;

          for (unsigned long  i = 0; i < numElements; i ++)
          {
            dummy = (fProgressIndicatorData->tickText[scale]).value (i);
            tempStore.append (dummy);
          }

          (fProgressIndicatorData->tickText[scale]).resize (0);


          if (fProgressIndicatorData->scaleFactor == 1)
          {
            for (unsigned long  i = 0; i < numElements; i ++)
            {
              dummy = tempStore.value (i);
              if (dummy.fKey < numberOfTicks)
              {
                (fProgressIndicatorData->tickText[scale]).append (dummy);
              }
              else
              {
                break;
              }

            } // for i

          } // if

          else
          {
            // Copy, from the rear of tempStore, only those elements which
            // satisfy the criterion:
            //
            //   prevNumberTicks - dummy.fKey - 1 < numberOfTicks
            //
            // While copying maintain the order.
            //


            //
            // First figure out how many elments you need to copy.
            // This is needed because IRawArray only provides an
            // 'append' function.
            //
            for (long i = numElements - 1; i >= 0; i--)
            {
              dummy = tempStore.value (i);
              unsigned long  tickNum = prevNumberTicks - dummy.fKey - 1;
              if (tickNum < numberOfTicks)
              {
              }
              else
              {
                break;
              }

            } // for i

            i = (i < 0) ? 0 : i;

            //
            // Now copy the elments from 'tempStore' into the final
            // resting place.
            //
            unsigned long j;
            for (j = i; j < numElements; j ++)
            {
              dummy = tempStore.value (j);
              unsigned long  tickNum = prevNumberTicks - dummy.fKey - 1;
              dummy.fKey = numberOfTicks - tickNum - 1;
              (fProgressIndicatorData->tickText[scale]).append (dummy);

            } // for j

            // Free memory of those elements of tempStore which were
            // not retained.

            for (j = 0; j < i; j ++)
            {
              dummy = tempStore.value (j);
              IGraphicText *  gText = (IGraphicText *) dummy.fObj;
              delete gText;

            } // for j

          } // else if scale is reversed

        } // if

         fProgressIndicatorData->setupNativeScale( scale );

         // Force a relayout of the trackbar.
         fProgressIndicatorData->layoutNativeSlider();
      }
   }
   else
   {
#endif
   SLDCDATA sldcData;
   sldcData.cbSize = sizeof(SLDCDATA);
   if (scale == scale1)
   {
      sldcData.usScale1Increments = (unsigned short)numberOfTicks;
      sldcData.usScale1Spacing =  (unsigned short)tickSpacing;
   }
   else
   {
      sldcData.usScale2Increments = (unsigned short)numberOfTicks;
      sldcData.usScale2Spacing = (unsigned short)tickSpacing;
   }
   if (tickSpacing == 0)
      fPMProgressIndicatorData->variableSpacing[scale] = true;
   else
      fPMProgressIndicatorData->variableSpacing[scale] = false;
#ifdef IC_WIN
      unsigned long ulResult =
            handle().sendEvent( SLM_SETSLIDERINFO,
                                IEventParameter1( SMA_SLIDERSCALEINFO,
                                                  SMA_SLIDERTICKCOUNT),
                                IEventParameter2( &sldcData ));
      if (ulResult == SLDERR_INVALID_PARAMETERS)
         ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable);

      ulResult =
            handle().sendEvent( SLM_SETSLIDERINFO,
                                IEventParameter1( SMA_SLIDERSCALEINFO,
                                                  SMA_SLIDERTICKSPACING),
                                IEventParameter2( &sldcData ));
      if (ulResult == SLDERR_INVALID_PARAMETERS)
         ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable);
   }
#else
   WNDPARAMS wndParams;
   unsigned long ulSuccess;

  // Set up buffer with query information
   wndParams.fsStatus = WPM_CTLDATA | WPM_CBCTLDATA;
   wndParams.cbCtlData = sizeof( SLDCDATA );
   wndParams.pCtlData = (PVOID)&sldcData;
   ulSuccess = handle().sendEvent(WM_QUERYWINDOWPARAMS,
                                  IEventParameter1(&wndParams),
                                  IEventParameter2((unsigned long)0));
   if (!ulSuccess)
      ITHROWLIBRARYERROR(IC_SLDR_GET_CTLDATA,
                         IBaseErrorInfo::accessError,
                         IException::recoverable);

   sldcData.cbSize = sizeof(SLDCDATA);
   if (scale == scale1)
   {
      sldcData.usScale1Increments = (unsigned short)numberOfTicks;
      sldcData.usScale1Spacing =  (unsigned short)tickSpacing;
   }
   else
   {
      sldcData.usScale2Increments = (unsigned short)numberOfTicks;
      sldcData.usScale2Spacing = (unsigned short)tickSpacing;
   }
   wndParams.fsStatus = WPM_CTLDATA;
   wndParams.cbCtlData = sizeof(SLDCDATA);
   wndParams.pCtlData = (void*)&sldcData;

   ulSuccess = handle().sendEvent(WM_SETWINDOWPARAMS,
                                  IEventParameter1(&wndParams),
                                  IEventParameter2((unsigned long)0));
   if (!ulSuccess)
      ITHROWLIBRARYERROR(IC_SLDR_SET_CTLDATA,
                         IBaseErrorInfo::accessError,
                         IException::recoverable);

#endif

   refresh();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Arg           args[16];
  Widget        tickWidget, labelWidget;
  int           i, n;
  Pixel         backgroundColor, foregroundColor;

  // Get the background color of the drawing area.  We need to use the
  // same color in the separator widgets.
  XtVaGetValues (handle(),
                 XmNbackground, &backgroundColor,
                 XmNforeground, &foregroundColor,
                 NULL);

  if (fProgressIndicatorData->tickForm[scale])
  {
     for (i = 0; i < fProgressIndicatorData->numberTicks[scale]; i++)
     {
        tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[scale], IString(i));
        XtDestroyWidget (tickWidget);

        labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[scale],
                                      IString (i) + IString ("L"));
        if (labelWidget)
        {
           XtDestroyWidget (labelWidget);
        }
     }
  }

  if (!fProgressIndicatorData->tickForm[scale] && numberOfTicks)
  {
     n = 0;
     XtSetArg (args[n], XmNborderWidth, 0); n++;
     XtSetArg (args[n], XmNmarginHeight, 0); n++;
     XtSetArg (args[n], XmNmarginWidth, 0); n++;
     XtSetArg (args[n], XmNhighlightThickness, 0); n++;
     XtSetArg (args[n], XmNshadowThickness, 0); n++;
     XtSetArg (args[n], XmNfractionBase, MAX(1, numberOfTicks)); n++;
     XtSetArg (args[n], XmNbackground, backgroundColor); n++;
     fProgressIndicatorData->tickForm[scale] = XtCreateWidget (
                                   IString (id()),
                                   xmFormWidgetClass,
                                   handle(),
                                   args, n);
  }
  else if (fProgressIndicatorData->tickForm[scale] && !numberOfTicks)
  {
     XtDestroyWidget( fProgressIndicatorData->tickForm[scale] );
     fProgressIndicatorData->tickForm[scale] = 0;
  }

  if (numberOfTicks)
  {
     XtVaSetValues (fProgressIndicatorData->tickForm[scale],
                    XmNfractionBase, MAX(1, numberOfTicks),
                    NULL);

     // Add tick marks using separator widgets.  All of the tick mark widgets
     // will be added to a form widget.  Position attachments are used to
     // uniformly space the tick marks.
     n = 0;
     XtSetArg (args[n], XmNseparatorType, XmSINGLE_LINE); n++;
     XtSetArg (args[n], XmNbackground, backgroundColor); n++;
     XtSetArg (args[n], XmNforeground, foregroundColor); n++;
     XtSetArg (args[n], XmNborderWidth, 0); n++;
     XtSetArg (args[n], XmNmargin, 0); n++;
     XtSetArg (args[n], XmNhighlightThickness, 0); n++;
     XtSetArg (args[n], XmNshadowThickness, 0); n++;
     if (isVertical())
     {
        XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNwidth, 0); n++;
     }
     else
     {
        XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNheight, 0); n++;
     }

     switch (scale) {
       case scale1:
        if (isVertical())
        {
           XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
           XtSetArg (args[n], XmNrightAttachment, XmATTACH_NONE); n++;
        }
        else
        {
           XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
           XtSetArg (args[n], XmNtopAttachment, XmATTACH_NONE); n++;
        }
        break;
       case scale2:
        if (isVertical())
        {
           XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
           XtSetArg (args[n], XmNleftAttachment, XmATTACH_NONE); n++;
        }
        else
        {
           XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
           XtSetArg (args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
        }
        break;
     }

     for (i = 0; i < numberOfTicks; i++)
     {
        if (isVertical())
           switch (homePosition()) {
             case homeTopRight:
              XtSetArg (args[n], XmNtopPosition, i); n++;
              XtSetArg (args[n], XmNbottomPosition, i+1); n++;
              break;
             case homeBottomLeft:
              XtSetArg (args[n],
                       XmNbottomPosition, numberOfTicks-i); n++;
              XtSetArg (args[n],
                       XmNtopPosition, numberOfTicks-i-1); n++;
              break;
           }
        else
           switch (homePosition()) {
             case homeTopRight:
              XtSetArg (args[n],
                       XmNrightPosition, numberOfTicks-i); n++;
              XtSetArg (args[n],
                       XmNleftPosition, numberOfTicks-i-1); n++;
              break;
             case homeBottomLeft:
              XtSetArg (args[n], XmNleftPosition, i); n++;
              XtSetArg (args[n], XmNrightPosition, i+1); n++;
              break;
           }
        tickWidget = XtCreateWidget (IString (i),
                                     xmSeparatorWidgetClass,
                                     fProgressIndicatorData->tickForm[scale],
                                     args, n);
        n = n - 2;
     }

     if (primaryScale() == scale1 && alignment() != topRight)
        XtManageChild (fProgressIndicatorData->tickForm[scale1]);
     else if (primaryScale() == scale2 && alignment() != bottomLeft)
        XtManageChild (fProgressIndicatorData->tickForm[scale2]);
  }

  fProgressIndicatorData->numberTicks[scale] = numberOfTicks;
  fProgressIndicatorData->tickSpacing[scale] = tickSpacing;
  fProgressIndicatorData->longestTick[scale] = 0;

  if (scale == primaryScale())
  {
     // If the tick spacing is 0, set the maximum value for the scale (or
     // scrollbar) to #ticks-1.  This will enable the user to set an arm
     // position prior to the slider being shown, and have the arm move to
     // the same relative position once the slider is sized at show time.
     if (fProgressIndicatorData->tickSpacing[scale] == 0)
        XtVaSetValues (fProgressIndicatorData->slider,
                       XmNmaximum, fProgressIndicatorData->numberTicks[scale] - 1,
                       NULL);

     // Move the slider arm to the home position.
     moveArmToTick (0);
     fProgressIndicatorData->layoutControl();
  }
#endif //IC_MOTIF

  setLayoutDistorted (IWindow::minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setTickLength                                            |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setTickLength (
                                                    unsigned long ulTickNum,
                                                    unsigned long ulTickLength )
{
   IMODTRACE_DEVELOP("IProgressIndicator::setTickLength");

#ifdef IC_PMWIN
   unsigned long ulResult = 1;

#ifdef IC_WIN
   if (!isPMCompatible())
      // Individual tick lengths not supported with native control.  We
      // can either turn them on or off (with the exception of the first
      // and last tick marks which will always be present).
   {
      Scale primary = primaryScale();

      if ((fProgressIndicatorData->visibleTicks[primaryScale()]
                .contains( ulTickNum ) && ulTickLength)
          || (!fProgressIndicatorData->visibleTicks[primaryScale()]
                .contains( ulTickNum ) && !ulTickLength)
          || ulTickNum == 0
          || ulTickNum == fProgressIndicatorData->numberTicks[primary] - 1)
         return *this;

      unsigned long ulPageSize, ulPosition;

      ulPageSize = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_GETPAGESIZE,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ));

      if (ulTickLength)
      {
         fProgressIndicatorData->visibleTicks[primary].add( ulTickNum );
         ulPosition = ulTickNum * ulPageSize * fProgressIndicatorData->scaleFactor;
         ulResult = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_SETTIC,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( ulPosition ));
      }
      else
      {
         fProgressIndicatorData->visibleTicks[primary]
               .remove( ulTickNum );
         fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_CLEARTICS,
                                       IEventParameter1( this->isVisible() ),
                                       IEventParameter2( 0 ));
         TickSet::Cursor cursor(
               fProgressIndicatorData->visibleTicks[primaryScale()] );
         forCursor (cursor)
         {
            ulPosition =
                  fProgressIndicatorData->visibleTicks[primaryScale()].elementAt( cursor )
                  * ulPageSize * fProgressIndicatorData->scaleFactor;
            ulResult = fProgressIndicatorData->trackbar.sendEvent(
                                    TBM_SETTIC,
                                    IEventParameter1( 0 ),
                                    IEventParameter2( ulPosition ));
         }
      }
      fProgressIndicatorData->needsTextMinCalc = true;
   }
   else
   {
#endif // IC_WIN
      fPMProgressIndicatorData->needsTextMinCalc = true;

      ulResult =
              handle().sendEvent(SLM_SETTICKSIZE,
                                 IEventParameter1((unsigned short)ulTickNum,
                                                  (unsigned short)ulTickLength),
                                 IEventParameter2((unsigned long)0 ));
#ifdef IC_WIN
   }
#endif // IC_WIN

   if (!ulResult)
   {
      ITHROWGUIERROR2("SLM_SETTICKSIZE",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
   }

#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (ulTickNum >= numberOfTicks (primaryScale()))
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_NUMBER,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);

  Dimension     tickHeight, tickWidth, labelHeight = 0, labelWidth = 0,
                atickHeight, atickWidth, alabelHeight, alabelWidth;
  Widget        tickWidget, labelWidget;

  // Force a minimum non-zero length of two - Motif minimum
  if (ulTickLength == 1)
     ulTickLength++;

  tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                               IString (ulTickNum));
  labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                IString (ulTickNum) + "L");
  XtVaGetValues (tickWidget,
                 XmNheight, &tickHeight,
                 XmNwidth,  &tickWidth,
                 NULL);
  if (isVertical())
     XtVaSetValues (tickWidget,
                    XmNheight, tickHeight,
                    XmNwidth,  (Dimension)ulTickLength,
                    NULL);
  else
     XtVaSetValues (tickWidget,
                    XmNheight, (Dimension)ulTickLength,
                    XmNwidth,  tickWidth,
                    NULL);

  if (ulTickLength && !XtIsManaged (tickWidget))
  {
     XtManageChild (tickWidget);
     if (labelWidget != NULL)
        // Need to change the label attachment from the form to the
        // separator widget.
        if (isVertical())
           switch (primaryScale()) {
             case scale1:
              XtVaSetValues (labelWidget,
                             XmNleftAttachment, XmATTACH_WIDGET,
                             XmNleftWidget,     tickWidget,
                             NULL);
              break;
             case scale2:
              XtVaSetValues (labelWidget,
                             XmNrightAttachment, XmATTACH_WIDGET,
                             XmNrightWidget,     tickWidget,
                             NULL);
              break;
           }
        else
           switch (primaryScale()) {
             case scale1:
              XtVaSetValues (labelWidget,
                             XmNbottomAttachment, XmATTACH_WIDGET,
                             XmNbottomWidget,     tickWidget,
                             NULL);
              break;
             case scale2:
              XtVaSetValues (labelWidget,
                             XmNtopAttachment, XmATTACH_WIDGET,
                             XmNtopWidget,     tickWidget,
                             NULL);
              break;
           }
  }
  else if (!ulTickLength && XtIsManaged (tickWidget))
  {
     if (labelWidget != NULL)
        // Need to change the label attachment from the separator widget to
        // the form.
        if (isVertical())
           switch (primaryScale()) {
             case scale1:
              XtVaSetValues (labelWidget,
                             XmNleftAttachment, XmATTACH_FORM,
                             NULL);
              break;
             case scale2:
              XtVaSetValues (labelWidget,
                             XmNrightAttachment, XmATTACH_FORM,
                             NULL);
              break;
           }
        else
           switch (primaryScale()) {
             case scale1:
              XtVaSetValues (labelWidget,
                             XmNbottomAttachment, XmATTACH_FORM,
                             NULL);
              break;
             case scale2:
              XtVaSetValues (labelWidget,
                             XmNtopAttachment, XmATTACH_FORM,
                             NULL);
              break;
           }
     XtUnmanageChild (tickWidget);
  }

  if (labelWidget != NULL)
  {
     XtVaGetValues (labelWidget,
                    XmNheight, &labelHeight,
                    XmNwidth,  &labelWidth,
                    NULL);
  }

  if (isVertical())
  {
     if (ulTickLength + labelWidth > fProgressIndicatorData->longestTick[primaryScale()])
     {
        fProgressIndicatorData->longestTick[primaryScale()] = ulTickLength + labelWidth;
     }
     else if ((tickWidth + labelWidth == fProgressIndicatorData->longestTick[primaryScale()]) &&
              (ulTickLength + labelWidth < fProgressIndicatorData->longestTick[primaryScale()]))
     {
        // Tick form may need to shrink.
        for (int i = 0, newLongestTick = 0;
             i < fProgressIndicatorData->numberTicks[primaryScale()] &&
                newLongestTick != fProgressIndicatorData->longestTick[primaryScale()];
             i++)
        {
           tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                        IString(i));
           if (XtIsManaged (tickWidget))
              XtVaGetValues (tickWidget,
                             XmNwidth,  &atickWidth,
                             NULL);
           else
              atickWidth = 0;
           labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                         IString(i) + "L");
           if (labelWidget == NULL)
              alabelWidth = 0;
           else
              XtVaGetValues (labelWidget,
                             XmNwidth,  &alabelWidth,
                             NULL);
           if (atickWidth + alabelWidth > newLongestTick)
              newLongestTick = atickWidth + alabelWidth;
        }
        fProgressIndicatorData->longestTick[primaryScale()] = newLongestTick;
     }
  }
  else
  {
     if (ulTickLength + labelHeight > fProgressIndicatorData->longestTick[primaryScale()])
     {
        fProgressIndicatorData->longestTick[primaryScale()] = ulTickLength + labelHeight;
     }
     else if ((tickHeight + labelHeight == fProgressIndicatorData->longestTick[primaryScale()]) &&
              (ulTickLength + labelHeight < fProgressIndicatorData->longestTick[primaryScale()]))
     {
        // Tick form may need to shrink.
        for (int i = 0, newLongestTick = 0;
             i < fProgressIndicatorData->numberTicks[primaryScale()] &&
                newLongestTick != fProgressIndicatorData->longestTick[primaryScale()];
             i++)
        {
           tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                        IString(i));
           if (XtIsManaged (tickWidget))
              XtVaGetValues (tickWidget,
                             XmNheight, &atickHeight,
                             NULL);
           else
              atickHeight = 0;
           labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                         IString(i) + "L");
           if (labelWidget == NULL)
              alabelHeight = 0;
           else
              XtVaGetValues (labelWidget,
                             XmNheight, &alabelHeight,
                             NULL);
           if (atickHeight + alabelHeight > newLongestTick)
              newLongestTick = atickHeight + alabelHeight;
        }
        fProgressIndicatorData->longestTick[primaryScale()] = newLongestTick;
     }
  }

  fProgressIndicatorData->layoutControl();
#endif //IC_MOTIF

  setLayoutDistorted (IWindow::minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setTickLength                                            |
|                                                                              |
| Set the size of ALL ticks (in pixels).                                       |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setTickLength (
                                                  unsigned long ulTickLength )
{
   IMODTRACE_DEVELOP("IProgressIndicator::setTickLength (all)");

#ifdef IC_PMWIN
   unsigned long ulResult = 1;

#ifdef IC_WIN
   if (!isPMCompatible())
   {
      // Tick lengths are minimally supported with native control.
      // The first and last ticks are always visible.  The remainder of the
      // tick marks are optionally visible.  The best we can do here is
      // remove the ticks when a 0 tick length is specified for all of the
      // ticks, and show them when a non-zero tick length is specified.

      if (fProgressIndicatorData->visibleTicks[primaryScale()].isEmpty()
                 && !ulTickLength)
         return *this;

      Scale primary = primaryScale();
      if (ulTickLength)
      {
         unsigned long ulPageSize, ulPosition, ulTickNum, numSizable;

         ulPageSize = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_GETPAGESIZE,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ));

         numSizable = fProgressIndicatorData->numberTicks[primary] - 2;
         for (ulTickNum = 1; ulTickNum <= numSizable; ulTickNum++)
         {
            if (!fProgressIndicatorData->visibleTicks[primary].contains( ulTickNum))
            {
               fProgressIndicatorData->visibleTicks[primary].add( ulTickNum );
               ulPosition = ulTickNum * ulPageSize
                            * fProgressIndicatorData->scaleFactor;

               ulResult = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_SETTIC,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( ulPosition ));
            }
         }
      }
      else
      {
         fProgressIndicatorData->visibleTicks[primary].removeAll();
         fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_CLEARTICS,
                                       IEventParameter1( this->isVisible() ),
                                       IEventParameter2( 0 ));
      }
      fProgressIndicatorData->needsTextMinCalc = true;
   }
   else
   {
#endif // IC_WIN
      fPMProgressIndicatorData->needsTextMinCalc = true;

      ulResult =
              handle().sendEvent(SLM_SETTICKSIZE,
                                 IEventParameter1((unsigned long)SMA_SETALLTICKS,
                                                  (unsigned short)ulTickLength),
                                 IEventParameter2((unsigned long)0 ));
#ifdef IC_WIN
   }
#endif // IC_WIN

   if (!ulResult)
   {
      ITHROWGUIERROR2("SLM_SETTICKSIZE",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
   }
#endif //IC_PMWIN

#ifdef IC_MOTIF
  for (int tick = 0; tick < fProgressIndicatorData->numberTicks[primaryScale()]; tick++)
     setTickLength (tick, ulTickLength);
#endif //IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setTickText                                              |
|                                                                              |
| Set the text above or text to a tick.                                        |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setTickText ( unsigned long ulTickNum,
                                                        const char* pszTickText)
{
   IMODTRACE_DEVELOP("IProgressIndicator::setTickText");

#ifdef IC_PMWIN
   unsigned long ulResult = 1;
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      Scale primary = primaryScale();
      if (ulTickNum > fProgressIndicatorData->numberTicks[primaryScale()])
         ulResult = 0;
      else
      {
         unsigned long    stringId;
         IGraphicText *   gText = 0;
         IKeyGraphicPair  dummy;
         unsigned long    numElements =
           fProgressIndicatorData->tickText[primary].numberOfValues ();

         // Set up IText using the window's current font.
         IText newText (pszTickText);
         if (newText != IText(""))
         {
            ITextStyleSet textStyles;
            IFont sliderFont = font();
            textStyles.add( ITextBitmapStyle( sliderFont.isBitmap() ));
            textStyles.add( ITextBoldfaceStyle( sliderFont.isBold() ));
            textStyles.add( ITextItalicStyle( sliderFont.isItalic() ));
            textStyles.add( ITextOutlineStyle( sliderFont.isOutline() ));
            textStyles.add( ITextPointSizeStyle( sliderFont.pointSize() ));
            textStyles.add( ITextStrikethroughStyle( sliderFont.isStrikeout() ));
            textStyles.add( ITextTypefaceStyle( sliderFont.name() ));
            textStyles.add( ITextUnderlineStyle( sliderFont.isUnderscore() ));
            textStyles.add( ITextUneditableStyle( true ));
            newText.addStyles( textStyles );
         }

         if (fProgressIndicatorData->scaleFactor == 1)
           stringId = ulTickNum;
         else
           stringId = numberOfTicks (primary) - ulTickNum - 1;

         //
         // Get the last element already in the tickText list.
         //

         if (numElements != 0)
         {
           dummy = (fProgressIndicatorData->
                               tickText[primary]).value (numElements - 1);
           gText = (IGraphicText *) dummy.fObj;
         }

         if (!gText || dummy.fKey < stringId)
         {
           if (newText != IText (""))
           {
              gText = new IGraphicText (newText, IGraphicText::kSingleLine);
              gText->setHitEnabled (false);
              IKeyGraphicPair   newOne;
              newOne.fObj = gText;
              newOne.fKey = stringId;
              (fProgressIndicatorData->tickText[primary]).append (newOne);
           }
         }
         else
         {
           //
           // ...means the new tick text to be set is for some intermediate
           // tick value.
           //

           for (long  i = 0; i < numElements; i ++)
           {
             dummy = (fProgressIndicatorData->tickText[primary]).value (i);
             IGraphicText *  gText = (IGraphicText *) dummy.fObj;

             if (dummy.fKey == stringId)
             {
               if (newText == IText (""))
               {
                 // Remove null strings.

                 delete gText;
                 gText = 0;

                 //
                 // Now that it is an array, you will have to move
                 // all subsequent elements up by one. That's the only
                 // way to delete the element.
                 //

                 for (unsigned long  j = i + 1; j < numElements; j++)
                 {
                   IKeyGraphicPair  temp =
                     (fProgressIndicatorData->tickText[primary]).value (j);

                   (fProgressIndicatorData->tickText[primary]).
                     setValue (j - 1, temp);
                 }

                 numElements --;

                 (fProgressIndicatorData->tickText[primary]).
                                                resize (numElements);

                 //
                 // No! we are not done yet, just adjust the index and
                 // continue iterating.
                 //

                 i --;

               } // if empty strings
               else
               {
                 // Set new text value.
                 gText->setText (newText);
                 break;
               }
             }
             else if (dummy.fKey > stringId)
             {
               if (newText != IText (""))
               {
                 IGraphicText *  newOne =
                   new IGraphicText (newText, IGraphicText::kSingleLine);
                 newOne->setHitEnabled (false);

                 IKeyGraphicPair  newKeyPair;
                 newKeyPair.fKey = stringId;
                 newKeyPair.fObj = newOne;

                 //
                 // This new element has to be added at index 'i'.
                 // So we'll have to make space for it by moving
                 // all subsequent elements down by 1.
                 //

                 (fProgressIndicatorData->tickText[primary]).
                                            resize (numElements + 1);

                 for (long  j = numElements - 1; j >= i; j--)
                 {
                   IKeyGraphicPair  temp =
                     (fProgressIndicatorData->tickText[primary]).value (j);
                   (fProgressIndicatorData->tickText[primary]).
                                               setValue (j + 1, temp);
                 }

                 //
                 // Now we have made the space. Add the new
                 // tick mark at index 'i'. Then we are done. So break out
                 // of the iterating for loop.

                 (fProgressIndicatorData->tickText[primary]).
                                            setValue (i, newKeyPair);
                 break;
               }
             }

           } // for

         } // else if new tick mark lies in-between

         // Force a re-layout.
         bool wasVisible = this->isVisible();
         if (wasVisible)
            disableUpdate();
         fProgressIndicatorData->layoutNativeSlider();
         if (wasVisible)
         {
            enableUpdate();
            refresh();
         }
      }
      fProgressIndicatorData->needsTextMinCalc = true;
   }
   else
   {
#endif
      fPMProgressIndicatorData->needsTextMinCalc = true;

      ulResult = handle().sendEvent(SLM_SETSCALETEXT,
                                 IEventParameter1(ulTickNum),
                                 IEventParameter2((void*)pszTickText));
#ifdef IC_WIN
   }
#endif

   if (!ulResult)
   {
      ITHROWGUIERROR2("SLM_SETSCALETEXT",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
   }
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Widget        labelWidget, tickWidget;
  IString       labelName, labelText;
  XmString      newString;
  Dimension     stringHeight, stringWidth, marginHeight, marginWidth,
                labelHeight = 0, labelWidth = 0,
                tickHeight = 0, tickWidth = 0,
                alabelHeight, alabelWidth, atickHeight, atickWidth;
  XmFontList    fontlist;
  Arg           args[20];
  int           n;
  Pixel         backgroundColor, foregroundColor;

  if (ulTickNum >= numberOfTicks (primaryScale()))
  {
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_NUMBER,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);
  }

  n = 0;
  labelText = IString (pszTickText);
  newString = XmStringCreateLtoR (labelText, XmFONTLIST_DEFAULT_TAG);
  XtSetArg (args[n], XmNlabelString, newString); n++;

  labelName = IString (ulTickNum) + "L";
  // Get the label widget.
  labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()], labelName);
  // Get the corresponding tick widget.
  tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                               IString (ulTickNum));
  if (labelWidget == NULL)
  {
     // Use same background color for label widget as drawing area uses.
     XtVaGetValues (handle(),
                    XmNbackground, &backgroundColor,
                    XmNforeground, &foregroundColor,
                    NULL);
     XtSetArg (args[n], XmNbackground, backgroundColor); n++;
     XtSetArg (args[n], XmNforeground, foregroundColor); n++;

     if (isVertical())
     {
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        switch (homePosition()) {
          case homeTopRight:
           XtSetArg (args[n], XmNtopPosition, ulTickNum); n++;
           XtSetArg (args[n], XmNbottomPosition, ulTickNum + 1); n++;
           break;
          case homeBottomLeft:
           XtSetArg (args[n], XmNtopPosition,
                    fProgressIndicatorData->numberTicks[primaryScale()] - ulTickNum - 1); n++;
           XtSetArg (args[n], XmNbottomPosition,
                     fProgressIndicatorData->numberTicks[primaryScale()] - ulTickNum); n++;
           break;
        }

        switch (primaryScale()) {
          case scale1:
           if (XtIsManaged (tickWidget))
           {
              XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
              XtSetArg (args[n], XmNleftWidget, tickWidget); n++;
           }
           else
           {
              XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
           }
           XtSetArg (args[n], XmNrightAttachment, XmATTACH_NONE); n++;
           XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
           break;
          case scale2:
           XtSetArg (args[n], XmNleftAttachment, XmATTACH_NONE); n++;
           if (XtIsManaged (tickWidget))
           {
              XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
              XtSetArg (args[n], XmNrightWidget, tickWidget); n++;
           }
           else
           {
              XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
           }
           XtSetArg (args[n], XmNalignment, XmALIGNMENT_END); n++;
           break;
        }
     }
     else
     {
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNalignment, XmALIGNMENT_CENTER); n++;
        switch (homePosition()) {
          case homeBottomLeft:
           XtSetArg (args[n], XmNleftPosition, ulTickNum); n++;
           XtSetArg (args[n], XmNrightPosition, ulTickNum + 1); n++;
           break;
          case homeTopRight:
           XtSetArg (args[n], XmNleftPosition,
                    fProgressIndicatorData->numberTicks[primaryScale()] - ulTickNum - 1); n++;
           XtSetArg (args[n], XmNrightPosition,
                     fProgressIndicatorData->numberTicks[primaryScale()] - ulTickNum); n++;
           break;
        }

        switch (primaryScale()) {
          case scale1:
           XtSetArg (args[n], XmNtopAttachment, XmATTACH_NONE); n++;
           if (XtIsManaged (tickWidget))
           {
              XtSetArg (args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
              XtSetArg (args[n], XmNbottomWidget, tickWidget); n++;
           }
           else
           {
              XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
           }
           break;
          case scale2:
           if (XtIsManaged (tickWidget))
           {
              XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
              XtSetArg (args[n], XmNtopWidget, tickWidget); n++;
           }
           else
           {
              XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
           }
           XtSetArg (args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
           break;
        }
     }

     XtSetArg (args[n], XmNmarginTop, 0); n++;
     XtSetArg (args[n], XmNmarginBottom, 0); n++;
     XtSetArg (args[n], XmNmarginLeft, 0); n++;
     XtSetArg (args[n], XmNmarginRight, 0); n++;
     labelWidget = XtCreateManagedWidget (labelName,
                                          xmLabelWidgetClass,
                                          fProgressIndicatorData->tickForm[primaryScale()],
                                          args, n);
  }
  else
  {
     XtVaGetValues (labelWidget,
                    XmNheight, &labelHeight,
                    XmNwidth,  &labelWidth,
                    NULL);
     XtSetValues (labelWidget, args, n);
  }

  // Determine height of new string.  The tick form may need to be re-sized
  // and re-positioned.
  XtVaGetValues (labelWidget,
                 XmNfontList,     &fontlist,
                 XmNmarginHeight, &marginHeight,
                 XmNmarginWidth,  &marginWidth,
                 NULL);
  XmStringExtent (fontlist,
                  newString,
                  &stringWidth,
                  &stringHeight);
  XmStringFree (newString);
  if (XtIsManaged (tickWidget))
  {
     XtVaGetValues (tickWidget,
                    XmNheight, &tickHeight,
                    XmNwidth,  &tickWidth,
                    NULL);
  }

  if (isVertical())
  {
     if (stringWidth + 2*marginWidth + tickWidth >
            fProgressIndicatorData->longestTick[primaryScale()])
        fProgressIndicatorData->longestTick[primaryScale()] =
            stringWidth + 2*marginWidth + tickWidth;
     else if ((labelWidth + tickWidth ==
                  fProgressIndicatorData->longestTick[primaryScale()]) &&
              (stringWidth + 2*marginWidth + tickWidth <
                  fProgressIndicatorData->longestTick[primaryScale()]))
     {
        // Tick form may need to shrink.
        for (int i = 0, newLongestTick = 0;
             i < fProgressIndicatorData->numberTicks[primaryScale()] &&
                newLongestTick != fProgressIndicatorData->longestTick[primaryScale()];
             i++)
        {
           tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                        IString(i));
           if (XtIsManaged (tickWidget))
              XtVaGetValues (tickWidget,
                             XmNwidth,  &atickWidth,
                             NULL);
           else
              atickWidth = 0;
           labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                         IString(i) + "L");
           if (labelWidget == NULL)
              alabelWidth = 0;
           else
              XtVaGetValues (labelWidget,
                             XmNwidth,  &alabelWidth,
                             NULL);
           if (atickWidth + alabelWidth > newLongestTick)
              newLongestTick = atickWidth + alabelWidth;
        }
        fProgressIndicatorData->longestTick[primaryScale()] = newLongestTick;
     }
  }
  else
  {
     if (stringHeight + 2*marginHeight + tickHeight >
            fProgressIndicatorData->longestTick[primaryScale()])
        fProgressIndicatorData->longestTick[primaryScale()] =
            stringHeight + 2*marginHeight + tickHeight;
     else if ((labelHeight + tickHeight == fProgressIndicatorData->longestTick[primaryScale()]) &&
              (stringHeight + 2*marginHeight + tickHeight <
                  fProgressIndicatorData->longestTick[primaryScale()]))
     {
        // Tick form may need to shrink.
        for (int i = 0, newLongestTick = 0;
             i < fProgressIndicatorData->numberTicks[primaryScale()] &&
                newLongestTick != fProgressIndicatorData->longestTick[primaryScale()];
             i++)
        {
           tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                        IString(i));
           if (XtIsManaged (tickWidget))
              XtVaGetValues (tickWidget,
                             XmNheight, &atickHeight,
                             NULL);
           else
              atickHeight = 0;
           labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                         IString(i) + "L");
           if (labelWidget == NULL)
              alabelHeight = 0;
           else
              XtVaGetValues (labelWidget,
                             XmNheight, &alabelHeight,
                             NULL);
           if (atickHeight + alabelHeight > newLongestTick)
              newLongestTick = atickHeight + alabelHeight;
        }
        fProgressIndicatorData->longestTick[primaryScale()] = newLongestTick;
     }
  }
  fProgressIndicatorData->layoutControl();
#endif //IC_MOTIF

  setLayoutDistorted (IWindow::minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setTickText                                              |
|                                                                              |
| Set the text above or below the tick.                                        |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setTickText ( unsigned long ulTickNum,
                                                        const IResourceId& text)
{
   IMODTRACE_DEVELOP("IProgressIndicator::setTickText");

   IString str = text.resourceLibrary().loadString(text);
   setTickText(ulTickNum,(char*)str);

   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::shaftPosition                                            |
|                                                                              |
| Return the lower left point of the shaft relative to the lower left          |
| of the control window.                                                       |
------------------------------------------------------------------------------*/
IPoint IProgressIndicator :: shaftPosition ( ) const
{
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("IProgressIndicator::shaftPosition");

#ifdef IC_PMWIN
   IRectangle shaftRect;
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      return IPoint();
   }
   else
   {
#endif
   IEventResult evtShaftData =
          handle().sendEvent(SLM_QUERYSLIDERINFO,
                             IEventParameter1((unsigned long)SMA_SHAFTPOSITION),
                             IEventParameter2((unsigned long)0));

   shaftRect = IRectangle(
                 IPoint((unsigned short)evtShaftData.number1(),
                        (unsigned short)evtShaftData.number2() ),
                 this->shaftSize() );
#ifdef IC_WIN
   }
#endif

   return ICoordinateSystem::convertToApplication(
             shaftRect,
             this->size() ).minXMinY() ;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  // Note that the position returned here assumes that the control is
  // already mapped!

  Dimension daHeight, scaleHeight;
  Position  scalex, scaley;
  XtVaGetValues (handle(),
                 XmNheight, &daHeight,
                 NULL);
  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNx,      &scalex,
                 XmNy,      &scaley,
                 XmNheight, &scaleHeight,
                 NULL);

  // Want to determine the lower left position wrt the drawing area.
  // Position will include the highlight border and shadow.
  return IPoint (scalex, daHeight - scaley - scaleHeight);
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IProgressIndicator::shaftSize                                                |
------------------------------------------------------------------------------*/
ISize IProgressIndicator :: shaftSize ( ) const
{
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("IProgressIndicator::shaftSize");

#ifdef IC_PMWIN
   ISize sz;
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      return ISize();
   }
   else
   {
#endif
   IEventResult evtShaftData =
        handle().sendEvent(SLM_QUERYSLIDERINFO,
                           IEventParameter1((unsigned long)SMA_SHAFTDIMENSIONS),
                           IEventParameter2((unsigned long)0));

   if (style() & vertical.asUnsignedLong())
   {
      sz.setWidth((unsigned short)evtShaftData.number2());
      sz.setHeight((unsigned short)evtShaftData.number1());
   }
   else
   {
      sz.setWidth((unsigned short)evtShaftData.number1());
      sz.setHeight((unsigned short)evtShaftData.number2());
   }
#ifdef IC_WIN
   }
#endif
   return sz;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  // Note that the size returned here assumes that the control is
  // already mapped!

  Dimension scaleHeight, scaleWidth, highlightThickness, shadowThickness;
  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNheight,             &scaleHeight,
                 XmNwidth,              &scaleWidth,
                 XmNhighlightThickness, &highlightThickness,
                 XmNshadowThickness,    &shadowThickness,
                 NULL);

  // Size will include the highlight border and shadow.
  return ISize (scaleWidth, scaleHeight);
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator::setShaftPosition                                         |
|                                                                              |
| Set the position of the progress indicator / slider shaft.                   |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setShaftPosition ( const IPoint& position )
{
   ITRACE_MOTIF_NOP();
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("IProgressIndicator::setShaftPosition");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
      return *this;
#endif
   IPoint nativePt = ICoordinateSystem::convertToNative(
                        IRectangle( position, this->shaftSize() ),
                        this->size() ).minXMinY() ;

   unsigned long ulResult =
        handle().sendEvent(SLM_SETSLIDERINFO,
                           IEventParameter1((unsigned long)SMA_SHAFTPOSITION),
                           IEventParameter2((unsigned short)nativePt.x(),
                                            (unsigned short)nativePt.y()));
   if (!ulResult)
      ITHROWGUIERROR2("SLM_SETSLIDERINFO",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);

#endif //IC_PMWIN
   return *this;

}

/*------------------------------------------------------------------------------
| IProgressIndicator::setShaftBreadth                                          |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setShaftBreadth (
                                                  unsigned long ulShaftBreadth )
{
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("IProgressIndicator::setShaftBreadth");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
      return *this;
#endif

   unsigned long ulResult =
        handle().sendEvent(SLM_SETSLIDERINFO,
                           IEventParameter1((unsigned long)SMA_SHAFTDIMENSIONS),
                           IEventParameter2(ulShaftBreadth));
   if (!ulResult)
      ITHROWGUIERROR2("SLM_SETSLIDERINFO",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Dimension highlightThickness, shadowThickness;

  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNhighlightThickness, &highlightThickness,
                 XmNshadowThickness,    &shadowThickness,
                 NULL);

  // Ignore the request if the breadth is too small.
  if (2 * (highlightThickness + shadowThickness) >= ulShaftBreadth)
     return *this;

  // The shaft breadth includes the shadow and highlight thicknesses.
  if (isVertical())
     if (XtIsSubclass (fProgressIndicatorData->slider, xmScaleWidgetClass))
     {
        XtVaSetValues (fProgressIndicatorData->slider,
                       XmNwidth,      ulShaftBreadth,
                       XmNscaleWidth, ulShaftBreadth,
                       NULL);
     }
     else
     {
        XtVaSetValues (fProgressIndicatorData->slider,
                       XmNwidth,      ulShaftBreadth,
                       NULL);
     }
  else
     if (XtIsSubclass (fProgressIndicatorData->slider, xmScaleWidgetClass))
     {
        XtVaSetValues (fProgressIndicatorData->slider,
                       XmNheight,      ulShaftBreadth,
                       XmNscaleHeight, ulShaftBreadth,
                       NULL);
     }
     else
     {
        XtVaSetValues (fProgressIndicatorData->slider,
                       XmNheight,      ulShaftBreadth,
                       NULL);
     }

  // Reposition the scale.
  fProgressIndicatorData->layoutControl();
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::armRange                                                 |
|                                                                              |
| Query the range over which the arm can be moved.                             |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: armRange ( ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::armRange");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      // Range here means the pixel extent of the slider.
      return (fProgressIndicatorData->pixelRange + 1);
   }
#endif

   IEventResult evtArmData =
      handle().sendEvent(SLM_QUERYSLIDERINFO,
                         IEventParameter1((unsigned long)SMA_SLIDERARMPOSITION,
                                          (unsigned long)SMA_RANGEVALUE ),
                         IEventParameter2((unsigned long)0 ));
   return evtArmData.number2();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int scaleMax;

  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNmaximum, &scaleMax,
                 NULL);
  return scaleMax + 1;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator::armPixelOffset                                           |
|                                                                              |
| Get position of arm as an offset from home in pixels.                        |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: armPixelOffset ( ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::armPixelOffset");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      unsigned long ulArmPosition = fProgressIndicatorData->armRangeOffset();
      if (isSnapToTickEnabled())
         // Need to calculate the arm offset in pixels from home position.
         ulArmPosition *= tickSpacing (primaryScale());
      else
      {
         unsigned long armPixelRange = this->armRange() - 1;
         unsigned long trackbarRange = fProgressIndicatorData->trackbarRange();
         if (trackbarRange)
           ulArmPosition = (ulArmPosition * armPixelRange + trackbarRange/2)
                           / trackbarRange;
      }
      return ulArmPosition;
   }
   else
   {
#endif

   IEventResult evtArmData =
      handle().sendEvent(SLM_QUERYSLIDERINFO,
                         IEventParameter1((unsigned long)SMA_SLIDERARMPOSITION,
                                          (unsigned long)SMA_RANGEVALUE),
                         IEventParameter2((unsigned long)0));
   return evtArmData.number1();
#ifdef IC_WIN
   }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int scaleValue, min, max;

  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNvalue, &scaleValue,
                 XmNminimum, &min,
                 XmNmaximum, &max,
                 NULL);

  return scaleValue;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator::armTickOffset                                            |
|                                                                              |
| Return the position of the arm as a tick number.  Ticks are numbered         |
| starting at 0.                                                               |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: armTickOffset ( ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::armTickOffset");

#ifdef IC_PMWIN
#ifdef IC_WIN
   // Native trackbar.
   if (!isPMCompatible())
   {
      unsigned long ulArmPosition
           = fProgressIndicatorData->armRangeOffset();

      if (!isSnapToTickEnabled())
      {
         unsigned long ulPageSize = fProgressIndicatorData->trackbar.sendEvent(
                                          TBM_GETPAGESIZE,
                                          IEventParameter1( 0 ),
                                          IEventParameter2( 0 ));
         if (ulPageSize)
           ulArmPosition = (ulArmPosition + ulPageSize / 2) / ulPageSize;
      }

      return ulArmPosition;
   }
   else
   {
#endif

      IEventResult evtArmData =
         handle().sendEvent(SLM_QUERYSLIDERINFO,
                            IEventParameter1((unsigned long)SMA_SLIDERARMPOSITION,
                                             (unsigned long)SMA_INCREMENTVALUE),
                            IEventParameter2((unsigned long)0));

      return evtArmData.number1();
#ifdef IC_WIN
   }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned long tickOffset = armPixelOffset() / tickSpacing (primaryScale());
  if (armPixelOffset() - tickOffset * tickSpacing (primaryScale()) >
      tickSpacing (primaryScale()) / 2)
     tickOffset++;
  return tickOffset;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator::moveArmToTick                                            |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: moveArmToTick (
                                                  unsigned long ulTickNum )
{
  IMODTRACE_DEVELOP("IProgressIndicator::moveArmToTick");

#ifdef IC_PMWIN
  /*******************************************************************/
  /* Only attempt to set it if we are attempting to set a new value. */
  /*******************************************************************/
  if (armTickOffset() != ulTickNum)
  {
    unsigned long ulResult = 1;
    bool          delayNotify = false;
#ifdef IC_WIN
    if (!isPMCompatible())
    {
       if (ulTickNum >= fProgressIndicatorData->numberTicks[primaryScale()])
          ulResult = 0;
       else
       {
          unsigned long ulNewPosition = ulTickNum;
          if (!isSnapToTickEnabled())
          {
             unsigned long ulPageSize = fProgressIndicatorData->trackbar.sendEvent(
                                          TBM_GETPAGESIZE,
                                          IEventParameter1( 0 ),
                                          IEventParameter2( 0 ));
             ulNewPosition *= ulPageSize;
          }
          fProgressIndicatorData->setArmRangeOffset( ulNewPosition );
       }
    }
    else
    {
#endif
       // There currently is a problem on OS/2 4.0 which results in a trap in
       // pmctls.dll if we attempt to set the arm position on a progress indicator
       // with variable tick spacing which hasn't been sized.  If we have that
       // case, just cache the desired arm offset.  Note that we won't move the
       // arm until we get a WM_SIZE event even if the application calls setTicks
       // and modifies the tick spacing.  This is reasonable since we want to isolate
       // the code for this workaround.  (This problem also exists in the CCL control.)
       if (size() == ISize() &&
           fPMProgressIndicatorData->variableSpacing[primaryScale()])
       {
          fPMProgressIndicatorData->initialArmOffset = ulTickNum;
          delayNotify = true;
       }
       else
       {
          ulResult =
             handle().sendEvent(SLM_SETSLIDERINFO,
                      IEventParameter1((unsigned long)SMA_SLIDERARMPOSITION,
                                       (unsigned long)SMA_INCREMENTVALUE),
                                       IEventParameter2(ulTickNum));
       }
#ifdef IC_WIN
    }
#endif // IC_WIN

#if (IC_OBSOLETE <= IC_OBSOLETE_2)
    if (ulResult && !delayNotify)
    {
      notifyObservers( INotificationEvent( IProgressIndicator::armTickOffsetId,
                                           *this,
                                           true,
                                           (void*)ulTickNum ));
    }
#endif

    if (!ulResult)
    {
      ITHROWGUIERROR2("SLM_SETSLIDERINFO",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
    }
  }
#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (ulTickNum >= numberOfTicks (primaryScale()))
  {
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_NUMBER,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);
  }

  XtVaSetValues (fProgressIndicatorData->slider,
                 XmNvalue, ulTickNum * tickSpacing (primaryScale()),
                 NULL);

  if (isRibbonStripEnabled())
     fProgressIndicatorData->fillWithRibbon();

  if (fProgressIndicatorData->arrowLeftDown)
     fProgressIndicatorData->setSensitivity();

#if (IC_OBSOLETE <= IC_OBSOLETE_2)
  this->notifyObservers(INotificationEvent( IProgressIndicator::armTickOffsetId,
                                            *this,
                                            true,
                                            (void*)ulTickNum ));
#endif

  // Generate SLN_CHANGE event.
  postEvent( IWindow::control,
             IEventData( (unsigned short)id(), SLN_CHANGE ),
             IEventData( armPixelOffset() ));

  // d7630 - Flush X events to see the slider update.
//  IWindowData::processPendingMsgs();
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| Function Name: IProgressIndicator :: moveArmToPixel                          |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: moveArmToPixel (
                                                  unsigned long ulArmOffset )
{
  IMODTRACE_DEVELOP("IProgressIndicator::moveArmToPixel");

#ifdef IC_PMWIN
  /*******************************************************************/
  /* Only attempt to set it if we are attempting to set a new value. */
  /*******************************************************************/
  if (armPixelOffset() != ulArmOffset)
  {
    unsigned long ulNewPosition = 0,
                  ulResult = 1;
    bool          delayNotify = false;
#ifdef IC_WIN
    if (!isPMCompatible())
    {
       if (ulArmOffset > this->armRange()-1)
          ulResult = 0;
       else
       {
          if (isSnapToTickEnabled())
          {
             // Adjust position to nearest tick.
             unsigned long spacing = tickSpacing( primaryScale() );
             if (spacing)
                ulNewPosition = (ulArmOffset + spacing / 2) / spacing;
          }
          else
          {
             // Calculate the range offset for the specified pixel.
             unsigned long armPixelRange = this->armRange()-1,
                           trackbarRange = fProgressIndicatorData->trackbarRange();
             if (armPixelRange)
               ulNewPosition = (ulArmOffset * trackbarRange + armPixelRange/2)
                               / armPixelRange;
          }

          fProgressIndicatorData->setArmRangeOffset( ulNewPosition );
       }
    }
    else
    {
#endif
       ulNewPosition = ulArmOffset;
       ulResult = handle().sendEvent(SLM_SETSLIDERINFO,
                             IEventParameter1((unsigned long)SMA_SLIDERARMPOSITION,
                                              (unsigned long)SMA_RANGEVALUE),
                             IEventParameter2(ulNewPosition));
#ifdef IC_WIN
    }
#endif

#if (IC_OBSOLETE <= IC_OBSOLETE_2)
    if (ulResult && !delayNotify)
    {
       notifyObservers( INotificationEvent( IProgressIndicator::armPixelOffsetId,
                                            *this,
                                            true,
                                            (void*)ulNewPosition ));
    }
#endif

    if (!ulResult)
    {
       ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                        IBaseErrorInfo::invalidParameter,
                        IException::recoverable );
    }
  }
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int maxValue;
  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNmaximum, &maxValue,
                 NULL);
  if (ulArmOffset > maxValue)
  {
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_PIXEL,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);
  }
          
  unsigned long ulNewPosition = 0;
  if (isSnapToTickEnabled())
  {
     // Adjust position to nearest tick.
     unsigned long spacing = tickSpacing( primaryScale() );
     if (spacing)
        ulNewPosition = spacing * ((ulArmOffset + spacing / 2) / spacing);
  }
  else
     ulNewPosition = ulArmOffset;
 
  XtVaSetValues (fProgressIndicatorData->slider,
                 XmNvalue, ulNewPosition,
                 NULL);

  if (isRibbonStripEnabled())
     fProgressIndicatorData->fillWithRibbon();

  if (fProgressIndicatorData->arrowLeftDown)
     fProgressIndicatorData->setSensitivity();

#if (IC_OBSOLETE <= IC_OBSOLETE_2)
  this->notifyObservers( INotificationEvent( IProgressIndicator::armPixelOffsetId,
                                             *this,
                                             true,
                                             (void*)ulNewPosition ));
#endif

  // Generate SLN_CHANGE event.
  postEvent( IWindow::control,
             IEventData( (unsigned short)id(), SLN_CHANGE ),
             IEventData( ulNewPosition ));

  // d7630 - Flush X events to see the slider update.
//  IWindowData::processPendingMsgs();
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::initialize                                               |
------------------------------------------------------------------------------*/
void IProgressIndicator :: initialize ( unsigned long     windowId,
                                        IWindowHandle     parent,
                                        IWindowHandle     owner,
                                        unsigned long     ulStyle,
                                        const IRectangle& rectInit,
                                        void *            sliderData )
{
  IMODTRACE_DEVELOP("IProgressIndicator::initialize");

#ifdef IC_PMWIN
  IWindowHandle wndhSlider;

#ifdef IC_WIN
  if (isPMCompatible())
  {
     // CCL control being used.  If needed load the DLL containing the control code.
     IControlStatics::loadControlDLL();
  }

  // Native trackbar.
  if (!isPMCompatible())
  {
     IProgressIndicatorData::registerSlider();

     fProgressIndicatorData = new IProgressIndicatorData();

     wndhSlider =
       this -> create( windowId,
                       0,
                       ulStyle | WS_CLIPCHILDREN,
                       WC_NATIVESLIDER,
                       parent,
                       owner,
                       rectInit,
                       0,
                       0,
                       defaultOrdering(),
                       convertToGUIStyle(
                         IProgressIndicator::Style( 0, extendedStyle() ),
                         true) );

     fProgressIndicatorData->createNativeSlider( wndhSlider,
                                                 wndhSlider,
                                                 ulStyle,
                                                 extendedStyle() );

     fProgressIndicatorData->fdefaultHandler.handleEventsFor( this );

     // Want to do startHandlingEventsFor here so we can use setTicks to
     // initialize the primary scale.
     startHandlingEventsFor(wndhSlider);

     SLDCDATA *psldcData = (SLDCDATA*)sliderData;
     this->setTicks ( (unsigned long)psldcData->usScale1Increments,
                      (unsigned long)psldcData->usScale2Increments,
                      (unsigned long)psldcData->usScale1Spacing,
                      (unsigned long)psldcData->usScale2Spacing );

     // Set the initial state of the buttons for sliders.
     fProgressIndicatorData->setButtonState();

     // Show now if visible style was specified.  (Trackbar is not created
     // with WS_VISIBLE style in order to prevent flashing during construction).
     if (ulStyle & visible.asUnsignedLong())
        this->show();
  }
  else

  // CCL Progress Indicator.
  {
#endif

     wndhSlider =
         this -> create( windowId,
                         0,
                         ulStyle,
                         WC_SLIDER,
                         parent,
                         owner,
                         rectInit,
                         sliderData,
                         0,
                         defaultOrdering(),
                         convertToGUIStyle(        // convert extended styles
                            IProgressIndicator::Style(0, extendedStyle() ),
                            true)     );

     fPMProgressIndicatorData = new IPMProgressIndicatorData( this );

     SLDCDATA *psldcData = (SLDCDATA*)sliderData;
     if ((unsigned long)psldcData->usScale1Spacing == 0)
        fPMProgressIndicatorData->variableSpacing[0] = true;
     if ((unsigned long)psldcData->usScale2Spacing == 0)
        fPMProgressIndicatorData->variableSpacing[1] = true;

     fPMProgressIndicatorData->fDefaultHandler.handleEventsFor( this );
#ifdef IC_WIN
     fPMProgressIndicatorData->fBrushHandler.handleEventsFor(this);
#endif

     startHandlingEventsFor(wndhSlider);
#ifdef IC_WIN
  }
#endif


#ifndef IC_WIN
  setBackgroundColor(defaultbackgroundcolor(wndhSlider));
#endif

	//Check if the default shaft is too large for the processor window.
	//If the shaft is too large, the shaft will be resized to fit the window
	//Note: If the window is too small, the control will not be drawn properly.

	ISize sizeOfShaft = shaftSize();
	if( isVertical() )
	{
	 	if( sizeOfShaft.width() >= rectInit.width() )
		{
			if( rectInit.width() > 9 )
			{
				setShaftBreadth( rectInit.width()-4);	
			}
		}
	}
	else
	{
	 	if( sizeOfShaft.height() >= rectInit.height() )
		{
			if( rectInit.height() > 9 )
			{
				setShaftBreadth( rectInit.height()-4);	
			}

		}
	}
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Widget        sliderda, arrowForm;
  Arg           args[15];
  int           i, n;
  sldcdata*     tickData = (sldcdata*) sliderData;
  Dimension     shadowThickness = 2, highlightThickness = 2;

  for (i = scale1; i <= scale2; i++)
  {
     fProgressIndicatorData->tickForm[i] = 0;
     fProgressIndicatorData->numberTicks[i] = tickData->increments[i];
     fProgressIndicatorData->tickSpacing[i] = tickData->spacing[i];
     fProgressIndicatorData->longestTick[i] = 0;
  }
  fProgressIndicatorData->arrowLeftDown = fProgressIndicatorData->arrowRightUp = 0;

  // Set primary scale.
  if (ulStyle & SLS_PRIMARYSCALE2)
     fProgressIndicatorData->primaryScale = scale2;
  else
     fProgressIndicatorData->primaryScale = scale1;

  // Set alignment.
  if (ulStyle & (SLS_TOP | SLS_RIGHT))
     fProgressIndicatorData->align = topRight;
  else if (ulStyle & (SLS_BOTTOM | SLS_LEFT))
     fProgressIndicatorData->align = bottomLeft;
  else
     fProgressIndicatorData->align = centered;

  fProgressIndicatorData->isRibbonStrip = (ulStyle & SLS_RIBBONSTRIP) ? true : false;
  fProgressIndicatorData->isSnapToTick = (ulStyle & SLS_SNAPTOINCREMENT) ? true : false;

  // Create the top level XmDrawingArea.
  n = 0;
  XtSetArg( args[n], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[n], XmNmarginWidth, 0 ); n++;
  XtSetArg( args[n], XmNresizePolicy, XmRESIZE_NONE ); n++;

  sliderda = Inherited::create( windowId,
                                0,
                                ulStyle,
                                (IXmCreateFunction)XmCreateDrawingArea,
                                parent,
                                owner,
                                rectInit,
                                args,
                                n,
                                defaultOrdering(),
                                convertToGUIStyle(
                                   IProgressIndicator::Style( 0, extendedStyle() ),
                                   true) );

  // Create the scale widget.
  n = 0;
  if (ulStyle & SLS_VERTICAL)
  {
     // Set buttons position.
     if (ulStyle & SLS_BUTTONSTOP)
        fProgressIndicatorData->buttonsPosition = ISlider::top;
     else if (ulStyle & SLS_BUTTONSBOTTOM)
        fProgressIndicatorData->buttonsPosition = ISlider::bottom;
     else
        fProgressIndicatorData->buttonsPosition = ISlider::none;

     XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
     if (ulStyle & IC_SLS_HOMEBOTTOM)
     {
        XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_TOP); n++;
     }
     else
     {
        XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_BOTTOM); n++;
     }
  }
  else
  {
     // Set buttons position.
     if (ulStyle & SLS_BUTTONSLEFT)
        fProgressIndicatorData->buttonsPosition = ISlider::left;
     else if (ulStyle & SLS_BUTTONSRIGHT)
        fProgressIndicatorData->buttonsPosition = ISlider::right;
     else
        fProgressIndicatorData->buttonsPosition = ISlider::none;

     XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
     if (ulStyle & SLS_HOMERIGHT)
     {
        XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_LEFT); n++;
     }
     else
     {
        XtSetArg (args[n], XmNprocessingDirection, XmMAX_ON_RIGHT); n++;
     }
  }
  XtSetArg (args[n], XmNborderWidth, 0); n++;
  XtSetArg (args[n], XmNhighlightThickness, highlightThickness); n++;
  XtSetArg (args[n], XmNshadowThickness, shadowThickness); n++;
  XtSetArg (args[n], XmNvalue, 0); n++;

  // For an IProgressIndicator with ribbonStrip style, use an XmScrollBar
  // widget.  Set the slider size to a minimum value of 1.
  if ((ulStyle & SLS_READONLY) && fProgressIndicatorData->isRibbonStrip)
  {
     XtSetArg (args[n], XmNsliderSize, 1); n++;
     XtSetArg (args[n], XmNshowArrows, False); n++;
     XtSetArg (args[n], XmNincrement, 1); n++;
     fProgressIndicatorData->slider = XtCreateManagedWidget (IString (windowId),
                                            xmScrollBarWidgetClass,
                                            sliderda,
                                            args, n);
  }
  else
  {
     XtSetArg (args[n], XmNnoResize, True); n++;
     fProgressIndicatorData->slider = XtCreateManagedWidget (IString (windowId),
                                            xmScaleWidgetClass,
                                            sliderda,
                                            args, n);
  }

  // Remove translations for the scroll widget if this is an IProgressIndicator.
  if (ulStyle & SLS_READONLY)
     if (XtIsSubclass (fProgressIndicatorData->slider, xmScaleWidgetClass))
        XtUninstallTranslations (XtNameToWidget (fProgressIndicatorData->slider, "Scrollbar"));
     else
        XtUninstallTranslations (fProgressIndicatorData->slider);

  if (~ulStyle & SLS_READONLY &&
      (ulStyle & (SLS_BUTTONSLEFT | SLS_BUTTONSRIGHT)))
  {
     // Create the arrow buttons.
     n = 0;
     XtSetArg (args[n], XmNfractionBase, 2); n++;
     arrowForm = XtCreateWidget (IString (windowId),
                                 xmFormWidgetClass,
                                 sliderda,
                                 args, n);

     n = 0;
     XtSetArg (args[n], XmNtraversalOn, False); n++;
     XtSetArg (args[n], XmNhighlightThickness, 0); n++;
     XtSetArg (args[n], XmNmultiClick, XmMULTICLICK_KEEP); n++;
     if (ulStyle & SLS_VERTICAL)
     {
        XtSetArg (args[n], XmNarrowDirection, XmARROW_DOWN); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNtopPosition, 1); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNbottomPosition, 2); n++;
     }
     else
     {
        XtSetArg (args[n], XmNarrowDirection, XmARROW_LEFT); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNleftPosition, 0); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNrightPosition, 1); n++;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
     }

     fProgressIndicatorData->arrowLeftDown = XtCreateManagedWidget (IString (windowId),
                                                   xmArrowButtonWidgetClass,
                                                   arrowForm,
                                                   args, n);

     n = 0;
     XtSetArg (args[n], XmNtraversalOn, False); n++;
     XtSetArg (args[n], XmNhighlightThickness, 0); n++;
     XtSetArg (args[n], XmNmultiClick, XmMULTICLICK_KEEP); n++;
     if (ulStyle & SLS_VERTICAL)
     {
        XtSetArg (args[n], XmNarrowDirection, XmARROW_UP); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNtopPosition, 0); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNbottomPosition, 1); n++;
     }
     else
     {
        XtSetArg (args[n], XmNarrowDirection, XmARROW_RIGHT); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNleftPosition, 1); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
        XtSetArg (args[n], XmNrightPosition, 2); n++;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
     }

     fProgressIndicatorData->arrowRightUp = XtCreateManagedWidget (IString (windowId),
                                                  xmArrowButtonWidgetClass,
                                                  arrowForm,
                                                  args, n);
     XtManageChild (arrowForm);
  }

  startHandlingEventsFor( sliderda );

  // Create forms to contain the tick text and marks.
  // Create the same width or height as the scale widget.
  setTicks (tickData->increments[scale1],
            tickData->increments[scale2],
            tickData->spacing[scale1],
            tickData->spacing[scale2]);

  if (ulStyle != IWindow::noStyle.asUnsignedLong())
     setStyle (ulStyle);

  // Register necessary callbacks and event handler.
  fProgressIndicatorData->registerCallbacks();

  // Set initial button sensitivity.
  if (fProgressIndicatorData->arrowLeftDown)
     fProgressIndicatorData->setSensitivity();
#endif //IC_MOTIF
}


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IProgressIndicator::initialize                                               |
------------------------------------------------------------------------------*/
void IProgressIndicator :: initialize ( IWindowHandle trackbar )
{
  IMODTRACE_DEVELOP("IProgressIndicator::initialize");

  IProgressIndicatorData::registerSlider();
  fProgressIndicatorData = new IProgressIndicatorData();

  fProgressIndicatorData->isWrappered = true;
  fProgressIndicatorData->trackbar = trackbar;

  // Get the styles directly from the trackbar.  We need to pass these styles
  // when we construct our composite window.  For some styles, we want to
  // remove them from the trackbar control; they should be set only on the
  // composite window.
  unsigned long trackbarStyle = ISTYLEOF( trackbar );

  unsigned long wrapperStyle = WS_CHILD | WS_CLIPCHILDREN;

  unsigned long trackbarExtStyle = GetWindowLong( trackbar, GWL_EXSTYLE );
  unsigned long wrapperExtStyle = 0;

  if (trackbarStyle & WS_VISIBLE)
     wrapperStyle |= WS_VISIBLE;

  if (trackbarStyle & WS_TABSTOP)
  {
     if (!(trackbarStyle & TBS_NOTHUMB))
        wrapperStyle |= IWindow::tabStop.asUnsignedLong();
     trackbarStyle &= ~WS_TABSTOP;
  }
  if (trackbarStyle & WS_GROUP)
  {
     if (!(trackbarStyle & TBS_NOTHUMB))
        wrapperStyle |= IWindow::group.asUnsignedLong();
     trackbarStyle &= ~WS_GROUP;
  }

  // Look at the trackbar styles to determine the primary scale and orientation.
  // (Do not change the logic below since TBS_HORZ == TBS_BOTTOM == TBS_RIGHT == 0).
  if (trackbarStyle & TBS_VERT)
  {
     wrapperStyle |= vertical.asUnsignedLong();
     if (trackbarStyle & TBS_BOTH)
        trackbarStyle &= ~TBS_BOTH;
     if (!(trackbarStyle & TBS_LEFT))
        wrapperStyle |= primaryScale2.asUnsignedLong();
     else
        wrapperExtStyle |= primaryScale1.asExtendedUnsignedLong();
  }
  else
  {
     wrapperExtStyle |= horizontal.asExtendedUnsignedLong();
     if (trackbarStyle & TBS_BOTH)
        trackbarStyle = (trackbarStyle & ~TBS_BOTH) | TBS_TOP;
     if (trackbarStyle & TBS_TOP)
        wrapperExtStyle |= primaryScale1.asExtendedUnsignedLong();
     else
        wrapperStyle |= primaryScale2.asUnsignedLong();
  }

  // Force the center style since alignment is an Open Class extension to the
  // trackbar.
  wrapperExtStyle |= alignCentered.asExtendedUnsignedLong();

  // Look for the TBS_NOTHUMB style which is equivalent to a read only control.
  // We can assume that if this style is set, we are constructing an
  // IProgressIndcator object.  Wrappering a read only trackbar as an ISlider
  // will not cause any real problems.  Conversely, we assume that a trackbar
  // without the TBS_NOTHUMB style is being wrappered as an ISlider object. 
  // Note that since buttons are also an Open Class extension and the
  // trackbars have no buttons, wrappered trackbars will never have buttons.
  if (trackbarStyle & TBS_NOTHUMB)
  {
     // Progress indicators must have the TBS_ENABLESELRANGE style since this is
     // used to provide a trough for filling.
     trackbarStyle |= TBS_ENABLESELRANGE;
     wrapperExtStyle |= Style( SLS_READONLY ).asExtendedUnsignedLong();
  }

  // Since we determine the home position in the trackbar via the range "min" and
  // "max" and since we don't honor the initial range values, take the default
  // home position.  This can be changed by the application after wrappering by
  // calling setHomePosition.
  if (trackbarStyle & TBS_VERT)
     wrapperExtStyle |= homeBottom.asExtendedUnsignedLong();
  else
     wrapperExtStyle |= homeLeft.asExtendedUnsignedLong();

  // Remove other styles that may cause conflicts.
  trackbarStyle &= ~(TBS_NOTICKS | TBS_FIXEDLENGTH | TBS_AUTOTICKS);

  // Reset the trackbar style with our adjusted style.
  ISETWINDOWSTYLE( trackbar, trackbarStyle );

  // Remove any border styles from the trackbar; we only want them on our
  // WC_NATIVESLIDER window.  Treat the WS_EX_STATICEDGE and WS_BORDER styles
  // like we do WS_EX_CLIENTEDGE since it is our best fit.
  if ((trackbarExtStyle & (WS_EX_CLIENTEDGE | WS_EX_STATICEDGE))
       || (trackbarStyle & WS_BORDER))
  {
     wrapperExtStyle |= border3D.asExtendedUnsignedLong();
     trackbarExtStyle &=
           ~(WS_EX_CLIENTEDGE | WS_EX_STATICEDGE);
     trackbarStyle &= ~WS_BORDER;
  }

  // If the trackbar is disabled, copy the WS_DISABLED style to the wrapper
  // window.
  if (trackbarStyle & WS_DISABLED)
     wrapperStyle |= WS_DISABLED;

  SetWindowLong( trackbar,
                 GWL_EXSTYLE,
                 convertToGUIStyle(
                    Style( 0, trackbarExtStyle ), true ));

  // Save our extended style.
  setExtendedStyle( wrapperExtStyle );

  // Create our composite (WC_NATIVESLIDER) window.  We will use the original
  // rectangle for the trackbar.
  RECT rectInit;
  IQUERYWINDOWRECT( trackbar, &rectInit );

  IWindowHandle hwndParent = IPARENTOF( trackbar );
  IWindowHandle hwndSlider = create( IIDOF( trackbar ),
                                     0,
                                     wrapperStyle,
                                     WC_NATIVESLIDER,
                                     hwndParent,
                                     hwndParent,
                                     IRectangle( rectInit ),
                                     0,
                                     0,
                                     defaultOrdering(),
                                     convertToGUIStyle(
                                        Style( 0, wrapperExtStyle ), true ));

  if (hwndSlider == 0)
  {
     ITHROWGUIERROR( IString( "WinCreateWindow: Id= ") + IString( IIDOF( trackbar ))+
        IString( " Class=" ) + IString( WC_NATIVESLIDER ));
  }

  // We want to put the new wrapper window in the same z-order as the trackbar.
  // Since the wrapper and trackbar windows are siblings, put the wrapper behind the
  // trackbar.  When we reparent the trackbar, we will have the desired z-order.
  SetWindowPos( hwndSlider,
                trackbar,
                0, 0, 0, 0,
                SWP_NOSIZE | SWP_NOMOVE | SWP_NOACTIVATE );


  // Now reparent the trackbar to our WC_NATIVESLIDER window.
  ISETPARENT( trackbar, hwndSlider, false );

  // Although our native slider window has already been created, we need to
  // subclass the trackbar window procedure and perform some initialization.
  fProgressIndicatorData->createNativeSlider( hwndSlider,
                                              hwndSlider,
                                              wrapperStyle,
                                              wrapperExtStyle );

  fProgressIndicatorData->fdefaultHandler.handleEventsFor( this );

  setAutoDestroyWindow( false );
  reserveUserWindowWord( false );
  startHandlingEventsFor( hwndSlider );

  // We assume that the trackbar range, page size, and line size have not been
  // set directly by the application prior to wrappering the control.  It is not
  // feasible to support the various trackbar messages that might have been sent
  // to the trackbar prior to wrappering.  We will set the tick count to 2
  // for the primary scale (which we infer from the TBS_TOP or TBS_BOTTOM
  // styles).  We will also set the tick spacing to 0.  Applications should use
  // the setTicks member function after wrappering the control in order to
  // initialize the scale.
  this->setTicks( this->primaryScale(), 2, 0 );
}
#endif // IC_WIN


/*------------------------------------------------------------------------------
| IProgressIndicator::calcMinimumSize                                          |
|                                                                              |
| Calculate the minimum screen size needed by the control.                     |
------------------------------------------------------------------------------*/
ISize  IProgressIndicator :: calcMinimumSize ( ) const
{
  IMODTRACE_DEVELOP("IProgressIndicator::calcMinimumSize");

  ISize         sizMin;
#ifdef IC_PMWIN
  bool          isVertical = this->isVertical(),
                isVariableSpacing[2];
  unsigned long shaftLength[2];
  int           i;
  IFont         font( this );

  for (i = scale1; i <= scale2; i++)
  {
#ifdef IC_WIN
     if (!isPMCompatible())
        isVariableSpacing[i] = (fProgressIndicatorData->tickSpacing[i] == 0) ?
                                  true : false;
     else
#endif // IC_WIN
        isVariableSpacing[i] = fPMProgressIndicatorData->variableSpacing[i];

     unsigned long spacing,
                   numTicks = numberOfTicks( (Scale)i );

     if (numTicks)
     {
        if (isVariableSpacing[i])
        {
           // Use a scaled default for tick spacing.
           if (numTicks <= 20)
              spacing = 20;
           else if (numTicks <= 40)
              spacing = 10;
           else if (numTicks <= 60)
              spacing = 5;
           else if (numTicks <= 100)
              spacing = 2;
           else
              spacing = 1;
        }
        else
        {
           spacing = tickSpacing( (Scale)i );
        }

        // Now determine the shaft length.
        shaftLength[i] = spacing * (numTicks - 1);
     }
     else
        shaftLength[i] = 0;
  }

  // Use the maximum for the two scales.
  if (isVertical)
     sizMin = ISize( 0, MAX( shaftLength[0], shaftLength[1] ));
  else
     sizMin = ISize( MAX( shaftLength[0], shaftLength[1] ), 0 );

  // If the range hasn't been set on either scale, return the
  // default minimum size.
  if (sizMin == ISize())
     return Inherited::calcMinimumSize();

  // Now add in the fixed portion of the slider length.

  ISlider *slider = dynamic_cast<ISlider*>((IProgressIndicator*)this);

  // Account for the areas beyond the first and last ticks.
#ifdef IC_WIN
  if (!isPMCompatible())
  {
     if (isVertical)
        sizMin += ISize( 0, 2*fProgressIndicatorData->tickPos0 );
     else
        sizMin += ISize( 2*fProgressIndicatorData->tickPos0, 0 );
  }
  else
  {
#endif // IC_WIN
     // For PM compatible sliders, add in the border width.
     if (isVertical)
        sizMin += ISize( 0, 4 );
     else
        sizMin += ISize( 4, 0 );

     // Add arm size.
     if (slider)
     {
        if (isVertical)
           sizMin += ISize( 0, slider->armSize().height() + 4 );
        else
           sizMin += ISize( slider->armSize().width() + 4, 0 );
     }
     else
     {
        if (isVertical)
           sizMin += ISize( 0, 1 );
        else
           sizMin += ISize( 1, 0 );
     }
#ifdef IC_WIN
  }
#endif // IC_WIN

  // Account for the buttons if present.
  long buttonLength = 0;
  ISlider::ButtonsPosition buttonsPosition = ISlider::none;
  if (slider)
     buttonsPosition = slider->buttonsPosition();

  if (buttonsPosition != ISlider::none)
  {
#ifdef IC_WIN
     if (!isPMCompatible())
     {
        buttonLength = 40;
     }
     else
     {
#endif // IC_WIN
        // Add the size of the buttons and button spacing.
        // The CCL control calculates the button length based on the
        // shaft breadth.
        unsigned long shaftBreadth;
        if (isVertical)
           shaftBreadth = shaftSize().width();
        else
           shaftBreadth = shaftSize().height();

        buttonLength = 7;
        if (shaftBreadth > 8)
           buttonLength += 3;
        if (shaftBreadth > 10)
           buttonLength += 2;

        if (shaftBreadth > 19)
           buttonLength += MIN( 18, (2 * ((shaftBreadth - 15) / 5)));

        buttonLength = (2 * buttonLength) + 2;
#ifdef IC_WIN
     }
#endif // IC_WIN
     if (isVertical)
        sizMin += ISize( 0, buttonLength );
     else
        sizMin += ISize( buttonLength, 0 );
  }

  // Now add a little extra space on the ends to account for tick text.
  // For a pmCompatible control, do this only if fixed spacing was
  // specified since the PM control's layout will not use the extra
  // room for the text.
  Scale primary = primaryScale();
#ifdef IC_WIN
  if (!isPMCompatible() || (isPMCompatible() && !isVariableSpacing[primary]))
#else
  if (!isVariableSpacing[primary])
#endif
  {
     IString      firstText = tickText( 0 ),
                  lastText = tickText( numberOfTicks(primary) - 1 );
     ISize        textSize,
                  armSize( 1, 1 );
     long         firstTextPad = 0,
                  lastTextPad = 0,
                  tickOffset = 0;
     HomePosition home = homePosition();

#ifdef IC_WIN
     if (!isPMCompatible())
        tickOffset = fProgressIndicatorData->tickPos0;
     else
#endif // IC_WIN
     {
        tickOffset = 2;
        if (slider)
           if (isVertical)
              tickOffset += slider->armSize().height() / 2;
           else
              tickOffset += slider->armSize().width() / 2;
     }

#ifdef IC_WIN
     IManagedPresSpaceHandle presSpace( (IProgressIndicator*)this );
     IRootGrafPort           port( presSpace );
#endif // IC_WIN

     if (firstText != IString())
     {
#ifdef IC_WIN
        if (!isPMCompatible())
        {
           IKeyGraphicPair textEntry; 
           IGraphicText *gText;
           IGRect2D textRect;
            
           textEntry = fProgressIndicatorData->tickText[primary].value( 0 );
           if (textEntry.fKey == 0)
           {
              gText = (IGraphicText*)textEntry.fObj;
              textRect = gText->looseFitBounds (&port);
              textSize = ISize( textRect.width(),
                                textRect.height() );
           }
        }
        else
        {
#endif // IC_WIN
           textSize = ISize( font.textWidth( firstText ),
                             font.maxCharHeight() );
#ifdef IC_WIN
        }
#endif // IC_WIN

        if (isVertical)
        {
           firstTextPad = MIN( 20, (textSize.height()+1)/2 - tickOffset );

           // Now consider the amount already added for the buttons if they are
           // at this end.  (Are we there yet?!)
           if (home == homeBottomLeft && buttonsPosition == ISlider::bottom ||
               home == homeTopRight && buttonsPosition == ISlider::top)
              firstTextPad =
                 MAX( firstTextPad - buttonLength, 0 );
        }
        else
        {
           firstTextPad = MIN( 50, (textSize.width()+1)/2 - tickOffset );

           if (home == homeBottomLeft && buttonsPosition == ISlider::left ||
               home == homeTopRight && buttonsPosition == ISlider::right)
              firstTextPad =
                 MAX( firstTextPad - buttonLength, 0 );
        }
     }
          
     if (lastText != IString())
     {                    
#ifdef IC_WIN
        if (!isPMCompatible())
        {
           IKeyGraphicPair textEntry; 
           IGraphicText *gText;
           IGRect2D textRect;
            
           textEntry = fProgressIndicatorData->tickText[primary].value(
                             fProgressIndicatorData->tickText[primary].numberOfValues() - 1 );
           if (textEntry.fKey == numberOfTicks( primary ) - 1)
           {
              gText = (IGraphicText*)textEntry.fObj;
              textRect = gText->looseFitBounds (&port);
              textSize = ISize( textRect.width(),
                                textRect.height() );
           }
        }
        else
        {
#endif // IC_WIN
           textSize = ISize( font.textWidth( lastText ),
                             font.maxCharHeight() );
#ifdef IC_WIN
        }
#endif // IC_WIN

        if (isVertical)
        {
           lastTextPad = MIN( 20, (textSize.height()+1)/2 - tickOffset );

           if (home == homeBottomLeft && buttonsPosition == ISlider::top ||
               home == homeTopRight && buttonsPosition == ISlider::bottom)
              lastTextPad =
                 MAX( lastTextPad - buttonLength, 0 );
        }
        else
        {
           lastTextPad = MIN( 50, (textSize.width()+1)/2 - tickOffset );

           if (home == homeBottomLeft && buttonsPosition == ISlider::right ||
               home == homeTopRight && buttonsPosition == ISlider::left)
              lastTextPad =
                 MAX( lastTextPad - buttonLength, 0 );
        }
     }

     // Note that we only calculate the text pad for the primary scale.
     // We can now determine whether adding this pad will increase the overall
     // control's length.
     long pad = 2 * MAX( firstTextPad, lastTextPad );
     if (primary == scale1)
     {
        if (shaftLength[scale1] < shaftLength[scale2])
           pad = MAX( 0, pad + (long)shaftLength[scale1] - (long)shaftLength[scale2] );
     }
     else
     {
        if (shaftLength[scale2] < shaftLength[scale1])
           pad = MAX( 0, pad + (long)shaftLength[scale2] - (long)shaftLength[scale1] );
     }

     if (pad > 0)
     {
        if (isVertical)
           sizMin += ISize( 0, pad );
        else
           sizMin += ISize( pad, 0 );
     }
  }

  // Add a small margin.
  if (isVertical)
     sizMin += ISize( 0, 4 );
  else
     sizMin += ISize( 4, 0 );

  // We have finished the "length" portion of the calculation.  Now for the
  // "breadth".  Check whether we need to calculate the tick and text length
  // portion of the minimum size.
#ifdef IC_WIN
  if (!isPMCompatible())
  {
     if (isVertical)
        sizMin.setWidth( 34 );
     else
        sizMin.setHeight( 34 );
  }
  else
  {
#endif // IC_WIN
     if (isVertical)
        if (slider)
           sizMin.setWidth( slider->armSize().width() );
        else
           sizMin.setWidth( shaftSize().width() );
     else
        if (slider)
           sizMin.setHeight( slider->armSize().height() );
        else
           sizMin.setHeight( shaftSize().height() );
#ifdef IC_WIN
  }
#endif // IC_WIN

  // We need to determine the alignment and add in the longest tick value for
  // the appropriate scales.
  Alignment     align = alignment();
  unsigned long longestTick[2];
#ifdef IC_WIN
  if (isPMCompatible() && fPMProgressIndicatorData->needsTextMinCalc)
#else
  if (fPMProgressIndicatorData->needsTextMinCalc)
#endif
  {
     fPMProgressIndicatorData->calcLongestTick();
     longestTick[0] = fPMProgressIndicatorData->longestTick[scale1];
     longestTick[1] = fPMProgressIndicatorData->longestTick[scale2];
  }
#ifdef IC_WIN
  else if (!isPMCompatible())
  {
     fProgressIndicatorData->calcLongestTick();
     longestTick[0] = fProgressIndicatorData->longestTick[scale1];
     longestTick[1] = fProgressIndicatorData->longestTick[scale2];
  }
#endif // IC_WIN
    
  if (isVertical)
  {
     switch (align)
     {
       case centered:
       {
         sizMin.setWidth(
                   sizMin.width() +
                   2 * (MAX( longestTick[scale1],
                             longestTick[scale2] )));
         break;
       }
     
       case bottomLeft:
       {
         sizMin.setWidth( sizMin.width() +
                          longestTick[scale1] );
         break;
       }

       case topRight:
       {
         sizMin.setWidth( sizMin.width() +
                          longestTick[scale2] );
         break;
       }
     }
  }
  else
  {
     switch (align)
     {
       case centered:
       {
         sizMin.setHeight(
                   sizMin.height() +
                   2 * (MAX( longestTick[scale1],
                             longestTick[scale2] )));
         break;
       }

       case bottomLeft:
       {
         sizMin.setHeight( sizMin.height() +
                           longestTick[scale1] );
         break;
       }

       case topRight:
       {
         sizMin.setHeight( sizMin.height() +
                           longestTick[scale2] );
         break;
       }
     }
  }

#ifdef IC_WIN
  if (!isPMCompatible())
  {
     if ( ( extendedStyle() & border3D.asExtendedUnsignedLong() ) &&
	  ( IPlatform::isWin9x() || IPlatform::isNTNewShell() ))
     {
        sizMin += ISize(4,4);
     }
  }
#endif // IC_WIN
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned long length, spacing, armSize, numTicks;
  Dimension     scaleHeight, scaleWidth, highlightThickness, shadowThickness,
                arrowFormHeight = 0, arrowFormWidth = 0;

  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNhighlightThickness, &highlightThickness,
                 XmNshadowThickness,    &shadowThickness,
                 XmNheight,             &scaleHeight,
                 XmNwidth,              &scaleWidth,
                 NULL);

  if (XtIsSubclass (fProgressIndicatorData->slider, xmScaleWidgetClass))
     armSize = ARM_SIZE;
  else
     armSize = 2;

  spacing = fProgressIndicatorData->tickSpacing[primaryScale()];
  numTicks = numberOfTicks( primaryScale() );
  if (!spacing)
  {
     // Use a scaled default for tick spacing.
     if (numTicks <= 10)
        spacing = 20;
     else if (numTicks <= 20)
        spacing = 10;
     else if (numTicks <= 40)
        spacing = 5;
     else if (numTicks <= 60)
        spacing = 3;
     else if (numTicks <= 100)
        spacing = 2;
     else
        spacing = 1;
  }

  if (spacing < armSize)
     length = spacing * (numTicks - 1) + armSize +
              2 * (highlightThickness + shadowThickness);
  else
     length = spacing * numTicks +
              2 * (highlightThickness + shadowThickness);

  if (fProgressIndicatorData->arrowLeftDown)
  {
     XtVaGetValues (XtParent (fProgressIndicatorData->arrowLeftDown),
                    XmNheight, &arrowFormHeight,
                    XmNwidth,  &arrowFormWidth,
                    NULL);
  }

  Alignment align = alignment();
  if (isVertical())
  {
     if (scaleWidth == 0)
        scaleWidth = 19;
     if (fProgressIndicatorData->arrowLeftDown && arrowFormHeight == 0)
        arrowFormHeight = 2 * scaleWidth;

     switch (align)
     {
       case centered:
         sizMin = ISize( scaleWidth +
                          2 * (MAX( fProgressIndicatorData->longestTick[scale1],
                                    fProgressIndicatorData->longestTick[scale2] )),
                         length + arrowFormHeight );
         break;

       case bottomLeft:
         sizMin = ISize( scaleWidth + fProgressIndicatorData->longestTick[scale1],
                         length + arrowFormHeight );
         break;

       case topRight:
         sizMin = ISize( scaleWidth + fProgressIndicatorData->longestTick[scale2],
                         length + arrowFormHeight );
         break;
     }
  }
  else
  {
     if (scaleHeight == 0)
        scaleHeight = 19;
     if (fProgressIndicatorData->arrowLeftDown && arrowFormWidth == 0)
        arrowFormWidth = 2 * scaleHeight;

     switch (align)
     {
       case centered:
         sizMin = ISize( length + arrowFormWidth,
                         scaleHeight +
                          2 * (MAX( fProgressIndicatorData->longestTick[scale1],
                                    fProgressIndicatorData->longestTick[scale2] )));
         break;

       case bottomLeft:
         sizMin = ISize( length + arrowFormWidth,
                         scaleHeight + fProgressIndicatorData->longestTick[scale1] );
         break;

       case topRight:
         sizMin = ISize( length + arrowFormWidth,
                         scaleHeight + fProgressIndicatorData->longestTick[scale2] );
         break;
     }
  }
#endif //IC_MOTIF

  return sizMin;
}


/*------------------------------------------------------------------------------
| IProgressIndicator::setLayoutDistorted                                       |
|                                                                              |
| Cause a font change to result in a new minimum size.                         |
------------------------------------------------------------------------------*/
IProgressIndicator&
  IProgressIndicator::setLayoutDistorted ( unsigned long layoutAttributesOn,
                                           unsigned long layoutAttributesOff )
{
  unsigned long flagsOn = layoutAttributesOn;
  if (layoutAttributesOn & IWindow::fontChanged)
  {
#ifdef IC_PMWIN
#ifdef IC_WIN
     if (!isPMCompatible())
     {
        // For native sliders, we have to cursor through all of the text
        // strings and change the font.
        
        Scale scale = primaryScale();
        unsigned long numElements
           = fProgressIndicatorData->tickText[scale].numberOfValues();
        if (numElements)
        {
           IFont            sliderFont = font();
           ITextStyleSet    textStyles;
           textStyles.add( ITextBitmapStyle( sliderFont.isBitmap() ));
           textStyles.add( ITextBoldfaceStyle( sliderFont.isBold() ));
           textStyles.add( ITextItalicStyle( sliderFont.isItalic() ));
           textStyles.add( ITextOutlineStyle( sliderFont.isOutline() ));
           textStyles.add( ITextPointSizeStyle( sliderFont.pointSize() ));
           textStyles.add( ITextStrikethroughStyle( sliderFont.isStrikeout() ));
           textStyles.add( ITextTypefaceStyle( sliderFont.name() ));
           textStyles.add( ITextUnderlineStyle( sliderFont.isUnderscore() ));

           for (unsigned long  i=0; i<numElements; i++)
           {
              IKeyGraphicPair
                 dummy = fProgressIndicatorData->tickText[scale].value( i );
              IGraphicText 
                 *gText = (IGraphicText*)dummy.fObj;
              IText
                 text( gText->text() );
              text.addStyles( textStyles );
              gText->setText( text );
           }
           if (isVisible())
              refresh();
        }
        fProgressIndicatorData->needsTextMinCalc = true;
     }
     else
#endif // IC_WIN
        fPMProgressIndicatorData->needsTextMinCalc = true;
#endif // IC_PMWIN
     
     flagsOn |= IWindow::minimumSizeChanged;
  }
  
  Inherited::setLayoutDistorted( flagsOn, layoutAttributesOff );
  return *this;
}


/*------------------------------------------------------------------------------
| IProgressIndicator::backgroundColor                                          |
|                                                                              |
| Returns the background color of the progress indicator.                      |
------------------------------------------------------------------------------*/
IColor IProgressIndicator::backgroundColor () const
{
  IMODTRACE_DEVELOP("IProgressIndicator::backgroundColor");

#ifdef IC_WIN
  if (!isPMCompatible())
  {
    return (IWindow::color(PP_BACKGROUNDCOLOR,
                           IGUIColor(IGUIColor::dialogBgnd)));
  }
  else
  {
     return Inherited::backgroundColor();
  }
#endif //IC_WIN

#ifdef IC_PM

  return (IWindow::color(PP_BACKGROUNDCOLOR,
                         IGUIColor(IGUIColor::defaultControl)));
#endif // IC_PM

#ifdef IC_MOTIF
  Pixel backgroundColor;

  XtVaGetValues (handle(),
                 XmNbackground, &backgroundColor,
                 NULL);
  return IColor (backgroundColor);
#endif //IC_MOTIF
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IProgressIndicator::foregroundColor                                          |
------------------------------------------------------------------------------*/
IColor IProgressIndicator::foregroundColor () const
{
  Pixel foregroundColor;

  XtVaGetValues (handle(),
                 XmNforeground, &foregroundColor,
                 NULL);
  return IColor (foregroundColor);
}

/*------------------------------------------------------------------------------
| IProgessIndicator :: setBackgroundColor                                      |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setBackgroundColor(const IColor& color)
{
  // the inherited setBackgroundColor calls setColor, so the code to set the
  // ticks color is in setColor.
  Inherited::setBackgroundColor (color);
  return *this;
}

/*------------------------------------------------------------------------------
| IProgessIndicator :: setForegroundColor                                      |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setForegroundColor(const IColor& color)
{
  // the inherited setForegroundColor calls setColor, so the code to set the
  // ticks color is in setColor.
  Inherited::setForegroundColor (color);
  return *this;
}

/*------------------------------------------------------------------------------
| IProgessIndicator :: setColor                                               |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setColor( unsigned long colorArea, 
                                          const IColor& color)
{
  // call the inherited setColor so colors can be propagated down to children.
  // not likely for a progress indicator, but still a good idea to call 
  // inherited function
  Inherited::setColor( colorArea, color );
  if (colorArea == PP_BACKGROUNDCOLOR)
  {
    int    scaleNum, tickNum;
    Widget tickWidget, labelWidget;
    long   newColorIndex = color.index();

    for (scaleNum = scale1; scaleNum <= scale2; scaleNum++)
    {
      if (fProgressIndicatorData->tickForm[scaleNum])
        XmChangeColor( fProgressIndicatorData->tickForm[scaleNum], 
                        newColorIndex);

      for ( tickNum = 0;
            tickNum < fProgressIndicatorData->numberTicks[scaleNum]; 
            tickNum++)
      { 
        tickWidget = XtNameToWidget(fProgressIndicatorData->tickForm[scaleNum],
                                     IString (tickNum));
        labelWidget= XtNameToWidget(fProgressIndicatorData->tickForm[scaleNum],
                                     IString (tickNum) + "L");
        XmChangeColor (tickWidget, newColorIndex);
        if (labelWidget != NULL)
          XmChangeColor (labelWidget, newColorIndex);
      }
    }
  }
  else if (colorArea == PP_FOREGROUNDCOLOR)
  {
    int    scaleNum, tickNum;
    Widget tickWidget, labelWidget;
    long   newColorIndex = color.index();
  
  
    for (scaleNum = scale1; scaleNum <= scale2; scaleNum++)
    {
      if (fProgressIndicatorData->tickForm[scaleNum])
        XtVaSetValues (fProgressIndicatorData->tickForm[scaleNum],
                       XmNforeground, newColorIndex,
                       NULL);
  
      for ( tickNum = 0; 
            tickNum < fProgressIndicatorData->numberTicks[scaleNum]; 
            tickNum++)
      {
        tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[scaleNum],
                                     IString (tickNum));
        labelWidget= XtNameToWidget (fProgressIndicatorData->tickForm[scaleNum],
                                      IString (tickNum) + "L");
        XtVaSetValues (tickWidget,
                       XmNforeground, newColorIndex,
                       NULL);
        if (labelWidget != NULL)
        {
          XtVaSetValues (labelWidget,
                         XmNforeground, newColorIndex,
                          NULL);
        }
      }
    }
  }
  return *this;
}
#endif //IC_MOTIF

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IProgressIndicator::getTickText                                              |
|                                                                              |
| Return the text associated with a tick.                                      |
------------------------------------------------------------------------------*/
char* IProgressIndicator :: getTickText ( unsigned long ulTickNum,
                                          char* pszBuffer,
                                          unsigned long ulBufLength ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::getTickText");

   char* pszTemp;

   if(pszBuffer==0 || ulBufLength==0)
   {
      ulBufLength = tickTextLength(ulTickNum)+1;
      pszTemp = new char[ulBufLength];
      pszTemp[0] = 0;
      pszBuffer = pszTemp;
   }

   if (ulBufLength > 1)
   {
      IEventResult evtTickData =
              handle().sendEvent(SLM_QUERYSCALETEXT,
                                 IEventParameter1((unsigned short)ulTickNum,
                                                  (unsigned short)ulBufLength),
                                 IEventParameter2((void*)pszBuffer));

      if (evtTickData.number1() == (USHORT)SLDERR_INVALID_PARAMETERS)
         ITHROWGUIERROR2("SLM_QUERYSCALETEXT",
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);
   }

   return pszBuffer;
}

/*------------------------------------------------------------------------------
| IProgressIndicator::tickTextLength                                           |
|                                                                              |
| Return the length of the text associated with a tick.                        |
------------------------------------------------------------------------------*/
unsigned long IProgressIndicator :: tickTextLength (
                                                unsigned long ulTickNum ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::tickTextLength");

   // There is a problem in both the CCL and PM slider controls such that
   // SLERR_INVALID_PARAMETERS is returned from the SLM_QUERYSCALETEXT
   // message for a tick having zero length and no text.  In order to work
   // around this, we will do the verification here that the tick number is
   // in range and assume a text length of 0 when SLERR_INVALID_PARAMETERS
   // is returned from the SLM_QUERYSCALETEXT.
   IEventResult evtTickData =
           handle().sendEvent(SLM_QUERYSCALETEXT,
                              IEventParameter1((unsigned short)ulTickNum, (unsigned long)0),
                              IEventParameter2((unsigned long)0));

   if (evtTickData.number1() == (USHORT)SLDERR_INVALID_PARAMETERS)
      if (ulTickNum < numberOfTicks( primaryScale() ))
         return 0;
      else
         ITHROWGUIERROR2("SLM_QUERYSCALETEXT",
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);

   return evtTickData.number1();
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IProgressIndicator :: alignment                                              |
------------------------------------------------------------------------------*/
IProgressIndicator::Alignment IProgressIndicator :: alignment ( ) const
{
  IMODTRACE_DEVELOP("IProgressIndicator::alignment");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      unsigned long extendedStyle = this->extendedStyle();
      if (extendedStyle & ISLS_BOTTOM)
         return bottomLeft;
      if (extendedStyle & ISLS_TOP)
         return topRight;
   }
   else
   {
#endif

   unsigned long ulStyle = style();
   if (ulStyle & alignLeft.asUnsignedLong())
      return bottomLeft;
   if (ulStyle & alignRight.asUnsignedLong())
      return topRight;
#ifdef IC_WIN
   }
#endif

   return centered;
#endif //IC_PMWIN

#ifdef IC_MOTIF
   return fProgressIndicatorData->align;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: homePosition                                           |
------------------------------------------------------------------------------*/
IProgressIndicator::HomePosition IProgressIndicator :: homePosition ( ) const
{
  IMODTRACE_DEVELOP("IProgressIndicator::homePosition");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      return( (extendedStyle() & ISLS_HOMETOP) ? homeTopRight
                                               : homeBottomLeft);
   }
   else
#endif
   return( (style() & homeTop.asUnsignedLong() ? homeTopRight
                                               : homeBottomLeft) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned char processDirect;

  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNprocessingDirection, &processDirect,
                 NULL);

  switch (processDirect)
  {
    case XmMAX_ON_TOP:
    case XmMAX_ON_RIGHT:
      return homeBottomLeft;
    default:
      return homeTopRight;
  }
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: setHomePosition                                        |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setHomePosition ( HomePosition home )
{
  IMODTRACE_DEVELOP("IProgressIndicator::setHomePosition");

#ifdef IC_PMWIN
  unsigned long ulStyle;
#ifdef IC_WIN
  if (!isPMCompatible())
  {
     ulStyle = extendedStyle();
     if (homePosition() != home)
     {
        bool wasVisible = this->isVisible();
        if (wasVisible)
           disableUpdate();

        fProgressIndicatorData->trackbar.sendEvent( TBM_CLEARTICS,
                                       IEventParameter1( true ),
                                       IEventParameter2( 0 ));

        unsigned long armOffset = fProgressIndicatorData->armRangeOffset();
        bool isRibbon = this->isRibbonStripEnabled();
        if (isRibbon)
           fProgressIndicatorData->setArmRangeOffset( 0 );

        unsigned long ulRangeMin = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_GETRANGEMIN,
                                       IEventParameter1 ((unsigned long)0),
                                       IEventParameter2 ((unsigned long)0) );

        unsigned long ulRangeMax = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_GETRANGEMAX,
                                       IEventParameter1 ((unsigned long)0),
                                       IEventParameter2 ((unsigned long)0) );

        // Swap the current min and max ranges if the home position has changed.
        fProgressIndicatorData->trackbar.sendEvent( TBM_SETRANGE,
                                   IEventParameter1 (true),
                                   IEventParameter2 (-ulRangeMax, -ulRangeMin) );
        fProgressIndicatorData->scaleFactor *= -1;

        // Reset the position.
        if (home == homeBottomLeft)
           setExtendedStyle( ulStyle & ~ISLS_HOMETOP );
        else
           setExtendedStyle( ulStyle | ISLS_HOMETOP );

        // Tick marks need to be reset relative to new home position.
        unsigned long ulPageSize, ulPosition, ulTickNum;
        ulPageSize = fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_GETPAGESIZE,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ));
        Scale primary = this->primaryScale();
        TickSet::Cursor cursor( fProgressIndicatorData->visibleTicks[primary]);

        forCursor( cursor )
        {
           ulTickNum = fProgressIndicatorData->visibleTicks[primary].elementAt( cursor );
           ulPosition = ulTickNum * ulPageSize * fProgressIndicatorData->scaleFactor;
           fProgressIndicatorData->trackbar.sendEvent(
                                       TBM_SETTIC,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( ulPosition ));
        }

        IKeyGraphicPair   dummy;

        unsigned long     numElements =
          (fProgressIndicatorData->tickText[primary]).numberOfValues ();

        // Tick text needs to be redrawn relative to new home position.
        unsigned long numTicks = numberOfTicks (primary);

        for (unsigned long  i = 0; i < numElements; i ++)
        {
          dummy = (fProgressIndicatorData->tickText[primary]).value (i);
          dummy.fKey = numTicks - dummy.fKey - 1;
        }

        // Force a re-layout.
        fProgressIndicatorData->layoutNativeSlider();
        fProgressIndicatorData->setArmRangeOffset( armOffset );

        if (wasVisible)
        {
           enableUpdate();
           refresh();
        }
     }
  }
  else
  {
#endif
  ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (home == homeBottomLeft)
  {
    ulStyle &= ~(homeTop.asUnsignedLong());
  }
  else
  {
    ulStyle |= homeTop.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();
  }
#ifdef IC_WIN
  }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned char processDirect;
  int           tickNum, scaleNum,
                leftPosition, rightPosition, topPosition, bottomPosition;
  Widget        tickWidget, labelWidget;

  // Verify that home position is being modified.
  if (homePosition() == home)
     return *this;

  switch (home) {
    case homeBottomLeft:
     processDirect = (unsigned char)(isVertical() ?
                                        XmMAX_ON_TOP : XmMAX_ON_RIGHT);
     break;
    case homeTopRight:
     processDirect = (unsigned char)(isVertical() ?
                                        XmMAX_ON_BOTTOM : XmMAX_ON_LEFT);
     break;
  }
  XtVaSetValues (fProgressIndicatorData->slider,
                 XmNprocessingDirection, (int) processDirect,
                 NULL);

  // Now need to reverse order of ticks and labels.
  for (scaleNum = scale1; scaleNum <= scale2; scaleNum++)
     for (tickNum = 0; tickNum < fProgressIndicatorData->numberTicks[scaleNum]; tickNum++)
     {
        tickWidget = XtNameToWidget (fProgressIndicatorData->tickForm[scaleNum],
                                     IString (tickNum));
        labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[scaleNum],
                                     IString (tickNum) + "L");
        if (isVertical())
        {
           XtVaGetValues (tickWidget,
                          XmNtopPosition,    &topPosition,
                          XmNbottomPosition, &bottomPosition,
                          NULL);
           XtVaSetValues (tickWidget,
                          XmNtopPosition,
                             fProgressIndicatorData->numberTicks[scaleNum] - bottomPosition,
                          XmNbottomPosition,
                             fProgressIndicatorData->numberTicks[scaleNum] - topPosition,
                          NULL);
           if (labelWidget != NULL)
           {
              XtVaSetValues (labelWidget,
                             XmNtopPosition,
                                fProgressIndicatorData->numberTicks[scaleNum] - bottomPosition,
                             XmNbottomPosition,
                                fProgressIndicatorData->numberTicks[scaleNum] - topPosition,
                             NULL);
           }
        }
        else
        {
           XtVaGetValues (tickWidget,
                          XmNleftPosition,  &leftPosition,
                          XmNrightPosition, &rightPosition,
                          NULL);
           XtVaSetValues (tickWidget,
                          XmNleftPosition,
                              fProgressIndicatorData->numberTicks[scaleNum] - rightPosition,
                          XmNrightPosition,
                             fProgressIndicatorData->numberTicks[scaleNum] - leftPosition,
                          NULL);
           if (labelWidget != NULL)
           {
              XtVaSetValues (labelWidget,
                             XmNleftPosition,
                                 fProgressIndicatorData->numberTicks[scaleNum] - rightPosition,
                             XmNrightPosition,
                                fProgressIndicatorData->numberTicks[scaleNum] - leftPosition,
                             NULL);
           }
        }
     }
  // Move the slider arm to the home position.
  moveArmToTick (0);

  fProgressIndicatorData->layoutControl();
  if (fProgressIndicatorData->arrowLeftDown)
     fProgressIndicatorData->setSensitivity();
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: primaryScale                                           |
------------------------------------------------------------------------------*/
IProgressIndicator::Scale IProgressIndicator :: primaryScale ( ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::primaryScale");

#ifdef IC_WIN
   if (!isPMCompatible())
   {
      if (!isVertical())
         return( (ISTYLEOF( fProgressIndicatorData->trackbar ) & TBS_TOP ?
                    scale1 : scale2) );
      else
         return( (ISTYLEOF( fProgressIndicatorData->trackbar ) & TBS_LEFT ?
                    scale2 : scale1) );
   }
   else
#endif

#ifdef IC_PMWIN
   return( (style() & primaryScale2.asUnsignedLong()) ? scale2 : scale1 );
#endif //IC_PMWIN

#ifdef IC_MOTIF
   return fProgressIndicatorData->primaryScale;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: setPrimaryScale                                        |
|                                                                              |
| Replaces the primary scale style of the progress indicator.                  |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator :: setPrimaryScale ( Scale primaryScale )
{
  IMODTRACE_DEVELOP("IProgressIndicator::setPrimaryScale");

  unsigned long ulStyle;
  unsigned long ulOldStyle;

#ifdef IC_PMWIN
#ifdef IC_WIN
  if (!isPMCompatible())
  {
     // Since the native trackbar only supports one range and set of tick
     // positions, we need to setup for the new primary scale using cached
     // info.
     ulStyle = ISTYLEOF( fProgressIndicatorData->trackbar );
     ulOldStyle = ulStyle;
     if (!isVertical())
        if (primaryScale == scale1)
           ulStyle |= TBS_TOP;
        else
           ulStyle &= ~TBS_TOP;
     else
        if (primaryScale == scale1)
           ulStyle &= ~TBS_LEFT;
        else
           ulStyle |= TBS_LEFT;

     if (ulStyle != ulOldStyle)
     {
        bool wasVisible = this->isVisible();
        if (wasVisible)
           disableUpdate();
        ISETWINDOWSTYLE( fProgressIndicatorData->trackbar, ulStyle );
        fProgressIndicatorData->setupNativeScale( primaryScale );
        fProgressIndicatorData->layoutNativeSlider();
        if (wasVisible)
           enableUpdate();
     }
  }
else
  {
#endif

  ulStyle = style();
  ulOldStyle = ulStyle;
  if (primaryScale == scale1)
  {
    ulStyle &= ~(primaryScale2.asUnsignedLong());
  }
  else
  {
    ulStyle |= primaryScale2.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    // If the longest tick length needs calculation (for future minimum size
    // determination), do it now.  We can only get the tick text or length for
    // the primary scale.
    if (fPMProgressIndicatorData->needsTextMinCalc)
       fPMProgressIndicatorData->calcLongestTick();

    setStyle(ulStyle);
  }
#ifdef IC_WIN
  }
#endif

  refresh();

  if (ulStyle != ulOldStyle)
    notifyObservers(INotificationEvent(IProgressIndicator::scaleId,
                                       *this, true, (void*)primaryScale));

#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (fProgressIndicatorData->numberTicks[primaryScale] < 2)
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_COUNT,
                         IBaseErrorInfo::accessError,
                         IException::recoverable);

  if (fProgressIndicatorData->primaryScale != primaryScale)
  {
     XtUnmanageChild (fProgressIndicatorData->tickForm[fProgressIndicatorData->primaryScale]);
     XtManageChild (fProgressIndicatorData->tickForm[primaryScale]);
     fProgressIndicatorData->primaryScale = primaryScale;
     fProgressIndicatorData->layoutControl();
     this->notifyObservers(INotificationEvent(
       IProgressIndicator::scaleId, *this,
       true, (void*)primaryScale));
  }

  // Move the slider arm to the home position.
  moveArmToTick (0);
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: isVertical                                             |
------------------------------------------------------------------------------*/
bool IProgressIndicator :: isVertical ( ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::isVertical");

#ifdef IC_WIN
   if (!isPMCompatible())
   {
      return( (ISTYLEOF( fProgressIndicatorData->trackbar ) & TBS_VERT) ?
                     true : false );
   }
   else
#endif

#ifdef IC_PMWIN
   return( (style() & vertical.asUnsignedLong()) ? true : false );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned char orientation;
  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNorientation, &orientation,
                 NULL);
  return (orientation == XmVERTICAL) ? true : false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: isRibbonStripEnabled                                   |
------------------------------------------------------------------------------*/
bool IProgressIndicator :: isRibbonStripEnabled ( ) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::isRibbonStripEnabled");

#ifdef IC_WIN
   if (!isPMCompatible())
      return( (ISTYLEOF( fProgressIndicatorData->trackbar ) &
                     TBS_ENABLESELRANGE) ? true : false );
   else
#endif

#ifdef IC_PMWIN
   return( (style() & ribbonStrip.asUnsignedLong()) ? true : false );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  return fProgressIndicatorData->isRibbonStrip;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: isPMCompatible                                         |
------------------------------------------------------------------------------*/
bool IProgressIndicator :: isPMCompatible ( ) const
{
   ITRACE_MOTIF_NOP();

   IMODTRACE_DEVELOP("IProgressIndicator::isPMCompatible");

#ifdef IC_WIN
   return ((extendedStyle() & pmCompatible.asExtendedUnsignedLong()) ?
             true : false );
#else
   return true;
#endif
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: enableRibbonStrip                                      |
|                                                                              |
| Sets the style to be ribbon-strip or not depending on the                    |
| input flag.                                                                  |
------------------------------------------------------------------------------*/
IProgressIndicator&  IProgressIndicator :: enableRibbonStrip ( bool ribbon)
{
  IMODTRACE_DEVELOP("IProgressIndicator::enableRibbonStrip");

#ifdef IC_PMWIN
  unsigned long ulStyle;      // get the current style
  unsigned long ulOldStyle;   // save it

#ifdef IC_WIN
  if (!isPMCompatible())
  {
     ulStyle = ISTYLEOF( fProgressIndicatorData->trackbar );
     ulOldStyle = ulStyle;

     if (ribbon)
        ulStyle |= TBS_ENABLESELRANGE;
     else
        ulStyle &= ~TBS_ENABLESELRANGE;

     if (ulStyle != ulOldStyle)
     {
        if (ulStyle & WS_VISIBLE)
           disableUpdate();

        unsigned long armOffset =
              fProgressIndicatorData->armRangeOffset();

        if (!ribbon)
           fProgressIndicatorData->trackbar.sendEvent(
                                        TBM_CLEARSEL,
                                        IEventParameter1( false ),
                                        IEventParameter2( 0 ));


        ISETWINDOWSTYLE( fProgressIndicatorData->trackbar, ulStyle );

        // If this is a native trackbar, we also need to set the selection if
        // the ribbon strip was enabled.  If the ribbon strip was disabled, we
        // need to set the slider position.
        fProgressIndicatorData->setArmRangeOffset( armOffset );
        if (ulStyle & WS_VISIBLE)
        {
           enableUpdate();
           refresh();
        }
     }
  }
  else
  {
#endif
  ulStyle = style();      // get the current style
  ulOldStyle = ulStyle;   // save it

  if (ribbon)
  {
    ulStyle |= ribbonStrip.asUnsignedLong();     //add ribbon-strip style
  }
  else
  {
    ulStyle &= ~(ribbonStrip.asUnsignedLong());  //remove ribbon-strip style
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();
  }
#ifdef IC_WIN
  }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (fProgressIndicatorData->isRibbonStrip == ribbon)
     return *this;

  Dimension     height, width;
  Position      posX, posY;
  int           value, maximum, scaleMultiple;
  unsigned char orientation, direction;

  XtTranslations transTable;
  if (ribbon)
  {
     // Determine whether the IProgressIndicator is read-only.
     XtVaGetValues (XtNameToWidget (fProgressIndicatorData->slider, "Scrollbar"),
                    XmNtranslations, &transTable,
                    NULL);

     // To enable ribbon strip style for a read-only IProgressIndicator,
     // replace the scale widget with a scroll bar.
     if (transTable == NULL)
     {
        XtVaGetValues (fProgressIndicatorData->slider,
                       XmNheight,              &height,
                       XmNwidth,               &width,
                       XmNx,                   &posX,
                       XmNy,                   &posY, 
                       XmNvalue,               &value,
                       XmNmaximum,             &maximum,
                       XmNorientation,         &orientation,
                       XmNprocessingDirection, &direction,
                       XmNscaleMultiple,       &scaleMultiple,
                       NULL);
        XtDestroyWidget (fProgressIndicatorData->slider);
        fProgressIndicatorData->slider = XtVaCreateManagedWidget (
                       IString (id()),
                       xmScrollBarWidgetClass,
                       handle(),
                       XmNheight,              height,
                       XmNwidth,               width,
                       XmNx,                   posX,
                       XmNy,                   posY,
                       XmNvalue,               value,
                       XmNmaximum,             maximum + 1,
                       XmNorientation,         orientation,
                       XmNprocessingDirection, direction,
                       XmNincrement,           1,
                       XmNsliderSize,          1,
                       XmNpageIncrement,       scaleMultiple,
                       XmNborderWidth,         0,
                       XmNshowArrows,          False,
                       NULL);

        // Remove translations for the scroll widget.
        XtUninstallTranslations (fProgressIndicatorData->slider);
        
        // Want to get expose events for scrollbar to repaint ribbon.  Attach
        // the event handler to the scrollbar.
        XtAddEventHandler (fProgressIndicatorData->slider,
                           ExposureMask,
                           False,
                           islExposeEventCallback,
                           (XtPointer*)fProgressIndicatorData);
     }
     else
     {
        // Want to get expose events for scrollbar to repaint ribbon.  The
        // event handler has to be attached to the scrollbar child of the
        // scale widget.
        XtAddEventHandler( XtNameToWidget( fProgressIndicatorData->slider, "Scrollbar" ),
                           ExposureMask,
                           False,
                           islExposeEventCallback,
                           (XtPointer*)fProgressIndicatorData);
     }
  }
  else
  {
     // Determine whether the IProgressIndicator is read-only.
     XtVaGetValues (fProgressIndicatorData->slider,
                    XmNtranslations, &transTable,
                    NULL);

     // To disable ribbon strip style for a read-only IProgressIndicator,
     // replace the scroll bar widget with a scale.
     if (transTable == NULL)
     {
        XtVaGetValues (fProgressIndicatorData->slider,
                       XmNheight,              &height,
                       XmNwidth,               &width,
                       XmNx,                   &posX,
                       XmNy,                   &posY, 
                       XmNvalue,               &value,
                       XmNmaximum,             &maximum,
                       XmNorientation,         &orientation,
                       XmNprocessingDirection, &direction,
                       XmNpageIncrement,       &scaleMultiple,
                       NULL);
        XtDestroyWidget (fProgressIndicatorData->slider);
        fProgressIndicatorData->slider = XtVaCreateManagedWidget (
                       IString (id()),
                       xmScaleWidgetClass,
                       handle(),
                       XmNheight,              height,
                       XmNwidth,               width,
                       XmNscaleWidth,          width,
                       XmNx,                   posX,
                       XmNy,                   posY,
                       XmNvalue,               value,
                       XmNmaximum,             maximum - 1,
                       XmNorientation,         orientation,
                       XmNprocessingDirection, direction,
                       XmNscaleMultiple,       scaleMultiple,
                       XmNborderWidth,         0,
                       XmNnoResize,            True,
                       NULL);

        // Remove translations for the scroll widget.
        XtUninstallTranslations (XtNameToWidget (fProgressIndicatorData->slider, "Scrollbar"));
     }
     else
     {
        XtRemoveEventHandler( XtNameToWidget( fProgressIndicatorData->slider, "Scrollbar" ),
                              ExposureMask,
                              False,
                              islExposeEventCallback,
                              (XtPointer*)fProgressIndicatorData );
     }
  }

  fProgressIndicatorData->isRibbonStrip = ribbon;

  if (isVisible())
     if (ribbon)
        fProgressIndicatorData->fillWithRibbon();
     else
        refresh();
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: disableRibbonStrip                                     |
------------------------------------------------------------------------------*/
IProgressIndicator&  IProgressIndicator :: disableRibbonStrip ( )
{
   enableRibbonStrip( false );
   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: isDrawItemEnabled                                      |
------------------------------------------------------------------------------*/
bool IProgressIndicator :: isDrawItemEnabled    ( ) const
{
  IMODTRACE_DEVELOP("IProgressIndicator::isDrawItemEnabled");

#ifdef IC_PMWIN
#ifdef IC_WIN
  if (!isPMCompatible())
     // Native Windows slider doesn't support owner drawing.
     return false;
#endif

   return( (style() & handleDrawItem.asUnsignedLong() ? true : false) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  return false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: enableDrawItem                                         |
------------------------------------------------------------------------------*/
IProgressIndicator&  IProgressIndicator :: enableDrawItem( bool drawItem )
{
  ITRACE_MOTIF_NOP();
  ITRACE_WIN_NOP();

  IMODTRACE_DEVELOP("IProgressIndicator::enableDrawItem");

#ifdef IC_PMWIN
#ifdef IC_WIN
  if (!isPMCompatible())
     // Native Windows slider doesn't support owner drawing.
     return *this;
#endif

  unsigned long ulStyle = style();    // get current style
  unsigned long ulOldStyle = ulStyle;   // save it

  if (drawItem)
  {
     ulStyle |= handleDrawItem.asUnsignedLong();
     // add owner draw style
  }
  else
  {
     ulStyle &= ~(handleDrawItem.asUnsignedLong());
     // remove previous draw style
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();
  }
#endif //IC_PMWIN

  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: disableDrawItem                                        |
------------------------------------------------------------------------------*/
IProgressIndicator&  IProgressIndicator :: disableDrawItem ( )
{
  ITRACE_MOTIF_NOP();
  ITRACE_WIN_NOP();

  IMODTRACE_DEVELOP("IProgressIndicator::disableDrawItem");

#ifdef IC_PMWIN
   enableDrawItem( false );
#endif //IC_PMWIN

   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: isSnapToTickEnabled                                    |
------------------------------------------------------------------------------*/
bool IProgressIndicator :: isSnapToTickEnabled  ( ) const
{
  IMODTRACE_DEVELOP("IProgressIndicator::isSnapToTickEnabled");

#ifdef IC_PMWIN
#ifdef IC_WIN
  if (!isPMCompatible())
     return( extendedStyle() & ISLS_SNAPTOTICK );
#endif

   return( (style() & snapToTickMark.asUnsignedLong() ? true : false) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  return fProgressIndicatorData->isSnapToTick;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: enableSnapToTick                                       |
|                                                                              |
| Sets the style to be snap-to-tick or not depending on the                    |
| input flag.                                                                  |
------------------------------------------------------------------------------*/
IProgressIndicator& IProgressIndicator::enableSnapToTick ( bool snapToTick )
{
  IMODTRACE_DEVELOP("IProgressIndicator::enableSnapToTick");

#ifdef IC_PMWIN
  unsigned long ulStyle;
  unsigned long ulOldStyle;

#ifdef IC_WIN
  if (!isPMCompatible())
  {
     ulStyle = extendedStyle();
     ulOldStyle = ulStyle;
     if (snapToTick)
        ulStyle |= ISLS_SNAPTOTICK;
     else
        ulStyle &= ~ISLS_SNAPTOTICK;
     if (ulStyle != ulOldStyle)
     {
        unsigned long position = armTickOffset();
        setExtendedStyle( ulStyle );
        if (ulStyle & WS_VISIBLE)
           disableUpdate();
        fProgressIndicatorData->setupNativeScale( primaryScale() );
        moveArmToTick( position );
        if (ulStyle & WS_VISIBLE)
        {
           enableUpdate();
           refresh();
        }
     }
  }
else
  {
#endif

  ulStyle = style();        // get the current style
  ulOldStyle = ulStyle;     // save it

  if (snapToTick)
  {
    ulStyle |= snapToTickMark.asUnsignedLong();
                                              //add snap-to-increment style
  }
  else
  {
    ulStyle &= ~(snapToTickMark.asUnsignedLong());
                                              //remove snap-to-increment style
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();
  }
#ifdef IC_WIN
  }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (snapToTick)
  {
     fProgressIndicatorData->isSnapToTick = true;
     moveArmToTick( armTickOffset() );
  }
  else
     fProgressIndicatorData->isSnapToTick = false;

#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: disableSnapToTick                                      |
------------------------------------------------------------------------------*/
IProgressIndicator&  IProgressIndicator :: disableSnapToTick ( )
{
   IMODTRACE_DEVELOP("IProgressIndicator::disableSnapToTick");

   enableSnapToTick( false );
   return *this;
}

/*------------------------------------------------------------------------------
| IProgressIndicator :: tickText                                               |
------------------------------------------------------------------------------*/
IString IProgressIndicator :: tickText(unsigned long ulTickNum) const
{
   IMODTRACE_DEVELOP("IProgressIndicator::tickText");

#ifdef IC_PMWIN
   IString strTickText;
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      if (ulTickNum >= this->numberOfTicks(this->primaryScale()))
         ITHROWLIBRARYERROR( IC_SLDR_GET_CTLDATA,
                             IBaseErrorInfo::accessError,
                             IException::recoverable);
      strTickText = IString();
      Scale primary = primaryScale();
      IKeyGraphicPair  dummy;
      unsigned long    numElements =
        (fProgressIndicatorData->tickText[primary]).numberOfValues ();

      if (numElements != 0)
      {
        unsigned long  stringId;

        if (fProgressIndicatorData->scaleFactor == 1)
          stringId = ulTickNum;
        else
          stringId = numberOfTicks (primary) - ulTickNum - 1;

        for (unsigned long  i = 0; i < numElements; i ++)
        {
          dummy = (fProgressIndicatorData->tickText[primary]).value (i);

          if (dummy.fKey == stringId)
          {
            IGraphicText *  gText = (IGraphicText *) dummy.fObj;
            strTickText = gText->text ();
            break;
          }
          else if (dummy.fKey > stringId)
          {
            break;
          }

        } // for i

      } // if
   }
   else
   {
#endif
   char* pszTickText = getTickText(ulTickNum);

   if (pszTickText)
      strTickText = IString(pszTickText);
   else
      strTickText = IString();

   delete(pszTickText);
#ifdef IC_WIN
   }
#endif

   return strTickText;

#endif //IC_PMWIN

#ifdef IC_MOTIF
  if (ulTickNum >= numberOfTicks (primaryScale()))
     ITHROWLIBRARYERROR (IC_SLDR_INVALID_TICK_NUMBER,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable);

  XmString labelString;
  char     *label=0;
  Widget labelWidget = XtNameToWidget (fProgressIndicatorData->tickForm[primaryScale()],
                                       IString (ulTickNum) + IString ("L"));
  if (labelWidget == NULL)
     return IString();

  XtVaGetValues (labelWidget,
                 XmNlabelString, &labelString,
                 NULL);
  XmStringGetLtoR (labelString, XmFONTLIST_DEFAULT_TAG, &label);
  XmStringFree (labelString);
  // d7950 - Need to free label.
  // return IString (label);
  IString result(label);
  XtFree( label );
  return result;
#endif //IC_MOTIF
}


#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IPMProgressIndicatorData :: IPMProgressIndicatorData                         |
------------------------------------------------------------------------------*/
IPMProgressIndicatorData::IPMProgressIndicatorData(
                                       IProgressIndicator *progressInd )
  : progressInd( progressInd )
  , needsTextMinCalc( false )
  , fDefaultHandler( this )
  , initialArmOffset( 0 )
#ifdef IC_WIN
  , fBrushHandler()
#endif
{
  longestTick[0] = 6;
  longestTick[1] = 6;
  variableSpacing[0] = false;
  variableSpacing[1] = false;
}


/*------------------------------------------------------------------------------
| IPMProgressIndicatorData :: calcLongestTick                                  |
------------------------------------------------------------------------------*/
void IPMProgressIndicatorData::calcLongestTick( )
{
  IProgressIndicator::Scale scale = progressInd->primaryScale();
  unsigned long tickNum,
                tickCount;
  IFont         font( progressInd );
  bool          isVertical = progressInd->isVertical();
  tickCount = progressInd->numberOfTicks( scale );
  longestTick[scale] = 0;
  
  for (tickNum = 0; tickNum < tickCount; tickNum++)
  {
     unsigned long curTickLength = progressInd->tickLength( tickNum );
     IString       tickString( progressInd->tickText( tickNum ));
     if ( tickString != IString() )
     {
        if (isVertical)
           curTickLength += font.textWidth( tickString ) + 2;
        else
           curTickLength += font.maxCharHeight() + 2;
     }
     curTickLength += 4;
     
     if (curTickLength > longestTick[scale])
        longestTick[scale] = curTickLength;
  }

  needsTextMinCalc = false;
}
#endif // IC_PMWIN


/*------------------------------------------------------------------------------
| Public slider styles                                                         |
------------------------------------------------------------------------------*/
const ISlider::Style
  ISlider::buttonsLeft       = SLS_BUTTONSLEFT,
  ISlider::buttonsRight      = SLS_BUTTONSRIGHT,
  ISlider::buttonsBottom     = SLS_BUTTONSBOTTOM,
  ISlider::buttonsTop        = SLS_BUTTONSTOP;
#ifdef IC_PMWIN
const ISlider::Style
  ISlider::classDefaultStyle ( (WS_VISIBLE |
                                SLS_BUTTONSLEFT),
                               (ISLS_HORIZONTAL |
                                ISLS_CENTER     |
                                ISLS_HOMELEFT   |
                                ISLS_PRIMARYSCALE1 |
                                IWS_BORDER3D ) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
const ISlider::Style
  ISlider::classDefaultStyle =
                              ( WS_VISIBLE           |
                                IC_SLS_HORIZONTAL    |
                                IC_SLS_CENTER        |
                                IC_SLS_HOMELEFT      |
                                IC_SLS_PRIMARYSCALE1 |
                                SLS_BUTTONSLEFT );
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| Default style for new slider objects (initial value)                         |
------------------------------------------------------------------------------*/
#ifdef IC_PMWIN
ISlider::Style
  ISlider::currentDefaultStyle ( (WS_VISIBLE |
                                  SLS_BUTTONSLEFT),
                                 (ISLS_HORIZONTAL |
                                  ISLS_CENTER     |
                                  ISLS_HOMELEFT   |
                                  ISLS_PRIMARYSCALE1 |
                                  IWS_BORDER3D ) );
#endif //IC_PMWIN

#ifdef IC_MOTIF
ISlider::Style ISlider::currentDefaultStyle =
   ISlider::classDefaultStyle;
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| ISlider::ISlider                                                             |
|                                                                              |
| Constructor to create a scale control on a standard window.                  |
------------------------------------------------------------------------------*/
ISlider :: ISlider( unsigned long     windowId,
                    IWindow*          parent,
                    IWindow*          owner,
                    const IRectangle& rectInit,
                    unsigned long     numberOfTicks,
                    unsigned long     tickSpacing,
                    const Style&      style )
                 : IProgressIndicator()
{
   IMODTRACE_DEVELOP("ISlider::ctor1");
   IASSERTPARM(parent!=0)

#ifdef IC_PMWIN
  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

#ifdef IC_WIN
   // If this is a native Windows slider, we have to save the snap to tick
   // style as an extended style.  This style is implemented with the native
   // trackbar by setting the max range to SHRT_MAX and interpreting tick
   // positions accordingly.  We also have to save the alignment, home
   // position, and button position as extended styles.
   if (!isPMCompatible())
   {
      unsigned long nativeExtended = 0;

      nativeExtended |= (style & snapToTickMark) ? ISLS_SNAPTOTICK : 0;

      nativeExtended |= (style & alignTop || style & alignRight)
                        ? ISLS_TOP
                      : ((style & alignBottom || style & alignLeft)
                        ? ISLS_BOTTOM : 0);

      nativeExtended |= (style & homeTop || style & homeRight)
                        ? ISLS_HOMETOP : 0;

      // If both button styles are on, then invalid so don't set native flags
      if (!((style & buttonsRight) && (style & buttonsLeft)))
      {
        nativeExtended |= (style & buttonsBottom || style & buttonsLeft)
                          ? ISLS_BUTTONSBOTTOM
                        : ((style & buttonsTop || style & buttonsRight)
                          ? ISLS_BUTTONSTOP : 0);
      }
      setExtendedStyle( extendedStyle() | nativeExtended );
   }
#endif

   SLDCDATA sldcData;                  //Slider control data
   sldcData.cbSize = sizeof(SLDCDATA);
   if (style & primaryScale2)
   {
      sldcData.usScale2Increments = (unsigned short)numberOfTicks;
      sldcData.usScale2Spacing = (unsigned short)tickSpacing;
      sldcData.usScale1Increments = (unsigned short)0;
      sldcData.usScale1Spacing = (unsigned short)0;
   }
   else
   {
      sldcData.usScale1Increments = (unsigned short)numberOfTicks;
      sldcData.usScale1Spacing = (unsigned short)tickSpacing;
      sldcData.usScale2Increments = (unsigned short)0;
      sldcData.usScale2Spacing = (unsigned short)0;
   }
   IWindowHandle::Value hwndOwner = 0;
   if (owner != 0)
      hwndOwner = owner->handle();

   initialize(windowId, parent->handleForChildCreation(), hwndOwner,
              convertToGUIStyle( style ), rectInit, (void*)&sldcData);
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned long ulStyle = style.asUnsignedLong();
  IWindowHandle ownerHandle = 0;
  if (owner != 0)
     ownerHandle = owner->handle();

  sldcdata sliderData;
  sliderData.increments[scale1] = numberOfTicks;
  sliderData.spacing[scale1] = tickSpacing;
  sliderData.increments[scale2] = 0;
  sliderData.spacing[scale2] = 0;

  // Create a private data object.
  fSliderData = new ISliderData();

  // Set buttons position.
  if (ulStyle & SLS_BUTTONSLEFT)
     fSliderData->buttonsPosition = ISlider::left;
  else if (ulStyle & SLS_BUTTONSRIGHT)
     fSliderData->buttonsPosition = ISlider::right;
  else if (ulStyle & SLS_BUTTONSTOP)
     fSliderData->buttonsPosition = ISlider::top;
  else if (ulStyle & SLS_BUTTONSBOTTOM)
     fSliderData->buttonsPosition = ISlider::bottom;
  else
     fSliderData->buttonsPosition = ISlider::none;

  initialize(windowId,
             parent->handleForChildCreation(),
             ownerHandle,
             ulStyle,
             rectInit,
             (void*)&sliderData);
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| ISlider::ISlider                                                             |
|                                                                              |
| Constructor to create a scale control on a standard window.                  |
------------------------------------------------------------------------------*/
ISlider :: ISlider ( unsigned long         windowId,
                     IWindow*              parent,
                     IWindow*              owner,
                     const IRectangle&     rectInit,
                     unsigned long         scale1NumberOfTicks,
                     unsigned long         scale1TickSpacing,
                     unsigned long         scale2NumberOfTicks,
                     unsigned long         scale2TickSpacing,
                     const Style&          style )
                 : IProgressIndicator()
{
   IMODTRACE_DEVELOP("ISlider::ctor");
   IASSERTPARM(parent!=0)

   // Save the extended style to make sure we have a copy of it stored
   setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

#ifdef IC_PMWIN
#ifdef IC_WIN
   // If this is a native Windows slider, we have to save the snap to tick
   // style as an extended style.  This style is implemented with the native
   // trackbar by setting the max range to SHRT_MAX and interpreting tick
   // positions accordingly.  We also have to save the alignment, home
   // position, and buttons position as extended styles.
   if (!isPMCompatible())
   {
      unsigned long nativeExtended = 0;

      nativeExtended |= (style & snapToTickMark) ? ISLS_SNAPTOTICK : 0;

      nativeExtended |= (style & alignTop || style & alignRight)
                        ? ISLS_TOP
                      : ((style & alignBottom || style & alignLeft)
                        ? ISLS_BOTTOM : 0);

      nativeExtended |= (style & homeTop || style & homeRight)
                        ? ISLS_HOMETOP : 0;

      // If both button styles are on, then invalid so don't set native flags
      if (!((style & buttonsRight) && (style & buttonsLeft)))
      {
        nativeExtended |= (style & buttonsBottom || style & buttonsLeft)
                          ? ISLS_BUTTONSBOTTOM
                        : ((style & buttonsTop || style & buttonsRight)
                          ? ISLS_BUTTONSTOP : 0);
      }
      setExtendedStyle( extendedStyle() | nativeExtended );
   }
#endif

   SLDCDATA sldcData;                  //Slider control data
   sldcData.cbSize = sizeof(SLDCDATA);
   sldcData.usScale1Increments = (unsigned short)scale1NumberOfTicks;
   sldcData.usScale2Increments = (unsigned short)scale2NumberOfTicks;
   sldcData.usScale1Spacing = (unsigned short)scale1TickSpacing;
   sldcData.usScale2Spacing = (unsigned short)scale2TickSpacing;

   IWindowHandle hwndOwner = 0;
   if (owner != 0)
#ifdef IC_WIN
      hwndOwner = owner->handle();

   initialize(windowId, parent->handleForChildCreation(), hwndOwner,
              convertToGUIStyle( style ), rectInit, (void*)&sldcData);
#else
      hwndOwner = owner->handle();

   initialize(windowId, parent->handleForChildCreation(), hwndOwner,
              convertToGUIStyle( style ), rectInit, (void*)&sldcData);
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned long ulStyle = style.asUnsignedLong();
  IWindowHandle ownerHandle = 0;
  if (owner != 0)
     ownerHandle = owner->handle();

  sldcdata sliderData;
  sliderData.increments[scale1] = scale1NumberOfTicks;
  sliderData.spacing[scale1] = scale1TickSpacing;
  sliderData.increments[scale2] = scale2NumberOfTicks;
  sliderData.spacing[scale2] = scale2TickSpacing;

  // Create a private data object.
  fSliderData = new ISliderData();

  // Set buttons position.
  if (ulStyle & IC_SLS_HORIZONTAL)
  {
     if (ulStyle & SLS_BUTTONSLEFT)
        fSliderData->buttonsPosition = ISlider::left;
     else if (ulStyle & SLS_BUTTONSRIGHT)
        fSliderData->buttonsPosition = ISlider::right;
     else
        fSliderData->buttonsPosition = ISlider::none;
  }
  else
  {
     if (ulStyle & SLS_BUTTONSTOP)
        fSliderData->buttonsPosition = ISlider::top;
     else if (ulStyle & SLS_BUTTONSBOTTOM)
        fSliderData->buttonsPosition = ISlider::bottom;
     else
        fSliderData->buttonsPosition = ISlider::none;
  }

  initialize(windowId,
             parent->handleForChildCreation(),
             ownerHandle,
             ulStyle,
             rectInit,
             (void*)&sliderData);
#endif //IC_MOTIF
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ISlider :: ISlider                                                           |
------------------------------------------------------------------------------*/
ISlider ::  ISlider( unsigned long ulId,
                      IWindow* pwndParent)
               : IProgressIndicator(ulId, pwndParent)
{ }

/*------------------------------------------------------------------------------
| ISlider :: ISlider                                                           |
------------------------------------------------------------------------------*/
ISlider :: ISlider( const IWindowHandle& wh )
               : IProgressIndicator(wh)
{ }
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| ISlider::~ISlider                                                            |
------------------------------------------------------------------------------*/
ISlider :: ~ISlider ( )
{
#ifdef IC_MOTIF
  delete fSliderData;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| ISlider::defaultStyle                                                        |
|                                                                              |
| Return the default style for new slider objects.                             |
------------------------------------------------------------------------------*/
ISlider::Style  ISlider :: defaultStyle ( )
{
   return currentDefaultStyle;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ISlider::setDefaultStyle                                                     |
|                                                                              |
| Sets the default style for new slider objects.                               |
------------------------------------------------------------------------------*/
void ISlider::setDefaultStyle( const ISlider::Style& style )
{
   currentDefaultStyle = style;
}
#endif

/*------------------------------------------------------------------------------
| ISlider::convertToGUIStyle                                                   |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long ISlider::convertToGUIStyle(const IBitFlag& guiStyle,
                                         bool bExtOnly) const
{
  IMODTRACE_DEVELOP("ISlider::convertToGUIStyle");

  // Obtain the style from the class (IProgressIndicator) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
#ifdef IC_WIN
    // Native trackbars don't support buttons, so no styles for them here.
    if (isPMCompatible())
    {
#endif
    // Note that since this class only deals with button related styles,
    // we only mask those out here.
    ulStyle |= guiStyle.asUnsignedLong() & ISLS_BUTTONMASK;
#ifdef IC_WIN
    }
#endif
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| ISlider::buttonsPosition                                                     |
|                                                                              |
| Returns the position of the slider's buttons.                                |
------------------------------------------------------------------------------*/
ISlider::ButtonsPosition ISlider :: buttonsPosition ( ) const
{
   IMODTRACE_DEVELOP("ISlider::buttonsPosition");

#ifdef IC_PMWIN
   unsigned long ulStyle;

#ifdef IC_WIN
   if (!isPMCompatible())
   {
     ulStyle = extendedStyle();
     if (ulStyle & ISLS_BUTTONSTOP)
        if (isVertical())
           return top;
        else
           return right;
     else if (ulStyle & ISLS_BUTTONSBOTTOM)
        if (isVertical())
           return bottom;
        else
           return left;
   }
   else
   {
#endif

   ulStyle = style();
   if (ulStyle & buttonsTop.asUnsignedLong())
   {
      if (ulStyle & vertical.asUnsignedLong())
         return top;
      else
         return right;
   }
   else if (ulStyle & buttonsBottom.asUnsignedLong())
   {
      if (ulStyle & vertical.asUnsignedLong())
         return bottom;
      else
         return left;
   }
#ifdef IC_WIN
   }
#endif

   return none;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  return fSliderData->buttonsPosition;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| ISlider::detentPosition                                                      |
|                                                                              |
| Returns the offset of a detent from the home position, in pixels.            |
------------------------------------------------------------------------------*/
unsigned long ISlider :: detentPosition ( unsigned long ulDetentId ) const
{
   ITRACE_MOTIF_NOP();
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("ISlider::detentPosition");

#ifdef IC_MOTIF
  return 0;
#endif //IC_MOTIF

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
      return 0;
#endif

   IEventResult evtDetentData =
           handle().sendEvent(SLM_QUERYDETENTPOS,
                              IEventParameter1(ulDetentId),
                              IEventParameter2((unsigned long)0));

   if ((short)(evtDetentData.number1()) == SLDERR_INVALID_PARAMETERS)
      ITHROWGUIERROR2("SLM_QUERYDETENTPOS",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);

   return evtDetentData.number1();
#endif
}

/*------------------------------------------------------------------------------
| ISlider::addDetent                                                           |
------------------------------------------------------------------------------*/
unsigned long ISlider :: addDetent (unsigned long ulOffset )
{
   ITRACE_MOTIF_NOP();
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("ISlider::addDetent");

#ifdef IC_MOTIF
  return 0;
#endif //IC_MOTIF

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
      return 0;
#endif

   unsigned long ulDetentID =
           handle().sendEvent(SLM_ADDDETENT,
                              IEventParameter1(ulOffset),
                              IEventParameter2((unsigned long)0));
   if (!ulDetentID)
   {
      ITHROWGUIERROR2("SLM_ADDDETENT",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
   }

   return ulDetentID;
#endif
}

/*------------------------------------------------------------------------------
| ISlider::removeDetent                                                        |
------------------------------------------------------------------------------*/
ISlider& ISlider :: removeDetent ( unsigned long ulDetentId )
{
   ITRACE_MOTIF_NOP();
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("ISlider::removeDetent");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
      return *this;
#endif

   unsigned long ulResult =
           handle().sendEvent(SLM_REMOVEDETENT,
                              IEventParameter1(ulDetentId),
                              IEventParameter2((unsigned long)0));
   if (!ulResult)
      ITHROWGUIERROR2("SLM_REMOVEDETENT",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
#endif //IC_PMWIN

   return *this;
}

/*------------------------------------------------------------------------------
| ISlider::armSize                                                             |
------------------------------------------------------------------------------*/
ISize ISlider :: armSize ( ) const
{
   IMODTRACE_DEVELOP("ISlider::armSize");

#ifdef IC_PMWIN
   ISize sz;
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      RECT armRect;
      handle().sendEvent( TBM_GETTHUMBRECT,
                          IEventParameter1((unsigned long)0),
                          IEventParameter2(&armRect) );
      sz = IRectangle( armRect ).size();
   }
   else
   {
#endif

   IEventResult evtShaftData =
      handle().sendEvent(SLM_QUERYSLIDERINFO,
                      IEventParameter1((unsigned long)SMA_SLIDERARMDIMENSIONS),
                      IEventParameter2((unsigned long)0));

   if (style() & vertical.asUnsignedLong())
   {
      sz.setWidth((unsigned short)evtShaftData.number2());
      sz.setHeight((unsigned short)evtShaftData.number1());
   }
   else
   {
      sz.setWidth((unsigned short)evtShaftData.number1());
      sz.setHeight((unsigned short)evtShaftData.number2());
   }
#ifdef IC_WIN
   }
#endif

   return sz;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Dimension     scaleHeight, scaleWidth, shadowThickness, highlightThickness;
  unsigned long armBreadth = ARM_SIZE;
  Widget        slider = NULL, *children;
  Cardinal      numChildren;

  // Get the scale widget.
  XtVaGetValues (handle(),
                 XmNnumChildren, &numChildren,
                 XmNchildren,    &children,
                 NULL);
  for (int i = 0; slider == NULL; i++)
  {
     // d8175 - check for child being destroyed.
     // if (XtIsSubclass (children[i], xmScaleWidgetClass))
     if ( XtIsSubclass (children[i], xmScaleWidgetClass)
       && !children[i]->core.being_destroyed )
        slider = children[i];
  }

  XtVaGetValues (slider,
                 XmNheight,             &scaleHeight,
                 XmNwidth,              &scaleWidth,
                 XmNshadowThickness,    &shadowThickness,
                 XmNhighlightThickness, &highlightThickness,
                 NULL);
  if (isVertical())
     return ISize (scaleWidth - 2 * (shadowThickness + highlightThickness),
                   armBreadth);
  else
     return ISize (armBreadth,
                   scaleHeight - 2 * (shadowThickness + highlightThickness));
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| ISlider::setArmSize                                                          |
------------------------------------------------------------------------------*/
ISlider& ISlider :: setArmSize(const ISize& sizSliderArm)
{
   ITRACE_MOTIF_NOP();
   ITRACE_WIN_NOP();

   IMODTRACE_DEVELOP("ISlider::setArmSize");

#ifdef IC_PMWIN
#ifdef IC_WIN
   if (!isPMCompatible())
   {
      // We can only set the thumb length for the native trackbar.
   }
   else
   {
#endif

   unsigned short us1;
   unsigned short us2;
   if (style() & vertical.asUnsignedLong())
   {
      us1 = (unsigned short)sizSliderArm.height();
      us2 = (unsigned short)sizSliderArm.width();
   }
   else
   {
      us1 = (unsigned short)sizSliderArm.width();
      us2 = (unsigned short)sizSliderArm.height();
   }

   unsigned long ulResult =
      handle().sendEvent(SLM_SETSLIDERINFO,
                       IEventParameter1((unsigned long)SMA_SLIDERARMDIMENSIONS),
                       IEventParameter2(us1, us2));
   if (!ulResult)
      ITHROWGUIERROR2("SLM_SETSLIDERINFO",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
#ifdef IC_WIN
   }
#endif
#endif //IC_PMWIN

   return *this;
}

#ifdef IC_MOTIFWIN
/*------------------------------------------------------------------------------
| ISlider::hasFocus                                                            |
|                                                                              |
| Is the window the focus window?                                              |
------------------------------------------------------------------------------*/
bool ISlider::hasFocus() const
{
#ifdef IC_WIN
   if (isPMCompatible())
      return Inherited::hasFocus();
   else
   {
      HWND hwndFocus = IQUERYFOCUS(HWND_DESKTOP);
      if (handle() == IPARENTOF( hwndFocus ))
      {
         IWindowClassName wcname( hwndFocus );
         if (wcname == TRACKBAR_CLASS)
            return true;
      }
      return false;
   }
#endif // IC_WIN

#ifdef IC_MOTIF
  // Get the current focus widget and see if it is our handle widget or one
  // of the widgets that is part of this composite.
  IWindowHandle handleWidget = handle();
  Widget focusWidget = XmGetFocusWidget( handleWidget );
  if (!focusWidget)
     return false;
  if (focusWidget == handleWidget)
     return true;
  if (XtParent( focusWidget ) == handleWidget &&
      XtIsSubclass( focusWidget, xmScaleWidgetClass ))
     return true;
  if (XtParent( XtParent( focusWidget )) == handleWidget &&
      (XtIsSubclass( focusWidget, xmArrowButtonWidgetClass ) ||
       XtIsSubclass( focusWidget, xmScrollBarWidgetClass )))
     return true;
  
  return false;
#endif // IC_MOTIF
}
#endif

#ifdef IC_MOTIF
#pragma info(none)
/*------------------------------------------------------------------------------
|  islMoveSlider is the procedure invoked when the slider timer elapses.  It   |
|  will move the slider arm and start a new timer.                             |
------------------------------------------------------------------------------*/
extern void _System islMoveSlider (XtPointer     client_data,
                                   XtIntervalId* id)
{
  int                     scaleMultiple, scaleMax;
  Widget                  slider = NULL, *children;
  Cardinal                numChildren;

  // Get the private data from the client data.
  IProgressIndicatorData* fProgressIndicatorData =
        (IProgressIndicatorData*)client_data;
  IProgressIndicator* pControl = fProgressIndicatorData->pControl;

  // Get the scale widget.
  XtVaGetValues (pControl->handle(),
                 XmNnumChildren, &numChildren,
                 XmNchildren,    &children,
                 NULL);
  for (int i = 0; slider == NULL; i++)
  {
     if ( XtIsSubclass (children[i], xmScaleWidgetClass)
       && !children[i]->core.being_destroyed )
        slider = children[i];
  }
  if (!slider)
     return;

  // Move the slider.
  XtVaGetValues (slider,
                 XmNscaleMultiple,       &scaleMultiple,
                 XmNmaximum,             &scaleMax,
                 NULL);
  if (fProgressIndicatorData->activeButton == fProgressIndicatorData->arrowLeftDown)
  {
     switch (pControl->homePosition()) {
       case IProgressIndicator::homeBottomLeft:
        pControl->moveArmToPixel (MAX (0,
                         (long) (pControl->armPixelOffset() - scaleMultiple)));
        break;
       case IProgressIndicator::homeTopRight:
        pControl->moveArmToPixel (MIN (scaleMax,
                         pControl->armPixelOffset() + scaleMultiple));
        break;
     }
  }
  else
  {
     switch (pControl->homePosition()) {
       case IProgressIndicator::homeBottomLeft:
        pControl->moveArmToPixel (MIN (scaleMax,
                         pControl->armPixelOffset() + scaleMultiple));
        break;
       case IProgressIndicator::homeTopRight:
        pControl->moveArmToPixel (MAX (0,
                         (long) (pControl->armPixelOffset() - scaleMultiple)));
        break;
     }
  }

  if (pControl->armPixelOffset() != 0 &&
      pControl->armPixelOffset() != scaleMax)
  {
     fProgressIndicatorData->timer = XtAppAddTimeOut
                       (XtWidgetToApplicationContext (fProgressIndicatorData->arrowLeftDown),
                        250,
                        (XtTimerCallbackProc)islMoveSlider,
                        client_data);
  }

  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  islArmCallback is the procedure invoked when one of the arrows is armed.    |
------------------------------------------------------------------------------*/
extern void _System islArmCallback (Widget w,
                                    XtPointer client_data,
                                    XtPointer call_data)
{
  int scaleMultiple, scaleMax;

  // Get the private data from the client data.
  IProgressIndicatorData* fProgressIndicatorData =
        (IProgressIndicatorData*)client_data;
  IProgressIndicator* pControl = fProgressIndicatorData->pControl;

  fProgressIndicatorData->activeButton = w;

  // Set focus to the slider if it does not already have it.
  if (XmGetFocusWidget (w) != fProgressIndicatorData->slider)
     XmProcessTraversal( fProgressIndicatorData->slider,
                         XmTRAVERSE_CURRENT );

  // Grab input to insure that we get the disarm event.  (Prevents callback
  // from doing a grab or otherwise changing input focus to another widget.)
  XtAddGrab (w, FALSE, FALSE);

  // Move the slider.
  XtVaGetValues (fProgressIndicatorData->slider,
                 XmNscaleMultiple, &scaleMultiple,
                 XmNmaximum,       &scaleMax,
                 NULL);
  if (w == fProgressIndicatorData->arrowLeftDown)
  {
     switch (pControl->homePosition()) {
       case IProgressIndicator::homeBottomLeft:
        pControl->moveArmToPixel (MAX (0,
                         (long) (pControl->armPixelOffset() - scaleMultiple)));
        break;
       case IProgressIndicator::homeTopRight:
        pControl->moveArmToPixel (MIN (scaleMax,
                         pControl->armPixelOffset() + scaleMultiple));
        break;
     }
  }
  else
  {
     switch (pControl->homePosition()) {
       case IProgressIndicator::homeBottomLeft:
        pControl->moveArmToPixel (MIN (scaleMax,
                         pControl->armPixelOffset() + scaleMultiple));
        break;
       case IProgressIndicator::homeTopRight:
        pControl->moveArmToPixel (MAX (0,
                         (long) (pControl->armPixelOffset() - scaleMultiple)));
        break;
     }
  }

  if (pControl->armPixelOffset() != 0 &&
      pControl->armPixelOffset() != scaleMax)
  {
     fProgressIndicatorData->timer = XtAppAddTimeOut (XtWidgetToApplicationContext (w),
                                   250,
                                   (XtTimerCallbackProc)islMoveSlider,
                                   client_data);
  }

  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  isbDisarmCallback is the callback invoked when the user releases one of the |
|  arrow buttons.                                                              |
------------------------------------------------------------------------------*/
extern void _System islDisarmCallback (Widget w,
                                       XtPointer client_data,
                                       XtPointer call_data)
{
  // Get the private data from the client data.
  IProgressIndicatorData* fProgressIndicatorData =
        (IProgressIndicatorData*)client_data;

  // Cancel the spin timer.
  if (fProgressIndicatorData->timer)
  {
     XtRemoveTimeOut (fProgressIndicatorData->timer);
     fProgressIndicatorData->timer = 0;
  }

  fProgressIndicatorData->activeButton = 0;

  // Remove the input grab.
  XtRemoveGrab (w);

  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  islValueChangedCallback is the procedure invoked when the slider arm is     |
|  moved.                                                                      |
------------------------------------------------------------------------------*/
extern void _System islValueChangedCallback (Widget w,
                                             XtPointer client_data,
                                             XtPointer call_data)
{
  // Get the private data from the client data.
  IProgressIndicatorData* fProgressIndicatorData =
        (IProgressIndicatorData*)client_data;
  IProgressIndicator* pControl = fProgressIndicatorData->pControl;

  if (pControl->isSnapToTickEnabled())
     // Get the current location.  Move slider arm to nearest tick mark.
     pControl->moveArmToTick (pControl->armTickOffset());
  else
     // moveArmToTick will handle ribbon strip is slider is snapToTick;
     // need to fill now if it is not
     if (pControl->isRibbonStripEnabled())
        fProgressIndicatorData->fillWithRibbon();

  if (fProgressIndicatorData->arrowLeftDown)
  {
     // If arm is at either end of slider, remove appropriate button
     // sensitivity.  Restore sensitivity as necessary.
     fProgressIndicatorData->setSensitivity();
  }

  // Generate SLN_CHANGE event.
  pControl->postEvent( IWindow::control,
                       IEventData( (unsigned short)(pControl->id()), SLN_CHANGE ),
                       IEventData( pControl->armPixelOffset() ));

  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  islDragCallback is the procedure invoked when the slider arm is dragged.    |
------------------------------------------------------------------------------*/
extern void _System islDragCallback (Widget    w,
                                     XtPointer client_data,
                                     XtPointer call_data)
{
  // Get the private data from the client data.
  IProgressIndicatorData* fProgressIndicatorData =
        (IProgressIndicatorData*)client_data;
  IProgressIndicator* pControl = fProgressIndicatorData->pControl;

  if (pControl->isRibbonStripEnabled())
  {
     fProgressIndicatorData->fillWithRibbon();

     if (fProgressIndicatorData->arrowLeftDown)
     {
        // If arm is at either end of slider, remove appropriate button
        // sensitivity.  Restore sensitivity as necessary.
        fProgressIndicatorData->setSensitivity();
     }
  }

  // We also need to send an SLN_SLIDERTRACK notification message to enable
  // the armTrackId notification and the ISliderArmHandler::moving member
  // function.
  pControl->postEvent( IWindow::control,
                       IEventData( (unsigned short)(pControl->id()), SLN_SLIDERTRACK ),
                       IEventData( pControl->armPixelOffset() ));

  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  islResizeCallback is the procedure invoked when the slider is               |
|  resized.  It is used to trigger a re-layout.                                |
------------------------------------------------------------------------------*/
extern void _System islResizeCallback (Widget    w,
                                       XtPointer client_data,
                                       XtPointer call_data)
{
  // Get the private data pointer from client data.
  IProgressIndicatorData* fProgressIndicatorData = (IProgressIndicatorData*) client_data;

  fProgressIndicatorData->layoutControl();
  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  islExposeEventCallback is the procedure invoked when the slider is exposed. |
|  It is used to repaint the ribbon strip.                                     |
------------------------------------------------------------------------------*/
extern void _System islExposeEventCallback (Widget    w,
                                            XtPointer client_data,
                                            XEvent    *event,
                                            ::Boolean *continueToDispatch)
{
  if (event->type == Expose && !w->core.being_destroyed)
  {
     // Get the private data pointer from client data.
     IProgressIndicatorData* fProgressIndicatorData = (IProgressIndicatorData*) client_data;

     fProgressIndicatorData->fillWithRibbon();
  }

  *continueToDispatch = True;
  return;
}
#pragma info(restore)

#pragma info(none)
/*------------------------------------------------------------------------------
|  islKeyPressCallback is the procedure invoked when any key is pressed while  |
|  focus is on the scrollbar.  We will only process arrow key events.          |
------------------------------------------------------------------------------*/
extern void _System islKeyPressCallback( Widget    w,
                                         XtPointer client_data,
                                         XEvent    *event,
                                         ::Boolean *continueToDispatch)
{
  *continueToDispatch = True;

  // Look for cursor key events.
  KeySym keysym = XKeycodeToKeysym( XtDisplay( w ), event->xkey.keycode, 0 );
  if (IsCursorKey( keysym ) == True &&
      !(event->xkey.state & (ControlMask | ShiftMask | Mod1Mask)))
  {
     // Get the private data pointer from client data.
     IProgressIndicatorData
        *fProgressIndicatorData = (IProgressIndicatorData*) client_data;
     IProgressIndicator
        *pControl = fProgressIndicatorData->pControl;
     int scaleMultiple,
         scaleMax;
     unsigned char orientation;

     // Move the slider.
     XtVaGetValues( fProgressIndicatorData->slider,
                    XmNscaleMultiple,       &scaleMultiple,
                    XmNmaximum,             &scaleMax,
                    XmNorientation,         &orientation,
                    NULL );
     if ((keysym == XK_Left && orientation == XmHORIZONTAL) ||
         (keysym == XK_Down && orientation == XmVERTICAL))
     {
        switch (pControl->homePosition()) {
          case IProgressIndicator::homeBottomLeft:
            pControl->moveArmToPixel(
                  MAX( 0, (long)(pControl->armPixelOffset() - scaleMultiple) ));
            break;
          case IProgressIndicator::homeTopRight:
            pControl->moveArmToPixel (
                  MIN( scaleMax, pControl->armPixelOffset() + scaleMultiple ));
            break;
        }
        *continueToDispatch = False;
     }
     else if ((keysym == XK_Right && orientation == XmHORIZONTAL) ||
              (keysym == XK_Up && orientation == XmVERTICAL))
     {
        switch (pControl->homePosition()) {
          case IProgressIndicator::homeBottomLeft:
            pControl->moveArmToPixel(
                  MIN( scaleMax, pControl->armPixelOffset() + scaleMultiple ));
            break;
          case IProgressIndicator::homeTopRight:
            pControl->moveArmToPixel(
                  MAX( 0, (long)(pControl->armPixelOffset() - scaleMultiple) ));
            break;
        }
        *continueToDispatch = False;
     }
  }

  return;
}
#pragma info(restore)

/*------------------------------------------------------------------------------
|  IProgressIndicatorData::setSensitivity                                      |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::setSensitivity ( )
{
  int                scaleMax;
  ::Boolean          sliderSensitive, leftDownSensitive, rightUpSensitive;
  unsigned long      pixelOffset;

  XtVaGetValues (pControl->handle(),
                 XmNsensitive, &sliderSensitive,
                 NULL);
  if (!sliderSensitive)
     return;

  XtVaGetValues (arrowLeftDown,
                 XmNsensitive, &leftDownSensitive,
                 NULL);
  XtVaGetValues (arrowRightUp,
                 XmNsensitive, &rightUpSensitive,
                 NULL);
  XtVaGetValues (slider,
                 XmNmaximum, &scaleMax,
                 NULL);
  IProgressIndicator::HomePosition home = pControl->homePosition();

  pixelOffset = pControl->armPixelOffset();
  if (pixelOffset == 0)
  {
     switch (home) {
       case IProgressIndicator::homeBottomLeft:
        if (leftDownSensitive)
           XtVaSetValues (arrowLeftDown,
                          XmNsensitive, False,
                          NULL);
        if (!rightUpSensitive)
           XtVaSetValues (arrowRightUp,
                          XmNsensitive, True,
                          NULL);
        break;
       case IProgressIndicator::homeTopRight:
        if (rightUpSensitive)
           XtVaSetValues (arrowRightUp,
                          XmNsensitive, False,
                          NULL);
        if (!leftDownSensitive)
           XtVaSetValues (arrowLeftDown,
                          XmNsensitive, True,
                          NULL);
        break;
     }
  }
  else if (pixelOffset == scaleMax)
  {
     switch (home) {
       case IProgressIndicator::homeBottomLeft:
        if (rightUpSensitive)
           XtVaSetValues (arrowRightUp,
                          XmNsensitive, False,
                          NULL);
        if (!leftDownSensitive)
           XtVaSetValues (arrowLeftDown,
                           XmNsensitive, True,
                           NULL);
        break;
       case IProgressIndicator::homeTopRight:
        if (leftDownSensitive)
           XtVaSetValues (arrowLeftDown,
                          XmNsensitive, False,
                          NULL);
        if (!rightUpSensitive)
           XtVaSetValues (arrowRightUp,
                          XmNsensitive, True,
                          NULL);
        break;
     }
  }
  else
  {
     if (!leftDownSensitive)
        XtVaSetValues (arrowLeftDown,
                       XmNsensitive, True,
                       NULL);
     if (!rightUpSensitive)
        XtVaSetValues (arrowRightUp,
                       XmNsensitive, True,
                       NULL);
  }
  return;
}

/*------------------------------------------------------------------------------
|  IProgressIndicatorData::fillWithRibbon                                      |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::fillWithRibbon ()
{
  // If using ribbonStrip style, get a GC and fill from home position to
  // slider with color.
  Widget             bbWidget;
  XGCValues          gcv;
  GC                 gc;
  Dimension          highlightThickness, shadowThickness,
                     sliderHeight, sliderWidth;
  Position           sliderx, slidery;
  XtTranslations     transTable;
  Pixel              troughColor, topShadow, bottomShadow, selectColor;
  Colormap           colorMap;
  unsigned long      armSize;

  bbWidget = pControl->handle();
  if (!XtIsManaged (bbWidget))
     return;

  // Get the translation table for the scrollbar; if it is NULL,
  // then we have an IProgressIndicator (read-only).  In that case,
  // the slider arm should not be visible (ribbon strip will extend
  // to trough color).
  if (XtIsSubclass (slider, xmScaleWidgetClass))
  {
     XtVaGetValues (XtNameToWidget (slider, "Scrollbar"),
                    XmNtranslations, &transTable,
                    XmNtroughColor,  &troughColor,
                    XmNcolormap,     &colorMap,
                    NULL);
  }
  else
  {
     XtVaGetValues (slider,
                    XmNtranslations, &transTable,
                    XmNtroughColor,  &troughColor,
                    XmNcolormap,     &colorMap,
                    NULL);
  }
  // Calculate the ribbon strip color with the XmGetColors routine using
  // the actual trough color as the background color.
  XmGetColors (XtScreen (slider),
               colorMap,
               troughColor,
               &gcv.foreground,
               &topShadow,
               &bottomShadow,
               &selectColor);

  gcv.subwindow_mode = IncludeInferiors;

  XtVaGetValues (slider,
                 XmNhighlightThickness, &highlightThickness,
                 XmNshadowThickness,    &shadowThickness,
                 XmNheight,             &sliderHeight,
                 XmNwidth,              &sliderWidth,
                 XmNx,                  &sliderx,
                 XmNy,                  &slidery,
                 NULL);

  // Determine the arm size.
  if (XtIsSubclass (slider, xmScaleWidgetClass))
     armSize = ARM_SIZE;
  else
     // Minimum arm size is actually one, but want to prevent roundoff
     // error.
     armSize = 2;

  gc = XCreateGC (XtDisplay (bbWidget),
                  RootWindow (XtDisplay (bbWidget), 0),
                  GCForeground | GCSubwindowMode,
                  &gcv);

  // Fill with foreground color for ribbon strip.  For a read-only
  // progress indicator, also want to fill over half of the slider arm
  // with the trough color.
  if (pControl->isVertical())
     if (pControl->homePosition() == IProgressIndicator::homeBottomLeft)
        if (transTable == NULL)
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + sliderHeight - pControl->armPixelOffset()
                              - highlightThickness - shadowThickness
                              - armSize / 2,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness),
                           pControl->armPixelOffset() + armSize / 2);
           XSetForeground (XtDisplay (bbWidget),
                           gc,
                           troughColor);
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness),
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness) -
                              pControl->armPixelOffset() - armSize / 2);
        }
        else
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + sliderHeight - pControl->armPixelOffset()
                              - highlightThickness - shadowThickness,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness),
                           pControl->armPixelOffset());
        }
     else
        if (transTable == NULL)
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness),
                           pControl->armPixelOffset() + armSize / 2);
           XSetForeground (XtDisplay (bbWidget),
                           gc,
                           troughColor);
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness +
                              pControl->armPixelOffset() + armSize / 2,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness),
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness) -
                              pControl->armPixelOffset() - armSize / 2);
        }
        else
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness),
                           pControl->armPixelOffset());
        }
  else
     if (pControl->homePosition() == IProgressIndicator::homeBottomLeft)
        if (transTable == NULL)
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness,
                           pControl->armPixelOffset() + armSize / 2,
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness));
           XSetForeground (XtDisplay (bbWidget),
                           gc,
                           troughColor);
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness +
                              pControl->armPixelOffset() + armSize / 2,
                           slidery + highlightThickness + shadowThickness,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness) -
                              pControl->armPixelOffset() - armSize / 2,
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness));
        }
        else
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness,
                           pControl->armPixelOffset(),
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness));
        }
     else
        if (transTable == NULL)
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + sliderWidth -
                              highlightThickness - shadowThickness -
                              pControl->armPixelOffset() - armSize / 2,
                           slidery + highlightThickness + shadowThickness,
                           pControl->armPixelOffset() + armSize / 2,
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness));
           XSetForeground (XtDisplay (bbWidget),
                           gc,
                           troughColor);
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx +
                              highlightThickness + shadowThickness,
                           slidery + highlightThickness + shadowThickness,
                           sliderWidth -
                              2 * (highlightThickness + shadowThickness) -
                              pControl->armPixelOffset() - armSize / 2,
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness));
        }
        else
        {
           XFillRectangle (XtDisplay (bbWidget),
                           XtWindow (bbWidget),
                           gc,
                           sliderx + sliderWidth -
                              highlightThickness - shadowThickness -
                              pControl->armPixelOffset(),
                           slidery + highlightThickness + shadowThickness,
                           pControl->armPixelOffset(),
                           sliderHeight -
                              2 * (highlightThickness + shadowThickness));
        }
  Display *fDisplay( XtDisplay(bbWidget) );
  XFlush( fDisplay );
  XSync( fDisplay, DefaultScreen( fDisplay ) );
  XFreeGC (fDisplay, gc);

  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData :: layoutControl                                      |
|                                                                              |
------------------------------------------------------------------------------*/
void IProgressIndicatorData :: layoutControl( )
{
  Dimension     daHeight, daWidth, scaleHeight, scaleWidth, tickFormHeight,
                tickFormWidth, arrowFormHeight, arrowFormWidth,
                highlightThickness, shadowThickness;
  Position      scalex, scaley, tickFormx, tickFormy, arrowFormx, arrowFormy;
  int           scaleMaximum, calcSpacing, scaleValue, n;
  unsigned long armSize;
  double        relativeOffset;
  Arg           args[15];

  // Get the internal dimensions of the drawing area.
  XtVaGetValues (pControl->handle(),
                 XmNheight, &daHeight,
                 XmNwidth,  &daWidth,
                 NULL);

  // If the drawing area has no dimensions, then just return.
  if (daHeight == 0)
     return;

  // Calculate the relative offset of the slider arm.  Want to restore the
  // arm to the same relative offset following resize.
  XtVaGetValues (slider,
                 XmNmaximum, &scaleMaximum,
                 XmNvalue,   &scaleValue,
                 NULL);
  relativeOffset = (double)scaleValue / (double)scaleMaximum;

  // Reset the location of the scale, tick form, and arrow form within the
  // drawing area.
  XtVaGetValues (slider,
                 XmNheight,             &scaleHeight,
                 XmNwidth,              &scaleWidth,
                 XmNhighlightThickness, &highlightThickness,
                 XmNshadowThickness,    &shadowThickness,
                 NULL);

  if (arrowLeftDown)
     XtVaGetValues (XtParent (arrowLeftDown),
                    XmNheight, &arrowFormHeight,
                    XmNwidth,  &arrowFormWidth,
                    NULL);
     else
        arrowFormHeight = arrowFormWidth = 0;

  // If this is the first time through layout, our queried widths and
  // heights may be 0.  We need to use the Motif default in our calculations.
  // Since there is no good way to query the default, use the values
  // witnessed during testing.
  if (pControl->isVertical())
  {
     if (scaleWidth == 0)
        scaleWidth = 19;
     if (arrowLeftDown)
     {
        if (arrowFormWidth == 0)
           arrowFormWidth = scaleWidth;
        if (arrowFormHeight == 0)
           arrowFormHeight = 2 * arrowFormWidth;
     }
  }
  else
  {
     if (scaleHeight == 0)
        scaleHeight = 19;
     if (arrowLeftDown)
     {
        if (arrowFormHeight == 0)
           arrowFormHeight = scaleHeight;
        if (arrowFormWidth == 0)
           arrowFormWidth = 2 * arrowFormHeight;
     }
  }

  if (XtIsSubclass (slider, xmScaleWidgetClass))
     armSize = ARM_SIZE;
  else
     armSize = 1;

  calcSpacing = tickSpacing[primaryScale];
  // If no tick spacing was specified, calculate the effective spacing.
  if (!calcSpacing)
  {
     // The slider arm is a fixed size and half the slider arm size must
     // be left on either end of the scale before the first and after
     // the last tick mark.  The tick form may therefore be wider
     // than the scale width or vice versa.  This depends on the
     // new rectangle size and the number of ticks (ie, the resulting tick
     // spacing).

     if (pControl->isVertical())
     {
        calcSpacing = (daHeight - arrowFormHeight -
            2 * (highlightThickness + shadowThickness) - armSize) /
            (numberTicks[primaryScale] - 1);
        if (calcSpacing > armSize)
           // Spacing will be constrained by tick form size.
           calcSpacing = (daHeight - 2 * (highlightThickness + shadowThickness)
                                  - arrowFormHeight) /
                                numberTicks[primaryScale];
     }
     else
     {
        calcSpacing = (daWidth - arrowFormWidth -
            2 * (highlightThickness + shadowThickness) - armSize) /
            (numberTicks[primaryScale] - 1);
        if (calcSpacing > armSize)
           // Spacing will be constrained by tick form size.
           calcSpacing = (daWidth - 2 * (highlightThickness + shadowThickness)
                                - arrowFormWidth) /
                                numberTicks[primaryScale];
     }
  }

  scaleMaximum = (numberTicks[primaryScale] - 1) * calcSpacing;
  if (pControl->isVertical())
  {
     tickFormHeight = (Dimension)(numberTicks[primaryScale] *
                          calcSpacing);
     scaleHeight = (Dimension) (scaleMaximum + armSize +
                    2 * (highlightThickness + shadowThickness));
     tickFormWidth = (Dimension) (longestTick[primaryScale]);
  }
  else
  {
     tickFormWidth = (Dimension) (numberTicks[primaryScale] *
                         calcSpacing);
     scaleWidth = (Dimension) (scaleMaximum + armSize +
                  2 * (highlightThickness + shadowThickness));
     tickFormHeight = (Dimension) (longestTick[primaryScale]);
  }

  if (pControl->isVertical())
  {
     if (buttonsPosition == ISlider::top)
     {
        arrowFormy = (Position)((daHeight - scaleHeight - arrowFormHeight)
                         / 2);
        scaley = (Position)(arrowFormy + arrowFormHeight);
     }
     else if (buttonsPosition == ISlider::bottom)
     {
        scaley = (Position)((daHeight - scaleHeight - arrowFormHeight) / 2);
        arrowFormy = (Position)(scaley + scaleHeight);
     }
     else
        scaley = (Position)((daHeight - scaleHeight) / 2);

     switch (align) {
       case IProgressIndicator::bottomLeft:
        scalex = 0;
        break;
       case IProgressIndicator::topRight:
        scalex = (Position)(daWidth - scaleWidth);
        break;
       default:
        scalex = (Position)((daWidth - scaleWidth) / 2);
     }
     arrowFormx = scalex;
  }
  else
  {
     if (buttonsPosition == ISlider::left)
     {
        arrowFormx = (Position)((daWidth - scaleWidth - arrowFormWidth) / 2);
        scalex = (Position)(arrowFormx + arrowFormWidth);
     }
     else if (buttonsPosition == ISlider::right)
     {
        scalex = (Position)((daWidth - scaleWidth - arrowFormWidth) / 2);
        arrowFormx = (Position)(scalex + scaleWidth);
     }
     else
        scalex = (Position)((daWidth - scaleWidth) / 2);

     switch (align) {
       case IProgressIndicator::bottomLeft:
        scaley = (Position)(daHeight - scaleHeight);
        break;
       case IProgressIndicator::topRight:
        scaley = 0;
        break;
       default:
        scaley = (Position)((daHeight - scaleHeight) / 2);
     }
     arrowFormy = scaley;
  }

  n = 0;
  XtSetArg (args[n], XmNx, scalex); n++;
  XtSetArg (args[n], XmNy, scaley); n++;
  XtSetArg (args[n], XmNheight, scaleHeight); n++;
  XtSetArg (args[n], XmNwidth, scaleWidth); n++;
  XtSetArg (args[n], XmNminimum, 0); n++;
  XtSetArg (args[n], XmNmaximum, scaleMaximum); n++;

  if (XtIsSubclass (slider, xmScaleWidgetClass))
  {
     XtSetArg (args[n], XmNscaleWidth, scaleWidth); n++;
     XtSetArg (args[n], XmNscaleMultiple,
                  scaleMaximum / (numberTicks[primaryScale] - 1)); n++;
  }
  else
  {
     XtSetArg (args[n], XmNpageIncrement,
                  scaleMaximum / (numberTicks[primaryScale] - 1)); n++;
  }
  // Want to reset the value so the slider arm is at the same relative offset
  // as it was before the sizing.
  XtSetArg (args[n], XmNvalue, relativeOffset * scaleMaximum); n++;
  XtSetValues (slider, args, n);

  // Now verify that the shaft size is large enough to accomodate tick spacing
  // and the slider arm.  If it is not, request to re-size was rejected by
  // the manager widget.  We want to throw an exception since the scroll bar
  // is either not visible (if a scroll bar) or it may be sized incorrectly
  // (if a scale widget).
  if (tickSpacing[primaryScale])
  {
     XtVaGetValues (slider,
                    XmNheight,  &scaleHeight,
                    XmNwidth,   &scaleWidth,
                    NULL);
     if ((pControl->isVertical() &&
            (scaleHeight - 2 * (highlightThickness + shadowThickness) !=
            (numberTicks[primaryScale]-1) * calcSpacing + armSize)) ||
         (!pControl->isVertical() &&
            (scaleWidth - 2 * (highlightThickness + shadowThickness) !=
            (numberTicks[primaryScale]-1) * calcSpacing + armSize)))
     {
        ITHROWLIBRARYERROR (IC_SLDR_RECT_TOO_SMALL,
                            IBaseErrorInfo::invalidParameter,
                            IException::recoverable);
     }
  }

  if (arrowLeftDown)
  {
     XtVaSetValues (XtParent (arrowLeftDown),
                    XmNx,      arrowFormx,
                    XmNy,      arrowFormy,
                    XmNheight, arrowFormHeight,
                    XmNwidth,  arrowFormWidth,
                    NULL);
  }

  switch (primaryScale) {
    case IProgressIndicator::scale1:
     // Set the position of the right/top tick form.
     if (pControl->isVertical() && align != IProgressIndicator::topRight)
     {
        tickFormx = (Position)(scalex + scaleWidth);
        tickFormy = (Position)(scaley + (scaleHeight - tickFormHeight) / 2);
     }
     else if (!pControl->isVertical() && align != IProgressIndicator::topRight)
     {
        tickFormx = (Position)(scalex + (scaleWidth - tickFormWidth) / 2);
        tickFormy = (Position)(scaley - tickFormHeight);
     }
     break;
    case IProgressIndicator::scale2:
     // Set the position of the left/bottom scale.
     if (pControl->isVertical() && align != IProgressIndicator::bottomLeft)
     {
        tickFormx = (Position)(scalex - tickFormWidth);
        tickFormy = (Position)(scaley + (scaleHeight - tickFormHeight) / 2);
     }
     else if (!pControl->isVertical() && align != IProgressIndicator::bottomLeft)
     {
        tickFormx = (Position)(scalex + (scaleWidth - tickFormWidth) / 2);
        tickFormy = (Position)(scaley + scaleHeight);
     }
     break;
  }

  XtVaSetValues (tickForm[primaryScale],
                 XmNx,      tickFormx,
                 XmNy,      tickFormy,
                 XmNheight, tickFormHeight,
                 XmNwidth,  tickFormWidth,
                 NULL);

  if (isRibbonStrip)
     fillWithRibbon();

  return;
}


/*------------------------------------------------------------------------------
| IProgressIndicatorData::registerCallbacks                                    |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::registerCallbacks ()
{
  // Add event handler for re-sizing control.
  XtAddCallback (pControl->handle(),
                 XmNresizeCallback,
                 islResizeCallback,
                 this);

  // Add the arrow button callbacks.
  if (arrowLeftDown)
  {
     XtAddCallback (arrowLeftDown,
                    XmNarmCallback,
                    islArmCallback,
                    this);
     XtAddCallback (arrowRightUp,
                    XmNarmCallback,
                    islArmCallback,
                    this);
     XtAddCallback (arrowLeftDown,
                    XmNdisarmCallback,
                    islDisarmCallback,
                    this);
     XtAddCallback (arrowRightUp,
                    XmNdisarmCallback,
                    islDisarmCallback,
                    this);
  }

  // Add a drag callback for all sliders.
  XtTranslations transTable;
  if (XtIsSubclass( slider, xmScaleWidgetClass ))
  {
     XtVaGetValues( XtNameToWidget( slider, "Scrollbar" ),
                    XmNtranslations, &transTable,
                    NULL );
  }
  else
  {
     XtVaGetValues (slider,
                    XmNtranslations, &transTable,
                    NULL);
  }
   
  if (transTable != NULL)
  {
     // Register callback to implement snap to tick behavior.  It will also
     // control arrow button sensitivity.
     XtAddCallback( slider,
                    XmNvalueChangedCallback,
                    islValueChangedCallback,
                    this );

     // Register callback to enable notifications, handler, and ribbon strip painting.
     XtAddCallback( slider,
                    XmNdragCallback,
                    islDragCallback,
                    this );

     // Want to get the key events for the scrollbar so we can process
     // arrow keys to move one full tick spacing.  Make sure we grab the key
     // press event first.
     XtInsertEventHandler( XtNameToWidget( slider, "Scrollbar" ),
                           KeyPressMask,
                           False,
                           islKeyPressCallback,
                           this,
                           XtListHead );

     // We also need to get focus events to enable the IFocusHandler.
     XtAddEventHandler( XtNameToWidget( slider, "Scrollbar" ),
                        FocusChangeMask,
                        False,
                        iwindowXEventCallback,
                        pControl );

     if (isRibbonStrip)
     {
        // Add event handler for re-painting ribbon strip on expose for a
        // slider.
        XtAddEventHandler( XtNameToWidget( slider, "Scrollbar" ),
                           ExposureMask,
                           False,
                           islExposeEventCallback,
                           this );
     }
  }
  else if (isRibbonStrip)
  {
        // Add event handler for re-painting ribbon strip on expose for a
        // progress indicator.
        XtAddEventHandler( slider,
                           ExposureMask,
                           False,
                           islExposeEventCallback,
                           this );
  }

  return;
}

/*------------------------------------------------------------------------------
| IProgressIndicatorData::unregisterCallbacks                                  |
------------------------------------------------------------------------------*/
void IProgressIndicatorData::unregisterCallbacks ()
{
  if (pControl->isValid())
  {
     // Resize callback.
     XtRemoveCallback ((Widget)pControl->handle(),
                       XmNresizeCallback,
                       islResizeCallback,
                       this);
  }
  
  if (arrowLeftDown && !arrowLeftDown->core.being_destroyed
                    && !arrowRightUp->core.being_destroyed )
  {
     // Arrow button callbacks.
     XtRemoveCallback (arrowLeftDown,
                       XmNarmCallback,
                       islArmCallback,
                       this);
     XtRemoveCallback (arrowRightUp,
                       XmNarmCallback,
                       islArmCallback,
                       this);
     XtRemoveCallback (arrowLeftDown,
                       XmNdisarmCallback,
                       islDisarmCallback,
                       this);
     XtRemoveCallback (arrowRightUp,
                       XmNdisarmCallback,
                       islDisarmCallback,
                       this);
  }

  // Determine whether the progress indicator is read-only.
  if ( !slider->core.being_destroyed )
  {
     XtTranslations transTable;
     if (XtIsSubclass( slider, xmScaleWidgetClass ))
     {
        XtVaGetValues( XtNameToWidget( slider, "Scrollbar" ),
                       XmNtranslations, &transTable,
                       NULL );
     }
     else
     {
        XtVaGetValues( slider,
                       XmNtranslations, &transTable,
                       NULL );
     }
   
     if (transTable != NULL)
     {
        XtRemoveCallback( slider,
                          XmNvalueChangedCallback,
                          islValueChangedCallback,
                          this );
   
        XtRemoveCallback( slider,
                          XmNdragCallback,
                          islDragCallback,
                          this );
   
        XtRemoveEventHandler( XtNameToWidget( slider, "Scrollbar" ),
                              KeyPressMask,
                              False,
                              islKeyPressCallback,
                              this );
   
        XtRemoveEventHandler( XtNameToWidget( slider, "Scrollbar" ),
                              FocusChangeMask,
                              False,
                              iwindowXEventCallback,
                              pControl );
                          
        if (isRibbonStrip)
        {
           // Remove event handler for re-painting ribbon strip on expose
           // for a slider.
           XtRemoveEventHandler( XtNameToWidget( slider, "Scrollbar" ),
                                 ExposureMask,
                                 False,
                                 islExposeEventCallback,
                                 this );
        }
     }
     else if (isRibbonStrip)
     {
           // Remove event handler for re-painting ribbon strip on expose
           // for a progress indicator.
           XtRemoveEventHandler (slider,
                                 ExposureMask,
                                 False,
                                 islExposeEventCallback,
                                 this);
     }
  }
}
#endif //IC_MOTIF

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IDefaultSliderHandler::IDefaultSliderHandler                                 |
------------------------------------------------------------------------------*/
IDefaultSliderHandler::IDefaultSliderHandler(
                                       IPMProgressIndicatorData *progressData )
  : IDefaultSliderHandler::Inherited( ),
    progressData( progressData )
{ }

/*------------------------------------------------------------------------------
| IDefaultSliderHandler::dispatchHandlerEvent                                  |
|                                                                              |
------------------------------------------------------------------------------*/
bool IDefaultSliderHandler::dispatchHandlerEvent( IEvent& event )
{
  IMODTRACE_ALL("IDefaultSliderHandler::dispatchHandlerEvent")

  bool fStopHandling( false );

  IWindowHandle hwndControl(event.controlHandle());

  if ( hwndControl )
  {
     IWindowClassName className( hwndControl );

     if (className == WC_SLIDER)
     {
       switch (event.eventId())
       {
#ifdef IC_WIN
         // Adding code to trap accelerator/mnemonics and send directly to
         // owner (allowing it to naturally progress through dispatcher results
         // in an audible beep caused by the button default window procedure
         // since it cannot find a match for a mnemonic that we process in
         // ICanvas or IFrameWindow), eating original event.
         case WM_SYSCHAR:
         {
           IWindow* window = event.window();
           if (window)
             window->owner()->sendEvent( event );
           fStopHandling = true;
         }
         break;

         case IC_UM_IS_AGGREGATE_CTRL:
         {
           event.setResult(true);
           fStopHandling = true;
         }
         break;
#endif // IC_WIN

         case WM_SIZE:
         {
           // In response to a resize, we want to readjust the arm position.
           IProgressIndicator *progressInd = progressData->progressInd;

           if (progressData->initialArmOffset)
           {
              // This is the first time the slider has been sized.  We have
              // cached an initial arm position and need to post the event
              // to set the arm position at this point.
              progressInd->postEvent( SLM_SETSLIDERINFO,
                    IEventParameter1( (unsigned short)SMA_SLIDERARMPOSITION,
                                      (unsigned short)SMA_INCREMENTVALUE ),
                    IEventParameter2( progressData->initialArmOffset ));
              progressData->initialArmOffset = 0;
           }
           else if (progressData->variableSpacing[progressInd->primaryScale()])
           {
              // If the slider has the snapToTick style, then save the arm's tick
              // position for resetting following the resize.  Otherwise, calculate
              // the arm's relative offset.
              double        relativeOffset;
              unsigned long newArmPosition;

              if (progressInd->isSnapToTickEnabled())
                 newArmPosition = progressInd->armTickOffset();
              else
                 relativeOffset = (double)(progressInd->armPixelOffset()) /
                                  (double)(progressInd->armRange());

              // Dispatch the remaining handlers - call the default procedure.
              progressInd->dispatchRemainingHandlers( event, true );
              
              // Layout for the control has now been completed.  We can reset the
              // arm position.
              unsigned long ulResult = 0;
              if (progressInd->isSnapToTickEnabled())
              {
                 // Use the tick position.
                 ulResult = progressInd->sendEvent( SLM_SETSLIDERINFO,
                              IEventParameter1( (unsigned long)SMA_SLIDERARMPOSITION,
                                                (unsigned long)SMA_INCREMENTVALUE ),
                              IEventParameter2( newArmPosition ))
                              .asUnsignedLong();
              }
              else
              {
                 newArmPosition =
                    (unsigned long)((relativeOffset * progressInd->armRange()) + 0.5);
                 ulResult = progressInd->sendEvent( SLM_SETSLIDERINFO,
                                 IEventParameter1( (unsigned long)SMA_SLIDERARMPOSITION,
                                                   (unsigned long)SMA_RANGEVALUE ),
                                 IEventParameter2( (unsigned long)newArmPosition ))
                                 .asUnsignedLong();
              }
              if (!ulResult)
              {
                 ITHROWGUIERROR2( "SLM_SETSLIDERINFO",
                                  IBaseErrorInfo::invalidParameter,
                                  IException::recoverable );
              }

              fStopHandling = true;
           }
         }
         break;
       }
     }
  }

  return fStopHandling;
}
#endif // IC_WIN
