// Revision: 40 1.35.1.5 source/ui/basectl/isetcv.cpp, canvas, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: isetcv.cpp                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in isetcv.hpp.                                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C"
{
  #include <iwindefs.h>
}

#include <isetcv.hpp>

#include <ialgnatt.hpp>
#include <ibidiset.hpp>
#include <icconst.h>
#include <icvobsrv.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <inotifev.hpp>
#include <ipoint.hpp>
#include <ireslib.hpp>
#include <iseq2.h>              // for ISequence
#include <isetcvob.hpp>         // for ISetCanvasObject
#include <isetcvwd.hpp>         // for ISetCanvasWindowData
#include <isizehdr.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#include <iwinlsts.hpp>
#include <iwposbuf.hpp>
#ifdef IC_MOTIF
  #include <ixconst.h>
  #include <isizrect.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ISETCV_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Public set canvas styles.                                                    |
------------------------------------------------------------------------------*/
const ISetCanvas::Style
  ISetCanvas::explicitAddsNeeded  ( 0, ICNV_EXPLICITADDS ),
  ISetCanvas::horizontalDecks     ( 0, ICNV_HORIZONTALDECKS ),
  ISetCanvas::verticalDecks       ( 0, ICNV_VERTICALDECKS ),
  ISetCanvas::packTight           ( 0, ICNV_PACKTIGHT ),
  ISetCanvas::packEven            ( 0, ICNV_PACKEVEN ),
  ISetCanvas::packExpanded        ( 0, ICNV_PACKEXPANDED ),
  ISetCanvas::leftAlign           ( 0, ICNV_LEFTALIGN ),
  ISetCanvas::rightAlign          ( 0, ICNV_RIGHTALIGN ),
  ISetCanvas::centerAlign         ( 0, ICNV_CENTERALIGN ),
  ISetCanvas::topAlign            ( 0, ICNV_TOPALIGN ),
  ISetCanvas::bottomAlign         ( 0, ICNV_BOTTOMALIGN ),
  ISetCanvas::centerVerticalAlign ( 0, ICNV_CENTERVERTICALALIGN ),
  ISetCanvas::decksByGroup        ( 0, ICNV_DECKSBYGROUP ),
  ISetCanvas::classDefaultStyle   ( WS_VISIBLE, (ICNV_HORIZONTALDECKS |
                                                 ICNV_PACKTIGHT       |
                                                 ICNV_LEFTALIGN       |
                                                 ICNV_TOPALIGN) );

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
ISetCanvas::Style
  ISetCanvas::currentDefaultStyle ( WS_VISIBLE, (ICNV_HORIZONTALDECKS |
                                                 ICNV_PACKTIGHT       |
                                                 ICNV_LEFTALIGN       |
                                                 ICNV_TOPALIGN) );
IWindow::DataHandle
  ISetCanvas::fClassDataHandle ( 0 );

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ISetCanvasShowHandler                                                        |
|                                                                              |
| Motif does not get the first WM_RESIZE message so we need a handler to look  |
| for the first time the window is shown.                                      |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class ISetCanvasShowHandler : public IHandler {
protected:
virtual bool
  dispatchHandlerEvent ( IEvent& event );
}; // ISetCanvasShowHandler

#pragma pack(pop)
#pragma enum(pop)

#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| ISetCanvasData                                                               |
|                                                                              |
| Private data class for ISetCanvas.                                           |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

/*------------------------------------------------------------------------------
|  ISetCanvasChildList                                                          |
|  Public data memeber for ISetCanvasData                                       |
--------------------------------------------------------------------------------*/

class ISetCanvasChildList : public ISequence < IWindow* > {

public :
	ISetCanvasChildList ( );
	~ISetCanvasChildList ( );
};//ISetCanvasChildList

class ISetCanvasData {
public:
ISetCanvasData ( )
  : layoutInProgress( 0 )
  , fAddCount( 0 )
#ifdef IC_ORIGINLOWERLEFT
  , positioningHeight( 0 )
#endif
  , pSetcvObserver( 0 )
  { }
unsigned long
  layoutInProgress,
  fAddCount;
#ifdef IC_ORIGINLOWERLEFT
unsigned long
  positioningHeight;
#endif
ISetCanvasObserver
 *pSetcvObserver;
 ISetCanvasChildList
  mSetcvsChildList;
#ifdef IC_MOTIF
ISetCanvasShowHandler
 *showHandler;
#endif
}; // ISetCanvasData


/*------------------------------------------------------------------------------
| ISetCanvasResizeHandler                                                      |
|                                                                              |
| Handle resize requests for the tool bars frame.                              |
------------------------------------------------------------------------------*/
class ISetCanvasResizeHandler : public IResizeHandler {
public:
  ISetCanvasResizeHandler ( );
virtual
 ~ISetCanvasResizeHandler ( );
protected:
virtual bool
  windowResize ( IResizeEvent& event );
}; // ISetCanvasResizeHandler

#pragma pack(pop)
#pragma enum(pop)

/*-----------------------------------------------------------------------------
| ISetCanvasChildList :: ISetCanvasChildList                                   |
-------------------------------------------------------------------------------*/
ISetCanvasChildList :: ISetCanvasChildList ( )
		:ISequence < IWindow* > ( )
{ }

/*------------------------------------------------------------------------------
| ISetCanvasChildList :: ~ISetCanvasChildList                                   |
--------------------------------------------------------------------------------*/

ISetCanvasChildList :: ~ ISetCanvasChildList ( )
{ }

/*------------------------------------------------------------------------------
| ISetCanvasResizeHandler::ISetCanvasResizeHandler                             |
|                                                                              |
------------------------------------------------------------------------------*/
ISetCanvasResizeHandler::ISetCanvasResizeHandler ( )
{ }

/*------------------------------------------------------------------------------
| ISetCanvasResizeHandler::~ISetCanvasResizeHandler                            |
|                                                                              |
------------------------------------------------------------------------------*/
ISetCanvasResizeHandler::~ISetCanvasResizeHandler ( )
{ }


/*------------------------------------------------------------------------------
| ISetCanvasObjectList                                                         |
|                                                                              |
| Implementation class.                                                        |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class ISetCanvasObjectList : public ISequence < ISetCanvasObject* > {
public:
  ISetCanvasObjectList ( );
 ~ISetCanvasObjectList ( );
}; // ISetCanvasObjectList

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| ISetCanvasObjectList::ISetCanvasObjectList                                   |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
ISetCanvasObjectList::ISetCanvasObjectList ( )
  : ISequence < ISetCanvasObject* > ( )
{ }

/*------------------------------------------------------------------------------
| ISetCanvasObjectList::~ISetCanvasObjectList                                  |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ISetCanvasObjectList::~ISetCanvasObjectList ( )
{ }


bool ISetCanvasDeleter ( ISetCanvasObject* const& element,
                         void* )
{
  delete element;
  return true;
}


/*------------------------------------------------------------------------------
| ISetCanvas::ISetCanvas                                                       |
|                                                                              |
| Constructor to create a set canvas and its presentation system window.       |
------------------------------------------------------------------------------*/
ISetCanvas::ISetCanvas ( unsigned long windowIdentifier,
                         IWindow* parent,
                         IWindow* owner,
                         const IRectangle& initial,
                         const Style& style )
  : ISetCanvas::Inherited( )
  , ulClNumDecks( 1 )
  , ulClGroupPad( 0 )
  , fExplicitAdds( false )
  , sizeHandler( 0 )
  , fSetCanvasData( new ISetCanvasData )
{
  this->initializeObject();
  this->initialize( windowIdentifier, parent, owner, initial,
                    style.asUnsignedLong(), style.asExtendedUnsignedLong() );
}

/*------------------------------------------------------------------------------
| ISetCanvas::ISetCanvas                                                       |
|                                                                              |
| Protected constructor for derived classes.                                   |
------------------------------------------------------------------------------*/
ISetCanvas::ISetCanvas ( )
  : ISetCanvas::Inherited( )
  , ulClNumDecks( 1 )
  , ulClGroupPad( 0 )
  , sizeHandler( 0 )
  , fSetCanvasData( new ISetCanvasData )
{
  this->initializeObject();
}

/*------------------------------------------------------------------------------
| ISetCanvas::initializeObject                                                 |
|                                                                              |
| Common code for object construction.                                         |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::initializeObject ( )
{
  fClassDataHandle = IWindow::dataHandleWithKey( "ISetCanvas" );
    // (Add check to do only do once?)          IC_NOT_YET

  // Create and attach any necessary handlers.
#ifdef IC_ORIGINLOWERLEFT
  sizeHandler = new ISetCanvasResizeHandler;
  sizeHandler->handleEventsFor( this );
#else
  if ( IBidiSettings::isBidiSupported() )
  {
     sizeHandler = new ISetCanvasResizeHandler;
     sizeHandler->handleEventsFor( this );
  }
#endif

#ifdef IC_MOTIF
  fSetCanvasData->showHandler = new ISetCanvasShowHandler;
  fSetCanvasData->showHandler->handleEventsFor( this );
#endif

  fSetCanvasData->pSetcvObserver = new ISetCanvasObserver(this);

  // Set the margin and pad defaults using the average character width
  // and maximum character height.
  if ( this->isValid() )
  {  // Window has been created already.
     // Use the current font.
     sizClMargin = this->characterSize();
  }
  else
  {  // No underlying window yet, so the canvas has no font yet either.
     // Use the desktop font.
     sizClMargin = IWindow::desktopWindow()->characterSize();
  }
  sizClMargin.setHeight( sizClMargin.height() / 2 );
  sizClPad = sizClMargin;

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::initialize                                                       |
|                                                                              |
| Create the presentation system window.                                       |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::initialize ( unsigned long     windowIdentifier,
                                     IWindow*          parent,
                                     IWindow*          owner,
                                     const IRectangle& initialRect,
                                     unsigned long     style,
                                     unsigned long     extendedStyle )
{
  Style
    userStyle( style, extendedStyle );
  // Check for an invalid style combination.
  if ( ( ( userStyle & horizontalDecks )  &&  ( userStyle & verticalDecks ) )  ||
       ( ( userStyle & packTight )  &&
         ( userStyle & ( packEven | packExpanded ) ) )   ||
       ( ( userStyle & packEven )  &&  ( userStyle & packExpanded ) )  ||
       ( ( userStyle & leftAlign )  &&
         ( userStyle & ( rightAlign | centerAlign ) ) )  ||
       ( ( userStyle & rightAlign )  &&  ( userStyle & centerAlign ) )  ||
       ( ( userStyle & topAlign )  &&
         ( userStyle & ( bottomAlign | centerVerticalAlign ) ) )  ||
       ( ( userStyle & bottomAlign )  &&  ( userStyle & centerVerticalAlign ) ) )
  {           // Mutually exclusive styles specified.
     ITHROWLIBRARYERROR( IC_INVALIDSTYLE,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable );
  }

  if ( userStyle & explicitAddsNeeded )
  {
     fExplicitAdds = true;
  }

  // Save the extended style to make sure we have a copy of it stored.
  setExtendedStyle( this->extendedStyle() | extendedStyle );

  // Create the presentation system window.
  this->Inherited::initialize( windowIdentifier,
                               parent,
                               owner,
                               initialRect,
                               this->convertToGUIStyle( userStyle ),
                               this->convertToGUIStyle( userStyle, true ) );

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::~ISetCanvas                                                      |
|                                                                              |
| Clean-up.                                                                    |
------------------------------------------------------------------------------*/
ISetCanvas::~ISetCanvas ( )
{
  // Iterate our child windows, calling stopHandlingNotificationsFor,
  // so we won't get any more notifications from child windows that
  // were not explicitly removed from the set canvas.
  ICanvasObserver *cObsrv = fSetCanvasData->pSetcvObserver;
  ISetCanvasChildList::Cursor cursor(fSetCanvasData->mSetcvsChildList);
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     IWindow *childWindow = ( IWindow* )fSetCanvasData->mSetcvsChildList.elementAt(cursor);
     if ( IAllAllocatedIWindows::isValid( childWindow ) )
     {
        // Validate that the child window objects have not yet been
        // deleted (passing a deleted IWindow object to
        // IObserver::stopHandlingNotificationsFor could cause a trap).
        // ISetCanvasObserver will remove child windows from this
        // collection as they are deleted, but that only occurs if the
        // child windows are enabled for notification. The above call to
        // IAllAllocatedIWindows::isValid covers the case where the child
        // windows never send the IWindow::deleteId notification.
        cObsrv->stopHandlingNotificationsFor( *childWindow,
                                              IWindow::attributeAddReplaceId );
        cObsrv->stopHandlingNotificationsFor( *childWindow,
                                              IWindow::attributeRemoveId );
        cObsrv->stopHandlingNotificationsFor( *childWindow,
                                              IWindow::deleteId );
     }
  }
  	
  delete fSetCanvasData->pSetcvObserver;

  if ( sizeHandler )
  {
     sizeHandler->stopHandlingEventsFor( this );
     delete sizeHandler;
  }
#ifdef IC_MOTIF
  if ( fSetCanvasData->showHandler )
  {
     fSetCanvasData->showHandler->stopHandlingEventsFor( this );
     delete fSetCanvasData->showHandler;
  }
#endif
  delete fSetCanvasData;
}

/*------------------------------------------------------------------------------
| ISetCanvas::defaultStyle                                                     |
|                                                                              |
| Returns the current default style.                                           |
| Notes: Cannot inline this since it is in the constructor declaration.        |
------------------------------------------------------------------------------*/
ISetCanvas::Style ISetCanvas::defaultStyle ( )
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| ISetCanvas::convertToGUIStyle                                                |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long ISetCanvas::convertToGUIStyle ( const IBitFlag& guiStyle,
                                              bool            bExtOnly ) const
{
  // Obtain the style from the class (ICanvas) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if ( bExtOnly )
  {
    // Use mask to only return extended styles in the user defined range.
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
    // Replace canvas window style with set canvas window style.
#ifdef IC_PMWIN
    ulStyle |= ( guiStyle.asUnsignedLong()
                  & (unsigned long)~IC_CVS_CANVAS )
                  | IC_CVS_SETCANVAS;
#else
    // All the styles for the setcanvas are extended styles, so we do not mask.
    ulStyle |= guiStyle.asUnsignedLong();
#endif

    // No additional processing required since ICanvas::convertToGUIStyle is
    // doing all the required work.
  }

  return ulStyle;
}

/*------------------------------------------------------------------------------
| ISetCanvas::add                                                              |
|                                                                              |
| Adds the specified child window so that the canvas will manage its size and  |
| position.  This function also shows the child window.  It has no real effect |
| if the canvas does not use the ISetCanvas::explicitAddsNeeded style (unless  |
| the child was previously removed using ISetCanvas::remove or                 |
| replaceChildWith).                                                           |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::add ( IWindow* child )
{
  IASSERTPARM( child->parent() == this );

  bool
    added = false;
  ISetCanvasWindowData
   *data = (ISetCanvasWindowData*)( child->windowData( fClassDataHandle ) );
  if ( ! data )
  {
     data = new ISetCanvasWindowData( this->handle() );
     data->addToLayout();
     child->adoptWindowData( fClassDataHandle, data );
     fSetCanvasData->fAddCount++;
     added = true;
  }
  else if ( ! data->isAddedToLayout()  ||
            ( data->parentHandle() != this->handle() ) )
  {
     data->setParentHandle( this->handle() );
     data->addToLayout();
     fSetCanvasData->fAddCount++;
     added = true;
  }
//( *data )
// .saveMinimumSize( child->minimumSize() );    IC_NOTYET

  if ( added )
  {  // (This check prevents IToolBar::show from getting caught in a
     // recursive loop.)
     child->show();
     this->setLayoutDistorted( IWindow::layoutChanged, 0 );

     // Add the child window to a collection that we use to track the
     // windows that the canvas observer has been added to.
     fSetCanvasData->mSetcvsChildList.addAsFirst( child );

     // Add an ISetCanvasObserver to the child window, to enable the
     // canvas to detect changes in the alignment attribute of the
     // child window. Clean up code is located in ISetCanvas::remove,
     // the ISetCanvas destructor, and ISetCanvasObserver.
     ICanvasObserver* cObsrv = fSetCanvasData->pSetcvObserver;
     cObsrv->handleNotificationsFor( *child,
                                     IWindow::attributeAddReplaceId );
     cObsrv->handleNotificationsFor( *child,
                                     IWindow::attributeRemoveId );
     cObsrv->handleNotificationsFor( *child,
                                     IWindow::deleteId );
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::remove                                                           |
|                                                                              |
| Removes the specified child window from the windows layed out by the canvas. |
| This function also modifies the child window to remove it from view in the   |
| following ways:                                                              |
|  - hides the child window                                                    |
|  - sizes the child window to (0, 0)                                          |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::remove ( IWindow* child )
{
  IASSERTPARM( child->parent() == this );

  bool
    removed = false;
  if ( this->areExplicitAddsNeeded() )
  {  // Because this canvas only lays out child windows that have
     // explicitly been added (and therefore have set canvas window
     // data), we can simply delete any existing window data.
     ISetCanvasWindowData
      *data = (ISetCanvasWindowData*)( child->windowData( fClassDataHandle ) );
     if ( data  &&
          data->isAddedToLayout()  &&
          ( data->parentHandle() == this->handle() ) )
     {  // Deleting a previously-added child window.
        fSetCanvasData->fAddCount--;
        child->adoptWindowData( fClassDataHandle, 0 );
        removed = true;
     }
  }
  else
  {  // Because this canvas can lay out child windows that have not
     // explicitly been added (and therefore do not have set canvas
     // window data), we need to add window data that marks this
     // child window as removed.
     ISetCanvasWindowData
      *data = (ISetCanvasWindowData*)( child->windowData( fClassDataHandle ) );
     if ( ! data )
     {
        data = new ISetCanvasWindowData( this->handle() );
        child->adoptWindowData( fClassDataHandle, data );
        removed = true;
     }
     else
     {
        if ( data  &&
             data->isAddedToLayout()  &&
             ( data->parentHandle() == this->handle() ) )
        {  // Deleting a previously-added child window.
           fSetCanvasData->fAddCount--;
           removed = true;
        }

        data->setParentHandle( this->handle() );
     }
     data->removeFromLayout();
  }

  if ( removed )
  {  // Now remove the child window from the screen.  Don't modify
     // the child if we consider it already removed (this prevents
     // IToolBar::hide from getting caught in a recursive loop).
     ( *child )
//      .hide()
      .sizeTo( ISize() );
     this->setLayoutDistorted( IWindow::layoutChanged, 0 );

     // Remove the child window from the collection we use in the
     // destructor to clean up the notifications we've enabled in
     // ISetCanvas::add.
     ISetCanvasChildList::Cursor cursor(fSetCanvasData->mSetcvsChildList);
     for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
     {
     	  if ( fSetCanvasData->mSetcvsChildList.elementAt(cursor) == child )
        {
     	     fSetCanvasData->mSetcvsChildList.removeAt(cursor);
           break;
        }
     }

     // We no longer need to process notifications from the child window.
     ICanvasObserver* cObsrv = fSetCanvasData->pSetcvObserver;
     cObsrv->stopHandlingNotificationsFor( *child,
                                IWindow::attributeAddReplaceId );
     cObsrv->stopHandlingNotificationsFor( *child,
                                IWindow::attributeRemoveId );
     cObsrv->stopHandlingNotificationsFor( *child,
                                IWindow::deleteId );
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::replaceChildWith                                                 |
|                                                                              |
| Replaces the specified child window from the windows layed out by the canvas |
| with another child window.  When removing the first child window, this       |
| function also modifies it to remove it from view in the following ways:      |
|  - hides the child window                                                    |
|  - sizes the child window to (0, 0)                                          |
| This function assigns the new child window the following characteristics of  |
| the window it is replacing:                                                  |
|  - position in the z-order                                                   |
|  - tab stop and group settings                                               |
| Additionally, it shows the new child window.                                 |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::replaceChildWith ( IWindow* existingChild,
                                           IWindow* newChild )
{
  IASSERTPARM( existingChild->parent() == this );
     // (The above check is duplicated in ISetCanvas::remove, but we also
     // include it here to avoid any processing before the check gets made.
  IASSERTPARM( newChild->parent() == this );

  newChild->positionBehindSibling( existingChild->handle() );

  // Copy tab stop and group styles to new child window.
  (*newChild)
   .enableTabStop( existingChild->isTabStop() )
   .enableGroup( existingChild->isGroup() );

  this->remove( existingChild );
  this->add( newChild );

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::isInLayout                                                       |
|                                                                              |
| Identifies if the set canvas manages the size and position of the specified  |
| child window.  This function returns false if the child window has been      |
| removed or replaced, or if the canvas uses the explicitAddsNeeded styles and |
| the child window has not been added to the set canvas.                       |
------------------------------------------------------------------------------*/
bool ISetCanvas::isInLayout ( IWindow* child ) const
{
  IASSERTPARM( (const IWindow*)child->parent() == this );

  bool
    inLayout = true;

  ISetCanvasWindowData
   *data = (ISetCanvasWindowData*)( child->windowData( fClassDataHandle ) );

  if ( ! data )
  {
     if ( this->areExplicitAddsNeeded() )
     {  // The child window must have been explicitly added to the canvas.
        inLayout = false;
     }
     else
     {  // Compatibility mode: This canvas lays out child windows that
        // have not explicitly been added.  Create window data for the
        // child now to store layout information.
        data = new ISetCanvasWindowData( this->handle() );
        child->adoptWindowData( fClassDataHandle, data );
     }
  }
  else if ( data->parentHandle() != this->handle() )
  {  // Found old window data from a former parent window.
     if ( this->areExplicitAddsNeeded() )
     {  // Ignore this child window because it was not explicitly
        // added to this set canvas.  Delete the old window data.
        child->adoptWindowData( fClassDataHandle, 0 );
        inLayout = false;
     }
     else
     {  // Compatibility mode: This canvas lays out child windows that
        // have not explicitly been added.  Replace the old window
        // data, however.
        data = new ISetCanvasWindowData( this->handle() );
        child->adoptWindowData( fClassDataHandle, data );
     }
  }
  else if ( this->areExplicitAddsNeeded() )
  {
     if ( ! data->isAddedToLayout() )
     {
        inLayout = false;
     }
  }
  else if ( data->isRemovedFromLayout() )
  {  // Compatibility mode: This canvas lays out child windows that
     // have not explicitly been added.
     inLayout = false;
  }

  return inLayout;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setDeckOrientation                                               |
|                                                                              |
| Sets the orientation to be used for laying out the contents of the canvas    |
| (vertical vs. horizontal).                                                   |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setDeckOrientation ( DeckOrientation orValue )
{
  unsigned long ulExtStyle = this->extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;

  switch ( orValue )
  {
     case horizontal:
        ulExtStyle &= ~verticalDecks.asExtendedUnsignedLong();
        ulExtStyle |= horizontalDecks.asExtendedUnsignedLong();
        break;
     case vertical:
        ulExtStyle &= ~horizontalDecks.asExtendedUnsignedLong();
        ulExtStyle |= verticalDecks.asExtendedUnsignedLong();
        break;
     default:                          // Uninitialized enum
        ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                            IBaseErrorInfo::invalidParameter,
                            IException::recoverable );
        break;
  }

  if ( ulExtStyle != ulOldExtStyle )
  {                                    // Orientation change.
     this->setExtendedStyle( ulExtStyle );
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
                                       // Cause canvas to re-layout.
     this->notifyObservers( INotificationEvent( ISetCanvas::deckOrientationId,
                                                *this, true,
                                                (void*)orValue ));
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::deckOrientation                                                  |
|                                                                              |
| Returns the current orientation setting.                                     |
------------------------------------------------------------------------------*/
ISetCanvas::DeckOrientation ISetCanvas::deckOrientation ( ) const
{
  return( (extendedStyle() & verticalDecks.asExtendedUnsignedLong()) ? vertical
                                                                : horizontal );
}

/*------------------------------------------------------------------------------
| ISetCanvas::setPackType                                                      |
|                                                                              |
| Replaces the packing type to be used for laying out the contents of the      |
| canvas (tight vs. even vs. expanded).                                        |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setPackType ( PackType packValue )
{
  unsigned long ulExtStyle = this->extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;

  switch ( packValue )
  {
    case tight:
       ulExtStyle &= ~(packEven.asExtendedUnsignedLong() |
                       packExpanded.asExtendedUnsignedLong());
       ulExtStyle |= packTight.asExtendedUnsignedLong();
       break;
    case even:
       ulExtStyle &= ~(packTight.asExtendedUnsignedLong() |
                       packExpanded.asExtendedUnsignedLong());
       ulExtStyle |= packEven.asExtendedUnsignedLong();
       break;
    case expanded:
       ulExtStyle &= ~(packTight.asExtendedUnsignedLong() |
                       packEven.asExtendedUnsignedLong());
       ulExtStyle |= packExpanded.asExtendedUnsignedLong();
       break;
    default:                          // Uninitialized enum
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }

  if ( ulExtStyle != ulOldExtStyle )
  {                                    // Packing change.
     this->setExtendedStyle( ulExtStyle );
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
                                      // Cause canvas to re-layout.
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::packType                                                         |
|                                                                              |
| Returns the current packing type.                                            |
------------------------------------------------------------------------------*/
ISetCanvas::PackType ISetCanvas::packType ( ) const
{
  unsigned long ulExtStyle = this->extendedStyle();
  PackType packValue = tight;          // Default value.

  if ( ulExtStyle & packEven.asExtendedUnsignedLong() )
  {
     packValue = even;
  }
  else if ( ulExtStyle & packExpanded.asExtendedUnsignedLong() )
  {
     packValue = expanded;
  }
  return packValue;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setAlignment                                                     |
|                                                                              |
| Replaces the alignment used to position controls within their grids          |
| (topLeft, topCenter, topRight, centerLeft, etc).                             |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setAlignment ( Alignment alignValue )
{
  unsigned long ulExtStyle = this->extendedStyle();
  unsigned long ulOldExtStyle = ulExtStyle;

  switch ( alignValue )
  {
    case topLeft:
       ulExtStyle &= ~(centerVerticalAlign.asExtendedUnsignedLong() |
                       bottomAlign.asExtendedUnsignedLong()         |
                       centerAlign.asExtendedUnsignedLong()         |
                       rightAlign.asExtendedUnsignedLong());
       ulExtStyle |= (topAlign.asExtendedUnsignedLong() |
                      leftAlign.asExtendedUnsignedLong());
       break;
    case topCenter:
       ulExtStyle &= ~(centerVerticalAlign.asExtendedUnsignedLong() |
                       bottomAlign.asExtendedUnsignedLong()         |
                       leftAlign.asExtendedUnsignedLong()           |
                       rightAlign.asExtendedUnsignedLong());
       ulExtStyle |= (topAlign.asExtendedUnsignedLong() |
                      centerAlign.asExtendedUnsignedLong());
       break;
    case topRight:
       ulExtStyle &= ~(centerVerticalAlign.asExtendedUnsignedLong() |
                       bottomAlign.asExtendedUnsignedLong()         |
                       leftAlign.asExtendedUnsignedLong()           |
                       centerAlign.asExtendedUnsignedLong());
       ulExtStyle |= (topAlign.asExtendedUnsignedLong() |
                      rightAlign.asExtendedUnsignedLong());
       break;
    case centerLeft:
       ulExtStyle &= ~(topAlign.asExtendedUnsignedLong()    |
                       bottomAlign.asExtendedUnsignedLong() |
                       rightAlign.asExtendedUnsignedLong()  |
                       centerAlign.asExtendedUnsignedLong());
       ulExtStyle |= (centerVerticalAlign.asExtendedUnsignedLong() |
                      leftAlign.asExtendedUnsignedLong());
       break;
    case centerCenter:
       ulExtStyle &= ~(topAlign.asExtendedUnsignedLong()    |
                       bottomAlign.asExtendedUnsignedLong() |
                       rightAlign.asExtendedUnsignedLong()  |
                       leftAlign.asExtendedUnsignedLong());
       ulExtStyle |= (centerVerticalAlign.asExtendedUnsignedLong() |
                      centerAlign.asExtendedUnsignedLong());
       break;
    case centerRight:
       ulExtStyle &= ~(topAlign.asExtendedUnsignedLong()    |
                       bottomAlign.asExtendedUnsignedLong() |
                       centerAlign.asExtendedUnsignedLong() |
                       leftAlign.asExtendedUnsignedLong());
       ulExtStyle |= (centerVerticalAlign.asExtendedUnsignedLong() |
                      rightAlign.asExtendedUnsignedLong());
       break;
    case bottomLeft:
       ulExtStyle &= ~(centerVerticalAlign.asExtendedUnsignedLong() |
                       topAlign.asExtendedUnsignedLong()            |
                       rightAlign.asExtendedUnsignedLong()          |
                       centerAlign.asExtendedUnsignedLong());
       ulExtStyle |= (bottomAlign.asExtendedUnsignedLong() |
                      leftAlign.asExtendedUnsignedLong());
       break;
    case bottomCenter:
       ulExtStyle &= ~(centerVerticalAlign.asExtendedUnsignedLong() |
                       topAlign.asExtendedUnsignedLong()            |
                       rightAlign.asExtendedUnsignedLong()          |
                       leftAlign.asExtendedUnsignedLong());
       ulExtStyle |= (bottomAlign.asExtendedUnsignedLong() |
                      centerAlign.asExtendedUnsignedLong());
       break;
    case bottomRight:
       ulExtStyle &= ~(centerVerticalAlign.asExtendedUnsignedLong() |
                       topAlign.asExtendedUnsignedLong()            |
                       centerAlign.asExtendedUnsignedLong()         |
                       leftAlign.asExtendedUnsignedLong());
       ulExtStyle |= (bottomAlign.asExtendedUnsignedLong() |
                      rightAlign.asExtendedUnsignedLong());
       break;
    default:                          // Uninitialized enum
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }

  if ( ulExtStyle != ulOldExtStyle )
  {                                    // Alignment change.
     this->setExtendedStyle( ulExtStyle );
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0);
                                      // Cause canvas to re-layout.
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::alignment                                                        |
|                                                                              |
| Returns the current alignment of controls in the canvas.                     |
------------------------------------------------------------------------------*/
ISetCanvas::Alignment ISetCanvas::alignment ( ) const
{
  unsigned long ulExtStyle = this->extendedStyle();
  Alignment alignValue = topLeft;      // Default value.

  if ( ulExtStyle & centerVerticalAlign.asExtendedUnsignedLong() )
  {
     if ( ulExtStyle & centerAlign.asExtendedUnsignedLong() )
     {
        alignValue = centerCenter;
     }
     else if ( ulExtStyle & rightAlign.asExtendedUnsignedLong() )
     {
        alignValue = centerRight;
     }
     else                              // leftAlign.
     {
        alignValue = centerLeft;
     }
  }
  else if ( ulExtStyle & bottomAlign.asExtendedUnsignedLong() )
  {
     if ( ulExtStyle & centerAlign.asExtendedUnsignedLong() )
     {
        alignValue = bottomCenter;
     }
     else if ( ulExtStyle & rightAlign.asExtendedUnsignedLong() )
     {
        alignValue = bottomRight;
     }
     else                              // leftAlign.
     {
        alignValue = bottomLeft;
     }
  }
  else                                 // topAlign.
  {
     if ( ulExtStyle & centerAlign.asExtendedUnsignedLong() )
     {
        alignValue = topCenter;
     }
     else if ( ulExtStyle & rightAlign.asExtendedUnsignedLong() )
     {
        alignValue = topRight;
     }
     else                              // leftAlign.
     {
        alignValue = topLeft;
     }
  }

  return alignValue;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setDeckCount                                                     |
|                                                                              |
| Sets the maximum number of decks used by the canvas during layout.           |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setDeckCount ( unsigned long decks )
{
  IASSERTPARM( decks != 0 );

  if ( ulClNumDecks != decks )
  {                                    // Deck number change.
     ulClNumDecks = decks;
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
                                       // Cause canvas to re-layout.
  }
  return *this;
}


/*------------------------------------------------------------------------------
| ISetCanvas::alignDecksByGroup                                                |
|                                                                              |
| Turns on or off the decksByGroup style.                                      |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::alignDecksByGroup ( bool alignDecksByGroup )
{

  unsigned long currentExtendedStyle = this->extendedStyle();
  unsigned long newExtendedStyle = currentExtendedStyle;

  if ( alignDecksByGroup )
     newExtendedStyle |= decksByGroup.asExtendedUnsignedLong();
  else
     newExtendedStyle &= ~(decksByGroup.asExtendedUnsignedLong());

  if(newExtendedStyle != currentExtendedStyle)
  {
    this->setExtendedStyle(newExtendedStyle);
    this->setLayoutDistorted( IWindow::layoutChanged |
                            IWindow::immediateUpdate, 0 );
  }                                     // Cause canvas to re-layout.

  return *this;
}


/*------------------------------------------------------------------------------
| ISetCanvas::hasDecksAlignedByGroup                                           |
|                                                                              |
| Returns the value of the decksByGroup style.                                 |
------------------------------------------------------------------------------*/
bool ISetCanvas::hasDecksAlignedByGroup ( ) const
{

  unsigned long currentExtendedStyle = this->extendedStyle();
  if (currentExtendedStyle & decksByGroup.asExtendedUnsignedLong())
     return true;
  else
     return false;

}

/*------------------------------------------------------------------------------
| ISetCanvas::setMargin                                                        |
|                                                                              |
| Sets the margin width and height which is the space between the edge of the  |
| canvas and the outermost child window.                                       |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setMargin ( const ISize& margin )
{
  IASSERTPARM( margin.width() >= 0  &&  margin.height() >= 0 );

  if ( sizClMargin != margin )
  {                                    // New margin specified.
     sizClMargin = margin;
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
                                       // Cause canvas to re-layout.
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setPad                                                           |
|                                                                              |
| Sets the pad width and height which is the space between child windows in a  |
| deck and between mulitiple decks.                                            |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setPad ( const ISize& pad )
{
  IASSERTPARM( pad.width() >= -1  &&  pad.height() >= -1 );

  if ( sizClPad != pad )
  {                                    // New pad specified.
     sizClPad = pad;
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
                                       // Cause canvas to re-layout.
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setGroupPad                                                      |
|                                                                              |
| Sets the pad width or height (depending on the orientation of the canvas).   |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setGroupPad ( unsigned long pad )
{
  if ( ulClGroupPad != pad )
  {                                    // New pad specified.
     ulClGroupPad = pad;
     this->setLayoutDistorted( IWindow::layoutChanged |
                               IWindow::immediateUpdate, 0 );
                                       // Cause canvas to re-layout.
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::groupPad                                                         |
|                                                                              |
| Returns the group pad.                                                       |
------------------------------------------------------------------------------*/
unsigned long ISetCanvas::groupPad ( ) const
{
  return ulClGroupPad;
}

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
/*------------------------------------------------------------------------------
| ISetCanvas::setText                                                          |
|                                                                              |
| Causes a group box with the given text to be drawn around the canvas.  If    |
| necessary, the canvas is grown to fill the width of the group box.  If no    |
| text is specified, an existing group box is removed.                         |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setText ( const char* text )
{
  if ( text  &&  *text )
  {  // Caller specified text to be displayed.
     this->setBorderText( text, true );
  }
  else
  {
     this->setBorderText( IString() );
     this->removeBorder();
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setText                                                          |
|                                                                              |
| Adds/removes text to/from the canvas window.                                 |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setText ( const IResourceId& residText)
{
  IString aString = residText.resourceLibrary().loadString( residText );
  return this->setText( (char*)aString );
}

/*------------------------------------------------------------------------------
| ISetCanvas::text                                                             |
|                                                                              |
| Returns the text of the group box.                                           |
------------------------------------------------------------------------------*/
IString ISetCanvas::text ( ) const
{
  return this->borderText();
}
#endif // IC_OBSOLETE

/*------------------------------------------------------------------------------
| ISetCanvas::expandForText                                                    |
|                                                                              |
| Grows the layout size of the canvas if necessary to be able to include the   |
| group box text of the canvas and its border.                                 |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::expandForText ( )
{
  ISize sizLayout = this->layoutSize();
  this->setLayoutSize( this->sizeWithBorder( sizLayout ) );

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::hasChildrenToLayout                                              |
|                                                                              |
| Identify whether the canvas contains any child windows to manage.  The       |
| application must have passed these child windows to the add or               |
| replaceChildWith function, or not be using the explicitAddsNeeded style.     |
| Non-wrappered child windows are not considered to be managed by the canvas.  |
------------------------------------------------------------------------------*/
bool ISetCanvas::hasChildrenToLayout ( ) const
{
  bool
    childToManage = false;

  if ( fSetCanvasData->fAddCount )
  {  // Have at least one explicitly-added child window.
     // (As a performance enhancement for tool bar, disregard the
     // possibility that the application may have reparented or
     // deleted the child window without calling ISetCanvas::remove.
     // We will revalidate the count later in ISetCanvas::layout.)
     childToManage = true;
  }
  else if ( ! this->areExplicitAddsNeeded() )
  {  // Check for implicitly-added child windows.
     ISetCanvas
      *nonConstThis = (ISetCanvas*)this;
     ChildCursor
       cursor( *nonConstThis );
     for ( cursor.setToFirst();
           !childToManage  &&  cursor.isValid();
           cursor.setToNext() )
     {
        IWindow
         *childWindow = IWindow::childWindowAt( cursor );
        if ( this->isInLayout( childWindow ) )
        {
           childToManage = true;
        }
     }
  }
  return childToManage;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setLayoutDistorted                                               |
|                                                                              |
| Provides the response specific for a set canvas to various distortion        |
| notifications.  Some will cause the canvas to need to update how it has      |
| layed out its child windows.                                                 |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::setLayoutDistorted ( unsigned long layoutAttributeOn,
                                             unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = layoutAttributeOn;

  // If the caller must explicitly add child windows, ignore when it simply
  // creates the child windows.
  if ( this->areExplicitAddsNeeded() )
  {
     if ( ulFlagsOn & IWindow::childWindowCreated )
     {  // Ignore this case.
        ulFlagsOn &= (unsigned long)~( IWindow::childWindowCreated );

        // This check could be improved if a pointer to the child window were
        // passed on the call, in which case we could separate this from
        // childWindowCreated and also check whether the window with the new
        // minimum size is one that has been added to the set canvas.  IC_NOTYET
        if ( ulFlagsOn & IWindow::childMinimumSizeChanged )
        {  // Assume this is a window that has just been created, so not
           // yet added to the canvas.
           ulFlagsOn &= (unsigned long)~( IWindow::childMinimumSizeChanged );
        }
     }
  }

  unsigned long parentFlagsOn =
       ( ulFlagsOn & (unsigned long)~IWindow::childMinimumSizeChanged );

  // Ignore child minimum size changes only during a layout (during a
  // layout, assume these are a by-product of calling calcMinimumSize,
  // as is the case for a child set or multicell canvas).
  if ( ( ulFlagsOn & IWindow::childMinimumSizeChanged )  &&
       fSetCanvasData->layoutInProgress == 0 )
  {                                    // Change to child window size
     parentFlagsOn |= IWindow::layoutChanged;  // Flag used by ICanvas.
     if ( ! ( ulFlagsOn & IWindow::fontPropogated )  &&
          ! isLayoutDistorted( IWindow::fontPropogated ) )
     {                                 // Change buffered til now?
        parentFlagsOn |= IWindow::immediateUpdate;
     }
  }

  // If there is no presentation system window yet, avoid passing flags to
  // IWindow::setLayoutDistorted that require it to access the window handle.
  if ( ! this->isValid() )
  {
     parentFlagsOn &= (unsigned long)~( IWindow::immediateUpdate );
  }

  Inherited::setLayoutDistorted( parentFlagsOn, layoutAttributeOff );

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvas::layout                                                           |
|                                                                              |
| Layout management routine.                                                   |
------------------------------------------------------------------------------*/
ISetCanvas& ISetCanvas::layout ( )
{
  IMODTRACE_DEVELOP("ISetCanvas::layout()");

  // Don't run layout if there is no presentation system window yet, or
  // if there have been no changes to any of the layout variables.
  if ( this->isValid()  &&
       this->isLayoutDistorted( IWindow::layoutChanged ))
  {
     ITRACE_DEVELOP("isLayoutDistorted(layoutChanged)==true");
     fSetCanvasData->layoutInProgress++;

     /**************************************************************/
     /* The layout is distorted so calculate the position and size */
     /* of each of the child controls.                             */
     /*                                                            */
     /* For the time being, one of two simple rules is used to     */
     /* divide child controls among multiple decks: (1) The child  */
     /* controls are divied up to best keep the decks even in the  */
     /* number of controls in each, or (2) The child windows with  */
     /* the IWindow::group style are used to start new decks.      */
     /* Neither algorithm takes into consideration the size of the */
     /* child controls or how best to balance the decks in terms   */
     /* of screen size.                                            */
     /*                                                            */
     /* Note: Until the controls are actually repositioned, the    */
     /*       position value kept in the ISetCanvasObject elements */
     /*       are maintained in a sort of "flipped" state          */
     /*       (top-left corners are actually the bottom-left       */
     /*       corners and uppermost controls are placed at the     */
     /*       bottom of the canvas window), since the coordinate   */
     /*       system is treated as if it were:                     */
     /*                                                            */
     /*            0    5   10   15   20   25   30   35   40       */
     /*          0 +----+----+----+----+----+----+----+----+-      */
     /*            |                                               */
     /*            |                                               */
     /*         10 |                                               */
     /*                                                            */
     /*        All this is adjusted at the end.                    */
     /**************************************************************/

     /**************************************************************/
     /* Build a list of the canvas' child controls, as enumerated  */
     /* from PM.  This list will be used as a workspace for        */
     /* positioning and sizing the child controls.                 */
     /*                                                            */
     /* Note that child controls are enumerated in the z-order     */
     /* (which is the default order that they are layed out and    */
     /* tabbed through).                                           */
     /**************************************************************/
     ITRACE_DEVELOP("Start getting all child controls");
     ISize sizLayout = layoutSize();  // Save original size for children
     unsigned long
       ulChildCount = 0,
       addCount = 0,
       ulExtStyle = extendedStyle(),
       ulCountPerDeck = 0,
       ulCountLeft = 0;

     ISetCanvasObjectList* pscvClObjList = new ISetCanvasObjectList;

     IWindow::ChildCursor childCursor(*this);
     IWindowHandle whChild;
     for (childCursor.setToFirst();
          childCursor.isValid();
          childCursor.setToNext())
     {                                 // Enumerate all child controls
        whChild = childAt(childCursor);
        IWindow* pwinChild = IWindow::windowWithHandle(whChild);
        if ( pwinChild  &&  this->isInLayout( pwinChild ) )
        {                              // Have child control object
           ISetCanvasObject* pscvChild =
                                     new ISetCanvasObject(pwinChild);

           if (areChildrenReversed() == false)
           {                         // enum is in reverse layout order
              pscvClObjList->addAsFirst(pscvChild);
           }
           else
           {                           // enum is in layout order
              pscvClObjList->addAsLast(pscvChild);
           }
           ulChildCount++;

           if (ulExtStyle & decksByGroup.asExtendedUnsignedLong())
           {            // Use group style to divide decks.
              if (pwinChild->isGroup())
              {         // This starts a new deck.
                 pscvChild->setFirstInDeck();
                 ulChildCount = 1;
              }
              if (ulChildCount > ulCountPerDeck)
              {         // Track deck with most children.
                 ulCountPerDeck = ulChildCount;
              }
           }            // End decksByGroup.

           // Revalidate fAddCount, since this may not reflect reality
           // if the application has reparented or deleted a child
           // window without calling ISetCanvas::remove.
           ISetCanvasWindowData
            *data = (ISetCanvasWindowData*)
                      ( pwinChild->windowData( fClassDataHandle ) );
           if ( data  &&  data->isAddedToLayout() )
           {  // We found an explicitly added child window.
              addCount++;
           }
        }               // Else ignore non-wrappered child.
     }                  // End for all child windows.

     // Update the value used by ISetCanvas::hasChildrenToLayout.
     fSetCanvasData->fAddCount = addCount;

     if (!(ulExtStyle & decksByGroup.asExtendedUnsignedLong()))
     {                  // Divide decks evenly by child count.
        ulCountPerDeck = ulChildCount / ulClNumDecks;
        ulCountLeft = ulChildCount % ulClNumDecks;
     }

     ITRACE_DEVELOP("End getting all child controls");

     /**************************************************************/
     /* Start with an initial position and size for each.          */
     /**************************************************************/
     ITRACE_DEVELOP("Start layout pass 1");

     long lXMargin = sizClMargin.width();   // White space from edge
     long lYMargin = sizClMargin.height();  //   to outer controls
     IPoint
       topLeftCorner( this->topLeftLayoutOffset() );
     long
       leftBorderMargin = topLeftCorner.x(),
       topBorderMargin = topLeftCorner.y();
     lXMargin += leftBorderMargin;
     lYMargin += topBorderMargin;

     long lXPad = sizClPad.width();    // White space between elements
     long lYPad = sizClPad.height();
     long lNextX = lXMargin;           // Offset to first child control
     long lNextY = lYMargin;

     ISize szMax(0, 0);       // Biggest child so far (for pack expanded)
     long lCanvasWidth = 0;            // Size of the canvas window
     long lCanvasHeight = 0;

     bool bRowLayout = (deckOrientation() == horizontal);
     bool bFirstInDeck = true;
     bool bFirstDeck = true;

     ISetCanvasObject* pscvFirstInDeck = 0;  // For pack tight/expanded
     unsigned long ulCountInDeck = 0;
     long lDeckWidth = 0;
     long lMaxDeckLength = 0;

     unsigned long ulCount = ulCountPerDeck;  // For pack even

     if (ulCountLeft)              // Need number of windows in a deck
     {                             // Adjust for overflow in some decks
        ulCount++;
     } /* endif */

     unsigned long *pulDeckCellLengths = new unsigned long [ulCount];
     unsigned long *pulDeckCellLength = pulDeckCellLengths;
     for (; ulCount; ulCount--)
     {                                 // Zero out the array of lengths
        *pulDeckCellLength++ = 0;
     }
     pulDeckCellLength = pulDeckCellLengths;  // Reset to start

     long lNextLPos = lNextX;          // Initialize direction-independent
     long lNextWPos = lNextY;          //   variables assuming
     long lLMargin  = lXMargin;        //   horizontal decks
     long lLPad     = lXPad;
     long lWPad     = lYPad;

     if (!bRowLayout)
     {                                 // Reset direction-independent
        lNextLPos = lNextY;            //   variables for vertical decks
        lNextWPos = lNextX;
        lLMargin  = lYMargin;
        lLPad     = lYPad;
        lWPad     = lXPad;
     }

     /**************************************************************/
     /* Divide child windows across decks, storing the following   */
     /* information in the ISetCanvasObjectList:                   */
     /*  - Store the minimum size of each window.                  */
     /*  - Mark each child window that starts a deck.              */
     /*  - Store the position of each window based on whether it   */
     /*    starts a new deck or is part of the current deck.       */
     /*  - Update the size of the current deck, and the position   */
     /*    for the next child window in the current deck.          */
     /**************************************************************/
     ISetCanvasObjectList::Cursor cursor(*pscvClObjList);
     for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
     {                                 // Process all contents

        ISetCanvasObject* pscvObj = pscvClObjList->elementAt(cursor);
        ISize szObj = pscvObj->window()->minimumSize();
                                       // Get screen size for each control
        pscvObj->sizeTo(szObj);
        long lHeight = szObj.height(); // Size of the
        long lWidth = szObj.width();   //   child control
        long lLength = lHeight;        // Assume vertical decks

        if (bRowLayout)
        {                              // Horizontal decks
           lLength = lWidth;           // Initialize direction-independent
           lWidth  = lHeight;          //   variables
        } /* endif */

        /***********************************************************/
        /* Position control, adding appropriate white space.       */
        /***********************************************************/
        if (bFirstInDeck  ||  pscvObj->isFirstInDeck())
        {                              // Start of new deck
           pscvObj->setFirstInDeck();
           pscvFirstInDeck = pscvObj;
           lNextWPos += lDeckWidth;    // Move past last row
           lDeckWidth = 0;
           pulDeckCellLength = pulDeckCellLengths;  // New deck
           ulCountInDeck = 1;
           lNextLPos = lLMargin;
           if (bFirstDeck)
           {                           // First deck
              bFirstDeck = false;      // Unset it for next time
           }
           else
           {                           // Not first deck
              lNextWPos += lWPad;      // Need pad before this deck
           }
           bFirstInDeck = false;       // Reset for next time.
        }
        else
        {                              // Follows another child window
           ulCountInDeck++;
           lNextLPos += lLPad;
           if (pscvObj->window()->isGroup())
           {            // Add in group pad.
              lNextLPos += groupPad();
           }
        }

        IPoint pt;
        if (bRowLayout)
        {                              // Horizontal decks
           pt.setX(lNextLPos).setY(lNextWPos);
        }
        else
        {                              // Vertical decks
           pt.setX(lNextWPos).setY(lNextLPos);
        }
        pscvObj->moveTo(pt);

        lNextLPos += lLength;          // Position for next child window
        if(lLength > *pulDeckCellLength)
        {                              // Length of the window
           *pulDeckCellLength = lLength;
        }

        if(lWidth > lDeckWidth)
        {                              // Track thickness of the deck
           lDeckWidth = lWidth;
           pscvFirstInDeck->setDeckWidth(lDeckWidth);
        }

        if (lNextLPos > lMaxDeckLength)
        {                              // Track longest deck
           lMaxDeckLength = lNextLPos;
        }
        pulDeckCellLength++;   // For the next window in the deck.

        if (!(ulExtStyle & decksByGroup.asExtendedUnsignedLong()))
        {               // Divide decks evenly by child count.
           if ((ulCountInDeck == ulCountPerDeck  &&
                ulCountLeft == 0)  ||
               ulCountInDeck > ulCountPerDeck)
           {            // Last child window in the deck.
              bFirstInDeck = true;   // Set flag for new row.
           }
           else if (ulCountInDeck == ulCountPerDeck)
           {            // Extra control for the deck.
              ulCountLeft--;
           }
        }

        szMax = szMax.maximum(szObj);  // New biggest control
     }                                 // End for all child controls
     lNextWPos += lDeckWidth;          // Move past last row.

     ITRACE_DEVELOP("End layout pass 1");

     /**************************************************************/
     /* Process specified pack and alignment styles now:           */
     /* - pack tight                                               */
     /*   Child windows are already packed tight, so the only      */
     /*   adjustment needed is aligning within the deck            */
     /*   (vertically in a horizontal deck, horizontally in a      */
     /*   vertical deck).                                          */
     /* - pack even                                                */
     /*   Child windows are aligned within the deck, as done for   */
     /*   pack tight, as well as horizontally in a horizontal deck */
     /*   and vertically in a vertical deck since each cell in a   */
     /*   deck may be expanded.                                    */
     /* - pack expanded                                            */
     /*   Child windows are resized to the largest child window,   */
     /*   and repositioned accordingly.  No alignment can be done  */
     /*   since everything is of "maximum" size.                   */
     /**************************************************************/
     if (bRowLayout)
     {                                 // Horizontal decks
        lNextX = lNextLPos;            // Map direction-independent
        lNextY = lNextWPos;            //   variables back out
     }
     else
     {                                 // Vertical decks
        lNextX = lNextWPos;            // Map direction-independent
        lNextY = lNextLPos;            //   variables back out
     }

     if (packType() == tight  ||  packType() == even ||
         packType() == expanded)
     {
        ITRACE_DEVELOP("Start layout pass 2 -- packType()==tight/even/expanded");

        unsigned long ulDeckWidth=0;
        unsigned long ulDeckHeight=0;
        unsigned long ulWidth=0;
        unsigned long ulHeight=0;
        unsigned long ulYDivisor=0;    // For top align
        unsigned long ulXDivisor=0;    // For left Align
        long lWidth=0;                 // For expanded
        long lHeight=0;

        if (packType() == expanded)
        {
          lWidth = szMax.width();   // Use largest for size
          lHeight = szMax.height(); //   of all child controls
          bFirstDeck = true;
        }
        else if ( (packType() == even) ||
                  ( (packType() == tight)  &&
                    (!(ulExtStyle & topAlign.asExtendedUnsignedLong())) ) )
        {
           // Need to reposition in a horizontal deck

           if (ulExtStyle & bottomAlign.asExtendedUnsignedLong())
           {                           // Need to adjust fully down
              ulYDivisor = 1;
           }
           else if (ulExtStyle & centerVerticalAlign.asExtendedUnsignedLong())
           {                           // Need to adjust halfway
              ulYDivisor = 2;
           }

           if (ulExtStyle & rightAlign.asExtendedUnsignedLong())
           {                           // Need to adjust fully over
              ulXDivisor = 1;
           }
           else if (ulExtStyle & centerAlign.asExtendedUnsignedLong())
           {                           // Need to adjust halfway
              ulXDivisor = 2;
           }
        }
        if (bRowLayout)                   // Layout everything in rows
        {
           for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
           {                           // Process all contents
              ISetCanvasObject* pscvObj =
                                    pscvClObjList->elementAt(cursor);

              /*************************************************************/
              /* Query window for possible attributes                      */
              /*************************************************************/
              const IAttribute* attr;
              const IAlignmentAttribute* algnAttr;
              bool algnAttrFound = false;
              IWindow* win = (IWindow*)pscvObj->window();
              IWindow::AttributeCursor cursor (*win);
              algnAttr = (const IAlignmentAttribute*) 0;
              for (cursor.setToFirst();
                  !algnAttrFound && cursor.isValid();
                   cursor.setToNext())
              {
                 IAttributeName attrName = win->attributeNameAt(cursor);
                 attr = win->attributeWithName(attrName);
                 algnAttr = dynamic_cast<const IAlignmentAttribute*>(attr);
                 if (algnAttr)
                    algnAttrFound = true;
              }
              if (packType() == expanded)
                 pscvObj->sizeTo(szMax);  // New bigger size

              /*****************************************************/
              /* Position control, adding appropriate white space. */
              /*****************************************************/
              if (pscvObj->isFirstInDeck())
              {
                 lNextX = lXMargin;         // For pack even only

                 if (packType() == expanded)
                 {
                    if (bFirstDeck)
                    {                       // First deck
                       lNextY = lYMargin;   // Offset to first row
                       bFirstDeck = false;  // Unset it for next time
                    }
                    else
                    {                                // Not first deck
                       lNextY += (lHeight + lYPad);  // Offset to row
                    }
                 }
                 else
                 {                          // Deck height stored here
                    ulDeckHeight = pscvObj->deckWidth();
                    pulDeckCellLength = pulDeckCellLengths;
                 }
              }
              else
              {                        // Preceding entry in deck
                 lNextX += lXPad;      //   for pack even only
              }

              // Process window alignment attribute
              if (algnAttr)
              {                        // Horizontal deck with alignment
                 unsigned long ulAlXDivisor, ulAlYDivisor;
                 ulAlXDivisor = ulAlYDivisor = 0;
                 if (algnAttr->horizontalAlignment() ==
                      IAlignmentAttribute::kRight)
                    ulAlXDivisor = 1;
                 else if (algnAttr->horizontalAlignment() ==
                      IAlignmentAttribute::kCenter)
                    ulAlXDivisor = 2;
                 else
                    ulAlXDivisor = 0;

                 if (algnAttr->verticalAlignment() ==
                      IAlignmentAttribute::kBottom)
                    ulAlYDivisor = 1;
                 else if (algnAttr->verticalAlignment() ==
                      IAlignmentAttribute::kCenterVertical)
                    ulAlYDivisor = 2;
                 else
                    ulAlYDivisor = 0;

                 // Hor pos.
                 if (packType() == expanded)
                    ulWidth = lWidth;
                 else
                    ulWidth = *pulDeckCellLength++;
                 unsigned long ulDelta =
                    (ulAlXDivisor ? ((ulWidth -
                                      pscvObj->size().width())
                                          / ulAlXDivisor) : 0);
                 if ( (packType() == expanded) || (packType() == even) )
                 {
                    pscvObj->moveLeftTo(lNextX + ulDelta);
                    lNextX += ulWidth;
                    if (lNextX > lMaxDeckLength)
                    {
                       lMaxDeckLength = lNextX;
                    }
                 }
                 else
                 {
                    pscvObj->moveLeftTo(pscvObj->topLeft().x() + ulDelta);
                 }

                 // Ver pos.
                 if (ulAlYDivisor)
                 {
                    if (packType() == expanded)
                       pscvObj->moveTopTo(lNextY);
                    else
                       pscvObj->moveTopTo(pscvObj->topLeft().y() +
                                       (ulDeckHeight -
                                        pscvObj->size().height())
                                               / ulAlYDivisor);
                 }
              }
              else
              {
                 if ( ( (packType() == tight) || (packType() == even) ) &&
                      (ulYDivisor) )
                 {                   // Vertically reposition child window
                    pscvObj->moveTopTo(pscvObj->topLeft().y() +
                                       (ulDeckHeight -
                                        pscvObj->size().height())
                                                     / ulYDivisor);
                 }

                 if (packType() == even)
                 {                   // Horizontally reposition child window
                    ulWidth = *pulDeckCellLength++;
                    unsigned long ulDelta =
                        (ulXDivisor ? ((ulWidth -
                                         pscvObj->size().width())
                                                     / ulXDivisor) : 0);
                    pscvObj->moveLeftTo(lNextX + ulDelta);
                    lNextX += ulWidth;    // For next child window

                    if (lNextX > lMaxDeckLength)
                    {                     // Track longest row
                       lMaxDeckLength = lNextX;
                    }
                 }
                 if (packType() == expanded)
                 {                         // Horizontally reposition child window
                    pscvObj->moveTo(IPoint(lNextX, lNextY));
                    lNextX += lWidth;        // For next control
                    if(lNextX > lMaxDeckLength)
                    {                        // Track longest row
                       lMaxDeckLength = lNextX;
                    }
                 }
              }
           }                              // End all contents
           if (packType() == expanded)
              lNextY += lHeight;          // Bottom of last row
        }                                 // End vertical aligning
        else                              // Layout everything in columns
        {
           if ( (packType() == even) ||
                ( (packType() == tight) &&
                  (!(ulExtStyle & leftAlign.asExtendedUnsignedLong())) ) )
           {
              ulDeckWidth = 0;
              ulHeight = 0;
              ulXDivisor = 0;  // For left align
              ulYDivisor = 0;  // For top align

              if (ulExtStyle & rightAlign.asExtendedUnsignedLong())
              {                           // Need to adjust fully over
                 ulXDivisor = 1;
              }
              else if (ulExtStyle & centerAlign.asExtendedUnsignedLong())
              {                           // Need to adjust halfway
                 ulXDivisor = 2;
              }
              if (ulExtStyle & bottomAlign.asExtendedUnsignedLong())
              {                           // Need to adjust fully down
                 ulYDivisor = 1;
              }
              else if (ulExtStyle & centerVerticalAlign.asExtendedUnsignedLong())
              {                           // Need to adjust halfway
                 ulYDivisor = 2;
              }
           }
           for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
           {                           // Process all contents
              ISetCanvasObject* pscvObj =
                                  pscvClObjList->elementAt(cursor);
              const IAttribute* attr;
              const IAlignmentAttribute* algnAttr;
              bool algnAttrFound = false;
              IWindow* win = (IWindow*)pscvObj->window();
              IWindow::AttributeCursor cursor (*win);
              algnAttr = (const IAlignmentAttribute*) 0;
              for (cursor.setToFirst();
                  !algnAttrFound && cursor.isValid();
                   cursor.setToNext())
              {
                 IAttributeName attrName = win->attributeNameAt(cursor);
                 attr = win->attributeWithName(attrName);
                 algnAttr = dynamic_cast<const IAlignmentAttribute*>(attr);
                 if (algnAttr)
                    algnAttrFound = true;
              }
              if (packType() == expanded)
                 pscvObj->sizeTo(szMax);    // New bigger size

              /*****************************************************/
              /* Position control, adding appropriate white space. */
              /*****************************************************/
              if (pscvObj->isFirstInDeck())
              {
                 lNextY = lYMargin;         // For pack even only
                 if (packType()==expanded)
                 {
                    if (bFirstDeck)
                    {                       // First deck
                       lNextX = lXMargin;   // Offset to first column
                       bFirstDeck = false;  // Unset it for next time
                    }
                    else
                    {                               // Not first deck
                       lNextX += (lWidth + lXPad);  // Offset to column
                    }
                 }
                 else
                 {                           // Deck width stored here
                    pulDeckCellLength = pulDeckCellLengths;
                    ulDeckWidth = pscvObj->deckWidth();
                 }
              }
              else
              {                        // Preceding entry in deck
                 lNextY += lYPad;      // For pack even only
              }
              // Process window alignment attribute
              if (algnAttr)
              {                           // Horizontal deck with alignment
                 unsigned long ulAlXDivisor, ulAlYDivisor;
                 ulAlXDivisor = ulAlYDivisor = 0;
                 if (algnAttr->horizontalAlignment() ==
                      IAlignmentAttribute::kRight)
                    ulAlXDivisor = 1;
                 else if (algnAttr->horizontalAlignment() ==
                      IAlignmentAttribute::kCenter)
                    ulAlXDivisor = 2;
                 else
                    ulAlXDivisor = 0;

                 if (algnAttr->verticalAlignment() ==
                      IAlignmentAttribute::kBottom)
                    ulAlYDivisor = 1;
                 else if (algnAttr->verticalAlignment() ==
                      IAlignmentAttribute::kCenterVertical)
                    ulAlYDivisor = 2;
                 else
                    ulAlYDivisor = 0;

                 // Ver pos.
                 if (packType() == expanded)
                    ulHeight = lHeight;
                 else
                    ulHeight = *pulDeckCellLength++;
                 unsigned long ulDelta =
                    (ulAlYDivisor ? ((ulHeight -
                                      pscvObj->size().height())
                                          /ulAlYDivisor) : 0);
                 if ( (packType() == even) || (packType() == expanded) )
                 {
                    pscvObj->moveTopTo(lNextY + ulDelta);
                    lNextY += ulHeight;
                    if (lNextY > lMaxDeckLength)
                    {
                       lMaxDeckLength = lNextY;
                    }
                 }
                 else
                 {
                    pscvObj->moveTopTo(pscvObj->topLeft().y() + ulDelta);
                 }

                 // Hor pos.
                 if (ulAlXDivisor)
                 {
                    if (packType() == expanded)
                       pscvObj->moveLeftTo(lNextX);
                    else
                       pscvObj->moveLeftTo(pscvObj->topLeft().x() +
                                    (ulDeckWidth -
                                     pscvObj->size().width())
                                                 / ulAlXDivisor);
                 }
              }
              else
              {
                 if ( ( (packType() == tight) || (packType() == even) ) &&
                      (ulXDivisor) )
                 {                    // Horizontally reposition child window
                    pscvObj->moveLeftTo(pscvObj->topLeft().x() +
                                        (ulDeckWidth -
                                         pscvObj->size().width())
                                                     / ulXDivisor);
                 }

                 if (packType() == even)
                 {                      // Vertically reposition child window
                    ulHeight = *pulDeckCellLength++;
                    unsigned long ulDelta =
                       (ulYDivisor ? ((ulHeight -
                                       pscvObj->size().height())
                                                     / ulYDivisor) : 0);
                    pscvObj->moveTopTo(lNextY + ulDelta);
                    lNextY += ulHeight;   // For next child window

                    if (lNextY > lMaxDeckLength)
                    {                     // Track longest column
                       lMaxDeckLength = lNextY;
                    }
                 }
                 if (packType() == expanded)
                 {
                    pscvObj->moveTo(IPoint(lNextX, lNextY));
                    lNextY += lHeight;       // For next control
                    if (lNextY > lMaxDeckLength)
                    {                        // Track longest column
                       lMaxDeckLength = lNextY;
                    }
                 }
              }
           }                           // End all contents
           if (packType()==expanded)
              lNextX += lWidth;        // End of last column
        }                              // End horizontal aligning

        ITRACE_DEVELOP("End layout pass 2");
     }                                 // End pack tight/even/expanded

     /**************************************************************/
     /* Have size of the canvas window.                            */
     /**************************************************************/
     if (bRowLayout)
     {                                 // Lay out everything in rows
        lNextX = lMaxDeckLength;
     }
     else
     {                                 // Lay out everything in columns
        lNextY = lMaxDeckLength;
     }

     /**************************************************************/
     /* If this is a group box, need to adjust size to include the */
     /* width of the text and grow the height to not overwrite any */
     /* child controls with the text.  Also expand to include the  */
     /* group box text.                                            */
     /**************************************************************/
     lCanvasWidth = lNextX + lXMargin - ( 2 * leftBorderMargin );
     lCanvasHeight = lNextY + lYMargin - ( 2 * topBorderMargin );
        // (ISetCanvas::expandForText will add the border sizes back in.)

     setLayoutSize(ISize(lCanvasWidth, lCanvasHeight));
     expandForText();   // Grow sizClLayout if necessary.
     ISize newLayoutSize( this->layoutSize() );

     ITRACE_DEVELOP( "Set canvas minimum size = (" +
                     IString( newLayoutSize.width() ) + ", " +
                     IString( newLayoutSize.height() ) + ")" );

     // Keep child windows at the top of the set canvas.
     lCanvasHeight = this->size().height();
#ifdef IC_ORIGINLOWERLEFT
     // Position child windows based on the current height of the
     // canvas, unless it has no height.  In this case, position
     // them based on the calculated minimum height to optimize the
     // case when/if the canvas is next sized to its minimum height.
     if ( lCanvasHeight == 0 )
     {
        lCanvasHeight = newLayoutSize.height();
     }
     fSetCanvasData->positioningHeight = lCanvasHeight;
#endif

     /**************************************************************/
     /* Size and position each child control, and clean up the     */
     /* sequential set.                                            */
     /**************************************************************/
     ITRACE_DEVELOP("Start layout pass 3 -- move/size controls");

#ifdef IC_PMWIN
     unsigned long style = this->style();
     if ( (style & WS_VISIBLE ))
       this->enableUpdate( false );      // Surpress screen updates.
#endif

     IWindowPosBuffer wposbuf = fixupChildren();
     for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
     {                                 // Process all contents
        ISetCanvasObject* pscvObj = pscvClObjList->elementAt(cursor);

#ifdef IC_ORIGINLOWERLEFT
        // Convert offsets from top-relative to native bottom-relative.
        pscvObj->moveTopTo(lCanvasHeight -
                           pscvObj->topLeft().y() -
                           pscvObj->size().height());
                                       // Adjust coordinate system
#endif

        // Adjust the control if necessary (primarily drop-down combos
        IRectangle rectAdjust = pscvObj->window()->layoutAdjustment();

        pscvObj->sizeTo(pscvObj->size()+rectAdjust.size());
#ifdef IC_ORIGINLOWERLEFT
        // With bottom-relative offsets, growing a child window
        // means moving its bottom edge down.
        pscvObj->moveTo(pscvObj->rect().bottomLeft()+rectAdjust.bottomLeft());
#endif

        wposbuf.moveSizeTo((IWindow*)(pscvObj->window()),
                           pscvObj->rect());  // Cast away const
     }                                 // End all contents

     pscvClObjList->removeAll(&ISetCanvasDeleter);
     delete pscvClObjList;
     wposbuf.apply();                  // Reposition/size child controls

     ITRACE_DEVELOP("End layout pass 3");

     setLayoutDistorted(0, IWindow::layoutChanged);

     /**************************************************************/
     /* The below follows setLayoutDistorted(), since calling      */
     /* WinEnableWindowUpdate may cause another WM_PAINT message   */
     /* and we don't want an infinite loop problem.                */
     /**************************************************************/
#ifdef IC_PMWIN
     if ( ( style & WS_VISIBLE ) )
     {
       // enable update will show the window so only call it if the window is
       // visible
       this->enableUpdate( true );       // Show screen updates.
     }
#endif

     setLayoutSize(newLayoutSize);     // Reset after fixupChildren.
     if (sizLayout != newLayoutSize)   // New minimum size for children.
     {                                 // Cause layout in parent canvas
        setLayoutDistorted(IWindow::minimumSizeChanged, 0);
     }
     delete (pulDeckCellLengths);

     fSetCanvasData->layoutInProgress--;
  }   /* end isDistorted */

  return *this;
}

/*------------------------------------------------------------------------------
| ISetCanvasResizeHandler::windowResize                                        |
|                                                                              |
| Causes the canvas to run its layout routine if necessary, based on a new     |
| size of the canvas.                                                          |
------------------------------------------------------------------------------*/
bool ISetCanvasResizeHandler::windowResize ( IResizeEvent& event )
{
  ISetCanvas* window = (ISetCanvas*)event.dispatchingWindow();
  bool layoutNeeded = false;
#ifdef IC_ORIGINLOWERLEFT
  long newHeight = event.newSize().height();
  if ( newHeight  &&
       newHeight != event.oldSize().height()  &&
       newHeight != window->fSetCanvasData->positioningHeight )
  {  // Force layout to run, since the canvas must update the y positions
     // of its child windows (the child windows not already positioned for
     // the new height).
     layoutNeeded = true;
  }
  else
#endif
  if ( IBidiSettings::isBidiSupported()  &&
       event.newSize().width() != event.oldSize().width() )
  {
     IBidiSettings settings( *window );
     if ( settings.windowLayout() == IBidiSettings::layoutRightToLeft )
     {
        layoutNeeded = true;
     }
  }

  if ( layoutNeeded )
  {
     window->setLayoutDistorted( IWindow::layoutChanged, 0 );
     window->postEvent( IC_UM_CANVAS_PAINT );
     // (Can't use the immediateUpdate flag or IWindow::refresh, since for
     // some reason these don't cause a toolbar container to get a WM_PAINT
     // message it needs.)
  }

  return false;
}

/*------------------------------------------------------------------------------
| ISetCanvas::setDefaultStyle                                                  |
|                                                                              |
| Sets the default style                                                       |
|                                                                              |
| Note: This function is inlined on all platforms except Windows due to        |
|       variables it references not being exported.                            |
|------------------------------------------------------------------------------*/
#ifdef IC_WIN
void  ISetCanvas::setDefaultStyle ( const ISetCanvas::Style& style )
  {
  currentDefaultStyle = style;
  }
#endif //IC_WIN

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ISetCanvasShowHandler::dispatchHandlerEvent                                  |
|                                                                              |
| This is motif specific because we don't get the first WM_RESIZE message.     |
| In motif we mapped the configureNotify event to the WM_RESIZE when we detect |
| the size has changed but the configureNotify event is not generated the      |
| first time a window is created.                                              |
------------------------------------------------------------------------------*/
bool ISetCanvasShowHandler::dispatchHandlerEvent ( IEvent& event )
{
  if ( event.eventId() == xEvent(MapNotify) &&
         event.parameter1().asUnsignedLong() == true )
  {
    ITRACE_DEVELOP("ISetCanvas - First time shown");
    IWindow* window = event.controlWindow();
    if (window )
    {                // Canvas needs to update y positions of child windows.
      window->setLayoutDistorted(IWindow::layoutChanged |
                                   IWindow::immediateUpdate, 0);
    }
  }

  return false;
}
#endif

ISetCanvasObserver& ISetCanvasObserver :: dispatchNotificationEvent ( const INotificationEvent& anEvent )
{
  INotificationId id(anEvent.notificationId());
  // Check the notification for ISetCanvas's ChildWindow
  // If it is, send the stopHandlingNotificationsFor to the particular Child Window
  if ( id == IWindow::deleteId )
  {
  	IWindow& window = (IWindow&)anEvent.notifierAddress().notifier();
  	ISetCanvasChildList::Cursor cursor(pSetCanvas->fSetCanvasData->mSetcvsChildList);
  	for ( cursor.setToFirst();cursor.isValid();cursor.setToNext() )
  	{
  		if ( pSetCanvas->fSetCanvasData->mSetcvsChildList.elementAt(cursor) == &window)
  		{
  			stopHandlingNotificationsFor(window,IWindow::attributeRemoveId);
  			stopHandlingNotificationsFor(window,IWindow::attributeAddReplaceId);
  			stopHandlingNotificationsFor(window,IWindow::deleteId);
  			pSetCanvas->fSetCanvasData->mSetcvsChildList.removeAt(cursor);
  			break;
  		}
  	}
  }
  else
  	Inherited :: dispatchNotificationEvent(anEvent);
  return *this;
 }
