// Revision: 32 1.30.1.1 source/ui/basectl/iscroll.cpp, slidectls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iscroll.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iscroll.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

extern "C" {
  #define INCL_WINERRORS
  #define INCL_WINMESSAGEMGR
  #define INCL_WINSCROLLBARS
  #define INCL_WINSYS
  #define INCL_WINWINDOWMGR
  #include <iwindefs.h>
}

#ifdef IC_MOTIF
  #include <Xm/Form.h>
  #include <Xm/ScrollBar.h>
  #include <Xm/ScrolledW.h>
  #include <Xm/MainW.h>
#endif // IC_MOTIF

#include <iscroll.hpp>
#include <icconst.h>
#include <icolor.hpp>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ihandle.hpp>
#include <ihandler.hpp>
#include <iinhratt.hpp>
#include <inotifev.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#ifdef IC_MOTIF
  #include <iapp.hpp>
  #include <icolmap.hpp>
  #include <iwindow.hpp>
  #include <ixdc.hpp>
#endif
#ifdef IC_WIN
  #include <ibrushdr.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ISCROLL_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_MOTIF
#pragma info(nouse)
void _System valueChangedCallback( Widget widget,
                                   XtPointer client_data,
                                   XtPointer call_data )
{
  if ( XtIsSubclass( widget, xmScrollBarWidgetClass ) )
  {
    IWindow *win = IWindow::windowWithHandle( widget );
    if ( win )
    {
      XmScrollBarCallbackStruct
        *scrollBarStruct = (XmScrollBarCallbackStruct *) call_data;
      switch (scrollBarStruct->reason)
      {
        case XmCR_INCREMENT:
        case XmCR_DECREMENT:
        case XmCR_DRAG:
        case XmCR_PAGE_DECREMENT:
        case XmCR_PAGE_INCREMENT:
        case XmCR_TO_BOTTOM:
        case XmCR_TO_TOP:
        case XmCR_VALUE_CHANGED:
          ((IScrollBar *) win)->setPrevScrollBoxPosition(
            scrollBarStruct->value );
        break;
      }
    }
  }
}
#pragma info(restore)
#endif // IC_MOTIF

#ifdef IC_MOTIF

#pragma enum(4)
#pragma pack(push,4)

class IScrollBarData {
public:
  IScrollBarData ( )
    : cClShaftColor( IColor::white ),
      cClDefaultShaftColor( IColor::white ),
      ulClPrevScrollBoxPosition( 0 ),
      ulClMinScrollIncrement( 1 ),
      ulClPageScrollIncrement( 0 ),
      width( 0 ),
      height( 0 ),
      vertical( false )
    {  }
 ~IScrollBarData ( )
    {  }
void
  initialize ( IScrollBar*       scrollBar,
               unsigned long     windowId,
               IWindow*          parent,
               IWindow*          owner,
               const IRectangle& initial,
               const IRange&     scrollableItems,
               unsigned long     visibleItemCount,
               unsigned long     scrollBoxPosition,
               unsigned long     ulmStyle );
IScrollBarData
 &wrapperInitialization ( IWindowHandle wh,
                          IScrollBar*   scrollBar );
void
  registerCallbacks     ( IScrollBar*   scrollBar );
void
  unregisterCallbacks   ( IScrollBar*   scrollBar );

IColor
  cClShaftColor,
  cClDefaultShaftColor;
unsigned long
  ulClPrevScrollBoxPosition,
  ulClMinScrollIncrement,
  ulClPageScrollIncrement;
Dimension
  width,
  height;
bool
  vertical;
private:
  IScrollBarData ( const IScrollBarData& );
IScrollBarData
 &operator=      ( const IScrollBarData& );
}; // IScrollBarData

#pragma pack(pop)
#pragma enum(pop)

#endif // IC_MOTIF

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| Default handler class for IScrollBar                                         |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IScrollBarHandler : public IHandler {
typedef IHandler
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
  IScrollBarHandler ( );

virtual
 ~IScrollBarHandler ( );

/*----------------------------- Event Processing -----------------------------*/
virtual bool
  dispatchHandlerEvent ( IEvent& event );
}; // class IScrollBarHandler

/*------------------------------------------------------------------------------
| Private data class for IScrollBar                                            |
------------------------------------------------------------------------------*/
class IScrollBarData {
public:
  IScrollBarData ( );

 ~IScrollBarData ( );

unsigned long
  fvisibleCount;

IScrollBarHandler
  fdefaultHandler;
IBrushHandler
  fBrushHandler;
private:
  IScrollBarData ( const IScrollBarData& );
IScrollBarData
 &operator=      ( const IScrollBarData& );
}; // IScrollBarData

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IScrollBarData::IScrollBarData                                               |
|                                                                              |
------------------------------------------------------------------------------*/
IScrollBarData::IScrollBarData ( )
  : fdefaultHandler( )
  , fBrushHandler( )
{ }

/*------------------------------------------------------------------------------
| IScrollBarData::~IScrollBarData                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IScrollBarData::~IScrollBarData ( )
{ }

/*------------------------------------------------------------------------------
| IScrollBarHandler::IScrollBarHandler                                         |
|                                                                              |
| Empty constructor here for page tuning.                                      |
------------------------------------------------------------------------------*/
IScrollBarHandler::IScrollBarHandler()
{ }

/*------------------------------------------------------------------------------
| IScrollBarHandler::~IScrollBarHandler                                        |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IScrollBarHandler::~IScrollBarHandler()
{ }

/*------------------------------------------------------------------------------
| IScrollBarHandler::dispatchHandlerEvent                                      |
|                                                                              |
| Intercepts control color event                                               |
|    This handler returns false for all other events.                          |
------------------------------------------------------------------------------*/
bool IScrollBarHandler::dispatchHandlerEvent( IEvent& event )
{
  IMODTRACE_ALL("IScrollBarHandler::dispatchHandlerEvent");
  bool
    result = false;

  switch( event.eventId() )
  {
    // Code to trap a mnemonic key press and send it directly to
    // the owner/parent window (allowing it to naturally progress
    // through the dispatcher results in an audible beep caused
    // by the default window procedure, because it cannot find a
    // match for a mnemonic that we process in ICanvas or
    // IFrameWindow), eating original event.
    case WM_SYSCHAR:
    {
      IWindow* window = event.window();
      if (window)
      {
         IWindow* owner = window->owner();
         if ( ! owner )
         {
            owner = window->parent();
         }
         if ( owner )
         {
            owner->sendEvent( event );
            result = true;
         }
      }
      break;
    }

    case WM_LBUTTONDOWN:
      // Need to prevent _pfnwpICWinProc from propagating this event up
      // the owner chain, since the scroll bar window proc does not
      // finish processing this message until the mouse button has been
      // released.  This can cause much havoc on an ISplitCanvas up the
      // owner chain.
      this->defaultProcedure( event );
      result = true;
      break;

    default:
      break;
  }
  return result;
}

#endif //IC_WIN


/*------------------------------------------------------------------------------
| Public scroll bar styles.                                                    |
------------------------------------------------------------------------------*/
const IScrollBar::Style
  IScrollBar::horizontal        ( 0, ISBS_HORIZONTAL ),
  IScrollBar::vertical          = SBS_VERT,
  IScrollBar::classDefaultStyle = WS_VISIBLE | SBS_VERT;

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
IScrollBar::Style
  IScrollBar::currentDefaultStyle ( WS_VISIBLE | SBS_VERT );


/*------------------------------------------------------------------------------
| IScrollBar::systemScrollBarWidth                                             |
|                                                                              |
| Static system value function.                                                |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: systemScrollBarWidth ( bool bVertical )
{
#ifdef IC_PMWIN
  return IQUERYSYSVALUE( bVertical ? SV_CXVSCROLL : SV_CYHSCROLL );
#endif
#ifdef IC_MOTIF
  return bVertical ? 16 : 16 ;
#endif
}

/*------------------------------------------------------------------------------
| IScrollBar::systemScrollBoxLength                                            |
|                                                                              |
| Static system value function.                                                |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: systemScrollBoxLength ( bool bVertical )
{
#ifdef IC_PMWIN
  return IQUERYSYSVALUE( bVertical ? SV_CYVSLIDER : SV_CXHSLIDER );
#endif
#ifdef IC_MOTIF
  return bVertical ? 18 : 20;
#endif
}

/*------------------------------------------------------------------------------
| IScrollBar::systemScrollButtonLength                                         |
|                                                                              |
| Static system value function.                                                |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: systemScrollButtonLength ( bool bVertical )
{
#ifdef IC_PMWIN
  return IQUERYSYSVALUE( bVertical ? SV_CYVSCROLLARROW :
                                     SV_CXHSCROLLARROW );
#endif
#ifdef IC_MOTIF
  return bVertical ? 18 : 18;
#endif
}


/*------------------------------------------------------------------------------
| IScrollBar::IScrollBar                                                       |
|                                                                              |
| Constructor to create a scroll bar control with defaulted values for         |
| scroll range and scroll box information on a standard window.                |
------------------------------------------------------------------------------*/
IScrollBar :: IScrollBar ( unsigned long windowId,
                           IWindow* parent,
                           IWindow* owner,
                           const IRectangle& initial,
                           const Style& style )
#ifdef IC_PMWIN
  : ulClMinScrollIncrement(1), ulClPageScrollIncrement(0),
#ifdef IC_WIN
    fScrollBarData (new IScrollBarData() )
#endif
#ifdef IC_PM
    fScrollBarData (0)
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
  : fScrollBarData (0)
#endif
{
  IMODTRACE_DEVELOP("IScrollBar::ctor");
  IASSERTPARM(parent != 0);

#ifdef IC_PMWIN
  initialize(windowId, parent, owner, initial,
         IRange(1, 100), 25, 1, style);
#endif

#ifdef IC_MOTIF
  fScrollBarData = new IScrollBarData;
  fScrollBarData->initialize(this, windowId, parent, owner, initial,
         IRange(1, 100), 25, 1, style.asUnsignedLong() );
#endif // IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window.  ScrollBar does
  // not inherit background or foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kNone );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif
#ifdef IC_WIN
  // the background and foreground color of the scrollbar is dialog independant.
  setColor( PP_BACKGROUNDCOLOR, IColor::kButtonMiddle );
  setColor( PP_FOREGROUNDCOLOR, IColor::kScrollBar );
#endif // IC_WIN
}


/*------------------------------------------------------------------------------
| IScrollBar::IScrollBar                                                       |
|                                                                              |
| Constructor to create a scroll bar control with scroll range and scroll      |
| box information on a standard window.                                        |
------------------------------------------------------------------------------*/
IScrollBar :: IScrollBar ( unsigned long windowId,
                           IWindow* parent, IWindow* owner,
                           const IRange& scrollableItems,
                           unsigned long visibleItemCount,
                           unsigned long scrollBoxPosition,
                           const IRectangle& initial,
                           const Style& style )
#ifdef IC_PMWIN
  : ulClMinScrollIncrement(1), ulClPageScrollIncrement(0),
#ifdef IC_WIN
    fScrollBarData (new IScrollBarData() )
#endif
#ifdef IC_PM
    fScrollBarData (0)
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
  : fScrollBarData(0)
#endif
{
  IMODTRACE_DEVELOP("IScrollBar::ctor");
  IASSERTPARM(parent != 0);

#ifdef IC_PMWIN
  initialize(windowId, parent, owner, initial,
         scrollableItems, visibleItemCount, scrollBoxPosition, style);
#endif

#ifdef IC_MOTIF
  fScrollBarData = new IScrollBarData;
  fScrollBarData->initialize(this, windowId, parent, owner, initial,
         scrollableItems, visibleItemCount, scrollBoxPosition,
         style.asUnsignedLong());
#endif // IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window.  ScrollBar does
  // not inherit background or foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kNone );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif
#ifdef IC_WIN
  // the background and foreground color of the scrollbar is dialog independant.
  setColor( PP_BACKGROUNDCOLOR, IColor::kButtonMiddle );
  setColor( PP_FOREGROUNDCOLOR, IColor::kScrollBar );
#endif // IC_WIN
}


/*------------------------------------------------------------------------------
| IScrollBar::IScrollBar                                                       |
|                                                                              |
| Constructor to instantiate a scroll bar object for a dialog template.        |
------------------------------------------------------------------------------*/
IScrollBar :: IScrollBar ( unsigned long windowId, IWindow* parent )
  : fScrollBarData( 0 )
#ifdef IC_PMWIN
  , ulClMinScrollIncrement(1)
  , ulClPageScrollIncrement(0)
#endif
{
  IMODTRACE_DEVELOP("IScrollBar::ctor");
#ifdef IC_WIN
  fScrollBarData = new IScrollBarData();
#endif
#ifdef IC_PMWIN
  setAutoDestroyWindow(false);
#ifdef IC_WIN
  fScrollBarData->fdefaultHandler.handleEventsFor( this );
  fScrollBarData->fBrushHandler.handleEventsFor(this);
#endif
  reserveUserWindowWord( false );
  startHandlingEventsFor(windowId, parent);
#endif // IC_PMWIN

#ifdef IC_MOTIF
  IASSERTPARM(parent != 0);
  fScrollBarData = new IScrollBarData;

  IWindowHandle wh =
            XtNameToWidget(parent->handle(), (char *)IString(windowId) );
  if (wh == 0)
    ITHROWLIBRARYERROR1( IC_CONTROL_NOT_FOUND,
                         IBaseErrorInfo::accessError,
                         IException::recoverable,
                         "XtNameToWidget (scroll bar)");

  setAutoDestroyWindow(false);
  fScrollBarData->wrapperInitialization( wh, this );
#endif // IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window.  ScrollBar does
  // not inherit background or foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kNone );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif
#ifdef IC_WIN
  // the background and foreground color of the scrollbar is dialog independant.
  setColor( PP_BACKGROUNDCOLOR, IColor::kButtonMiddle );
  setColor( PP_FOREGROUNDCOLOR, IColor::kScrollBar );
#endif // IC_WIN
}


/*------------------------------------------------------------------------------
| IScrollBar::IScrollBar                                                       |
|                                                                              |
| Constructor to instantiate from an existing scroll bar control.              |
------------------------------------------------------------------------------*/
IScrollBar :: IScrollBar ( const IWindowHandle & wh )
#ifdef IC_PMWIN
  :   ulClMinScrollIncrement(1), ulClPageScrollIncrement(0),
#ifdef IC_WIN
    fScrollBarData (new IScrollBarData() )
#endif
#ifdef IC_PM
    fScrollBarData (0)
#endif
#endif // IC_PMWIN
#ifdef IC_MOTIF
  : fScrollBarData( 0 )
#endif
{
  IMODTRACE_DEVELOP("IScrollBar::ctor");
#ifdef IC_PMWIN
  setAutoDestroyWindow(false);
  reserveUserWindowWord( false );
  startHandlingEventsFor(wh);
#ifdef IC_WIN
  fScrollBarData->fdefaultHandler.handleEventsFor( this );
  fScrollBarData->fBrushHandler.handleEventsFor(this);
#endif
#endif // IC_PMWIN

#ifdef IC_MOTIF
  IASSERTPARM(wh!=0);
  fScrollBarData = new IScrollBarData;

  setAutoDestroyWindow(false);
  fScrollBarData->wrapperInitialization( wh, this );
#endif // IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window.  ScrollBar does
  // not inherit background or foreground color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kNone );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
                               inheritColor );
#endif
#ifdef IC_WIN
  // the background and foreground color of the scrollbar is dialog independant.
  setColor( PP_BACKGROUNDCOLOR, IColor::kButtonMiddle );
  setColor( PP_FOREGROUNDCOLOR, IColor::kScrollBar );
#endif // IC_WIN
}


/*------------------------------------------------------------------------------
| IScrollBar::~IScrollBar                                                      |
|                                                                              |
| Empty destructor for page tuning.                                            |
------------------------------------------------------------------------------*/
IScrollBar :: ~IScrollBar ( )
{
#ifdef IC_WIN
  if (fScrollBarData)
  {
    fScrollBarData->fdefaultHandler.stopHandlingEventsFor( this );
    fScrollBarData->fBrushHandler.stopHandlingEventsFor( this );
    delete fScrollBarData;
  }
#endif // IC_WIN

#ifdef IC_MOTIF
  if (fScrollBarData)
  {
     if (isValid())
        fScrollBarData->unregisterCallbacks( this );
     delete fScrollBarData;
  }
#endif // IC_MOTIF
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IScrollBarData::wrapperInitialize                                            |
|                                                                              |
| Initialize the values for the Motif wrapper.                                 |
------------------------------------------------------------------------------*/
IScrollBarData &IScrollBarData::wrapperInitialization( IWindowHandle wh,
                                                 IScrollBar *scrollBar )
{
  if (wh == 0)
    ITHROWLIBRARYERROR( IC_NAME_NOT_FOUND,
      IBaseErrorInfo::invalidRequest,
      IException::recoverable );

  if ( !XtIsSubclass( wh, xmScrollBarWidgetClass ) )
     ITHROWLIBRARYERROR1( IC_WRONG_CONTROL_TYPE,
                          IBaseErrorInfo::accessError,
                          IException::recoverable,
                          "XmScrollBar");

  scrollBar->prepareForUse(wh);
  scrollBar->startHandlingEventsFor( wh );
  registerCallbacks( scrollBar );

  unsigned char orientation;
  int upperBound;

  XtVaGetValues( wh,
    XmNheight, &height,
    XmNwidth, &width,
    XmNmaximum, &upperBound,
    XmNvalue, &ulClPrevScrollBoxPosition,
    XmNincrement, &ulClMinScrollIncrement,
    XmNpageIncrement, &ulClPageScrollIncrement,
    XmNorientation, &orientation,
    NULL );

  if ( orientation == XmHORIZONTAL )
    vertical = false;
  else
    vertical = true;

  // Adding one to the upper bound is necessary to get the same behavior
  // as the OS/2 version.
  XtVaSetValues( wh,
    XmNmaximum, upperBound + 1,
    NULL );

  // The trough color needs to be saved to get around a Motif bug
  cClShaftColor = scrollBar->foregroundColor();
  cClDefaultShaftColor = scrollBar->foregroundColor();

  // Check to see if we are wrappering the horizontal or vertical scroll bar in
  // the frame window.  If so, we want to set the owner of the scroll bar.
  Widget scrollParent = XtParent( wh );
  if (XtIsSubclass( scrollParent, xmMainWindowWidgetClass ))
  {
     Widget frameShell = XtParent( scrollParent );
     IWindow *frame = IWindow::windowWithHandle( frameShell );
     scrollBar->setOwner( frame );
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IScrollBarData::initialize                                                   |
|                                                                              |
| Initialize the values for the Motif resources.                               |
------------------------------------------------------------------------------*/
void IScrollBarData::initialize ( IScrollBar *scrollBar,
                                     unsigned long windowId,
                                     IWindow* parent, IWindow* owner,
                                     const IRectangle & initial,
                                     const IRange & scrollableItems,
                                     unsigned long visibleItemCount,
                                     unsigned long scrollBoxPosition,
                                     unsigned long ulmStyle )
{
  IRectangle createRect( initial );
  Arg args[20];
  int i;

  i=0;
  height = (Dimension)initial.height(),
  width  = (Dimension)initial.width();
  if ( ulmStyle & IScrollBar::vertical.asUnsignedLong() )
  {
    vertical = true;
    XtSetArg(args[i], XmNorientation, XmVERTICAL);i++;
    // Xm gives an error if a 0 width, height scrollbar is created.
    if ( width == 0 )
      width = (Dimension)scrollBar->systemScrollBarWidth( true );
    if ( height == 0 )
      height = (Dimension)scrollBar->systemScrollBoxLength( true );
  }
  else
  {
    vertical = false;
    XtSetArg(args[i], XmNorientation, XmHORIZONTAL);i++;
    // Xm gives an error if a 0 width, height scrollbar is created.
    if ( width == 0 )
      width = (Dimension)scrollBar->systemScrollBarWidth( false );
    if ( height == 0 )
      height = (Dimension)scrollBar->systemScrollBoxLength( false );
  }
  createRect.sizeTo( ISize( width, height ));

  XtSetArg(args[i], XmNmaximum, scrollableItems.upperBound() + 1);i++;
  XtSetArg(args[i], XmNminimum, scrollableItems.lowerBound());i++;
  XtSetArg(args[i], XmNsliderSize, visibleItemCount);i++;
  XtSetArg(args[i], XmNvalue, scrollBoxPosition);i++;
  XtSetArg(args[i], XmNincrement, ulClMinScrollIncrement); i++;
  // Set XmNpageIncrement via IScrollBar::pageScrollIncrement below.

  // Create the scrollbar widget
  Widget whScrollbar =
                scrollBar->create( windowId,
                                   NULL,
                                   ulmStyle,
                                   (IXmCreateFunction) XmCreateScrollBar,
                                   parent->handleForChildCreation(),
                                   owner->handle(),
                                   createRect,
                                   args, i );
  scrollBar->startHandlingEventsFor( whScrollbar );
  registerCallbacks( scrollBar );

  // Calling IScrollBar::pageScrollIncrement has the side effect of
  // setting XmNpageIncrement with a default value, if necessary.
  unsigned long
    pageIncrement = scrollBar->pageScrollIncrement();

  // The trough color needs to be saved to get around a Motif bug
  cClShaftColor = scrollBar->foregroundColor();
  cClDefaultShaftColor = scrollBar->foregroundColor();
  ulClPrevScrollBoxPosition = scrollBoxPosition;
}

/*------------------------------------------------------------------------------
| IScrollBarData::registerCallbacks                                            |
|                                                                              |
| registerCallbacks callbacks for XmScrollBar widget.                          |
------------------------------------------------------------------------------*/
void IScrollBarData::registerCallbacks( IScrollBar *scrollBar )
{
  Widget w = scrollBar->handle();
  XtAddCallback( w,
                 XmNdecrementCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNdecrementCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNincrementCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNincrementCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNpageDecrementCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNpageDecrementCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNpageIncrementCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNpageIncrementCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNdragCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNdragCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNvalueChangedCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNvalueChangedCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNtoBottomCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNtoBottomCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );

  XtAddCallback( w,
                 XmNtoTopCallback,
                 iwindowMotifCallback,
                 (XtPointer)scrollBar );
  XtAddCallback( w,
                 XmNtoTopCallback,
                 valueChangedCallback,
                 (XtPointer)scrollBar );
}

/*------------------------------------------------------------------------------
| IScrollBar::unregisterCallbacks()                                            |
|                                                                              |
| unregisterCallbacks for XmScrollBar widget.                                  |
------------------------------------------------------------------------------*/
void IScrollBarData::unregisterCallbacks( IScrollBar* scrollBar )
{
  Widget w = scrollBar->handle();

  XtRemoveCallback( w,
                    XmNdecrementCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNdecrementCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNincrementCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNincrementCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNpageDecrementCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNpageDecrementCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNpageIncrementCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNpageIncrementCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNdragCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNdragCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNvalueChangedCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNvalueChangedCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNtoBottomCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNtoBottomCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );

  XtRemoveCallback( w,
                    XmNtoTopCallback,
                    valueChangedCallback,
                    (XtPointer)scrollBar );
  XtRemoveCallback( w,
                    XmNtoTopCallback,
                    iwindowMotifCallback,
                    (XtPointer)scrollBar );
}
#endif // IC_MOTIF

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IScrollBar::initialize                                                       |
------------------------------------------------------------------------------*/
void  IScrollBar :: initialize ( unsigned long windowId,
                                 IWindow* pParent,
                                 IWindow* pOwner,
                                 const IRectangle& initial,
                                 const IRange& scrollableItems,
                                 unsigned long visibleItemCount,
                                 unsigned long scrollBoxPosition,
                                 const Style& style )
{
  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

#ifdef IC_PM
#if (IC_OBSOLETE <= IC_OBSOLETE_3)
  /*****************************************************************/
  /* Check for autosizing of the scroll bar (PM only autosizes if  */
  /* if the appropriate rectangle dimension is <= 0).  This        */
  /* means that you can't create a scroll bar with a size of       */
  /* (0, 0), unless you size it after you create it.               */
  /*****************************************************************/
  bool bAutoWidthSize = false;
  bool bAutoHeightSize = false;
  if (style & autoSize)
  {                                    //Scroll bar should be auto-sized
     if (style & vertical)
     {                                 //Vertical scroll bar
        bAutoWidthSize = true;         //Autosize vertical scroll bar's width
     }
     else
     {                                 //Horizontal scroll bar
        bAutoHeightSize = true;        //Autosize horizontal scroll bar's
     }                                 //height
  }
#endif

  /*****************************************************************/
  /* Build structure for passing control data.                     */
  /*****************************************************************/
  SBCDATA sbcdata;
  initializeControlData(scrollableItems, visibleItemCount, &sbcdata);
                                       //Initialize the SBCDATA structure
  sbcdata.sHilite = 0;
  sbcdata.posThumb = (short)scrollBoxPosition;

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
  IRectangle rectInit(initial.bottomLeft(),
                      ISize(bAutoWidthSize ? 0 : initial.width(),
                            bAutoHeightSize ? 0 : initial.height()));
#endif

#endif // IC_PM

  IWindowHandle owner = (pOwner == 0) ? IWindowHandle(0) : pOwner->handle();

  IWindowHandle wh =
      this -> create( windowId,
                      0,
                      convertToGUIStyle( style ),
                      WC_SCROLLBAR,
                      pParent->handleForChildCreation(),
                      owner,
#ifdef IC_WIN
                      initial,
                      0,
#else
#if (IC_OBSOLETE <= IC_OBSOLETE_3)
                      rectInit,
#else
                      initial,
#endif
                      &sbcdata,
#endif // IC_WIN
                      0,
                      defaultOrdering(),
                      convertToGUIStyle( style, true ) );

#ifdef IC_WIN
  fScrollBarData->fdefaultHandler.handleEventsFor( this );
  fScrollBarData->fBrushHandler.handleEventsFor( this );

  SCROLLINFO scrollinfo;
  initializeControlData(scrollableItems, visibleItemCount, &scrollinfo);
  scrollinfo.fMask  = SIF_PAGE | SIF_RANGE | SIF_POS;
  scrollinfo.nPos   = (int)scrollBoxPosition;
  SetScrollInfo( wh,
                 SB_CTL,
                 &scrollinfo,
                 (style & IWindow::disabled) ? TRUE : FALSE );
#endif // IC_WIN
  startHandlingEventsFor(wh);
}


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IScrollBar::initializeControlData                                            |
|                                                                              |
| Fill in control data structure for a scroll bar control.                     |
------------------------------------------------------------------------------*/
void  IScrollBar :: initializeControlData ( const IRange & scrollableItems,
                                            unsigned long visibleItemCount,
                                            struct tagSCROLLINFO* controlData )
{
  int lower = (int)scrollableItems.lowerBound();
  int upper = (int)scrollableItems.upperBound();
  if (lower > upper)
  {                                    //Swap the bounds
     int ulTemp = lower;
     lower = upper;
     upper = ulTemp;
  }

  /*****************************************************************/
  /* Store control data values now.                                */
  /*****************************************************************/
  controlData->cbSize = sizeof(SCROLLINFO);
  controlData->nMin   = lower;
  controlData->nMax   = upper;
  controlData->nPage    = (unsigned int)visibleItemCount;
}
#endif // IC_WIN

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IScrollBar::initializeControlData                                            |
|                                                                              |
| Fill in control data structure for a scroll bar control.                     |
------------------------------------------------------------------------------*/
void  IScrollBar :: initializeControlData ( const IRange & scrollableItems,
                                            unsigned long visibleItemCount,
                                            struct _SBCDATA* controlData )
{
  unsigned long ulLower = scrollableItems.lowerBound();
  unsigned long ulUpper = scrollableItems.upperBound();
  if (ulLower > ulUpper)
  {                                    //Swap the bounds
     unsigned long ulTemp = ulLower;
     ulLower = ulUpper;
     ulUpper = ulTemp;
  }
  unsigned long ulTotalItems = ulUpper - ulLower + 1;

  /*****************************************************************/
  /* Store control data values now.                                */
  /*****************************************************************/
  controlData->cb = sizeof(SBCDATA);
  controlData->posFirst = (short)ulLower;
  controlData->posLast = (short)((visibleItemCount > ulTotalItems) ?
                            ulLower : ulUpper - visibleItemCount + 1);
  controlData->cVisible = (short)visibleItemCount;
  controlData->cTotal = (short)ulTotalItems;
}


/*------------------------------------------------------------------------------
| IScrollBar::windowData                                                       |
|                                                                              |
| Return window parameters and control data for a scroll bar control.  The     |
| caller is responsible for setting up the pointer and count fields for the    |
| data desired.  This function will initialize other fields in the _WNDPARAMS  |
| structure.                                                                   |
------------------------------------------------------------------------------*/
void  IScrollBar :: windowData ( struct _WNDPARAMS* windowData ) const
{
  // Assume caller has set up return buffers and sizes.
  windowData->fsStatus = 0;

  if (windowData->pszText)
  {                                    // Caller wants text
     windowData->fsStatus |= (WPM_TEXT | WPM_CCHTEXT);
  }
  else
  {
     windowData->cchText = 0;
  }

  if (windowData->pPresParams)
  {                           // Caller wants presentation parameters
     windowData->fsStatus |= (WPM_PRESPARAMS | WPM_CBPRESPARAMS);
  }
  else
  {
     windowData->cbPresParams = 0;
  }

  if (windowData->pCtlData)
  {                                    // Caller wants control data
     windowData->fsStatus |= (WPM_CTLDATA | WPM_CBCTLDATA);
  }
  else
  {
     windowData->cbCtlData = 0;
  }

  if (!handle().sendEvent( WM_QUERYWINDOWPARAMS,
                           IEventParameter1( windowData ),
                           IEventParameter2( 0 ))
            .asUnsignedLong())
  {                                    // Something went wrong
     ITHROWGUIERROR2("WM_QUERYWINDOWPARAMS",
                     IBaseErrorInfo::accessError,
                     IException::recoverable);
  }
  /*****************************************************************/
  /* Note: windowData->fsStatus does not get cleared on success,   */
  /*       as advertised, so just assume we have all we requested. */
  /*****************************************************************/
}
#endif // IC_PM
#endif // IC_PMWIN

/*------------------------------------------------------------------------------
| IScrollBar::defaultStyle                                                     |
|                                                                              |
| Return the current default style for creating new scroll bar objects.        |
------------------------------------------------------------------------------*/
IScrollBar::Style  IScrollBar :: defaultStyle ( )
{
  return currentDefaultStyle;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IScrollBar::setDefaultStyle                                                  |
|                                                                              |
| Sets the current default style for creating new scroll bar objects.          |
------------------------------------------------------------------------------*/
void IScrollBar::setDefaultStyle ( const IScrollBar::Style& style )
{
  currentDefaultStyle = style;
}
#endif // IC_WIN

/*------------------------------------------------------------------------------
| IScrollBar::convertToGUIStyle                                                |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IScrollBar::convertToGUIStyle(const IBitFlag& guiStyle,
                                            bool bExtOnly) const
{
  // Obtain the style from the class (IControl) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
    // Let the clean SBS_ styles, flow thru
    ulStyle |= guiStyle.asUnsignedLong() & ISBS_MASK;

#ifdef IC_WIN
    // For Windows it is always a child window
    ulStyle |= WS_CHILD;
#endif

#ifdef IC_PM
    // Add the thumb sizing style
    ulStyle |= SBS_THUMBSIZE;
#endif
  }

  return( ulStyle );
}

/*------------------------------------------------------------------------------
| IScrollBar::isHorizontal                                                     |
------------------------------------------------------------------------------*/
bool  IScrollBar :: isHorizontal ( ) const
{
  return !isVertical();
}


/*------------------------------------------------------------------------------
| IScrollBar::isVertical                                                       |
------------------------------------------------------------------------------*/
bool  IScrollBar :: isVertical ( ) const
{
#ifdef IC_PMWIN
   return ((style() & vertical.asUnsignedLong()) ? true : false);
#endif

#ifdef IC_MOTIF
  unsigned char orientation;

  XtVaGetValues(handle(),
               XmNorientation, &orientation,
               NULL);
  return ((orientation & XmVERTICAL) ? true : false);
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IScrollBar::setScrollBar                                                     |
|                                                                              |
| Set the scroll bar range and scroll box size.                                |
------------------------------------------------------------------------------*/
IScrollBar &  IScrollBar :: setScrollBar ( const IRange & minMax,
                                           unsigned long visibleCount )
{
#ifdef IC_PMWIN
#ifdef IC_WIN
  SCROLLINFO scrollinfo;
  initializeControlData(minMax, visibleCount, &scrollinfo);
  scrollinfo.fMask  = SIF_PAGE | SIF_RANGE;

  // Note:  Only redraw the scrollbar if it is enabled.
  SetScrollInfo( this->handle(),
                 SB_CTL,
                 &scrollinfo,
                 isEnabled() );
#endif // IC_WIN

#ifdef IC_PM
  SBCDATA sbcdata;
  initializeControlData(minMax, visibleCount, &sbcdata);
                                    // Initialize the SBCDATA structure
  unsigned long ulPos = scrollBoxPosition();
  unsigned long ulTotalItems = sbcdata.cTotal;

  // The call to SBM_SETSCROLLBAR will redraw and enable the control.
  // Need to restore the enabled/disabled state.
  bool isDisabled = !isEnabled();

  handle().sendEvent(SBM_SETSCROLLBAR,
                     IEventParameter1(ulPos),
                     IEventParameter2(sbcdata.posFirst,
                                      sbcdata.posLast));
  handle().sendEvent(SBM_SETTHUMBSIZE,
                     IEventParameter1((unsigned short)visibleCount,
                                      (unsigned short)ulTotalItems),
                     IEventParameter2());

  if (isDisabled)
     disable();
#endif // IC_PM
#endif // IC_PMWIN

#ifdef IC_MOTIF
  XtVaSetValues( handle(),
    XmNmaximum, minMax.upperBound() + 1,
    XmNminimum, minMax.lowerBound(),
    XmNsliderSize, visibleCount,
    XmNincrement, fScrollBarData->ulClMinScrollIncrement,
    NULL );

  // Calling IScrollBar::pageScrollIncrement has the side effect of
  // setting XmNpageIncrement with a default value, if necessary.
  unsigned long
    pageIncrement = this->pageScrollIncrement();
#endif // IC_MOTIF

  return *this;
}


/*------------------------------------------------------------------------------
| IScrollBar::setScrollableRange                                               |
|                                                                              |
| Set the range of all items to which the scroll box can scroll.               |
------------------------------------------------------------------------------*/
IScrollBar &  IScrollBar :: setScrollableRange ( const IRange & minMax )
{
#ifdef IC_PMWIN
  if (this->scrollableRange() != minMax)
  {
    unsigned long ulVisible = visibleCount();
    setScrollBar(minMax, ulVisible);
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  XtVaSetValues(handle(),
    XmNmaximum, minMax.upperBound() + 1,
    XmNminimum, minMax.lowerBound(),
    NULL);
#endif // IC_MOTIF
  return *this;
}


/*------------------------------------------------------------------------------
| IScrollBar::setVisibleCount                                                  |
|                                                                              |
| Set the size of the scroll box.                                              |
------------------------------------------------------------------------------*/
IScrollBar &  IScrollBar :: setVisibleCount ( unsigned long rangeUnits )
{
#ifdef IC_PMWIN
  if (this->visibleCount() != rangeUnits)
  {
    IRange rangeScroll = scrollableRange();
    setScrollBar(rangeScroll, rangeUnits);
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  int scrollBarPos, sliderSize, incrementAmount, pageIncrementAmount;
  IRange scrollRange = scrollableRange();

  if ( rangeUnits > ( (scrollRange.upperBound() + 1) - scrollRange.lowerBound()) )
    ITHROWLIBRARYERROR( IC_INVALID_VISIBLECOUNT,
      IBaseErrorInfo::invalidRequest,
      IException::recoverable);

  XmScrollBarGetValues(handle(),
                       &scrollBarPos,
                       &sliderSize,
                       &incrementAmount,
                       &pageIncrementAmount);

  // Can not let the page increment get less than 1.
  int pageIncrement = rangeUnits - incrementAmount;
  if ( pageIncrement < 1 )
    pageIncrement = 1;
  XmScrollBarSetValues(handle(),
                       scrollBarPos,
                       rangeUnits,
                       incrementAmount,
                       pageIncrement,
                       true);
#endif // IC_MOTIF
  return *this;
}


/*------------------------------------------------------------------------------
| IScrollBar::moveScrollBoxTo                                                  |
------------------------------------------------------------------------------*/
IScrollBar &  IScrollBar :: moveScrollBoxTo ( unsigned long bottomItem )
{
#ifdef IC_PMWIN
  // Get the current slider position so that we can notify any
  // observers of position changes
  unsigned long sliderPosition;

#ifdef IC_WIN
  sliderPosition = GetScrollPos(this->handle(),SB_CTL);
#endif

#ifdef IC_PM
  sliderPosition = this->sendEvent(SBM_QUERYPOS).asUnsignedLong();
#endif

  // Check if a delayed notification message is already queued
  bool            notify = false;
  IAnchorBlockHandle hab = IThread::current().anchorBlock();
  QMSG               msg;
  if ( !IPEEKMSG(hab,&msg,this->handle(),
                 IC_UM_DELAYNOTIFY,IC_UM_DELAYNOTIFY,PM_NOREMOVE))
  {
    notify = true;
  }

  if (this->scrollBoxPosition() != bottomItem)
  {
#ifdef IC_WIN
    // Windows will enable a disabled scrollbar if the redraw
    // flag is set to true.  Only redraw when the scrollbar is
    // enabled.
    handle().sendEvent( SBM_SETPOS,
                        bottomItem,
                        isEnabled() );
#endif // IC_WIN

#ifdef IC_PM
    unsigned long lparam = 0;
    handle().sendEvent( SBM_SETPOS, bottomItem, lparam );
#endif
  }

  // Notify observers (if needed)
  if ( notify )
  {
    this->sendEvent( IC_UM_DELAYNOTIFY,
                     IEventParameter1( sliderPosition ),
                     IEventParameter2( SB_SLIDERPOSITION ));
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  // Need to use XmScrollBarSetValues so that notify can be set true. When
  // wrappering a scrolled window scroll bar this notify is required for the
  // scrolled window to detect that the scroll bar has been moved.

  int value, increment, pageIncrement, sliderSize;

  XmScrollBarGetValues( handle(),
    &value,
    &sliderSize,
    &increment,
    &pageIncrement );
  XmScrollBarSetValues( handle(),
    (int) bottomItem,
    sliderSize,
    increment,
    pageIncrement,
    true );
#endif // IC_MOTIF

  return *this;
}


/*------------------------------------------------------------------------------
| IScrollBar::scrollableRange                                                  |
|                                                                              |
| Return the range of all scrollable items.                                    |
------------------------------------------------------------------------------*/
IRange  IScrollBar :: scrollableRange ( ) const
{
#ifdef IC_WIN
  INT  minPos, maxPos;

  handle().sendEvent( SBM_GETRANGE, &minPos, &maxPos );

  return IRange(minPos, maxPos);
#endif // IC_WIN
#ifdef IC_PM
  SBCDATA sbcdata;
  WNDPARAMS wndparams;

  wndparams.pszText = 0;
  wndparams.pPresParams = 0;
  wndparams.cbCtlData = sizeof(SBCDATA);
  wndparams.pCtlData = &sbcdata;
  windowData(&wndparams);

  unsigned long start = sbcdata.posFirst;
  return IRange(start, start + sbcdata.cTotal - 1);
#endif // IC_PM

#ifdef IC_MOTIF
  IRange range = scrollBoxRange() + IRange(0, visibleCount() - 1);
                              // add scroll box length to upper bound
  return range;
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IScrollBar::scrollBoxRange                                                   |
|                                                                              |
| Return the range in which the scroll box can be positioned.                  |
------------------------------------------------------------------------------*/
IRange  IScrollBar :: scrollBoxRange ( ) const
{
#ifdef IC_WIN
  INT  minPos, maxPos;

  handle().sendEvent( SBM_GETRANGE, &minPos, &maxPos );
  // Because the thumb size is variable, if set greater than 1, we
  // will need to adjust the range accordingly
  if (this->visibleCount() > 1)
    maxPos -= this->visibleCount() - 1;
  return IRange( minPos, maxPos );
#endif // IC_WIN

#ifdef IC_PM
  IEventResult evtRc = handle().sendEvent(SBM_QUERYRANGE, 0, 0);
  return IRange(evtRc.number1(), evtRc.number2());
#endif

#ifdef IC_MOTIF
  int minimum, maximum;

  XtVaGetValues(handle(),
                XmNmaximum, &maximum,
                XmNminimum, &minimum,
                NULL);

  return IRange(minimum,maximum) - IRange(0, visibleCount());
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IScrollBar::visibleCount                                                     |
|                                                                              |
| Return the scroll box size (also used to determine the size of               |
| a page scroll).                                                              |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: visibleCount ( ) const
{
#ifdef IC_WIN
  SCROLLINFO scrollinfo;
  scrollinfo.cbSize = sizeof(SCROLLINFO);
  scrollinfo.fMask  = SIF_PAGE;
  GetScrollInfo( this->handle(),
                 SB_CTL,
                 &scrollinfo );
  return scrollinfo.nPage;
#endif // IC_WIN

#ifdef IC_PM
  SBCDATA sbcdata;
  WNDPARAMS wndparams;

  wndparams.pszText = 0;
  wndparams.pPresParams = 0;
  wndparams.cbCtlData = sizeof(SBCDATA);
  wndparams.pCtlData = &sbcdata;
  windowData(&wndparams);

  return sbcdata.cVisible;
#endif // IC_PM

#ifdef IC_MOTIF
  int sliderSize;

  XtVaGetValues( handle(),
                 XmNsliderSize, &sliderSize,
                 NULL );
  return (unsigned long) sliderSize;
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IScrollBar::scrollBoxPosition                                                |
|                                                                              |
| Return the position of the scroll box (the lower boundary).                  |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: scrollBoxPosition ( ) const
{
#ifdef IC_PMWIN
  return (unsigned long) handle().sendEvent(SBM_QUERYPOS, 0, 0);
#endif

#ifdef IC_MOTIF
  int scrollBarPos;

  XtVaGetValues( handle(),
                 XmNvalue, &scrollBarPos,
                 NULL );
  return (unsigned long) scrollBarPos;
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IScrollBar::pageScrollIncrement                                              |
|                                                                              |
| Return the page scrolling increment (the default value is the visible page   |
| size minus the minimum scrolling increment).                                 |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: pageScrollIncrement ( ) const
{
#ifdef IC_PMWIN
  unsigned long ulScroll = ulClPageScrollIncrement;
#endif
#ifdef IC_MOTIF
  unsigned long ulScroll = fScrollBarData->ulClPageScrollIncrement;
#endif
  if (ulScroll == 0)
  {  // No application-specified value, so calculate a default.
     unsigned long
       minScroll = this->minScrollIncrement();
     long lDelta = this->visibleCount() - minScroll;
     ulScroll = ( lDelta > 0 ) ? lDelta : minScroll;

#ifdef IC_MOTIF
     XtVaSetValues( this->handle(),
                    XmNpageIncrement, ulScroll,
                    0 );
#endif
  }
  return ulScroll;
}


/*------------------------------------------------------------------------------
| IScrollBar::calcMinimumSize                                                  |
|                                                                              |
| Return the size of a "typical" scroll bar.                                   |
------------------------------------------------------------------------------*/
ISize  IScrollBar :: calcMinimumSize ( ) const
{
  bool bVertical = isVertical();
  unsigned long ulLength = 2 * systemScrollButtonLength(bVertical) +
                             3 * systemScrollBoxLength(bVertical);
  unsigned long ulWidth = systemScrollBarWidth(bVertical);

  ISize sizMin = bVertical ? ISize(ulWidth, ulLength) :
                             ISize(ulLength, ulWidth);
#ifdef IC_MOTIF
  // ignore the border for now. The default value for the border is 0.
//  Dimension borderWidth;
//  XtVaGetValues( handle(),
//                 XmNborderWidth, &borderWidth,
//                 NULL );
//  sizMin += ISize( 2 * borderWidth, 2 * borderWidth );
#endif // IC_MOTIF
#ifdef IC_PMWIN
  sizMin += ISize(2 * IQUERYSYSVALUE( SV_CXBORDER ),
                  2 * IQUERYSYSVALUE( SV_CYBORDER ));
#endif
  return sizMin;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IScrollBar::isVisible                                                        |
|                                                                              |
| Override of IWindow::isVisible.  This override is needed since scroll bar    |
| children of XmScrolledWindow widgets with AS_NEEDED scrollBarDisplayPolicy   |
| won't be unmanaged by the ScrolledWindow when they are not needed (and       |
| are apparently not visible).  Their position will be outside their parent's  |
| window and they will be clipped.                                             |
------------------------------------------------------------------------------*/
bool IScrollBar::isVisible( ) const
{
  // Call to IWindow::isVisible.  There, we check for the map_state window
  // attribute.  If it is IsViewable, the function returns true.
  bool isVisible = Inherited::isVisible();

  // Now we need to check whether we have an AS_NEEDED scroll bar child of an
  // XmScrolledWindow widget.  In that case, even if the window is viewable,
  // we need to check to determine whether the scroll bar is fully obscured.
  if (isVisible)
  {
     Widget widget = handle(),
            parentWidget = parent()->handle();
     if (XmIsScrolledWindow( parentWidget ))
     {
       unsigned char dispPolicy;
        XtVaGetValues( parentWidget,
                       XmNscrollBarDisplayPolicy, &dispPolicy,
                       NULL );
        if (dispPolicy == XmAS_NEEDED)
        {
           Position  xScroll,
                     yScroll;
           Dimension parentHeight,
                     parentWidth,
                     borderWidth;

           XtVaGetValues( widget,
                          XmNx, &xScroll,
                          XmNy, &yScroll,
                          NULL );
           XtVaGetValues( parentWidget,
                          XmNheight,      &parentHeight,
                          XmNwidth,       &parentWidth,
                          XmNborderWidth, &borderWidth,
                          NULL );
           if ((xScroll >= (parentWidth + borderWidth)) ||
               (yScroll >= (parentHeight + borderWidth)))
              isVisible = false;
        }
     }
  }

  return isVisible;
}

/*------------------------------------------------------------------------------
| IScrollBar::setPrevScrollBoxPosition                                         |
|                                                                              |
| Allows saving the current position of the scroll bar so that it can be used  |
| as the previous position for the next event.                                 |
------------------------------------------------------------------------------*/
IScrollBar &IScrollBar::setPrevScrollBoxPosition( unsigned long value )
{
  fScrollBarData->ulClPrevScrollBoxPosition = value;
  return *this;
}

/*------------------------------------------------------------------------------
| IScrollBar::prevScrollBoxPosition                                            |
|                                                                              |
| Returns the prevScrollBoxPosition.                                           |
------------------------------------------------------------------------------*/
unsigned long IScrollBar::prevScrollBoxPosition() const
{
  return fScrollBarData->ulClPrevScrollBoxPosition;
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| WIN32 Color Support                                                          |
|                                                                              |
| Windows has no API equivalent for Presentation Parameters.  The following    |
| scrollBar color functions are NOPed for Windows.  The query functions return |
| an appropriate default IColor object.                                        |
------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
| IScrollBar::foregroundColor                                                  |
|                                                                              |
| Returns the foreground color of the scrollbar.                               |
------------------------------------------------------------------------------*/
IColor IScrollBar::foregroundColor () const
{
#ifdef IC_PM
  return (IWindow::color(PP_FOREGROUNDCOLOR,
                         IGUIColor(IGUIColor::defaultControl)));
#endif

#ifdef IC_WIN
  return Inherited::foregroundColor();
#endif

#ifdef IC_MOTIF
  Pixel colorArea;
  XtVaGetValues( handle(),
                 XmNtroughColor, &colorArea,
                 NULL );
  return IColor( colorArea, &IApplication::current().colorMap() );
#endif // IC_MOTIF
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IScrollBar::setForegroundColor.                                              |
|                                                                              |
| Sets the color of the shaft of the scroll bar.                               |
------------------------------------------------------------------------------*/
IScrollBar &IScrollBar::setForegroundColor( const IColor &color )
{
  XtVaSetValues( handle(),
    XmNtroughColor, color.asPixel(),
    NULL );
  // Need to save color to get around bug in Motif
  fScrollBarData->cClShaftColor = color;
  return *this;
}

/*------------------------------------------------------------------------------
| IScrollBar::setColor.                                                        |
|                                                                              |
| Sets the foreground color of the scroll bar, all other areas are passed onto |
| the inherited version of setColor
------------------------------------------------------------------------------*/
IScrollBar &IScrollBar::setColor( unsigned long colorArea,
                                  const IColor &color )
{
  if ( colorArea == PP_FOREGROUNDCOLOR )
  {
    XtVaSetValues( handle(),
      XmNtroughColor, color.asPixel(),
      NULL );
    // Need to save color to get around bug in Motif
    fScrollBarData->cClShaftColor = color;
  }
  else
    Inherited::setColor( colorArea, color );
  return *this;
}


/*------------------------------------------------------------------------------
| IScrollBar::resetForegroundColor.                                            |
|                                                                              |
| Resets the color of the shaft to the original color when the widget was      |
| created.                                                                     |
------------------------------------------------------------------------------*/
IScrollBar &IScrollBar::resetForegroundColor()
{
  setForegroundColor( fScrollBarData->cClDefaultShaftColor );
  return *this;
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| IScrollBar::hiliteForegroundColor                                            |
|                                                                              |
| Returns the hilite foreground color of the scrollbar.                        |
------------------------------------------------------------------------------*/
IColor IScrollBar::hiliteForegroundColor () const
{
#ifdef IC_PM
  return (IWindow::color(PP_HILITEFOREGROUNDCOLOR,
                         IGUIColor(IGUIColor::buttonMiddle)));
#endif

#ifdef IC_WIN
  return IGUIColor( IGUIColor::buttonMiddle );
#endif

#ifdef IC_MOTIF
  return Inherited::backgroundColor( );
#endif
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IScrollBar::setHiliteForegroundColor.                                        |
|                                                                              |
| Set the scroll box color of the scroll bar. In Motif this is the same as the |
| background color.                                                            |
------------------------------------------------------------------------------*/
IScrollBar &IScrollBar::setHiliteForegroundColor( const IColor &color )
{
  // A bug in Motif resets the trough color when setting the slider.
  // The workaround is to set the trough color to something different
  // than it was before.
  IColor troughColor = foregroundColor();

  // if the trough pixel value has not changed then set it to something
  // different
  if ( troughColor.asPixel() == fScrollBarData->cClShaftColor.asPixel() )
  {
    if ( troughColor.redMix() == 255 &&
         troughColor.greenMix() == 255 &&
         troughColor.blueMix() == 255 )
      setForegroundColor( IColor::blue );
    else
      setForegroundColor( IColor::white );
  }
    // Now set the trough and slider colors.
  Inherited::setBackgroundColor( color );
  setForegroundColor( troughColor );
  return *this;
}

/*------------------------------------------------------------------------------
| IScrollBar::resetHiliteForegroundColor.                                      |
|                                                                              |
| Reset the scroll box color of the scroll bar. In Motif this is the same as   |
| the background color.                                                        |
------------------------------------------------------------------------------*/
IScrollBar &IScrollBar::resetHiliteForegroundColor( )
{
  Inherited::resetBackgroundColor();
  return *this;
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| IScrollBar::setMinScrollIncrement                                            |
|                                                                              |
| Sets the minimum scroll increment for the scrollbar.                         |
------------------------------------------------------------------------------*/
IScrollBar& IScrollBar :: setMinScrollIncrement ( unsigned long value )
{
#ifdef IC_PMWIN
  if (this->minScrollIncrement() != value)
  {
    ulClMinScrollIncrement = value;
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  XtVaSetValues( handle(),
                 XmNincrement, (int) value,
                 NULL );
  fScrollBarData->ulClMinScrollIncrement = (unsigned long ) value;
#endif // IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IScrollBar::setPageScrollIncrement                                           |
|                                                                              |
| Sets the page scroll increment for the scrollbar.                            |
------------------------------------------------------------------------------*/
IScrollBar& IScrollBar :: setPageScrollIncrement ( unsigned long value )
{
#ifdef IC_PMWIN
  if (this->pageScrollIncrement() != value)
  {
    ulClPageScrollIncrement = value;
  }
#endif // IC_PMWIN

#ifdef IC_MOTIF
  XtVaSetValues( handle(),
                 XmNpageIncrement, (int) value,
                 NULL );
  fScrollBarData->ulClPageScrollIncrement = value;
#endif // IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IScrollBar::minScrollIncrement                                               |
|                                                                              |
| Returns the minimum page scroll increment for the scrollbar.                 |
------------------------------------------------------------------------------*/
unsigned long  IScrollBar :: minScrollIncrement ( ) const
{
#ifdef IC_PMWIN
  return ulClMinScrollIncrement;
#endif

#ifdef IC_MOTIF
  return fScrollBarData->ulClMinScrollIncrement;
#endif
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IScrollBar::passEventToOwner                                                 |
|                                                                              |
| Returns whether or not the event can be passed up to the owner of this       |
| control.                                                                     |
------------------------------------------------------------------------------*/
bool IScrollBar::passEventToOwner( IEvent& event )
{
  switch ( event.eventId() )
  {
    case WM_BUTTON1CLICK:
    case WM_BUTTON2CLICK:
    case WM_BUTTON3CLICK:
    case WM_CHORD:
      event.setPassToOwner( false );
      break;
    default:
      Inherited::passEventToOwner( event );
      break;
  }
  return event.passToOwner();
}
#endif // IC_MOTIF
