#ifndef _ILISTBX3_
#define _ILISTBX3_

// Revision: 97 1.11.2.3 source/ui/basectl/ilistbx3.hpp, listctls, ioc.v400, 001006 
/*NOSHIP*/
/*******************************************************************************
* FILE NAME: ilistbx3.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in both ilistbas.hpp and icombobs.hpp.                                     *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#ifdef IC_MOTIF
  #include <Xm/List.h>
  #include <Xm/ScrolledW.h>
  #include <stdio.h>
  #include <ilistbox.hpp>
  #include <istring.hpp>
  #include <irect.hpp>
  #include <ireslib.hpp>
  #include <iapp.hpp>
  #include <ihandle.hpp>
  #include <iexcept.hpp>
#endif //IC_MOTIF

#include <itrace.hpp>

#ifdef IC_USE_CB
  #define IC_CLASSNAME IBaseComboBox
#else
  #define IC_CLASSNAME IBaseListBox
#endif

#ifdef IC_MOTIF
// Carry over from PM implementation.
  #define LIT_ERROR           (-3)

#ifdef IC_USE_CB
  #define IC_CLASSPRIV IBaseComboBoxData
  #define PRIVATEDATA  fBaseComboBoxData
#else
  #define IC_CLASSPRIV IBaseListBoxData
  #define PRIVATEDATA  fBaseListBoxData
#endif
#endif //IC_MOTIF

#ifdef IC_WIN

#ifdef IC_USE_CB
#define IC_CLASSNAMECURSORDATA IBaseComboBoxCursorData
#else
#define IC_CLASSNAMECURSORDATA IBaseListBoxCursorData
#endif

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

#pragma enum(4)
#pragma pack(push,4)

// Private data class for the nested Cursor class
class IC_CLASSNAMECURSORDATA {
public:
  IC_CLASSNAMECURSORDATA() : fcount(0), fcurrent(0), flist(0) {}
  ~IC_CLASSNAMECURSORDATA() { if (flist) delete [] flist; }

  long  fcount;         // number of selected items
  long  fcurrent;       // current cursor position in flist
  long* flist;          // list of listbox indices
};

#pragma pack(pop)
#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IItemHandleElement key function                                              |
|                                                                              |
| Used by IItemHandlesList collection to obtain key when searching.            |
------------------------------------------------------------------------------*/
unsigned long const& key (IItemHandleElement* const& k)
{
  return k->itemKey;
}
#endif


/*------------------------------------------------------------------------------
| IC_CLASSNAME::isHorizontalScroll                                             |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::isHorizontalScroll() const
{
#ifdef IC_WIN
#ifdef IC_USE_CB
  return((extendedStyle() & horizontalScroll.asUnsignedLong()) ? true : false);
#else
  return((style() & horizontalScroll.asUnsignedLong()) ? true : false);
#endif
#endif //IC_WIN

#ifdef IC_PM
  return( (style() & horizontalScroll.asUnsignedLong()) ? true : false );
#endif //IC_PM

#ifdef IC_MOTIF
  unsigned char horizontalPolicy;
  Widget        listBox, horzScroll, scrolledWindow;

  listBox = IC_CLASSPRIV::listBoxHandle (*this);
  scrolledWindow = XtParent (listBox);
  if (XtIsSubclass (scrolledWindow, xmScrolledWindowWidgetClass))
  {
     XtVaGetValues (scrolledWindow,
                    XmNhorizontalScrollBar, &horzScroll,
                    NULL);
     if (horzScroll != NULL)

     // In Motif, a scrolled list will not have a horizontal scroll
     // bar if XmNlistSizePolicy = XmVARIABLE.
     {
        XtVaGetValues (listBox,
                       XmNlistSizePolicy, &horizontalPolicy,
                       NULL);

        if (horizontalPolicy != XmVARIABLE)
           return true;
     }
  }
  return false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::itemText                                                       |
|                                                                              |
| Get the text of an item.                                                     |
------------------------------------------------------------------------------*/
IString IC_CLASSNAME::itemText ( unsigned long lIndex ) const
{
#ifdef IC_PMWIN
   IEventResult evt = handle().sendEvent(LM_QUERYITEMTEXTLENGTH,
                                         IEventParameter1(lIndex),
                                         IEventParameter2(0));
   if (evt.asLong() == LIT_ERROR)
      ITHROWLIBRARYERROR1(IC_INVALID_INDEX,
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable,
                          IString(lIndex));
   unsigned long ulBufSize = evt.asUnsignedLong();

   IString returnString(0, ulBufSize);
   #ifdef IC_WIN
   handle().sendEvent(LM_QUERYITEMTEXT,
                      IEventParameter1(lIndex),
                      IEventParameter2((char*)returnString));
   #else
   handle().sendEvent(LM_QUERYITEMTEXT,
                      IEventParameter1((unsigned short)lIndex,
                                       (unsigned short)(ulBufSize+1)),
                      IEventParameter2((char*)returnString));
   #endif
   return returnString;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int           lbItemCount;
  XmStringTable itemList;
  char*         item;

  XtVaGetValues (IC_CLASSPRIV::listBoxHandle(*this),
                 XmNitemCount, &lbItemCount,
                 XmNitems, &itemList,
                 NULL);

  if (lIndex >= lbItemCount)
  {
     ITHROWLIBRARYERROR1 (IC_INVALID_INDEX,
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable,
                          IString (lIndex));
  }

  if (XmStringGetLtoR (itemList[lIndex],
                       XmSTRING_DEFAULT_CHARSET,
                       &item))
     // d7950 - Need to free item.
     // return IString (item);
  {
     IString result( item );
     XtFree( item );
     return result;
  }
  else
     return IString("");
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IC_CLASSNAME::setItemText                                                    |
|                                                                              |
| Change the text of an item currently in the list.                            |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::setItemText ( unsigned long lIndex,
                                          const char* pszString
#ifdef IC_USE_CB
                                        , bool updateEntryField )
#else
                                         )
#endif
{
#ifdef IC_PM
   IEventResult evt = handle().sendEvent(LM_SETITEMTEXT,
                              IEventParameter1(lIndex),
                              IEventParameter2((unsigned long)pszString));
   if (!(evt.asUnsignedLong()))
      ITHROWGUIERROR("LM_SETITEMTEXT");
#endif //IC_PM

#ifdef IC_WIN
   bool wasSelected = isSelected( lIndex );
   IEventResult itd = handle().sendEvent(LM_QUERYITEMHANDLE,
                              IEventParameter1(lIndex),
                              IEventParameter2(0));
   if (itd.asLong() == LIT_ERROR)
      ITHROWSYSTEMERROR(LIT_ERROR, "setItemText", IBaseErrorInfo::accessError,
                        IException::recoverable);
   IEventResult evt = handle().sendEvent(LM_DELETEITEM,
                              IEventParameter1(lIndex),
                              IEventParameter2(0));
   if (evt.asLong() == LIT_ERROR)
      ITHROWSYSTEMERROR(LIT_ERROR, "setItemText", IBaseErrorInfo::accessError,
                        IException::recoverable);
   evt = handle().sendEvent( LM_INSERTITEM,
                             IEventParameter1(lIndex),
                             IEventParameter2((unsigned long)pszString));
   if ( (evt.asLong() == LIT_ERROR) ||
        (evt.asLong() == LIT_MEMERROR) )
      ITHROWSYSTEMERROR(evt.asUnsignedLong(), "setItemText",
                        IBaseErrorInfo::accessError,
                        IException::recoverable);
   evt = handle().sendEvent( LM_SETITEMHANDLE,
                             IEventParameter1(lIndex),
                             IEventParameter2(itd));
   if (evt.asLong() == LIT_ERROR)
      ITHROWSYSTEMERROR(LIT_ERROR, "setItemText", IBaseErrorInfo::accessError,
                        IException::recoverable);
   if (wasSelected)
      IC_CLASSNAME::select( lIndex );
#endif //IC_WIN

#ifdef IC_PMWIN
#ifdef IC_USE_CB
   // For comboboxes, if the item being changed is the selected item,
   // then update the entry field string as well with the new text
   if (( updateEntryField ) && ( isSelected(lIndex) ))
     setText( pszString );
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
   Widget        listBox;
   char          *replaceItem;
   XmString      string;
   bool       isSelected;
   unsigned long position;

   listBox = IC_CLASSPRIV::listBoxHandle (*this);

   // See if the index is valid.
   if (lIndex >= count())
   {
      ITHROWLIBRARYERROR1 (IC_INVALID_INDEX,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable,
                           IString (lIndex));
   }

   position = lIndex + 1;
   // Check to see whether original string is selected.
   isSelected = XmListPosSelected (listBox, position);

   // Create compound string.
   replaceItem = XtMalloc (strlen (pszString) + 1);
   sprintf (replaceItem, "%s", pszString);
   string = XmStringCreateLocalized (replaceItem);

   XmListReplaceItemsPos (listBox,
                          &string,
                          1,
                          position);

   XmStringFree (string);
   XtFree (replaceItem);

   if (isSelected)
   {
      XmListUpdateSelectedList (listBox);
      select (position-1);
   }
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::setItemText                                                    |
|                                                                              |
| Change the text of an item currently in the list.                            |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::setItemText ( unsigned long lIndex,
                                          const IResourceId& resid
#ifdef IC_USE_CB
                                        , bool updateEntryField )
#else
                                         )
#endif
{
  IString newText = resid.resourceLibrary().loadString(resid);

#ifdef IC_USE_CB
  return setItemText( lIndex, (char *)newText, updateEntryField );
#else
  return setItemText( lIndex, (char *)newText );
#endif
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IC_CLASSNAME::select                                                         |
|                                                                              |
| Set an item as selected in the list.                                         |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::select ( unsigned long index, bool select )
{
   IEventResult evt;
#ifdef IC_WIN
#ifndef IC_USE_CB
   // For multiple select listboxes in Windows, use a different API
   if (isExtendedSelect() || isMultipleSelect())
   {
     evt = handle().sendEvent( LB_SETSEL,
                               IEventParameter1(select),
                               IEventParameter2(index));
   }
   else
#endif
   {
     evt = handle().sendEvent( LM_SELECTITEM,
                               IEventParameter1(select ? index : -1),
                               IEventParameter2(0) );
     evt = parent()->handle().sendEvent( WM_COMMAND,
                               IEventParameter1(id(),CBN_SELENDOK),
                               IEventParameter2(handle()));
     if (!select )
     {
        // If LB_SETCURSEL is called with -1 (deselect all items),
        // LB_ERR is always returned.  To be consistent, check if
        // index is out of range.
        if (index >= this->count() && index != LIT_NONE)
           evt = LB_ERR;
        else
           evt = LB_OKAY;
     }
   }

   if (evt.asLong() == LIT_NONE)
#else
     evt = handle().sendEvent( LM_SELECTITEM,
                               IEventParameter1(index),
                               IEventParameter2(select));

   if (!(evt.asUnsignedLong()))
#endif
      ITHROWGUIERROR("LM_SELECTITEM");

#ifdef IC_WIN
#ifndef IC_USE_CB
   // If the item selected is not in view, Windows incorrectly scrolls the list.
   // Check here to make sure the item selected is in view.
   if (select)
   {
      IEventData topIndex = handle().sendEvent(
                                       LB_GETTOPINDEX,
                                       IEventParameter1( 0 ),
                                       IEventParameter2( 0 ));
      if (topIndex > index)
         evt = handle().sendEvent( LM_SETTOPINDEX,
                                   IEventParameter1( index ),
                                   IEventParameter2( 0 ));
      if (evt.asLong() == LIT_NONE)
         ITHROWGUIERROR("LM_SETTOPINDEX");
   }
#endif

   // Windows does not send the selected notification message if we
   // select via API.  Generate one so select handler works.
   ITRACE_ALL("Sending select event to parent");
   IWindowHandle parent = this->parent()->handle();
   parent.sendEvent( WM_COMMAND,
                     IEventParameter1( (unsigned short)this->id(),
#ifdef IC_USE_CB
                                       CBN_LBSELECT ),
#else
                                       IC_UM_LN_SELECT ),
#endif
                     IEventParameter2( this->handle() ) );
#endif
   return *this;
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IC_CLASSNAME::deselect                                                       |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::deselect ( unsigned long index )
{
   select(index, false);
   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::isSelected                                                     |
|                                                                              |
| Query whether given item is selected.                                        |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::isSelected ( unsigned long index ) const
{
#ifdef IC_PMWIN
   IEventResult evt = handle().sendEvent(LM_QUERYSELECTION,
#ifdef IC_PM
                              IEventParameter1(index - 1),
#endif
#ifdef IC_WIN
#ifdef IC_USE_CB
                              IEventParameter1(0),
#else
                              IEventParameter1(index),
#endif // IC_USE_CB
#endif // IC_WIN
                              IEventParameter2(0));

   if ((evt.asLong() != LIT_NONE)
#ifdef IC_PM
        && (evt.asUnsignedLong() == index))
#endif
#ifdef IC_WIN
#ifdef IC_USE_CB
        && (evt.asUnsignedLong() == index))
#else
        && (evt.asLong() > 0))
#endif // IC_USE_CB
#endif // IC_WIN
      return true;
   else
      return false;
#endif //IC_PMWIN

#ifdef IC_MOTIF
   Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

   return (bool) XmListPosSelected (listBox, index + 1);
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::deselectAll                                                    |
|                                                                              |
| Set all items as unselected.                                                 |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::deselectAll ( )
{
#ifdef IC_PMWIN
#ifndef IC_USE_CB
#ifdef IC_PM
   // Workaround for OS/2 problem with deselecting all items in a listbox for
   // which the LS_EXTENDEDSEL style is set and the LS_MULTIPLESEL style is
   // not set.  By adding the LS_MULTIPLESEL style the deselection will take
   // place.
   if (isExtendedSelect()  &&  !isMultipleSelect())
   {
      enableMultipleSelect();
      deselectAll();
      disableMultipleSelect();
   }
   else
#endif
#endif
   {
     IC_CLASSNAME::select( LIT_NONE, false );
   }
#endif //IC_PMWIN

#ifdef IC_MOTIF
   Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

   XmListDeselectAllItems (listBox);

   notifyObservers(INotificationEvent(IC_CLASSNAME::selectId, *this));
#endif //IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::numberOfSelections                                             |
|                                                                              |
| Return the number of selected items in the list.                             |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::numberOfSelections ( ) const
{
#ifdef IC_PMWIN
   unsigned long
     workSelect,
     selCount = 0;
   workSelect = LIT_FIRST;
#ifndef IC_USE_CB
   // For a multiple or extended selection IListBox, iterate through the
   // selected items.
   if ((style() & multipleSelect.asUnsignedLong())  ||
       ((style() & extendedSelect.asUnsignedLong())
            == extendedSelect.asUnsignedLong()) )
   {
#ifdef IC_PM
      for ( ;
            (workSelect = handle().sendEvent(LM_QUERYSELECTION,
                                             IEventParameter1(workSelect),
                                             IEventParameter2(0)))
                                          != LIT_NONE;
            selCount++ )
      { }
#endif
#ifdef IC_WIN
      // If multiple select in Windows, use different API to get count
      selCount = handle().sendEvent( LB_GETSELCOUNT,
                                     IEventParameter1(0),
                                     IEventParameter2(0) );
#endif
   }
   else
#endif
   {
#ifdef IC_PM
      IEventResult evt = handle().sendEvent(LM_QUERYSELECTION,
                                            IEventParameter1(workSelect),
                                            IEventParameter2(0));
#endif
#ifdef IC_WIN
      IEventResult evt = handle().sendEvent(IC_QUERYCURSELECTION,
                                            IEventParameter1(0),
                                            IEventParameter2(0));
#endif

      if (evt.asLong() != LIT_NONE)
         selCount = 1;
   }
   return selCount;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int     selectedItemCount;
  Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

  XtVaGetValues (listBox,
                 XmNselectedItemCount, &selectedItemCount,
                 NULL);

  return selectedItemCount;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::selection                                                      |
|                                                                              |
| Return the index of the first selected item (or LIT_NONE).                   |
------------------------------------------------------------------------------*/
long IC_CLASSNAME::selection ( ) const
{
#ifdef IC_PMWIN
#ifdef IC_PM
   IEventResult evt = handle().sendEvent(LM_QUERYSELECTION,
                                         IEventParameter1(LIT_FIRST),
                                         IEventParameter2(0));
#endif
#ifdef IC_WIN
   IEventResult evt;
#ifndef IC_USE_CB
   if (isMultipleSelect() || isExtendedSelect())
   {
      unsigned long firstSelection;
      evt = handle().sendEvent( LB_GETSELITEMS,
                                IEventParameter1( 1 ),
                                IEventParameter2( &firstSelection ));
      if (evt.asLong())
         return firstSelection;
      else
         return LIT_NONE;
   }
   else
#endif
   {
      evt = handle().sendEvent( IC_QUERYCURSELECTION,
                                IEventParameter1(0),
                                IEventParameter2(0));
   }
#endif
   return evt.asLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int     positionCount, *positionList;
  ulong   position;
  Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

  if (XmListGetSelectedPos (listBox,
                            &positionList,
                            &positionCount))
  {
     position = positionList[0];
     XtFree ((char*) positionList);
     return position - 1;
  }
  else
     return (long)notFound;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::setTop                                                         |
|                                                                              |
| Scroll the list so the provided item is at the top.                          |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::setTop ( unsigned long lIndex )
{
#ifdef IC_PMWIN
#if ((defined IC_WIN) && (defined IC_USE_CB))
   if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
   {
      IEventResult evt = sendEvent( CB_SETTOPINDEX,
                                    IEventParameter1( lIndex ),
                                    IEventParameter2( 0 ));
      if (evt.asLong() == LIT_ERROR)
         ITHROWSYSTEMERROR(LIT_ERROR,
                           "setTop",
                           IBaseErrorInfo::accessError,
                           IException::recoverable);
   }
#endif

// If PM listbox/combobox OR Windows listbox 
#if (defined (IC_PM) || !defined (IC_USE_CB))
   handle().sendEvent(LM_SETTOPINDEX,
                      IEventParameter1(lIndex),
                      IEventParameter2(0));
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

  XmListSetPos (listBox, lIndex+1);
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::top                                                            |
|                                                                              |
| Return the index of the item currently at the top of the visible list.       |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::top ( ) const
{
#ifdef IC_PMWIN
   IEventResult evt = 0;
#if ((defined IC_WIN) && (defined IC_USE_CB))
   if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
   {
      evt = handle().sendEvent( CB_GETTOPINDEX,
                                IEventParameter1( 0 ),
                                IEventParameter2( 0 ));
   }
#endif
// If PM listbox/combobox OR Windows listbox 
#if (defined (IC_PM) || !defined (IC_USE_CB))
   evt = handle().sendEvent(LM_QUERYTOPINDEX,
                            IEventParameter1(0),
                            IEventParameter2(0));
#endif
   if (evt.asLong() == LIT_NONE)
      ITHROWGUIERROR("LM_QUERYTOPINDEX");
   return evt.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int     topItemPosition, itemCount;
  Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

  XtVaGetValues (listBox,
                 XmNtopItemPosition, &topItemPosition,
                 XmNitemCount, &itemCount,
                 NULL);

  if (itemCount == 0)
  {
     ITHROWLIBRARYERROR (IC_EMPTY_LIST,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
  }

  return topItemPosition - 1;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::setItemHandle                                                  |
|                                                                              |
| Set the handle of an item.                                                   |
------------------------------------------------------------------------------*/
IC_CLASSNAME& IC_CLASSNAME::setItemHandle ( unsigned long lIndex,
                                            unsigned long ulHandle )
{
#ifdef IC_PMWIN
   IEventResult evt = handle().sendEvent(LM_SETITEMHANDLE,
                                         IEventParameter1(lIndex),
                                         IEventParameter2(ulHandle));
#ifdef IC_WU
   // For Wind/U a result of Zero means the SetItemHandle is successful
   if (  evt.asUnsignedLong() != 0  )
#else
   if (!(evt.asUnsignedLong()))
#endif
      ITHROWGUIERROR("LM_SETITEMHANDLE");
#endif //IC_PMWIN

#ifdef IC_MOTIF
  // See if the index is valid.
  if (lIndex >= count())
  {
     ITHROWLIBRARYERROR1 (IC_INVALID_INDEX,
                          IBaseErrorInfo::invalidParameter,
                          IException::recoverable,
                          IString (lIndex));
  }

  if( PRIVATEDATA->itemHandlesList->containsElementWithKey(lIndex) )
  {
    IItemHandleElement* handleElement = PRIVATEDATA->itemHandlesList->elementWithKey (lIndex);
    PRIVATEDATA->itemHandlesList->removeElementWithKey(lIndex);
    delete handleElement;
  }

  IItemHandleElement* newElement = new IItemHandleElement (lIndex, ulHandle);
  PRIVATEDATA->itemHandlesList->addOrReplaceElementWithKey (newElement);
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::itemHandle                                                     |
|                                                                              |
| Return an item's handle.                                                     |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::itemHandle ( unsigned long lIndex ) const
{
#ifdef IC_PMWIN
   IEventResult evt = handle().sendEvent(LM_QUERYITEMHANDLE,
                                         IEventParameter1(lIndex),
                                         IEventParameter2(0));
#ifdef IC_WIN
   // OS/2 docs do not state whether an error is returned if lIndex
   // is invalid, so for now don't check for return code on OS/2.
   if (evt.asLong() == LIT_NONE)
      ITHROWGUIERROR("LM_QUERYITEMHANDLE");
#endif

   return evt.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  IItemHandleElement* itemElement;
  try {
    itemElement = PRIVATEDATA->itemHandlesList->elementWithKey (lIndex);
  }
  catch (INotContainsKeyException) {
    return 0;
  }

  return itemElement->itemHandle();
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::locateText                                                     |
|                                                                              |
| Return the index of the item which matches the search string.  A search can  |
| be constrained with a case sensitivity flag, a check substring flag, and/or  |
| a starting index.                                                            |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::locateText ( const char* pszSearchString,
                                         bool caseSensitive,
                                         SearchType search,
                                         unsigned long lIndex ) const
{
#ifdef IC_PMWIN
   unsigned long eventId = LM_SEARCHSTRING;
   IEventResult evt;
   if (lIndex == IC_CLASSNAME::first)
      lIndex = LIT_FIRST;

#ifdef IC_PM
   unsigned long ulFindOptions = 0;

   if (caseSensitive)
      ulFindOptions |= LSS_CASESENSITIVE;
   switch (search)
   {
     case prefix:
          ulFindOptions |= LSS_PREFIX;
          break;
     case substring:
          ulFindOptions |= LSS_SUBSTRING;
          break;
     //Don't have to do anything for exactMatch, since it is the
     //OS/2 default if neither of the above two styles isn't set
   }

   evt =
      handle().sendEvent( eventId,
                          IEventParameter1((unsigned short)ulFindOptions,
                                           (unsigned short)lIndex),
                          IEventParameter2((unsigned long)pszSearchString));
#endif //IC_PM
#ifdef IC_WIN
   if (search == exactMatch)
      eventId = IC_SEARCHSTRINGEXACT;
   else if (search == prefix)
      eventId = LM_SEARCHSTRING;

   bool searching = true;
   unsigned long searchIndex = lIndex,
                 prevSearchIndex,
                 itemCount = this->count(),
                 searchCount = 0;

   IString searchString( pszSearchString ),
           matchString;
   if (!caseSensitive)
      searchString.lowerCase();
   while (searching)
   {
      if (search == substring)
      {
         // Case for substring search.
         searchIndex++;
         if (searchIndex == itemCount)
            searchIndex = LIT_ERROR;
      }
      else
      {
         prevSearchIndex = searchIndex;
         searchIndex = handle().sendEvent( eventId,
                                IEventParameter1(searchIndex),
                                IEventParameter2((unsigned long)pszSearchString));
         searchCount++;
         // Windows doesn't support case-sensitive search, so worst case is if there's a match
         // but it's not case-sensitive.  Return not found if attempt search more times than items
         // in the list.
         if (searchCount > itemCount)
           searchIndex = LIT_ERROR;

      }

      if (searchIndex != LIT_ERROR)
      {
         if (search == substring || caseSensitive)
         {
            IString matchString( itemText( searchIndex ));
            if (!caseSensitive)
               matchString.lowerCase();

            if (matchString.includes( searchString ))
               searching = false;
         }
         else
            searching = false;
      }
      else
         searching = false;
   }
   evt = searchIndex;
#endif //IC_WIN

   switch (evt.asLong())
   {
#ifdef IC_PM
      case LIT_ERROR:
         ITHROWGUIERROR("LM_SEARCHSTRING");
         break;
#endif
      case LIT_NONE:
         return IC_CLASSNAME::notFound;
      default:
         break;
   }
   return evt.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
   Widget        listBox;
   char*         listItem;
   IString       *listString, *locateString;
   XmStringTable itemList;
   int           i = lIndex, searchCount = 0, lbItemCount;
   bool       found=false;

   listBox = IC_CLASSPRIV::listBoxHandle (*this);

   XtVaGetValues (listBox,
                  XmNitemCount, &lbItemCount,
                  XmNitems, &itemList,
                  NULL);

   if (caseSensitive)
      locateString = new IString (pszSearchString);
   else
      locateString = new IString (IString::lowerCase ((IString) pszSearchString));

  if( i == IC_CLASSNAME::first )
    i=0;

  if( ( i >= lbItemCount ) || (i<0) )//start index is not in the list
    i=0;

   while ( !found && (searchCount < lbItemCount) )
   {
      if (XmStringGetLtoR (itemList[i],
                           XmFONTLIST_DEFAULT_TAG,
                           &listItem))
      {
         if (caseSensitive)
            listString = new IString (listItem);
         else
            listString = new IString (IString::lowerCase (listItem));
         // d7950 - Need to free listItem
         XtFree( listItem );

         switch (search) {
         case prefix:
            if (listString -> indexOf (*locateString) == 1)
               found = true;
            break;
         case substring:
            if (listString -> includes (*locateString))
               found = true;
            break;
         case exactMatch:
            if (*listString == *locateString)
               found = true;
            break;
         }
         delete listString;
      }

      // increment index; wrap if necessary
      if ( (++i == lbItemCount) && !found )
        i = 0; 
      searchCount++;
   }

   delete locateString;
   if (!found)
      return notFound;

   return i-1;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::count                                                          |
|                                                                              |
| Return the number of items in this list.                                     |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::count ( ) const
{
#ifdef IC_PMWIN
   IEventResult evt = handle().sendEvent(LM_QUERYITEMCOUNT,
                                         IEventParameter1(0),
                                         IEventParameter2(0));
   return evt.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int itemCount;
  Widget  listBox = IC_CLASSPRIV::listBoxHandle (*this);

  XtVaGetValues (listBox,
                 XmNitemCount, &itemCount,
                 NULL);
  return (unsigned long) itemCount;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: Cursor                                               |
|                                                                              |
| Construct a cursor for the IComboBox or IListBox.                            |
------------------------------------------------------------------------------*/
IC_CLASSNAME::Cursor :: Cursor( const IC_CLASSNAME& rlb,
                                Filter type )
  : rlbCl( rlb ),
    cursorTypeCl( type ),
#ifdef IC_WIN
    fCursorData( new IC_CLASSNAMECURSORDATA )
#endif //IC_WIN
#ifdef IC_MOTIFPM
    fCursorData( 0 )
#endif
{
  // The cursor is initially invalid until it is positioned.
  invalidate();
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: ~Cursor                                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IC_CLASSNAME::Cursor :: ~Cursor()
{
#ifdef IC_WIN
  delete [] fCursorData;
#endif
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: setToFirst                                           |
|                                                                              |
| Position cursor to the first item in the list according to cursor type.      |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::Cursor :: setToFirst()
{
  bool bSuccess = false;

  if (!rlbCl.isEmpty())
  {
    if(cursorTypeCl==selectedItems)
    {
#ifdef IC_WIN
       //  Get a list of selected items from the listbox.  Store the
       // list in the cursor private data, as well as its size and
       // current index.
       if (fCursorData->flist)
         {
         delete [] fCursorData->flist;
         fCursorData->flist = 0;
         }
       unsigned long lbcount = rlbCl.numberOfSelections();
       if (lbcount)
         {
         fCursorData->flist = new long[ lbcount ];
#ifndef IC_USE_CB
         if ( rlbCl.isMultipleSelect() || rlbCl.isExtendedSelect() )
            {
            fCursorData->fcount =
              rlbCl.handle().sendEvent(
                 LB_GETSELITEMS,
                 IEventParameter1(lbcount),
                 IEventParameter2(fCursorData->flist) ).asLong();
            }
         else
#endif
            {
            // Don't need the array and such for combobox or single
            // select listbox but the rest of the code still works
            // if we fake it here.
            fCursorData->flist[0] =
               rlbCl.handle().sendEvent( IC_QUERYCURSELECTION ).asLong();
            if (fCursorData->flist[0] != LIT_ERROR)
               fCursorData->fcount = 1;
            }
         }
       if ((lbcount) && (lbcount == fCursorData->fcount))
         {
         fCursorData->fcurrent = 0;
         lClCurrent = fCursorData->flist[0];
         bSuccess = true;
         sameValidation = rlbCl.needValidation;
         }
#endif //IC_WIN

#ifdef IC_PM
       IEventResult evt =
          rlbCl.handle().sendEvent(LM_QUERYSELECTION,
                                   IEventParameter1(LIT_FIRST),
                                   IEventParameter2(0));
       if (evt.asLong() != LIT_NONE)
       {
          lClCurrent = evt.asUnsignedLong();
          bSuccess = true;
          sameValidation = rlbCl.needValidation;
       }
#endif //IC_PM

#ifdef IC_MOTIF
        int   positionCount, *positionList;

        if (XmListGetSelectedPos (IC_CLASSPRIV::listBoxHandle(this->rlbCl),
                                  &positionList,
                                  &positionCount))
        {
           this->lClCurrent = positionList[0] - 1;
           XtFree ((char*) positionList);
           bSuccess = true;
           this->sameValidation = this->rlbCl.needValidation;
        }
#endif //IC_MOTIF
    }
    else
    {
       lClCurrent = 0;
       bSuccess = true;
       sameValidation = rlbCl.needValidation;
    }
  }

  return bSuccess;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor::setToNext                                              |
|                                                                              |
| Position cursor to the next item in the list according to cursor type.       |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::Cursor :: setToNext()
{
  bool bSuccess = false;

  if (isValid()  &&  (lClCurrent < (rlbCl.count() - 1)))
  {
     if (cursorTypeCl==selectedItems)
     {
#ifdef IC_WIN
        fCursorData->fcurrent++;
        if (fCursorData->fcount > fCursorData->fcurrent)
           {
           lClCurrent = fCursorData->flist[fCursorData->fcurrent];
           bSuccess = true;
           }
#endif //IC_WIN

#ifdef IC_PM
        IEventResult evt =
           rlbCl.handle().sendEvent(LM_QUERYSELECTION,
                                    IEventParameter1(lClCurrent),
                                    IEventParameter2(0));
        if ((evt.asLong() != LIT_NONE)  &&
            (evt.asUnsignedLong() != lClCurrent))
        {
           lClCurrent = evt.asUnsignedLong();
           bSuccess = true;
        }
#endif //IC_PM

#ifdef IC_MOTIF
        int   i, positionCount, *positionList;

        if (XmListGetSelectedPos (IC_CLASSPRIV::listBoxHandle(this->rlbCl),
                                  &positionList,
                                  &positionCount))
        {
           for (i = 0; i < positionCount && !bSuccess; i++)
              if (positionList[i]-1 > this->lClCurrent)
              {
                 this->lClCurrent = positionList[i]-1;
                 XtFree ((char*) positionList);
                 bSuccess = true;
              }
        }
#endif //IC_MOTIF
     }
     else
     {
        lClCurrent++;
        bSuccess = true;
     }
  }


  if (!bSuccess)
     invalidate();

  return bSuccess;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: setToPrevious                                        |
|                                                                              |
| Position cursor to previous item in the list according to cursor type.       |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::Cursor :: setToPrevious ( )
{
  bool bSuccess = false;

  if (isValid()  &&  (lClCurrent != 0))
  {
     if (cursorTypeCl == selectedItems)
     {
#ifdef IC_WIN
        if (fCursorData->fcurrent > 0)
           {
           fCursorData->fcurrent--;
           lClCurrent = fCursorData->flist[fCursorData->fcurrent];
           bSuccess = true;
           }
#endif //IC_WIN

#ifdef IC_PM
        long
          lLast  = -1,
          lFound = -1;
        IC_CLASSNAME::Cursor tempCurs(rlbCl,selectedItems);

        for (tempCurs.setToFirst();
             tempCurs.isValid()  &&  lFound == -1;
             tempCurs.setToNext())
        {
           if (tempCurs.asIndex() == asIndex())
           {
              lFound = lLast;
           }
           lLast = tempCurs.asIndex();
        }

        if (lFound != -1)
        {
           lClCurrent = lFound;
           bSuccess = true;
        }
#endif //IC_PM

#ifdef IC_MOTIF
        int   i, positionCount, *positionList;

        if (XmListGetSelectedPos (IC_CLASSPRIV::listBoxHandle(this->rlbCl),
                                  &positionList,
                                  &positionCount))
        {
           for (i = positionCount - 1; i >= 0 && !bSuccess; i--)
           {
              if (positionList[i]-1 < this->lClCurrent)
              {
                 this->lClCurrent = positionList[i] - 1;
                 XtFree ((char*) positionList);
                 bSuccess = true;
              }
           }
        }
#endif //IC_MOTIF
     }
     else
     {
       lClCurrent--;
       bSuccess = true;
     }
  }

  if (!bSuccess)
     invalidate();
  return bSuccess;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: setToLast                                            |
|                                                                              |
| Position cursor to the last item in the list according to cursor type.       |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::Cursor :: setToLast ( )
{
  bool bSuccess = false;

  if (!rlbCl.isEmpty())
  {
     if (cursorTypeCl == selectedItems)
     {
#ifdef IC_WIN
        if ( this->setToFirst() )
           {
           fCursorData->fcurrent = fCursorData->fcount - 1;
           lClCurrent = fCursorData->flist[fCursorData->fcurrent];
           bSuccess = true;
           }
#endif //IC_WIN

#ifdef IC_PM
        long lLast = -1;
        IC_CLASSNAME::Cursor tempCurs(rlbCl,selectedItems);

        for (tempCurs.setToFirst();
             tempCurs.isValid();
             tempCurs.setToNext())
        {
           lLast = tempCurs.asIndex();
        }
        if (lLast != -1)
        {
           lClCurrent = lLast;
           bSuccess = true;
           sameValidation = rlbCl.needValidation;
        }
#endif  //IC_PM

#ifdef IC_MOTIF
        int   positionCount, *positionList;

        if (XmListGetSelectedPos (IC_CLASSPRIV::listBoxHandle(this->rlbCl),
                                  &positionList,
                                  &positionCount))
        {
           this->lClCurrent = positionList[positionCount-1] - 1;
           XtFree ((char*) positionList);
           bSuccess = true;
        }
#endif //IC_MOTIF
     }
     else
     {
        lClCurrent = rlbCl.count() - 1;
        bSuccess = true;
        sameValidation = rlbCl.needValidation;
     }
  }

  return bSuccess;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: setToIndex                                           |
|                                                                              |
| Position cursor to specified index regardless of cursor type.                |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::Cursor :: setToIndex ( unsigned long lIndex )
{
  bool bSuccess = false;

  if ((long)lIndex >= 0  &&  lIndex < rlbCl.count())
  {
     lClCurrent = lIndex;
     bSuccess = true;
     sameValidation = rlbCl.needValidation;
  }

  return bSuccess;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: isValid                                              |
|                                                                              |
| Determine whether the cursor is valid.  The list's validation count must be  |
| equal to the cursor's current validation count.  For a selected item cursor, |
| the item it points to must be also be selected.                              |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::Cursor :: isValid ( ) const
{
  bool bValid = false;
  ITRACE_ALL( IString("Cursor::isValid lclCurrent=") +
              IString(lClCurrent) +
              IString(" sameValidation=") +
              IString(sameValidation) );

  if ((long)lClCurrent >= 0  &&
      lClCurrent < rlbCl.count()  &&
      lClCurrent != LIT_ERROR)
  {
     if (cursorTypeCl == selectedItems)
     {
#ifdef IC_WIN
        if ( (fCursorData->flist) &&
             (sameValidation == rlbCl.needValidation) )
           bValid = true;
#endif //IC_WIN

#ifdef IC_PM
        IEventResult evt =
           rlbCl.handle().sendEvent(LM_QUERYSELECTION,
                                    IEventParameter1(lClCurrent - 1),
                                    IEventParameter2(0));
        if ((evt.asUnsignedLong() == lClCurrent)  &&
            (sameValidation == rlbCl.needValidation))
           bValid = true;
#endif  //IC_PM

#ifdef IC_MOTIF
        bValid = XmListPosSelected (IC_CLASSPRIV::listBoxHandle(this->rlbCl),
                                    this->lClCurrent + 1);
#endif //IC_MOTIF
     }
     else
     {
        if (sameValidation == rlbCl.needValidation)
           bValid = true;
     }
  }

  return bValid;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: invalidate                                           |
------------------------------------------------------------------------------*/
void IC_CLASSNAME::Cursor :: invalidate ( )
{
  lClCurrent = LIT_ERROR;
#ifdef IC_WIN
  if (fCursorData->flist)
    {
    delete [] fCursorData->flist;
    fCursorData->flist = 0;
    }
#endif
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::Cursor :: asIndex                                              |
------------------------------------------------------------------------------*/
unsigned long IC_CLASSNAME::Cursor :: asIndex ( ) const
{
  return lClCurrent;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::elementAt                                                      |
|                                                                              |
| Return the item string at the cursor's position.                             |
------------------------------------------------------------------------------*/
IString IC_CLASSNAME::elementAt ( const IC_CLASSNAME::Cursor& cursor ) const
{
  IASSERTSTATE(cursor.isValid());
  unsigned long ulIndex = cursor.asIndex();
  return itemText(ulIndex);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::isEmpty                                                        |
------------------------------------------------------------------------------*/
bool IC_CLASSNAME::isEmpty ( ) const
{
  return (count() == 0);
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::changeCount                                                    |
------------------------------------------------------------------------------*/
unsigned long  IC_CLASSNAME :: changeCount ( ) const
{
  return needValidation;
}

/*------------------------------------------------------------------------------
| IC_CLASSNAME::incrementChangeCount                                           |
------------------------------------------------------------------------------*/
void IC_CLASSNAME :: incrementChangeCount  ( )
{
  needValidation++;
}
#endif // ILISTBX3 header guard
