// Revision: 91 1.34.3.1 source/ui/basectl/ilistbas.cpp, listctls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ilistbas.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ilistbas.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

extern "C"
{
  #define INCL_WINLISTBOXES      // LIT_xxx, LM_xxx, LS_xxx
  #define INCL_WINSYS            // SV_xxx, PP_xxx
  #define INCL_GPIPRIMITIVES     // GpiQueryTextBox, TXTBOX_xxx
  #include <iwindefs.h>
  #include <ilistbox.h>
}

#ifdef IC_MOTIF
  #include <X11/IntrinsicP.h>     // for CoreP.h
  #include <X11/CoreP.h>          // for core.being_destroyed
  #include <Xm/List.h>
  #include <Xm/ListP.h>
  #include <Xm/ScrolledW.h>
#endif //IC_MOTIF

#include <inotifev.hpp>
#include <ilistbas.hpp>
#ifdef IC_MOTIF
  #include <ilistprv.hpp>
  #include <iclcreat.h>
#endif //IC_MOTIF
#include <istring.hpp>
#include <ireslib.hpp>
#include <iapp.hpp>
#include <ihandle.hpp>
#include <ihandler.hpp>
#include <iinhratt.hpp>
#include <iexcept.hpp>
#include <icconst.h>
#include <itrace.hpp>
#include <icolor.hpp>
#include <ifont.hpp>
#include <iplatfrm.hpp>
#include <ilanglvl.hpp>
#include <ikeyevt.hpp>
#ifdef IC_WIN
  #include <ibrushdr.hpp>
#endif
// Segment definitions
#ifdef IC_PAGETUNE
  #define _ILISTBAS_CPP_
  #include <ipagetun.h>
#endif


/***************************************************************/
/* Public list box styles.                                     */
/***************************************************************/
const IBaseListBox::Style
  IBaseListBox::multipleSelect    = LS_MULTIPLESEL,            // 0x00000001
#ifdef IC_PMWIN
  IBaseListBox::drawItem          = LS_OWNERDRAW,              // 0x00000002
#endif //IC_PMWIN
  IBaseListBox::noAdjustPosition  = LS_NOADJUSTPOS,            // 0x00000004
  IBaseListBox::horizontalScroll  = LS_HORZSCROLL,             // 0x00000008
  IBaseListBox::extendedSelect    = LS_EXTENDEDSEL,            // 0x00000010
#ifdef IC_PMWIN
  IBaseListBox::border3D          ( 0, IWS_BORDER3D ),
#endif
#ifdef IC_MOTIF
  IBaseListBox::border3D          ( 0 ),
#endif
#ifdef IC_PMWIN
  IBaseListBox::classDefaultStyle ( LS_HORZSCROLL  |
				    LS_NOADJUSTPOS |
				    WS_VISIBLE,
				    IWS_BORDER3D );
#endif
#ifdef IC_MOTIF
  IBaseListBox::classDefaultStyle ( LS_HORZSCROLL  |
				    LS_NOADJUSTPOS |
				    WS_VISIBLE );
#endif

const unsigned long
  IBaseListBox::notFound = (unsigned long)ULONG_MAX,
  IBaseListBox::first    = (unsigned long)ULONG_MAX;

/**********************************************************************/
/* Default style for new objects (initial value)                      */
/**********************************************************************/
#ifdef IC_PMWIN
  IBaseListBox::Style IBaseListBox::currentDefaultStyle ( LS_HORZSCROLL |
							  LS_NOADJUSTPOS |
							  WS_VISIBLE,
							  IWS_BORDER3D );
#endif //IC_PMWIN
#ifdef IC_MOTIF
  IBaseListBox::Style IBaseListBox::currentDefaultStyle ( LS_HORZSCROLL |
							  LS_NOADJUSTPOS |
							  WS_VISIBLE);
#endif //IC_MOTIF

#ifdef IC_WIN
#pragma enum(4)
#pragma pack(push,4)

class IBaseListHandler : public IHandler {
/*******************************************************************************
* The IBaseListHandler class is a default handler for list boxes.              *
*******************************************************************************/
typedef IHandler
  Inherited;
public:

/*------------------------------- Constructors -------------------------------*/
  IBaseListHandler ( );

virtual
  ~IBaseListHandler ( );

/*----------------------------- Event Processing -----------------------------*/
virtual bool
  dispatchHandlerEvent ( IEvent& event );

}; // class IBaseListHandler

#pragma pack(pop)
#pragma enum(pop)
#endif //IC_WIN

#ifdef IC_PMWIN
#pragma enum(4)
#pragma pack(push,4)

class IBaseListBoxData {
/*******************************************************************************
* The IBaseListBoxData class encapsulates private data and functions used by the *
* IBaseListBox class.  An object of this class is created in the IBaseListBox  *
* constructors.                                                                *
*******************************************************************************/
public:
  IBaseListBoxData ( ) ;

 ~IBaseListBoxData ( );

unsigned long
  minimumRows,
  minimumCharacters;

#ifdef IC_WIN
IBaseListHandler
  fdefaultHandler;
IBrushHandler
  fBrushHandler;
#endif // IC_WIN

private:
  IBaseListBoxData ( const IBaseListBoxData& );
IBaseListBoxData
 &operator=    ( const IBaseListBoxData& );

}; // IBaseListBoxData

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IBaseListBoxData::IBaseListBoxData                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IBaseListBoxData::IBaseListBoxData ( )
    : minimumRows( 4 )
    , minimumCharacters( 25 )
#ifdef IC_WIN
    , fBrushHandler( )
#endif // IC_WIN
#ifdef IC_MOTIFWIN
    , fdefaultHandler( )
#endif //_MOTIFWIN
{ }

/*------------------------------------------------------------------------------
| IBaseListBoxData::~IBaseListBoxData                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IBaseListBoxData::~IBaseListBoxData ( )
{
}
#endif //IC_PMWIN

#if IC_EXTENDED_RTTI_ENABLED
IBaseListBoxData::IBaseListBoxData(const IBaseListBoxData&)
{ }
#endif // IC_EXTENDED_RTTI_ENABLED

#ifdef IC_MOTIF

IFont
IBaseListBox::font() const
{
    IFont font;
    XmFontList fontList = 0;
    XtVaGetValues(fBaseListBoxData->lbHandle,
                  XmNfontList, &fontList, NULL);
    return IFont(fontList);
}

/*------------------------------------------------------------------------------
| IBaseListBoxData::listBoxHandle                                              |
|                                                                              |
| Returns the handle of the XmList widget.                                     |
------------------------------------------------------------------------------*/
Widget IBaseListBoxData :: listBoxHandle (const IBaseListBox& listBox)
{
  Cardinal numChildren;
  WidgetList children;
  XtVaGetValues( ( Widget) listBox.handle(),
    XmNnumChildren, &numChildren,
    XmNchildren, &children,
    NULL );

  for ( int i = 0; i < numChildren; i++ )
    {
      if (XtIsSubclass((Widget)children[i], xmListWidgetClass))
	 return (Widget)children[i];
    }
  return 0;
}


/*-----------------------------------------------------------------------------
| ilistMotionXEventCallback:Callback procedure for passing mouse Motion events | 
-----------------------------------------------------------------------------*/
extern void _System ilistMotionXEventCallback(
    Widget    handle,
    XtPointer client_data,
    XEvent    *event,
    ::Boolean *continue_to_dispatch )
{
  IMODTRACE_ALL("IListBox::ilistMotionXEventCallback");

  // Retrieve the IWindow associated with the handle (Widget ID )
  IWindow *pwin = (IWindow *) client_data;
  if ( pwin != 0 )
  {
    pwin->sendEvent( xEvent(MotionNotify) ,            
		     (unsigned long) 0,
		     (unsigned long) event );
  }
} // end ilistXEventCallback()


/*------------------------------------------------------------------------------
| IBaseListBoxData::listBoxWidget                                              |
------------------------------------------------------------------------------*/
Widget IBaseListBoxData :: listBoxWidget ( Widget topParent )
{
  Cardinal numChildren;
  WidgetList children;
  XtVaGetValues( ( Widget) topParent,
    XmNnumChildren, &numChildren,
    XmNchildren, &children,
    NULL );

  for ( int i = 0; i < numChildren; i++ )
    {
      if (XtIsSubclass((Widget)children[i], xmListWidgetClass))
	 {
	 XtManageChild(children[i]);
	 return (Widget)children[i];
	 }
    }
  return 0;

}

/*------------------------------------------------------------------------------
| IBaseListBoxData::registerCallbacks                                          |
------------------------------------------------------------------------------*/
void IBaseListBoxData :: registerCallbacks (IBaseListBox* listBox)
{
  // Register the browseSelect callback.
  XtAddCallback ((Widget)lbHandle,
		 XmNbrowseSelectionCallback,
		 iwindowMotifCallback,
		 (XtPointer) listBox);

  // Register the enter callback.
  XtAddCallback ((Widget)lbHandle,
		 XmNdefaultActionCallback,
		 iwindowMotifCallback,
		 (XtPointer) listBox);

  // Register the extendedSelect callback.
  XtAddCallback ((Widget)lbHandle,
		 XmNextendedSelectionCallback,
		 iwindowMotifCallback,
		 (XtPointer) listBox);

  // Register the multipleSelect callback.
  XtAddCallback ((Widget)lbHandle,
		 XmNmultipleSelectionCallback,
		 iwindowMotifCallback,
		 (XtPointer) listBox);

  // Register the singleSelect callback.  Although we do not
  // provide a class for Motif single selection (browse is
  // default), XmSINGLE_SELECT may be specified in a defaults
  // file.
  XtAddCallback ((Widget)lbHandle,
		 XmNsingleSelectionCallback,
		 iwindowMotifCallback,
		 (XtPointer) listBox);


  // Explicitly add X event handler for listbox focus events
  XtAddEventHandler(
     (Widget)lbHandle,               // widget
     FocusChangeMask,                // ask for FocusIn and FocusOut events
     True,                           // indicate we also want nonmaskable events
     iwindowXEventCallback,          // event handler routine
     (XtPointer) listBox);          // save reference to this IWindow in client data

  XtAddEventHandler(
     (Widget)lbHandle,               // widget
     PointerMotionMask |
     Button1MotionMask |
     Button2MotionMask |
     Button3MotionMask |
     Button4MotionMask |
     Button5MotionMask,              // ask for FocusIn and FocusOut events
     True,                           // indicate we also want nonmaskable events
     ilistMotionXEventCallback,          // event handler routine
     (XtPointer) listBox);         

}

/*------------------------------------------------------------------------------
| IBaseListBoxData::unregisterCallbacks                                        |
------------------------------------------------------------------------------*/
void IBaseListBoxData :: unregisterCallbacks (IBaseListBox* listBox)
{
  if (listBox->isValid())
  {
    Widget wHandle = (Widget)listBox->handle();
    // Only unregister callbacks if widget not to be destroyed
    if (!wHandle->core.being_destroyed)
    {
      // Remove the browseSelect callback.
      XtRemoveCallback ((Widget)lbHandle,
			XmNbrowseSelectionCallback,
			iwindowMotifCallback,
			(XtPointer) listBox);

      // Remove the enter callback.
      XtRemoveCallback ((Widget)lbHandle,
			XmNdefaultActionCallback,
			iwindowMotifCallback,
			(XtPointer) listBox);

      // Remove the extendedSelect callback.
      XtRemoveCallback ((Widget)lbHandle,
			XmNextendedSelectionCallback,
			iwindowMotifCallback,
			(XtPointer) listBox);

      // Remove the multipleSelect callback.
      XtRemoveCallback ((Widget)lbHandle,
			XmNmultipleSelectionCallback,
			iwindowMotifCallback,
			(XtPointer) listBox);

      // Remove the singleSelect callback.
      XtRemoveCallback ((Widget)lbHandle,
			XmNsingleSelectionCallback,
			iwindowMotifCallback,
			(XtPointer) listBox);

      XtRemoveEventHandler(
	 (Widget)lbHandle,               // widget
	 FocusChangeMask,                // ask for FocusIn and FocusOut events
	 True,                           // indicate we also want nonmaskable events
	 iwindowXEventCallback,          // event handler routine
	 (XtPointer) listBox);          // save reference to this IWindow in client data

      XtRemoveEventHandler(
	 (Widget)lbHandle,               // widget
	 PointerMotionMask |
	 Button1MotionMask |
	 Button2MotionMask |
	 Button3MotionMask |
	 Button4MotionMask |
	 Button5MotionMask,              // ask for FocusIn and FocusOut events
	 True,                           // indicate we also want nonmaskable
	 ilistMotionXEventCallback,          // event handler routine
	 (XtPointer) listBox);          // save reference to this IWindow in
    }
  }
}
#endif //IC_MOTIF


#ifdef IC_MOTIFWIN

/*------------------------------------------------------------------------------
| IBaseListHandler::IBaseListHandler                                           |
|                                                                              |
| Empty constructor here for page tuning.                                      |
------------------------------------------------------------------------------*/
IBaseListHandler::IBaseListHandler()
{ }

/*------------------------------------------------------------------------------
| IBaseListHandler::~IBaseListHandler                                          |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IBaseListHandler::~IBaseListHandler()
{ }

/*------------------------------------------------------------------------------
| IBaseListHandler::dispatchHandlerEvent                                       |
|                                                                              |
| Intercepts control color event                                               |
|    This handler returns false for all other events.                          |
------------------------------------------------------------------------------*/
bool IBaseListHandler::dispatchHandlerEvent( IEvent& event )
{
  IMODTRACE_ALL("IBaseListHandler::dispatchHandlerEvent");
  bool
    result = false;

  switch( event.eventId() )
  {
#ifdef IC_WIN
    case WM_SIZE:
      /* This is a workaround for a possible Windows API problem.
	 Basically, there are two problems with a RTL listbox
	 control in Windows:
	 1. If there is no horizontal scrollbar, the entire RTL listbox
	    must be updated whenever the control is resized.
	 2. If there is a horizontal scrollbar, the text in the listbox
	    is not properly aligned if the listbox is then resized.

	 To overcome the first problem, we call InvalidateRect on the
	 entire RTL listbox control to force an update.
	 To overcome the second problem, we add and remove an empty
	 string so that some internal alignment values are reset during
	 resize.
      */
	 
      if (IBidiSettings::isBidiSupported()) {
	  IBidiSettings settings = IBidiSettings::applicationDefaults();
	  if (settings.windowLayout() == IBidiSettings::layoutRightToLeft) {
	      IBaseListBox *listBox = (IBaseListBox*)(event.controlWindow());
	      HWND hWnd = listBox->handle();
	      if (listBox->isHorizontalScroll()) {
		  // Add and remove an empty string to ensure proper
		  // text alignment.
		  // Notice that we are adding to the end of the listbox
		  // and we disable update during the operation
		  // to ensure minimal change to the existing content.
		  listBox->disableUpdate();
		  DWORD rc = SendMessage(hWnd, LB_INSERTSTRING, -1, 
					 (LPARAM)(LPCTSTR)"");
		  if (rc != LB_ERR && rc != LB_ERRSPACE) {
		      // Don't delete anything unless we have
		      // successfully added the empty string.
		      SendMessage(hWnd, LB_DELETESTRING, rc, 0);
		  }
		  listBox->enableUpdate();
	      }
	      InvalidateRect(hWnd, NULL, TRUE);
	  }
      }
      break;
    case WM_KEYDOWN:
      // If this is an owner draw listbox on the oldshell, we need to handle
      // page down processing due to a Windows bug.
      if (event.parameter1() == VK_PAGEDOWN)
      {
	 IBaseListBox *listBox = (IBaseListBox*)(event.controlWindow());
	 if (listBox && listBox->isDrawItem() &&
	     !(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
	 {
	    IWindow *owner = listBox->owner();

	    // Get number of items in the list box.
	    unsigned long itemCount = listBox->count();
	    if (!itemCount || !listBox->numberOfSelections())
	       break;

	    // Get the index for the item which is to be at the top.
	    unsigned long topIndex = listBox->selection();
	    if (topIndex == itemCount - 1)
	       break;

	    // Get the height of the client area of the list box.
	    RECT rect;
	    GetClientRect( listBox->handle(), &rect );
	    unsigned long lbHeight = rect.bottom - rect.top;

	    // Get the index for the last visible item in the list
	    // box based on the new top item.
	    unsigned long bottomIndex = 0,
			  itemHeights = 0;

	    // Determine the last item in the listbox which should be
	    // visible after the page down.
	    MEASUREITEMSTRUCT mis;
	    mis.CtlType = ODT_LISTBOX;
	    mis.CtlID = listBox->id();
	    mis.itemID = topIndex;
	    mis.itemWidth  = 0;
	    mis.itemHeight = 0;
	    while (!bottomIndex)
	    {
	       // Get the item height of the next item.
	       mis.itemData = listBox->sendEvent( LB_GETITEMDATA,
				       (unsigned long)mis.itemID,
				       0 );
	       owner->sendEvent( WM_MEASUREITEM,
				 (unsigned long)mis.CtlID,
				 (void*)&mis );
	       itemHeights += mis.itemHeight;

	       if (itemHeights <= lbHeight )
		  if (mis.itemID == (itemCount - 1))
		     bottomIndex = mis.itemID;
		  else
		     mis.itemID++;
	       else
		  bottomIndex = --mis.itemID;
	    }

	    // Special case: When the height of the list box is less than or
	    // equal to the height of an item, the list box displays only one
	    // item or part of an item at a time. So, when paging down we need
	    // to display the next item in the list.
	    if (bottomIndex <= topIndex)
	       bottomIndex = ++topIndex;

	    // Move the item that had the focus to the top, then set the
	    // selection to the last visible item in the list box.
	    listBox->disableUpdate();
	    listBox->select( bottomIndex );
	    listBox->setTop( topIndex );
	    listBox->enableUpdate();
	    listBox->refresh();

	    // Disabling update prevents the scroll bar from being updated.
	    // Set the new position explicitly.
	    unsigned long pos = GetScrollPos(listBox->handle(), SB_VERT);
	    SetScrollPos(listBox->handle(), SB_VERT, pos, TRUE);

	    result = true;
	 }
      }
      break;

    // Code to trap a mnemonic key press and send it directly to
    // the owner/parent window (allowing it to naturally progress
    // through the dispatcher results in an audible beep caused
    // by the default window procedure, because it cannot find a
    // match for a mnemonic that we process in ICanvas or
    // IFrameWindow), eating original event.
    case WM_SYSCHAR:
    {
      IWindow* window = event.window();
      if (window)
      {
	 IWindow* owner = window->owner();
	 if ( ! owner )
	 {
	    owner = window->parent();
	 }
	 if ( owner )
	 {
	    owner->sendEvent( event );
	    result = true;
	 }
      }
      break;
    }
#endif //IC_WIN

#ifdef IC_MOTIF
    case LM_INSERTITEM:
    {
      ITRACE_DEVELOP("In IBaseListHandler --- LM_INSERTITEM");

      IBaseListBox *listBox = (IBaseListBox*)(event.controlWindow());
      Widget listWidget = listBox->fBaseListBoxData->lbHandle;

      XmString string;
      char* replaceItem;
      char* text = (char*)event.parameter2();
      int position = (int)event.parameter1() + 1;

      // Create compound string.
      replaceItem = XtMalloc (strlen (text) + 1);
      sprintf (replaceItem, "%s", text);
      string = XmStringCreateLocalized (replaceItem);

      XmListAddItemUnselected(listWidget,
			      (XmString) string,
			      position);

      XmStringFree (string);
      XtFree (replaceItem);

      result = true;
      break;
    }

    case LM_DELETEITEM:
    {

      IBaseListBox *listBox = (IBaseListBox*)(event.controlWindow());
      Widget listWidget = listBox->fBaseListBoxData->lbHandle;

      int position = (int)event.parameter1() + 1;
      XmListDeleteItemsPos(listWidget,
			   1,
			   position);
      result = true;
      break;
    }

    case LM_DELETEALL:
    {

      IBaseListBox *listBox = (IBaseListBox*)(event.controlWindow());
      Widget listWidget = listBox->fBaseListBoxData->lbHandle;

      XmListDeleteAllItems(listWidget);
      result = true;
      break;
    }
#endif //IC_MOTIF

    default:
      break;
  }
  return result;
}

#endif //IC_MOTIFWIN

/*------------------------------------------------------------------------------
| IBaseListBox::IBaseListBox                                                           |
|                                                                              |
| Construct an IBaseListBox on an IWindow.                                         |
------------------------------------------------------------------------------*/
IBaseListBox::IBaseListBox ( unsigned long     Id,
		     IWindow*          pParent,
		     IWindow*          pOwner,
		     const IRectangle& rectInit,
		     const Style&      style )
{
  // assertions on input parms
  IASSERTPARM(pParent!=0);

  // Create a private data object.
  fBaseListBoxData = new IBaseListBoxData();

#ifdef IC_PMWIN
  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );
  IWindowHandle parentHandle(0);
  IWindowHandle ownerHandle(0);
  if (pParent)
    parentHandle = pParent->handleForChildCreation();
  else
    parentHandle = IWindow::desktopWindow()->handle();
  if (pOwner)
    ownerHandle = pOwner->handle();

  IWindowHandle whListBox =
      this -> create( Id,
		      0,
		      convertToGUIStyle( style ),
		      WC_LISTBOX,
		      parentHandle,
		      ownerHandle,
		      rectInit,
		      0,
		      0,
		      defaultOrdering(),
		      convertToGUIStyle( style, true ) );

#ifdef IC_WIN
   fBaseListBoxData->fdefaultHandler.handleEventsFor( this );
   fBaseListBoxData->fBrushHandler.handleEventsFor(this);
#endif
   startHandlingEventsFor(whListBox);

#ifdef IC_WIN
   // Initially set horizontal extent to screen width so as not to break
   // collectionview listbox.
   // Extent now set appropriately based on adds/deletes to the widest
   // string in current font.
   if ( (style & IBaseListBox::horizontalScroll) &&
	(!this->isDrawItem()) )
      {
      whListBox.sendEvent( LB_SETHORIZONTALEXTENT,
			   (unsigned long)IWindow::desktopWindow()->size().width(),
			   0 );
      }
#endif

#endif //IC_PMWIN

#ifdef IC_MOTIF
  fBaseListBoxData->parentIsTop = true;

  int    i = 0;
  Arg    args[10];
  Widget parentWidget, scrolledWindow;

  unsigned long ulStyle = style.asUnsignedLong();
  parentWidget = pParent->handle();

  // Set the selection policy.
  if (ulStyle & LS_MULTIPLESEL)
  {
    XtSetArg (args[i], XmNselectionPolicy, XmMULTIPLE_SELECT); i++;
  }
  else if (ulStyle & LS_EXTENDEDSEL)
  {
    XtSetArg (args[i], XmNselectionPolicy, XmEXTENDED_SELECT); i++;
  }

  // Set XmNlistSizePolicy to indicate that the horizontal size should
  // be constant.
  if (!(ulStyle & IWindow::noStyle.asUnsignedLong()))
  {
     XtSetArg (args[i], XmNlistSizePolicy, XmCONSTANT); i++;
  }

  // Set XmNscrollBarDisplayPolicy to indicate that the scroll bars should
  // be static.
  if (!(ulStyle & IWindow::noStyle.asUnsignedLong()))
  {
     XtSetArg (args[i], XmNscrollBarDisplayPolicy, XmSTATIC); i++;
  }

  IWindowHandle parentHandle(0);
  IWindowHandle ownerHandle(0);
  if (pParent)
    parentHandle = pParent->handleForChildCreation();
  else
    parentHandle = IWindow::desktopWindow()->handle();
  if (pOwner)
    ownerHandle = pOwner->handle();
  // Create the list widget.  Initial location and IWindow styles
  // are handled by create()
  scrolledWindow =
      Inherited::create(
	 Id,
	 NULL,
	 ulStyle,
	 (IXmCreateFunction)XiclCreateScrolledList,
	 parentHandle,
	 ownerHandle,
	 rectInit,
	 args,
	 i);

  // Stash away listbox widget handle
  if (XtIsSubclass((Widget)scrolledWindow, xmListWidgetClass))
    fBaseListBoxData->lbHandle = scrolledWindow;
  else
    fBaseListBoxData->lbHandle = fBaseListBoxData->listBoxWidget(scrolledWindow);

  fBaseListBoxData->fdefaultHandler.handleEventsFor( this );

  startHandlingEventsFor(scrolledWindow);
  fBaseListBoxData->registerCallbacks(this);

  // If only vertical scroll bar requested, remove horizontal one from the
  // scrolled window children.
  Widget    vertScroll, horzScroll, workArea;

  XtVaGetValues (scrolledWindow,
		 XmNhorizontalScrollBar, &horzScroll,
		 XmNverticalScrollBar, &vertScroll,
		 XmNworkWindow, &workArea,
		 NULL);

  if (!(ulStyle & IWindow::noStyle.asUnsignedLong()) &&
      !(ulStyle & LS_HORZSCROLL))
  {
     fBaseListBoxData->horizontalScroll = horzScroll;
     XmScrolledWindowSetAreas (scrolledWindow,
			       NULL,
			       vertScroll,
			       workArea);
  }
#endif //IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window. ListBox does
  // not inherit background color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kForegroundColor );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
			       inheritColor );
#endif
#ifdef IC_WIN
  // the background color of the list box is dialog independant.
  setColor( PP_BACKGROUNDCOLOR, IColor::listBoxBgnd );
#endif // IC_WIN

}

/*------------------------------------------------------------------------------
| IBaseListBox::IBaseListBox                                                   |
|                                                                              |
| Construct an IBaseListBox on a dialog.                                       |
------------------------------------------------------------------------------*/
IBaseListBox::IBaseListBox ( unsigned long id,
		     IWindow* parent )
{
#ifdef IC_PMWIN
  // Create a private data object.
  fBaseListBoxData = new IBaseListBoxData();

  setAutoDestroyWindow(false);
#ifdef IC_WIN
   fBaseListBoxData->fdefaultHandler.handleEventsFor( this );
   fBaseListBoxData->fBrushHandler.handleEventsFor(this);
#endif
  reserveUserWindowWord( false );
  startHandlingEventsFor(id, parent);

#ifdef IC_WIN
   // Save extended style information
   unsigned long ulStyle = style();
   if (ulStyle & IWS_BORDER3D)
   {
      unsigned long ulExtStyle = extendedStyle();
      ulExtStyle |= IWS_BORDER3D;
      setExtendedStyle (ulExtStyle);
   }
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  IASSERT(parent!=0);
  Widget handle=NULL;

  handle =  XtNameToWidget (parent->handle(), (char *)IString(id));
  if (handle == 0)
  {
    Widget swHandle = XtNameToWidget (parent->handle(),
			      (char *)(IString(id) + "SW"));
    if (swHandle)
       handle = XtNameToWidget (swHandle, (char*)IString(id));

    if (handle == 0)
      ITHROWLIBRARYERROR1( IC_CONTROL_NOT_FOUND,
			   IBaseErrorInfo::accessError,
			   IException::recoverable,
			   "XtNameToWidget (list box)");
  }

  fBaseListBoxData->fdefaultHandler.handleEventsFor( this );

  // Stash away listbox widget handle
  if (XtIsSubclass((Widget)handle, xmListWidgetClass))
    fBaseListBoxData->lbHandle = handle;
  else
    fBaseListBoxData->lbHandle = fBaseListBoxData->listBoxWidget(handle);

  fBaseListBoxData->registerCallbacks(this);
  wrapIt (handle);
#endif //IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window. ListBox does
  // not inherit background color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kForegroundColor );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
			       inheritColor );
#endif
}

/*------------------------------------------------------------------------------
| IBaseListBox::IBaseListBox                                                   |
|                                                                              |
| Wrapper an existing ListBox.                                                 |
------------------------------------------------------------------------------*/
IBaseListBox::IBaseListBox ( const IWindowHandle& handle )
{
#ifdef IC_PMWIN
   // Create a private data object.
   fBaseListBoxData = new IBaseListBoxData();

   setAutoDestroyWindow(false);
#ifdef IC_WIN
   fBaseListBoxData->fdefaultHandler.handleEventsFor( this );
   fBaseListBoxData->fBrushHandler.handleEventsFor(this);
#endif
   reserveUserWindowWord( false );
   startHandlingEventsFor(handle);

#ifdef IC_WIN
   // Save extended style information
   unsigned long ulStyle = style();
   if (ulStyle & IWS_BORDER3D)
   {
      unsigned long ulExtStyle = extendedStyle();
      ulExtStyle |= IWS_BORDER3D;
      setExtendedStyle (ulExtStyle);
   }
    handle.sendEvent( LB_SETHORIZONTALEXTENT,
		    (unsigned long)IWindow::desktopWindow()->size().width(),
		     0 );


#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
  IASSERTPARM(handle!=0);

  fBaseListBoxData->fdefaultHandler.handleEventsFor( this );

  // Stash away listbox widget handle
  if (XtIsSubclass((Widget)handle, xmListWidgetClass))
    fBaseListBoxData->lbHandle = handle;
  else
    fBaseListBoxData->lbHandle = fBaseListBoxData->listBoxWidget(handle);

  fBaseListBoxData->registerCallbacks(this);
  wrapIt (handle);

#endif //IC_MOTIF
#ifdef IC_MOTIFWIN
  // add the inheritColor attribute to the window. ListBox does
  // not inherit background color.
  IInheritColorAttribute
    inheritColor( IInheritColorAttribute::kForegroundColor );
  this->addOrReplaceAttribute( IAttributeName("IInheritColorAttribute"),
			       inheritColor );
#endif
}

/*------------------------------------------------------------------------------
| IBaseListBox::~IBaseListBox                                                  |
|                                                                              |
| Default destructor here for page tuning.                                     |
------------------------------------------------------------------------------*/
IBaseListBox::~IBaseListBox()
{
#ifdef IC_PMWIN
#ifdef IC_WIN
  fBaseListBoxData->fdefaultHandler.stopHandlingEventsFor( this );
  fBaseListBoxData->fBrushHandler.stopHandlingEventsFor(this);
#endif
  delete fBaseListBoxData;
#endif //IC_PMWIN

#ifdef IC_MOTIF
  fBaseListBoxData->unregisterCallbacks(this);
  if (XtIsWidget(fBaseListBoxData->horizontalScroll))
     XtDestroyWidget (fBaseListBoxData->horizontalScroll);
  fBaseListBoxData->fdefaultHandler.stopHandlingEventsFor( this );
  delete fBaseListBoxData;
#endif //IC_MOTIF
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IBaseListBox::wrapIt                                                         |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox :: wrapIt(const IWindowHandle& handle)
{
  Widget    immedParent=NULL, workarea=NULL, horizontalScroll, verticalScroll;

  if (!XtIsSubclass((Widget)handle, xmListWidgetClass) )
     ITHROWLIBRARYERROR1( IC_WRONG_CONTROL_TYPE,
			  IBaseErrorInfo::accessError,
			  IException::recoverable,
			  "XmList");

  setAutoDestroyWindow(false);

  fBaseListBoxData = new IBaseListBoxData();
  fBaseListBoxData->parentIsTop = false;
  // Determine if this should be a scrolled IListBox object or not.
  immedParent = XtParent((Widget)handle);
  if ((XtIsSubclass(immedParent, xmScrolledWindowWidgetClass)) &&
      (windowWithHandle(immedParent) == 0)) // and it's not in another IWindow
  {
    // Ok, we have an available scrolled window parent; now see if it has us
    // as its workWindow (otherwise, we might just be a plain old child).
    XtVaGetValues (immedParent,
		   XmNworkWindow, &workarea,
		   XmNhorizontalScrollBar, &horizontalScroll,
		   XmNverticalScrollBar,   &verticalScroll,
		   NULL);
    if (workarea == (Widget)handle)
    {
       fBaseListBoxData->parentIsTop = true;
       // Do not manage the child listbox; the caller decides when to do that.
    }
  }

  prepareForUse(handle);
  startHandlingEventsFor(handle);

  return *this;
}
#endif //IC_MOTIF


/*------------------------------------------------------------------------------
| IBaseListBox::isMultipleSelect                                               |
|                                                                              |
------------------------------------------------------------------------------*/
bool IBaseListBox::isMultipleSelect() const
{
#ifdef IC_PMWIN
  return( (style() & multipleSelect.asUnsignedLong()) ? true : false );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned char selectType;

  XtVaGetValues ((Widget)fBaseListBoxData->lbHandle,
		 XmNselectionPolicy, &selectType,
		 NULL);

  if (selectType == XmMULTIPLE_SELECT)
    return true;
  else
    return false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IBaseListBox::isExtendedSelect                                               |
------------------------------------------------------------------------------*/
bool IBaseListBox::isExtendedSelect()  const
{
#ifdef IC_PMWIN
  return( (style() & extendedSelect.asUnsignedLong()) ? true : false );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  unsigned char selectType;

  XtVaGetValues ((Widget)fBaseListBoxData->lbHandle,
		 XmNselectionPolicy, &selectType,
		 NULL);

  if (selectType == XmEXTENDED_SELECT)
    return true;
  else
    return false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IBaseListBox::isDrawItem                                                     |
------------------------------------------------------------------------------*/
bool IBaseListBox::isDrawItem()  const
{
  ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  return( (style() & drawItem.asUnsignedLong()) ? true : false );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  return false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IBaseListBox::isNoAdjustPosition                                             |
------------------------------------------------------------------------------*/
bool IBaseListBox::isNoAdjustPosition()  const
{
  ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  return( (style() & noAdjustPosition.asUnsignedLong()) ? true : false );
#endif //IC_PMWIN

#ifdef IC_MOTIF
  return false;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableMultipleSelect                                           |
|                                                                              |
| Either turnon or turnoff the MultipleSelect style.                           |
| In Motif, turning off multiple selection will result in the                  |
| default selection policy being set.  (In PM, the listbox                     |
| may still be using extended selection after turning off                      |
| multiple selection).                                                         |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableMultipleSelect ( bool enable )
{
  ITRACE_WIN_NOP();
#ifdef IC_PM
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);
  bool       refreshListBox(false);

  if (enable)
  {
    ulStyle |= multipleSelect.asUnsignedLong();
  }
  else
  {
    // Since extended and multiple select can be set at the
    // same time, do not deselect all items if extended select
    // is still set.
    if (!(ulStyle & extendedSelect.asUnsignedLong()))
    {
      deselectAll();
      refreshListBox = true;
    }
    ulStyle &= ~multipleSelect.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    if (refreshListBox)
      refresh();
  }
#endif //IC_PM

#ifdef IC_MOTIF
  if (enable)
  {
     XtVaSetValues ((Widget)fBaseListBoxData->lbHandle,
		    XmNselectionPolicy, XmMULTIPLE_SELECT,
		    NULL);
  }
  else
  {
     XtVaSetValues ((Widget)fBaseListBoxData->lbHandle,
		    XmNselectionPolicy, XmBROWSE_SELECT,
		    NULL);
  }
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableMultipleSelect                                          |
|                                                                              |
| Turnoff the MultipleSelect style bit.  In Motif, turning                     |
| off multiple selection will result in the default selection                  |
| policy being set.  (In PM, the listbox may still be using                    |
| extended selection after turning off multiple selection).                    |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableMultipleSelect()
{
  ITRACE_WIN_NOP();
  enableMultipleSelect(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableExtendedSelect                                           |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableExtendedSelect ( bool enable )
{
  ITRACE_WIN_NOP();
#ifdef IC_PM
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);
  bool       refreshListBox(false);

  if (enable)
  {
    ulStyle |= extendedSelect.asUnsignedLong();
  }
  else
  {
    // Since extended and multiple select can be set at the
    // same time, do not deselect all items if multiple select
    // is still set.
    if (!(ulStyle & multipleSelect.asUnsignedLong()))
    {
      deselectAll();
      refreshListBox = true;
    }
    ulStyle &= ~extendedSelect.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    if (refreshListBox)
      refresh();
  }
#endif //IC_PM

#ifdef IC_MOTIF
  Widget listBox = fBaseListBoxData->lbHandle;

  if (enable)
  {
     XtVaSetValues (listBox,
		    XmNselectionPolicy, XmEXTENDED_SELECT,
		    NULL);
  }
  else
  {
     XtVaSetValues (listBox,
		    XmNselectionPolicy, XmBROWSE_SELECT,
		    NULL);
  }
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableExtendedSelect                                          |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableExtendedSelect()
{
  ITRACE_WIN_NOP();
  enableExtendedSelect(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableDrawItem                                                 |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableDrawItem ( bool enable )
{
  ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);

  if (enable)
  {
    ulStyle |= drawItem.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~drawItem.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);

    if ( enable )
    {         // Force draw-item handlers to be called.
       IWindow* owner = this->owner();
       if (owner)
       {
#ifdef IC_PM
	  owner->sendEvent( WM_MEASUREITEM,
			    IEventParameter1( this->id() ),
			    0 );
#endif
#ifdef IC_WIN
	  MEASUREITEMSTRUCT mis;
	  mis.CtlType = ODT_LISTBOX;
	  mis.CtlID   = this->id();
	  mis.itemID  = 0;
	  mis.itemWidth  = 0;
	  mis.itemHeight = 0;
	  // get the value previously set if any
	  mis.itemData =
	     this->sendEvent( LB_GETITEMDATA,
			      IEventParameter1( (unsigned long)mis.itemID ),
			      IEventParameter2( 0 ) );
	  owner->sendEvent( WM_MEASUREITEM,
			    IEventParameter1( (unsigned long)mis.CtlID ),
			    IEventParameter2( (void*) &mis ) );
#endif
       }
    }

    refresh();
  }
#endif //IC_PMWIN

  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableDrawItem                                                |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableDrawItem()
{
  ITRACE_MOTIF_NOP();
  enableDrawItem(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::enableNoAdjustPosition                                         |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::enableNoAdjustPosition ( bool enable )
{
  ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
  unsigned long ulStyle(style());
  unsigned long ulOldStyle(ulStyle);

  if (enable)
  {
    ulStyle |= noAdjustPosition.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~noAdjustPosition.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();
  }
#endif //IC_PMWIN

  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::disableNoAdjustPosition                                        |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::disableNoAdjustPosition()
{
  ITRACE_MOTIF_NOP();
  enableNoAdjustPosition(false);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::Style IBaseListBox::defaultStyle                               |
------------------------------------------------------------------------------*/
IBaseListBox::Style IBaseListBox::defaultStyle()
{
   return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setDefaultStyle                                                |
------------------------------------------------------------------------------*/
void IBaseListBox::setDefaultStyle(const Style& style)
{
  currentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IBaseListBox::convertToGUIStyle                                              |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IBaseListBox::convertToGUIStyle(const IBitFlag& guiStyle,
					  bool bExtOnly) const
{
  // Obtain the style from the class (IControl) that we inherit from
  unsigned long ulStyle( Inherited::convertToGUIStyle( guiStyle, bExtOnly ));

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= guiStyle.asExtendedUnsignedLong() & ILS_EXTGUIMASK;
  }
  else
  {
    // LS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= guiStyle.asUnsignedLong() & ILS_MASK;

#ifdef IC_WIN
    ulStyle |= WS_CHILD | LBS_NOTIFY | WS_BORDER | LBS_HASSTRINGS | WS_VSCROLL;
#endif
  }

  return( ulStyle );
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IBaseListBox::select                                                         |
|                                                                              |
| Select (or deselect) an item in the list.                                    |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::select(unsigned long lIndex, bool turnOn)
{
   Widget  listBox = fBaseListBoxData->lbHandle;

   // See if the index is valid.
   if (lIndex >= count())
   {
      ITHROWLIBRARYERROR1 (IC_INVALID_INDEX,
			   IBaseErrorInfo::invalidParameter,
			   IException::recoverable,
			   IString (lIndex));
   }

   if (turnOn)
   {
      // If it is already selected, leave it alone.  Do not want to toggle it.
      if (!isSelected (lIndex))
      {
	 if (isMultipleSelect() || isExtendedSelect())
	 {
	    if (isExtendedSelect()){
		  enableMultipleSelect();
		  XmListSelectPos (listBox, lIndex+1, true);
		  enableExtendedSelect();
	    }
	    else
	       XmListSelectPos (listBox, lIndex+1, true);
	 }
	 else
	 {
	   XmListSelectPos (listBox, lIndex+1, true);
	   scrollToItem(lIndex);
	 }
      }
      else
      {
	//if single selection, scroll item into view
	if (!isMultipleSelect() && !isExtendedSelect())
	  scrollToItem(lIndex);
      }
   }
   else
   {
     XmListDeselectPos (listBox, lIndex + 1);
     this->notifyObservers(INotificationEvent(IBaseListBox::selectId, *this));
   }

   return *this;
}
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| IBaseListBox::selectAll                                                      |
|                                                                              |
| Sequentially select all items in listbox.                                    |
------------------------------------------------------------------------------*/
IBaseListBox&  IBaseListBox::selectAll()
{
#ifdef IC_PMWIN
/* Don't throw exception on empty list
   // Verify that the list contains items and if not, fail call
   if (isEmpty())
     ITHROWLIBRARYERROR( IC_EMPTY_LIST,
			 IBaseErrorInfo::invalidRequest,
			 IException::recoverable );
*/

   // Verify that the list supports selection of multiple items
   // i.e. it is not a single select list box
   if ( isMultipleSelect() || isExtendedSelect() )
   {
     unsigned long
       index,
       lastIndex;
     for (index = 0, lastIndex = count(); index < lastIndex; index++)
     {
       IBaseListBox::select(index);
     }
   }
   else
     ITHROWLIBRARYERROR( IC_SINGLE_SELECT_LIST,
			 IBaseErrorInfo::invalidRequest,
			 IException::recoverable );
#endif //IC_PMWIN

#ifdef IC_MOTIF
   enum selectType {extended, multiple};
   Widget     listBox;
// d8269
//   int        i, initialSelectionCount, itemCount;
//   XmString   *initialSelectedItems, *itemList, *newSelectedItems;
   int        itemCount;
   XmString   *itemList;

   listBox = fBaseListBoxData->lbHandle;
   if (isExtendedSelect() || isMultipleSelect())
   {
// d8269 - You should not free what you get back from XmNselectedItems.
//         See O'Reilly V6B, XmList Widget, XmNselectedItems.
//
//      XtVaGetValues (listBox,
//                     XmNselectedItems, &initialSelectedItems,
//                     XmNselectedItemCount, &initialSelectionCount,
//                     XmNitems, &itemList,
//                     XmNitemCount, &itemCount,
//                     NULL);
//      newSelectedItems = (XmString*) XtMalloc (itemCount * sizeof (XmString));
//      for (i=0; i<itemCount; i++)
//      {
//         newSelectedItems[i] = XmStringCopy (itemList[i]);
//      }
//
//      XtVaSetValues (listBox,
//                     XmNselectedItems, newSelectedItems,
//                     XmNselectedItemCount, itemCount,
//                     NULL);
//
//      for (i=0; i<initialSelectionCount; i++)
//      {
//         XmStringFree (initialSelectedItems [i]);
//      }
//      XtFree ((char*) initialSelectedItems);
//
      XtVaGetValues (listBox,
		     XmNitems, &itemList,
		     XmNitemCount, &itemCount,
		     NULL);
      XtVaSetValues (listBox,
		     XmNselectedItems, itemList,
		     XmNselectedItemCount, itemCount,
		     NULL);
   }
   else
   // If this is a single or browse selection listbox, just select the last
   // item.  Resulting behavior is consistent with PM.
      XmListSelectPos (listBox, itemCount, true);
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setItemHeight                                                  |
|                                                                              |
| Set height of the items in a listbox.                                        |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox::setItemHeight(unsigned long lNewHeight)
{
#ifdef IC_WIN
   IEventResult evt;
   if (this->isDrawItem())
   {
      // For owner draw listboxes, we have to send the LB_SETITEMHEIGHT to
      // each item in the list.
      unsigned long index,
		    itemCount = this->count();
      for (index = 0; index < itemCount; index++)
      {
	 evt = handle().sendEvent( LM_SETITEMHEIGHT,
				   IEventParameter1( index ),
				   IEventParameter2( lNewHeight ));
	 if (evt.asLong() == LIT_ERROR)
	    ITHROWGUIERROR( "LB_SETITEMHEIGHT" );
      }
   }
   else
   {
      evt = handle().sendEvent( LM_SETITEMHEIGHT,
				IEventParameter1( 0 ),
				IEventParameter2( lNewHeight ));
      if (evt.asLong() == LIT_ERROR)
	 ITHROWGUIERROR( "LB_SETITEMHEIGHT" );
   }
   refresh();
#endif //IC_WIN

#ifdef IC_PM
   IEventResult evt;
   evt = handle().sendEvent( LM_SETITEMHEIGHT,
			     IEventParameter1((unsigned long)lNewHeight),
			     IEventParameter2(0));
   if (!(evt.asUnsignedLong()))
      ITHROWGUIERROR("LM_SETITEMHEIGHT");
#endif //IC_PM

#ifdef IC_MOTIF
   Widget        listBox;
   Dimension     highlightThickness, itemHeight, newItemSpacing;
   XmFontList    fontList;
   ISize         lbSize = size();

   listBox = fBaseListBoxData->lbHandle;
   XtVaGetValues (listBox,
		  XmNhighlightThickness, &highlightThickness,
		  XmNfontList, &fontList,
		  NULL);

   // Calculate the line spacing and the new height of the list.
   IFont lbFont = IFont(this);
   itemHeight = (Dimension)lbFont.maxCharHeight();
   newItemSpacing = (Dimension) MAX(0, (Dimension)lNewHeight -
				   ((2 * highlightThickness) + itemHeight));

   XtVaSetValues (listBox,
		  XmNlistSpacing, newItemSpacing,
		  NULL);

   sizeTo (lbSize);
   setLayoutDistorted (minimumSizeChanged, 0);
#endif //IC_MOTIF

   return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::itemHeight                                                     |
|                                                                              |
| Queries the item height in a listbox.                                        |
------------------------------------------------------------------------------*/
unsigned long IBaseListBox::itemHeight ( unsigned long index ) const
{
#ifdef IC_WIN
  IEventResult evt = handle().sendEvent(LB_GETITEMHEIGHT,
					IEventParameter1( index ),
					IEventParameter2(0));
  if (evt.asLong() == LB_ERR)
    ITHROWSYSTEMERROR( LB_ERR, "LB_GETITEMHEIGHT", IBaseErrorInfo::accessError,
		       IException::recoverable );
  return evt.asUnsignedLong();
#endif

#ifdef IC_PM
  // if owner draw listbox
  if (isDrawItem())
    {
    // Items in OS/2 are fixed height, so send WM_MEASUREITEM to
    // determine item height
    IEventResult result = owner()->sendEvent( WM_MEASUREITEM,
			  IEventParameter1( id() ),
			  0 );
    return (unsigned long)(result.number1());
    }
  else         // Normal listbox
    {
    // For a normal listbox, items are the size of the font.
    // To determine the size, get the bounding box of a space.
    IPresSpaceHandle hps = presSpace();
    POINTL Points[2];

    GpiQueryTextBox(hps, 1L, " ", 2, Points);
    releasePresSpace(hps);

    return (unsigned long)(Points[TXTBOX_TOPLEFT].y - Points[TXTBOX_BOTTOMLEFT].y);
    }
#endif

#ifdef IC_MOTIF
   XmListRec* listRec = (XmListRec*)(Widget)(fBaseListBoxData->lbHandle);
   XmListPart& listPart = listRec->list;

   // Item height is the maximum item height plus space between items
   return listPart.MaxItemHeight + listPart.spacing;

#endif //IC_MOTIF
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IBaseListBox::backgroundColor                                                |
|                                                                              |
| Returns the background color of the IBaseListBox.                            |
------------------------------------------------------------------------------*/
IColor IBaseListBox::backgroundColor () const
{
  return (IWindow::color(PP_BACKGROUNDCOLOR,
			 IGUIColor(IGUIColor::listBoxBgnd)));
}
#endif //IC_PMWIN
#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IBaseListBox :: setForegroundColor                                           |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox :: setForegroundColor(const IColor& color)
{
  Inherited::setForegroundColor (color);
  XtVaSetValues (fBaseListBoxData->lbHandle,
		 XmNforeground, color.index(),
		 NULL);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox :: setBackgroundColor                                           |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox :: setBackgroundColor(const IColor& color)
{
  Inherited::setBackgroundColor(color);

  XmChangeColor (fBaseListBoxData->lbHandle, color.index());
  return *this;
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IBaseListbox::show                                                           |
|                                                                              |
| Make the window visible.                                                     |
------------------------------------------------------------------------------*/
IWindow& IBaseListBox::show(bool bShow)

  {
  // We need to manage and unmanage the listbox child of a scrolled list.
  if (bShow)
    XtManageChild((Widget)fBaseListBoxData->lbHandle);
  else
    XtUnmanageChild((Widget)fBaseListBoxData->lbHandle);

  Inherited::show (bShow);

  return *this;
  }

#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| IBaseListBox::calcMinimumSize                                                |
|                                                                              |
| Calculate the minimum size needed by the control.                            |
------------------------------------------------------------------------------*/
ISize  IBaseListBox :: calcMinimumSize() const
{
  ISize sizMin;

#ifdef IC_PMWIN
  IFont fnt(this);

  sizMin.setHeight ( fBaseListBoxData->minimumRows * fnt.maxCharHeight() +
#ifdef IC_WIN
		     2 * IQUERYSYSVALUE(SV_CYBORDER) );
#else
		     2 * IQUERYSYSVALUE(SV_CYBORDER) + 2);
#endif

  sizMin.setWidth  ( fBaseListBoxData->minimumCharacters * fnt.avgCharWidth() +
		     2 * IQUERYSYSVALUE(SV_CXBORDER) +
		     IQUERYSYSVALUE(SV_CXVSCROLL));

  if (isHorizontalScroll())
  {
#ifdef IC_WIN
     sizMin.setHeight (sizMin.height() + IQUERYSYSVALUE(SV_CYHSCROLL) - 1);
#else
     sizMin.setHeight (sizMin.height() + IQUERYSYSVALUE(SV_CYHSCROLL));
#endif
  }

  if ( ( extendedStyle() & border3D.asExtendedUnsignedLong() ) &&
       ( IPlatform::isWin9x() || IPlatform::isNTNewShell() ))
  {
     sizMin += ISize(4,4);
  }
#endif //IC_PMWIN

#ifdef IC_MOTIF
  Dimension listMarginHeight, listMarginWidth, listSpacing, highlightThickness,
	    shadowThickness, borderWidth;
  XtVaGetValues ((Widget)fBaseListBoxData->lbHandle,
		 XmNlistMarginHeight,   &listMarginHeight,
		 XmNlistMarginWidth,    &listMarginWidth,
		 XmNlistSpacing,        &listSpacing,
		 XmNhighlightThickness, &highlightThickness,
		 XmNshadowThickness,    &shadowThickness,
		 XmNborderWidth,        &borderWidth,
		 NULL);
  sizMin.setWidth(fBaseListBoxData->minimumCharacters * characterSize().width() +
		  2 * (highlightThickness + borderWidth + shadowThickness));
  // Add width of vertical scroll bar.
  Dimension scrollWidth, spacing;
  XtVaGetValues (XtNameToWidget (XtParent ((Widget)fBaseListBoxData->lbHandle), "VertScrollBar"),
		 XmNwidth, &scrollWidth,
		 NULL);
  XtVaGetValues (XtParent ((Widget)fBaseListBoxData->lbHandle),
		 XmNspacing, &spacing,
		 NULL);
  sizMin.setWidth (sizMin.width() + spacing +
		   (scrollWidth ? scrollWidth : 15));

  sizMin.setHeight(fBaseListBoxData->minimumRows * characterSize().height() +
		   2 *fBaseListBoxData->minimumRows * highlightThickness +
		   (fBaseListBoxData->minimumRows - 1) * listSpacing +
		   2 * (borderWidth + shadowThickness));

  // Add height of horizontal scroll bar.
  if (isHorizontalScroll())
  {
     Dimension scrollHeight;
     XtVaGetValues (XtNameToWidget (XtParent ((Widget)fBaseListBoxData->lbHandle), "HorScrollBar"),
		    XmNheight, &scrollHeight,
		    NULL);
     sizMin.setHeight (sizMin.height() + spacing +
		       (scrollHeight ? scrollHeight : 15));
  }
#endif //IC_MOTIF

  return sizMin;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IBaseListBox::setLayoutDistorted                                             |
|                                                                              |
| Cause a font change to result in a new minimum size.                         |
------------------------------------------------------------------------------*/
IBaseListBox&
  IBaseListBox::setLayoutDistorted ( unsigned long layoutAttributesOn,
				     unsigned long layoutAttributesOff )
{
  unsigned long flagsOn = layoutAttributesOn;
  if ( layoutAttributesOn & IWindow::fontChanged )
  {                // New font means a new minimum size.
     flagsOn |= IWindow::minimumSizeChanged;

#ifdef IC_WIN
     // If this is an owner draw listbox, the listbox items need to be redrawn.
     // Since Windows doesn't send a WM_MEASUREITEM with font changes, we need
     // to force the draw handler's setItemHeight to be called for each item
     // in the list.
     unsigned long itemId,
		   itemCount = this->count();
     if (isDrawItem())
     {
	for (itemId = 0; itemId < itemCount; itemId++)
	{
	   MEASUREITEMSTRUCT mis;
	   mis.CtlType = ODT_LISTBOX;
	   mis.CtlID   = this->id();
	   mis.itemID = itemId;
	   mis.itemWidth  = 0;
	   mis.itemHeight = 0;
	   // get the value previously set if any
	   mis.itemData =
	      this->sendEvent( LB_GETITEMDATA,
			       IEventParameter1( (unsigned long)itemId ),
			       IEventParameter2( 0 ) );
	   owner()->sendEvent( WM_MEASUREITEM,
			     IEventParameter1( (unsigned long)mis.CtlID ),
			     IEventParameter2( (void*) &mis ) );
	}
     }
#endif
  }
  Inherited::setLayoutDistorted( flagsOn, layoutAttributesOff );
  return *this;
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IBaseListBox::setMinimumCharacters                                           |
|                                                                              |
| Sets the number of characters in an item of a minimum size listbox.          |
------------------------------------------------------------------------------*/
IBaseListBox&  IBaseListBox :: setMinimumCharacters (unsigned long minimumCharacters)
{
  fBaseListBoxData->minimumCharacters = minimumCharacters;
  setLayoutDistorted (minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::minimumCharacters                                              |
|                                                                              |
| Returns the number of characters in an item in a minimum size listbox.       |
------------------------------------------------------------------------------*/
unsigned long  IBaseListBox ::  minimumCharacters() const
{
  return fBaseListBoxData->minimumCharacters;
}

/*------------------------------------------------------------------------------
| IBaseListBox::setMinimumRows                                                 |
|                                                                              |
| Sets the number of rows in a minimum size listbox.                           |
------------------------------------------------------------------------------*/
IBaseListBox&  IBaseListBox :: setMinimumRows (unsigned long minimumRows)
{
  fBaseListBoxData->minimumRows = minimumRows;
  setLayoutDistorted (minimumSizeChanged, 0);
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseListBox::minimumRows                                                    |
|                                                                              |
| Returns the number of rows in a minimum size listbox.                        |
------------------------------------------------------------------------------*/
unsigned long  IBaseListBox :: minimumRows ( ) const
{
  return fBaseListBoxData->minimumRows;
}

/*------------------------------------------------------------------------------
| IBaseListBox::scrollToItem                                                   |
|                                                                              |
| Scroll requested item into view                                              |
------------------------------------------------------------------------------*/
IBaseListBox& IBaseListBox :: scrollToItem (unsigned long index)
{
#ifdef IC_PMWIN
  IMODTRACE_DEVELOP("IBaseListBox::scrollToItem");

  unsigned long firstVisible = top();

  // If item is above current view, just set to the top
  if (index < firstVisible)
    {
    setTop(index);
    return *this;
    }

  // Get the height of the client area of the list box.
  unsigned long lbHeight = 0;
#ifdef IC_WIN
  RECT rect;
  GetClientRect( handle(), &rect );
  lbHeight = rect.bottom - rect.top;
#endif
#ifdef IC_PM
  RECTL rect;
  WinQueryWindowRect( handle(), &rect);
  lbHeight = rect.yTop - rect.yBottom;
#endif
  ITRACE_DEVELOP("lbHeight: " + IString(lbHeight));

  unsigned long numOfItemsInView = 0,
		itmHeight = 0;

#ifdef IC_WIN
  // If listbox is owner draw
  if (isDrawItem())
    {
    // Items can be variable height in Windows.
    // Calculate the height of each item in the listbox view
    unsigned long itemHeights = 0,
		  itemIndex = firstVisible,
		  itemCount = count();

    // Determine the number of items in the current listbox view
    // while the cumulative item heights does not exceed the listbox
    // height and the end of the listbox items not reached
    while ((itemHeights <= lbHeight) && (itemIndex < itemCount))
      {
      itemHeights += itemHeight(itemIndex);

      // Increment count of items in view
      numOfItemsInView++;

      // Increment item index to measure next item
      itemIndex++;
      }

    // Check for partial item visible at bottom of listbox view.
    RECT itemRect;
    IEventResult evt = handle().sendEvent(LB_GETITEMRECT,
				  IEventParameter1( firstVisible + numOfItemsInView - 1 ),
				  IEventParameter2( &itemRect ));

    IRectangle lbRect1(rect);
    IRectangle itemRect1(itemRect);

    // Special case: If the last item in the listbox is partially displayed,
    // decrement number of items in the view
    if (!lbRect1.contains(itemRect1) && (numOfItemsInView > 1))
      numOfItemsInView--;

    }
  else         // Normal listbox
#endif
    {
    itmHeight = itemHeight();

    if (itmHeight)
      numOfItemsInView = lbHeight / itmHeight;
    }

  // Return if item already in view.  Nothing to do.
  if ((index >= firstVisible) && (index <= firstVisible + (numOfItemsInView - 1)))
    return *this;

  // calculate the new top index such that the requested item will
  // be scrolled into view at the bottom of the listbox
  unsigned long bottomIndex = firstVisible + numOfItemsInView - 1;
  setTop(firstVisible + (index - bottomIndex));
#endif // IC_PMWIN

#ifdef IC_MOTIF
  int top, visible;
  XtVaGetValues (fBaseListBoxData->lbHandle,
		 XmNtopItemPosition,   &top,
		 XmNvisibleItemCount,  &visible,
		 NULL);

  if (index < top)
    setTop(index);
  else if (index >= top+visible)
    XmListSetBottomPos(fBaseListBoxData->lbHandle, index + 1);

#endif //IC_MOTIF
  return *this;
}

#ifdef IC_MOTIFWIN
/*------------------------------------------------------------------------------
| IBaseListBox::passEventToOwner                                               |
|                                                                              |
| Returns whether or not the event can be passed up to the owner of this       |
| control.                                                                     |
------------------------------------------------------------------------------*/
bool IBaseListBox::passEventToOwner( IEvent& event )
{
  switch ( event.eventId() )
  {
#ifdef IC_WIN
    case WM_CHAR:
      {
	IKeyboardEvent keyEvent(event);
	// Don't pass WM_CHAR up the owner chain.  Used by the windows listbox
	// control for keyboard selection
	if (keyEvent.isCharacter())
	  event.setPassToOwner(false);
      }
      break;
#endif //IC_WIN
#ifdef IC_MOTIF
    case WM_BUTTON1CLICK:
    case WM_BUTTON2CLICK:
    case WM_BUTTON3CLICK:
    case WM_CHORD:
      event.setPassToOwner( false );
      break;
    case xEvent(KeyPress):
    case xEvent(KeyRelease):
    {
      event.setPassToOwner( false );
      // Pass enter and esc key up the owner chain
      IKeyboardEvent keyEvent( event );
      if (keyEvent.isVirtual() )
      {
	if ( ( keyEvent.virtualKey() == IKeyboardEvent::enter ) ||
	     ( keyEvent.virtualKey() == IKeyboardEvent::esc ) )
	  event.setPassToOwner(true);
      }
      break;
    }
#endif //IC_MOTIF
    default:
      Inherited::passEventToOwner( event );
      break;
  }
  return event.passToOwner();
}
#endif //IC_MOTIFWIN

#include <ilistbx3.hpp>
