// Revision: 75 1.9.2.1 source/ui/basectl/igraphbt.cpp, buttonctls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: igraphbt.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in igraphbt.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

extern "C" {
  #define INCL_WINBUTTONS
  #define INCL_WINERRORS
  #define INCL_WINMESSAGEMGR
  #define INCL_WINSYS
  #define INCL_GPIPRIMITIVES
  #define INCL_WININPUT
  #include <iwindefs.h>
}

#ifdef IC_MOTIF
  #include <Xm/PushB.h>
#endif

#include <igraphbt.hpp>

#include <ibmpdata.hpp>
#include <ibmpstat.hpp>
#include <icconst.h>
#include <icustbdd.hpp>
#include <idefstyl.h>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <igraphdr.hpp>
#include <igrszhdr.hpp>
#include <ihandle.hpp>
#include <iiconctl.hpp>
#include <ipoint.hpp>
#include <ireslib.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <itrace.hpp>

#ifdef IC_PAGETUNE
  #define _IGRAPHBT_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| Public push button styles.                                                   |
------------------------------------------------------------------------------*/

const IGraphicPushButton::Style
#if (IC_OBSOLETE <= IC_OBSOLETE_3)
  IGraphicPushButton::sizeToGraphic     (1, IGraphicPushButton__sizeToGraphic    ),
#endif // IC_OBSOLETE
  IGraphicPushButton::sizeImageToWindow (1, IGraphicPushButton__sizeImageToWindow),
  IGraphicPushButton::sizeWindowToImage (1, IGraphicPushButton__sizeWindowToImage),
  IGraphicPushButton::classDefaultStyle (2, IGraphicPushButton__classDefaultStyle);

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
IGraphicPushButton::Style
  IGraphicPushButton::currentDefaultStyle (2, IGraphicPushButton__classDefaultStyle);


// Convenience definitions:
// The member containing the images is the IIconControl in PMWIN and the
// private data class in Motif.  Due to binary compatibility requirements
// the class member layout cannot be changed at this time.
// Also refresh is forced differently.
#ifdef IC_PMWIN
  #define IGRAPHIC_OBJECT aIcon
#endif //IC_PMWIN
#ifdef IC_MOTIF
  #define IGRAPHIC_OBJECT fGraphicPushButtonData
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| IGraphicPushButton::IGraphicPushButton                                       |
|                                                                              |
| Constructor to create a graphic push button control on a standard window.    |
------------------------------------------------------------------------------*/
IGraphicPushButton::IGraphicPushButton(unsigned long      id,
                                       IWindow*           parent,
                                       IWindow*           owner,
                                       const IResourceId& bitmapOrIconId,
                                       const IRectangle&  initial,
                                       const Style&       style)
    : IPushButton(id, parent,owner,IRectangle(),IWindow::noStyle)
#ifdef IC_PMWIN
    , returnBack(false)
    , border(3,3)
#endif // IC_PMWIN
#ifdef IC_MOTIF
    , border( 0,0 )
#endif
    , fGraphicPushButtonData( new IGraphicPushButtonData() )

{
  IASSERTPARM(parent!=0);
  initialize(style, initial, id);
  setGraphic(bitmapOrIconId);               // load and set graphic
  if (style & IWindow::visible)
  {
     show();
#ifdef IC_PMWIN
     refresh(IWindow::immediate);
#endif // IC_PMWIN
  }
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::IGraphicPushButton                                       |
|                                                                              |
| Constructor to create a graphic push button control on a standard window.    |
------------------------------------------------------------------------------*/
IGraphicPushButton::IGraphicPushButton(unsigned long     id,
                                       IWindow*          parent,
                                       IWindow*          owner,
                                       unsigned long     bitmapOrIconId,
                                       const IRectangle& initial,
                                       const Style&      style)
    : IPushButton(id, parent,owner, IRectangle(),IWindow::noStyle)
#ifdef IC_PMWIN
    , returnBack(false)
    , border(3,3)
#endif // IC_PMWIN
#ifdef IC_MOTIF
    , border( 0,0 )
#endif
    , fGraphicPushButtonData( new IGraphicPushButtonData()  )
{
  IASSERTPARM(parent!=0);
  initialize(style, initial, id);
  setGraphic(bitmapOrIconId);               // load and set graphic
  if (style & IWindow::visible)
  {
     show();
#ifdef IC_PMWIN
     refresh(IWindow::immediate);
#endif // IC_PMWIN
  }
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::IGraphicPushButton                                       |
|                                                                              |
| Constructor to create a graphic push button control on a standard window.    |
------------------------------------------------------------------------------*/
IGraphicPushButton::IGraphicPushButton(unsigned long        id,
                                       IWindow*             parent,
                                       IWindow*             owner,
                                       const IBitmapHandle& bitmap,
                                       const IRectangle&    initial,
                                       const Style&         style)
    : IPushButton(id, parent,owner, IRectangle(),IWindow::noStyle)
#ifdef IC_PMWIN
    , returnBack(false)
    , border(3,3)
#endif
#ifdef IC_MOTIF
    , border( 0,0 )
#endif
    , fGraphicPushButtonData( new IGraphicPushButtonData() )
{
  IASSERTPARM(parent!=0);
  initialize(style, initial, id);
  setGraphic(bitmap);               // load and set graphic
  if (style & IWindow::visible)
  {
     show();
#ifdef IC_PMWIN
     refresh(IWindow::immediate);
#endif // IC_PMWIN
  }
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::IGraphicPushButton                                       |
|                                                                              |
| Constructor to create a graphic push button control on a standard window.    |
------------------------------------------------------------------------------*/
IGraphicPushButton::IGraphicPushButton(unsigned long         id,
                                       IWindow*              parent,
                                       IWindow*              owner,
                                       const IPointerHandle& icon,
                                       const IRectangle&     initial,
                                       const Style&          style)
    : IPushButton(id, parent,owner, IRectangle(),IWindow::noStyle)
#ifdef IC_PMWIN
    , returnBack(false)
    , border(3,3)
#endif
#ifdef IC_MOTIF
    , border(0,0)
#endif
    , fGraphicPushButtonData( new IGraphicPushButtonData() )
{
  IASSERTPARM(parent!=0);
  initialize(style, initial, id);
  setGraphic(icon);               // load and set graphic
  if (style & IWindow::visible)
  {
     show();
#ifdef IC_PMWIN
     refresh(IWindow::immediate);
#endif // IC_PMWIN
  }
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::IGraphicPushButton                                       |
|                                                                              |
| Constructor to instantiate an object for a dialog template.                  |
------------------------------------------------------------------------------*/
IGraphicPushButton::IGraphicPushButton(unsigned long ulId,
                                       IWindow*      pdlgwndParent)
  : IPushButton(ulId,pdlgwndParent)
#ifdef IC_PMWIN
  , returnBack(true)
  , border(3,3)
#endif
#ifdef IC_MOTIF
  , border( 0,0)
#endif
  , fGraphicPushButtonData( new IGraphicPushButtonData() )
{
  IASSERTPARM(pdlgwndParent!=0);

#ifdef IC_PMWIN
#ifdef IC_PM
  BTNCDATA btn;
  WNDPARAMS wnd;
  wnd.fsStatus  = WPM_CTLDATA;
  wnd.cbCtlData = btn.cb = sizeof(BTNCDATA);
  wnd.pCtlData  = &btn;
  if (!(unsigned long)(handle().sendEvent(WM_QUERYWINDOWPARAMS, IEventParameter1(&wnd),
                                                 IEventParameter2(0))))
     ITHROWGUIERROR("WM_QUERYWINDOWPARAMS");
  LHANDLE hImage = btn.hImage;
#endif

#ifdef IC_WIN
  // The BS_ICON and BS_BITMAP styles are Win95 only.   Attempts to
  // use this function on NT 3.51 will probably result in the exception
  // being thrown.  BM_GETIMAGE is also documented as Win95 only.
  HANDLE hImage = (HANDLE) handle().sendEvent( BM_GETIMAGE );
  if (!hImage)
     ITHROWGUIERROR("BM_GETIMAGE");
#endif

  aIcon = new IIconControl(1,this,this);
  // Neither the image or the window are resized by default with a wrappered
  // control.
  enableSizeImageToWindow( false );
  enableSizeWindowToImage( false );
  if (style() & BS_BITMAP)
  {
     aIcon->setBitmap( IBitmapHandle( (IBitmapHandle::Value) hImage ) );
     contains = bitmapType;
  }
  else
  {
     aIcon->setIcon( IPointerHandle( (IPointerHandle::Value) hImage ) );
     contains = iconType;
  }

  fGraphicPushButtonData->fsizeHandler.handleEventsFor( this );
  fGraphicPushButtonData->fpaintHandler.handleEventsFor( this );
  fGraphicPushButtonData->ficonHandler.handleEventsFor( aIcon );
#endif // IC_PMWIN
#ifdef IC_MOTIF
  // Install window data for use by the default handler.
  this->adoptWindowData( ICustomButtonDrawData::dataHandle(),
                         new ICustomButtonDrawData );

  fGraphicPushButtonData->fsizeHandler.handleEventsFor( this );
  fGraphicPushButtonData->setFromWidget( this->handle() );
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::IGraphicPushButton                                       |
|                                                                              |
| Constructor to instantiate from an existing push button control.             |
------------------------------------------------------------------------------*/
IGraphicPushButton::IGraphicPushButton(const IWindowHandle& wh)
    : IPushButton(wh)
#ifdef IC_PMWIN
    , returnBack(true)
    , border(3,3)
#endif
#ifdef IC_MOTIF
    , border( 0,0)
#endif
    , fGraphicPushButtonData( new IGraphicPushButtonData() )
{
#ifdef IC_PMWIN
#ifdef IC_PM
  BTNCDATA btn;
  WNDPARAMS wnd;
  wnd.fsStatus  = WPM_CTLDATA;
  wnd.cbCtlData = btn.cb = sizeof(BTNCDATA);
  wnd.pCtlData  = &btn;
  if (!(unsigned long)(handle().sendEvent(WM_QUERYWINDOWPARAMS, IEventParameter1(&wnd),
                                                 IEventParameter2(0))))
     ITHROWGUIERROR("WM_QUERYWINDOWPARAMS");
  LHANDLE hImage = btn.hImage;
#endif

#ifdef IC_WIN
  // The BS_ICON and BS_BITMAP styles are Win95 only.   Attempts to
  // use this function on NT 3.51 will probably result in the exception
  // being thrown.  BM_GETIMAGE is also documented as Win95 only.
  HANDLE hImage = (HANDLE) handle().sendEvent( BM_GETIMAGE );
  if (!hImage)
     ITHROWGUIERROR("BM_GETIMAGE");
#endif

  aIcon = new IIconControl(1,this,this);
  // Neither the image or the window are resized by default with a wrappered
  // control.
  enableSizeImageToWindow( false );
  enableSizeWindowToImage( false );

  if (style() & BS_BITMAP)
  {
     aIcon->setBitmap( IBitmapHandle( (IBitmapHandle::Value) hImage ) );
     contains = bitmapType;
  }
  else
  {
     aIcon->setIcon( IPointerHandle( (IPointerHandle::Value) hImage ) );
     contains = iconType;
  }

  fGraphicPushButtonData->fsizeHandler.handleEventsFor( this );
  fGraphicPushButtonData->fpaintHandler.handleEventsFor( this );
  fGraphicPushButtonData->ficonHandler.handleEventsFor( aIcon );
#endif
#ifdef IC_MOTIF
  // Install window data for use by the default handler.
  this->adoptWindowData( ICustomButtonDrawData::dataHandle(),
                         new ICustomButtonDrawData );
  fGraphicPushButtonData->fsizeHandler.handleEventsFor( this );

  fGraphicPushButtonData->setFromWidget( this->handle() );
#endif
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::~IGraphicPushButton                                      |
------------------------------------------------------------------------------*/
IGraphicPushButton::~IGraphicPushButton()
{
#ifdef IC_PMWIN
   if ( returnBack && isValid() )
   {
      // Insure correct style settings for native standalone control.
      // The returnBack flag is only set when our ctor did not create
      // the control.
      if ( currentGraphicType() == bitmapType )
         {
         setStyle((style() & ~BS_ICON) | BS_BITMAP);
         setControlData(bitmap());
         }
      else
         {
         setStyle((style() & ~BS_BITMAP) | BS_ICON);
         setControlData( (IBitmapHandle::Value)
                         (IPointerHandle::Value) this->icon() );
         }
   } /* endif */
  fGraphicPushButtonData->fsizeHandler.stopHandlingEventsFor( this );
  fGraphicPushButtonData->fpaintHandler.stopHandlingEventsFor( this );
  if (aIcon)
  {
     fGraphicPushButtonData->ficonHandler.stopHandlingEventsFor( aIcon );
     delete aIcon;
  }
#endif // IC_PMWIN
#ifdef IC_MOTIF
  fGraphicPushButtonData->fsizeHandler.stopHandlingEventsFor( this );
#endif // IC_MOTIF
  delete fGraphicPushButtonData;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::defaultStyle                                             |
------------------------------------------------------------------------------*/
IGraphicPushButton::Style  IGraphicPushButton::defaultStyle()
{
  return currentDefaultStyle;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::setDefaultStyle                                          |
------------------------------------------------------------------------------*/
void  IGraphicPushButton::setDefaultStyle(const IGraphicPushButton::Style& pbsStyle)
{
  currentDefaultStyle = pbsStyle;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::convertToGUIStyle                                        |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IGraphicPushButton::convertToGUIStyle(const IBitFlag& guiStyle,
                                                    bool bExtOnly) const
{
  // Obtain the style from the class (IPushButton) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );
#ifdef IC_PM
  if (!bExtOnly)
  {
    // In PM BS_PUSHBUTTON added by IPushButton.
    ulStyle |= BS_BITMAP;
  }
#endif //IC_PM
    // In Windows WS_CHILD and BS_PUSHBUTTON or BS_DEFPUSHBUTTON added
    // by IPushButton.

  return( ulStyle );
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::setGraphic                                               |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::setGraphic(const IResourceId& bitmapOrIconId)
{
   IBitmapHandle bmp = bitmapOrIconId.resourceLibrary().tryToLoadBitmap(bitmapOrIconId);
   if (bmp == 0)
   {
      setGraphic(bitmapOrIconId.resourceLibrary().loadIcon(bitmapOrIconId));
   }
   else
      setGraphic(bmp);

  return *this;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::setGraphic                                               |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::setGraphic(unsigned long bitmapOrIconId)
{

  setGraphic(IResourceId(bitmapOrIconId));
  return *this;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::setGraphic                                               |
------------------------------------------------------------------------------*/
IGraphicPushButton&  IGraphicPushButton::setGraphic(const IBitmapHandle& bitmapHandle)
{
  // If last image was an icon, remove it.
  if (currentGraphicType() == iconType)
     IGRAPHIC_OBJECT->setIcon(IPointerHandle(0));

#ifdef IC_PMWIN
  contains=bitmapType;
#endif // IC_PMWIN
#ifdef IC_MOTIF
  fGraphicPushButtonData->contains = bitmapType;
#endif // IC_MOTIF

  IGRAPHIC_OBJECT->setBitmap(bitmapHandle);
  setLayoutDistorted(IWindow::minimumSizeChanged,0);
  if ( isSizeWindowToImageEnabled() )
  {
     sizeTo( this->calcMinimumSize( ) );
  }

  refresh();

  return *this;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::setGraphic                                               |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::setGraphic(const IPointerHandle& handle)
{
  // If last image was a bitmap, remove it.
  if (currentGraphicType() == bitmapType)
     IGRAPHIC_OBJECT->setBitmap(IBitmapHandle(0));

#ifdef IC_PMWIN
  contains = iconType;
#endif // IC_PMWIN
#ifdef IC_MOTIF
  fGraphicPushButtonData->contains = iconType;
#endif //IC_MOTIF

  IGRAPHIC_OBJECT->setIcon(handle);
  setLayoutDistorted(IWindow::minimumSizeChanged,0);
  if ( isSizeWindowToImageEnabled() )
  {
     sizeTo( this->calcMinimumSize( ) );
  }

  refresh();

  return *this;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IGraphicPushButton::graphicWindow                                            |
------------------------------------------------------------------------------*/
IIconControl&  IGraphicPushButton::graphicWindow() const
{
   return *aIcon;
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IGraphicPushButton::bitmap                                                   |
------------------------------------------------------------------------------*/
IBitmapHandle  IGraphicPushButton::bitmap() const
{
   if (currentGraphicType() == bitmapType)
      return IGRAPHIC_OBJECT->bitmap();
   else
      return 0;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::icon                                                     |
------------------------------------------------------------------------------*/
IPointerHandle  IGraphicPushButton::icon() const
{
   if (currentGraphicType() == iconType)
   {
      return IGRAPHIC_OBJECT->icon();
   }
   else
      return 0;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::currentGraphicType                                       |
------------------------------------------------------------------------------*/
IGraphicPushButton::GraphicType IGraphicPushButton::currentGraphicType ( ) const
{
#ifdef IC_PMWIN
   return contains;
#endif // IC_PMWIN
#ifdef IC_MOTIF
   return fGraphicPushButtonData->contains;
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::setControlData                                           |
------------------------------------------------------------------------------*/
IGraphicPushButton&  IGraphicPushButton::setControlData(const IBitmapHandle& aHandle)
{
#ifdef IC_PM
   BTNCDATA btn;
   WNDPARAMS wnd;
   wnd.fsStatus  = WPM_CTLDATA;
   wnd.cbCtlData = btn.cb = sizeof(BTNCDATA);
   wnd.pCtlData  = &btn;
   btn.hImage = aHandle;
   btn.fsCheckState = 0;
   btn.fsHiliteState = 0;
   if (!(unsigned long)(handle().sendEvent(WM_SETWINDOWPARAMS, IEventParameter1(&wnd),
                                                  IEventParameter2(0))))
      ITHROWGUIERROR("WM_SETWINDOWPARAMS");
#endif
#ifdef IC_WIN
   // Message documented as Win95 only.  We only call this for wrappered
   // controls being destructed in Windows.
   handle().sendEvent(BM_SETIMAGE,
                      IEventParameter1( 0 ),
                      IEventParameter2( aHandle.asUnsigned() ) );
#endif

   return *this;
}

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
/*------------------------------------------------------------------------------
| IGraphicPushButton::enableSizeToGraphic                                      |
------------------------------------------------------------------------------*/
IGraphicPushButton&  IGraphicPushButton::enableSizeToGraphic( bool enable)
{
   // The sizeToGraphic style is a misnomer.  It acually controls whether
   // the image is stretched to fit the window or not.
   return this->enableSizeImageToWindow( !enable );
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::disableSizeToGraphic                                     |
------------------------------------------------------------------------------*/
IGraphicPushButton&  IGraphicPushButton::disableSizeToGraphic()
{
   enableSizeToGraphic(false);
   return *this;
}


/*------------------------------------------------------------------------------
| IGraphicPushButton::isSizeToGraphic                                          |
------------------------------------------------------------------------------*/
bool  IGraphicPushButton::isSizeToGraphic() const
{
   return !this->isSizeImageToWindowEnabled();
}
#endif // IC_OBSOLETE

/*------------------------------------------------------------------------------
| IGraphicPushButton::calcMinimumSize                                          |
|                                                                              |
| Calculate the minimum screen size needed by the control.  The height is      |
| based on the bitmap/icon height and border height and the width is based on  |
| the bitmap/icon width and the border width.                                  |
------------------------------------------------------------------------------*/
ISize  IGraphicPushButton::calcMinimumSize() const
{
#ifdef IC_PMWIN
  ISize size = aIcon->minimumSize();
#endif
#ifdef IC_MOTIF
  Dimension shadowThickness(0), borderWidth(0), highlightThickness(0),
            defaultButtonShadowThickness(0), showAsDefault(0),
            marginHeight(0), marginWidth(0),
            marginLeft(0),   marginRight(0), marginTop(0), marginBottom(0);

  XtVaGetValues(this->handle(),
                XmNshadowThickness,    &shadowThickness,
                XmNborderWidth,        &borderWidth,
                XmNhighlightThickness, &highlightThickness,
                XmNdefaultButtonShadowThickness, &defaultButtonShadowThickness,
                XmNshowAsDefault,      &showAsDefault,
                XmNmarginHeight,       &marginHeight,
                XmNmarginWidth,        &marginWidth,
                XmNmarginLeft,         &marginLeft,
                XmNmarginRight,        &marginRight,
                XmNmarginTop,          &marginTop,
                XmNmarginBottom,       &marginBottom,
                0);


  // Hack.  There is a private value Xm3D_ENHANCE_PIXEL with
  // value of 2 which is added to the highlight thickness
  // value in XmPushButton.  This occurs ONLY if
  // defaultButtonShadowThickness is != 0.  There are also some differences
  // in the button size when there is no default button shadow at all.
  int enhance3D   ( defaultButtonShadowThickness ? 2 : 0 );
  int defThickness( enhance3D ?
                    defaultButtonShadowThickness : showAsDefault);
  int edges;
  if (defThickness)
     edges = 2*(shadowThickness + defThickness) +
             highlightThickness + enhance3D + borderWidth;
  else
     edges =  shadowThickness + highlightThickness + borderWidth;
  ISize size(2*edges,2*edges);   //Base size of widget

  // Compute size of image and add it in.
  ISize imageSize;
  if (this->currentGraphicType() == bitmapType)
  {
      IBitmapHandle bmp = this->bitmap();
      if (bmp)
      {
         IGImage image(bmp);
         imageSize = IPoint(image.rasterBounds().size());
      }
  }
  else
  {
      IPointerHandle icon = this->icon();
      if (icon)
      {
  #ifdef IC_NOTYET
         IGImage image(icon);
         imageSize = IPoint(image.rasterBounds().size());
  #else
         IBitmapData* bmpdata =
             IBitmapStaticPtr::pointerSet().find( icon );
         imageSize =  bmpdata->size() ;
  #endif //IC_NOTYET
      }
  }

  if (this->textLength() > 0)
  {
     ISize textSize( displaySize( text() ) );
     textSize += ISize( marginLeft + marginRight,
                        marginTop + marginBottom);
     imageSize = imageSize.maximum( textSize );
  }
  size += imageSize;
#endif //IC_MOTIF
  return ISize(size.width()+2*border.width(),size.height()+2*border.height());
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::initialize                                               |
------------------------------------------------------------------------------*/
IGraphicPushButton&  IGraphicPushButton::initialize(const Style& bmsStyle,
                                                    const IRectangle& initial,
                                                    unsigned long id)
{
#ifdef IC_PMWIN
  aIcon = new IIconControl(id,this,this);
  aIcon->setBackgroundColor( this->backgroundColor());
  aIcon->setAlignment(IStaticText::centerCenter);

  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | bmsStyle.asExtendedUnsignedLong() );

  fGraphicPushButtonData->fsizeHandler.handleEventsFor( this );
  fGraphicPushButtonData->fpaintHandler.handleEventsFor( this );
  fGraphicPushButtonData->ficonHandler.handleEventsFor( aIcon );

  unsigned long ulStyle = convertToGUIStyle( bmsStyle );
  if (bmsStyle & IWindow::visible)
     ulStyle &= ~WS_VISIBLE;   // We enable visibility later.
  setStyle(ulStyle);
#endif // IC_PMWIN
#ifdef IC_MOTIF
  // Install window data for use by the default handler.
  this->adoptWindowData( ICustomButtonDrawData::dataHandle(),
                         new ICustomButtonDrawData );
  fGraphicPushButtonData->fsizeHandler.handleEventsFor( this );

  // Remove fillOnArm so that image stays in the button when it is clicked.
  XtVaSetValues( this->handle(),  XmNfillOnArm, False, 0);

  // Handle Push Button styles
  if (bmsStyle & IPushButton::noBorder)
    addBorder( false);
  enableDefault( bmsStyle & IPushButton::defaultButton );
  enableSystemCommand( bmsStyle & IPushButton::systemCommand );
  enableMouseClickFocus( (bmsStyle & noPointerFocus) ? false : true );
#endif //IC_MOTIF

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
  if (bmsStyle & IGraphicPushButton::sizeToGraphic )
     enableSizeToGraphic(true);
#endif // IC_OBSOLETE
  if (bmsStyle & IGraphicPushButton::sizeImageToWindow )
     enableSizeImageToWindow(true);
  if (bmsStyle & IGraphicPushButton::sizeWindowToImage )
     enableSizeWindowToImage(true);

  moveSizeTo(initial);

  return *this;
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::marginSize                                               |
------------------------------------------------------------------------------*/
ISize  IGraphicPushButton::marginSize() const
{
   return border;
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::setMarginSize                                            |
------------------------------------------------------------------------------*/
IGraphicPushButton&  IGraphicPushButton::setMarginSize(const ISize& newSize)
{
   border = newSize;
#ifdef IC_PMWIN
   long inith  = newSize.height(),
        initw  = newSize.width(),
        width  = -initw,
        height = -inith;

   ISize window = size();
   if ((width += window.width() ) < 0)
      width = 0;
   if ((height+= window.height()) < 0)
      height = 0;
   aIcon->moveSizeTo(IRectangle(initw,inith,width,height));
#endif
#ifdef IC_MOTIF
   refresh();
#endif //IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::enableSizeImageToWindow                                  |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::enableSizeImageToWindow  ( bool enable )
{
#ifdef IC_MOTIF
   bool needRefresh (
      ( this->fGraphicPushButtonData->fsizeImageToWindow && !enable ) ||
      ( !this->fGraphicPushButtonData->fsizeImageToWindow && enable ) );
#endif //IC_MOTIF

   this->fGraphicPushButtonData->fsizeImageToWindow = enable;

#ifdef IC_PMWIN
   aIcon->enableSizeImageToWindow( enable );
   // The IIconControl also handles refresh.
#endif //IC_PMWIN
#ifdef IC_MOTIF
   if (needRefresh)
      refresh();
#endif //IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::disableSizeImageToWindow                                 |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::disableSizeImageToWindow ( )
{
   return this->enableSizeImageToWindow( false );
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::isSizeImageToWindowEnabled                               |
------------------------------------------------------------------------------*/
bool  IGraphicPushButton::isSizeImageToWindowEnabled( ) const
{
  return this->fGraphicPushButtonData->fsizeImageToWindow;
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::enableSizeWindowToImage                                  |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::enableSizeWindowToImage  ( bool enable )
{
   this->fGraphicPushButtonData->fsizeWindowToImage = enable;
#ifdef IC_PMWIN
   aIcon->enableSizeWindowToImage( enable );
#endif //IC_PMWIN
   return *this;
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::disableSizeWindowToImage                                 |
------------------------------------------------------------------------------*/
IGraphicPushButton& IGraphicPushButton::disableSizeWindowToImage ( )
{
   return this->enableSizeWindowToImage( false );
}

/*------------------------------------------------------------------------------
| IGraphicPushButton::isSizeWindowToImageEnabled                               |
------------------------------------------------------------------------------*/
bool  IGraphicPushButton::isSizeWindowToImageEnabled( ) const
{
   return this->fGraphicPushButtonData->fsizeWindowToImage;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IGraphicPushButton::enable                                                   |
|                                                                              |
| Enable the object to accept input.  This override provided to force a        |
| refresh of the graphic image.                                                |
------------------------------------------------------------------------------*/
IWindow& IGraphicPushButton::enable ( bool bEnable )
{
  // First check to see if we need to do anything.  If not, return.
  // This has the effect of NOT performing unnecessary drawing.
  if ( bEnable && isEnabled() )
     return *this;
  else if ( !bEnable && !isEnabled() )
     return *this;

  // Set the enabled state in the usual way.
  Inherited::enable( bEnable );

  // Refresh the image.
  refresh();
  return *this;
}
#endif //IC_MOTIF

