// Revision: 15 1.9.2.2 source/ui/baseapp/iwposbuf.cpp, datatype, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iwposbuf.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iwposbuf.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINWINDOWMGR      // WinSetMultWindowPos, WinQueryWindow
#include <iwindefs.h>

#include <iwposbuf.hpp>
#include <ibidiset.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#include <iwindow.hpp>
#include <ikeyset2.h>


/*------------------------------------------------------------------------------
| IWindowPosSet                                                                |
------------------------------------------------------------------------------*/
class IWinPosSet : public IKeySet < IWindowPosBuffer::Item*,
                                    unsigned long >  {
public:
  IWinPosSet ( );
 ~IWinPosSet ( );
}; // IWinPosSet


// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IWPOSBUF_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| IWinPosSet::IWinPosSet                                                       |
|                                                                              |
| Constructor here for page tuning.                                            |
------------------------------------------------------------------------------*/
IWinPosSet::IWinPosSet ( )
  : IKeySet < IWindowPosBuffer::Item*, unsigned long > ( )
{ }

/*------------------------------------------------------------------------------
| IWinPosSet::~IWinPosSet                                                      |
|                                                                              |
| Destructor here for page tuning.                                             |
------------------------------------------------------------------------------*/
IWinPosSet::~IWinPosSet ( )
{ }


/*------------------------------------------------------------------------------
| IWindowPosBufferDeleter                                                      |
------------------------------------------------------------------------------*/
bool
  IWindowPosBufferDeleter ( IWindowPosBuffer::Item* const& element,
                            void* /*unused*/ )
{
  delete element;
  return true;
}


/*------------------------------------------------------------------------------
| IWindowPosBuffer::IWindowPosBuffer                                           |
|                                                                              |
| Constructor                                                                  |
------------------------------------------------------------------------------*/
IWindowPosBuffer::IWindowPosBuffer ( IWindow* pwndParent )
  : pwndClParent(pwndParent)
{
   IASSERTPARM(pwndParent != 0);
   pClBufSet = new IWinPosSet();
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::IWindowPosBuffer                                           |
|                                                                              |
| Copy constructor.                                                            |
------------------------------------------------------------------------------*/
IWindowPosBuffer::IWindowPosBuffer ( const IWindowPosBuffer& original )
  : pwndClParent(original.pwndClParent)
{
   pClBufSet = new IWinPosSet();
   IWinPosSet::Cursor cursor(*(original.pClBufSet));

   for(cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
   {                        // Copy set elements from original to new
      Item* pOrigItem = original.pClBufSet->elementAt(cursor);
      Item* pNewItem = new Item(*pOrigItem);
      pClBufSet->add(pNewItem);
   } /* endfor */
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::~IWindowPosBuffer                                          |
------------------------------------------------------------------------------*/
IWindowPosBuffer::~IWindowPosBuffer ( )
{
   if (pClBufSet) {
      empty();
      delete pClBufSet;
   } /* endif */
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::position                                                   |
|                                                                              |
| Returns the proposed position of the specified child window.  If the child   |
| is not in the collection, the actual window position is returned.            |
------------------------------------------------------------------------------*/
IPoint IWindowPosBuffer::position ( const IWindow* pwndChild ) const
{
   IASSERTPARM(pwndChild != 0);
   IASSERT(isParentOk(pwndChild->handle()) );

   if (pClBufSet->containsElementWithKey(pwndChild->handle().asUnsigned() ))
   {                                         // Already entry for window
      Item* pItem = pClBufSet->elementWithKey(pwndChild->handle().asUnsigned());
      if (pItem->isMove())
      {                                      // Window to be moved
         return pItem->position();           // Return new value
      }
   } /* endif */

   return pwndChild->position();
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::size                                                       |
|                                                                              |
| Returns the proposed size of the specified child window.  If the child is    |
| not in the collection, the actual window size is returned.                   |
------------------------------------------------------------------------------*/
ISize IWindowPosBuffer::size ( const IWindow* pwndChild ) const
{
   IASSERTPARM(pwndChild != 0);
   IASSERT(isParentOk(pwndChild->handle()) );

   if (pClBufSet->containsElementWithKey(pwndChild->handle().asUnsigned() ))
   {                                    // Already entry for window
      Item* pItem = pClBufSet->elementWithKey(pwndChild->handle().asUnsigned());
      if (pItem->isSize( ) )
      {                                // Window to be sized
         return pItem->size();         // Return new value
      }
   } /* endif */

   return pwndChild->size();
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::rect                                                       |
|                                                                              |
| Returns the proposed position and size of the specfied child window.  If the |
| child is not in the collection, the actual window position and size is       |
| returned.                                                                    |
------------------------------------------------------------------------------*/
IRectangle IWindowPosBuffer::rect ( const IWindow* pwndChild ) const
{
   IASSERTPARM(pwndChild != 0);
   IASSERT(isParentOk(pwndChild->handle()) );

   IPoint ptChild = pwndChild->position();
   ISize szChild = pwndChild->size();
   if (pClBufSet->containsElementWithKey(pwndChild->handle().asUnsigned() ))
   {                              // Already entry for window
      Item* pItem = pClBufSet->elementWithKey(pwndChild->handle().asUnsigned());

      if ( pItem->isMove() )
      {                           // Window to be moved
         ptChild = pItem->position();  // Use new position
      }
      if ( pItem->isSize() )
      {                           // Window to be sized
         szChild = pItem->size();        // Use new size
      }
   } /* endif */

   return IRectangle(ptChild, szChild);
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::moveTo                                                     |
|                                                                              |
| Specifies a new position for the child window.                               |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::moveTo ( IWindow* pwndChild,
                                             const IPoint& pt )
{
   IASSERTPARM(pwndChild != 0);
   return moveTo(pwndChild->handle(), pt);
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::moveTo                                                     |
|                                                                              |
| Specifies a new position for the child window.                               |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::moveTo ( const IWindowHandle& hwndChild,
                                             const IPoint& pt )
{
   IASSERT(isParentOk(hwndChild) );

   Item* pItem = createOrLocate(hwndChild);
   pItem->setMove( true ).setNoAdjust(true);
   pItem->setPosition(pt);
   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::sizeTo                                                     |
|                                                                              |
| Specifies a new size for the child window.                                   |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::sizeTo ( IWindow* pwndChild,
                                             const ISize& siz )
{
   IASSERTPARM(pwndChild != 0);
   return sizeTo(pwndChild->handle(), siz);
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::sizeTo                                                     |
|                                                                              |
| Specifies a new size for the child window.                                   |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::sizeTo ( const IWindowHandle& hwndChild,
                                             const ISize& siz )
{
   IASSERT(isParentOk(hwndChild) );

   Item* pItem = createOrLocate(hwndChild);
   pItem->setSizeFlag( true ).setNoAdjust( true );
   pItem->setSize(siz);
   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::moveSizeTo                                                 |
|                                                                              |
| Specifies a new position and size for the child window.                      |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::moveSizeTo ( IWindow* pwndChild,
                                                 const IRectangle& rect )
{
   IASSERTPARM(pwndChild != 0);

#ifdef IC_MOTIF
   // The rectangle is in native coordinates. Recoordinate it if
   // necessary before calling IWindow::moveSizeTo, which expects
   // a rectangle in application coordinates.
   if ( ICoordinateSystem::isConversionNeeded() )
   {
      ISize parentSize = pwndClParent->size();
      IRectangle applRect =
         ICoordinateSystem::convertToApplication( rect, parentSize );
      pwndChild->moveSizeTo( applRect );

      // Recoordination causes a problem for a IMultiCellCanvas with
      // no size. IMultiCellCanvas has an optimization for this case
      // when it runs its layout routine. To try to eliminate a layout
      // when it does get sized, it sizes and positions its child
      // windows relative to its minimum size. This way if the canvas
      // is later sized to its minimum size, it can skip running
      // layout. However, if the child windows have been recoordinated
      // for a zero-size parent window, this optimization doesn't
      // work. So for this case, we reposition the child based on its
      // position in native coordinates. This code isn't ideal, since
      // any specialized code performed by an override of
      // IWindow::moveSizeTo could be negated. However,
      // IWindowPosBuffer doesn't even go through IWindow::moveSizeTo
      // on intel.
      if ( parentSize == ISize() )
      {
         this->moveTo( pwndChild, rect.minXMinY() );
      }
   }
   else
   {
      pwndChild->moveSizeTo( rect );
   }
   return *this;
#else
   return moveSizeTo(pwndChild->handle(), rect);
#endif
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::moveSizeTo                                                 |
|                                                                              |
| Specifies a new position and size for the child window.                      |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::moveSizeTo ( const IWindowHandle& hwndChild,
                                                 const IRectangle& rect )
{
#ifdef IC_MOTIF
   return moveSizeTo( (IWindow::windowWithHandle(hwndChild)), rect);
#else
  moveTo( hwndChild, rect.minXMinY() );
  sizeTo( hwndChild, rect.size() );
   
// IASSERT(isParentOk(hwndChild) );

// Item* pItem = createOrLocate(hwndChild);
// pItem->flClOption |= (SWP_SIZE | SWP_MOVE | SWP_NOADJUST);
// pItem->rectCl = rect;
   return *this;
#endif   
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::show                                                       |
|                                                                              |
| Specifies that a child window should be made visible.                        |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::show ( IWindow* pwndChild )
{
   IASSERTPARM(pwndChild != 0);
   return show(pwndChild->handle());
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::show                                                       |
|                                                                              |
| Specifies that a child window should be made visible.                        |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::show ( const IWindowHandle& hwndChild )
{
   IASSERT(isParentOk(hwndChild) );

   Item* pItem = createOrLocate(hwndChild);
   pItem->setShow( true ).setHide( false );
   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::hide                                                       |
|                                                                              |
| Specifies that a child window should be made invisible.                      |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::hide ( IWindow* pwndChild )
{
   IASSERTPARM(pwndChild != 0);
   return hide(pwndChild->handle());
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::hide                                                       |
|                                                                              |
| Specifies that a child window should be made invisible.                      |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::hide ( const IWindowHandle& hwndChild )
{
   IASSERT(isParentOk(hwndChild) );

   Item* pItem = createOrLocate(hwndChild);
   pItem->setShow( false ).setHide( true );
   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::orderAfter                                                 |
|                                                                              |
| Specifies the child window to be moved after another child in the z-order.   |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::orderAfter ( IWindow* pwndChild,
                                                 IWindow* pwndSibling )
{
   IASSERTPARM(pwndChild != 0);
   IASSERTPARM(pwndSibling != 0);
   return orderAfter(pwndChild->handle(), pwndSibling->handle());
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::orderAfter                                                 |
|                                                                              |
| Specifies the child window to be moved after another child in the z-order.   |
------------------------------------------------------------------------------*/
IWindowPosBuffer&
  IWindowPosBuffer::orderAfter ( const IWindowHandle& hwndChild,
                                 const IWindowHandle& hwndSibling )
{
   IASSERT(isParentOk(hwndChild));
   IASSERT(isParentOk(hwndSibling));

   return setOrder(hwndChild, hwndSibling);
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::orderAsFirst                                               |
|                                                                              |
| Specifies the child window to be made first in the z-order.                  |
| Notes: Making more than one child window "first" with this call gives        |
|          unpredictable results (depends on the order of Item objects, based  |
|          on their keys).                                                     |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::orderAsFirst ( IWindow* pwndChild )
{
   IASSERTPARM(pwndChild != 0);
   return orderAsFirst(pwndChild->handle());
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::orderAsFirst                                               |
|                                                                              |
| Specifies the child window to be made first in the z-order.                  |
| Notes: Making more than one child window "first" with this call gives        |
|          unpredictable results (depends on the order of Item objects, based  |
|          on their keys).                                                     |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::orderAsFirst ( const IWindowHandle& hwndChild )
{
   IASSERT(isParentOk(hwndChild));
#ifdef IC_PMWIN
   return setOrder(hwndChild, IWindowHandle(HWND_TOP));
#endif
#ifdef IC_MOTIF
   return setOrder(hwndChild, hwndChild );
#endif
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::orderAsLast                                                |
|                                                                              |
| Specifies the child window to be made last in the z-order.                   |
| Notes: Making more than one child window "last" with this call gives         |
|          unpredictable results (depends on the order of Item objects, based  |
|          on their keys.                                                      |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::orderAsLast ( IWindow* pwndChild )
{
   IASSERTPARM(pwndChild != 0);
   return orderAsLast(pwndChild->handle());
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::orderAsLast                                                |
|                                                                              |
| Specifies the child window to be made last in the z-order.                   |
| Notes: Making more than one child window "last" with this call gives         |
|          unpredictable results (depends on the order of Item objects, based  |
|          on their keys.                                                      |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::orderAsLast ( const IWindowHandle& hwndChild )
{
   IASSERT(isParentOk(hwndChild));
#ifdef IC_PMWIN
   return setOrder(hwndChild, IWindowHandle(HWND_BOTTOM));
#endif
#ifdef IC_MOTIF
   return setOrder(hwndChild, hwndChild );
#endif
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::setOrder                                                   |
|                                                                              |
| Add the z-order change of the child window to the buffer.                    |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::setOrder ( const IWindowHandle& hwndChild,
                                               const IWindowHandle& hwndSibling )
{
   Item* pItem = createOrLocate(hwndChild);
   pItem->setZOrder( true );
   pItem->behind = hwndSibling;
   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::apply                                                      |
|                                                                              |
| Causes all proposed window changes to be made effective.                     |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::apply ( bool checkBidi )
{
   IMODTRACE_DEVELOP("IWindowPosBuffer::apply");

   // Check if the parent window orientation is right-to-left.
   bool  rightToLeft = false;
   ISize parentSize;
   if ( checkBidi && IBidiSettings::isBidiSupported() )
   {
      IBidiSettings settings(*pwndClParent);
      if ( settings.windowLayout() == IBidiSettings::layoutRightToLeft )
      {
         rightToLeft = true;
         // parentSize is only needed for rightToLeft layout.
         parentSize = pwndClParent->size();
      }
   }

   unsigned long ulCount = pClBufSet->numberOfElements();
   if (ulCount != 0)
   {                             // At least one child window changed
#ifdef   IC_MOTIFPM
      ISWP* paswp = new ISWP[ulCount];
#endif   //IC_PM
#ifdef   IC_WIN
      ISWP* paswp = new ISWP[1];    // only need one of these
      HDWP  hdwp = BeginDeferWindowPos( (int)ulCount );
      if (!hdwp)
      {
         ITHROWGUIERROR2("BeginDeferWindowPos",
                         IBaseErrorInfo::outOfSystemResource,
                         IException::unrecoverable);
      }
#endif   //IC_WIN

      ISWP* pswpPtr = paswp;
      ISWP  swpCurrent;
      IWinPosSet::Cursor cursor(*pClBufSet);

      for(cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
         {                         // Iterate list of change entries
         bool bChange = false;  // Assume no change to entry
         Item* pItem = pClBufSet->elementAt(cursor);
         // Fix position for right-to-left layout (if needed)
         if ( rightToLeft )
         {
            pItem->setPosition ( IPoint ( parentSize.width() -
                                          pItem->size().width() -
                                          pItem->position().x(),
                                          pItem->position().y() ));
         }
         // copy the current element
         *pswpPtr = *pItem;
         // get position
#ifdef   IC_PM
         WinQueryWindowPos(pswpPtr->hwnd, (SWP*)&swpCurrent);
#endif   //IC_PM
#ifdef   IC_WIN
         {
            RECT wrect;
            GetWindowRect( pswpPtr->hwnd, &wrect);
            MapWindowPoints( HWND_DESKTOP,
                             pwndClParent->handle(),
                             (POINT*)&wrect, 2);
            swpCurrent.x = wrect.left;
            swpCurrent.y = wrect.top;
            swpCurrent.cx = wrect.right - wrect.left;
            swpCurrent.cy = wrect.bottom - wrect.top;
            swpCurrent.behind = GetWindow( pswpPtr->hwnd, GW_HWNDPREV );
            swpCurrent.hwnd = pswpPtr->hwnd;
            swpCurrent.fl = SWP_NOZORDER | SWP_FRAMECHANGED |
                            SWP_SHOWWINDOW | SWP_HIDEWINDOW ;
            // Always leave show/hide flags on.  Disabling updates
            // (e.g. notebooks) affects visiblity
         }
#endif   //IC_WIN
#ifdef   IC_MOTIF
         {
            Dimension x, y, width, height, border;
            XtVaGetValues( pswpPtr->hwnd,
              XmNx, &x,
              XmNy, &y,
              XmNwidth, &width,
              XmNheight, &height,
              XmNborder, &border,
              NULL );
            // if the widget is not managed then its size is 0!
            if (!XtIsManaged( pswpPtr->hwnd))
            {
              width = 0;
              height = 0;
            }

            swpCurrent.x = x;
            swpCurrent.y = y;
            swpCurrent.cx = width - 2*border;
            swpCurrent.cy = height - 2*border;
            //swpCurrent.behind = GetWindow( pswpPtr->hwnd, GW_HWNDPREV );
            swpCurrent.hwnd = pswpPtr->hwnd;
            swpCurrent.fl = SWP_SHOW | SWP_HIDE ;
         }
#endif   //IC_MOTIF

         ITRACE_ALL( IString("swpCurrent hwnd=") +
                     IString( swpCurrent.hwnd.asUnsigned() ).d2x() +
                     IString(" origin=") + swpCurrent.position().asString() +
                     IString(" size=") + swpCurrent.size().asString() +
                     IString(" flags=") + IString( swpCurrent.fl ).d2x() );

         // unsigned long fulOption = pItem->flags();
         if ( pItem->isSize() )
         {                      // Size change
            if (pswpPtr->cx != swpCurrent.cx  ||
                pswpPtr->cy != swpCurrent.cy)
            {                   // Size really changing
               bChange = true;  // Flag a change
            }
            else
            {                   // No size change
               pswpPtr->setSizeFlag( false );
            }
         }
         else
         {
            // Need to set a valid size for recoordination
            pswpPtr->setSize( swpCurrent.size() );
         }

         if ( pItem->isMove() )
         {                      // Position change
            // No coordinate system adjustments needed, since
            // IWindowPosBuffer only deals with native coordinates,
            // as does the API that returns the information in swpCurrent.

            if (pswpPtr->x != swpCurrent.x  ||
                pswpPtr->y != swpCurrent.y)
            {                   // Position really changing
               bChange = true;  // Flag a change
            }
            else
            {                   // No position change
               pswpPtr->setMove( false ); // Unflag it
            }
         }
         else
         {
            pswpPtr->setPosition( swpCurrent.position() );
         }

         if ( pItem->isZOrder() )
         {                      // Z-order change
            if (pswpPtr->behind != swpCurrent.behind)
            {                   // Z-order really changing
               bChange = true;     // Flag a change
            }
            else
            {                   // No z-order change
               pswpPtr->setZOrder( false );  // Unflag it
            }
         }
         if ( pItem->isShow() && swpCurrent.isHide() )
         {                      // Visibility change
            bChange = true;     // Flag a change
         }
         else
         {                      // No visibility change
            pswpPtr->setShow( false ); // Unflag it
         }
         if ( pItem->isHide() && swpCurrent.isShow( ) )
         {                      // Visibility change
            bChange = true;     // Flag a change
         }
         else
         {                      // No visibility change
            pswpPtr->setHide( false );  // Unflag it
         }

         if (bChange)
         {                      // Change to the window
#ifdef   IC_MOTIFPM
            ITRACE_ALL( IString("WindowPos hwnd=") +
                        IString( pswpPtr->hwnd.asUnsigned() ).d2x() +
                        IString(" origin=") + pswpPtr->position().asString() +
                        IString(" size=") + pswpPtr->size().asString() +
                        IString(" flags=") + IString( pswpPtr->fl ).d2x() );
            pswpPtr++;             // Save this entry
#endif   //IC_PM
#ifdef   IC_WIN
             // Tell Windows about the changes
            ITRACE_ALL( IString("DeferWindowPos hwnd=") +
                        IString( pswpPtr->hwnd.asUnsigned() ).d2x() +
                        IString(" origin=") + pswpPtr->position().asString() +
                        IString(" size=") + pswpPtr->size().asString() +
                        IString(" flags=") + IString( pswpPtr->fl ).d2x() );
            hdwp = DeferWindowPos(  hdwp,
                                    pswpPtr->hwnd,
                                    pswpPtr->behind,
                                    (int)pswpPtr->x,
                                    (int)pswpPtr->y,
                                    (int)pswpPtr->cx,
                                    (int)pswpPtr->cy,
                                    (unsigned int)pswpPtr->fl );
#endif   //IC_WIN
         }
         else
         {                      // No change to window
            ulCount--;          // Throw out this entry
         }
      } /* endfor */

#ifdef   IC_PM
      if (pswpPtr != paswp)
      {                         // Change to a window(s)
         if (WinSetMultWindowPos(IThread::current().anchorBlock(),
                                 (SWP*)paswp, ulCount) == false)
         {                      // Error from WinSetMultWindowPos
            ITHROWGUIERROR2("WinSetMultWindowPos",
                            IBaseErrorInfo::invalidParameter,
                            IException::unrecoverable);
         } /* endif */
      }
#endif   //IC_PM
#ifdef   IC_WIN
      if ( !EndDeferWindowPos(hdwp) )
      {
         ITHROWGUIERROR2("EndDeferWindowPos",
                         IBaseErrorInfo::invalidParameter,
                         IException::unrecoverable);
      }
#endif   //IC_WIN
#ifdef IC_MOTIF
      for ( int swpIndex = 0; swpIndex < ulCount; swpIndex++ )
      {
        Dimension border;
        XtVaGetValues( paswp[swpIndex].hwnd,
          XmNborderWidth, &border,
          NULL );
        Dimension width, height;
        width = paswp[swpIndex].cx - 2*border;
        height = paswp[swpIndex].cy - 2*border;

        if (paswp[swpIndex].isShow() )
          XtVaSetValues( paswp[swpIndex].hwnd,
                         XmNmappedWhenManaged, true,
                         NULL );

        if ( width && height )
        {
          // if the widget is not managed then it was sized to 0
          // at some point. Now that it has a size make sure it
          // is managed.
          XtVaSetValues( paswp[swpIndex].hwnd,
            XmNx, paswp[swpIndex].x,
            XmNy, paswp[swpIndex].y,
            XmNwidth, width,
            XmNheight, height,
            NULL );
          if ( !XtIsManaged( paswp[swpIndex].hwnd ) )
            XtManageChild( paswp[swpIndex].hwnd );
        }
        else
          XtUnmanageChild( paswp[swpIndex].hwnd );
      }
#endif
      delete [] paswp;
      empty();
      } /* endif */

   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::empty                                                      |
|                                                                              |
| Removes all window change entries from the collection.                       |
------------------------------------------------------------------------------*/
IWindowPosBuffer& IWindowPosBuffer::empty ( )
{
   pClBufSet->removeAll(&IWindowPosBufferDeleter);
   return *this;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::isParentOk                                                 |
|                                                                              |
| Verify parent is valid.                                                      |
------------------------------------------------------------------------------*/
bool IWindowPosBuffer::isParentOk ( const IWindowHandle& hwndChild ) const
{
   bool bValid = true;
#ifdef IC_DEVELOP
#ifdef IC_PMWIN
   bValid =
     (IPARENTOF(hwndChild) == pwndClParent->handle());
#endif
#endif
   return bValid;
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::createOrLocate                                             |
|                                                                              |
| Return the entry for the specified window if it already exists.  Otherwise   |
| create an entry and add it to the keyed set of entries ( the entry is        |
| assumed to be deleted when the keyed set is cleaned up).                     |
------------------------------------------------------------------------------*/
IWindowPosBuffer::Item*
  IWindowPosBuffer::createOrLocate ( const IWindowHandle& hwndChild )
{
   Item* pItem = 0;
   if (pClBufSet->containsElementWithKey( hwndChild.asUnsigned() ))
   {                              // Already entry for window
      pItem = pClBufSet->elementWithKey( hwndChild.asUnsigned() );
   }
   else
   {
      pItem = new Item(hwndChild);
      pClBufSet->add(pItem);
   }
   return pItem;
}


/*------------------------------------------------------------------------------
| IWindowPosBuffer::Item::Item                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowPosBuffer::Item::Item ( const IWindowHandle& handle )
  : ISWP (),
    fkeyHwnd( handle.asUnsigned() )
{
   hwnd = handle;
#ifdef IC_WIN
   flags() = SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_FRAMECHANGED;
#endif
}

/*------------------------------------------------------------------------------
| IWindowPosBuffer::Item::Item                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowPosBuffer::Item::Item ( const IWindowPosBuffer::Item& original )
  : ISWP (original),
    fkeyHwnd( original.fkeyHwnd )
{ }

/*------------------------------------------------------------------------------
| IWindowPosBuffer::Item::~Item                                                |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowPosBuffer::Item::~Item ( )
{ }
