// Revision: 06 1.25 source/ui/baseapp/iwindow4.cpp, basewin, ioc.v400
/*******************************************************************************
* FILE NAME: iwindow4.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iwindow.hpp that are Windows unique.                                    *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

/*
   Portability Problems:
    ITheObjectWindow     ==> WinQueryObjectWindow
    isShowing     == >  not supported in Windows; implemented as isVisible
    color                ==> WinQueryPresParam
    setColor             ==> WinSetPresParam
                             We have implemented our own support (color/setColor)
                             for WinQueryPresParam and WinSetPresParam
*/


extern "C" {
  #define IUSING_OLE
  #ifdef WIN32_LEAN_AND_MEAN
    #undef WIN32_LEAN_AND_MEAN
  #endif
  #ifdef _OLE2_H_
    #undef _OLE2_H_
  #endif
  #include <iwindefs.h>
}

#ifdef IC_WIN  // Ensure clean compile in PM/AIX environment

#include <iwindow.hpp>
#include <iwinpriv.hpp>
#include <iwinlsts.hpp>

#include <icconst.h>
#include <icolor.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <imcevt.hpp>
#include <iplatfrm.hpp>
#include <irect.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IWINDOW4_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| ITheObjectWindowHandle::ITheObjectWindowHandle                               |
------------------------------------------------------------------------------*/
ITheObjectWindowHandle::ITheObjectWindowHandle( )
   : fhandle(0)
{
   IMODTRACE_DEVELOP("ITheObjectWindowHandle::ITheObjectWindowHandle");
   IWindowHandle focusWindow = IQUERYFOCUS(0);

   fhandle = ICREATEWINDOW(
              0,                        // parent
              WC_STATIC,                // class
              "ITheObjectWindow",       // window text
              WS_POPUP | SS_GRAYRECT,   // Note not WS_VISIBLE
              0,                        // x
              0,                        // y
              0,                        // width
              0,                        // height
              0,                        // owner (unused in WIN)
              0,                        // behind (unused in WIN)
              0,                        // id/menu handle
              0,                        // window creation data
              0,                        // unused in win
              0     );                  // hInst
   if (fhandle == 0)
      {
      ITHROWGUIERROR2("CreateWindow",
                      IBaseErrorInfo::accessError,
                      IException::recoverable);
      }
   // restore focus back to original window
   ISETFOCUS(0, focusWindow);
}

/*------------------------------------------------------------------------------
| ITheObjectWindowHandle::~ITheObjectWindowHandle                              |
------------------------------------------------------------------------------*/
ITheObjectWindowHandle::~ITheObjectWindowHandle( )
{
  IMODTRACE_DEVELOP("ITheObjectWindowHandle::~ITheObjectWindowHandle");
  IDESTROYWINDOW(fhandle);
}

/*------------------------------------------------------------------------------
| IWindow::size                                                                |
|                                                                              |
| Get the window size.                                                         |
------------------------------------------------------------------------------*/
ISize IWindow::size() const
{
   return rect().size();
}

/*------------------------------------------------------------------------------
| IWindow::parentSize                                                          |
------------------------------------------------------------------------------*/
ISize IWindow::parentSize ( ) const
{
   return parentSize( this->handle() );
}

/*------------------------------------------------------------------------------
| IWindow::parentSize                                                          |
------------------------------------------------------------------------------*/
ISize IWindow::parentSize ( const IWindowHandle& windowhandle )
{
   IMODTRACE_ALL("parentSize");
   IWindowHandle  parentHwnd = IPARENTOF( windowhandle );
   if ( parentHwnd == IWindowHandle(0) )
      {
      parentHwnd = IWindow::desktopWindow()->handle();
      }

   // the size needed is the size of the coordinate space which is used
   // to position child windows.
#ifdef   IC_WIN
   // If the window is a popup, use the desktop as the parent since
   //    1) By default this window has no parent
   //    2) A query for the parent always returns the owner window
   if ( ISTYLEOF( windowhandle ) & WS_POPUP )
     parentHwnd = IWindow::desktopWindow()->handle();

   RECT rect;
   GetClientRect(parentHwnd, &rect);

   ITRACE_ALL( IString("parentSize=") +
               ISize(rect.right - rect.left, rect.bottom - rect.top ).asString() );
   return  ISize(rect.right - rect.left, rect.bottom - rect.top );
#endif   //IC_WIN
#ifdef   IC_PM
   SWP swp;
   WinQueryWindowPos(parentHwnd, &swp);
   ITRACE_ALL( IString("parentSize=") +
               ISize(swp.cx, swp.cy ).asString() );
   return (ISize(swp.cx,swp.cy));
#endif   //IC_PM
}

/*------------------------------------------------------------------------------
| IWindow::position                                                            |
|                                                                              |
| Gets the object position.                                                    |
------------------------------------------------------------------------------*/
IPoint IWindow::position() const
{
   return rect().minXMinY();
}

/*------------------------------------------------------------------------------
| IWindow::rect                                                                |
|                                                                              |
| Gets the object position and size.                                           |
------------------------------------------------------------------------------*/
IRectangle IWindow::rect() const
{
   if ( ICoordinateSystem::isConversionNeeded() )
      {
      return ICoordinateSystem::convertToApplication( nativeRect( ),
                                                      this->parentSize( ) );
      }
   else
      {
      return nativeRect( );
      }
}

/*------------------------------------------------------------------------------
| IWindow::nativeRect                                                          |
------------------------------------------------------------------------------*/
IRectangle IWindow::nativeRect ( ) const
{
   RECT   rect;
   IQUERYWINDOWRECT(handle(), &rect);
   IWindowHandle  parentHwnd = IPARENTOF( handle() );

#ifdef   IC_WIN
   // If the window is a popup, use the desktop as the parent since
   //    1) By default this window has no parent
   //    2) A query for the parent always returns the owner window
   if ( ISTYLEOF( handle() ) & WS_POPUP )
     parentHwnd = IWindow::desktopWindow()->handle();
#endif   //IC_WIN

   MapWindowPoints( HWND_DESKTOP, parentHwnd, (POINT*)&rect, 2);
   return IRectangle( rect );
}

/*------------------------------------------------------------------------------
| IWindow::sizeTo                                                              |
|                                                                              |
| Size the window.                                                             |
------------------------------------------------------------------------------*/
IWindow& IWindow::sizeTo(const ISize& siz)
{
   IMODTRACE_ALL("Win::sizeTo");
   ITRACE_ALL(siz.asString());
   moveSizeTo(IRectangle( position(), siz) );
   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::moveTo                                                              |
|                                                                              |
| Move the object.                                                             |
------------------------------------------------------------------------------*/
IWindow& IWindow::moveTo(const IPoint& pt)
{
   IMODTRACE_ALL("Win::moveTo");
   ITRACE_ALL(pt.asString());
   moveSizeTo(IRectangle( pt, size()) );
   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::moveSizeTo                                                          |
|                                                                              |
| Move and size the object.                                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::moveSizeTo(const IRectangle& rect)
{
   IMODTRACE_ALL("Win::moveSizeTo");
   ITRACE_ALL(rect.asString());

   IWindowHandle  hwnd  = this->handle();

   // Sorry, but you need to mess about with display drivers to change
   // desktop window size.
   if ( hwnd == IWindow::desktopWindow()->handle() )
      return *this;

   // recoordination ... translate to native coordinate system
   IRectangle nativeRectangle = rect;
   if ( ICoordinateSystem::isConversionNeeded() )
      {
      nativeRectangle = ICoordinateSystem::convertToNative( rect,
                                                            parentSize() );
      }

   // set the window position
   ITRACE_ALL( IString("SetWindowPos hwnd=") +
               IString(hwnd.asUnsigned()).d2x() +
               IString(" position=") +
               nativeRectangle.minXMinY().asString() +
               IString(" size=") +
               nativeRectangle.size().asString()  );
   if (! SetWindowPos( hwnd, 0,
                       (int)nativeRectangle.minX(),
                       (int)nativeRectangle.minY(),  // really X origin
                       (int)nativeRectangle.width(),
                       (int)nativeRectangle.height(),
                       SWP_NOZORDER | SWP_NOACTIVATE ) )
      {
      ITHROWGUIERROR2("SetWindowPos",
                      IBaseErrorInfo::invalidParameter,
                      IException::recoverable);
      }

#ifdef IC_TRACE_DEVELOP
   RECT tRect;
   GetWindowRect( hwnd, &tRect);
   IRectangle  newRect(tRect);
   // GetWindowRect always returns screen coordinates
   // and not relative to parent.
   ITRACE_DEVELOP( IString(" real rect=") + newRect.asString() );
#endif

   return *this;
}



/*------------------------------------------------------------------------------
| IWindow::positionBehindSiblings                                              |
|                                                                              |
| Put this window in the Z order behind all of its siblings.                   |
------------------------------------------------------------------------------*/
IWindow& IWindow::positionBehindSiblings ( )
{
   IMODTRACE_DEVELOP("Win::positionBehindSiblings");
   SetWindowPos(handle(), HWND_BOTTOM, 0,0,0,0,
                SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);

   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::positionOnSiblings                                                  |
|                                                                              |
| Put this window in the Z order on top of all of its siblings.                |
------------------------------------------------------------------------------*/
IWindow& IWindow::positionOnSiblings ( )
{
   IMODTRACE_DEVELOP("Win::positionBehindSiblings");
   SetWindowPos(handle(), HWND_TOP, 0,0,0,0,
                SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE );

   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::positionBehind                                                      |
|                                                                              |
| Put this window in the Z order behind the passed window.                     |
------------------------------------------------------------------------------*/
IWindow& IWindow::positionBehindSibling ( const IWindowHandle& siblingWindow)
{
   IMODTRACE_DEVELOP("Win::positionBehindSiblings");
   SetWindowPos(handle(), siblingWindow, 0,0,0,0,
                SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);

   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::isShowing                                                           |
|                                                                              |
| Is the window showing?  True if the window not being moved                   |
| and some part of the window is currently on the screen.                      |
------------------------------------------------------------------------------*/
bool IWindow::isShowing() const
{
// return WinIsWindowShowing(handle());

// Not supported in Windows.  For now if the window is visible, we will
// assume it is also showing
//
   return isVisible();
}

/*------------------------------------------------------------------------------
| IChildCursorData::IChildCursorData                                           |
|                                                                              |
------------------------------------------------------------------------------*/
IChildCursorData::IChildCursorData( IWindowHandle parentHandle )
    : hwndParent( parentHandle )
{
}

/*------------------------------------------------------------------------------
| IChildCursorData::~IChildCursorData                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IChildCursorData::~IChildCursorData()
{
}

/*------------------------------------------------------------------------------
| IChildCursorData::setToFirst                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
bool IChildCursorData::setToFirst( IWindowHandle &hwnd )
{
  hwnd = GetWindow( hwndParent, GW_CHILD );
  if ( hwnd )
    return true;

  // No exception, since there either is a handle (return true) or
  // not (return false).
  return false;
}

/*------------------------------------------------------------------------------
| IChildCursorData::setToNext                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
bool IChildCursorData::setToNext(IWindowHandle &hwnd)
{
  // If First, get next, else return first
  if ( !hwnd )
    return setToFirst(hwnd);
  else
  {
    hwnd = GetWindow( hwnd, GW_HWNDNEXT );
    if ( hwnd )
    // Got it, return true...
      return true;
  }

  // Failure...
  return false;
}

/*------------------------------------------------------------------------------
| IChildCursorData::invalidate                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
void IChildCursorData::invalidate(IWindowHandle &hwnd)
{
   hwnd = IWindowHandle(0);
}

/*------------------------------------------------------------------------------
| IWindow::ChildCursor::ChildCursor                                            |
|                                                                              |
| The constructor simply saves the parent handle and initializes the cursor    |
| state to "invalid" (enumHandle = windowHandle == 0).                         |
------------------------------------------------------------------------------*/
IWindow::ChildCursor::ChildCursor ( IWindow &parent, bool onlyIWindowChildren )
    : hwnd( 0 )
    , pCursorData( new IChildCursorData( parent.handle()) )
{
}

/*------------------------------------------------------------------------------
| IWindow::ChildCursor::~ChildCursor                                           |
|                                                                              |
| If the cursor enum handle is valid, issue WinEndEnumWindows.                 |
------------------------------------------------------------------------------*/
IWindow::ChildCursor::~ChildCursor ( )
{
  delete pCursorData;
}

/*------------------------------------------------------------------------------
| IWindow::ChildCursor::setToFirst                                             |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::ChildCursor::setToFirst ( )
{
  return pCursorData->setToFirst( hwnd);
}

/*------------------------------------------------------------------------------
| IWindow::ChildCursor::setToNext                                              |
|                                                                              |
| Set the saved window handle to the value returned by WinGetNextWindow.       |
------------------------------------------------------------------------------*/
bool IWindow::ChildCursor::setToNext ( )
{
  return pCursorData->setToNext(hwnd);
}

/*------------------------------------------------------------------------------
| IWindow::ChildCursor::isValid                                                |
|                                                                              |
| Return true if the saved window handle isn't null.                           |
------------------------------------------------------------------------------*/
bool IWindow::ChildCursor::isValid ( ) const
{
  return hwnd != 0;
}

/*------------------------------------------------------------------------------
| IWindow::ChildCursor::invalidate                                             |
|                                                                              |
| If enum handle is valid, end the enumeration.                                |
------------------------------------------------------------------------------*/
void IWindow::ChildCursor::invalidate ( )
{
  pCursorData->invalidate( hwnd );
}

/*------------------------------------------------------------------------------
| IWindow::foregroundColor                                                     |
|                                                                              |
| Returns the foreground color of the window.                                  |
------------------------------------------------------------------------------*/
IColor IWindow::foregroundColor () const
{
  return this->color( PP_FOREGROUNDCOLOR );
}

/*------------------------------------------------------------------------------
| IWindow::backgroundColor                                                     |
|                                                                              |
| Returns the background color of the window.                                  |
------------------------------------------------------------------------------*/
IColor IWindow::backgroundColor () const
{
  return this->color( PP_BACKGROUNDCOLOR );
}


/*------------------------------------------------------------------------------
| IDialogControl::IDialogControl                                               |
------------------------------------------------------------------------------*/
IDialogControl::IDialogControl( )
  : fHwnd        (0),
    fOriginalProc(0),
    fDialog      (0)
{}

/*------------------------------------------------------------------------------
| IDialogControl::IDialogControl                                               |
------------------------------------------------------------------------------*/
IDialogControl::IDialogControl( const IDialogControl& control)
  : fHwnd        ( control.fHwnd ),
    fOriginalProc( control.fOriginalProc ),
    fDialog      ( control.fDialog )
{}

/*------------------------------------------------------------------------------
| IDialogControl::IDialogControl                                               |
------------------------------------------------------------------------------*/
IDialogControl& IDialogControl:: operator= ( const IDialogControl& control )
{
  fHwnd         = control.fHwnd;
  fOriginalProc = control.fOriginalProc;
  fDialog       = control.fDialog;
  return *this;
}

/*------------------------------------------------------------------------------
| IDialogControl::IDialogControl                                               |
------------------------------------------------------------------------------*/
IDialogControl::~IDialogControl   ()
{
}

/*------------------------------------------------------------------------------
| IDialogControls::IDialogControls                                             |
------------------------------------------------------------------------------*/
IDialogControls::IDialogControls () :
     IKeySetAsHshTable< IDialogControl, unsigned long>(19)
{
}

/*------------------------------------------------------------------------------
| IDialogControls::~IDialogControls                                            |
------------------------------------------------------------------------------*/
IDialogControls::~IDialogControls ( )
{
  // Remove the subclassing for any valid windows
  IDialogControls::Cursor cursor( *this );
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
     {
     if ( IsWindow( (HWND)cursor.element().fHwnd ) )
        {
        // Verify that no additional subclassing has occurred.  If not,
        // put back the original proc.  We can't do anything if the
        // control has been subclassed again.
        unsigned long  currentP = GetWindowLong( (HWND)cursor.element().fHwnd,
                                                 GWL_WNDPROC) ;
        if (currentP == (unsigned long)pfnwpICControlProc)
           {
           ISUBCLASSWINDOW( (HWND)cursor.element().fHwnd,
                            (unsigned long)cursor.element().fOriginalProc );
           }
        }
     }
}

/*------------------------------------------------------------------------------
| _pfnwpICControlProc                                                          |
------------------------------------------------------------------------------*/
unsigned long __stdcall pfnwpICControlProc(HWND          hwnd,
                                           unsigned long ulMsg,
                                           void*         mp1,
                                           void*         mp2)
{
   IMODTRACE_ALL("pfnwpICControlProc");
   IDialogControls*        dialogControls =
      IDialogControls::listForThread(IThread::currentId());
   IDialogControls::Cursor cursor( *dialogControls );
 #ifdef IC_WIN_STRICT
   WNDPROC                 originalProc = 0;
   if (dialogControls->locateElementWithKey( (unsigned long)hwnd, cursor) )
      {
      originalProc = (WNDPROC) cursor.element().fOriginalProc;
      }
 #else
   FARPROC                 originalProc = 0;
   if (dialogControls->locateElementWithKey( (unsigned long)hwnd, cursor) )
      {
      originalProc = (FARPROC) cursor.element().fOriginalProc;
      }
 #endif

   if (originalProc)
      {
      switch (ulMsg)
         {
         case  WM_KEYDOWN:
         case  WM_KEYUP:
         case  WM_CHAR:
         case  WM_SYSKEYDOWN:
         case  WM_SYSKEYUP:
         case  WM_SYSCHAR:
            // We need to route these messages to the parent IWindow to allow for
            // keyboard navigation.  We can't use the Windows IsDialogMessage
            // function because it requires knowledge of which dialog (read
            // IFrameWindow) we want to route events to.
            {
            ITRACE_DEVELOP( IString("pfnwpICControlProc hwnd=") +
                            IString((unsigned long)hwnd).d2x()+
                            IString(", Msg=") + IString(ulMsg).d2x()+
                            IString(", wParam=") + IString((unsigned long)mp1).d2x()+
                            IString(", lParam=") + IString((unsigned long)mp2).d2x() );
            unsigned long result = 0;
            IWindowHandle dialog = cursor.element().fDialog;

            if (!IWindow::windowWithHandle( dialog, false ))
               {
               // For some reason the dialog is no longer an IWindow.
               // Just call the original procedure and return.
               result =
                  CallWindowProc( originalProc,
                                  hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
               }
            else
               {
               // get the dialog code to see what this control is.
               unsigned long dlgCode =
                  SendMessage( hwnd, WM_GETDLGCODE, 0, 0);

               ITRACE_DEVELOP( IString("GETDLGCODE=") + IString(dlgCode).d2x() );

               // Special case for edit fields
               if ( ( ulMsg == WM_CHAR ) && ( dlgCode & DLGC_HASSETSEL ) )
                  {
                  bool toDialog = true;

                  // An edit field.  Need to check multiline edit controls
                  // for ES_WANTRETURN to determine return behavior.
                  unsigned long ulStyle = ISTYLEOF(hwnd);

                  // Note that WC_MLE and WC_ENTRYFIELD are same class in Win
                  // Have to check the style.
                  if ( ( IWindowClassName(hwnd) == WC_ENTRYFIELD ) &&
                       ( ulStyle & ES_MULTILINE) )
                     {
                     if ( ((unsigned long)mp1 != VK_RETURN ) ||
                          (ulStyle & ES_WANTRETURN) )
                        toDialog = false;
                     }

                  if ( toDialog &&
                       ( ((unsigned long)mp1 == VK_RETURN ) ||
                         ((unsigned long)mp1 == VK_ESCAPE ) ||
                         ((unsigned long)mp1 == VK_TAB    ) ) )
                     // Send to dialog and return result now
                     return dialog.sendEvent( ulMsg, mp1, mp2 );
                  }

               // Call the default procedure for the control.
               result =
                  CallWindowProc( originalProc,
                                  hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);

               // If not processed, send event to the dialog.
               if (!result)
                  {
                  ITRACE_ALL("pfnwpICControlProc - to dialog");
                  result = dialog.sendEvent(ulMsg, mp1, mp2);
                  }
               } // else dialog is IWindow
            return result;
            }

         case  WM_MATCHMNEMONIC:
            // We implement WM_MATCHMNEMONIC  to allow canvas and frame
            // navigation.
            // IWindow
            return IWindowPrivateData::matchMnemonicHwnd(
                                   LOWORD((unsigned long)mp1),
                                   hwnd );

         case  WM_KILLFOCUS:
         {
            // Catch dialog control setfocus events, and pass a new
            // user msg up to the frame to process the focus event
            IWindowHandle dialog = cursor.element().fDialog;

            // if the dialog proc is not wrappered, then simply pass
            // the message on or if this msg is for the dialog, then
            // simply pass it on.  DO not send it again.
            if (!IWindow::windowWithHandle( dialog, false ) ||
                 dialog == hwnd )
                // pass on to default proc
                return CallWindowProc( originalProc,
                                hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
            else
            {
                // Dialog proc is wrappered, so let default proc have
                // a chance at it, and then we look at it to see if we
                // want to do anything extra
                CallWindowProc( originalProc,
                             hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
                // send new user msg, p1=losing focus hwnd, p2=getting focushwnd
                // NOTE must change msg: FFAB to a macro before checking in.
                unsigned long dlgCode =
                    PostMessage( dialog, IC_UM_DLGCNTL_KILLFOCUS, (WPARAM)mp1, (LPARAM)hwnd );
                return dlgCode;
            }
          }
            break;

         case  WM_SETFOCUS:
         {
            // Catch dialog control setfocus events, and pass a new
            // user msg up to the frame to process the focus event
            IWindowHandle dialog = cursor.element().fDialog;

            // if the dialog proc is not wrappered, then simply pass
            // the message on or if this msg is for the dialog, then
            // simply pass it on.  DO not send it again.
            if (!IWindow::windowWithHandle( dialog, false ) ||
                 dialog == hwnd )
                // pass on to default proc
                return CallWindowProc( originalProc,
                                hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
            else
            {
                // Dialog proc is wrappered, so let default proc have
                // a chance at it, and then we look at it to see if we
                // want to do anything extra
                CallWindowProc( originalProc,
                             hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
                // send new user msg, p1=losing focus hwnd, p2=getting focushwnd
                // NOTE must change msg: FFAB to a macro before checking in.
                unsigned long dlgCode =
                    PostMessage( dialog, IC_UM_DLGCNTL_SETFOCUS, (WPARAM)mp1, (LPARAM)hwnd );
                return dlgCode;
            }
          }
            break;

         case  WM_DESTROY:
            // remove this control from the list.  Don't bother removing
            // the subclass since the window is being destroyed
            dialogControls->removeAt( cursor );
            return 0;

         default:
            return CallWindowProc( originalProc,
                                   hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
         }  // switch
      }  // if originalProc
   else
      {
      // Can't find IWindow ... call default proc
      return DefWindowProc( hwnd, (UINT)ulMsg, (WPARAM)mp1, (LPARAM)mp2);
      }
   return 0;
}


/*------------------------------------------------------------------------------
| IWindowPrivateData::matchMnemonicHwnd                                        |
| This function used to implement WM_MATCHMNEMONIC.  Not to be confused        |
| with the non static matchForMnemonic which is overridden by canvases         |
| to check children for mnemonics.                                             |
------------------------------------------------------------------------------*/
unsigned long IWindowPrivateData::matchMnemonicHwnd ( unsigned short character,
                                                      IWindowHandle hwnd )
{
   bool result = false;
   // only visible, enabled, child controls need to do anything
   if (  (ISTYLEOF( hwnd ) & (WS_CHILD | WS_DISABLED | WS_VISIBLE)) ==
         (WS_CHILD | WS_VISIBLE)  )
      {
      char  textBuffer[256];   // arbitrary max len
      int copied = GetWindowText(hwnd, textBuffer, 255);
      if (copied)
         {
         IString text(textBuffer);
         // find first non-escaped & character
         unsigned beginPos = text.indexOf('&');
         while ((beginPos != 0) &&
                (text.indexOf('&', beginPos+1) == beginPos+1) )
            beginPos = text.indexOf('&', beginPos + 2);

         if (beginPos)
            {
            IString target =
               IString("&") +
               IString( (char)character ) ;

            // See if there is a match
            if ((text.indexOf(target.upperCase(), beginPos) ==
                 beginPos) ||
                (text.indexOf(target.lowerCase(), beginPos) ==
                 beginPos))
               result =  true ;
            }  // found &
         }  // if has text
      }  // if child
   return result;
}


/*------------------------------------------------------------------------------
| IWindow::isDragStarting                                                       |
|
| Windows does not generate drag messages, so therefore it is up to a
| a control to decide if the conditions exist to start a drag. This is
| is the basic implementation that will just look for a down followed by
| a move more than xxx. If this occurs before we have an up event then
| we have a drag.
|
| Note: if you don't want the event to continue up the owner chain,
|       then set the event result to true.
------------------------------------------------------------------------------*/
bool IWindow::isDragStarting( IEvent &event )
{
  IMODTRACE_DEVELOP("IWindow::isDragStarting");
  bool brc = false;

  switch ( event.eventId() )
  {
    case WM_LBUTTONDOWN:
    case WM_RBUTTONDOWN:
    {
      ITRACE_DEVELOP("Got a button down, start to look for drag");
      IMouseClickEvent mevt( event );

      // We need to use ChildWindowFromPoint instead of WindowFromPoint
      // which IMouseEvent::windowUnderPointer uses to find disabled window
      POINT pt = { (short)mevt.parameter2().number1(),
                   (short)mevt.parameter2().number2() };
      IWindowHandle childHwnd ( ChildWindowFromPoint( mevt.handle(), pt) );

      if ( (childHwnd) && !(ISTYLEOF( childHwnd ) & WS_DISABLED) )
      {
        // In Windows 95, the system apparently captures the pointer itself.
        // during the down event and releases it in the up event.  If we
        // mess with this incorrectly it causes no WM_COMMANDs for the
        // button clicks to be sent.
        IWindowHandle captureHandle = GetCapture( );
        if (( captureHandle != NULL ) && (captureHandle != event.handle()) )
        {
          // This means that someone already has a capture...do nothing
//          SetCapture( captureHandle );
          ITRACE_DEVELOP(IString("Pointer already captured hwnd=") +
                         IString(captureHandle.asUnsigned()).d2x() +
                         IString(" event.handle()=")+
                         IString(event.handle().asUnsigned()).d2x());
        }
        else
        {
          if (captureHandle == NULL)
             {
             ITRACE_DEVELOP(IString("capturing pointer hwnd=") +
                            IString(event.handle().asUnsigned()).d2x());
             SetCapture( event.handle() );
             }
          pWindowData->bCheckingDragStart = True;
          pWindowData->pointLButtonDown =
              mevt.mousePosition();
        }
      }
    break;
    }

    case WM_LBUTTONUP:
    case WM_RBUTTONUP:
    {
      ITRACE_DEVELOP("Got a button up event");
      if ( pWindowData->bCheckingDragStart)
      {
         if (!( IPlatform::isWin9x() || IPlatform::isNTNewShell() ))
           {
             // Win95 appears to release the capture when processing buttonup
             // If we release it we get no WM_COMMAND for button clicks.
             ITRACE_DEVELOP(IString("Releasing capture GetCapture=") +
                            IString((unsigned long)GetCapture()).d2x());
             SetCapture( NULL );
           }
           else
           {
             // On 95, if captured by us, and not already released by WM_MOUSEMOVE
             // processing, (for example user clicks and releases button, without moving
             // mouse) release it here.

              IWindowClassName className( event.handle());
              // defect 29828
              // There are some controls on 9x and NT that release the capture
              // by themselves (ie. they get capture when the mouse is down, thus
              // they release it also).  For
              //    WC_BUTTON
              //    WC_ENTRYFIELD
              //    WC_LISTBOX
              // the os tracks these messages as we can see by holding the
              // mouse down and then releasing them outside the current
              // application.
              // If we release the capture here, we will mess up following
              // command events to the control, thus in the above cases
              // we do not release the capture but let the os do it.
              if (!(className == WC_BUTTON || className == WC_ENTRYFIELD ||
                    className == WC_LISTBOX                             ) )
              {
               IWindowHandle captureHandle = GetCapture( );
               if (( captureHandle != NULL ) && (captureHandle == event.handle()) )
               {
                  ITRACE_DEVELOP(IString("Releasing capture GetCapture=") +
                              IString((unsigned long)GetCapture()).d2x()  );
                  SetCapture( NULL );
               }
              }
           }
      }
      pWindowData->bCheckingDragStart = False;
      pWindowData->pointLButtonDown = IPoint(0,0);
      break;
    }

    case WM_MOUSEMOVE:
    {
      if (pWindowData->bCheckingDragStart)
      {
        ITRACE_DEVELOP("Got a move and DragStart is true");
        IMouseClickEvent mevt( event );
        IPoint mPoint( pWindowData->pointLButtonDown );
        if ( abs(mevt.mousePosition().x() - mPoint.x()) >
                 DD_DEFDRAGMINDIST ||
             abs(mevt.mousePosition().y() - mPoint.y()) >
                 DD_DEFDRAGMINDIST )
        {
          ITRACE_DEVELOP("We have a drag" );
          SetCapture( NULL );
          pWindowData->bCheckingDragStart = FALSE;
          //We have detected a drag occurring
          brc = true;
        }
      }
    }
    break;
#ifdef IC_TRACE_DEVELOP
    // Windows 95 only
    case WM_CAPTURECHANGED:
    {
       ITRACE_DEVELOP(IString("WM_CAPTURECHANGED new capture=") +
                      IString(event.parameter2().asUnsignedLong()).d2x() +
                      IString(" event.handle()=")+
                      IString(event.handle().asUnsigned()).d2x());
    }
    break;
#endif
  }
  return brc;
}

/*------------------------------------------------------------------------------
| IWindow::passEventToOwner                                                    |
|                                                                              |
| Returns whether the event should be passed to its owner or not. This will    |
| help simulate PM's behavior.                                                 |
------------------------------------------------------------------------------*/
bool IWindow::passEventToOwner ( IEvent& evt )
{
  evt.setPassToOwner( false );

  switch ( evt.eventId() )
  {
    case WM_KEYDOWN      :
    case WM_KEYUP        :
    case WM_CHAR         :
    case WM_DEADCHAR     :
    case WM_SYSKEYDOWN   :
    case WM_SYSKEYUP     :
    case WM_SYSCHAR      :
    case WM_SYSDEADCHAR  :
//    case WM_MOUSEMOVE    :
    case WM_LBUTTONDOWN  :
    case WM_LBUTTONUP    :
    case WM_LBUTTONDBLCLK:
    case WM_RBUTTONDOWN  :
    case WM_RBUTTONUP    :
    case WM_RBUTTONDBLCLK:
    case WM_MBUTTONDOWN  :
    case WM_MBUTTONUP    :
    case WM_MBUTTONDBLCLK:
      evt.setPassToOwner( true );
      break;
    case IC_UM_LBUTTONDBLCLK:
      if ( IISWINDOW( 0, evt.handle()))
        evt.setPassToOwner( true );
      break;
  }
  return evt.passToOwner();
}

#endif // IC_WIN
