// Revision: 02 1.37.2.5 source/ui/baseapp/iwindow0.cpp, basewin, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iwindow0.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in iwindow.hpp that are common to all environments.                        *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
// Priority is 10 more (more negative) than controls or rest of IWindow,
// meaning statics in here get constructed before and destructed after
// controls or other IWindow statics.
#pragma priority( -2147482122 )

extern "C" {
  #include <iwindefs.h>
  #ifdef IC_MOTIF
    #include <Xm/BulletinB.h>
  #endif
}

#include <iwindow.hpp>
#include <iacceltb.hpp>
#ifdef IC_MOTIF
  #include <iarglist.hpp>
#endif
#include <iatname.hpp>
#include <iattribt.hpp>
#include <icconst.h>
#include <ievent.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <iguilock.hpp>
#include <ihandler.hpp>
#include <iinhratt.hpp>
#include <ilanglvl.hpp>
#include <inotifev.hpp>
#include <iperfset.hpp>
#include <iplatfrm.hpp>
#include <ipoint.hpp>
#include <iprimlck.hpp>
#include <irect.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#include <ithread.hpp>
#include <iwinlsts.hpp>
#include <iwinpriv.hpp>
#include <istdntfy.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IWINDOW0_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Static Data.                                                                 |
------------------------------------------------------------------------------*/

  //--------------------------------------------------------------------------
  // Static data requiring a constructor and/or destructor.
  //--------------------------------------------------------------------------
static IWindowStaticData  gStatics;    // Lone instance of IWindowStaticData

                                       // used to manage the following items.
  IPrivateResource*       IWindowStaticData::flibKey = 0;
  IWindowThreadObserver*  IWindowStaticData::fgObserver = 0;
  IWindowThreadDataList*  IWindowStaticData::fgThreadList = 0;
  IDataHandleSet*         IWindowStaticData::fgDataHandleSet = 0;
  IPrivateResource*       IWindowStaticData::fgDictionaryKey = 0;
#ifdef IC_MOTIF
  IWindowSortedSet*       IWindowStaticData::fgAllIWindowsSet = 0;
#endif
  IWindowSortedSet*       IWindowStaticData::fgAllAllocatedIWindowsSet = 0;
  ITheDesktopWindow*      IWindowStaticData::desktopWindow = 0;
  ITheObjectWindow*       IWindowStaticData::objectWindow = 0;

  //--------------------------------------------------------------------------
  // Static data not requiring constructors or destructors
  //--------------------------------------------------------------------------
  IWindow::ExceptionFn* IWindowPrivateData::exceptionFunction = 0;
  IWindow::SiblingOrder IWindowPrivateData::siblingCreateOrder =
                                            IWindow::behindSiblings;

  //This flag disables the fast window procedure optimization on a
  // global basis.  In Windows and PM, the optimization is enabled.
  // It is disabled in Motif.
#ifdef IC_PMWIN
  bool   IWindowPrivateData::fgFastWindowProcEnabled = true;
#else
  bool   IWindowPrivateData::fgFastWindowProcEnabled = false;
#endif

#ifdef IC_MOTIFWIN
unsigned long        propagationInProgress = 0;
#endif  // IC_MOTIFWIN


#ifdef IC_DEVELOP
unsigned long IWindowList::totalEvents = 0;
unsigned long IWindowList::totalWindowsAdded = 0;
unsigned long IWindowList::totalWindowsDeleted = 0;
unsigned long IWindowList::cacheUsedCount = 0;
#endif

#pragma enum(4)
#pragma pack(push,4)

/*------------------------------------------------------------------------------
| IWindowAttributeCursorData class                                             |
------------------------------------------------------------------------------*/
class IWindowAttributeCursorData {
public:
  IWindowAttributeCursorData ( const IWindowAttributeKeySet* keySet );
 ~IWindowAttributeCursorData ( );

IWindowAttributeKeySet::Cursor
  fCursor;
};  // IWindowAttributeCursorData

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IWindowAttributeCursorData::IWindowAttributeCursorData                       |
------------------------------------------------------------------------------*/
IWindowAttributeCursorData::IWindowAttributeCursorData
                                 ( const IWindowAttributeKeySet* keySet )
  : fCursor( *keySet )
{ }

/*------------------------------------------------------------------------------
| IWindowAttributeCursorData::~IWindowAttributeCursorData                      |
------------------------------------------------------------------------------*/
IWindowAttributeCursorData::~IWindowAttributeCursorData ( )
{ }

/*------------------------------------------------------------------------------
| IWindowPrivateData::IWindowPrivateData                                       |
------------------------------------------------------------------------------*/
IWindowPrivateData::IWindowPrivateData ( IWinProc*     windowProcedure,
                                         unsigned long defState )
       : defaultProc( windowProcedure )
       , state( defState )
       , layoutChange( 0 )
       , helpId( 0 )
       , ulExtGUIStyle( 0 )
       , handlerList( 0 )
       , minimumSize( -1, -1 )
       , fUpdateEnabled( true )
#ifdef IC_PMWIN
       , fCachedMinimumSize( 0, 0 )
#endif
       , sizClChar( 0, 0 )
       , _observerData( 0 )
       , fHandlerRemovePending( false )
       , fSetMinSizeInProgress( false )
       , fAccelHandle( 0 )
#ifdef IC_MOTIFWIN
       , fhwndOwner( 0 )
       , oldHeight(0)
       , oldWidth(0)
       , clrforeground( IColor::kWindowText )
       , clrbackground( IColor::kWindowBgnd )
       , fPrevForegroundColor( IColor::kWindowText )
       , fPrevBackgroundColor( IColor::kWindowBgnd )
#endif
#ifdef IC_WIN
       , fFont( 0 )
       , lastButtonTime( 0 )
       , lastButtonEventHandle( 0 )
       , lastButtonEventId( 0 )
       , lastButtonPos( IPoint( 0, 0 ) )
       , eat_BN_CLICKED( false )
       , pointLButtonDown( 0, 0 )
       , bCheckingDragStart( false )
#endif
#ifdef IC_MOTIF
       , windowGrabbed( false )
       , windowSize( ISize(0, 0) )
       , windowPosition( IPoint(0,0) )
       , oldX(0)
       , oldY(0)
       , motifState( 0 )               // Motif only data members
       , gc(0)
       , gcUseCount(0)
#endif   // IC_MOTIF
#ifdef IC_PMWIN
       , fSomeMouseMovesCount( 1 )
       , fAllMouseMovesCount( 1 )
       , fMouseEntersLeavesCount( 1 )
#else
       , fSomeMouseMovesCount( 0 )
       , fAllMouseMovesCount( 0 )
       , fMouseEntersLeavesCount( 0 )
#endif
       , fMousePtr( 0 )
#ifdef IC_PMWIN
       , fMousePointerHandler( 0 )
#endif
       , fWindowDirection( IWindowPrivateData::kDefaultDirection )
{
#ifdef IC_MOTIF
  motifState |= IWindowPrivateData::neverBeenShown;
#endif
}

/*------------------------------------------------------------------------------
| IWindowPrivateData::~IWindowPrivateData                                      |
------------------------------------------------------------------------------*/
IWindowPrivateData::~IWindowPrivateData ( )
{
  if ( this->fAccelHandle )
  {  // Accelerator table cleanup.
     // Ignore any errors, since the application could have already
     // destroyed this using system APIs as part of its window cleanup.
     IAcceleratorTable::destroyHandle( this->fAccelHandle );
  }
#ifdef IC_WIN
  if ( fFont )
  {
     delete fFont;
  }
#endif
#ifdef IC_PMWIN
  if ( fMousePointerHandler )
  {
     delete fMousePointerHandler;
  }
#endif
}

/*------------------------------------------------------------------------------
| IWindowPrivateData::checkForPrimaryWindow                                    |
------------------------------------------------------------------------------*/
void IWindow::checkForPrimaryWindow( const IWindowHandle& hwndParent,
                                     const IWindowHandle& hwndOwner )
{
  bool wasBound = this->isBoundToMessageQueue();

  if ( (hwndOwner == 0 || hwndOwner == IWindow::desktopWindow()->handle() ) &&
       (hwndParent == 0 || hwndParent == IWindow::desktopWindow()->handle()) )
     pWindowData->state |= IWindow::primaryWindow;
  else
     pWindowData->state &= (unsigned long)~IWindow::primaryWindow;

  bool isBound = this->isBoundToMessageQueue();
  if (wasBound != isBound )
     {
     // A state change occurred in the window's bound state.  Adjust counter.
     #ifdef IC_PMWIN
       unsigned long
         threadId,
         processId;
       IQUERYWINDOWPROCESS( this->handle(), &processId, &threadId );
     #endif
     #ifdef IC_MOTIF
       IThreadId
          threadId = IThread::current().id();
     #endif
     IWindowThreadData* data = IWindowThreadData::dataForThread( threadId );
     if (data)
        {
        if ( isBound )
           data->addBoundWindow();
        else
           data->removeBoundWindow();
        }
     }
}

/*------------------------------------------------------------------------------
| ITheDesktopWindow::handle                                                    |
------------------------------------------------------------------------------*/
IWindowHandle ITheDesktopWindow::handle ( ) const
{
  return this->hwnd;
}

/*------------------------------------------------------------------------------
| ITheDesktopWindow::~ITheDesktopWindow                                        |
------------------------------------------------------------------------------*/
ITheDesktopWindow::~ITheDesktopWindow ( )
{ }


/*------------------------------------------------------------------------------
|ITheObjectWindow::ITheObjectWindow                                            |
------------------------------------------------------------------------------*/
ITheObjectWindow::ITheObjectWindow()
       : hwnd(0)  // handle is initialized on demand
{ }

/*------------------------------------------------------------------------------
|ITheObjectWindow::~ITheObjectWindow                                           |
------------------------------------------------------------------------------*/
ITheObjectWindow::~ITheObjectWindow ( )
{ }

/*------------------------------------------------------------------------------
|ITheObjectWindow::handle                                                      |
------------------------------------------------------------------------------*/
IWindowHandle ITheObjectWindow::handle ( ) const
{
  // In PM and Motif, the object window handle is initialized on demand
  // so that the GUI does not get automatically initialized at load time.
  // In Windows, an ITheObjectWindowHandle object is created at DLL load time
  // to insure its persistence across threads.  GUI initialization is not
  // an issue in Windows because it is automatic.
#ifdef IC_PM
  if ( this->hwnd == 0 )
  {
     ITheObjectWindow* localthis = (ITheObjectWindow*)this;
     localthis->hwnd = WinQueryObjectWindow(desktopWindow()->handle());
  }
#endif
#ifdef IC_WIN
  if ( this->hwnd == 0 )
  {
     ITheObjectWindow* localthis = (ITheObjectWindow*)this;
     localthis->hwnd = gStatics.fObjectWindowHandle.handle();
  }
#endif
#ifdef IC_MOTIF
  if ( this->hwnd == 0 )
  {
     ITheObjectWindow* localthis = (ITheObjectWindow*)this;

     IArgList args;
     args.setResource( XmNx, 0 );
     args.setResource( XmNy, 0 );
     args.setResource( XmNheight, 1 );
     args.setResource( XmNwidth, 1 );
     args.setResource( XmNmarginHeight, 0 );
     args.setResource( XmNmarginWidth, 0 );
     args.setResource( XmNnoResize, true );
     args.setResource( XmNresizePolicy, XmRESIZE_NONE );
     args.setResource( XmNallowOverlap, True );
     args.setResource( XmNmappedWhenManaged, False );
     localthis->hwnd = XtCreateManagedWidget(
                           "ITheObjectWindow",
                           xmBulletinBoardWidgetClass,
                           IWindow::desktopWindow()->handle(),
                           (ArgList)args.argList(),
                           args.numberOfArgs());
  }
#endif
  return this->hwnd;
}


/*------------------------------------------------------------------------------
| IWindow::handle                                                              |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::handle ( ) const
{
#ifndef IC_MOTIF
  if ( IPerformanceSettings::isSettingEnabled(
          IPerformanceSettings::validateWindowHandle) )
  {
     // Throw exception if window handle is not valid window.
     if ( !isValid() )
     {
        ITHROWLIBRARYERROR( IC_INVALIDHANDLE,
                            IBaseErrorInfo::invalidRequest,
                            IException::recoverable );
     }
  }
#endif  // ifndef IC_MOTIF

  return (IWindowHandle::Value)this->hwnd;
}

/*------------------------------------------------------------------------------
| IWindow::hide                                                                |
|                                                                              |
| Hide the window.                                                             |
------------------------------------------------------------------------------*/
IWindow& IWindow::hide ( )
{
  return show( false );
}

/*------------------------------------------------------------------------------
| IWindow::disableUpdate                                                       |
|                                                                              |
| Cause updates to the window not to be painted to the display.                |
------------------------------------------------------------------------------*/
IWindow& IWindow::disableUpdate ( )
{
  return enableUpdate( false );
}

/*------------------------------------------------------------------------------
| IWindow::isUpdateEnabled                                                      |
|                                                                              |
| returns whether updates to the window are enabled.          .                |
------------------------------------------------------------------------------*/
bool IWindow::isUpdateEnabled ( ) const
{
  bool fUpdateEnabled( pWindowData->fUpdateEnabled);
#ifdef IC_DEVELOP
  IString strTrace(getenv("ICLUI_USENEWUPDATE"));
  if (strTrace.length())
  {
    strTrace.strip().upperCase();
    if (strTrace == "FALSE")
    {
      ITRACE_DEVELOP("use new update false, isUpdateEnable will return true");
      fUpdateEnabled = true;
    }
  }
#endif
  return fUpdateEnabled;
}

/*------------------------------------------------------------------------------
| IWindow::disable                                                             |
|                                                                              |
| Disable the object - do not accept input.                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::disable ( )
{
  return enable( false );
}

/*------------------------------------------------------------------------------
| IWindow::setAutoDeleteObject                                                 |
|                                                                              |
| Set flag to delete the object (this pointer) when the                        |
| window handle is destroyed.                                                  |
------------------------------------------------------------------------------*/
IWindow& IWindow::setAutoDeleteObject ( bool autoDelete )
{
  if ( autoDelete )
     pWindowData->state |= autoDeleteObject;
  else
     pWindowData->state &= (unsigned long)(~autoDeleteObject);
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setAutoDestroyWindow                                                |
|                                                                              |
| Set flag to destroy the window handle when the object                        |
| is destructed.                                                               |
------------------------------------------------------------------------------*/
IWindow& IWindow::setAutoDestroyWindow ( bool autoDestroy )
{
   if ( autoDestroy )
      pWindowData->state |= autoDestroyWindow;
   else
      pWindowData->state &= (unsigned long)(~autoDestroyWindow);
   return *this;
}

/*------------------------------------------------------------------------------
| IWindow::startHandlingEventsFor                                                  |
|                                                                              |
| This function subclasses the PM window procedure to                          |
| begin dispatch events.                                                       |
------------------------------------------------------------------------------*/
IWindow& IWindow::startHandlingEventsFor ( unsigned long identifier,
                                           IWindow*      parent )
{
  // Assertions on input parms.
  IASSERTPARM( parent != 0 );

  IWindowHandle wndh =
            IWindow::handleWithParent( identifier, parent->handle() );
  if ( wndh == 0 )
     ITHROWLIBRARYERROR( IC_NO_WINDOW_WITHID,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
#ifdef IC_WIN
  // Set up the parent dialog as owner to simulate PM message routing.
  if ( ISTYLEOF( wndh ) & WS_CHILD )
  {
     // Owner is simulated.
     pWindowData->fhwndOwner = parent->handle();
  }
#endif

  return startHandlingEventsFor( wndh );
}

/*------------------------------------------------------------------------------
| IWindow::cleanUpHandler                                                      |
|                                                                              |
| Remove the argument handler from any windows it still might be trying        |
| to handle events for.                                                        |
|                                                                              |
| Currently, just searches the current thread's window list.  If the handler   |
| is handling events for windows on other threads, then log a "warning."       |
| At some point, we should try to remove this handler from windows on          |
| other threads.                                                               |
|                                                                              |
| This code presumes the window list will not be twiddled with by code on      |
| another thread.                                                              |
------------------------------------------------------------------------------*/
void IWindow::cleanUpHandler ( IHandler &handler )
{
  IMODTRACE_DEVELOP( "IWindow::cleanUpHandler" );
  IWindowList
   *pwinlist = IWindowList::current();
  if ( pwinlist !=0 && pwinlist->numberOfElements() != 0 )
  {
     IWindowList::Cursor cursor( *pwinlist );
     for ( cursor.setToFirst();
           cursor.isValid() && handler.numWindows;
           cursor.setToNext() )
        pwinlist->elementAt( cursor ) -> removeHandler( &handler );
     if ( handler.numWindows )
     {
        ITRACE_DEVELOP( "Active Handler Destroyed!" );
     }
  }
}

/*------------------------------------------------------------------------------
| IWindow::windowWithOwner                                                     |
|                                                                              |
| Search the internal collection for the iwindow with                          |
| the passed id and owner.                                                     |
------------------------------------------------------------------------------*/
IWindow* IWindow::windowWithOwner ( unsigned long identifier,
                                    const IWindow* owner,
                                    bool allThreads )
{
  IWindow* pwin;
  IWindowList* pwinlist;

  if ( allThreads )
  {
     // Lock resources for cross thread search.
     IGUIResourceLock lock( IWindowStaticData::libraryKey() );
     IWindowThreadDataList* list = IWindowThreadData::list();
     if (list)
     {
        IWindowThreadDataList::Cursor cursor( *list );
        for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
        {
           pwinlist = cursor.element()->windowList();
           IWindowList::Cursor wlcursor( *pwinlist );
           for ( wlcursor.setToFirst(); wlcursor.isValid(); wlcursor.setToNext() )
           {
              pwin = pwinlist->elementAt( wlcursor );

              // Ignore entries without a PM window.  Calling
              // IThread::stopProcessingMsgs, or falling out of
              // IThread::processMsgs with an exception, can leave
              // invalid entries for a short time.
              if ( pwin->isValid()  &&
                   pwin->id() == identifier  &&
                   pwin->owner() == owner )
              {
                 return pwin;
              }
           } /* endfor */
        }
     }
  }
  else
  {
     pwinlist = IWindowList::current();
     IWindowList::Cursor cursor( *pwinlist );
     unsigned long windowsRemaining;
     do
     {  // Repeat if not able to iterate through all windows
        // (cursor may have been invalidated by add/remove
        // done from another thread).
        windowsRemaining = pwinlist->numberOfElements();
        for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
        {
           pwin = pwinlist->elementAt( cursor );
           if ( pwin->isValid()  &&
                pwin->id() == identifier  &&
                pwin->owner() == owner )
           {
              return pwin;
           }
           windowsRemaining--;
        } /* endfor */
     } /* enddo */
     while ( windowsRemaining );
  }  // end if/then/else

  return 0;
}

/*------------------------------------------------------------------------------
| IWindow::windowWithParent                                                    |
|                                                                              |
| Search the internal collection for the iwindow with                          |
| the passed id and parent.                                                    |
------------------------------------------------------------------------------*/
IWindow* IWindow::windowWithParent ( unsigned long identifier,
                                     const IWindow* parent,
                                     bool allThreads )
{
  IASSERTPARM( parent != 0 );
  return (windowWithHandle( IWINDOWFROMID( parent->handle(), identifier ),
                            allThreads ));
}

/*------------------------------------------------------------------------------
| IHandlerListObject::IHandlerListObject                                       |
------------------------------------------------------------------------------*/
IHandlerListObject::IHandlerListObject ( IHandler* handler )
  : fHandler ( handler ),
    fRemovePending ( false )
{ }

/*------------------------------------------------------------------------------
| IHandlerListObject::~IHandlerListObject                                      |
------------------------------------------------------------------------------*/
IHandlerListObject::~IHandlerListObject ( )
{ }

/*------------------------------------------------------------------------------
| IHandlerListObject::IHandlerListObject (copy constructor)                    |
------------------------------------------------------------------------------*/
IHandlerListObject::IHandlerListObject ( const IHandlerListObject& object )
  : fHandler ( object.fHandler ),
    fRemovePending ( object.fRemovePending )
{ }

/*------------------------------------------------------------------------------
| IHandlerListObject::operator=                                                |
------------------------------------------------------------------------------*/
IHandlerListObject& IHandlerListObject::operator= ( const IHandlerListObject& object )
{
  fHandler = object.fHandler;
  fRemovePending = object.fRemovePending;
  return *this;
}

/*------------------------------------------------------------------------------
| IHandlerList::IHandlerList                                                   |
------------------------------------------------------------------------------*/
IHandlerList::IHandlerList ( )
  : ISequence<IHandlerListObject>( 10 ),
    fcursor( *this ),
    totalHandlers( 0 ),
    maxHandlers( 0 )
{ }

/*------------------------------------------------------------------------------
| IHandlerList::~IHandlerList                                                  |
|                                                                              |
| IHandlerList Destructor - total statistics.                                  |
------------------------------------------------------------------------------*/
IHandlerList::~IHandlerList ( )
{
  IMODTRACE_DEVELOP( "handlerList::~HandlerList" );
  ITRACE_DEVELOP( IString( "Total Handlers added: " ) +
                  IString( this->totalHandlers ));
  ITRACE_DEVELOP( IString( "Max Handlers: " ) + IString( this->maxHandlers ));
}

/*------------------------------------------------------------------------------
| IWindow::isLayoutDistorted                                                   |
------------------------------------------------------------------------------*/
bool IWindow::isLayoutDistorted( unsigned long layoutAttribute ) const
{
   if ( layoutAttribute & pWindowData->layoutChange )
      return true;
   else
      return false;
}

/*------------------------------------------------------------------------------
| IWindow::calcMinimumSize                                                     |
|                                                                              |
| Virtual function returning the minimun allowable size of                     |
| the window, used by canvas layout routined. Default                          |
| operation returns the minimun size set by users. Typically,                  |
| this should calculate the minimun size based on text and                     |
| current font setting.                                                        |
------------------------------------------------------------------------------*/
ISize IWindow::calcMinimumSize ( ) const
{
  IMODTRACE_DEVELOP( "Ctl::calcMinimumSize" );
  ITRACE_DEVELOP( "Default minimize size WIDTH=100 and HEIGHT=100 is used." );

  return ISize( 100,100 );
}

/*------------------------------------------------------------------------------
| IWindow::layoutAdjustment                                                    |
|                                                                              |
| Virtual function returning an IRectangle used to adjust the position         |
| and size of a window after layout.                                           |
------------------------------------------------------------------------------*/
IRectangle IWindow::layoutAdjustment ( ) const
{
  return IRectangle();
}

/*------------------------------------------------------------------------------
| IWindow::visibleRectangle                                                    |
|                                                                              |
| Virtual function returning an IRectangle representing the visible (painted)  |
| area of the control.                                                         |
------------------------------------------------------------------------------*/
IRectangle IWindow::visibleRectangle ( ) const
{
  return rect();
}

/*------------------------------------------------------------------------------
| IWindow::ExceptionFn::~ExceptionFn                                           |
|                                                                              |
| Keep the compiler from generating destructors.                               |
------------------------------------------------------------------------------*/
IWindow::ExceptionFn::~ExceptionFn ( )
{ }

/*------------------------------------------------------------------------------
| IWindow::setExceptionFunction                                                |
------------------------------------------------------------------------------*/
IWindow::ExceptionFn*
  IWindow::setExceptionFunction ( IWindow::ExceptionFn* pexcFn )
{
  IWindow::ExceptionFn* pcurrent = IWindowPrivateData::exceptionFunction;
  IWindowPrivateData::exceptionFunction = pexcFn;
  return pcurrent;
}

/*------------------------------------------------------------------------------
| IWindow::exceptionFunction                                                   |
------------------------------------------------------------------------------*/
IWindow::ExceptionFn* IWindow::exceptionFunction ( )
{
   return IWindowPrivateData::exceptionFunction;
}

/*------------------------------------------------------------------------------
| IWindow::handleException                                                     |
------------------------------------------------------------------------------*/
bool IWindow::handleException ( IException& exc,
                                          IEvent&     event )
{
  IWindow::ExceptionFn* pexcFn = exceptionFunction();
  if ( pexcFn )
     return pexcFn->handleException( exc, event );
  else
     return false;
}

/*------------------------------------------------------------------------------
| IWindow::childAt                                                             |
|                                                                              |
| Return the windowhandle from the argument cursor.                            |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::childAt ( const ChildCursor &cursor ) const
{
  return cursor.hwnd;
}

/*------------------------------------------------------------------------------
| IWindow::childWindowAt                                                             |
|                                                                              |
| Return the windowhandle from the argument cursor.                            |
------------------------------------------------------------------------------*/
IWindow* IWindow::childWindowAt ( const ChildCursor &cursor ) const
{
  return IWindow::windowWithHandle( cursor.hwnd );
}

/*------------------------------------------------------------------------------
| IWindow::setDefaultOrdering                                                  |
|                                                                              |
| Static function that sets the default placement of new windows.              |
------------------------------------------------------------------------------*/
void IWindow::setDefaultOrdering ( IWindow::SiblingOrder order )
{
  IWindowPrivateData::siblingCreateOrder = order;
}

/*------------------------------------------------------------------------------
| IWindow::defaultOrdering                                                     |
|                                                                              |
| Static function that returns the default placement of new windows.           |
------------------------------------------------------------------------------*/
IWindow::SiblingOrder IWindow::defaultOrdering ( )
{
  return IWindowPrivateData::siblingCreateOrder;
}

/*------------------------------------------------------------------------------
| IWindow::showSourceEmphasis                                                  |
|                                                                              |
| Overloaded to show source emphasis.                                          |
------------------------------------------------------------------------------*/
IWindow& IWindow::showSourceEmphasis ( bool /*unused*/ )
{
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::hideSourceEmphasis                                                  |
|                                                                              |
| Overloaded to remove source emphasis.                                        |
------------------------------------------------------------------------------*/
IWindow& IWindow::hideSourceEmphasis ( )
{
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::postEvent                                                           |
|                                                                              |
| Construct an IEvent from the arguments and post it to this window.           |
------------------------------------------------------------------------------*/
const IWindow& IWindow::postEvent ( unsigned long         eventId,
                                    const IEventParameter1 &parm1,
                                    const IEventParameter2 &parm2 ) const
{
  handle().postEvent( eventId, parm1, parm2 );
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::postEvent                                                           |
|                                                                              |
| Construct an IEvent from the arguments and post it to this window.           |
------------------------------------------------------------------------------*/
const IWindow& IWindow::postEvent ( const IEvent& event) const
{
  handle().postEvent( event.eventId(),
                      event.parameter1(),
                      event.parameter2() );
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::sendEvent                                                           |
|                                                                              |
| Construct an IEvent from the arguments and send it to this window.  Return   |
| the event result.                                                            |
------------------------------------------------------------------------------*/
IEventResult IWindow::sendEvent ( const IEvent& event ) const
{
  return handle().sendEvent( event.eventId(),
                             event.parameter1(),
                             event.parameter2() );
}

/*------------------------------------------------------------------------------
| IWindow::sendEvent                                                           |
|                                                                              |
| Construct an IEvent from the arguments and send it to this window.  Return   |
| the event result.                                                            |
------------------------------------------------------------------------------*/
IEventResult IWindow::sendEvent ( unsigned long   eventId,
                                  const IEventParameter1 &parm1,
                                  const IEventParameter2 &parm2 ) const
{
  return handle().sendEvent( eventId, parm1, parm2 );
}

/*------------------------------------------------------------------------------
| IWindow::setDefaultProcedure                                                 |
|                                                                              |
| Simply set the "default procedure" data member.  Returns the previous        |
| value.                                                                       |
------------------------------------------------------------------------------*/
IWinProc* IWindow::setDefaultProcedure ( IWinProc *newDefaultProc )
{
  IWinProc
   *oldDefaultProc = pWindowData->defaultProc;
  pWindowData->defaultProc = newDefaultProc;
  return oldDefaultProc;
}

/*------------------------------------------------------------------------------
| IWindow::isAutoDeleteObject                                                  |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow ::isAutoDeleteObject ( ) const
{
  return ((pWindowData->state & IWindow::autoDeleteObject) ? true : false);
}

/*------------------------------------------------------------------------------
| IWindow::isAutoDestroyWindow                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::isAutoDestroyWindow ( ) const
{
  return ((pWindowData->state & IWindow::autoDestroyWindow) ? true : false);
}

/*------------------------------------------------------------------------------
| IWindow::deleteIsInProcess                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::deleteIsInProcess ( ) const
{
  return ((pWindowData->state & IWindow::deleteInProcess) ? true : false);
}

/*------------------------------------------------------------------------------
| IWindow::isPrimaryWindow                                                     |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::isPrimaryWindow ( ) const
{
  return ((pWindowData->state & IWindow::primaryWindow) ? true : false);
}

/*------------------------------------------------------------------------------
| IWindow::setDeleteInProcess                                                  |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::setDeleteInProcess ( )
{
  pWindowData->state |= IWindow::deleteInProcess;
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::resetMinimumSize                                                    |
|                                                                              |
|  Reset the minimum size to the default.                                      |
------------------------------------------------------------------------------*/
IWindow& IWindow::resetMinimumSize ( )
{
  pWindowData->minimumSize = ISize( -1, -1 );
  this->saveMinimumSize( ISize( 0, 0 ) );
  setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setMinimumSize                                                      |
|                                                                              |
|  Set the minimum allowable size of the window. In the                        |
|  library this only apply to window on a canvas.                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::setMinimumSize ( const ISize& sizMin )
{
  pWindowData->fSetMinSizeInProgress = true;
  pWindowData->minimumSize = sizMin;
  setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
  pWindowData->fSetMinSizeInProgress = false;
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::minimumSize                                                         |
|                                                                              |
|  Returns the minimum allowable size set by the user's class                  |
------------------------------------------------------------------------------*/
ISize IWindow::minimumSize ( bool useCalculatedSize ) const
{
  ISize minSize;

  if ( useCalculatedSize  ||
       ( pWindowData->minimumSize.width() == -1  &&
         pWindowData->minimumSize.height() == -1 ) )
  {           // Get the value calculated by the window.
     if ( this->isLayoutDistorted( IWindow::minimumSizeChanged
                                    | IWindow::layoutChanged )  ||
          this->savedMinimumSize() == ISize( 0, 0 )  ||
          !( this->isMinimumSizeCachingEnabled() ) )
     {        // Need new calculated value.
        // No previously cached calculated value, the cached value is
        // no longer valid, or don't use cached values for this window.
        minSize = this->calcMinimumSize();
        (*(IWindow*)this)
         .saveMinimumSize( minSize )
         .setLayoutDistorted( 0, IWindow::minimumSizeChanged );
     }
     else
     {  // The last calculated value is still valid, so return it.
        minSize = this->savedMinimumSize();
     }
  }
  else
  {
     // Return the value stored by the application via
     // IWindow::setMInimumSize.
     minSize = pWindowData->minimumSize;
     (*(IWindow*)this)
      .setLayoutDistorted( 0, IWindow::minimumSizeChanged );
  }

  return minSize;
}

/*------------------------------------------------------------------------------
| IWindow::enableMinimumSizeCaching                                            |
|                                                                              |
|  Enable or disable minimum size caching as an optimization.  Enabling        |
|  caching causes IWindow::minimumSize to return the next calculated size      |
|  until the IWindow::minimumSizeChanged flag gets turned on.  Disabling       |
|  caching causes IWindow::minimumSize to call calcMinimumSize.                |
------------------------------------------------------------------------------*/
IWindow& IWindow::enableMinimumSizeCaching ( bool enableCaching )
{
  if ( enableCaching )
  {
     pWindowData->state |= IWindow::cacheMinimumSize;
  }
  else
  {
     pWindowData->state &= (unsigned long)(~IWindow::cacheMinimumSize);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::disableMinimumSizeCaching                                           |
|                                                                              |
|  Disable minimum size caching as an optimization, so that                    |
|  IWindow::minimumSize will not bypass calcMinimumSize.                       |
------------------------------------------------------------------------------*/
IWindow& IWindow::disableMinimumSizeCaching ( )
{
  return this->enableMinimumSizeCaching( false );
}

/*------------------------------------------------------------------------------
| IWindow::isMinimumSizeCachingEnabled                                         |
|                                                                              |
|  Disable minimum size caching as an optimization, so that                    |
|  IWindow::minimumSize will not bypass calcMinimumSize.                       |
------------------------------------------------------------------------------*/
bool IWindow::isMinimumSizeCachingEnabled ( ) const
{
  return ( pWindowData->state & IWindow::cacheMinimumSize ) ? true : false;
}

/*------------------------------------------------------------------------------
| IWindow::saveMinimumSize                                                     |
|                                                                              |
|  Save the minimum size as an optimization.  IWindow::minimumSize             |
|  will return this value until the IWindow::minimumSizeChanged                |
|  flag gets turned on.                                                        |
------------------------------------------------------------------------------*/
IWindow& IWindow::saveMinimumSize ( const ISize& size )
{
  pWindowData->fCachedMinimumSize = size;
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::savedMinimumSize                                                    |
|                                                                              |
|  Returns the minimum size saved via IWindow::saveMinimumSize.                |
|  This function is used as an optimization step to avoid having               |
|  to constantly call calcMinimumSize.                                         |
------------------------------------------------------------------------------*/
ISize IWindow::savedMinimumSize ( ) const
{
  return pWindowData->fCachedMinimumSize;
}

/*------------------------------------------------------------------------------
| IWindow::helpId                                                              |
|                                                                              |
| Get the context help topic identifier.                                       |
------------------------------------------------------------------------------*/
unsigned long IWindow::helpId() const
{
  unsigned long ulId = pWindowData->helpId;

#if defined(IC_WIN)
  ulId = GetWindowContextHelpId( this->handle() );
#endif

  return ulId;
}

/*------------------------------------------------------------------------------
| IWindow::setHelpId                                                           |
|                                                                              |
| Set the context help topic identifier.                                       |
------------------------------------------------------------------------------*/
IWindow& IWindow::setHelpId ( unsigned long helpTopicId )
{
#if defined(IC_WIN)
  SetWindowContextHelpId( this->handle(), helpTopicId );
#endif
  pWindowData->helpId = helpTopicId;

  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::defaultProcedure                                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::defaultProcedure ( IEvent& evt )
{
#ifdef IC_WIN
  IEventResult er( (unsigned long)
 #ifdef IC_WIN_STRICT
                   CallWindowProc( (WNDPROC) pWindowData->defaultProc,
 #else
                   CallWindowProc( (FARPROC) pWindowData->defaultProc,
 #endif
                                   evt.handle(),
                                   (unsigned int) evt.eventId(),
                                   evt.parameter1(),
                                   evt.parameter2() ) );
#else
  IEventResult er( pWindowData->defaultProc(
                                    evt.handle().asUnsigned(),
                                    evt.eventId(),
                                    evt.parameter1(),
                                    evt.parameter2()) );
#endif
  evt.setResult( er );
  return *this;
}


#pragma info(none)
/*------------------------------------------------------------------------------
| IWindowList::IWindowList                                                     |
------------------------------------------------------------------------------*/
IWindowList::IWindowList ( )
  : IVPtrKeySetAsHshTable<IWindow*, unsigned long>( 300 )
{ }
#pragma info(restore)

/*------------------------------------------------------------------------------
| IWindowList::~IWindowList                                                    |
|                                                                              |
| IWindowList Destructor - total statistics.                                   |
------------------------------------------------------------------------------*/
IWindowList::~IWindowList ( )
{
  IMODTRACE_DEVELOP( "IWindowList::~IWindowList" );

  this->removeAll();

#ifdef IC_DEVELOP
  ITRACE_DEVELOP( IString( "Total Windows added: " ) +
                  IString( this->totalWindowsAdded ));
  ITRACE_DEVELOP( IString( "Total Events: " ) +
                  IString( this->totalEvents ));
  ITRACE_DEVELOP( IString( "Successful Cache Requests: " ) +
                  IString( this->cacheUsedCount ));
#endif
}

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
/*------------------------------------------------------------------------------
| IWindow::enableFastWindowWithHandle                                          |
------------------------------------------------------------------------------*/
void IWindow::enableFastWindowWithHandle   ( bool enable )
{
#ifdef IC_PMWIN
   IWindowPrivateData::fgFastWindowProcEnabled = enable;
#endif
}
#endif //IC_OBSOLETE

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
/*------------------------------------------------------------------------------
| IWindow::isFastWindowWithHandleEnabled                                       |
------------------------------------------------------------------------------*/
bool IWindow::isFastWindowWithHandleEnabled( )
{
  return IWindowPrivateData::fgFastWindowProcEnabled;
}
#endif //IC_OBSOLETE

/*------------------------------------------------------------------------------
| IWindowThreadObserver::IWindowThreadObserver                                 |
------------------------------------------------------------------------------*/
IWindowThreadObserver::IWindowThreadObserver()
{ }

/*------------------------------------------------------------------------------
| IWindowThreadObserver::~IWindowThreadObserver                                |
------------------------------------------------------------------------------*/
IWindowThreadObserver::~IWindowThreadObserver()
{ }

/*------------------------------------------------------------------------------
| IWindowThreadObserver::dispatchNotificationEvent                             |
------------------------------------------------------------------------------*/
IWindowThreadObserver& IWindowThreadObserver::dispatchNotificationEvent
             ( const INotificationEvent& event )
{
   if ( event.notificationId() == INotifier::deleteId )
   {
      IWindowThreadData::threadTerminated( IThread::currentId() );
   }
   return *this;
}


/*------------------------------------------------------------------------------
| IWindowThreadData::IWindowThreadData                                         |
------------------------------------------------------------------------------*/
IWindowThreadData::IWindowThreadData( )
  : fwindowList         ( new IWindowList ),
#ifdef IC_WIN
    fdialogControls     ( new IDialogControls ),
#endif
    fKey                ( INonGUIThread::currentId().asUnsigned() ),
    fboundCount         ( 0 )
{
  IMODTRACE_DEVELOP("IWindowThreadData::IWindowThreadData");
}

/*------------------------------------------------------------------------------
| IWindowThreadData::~IWindowThreadData                                        |
------------------------------------------------------------------------------*/
IWindowThreadData::~IWindowThreadData( )
{
  IMODTRACE_DEVELOP("IWindowThreadData::~IWindowThreadData");
  delete fwindowList;
#ifdef IC_WIN
  delete fdialogControls;
#endif
}

/*------------------------------------------------------------------------------
| IWindowThreadData::dataForThread                                             |
------------------------------------------------------------------------------*/
IWindowThreadData* IWindowThreadData::dataForThread(const IThreadId& threadId)
{
   if ( !IWindowThreadData::list() )
   {
      // Lock resources for update.
      IGUIResourceLock lock( IWindowStaticData::libraryKey() );
      // If still 0 then allocate a list.  Allocate the observer also because
      // we are going to need it.
      if ( !IWindowStaticData::fgThreadList )
         IWindowStaticData::fgThreadList = new IWindowThreadDataList;
      if ( !IWindowStaticData::fgObserver )
         IWindowStaticData::fgObserver = new IWindowThreadObserver;
   }
   IWindowThreadDataList::Cursor cursor( *IWindowThreadData::list() );
   if ( IWindowThreadData::list()->locateElementWithKey(
           threadId.asUnsigned(), cursor ) )
   {
      return IWindowThreadData::list()->elementAt( cursor );
   }
   else
   {
      // If it is the current thread, create the element for the thread.
      unsigned long key = threadId.asUnsigned();
      if ( key == IThread::currentId().asUnsigned() )
      {
         // Lock resources for update.
         IGUIResourceLock lock( IWindowStaticData::libraryKey() );
         IWindowThreadData* newElement = new IWindowThreadData;
         IWindowThreadData::list()->add( newElement );
         // Register to be notified when thread goes away
         IStartedThread::notifyOnCurrentThreadEnd(
            IWindowStaticData::fgObserver );
         return newElement;
      }
   }
   return 0;
}

/*------------------------------------------------------------------------------
| IWindowThreadData::current                                                   |
------------------------------------------------------------------------------*/
const IWindowThreadData* IWindowThreadData::current( )
{
   return IWindowThreadData::dataForThread( IThread::currentId() );
}

/*------------------------------------------------------------------------------
| IWindowThreadData::threadTerminated                                          |
------------------------------------------------------------------------------*/
void IWindowThreadData::threadTerminated( const IThreadId& threadId )
{
   IMODTRACE_DEVELOP("IWindowThreadData::threadTerminated");
   // Lock resources for update.
   IGUIResourceLock lock( IWindowStaticData::libraryKey() );
   if ( IWindowThreadData::list() )
   {
      IWindowThreadDataList::Cursor cursor( *IWindowThreadData::list() );
      if (IWindowThreadData::list()->locateElementWithKey(
                                     threadId.asUnsigned(), cursor) )
      {
         IWindowThreadData* element =
            IWindowThreadData::list()->elementAt( cursor );
         IWindowThreadData::list()->removeAt( cursor );
         delete element;
         // Removed an element.  If it was the last one delete the collection.
         if ( IWindowThreadData::list()->isEmpty() )
         {
            ITRACE_DEVELOP("IWindowThreadData::threadTerminated - removing fgThreadList");
            delete IWindowStaticData::fgThreadList;
            IWindowStaticData::fgThreadList = 0;
         }
      }
   }
}

/*------------------------------------------------------------------------------
| IWindowThreadDataList::IWindowThreadDataList                                 |
------------------------------------------------------------------------------*/
IWindowThreadDataList::IWindowThreadDataList( )
  : IVPtrKeySetAsHshTable<IWindowThreadData*, unsigned long>( 5 )
{ }

/*------------------------------------------------------------------------------
| IWindowThreadDataList::~IWindowThreadDataList                                |
------------------------------------------------------------------------------*/
IWindowThreadDataList::~IWindowThreadDataList( )
{ }

#ifdef IC_DEVELOP
/*------------------------------------------------------------------------------
| IWindowList::dump                                                            |
|                                                                              |
| Debug function to write the contents of the current windowlist.              |
------------------------------------------------------------------------------*/
void IWindowList::dump ( )
{
  IMODTRACE_DEVELOP( "IWindowList::dump" );
  IWindowList* pwinlist = IWindowList::current();
  IWindowList::Cursor cursor( *pwinlist );
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     IWindow* pwin = pwinlist->elementAt( cursor );
     if ( pwin->isValid() )
     {            // Entry represents a valid window.
        ITRACE_DEVELOP( IString( "pwin =")+
                        IString( (unsigned long)pwin ).d2x() +
                        IString( " id =") +
                        IString( pwin->id() ) +
                        IString( " handle =")+
                        IString( pwin->handle().asUnsigned() ).d2x());
     }
     else
     {            // Window has been destroyed.
        ITRACE_DEVELOP( IString( "pwin =") +
                        IString( (unsigned long)pwin ).d2x() +
                        IString( " handle =none" ));
     }
  }
  return;
}
#endif  //IC_DEVELOP


/*------------------------------------------------------------------------------
| IWindowData::IWindowData                                                     |
------------------------------------------------------------------------------*/
IWindowData::IWindowData( )
{ }

/*------------------------------------------------------------------------------
| IWindowData::~IWindowData                                                    |
------------------------------------------------------------------------------*/
IWindowData::~IWindowData( )
{ }


#pragma enum(4)
#pragma pack(push,4)
/*------------------------------------------------------------------------------
| IDataHandleElement class                                                     |
------------------------------------------------------------------------------*/
class IDataHandleElement {
public:
  IDataHandleElement ( const char* keyName, IWindow::DataHandle index )
    : dataKeyName( keyName )
    , elemArrayIndex( index )
  { }

 ~IDataHandleElement ( )
  { }

bool
  operator== ( const IDataHandleElement& key ) const;

IWindow::DataHandle
  dataHandle ( )
  { return elemArrayIndex; }

private:
IString
  dataKeyName;
IWindow::DataHandle
  elemArrayIndex;

friend IString const&
  key ( IDataHandleElement* const& pDataHandleElement );
}; // IDataHandleElement

/*------------------------------------------------------------------------------
| IDataHandleSet class                                                         |
------------------------------------------------------------------------------*/
class IDataHandleSet : public IKeySet <IDataHandleElement*, IString> {
public:
  IDataHandleSet ( )
    : IKeySet <IDataHandleElement*, IString>(10)
    , arrayIndex(0)
  { }

 ~IDataHandleSet ( );

IWindow::DataHandle
  arrayIndex;

private:
#if IC_EXTENDED_RTTI_ENABLED
  IDataHandleSet ( const IDataHandleSet& x )
  { };
#else
  IDataHandleSet ( const IDataHandleSet& x );  /* Copy Ctor */
#endif // IC_EXTENDED_RTTI_ENABLED
IDataHandleSet
 &operator=      ( const IDataHandleSet& );    /* assign op */
}; // IDataHandleSet

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IDelDataHandleElement Function                                               |
| for the purpose of deleting elements in set when removed                     |
------------------------------------------------------------------------------*/
static bool
  IDelDataHandleElement(IDataHandleElement* const& element, void* /*anything*/)
{
  delete (IDataHandleElement*)element;
  return true;
}

/*------------------------------------------------------------------------------
| key function for IDataHandleElement                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IString const& key(IDataHandleElement* const& pDataHandleElement)
{
  return pDataHandleElement->dataKeyName;
}



/*------------------------------------------------------------------------------
| IDataHandleSet::~IDataHandleSet                                              |
|                                                                              |
| Remove and delete all elements                                               |
|                                                                              |
------------------------------------------------------------------------------*/
IDataHandleSet::~IDataHandleSet()
{
  removeAll(&IDelDataHandleElement);
}

/*------------------------------------------------------------------------------
| IWindow::dataHandleWithKey                                                   |
|                                                                              |
| This thread-safe function increments a counter so each new request           |
| gets a new offset.  Duplicate requests return the same DataHandle.           |
------------------------------------------------------------------------------*/
IWindow::DataHandle IWindow::dataHandleWithKey(const char* dataKeyName)
{
  IGUIResourceLock resLock( IWindowStaticData::libraryKey() );

  if (!IWindowStaticData::fgDataHandleSet)
     IWindowStaticData::fgDataHandleSet = new IDataHandleSet;

  IDataHandleSet& handleSet( *IWindowStaticData::fgDataHandleSet );

  /**********************************************************************/
  /* If key is already in collection, return DataHandle associated with */
  /* key                                                                */
  /**********************************************************************/
  IString dataKeyNameString(dataKeyName);
  if (handleSet.containsElementWithKey( dataKeyNameString ) )
  {
    IDataHandleElement* dataHandleElement;
    dataHandleElement = handleSet.elementWithKey(
                        dataKeyNameString );
    return (dataHandleElement->dataHandle() );
  }
  /**********************************************************************/
  /* Else, key is not in collection.  Associate the next available      */
  /* DataHandle with the key, add the key to the collection, and        */
  /* return the new DataHandle                                          */
  /**********************************************************************/
  else
  {
    handleSet.arrayIndex++;

    IDataHandleElement* dataHandleElement =
      new IDataHandleElement( dataKeyName,
                              handleSet.arrayIndex );

    handleSet.add(dataHandleElement);
    return (handleSet.arrayIndex);
  }
}

/*------------------------------------------------------------------------------
| IWindow::adoptWindowData                                                     |
|                                                                              |
| This member checks to see if an array of IWindowData exists for the          |
| window and allocates it if it doesn't.  It also checks to see if the array   |
| is large enough to contain the index and reallocates it if necessary.        |
| It then stores the pointer in the array.  If an entry already exists at the  |
| handle location, it is deleted before the new window data is stored.         |
------------------------------------------------------------------------------*/
IWindow& IWindow::adoptWindowData(const DataHandle& typeToken,
                                  IWindowData* windowData)
{
  if (typeToken != 0)
  {
    if (!windowDataArray)
    {
      ulWindowDataArraySize = 10;  //Allocate in increments of 10
      windowDataArray = new IWindowData*[ulWindowDataArraySize];

      //Need to zero out array for check done for existing element.
      for (int i=0; i < ulWindowDataArraySize; i++)
        windowDataArray[i] = 0;
    }
    else
    {
      if (ulWindowDataArraySize < typeToken)
      {
        //Array is too small.  Re-allocate it to a larger size.
        IWindowData** newWindowDataArray =
                    new IWindowData*[ulWindowDataArraySize+10];

        //Copy data to new array.
        for (int j=0; j < ulWindowDataArraySize; j++)
          newWindowDataArray[j] = windowDataArray[j];

        //Need to zero out new slots for check of existing elements.
        for (int k=ulWindowDataArraySize; k<ulWindowDataArraySize+10; k++)
          newWindowDataArray[k] = 0;

        ulWindowDataArraySize += 10;

        delete [] windowDataArray;
        windowDataArray = newWindowDataArray;
      }
    }
    // If entry already exists at location, delete it first.
    if (windowDataArray[typeToken-1])
      delete windowDataArray[typeToken-1];

    // Place new windowData pointer into array.
    windowDataArray[typeToken-1] = windowData;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::addOrReplaceAttribute
|
| This member function adds or replaces the attribute to the key collection.
| The key is the attribute name.
------------------------------------------------------------------------------*/
IWindow& IWindow::addOrReplaceAttribute( const IAttributeName &name,
                                         const IAttribute& attribute )
{
  //If there is an existing attribute we are replacing,free the
  //storage associated with that entry before adding the new one.
  IAttribute *attr = ( IAttribute* ) attributeWithName( name );
  if ( attr ) delete attr;

  IWindowAttributeName attName(name);
  // A clone of the attribute is created here and stored in the IWindow key set.
  INamedAttribute newKey( attribute.clone(), attName );
  pWindowData->fAttributeKeySet.addOrReplaceElementWithKey( newKey );

#ifdef IC_PM
  IAttribute* pClone = attribute.clone();
  IInheritColorAttribute* pAttribute = dynamic_cast<IInheritColorAttribute*>(pClone);
  if( pAttribute )
  {
    // explicitly set colour so PM won't automatically set this window's colours based on its owner
    IColor foreColour( foregroundColor());
    setForegroundColor( foreColour );

    IColor backColour( backgroundColor());
    setBackgroundColor( backColour );
  }
  delete pClone;
#endif

  // Send a notification
  IWindow *thisWindow = (IWindow *) this;
  IString attString( name.getString() );
  this->notifyObservers( INotificationEvent(IWindow::attributeAddReplaceId,
                                 *thisWindow, true, &attString ));
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::removeAttribute
|
| This member function removes the attribute from the attribute collection.
| The key is the attribute name.
------------------------------------------------------------------------------*/
IWindow& IWindow::removeAttribute( const IAttributeName &name )
{
  IAttribute *attribute = (IAttribute *) attributeWithName( name );
  if ( pWindowData->fAttributeKeySet.removeElementWithKey(
                                                  IWindowAttributeName(name) ))
  {
    if (attribute)
    {
      IWindow *thisWindow = (IWindow *) this;
      IString attString( name.getString() );
      this->notifyObservers( INotificationEvent(IWindow::attributeRemoveId,
                                 *thisWindow, true, &attString ));
      // We cloned so we need to clean up.
      delete attribute;
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::removeAllAttributes
|
| This member function removes the attribute from the attribute collection.
| The key is the attribute name. We have to visit each member of the collection
| so we can clean up the memory from the clone of the attribute.
------------------------------------------------------------------------------*/
IWindow& IWindow::removeAllAttributes( )
{
  // See if there is faster way to implement this. The problem is that we are
  // trying to cut down on the number of times an attribute is cloned, so we
  // moved the problem to destruction.

  // Should this routine be protected for thread safety.
  IWindowAttributeKeySet::Cursor cursor(pWindowData->fAttributeKeySet);
  IWindow *thisWindow = (IWindow *) this;
  for( cursor.setToFirst();cursor.isValid();cursor.setToNext() )
  {
    IAttribute *attribute = (IAttribute *)cursor.element().attribute();
    if (attribute)
    {
      IString attName( cursor.element().attributeName().getString() );
      this->notifyObservers( INotificationEvent(IWindow::attributeRemoveId,
                                 *thisWindow, true, &attName ));
      // clean up our clone.
      delete attribute;
    }
  }

  pWindowData->fAttributeKeySet.removeAll( );
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::attributeWithName
|
| This member function looks up the attribute from the attribute collection.
| The key is the attribute name and a search capability allows searching up
| the parent chain for a particular attribute.
------------------------------------------------------------------------------*/
const IAttribute* IWindow::attributeWithName( const IAttributeName &name,
                                              ESearchType search ) const
{
  // convert the attribute name into a window attribute name.
  IWindowAttributeName theName( name );
  IWindow *theSearchWin = (IWindow *)this;

  do {
    // Be real nice to add reference counting to attributes for maintaining
    // the cloned attribute.
    IWindowAttributeKeySet::Cursor cursor(theSearchWin->pWindowData->fAttributeKeySet);
    if ( theSearchWin->pWindowData->fAttributeKeySet.locateElementWithKey(
           theName, cursor ) )
    {
      INamedAttribute namedAttribute(
        theSearchWin->pWindowData->fAttributeKeySet
          .elementAt( cursor ));

      return namedAttribute.attribute();
    }
    if (search == kWindowAndParents)
      theSearchWin = theSearchWin->parent();
    else
      break;

  } while ( theSearchWin != 0 );

  // if we did not find the attribute then return 0.
  return 0;
}

/*------------------------------------------------------------------------------
| IWindow::attributeNameAt                                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IAttributeName IWindow::attributeNameAt( const AttributeCursor& cursor ) const
{
  return cursor.attributeName();
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::AttributeCursor                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow::AttributeCursor::AttributeCursor( IWindow &win )
    : fData( new IWindowAttributeCursorData( &(win.pWindowData->fAttributeKeySet ) ))
{ }

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::~AttributeCursor                                    |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow::AttributeCursor::~AttributeCursor( )
{
  delete this->fData;
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::setToFirst                                         |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::AttributeCursor::setToFirst( )
{
  return this->fData->fCursor.setToFirst();
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::setToNext                                          |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::AttributeCursor::setToNext( )
{
  return this->fData->fCursor.setToNext();
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::isValid                                            |
|                                                                              |
------------------------------------------------------------------------------*/
bool IWindow::AttributeCursor::isValid( ) const
{
  return this->fData->fCursor.isValid();
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::invalidate                                         |
|                                                                              |
------------------------------------------------------------------------------*/
void IWindow::AttributeCursor::invalidate( )
{
  this->fData->fCursor.invalidate();
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::attributeName                                      |
|                                                                              |
------------------------------------------------------------------------------*/
IAttributeName IWindow::AttributeCursor::attributeName( ) const
{
  INamedAttribute elementKey = this->fData->fCursor.element();
  return elementKey.attributeName();
}

/*------------------------------------------------------------------------------
| IWindow::AttributeCursor::attributeName                                      |
|                                                                              |
------------------------------------------------------------------------------*/
const IAttribute* IWindow::AttributeCursor::attribute( ) const
{
  INamedAttribute elementKey = this->fData->fCursor.element();
  return elementKey.attribute();
}

/*----------------------------- INamedAttribute -----------------------------*/
INamedAttribute::INamedAttribute( const IAttribute *attribute,
                                  const IWindowAttributeName& name )
  : fAttributeName( name )
  , fAttribute( (IAttribute *)attribute )
{
  // Note that the attribute is not cloned. Do not delete it in the dtor.
}


INamedAttribute::INamedAttribute()
  : fAttributeName( )
  , fAttribute( 0 )
{
}

INamedAttribute::~INamedAttribute()
{
  // Do Not Delete the attribute!
  //delete this->fAttribute;
}

INamedAttribute::INamedAttribute( const INamedAttribute& key )
    : fAttributeName( key.fAttributeName )
    , fAttribute( key.fAttribute )
{ }

INamedAttribute &INamedAttribute::operator= ( const INamedAttribute& key )
{
  if (*this != key)
  {
    this->fAttributeName = key.fAttributeName;
    this->fAttribute     = key.fAttribute;
  }
  return *this;
}

bool INamedAttribute::operator== ( const INamedAttribute& nattr ) const
{
  if ( (this->fAttributeName == nattr.fAttributeName ) &&
       (this->fAttribute == nattr.fAttribute) )
  {
    return true;
  }
  else
  {
    return false;
  }
}

bool INamedAttribute::operator!= ( const INamedAttribute& key ) const
{
  return !(*this == key);
}

bool INamedAttribute::operator< ( const INamedAttribute& nattr ) const
{
  if ( this->fAttributeName < nattr.fAttributeName )
  {
    return true;
  }
  else
  {
    return false;
  }
}

const IAttribute* INamedAttribute::attribute() const
{
  return this->fAttribute;
}

const IAttributeName INamedAttribute::attributeName() const
{
  return this->fAttributeName;
}

/*-------------------------- IWindowAttributeName --------------------------*/
IWindowAttributeName INamedAttribute::uniqueKeyFor(const INamedAttribute& key )
{
  return key.fAttributeName;
}

/*------------------------------------------------------------------------------
| ::key                                                                        |
|                                                                              |
| Global function for key set of IAttributeKey objects.  Allow only one      |
| attribute table entry for a given attribute name.                            |
------------------------------------------------------------------------------*/
IWindowAttributeName const& key ( INamedAttribute const& item )
{
  return item.fAttributeName;
}
/*-----------------------------------------------------------------------------
| operator <
|
| The collections use the < operator for the element comparisons. So even
| though the key set is not sorted this operator still needs to be implemented.
| The collections use the following logic
|
| if ( e1 < e2) return -1;
| else if ( e2 < e1 ) return 1;
| else return 0;
|
-----------------------------------------------------------------------------*/
bool IWindowAttributeName::operator < (IWindowAttributeName const& k) const
{
  return ( IString(this->getString()) < IString( k.getString() ));
}

IWindowAttributeName::IWindowAttributeName( const IAttributeName& name )
    : IAttributeName( name )
{ }

IWindowAttributeName::IWindowAttributeName( )
    : IAttributeName( "" )
{ }

IWindowAttributeName::IWindowAttributeName( const IWindowAttributeName& other )
    : IAttributeName( other )
{ }

IWindowAttributeName::~IWindowAttributeName()
{ }

/*------------------------------------------------------------------------------
| IWindow::bindMessageQueue                                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::bindMessageQueue( bool bindToMessageQueue )
{
  bool wasBound( this->isBoundToMessageQueue() );

  if (bindToMessageQueue )
  {
    pWindowData->state |= IWindow::boundToMessageQueue;
    if ( !wasBound )
    {
       // A state change occurred in the window's bound state.  Adjust counter.
  #ifdef IC_PMWIN
       unsigned long
         threadId,
         processId;
       IQUERYWINDOWPROCESS( this->handle(), &processId, &threadId );
  #endif
  #ifdef IC_MOTIF
       IThreadId
         threadId = IThread::current().id();
  #endif
       IWindowThreadData* data = IWindowThreadData::dataForThread( threadId );
       if ( data )
          data->addBoundWindow();
    }
  }
  else
  {
    pWindowData->state &=
            (unsigned long) ~IWindow::boundToMessageQueue;
    if (wasBound)
    {
       // A state change occurred in the window's bound state.  Adjust counter.
  #ifdef IC_PMWIN
       unsigned long
         threadId,
         processId;
       IQUERYWINDOWPROCESS( this->handle(), &processId, &threadId );
  #endif
  #ifdef IC_MOTIF
       IThreadId
         threadId = IThread::current().id();
  #endif
       IWindowThreadData* data = IWindowThreadData::dataForThread( threadId );
       if ( data )
          data->removeBoundWindow();
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::unbindMessageQueue                                                    |
------------------------------------------------------------------------------*/
IWindow& IWindow::unbindMessageQueue( )
{
  return bindMessageQueue(false);
}

/*------------------------------------------------------------------------------
| IWindow::isBoundToMessageQueue                                               |
------------------------------------------------------------------------------*/
bool IWindow::isBoundToMessageQueue ( ) const
{
  if ( (pWindowData->state & IWindow::boundToMessageQueue ) ||
       this->isPrimaryWindow() )
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| IWindow::setDefaultPushButton                                                |
|                                                                              |
| Set the defaultButton. Information is maintained by the frame window so keep |
| passing the information up until a IFrameWindow override of this member      |
| function is called.                                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::setDefaultPushButton ( const IWindowHandle& defaultPushButton )
{
  IWindow* pwinParent = this->parent();
  if ( pwinParent )
  {
    pwinParent->setDefaultPushButton( defaultPushButton );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::setDefaultEmphasisButton                                            |
|                                                                              |
| Set the defaultEmphasisButton. Information is maintained by the frame window |
| so keep passing the information up until a IFrameWindow override of this     |
| member function is called.                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow&
  IWindow::setDefaultEmphasisButton( const IWindowHandle& defaultEmphasisButton,
                                     bool  enable)
{
  IWindow* pwinParent = this->parent();
  if ( pwinParent )
  {
    pwinParent->setDefaultEmphasisButton( defaultEmphasisButton, enable );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IWindow::defaultPushButton                                                   |
|                                                                              |
| Return the defaultButton. Information is maintained by the frame window      |
| so keep calling up the parent chain until a IFrameWindow override of this    |
| member function is called.                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::defaultPushButton ( ) const
{
  IWindow* pwinParent = this->parent();
  if ( pwinParent )
  {
    return pwinParent->defaultPushButton();
  }
  return IWindowHandle( 0 );
}

/*------------------------------------------------------------------------------
| IWindow::defaultEmphasisButton                                               |
|                                                                              |
| Return the defaultEmphasisButton. Information is maintained by the frame     |
| window so keep calling up the parent chain until a IFrameWindow override of  |
| this member function is called.                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowHandle IWindow::defaultEmphasisButton ( ) const
{
  IWindow* pwinParent = this->parent();
  if ( pwinParent )
  {
    return pwinParent->defaultEmphasisButton();
  }
  return IWindowHandle( 0 );
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IWindow::setBidiSettings                                                     |
|                                                                              |
| Provide window-specific processing for a change to its bidirectional         |
| attributes.                                                                  |
|                                                                              |
------------------------------------------------------------------------------*/
IWindow& IWindow::setBidiSettings ( const IBidiSettings& bidiSettings,
                                    bool childInherit,
                                    bool refresh )
{
#ifdef IC_WIN
  // Window orientation and text direction are the only attributes
  // we can control.
  long
    extendedStyle = GetWindowLong( this->handle(), GWL_EXSTYLE );

  if ( bidiSettings.windowLayout() == IBidiSettings::layoutLeftToRight )
  {  // Need left-to-right window orientation.
     extendedStyle &= (unsigned long) ~( WS_EX_RIGHT | WS_EX_LEFTSCROLLBAR );
     extendedStyle |= ( WS_EX_LEFT | WS_EX_RIGHTSCROLLBAR );
  }
  else
  {  // Need right-to-left window orientation.
     extendedStyle &= (unsigned long) ~( WS_EX_LEFT | WS_EX_RIGHTSCROLLBAR );
     extendedStyle |= ( WS_EX_RIGHT | WS_EX_LEFTSCROLLBAR );
  }

  if ( bidiSettings.textOrientation() == IBidiSettings::textLeftToRight )
  {  // Need left-to-right text.
     extendedStyle &= (unsigned long) ~WS_EX_RTLREADING;
     extendedStyle |= WS_EX_LTRREADING;
  }
  else
  {  // Need right-to-left text.
     extendedStyle &= (unsigned long) ~WS_EX_LTRREADING;
     extendedStyle |= WS_EX_RTLREADING;
  }

  SetWindowLong( this->handle(), GWL_EXSTYLE, extendedStyle );
#endif // IC_WIN

  return *this;
}
#endif // IC_PMWIN

/*------------------------------------------------------------------------------
| IWindowStaticData::IWindowStaticData                                         |
------------------------------------------------------------------------------*/
IWindowStaticData::IWindowStaticData( )
#ifdef IC_WIN
  : fObjectWindowHandle ( )
#endif
{
   IMODTRACE_DEVELOP("IWindowStaticData::IWindowStaticData");
   IWindowStaticData::initializeStatics( true );
}

/*------------------------------------------------------------------------------
| IWindowStaticData::~IWindowStaticData                                        |
------------------------------------------------------------------------------*/
IWindowStaticData::~IWindowStaticData( )
{
   IMODTRACE_DEVELOP("IWindowStaticData::~IWindowStaticData");
   IWindowStaticData::finalizeStatics();
}

/*------------------------------------------------------------------------------
| IWindowStaticData::initializeStatics                                         |
------------------------------------------------------------------------------*/
void IWindowStaticData::initializeStatics( bool fromConstructor )
{
    IPrimalLock lock;
    if (!IWindowStaticData::flibKey)
    {
        IWindowStaticData::flibKey = new IPrivateResource;
        // Other static objects are demand allocated if/when needed
#ifdef IC_TRACE_DEVELOP
        if (!fromConstructor)
           ITRACE_DEVELOP("IWindowStaticData::flibKey was demand allocated.");
        else
           ITRACE_DEVELOP("IWindowStaticData::flibKey was allocated during static init.");
#endif
#ifdef IC_MOTIF
        IWindowStaticData::fgAllIWindowsSet =
                              new IWindowSortedSet( 100 );
#endif
        IWindowStaticData::fgAllAllocatedIWindowsSet =
                              new IWindowSortedSet( 100 );
    }
}

/*------------------------------------------------------------------------------
| IWindowStaticData::finalizeStatics                                           |
------------------------------------------------------------------------------*/
void IWindowStaticData::finalizeStatics( )
{
    // This cleanup is not done when we cannot control the order of
    // static initialization/termination and insure that it occurs in
    // process/module termination.
#if (IC_STATIC_PRIORITY_SUPPORTED)
    if (IWindowStaticData::flibKey)
    {
        delete IWindowStaticData::objectWindow;
        delete IWindowStaticData::desktopWindow;
        delete IWindowStaticData::fgDictionaryKey;
        delete IWindowStaticData::fgObserver;
        delete IWindowStaticData::fgThreadList;
        delete IWindowStaticData::fgDataHandleSet;
#ifdef IC_MOTIF
        delete IWindowStaticData::fgAllIWindowsSet;
#endif
        delete IWindowStaticData::fgAllAllocatedIWindowsSet;

        // Do this one last, and set it to 0 as a flag.
        delete IWindowStaticData::flibKey;
        IWindowStaticData::flibKey = 0;
    }
#endif
}

