// Revision: 92 1.4.2.3 source/ui/baseapp/ithreads.cpp, thread, ioc.v400, 001006 
/*----------------------------------------------------------------------------*/
/* FILE NAME: ithreads.cpp                                                    */
/*                                                                            */
/* DESCRIPTION:                                                               */
/*   This file contains the implementation of certain IThread member          */
/*   funcions, removing the dependency on beginthread for single-threaded     */
/*   applications.                                                            */
/*                                                                            */
/* COPYRIGHT:                                                                 */
/*   IBM Open Class Library                                                   */
/*   Licensed Materials - Property of IBM                                     */
/*                                                                            */
/*   5645-001                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997                                 */
/*                                                                            */
/*   US Government Users Restricted Rights - Use, duplication, or             */
/*   disclosure restricted by GSA ADP Schedule Contract with IBM              */
/*   Corp.                                                                    */
/*                                                                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/
// If the compiler were to support:
//    #pragma options /Gm+
// then we wouldn't need the declaration of _beginthread below.
// The declaration is lifted from stdlib.h, because it won't get
// generated unless /Gm+ has been set for the compile.
// If the declaration changes in stdlib.h, then we have to update here.

#include <ibase.hpp>

#ifdef IC_PMWIN
  #include <iwindefs.h>
#endif

extern "C" {
  #include <stdlib.h>
}

#include <ithread.hpp>
#include <ingthrdp.hpp>
#include <iexcept.hpp>
#include <itrace.hpp>
#include <istring.hpp>

#ifdef IC_WIN
  #include <iplatfrm.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ITHREADS_CPP_
  #include <ipagetun.h>
#endif


class IStartedThread;


/*------------------------------------------------------------------------------
  Class IGUIThreadFn is used by the IThread::start function to arrange for
  a chance to perform GUI initialization for the new thread.  Its run function
  performs GUI initialization if requested, and then calls the run function
  of the caller supplied IThreadFn object.
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IGUIThreadFn : public IThreadFn {
public:
  IGUIThreadFn ( IThreadFn* aFnObjRef );
 ~IGUIThreadFn ( );
void
  run ( );
private:
  IGUIThreadFn ( const IGUIThreadFn& )
  { }
IGUIThreadFn
  &operator=   ( const IGUIThreadFn& );
IThreadFn*
  fnObj;
}; // IGUIThreadFn

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IGUIThreadFn::IGUIThreadFn                                                   |
------------------------------------------------------------------------------*/
IGUIThreadFn::IGUIThreadFn( IThreadFn* aFnObjRef )
 : fnObj( aFnObjRef )
{
  IMODTRACE_DEVELOP("IGUIThreadFn::IGUIThreadFn");
}

/*------------------------------------------------------------------------------
| IGUIThreadFn::~IGUIThreadFn                                                  |
------------------------------------------------------------------------------*/
IGUIThreadFn::~IGUIThreadFn()
{
  IMODTRACE_DEVELOP("IGUIThreadFn::~IGUIThreadFn");
  // Delete the application's IThreadFn object.
  if ( fnObj )
  {
     delete fnObj;
  }
}

/*------------------------------------------------------------------------------
| IGUIThreadFn::run                                                            |
------------------------------------------------------------------------------*/
void IGUIThreadFn::run()
{
   IMODTRACE_DEVELOP("IGUIThreadFn::run");
   bool initializedGUI = false;
   // Initialize the GUI if requested.
   if (IThread::current().autoInitGUI() && ! IThread::current().isGUIInitialized())
   {
       IThread::current().initializeGUI( IThread::current().queueSize() );
       initializedGUI = true;
   }
   
   // Dispatch user code.
   fnObj->run();

#ifndef IC_MOTIF
   // In general, don't terminateGUI on thread exit.
   // Terminate the GUI if we initialized it.
   if (initializedGUI)
   {
       IThread::current().terminateGUI();
   }
#endif
}

/*------------------------------------------------------------------------------
| IGUIOptlinkFn class : An IThreadFn subclass to dispatch an optlink function  |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IGUIOptlinkFn : public IThreadFn {
public:
  IGUIOptlinkFn ( INonGUIThread::OptlinkFnPtr pfn,
                  void*                       p );
 ~IGUIOptlinkFn ( );

void
  run ( );

INonGUIThread::OptlinkFnPtr
  function;
void
 *arg;

private:
  IGUIOptlinkFn ( const IGUIOptlinkFn& )
  { }
IGUIOptlinkFn
 &operator=     ( const IGUIOptlinkFn& );
}; // IGUIOptlinkFn

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IGUIOptlinkFn::IGUIOptlinkFn                                                 |
------------------------------------------------------------------------------*/
IGUIOptlinkFn::IGUIOptlinkFn ( INonGUIThread::OptlinkFnPtr pfn, void* p )
  : function( pfn ), arg( p )
{ }

/*------------------------------------------------------------------------------
| IGUIOptlinkFn::~IGUIOptlinkFn                                                |
------------------------------------------------------------------------------*/
IGUIOptlinkFn::~IGUIOptlinkFn ( )
{ }

/*------------------------------------------------------------------------------
| IGUIOptlinkFn::run                                                           |
------------------------------------------------------------------------------*/
void IGUIOptlinkFn::run ( )
{
  function( arg );
}

/*------------------------------------------------------------------------------
| IGUISystemFn class : An IThreadFn subclass to dispatch a system function     |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IGUISystemFn : public IThreadFn {
public:
  IGUISystemFn ( INonGUIThread::SystemFnPtr pfn,
                 unsigned long              p );
 ~IGUISystemFn ( );

void
  run ( );

INonGUIThread::SystemFnPtr
  function;
unsigned long
  arg;

private:
  IGUISystemFn  ( const IGUISystemFn& )
  { }
IGUISystemFn
 &operator=     ( const IGUISystemFn& );
}; // IGUISystemFn

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
| IGUISystemFn::IGUISystemFn                                                   |
------------------------------------------------------------------------------*/
IGUISystemFn::IGUISystemFn ( INonGUIThread::SystemFnPtr pfn, unsigned long p )
  : function( pfn ), arg( p )
{ }

/*------------------------------------------------------------------------------
| IGUISystemFn::~IGUISystemFn                                                  |
------------------------------------------------------------------------------*/
IGUISystemFn::~IGUISystemFn ( )
{ }

/*------------------------------------------------------------------------------
| IGUISystemFn::run                                                            |
------------------------------------------------------------------------------*/
void IGUISystemFn::run ( )
{
  function( arg );
}


/*------------------------------------------------------------------------------
  All of the IThread constructors which result in the dispatch of the thread
  are implemented by calling the INonGUIThread default constructor, and then
  calling IThread::start.  IThread::start arranges for autoInitGUI processing
  to occur by using IGUIThreadFn before calling INonGUIThread::start.
------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
------------------------------------------------------------------------------*/
IThread::IThread ( IThreadFn* aFnObjRef,
                   bool       autoInitGUI )
  : INonGUIThread( )
  , fThreadData( 0 )
{
  this->start( aFnObjRef, autoInitGUI );
}

/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
|                                                                              |
| IThread constructor for functions that use the _Optlink linkage              |
------------------------------------------------------------------------------*/
IThread::IThread ( OptlinkFnPtr  pfn,
                   void*         anArg,
                   bool          autoInitGUI )
  : INonGUIThread( )
  , fThreadData( 0 )
{
  this->start( pfn, anArg, autoInitGUI );
}

/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
|                                                                              |
| IThread constructor for functions that use _System linkage                   |
------------------------------------------------------------------------------*/
IThread::IThread ( SystemFnPtr   pfn,
                   unsigned long anArg,
                   bool          autoInitGUI )
  : INonGUIThread( )
  , fThreadData( 0 )
{
  this->start( pfn, anArg, autoInitGUI );
}

/*------------------------------------------------------------------------------
| IThread::start                                                               |
|                                                                              |
| All versions are implemented via the function that accepts                   |
| as argument a reference to an IThread::Function object.  The other           |
| versions construct such an object from the other arguments.                  |
|                                                                              |
| In order to support autoInitGUI, this overload of start that accepts the     |
| autoInitGUI argument creates an IGUIThreadFn object and places in it a       |
| pointer to the caller supplied IThreadFn.  The IGUIThreadFn::run() function  |
| handles initialization and termination of the GUI.                           |
| The other overloads of start that accept function pointers and autoInitGUI   |
| create IThreadFn objects and then call this overload.                        |
------------------------------------------------------------------------------*/
void IThread::start ( IThreadFn* aFnObjRef,
                      bool       autoInitGUI )
{
  IMODTRACE_DEVELOP( "IThread::start (gui)" );
  if( autoInitGUI )
  {
    this->setAutoInitGUI( autoInitGUI );
    Inherited::start( new IGUIThreadFn( aFnObjRef ) );
  } 
  else
  {
    Inherited::start( aFnObjRef );
  }
}

/*------------------------------------------------------------------------------
| IThread::start                                                               |
| Supports autoInitGUI                                                         |
------------------------------------------------------------------------------*/
void IThread::start ( OptlinkFnPtr  pfn,
                      void*         anArg,
                      bool          autoInitGUI )
{
  this->start( new IGUIOptlinkFn( pfn, anArg ), autoInitGUI );
}

/*------------------------------------------------------------------------------
| IThread::start                                                               |
| Supports autoInitGUI                                                         |
------------------------------------------------------------------------------*/
void IThread::start ( SystemFnPtr   pfn,
                      unsigned long anArg,
                      bool          autoInitGUI )
{
  this->start( new IGUISystemFn( pfn, anArg ), autoInitGUI );
}


// The following overloads of start() are identical to their INonGUIThread
// counterparts.  They are overridden only to allow the new overloads in
// IThread which support autoInitGUI.
/*------------------------------------------------------------------------------
| IThread::start                                                               |
------------------------------------------------------------------------------*/
void IThread::start ( IThreadFn* aFnObjRef )
{
  IMODTRACE_DEVELOP( "IThread::start" );
  bool  autoInitGUI = defaultAutoInitGUI();
  start( aFnObjRef, autoInitGUI);
}

/*------------------------------------------------------------------------------
| IThread::start                                                               |
------------------------------------------------------------------------------*/
void IThread::start ( OptlinkFnPtr  pfn,
                      void*         anArg )
{
  Inherited::start( pfn, anArg );
}

/*------------------------------------------------------------------------------
| IThread::start                                                               |
------------------------------------------------------------------------------*/
void IThread::start ( SystemFnPtr   pfn,
                      unsigned long anArg )
{
  Inherited::start( pfn, anArg );
}

