// Revision: 79 1.10.2.1 source/ui/baseapp/istathdr.cpp, staticctls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: istathdr.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in istathdr.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// #define IC_TRACE_ALL

extern "C" {
  #define INCL_GPILOGCOLORTABLE
  #define INCL_GPIPRIMITIVES
  #define INCL_WINSYS
  #include <iwindefs.h>
}

#include <istathdr.hpp>

#include <ibidiset.hpp>
#include <icconst.h>
#include <icolmap.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <ihandle.hpp>
#include <ipainevt.hpp>
#include <istattxt.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#include <iwcname.hpp>

#ifdef IC_PM
  #include <ipmbidi.hpp>        // OS/2 specific BIDI functions
#endif
#ifdef IC_MOTIF
  #include <isizrect.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ISTATHDR_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IStaticTextHandler::IStaticTextHandler                                       |
------------------------------------------------------------------------------*/
IStaticTextHandler::IStaticTextHandler( )
  : fstaticText( 0 )
{ }

/*------------------------------------------------------------------------------
| IStaticTextHandler::~IStaticTextHandler                                      |
------------------------------------------------------------------------------*/
IStaticTextHandler::~IStaticTextHandler()
{ }

/*------------------------------------------------------------------------------
| IStaticTextHandler::drawBackground                                           |
------------------------------------------------------------------------------*/
bool IStaticTextHandler::drawBackground ( IStaticText*      staticText,
                                          IPresSpaceHandle& ps,
                                          const IRectangle& updateRect )
{
   IMODTRACE_ALL( "IStaticTextHandler::drawBackground" );
#ifdef IC_PMWIN
   if ( staticText->hasFillBackground() )
   {
      // Fill the window area with the fill color.
      ISize winsize = staticText->size();
      RECTL lprc;
      lprc.MINX = 0;
      lprc.MINY = 0;
      lprc.MAXX = winsize.width();
      lprc.MAXY = winsize.height();

      IColor clrFill = staticText->fillColor();

   #ifdef IC_PM
      WinFillRect( ps, &lprc, clrFill.asRGBLong() );
   #endif
   #ifdef IC_WIN
      // Need to adjust rectangle for 3D border if present.
      if ( staticText->extendedStyle() &
              IStaticText::border3D.asExtendedUnsignedLong() )
      {
         lprc.MAXX -= 4;
         lprc.MAXY -= 4;
      }
      HBRUSH hBrush = (HBRUSH)
          SelectObject( ps,
                        CreateSolidBrush(
                           NEARESTPALETTECOLOR( clrFill.asRGBLong() ) ) );
      HPEN hPen = (HPEN)
          SelectObject( ps,
                        CreatePen(
                           PS_SOLID,
                           1,
                           NEARESTPALETTECOLOR( clrFill.asRGBLong() ) ) );

      Rectangle( ps, (int) lprc.left, (int) lprc.top,
                 (int) lprc.right, (int) lprc.bottom );

      DeleteObject( SelectObject( ps, hBrush ) );
      DeleteObject( SelectObject( ps, hPen ) );
   #endif //IC_WIN
   }
#endif //IC_PMWIN
   return true;
}

/*------------------------------------------------------------------------------
| IStaticTextHandler::drawForeground                                           |
------------------------------------------------------------------------------*/
bool IStaticTextHandler::drawForeground ( IStaticText*      staticText,
                                          IPresSpaceHandle& ps,
                                          const IRectangle& updateRect )
{
   IMODTRACE_ALL( "IStaticTextHandler::drawForeground" );
#ifdef IC_PMWIN
   unsigned long ulLength = staticText->textLength();
   if ( ulLength )
   {
      //Save the style flags for frequent reference.
      unsigned long drawFlags = 0;
      unsigned long ulStyles = staticText->style();
      unsigned long ulExtStyles = staticText->extendedStyle();

      // Setup the window size in a native rectangle structure.
      ISize winsize = staticText->size();
      RECTL lprc;
      lprc.MINX = 0;
      lprc.MINY = 0;
      lprc.MAXX = winsize.width();
      lprc.MAXY = winsize.height();

  #ifdef IC_PM
      IString*      bidiText         = 0;
      LAYOUT_OBJECT bidiLayoutObject = 0;
  #endif //IC_PM
  #ifdef IC_WIN
      // Need to adjust rectangle for 3D border if present.
      if ( ulExtStyles & IStaticText::border3D.asExtendedUnsignedLong() )
      {
         lprc.MAXX -= 4;
         lprc.MAXY -= 4;
      }
  #endif //IC_WIN

      //---------------------------------------------------------------
      // Check for BIDI support
      //---------------------------------------------------------------
      bool rightToLeftText = false;
      if ( IBidiSettings::isBidiSupported() )
      {
         //-------------------------------------------------------------
         // Query the current BIDI attributes
         //-------------------------------------------------------------
         IBidiSettings settings( *staticText );

  #ifdef IC_PM
         //-------------------------------------------------------------
         // BIDI text strings need to be converted before display, so
         // we need to make a copy of the text (to work with)
         //-------------------------------------------------------------
         bidiText = new IString( staticText->text() );

         //-------------------------------------------------------------
         // Create a layout object to use when drawing BIDI text
         //-------------------------------------------------------------
         bidiLayoutObject = bidiCreateLayoutObject();

         //-------------------------------------------------------------
         // Convert the BIDI text for display
         // Note: We need to turn off symmetric swapping because during
         //       conversion (it will be handled later)
         //-------------------------------------------------------------
         {
            IBidiSettings tmpSettings( *staticText );
            unsigned long tilde = bidiText->indexOf( '~' );
            if ( tilde )
               bidiText->remove( tilde, 1 );
            tmpSettings.disableSymmetricSwapping();
            bidiPrepareForDisplay( bidiLayoutObject, *bidiText, tmpSettings, tilde );
            if ( tilde )
               bidiText->insert( "~", tilde );
         }
  #endif   //IC_PM

#ifdef IC_PM
         //-------------------------------------------------------------
         // If right-to-left layout, swap left and right justify flags
         //-------------------------------------------------------------
         if ( settings.windowLayout() == IBidiSettings::layoutRightToLeft )
         {
            if ( ulStyles & IStaticText::right.asUnsignedLong() )
            {
               ulStyles &= ~IStaticText::right.asUnsignedLong();
               ulExtStyles |= IStaticText::left.asExtendedUnsignedLong();
            }
            else
            {
               if ( ! ( ulStyles & IStaticText::center.asUnsignedLong() ) )
               {
                  ulStyles |= IStaticText::right.asUnsignedLong();
                  ulExtStyles &= ~IStaticText::left.asExtendedUnsignedLong();
               }
            }
         }
#endif // IC_PM

         if ( settings.textOrientation() == IBidiSettings::textRightToLeft )
         {
            rightToLeftText = true;
         }
      } // isBidiSupported

  #ifdef IC_PM
      if ( ulExtStyles & IStaticText::underscore.asExtendedUnsignedLong() )
         ulStyles |= DT_UNDERSCORE;
      if ( ulExtStyles & IStaticText::strikeout.asExtendedUnsignedLong() )
         ulStyles |= DT_STRIKEOUT;

      drawFlags |= (ulStyles & (DT_RIGHT | DT_CENTER | DT_BOTTOM |
                              DT_VCENTER | DT_WORDBREAK | DT_MNEMONIC |
                              DT_HALFTONE | DT_UNDERSCORE | DT_STRIKEOUT));

      if ( ! staticText->isEnabled() )
         drawFlags |= DT_HALFTONE;

      // Draw the text
      IString pszText = staticText->text();
      if ( bidiText )
         pszText = *bidiText;

      ISize fm = staticText->characterSize();
      unsigned long cchDrawn = 0;

      // Loop to handle multiple line static text
      IColor clrFgnd = staticText->foregroundColor();
      IColor clrBgnd = staticText->backgroundColor();
      while ( cchDrawn != ulLength )
      {
         RECTL lprc2;
         lprc2.MINX = lprc.MINX;
         lprc2.MAXX = lprc.MAXX;
         lprc2.MINY = lprc.MINY;
         lprc2.MAXY = lprc.MAXY;

         //-------------------------------------------------------------
         // Check for BIDI support
         //-------------------------------------------------------------
         if ( IBidiSettings::isBidiSupported() )
         {
            // For BIDI systems, call specialized draw routine
            IBidiSettings settings( *staticText );
            cchDrawn += bidiDrawText( ps,
                                      pszText,
                                      cchDrawn,
                                      bidiLayoutObject,
                                      settings,
                                      drawFlags,
                                      clrFgnd,
                                      clrBgnd,
                                      IRectangle( lprc2 ) );
         }
         else
         {
             // Calculate the size of the drawing rectangle
             WinDrawText( ps,
                          ulLength - cchDrawn,
                          (PCHAR)pszText + cchDrawn,
                          &lprc2,
                          0,
                          0,
                          drawFlags | DT_QUERYEXTENT );
             if ( staticText->hasFillBackground() )
                WinFillRect( ps, &lprc2, clrBgnd.asRGBLong() );

             // Draw the static text
             cchDrawn += WinDrawText( ps,
                                      ulLength - cchDrawn,
                                      (PCHAR)pszText + cchDrawn,
                                      &lprc,
                                      clrFgnd.asRGBLong(),
                                      clrBgnd.asRGBLong(),
                                      drawFlags );
         }

          // Don't loop if no word breaking, or if it is not top aligned.
         if ( ! ( drawFlags & DT_WORDBREAK ) )
            break;

         if ( drawFlags & ( DT_VCENTER | DT_BOTTOM ) )
            break;

          // If out of room for drawing, exit.
         if ( ( lprc.MAXY -= fm.height() ) <= lprc.MINY )
            break;
      } /* end while */
  #endif  // IC_PM

  #ifdef IC_WIN
      drawFlags = DT_NOPREFIX ;     // implied DT_LEFT | DT_TOP
      // Set up the DrawText flags based on the control extended styles.
      if ( ulStyles & SS_RIGHT )
         drawFlags |= DT_RIGHT;
      if ( ulStyles & SS_CENTER )
         drawFlags |= DT_CENTER ;
      if ( ulExtStyles & IDT_BOTTOM )
         drawFlags |= DT_BOTTOM | DT_SINGLELINE;
      if ( ulExtStyles & IDT_VCENTER )
         drawFlags |= DT_VCENTER | DT_SINGLELINE;
      if ( ulExtStyles & IDT_WORDBREAK )
         drawFlags |= DT_WORDBREAK;
      if ( ulExtStyles & IDT_MNEMONIC )
         drawFlags &= (unsigned long)~DT_NOPREFIX;

      // If running on a bidi system, add in the right-to-left style.
      if ( rightToLeftText )
         drawFlags |= DT_RTLREADING;

      bool grayText = ( ( ulExtStyles & IDT_HALFTONE ) ||
                        ( ! staticText->isEnabled() ) );

      // Draw the text.
      IString pszText = staticText->text();
      COLORREF fgOld, bgOld;
      if ( ! grayText )
      {
         fgOld = SetTextColor( ps, staticText->foregroundColor().asRGBLong() );
      }
      else
      {
         DWORD grayColor = GetSysColor( COLOR_GRAYTEXT );
         fgOld = SetTextColor( ps, grayColor );
      }
      if ( fgOld == (COLORREF)CLR_INVALID )
      {
         ITHROWGUIERROR2( "SetTextColor",
                          IBaseErrorInfo::accessError,
                          IException::unrecoverable );
      }
      bgOld = SetBkColor( ps, staticText->backgroundColor().asRGBLong() );
      int bmOld = SetBkMode(
                     ps,
                     staticText->hasFillBackground() ? OPAQUE : TRANSPARENT );
      if ( bgOld == (COLORREF)CLR_INVALID )
      {
         ITHROWGUIERROR2( "SetTextColor",
                          IBaseErrorInfo::accessError,
                          IException::unrecoverable );
      }

      // Select the font to use...need a copy of the font.  We use the font
      // to implement strikeout and underscore.
      IFont winFont = staticText->font();
      if ( staticText->isUnderscore() )
         winFont.setUnderscore();
      if ( staticText->isStrikeout() )
         winFont.setStrikeout();
      winFont.beginUsingFont( ps );

      // Draw the static text
      DrawText( ps, (PCHAR)pszText, (int)ulLength , (RECT*)&lprc, (UINT)drawFlags );

      winFont.endUsingFont( ps );

      fgOld = SetTextColor( ps, fgOld );
      bgOld = SetBkColor( ps, bgOld );
      SetBkMode( ps, bmOld );
  #endif // IC_WIN

  #ifdef IC_PM
      //---------------------------------------------------------------
      // Clean up any temporary storage used for BIDI support
      //---------------------------------------------------------------
      if ( bidiText )
         delete bidiText;
      if ( bidiLayoutObject )
         bidiDeleteLayoutObject( bidiLayoutObject );
  #endif //IC_PM

   }  // if length > 0
#endif // IC_PMWIN

#ifdef IC_MOTIF
   // Need to recompute the text alignment based on the current size of the
   // window.
   staticText->setAlignment( staticText->alignment() );
#endif //IC_MOTIF

   return true;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IStaticTextHandler::paintWindow                                              |
|                                                                              |
| Process the paint events for a static window.                                |
| This function always returns true.  Derived classes must perform all         |
| required drawing in drawForeground and drawBackground.                       |
------------------------------------------------------------------------------*/
bool IStaticTextHandler::paintWindow( IPaintEvent& evt )
{
   IMODTRACE_ALL( "IStaticTextHandler::paintWindow" );
  // The following is an optimization which relies upon the fact that we
  // have an instance of this handler for each IStaticText object.  Should that
  // ever change, use evt.window() to get the IStaticText object.
  IStaticText* pStatic = this->staticText();
  // If the pointer to the control is null, cache pointer to window for future paints.
  if ( ! pStatic )
  {
      pStatic = dynamic_cast< IStaticText* >( evt.window() );
      this->setStaticText( pStatic );
      if ( ! pStatic )
         return false;    // Cannot process event...not IStaticText
  }


  IPresSpaceHandle ps( evt.presSpaceHandle() );
  IRectangle       updateRect( evt.rect() );

#ifdef IC_PM
  ICREATELOGCOLORTABLE( ps, 0L,  LCOLF_RGB, 0L, 0L, 0 );
#endif
#ifdef IC_WIN
  // Insure that the palette is selected into the ps/dc.  This is automatic
  // if an IGrafPort is used but there are places in this handler where the
  // ps is used directly.
  IColorMapContext* context = IColorMap::hasColorMapSupport() ?
     new IColorMapContext( ps, &(IColorMap::defaultColorMap()) ) : 0;
#endif

  drawBackground( pStatic, ps, updateRect );
  drawForeground( pStatic, ps, updateRect );

#ifdef IC_WIN
  if (context)
     delete context;
#endif

  return true;
}
#endif // IC_PMWIN

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IStaticTextHandler::windowResize                                             |
| This function always returns true.  Derived classes must perform all         |
| required drawing in drawForeground and drawBackground.                       |
------------------------------------------------------------------------------*/
bool IStaticTextHandler::windowResize ( IResizeEvent& evt )
{
   IMODTRACE_ALL("IStaticTextHandler::windowResize");
   // The following is an optimization which relies upon the fact that we
   // have an instance of this handler for each IStaticText object.  Should that
   // ever change, use evt.window() to get the IStaticText object.
   IStaticText* pStatic = this->staticText();
   // If the pointer to the control is null, cache pointer to window for future paints.
   if ( ! pStatic )
   {
      pStatic = dynamic_cast< IStaticText* >( evt.window() );
      this->setStaticText( pStatic );
      if ( ! pStatic )
         return false;    // Cannot process event...not IStaticText
   }

   IPresSpaceHandle dummyPs;   // Dummy, unused in Motif.
   drawBackground( pStatic, dummyPs, IRectangle() );
   drawForeground( pStatic, dummyPs, IRectangle() );
   return true;
}

/*------------------------------------------------------------------------------
| IStaticTextHandler::dispatchHandlerEvent                                     |
------------------------------------------------------------------------------*/
bool IStaticTextHandler::dispatchHandlerEvent ( IEvent& event )
{
   if ( ( event.eventId() == IC_UM_PAINT ) ||
        ( event.eventId() == xEvent( MapNotify ) ) )
   {  // Respond to refresh() by redrawing button.

      // The following is an optimization which relies upon the fact that we
      // have an instance of this handler for each IStaticText object.  Should that
      // ever change, use evt.window() to get the IStaticText object.
      IStaticText* pStatic = this->staticText();
      // If the pointer to the control is null, cache pointer to window for future paints.
      if ( ! pStatic )
      {
         pStatic = dynamic_cast< IStaticText* >( event.window() );
         this->setStaticText( pStatic );
         if ( ! pStatic )
            return false;    // Cannot process event...not IStaticText
      }

      IPresSpaceHandle dummyPs;   // dummy, unused in Motif.
      drawBackground( pStatic, dummyPs, IRectangle() );
      drawForeground( pStatic, dummyPs, IRectangle() );
      return true;
   }
   return Inherited::dispatchHandlerEvent( event );
}
#endif // IC_MOTIF

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IStaticTextHandler::dispatchHandlerEvent                                     |
|                                                                              |
| We override this function to detect resizing of the static text window.      |
| When the window is resized we need to invalidate the entire window so that   |
| text (and graphics) get positioned consistently.  Without this, painting     |
| problems appear in some cases (for example, the asplitcv sample).            |
| We could use a resize handler to do this but it is done this way to          |
| keep only one default handler.                                               |
------------------------------------------------------------------------------*/
bool IStaticTextHandler::dispatchHandlerEvent ( IEvent& event )
{
   bool result = false;
   if ( event.eventId() == WM_WINDOWPOSCHANGED )
   {
      PWINDOWPOS pswp = (PWINDOWPOS)(char*)event.parameter2();
      if ( ! ( pswp->flags & SWP_NOSIZE ) )
      {
         // Invalidate the entire window
         IWindow* win = event.dispatchingWindow();
         if ( win )
            win->refresh( IWindow::paintAll );
      }
   }
   else if ( event.eventId() == WM_ENABLE )
   {
      // Bypass the Static class window procedure, since this seems to
      // do direct painting (no WM_PAINT generated) to give its text a
      // disabled/enabled look.  This direct painting does use colors
      // assigned by the application to the IStaticText object.
      // Invalidate the entire window to allow the IStaticText paint
      // handler to repaint the text appropriately.
      IWindow* win = event.dispatchingWindow();
      if ( win )
      {
         win->refresh( IWindow::paintAll );
         DefWindowProc( event.handle(),
                        (UINT)event.eventId(),
                        event.parameter1(),
                        event.parameter2() );
         result = true;
      }
   }
   else if ( event.eventId() == WM_NCHITTEST )
   {
      IWindowHandle parent = IPARENTOF( event.handle() );
      if ( ( ! parent ) ||
           ( IWindowClassName( parent ) != WC_BUTTON ) )
      {
         // Avoid HT_TRANSPARENT from WC_STATIC window procedure to
         // allow mouse button presses on static controls.
         MRESULT mr = IDEFWINDOWPROC( event.handle(),
                                      (UINT)event.eventId(),
                                      event.parameter1(),
                                      event.parameter2() );
         event.setResult( IEventResult( (unsigned long)mr ) );
         result = true;
      }
      else
      {
         // We want the default behavior when the parent is a button
         // such as IGraphicPushButton.   Can't click in graphic otherwise.
         result = Inherited::dispatchHandlerEvent( event );
      }
   }
   else
   {
      result = Inherited::dispatchHandlerEvent( event );
   }
   return result;
}
#endif // IC_WIN
