// Revision: 48 1.8.3.3 source/ui/baseapp/iprofile.cpp, profile, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iprofile.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the class(es):                                           *
*     IProfile                                                                 *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
extern "C" {
  #define INCL_WINSHELLDATA
  #define INCL_SHLERRORS
  #include <iwindefs.h>
  #include <stdio.h>
  #ifdef IC_MOTIF
    #include <X11/StringDefs.h>
    #include <X11/Xresource.h>
  #endif //IC_MOTIF
}

#ifdef IC_MOTIF
  #define STRBUFFERSIZE 2048
  #define max(x,y) (((x)>(y))?(x):(y))
#endif //IC_MOTIF

#include <iprofile.hpp>
#include <iprofprv.hpp>
#include <iexcept.hpp>
#include <istring.hpp>
#include <istparse.hpp>
#include <ithread.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IPROFILE_CPP_
  #include <ipagetun.h>
#endif


#ifdef IC_MOTIF

// The following data declared in non-const segment, because Istring
// construction calls new()
// used in creating resource entries
const IString iprofOpenFlag("a+");
const IString iprofDot(".");
const IString iprofColon(":");
const IString iprofApplKey("0.0");

static int strBufferSize;
XrmQuark  XrmQString;

#endif //IC_MOTIF

#ifdef IC_PMWIN
// New static (not thread safe) for determining when to translate
// profile, application and key names and when not to.  Default is
// true to remain compatible with GA release.
static bool fgMapKey = false;
#endif
#ifdef IC_WIN
/*------------------------------------------------------------------------------
| Modify a key name to meet Win32 registry requirements                        |
------------------------------------------------------------------------------*/
static IString IRegFixKey( const char* name )
{
  IString keyName(name);

  // Convert invalid key name characters for compatibility with OS/2
  // Also converts hierarchical characters to flatten tree.
  if (IProfile::isKeyMappingEnabled())
  {
    // Convert slashes first, to allow us to differentiate later
    // between an existing combo and one we created.
    keyName.change("/","//");

    // Convert illegal characters for Win32 Registry.
    keyName.change("*","/1");
    keyName.change("?","/0");

    // Convert backslash to flatten hierarchy.
    keyName.change("\\","/_");
  }
  return keyName;
}

/*------------------------------------------------------------------------------
| Check if a Win32 registry contains the specified key                         |
------------------------------------------------------------------------------*/
static bool IRegTestKey( HKEY parentKey, const char* name )
{
  HKEY          openKey;
  IString       keyName = IRegFixKey( name );
  unsigned long rc;

  rc = RegOpenKeyEx( parentKey, keyName, 0, KEY_READ, &openKey );
  if ( rc == ERROR_SUCCESS )
  {
    RegCloseKey(openKey);
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| Open a key in the Win32 registry                                             |
------------------------------------------------------------------------------*/
static HKEY IRegOpenKey( HKEY parentKey, const char* name )
{
  HKEY    openKey;
  IString keyName = IRegFixKey( name );
  unsigned long disposition;
  if ( RegCreateKeyEx ( parentKey, keyName, 0, "",
                        REG_OPTION_NON_VOLATILE, (KEY_READ | KEY_WRITE),
                        NULL, &openKey, &disposition ) != ERROR_SUCCESS )
  {
    ITHROWGUIERROR("RegCreateKeyEx");
  }
  return openKey;
}

/*------------------------------------------------------------------------------
| Delete a key from the Win32 registry                                         |
------------------------------------------------------------------------------*/
static void IRegDeleteKey( HKEY parentKey, const char* name )
{
  IString keyName = IRegFixKey( name );
  RegDeleteKey( parentKey, keyName );
}

/*------------------------------------------------------------------------------
| Open the profile name passed, returning the key for it.  Also set flag       |
| to denote whether this is a system profile or not                            |
------------------------------------------------------------------------------*/
static HKEY IRegOpenProfile( const char *profName, bool* isSystemProfile )
{
  HKEY openKey, parentKey = 0;
  IString token1, token2;
  IString token3 = profName;

  token3 >> token1 >> "\\" >> token3;

  if (token1 == "HKEY_CURRENT_USER")
    parentKey = HKEY_CURRENT_USER;

  if (token1 == "HKEY_LOCAL_MACHINE")
    parentKey = HKEY_LOCAL_MACHINE;

  if (token1 == "HKEY_USERS")
    parentKey = HKEY_USERS;

  if (token1 == "HKEY_CLASSES_ROOT")
    parentKey = HKEY_CLASSES_ROOT;

  // If the first token matches a system defined keyword and
  // has additional key information as well
  if ((parentKey) && (token3 != ""))
  {
    token3 >> token2 >> "\\" >> token3;

    // If only one subkey provided, open that subkey
    if (token3 == "")
    {
      if (token2 == "Software")
        (*isSystemProfile) = TRUE;
      openKey = IRegOpenKey( parentKey, token2 );
    }
    else
    {
      // Otherwise open the parent, then the subkey, and close parent
      HKEY tempKey = IRegOpenKey( parentKey, token2 );
      openKey = IRegOpenKey( tempKey, token3 );
      RegCloseKey( tempKey );
    }
  }
  else
  {
    // If not system profile, then open app profile under local area.
    HKEY tempKey = IRegOpenKey( HKEY_LOCAL_MACHINE, "Software" );
    openKey = IRegOpenKey( tempKey, profName );
    RegCloseKey( tempKey );
  }
  return openKey;
}

/*------------------------------------------------------------------------------
| Get the data type for a key from the Win32 registry                          |
------------------------------------------------------------------------------*/
static unsigned long IRegDataType ( HKEY openKey )
{
  char          buffer[10];
  unsigned long rc;
  unsigned long dataType = REG_NONE;
  long          bufferSize = 10;

  // Query the operating system for the type of data stored at key
  rc = RegQueryValueEx( openKey, "DataType", 0, 0,
                        (LPBYTE)buffer, (LPDWORD)&bufferSize );
 // If successful, then set data type flag for future operations
  if ( rc == ERROR_SUCCESS )
  {
     buffer[bufferSize] = '\0';
     IString dType(buffer);
     if ( dType == "Binary" )
       dataType = REG_BINARY;
     if ( dType == "Integer" )
       dataType = REG_DWORD;
     if ( dType == "String" )
       dataType = REG_SZ;
  }
  return dataType;
}

/*------------------------------------------------------------------------------
| Get the string for a key from the Win32 registry                             |
------------------------------------------------------------------------------*/
static IString IRegGetString ( HKEY openKey )
{
  IString       str;
  char*         buffer;
  long          bufferSize;
  unsigned long rc;

  // First get the size of the buffer needed to retrieve the string.
    rc = RegQueryValueEx( openKey, "Data", 0, 0, 0, (LPDWORD)&bufferSize );
  // If it fails, throw an exception
  if ( rc != ERROR_SUCCESS )
    ITHROWGUIERROR("RegQueryValueEx");

  // Next, allocate a buffer and retrieve the string data
  buffer = new char[bufferSize+1];
  rc = RegQueryValueEx( openKey, "Data", 0, 0, (LPBYTE)buffer,
                        (LPDWORD)&bufferSize );
  // Finally terminate the buffer, create a string object,
  // free the character buffer and return the string.
  buffer[bufferSize] = '\0';
  str = IString(buffer);
  delete [] buffer;
  return str;
}

/*------------------------------------------------------------------------------
| Get the integer for a key from the Win32 registry                            |
------------------------------------------------------------------------------*/
static long IRegGetInteger ( HKEY openKey )
{
  long          value;
  long          bufferSize;
  // If regular 32-bit platform, then data can be stored as an integer
  // so query the value directly
  bufferSize = sizeof(value);
  if ( RegQueryValueEx( openKey, "Data", 0, 0, (LPBYTE)&value,
                        (LPDWORD)&bufferSize ) != ERROR_SUCCESS )
  {
    ITHROWGUIERROR("RegQueryValueEx");
  }
  return value;
}

/*------------------------------------------------------------------------------
| Get the binary data for a key from the Win32 registry                        |
------------------------------------------------------------------------------*/
static IString IRegGetBinary ( HKEY openKey )
{
  IString       str;
  unsigned long cnt;
  unsigned long rc = ERROR_SUCCESS;
  long          bufferSize;
  for ( cnt = 0; rc == ERROR_SUCCESS; cnt++ )
  {
    IString valueName(cnt);
    rc = RegQueryValueEx( openKey, valueName, 0, 0, 0,
                          (LPDWORD)&bufferSize );
    if ( rc == ERROR_SUCCESS )
    {
      char* buffer = new char[bufferSize+1];
      rc = RegQueryValueEx( openKey, valueName, 0, 0,
                            (LPBYTE)buffer, (LPDWORD)&bufferSize );
      IString tmp(buffer,bufferSize);
      str += tmp;
      delete [] buffer;
    }
  }
  return str;
}

/*------------------------------------------------------------------------------
| Write a string in the Win32 registry                                         |
------------------------------------------------------------------------------*/
static void IRegWriteString ( HKEY openKey, IString data )
{
  long          bufferSize;
  unsigned long rc;
  bufferSize = 6;
  rc = RegSetValueEx ( openKey, "DataType", 0, REG_SZ,
                       (const BYTE*) "String", bufferSize );
  if ( rc != ERROR_SUCCESS )
    ITHROWGUIERROR("RegSetValueEx");
  bufferSize = data.length();
  rc = RegSetValueEx ( openKey, "Data", 0, REG_SZ, data, bufferSize );
  if ( rc != ERROR_SUCCESS )
    ITHROWGUIERROR("RegSetValueEx");
}

/*------------------------------------------------------------------------------
| Get the integer for a key from the Win32 registry                            |
------------------------------------------------------------------------------*/
static void IRegWriteInteger ( HKEY openKey, long data )
{
  long          bufferSize;
  unsigned long rc;
  bufferSize = 7;
  rc = RegSetValueEx( openKey, "DataType", 0, REG_SZ,
                      (const BYTE*) "Integer", bufferSize );
  if ( rc != ERROR_SUCCESS )
    ITHROWGUIERROR("RegSetValueEx");
  bufferSize = sizeof(data);
  rc = RegSetValueEx( openKey, "Data", 0, REG_DWORD,
                      (const BYTE*) &data, bufferSize );
  if ( rc != ERROR_SUCCESS )
    ITHROWGUIERROR("RegSetValueEx");
}

/*------------------------------------------------------------------------------
| Get the binary data for a key from the Win32 registry                        |
------------------------------------------------------------------------------*/
static void IRegWriteBinary ( HKEY openKey, IString data )
{
  long          bufferSize;
  unsigned long rc;
  unsigned long cnt;
  unsigned long written = 0;

  bufferSize = 6;
  rc = RegSetValueEx ( openKey, "DataType", 0, REG_SZ,
                       (const BYTE*) "Binary", bufferSize );
  if ( rc != ERROR_SUCCESS )
    ITHROWGUIERROR("RegSetValueEx");

  for ( cnt = 0; written < data.length(); cnt++ )
  {
     IString tmp;
     IString valueName(cnt);
     if (( data.length() - written ) > 16000 )
        tmp = data.subString(written+1,16000);
     else
        tmp = data.subString(written+1);
     written += tmp.length();
     bufferSize = tmp.length();
     rc = RegSetValueEx ( openKey, valueName, 0, REG_BINARY, tmp, bufferSize );
     if ( rc != ERROR_SUCCESS )
       ITHROWGUIERROR("RegSetValueEx");
  }
}
#endif  //IC_WIN

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| Open a key in the Win32 registry                                             |
------------------------------------------------------------------------------*/
static unsigned long IRegOpenOrCreateKey( HKEY parentKey, const char* name, HKEY *openKey, bool mod )
{
  IString keyName = IRegFixKey( name );
  unsigned long disposition;
  unsigned long rc;
  if (mod){
	 rc = RegCreateKeyEx ( parentKey, keyName, 0, "",
				  REG_OPTION_NON_VOLATILE, (KEY_READ | KEY_WRITE),
				  NULL, openKey, &disposition ); 
  }
  else{
    rc = RegOpenKeyEx( parentKey, keyName, 0, KEY_READ, openKey ); 
  }
  return rc;
}

/*------------------------------------------------------------------------------
| IProfile::IProfile                                                           |
------------------------------------------------------------------------------*/
IProfile::IProfile( const char *profileName , EType pType, EAccess pAccess)
  : profileName(profileName), defApplName(), fProfileData(new IProfileData())
{
  IMODTRACE_DEVELOP("IProfile::ctor (char *, EType, EAccess)");
  HKEY tempKey;
  unsigned long rc;
  IASSERTPARM(profileName != 0);

  fProfileData->newConstrctr = 1;
  if( (pType == kSystem) && (pAccess == kReadWrite) ) {
	   fProfileData->parentKey = HKEY_LOCAL_MACHINE ;
	   fProfileData->accessMode = kReadWrite ;
  		rc = IRegOpenOrCreateKey( HKEY_LOCAL_MACHINE, "Software", &tempKey, true);
      if ( rc == ERROR_SUCCESS ){
			if( IRegOpenOrCreateKey( tempKey, profileName, 
			               &fProfileData->hKey, true) == ERROR_SUCCESS){
  				RegCloseKey( tempKey );
			}
			else {
  				RegCloseKey( tempKey );
				ITHROWGUIERROR("IRegOpenOrCreateKey");
			}
      }
		else {
			ITHROWGUIERROR("IRegOpenOrCreateKey");
		}
  }
  else if( (pType == kSystem) && (pAccess == kReadOnly) ){
	   fProfileData->parentKey = HKEY_LOCAL_MACHINE ;
	   fProfileData->accessMode = kReadOnly ;
  		rc = IRegOpenOrCreateKey( HKEY_LOCAL_MACHINE, "Software", &tempKey, false);
      if ( rc == ERROR_SUCCESS ){
			if( IRegOpenOrCreateKey( tempKey, profileName, 
			               &fProfileData->hKey, false) == ERROR_SUCCESS){
  				RegCloseKey( tempKey );
			}
			else {
  				RegCloseKey( tempKey );
				ITHROWGUIERROR("IRegOpenOrCreateKey");
			}
      }
		else {
			ITHROWGUIERROR("IRegOpenOrCreateKey");
		}
  }
  else if( (pType == kUser) && (pAccess == kReadWrite) ){
	   fProfileData->parentKey = HKEY_CURRENT_USER;
	   fProfileData->accessMode = kReadWrite;
  		rc = IRegOpenOrCreateKey( HKEY_CURRENT_USER, "Software", &tempKey, true);
      if ( rc == ERROR_SUCCESS ){
			if( IRegOpenOrCreateKey( tempKey, profileName, 
			               &fProfileData->hKey, true) == ERROR_SUCCESS){
  				RegCloseKey( tempKey );
			}
			else {
  				RegCloseKey( tempKey );
				ITHROWGUIERROR("IRegOpenOrCreateKey");
			}
      }
		else {
			ITHROWGUIERROR("IRegOpenOrCreateKey");
		}
  }
  else if( (pType == kUser) && (pAccess == kReadOnly) ){
	   fProfileData->parentKey = HKEY_CURRENT_USER ;
	   fProfileData->accessMode = kReadOnly ;
  		rc = IRegOpenOrCreateKey( HKEY_CURRENT_USER, "Software", &tempKey, false);
      if ( rc == ERROR_SUCCESS ){
			if( IRegOpenOrCreateKey( tempKey, profileName, 
			               &fProfileData->hKey, false) == ERROR_SUCCESS){
  				RegCloseKey( tempKey );
			}
			else {
  				RegCloseKey( tempKey );
				ITHROWGUIERROR("IRegOpenOrCreateKey");
			}
      }
		else {
			ITHROWGUIERROR("IRegOpenOrCreateKey");
		}
  }
}
#endif

/*------------------------------------------------------------------------------
| IProfile::IProfile                                                           |
------------------------------------------------------------------------------*/
IProfile::IProfile( const char *profName )
#ifdef IC_PMWIN
  : profileName(profName), defApplName(), fProfileData(new IProfileData())
#else
  : profileName(profName), defApplName(), fProfileData(new IProfileData(this))
#endif
{
  IMODTRACE_DEVELOP("IProfile::ctor (char *)");
  IASSERTPARM(profName != 0);

#ifdef IC_PM
  //Note:  Create a temporary IString for the profile name to
  //       workaround a PMSHAPI bug that attempts to write to
  //       read-only storage
  hIni = PrfOpenProfile( IThread::current().anchorBlock(),
                         IString( profName ) );
  if (!hIni)
    ITHROWGUIERROR("PrfOpenProfile");
#endif //IC_PM
#ifdef IC_WIN
  fProfileData->hKey = IRegOpenProfile( profName,
                                        &(fProfileData->isSystemProfile) );
#endif
#ifdef IC_MOTIF
   IString theProfName(profName);
   unsigned occurs = theProfName.occurrencesOf("/");
   if (!occurs)
   {
     IString newProfile = (char *)(getenv("PWD"));
     if (newProfile.size())
     {
       newProfile += "/";
       newProfile.change("//", "/");
       newProfile += profName;
       profileName = newProfile;
     }
   }
   hIni = fProfileData->openProfile(profileName);
   if (!hIni)
   {
      ITHROWGUIERROR("PrfOpenProfile");
   }
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IProfile::IProfile                                                           |
| Copy constructor                                                             |
------------------------------------------------------------------------------*/
IProfile::IProfile ( const IProfile& aProfile )
#ifdef IC_PMWIN
  : fProfileData(new IProfileData())
#endif
{
#ifdef IC_PM
  if ((aProfile.hIni != HINI_PROFILE) &&
      (aProfile.hIni != HINI_USERPROFILE) &&
      (aProfile.hIni != HINI_SYSTEMPROFILE))
  {
     hIni = PrfOpenProfile(IThread::current().anchorBlock(),
                            (PSZ)aProfile.profileName);
     if (!hIni)
        ITHROWGUIERROR("PrfOpenProfile");
  }
  else
     this->hIni = aProfile.hIni;
#endif //IC_PM
#ifdef IC_MOTIF
   fProfileData = new IProfileData(this);
   hIni = fProfileData->openProfile(aProfile.profileName);
   if (!hIni) {
      ITHROWGUIERROR("PrfOpenProfile");
   } /* endif */
#endif //IC_MOTIF

  this->defApplName = aProfile.defApplName;
  this->profileName = aProfile.profileName;

#ifdef IC_WIN
  this->fProfileData->parentKey = aProfile.fProfileData->parentKey;
  this->fProfileData->accessMode = aProfile.fProfileData->accessMode;
  this->fProfileData->newConstrctr = aProfile.fProfileData->newConstrctr;
  this->fProfileData->hKey =
    IRegOpenProfile( this->profileName, &(this->fProfileData->isSystemProfile));
#endif
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IProfile::IProfile                                                           |
|  Private Ctor. to create from Profile Handle                                 |
------------------------------------------------------------------------------*/
IProfile::IProfile ( const IProfileHandle& profile )
#ifdef IC_PM
  : hIni(profile), defApplName(), fProfileData(new IProfileData())
#else
  : hIni(profile), defApplName(), fProfileData(new IProfileData(this))
#endif
{
#ifdef IC_PM
  HAB hab = WinQueryAnchorBlock(HWND_DESKTOP);
  PRFPROFILE Profile;
  Profile.cchUserName = 0;
  Profile.pszUserName = 0;
  Profile.cchSysName = 0;
  Profile.pszSysName = 0;
  char Buffer[100];

  if (PrfQueryProfile(hab, &Profile))
  {
    if (hIni == HINI_USERPROFILE)
    {
      char* pszData = new char[Profile.cchUserName+1];
      Profile.pszUserName = (PSZ)pszData;
      Profile.cchSysName = 100;
      Profile.pszSysName = (PSZ)Buffer;
      PrfQueryProfile(hab, &Profile);
      profileName = IString(pszData);
      delete [] pszData;
    }
    else
    {
      char* pszData = new char[Profile.cchSysName];
      Profile.cchUserName = 100;
      Profile.pszUserName = (PSZ)Buffer;
      Profile.pszSysName = (PSZ)pszData;
      PrfQueryProfile(hab, &Profile);
      profileName = IString(pszData);
      delete [] pszData;
    } /* endif */
  } /* endif */
#endif // IC_PM
}
#endif  //IC_MOTIFPM

/*------------------------------------------------------------------------------
| IProfile::systemProfile                                                      |
------------------------------------------------------------------------------*/
IProfile IProfile::systemProfile ( )
{
  IMODTRACE_DEVELOP("IProfile::systemProfile");

#ifdef IC_PM
  return IProfile(HINI_SYSTEMPROFILE);
#endif
#ifdef IC_WIN
  return IProfile("HKEY_LOCAL_MACHINE\\Software");
#endif
#ifdef IC_MOTIF
  return (IProfile)NULL;
#endif

}

/*------------------------------------------------------------------------------
| IProfile::userProfile                                                        |
------------------------------------------------------------------------------*/
IProfile IProfile::userProfile ( )
{
   IMODTRACE_DEVELOP("IProfile::userProfie");

#ifdef IC_PM
  return IProfile(HINI_USERPROFILE);
#endif
#ifdef IC_WIN
  return IProfile("HKEY_CURRENT_USER\\Software");
#endif
#ifdef IC_MOTIF
  return (IProfile)NULL;
#endif
}

/*------------------------------------------------------------------------------
| IProfile::operator =                                                         |
------------------------------------------------------------------------------*/
IProfile& IProfile::operator =(const IProfile& aProfile)
{
#ifdef IC_PM
  /* Note: Ensure self assignment works */
  if (this->hIni != aProfile.hIni)
  {
     if ((hIni != HINI_PROFILE) &&
         (hIni != HINI_USERPROFILE) &&
         (hIni != HINI_SYSTEMPROFILE))
     {
        // close file if not system & user
        PrfCloseProfile(hIni);

        hIni = PrfOpenProfile(IThread::current().anchorBlock(),
                              (PSZ)aProfile.profileName);
        if (!hIni)
           ITHROWGUIERROR("PrfOpenProfile");
     }
     else
        this->hIni = aProfile.hIni;

     this->defApplName = aProfile.defApplName;
     this->profileName = aProfile.profileName;
  } /* endif */
#endif

#ifdef IC_WIN
  this->defApplName = aProfile.defApplName;
  this->profileName = aProfile.profileName;
  this->fProfileData->parentKey = aProfile.fProfileData->parentKey;
  this->fProfileData->accessMode = aProfile.fProfileData->accessMode;
  this->fProfileData->newConstrctr = aProfile.fProfileData->newConstrctr;
  this->fProfileData->hKey =
    IRegOpenProfile( this->profileName, &(this->fProfileData->isSystemProfile));
#endif

#ifdef IC_MOTIF
  /* Note: Ensure self assignment works */
  if(this->hIni != aProfile.hIni)
  {
    fProfileData->closeProfile(hIni);         // write DB to file if not system & user
    hIni = fProfileData->openProfile(aProfile.profileName);
    if (!hIni)
    {
       ITHROWGUIERROR("PrfOpenProfile");
    } /* endif */

    this->defApplName = aProfile.defApplName;
    this->profileName = aProfile.profileName;
  } /* endif */
#endif

  return *this;
}

/*------------------------------------------------------------------------------
| IProfile::name                                                               |
| Return profile name                                                          |
------------------------------------------------------------------------------*/
IString IProfile::name ( ) const
{
  return profileName;
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IProfile::handle                                                             |
| Return the profile handle                                                    |
------------------------------------------------------------------------------*/
IProfileHandle IProfile::handle() const
{
  return hIni;
}
#endif //IC_MOTIFPM

/*------------------------------------------------------------------------------
| IProfile::defaultApplicationName                                             |
| Return default application name                                              |
------------------------------------------------------------------------------*/
IString IProfile::defaultApplicationName ( ) const
{
  return defApplName;
}

/*------------------------------------------------------------------------------
| IProfile::setDefaultApplicationName                                          |
| Sets the default application name                                            |
------------------------------------------------------------------------------*/
IProfile& IProfile::setDefaultApplicationName ( const char *applName )
{
  defApplName = IString(applName);
  return *this;
}

/*------------------------------------------------------------------------------
| IProfile::~IProfile                                                          |
------------------------------------------------------------------------------*/
IProfile::~IProfile ( )
{
#ifdef IC_PM
  if ((hIni != HINI_PROFILE) &&
      (hIni != HINI_USERPROFILE) &&
      (hIni != HINI_SYSTEMPROFILE))
  {
    PrfCloseProfile(hIni);           // close file if not system & user
  } /* endif */
#endif

#ifdef IC_WIN
  RegCloseKey(fProfileData->hKey);
#endif

#ifdef IC_MOTIF
  fProfileData->closeProfile(hIni);           // write DB to file if not system & user
#endif

  delete fProfileData;
}

/*------------------------------------------------------------------------------
| IProfile::elementWithKey                                                     |
------------------------------------------------------------------------------*/
IString IProfile::elementWithKey ( const char *key, const char *applName )
{
  IMODTRACE_DEVELOP("IProfile::elementWithKey");
  IASSERTPARM(key != 0);

  if (!applName)
    IASSERTPARM(defApplName.length() > 0);

#ifdef IC_PM
  unsigned long ulLen = length(key, applName);

  if (ulLen > 0)
  {
    char* buffer = new char[ulLen + 1];
    /**************************/
    /* Query string first.    */
    /**************************/
    IGUIErrorInfo clear("Clear Last Error");
    PrfQueryProfileString( hIni,
           (applName) ? (PSZ)applName : (PSZ)defApplName,
           (PSZ)key, (PSZ)"", (PSZ)buffer, ulLen + 1);

    IGUIErrorInfo err("PrfQueryProfileString");
    if (err.errorId() != 0)
    {
      if (err.errorId() == PMERR_INVALID_ASCIIZ )
      {
        /**************************/
        /* Query binary data.     */
        /**************************/
        if (!PrfQueryProfileData( hIni,
               (applName) ? (PSZ)applName : (PSZ)defApplName,
               (PSZ)key, buffer, &ulLen + 1) )
        {
          ITHROWGUIERROR("PrfQueryProfileData");
        }
      }
      else
      {
        IInvalidRequest ex(err.text(), err.errorId(), IException::unrecoverable);
        ITHROW(ex);
      }
    }

    IString str(buffer, ulLen);
    delete [] buffer;
    return str;

  } /* endif */
  return IString();
#endif //IC_PM

#ifdef IC_WIN
  IString str;

  IString appName(applName);
  if (!applName)
    appName = defApplName;

  // Make sure the application name is ok
  if (!IRegTestKey( fProfileData->hKey, appName ))
  {
    ITHROWGUIERROR("RegOpenKey");
  }

  HKEY appKey = IRegOpenKey( fProfileData->hKey, appName );

  // Make sure the key name is ok
  if (!IRegTestKey( appKey, key ))
  {
    RegCloseKey(appKey);
    ITHROWGUIERROR("RegOpenKey");
  }

  HKEY dataKey = IRegOpenKey(appKey,key);

  // Close the application key
  RegCloseKey(appKey);

  // Get the value type
  unsigned long dataType = IRegDataType(dataKey);

  // Check if data is not in supported format
  if ( dataType == REG_NONE )
  {
    RegCloseKey(dataKey);
    ITHROWGUIERROR("RegQueryKey");
  }

  // Get data
  switch ( dataType )
  {
    case REG_SZ:
      str = IRegGetString(dataKey);
      break;
    case REG_DWORD:
    {
      long tmp = IRegGetInteger(dataKey);
      str = IString((char*)&tmp,sizeof(tmp));
      break;
    }
    case REG_BINARY:
      str = IRegGetBinary(dataKey);
      break;
  }

  // Close the key and return
  RegCloseKey(dataKey);
  return str;
#endif //IC_WIN

#ifdef IC_MOTIF

  IString theKey(key);
  IString theApplName;
  if (applName)
     theApplName = applName;
  else
     theApplName = defApplName;

  if (!applName)
  {
      IASSERTPARM(defApplName.length() > 0);
  } /* endif */

  unsigned long ulLen = length(theKey, theApplName);

  if (ulLen > 0)
  {
    char* buffer = new char[ulLen + 1];

//  First close the profile and then reopen it to ensure that any other 
//  information updated by an app sharing the file is the latest.    

    fProfileData->closeProfile(hIni);
    hIni = fProfileData->openProfile(profileName);

    /**************************/
    /* Query string first.    */
    /**************************/

    fProfileData->queryProfileString(hIni, theApplName,
               theKey, buffer, ulLen + 1);
    IString str(buffer, ulLen);

// Check for special mark written to indicate a deleted record
   if ((buffer[0] == '~') && (buffer[1] == '/') && (buffer[2] == '~'))
   {
     ITHROWGUIERROR("PrfQueryProfileData");
   } 

    // If this data was written out in IOC's binary format, convert back to original format
    if ((buffer[0] == '~') && (buffer[1] =='~')) {
       str.stripLeading("~~");
       str.x2c();
    }
    delete buffer;
    return str;

  } /* endif */
  return IString();

#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| IProfile::integerWithKey                                                     |
------------------------------------------------------------------------------*/
long IProfile::integerWithKey ( const char *key, const char *applName )
{
  IMODTRACE_DEVELOP("IProfile::integerWithKey");
  IASSERTPARM( key != 0 );
  if (!applName)
    IASSERTPARM( defApplName.length() > 0 );

  long lResult;

#ifdef IC_PM
  unsigned long resultSize = sizeof(lResult);
  if (!PrfQueryProfileData( hIni,
             (applName) ? (PSZ)applName : (PSZ)defApplName,
             (PSZ)key, (PVOID)&lResult, &resultSize))
  {
    ITHROWGUIERROR("PrfQueryProfileData");
  }
#endif //IC_PM

#ifdef IC_WIN
  IString appName(applName);
  if (!applName)
    appName = defApplName;

  // Make sure the application name is ok
  if (!IRegTestKey( fProfileData->hKey, appName ))
  {
    ITHROWGUIERROR("RegOpenKey");
  }

  HKEY appKey = IRegOpenKey( fProfileData->hKey, appName );

  // Make sure the key name is ok
  if (!IRegTestKey( appKey, key ))
  {
    RegCloseKey(appKey);
    ITHROWGUIERROR("RegOpenKey");
  }

  HKEY dataKey = IRegOpenKey(appKey,key);

  // Close the application key
  RegCloseKey(appKey);

  // Get the value type
  unsigned long dataType = IRegDataType(dataKey);

  // Check if data is not in supported format
  if ( dataType != REG_DWORD )
  {
    RegCloseKey(dataKey);
    ITHROWGUIERROR("RegQueryKey");
  }

  // Get data
  lResult = IRegGetInteger(dataKey);

  // Close the key
  RegCloseKey(dataKey);
#endif //IC_WIN

#ifdef IC_MOTIF
   IString theKey(key);
   IString theApplName;
   if (applName)
     theApplName = applName;
   else
     theApplName = defApplName;

//  First close the profile and then reopen it to ensure that any other 
//  information updated by an app sharing the file is the latest.    
   fProfileData->closeProfile(hIni);
   hIni = fProfileData->openProfile(profileName);
   fProfileData->queryProfileData(hIni, theApplName, theKey, &lResult);
#endif //IC_MOTIF

  return lResult;
}


/*------------------------------------------------------------------------------
| IProfile::addOrReplaceElementWithKey                                         |
|   Write string data to file.                                                 |
------------------------------------------------------------------------------*/
IProfile& IProfile::addOrReplaceElementWithKey( const char *key,
                                                const IString& data,
                                                const char *applName)
{
  IMODTRACE_DEVELOP("IProfile::addOrReplaceElementWithKey");
  IASSERTPARM(key != 0);
  if (!applName) {
     IASSERTPARM(defApplName.length() > 0);
  } /* endif */

#ifdef IC_PM
  if (!PrfWriteProfileData(hIni,
                           (applName) ? (PSZ)applName : (PSZ)defApplName,
                           (PSZ)key, (PSZ)data, data.length() )) {
     ITHROWGUIERROR("PrfWriteProfileString");
  } /* endif */
#endif //IC_PM

#ifdef IC_WIN
  IString appName(applName);
  if (!applName)
    appName = defApplName;

  // Open the application key
  HKEY appKey = IRegOpenKey( fProfileData->hKey, appName );

  // If the value already has data then delete it
  if (IRegTestKey( appKey, key ))
  {
    RegCloseKey(appKey);
    deleteElementWithKey( key, appName );
    appKey = IRegOpenKey( fProfileData->hKey, appName );
  }

  HKEY dataKey = IRegOpenKey(appKey,key);
  RegCloseKey(appKey);

  // Set the key value
  if (data.isPrintable())
  {
    IRegWriteString(dataKey,data);
  }
  else
  {
    IRegWriteBinary(dataKey,data);
  }

  RegCloseKey(dataKey);
#endif //IC_WIN

#ifdef IC_MOTIF
  IString theKey(key);
  IString theApplName;
  if (applName)
    theApplName = applName;
  else
    theApplName = defApplName;

  IString cdata(data);
  if (!data.isPrintable())
     cdata = "~~" + cdata.c2x();

//  First close the profile and then reopen it to ensure that any other 
//  information updated by an app sharing the file is the latest.  Lock
//  the file during write to protect against simultaneous updates.    
  fProfileData->closeProfile(hIni);
  fProfileData->profileRes->lock(-1);
  hIni = fProfileData->openProfile(profileName);
  fProfileData->writeProfileString(hIni, theApplName,
               theKey, (char *)cdata, cdata.length() );
  fProfileData->profileRes->unlock();
#endif //IC_MOTIF

  return *this;

}

/*------------------------------------------------------------------------------
| IProfile::addOrReplaceElementWithKey                                         |
|   Write binary data to file.                                                 |
------------------------------------------------------------------------------*/
IProfile& IProfile::addOrReplaceElementWithKey
                            ( const char *key, long data, const char *applName)
{
  IMODTRACE_DEVELOP("IProfile::addOrReplaceElementWithKey");
  IASSERTPARM( key != 0 );
  if (!applName)
    IASSERTPARM( defApplName.length() > 0 );

#ifdef IC_PM
  if (!PrfWriteProfileData( hIni, (applName) ? (PSZ)applName :
          (PSZ)defApplName, (PSZ)key, (void*)&data, sizeof(data)))
  {
    ITHROWGUIERROR("PrfWriteProfileData");
  } /* endif */
#endif //IC_PM

#ifdef IC_WIN
  IString appName(applName);
  if (!applName)
    appName = defApplName;

  // Open the application key
  HKEY appKey = IRegOpenKey( fProfileData->hKey, appName );

  // If the value already has data then delete it
  if (IRegTestKey( appKey, key ))
  {
    RegCloseKey(appKey);
    deleteElementWithKey( key, appName );
    appKey = IRegOpenKey( fProfileData->hKey, appName );
  }

  HKEY dataKey = IRegOpenKey(appKey,key);
  RegCloseKey(appKey);

  // Set the key value
  IRegWriteInteger( dataKey, data );

  RegCloseKey(dataKey);
#endif //IC_WIN

#ifdef IC_MOTIF
  IString theKey(key);
  IString theApplName;
  if (applName)
    theApplName = applName;
  else
    theApplName = defApplName;

  long  lData = data;
  unsigned long dataSize = sizeof(long);
//  First close the profile and then reopen it to ensure that any other 
//  information updated by an app sharing the file is the latest.  Lock
//  the file during write to protect against simultaneous updates.    
  fProfileData->closeProfile(hIni);
  fProfileData->profileRes->lock(-1);
  hIni = fProfileData->openProfile(profileName);
  fProfileData->writeProfileData(hIni, theApplName, theKey, &lData);
  fProfileData->profileRes->unlock();
#endif //IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| IProfile::deleteElementWithKey                                               |
------------------------------------------------------------------------------*/
IProfile& IProfile::deleteElementWithKey ( const char *key,
                                           const char *applName)
{
  IMODTRACE_DEVELOP("IProfile::deleteElementWithKey");
  IASSERTPARM(key != 0);
  if (!applName) {
     IASSERTPARM(defApplName.length() > 0);
  } /* endif */

#ifdef IC_PM
  if (!PrfWriteProfileString( hIni,
       (applName) ? (PSZ)applName : (PSZ)defApplName, (PSZ)key, 0) )
    ITHROWGUIERROR("PrfWriteProfileString");
#endif //IC_PM

#ifdef IC_WIN
  IString appName(applName);
  if (!applName)
    appName = defApplName;

  // Make sure the application name is ok
  if (!IRegTestKey( fProfileData->hKey, appName ))
    ITHROWGUIERROR("RegOpenKey");

  // Open the application
  HKEY appKey = IRegOpenKey( fProfileData->hKey, appName );

  // Delete the key value
  IRegDeleteKey( appKey, key );

  // Check if there is another subkey
  char          buffer[MAX_PATH] = "";
  unsigned long keys = 0;
  if ( RegEnumKey ( appKey, 0, buffer, MAX_PATH ) == ERROR_SUCCESS )
    keys = 1;

  // Close open application key
  RegCloseKey( appKey );

  // If no other data then delete the application
  if ( !keys )
    IRegDeleteKey( fProfileData->hKey, appName );

#endif //IC_WIN

#ifdef IC_MOTIF
  IString theKey(key);
  IString theApplName;
  if (applName)
    theApplName = applName;
  else
    theApplName = defApplName;
//  First close the profile and then reopen it to ensure that any other 
//  information updated by an app sharing the file is the latest.  Lock
//  the file during write to protect against simultaneous updates.    
  fProfileData->closeProfile(hIni);
  fProfileData->profileRes->lock(-1);
  hIni = fProfileData->openProfile(profileName);
  fProfileData->deleteProfileString(hIni, theApplName, theKey);
  fProfileData->profileRes->unlock();
#endif //IC_MOTIF

  return *this;
}


/*------------------------------------------------------------------------------
| IProfile::deleteElementWithApplication                                       |
------------------------------------------------------------------------------*/
IProfile& IProfile::deleteElementWithApplication ( const char *applName)
{
  IMODTRACE_DEVELOP("IProfile::deleteElementWithApplication");
  if (!applName) {
     IASSERTPARM(defApplName.length() > 0);
  } /* endif */

#ifdef IC_PM
  if (!PrfWriteProfileString( hIni,
                 (applName) ? (PSZ)applName : (PSZ)defApplName, 0, 0 ))
  {
    ITHROWGUIERROR("PrfWriteProfileString");
  } /* endif */
#endif //IC_PM

#ifdef IC_WIN
  IProfile::Cursor cursor(*this,applName);
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
  {
    IString subkey = applicationOrKeyAt(cursor);
    deleteElementWithKey(subkey,applName);
  }
#endif //IC_WIN

#ifdef IC_MOTIF
  IProfile::Cursor cursor((IProfile&)*this,
                          (applName) ? (char *)applName : (char *)defApplName );

   unsigned long
     result = 0;

   for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() ) {
     deleteElementWithKey((char *)applicationOrKeyAt(cursor),
       (applName) ? (char *)applName : (char *)defApplName );
   } /* endfor */

//  First close the profile and then reopen it to ensure that any other 
//  information updated by an app sharing the file is the latest.  Lock
//  the file during write to protect against simultaneous updates.    
  fProfileData->closeProfile(hIni);
  fProfileData->profileRes->lock(-1);
  hIni = fProfileData->openProfile(profileName);
   fProfileData->deleteProfileAppl(hIni,
     (applName) ? (char *)applName : (char *)defApplName );
  fProfileData->profileRes->unlock();
#endif //IC_MOTIF

  return *this;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IProfile::deleteProfile                                                      |
------------------------------------------------------------------------------*/
void IProfile::deleteProfile ( const char* profileName )
{
  // Do not allow the user or system profiles to be removed
  if (( IProfile::systemProfile().name() == profileName ) ||
      ( IProfile::userProfile().name()   == profileName ))
  {
    return;
  }

#ifdef IC_WIN
  // First delete any applications
  {
    IString subKey;
    IProfile tmpProfile( profileName );
    IProfile::Cursor cursor(tmpProfile);
    for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
    {
      subKey = tmpProfile.applicationOrKeyAt(cursor);
      tmpProfile.deleteElementWithApplication(subKey);
    }
  }
  // Now we can delete the profile
  HKEY softwareKey = IRegOpenKey(HKEY_LOCAL_MACHINE,"Software");
  IRegDeleteKey( softwareKey, profileName );
  RegCloseKey( softwareKey );
#endif

#ifndef IC_WIN
  // Delete the profile file
  remove(profileName);
#endif
}

/*------------------------------------------------------------------------------
| IProfile::enableKeyMapping - Enables the mapping of specific characters      |
|   that are invalid for a Win32 registry entry as a key to valid chars.       |
------------------------------------------------------------------------------*/
void IProfile::enableKeyMapping( bool mapKey )
{
  fgMapKey = mapKey;
}

/*------------------------------------------------------------------------------
| IProfile::disableKeyMapping - Disables the mapping of specific characters    |
|   that are invalid for a Win32 registry entry as a key to valid chars.       |
------------------------------------------------------------------------------*/
void IProfile::disableKeyMapping( )
{
  enableKeyMapping( false );
}

/*------------------------------------------------------------------------------
| IProfile::isKeyMappingEnabled - Returns the enablement state of mapping of   |
|   specific characters that are invalid for a Win32 registry entry as a key.  |
------------------------------------------------------------------------------*/
bool IProfile::isKeyMappingEnabled( )
{
  return ( fgMapKey );
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IProfile::numberOfApplications                                               |
------------------------------------------------------------------------------*/
unsigned long IProfile::numberOfApplications () const
{
  IMODTRACE_DEVELOP("IProfile::numberOfApplications");

  unsigned long
    result = 0;

  IProfile::Cursor cursor((IProfile&)*this);

  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
    result++;

  return result;
}

/*------------------------------------------------------------------------------
| IProfile::numberOfKeys                                                       |
------------------------------------------------------------------------------*/
unsigned long IProfile::numberOfKeys ( const char *applName) const
{
  IMODTRACE_DEVELOP("IProfile::numberOfKeys");
  if (!applName)
    IASSERTPARM( defApplName.length() > 0 );

  IProfile::Cursor cursor((IProfile&)*this,
                          (applName) ? (char *)applName : (char *)defApplName );

  unsigned long
    result = 0;

  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
    result++;

  return result;
}

/*------------------------------------------------------------------------------
| IProfile::containsApplication                                                |
------------------------------------------------------------------------------*/
bool IProfile::containsApplication ( const char *applName ) const
{
  IMODTRACE_DEVELOP("IProfile::containsApplication");
  IASSERTPARM(applName != 0);

  IProfile::Cursor cursor((IProfile&)*this);

  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
#ifdef IC_WIN
    IString currentApp(applicationOrKeyAt(cursor));
    IString applNameU = IRegFixKey(applName);
    if (currentApp.upperCase() == applNameU.upperCase())
      return true;
#endif
#ifdef IC_MOTIFPM
    if (applicationOrKeyAt(cursor) == applName)
      return true;
#endif
  } /* endfor */
  return false;
}

/*------------------------------------------------------------------------------
| IProfile::containsKeyName                                                    |
------------------------------------------------------------------------------*/
bool IProfile::containsKeyName( const char *key,
                                const char *applName )
{
  IMODTRACE_DEVELOP("IProfile::containsKeyName");

#ifdef IC_PMWIN
  IProfile::Cursor cursor((IProfile&)*this,
                          (applName) ? applName : (const char*)defApplName );

  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
#ifdef IC_WIN
    IString currentKey(applicationOrKeyAt(cursor));
    IString keyNameU = IRegFixKey(key);
    if (currentKey.upperCase() == keyNameU.upperCase())
      return true;
#endif //IC_WIN
#ifdef IC_PM
    if (applicationOrKeyAt(cursor) == key)
    {
      try 
        {   
          IString buff = elementWithKey(key, applName);
        }
      catch(...)
        {
          return false;
        }
      return true;
    }
#endif //IC_PM
  } /* endfor */
  return false;
#endif //IC_PMWIN
#ifdef IC_MOTIF
      try 
        {   
          IString buff = elementWithKey(key, applName);
          if (buff == IString()) return false;
        }
      catch(...)
        {
          return false;
        }
      return true;
#endif //IC_MOTIF
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IProfile::length                                                             |
|   Private function to return the length of the profile data                  |
------------------------------------------------------------------------------*/
unsigned long IProfile::length(const char *keyName, const char *applName) const
{
   unsigned long ulSize = 0;

#ifdef IC_PM
   if (PrfQueryProfileSize( hIni,
              (applName) ? (PSZ)applName : (PSZ)defApplName,
              (PSZ)keyName, (unsigned long *) &ulSize))
   {
     return ulSize;
   }
   else
     ITHROWGUIERROR("PrfQueryProfileSize");
#endif //IC_PM

#ifdef IC_MOTIF
   IString theKey(keyName);
   IString theApplName;
   if (applName)
     theApplName = applName;
   else
     theApplName = defApplName;

   ulSize = fProfileData->queryProfileSize(hIni, theApplName, theKey);
#endif //IC_MOTIF

   return ulSize;
}
#endif //IC_MOTIFPM

/*------------------------------------------------------------------------------
| IProfile::applicationOrKeyAt                                                 |
------------------------------------------------------------------------------*/
IString IProfile::applicationOrKeyAt ( const Cursor &cursor ) const
{
  IMODTRACE_DEVELOP("IProfile::applicationOrKeyAt");
  IASSERTPARM(cursor.isValid());

  // cursor pos marks start of app/key name
  return IString( ((char*)cursor.strCl) + cursor.pos - 1 );
}

/*------------------------------------------------------------------------------
| IProfile::Cursor::~Cursor                                                    |
------------------------------------------------------------------------------*/
IProfile::Cursor::~Cursor ( )
{
  delete fCursorData;
}

/*------------------------------------------------------------------------------
| IProfile::Cursor::Cursor                                                     |
------------------------------------------------------------------------------*/
IProfile::Cursor::Cursor ( IProfile  &profile )
#ifdef IC_PMWIN
  : pos(1), strCl(), applNameCl(), fCursorData(new IProfileCursorData())
#else
  : pos(1), strCl(), applNameCl(), fCursorData(new IProfileCursorData(this))
#endif
{
  IMODTRACE_DEVELOP("IProfile::Cursor::ctor1");
#ifdef IC_MOTIF
  fCursorData->thisProfile = &profile;
#endif
#ifdef IC_MOTIFPM
  initCursor( profile.handle(), 0 );
#endif
#ifdef IC_WIN
  fCursorData->thisProfile = &profile;
  initCursor( profile.name(), NULL );
#endif

}

/*------------------------------------------------------------------------------
| IProfile::Cursor::Cursor                                                     |
------------------------------------------------------------------------------*/
IProfile::Cursor:: Cursor ( IProfile  &profile, const char *applName)
#ifdef IC_PMWIN
  : pos(1), strCl(), fCursorData(new IProfileCursorData())
#else
  : pos(1), strCl(), fCursorData(new IProfileCursorData(this))
#endif
{
  IMODTRACE_DEVELOP("IProfile::Cursor::ctor2");

  if (applName)
    applNameCl = applName;
  else
    applNameCl = profile.defaultApplicationName();

#ifdef IC_MOTIF
  fCursorData->thisProfile = &profile;
#endif
#ifdef IC_MOTIFPM
  initCursor( profile.handle(), applNameCl );
#endif
#ifdef IC_WIN
  fCursorData->thisProfile = &profile;
  initCursor( profile.name(), applNameCl );
#endif

}

/*------------------------------------------------------------------------------
| IProfile::Cursor::setToFirst                                                 |
| Have the cursor point to the first element                                   |
------------------------------------------------------------------------------*/
bool IProfile::Cursor :: setToFirst ()
{
  this->pos = 1;
#ifdef IC_MOTIF
// Check for a deleted item
  try
  {
     if (applNameCl.length())
     {
        fCursorData->thisProfile->elementWithKey(
                   fCursorData->thisProfile->applicationOrKeyAt(*this),
                   applNameCl);
     }
     else
     {
        fCursorData->thisProfile->applicationOrKeyAt(*this);
     }
  }
  catch(...)
  {
     if ((this->pos < this->strCl.length()) && (this->isValid()))
           return this->setToNext();
     else return false;
  }
#endif
  // Valid if not off end...
  return this->pos < this->strCl.length();
}

/*------------------------------------------------------------------------------
| IProfile::Cursor::setToNext                                                  |
| Have the cursor point to the next element                                    |
------------------------------------------------------------------------------*/
bool IProfile::Cursor :: setToNext ()
{
  // Skip to next null...
  this->pos = this->strCl.indexOf( '\0', this->pos );
  // If not found, set position off end...
  if ( this->pos == 0 )
    this->pos = this->strCl.length() + 1;
  // Else, point to next byte...
  else
    this->pos++;
  // Valid if not off end...
#ifdef IC_MOTIF
// Check for a deleted item
  try
  {
     if (applNameCl.length())
     {
        fCursorData->thisProfile->elementWithKey(
                   fCursorData->thisProfile->applicationOrKeyAt(*this),
                   applNameCl);
     }
     else
     {
        fCursorData->thisProfile->applicationOrKeyAt(*this);
     }
  }
  catch(...)
  {
     if ((this->pos < this->strCl.length()) && (this->isValid()))
           return this->setToNext();
     else return false;
  }
#endif
  return this->pos < this->strCl.length();
}

/*------------------------------------------------------------------------------
| IProfile::Cursor::setToPrevious                                              |
| Have the cursor point to the previous element                                |
------------------------------------------------------------------------------*/
bool IProfile::Cursor :: setToPrevious ()
{
  // If not at start, back up to previous null...
  if ( this->pos > 1 && this->pos < this->strCl.length() )
    this->pos = this->strCl.lastIndexOf( '\0', this->pos - 2 );
  // Else, leave invalid...
  else
    this->pos = this->strCl.length();
  // If not found, then we must now be at start...
  if ( this->pos == 0 )
    this->pos = 1;
  // Else, position at byte past the null...
  else
    this->pos++;

  // Valid if not off end...
#ifdef IC_MOTIF
// Check for a deleted item
  try
  {
     if (applNameCl.length())
     {
        fCursorData->thisProfile->elementWithKey(
                   fCursorData->thisProfile->applicationOrKeyAt(*this),
                   applNameCl);
     }
     else
     {
        fCursorData->thisProfile->applicationOrKeyAt(*this);
     }
  }
  catch(...)
  {
     if ((this->pos < this->strCl.length()) && (this->isValid()))
           return this->setToPrevious();
     else return false;
  }
#endif

  return this->pos < this->strCl.length();
}

/*------------------------------------------------------------------------------
| IProfile::Cursor::setToLast                                                  |
| Have the cursor point to the last element                                    |
------------------------------------------------------------------------------*/
bool IProfile::Cursor :: setToLast ()
{
  // Position at last null (before the terminating one).
  this->pos = this->strCl.lastIndexOf( '\0', this->strCl.length() - 2 );
  // Advance to next byte.
  this->pos++;

  // Valid if not off end...
#ifdef IC_MOTIF
// Check for a deleted item
  try
  {
     if (applNameCl.length())
     {
        fCursorData->thisProfile->elementWithKey(
                   fCursorData->thisProfile->applicationOrKeyAt(*this),
                   applNameCl);
     }
     else
     {
        fCursorData->thisProfile->applicationOrKeyAt(*this);
     }
  }
  catch(...)
  {
     if ((this->pos < this->strCl.length()) && (this->isValid()))
           return this->setToPrevious();
     else return false;
  }
#endif
  return this->pos < this->strCl.length();
}

/*------------------------------------------------------------------------------
| IProfile::Cursor :: isValid                                                  |
------------------------------------------------------------------------------*/
bool IProfile::Cursor :: isValid (bool bCheckFile) const
{
  if ( this->pos >= this->strCl.length() )
    return false;
  else
  {
#ifdef IC_PMWIN
    if (bCheckFile)
    {
      const char* p = ( applNameCl.length() == 0 ) ?
                              (char*)0 : (char*)applNameCl;
#ifdef IC_PM
      unsigned long ulLen = 0;
      PrfQueryProfileSize(hProfileCl, p, 0, &ulLen);
      IString current( 0, ulLen );
      if (ulLen)
      {
        PrfQueryProfileString( hProfileCl, p, 0, "",
                               (char *)current, ulLen+1 );
      }
      return (current == strCl);
#endif
#ifdef IC_WIN
      // Create another cursor and see if the data strings are the same
      IProfile tmpProfile( fCursorData->profileName );
      IProfile::Cursor cursor(tmpProfile,p);
      return (strCl == cursor.strCl);
#endif
    }
  else
#endif //IC_PMWIN
      return true;
  } /* endif */
}

/*------------------------------------------------------------------------------
| IProfile::Cursor::invalidate                                                 |
| Invalidate the cursor                                                        |
------------------------------------------------------------------------------*/
void IProfile::Cursor :: invalidate ()
{
  this->pos = this->strCl.length();
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IProfile::Cursor::initCursor                                                 |
|   Read profile for all application names or key names                        |
|                                                                              |
|   On exit, strCl will contain:                                               |
|      | entry1 | 0 | entry2 | 0 | ... | entry n | 0 | 0 |                     |
|                                                      A                       |
|                  Terminating null of the IString -----                       |
------------------------------------------------------------------------------*/
void IProfile::Cursor::initCursor( const IProfileHandle& hProfile,
                                   const char *applName )
{
  IMODTRACE_DEVELOP("IProfile::Cursor::initCursor");

#ifdef IC_PM
  hProfileCl = hProfile;
  unsigned long ulLen;

  PrfQueryProfileSize( hProfile, applName, 0, &ulLen );

  this->strCl = IString( 0, ulLen );

  if (ulLen)
    PrfQueryProfileString( hProfile, applName, 0, "",
                           (char*)strCl, ulLen+1 );
  else
    strCl = "";

#endif //IC_PM
#ifdef IC_MOTIF
  IString apName(applName);
  strCl = fCursorData->initProfileCursor(hProfile, apName);
#endif //IC_MOTIF

}
#endif //IC_MOTIFPM

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IProfile::Cursor::initCursor                                                 |
|   Read profile for all application names or key names                        |
|                                                                              |
|   On exit, strCl will contain:                                               |
|      | entry1 | 0 | entry2 | 0 | ... | entry n | 0 | 0 |                     |
|                                                      A                       |
|                  Terminating null of the IString -----                       |
------------------------------------------------------------------------------*/
void IProfile::Cursor::initCursor( const char * profileName,
                                   const char * applName )
{
  IMODTRACE_DEVELOP("IProfile::Cursor::initCursor");

  fCursorData->profileName = profileName;
  strCl = "";

  // Open the profile
  HKEY hKeyProfile, tempKey;
  unsigned long rc;
  if (fCursorData->thisProfile->fProfileData->newConstrctr){
      hKeyProfile = 0;
  		rc = IRegOpenOrCreateKey( 
                  fCursorData->thisProfile->fProfileData->parentKey,
					   "Software", &tempKey, false);
      if ( rc == ERROR_SUCCESS ){
			if( IRegOpenOrCreateKey( tempKey, profileName, 
			               &hKeyProfile, false) == ERROR_SUCCESS){
  				RegCloseKey( tempKey );
			}
			else {
  				RegCloseKey( tempKey );
				//ITHROWGUIERROR("IRegOpenOrCreateKey");
			}
      }
		else {
			//ITHROWGUIERROR("IRegOpenOrCreateKey");
		}
  }
  else{
  	 hKeyProfile = IRegOpenProfile( profileName,
                                      &(fCursorData->isSystemProfile) );
  }
  
  HKEY enumKey = 0;
  if ( applName != NULL )
  {
    if ( IRegTestKey ( hKeyProfile, applName ))
      enumKey = IRegOpenKey( hKeyProfile, applName );
  }
  else
    enumKey = hKeyProfile;

  if ( enumKey )
  {
    char          buffer[MAX_PATH] = "";
    unsigned long key;
    for ( key = 0;
          RegEnumKey ( enumKey, key, buffer, MAX_PATH ) == ERROR_SUCCESS;
          key++ )
    {
      // Construct a temporary string for the current key
      bool invalid = false;
      IString keyName(buffer);

      // If translate characters on, restore characters converted
      if (IProfile::isKeyMappingEnabled())
      {
        unsigned long tmp = keyName.indexOf("/");
        if ( tmp != 0 )
        {
          do
          {
            tmp++;
            switch ( keyName[tmp] )
            {
              case '_':
                keyName[tmp] = '\\';
                break;
              case '0':
                keyName[tmp] = '?';
                break;
              case '1':
                keyName[tmp] = '*';
                break;
              case '/':
                break;
              default:
                invalid = true;
            }
            tmp = keyName.indexOf("/",tmp+1);
          }
          while ( tmp != 0 );
          keyName.change("/\\","\\");
          keyName.change("/*","*");
          keyName.change("/?","?");
          keyName.change("//","/");
        }
      }
      // If application name passed
      if ( applName != NULL )
      {
        // Verify that the name is valid before adding to string
        HKEY dataKey = IRegOpenKey( enumKey, keyName );
        unsigned long dataType = IRegDataType(dataKey);
        RegCloseKey(dataKey);
        if ( dataType == REG_NONE )
          invalid = true;
      }
      if ( !invalid )
      {
        strCl += keyName;
        strCl += IString('\0');
      }
    }
  }

  if ( applName != NULL )
  {
    if ( enumKey )
      RegCloseKey(enumKey);
  }
  // Close the key for the profile
  RegCloseKey(hKeyProfile);

}
#endif  // IC_WIN

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IProfileData::IProfileData                                                   |
|                                                                              |
| This class contains private data used by IProfile                            |
------------------------------------------------------------------------------*/
IProfileData::IProfileData()
#ifdef IC_WIN
  : isSystemProfile(false), defaultString("IProfile-default string"), hKey(0),
    parentKey(0), accessMode(), newConstrctr(0)
#endif
{
  IMODTRACE_DEVELOP("IProfileData::IProfileData::ctor");
}

/*------------------------------------------------------------------------------
| IProfileData::~IProfileData                                                  |
------------------------------------------------------------------------------*/
IProfileData::~IProfileData()
{}

/*------------------------------------------------------------------------------
| IProfileCursorData::IProfileCursorData                                       |
|                                                                              |
| This class contains private data used by IProfile::Cursor                    |
------------------------------------------------------------------------------*/
IProfileCursorData::IProfileCursorData()
#ifdef IC_WIN
  : profileName(), isSystemProfile(false),
    defaultString("IProfile-default string")
#endif
{
  IMODTRACE_DEVELOP("IProfileCursorData::IProfileCursorData::ctor");
}

/*------------------------------------------------------------------------------
| IProfileCursorData::~IProfileCursorData                                      |
------------------------------------------------------------------------------*/
IProfileCursorData::~IProfileCursorData()
{}
#endif //IC_PMWIN


#ifdef IC_MOTIF
Bool _System iprofileXlibCallback( XrmDatabase*,
                           XrmBindingList,
                           XrmQuarkList,
                           XrmRepresentation*,
                           XrmValue*,
                           XPointer);

/*------------------------------------------------------------------------------
   iprofilmXlibCallback is the callback function for Xlib XrmEnumerateDatabase.
   This callback function will be called for each entry in the specified
   database allowing the ICL IProfile::Cursor to be implemented.
------------------------------------------------------------------------------*/
extern Bool _System iprofileXlibCallback(XrmDatabase*        pDB,
                                         XrmBindingList      bList,
                                         XrmQuarkList        qList,
                                         XrmRepresentation*  typeList,
                                         XrmValue*           value,
                                         XPointer            bufOfIStrsPtr)
{
   IMODTRACE_ALL("iprofileXlibCallback");
   XrmValue localValue = *value;
   if (
//  Removed to fix defect 27331 
//  (*typeList == XrmQString) && 
       (localValue.size) )
   {
     char *pos;
     XPointer bufOfIStrs;
     int  ch = '\0';
     /*************************************************************************/
     /* Increment quark list past the applName to the actual key name         */
     /*************************************************************************/
     qList++;
     char *keyName = XrmQuarkToString((XrmQuark)*qList);
     int i = 0;
     bufOfIStrs = *(XPointer *)bufOfIStrsPtr;
     pos = bufOfIStrs;
     /*************************************************************************/
     /* Set pos to end of the buffer of IStrings; the current keyName will    */
     /* be added to the buffer at this position.                              */
     /*************************************************************************/
     if (strlen(bufOfIStrs)) {
        pos = (strchr(bufOfIStrs, ch)) + 1;
        i = (pos - bufOfIStrs);
        while ( (i < strBufferSize) && (*(pos) != '\0') ) {
          pos = (strchr(pos, ch)) + 1;
          i = (pos - bufOfIStrs);
        }
     }
     /*************************************************************************/
     /* Expand buffer if necessary.                                           */
     /*************************************************************************/
     if (i+strlen(keyName)+2 > strBufferSize) {
         char *newBuffer;
         strBufferSize += max(i+strlen(keyName)+2-strBufferSize,STRBUFFERSIZE);
         newBuffer = new char[strBufferSize];
         memcpy(newBuffer, (char *)bufOfIStrs, i);
         memset(newBuffer+i, '\0', strBufferSize-i);
         delete bufOfIStrs;
         *(XPointer *)bufOfIStrsPtr = newBuffer;
         bufOfIStrs = newBuffer;
     }
     /*************************************************************************/
     /* Append key name to string of IStrings                                 */
     /*************************************************************************/
     char *p = bufOfIStrs + i;
     memmove(p, keyName, strlen(keyName));
   }
   return false;
}

/*------------------------------------------------------------------------------
| IProfileData::IProfileData                                                   |
------------------------------------------------------------------------------*/
IProfileData :: IProfileData ( IProfile* profile )  :
        pProfile(profile), profileRes(0)
   {
     int ix, ix2;
      /*
       * construct a simple pathname for ISharedResource (so the
       * file created to fabricate a key is created under /tmp).
       */
      IString profileResString("");
      ix2 = profile->profileName.size();
      if (profile->profileName[ix2] == '/') {
          --ix2;
      }
      for (ix = ix2; ix >= 1; ix--) {
          if (profile->profileName[ix] == '/') {
              break;
          }
      }
      ++ix;
      profileResString = profile->profileName.subString(ix, (ix2 - ix + 1));
      profileResString = "__" + profileResString;
      profileRes = new ISharedResource(profileResString);
   }

/*------------------------------------------------------------------------------
| IProfileData::~IProfileData                                                  |
------------------------------------------------------------------------------*/
IProfileData :: ~IProfileData ( )
   {
       if (profileRes) {
           delete profileRes;
           profileRes = 0;
       }
   }

/*------------------------------------------------------------------------------
| IProfileData::openProfile                                                    |
------------------------------------------------------------------------------*/
IProfileHandle IProfileData :: openProfile(const IString profName)
{
   IProfileHandle pDB;
   FILE * hFile;

   theProfile = profName;

   /*************************************************************************/
   /* Read profile database.                                                */
   /*************************************************************************/
   if ( (pDB = XrmGetFileDatabase((char *)theProfile)) == (IProfileHandle)0) {
     /***********************************************************************/
     /* Create empty profile.                                               */
     /***********************************************************************/
     if ( (hFile = fopen(theProfile, iprofOpenFlag)) == (FILE *)0) {
       ITHROWCLIBERROR("fopen",
                        IBaseErrorInfo::accessError,
                        IException::recoverable);
     }
     else {
       /*********************************************************************/
       /* Close empty file, then create empty database so that an           */
       /* IProfileHandle exists.  Entries can be added to an empty          */
       /* database.                                                         */
       /*********************************************************************/
       fflush(hFile);
       fclose(hFile);
       pDB = XrmGetFileDatabase((char *)theProfile);
     }    // end else
   }   // endif
   /*********************************************************************/
   /* Set the application class name                                    */
   /*********************************************************************/
   char *iname;
   char *cname;
   if (!(IThread::current().isGUIInitialized()))
   {
      // Initialize GUI...
      IThread::current().initializeGUI();
   }
   XtGetApplicationNameAndClass(XtDisplay(IThread::current().applicationShell()),
     &iname, &cname);
   applClass = cname;
   return pDB;
}

/*------------------------------------------------------------------------------
| IProfileData::closeProfile                                                   |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: closeProfile(const IProfileHandle pDB)
{
   /*************************************************************************/
   /* Write database to profile file.                                       */
   /*************************************************************************/
   if (pDB == pProfile->handle()) {
     XrmDestroyDatabase(pDB);
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::queryProfileString                                             |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: queryProfileString(const IProfileHandle pDB,
                          const IString applName, const IString key, char * strBuf,
                          unsigned long strBufLen )
{
   char *returnType;
   XrmValue returnStruct;
   IString pfKey(applName);
   pfKey+=iprofDot;
   pfKey+=key;
   /*************************************************************************/
   /* Retrieve keyed string value from database into XrmValue structure.    */
   /* Then copy returned string value into strBuf.                          */
   /*************************************************************************/
   XrmGetResource (pDB, (char *)pfKey, (char *)applClass, &returnType,
        &returnStruct);
   if (returnStruct.size)
     strncpy(strBuf, returnStruct.addr, returnStruct.size - 1);
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::queryProfiledata                                               |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: queryProfileData(const IProfileHandle pDB,
                          const IString applName, const IString key, long * lData)
{
   char *returnType;
   XrmValue returnStruct;
   IString longStrBuf;
   IString pfKey(applName);
   pfKey+=iprofDot;
   pfKey+=key;
   /*************************************************************************/
   /* Retrieve keyed string (long) value from database into XrmValue        */
   /* structure.  Then copy returned string (long) value into longStrBuf.   */
   /*************************************************************************/
   XrmGetResource (pDB, (char *)pfKey, (char *)applClass, &returnType,
        &returnStruct);
   if ( (returnStruct.size) && (*returnStruct.addr != '\0') )
   {
     longStrBuf=IString(returnStruct.addr, returnStruct.size);
// Check for special sequence indicating deleted record
     if ( (longStrBuf[1] == '~') && (longStrBuf[2] == '/') 
                       && (longStrBuf[3] == '~') ) 
     {  
        ITHROWGUIERROR ("PrfQueryProfileData");
     }
     else *lData = longStrBuf.asInt();
   }
   else 
   {
     ITHROWGUIERROR("PrfQueryProfileData");
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::queryProfileSize                                               |
------------------------------------------------------------------------------*/
unsigned long IProfileData :: queryProfileSize(const IProfileHandle pDB,
                          const IString applName, const IString key)
{
   unsigned long returnValue;
   char *returnType;
   XrmValue returnStruct;
   IString pfKey(applName);
   pfKey+=iprofDot;
   pfKey+=key;
   /*************************************************************************/
   /* Retrieve keyed value from database into XrmValue structure.  Then     */
   /* return the structure size value.                                      */
   /*************************************************************************/
   if (XrmGetResource (pDB, (char *)pfKey, (char *)applClass, &returnType,
                       &returnStruct)) {
       returnValue = returnStruct.size - 1;
       return returnValue;
   } else {
       ITHROWGUIERROR("queryProfileSize");
   }
   return 0;
}

/*------------------------------------------------------------------------------
| IProfileData::writeProfileString                                             |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: writeProfileString(const IProfileHandle pDB,
                          const IString applName, const IString key, char * strBuf,
                          unsigned long strBufLen )
{
   IString pfKey(applName);
   pfKey+=iprofDot;
   pfKey+=key;
   IString value((char *)strBuf);
   /*************************************************************************/
   /* Add applName to database.                                             */
   /*************************************************************************/
   addProfileAppl(pDB, applName);
   /*************************************************************************/
   /* Update database with keyed value.                                     */
   /*************************************************************************/
   if (strBufLen == 0) {
     pfKey+=iprofColon;
     XrmPutLineResource ((XrmDatabase *)&pDB, (char *)pfKey);
   }
   else {
     XrmPutStringResource ((XrmDatabase *)&pDB, (char *)pfKey, (char *)value);
   }
   XrmPutFileDatabase(pDB, theProfile);
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::writeProfileData                                               |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: writeProfileData(const IProfileHandle pDB,
                          const IString applName, const IString key, long * lData)
{
   /*************************************************************************/
   /* Assume lData is a long and not binary; convert it to IString before   */
   /* updating its key in the database.                                     */
   /*************************************************************************/
   long    data = *lData;
   IString strlData(data);
   writeProfileString(pDB, applName, key, (char *)strlData, strlData.length());
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::addProfileAppl                                                 |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: addProfileAppl(const IProfileHandle pDB,
                          const IString applName)
{
   char *returnType;
   XrmValue returnStruct;
   IString strBuf;
   bool found = false;
   /*************************************************************************/
   /* Retrieve keyed string ("0:0") value from database into XrmValue       */
   /* structure.                                                            */
   /*************************************************************************/
   XrmGetResource (pDB, (char *)iprofApplKey, (char *)applClass,
        &returnType, &returnStruct);
   strBuf.insert((char *)returnStruct.addr);
   if (returnStruct.size) {
     IString s(strBuf);
     s.change(iprofDot, ' ');
     unsigned curPos = s.indexOf(applName);
     if (curPos)
       found = true;
   }
   /*************************************************************************/
   /* If applName not found, add it                                         */
   /*************************************************************************/
   if (!found) {
     IString newStrBuf(strBuf);
     newStrBuf+=applName;
     newStrBuf+=iprofDot;
     /***********************************************************************/
     /* Update database with keyed value.                                   */
     /***********************************************************************/
     XrmPutStringResource ((XrmDatabase *)&pDB, (char *)iprofApplKey,
       (char *)newStrBuf);
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::deleteProfileString                                            |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: deleteProfileString(const IProfileHandle pDB,
                          const IString applName, const IString key)
{
   IString strData("~/~");
   writeProfileString(pDB, applName, key, (char *)strData, strData.length());
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::deleteProfileAppl                                              |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: deleteProfileAppl(const IProfileHandle pDB,
                          const char *applName)
{
   char *returnType;
   XrmValue returnStruct;
   IString strBuf;
   bool found = false;
   /*************************************************************************/
   /* Retrieve keyed string ("0:0") value from database into XrmValue       */
   /* structure.                                                            */
   /*************************************************************************/
   XrmGetResource (pDB, (char *)iprofApplKey, (char *)applClass,
        &returnType, &returnStruct);
   strBuf.insert((char *)returnStruct.addr);
   if (returnStruct.size) {
     IString s(strBuf);
     s.change(iprofDot, ' ');
     unsigned curPos = s.indexOf(applName);
     unsigned stopPos = s.indexOf(' ', curPos);
     if ( (curPos) && (stopPos > curPos) ) {
       found = true;
       strBuf.remove(curPos, (stopPos-curPos) + 1);
     }
   }

   /*************************************************************************/
   /* If applName found, remove it                                          */
   /*************************************************************************/
   if (found) {
     IString newStrBuf(strBuf);
     /***********************************************************************/
     /* Update database with keyed value.                                   */
     /***********************************************************************/
     XrmPutStringResource ((XrmDatabase *)&pDB, (char *)iprofApplKey,
       (char *)newStrBuf);
     XrmPutFileDatabase(pDB, theProfile);
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IProfileData::deleteProfile                                                  |
------------------------------------------------------------------------------*/
IProfileData & IProfileData :: deleteProfile(const IProfileHandle pDB)
{
   if (pDB == pProfile->handle()) {
     if (remove(theProfile) != 0)
       ITHROWGUIERROR("DeleteOldProfile");
   }
   return *this;
}


/*------------------------------------------------------------------------------
| IProfileCursorData::IProfileCursorData                                       |
------------------------------------------------------------------------------*/
IProfileCursorData :: IProfileCursorData 
             ( IProfile::Cursor* profileCursor )  :
        pProfileCursor(profileCursor)
   { }

/*------------------------------------------------------------------------------
| IProfileCursorData::~IProfileCursorData                                      |
------------------------------------------------------------------------------*/
IProfileCursorData :: ~IProfileCursorData ( )
   { }

/*------------------------------------------------------------------------------
| IProfileCursorData::initProfileCursor                                        |
|   Read profile for all application names or key names                        |
|                                                                              |
|   On exit, strCl will contain:                                               |
|      | entry1 | 0 | entry2 | 0 | ... | entry n | 0 | 0 |                     |
|                                                      A                       |
              Terminating null of the IString -----                       |
------------------------------------------------------------------------------*/
IString IProfileCursorData :: initProfileCursor (const IProfileHandle hProfile,
                                    IString applName )
{
  IMODTRACE_DEVELOP("IProfile::Cursor::initCursor");

  int (*profileCallback)( XrmDatabase*,
                         XrmBindingList,
                         XrmQuarkList,
                         XrmRepresentation*,
                         XrmValue*,
                         XPointer);
  XrmName names[3];
  XrmClass classes[3];
  char *buffer;
  char *iname;
  char *cname;
  char *pos;

  int mode = XrmEnumAllLevels;
  int  ch = '\0';
  XrmDatabase pDB = hProfile;
  XrmQString = XrmPermStringToQuark("String");
  /**************************************************************************/
  /* Get the correct className to use for database query/enumeration.       */
  /**************************************************************************/
  XtGetApplicationNameAndClass(XtDisplay(IThread::current().applicationShell()),
    &iname, &cname);
  /**************************************************************************/
  /* If no applName is specified, query available applNames from the        */
  /* database via XrmGetResource                                            */
  /**************************************************************************/
  if (!applName.size()) {
    XrmValue returnStruct;
    char *returnType;
    /************************************************************************/
    /* Retrieve keyed string value from database into XrmValue structure.   */
    /* Then copy string to buffer and change . to \0.                       */
    /************************************************************************/
    XrmGetResource (pDB, (char *)iprofApplKey, iname,
      &returnType, &returnStruct);
    /************************************************************************/
    /* Allocate and NULL buffer to contain appl names.                      */
    /************************************************************************/
    strBufferSize = max(returnStruct.size+1, STRBUFFERSIZE);
    buffer = new char[strBufferSize];
    memset (buffer, '\0', strBufferSize);
    /************************************************************************/
    /* copy string to buffer and change . to \0.                            */
    /************************************************************************/
    strncpy(buffer, returnStruct.addr, returnStruct.size);
    pos = buffer;
    int i = (pos - buffer);
    while ( (i < strBufferSize) && (*(pos) != '\0') ) {
      if (*(pos) == iprofDot)
        *(pos) = '\0';
      pos++;
      i++;
    }
  }
  /**************************************************************************/
  /* else if applName is specified, enumerate the keys for the applName     */
  /* in the database via XrmEnumerateDatabase                               */
  /**************************************************************************/
  else {
    /************************************************************************/
    /* Build ClassList and NameList                                         */
    /************************************************************************/
    XrmStringToClassList(iname, classes);
    XrmStringToNameList((char *)applName, names);
    /************************************************************************/
    /* Allocate and NULL buffer to contain IStrings of key names that are   */
    /* retrieved from the database for the specified applName.              */
    /* Enumerate key names for specified applName.                          */
    /************************************************************************/
    buffer = new char[STRBUFFERSIZE];
    memset (buffer, '\0', STRBUFFERSIZE);
    strBufferSize = STRBUFFERSIZE;
    profileCallback = &iprofileXlibCallback;
    Bool rc = XrmEnumerateDatabase(hProfile, names, classes, mode,
      profileCallback, (XPointer) &buffer);
  }
  /**************************************************************************/
  /* Determine size of list of appl or key names and create an IString of   */
  /* that size to be returned.                                              */
  /**************************************************************************/
  pos = buffer;
  int i = (pos - buffer);
  while ( (i < strBufferSize) && (*(pos) != '\0') ) {
    pos = (strchr(pos, ch)) + 1;
    i = (pos - buffer);
  }
  IString str('\0');      // assume no keys
  if (i > 0)  {
    IString str1((void *)buffer, i);
    str = str1;
  }
  /**************************************************************************/
  /* Free allocated buffer                                                  */
  /**************************************************************************/
  delete buffer;
  return str;
}
#endif //IC_MOTIF
