// Revision: 44 1.16.2.1 source/ui/baseapp/ipopmenu.cpp, menu, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ipopmenu.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ipopmenu.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// #define IC_TRACE_ALL

extern "C" {
   #define INCL_WINPOINTERS
   #define INCL_WINWINDOWMGR
   #define INCL_WINMENUS
   #define INCL_WININPUT
   #include <iwindefs.h>
   #ifdef IC_MOTIF
     #include <Xm/RowColumn.h>
     #include <Xm/CascadeBG.h>
     #include <Xm/MenuShell.h>
   #endif //IC_MOTIF
}

#include <ipopmenu.hpp>
#include <ibidiset.hpp>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <imenuprv.hpp>
#include <inotifev.hpp>
#include <iplatfrm.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <iwindow.hpp>

#ifdef IC_MOTIF
  #include <iwinpriv.hpp>
#endif //IC_MOTIF

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IPOPMENU_CPP_
  #include <ipagetun.h>
#endif


#ifdef IC_MOTIF
  // defined in ithread.cpp
  extern XButtonPressedEvent  ithreadLastButtonEvent;
#endif //IC_MOTIF


#ifdef IC_MOTIFPM
class PopUpMenuWindow : public MenuWindow {
public:
  PopUpMenuWindow ( unsigned long        windowId,
                    unsigned long        style,
                    const IWindowHandle& owner);
#ifdef IC_PM
  PopUpMenuWindow ( const IWindowHandle& handle,
                    IWindow*             rootMenu );
#endif
virtual
 ~PopUpMenuWindow();

virtual IWindowHandle
  handleForChildCreation( ) const;

PopUpMenuWindow
 &cleanUpMenuObject  ( bool cleanUp = true );
bool
  isDeleteInProgress ( ) const;

private:
bool
  fCleanUpMenuObject,
  fDeleteInProgress;
#ifdef IC_MOTIF
IWindowHandle
  fRowCol;
#endif

  PopUpMenuWindow ( const PopUpMenuWindow& );
PopUpMenuWindow
 &operator=       ( const PopUpMenuWindow& );
}; // PopUpMenuWindow

/*------------------------------------------------------------------------------
| PopUpMenuWindow::PopUpMenuWindow                                             |
------------------------------------------------------------------------------*/
PopUpMenuWindow::PopUpMenuWindow( unsigned long        windowId,
                                  unsigned long        style,
                                  const IWindowHandle& owner)
  : MenuWindow()
  , fCleanUpMenuObject( true )
  , fDeleteInProgress( false )
#ifdef IC_MOTIF
  , fRowCol( 0 )
#endif
{
  IFUNCTRACE_DEVELOP();
#ifdef IC_PM
  IWindowHandle whMenu =
                create( windowId,
                        0,
                        style,
                        WC_MENU,
                        IWindow::desktopWindow()->handle(),
                        owner,
                        IRectangle(),
                        0,
                        0,
                        IWindow::onTopOfSiblings );
  startHandlingEventsFor(whMenu);
#endif // IC_PM
#ifdef IC_MOTIF
   //Create popupMenu widget using convenience function, to gain added
   //behavior such as automatic popdown, etc.
   Arg msArgs [5];
   int n=0;
   Position pos=1;
   Dimension dim=1;
   XtSetArg(msArgs[n], XmNx, pos); n++;
   XtSetArg(msArgs[n], XmNy, pos); n++;
   XtSetArg(msArgs[n], XmNwidth, dim); n++;
   XtSetArg(msArgs[n], XmNheight, dim); n++;
   // Disable accelerators and mnemonics in popups so that our handlers
   // can post the menu.  With this resource enabled, Motif installs
   // accelerators on the owner to popup the menu, which conflicts with
   // our handlers.
   XtSetArg(msArgs[n], XmNpopupEnabled, False); n++;
   Widget whMenu = XmCreatePopupMenu(
                                owner,
                                (char *)IString( windowId ),
                                msArgs,
                                n);
   fRowCol = whMenu;
   Widget menuShell = XtParent(whMenu);
   ITRACE_ALL( IString("menuShell=") +
               IString((unsigned long)menuShell).d2x() +
               IString(" whMenu=") +
               IString((unsigned long)whMenu).d2x() );
   // Use the handle of the menu shell as our "window()->handle()";
   // create a IWindow wrapper for menuShell & make it our non-portable "menu" window
   startHandlingEventsFor( menuShell );
#endif // IC_MOTIF
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| PopUpMenuWindow::PopUpMenuWindow                                             |
------------------------------------------------------------------------------*/
PopUpMenuWindow::PopUpMenuWindow ( const IWindowHandle& handle,
                                   IWindow*             rootMenu )
  : MenuWindow( handle, rootMenu )
  , fCleanUpMenuObject( true )
  , fDeleteInProgress( false )
{
  IFUNCTRACE_DEVELOP();
}
#endif // IC_PM

/*------------------------------------------------------------------------------
| PopUpMenuWindow::~PopUpMenuWindow                                            |
------------------------------------------------------------------------------*/
PopUpMenuWindow::~PopUpMenuWindow()
{
  IFUNCTRACE_DEVELOP();

  // Set a flag to prevent the IPopUpMenu dtor from trying to delete
  // this object a second time during auto-delete object processing.
  fDeleteInProgress = true;

  // Logic to auto-delete the IPopUpMenu object.
  if ( fCleanUpMenuObject )
  {  // This destructor is being called before the IPopUpMenu destructor,
     // which should happen only if auto-delete object is enabled.
     // So delete the IPopUpMenu object.
     IPopUpMenu* pmenu = dynamic_cast< IPopUpMenu* >( this->menu() );
     if ( pmenu )
     {
        ITRACE_DEVELOP( IString( " Deleting root popup menu shell: " ) +
                        IString( (unsigned long) pmenu).d2x() );
        delete pmenu;
     }
  }
}


/*------------------------------------------------------------------------------
| PopUpMenuWindow::handleForChildCreation                                      |
| In Motif, this returns the handle of the RowColumn widget for the popup      |
| instead of the MenuShell which is returned by handle().                      |
------------------------------------------------------------------------------*/
IWindowHandle PopUpMenuWindow::handleForChildCreation( ) const
{
  IFUNCTRACE_ALL();
#ifdef IC_MOTIF
  return fRowCol;
#endif
#ifdef IC_PM
  return this->handle();
#endif
}

/*------------------------------------------------------------------------------
| PopUpMenuWindow::cleanUpMenuObject                                           |
| Set whether this object should delete its IPopUpMenu object.                 |
------------------------------------------------------------------------------*/
PopUpMenuWindow& PopUpMenuWindow::cleanUpMenuObject ( bool cleanUp )
{
  fCleanUpMenuObject = cleanUp;
  return *this;
}

/*------------------------------------------------------------------------------
| PopUpMenuWindow::isDeleteInProgress                                          |
| Identifies if the object is currently running its destructor code.           |
------------------------------------------------------------------------------*/
bool PopUpMenuWindow::isDeleteInProgress ( ) const
{
  return fDeleteInProgress;
}
#endif // IC_MOTIFPM


/*------------------------------------------------------------------------------
| IPopUpMenu::IPopUpMenu                                                       |
|  Constructor. Empty pop-up for dynamic menu.                                 |
------------------------------------------------------------------------------*/
IPopUpMenu::IPopUpMenu( IWindow*           owner,
                        unsigned long      windowId,
                        const Style&       style )
#ifdef IC_PMWIN
                      : fPopUpMenuData( 0 )
#endif
{
   IMODTRACE_DEVELOP("IPopUpMenu::IPopUpMenu");
   IASSERTPARM(owner != 0);
#ifdef IC_WIN
   fAutoDeleteObject = false;
   fhMenu = CreatePopupMenu();
   fowner = owner->handle();
   fmenuFrame = IMenuPrivate::locateFrame(owner);

   /*******************************************************************/
   /* Add an entry for the main popup, identified by the window id    */
   /*******************************************************************/
   IMenuPrivate::addToLookUpTable( fmenuFrame, fhMenu, windowId );

   // We can't give the menu bar bidirectional attributes until
   // it has at least one item, since bidi attributes can only
   // be assigned to menu items.
#endif // IC_WIN
#ifdef IC_PM
   IWindowHandle ownerHandle(0);
   if (owner)
      ownerHandle = owner->handle();
   PopUpMenuWindow* menuWindow =
       new PopUpMenuWindow( windowId,
                             style.asUnsignedLong(),
                             owner->handle());

   menuWindow->setMenu( this );
   setWindow(menuWindow);
#endif // IC_PM
#ifdef IC_MOTIF
   PopUpMenuWindow* menuWindow = new PopUpMenuWindow( windowId,
                                                      0,
                                                      owner->handle() );
   menuWindow->setMenu( this );
   setWindow(menuWindow);
   registerCallbacks();  //call private method to register necessary menu callbacks
   menuWindow->setOwner(owner);
   if (style != IWindow::noStyle)
     menuWindow->setStyle (style.asUnsignedLong());
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IPopUpMenu::IPopUpMenu                                                       |
|  Constructor. Pop-up from .rc.                                               |
------------------------------------------------------------------------------*/
IPopUpMenu::IPopUpMenu( const IResourceId& menuResId,
                        IWindow* owner )
#ifdef IC_PMWIN
                      : fPopUpMenuData( 0 )
#endif //IC_PMWIN
{
   IMODTRACE_DEVELOP("IPopUpMenu::IPopUpMenu(resId, owner)");
#ifdef IC_WIN
   fAutoDeleteObject = false;
   char  menuText[IC_MENUTEXTLENGTH];

   /*******************************************************************/
   /* Since Windows does not allow you to load a popup directly, we   */
   /* must create an empty popup, load the menu resource and then     */
   /* copy each item over to the new menu                             */
   /*******************************************************************/
   fhMenu = CreatePopupMenu();
   HMENU menuLoaded = menuResId.resourceLibrary().loadMenu(menuResId, owner);
   fowner = owner->handle();
   fmenuFrame = IMenuPrivate::locateFrame(owner);

   /*******************************************************************/
   /* Begin with first item (0 based) and get count of items          */
   /*******************************************************************/
   int itemPos = 0;
   int itemCount = GetMenuItemCount( menuLoaded );

   /*******************************************************************/
   /* Loop through menu, transferring each item found to the new menu */
   /*******************************************************************/
   while ( itemPos < itemCount )
   {
     /*****************************************************************/
     /* If on Windows 95 or new shell, use extended menu api to xfer  */
     /*****************************************************************/
     if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
     {
       MENUITEMINFO miItem;
       miItem.cbSize = sizeof(MENUITEMINFO);
       miItem.fMask = MIIM_STATE | MIIM_ID | MIIM_SUBMENU | MIIM_TYPE |
                      MIIM_DATA;
       miItem.dwTypeData = menuText;
       miItem.cch = IC_MENUTEXTLENGTH;

       /***************************************************************/
       /* Get all of the information about the menuitem.              */
       /***************************************************************/
       GetMenuItemInfo( menuLoaded, itemPos, TRUE, &miItem );

       /***************************************************************/
       /* Insert the item as is into the newly created popup menu     */
       /***************************************************************/
       InsertMenuItem( fhMenu, itemPos, TRUE, &miItem );
     }
     /*****************************************************************/
     /* otherwise use old menu API to xfer items over                 */
     /*****************************************************************/
     else
     {
       ULONG ulItem = (ULONG)GetSubMenu( menuLoaded, itemPos );
       ULONG uState = GetMenuState( menuLoaded, itemPos, MF_BYPOSITION );
       if (ulItem)
         uState &= 0x000000FF;

       if (!ulItem)
         ulItem = GetMenuItemID( menuLoaded, itemPos );
       GetMenuString( menuLoaded, itemPos, menuText, IC_MENUTEXTLENGTH,
                      MF_BYPOSITION );
       InsertMenu( fhMenu, 0xFFFFFFFF, (UINT)(MF_BYPOSITION | uState),
                   (UINT)ulItem, menuText );
     }
     itemPos++;
   }

   /*******************************************************************/
   /* Loop through menu again to remove submenus so they aren't       */
   /*  destroyed (which only is a problem on Windows 95)              */
   /*******************************************************************/
   for ( itemPos = 0; itemPos < itemCount; itemPos++ )
     RemoveMenu( menuLoaded, ((itemCount-1)-itemPos), MF_BYPOSITION );

   /*******************************************************************/
   /* Destroy loaded menu once items are transferred                  */
   /*******************************************************************/
   DestroyMenu( menuLoaded );

   /*******************************************************************/
   /* For popups, always add to lookup table for either menu system   */
   /*******************************************************************/
   IMenuPrivate::buildLookUpTableFromResourceLoad( fmenuFrame, fhMenu,
                                                   menuResId );

   /*******************************************************************/
   /* For bidi systems, if the owner window has a right-to-left       */
   /* layout, reverse the menu order and justification.               */
   /*******************************************************************/
   if (IBidiSettings::isBidiSupported())
   {
      IBidiSettings
        bidiSettings( IBidiSettings::applicationDefaults() );
      if ( owner )
      {
         bidiSettings = IBidiSettings( *owner );
      }
      IMenuPrivate::setBidiAttributes( fhMenu, bidiSettings );
   }
#endif // IC_WIN
#ifdef IC_PM
   IWindowHandle handleClPopUp =
       menuResId.resourceLibrary().loadMenu(menuResId, owner);
   WinSetParent(handleClPopUp, HWND_DESKTOP, FALSE);
   WinSetWindowUShort(handleClPopUp,           //PM load with FID_MENU
                      QWS_ID, menuResId);      //Change it.
   // Create a IWindow wrapper for the popup handle & make it our non-portable "menu" window
   PopUpMenuWindow* menuWindow = new PopUpMenuWindow( handleClPopUp, 0 );
   menuWindow->reserveUserWindowWord( false );
   menuWindow->setMenu( this );
   setWindow(menuWindow);
#endif // IC_PM
#ifdef IC_MOTIF
   IASSERTPARM(owner != 0);
   PopUpMenuWindow* menuWindow = new PopUpMenuWindow(menuResId.id(),
                                                     0,
                                                     owner->handle());
   menuWindow->setMenu( this );
   setWindow(menuWindow);
   menuWindow->setOwner(owner);
   menuResId.resourceLibrary().loadMenu(menuResId.id(),
                                        menuWindow->handleForChildCreation(),
                                        this);
   registerCallbacks();   //register menu callbacks after adding any menu items
                          //  This way, button gadgets don't get this callback,
                          //  and all menu-related events passed to the menubar.
#endif // IC_MOTIF
}


/*------------------------------------------------------------------------------
| IPopUpMenu::~IPopUpMenu                                                      |
|  Define to override IMenu class definition.                                  |
------------------------------------------------------------------------------*/
IPopUpMenu::~IPopUpMenu()
{
   IMODTRACE_DEVELOP( "IPopUpMenu::~IPopUpMenu" );
#ifdef IC_WIN
  if ( fhMenu )
     DestroyMenu( fhMenu );
#endif

#ifdef IC_MOTIFPM
  IWindow
   *window = this->window();
  PopUpMenuWindow
   *popupMenuWindow = dynamic_cast< PopUpMenuWindow* >( window );

  if ( window )
  {
     // Even if auto-delete object is enabled and this dtor is
     // being called from the PopUpMenuWindow dtor, the
     // PopUpMenuWindow is still a valid object for calling
     // functions.
#ifdef IC_MOTIF
     // If underlying HWND/widget exists, we must clean it up first
     if ( window->isValid() )
     {
        this->unregisterCallbacks();
     }
#endif

     // Clean-up processing for the PopUpMenuWindow object.
     if ( popupMenuWindow )
     {
        if ( popupMenuWindow->isDeleteInProgress() )
        {
           // We are being called from the PopUpMenuWindow dtor thru
           // auto-delete object processing. Don't delete the
           // PopUpMenuWindow again, and prevent the IMenu dtor from
           // doing this by resetting the menu window stored in IMenu.
           fWindow = 0;
        }
        else
        {
           // Delete the PopUpMenuWindow object via reference counting
           // in IMenu::setWindow. Beforehand, ensure the PopUpMenuWindow
           // dtor does not try to delete this IPopUpMenu object again
           // by removing the reference to it from the PopUpMenuWindow
           // object.
           popupMenuWindow->cleanUpMenuObject( false );
           this->setWindow( 0 );
        }
     }
  }
#endif // IC_MOTIFPM
}


/*------------------------------------------------------------------------------
| IPopUpMenu::show - show this popup at the given point (in window-relative    |
| application coordinates, to match IMenuEvent::mousePosition)                 |
------------------------------------------------------------------------------*/
IPopUpMenu& IPopUpMenu::show( const IPoint& ptAt, ETrackButton trackButton )
{
   IMODTRACE_DEVELOP("IPopUpMenu::show");

#ifdef IC_WIN
   IPoint nativePoint = ICoordinateSystem::convertToNative(
                           ptAt, owner()->size() );
   POINTL scrPoint = nativePoint.asPOINTL();
   ClientToScreen( fowner, (LPPOINT)&scrPoint );
   int uFlags = TPM_LEFTALIGN | TPM_TOPALIGN;

   // For bidi systems, if the owner window has a right-to-left
   // layout, then the menu is right aligned.  In this case, also
   // position the pop-up to the left of the specified point.
   if (IBidiSettings::isBidiSupported())
   {
      if ( this->owner() )
      {
         // Give the menu the bidi attributes of the owner window.
         IBidiSettings
           bidiSettings( *( this->owner() ) );
         if ( bidiSettings.windowLayout() ==
                              IBidiSettings::layoutRightToLeft )
         {
            uFlags = TPM_RIGHTALIGN | TPM_TOPALIGN;
         }
      }
      else
      {
         // Give the menu default bidi attributes.
         IBidiSettings
           bidiSettings( IBidiSettings::applicationDefaults() );
         if ( bidiSettings.windowLayout() ==
                              IBidiSettings::layoutRightToLeft )
         {
            uFlags = TPM_RIGHTALIGN | TPM_TOPALIGN;
         }
      }
   }

   if ( trackButton == kLeftButton )
   {
      uFlags |= TPM_LEFTBUTTON;
   }
   else if ( trackButton == kRightButton )
   {
      uFlags |= TPM_RIGHTBUTTON;
   }

   // Show the popup - Align topleft corner of menu with mouse pointer
   TrackPopupMenu( fhMenu,
                   uFlags,
                   (int)scrPoint.x, (int)scrPoint.y,
                   0, fowner, NULL );
   // TrackPopupMenu returns false when menu already showing, so we can't detect a failure
   // We can't reposition existing popup, or dismiss/reshow. Have to do nothing.

   // TrackPopupMenu returns AFTER popup is dismissed, so we have to handle
   // autoDeleteObject here and now (WIN32 IMenus don't have an hwnd to post to)
   if (isAutoDeleteObject()) {
      delete this;        //Yes, it is unusual to delete yourself, but it's OK here.
      return *this;
   }
#endif
#ifdef IC_PM
   IPoint nativePoint = ICoordinateSystem::convertToNative(
                           ptAt, owner()->size() );
   IWindowHandle ownerHwnd = owner()->handle();
   WinPopupMenu(ownerHwnd, ownerHwnd,
                window()->handle(),
                nativePoint.x(), nativePoint.y(),
                0,
                PU_HCONSTRAIN | PU_VCONSTRAIN |
                PU_KEYBOARD | PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2);
   // WinPopupMenu returns false when menu already showing, so we can't detect errors.
   // To be consistent with WIN, do nothing.
#endif

#ifdef IC_MOTIF
   // IThread keeps a copy of the most recent Button event in
   // the variable ithreadLastButtonEvent.  We use this to obtain
   // a valid XButtonPressedEvent.
   XButtonPressedEvent  event = ithreadLastButtonEvent;

   // Obtain an IWindow with which to do coordinate conversion.
   // Try to use the window which experienced the last button event.
   // If that is not an IWindow, use the owner of this PopUpMenu.
   // Last ditch is desktopWindow.
   IWindowHandle hwnd = XtWindowToWidget(
                           event.display, event.window );
   IWindow      *pwin = IWindow::windowWithHandle( hwnd );
   if (pwin == 0)
      pwin = this->owner();

   // Map the native point from "relative to app window" to "relative to desktop",
   //   since the original MB2 click was in the app window but the popup menu
   //   is a child of the desktop.
   IPoint tempPoint = IWindow::mapPoint( ptAt,
                                         pwin->handle(),
                                         IWindow::desktopWindow()->handle() );
   IPoint nativePoint = ICoordinateSystem::convertToNative(
                           tempPoint, IWindow::desktopWindow()->size() );

   // Obtain the rowcolumn widget for this popup menu (window()->handle() is
   // its XmMenuShell parent)
   IWindowHandle rowcolumn = window()->handleForChildCreation();

   // d7541 - Save and restore the event fields.  They may still be
   //         needed.
   Position saveXRoot = event.x_root;
   Position saveYRoot = event.y_root;

   event.x_root = nativePoint.x();
   event.y_root = nativePoint.y();
   ITRACE_DEVELOP( IString("RETRO atLocation=") + ptAt.asString() +
                   IString(" nativePoint.x()=") + IString(nativePoint.x()) +
                   IString(" nativePoint.y()=") + IString(nativePoint.y())  );
   XmMenuPosition( rowcolumn, &event );
   // d7541 - restore the event fields.
   event.x_root = saveXRoot;
   event.y_root = saveYRoot;

   XtManageChild(rowcolumn);  //AJ26329

   this->notifyObservers(INotificationEvent(
         IWindow::visibleId, *this, true, true));
#endif //IC_MOTIF

   return *this;
}


/*------------------------------------------------------------------------------
| IPopUpMenu::setAutoDeleteObject                                              |
|                                                                              |
| Mark if this IPopUpMenu object is deleted when related window is destroyed.  |
------------------------------------------------------------------------------*/
void IPopUpMenu::setAutoDeleteObject( bool autoDelete )
{
#ifdef IC_WIN
  fAutoDeleteObject = autoDelete;
#else
  ITRACE_ALL( IString("IPopUpMenu::setAutoDeleteObject(") +
              IString( autoDelete ? "true" : "false") +
              IString(" handle=") +
              IString( window()->isValid() ?
                       window()->handle().asUnsigned() : 0 ).d2x() );
  window()->setAutoDeleteObject(autoDelete);
#endif
  return;
}


/*------------------------------------------------------------------------------
| IPopUpMenu::isAutoDeleteObject                                               |
|                                                                              |
| Return if this IPopUpMenu object is deleted when related window is destroyed.|
------------------------------------------------------------------------------*/
bool IPopUpMenu::isAutoDeleteObject( ) const
{
#ifdef IC_WIN
  return fAutoDeleteObject;
#else
  return window()->isAutoDeleteObject();
#endif
}


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IPopUpMenu::id                                                               |
|                                                                              |
| Get the menu identity. For popups, we have to check the rowcolumn, not shell.|
------------------------------------------------------------------------------*/
unsigned long IPopUpMenu::id() const
{
   IFUNCTRACE_ALL();
   // Obtain the rowcolumn widget for this popup menu (window()->handle() is its XmMenuShell parent)
   IWindowHandle rowcolumn = window()->handleForChildCreation();
   return IString(XtName( (Widget)rowcolumn )).asInt();     // Keep in synch with IIDOF , in iwindow2.cpp
}

/*------------------------------------------------------------------------------
| IPopUpMenu::registerCallbacks                                                |
|                                                                              |
| Add callbacks and X event handlers for events which entry fields can         |
| experience. Private method                                                   |
------------------------------------------------------------------------------*/
void IPopUpMenu::registerCallbacks()
{
   IFUNCTRACE_ALL();
   // "window()->handle()" returns the menuShell, but it's one and only child should be the
   //   row-column widget which is the actual menu we need.
   IWindowHandle widget = window()->handleForChildCreation();

   // Add the callback to handle the changed flag
   XtAddCallback(
      (Widget)widget,           // widget
      XmNentryCallback,         // callback name
      imenuMotifCallback,       // callback routine
      NULL);                    // client data
   XtAddCallback(
      (Widget)widget,
      XmNmapCallback,
      imenuMotifCallback,
      (XtPointer)this->id());
  XtAddCallback(
      (Widget)widget,
      XmNunmapCallback,
      imenuMotifCallback,
      (XtPointer)this->id());

   // Install callbacks for detecting activation and dismissal of
   // submenus of the menubar and popup menus.  The popdown callback is
   // also used to handle autoDeleteObject for the popup menu.
   XtAddCallback((Widget)window()->handle(),       //Want the MenuShell here
                 XmNpopdownCallback, imenuPopDownCallback,
                (XtPointer)window() );
   XtAddCallback((Widget)window()->handle(),       //Want the MenuShell here
                 XmNpopupCallback,   imenuPopUpCallback,
                (XtPointer)window() );
}

/*------------------------------------------------------------------------------
| IPopUpMenu::unregisterCallbacks                                              |
|                                                                              |
| Remove callbacks and X event handlers added in registerCallbacks(). Private. |
------------------------------------------------------------------------------*/
void IPopUpMenu::unregisterCallbacks()
{
   IFUNCTRACE_ALL();
   // "window()->handle()" returns the menuShell, but it's one and only child should be the
   //   row-column widget which is the actual menu we need.
   IWindowHandle widget = window()->handleForChildCreation();

   // Remove the callback to handle the changed flag
   XtRemoveCallback(
      (Widget)widget,           // widget
      XmNentryCallback,         // callback name
      imenuMotifCallback,       // callback routine
      NULL);                    // client data
   XtRemoveCallback(
      (Widget)widget,
      XmNmapCallback,
      imenuMotifCallback,
      (XtPointer)this->id());
   XtRemoveCallback(
      (Widget)widget,
      XmNunmapCallback,
      imenuMotifCallback,
      (XtPointer)this->id());

   XtRemoveCallback((Widget)window()->handle(),       //Want the MenuShell here
                    XmNpopdownCallback, imenuPopDownCallback,
                    (XtPointer)window() );
   XtRemoveCallback((Widget)window()->handle(),       //Want the MenuShell here
                    XmNpopupCallback,   imenuPopUpCallback,
                    (XtPointer)window() );
}

#endif //IC_MOTIF
