// Revision: 16 1.11.2.1 source/ui/baseapp/imoushdr.cpp, userinput, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: imoushdr.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imoushdr.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
extern "C"
{
  #define INCL_WININPUT
  #define INCL_WINPOINTERS      // For WinQueryPointerPos.
  #define INCL_WINWINDOWMGR     // For WinWindowFromPoint.
  #include <iwindefs.h>
}

#include <imoushdr.hpp>

#include <icconst.h>
#include <idefstyl.h>
#include <iexcept.hpp>
#include <iplatfrm.hpp>
#include <itimer.hpp>
#include <iwindow.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IMOUSHDR_CPP_
  #include <ipagetun.h>
#endif

const IMouseHandler::Style
  IMouseHandler::noMouseMoves      ( 1, IMouseHandler__noMoves ),
  IMouseHandler::someMouseMoves    ( 1, IMouseHandler__someMoves ),
  IMouseHandler::allMouseMoves     ( 1, IMouseHandler__allMoves ),
  IMouseHandler::mouseEntersLeaves ( 1, IMouseHandler__entersLeaves ),
  IMouseHandler::classDefaultStyle ( 1, IMouseHandler__classDefaultStyle );

IMouseHandler::Style
  IMouseHandler::fgCurrentDefaultStyle ( 1, IMouseHandler__classDefaultStyle );


/*------------------------------------------------------------------------------
| IMouseHandlerData                                                            |
|                                                                              |
| Private data class for IMouseHandler.                                        |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IMouseHandlerData {
public:
  IMouseHandlerData ( );
 ~IMouseHandlerData ( );
#ifdef IC_PMWIN
static IWindowHandle
  fgLastWindowWithMouse;
    // Store this per thread so we can simply serialize access to it
    // through the message queue.                                    IC_NOTYET
static ITimer
  fgMouseLeaveTimer;
    // Store this per thread because it would disappear if its message
    // queue is destroyed, and no way to tell which message is the
    // best one to create it on.                                     IC_NOTYET
void
  timerPop ( );
#endif
private:
friend class IMouseHandler;
}; // IMouseHandlerData

#pragma pack(pop)
#pragma enum(pop)

#ifdef IC_PMWIN
IWindowHandle
  IMouseHandlerData::fgLastWindowWithMouse( 0 );
ITimer
  IMouseHandlerData::fgMouseLeaveTimer;
    // Need to stop this timer somewhere.                            IC_NOTYET
#endif

#define MOUSELEAVETIMER_INTERVAL  100

/*------------------------------------------------------------------------------
| IMouseHandlerData::IMouseHandlerData                                         |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IMouseHandlerData::IMouseHandlerData ( )
{ }

/*------------------------------------------------------------------------------
| IMouseHandlerData::~IMouseHandlerData                                        |
|                                                                              |
| Destructor for clean-up code.                                                |
------------------------------------------------------------------------------*/
IMouseHandlerData::~IMouseHandlerData ( )
{ }

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IMouseHandlerData::timerPop                                                  |
|                                                                              |
| This function is called from a timer to help determine if the mouse pointer  |
| has left a window.  It is mainly needed to detect if the mouse has entered   |
| a window that is not wrappered by an IWindow object, and therefore does not  |
| cause the window dispatcher to be called.                                    |
------------------------------------------------------------------------------*/
void IMouseHandlerData::timerPop ( )
{
  IWindowHandle
    lastMouseWindow( IMouseHandlerData::fgLastWindowWithMouse );
  if ( lastMouseWindow )
  {
     // The mouse pointer was last over a window that we know about.
     // Therefore, a mouse leave condition was not already detected
     // from a mouse move event, with a mouse leave event synthesized.
     // See if a mouse leave condition has just occurred.

     // Find the handle of the window under the mouse.  Unfortunately,
     // disabled controls typically are ignored because the default
     // window proc says they are transparent as a result of hit testing.
     // The below code is independent of the coordinate system.
     unsigned long
       mouseX,
       mouseY;
     IQUERYPOINTERPOSITION( mouseX, mouseY );
     IPoint
       mousePoint( mouseX, mouseY );
     POINTL
       pt( mousePoint.asPOINTL() );
     IWindowHandle
       handleWithMouse( IWINDOWFROMPOINT( IWindow::desktopWindow()->handle(),
                                          &pt, true ) );

     if ( handleWithMouse != lastMouseWindow )
     {
        // The mouse has left the last window we knew it to be in.
        // Dispatch a WM_MOUSELEAVE message to signify that the mouse
        // has left the window.
        IMouseHandlerData::fgLastWindowWithMouse = 0;
        if ( lastMouseWindow.isValid() )
        {  // The last window is still around.
#ifdef IC_PM
           lastMouseWindow.sendEvent( WM_MOUSELEAVE );
#endif
#ifdef IC_WIN
           lastMouseWindow.sendEvent( IC_UM_MOUSELEAVE );
#endif
        }
     }
  }
}
#endif // IC_PMWIN


/*------------------------------------------------------------------------------
| IMouseHandler::IMouseHandler                                                 |
|                                                                              |
| Default constructor here for page tuning.                                    |
------------------------------------------------------------------------------*/
IMouseHandler::IMouseHandler ( const Style& style )
  : IMouseHandler::Inherited ( )
  , fStyle( style )
  , fMouseHandlerData( new IMouseHandlerData )
{
  // Check for an invalid style combination.
  if ( ( ( style & noMouseMoves )  &&  ( style & someMouseMoves ) )  ||
       ( ( style & noMouseMoves )  &&  ( style & allMouseMoves ) )  ||
       ( ( style & someMouseMoves )  &&  ( style & allMouseMoves ) ) )
  {
     ITHROWLIBRARYERROR( IC_INVALIDSTYLE,
                         IBaseErrorInfo::invalidParameter,
                         IException::recoverable );
  }

  if ( ! ( fStyle & someMouseMoves )  &&
       ! ( fStyle & allMouseMoves ) )
  {
     // noMouseMoves is implied if neither someMouseMoves nor
     // allMouseMoves is specified.
     fStyle |= noMouseMoves;
  }
}


/*------------------------------------------------------------------------------
| IMouseHandler::~IMouseHandler                                                |
|                                                                              |
| Destructor here for clean up.                                                |
------------------------------------------------------------------------------*/
IMouseHandler::~IMouseHandler ( )
{
  delete fMouseHandlerData;
}

/*------------------------------------------------------------------------------
| IMouseHandler::defaultStyle                                                  |
|                                                                              |
| Return the current default style of the class.                               |
------------------------------------------------------------------------------*/
IMouseHandler::Style IMouseHandler::defaultStyle ( )
{
  return fgCurrentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IMouseHandler::setDefaultStyle                                               |
|                                                                              |
| Set the current default style of the class.                                  |
------------------------------------------------------------------------------*/
void IMouseHandler::setDefaultStyle ( const Style& style )
{
  fgCurrentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IMouseHandler::handleEventsFor                                               |
|                                                                              |
| Attach the handler to a window.                                              |
------------------------------------------------------------------------------*/
IMouseHandler& IMouseHandler::handleEventsFor ( IWindow* window )
{
  Inherited::handleEventsFor( window );

  if ( fStyle & someMouseMoves )
  {
     window->startHandling( IWindow::someMouseMoves );
  }
  else if ( fStyle & allMouseMoves )
  {
     window->startHandling( IWindow::allMouseMoves );
  }

  if ( fStyle & mouseEntersLeaves )
  {
#ifdef IC_PM
#if 0
     if ( IPlatform::isOS2Warp3() )      // IC_NOTYET
     {  // To process mouse enter and leave events in OS/2 Warp 3.0,
        // we must synthesize them from mouse move events.
        window->startHandling( IWindow::mouseEntersLeaves |
                               IWindow::someMouseMoves );
     }
     else
#endif
     {  // Starting with Warp 4.0 (Merlin), mouse enter/leave is a
        // separate event.
        window->startHandling( IWindow::mouseEntersLeaves );
     }
#endif // IC_PM
#ifdef IC_WIN
     // To process mouse enter and leave events in Windows 95 and
     // Windows NT, we must synthesize them from mouse move events.
     window->startHandling( IWindow::mouseEntersLeaves |
                            IWindow::someMouseMoves );
#endif
#ifdef IC_MOTIF
     // Mouse enter/leave is a specific event type.
     window->startHandling( IWindow::mouseEntersLeaves );
#endif
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IMouseHandler::stopHandlingEventsFor                                         |
|                                                                              |
| Remove the handler from a window.                                            |
------------------------------------------------------------------------------*/
IMouseHandler& IMouseHandler::stopHandlingEventsFor ( IWindow* window )
{
  if ( fStyle & someMouseMoves )
  {
     window->stopHandling( IWindow::someMouseMoves );
  }
  else if ( fStyle & allMouseMoves )
  {
     window->stopHandling( IWindow::allMouseMoves );
  }

  if ( fStyle & mouseEntersLeaves )
  {
#ifdef IC_PM
#if 0
     if ( IPlatform::isOS2Warp3() )    // IC_NOTYET
     {  // For OS/2 Warp 3.0, we synthesize mouse enter/leave events
        // from mouse move events.
        window->stopHandling( IWindow::mouseEntersLeaves |
                              IWindow::someMouseMoves );
     }
     else
#endif
     {  // Starting with Warp 4.0 (Merlin), mouse enter/leave is a
        // separate event.
        window->stopHandling( IWindow::mouseEntersLeaves );
     }
#endif
#ifdef IC_WIN
     // For Windows 95 and NT, we synthesize mouse enter/leave events
     // from mouse move events.
     window->stopHandling( IWindow::mouseEntersLeaves |
                           IWindow::someMouseMoves );
#endif
#ifdef IC_MOTIF
     // Mouse enter/leave is a specific event type.
     window->stopHandling( IWindow::mouseEntersLeaves );
#endif
  }

  Inherited::stopHandlingEventsFor( window );
  return *this;
}

/*------------------------------------------------------------------------------
| IMouseHandler::dispatchHandlerEvent                                          |
|                                                                              |
| Dispatch mouse click events for this handler.                                |
------------------------------------------------------------------------------*/
bool IMouseHandler::dispatchHandlerEvent ( IEvent& event )
{
  bool bRc = false;

  switch ( event.eventId() )
  {
     case WM_BUTTON1DOWN:
     case WM_BUTTON1UP:
     case WM_BUTTON1DBLCLK:
     case WM_BUTTON2DOWN:
     case WM_BUTTON2UP:
     case WM_BUTTON2DBLCLK:
     case WM_BUTTON3DOWN:
     case WM_BUTTON3UP:
     case WM_BUTTON3DBLCLK:
     case WM_BUTTON1CLICK:
     case WM_BUTTON2CLICK:
     case WM_BUTTON3CLICK:
     case WM_CHORD:
     {
        IMouseClickEvent mouseClickEvent( event );
        bRc = this->mouseClicked( mouseClickEvent );  // Call virtual function.
        if ( bRc )
        {                    // All processing is complete.
           event.setResult( mouseClickEvent.result() );
        }
        break;
     }

#ifdef IC_PMWIN
     case WM_MOUSEMOVE:
#endif
#ifdef IC_MOTIF
     case xEvent( MotionNotify ):
#endif
     {
#ifdef IC_PMWIN
        // OS/2 Warp 3.0, Windows 95, and Windows NT lack a WM_MOUSEENTER
        // message.  For these platforms, synthesize a mouse enter event
        // from a mouse move event when applicable.
#if 0
        bool
          synthesizeEnterLeave = true;
#ifdef IC_PM
        if ( ! IPlatform::isOS2Warp3() )            // IC_NOTYET
        {
           // No need to generate WM_MOUSEENTER, beginning with
           // OS/2 Warp 4.0 (Merlin).
           synthesizeEnterLeave = false;
        }
#endif
        if ( synthesizeEnterLeave )
#endif // 0
#ifdef IC_WIN
        {
           // Generate a mouse enter event if it looks like the mouse
           // is now entering a new window.  Do this by comparing the
           // the current window with the mouse with the last one we
           // know about, which is about as good as we can do without
           // hooking all mouse messages.
           IWindowHandle
             lastMouseWindow( IMouseHandlerData::fgLastWindowWithMouse );
           if ( lastMouseWindow != event.handle() )
           {
              // The mouse pointer is in a new window now.
              if ( lastMouseWindow )
              {
                 // Dispatch a WM_MOUSELEAVE message to the previous
                 // window to signify that the mouse has left that window.
                 IMouseHandlerData::fgLastWindowWithMouse = 0;
                 if ( lastMouseWindow.isValid() )
                 {  // The last window is still around.
                    lastMouseWindow.sendEvent( IC_UM_MOUSELEAVE );
                 }
              }

              // Change the window containing the mouse pointer.
              IMouseHandlerData::fgLastWindowWithMouse = event.handle();

              // Need to dispatch a WM_MOUSEENTER msg to signify that
              // the mouse has entered a new window.
              // Note that this code really needs to be in
              // IWindow::dispatch or the event loop to properly handle
              // the case of multiple mouse handlers attached to the
              // same window.                       IC_NOTYET
              event.dispatchingWindow()->sendEvent( WM_MOUSEENTER );

              // OS/2 Warp 3.0 and Windows 95 also lack a WM_MOUSELEAVE
              // message.  While Windows NT 4.0 supports WM_MOUSELEAVE
              // message, it does so via an API call (TrackMouseEvent).
              // Binding to this API without introducing a different
              // version of executable code for Windows 95 unfortunately
              // prevents use of the API here.  Therefore, for all the
              // above platforms, start a timer to help detect when a
              // mouse leave has occurred.
              if ( ! IMouseHandlerData::fgMouseLeaveTimer.isStarted() )
              {
                 IMouseHandlerData::fgMouseLeaveTimer
                  .start( new ITimerMemberFn0< IMouseHandlerData >
                                         ( *fMouseHandlerData,
                                           &IMouseHandlerData::timerPop ),
                          MOUSELEAVETIMER_INTERVAL );
              }
           }
        }
#endif // IC_WIN
#endif // IC_PMWIN
#ifdef IC_MOTIF
        // Allow the application the opportunity to change the mouse
        // pointer on AIX.
        IMousePointerEvent
          mousePointerEvent( event );
        bRc = this->mousePointerChange( mousePointerEvent );
        if ( bRc )
        {  // The application has a specific mouse pointer to use.
           event.dispatchingWindow()
             ->setMousePointer( mousePointerEvent.mousePointer() );
           bRc = false;   // Reset for mouse move processing.
        }
#endif // IC_MOTIF

        // Now process the mouse move event itself, but only call the
        // virtual function if this handler was constructed with the
        // someMouseMoves or allMouseMoves style (otherwise the
        // application is not interested in the event at this time.
        if ( ! ( fStyle & noMouseMoves ) )
        {
           IMouseEvent mouseEvent( event );
           bRc = this->mouseMoved( mouseEvent );
           if ( bRc )
           {
              this->defaultProcedure( event );
              event.setResult( mouseEvent.result() );
           }
        }

        break;
     }

#ifdef IC_PMWIN
     case WM_MOUSEENTER:
#endif
#ifdef IC_MOTIF
     case xEvent( EnterNotify ):
#endif
     {
        // Only call the virtual function to process the event if this
        // handler was constructed with the mouseEntersLeave style.
        if ( fStyle & mouseEntersLeaves )
        {
#ifdef IC_PM
           if ( event.parameter1() == event.handle() )
           {  // Check that the mouse pointer is really entering the
              // dispatching window.  Otherwise, the message may have
              // been propagated to the dispatching window (from the
              // frame to the client window, for example).
              bRc = this->mouseEnter( event );
           }
#else
           bRc = this->mouseEnter( event );
#endif
        }
        break;
     }

#ifdef IC_PM
     case WM_MOUSELEAVE:
#endif
#ifdef IC_WIN
     case IC_UM_MOUSELEAVE:
#endif
#ifdef IC_MOTIF
     case xEvent( LeaveNotify ):
#endif
     {
        // Only call the virtual function to process the event if this
        // handler was constructed with the mouseEntersLeaves style.
        if ( fStyle & mouseEntersLeaves )
        {
#ifdef IC_PM
           if ( event.parameter1() == event.handle() )
           {  // Check that the mouse pointer is really leaving the
              // dispatching window.  Otherwise, the message may have
              // been propagated to the dispatching window (from the
              // frame to the client window, for example).
              bRc = this->mouseLeave( event );
           }
#else
           bRc = this->mouseLeave( event );
#endif
        }
        break;
     }

#ifdef IC_PMWIN
     case WM_CONTROLPOINTER:
     {
        IMousePointerEvent mousePointerEvent( event );
        bRc = this->mousePointerChange( mousePointerEvent );
        event.setResult( mousePointerEvent.result() );
        break;
     }
#endif // IC_PMWIN

     default:
        break;
  } /* endswitch */

  return bRc;
}

/*------------------------------------------------------------------------------
| IMouseHandler::mouseClicked                                                  |
|                                                                              |
| Process a change in the button state of the mouse.                           |
------------------------------------------------------------------------------*/
bool IMouseHandler::mouseClicked ( IMouseClickEvent& event )
{
  return false;
}

/*------------------------------------------------------------------------------
| IMouseHandler::mouseMoved                                                    |
|                                                                              |
| Process movement of the mouse pointer.                                       |
------------------------------------------------------------------------------*/
bool IMouseHandler::mouseMoved ( IMouseEvent& event )
{
  return false;
}

/*------------------------------------------------------------------------------
| IMouseHandler::mouseEnter                                                    |
|                                                                              |
| Process the mouse pointer entering a window.                                 |
------------------------------------------------------------------------------*/
bool IMouseHandler::mouseEnter ( IEvent& event )
{
  return false;
}

/*------------------------------------------------------------------------------
| IMouseHandler::mouseLeave                                                    |
|                                                                              |
| Process the mouse pointer leaving a window.                                  |
------------------------------------------------------------------------------*/
bool IMouseHandler::mouseLeave ( IEvent& event )
{
  return false;
}

/*------------------------------------------------------------------------------
| IMouseHandler::mousePointerChange                                            |
|                                                                              |
| Allow for setting of the mouse pointer shape.                                |
------------------------------------------------------------------------------*/
bool IMouseHandler::mousePointerChange ( IMousePointerEvent& event )
{
  return false;
}
