// Revision: 13 1.5.1.7 source/ui/baseapp/imousevt.cpp, userinput, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: imousevt.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imousevt.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINFRAMEMGR
#define INCL_WINMESSAGEMGR   // For WinQueryMsgPos.
#define INCL_WINWINDOWMGR
#define INCL_WININPUT
extern "C" {
  #include <iwindefs.h>
  }
#ifdef IC_MOTIF
  #include <X11/Xlib.h>      // For XEvent.
#endif

#include <imousevt.hpp>
#include <icconst.h>
#include <icoordsy.hpp>
#include <iexcept.hpp>
#include <ihandle.hpp>
#include <ikey.hpp>
#include <ipoint.hpp>
#include <ithread.hpp>
#include <iwindow.hpp>

#ifdef IC_MOTIF
#include <iwinpriv.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IMOUSEVT_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| IMouseEvent::IMouseEvent                                                     |
|                                                                              |
| Constructor here for page tuning.                                            |
------------------------------------------------------------------------------*/
IMouseEvent::IMouseEvent ( const IEvent& event )
  : IMouseEvent::Inherited( event )
{ }

/*------------------------------------------------------------------------------
| IMouseEvent::~IMouseEvent                                                    |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IMouseEvent::~IMouseEvent ( )
{ }

/*------------------------------------------------------------------------------
| IMouseEvent::mousePosition                                                   |
|                                                                              |
| Return the mouse pointer position.                                           |
------------------------------------------------------------------------------*/
IPoint IMouseEvent::mousePosition ( ) const
{
#ifdef IC_WIN
  IPoint pt( (short) this->parameter2().number1(),
             (short) this->parameter2().number2() );
  RECT rect;
  GetWindowRect( this->handle(), &rect );
  ISize size( rect.right - rect.left, rect.bottom - rect.top );
#endif
#ifdef IC_PM
  IPoint pt( (short) this->parameter1().number1(),
             (short) this->parameter1().number2() );
  SWP swp;
  WinQueryWindowPos( this->handle(), &swp );
  ISize size( swp.cx, swp.cy );
#endif
#ifdef IC_MOTIF
  XEvent *xevt = (XEvent *) this->parameter2().asUnsignedLong();
  IPoint pt( xevt->xbutton.x, xevt->xbutton.y );
  Dimension width, height;
  XtVaGetValues( this->handle(),
                 XmNwidth, &width,
                 XmNheight, &height,
                 NULL );
  ISize size( width, height );
#endif
  return ICoordinateSystem::convertToApplication( pt, size );
}

/*------------------------------------------------------------------------------
| IMouseEvent::windowUnderPointer                                              |
|                                                                              |
| Get the window handle that the mouse is positioned over.                     |
------------------------------------------------------------------------------*/
IWindowHandle IMouseEvent::windowUnderPointer ( ) const
{
#ifdef IC_PMWIN
#ifdef IC_WIN
  RECT rect;
  GetWindowRect( this->handle(), &rect );
  ISize size( rect.right - rect.left, rect.bottom - rect.top );
#else
  SWP swp;
  WinQueryWindowPos( this->handle(), &swp );
  ISize size( swp.cx, swp.cy );
#endif
  POINTL ptMouse =
     ICoordinateSystem::convertToNative( this->mousePosition(), size ).asPOINTL();
#ifdef IC_WIN
  // Map the point to screen coordinates.  This is necessary because the
  // macro IWINDOWFROMPOINT uses WindowFromPoint instead of
  // ChildWindowFromPoint.  It appears that ChildWindowFromPoint does not
  // return any window beyond its immediate children.
  IMAPWINDOWPOINTS( this->handle(), 0, &ptMouse, 1 );
#endif

  return IWINDOWFROMPOINT( this->handle(), &ptMouse, true );
  // No exception is thrown because handle is always valid in an event object.
#endif // IC_PMWIN

#ifdef IC_MOTIF
  // It is debatable how to handle this on Motif.  We have several controls
  // that are made up of by an aggregate of widgets, such as list box.  We
  // could return the widget in that aggregate that the mouse is over, but
  // this widget may not be the handle for the control.  This means that
  // IWindow::windowWithHandle would return 0 in that case.
  //
  // A solution is to walk up the widget hierarchy until we find the widget
  // that is the top widget for the aggregate.  However, the function to
  // do this returns an IWindowHandle (not an IWindow), so you have to
  // assume that it should return the actual widget that the mouse was over,
  // rather than the top most widget.  This behavior seems necessary for
  // portability reasons.  To get different behavior, the client can access
  // the actual XEvent and can easily get the widget if they need to.
  //
  // Here is the previous implementation:
  //  XEvent *xEvent = (XEvent *) this->parameter2().asUnsignedLong();
  //  return (IWindowHandle) XtWindowToWidget(
  //                 xEvent->xbutton.display, xEvent->xbutton.window );

  XEvent *xEvent = (XEvent *) this->parameter2().asUnsignedLong();
  Widget fWidget = XtWindowToWidget( xEvent->xany.display,
                                     xEvent->xany.window );

  // For the sake of aggregates, let's get the IWindowHandle of the top-
  // most widget that makes up the control.
  IWindow *win = IWindow::windowWithHandle( fWidget);

  // Walk up the chain until we find a widget that is wrappered.
  while ( !win  &&  fWidget )
  {
     fWidget = XtParent( (Widget) fWidget );
     win = IWindow::windowWithHandle( fWidget );
  }

return (IWindowHandle) fWidget;
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| IMouseEvent::isAltKeyDown                                                    |
|                                                                              |
| Determine if the Alt key is also pressed.                                    |
------------------------------------------------------------------------------*/
bool IMouseEvent::isAltKeyDown ( ) const
{
#ifdef IC_WIN
  return (bool) HIBYTE( GetKeyState( VK_ALT ) );
#endif
#ifdef IC_PM
  return ( this->parameter2().number2() & KC_ALT ) ? true : false;
#endif
#ifdef IC_MOTIF
  XEvent
   *xevt = (XEvent*) this->parameter2().asUnsignedLong();
  unsigned int
    modifierMask = xevt->xbutton.state;
  return ( modifierMask & IKey::altModifier() ) ? true : false;
#endif
}

/*------------------------------------------------------------------------------
| IMouseEvent::isCtrlKeyDown                                                   |
|                                                                              |
| Determine if the Ctrl key is also pressed.                                   |
------------------------------------------------------------------------------*/
bool IMouseEvent::isCtrlKeyDown ( ) const
{
#ifdef IC_WIN
  return ( this->parameter1() & MK_CONTROL ) ? true : false;
#endif
#ifdef IC_PM
  return ( this->parameter2().number2() & KC_CTRL ) ? true : false;
#endif
#ifdef IC_MOTIF
  XEvent
   *xevt = (XEvent*) this->parameter2().asUnsignedLong();
  unsigned int
    modifierMask = xevt->xbutton.state;
  return ( modifierMask & ControlMask ) ? true : false;
#endif
}

/*------------------------------------------------------------------------------
| IMouseEvent::isShiftKeyDown                                                  |
|                                                                              |
| Determine if the Shift key is also pressed.                                  |
------------------------------------------------------------------------------*/
bool IMouseEvent::isShiftKeyDown ( ) const
{
#ifdef IC_WIN
  return ( this->parameter1() & MK_SHIFT ) ? true : false;
#endif
#ifdef IC_PM
  return ( this->parameter2().number2() & KC_SHIFT ) ? true : false;
#endif
#ifdef IC_MOTIF
  XEvent
   *xevt = (XEvent*) this->parameter2().asUnsignedLong();
  unsigned int
    modifierMask = xevt->xbutton.state;
  return ( modifierMask & ShiftMask ) ? true : false;
#endif
}

/*------------------------------------------------------------------------------
| IMouseClickEvent::IMouseClickEvent                                           |
|                                                                              |
| Constructor here for page tuning.                                            |
------------------------------------------------------------------------------*/
IMouseClickEvent::IMouseClickEvent ( const IEvent& event )
  : IMouseClickEvent::Inherited( event )
{ }

/*------------------------------------------------------------------------------
| IMouseClickEvent::~IMouseClickEvent                                          |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IMouseClickEvent::~IMouseClickEvent ( )
{ }

/*------------------------------------------------------------------------------
| IMouseClickEvent::mouseButton                                                |
|                                                                              |
| Determine which mouse button is generating the event.                        |
------------------------------------------------------------------------------*/
IMouseClickEvent::MouseButton IMouseClickEvent::mouseButton ( ) const
{
  MouseButton button = button1;  // Default value.

#ifdef IC_PMWIN
  switch ( this->eventId() )
  {
     case WM_BUTTON1DOWN:
     case WM_BUTTON1UP:
     case WM_BUTTON1CLICK:
     case WM_BUTTON1DBLCLK:
        break;                   // Use the default value.

     case WM_BUTTON2DOWN:
     case WM_BUTTON2UP:
     case WM_BUTTON2CLICK:
     case WM_BUTTON2DBLCLK:
        button = button2;
        break;

     case WM_BUTTON3DOWN:
     case WM_BUTTON3UP:
     case WM_BUTTON3CLICK:
     case WM_BUTTON3DBLCLK:
        button = button3;
        break;

     case WM_CHORD:
        button = buttonChord;
        break;

     default:
        ITHROWLIBRARYERROR( IC_INVALIDEVENT,
                            IBaseErrorInfo::invalidRequest,
                            IException::recoverable );
        break;
   } /* endswitch */
#endif // IC_PMWIN

#ifdef IC_MOTIF
  XEvent *xEvent = (XEvent *) this->parameter2().asUnsignedLong();
  if ( this->eventId() == WM_CHORD )
  {
     button = buttonChord;
  }
  else
  {
     switch ( xEvent->xbutton.button )
     {
        case 1:
           button = button1;
           break;
        case 2:
           button = button2;
           break;
        case 3:
           button = button3;
           break;
        default :
           ITHROWLIBRARYERROR( IC_INVALIDEVENT,
                               IBaseErrorInfo::invalidRequest,
                               IException::recoverable );
     }
  }
#endif

  return button;
}

/*------------------------------------------------------------------------------
| IMouseClickEvent::mouseAction                                                |
|                                                                              |
| Determine the action taken on the mouse.                                     |
------------------------------------------------------------------------------*/
IMouseClickEvent::MouseAction IMouseClickEvent::mouseAction ( ) const
{
  MouseAction action = down;     // Default value.
  switch ( this->eventId() )
  {
     case WM_BUTTON1DOWN:
     case WM_BUTTON2DOWN:
     case WM_BUTTON3DOWN:
     case WM_CHORD:              // Button 1 and 2 held down.
        break;                   // Use the default value.

     case WM_BUTTON1UP:
     case WM_BUTTON2UP:
     case WM_BUTTON3UP:
        action = up;
        break;

     case WM_BUTTON1CLICK:
     case WM_BUTTON2CLICK:
     case WM_BUTTON3CLICK:
        action = click;
        break;

     case WM_BUTTON1DBLCLK:
     case WM_BUTTON2DBLCLK:
     case WM_BUTTON3DBLCLK:
        action = doubleClick;
        break;
     default:
        ITHROWLIBRARYERROR( IC_INVALIDEVENT,
                            IBaseErrorInfo::invalidRequest,
                            IException::recoverable );
        break;
  } /* endswitch */

  return action;
}


#ifdef IC_MOTIFWIN
class IMousePointerEventData {
public:
  IMousePointerEventData ( );
 ~IMousePointerEventData ( );
IPointerHandle
  fNewMousePointer;
}; // IMousePointerEventData

IMousePointerEventData::IMousePointerEventData ( )
 : fNewMousePointer( 0 )
{ }

IMousePointerEventData::~IMousePointerEventData ( )
{ }
#endif // IC_MOTIFWIN

/*------------------------------------------------------------------------------
| IMousePointerEvent::IMousePointerEvent                                       |
|                                                                              |
| Constructor here for page tuning.                                            |
------------------------------------------------------------------------------*/
IMousePointerEvent::IMousePointerEvent ( const IEvent& event )
  : IMousePointerEvent::Inherited( event )
#ifdef IC_MOTIFWIN
  , fData( new IMousePointerEventData() )
#else
  , fData( 0 )
#endif
{ }

/*------------------------------------------------------------------------------
| IMousePointerEvent::~IMousePointerEvent                                      |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IMousePointerEvent::~IMousePointerEvent ( )
{
#ifdef IC_MOTIFWIN
  delete fData;
#endif
}

/*------------------------------------------------------------------------------
| IMousePointerEvent::windowId                                                 |
|                                                                              |
| Identifier of the window under the mouse pointer.                            |
------------------------------------------------------------------------------*/
unsigned long IMousePointerEvent::windowId ( ) const
{
#ifdef IC_WIN
  return IIDOF( this->parameter1() );
#endif
#ifdef IC_PM
  return this->parameter1().number1();
#endif
#ifdef IC_MOTIF
  // Model this function after IMouseEvent::windowUnderPointer.
  XEvent
   *xEvent = ( XEvent* ) this->parameter2().asUnsignedLong();
  Widget
    widget = XtWindowToWidget( xEvent->xany.display,
                               xEvent->xany.window );

  // For the sake of aggregates, let's get the IWindowHandle of
  // the topmost widget that makes up the control.
  IWindow
   *mouseWindow = IWindow::windowWithHandle( widget );

  // Walk up the parent window chain until we find a widget
  // that is wrappered.
  while ( !mouseWindow  &&  widget )
  {
     widget = XtParent( ( Widget ) widget );
     mouseWindow = IWindow::windowWithHandle( widget );
  }
  return ( mouseWindow ? mouseWindow->id() : 0 );
#endif
}

/*------------------------------------------------------------------------------
| IMousePointerEvent::setMousePointer                                          |
|                                                                              |
| Set the mouse pointer to be displayed at this time.                          |
------------------------------------------------------------------------------*/
IMousePointerEvent& IMousePointerEvent::setMousePointer
                                          ( const IPointerHandle& mousePointer )
{
#ifdef IC_PM
  this->setResult( (unsigned long) mousePointer );
#endif
#ifdef IC_WIN
  fData->fNewMousePointer = mousePointer;
  ISETPOINTER( mousePointer );
  this->setResult( true );
#endif
#ifdef IC_MOTIF
  fData->fNewMousePointer = mousePointer;
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IMousePointerEvent::mousePointer                                             |
|                                                                              |
| Return the new mouse pointer set into the event.                             |
------------------------------------------------------------------------------*/
IPointerHandle IMousePointerEvent::mousePointer ( ) const
{
#ifdef IC_PM
  IPointerHandle
    newMousePtr( (unsigned long) this->result() );
#endif
#ifdef IC_MOTIFWIN
  IPointerHandle
    newMousePtr( fData->fNewMousePointer );
#endif
  return newMousePtr;
}

/*------------------------------------------------------------------------------
| IMousePointerEvent::defaultMousePointer                                      |
|                                                                              |
| Return the mouse pointer used if no action is taken.                         |
------------------------------------------------------------------------------*/
IPointerHandle IMousePointerEvent::defaultMousePointer ( ) const
{
#ifdef IC_WIN
  return IPointerHandle( GetCursor() );
#endif
#ifdef IC_PM
  return IPointerHandle( (unsigned long) this->parameter2() );
#endif
#ifdef IC_MOTIF
  return this->dispatchingWindow()->mousePointer();
#endif
}

/*------------------------------------------------------------------------------
| IMousePointerEvent::mousePosition                                            |
|                                                                              |
| Return the position of the mouse pointer within the dispatching window.      |
------------------------------------------------------------------------------*/
IPoint IMousePointerEvent::mousePosition ( ) const
{
#ifdef IC_PMWIN
  POINTL
    pointl;
#ifdef IC_PM
  IQUERYMSGPOS( IThread::current().anchorBlock(), &pointl );
#else
  IQUERYMSGPOS( 0, &pointl );
#endif
  IPoint
    screenPoint( pointl );
  screenPoint =
    ICoordinateSystem::convertToApplication( screenPoint,
                                             IWindow::desktopWindow()->size() );
  IPoint
    mousePoint( IWindow::mapPoint( screenPoint,
                                   IWindow::desktopWindow()->handle(),
                                   this->dispatchingWindow()->handle() ) );
#endif
#ifdef IC_MOTIF
  XEvent *xevt = (XEvent *) this->parameter2().asUnsignedLong();
  IPoint
    pt( xevt->xbutton.x, xevt->xbutton.y );
  Dimension
    width, height;
  XtVaGetValues( this->handle(),
                 XmNwidth, &width,
                 XmNheight, &height,
                 NULL );
  ISize
    windowSize( width, height );
  IPoint
    mousePoint( ICoordinateSystem::convertToApplication( pt, windowSize ) );
#endif
  return mousePoint;
}
