// Revision: 10 1.5.2.10 source/ui/baseapp/imnitem.cpp, menu, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: imnitem.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imnitem.cpp                                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C"
{
  #define INCL_WINMENUS
  #define INCL_WINWINDOWMGR
  #include <iwindefs.h>
}

#include <imenu.hpp>
#include <iexcept.hpp>
#include <imnitem.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMNITEM_CPP_
  #include <ipagetun.h>
#endif

//***************************************************************
// Public styles.
//***************************************************************

const IMenuItem::Style
  IMenuItem::noStyle            = 0,
#ifdef IC_WIN
  IMenuItem::textItem           ( 0, IMIS_TEXT ),         // 0x00000001
  IMenuItem::unavailable        ( 0, IMIS_STATIC ),       // 0x00000002
#endif
#ifdef IC_MOTIFPM
  IMenuItem::textItem           = MIS_TEXT,               // 0x00000001
  IMenuItem::unavailable        = MIS_STATIC,             // 0x00000100
#endif
  IMenuItem::bitmapItem         = MIS_BITMAP,             // 0x00000002
  IMenuItem::submenu            = MIS_SUBMENU,            // 0x00000010
  IMenuItem::separator          = MIS_SEPARATOR,          // 0x00000004
  IMenuItem::drawItem           = MIS_OWNERDRAW,          // 0x00000008
  IMenuItem::postSystemCommand  = MIS_SYSCOMMAND,         // 0x00000040
  IMenuItem::postHelp           = MIS_HELP,               // 0x00000080
  IMenuItem::buttonSeparator    = MIS_BUTTONSEPARATOR,    // 0x00000200
  IMenuItem::split              = MIS_BREAK,              // 0x00000400
  IMenuItem::splitWithSeparator = MIS_BREAKSEPARATOR,     // 0x00000800
  IMenuItem::classDefaultStyle  = 0;
  const unsigned long IMenuItem::atEnd = (unsigned long)ULONG_MAX;


  IMenuItem::Style IMenuItem::currentDefaultStyle = 0;

//***************************************************************
// Public attributes
//***************************************************************

const IMenuItem::Attribute
  IMenuItem::noAttribute           = 0,
  IMenuItem::noDismiss             = MIA_NODISMISS,       // 0x00000020
  IMenuItem::framed                = MIA_FRAMED,          // 0x00001000
  IMenuItem::checked               = MIA_CHECKED,         // 0x00002000
  IMenuItem::disabled              = MIA_DISABLED,        // 0x00004000
  IMenuItem::highlighted           = MIA_HILITED,         // 0x00008000
  IMenuItem::classDefaultAttribute = noAttribute;


  IMenuItem::Attribute IMenuItem::currentDefaultAttribute = noAttribute;

/*------------------------------------------------------------------------------
| IMenuItem::IMenuItem                                                         |
------------------------------------------------------------------------------*/
IMenuItem::IMenuItem  ( unsigned long       id,
                        const Style         &style,
                        const Attribute     &attribute)
 : strClItemText(),
   iPosition(IMenuItem::atEnd),
   afStyle(style.asUnsignedLong()),
   afExtendedStyle(style.asExtendedUnsignedLong()),
   afHelpId(0),
   afAttribute(attribute.asUnsignedLong()),
   itemId(id),
   hwndSubmenu(0),
   hBitmapHandle(0)
{ }

/*------------------------------------------------------------------------------
| IMenuItem::IMenuItem                                                         |
------------------------------------------------------------------------------*/
IMenuItem::IMenuItem( const IMenuItem&  menuItem )
  : strClItemText( menuItem.strClItemText ),
    iPosition( menuItem.iPosition ),
    afStyle( menuItem.afStyle ),
    afExtendedStyle( menuItem.afExtendedStyle ),
    afAttribute( menuItem.afAttribute ),
    itemId( menuItem.itemId ),
    afHelpId( menuItem.afHelpId ),
    hwndSubmenu( menuItem.hwndSubmenu ),
    hBitmapHandle( menuItem.hBitmapHandle )
{ }

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IMenuItem::operator=                                                         |
------------------------------------------------------------------------------*/
IMenuItem& IMenuItem::operator=( const IMenuItem&  menuItem )
{
  if (this == &menuItem)
    return *this;

  strClItemText   = menuItem.strClItemText;
  iPosition       = menuItem.iPosition;
  afStyle         = menuItem.afStyle;
  afExtendedStyle = menuItem.afExtendedStyle;
  afAttribute     = menuItem.afAttribute;
  itemId          = menuItem.itemId;
  afHelpId        = menuItem.afHelpId;
  hwndSubmenu     = menuItem.hwndSubmenu;
  hBitmapHandle   = menuItem.hBitmapHandle;

  return *this;
}
#endif

/*------------------------------------------------------------------------------
| IMenuItem::~IMenuItem                                                        |
------------------------------------------------------------------------------*/
IMenuItem::~IMenuItem()
{ }

/*------------------------------------------------------------------------------
| IMenuItem::setDrawItem                                                       |
------------------------------------------------------------------------------*/
IMenuItem&  IMenuItem::setDrawItem    ( bool  state)
{
//  ITRACE_MOTIF_NOP();

//#ifdef IC_PMWIN                           //AJ F26433
   if (state)
   {
#ifdef IC_WIN
     afExtendedStyle &= ~(textItem.asExtendedUnsignedLong());
#else
     afStyle &= ~(textItem.asUnsignedLong());
#endif
     afStyle &= ~(separator.asUnsignedLong() | bitmapItem.asUnsignedLong() );
     afStyle |= drawItem.asUnsignedLong();
     hBitmapHandle = IBitmapHandle();
     strClItemText = IString();
   }
   else
     afStyle &= ~drawItem.asUnsignedLong();

//#endif  // IC_PMWIN

//#ifdef IC_MOTIF                         //AJ F26433
//  hBitmapHandle = IBitmapHandle();
 // strClItemText = IString();
 // // X/Motif does not support this function
//#endif // IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IMenuItem::setSeparator                                                      |
------------------------------------------------------------------------------*/
IMenuItem& IMenuItem::setSeparator   ( )
{
#ifdef IC_PMWIN
#ifdef IC_WIN
   afExtendedStyle &= ~(textItem.asExtendedUnsignedLong());
#else
   afStyle &= ~(textItem.asUnsignedLong());
#endif
   afStyle &= ~(drawItem.asUnsignedLong() | bitmapItem.asUnsignedLong() );
   afStyle |= separator.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
   afStyle &= ~(MIS_TEXT | MIS_BITMAP |
                drawItem.asUnsignedLong() );
   afStyle |= separator.asUnsignedLong();
#endif //IC_MOTIF

   hBitmapHandle = IBitmapHandle();
   strClItemText = IString();
   return *this;
}

/*------------------------------------------------------------------------------
| IMenuItem::setText                                                           |
------------------------------------------------------------------------------*/
IMenuItem&  IMenuItem::setText( const char*         newText )
{
#ifdef IC_PMWIN
   afStyle &= ~(bitmapItem.asUnsignedLong() | separator.asUnsignedLong() |
                drawItem.asUnsignedLong() );
#ifdef IC_WIN
   afExtendedStyle |= textItem.asExtendedUnsignedLong();
#else
   afStyle |= textItem.asUnsignedLong();
#endif
#endif //IC_PMWIN

#ifdef IC_MOTIF
   afStyle &= ~(MIS_BITMAP | separator.asUnsignedLong() |
                drawItem.asUnsignedLong() );
   afStyle |= MIS_TEXT;
#endif // IC_MOTIF

   strClItemText = IString(newText);
   hBitmapHandle = 0;
   return *this;
}

/*------------------------------------------------------------------------------
| IMenuItem::setBitmap                                                         |
------------------------------------------------------------------------------*/
IMenuItem&  IMenuItem::setBitmap(const IBitmapHandle& hbmpMenuItem)
{
#ifdef IC_PMWIN
#ifdef IC_WIN
   afExtendedStyle &= ~(textItem.asExtendedUnsignedLong());
#else
   afStyle &= ~(textItem.asUnsignedLong());
#endif
   afStyle &= ~(drawItem.asUnsignedLong() | separator.asUnsignedLong() );
   afStyle |= bitmapItem.asUnsignedLong();
#endif //IC_PMWIN

#ifdef IC_MOTIF
   afStyle &= ~(MIS_TEXT | separator.asUnsignedLong() |
                drawItem.asUnsignedLong() );
   afStyle |= MIS_BITMAP;
#endif //IC_MOTIF

   strClItemText = IString();
   hBitmapHandle = hbmpMenuItem;
   return *this;
}


/*------------------------------------------------------------------------------
| IMenuItem::setLayout                                                         |
------------------------------------------------------------------------------*/
IMenuItem& IMenuItem::setLayout(IMenuItem::LayoutType newLayout)
{
#ifdef IC_WIN
   afStyle &= ~(split.asUnsignedLong() | splitWithSeparator.asUnsignedLong() );
#else
   afStyle &= ~(split.asUnsignedLong() | splitWithSeparator.asUnsignedLong() |
                buttonSeparator.asUnsignedLong());
#endif

   switch(newLayout)
   {
     case splitLayout:
       afStyle |= split.asUnsignedLong();
       break;
#ifdef IC_PMWIN
     case splitWithSeparatorLayout:
       afStyle |= splitWithSeparator.asUnsignedLong();
       break;
#endif
#ifdef IC_PM
     case buttonSeparatorLayout:
       afStyle |= buttonSeparator.asUnsignedLong();
       break;
#endif
   } /* endswitch */
   return *this;
}

/*------------------------------------------------------------------------------
| IMenuItem::setCommand                                                        |
------------------------------------------------------------------------------*/
IMenuItem& IMenuItem::setCommand(IMenuItem::CommandType newMessageType)
{
  ITRACE_WIN_NOP();

#ifdef IC_MOTIFPM
   afStyle &= ~( postSystemCommand.asUnsignedLong() |
                 postHelp.asUnsignedLong() );
   switch (newMessageType)
   {
     case systemCommand:
       afStyle |= postSystemCommand.asUnsignedLong();
       break;
     case helpCommand:
       afStyle |= postHelp.asUnsignedLong();
       break;
   } /* endswitch */
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMenuItem::layoutType                                                        |
------------------------------------------------------------------------------*/
IMenuItem::LayoutType IMenuItem::layoutType  ( ) const
{
   if ( afStyle & split.asUnsignedLong() )
      return splitLayout;
#ifdef IC_PMWIN
   if ( afStyle & splitWithSeparator.asUnsignedLong() )
      return splitWithSeparatorLayout;
#ifdef IC_PM
   if ( afStyle & buttonSeparator.asUnsignedLong() )
      return buttonSeparatorLayout;
#endif //IC_PM
#endif //IC_PMWIN
   return normalLayout;
}


/*------------------------------------------------------------------------------
| IMenuItem::commandType                                                       |
------------------------------------------------------------------------------*/
IMenuItem::CommandType IMenuItem::commandType ( ) const
{
#ifdef IC_MOTIFPM
   if (afStyle & postSystemCommand.asUnsignedLong() )
      return systemCommand;
   else
      if (afStyle & postHelp.asUnsignedLong() )
         return helpCommand;
      else
#endif
         return command;
}

/*------------------------------------------------------------------------------
| IMenuItem::defaultStyle                                                      |
|                                                                              |
| Return the default style for new menu item objects.                          |
------------------------------------------------------------------------------*/
IMenuItem::Style  IMenuItem::defaultStyle()
{
  return currentDefaultStyle;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMenuItem::setDefaultStyle                                                   |
|                                                                              |
| Set the default style for new menu item objects.                             |
------------------------------------------------------------------------------*/
void IMenuItem::setDefaultStyle( const IMenuItem::Style& newDefault )
{
  currentDefaultStyle = newDefault;
}
#endif

/*------------------------------------------------------------------------------
| IMenuItem::defaultAttribute                                                  |
|                                                                              |
| Return the default attribute for new menu item objects.                      |
------------------------------------------------------------------------------*/
IMenuItem::Attribute  IMenuItem::defaultAttribute()
{
  return currentDefaultAttribute;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMenuItem::setDefaultAttribute                                               |
|                                                                              |
| Set the default attribute for new menu item objects.                         |
------------------------------------------------------------------------------*/
void IMenuItem::setDefaultAttribute( const IMenuItem::Attribute& newDefault )
{
  currentDefaultAttribute = newDefault;
}
#endif

/*------------------------------------------------------------------------------
| IMenuItem::setSubmenuHandle                                                  |
------------------------------------------------------------------------------*/
IMenuItem& IMenuItem::setSubmenuHandle ( const IMenuHandle &menuHandle )
{
  // Set the handle to the one passed and then check if for validity
  hwndSubmenu = menuHandle;
#ifdef IC_PMWIN
  if ( hwndSubmenu )
    // If non-null, then set style bit for submenu
    afStyle |= submenu.asUnsignedLong();
  else
    // otherwise if null, remove style bit for submenu
    afStyle &= ~(submenu.asUnsignedLong());
#endif //IC_PMWIN

#ifdef IC_MOTIF
  hwndSubmenu = menuHandle;
  afStyle |= submenu.asUnsignedLong();
#endif //IC_MOTIF

  return *this;
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IMenuItem::convertToGUIStyle                                                 |
|                                                                              |
| Returns base/extended style for the underlying GUI.                          |
------------------------------------------------------------------------------*/
unsigned long IMenuItem::convertToGUIStyle(const IBitFlag& guiStyle,
                                           bool bExtOnly) const
{
  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    return( (extendedStyle() & IMIS_EXTMASK) );
  }

  // MIS_ styles have a one-to-one correspondence to our style bits, and
  // inhabit the lower word of the base GUI style.  Therefore, return
  // that portion asis, masking out the upper word.
  return( (guiStyle.asUnsignedLong() & IMIS_MASK) );
}
#endif //IC_PMWIN

/*------------------------------------------------------------------------------
| IMenuItem::setText                                                           |
------------------------------------------------------------------------------*/
IMenuItem&  IMenuItem::setText(const IResourceId& textResId)
{
   return(setText((char*)textResId.resourceLibrary().loadString(textResId)));
}

/*------------------------------------------------------------------------------
| IMenuItem::setBitmap                                                         |
------------------------------------------------------------------------------*/
IMenuItem&  IMenuItem::setBitmap(const IResourceId& bitmapResId)
{
  return(setBitmap(bitmapResId.resourceLibrary().loadBitmap(bitmapResId)) );
}

IMenuItem& IMenuItem :: setSelectable(bool state)
{
#ifndef IC_WIN
   if (!state)
     afStyle |= unavailable.asUnsignedLong();
   else
     afStyle &= ~unavailable.asUnsignedLong();
#endif
#ifdef IC_WIN
   if (!state)
     afExtendedStyle |= unavailable.asExtendedUnsignedLong();
   else
     afExtendedStyle &= ~(unavailable.asExtendedUnsignedLong());
#endif
   return *this;
}

#ifdef IC_PM
IMenuItem& IMenuItem :: setNoDismiss(bool state)
{
   ITRACE_WIN_NOP();
   ITRACE_MOTIF_NOP();

   if (state)
     afAttribute |= noDismiss.asUnsignedLong();
   else
     afAttribute &= ~noDismiss.asUnsignedLong();
   return *this;
}

IMenuItem& IMenuItem :: setFramed(bool state)
{
   ITRACE_WIN_NOP();
   ITRACE_MOTIF_NOP();

   if (state)
     afAttribute |= framed.asUnsignedLong();
   else
     afAttribute &= ~framed.asUnsignedLong();
   return *this;
}
#endif

#ifdef IC_MOTIFWIN
#ifndef IC_MOTIFWIN_FLAGNOP
IMenuItem& IMenuItem :: setNoDismiss(bool)
{
   ITRACE_WIN_NOP();
   ITRACE_MOTIF_NOP();
   return *this;
}

IMenuItem& IMenuItem :: setFramed(bool)
{
   ITRACE_WIN_NOP();
   ITRACE_MOTIF_NOP();
   return *this;
}
#endif  // IC_MOTIFWIN_FLAGNOP
#endif  // MotifWin

IMenuItem& IMenuItem :: setChecked(bool state)
{
   if (state)
     afAttribute |= checked.asUnsignedLong();
   else
     afAttribute &= ~checked.asUnsignedLong();
   return *this;
}

IMenuItem& IMenuItem :: setDisabled(bool state)
{
   if (state)
     afAttribute |= disabled.asUnsignedLong();
   else
     afAttribute &= ~disabled.asUnsignedLong();
   return *this;
}

#ifndef IC_MOTIF
IMenuItem& IMenuItem :: setHighlighted(bool state)
{
  ITRACE_MOTIF_NOP();

   if (state)
     afAttribute |= highlighted.asUnsignedLong();
   else
     afAttribute &= ~highlighted.asUnsignedLong();
   return *this;
}
#endif  // not Motif

#ifdef IC_MOTIF
#ifndef IC_MOTIF_FLAGNOP
IMenuItem& IMenuItem :: setHighlighted(bool)
{
  ITRACE_MOTIF_NOP();
  // X/Motif does not support this behavior
   return *this;
}
#endif  // IC_MOTIF_FLAGNOP
#endif  // Motif

bool IMenuItem :: isSelectable() const
{
#ifndef IC_WIN
   return !(afStyle & unavailable.asUnsignedLong());
#endif
#ifdef IC_WIN
   return !(afExtendedStyle & unavailable.asExtendedUnsignedLong() );
#endif
}

bool IMenuItem :: isDrawItem() const
{
   return ((bool) (afStyle & drawItem.asUnsignedLong()) ? TRUE : FALSE );
}

bool IMenuItem :: isSeparator() const
{
   return ((bool)(afStyle & separator.asUnsignedLong()) ? TRUE : FALSE );
}

bool IMenuItem :: isText() const
{
#ifdef IC_WIN
   return ((bool)(afExtendedStyle & textItem.asExtendedUnsignedLong()) ? TRUE : FALSE);
#endif
#ifndef IC_WIN
   return ((bool)(afStyle & textItem.asUnsignedLong()) ? TRUE : FALSE);
#endif
}

bool IMenuItem :: isBitmap() const
{
   return ((bool)(afStyle & bitmapItem.asUnsignedLong()) ? TRUE : FALSE);
}

bool IMenuItem :: isSubmenu() const
{
   return ((bool)(afStyle & submenu.asUnsignedLong()) ? TRUE : FALSE);
}


#ifdef IC_PM
bool IMenuItem :: isNoDismiss() const
{
   return ((bool)(afAttribute & noDismiss.asUnsignedLong()) ? TRUE : FALSE);
}

bool IMenuItem :: isFramed() const
{
   return ((bool)(afAttribute & framed.asUnsignedLong()) ? TRUE : FALSE);
}
#endif  // IC_PM


#ifdef IC_MOTIFWIN
#ifndef IC_MOTIFWIN_FLAGNOP
bool IMenuItem :: isNoDismiss() const
{
   return false;
}

bool IMenuItem :: isFramed() const
{
   return false;
}
#endif  // IC_MOTIFWIN_FLAGNOP
#endif  // IC_MOTIFWIN


bool IMenuItem :: isChecked() const
{
   return ((bool)(afAttribute & checked.asUnsignedLong()) ? TRUE : FALSE);
}

bool IMenuItem :: isDisabled() const
{
   return ((bool)(afAttribute & disabled.asUnsignedLong()) ? TRUE : FALSE);
}

#ifndef IC_MOTIF_FLAGNOP
bool IMenuItem :: isHighlighted() const
{
   return ((bool)(afAttribute & highlighted.asUnsignedLong()) ? TRUE : FALSE);
}
#endif  // IC_MOTIF_FLAGNOP
