// Revision: 04 1.4.2.4 source/ui/baseapp/imenuprv.cpp, menu, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: imenuprv.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imenuprv.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1997             *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

// #define IC_TRACE_ALL

extern "C"
{
  #include <iwindefs.h>
   #ifdef IC_MOTIF
     #include <Xm/RowColumn.h>
     #include <Xm/MenuShell.h>
     #include <Xm/MainW.h>
     #include <X11/IntrinsicP.h>
     #include <X11/CoreP.h>
   #endif //IC_MOTIF
}

#include <imenuprv.hpp>
#include <ibidiset.hpp>
#include <iframe.hpp>
#include <iframprv.hpp>
#include <imenu.hpp>
#include <imnitem.hpp>
#include <iplatfrm.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <iwcname.hpp>
#include <iwindow.hpp>
#ifdef IC_MOTIF
  #include <ithread.hpp>
  #include <ithreadp.hpp>
#endif

#ifdef IC_WU
  #include "wuexten.h"   // Get Wind/U Extension functions prototype
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMENUPRV_CPP_
  #include <ipagetun.h>
#endif

// Initialize static objects and notification strings.
#pragma data_seg(ICLStaticConst)
#ifdef IC_MOTIFPM
  char* const IMenuAttribute::menuAttrName = "ThisWindowIsAnIMenu";
#endif
#pragma data_seg()

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IMenuItemHandle::isOwnerDrawStyle                                            |
------------------------------------------------------------------------------*/
bool IMenuItemHandle::isOwnerDrawStyle() const
{
   return IMenuData::isHandleOwnerDrawStyle( this->handle() );
}
#endif //IC_MOTIF

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IMenuAttribute::IMenuAttribute                                               |
------------------------------------------------------------------------------*/
IMenuAttribute::IMenuAttribute(IMenu* menu)
    : fMenu(menu) {}

/*------------------------------------------------------------------------------
| IMenuAttribute::operator=                                                    |
------------------------------------------------------------------------------*/
bool IMenuAttribute::operator=(const IAttribute& anAttribute)
{
  const IMenuAttribute& rhs = dynamic_cast<const IMenuAttribute&>(anAttribute);
  fMenu = rhs.fMenu;
  return true;
}

/*------------------------------------------------------------------------------
| IMenuAttribute::operator==                                                   |
------------------------------------------------------------------------------*/
bool IMenuAttribute::operator==(const IAttribute& anAttribute) const
{
  const IMenuAttribute& rhs = dynamic_cast<const IMenuAttribute&>(anAttribute);
  if (rhs.fMenu == this->fMenu)
     return true;
  else
     return false;
}

/*------------------------------------------------------------------------------
| IMenuAttribute::clone                                                        |
------------------------------------------------------------------------------*/
IAttribute* IMenuAttribute::clone() const
{
  IMenuAttribute* newOne = new IMenuAttribute(fMenu);
  return newOne;
}
#endif //IC_MOTIFPM


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| findIdForPopup                                                               |
|  Local function to locate a menuhandle within a specific menu                |
------------------------------------------------------------------------------*/
unsigned long IMenuPrivate::findIdForPopup( const IMenuHandle& hBaseMenu,
                                            const IMenuHandle& hMenuToFind )
{
  unsigned long idFound = 0;
  if ((hBaseMenu) && (hMenuToFind))
  {
    //*******************************************************************
    // Begin with first item (0 based) and get count of items
    //*******************************************************************
    int itemPos = 0;
    int itemCount = GetMenuItemCount( hBaseMenu );

    //*******************************************************************
    // Look at each item in the menu to see if it has the specified id
    //*******************************************************************
    while ( itemPos < itemCount )
    {
      //*****************************************************************
      // Get id for this menu item
      //*****************************************************************
      MENUITEMINFO miItem;
      miItem.cbSize = sizeof(MENUITEMINFO);
      miItem.fMask = MIIM_STATE | MIIM_ID | MIIM_SUBMENU;
      GetMenuItemInfo( hBaseMenu, itemPos, TRUE, &miItem );

      idFound = miItem.wID;

      //*****************************************************************
      // Check for match and if found save position and exit
      //*****************************************************************
      if ( miItem.hSubMenu == hMenuToFind )
        return idFound;

      //*****************************************************************
      // Otherwise if submenu found, navigate it before moving on
      //*****************************************************************
      if ( miItem.hSubMenu )
        if (idFound = findIdForPopup( miItem.hSubMenu, hMenuToFind ))
          return idFound;

      //*****************************************************************
      // Otherwise no match so go to next menu position
      //*****************************************************************
      itemPos++;
    }
  }
  return 0;
}

/*------------------------------------------------------------------------------
| locateMenuItem                                                               |
|  Local function to locate a menu item within a specific menu                 |
------------------------------------------------------------------------------*/
bool IMenuPrivate::locateMenuItem( IFrameWindow*      owner,
                                      const IMenuHandle& hMenu,
                                      unsigned long      itemId,
                                      long*              pItemPos,
                                      IMenuHandle*       pMenu )
{
  // If id is 0, invalid so exit
  if (itemId)
  {
    //*******************************************************************
    // Begin with first item (0 based) and get count of items
    //*******************************************************************
    int itemPos = 0;
    int itemCount = GetMenuItemCount( hMenu );
    int idFound = 0;

    //*******************************************************************
    // Look at each item in the menu to see if it has the specified id
    //*******************************************************************
    while ( itemPos < itemCount )
    {
      //*****************************************************************
      // Depending on platform, get id differently
      //*****************************************************************
      if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
      {
        //***************************************************************
        // Get the id of the item at the current position
        //***************************************************************
        idFound = GetMenuItemID( hMenu, itemPos );

        //***************************************************************
        // If error, item might be a submenu so query for a menu handle
        //***************************************************************
        if ( idFound == 0xFFFFFFFFul )
        {
          HMENU hPop = GetSubMenu( hMenu, itemPos );

          //*************************************************************
          // If a popup, navigate its menu before checking this item
          //*************************************************************
          if (locateMenuItem( owner, hPop, itemId, pItemPos, pMenu ))
            return true;

          //*************************************************************
          // If a menu handle is returned, then get id from lookup table
          //*************************************************************
          if (hPop)
            idFound = (int)getItemIdFromLookUpTable( owner, hPop );
        }
      }
      //*****************************************************************
      // Otherwise, use extended menu api to get id
      //*****************************************************************
      else
      {
        //***************************************************************
        // Get the info for the item at the current position
        //***************************************************************
        MENUITEMINFO miItem;
        miItem.cbSize = sizeof(MENUITEMINFO);
        miItem.fMask = MIIM_STATE | MIIM_ID | MIIM_SUBMENU;
        GetMenuItemInfo( hMenu, itemPos, TRUE, &miItem );

        //***************************************************************
        // Check if this item is a pulldown menu
        //***************************************************************
        if ( miItem.hSubMenu )
        {
          //*************************************************************
          // If a pulldown, navigate its menu before checking this item
          //*************************************************************
          if (locateMenuItem( owner, miItem.hSubMenu, itemId, pItemPos, pMenu ))
            return true;
        }
        //***************************************************************
        // If not a pulldown, or no match in pulldown, use this id
        //***************************************************************
        idFound = miItem.wID;
      }

      //*****************************************************************
      // Check for match and if found save position and exit
      //*****************************************************************
      if ( idFound == itemId )
      {
        *pItemPos = (long)itemPos;
        if (pMenu)
          *pMenu = hMenu;
        return true;
      }
      //*****************************************************************
      // Otherwise no match so go to next menu position
      //*****************************************************************
      else itemPos++;
    }
  }
  return false;
}

/*------------------------------------------------------------------------------
| locateFrame                                                                  |
|  Local function to find the IFrameWindow for an IWindow passed               |
------------------------------------------------------------------------------*/
IFrameWindow* IMenuPrivate::locateFrame( IWindow* window )
{
  HWND tempWindow = window->handle();
  bool exit = false;

  while (!exit)
  {
    // Query if window to check is a frame window, if so return this window
    IWindowClassName className( tempWindow );
    if ((className.asString().indexOf(WC_FRAME) == 1) ||
        (className == (const char*)WC_DIALOG ) )
      exit = true;
    else
    {
      // Get the parent handle and if valid, loop again, otherwise exit.
      HWND parentWindow = IPARENTOF( tempWindow );
      if (parentWindow)
        tempWindow = parentWindow;
      else
      {
        tempWindow=0;
        exit = true;
      }
    }
  }
  // If we have a window handle, then see if it is already wrappered
  if (tempWindow)
  {
    // If wrappered, then use it to access the lookup table later.
    IWindow* pwin = IWindow::windowWithHandle(tempWindow);
    if (pwin)
      return (IFrameWindow*)pwin;
  }
  return 0;
}

/*------------------------------------------------------------------------------
| addToLookUpTable                                                             |
|  Local function used to build mapping table for menu handles to item ids     |
|  since this is NOT supported in Windows 3.1 or NT old shell                  |
------------------------------------------------------------------------------*/
void IMenuPrivate::addToLookUpTable( IFrameWindow*      owner,
                                     const IMenuHandle& menuHandle,
                                     unsigned long      itemId )
{
  /********************************************************************/
  /* If invalid owner, just exit (don't touch table)                  */
  /********************************************************************/
  if (!owner)
    return;

  /********************************************************************/
  /* Create a new menu lookup table entry and fill with info passed   */
  /********************************************************************/
  FMTABLE* pFmTable = new FMTABLE;
  pFmTable->fId = itemId;
  pFmTable->fHMenu = (HMENU)menuHandle;

  /********************************************************************/
  /* Default the next entry to null                                   */
  /********************************************************************/
  pFmTable->pfmtNext = 0;

  /********************************************************************/
  /* Check if there are current entries, and if so chain to list      */
  /********************************************************************/
  if (owner->fFrameWindowData->fpSubmenuList)
    pFmTable->pfmtNext = (FMTABLE*)(owner->fFrameWindowData->fpSubmenuList);

  /********************************************************************/
  /* Finally, since this is now the start of the list, save it        */
  /********************************************************************/
  owner->fFrameWindowData->fpSubmenuList = (void*)pFmTable;
  return;
}


/*------------------------------------------------------------------------------
| removeFromLookUpTable                                                        |
|  Local function used to remove item id from mapping table                    |
------------------------------------------------------------------------------*/
void IMenuPrivate::removeFromLookUpTable( IFrameWindow*      owner,
                                          const IMenuHandle& menuHandle,
                                          unsigned long      itemId )
{
  /********************************************************************/
  /* If invalid owner, just exit (don't touch table)                  */
  /********************************************************************/
  if (!owner)
    return;

  /********************************************************************/
  /* Start at top and keep searching as long as entries exist         */
  /********************************************************************/
  bool  bFound = false;
  FMTABLE* pFmPrev = 0;
  FMTABLE* pFmTable = (FMTABLE*)(owner->fFrameWindowData->fpSubmenuList);
  while (pFmTable)
  {
    /******************************************************************/
    /* If match is found, set found flag and stop searching           */
    /******************************************************************/
    if ((pFmTable->fId == itemId) && (pFmTable->fHMenu == (HMENU)menuHandle))
    {
      bFound = true;
      break;
    }
    /******************************************************************/
    /* Otherwise, go to next entry, but save this as previous         */
    /******************************************************************/
    pFmPrev = pFmTable;
    pFmTable = pFmTable->pfmtNext;
  }

  /********************************************************************/
  /* If the entry was found, then update list and delete item         */
  /********************************************************************/
  if (bFound)
  {
    /******************************************************************/
    /* Check for special case of first entry                          */
    /******************************************************************/
    if (pFmPrev == 0)
      owner->fFrameWindowData->fpSubmenuList = (void*)pFmTable->pfmtNext;

    /******************************************************************/
    /* otherwise remove middle entry and point previous to next       */
    /******************************************************************/
    else
      pFmPrev->pfmtNext = pFmTable->pfmtNext;

    /******************************************************************/
    /* Delete the actual item found to be removed and exit            */
    /******************************************************************/
    delete pFmTable;
  }
  return;
}


/*------------------------------------------------------------------------------
| getHMenuFromLookUpTable                                                      |
|  Local function used to retrieve the menu handle associated with a specific  |
|  item id                                                                     |
------------------------------------------------------------------------------*/
HMENU IMenuPrivate::getHMenuFromLookUpTable( IFrameWindow*  owner,
                                             unsigned long  itemId )
{
  /*******************************************************************/
  /* If a valid owner passed, check table for id passed              */
  /*******************************************************************/
  if (owner)
  {
    /*****************************************************************/
    /* Start at top and keep searching as long as entries exist      */
    /*****************************************************************/
    FMTABLE* pFmTable = (FMTABLE*)(owner->fFrameWindowData->fpSubmenuList);
    while (pFmTable)
    {
      /***************************************************************/
      /* If match is found, return match, otherwise go to next entry */
      /***************************************************************/
      if ( pFmTable->fId == itemId )
        return pFmTable->fHMenu;
      pFmTable = pFmTable->pfmtNext;
    }
  }
  /*******************************************************************/
  /* If no match, return null handle                                 */
  /*******************************************************************/
  return NULL;
}

/*------------------------------------------------------------------------------
| getItemIdFromLookUpTable                                                     |
|  Local function used to retrieve the item id associated with a specific      |
|  menu handle                                                                 |
------------------------------------------------------------------------------*/
long IMenuPrivate::getItemIdFromLookUpTable( IFrameWindow*      owner,
                                             const IMenuHandle& hMenu )
{
  /*******************************************************************/
  /* If a valid owner passed, check table for menuhandle passed      */
  /*******************************************************************/
  if (owner)
  {
    /*****************************************************************/
    /* Start at top and keep searching as long as entries exist      */
    /*****************************************************************/
    FMTABLE* pFmTable = (FMTABLE*)(owner->fFrameWindowData->fpSubmenuList);
    while (pFmTable)
    {
      /***************************************************************/
      /* If match is found, return match, otherwise go to next entry */
      /***************************************************************/
      if ( pFmTable->fHMenu == hMenu )
        return (long)pFmTable->fId;
      pFmTable = pFmTable->pfmtNext;
    }
  }
  /*******************************************************************/
  /* If no match, return invalid value (-1) to indicate this         */
  /*******************************************************************/
  return (long)0xFFFFFFFF;
}


/*------------------------------------------------------------------------------
| buildLookUpTableFromResourceLoad                                             |
|  Local function to prime a lookup table from the loaded menu                 |
------------------------------------------------------------------------------*/
bool IMenuPrivate::buildLookUpTableFromResourceLoad( IFrameWindow*      owner,
                                                 const IMenuHandle& hMenu,
                                                 const IResourceId& menuResId )
{
  HMENU hMenuParent[9];     // Array to save parent menu handle
  int   iMenuPosition[9];   // Array to save nested menu position

  // Get the module (instance) handle to load resource file from
  HMODULE hModule = menuResId.resourceLibrary().handle();

  // Find the menu resource, load it and get a pointer to the data
  HRSRC hrc = FindResource( hModule, MAKEINTRESOURCE(menuResId), RT_MENU );
  HGLOBAL hgb = LoadResource( hModule, hrc );
  LPVOID pRes = LockResource( hgb );

  // Convert pointer to dword since we need to remain on boundary
  LPDWORD pb = (LPDWORD)pRes;

  // Initialize global variables and flags needed for processing
  unsigned long menuItemId = 0;
  BOOL bPopUp, bFoundEnd;
  DWORD dwString;
  hMenuParent[1] = (HMENU)hMenu;
  iMenuPosition[1] = -1;

  // Check for correct version of menu resource (Extended Menu)
  // Version is 1 with offset 4 to start of menu template
  DWORD dwVersion = *pb;
#ifdef IC_WU
  wuSWAP32(dwVersion);  // Swap the bytes
#endif
  if (dwVersion == 0x00040001)
    pb += 2;         // If correct, move pointer to start of menu info
  else
    return false;    // otherwise exit with error

  // Start at menubar level (1) until end resets this to 0
  UINT iCascadeLevel = 1;
  UINT iNestedLevel = 1;
  while (iNestedLevel)
  {
    // Reset popup flag for each item, and bump position count by one
    bPopUp = FALSE;
    (iMenuPosition[iNestedLevel])++;

    // First move pointer past dwType & dwState, to menu id and get it
    pb += 2;
    menuItemId = (unsigned long)*pb;
#ifdef IC_WU
    wuSWAP32(menuItemId);  // Swap the bytes
#endif
    pb++;

    // Then check the next byte for the template item type
    dwString = *pb;
#ifdef IC_WU
    wuSWAP32(dwString);  // Swap the bytes
#endif
    if ( LOBYTE(LOWORD(dwString)) & MFR_POPUP )
    {
      // If this is a popup, get the submenu handle for this popup
      HMENU hPop = GetSubMenu( hMenuParent[iNestedLevel],
                               iMenuPosition[iNestedLevel] );

      // and add it to the lookup table for this menu
      addToLookUpTable( owner, hPop, menuItemId );

      // Now bump the nesting level up one (for new popup), set flag to
      // indicate this is a popup, save the menu handle and reset position
      iNestedLevel++;
      bPopUp = TRUE;
      hMenuParent[iNestedLevel] = hPop;
      iMenuPosition[iNestedLevel] = -1;
    }

    // If this is the end of popup (or menubar), decrement level by 1
    if ( LOBYTE(LOWORD(dwString)) & MFR_END )
    {
      if (bPopUp)
        iCascadeLevel++;
      else
      {
        iNestedLevel -= iCascadeLevel;
        iCascadeLevel = 1;
      }
    }

    // Initialize flag for looking for string end with first character compare
    if ( bFoundEnd = (HIWORD(dwString) == 0))
      pb++;

    // Loop through string until end of string
    while ( !bFoundEnd )
    {
      // If character, then move to next block to check
      pb++;

      // Get next block and check both characters for NULL
      dwString = *pb;
#ifdef IC_WU
      wuSWAP32(dwString);  // Swap the bytes
#endif

      if ( (LOWORD(dwString) == 0) || (HIWORD(dwString) == 0))
      {
        // If null termination, then set flag to show string end found
        // and move pointer after this block
        bFoundEnd = TRUE;
        pb++;
      }
    }
    // If popup, then skip over extra dword for help id as well.
    if (bPopUp)
      pb++;
  }
  // Finally, add the toplevel handle to the table with its id
  addToLookUpTable( owner, hMenu, menuResId.id() );
  return true;
}


/*------------------------------------------------------------------------------
| deleteLookUpTable                                                            |
|  Local function to delete a previously built lookup table                    |
------------------------------------------------------------------------------*/
void IMenuPrivate::deleteLookUpTable( IFrameWindow* owner )
{
  if (owner->fFrameWindowData->fpSubmenuList)
  {
    FMTABLE* pFmTable = (FMTABLE*)(owner->fFrameWindowData->fpSubmenuList);
    FMTABLE* pNext;
    do {
      pNext = pFmTable->pfmtNext;
      delete pFmTable;
      pFmTable = pNext;
    } while (pFmTable);
  }
  owner->fFrameWindowData->fpSubmenuList = 0;
}


/**********************************************************************/
/* Procedure: EnumAllFrames - Callback function for EnumWindows.      */
/*            Receives the handles of top-level windows and checks    */
/*            each one looking for a lookup table function            */
/**********************************************************************/
int __stdcall IMenuPrivate::EnumAllFrames( HWND hwnd, long lpData )
{
  PFRAMEINFO pfrminfo = (PFRAMEINFO)lpData;

  /*------------------------------------------------------------------*/
  /* Check if there is an IWindow object for this window              */
  /*------------------------------------------------------------------*/
  IWindow* pwndTop = IWindow::windowWithHandle( hwnd );
  if ( pwndTop )
  {
    long id = 0;
    if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
    {
      /*--------------------------------------------------------------*/
      /* For the frame passed, check if it contains a lookup list     */
      /*--------------------------------------------------------------*/
      IFrameWindow* tempFrame = (IFrameWindow*)GetProp( hwnd,
                                                  PROP_MENUBAR_OBJECT );
      if (tempFrame)
      {
        /*------------------------------------------------------------*/
        /* If it does, check if the lookup table contains the HMENU   */
        /*------------------------------------------------------------*/
        id = getItemIdFromLookUpTable( tempFrame, pfrminfo->hmenu );
        if ( id != 0xFFFFFFFFul )
        {
          /*----------------------------------------------------------*/
          /* If it contains the menu, then return the IMenu* of the   */
          /*  table and stop the enumeration                          */
          /*----------------------------------------------------------*/
          pfrminfo->owner = hwnd;
          pfrminfo->frame = tempFrame;
          return FALSE;
        }
      }
    }
    else
    {
      /*--------------------------------------------------------------*/
      /* See if this menu is contained somewhere in this window       */
      /*--------------------------------------------------------------*/
      HMENU hmTemp = GetMenu(hwnd);
      id = findIdForPopup( hmTemp, pfrminfo->hmenu );
      if ( id || (hmTemp == pfrminfo->hmenu))
      {
        /*------------------------------------------------------------*/
        /* If found, then save the window as the owner                */
        /*------------------------------------------------------------*/
        pfrminfo->owner = hwnd;

        /*------------------------------------------------------------*/
        /* For the window passed, get the frame window object (if any)*/
        /*------------------------------------------------------------*/
        pfrminfo->frame = (IFrameWindow*)GetProp( hwnd, PROP_MENUBAR_OBJECT );
        return FALSE;
      }
    }
  }
  /*------------------------------------------------------------------*/
  /* If it fails above, return true to continue searching for frame   */
  /*------------------------------------------------------------------*/
  return (TRUE);
}

/*------------------------------------------------------------------------------
| IMenuPrivate::setBidiAttributes                                              |
| Static function to set the bidirectional attributes of a menu.               |
------------------------------------------------------------------------------*/
void IMenuPrivate::setBidiAttributes( IMenuHandle& menuHandle,
                                      const IBidiSettings& bidiSettings )
{
  // First query the appropriate type flags for the menu.
  char
    menuText[ IC_MENUTEXTLENGTH ];
  MENUITEMINFO
    menuItemInfo;
  menuItemInfo
   .cbSize     = sizeof( MENUITEMINFO );
  menuItemInfo
   .fMask      = MIIM_TYPE;
  menuItemInfo
   .dwTypeData = menuText;
  menuItemInfo
   .cch        = IC_MENUTEXTLENGTH;
  GetMenuItemInfo( (HMENU) menuHandle, 0, TRUE, &menuItemInfo );

  // Set the appropriate type flags for the menu.
  menuItemInfo
   .fMask = MIIM_TYPE;
  unsigned int
    origType = menuItemInfo.fType,
    newType  = origType;
  if ( bidiSettings.textOrientation() == IBidiSettings::textLeftToRight )
  {
     newType &= (unsigned int) ~MFT_RIGHTORDER;
  }
  else
  {
     newType |= MFT_RIGHTORDER;
  }
  if ( bidiSettings.windowLayout() == IBidiSettings::layoutLeftToRight )
  {
     newType &= (unsigned int) ~MFT_RIGHTJUSTIFY;
  }
  else
  {
     newType |= MFT_RIGHTJUSTIFY;
  }
  if ( origType != newType )
  {
     menuItemInfo
      .fType = newType;
     SetMenuItemInfo( (HMENU) menuHandle, 0, TRUE, &menuItemInfo );
  }
}
#endif // IC_WIN


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IMenuPrivate::rootMenuWindowForHandle                                        |
| Locates the root MenuWindow (if any) containing the given handle.            |
------------------------------------------------------------------------------*/
MenuWindow* IMenuPrivate::rootMenuWindowForHandle( const IMenuHandle& handle )
{
  IFUNCTRACE_DEVELOP();
  MenuWindow* pwin(0);
  Widget tempHandle = handle;
  // Locate the "root" parent of this submenu Widget.
  while (tempHandle != 0)
  {
    // A MenuWindow object exists for all IMenuBar/IPopUpMenu object that
    // we are looking for.  It can also exist for intervening ISubMenu objects
    // as well.
    pwin = dynamic_cast<MenuWindow*>(
              IWindow::windowWithHandle(tempHandle) );
    if ( pwin )
    {
       // Filter out intervening ISubMenu windows.
       if ( XmIsRowColumn(tempHandle) )
       { // MenuBar or Submenu
         unsigned char rcType;
         XtVaGetValues(tempHandle, XmNrowColumnType, &rcType, 0);
         if ( rcType == XmMENU_BAR )
         {  // We have found the menu bar.
            ITRACE_DEVELOP(IString("Root menubar: RowColumn=") +
                           IString((unsigned long)tempHandle).d2x() +
                           IString(" MenuWindow=") +
                           IString((unsigned long)pwin).d2x());
            tempHandle = 0;   //Stop now
         }
         else
         {  // Probably a submenu.  Not finished yet.
            ITRACE_DEVELOP(IString("Submenu:      RowColumn=") +
                           IString((unsigned long)tempHandle).d2x() +
                           IString(" MenuWindow=") +
                           IString((unsigned long)pwin).d2x() );
            pwin = 0;     //Avoid spurious results.
            tempHandle = XtParent(tempHandle);
         }
       }
       else
       { // tempHandle is the MenuShell widget of a popup.
         ITRACE_DEVELOP(IString(   "Root popup:   MenuShell=") +
                        IString((unsigned long)tempHandle).d2x() +
                        IString(" MenuWindow=") +
                        IString((unsigned long)pwin).d2x() );
         tempHandle = 0;   //Stop now
       }
    }
    else
    {
       ITRACE_DEVELOP(IString(     "no MenuWindow:   handle=") +
                      IString((unsigned long)tempHandle).d2x());
       tempHandle = XtParent(tempHandle);
    }
  } //endwhile
  return pwin;
}

/*------------------------------------------------------------------------------
| IMenuPrivate::menuWindowForHandle                                            |
------------------------------------------------------------------------------*/
MenuWindow* IMenuPrivate::menuWindowForHandle( const IMenuHandle& handle )
{
  IFUNCTRACE_DEVELOP();
  MenuWindow* pwin(0);
  Widget tempHandle = handle;
  // Locate the first MenuWindow in the ancestor tree of this submenu Widget.
  while (tempHandle != 0)
  {
    // A MenuWindow object exists for all IMenu objects.
    pwin = dynamic_cast<MenuWindow*>(
              IWindow::windowWithHandle(tempHandle) );
    if ( pwin )
    {  // We have found the window we want bar.
       ITRACE_DEVELOP(IString("MenuWindow:    handle=") +
                      IString((unsigned long)tempHandle).d2x() +
                      IString(" MenuWindow=") +
                      IString((unsigned long)pwin).d2x());
       tempHandle = 0;   //Stop now
    }
    else
    {
       ITRACE_DEVELOP(IString("no MenuWindow: handle=") +
                      IString((unsigned long)tempHandle).d2x());
       tempHandle = XtParent(tempHandle);
    }
  } //endwhile
  return pwin;
}

/*------------------------------------------------------------------------------
| IMenuPrivate::refreshMenuBar                                                 |
------------------------------------------------------------------------------*/
void IMenuPrivate::refreshMenuBar( const IMenuHandle& menuItemHandle )
{
   IFUNCTRACE_ALL();
   Widget rowcol ( (menuItemHandle != 0) ?
                   XtParent((Widget)menuItemHandle) : 0 );
   unsigned char rcType = XmWORK_AREA;   //Default to something we DO NOT want
   if ( rowcol && XmIsRowColumn( rowcol ) && XtIsManaged( rowcol ) )
   {
      XtVaGetValues(rowcol, XmNrowColumnType, &rcType, 0);
   }
   if ( rcType == XmMENU_BAR )
   {
      Widget mainWindow( XtParent(rowcol) );
      if ( mainWindow && XmIsMainWindow( mainWindow ) )
      {
         Dimension menuHeight = 0, newMenuHeight = 0;
         XtVaGetValues( rowcol, XmNheight, &menuHeight, 0 );

         //Force the main window to refresh itself.
         ITRACE_DEVELOP("Calling core_class.resize on mainWindow and menuBar" );
         // We have trouble with the menubar getting sized when items are
         // added and a wrap is needed.  Nudge the main window and the rowcolumn.
         (mainWindow->core.widget_class->core_class.resize)( mainWindow );
         (rowcol->core.widget_class->core_class.resize)( rowcol );

         //Check to see if menu height changed.  If so, refresh the frame.
         XtVaGetValues( rowcol, XmNheight, &newMenuHeight, 0 );
         ITRACE_DEVELOP( IString("menuHeight=") + IString(menuHeight) +
                         IString(" newMenuHeight=") + IString(newMenuHeight));
         if (newMenuHeight != menuHeight)
         {
            IWindow* frame = IWindow::windowWithHandle( XtParent(mainWindow) );
            if (frame && frame->isShowing())
               frame->postEvent(IC_UM_UPDATEFRAME);
         }
      }
   }
}

/*------------------------------------------------------------------------------
| IMenuPrivate::setActiveMenu                                                  |
------------------------------------------------------------------------------*/
void IMenuPrivate::setActiveMenu( const IMenuHandle& handle )
{
   IFUNCTRACE_DEVELOP();
   static IMenuHandle activeMenu(0);   // Currently active popup or submenu

   ITRACE_DEVELOP( IString("new handle=") +
                   IString(handle.asUnsigned()).d2x() );
   if ( activeMenu  &&  handle  &&  ( activeMenu != handle )  &&
        ( (Widget) activeMenu )->core.being_destroyed == false )
   {
       // A new menu is being activated while another one is active.  This
       // typically does not occur, because Motif dismisses an active menu
       // before activating a new one.  However, it can happen in some cases
       // where non-standard events are being used to activate popup menus.
       // An example of this is using Button2 (middle button) to activate
       // popup menus.  Motif treats this as BTransfer and does not dismiss
       // any pre-existing menus in this case.
       // We force the old menu to be popped down now.  Note that this may
       // cause a recursive invocation of this function (with handle == 0).
       ITRACE_DEVELOP( IString("Calling MenuShellPopdownDone on activeMenu=") +
                       IString(activeMenu.asUnsigned()).d2x() );
       XtCallActionProc(activeMenu, "MenuShellPopdownDone", NULL, NULL, 0);
   }
   // Update the currently active menu.
   activeMenu = handle;
}
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| setMenuItemHelpId                                                            |
|  Local function used to build help id table for menu items                   |
------------------------------------------------------------------------------*/
void IMenuPrivate::setMenuItemHelpId( IFrameWindow*  owner,
                                      unsigned long  menuItemId,
                                      unsigned long  helpTopicId )
{
  /********************************************************************/
  /* If a valid owner passed, check table for id passed               */
  /********************************************************************/
  if (owner)
  {
    /******************************************************************/
    /* Start at top and keep searching as long as entries exist       */
    /******************************************************************/
    MIHELPTBL* pMiHTable = (MIHELPTBL*)(owner->fFrameWindowData->fpHelpList);
    while (pMiHTable)
    {
      /****************************************************************/
      /* If match is found, return match, otherwise go to next entry  */
      /****************************************************************/
      if ( pMiHTable->fId == menuItemId )
      {
        pMiHTable->fHelpId = helpTopicId;
        return;
      }
      pMiHTable = pMiHTable->pmitNext;
    }

    /******************************************************************/
    /* Create a new menu lookup table entry and fill with info passed */
    /******************************************************************/
    MIHELPTBL* pMi2Table = new MIHELPTBL;
    pMi2Table->fId = menuItemId;
    pMi2Table->fHelpId = helpTopicId;

    /******************************************************************/
    /* Chain list (if any) to list                                    */
    /******************************************************************/
    pMi2Table->pmitNext = (MIHELPTBL*)(owner->fFrameWindowData->fpHelpList);

    /******************************************************************/
    /* Finally, since this is now the start of the list, save it      */
    /******************************************************************/
    owner->fFrameWindowData->fpHelpList = (void*)pMi2Table;
  }
  return;
}


/*------------------------------------------------------------------------------
| queryMenuItemHelpId                                                          |
|  Local function used to build help id table for menu items                   |
------------------------------------------------------------------------------*/
unsigned long IMenuPrivate::queryMenuItemHelpId( IFrameWindow*  owner,
                                                 unsigned long  menuItemId )
{
  /********************************************************************/
  /* If a valid owner passed, check table for id passed               */
  /********************************************************************/
  if (owner)
  {
    /******************************************************************/
    /* Start at top and keep searching as long as entries exist       */
    /******************************************************************/
    MIHELPTBL* pMiHTable = (MIHELPTBL*)(owner->fFrameWindowData->fpHelpList);
    while (pMiHTable)
    {
      /****************************************************************/
      /* If match is found, return match, otherwise go to next entry  */
      /****************************************************************/
      if ( pMiHTable->fId == menuItemId )
        return pMiHTable->fHelpId;

      pMiHTable = pMiHTable->pmitNext;
    }
  }
  return 0;
}


/*------------------------------------------------------------------------------
| deleteHelpIdTable                                                            |
|  Local function to delete a previously built help id table                   |
------------------------------------------------------------------------------*/
void IMenuPrivate::deleteHelpIdTable( IFrameWindow* owner )
{
  if (owner->fFrameWindowData->fpHelpList)
  {
    MIHELPTBL* pMiHTable = (MIHELPTBL*)(owner->fFrameWindowData->fpHelpList);
    MIHELPTBL* pNext;
    do {
      pNext = pMiHTable->pmitNext;
      delete pMiHTable;
      pMiHTable = pNext;
    } while (pMiHTable);
  }
  owner->fFrameWindowData->fpHelpList = 0;
}


#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| MenuWindow::MenuWindow    26329                                              |
|                                                                              |
------------------------------------------------------------------------------*/
MenuWindow::MenuWindow ( )         //used by imenubar & ipopmenu, who derive their own class
    : fRootMenu(0),
      fUniversalRedispatcher(this)
{
  IFUNCTRACE_DEVELOP();
  fUniversalRedispatcher.handleEventsFor(this);
}

/*------------------------------------------------------------------------------
| MenuWindow::MenuWindow                                                       |
// For ISubmenu menu windows, or sometimes for popup menu windows (rootMenu=0)
------------------------------------------------------------------------------*/
MenuWindow::MenuWindow ( const IWindowHandle& handle, IWindow* rootMenu )
    : IWindow(handle),
      fRootMenu( dynamic_cast<MenuWindow*>(rootMenu) ),
      fUniversalRedispatcher(this)
{
  IFUNCTRACE_DEVELOP();
  ITRACE_DEVELOP( IString("handle=") +
                  IString(handle.asUnsigned()).d2x() +
                  ", this=" + IString((unsigned long)this).d2x() );
  fUniversalRedispatcher.handleEventsFor(this);
}


/*------------------------------------------------------------------------------
| MenuWindow::~MenuWindow                                                      |
------------------------------------------------------------------------------*/
MenuWindow::~MenuWindow ( )
{
  IFUNCTRACE_DEVELOP();
  ITRACE_DEVELOP( IString("this=") +
                  IString((unsigned long)this).d2x() );

  fUniversalRedispatcher.stopHandlingEventsFor(this);
}

/*------------------------------------------------------------------------------
| MenuWindow::menu                                                             |
------------------------------------------------------------------------------*/
IMenu* MenuWindow::menu   ( ) const
{
  IMenu* result(0);
  const IAttributeName menuName(IMenuAttribute::menuAttrName);
  const IMenuAttribute* menuAttr =
     dynamic_cast<const IMenuAttribute*>(this->attributeWithName(menuName));
  if (menuAttr)
     result = menuAttr->fMenu;
  return result;
}

/*------------------------------------------------------------------------------
| MenuWindow::setMenu                                                          |
------------------------------------------------------------------------------*/
MenuWindow& MenuWindow::setMenu( IMenu* menu )
{
   const IAttributeName menuName(IMenuAttribute::menuAttrName);
   if (menu)
   {  //Add or replace attribute.
      IMenuAttribute menuAttr(menu);
      this->addOrReplaceAttribute(menuName, menuAttr);
   }
   else
   {  //remove attribute.
      this->removeAttribute(menuName);
   }
   ITRACE_ALL( IString("MenuWindow::setWindow this=") +
               IString((unsigned long)this).d2x() +
               IString(" menu=") + IString((unsigned long)menu).d2x() );
   return *this;
}

#ifdef IC_MOTIF
bool MenuWindow::passEventToOwner ( IEvent& event )
{
   // Don't propagate pop-up menu events.
   if ( event.eventId() == WM_CONTEXTMENU )
   {
      event.setPassToOwner( false );
   }
   else
   {
      this->IWindow::passEventToOwner( event );
   }
   return event.passToOwner();
}
#endif

/*------------------------------------------------------------------------------
| MenuWindow::redispatchToRootMenu                                             |
------------------------------------------------------------------------------*/
bool MenuWindow::redispatchToRootMenu(IEvent& evt)
{
  if (fRootMenu)
    return fRootMenu->dispatch(evt);
  else
    return false;
}


/*------------------------------------------------------------------------------
| UniversalRedispatcher::UniversalRedispatcher     26329                       |
|                                                                              |
------------------------------------------------------------------------------*/
UniversalRedispatcher::UniversalRedispatcher(MenuWindow* menuWindow)
    : fMenuWindow(menuWindow) {}


/*------------------------------------------------------------------------------
| UniversalRedispatcher::~UniversalRedispatcher                                |
------------------------------------------------------------------------------*/
UniversalRedispatcher::~UniversalRedispatcher() {}


/*------------------------------------------------------------------------------
| UniversalRedispatcher::dispatchHandlerEvent                                  |
------------------------------------------------------------------------------*/
bool UniversalRedispatcher::dispatchHandlerEvent(IEvent& evt)
{
  switch (evt.eventId())      //27183 - don't redispatch ALL events
  {
    case WM_INITMENU:
    case WM_MENUSELECT:
    case WM_MENUEND:
#ifdef IC_MOTIF
    case WM_MENUUNMAP:
#endif //IC_MOTIF
    {
      IMODTRACE_DEVELOP("UniversalRedispatcher forwarding event");
      #ifdef IC_TRACE_ALL
      {
         switch (evt.eventId())
         {
            case WM_INITMENU:    ITRACE_ALL("WM_INITMENU:  "); break;
            case WM_MENUSELECT:  ITRACE_ALL("WM_MENUSELECT:"); break;
            case WM_MENUEND:     ITRACE_ALL("WM_MENUEND:   "); break;
#ifdef IC_MOTIF
            case WM_MENUUNMAP:   ITRACE_ALL("WM_MENUUNMAP: "); break;
#endif //IC_MOTIF
         }
      }
      #endif
      return fMenuWindow->redispatchToRootMenu(evt);
      break;
    } /* endcase */

#ifdef IC_MOTIF
    case WM_BUTTON1DOWN:
    case WM_BUTTON2DOWN:
    case WM_BUTTON3DOWN:
    {
       // Find the actual widget for which this occurred.  In the case
       // of a press on another application reported to the menu because
       // of a pointer grab, it should be the row column.  In other cases,
       // such as a menu item being selected, it will be something else.
       XButtonEvent* xevent = (XButtonEvent*)(char*)evt.parameter2();
       Widget widget = XtWindowToWidget( xevent->display, xevent->window );
       if ( widget && XmIsRowColumn( widget ))
       {
          unsigned char rctype = '\0';
          XtVaGetValues( widget, XmNrowColumnType, &rctype, 0);

          // Hack city.  The event is a ButtonPress event reported to the
          // menu as the result of the pointer grab that the menu sets up.
          // When this occurs over another application, we will never see
          // the corresponding ButtonRelease.   Therefore, reset the state
          // of the IMouseEventDetect object for the current thread so
          // that it does not expect to see the ButtonRelease.  This
          // avoids an apparent disabling of clicks until the same button
          // is pressed/released in this app.
          ICurrentThreadData* threadData =
             IThreadData::currentThreadData( IThread::current() );
          if (threadData && (rctype == XmMENU_POPUP))
          {
             ITRACE_DEVELOP(IString("Resetting IMouseEventDetect  widget=") +
                            IString((unsigned long)widget).d2x() );
             // This resets the state for the button held in IMouseEventDetect.
             threadData->resetButton( xevent->button );
          }
       }
    }
    break;
#endif

    default:
      ITRACE_ALL( IString("UniversalRedispatcher ate event: eventId=") +
                  IString(evt.eventId()) );
      break;


  } /* endswitch */     //end 27183

  return false;
}
#endif //IC_MOTIFPM


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| imenuPopUpCallback                                                           |
------------------------------------------------------------------------------*/
void imenuPopUpCallback  ( _WidgetRec* handle,
                           void*       client_data,
                           void*       call_data)
{
   IFUNCTRACE_ALL();
   //This callback is called when a popup menu or submenu of the menu bar
   //is popped up.

   // Store the fact that this menu is now active.
   IMenuPrivate::setActiveMenu( handle );
}

/*------------------------------------------------------------------------------
| imenuPopDownCallback                                                         |
------------------------------------------------------------------------------*/
void imenuPopDownCallback( _WidgetRec* handle,
                           void*       client_data,
                           void*       call_data)
{
   IFUNCTRACE_ALL();
   //This callback is called when a popup menu or submenu of the menu bar
   //is popped down.

   // Indicate that this menu is no longer active.
   IMenuPrivate::setActiveMenu( 0 );

   // If this callback was installed on an IPopUpMenu, the client_data will
   // contain a pointer to the MenuWindow object for it.  Initiate the auto
   // destruction of the object if needed.
   MenuWindow* rootWin = (MenuWindow*) client_data;
   ITRACE_DEVELOP(IString("Dismissed the menu rootWin=") +
                  IString((unsigned long)rootWin).d2x() +
                  IString(" MenuShell=" ) +
                  IString( (unsigned long)handle).d2x() );

   if ( rootWin && rootWin->isAutoDeleteObject())
   {
      ITRACE_ALL("Posting WM_POPUPDESTROY");
      // we need to destroy the popup menu object
      rootWin->postEvent( WM_POPUPDESTROY,
                          IEventData(0), IEventData(0) );
   }  // if autodelete popup
}
#endif //IC_MOTIF
