// Revision: 07 1.14.3.6 source/ui/baseapp/ifontprv.cpp, font, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ifontprv.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ifontprv.hpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   Licensed Materials - Property of IBM                                       *
*   (C) Copyright IBM Corporation 1992, 1997 All Rights Reserved.              *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#include <ilanglvl.hpp>

extern "C"
  {
  #define INCL_GPIPRIMITIVES
#ifdef IC_WIN
  #include <windows.h>
#endif // IC_WIN
#ifdef IC_PM
#define INCL_DOSPROCESS
#define INCL_GPI

  #include <os2.h>
#endif // IC_PM
#ifdef IC_MOTIF
  #include <Xm/AtomMgr.h>
  #include <Xm/Xm.h>
  #include <Xm/Label.h>            // for temp label widget to get "default" fontlist
#endif //IC_MOTIF
  }

#include <ifontprv.hpp>

#include <iexcept.hpp>
#include <itrace.hpp>
#include <iwindow.hpp>
#include <istparse.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _IFONTPRV_CPP_
#include <ipagetun.h>
#endif

#ifdef IC_MOTIF
#include <ixdc.hpp>
#include <ixfntset.hpp>
#endif // IC_MOTIF

#ifdef IC_WIN
#define CHDIRN_DEFAULT              0
#endif // IC_WIN

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData() : flClStyles(0),
                                       lClFontDir(CHDIRN_DEFAULT),
                                       pntClAngle(1,0),
                                       pntClShear(0,1),
                                       fxClPointSize(0),
#ifdef IC_PM
                                       pfontmetCl(0),
                                       pfattrsCl(0),
                                       plClCharWidths(0),
                                       fxClHeight(0),
                                       fxClWidth(0),
                                       lClLCId(0),
#endif
#ifdef IC_WIN
                                       fpFontHandle(new IFontHandle()),
                                       prevFont(NULL),
                                       systemFont( (HFONT)(void*)
                                                   GetStockObject( SYSTEM_FONT ) ),
                                       fcharSet(DEFAULT_CHARSET),
                                       plogfontCl(0),
                                       ptextmetricCl(0),
                                       inUseCount(0),
#endif
                                       refs(1)
{
#ifdef IC_WIN
  /********************************************************************/
  /* Request information for the system font (Stock object) and if    */
  /* retrieved, use character set from it, otherwise leave as default */
  /* Only use it if it is not ANSI, since using ANSI will force the   */
  /* font mapper to select True type fonts over vector or symbol fonts*/
  /********************************************************************/
  LOGFONT logfont;
  if (GetObject(systemFont, sizeof(LOGFONT), &logfont))
    if (logfont.lfCharSet)
      fcharSet = logfont.lfCharSet;
#endif
}


/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
|                                                                              |
| Construct font private data based on input parameters.                       |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData(const IFontPrivateData& fpd)
  : flClStyles(fpd.flClStyles),
    lClFontDir(fpd.lClFontDir),
    pntClAngle(fpd.pntClAngle),
    pntClShear(fpd.pntClShear),
    fxClPointSize(fpd.fxClPointSize),
#ifdef IC_PM
    pfontmetCl(0),
    pfattrsCl(0),
    plClCharWidths(0),
    fxClHeight(fpd.fxClHeight),
    fxClWidth(fpd.fxClWidth),
    lClLCId(0),
#endif
#ifdef IC_WIN
    prevFont(NULL),
    systemFont(fpd.systemFont),
    fcharSet(fpd.fcharSet),
    plogfontCl(0),
    ptextmetricCl(0),
    inUseCount(0),
#endif
    refs(1)
{
#ifdef IC_PM
  /* Initialize the font metrics structure. */
  if (fpd.pfontmetCl)
  {
    pfontmetCl = new FONTMETRICS;
    *pfontmetCl = *(fpd.pfontmetCl);
  }

  /* Initialize font attributes structure */
  if (fpd.pfattrsCl)
  {
    pfattrsCl = new FATTRS;
    *pfattrsCl = *(fpd.pfattrsCl);
  }
#endif
#ifdef IC_WIN
  fpd.fpFontHandle->addRef();
  fpFontHandle = fpd.fpFontHandle;
  /* Initialize the log font structure. */
  if (fpd.plogfontCl)
  {
    plogfontCl = new LOGFONT;
    *plogfontCl = *(fpd.plogfontCl);
  }
  if (fpd.ptextmetricCl)
  {
    ptextmetricCl = new TEXTMETRIC;
    *ptextmetricCl = *(fpd.ptextmetricCl);
  }
#endif
}

/*------------------------------------------------------------------------------
| IFontPrivateData::removeRef                                                  |
|                                                                              |
| Decrement the ref count and delete the object if refcount is zero.           |
------------------------------------------------------------------------------*/
void IFontPrivateData::removeRef()
{
  if (--refs == 0)
  {
#ifdef IC_PM
    delete pfontmetCl;
    delete [] plClCharWidths;
    delete pfattrsCl;
#endif
#ifdef IC_WIN
    /* Get rid of stock font requested */
    DeleteObject(systemFont);

    /* Delete structures allocated by us for storing font information */
    delete plogfontCl;
    delete ptextmetricCl;
    fpFontHandle->removeRef();
#endif
    delete this;
  }
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IFontHandle::~IFontHandle                                                    |
------------------------------------------------------------------------------*/
IFontHandle::~IFontHandle()
{ }


/*------------------------------------------------------------------------------
| IFontHandle::removeRef                                                       |
|                                                                              |
| Decrement the ref count and delete the object if refcount is zero.           |
| Also delete the font object if this was allocated by the IFont code          |
------------------------------------------------------------------------------*/
void IFontHandle::removeRef()
{
  if (--refs == 0)
  {
    if (fAllocatedByIFont)
    {
 #ifdef IC_WIN_STRICT
      DeleteObject( fValue );
 #else
      DeleteObject( (HFONT) fValue );
 #endif
    }
    delete this;
  }
}
#endif //IC_WIN
#endif //IC_PMWIN

#ifdef IC_MOTIF
// These two little classes provide reference counting, so we don't
// free our font data while it is referenced by others.
//
// The previous implementation from the first GA had the following problem:
//
// "It looks like the copy constructors and assignment operators copy
//  pointers, (but always set deletePFontStruct to false).  So the
//  copied pointers will be dangling if the original object is deleted."
//
// I have changed IFontPrivateData to use IFontPrivateFontList::Handle
// to do reference counting.  ..Rob.
//

// Dissallow copy and assignment of this class.
// These null implementations are here to satisfy SOM.
IFontPrivateFontList::IFontPrivateFontList(const IFontPrivateFontList&)
{}

IFontPrivateFontList&
IFontPrivateFontList::operator=(const IFontPrivateFontList&)
{
 return *this;
}

XmFontList IFontPrivateFontList::createFontList(XtPointer fontOrFontSet,
                                                 XmFontType type,
                                                 const char* tag)
{
 //
 // Create an XmFontList containing just one entry.
 //
 if (!fontOrFontSet)
  {
   // The caller gave us a null pointer.
   ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                       IBaseErrorInfo::accessError,
                       IException::recoverable);
  }

 // Create the new font list. This could only fail if fontOrFontSet were
 // NULL and we just checked that it's not.
 XmFontListEntry
 entry = XmFontListEntryCreate((char*)tag, type, fontOrFontSet);
 XmFontList fontList = 0;
 fontList = XmFontListAppendEntry(fontList, entry);

// Here is an old comment from the first GA:
//
//  "OReilly V 6B says there is a bug in XmFontListEntryFree in Motif 1.2,
//   and to use XtFree instead."
//   XtFree ((char *)entry);
//
// I can't find any mention of the bug, in fact it says that using XtFree
// would be wrong. ..Rob.
//
 XmFontListEntryFree(&entry);
 return fontList;
}

/*------------------------------------------------------------------------------
| IFontPrivateFontList::getProp                                                |
| Utility for getting font properties.                                         |
------------------------------------------------------------------------------*/
unsigned long
IFontPrivateFontList::getProp(Atom atomId, XFontStruct* pFontStruct)
{
    unsigned long propValue;
    if (XGetFontProperty(pFontStruct, atomId, &propValue))
        return propValue;
    else
        return 0;
}

char *
IFontPrivateFontList::getFontName(XFontStruct* aFontStructPtr)
{
    // Try to get the fontName.
    char* name = 0;
    unsigned long propValue = getProp(XA_FONT, aFontStructPtr);
    if (propValue)
        name = XmGetAtomName(dpy, (Atom) propValue);
    return name; // Caller must free using XtFree.
}

IFontPrivateFontList &
IFontPrivateFontList::initialize(XmFontList aFontList,
                                 Display* aDpy)
{
    dpy = aDpy;
    fontList = aFontList;
    //
    // Get a fontStruct and possibly a fontSet from the fontList.
    // Get the one that matches XmFONTLIST_DEFAULT_TAG.  That is either the
    // first one with a tag of XmFONTLIST_DEFAULT_TAG, or the first one in
    // the list.
    // (See O'Reilly V6a, 19.2.2 "The Default Font List Tag")
    //
    // The fontStruct is not to be freed.
    // (See O'Reilly V2 XFontsOfFontSet, and V6B XmFontListEntryGetFont)
    //
    XmFontContext context;
    if (!(XmFontListInitFontContext(&context, fontList)))
        ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                           IBaseErrorInfo::accessError,
                           IException::recoverable);

    bool found = false;
    XmFontListEntry fontEntry = 0;
    while (!found && (fontEntry = XmFontListNextEntry(context)))
    {
        char* strTag = XmFontListEntryGetTag(fontEntry);
        //
        // Use the first entry with XmFONTLIST_DEFAULT_TAG, or just the first entry.
        //
        if ((found = !strcmp(strTag, XmFONTLIST_DEFAULT_TAG)) || !fontStructList)
        {
            XmFontType fontType;
            XtPointer fontPointer = XmFontListEntryGetFont(fontEntry, &fontType);
            if (fontType == XmFONT_IS_FONT)
            {
                fontSet = 0;
                ulNumFontStructs = 1;
                fontStructList = &pFontStruct;
                fontStructList[0] = (XFontStruct *) fontPointer;
                fontNameList = &fontName;
            }
            else if (fontType == XmFONT_IS_FONTSET)
            {
                // In this case the XtPointer is an XFontSet
                fontSet = (XFontSet)fontPointer;
                ulNumFontStructs
                    = XFontsOfFontSet(fontSet, &fontStructList, &fontNameList);
                // Existence of the per_char metrics in the fontStruct is undefined.
                // See O'Reilly V2, XFontsOfFontSet.
            }
        }
        XtFree(strTag);
    }
    // cleanup
    XmFontListFreeFontContext(context);

    if (!fontStructList)
        ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                           IBaseErrorInfo::accessError,
                           IException::recoverable);
    //
    // For some reason, sometimes all the fonts in a font set are the same.
    // In this case, set ulNumFontStructs to 1, so we can use properties
    // for xaXHeight and xaCapHeight.  Start with the last in the list,
    // if it is the same as the previous, decrement ulNumFontStructs.
    // Stop if you don't decrement ulNumFontStructs.
    //
    for (unsigned long index=ulNumFontStructs-1;
         index && index==(ulNumFontStructs-1);
         --index)
    {
        if (fontStructList[index] == fontStructList[index-1]) --ulNumFontStructs;
    }

    if (!fontSet && ulNumFontStructs == 1)
    {
        // Try to get the fontName.
        fontName = getFontName(fontStructList[0]);
    }
    return *this;
}

IFontPrivateFontList &
IFontPrivateFontList::initialize(XmFontList aFontList,
                                 XFontStruct* aFontStructPtr,
                                 Display* aDpy)
{
    // This version of initialize is a bit faster and handles the degenerate case
    // where the fontList has a single element which is a fontStruct.
    dpy = aDpy;
    fontList = aFontList;

    fontSet = 0;
    ulNumFontStructs = 1;
    fontStructList = &pFontStruct;
    fontNameList = &fontName;

    pFontStruct = aFontStructPtr;
    fontName = getFontName(pFontStruct);

    return *this;
}

IFontPrivateFontList::IFontPrivateFontList()
        : dpy(0),
          fontList(0),
          fontSet(0),
          ulNumFontStructs(0),
          fontStructList(0),
          pFontStruct(0),
          fontNameList(0),
          fontName(0)
{}

IFontPrivateFontList::IFontPrivateFontList(XmFontList aFontList,
                                            Display* aDpy)
: dpy(0),
  fontList(0),
  fontSet(0),
  ulNumFontStructs(0),
  fontStructList(0),
  pFontStruct(0),
  fontNameList(0),
  fontName(0)
{
 initialize(aFontList, aDpy);
}

IFontPrivateFontList::IFontPrivateFontList(XFontSet aFontSet, Display* aDpy)
: dpy(0),
  fontList(0),
  fontSet(0),
  ulNumFontStructs(0),
  fontStructList(0),
  pFontStruct(0),
  fontNameList(0),
  fontName(0)
{
 XmFontList aFontList = createFontList(aFontSet,
                                        XmFONT_IS_FONTSET,
                                        XmFONTLIST_DEFAULT_TAG);
 initialize(aFontList, aDpy);
}

IFontPrivateFontList::~IFontPrivateFontList()
{
 if (fontList)
   XmFontListFree(fontList);  // Does not free the FontSet's and FontStruct's
 if (fontName)
   XtFree(fontName);
}

IFontPrivateFontStruct::IFontPrivateFontStruct(XFontStruct* aFontStructPtr,
                                                Display* aDpy)
: IFontPrivateFontList()
{
 XmFontList aFontList = createFontList(aFontStructPtr,
                                        XmFONT_IS_FONT,
                                        XmFONTLIST_DEFAULT_TAG);
 initialize(aFontList, aFontStructPtr, aDpy);
}

IFontPrivateFontStruct::~IFontPrivateFontStruct()
{
// Here is an old comment from the first GA:
//
//  "HACK ALERT:
//     Close to GA we determined that on 3.2.5 systems with latest AixWindows
//     CSD applied or on a 4.1 system there were problems if we did the
//     XFreeFont for the IFont object about to be destructed.
//     For now we'll just set deletePFontStruct to false all this time."
//
// The problem still seems to exist, so the following two lines are
// commented out:
//
// if (xFontStruct())
//   XFreeFont(display(), xFontStruct());
}

IFontPrivateFontSet::IFontPrivateFontSet(XFontSet aFontSet, Display* aDpy)
: IFontPrivateFontList()
{
 XmFontList aFontList = createFontList(aFontSet,
                                        XmFONT_IS_FONTSET,
                                        XmFONTLIST_DEFAULT_TAG);
 initialize(aFontList, aDpy);
}

IFontPrivateFontSet::~IFontPrivateFontSet()
{
    //if (xFontSet())
    // XFreeFontSet(display(), xFontSet());
}

/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData() :
   hFontList(0),
   prevFont(0),
   facename(),
   xpointSize(0),
   avgWidth(0),
   flags(0),
   resX(0),
   resY(0),
   resXexact(0),
   resYexact(0),
   dpy(0),
   maxSize(0,0),
   maxAscDesc(0,0),
   xaXHeight(0),
   xaCapHeight(0),
   XLFDstr(0)
{
   IMODTRACE_ALL("IFontPrivateData - default ctor");

   initialize();

   // Construct a base font name
   // Intentionally leave char-set element to get the appropriate font
   // for the right locale
   IString base_font_name("-*-*-*-*-*-*-*-*-");
   base_font_name += IString(resX) + IString("-");
   base_font_name += IString(resY) + IString("-");
   base_font_name += IString("*-*");

   ITRACE_DEVELOP(IString("base font name") + base_font_name);

   XFontSet aFontSet;

   aFontSet = IXCreateFontSet(dpy, base_font_name);

   if (! aFontSet) {
       // No matching font found, call XCreateFontSet again with
       // the most generic form.
       ITRACE_DEVELOP("No matching font found. Try again with '*'");
       aFontSet = IXCreateFontSet(dpy, "*");
   }

   IFontPrivateFontList::Handle temph =
       new IFontPrivateFontSet(aFontSet, dpy);
   hFontList = temph; // Need temph to get reference count initialized to 1.

   // Set the private data using the font set in hFontList.
   getFontInfo();
}

/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData(XmFontList xmfontList) :
   hFontList(0),
   prevFont(0),
   facename(),
   xpointSize(0),
   avgWidth(0),
   flags(0),
   resX(0),
   resY(0),
   resXexact(0),
   resYexact(0),
   dpy(0),
   maxSize(0,0),
   maxAscDesc(0,0),
   xaXHeight(0),
   xaCapHeight(0),
   XLFDstr(0)
{
   IMODTRACE_ALL("IFontPrivateData - XmFontList ctor");
   initialize();

   IFontPrivateFontList::Handle
   temph = new IFontPrivateFontList(XmFontListCopy(xmfontList), dpy);
   hFontList = temph; // Need temph to get reference count initialized to 1.

   // Retrieve the font name and fill in the information
   getFontInfo();
}

/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData(const char*   pszFaceName,
                                   unsigned long ulPointSize,
                                   Boolean       useFixedFont,
                                   Boolean       useVectorFont,
                                   const IPresSpaceHandle& hpsInit) :
   hFontList(0),
   prevFont(0),
   facename(pszFaceName),
   xpointSize(ulPointSize*10),
   avgWidth(0),
   flags(0),
   resX(0),
   resY(0),
   resXexact(0),
   resYexact(0),
   dpy(0),
   maxSize(0,0),
   maxAscDesc(0,0),
   xaXHeight(0),
   xaCapHeight(0),
   XLFDstr(0)
{
   IMODTRACE_ALL("IFontPrivateData - specifications ctor");
   initialize();

   // pszFaceName could contain "qualified Name"...if so, parse out XLFD string
   IString qualName = pszFaceName;
   IString fontName;

   if (qualName.includes("IOC:"))
     {
     qualName >> "IOC:" >> fontName;
     loadXLFD(fontName);
     }
   else
     {
     setFixed(useFixedFont);
     setVector(useVectorFont);
     checkForMatch(hpsInit);
     }
}

/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData(const IPresSpaceHandle& presSpace) :
   hFontList(0),
   prevFont(0),
   facename(),
   xpointSize(0),
   avgWidth(0),
   flags(0),
   resX(0),
   resY(0),
   resXexact(0),
   resYexact(0),
   dpy(0),
   maxSize(0,0),
   maxAscDesc(0,0),
   xaXHeight(0),
   xaCapHeight(0),
   XLFDstr(0)
{
   IMODTRACE_ALL("IFontPrivateData - IPresSpaceHandle ctor");
   initialize();
   XFontStruct* pFontStruct = XQueryFont(dpy, XGContextFromGC(((IXDC*)presSpace)->gc()));

   // Query default font information if invalid presSpace
   if (!pFontStruct)
     pFontStruct = XQueryFont(dpy, XGContextFromGC(DefaultGC(dpy, DefaultScreen(dpy))));

   IFontPrivateFontList::Handle
   temph = new IFontPrivateFontStruct(pFontStruct, dpy);
   hFontList = temph; // Need temph to get reference count initialized to 1.

   getFontInfo();
}

/*------------------------------------------------------------------------------
| IFontPrivateData::IFontPrivateData                                           |
------------------------------------------------------------------------------*/
IFontPrivateData::IFontPrivateData(const IFontPrivateData& orig) :
  hFontList(0),
  prevFont(orig.prevFont),
  facename(orig.facename),
  xpointSize(orig.xpointSize),
  avgWidth(orig.avgWidth),
  flags(orig.flags),
  resX(orig.resX),
  resY(orig.resY),
  resXexact(orig.resXexact),
  resYexact(orig.resYexact),
  dpy(orig.dpy),
  maxSize(orig.maxSize),
  maxAscDesc(orig.maxAscDesc),
  xaXHeight(orig.xaXHeight),
  xaCapHeight(orig.xaCapHeight),
  XLFDstr(orig.XLFDstr)
{
   IMODTRACE_ALL("IFontPrivateData - copy ctor");
   hFontList = orig.hFontList;  // Explicit assignment for reference counting.
}

/*------------------------------------------------------------------------------
| IFontPrivateData::~IFontPrivateData                                          |
------------------------------------------------------------------------------*/
IFontPrivateData::~IFontPrivateData()
{
   IMODTRACE_ALL("IFontPrivateData - dtor");
}

/*------------------------------------------------------------------------------
| IFontPrivateData::operator=                                                  |
------------------------------------------------------------------------------*/
IFontPrivateData& IFontPrivateData::operator=(const IFontPrivateData& orig)
{
   if (&orig != this)
      {
      hFontList   = orig.hFontList;
      prevFont    = orig.prevFont;
      facename    = orig.facename;
      xpointSize  = orig.xpointSize;
      avgWidth    = orig.avgWidth;
      flags       = orig.flags;
      resX        = orig.resX;
      resY        = orig.resY;
      resXexact   = orig.resXexact;
      resYexact   = orig.resYexact;
      dpy         = orig.dpy;
      maxSize     = orig.maxSize;
      maxAscDesc  = orig.maxAscDesc;
      xaXHeight   = orig.xaXHeight;
      xaCapHeight = orig.xaCapHeight;
      XLFDstr     = XLFDstr;
      }
   return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::initialize                                                 |
------------------------------------------------------------------------------*/
void IFontPrivateData::initialize()
{
  // compute exact resolutions for scaling vector fonts
  dpy   = IXDisplay::display();

  // if display not initialized
  if (!dpy)
    {
    // open display explicitly
    dpy = XOpenDisplay("");
    }

  resXexact = DisplayWidth(dpy, 0) /
              (unsigned long)(DisplayWidthMM(dpy,0)/25.4);
  resYexact = DisplayHeight(dpy, 0) /
              (unsigned long)(DisplayHeightMM(dpy,0)/25.4);

  // for bitmap fonts use 75 or 100dpi whichever is closest to exact
  resX = (resXexact < 88 ? 75 : 100);
  resY = (resYexact < 88 ? 75 : 100);
}

/*------------------------------------------------------------------------------
| IFontPrivateData::getFontInfo                                                |
| Set private data based on hFontList font structs and font names.             |
| All font names are used to calculate avgWidth, but only the first font name  |
| is used for the other information.                                           |
------------------------------------------------------------------------------*/
void IFontPrivateData::getFontInfo()
{
 avgWidth = 0;
 maxSize = ISize(0,0);
 maxAscDesc = ISize(0,0);

 XLFDstr = IXLFDString(hFontList->xFontName(0));
        facename = XLFDstr.familyName();
        xpointSize = XLFDstr.pointSize() * 10;
        setBold(XLFDstr.isBold());
        setItalic(XLFDstr.isItalic());
        setFixed(XLFDstr.isFixed());
        setVector(XLFDstr.isVector());

 for (unsigned long index=0; index<hFontList->numFontStructs(); ++index)
  {
   const XFontStruct* pFontStruct = hFontList->xFontStruct(index);
   short min_width   = pFontStruct->min_bounds.width;
   short max_width   = pFontStruct->max_bounds.width;
   short max_ascent  = pFontStruct->max_bounds.ascent;
   short max_descent = pFontStruct->max_bounds.descent;

   IString fntName;
   const char* fn = hFontList->xFontName(index);

   if (fn)
     fntName = fn;

    fntName.lowerCase();

   // Check to see if we have a complete font name.
   // Should have 14 -'s in the name.
   // We no longer THROW an exception if we couldn't get properties.
   // If we can't get the properties, then the worst that can happen is that
   // some accessor functions such as faceName() and pointSize() will return
   // a NULL string and 0 respectively (ie. what the member data is initialized
   // to in the cstor).
   // We can't rely on the XA_FONT property being there.
    if (fntName.occurrencesOf("-") == 14)
    {
      int currPos, nextPos, field;
      IString currWord;

      unsigned long avgw = currWord.asUnsigned();
      if (avgw)
        avgWidth += avgw;
      else
        avgWidth += ((min_width + max_width) / 2) * 10;
    }
    else
      // Get value of avgWidth from fontStruct rather than relying on XA_FONT
      avgWidth += ((min_width + max_width) / 2) * 10;
   //
   // Accumulate various maximums.
   //
   if (max_width > maxSize.width())
     maxSize.setWidth((ISize::Coord)max_width);
   if (max_ascent > maxAscDesc.coord1())
     maxAscDesc.setCoord1(max_ascent);
   if (max_descent > maxAscDesc.coord2())
     maxAscDesc.setCoord2(max_descent);
  } // for index
 maxSize.setHeight(maxAscDesc.coord1() + maxAscDesc.coord2());

 if (hFontList->numFontStructs() > 1)
   avgWidth = avgWidth / hFontList->numFontStructs();
 //
 // Set the height of a small x, and the height of a capital letter.
 // The heights do not include any interline spacing or descenders.
 //
 xaXHeight = 0;
 xaCapHeight = 0;
 if (hFontList->numFontStructs() == 1)
  {
   xaXHeight = getProp(XA_X_HEIGHT); // Sometimes this returns -2.
   xaCapHeight = getProp(XA_CAP_HEIGHT);
  }
 if (xaXHeight <= 0)
   xaXHeight = inkSize("x").height();
 if (xaCapHeight <= 0)
   xaCapHeight = inkSize("X").height();
}

/*------------------------------------------------------------------------------
| IFontPrivateData::getProp                                                    |
| Utility for getting font properties.                                         |
| Use the first font struct from hFontList that has the property.              |
------------------------------------------------------------------------------*/
unsigned long IFontPrivateData::getProp(Atom atomId)
{
 unsigned long propValue;
 Bool done = 0;
 unsigned long numFonts = hFontList->numFontStructs();
 for (unsigned long index=0; !done && index<numFonts; ++index)
  {
   done = XGetFontProperty(hFontList->xFontStruct(index), atomId, &propValue);
  }
 if (done)
   return propValue;
 else
   return 0;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::inkSize                                                    |
| Ink extents of a text string.                                                |
------------------------------------------------------------------------------*/
ISize IFontPrivateData::inkSize(const char* text) const
{
 if (hFontList->xFontSet())
  {
   XRectangle overall_ink_return, overall_logical_return;
   XmbTextExtents(hFontList->xFontSet(), text, strlen(text),
                   &overall_ink_return, &overall_logical_return);
   return ISize(abs(overall_ink_return.width), abs(overall_ink_return.height));
  }
 else
  {
   int direction_return, font_ascent_return, font_descent_return;
   XCharStruct overall_return;
   XTextExtents(hFontList->xFontStruct(0), text, strlen(text),
                 &direction_return, &font_ascent_return,
                 &font_descent_return, &overall_return);
   return ISize(abs(overall_return.width),
                 abs(overall_return.ascent) + abs(overall_return.descent));
  }
}

/*------------------------------------------------------------------------------
| IFontPrivateData::logicalSize                                                |
| Logical extents of a text string.                                            |
------------------------------------------------------------------------------*/
ISize IFontPrivateData::logicalSize(const char* text) const
{
 if (hFontList->xFontSet())
  {
   XRectangle overall_ink_return, overall_logical_return;
   XmbTextExtents(hFontList->xFontSet(), text, strlen(text),
                   &overall_ink_return, &overall_logical_return);
   return ISize(abs(overall_logical_return.width),
                abs(overall_logical_return.height));
  }
 else
  {
   int direction_return, font_ascent_return, font_descent_return;
   XCharStruct overall_return;
   XTextExtents(hFontList->xFontStruct(0), text, strlen(text),
                 &direction_return, &font_ascent_return,
                 &font_descent_return, &overall_return);
   return ISize(abs(overall_return.width),
                 abs(font_ascent_return) + abs(font_descent_return));
  }
}

/*------------------------------------------------------------------------------
| Definitions for the library quick sort routine, qsort.                       |
------------------------------------------------------------------------------*/
typedef struct
 {
  unsigned long desirability;
  unsigned long strLength;
  char* fontName;
 } QSortElement;

int qsortCompare(const void* pElement1, const void* pElement2)
 {
  // Sort in descending order of desirability.
  // So most desirable will be at the beginning of the array.
  const QSortElement& element1 = *(const QSortElement*)pElement1;
  const QSortElement& element2 = *(const QSortElement*)pElement2;
  int result=-1;
  if (element1.desirability == element2.desirability) result=0;
  else
  if (element1.desirability < element2.desirability) result=1;
  return result;
 }

// d6893
// Determine length of string truncated at n'th last separator.
// Note: for MBCS, char must be less than 0x40.
//
size_t truncatedLength(const char* pStr,
                        size_t strLen,
                        unsigned long n,
                        char separator)
{
 const char* pChar = &(pStr[strLen]);
 int nSep = 0;
 while (--pChar >= pStr && nSep < n)
  {
   if (*pChar == separator)
     ++nSep;
  }
 return 1+pChar-pStr;
}


/*------------------------------------------------------------------------------
| IFontPrivateData::buildXLFD                                                  |
| Build an XLFD string to be used to find the closest match in all installed   |
| fonts.                                                                       |
------------------------------------------------------------------------------*/
void IFontPrivateData::buildXLFD( IXLFDString& strXLFD, bool useFontSet)
{		
  // To get a vector font a fully qualified name is needed.
  // The point sixe, resx, resy need to be specified.
  // O'Reilly V1, pg, 569

  // fixed p,m,c
  // slannts can be r,i,o

  //new font being created

    strXLFD.setWord(IString(resX),IXLFDString::RESX);
    strXLFD.setWord(IString(resY),IXLFDString::RESY);

  if( isBold() )
    strXLFD.setWord("bold",IXLFDString::WEIGHT);
  else
    strXLFD.setWord("medium",IXLFDString::WEIGHT);

  if( isItalic() )
    strXLFD.setWord("i",IXLFDString::SLANT);     //since we can have a couple of values
  else
     strXLFD.setWord("r",IXLFDString::SLANT);

  if (isFixed())
     strXLFD.setWord("m",IXLFDString::SPACING); //since we can have a couple of values
  else
    strXLFD.setWord("p",IXLFDString::SPACING);

  if( isVector() )
  {
    strXLFD.setWord("0",IXLFDString::PIXELS);
    strXLFD.setWord(IString(xpointSize),IXLFDString::POINTSIZE);
    strXLFD.setWord("0",IXLFDString::AVGWIDTH);
  }
  else
  {
    strXLFD.setWord("*",IXLFDString::PIXELS);  //indicates we want a bitmap font
    strXLFD.setWord("*",IXLFDString::AVGWIDTH);

    if( xpointSize>0)
      strXLFD.setWord(IString(xpointSize),IXLFDString::POINTSIZE);
    else if( strXLFD.getWord(IXLFDString::POINTSIZE) == "0" )
      strXLFD.setWord("100",IXLFDString::POINTSIZE);  //default a font size to 10
  }
}

/*------------------------------------------------------------------------------
| IFontPrivateData::checkForMatchSimple                                              |
| Search for the closest match in all installed fonts.  Then setup private     |
| data based on the first font in the font set found.                          |
------------------------------------------------------------------------------*/

//NOTE: only called by setBold, setItalic, setPointsize
IFontPrivateData&
IFontPrivateData::checkForMatchSimple(const IPresSpaceHandle& hpsInit)
{
  IXFontSet::PRIORITY priority = IXFontSet::NO_PRIORITY;
  XFontSet aFontSet=0;
  XmFontList fontlist=0;

  IXLFDString srchXLFD( XLFDstr );

  //Try to find an fairly exact list first to speed things up.
  buildXLFD(srchXLFD,true);

  if(hFontList->xFontSet() == NULL)  //was created with a fully qualified name
  {
    // we have to see if the font exist on the system
    srchXLFD.setWord("*",IXLFDString::PIXELS);  //indicates we want a bitmap font
    srchXLFD.setWord("*",IXLFDString::AVGWIDTH);

    unsigned long maxNumFonts=5000; // Arbitrarily large limit for XListFonts.
    char** namesArray;
    int foundCount = 0;
    namesArray = XListFonts(dpy,
                            srchXLFD.asString(),
                            maxNumFonts,     // arbitrary large value
                            &foundCount);

    if(( foundCount == 0) && srchXLFD.isItalic() )
    {
      srchXLFD.setWord("o",IXLFDString::SLANT);
      namesArray = XListFonts(dpy,
                            srchXLFD.asString(),
                            maxNumFonts,     // arbitrary large value
                            &foundCount);
    }

    if( foundCount > 0 )
    {
  	  XFontStruct* pFontStruct;
    	XmFontListEntry entry1 =  XmFontListEntryLoad(dpy
    	                            ,(char*)srchXLFD.asString()
      	                          ,XmFONT_IS_FONT,"NON_LOCALE_TAG");
  		fontlist = XmFontListAppendEntry (NULL,entry1);
    	XmFontListEntryFree(&entry1);
      XFreeFontNames(namesArray);
    }
    		
    if( fontlist )
    {
      IFontPrivateFontList::Handle
      temph = new IFontPrivateFontList(XmFontListCopy(fontlist), dpy);
      hFontList = temph; // Need temph to get reference count initialized to 1.
    }
  }
  else  //use the locale friendly fontSet
  {
    // Search for and load a fontSet that satisfies the current locale.
    if( isFixed() != XLFDstr.isFixed() )
      priority = IXFontSet::PROP_PRIORITY;

    if( isVector() != XLFDstr.isVector() )
      priority = IXFontSet::VECTOR_PRIORITY;

    if( pointSize() != srchXLFD.pointSize() )
      priority = IXFontSet::PTSIZE_PRIORITY;

    aFontSet = IXCreateFontSet( dpy, srchXLFD, true,priority);
    if( aFontSet )
    {
      IFontPrivateFontList::Handle
                            temph = new IFontPrivateFontSet(aFontSet, dpy);
      hFontList = temph; // Need temph to get reference count initialized to 1.
    }
  }

  if ((!aFontSet)  && (hFontList==NULL))
  {
    aFontSet = IXCreateFontSet(dpy, "fixed");
    if (!aFontSet)
    {
        ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                           IBaseErrorInfo::accessError,
                           IException::recoverable);
    }

    IFontPrivateFontList::Handle
      temph = new IFontPrivateFontSet(aFontSet, dpy);
    hFontList = temph; // Need temph to get reference count initialized to 1.
  }

   // Retrieve the font name and fill in the information
  getFontInfo();
  return *this;
}


/*------------------------------------------------------------------------------
| IFontPrivateData::checkForMatch                                              |
| Search for the closest match in all installed fonts.  Then setup private     |
| data based on the first font in the font set found.                          |
------------------------------------------------------------------------------*/
IFontPrivateData&
IFontPrivateData::checkForMatch(const IPresSpaceHandle& hpsInit)
{
	bool localeIndependent(false);
  XFontSet aFontSet=0;
  XmFontList fontlist=0;


  IString testName(facename);
  testName.stripTrailing(" ");

  IXLFDString srchXLFD = IXLFDString(facename);
  // If a fully qualified name was passed in with the charsets set
  // The assumption is that the user wants to use an exact font
  // which is locale independent
  if( testName.length()>0 )
  {
    if (!((srchXLFD.charsetRegistry() == "*") || (srchXLFD.charsetRegistry()=="|")))
      localeIndependent = true;

    if( localeIndependent == false ) //get a font set
    {
      //Try to find an fairly exact list first to speed things up.
      buildXLFD(srchXLFD,true);
      // Search for and load a fontSet that satisfies the current locale.
      aFontSet = IXCreateFontSet(dpy, srchXLFD,false);
      if( aFontSet )
      {
        IFontPrivateFontList::Handle
                              temph = new IFontPrivateFontSet(aFontSet, dpy);
        hFontList = temph; // Need temph to get reference count initialized to 1.
      }
    }
    else //localeIndependent
    {
      //check to see if the font exists on the system
      unsigned long maxNumFonts=5000; // Arbitrarily large limit for XListFonts.
      char** namesArray;
      int foundCount = 0;
      namesArray = XListFonts(dpy,
                              srchXLFD.asString(),
                              maxNumFonts,     // arbitrary large value
                              &foundCount);

      if( foundCount > 0 )
      {
  	    XFontStruct* pFontStruct;
    	  XmFontListEntry entry1 =  XmFontListEntryLoad(dpy
    	                                                ,(char*)srchXLFD.asString()
      	                                              ,XmFONT_IS_FONT,"NON_LOCALE_TAG");
  		  fontlist = XmFontListAppendEntry (NULL,entry1);
    	  XmFontListEntryFree(&entry1);
        XFreeFontNames(namesArray);
    	}
  		
      if( fontlist )
      {
        IFontPrivateFontList::Handle
        temph = new IFontPrivateFontList(XmFontListCopy(fontlist), dpy);
        hFontList = temph; // Need temph to get reference count initialized to 1.
      }
    } //end else localeIndependent
  }

  if ((!aFontSet)  && (hFontList==NULL))
  {
    //check for valid font
    IFontList* validFontList = IXLFDString::getFontListForCurrentLocale(dpy);
    IFontList::Cursor myCursor( *validFontList );

    IXLFDString IXLFDTestStr("-*-fixed");
      myCursor.setToFirst();

    IString theKey = IXLFDTestStr.keyName();
    if( validFontList->locateElementWithKey(theKey,myCursor) )
       aFontSet = IXCreateFontSet(dpy, "fixed");

    if( !aFontSet)
    {
      myCursor.setToFirst();
      IXLFDString baseXLFDstr = validFontList->elementAt(myCursor);

      if( xpointSize>0)
        baseXLFDstr.setWord(IString(xpointSize),IXLFDString::POINTSIZE);
      else if( baseXLFDstr.getWord(IXLFDString::POINTSIZE) == "0" )
        baseXLFDstr.setWord("100",IXLFDString::POINTSIZE);  //default a font size to 10

      buildXLFD(baseXLFDstr,true);

      aFontSet = IXCreateFontSet(dpy, baseXLFDstr,false);
    }

    if (!aFontSet)
    {
        ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                           IBaseErrorInfo::accessError,
                           IException::recoverable);
    }

    IFontPrivateFontList::Handle
      temph = new IFontPrivateFontSet(aFontSet, dpy);
    hFontList = temph; // Need temph to get reference count initialized to 1.
  }

   // Retrieve the font name and fill in the information
  getFontInfo();
  return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::loadXLFD                                                   |
------------------------------------------------------------------------------*/
// This method is used by IFontDialog.
IFontPrivateData &
IFontPrivateData::loadXLFD(char * xlfd)
{
    XFontSet aFontSet = IXCreateFontSet(dpy, xlfd);

    if (!aFontSet)
    {
        ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                           IBaseErrorInfo::accessError,
                           IException::recoverable);
    }


    IFontPrivateFontList::Handle
        temph = new IFontPrivateFontSet(aFontSet, dpy);
    hFontList = temph; // Need temph to get reference count initialized to 1.

    // Set the private data using the font in hFontList.
    getFontInfo();

    return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::loadXLFD                                                   |
------------------------------------------------------------------------------*/
// This method is used by IFontDialog.
IFontPrivateData &
IFontPrivateData::loadXLFD(IXLFDString baseXLFDstr)
{
    XFontSet aFontSet = IXCreateFontSet(dpy, baseXLFDstr,false);
    if (!aFontSet)
    {
        ITHROWLIBRARYERROR(IC_FONT_ACCESS_ERROR,
                           IBaseErrorInfo::accessError,
                           IException::recoverable);
    }


    IFontPrivateFontList::Handle
        temph = new IFontPrivateFontSet(aFontSet, dpy);
    hFontList = temph; // Need temph to get reference count initialized to 1.

    // Set the private data using the font in hFontList.
    getFontInfo();

    return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::beginUsingFont                                             |
------------------------------------------------------------------------------*/
IFontPrivateData&
IFontPrivateData::beginUsingFont(const IPresSpaceHandle& hps)
{
   IMODTRACE_ALL("IFontPrivateData::beingUsingFont");

   // If it appears that beginUsingFont called already without a endUsingFont
   // having been called, don't bother saving original Font.
   if (prevFont == 0)
      prevFont = XGContextFromGC(((IXDC*)hps)->gc());

   // XSetFont does not return a return code.
   // Refer to O'Reilly Volume 2, Appendix B.  XSetFont appears in Table B-2.
   XSetFont(display(), ((IXDC*)hps)->gc(), hFontList->xFontStruct(0)->fid);
   return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::endUsingFont                                               |
------------------------------------------------------------------------------*/
IFontPrivateData &
IFontPrivateData::endUsingFont(const IPresSpaceHandle& hps)
{
    if (prevFont)
    {
        // XSetFont does not return a return code.
        // Refer to O'Reilly Volume 2, Appendix B.  XSetFont appears in Table B-2.
        XSetFont(display(), ((IXDC*)hps)->gc(), prevFont);
    }

    prevFont = 0;
    return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::setBold                                                    |
------------------------------------------------------------------------------*/
IFontPrivateData &
IFontPrivateData::setBold(Boolean bTurnOn)
{
    if (bTurnOn)
        flags |= bold;
    else
        flags &= ~bold;
    return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::setItalic                                                  |
------------------------------------------------------------------------------*/
IFontPrivateData &
IFontPrivateData::setItalic(Boolean bTurnOn)
{
   if (bTurnOn)
      flags |= italic;
   else
      flags &= ~italic;
   return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::setFixed                                                   |
------------------------------------------------------------------------------*/
IFontPrivateData &
IFontPrivateData::setFixed(Boolean bTurnOn)
{
   if (bTurnOn)
      flags |= fixed;
   else
      flags &= ~fixed;
   return *this;
}

/*------------------------------------------------------------------------------
| IFontPrivateData::setVector                                                  |
------------------------------------------------------------------------------*/
IFontPrivateData &
IFontPrivateData::setVector(Boolean bTurnOn)
{
   if (bTurnOn)
      flags |= vector;
   else
      flags &= ~vector;
   return *this;
}

#endif //IC_MOTIF
