// Revision: 27 1.12.3.5 source/ui/baseapp/ifont.cpp, font, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ifont.cpp                                                         *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ifont.hpp.                                                              *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C"
{
   #define INCL_GPILCIDS
   #define INCL_GPIPRIMITIVES
   #define INCL_WINMLE
   #define INCL_WINSTDFONT
   #define INCL_WINSYS
   #define INCL_WINERRORS
   #define IUSING_OLE
   #ifdef WIN32_LEAN_AND_MEAN
     #undef WIN32_LEAN_AND_MEAN
   #endif
   #include <iwindefs.h>
   #include <string.h>
   #include <stdlib.h>
   #include <math.h>
   #include <ctype.h>

   #ifdef IC_MOTIF
     #include <Xm/Xm.h>
     #include <Xm/AtomMgr.h>
     #include <Xm/BulletinB.h>
     #include <Xm/CascadeBG.h>
     #include <Xm/CascadeB.h>      // 27912
     #include <Xm/FileSB.h>
     #include <Xm/MenuShell.h>
     #include <Xm/RowColumn.h>
     #include <Xm/ToggleBG.h>
     #include <Xm/ScrolledW.h>
     #include <Xm/Text.h>
     #include <Xm/Label.h>
     #include <Xm/Form.h>
     #include <Xm/DrawingA.h>
     #include <X11/IntrinsicP.h>   // d8175 - for CoreP.h.
     #include <X11/CoreP.h>        // d8175 - for being_destroyed.
   #endif
}

#include <ifont.hpp>
#include <ifontprv.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#include <iexcept.hpp>
#include <iwindow.hpp>
#include <iwinpriv.hpp>
#include <ifontdlg.hpp>
#include <iwcname.hpp>
#include <istparse.hpp>
#include <istylset.hpp>
#include <icombobs.hpp>
#include <ispinbas.hpp>
#include <istattxt.hpp>
#include <iprimlck.hpp>
#include <istatics.hpp>

#ifdef IC_PM
  #include <ipmbidi.hpp>
#endif
#ifdef IC_MOTIF
  #include <icnrobj.hpp>
  #include <cnr.h>
  #include <icong.h>
  #include <icnrrec.hpp>
  #include <icnrctl.hpp>
  #include <icnrctlm.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IFONT_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_MOTIFWIN
extern unsigned long propagationInProgress;
#endif

#ifdef IC_MOTIF
static IFontPrivateData *gFontPrivateData = 0;
#endif

#ifdef IC_PMWIN

// Local non-PM flags used by this class

#define IC_BITMAP_FONTS_ONLY              0x01
#define IC_VECTOR_FONTS_ONLY              0x02
#define IC_NONPROPORTIONAL_FONTS_ONLY     0x04
#define IC_MUSTUSEVECTORFONT              0x08

#ifdef IC_PM
const int iCharArrayWidth = 256;
#endif

/*------------------------------------------------------------------------------
| Note:                                                                        |
|                                                                              |
| Because of reference counting, any new or existing member function which     |
| causes a change to the font should call copyPrivate().  This ensures that    |
| existing font objects which point to the same font private data object       |
| will not be changed.                                                         |
------------------------------------------------------------------------------*/


#ifdef IC_WIN
/*------------------------------------------------------------------------------
| CalcPointSize                                                                |
|                                                                              |
| Calculate the pointsize based on the height of the font                      |
| requested and the resolution of the device passed in                         |
------------------------------------------------------------------------------*/
long CalcPointSize( IPresSpaceHandle hps, long tmHeight, long tmIntlLeading )
{
  /*------------------------------------------------------------------*/
  /* Get device resolution for device passed for conversion to points */
  /*------------------------------------------------------------------*/
  int vyPelsPerLogInch = GetDeviceCaps( hps, LOGPIXELSY );

  /*------------------------------------------------------------------*/
  /* if the return value is zero, the problem may be caused by using  */
  /* a metafile device context.                                       */
  /*------------------------------------------------------------------*/
  if (vyPelsPerLogInch == 0)
  {
    IPresSpaceHandle desktopPs = IGETPS(HWND_DESKTOP);

    /*----------------------------------------------------------------*/
    /* Get device resolution for desktop for conversion to point size */
    /*----------------------------------------------------------------*/
    vyPelsPerLogInch = GetDeviceCaps( desktopPs, LOGPIXELSY );

    /*----------------------------------------------------------------*/
    /* Release any device context acquired                            */
    /*----------------------------------------------------------------*/
    IRELEASEPS( HWND_DESKTOP, desktopPs );
  }

  /*------------------------------------------------------------------*/
  /* compute the twip size                                            */
  /*------------------------------------------------------------------*/
  DWORD wTwipSize = (1440 * (tmHeight - tmIntlLeading)) / vyPelsPerLogInch;

  /*------------------------------------------------------------------*/
  /* Set the point size internally, rounding to the nearest point     */
  /*------------------------------------------------------------------*/
  return (long)((wTwipSize + 10) / 20);
}


/*------------------------------------------------------------------------------
| CalcFontHeight                                                               |
|                                                                              |
| Calculate the height of the font (in pixels) based on the point size         |
| requested and the resolution of the device passed in                         |
------------------------------------------------------------------------------*/
int CalcFontHeight( IPresSpaceHandle presSpaceHandle, unsigned long ulPointSize )
{
  /*------------------------------------------------------------------*/
  /* Get device context for desktop if no context passed              */
  /*------------------------------------------------------------------*/
  IPresSpaceHandle hps;
  if (presSpaceHandle == 0)
    hps = IGETPS(HWND_DESKTOP);
  else
    hps = presSpaceHandle;

  /*------------------------------------------------------------------*/
  /* Get device resolution for desktop for conversion to point size   */
  /*------------------------------------------------------------------*/
  int vyPelsPerLogInch = GetDeviceCaps( hps, LOGPIXELSY );

  /*------------------------------------------------------------------*/
  /* Release any device context acquired                              */
  /*------------------------------------------------------------------*/
  if (presSpaceHandle == 0)
    IRELEASEPS( HWND_DESKTOP, hps );

  /*------------------------------------------------------------------*/
  /* Calculate the height, converting based on device resolution and  */
  /* adding in a roundoff value.  Change the sign of final result to  */
  /* negative to indicate that this is a character height value, not  */
  /* a cell height value (i.e. no allowance for internal leading)     */
  /*------------------------------------------------------------------*/
  return -((((int)ulPointSize*vyPelsPerLogInch)+36)/72);
}
#endif


/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
|                                                                              |
| Construct a font for the font being used in an IWindow                       |
------------------------------------------------------------------------------*/
IFont :: IFont (const IWindow* pwindow)
         : ppd(new IFontPrivateData())
{
   IMODTRACE_DEVELOP("IFont-ctor(1)");

   IWindow
    *fontWindow = (IWindow*)pwindow;   // Cast away const.
   if ( pwindow == 0 )
      fontWindow = IWindow::desktopWindow();

   IPresSpaceHandle hps = fontWindow->presSpace();

#ifdef IC_PM
   /*******************************************************************/
   /* For most windows, we can get the font information needed from   */
   /* the presentation space handle                                   */
   /*******************************************************************/
   initialize(hps);

   IString fontName( 0, 120 );
   unsigned long ulPresParams;

   /*******************************************************************/
   /* If the window passed is non-null, then do some additional       */
   /* processing to set the font up correctly                         */
   /*******************************************************************/
   if (pwindow != 0)
   {
     /*****************************************************************/
     /* First check the presentation parameters for any additional    */
     /* attributes being used by the graphics engine to display font  */
     /*****************************************************************/
     if ( WinQueryPresParam( fontWindow->handle(),
                             PP_FONTNAMESIZE,
                             0,
                             &ulPresParams,
                             (ULONG)120,
                             (PVOID)(char*)fontName,
                             QPF_NOINHERIT ) != 0 )
     {

       if (fontName.includes(".Bold"))
         ppd->pfattrsCl->fsSelection |= FATTR_SEL_BOLD;
       if (fontName.includes(".Italic"))
         ppd->pfattrsCl->fsSelection |= FATTR_SEL_ITALIC;
       else
         ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_ITALIC;
       if (fontName.includes(".Outline"))
         ppd->pfattrsCl->fsSelection |= FATTR_SEL_OUTLINE;
       else
         ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_OUTLINE;
       if (fontName.includes(".Underscore"))
         ppd->pfattrsCl->fsSelection |= FATTR_SEL_UNDERSCORE;
       else
         ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_UNDERSCORE;

       if (fontName.includes(".Strikeout"))
         ppd->pfattrsCl->fsSelection |= FATTR_SEL_STRIKEOUT;
       else
         ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_STRIKEOUT;

     }

     /*****************************************************************/
     /* Then check if window is an MLE control                        */
     /*****************************************************************/
     IWindowHandle hMle = fontWindow->handle();
     IWindowClassName className( hMle );
     if ( className == WC_MLE )
     {
       /***************************************************************/
       /* If it is MLE, need to do some special case processing to    */
       /* create an IFont with the correct information since you      */
       /* cannot get at the hps for an MLE (as info set above is bad) */
       /***************************************************************/
       hMle.sendEvent( MLM_QUERYFONT,
                       IEventParameter1(ppd->pfattrsCl),
                       IEventParameter2(0) );
       HDC hdc(GpiQueryDevice(hps));
       long vRes;
       DevQueryCaps(hdc, CAPS_VERTICAL_FONT_RES, 1L, &vRes);

       // Add 0.5 before doing integer conversion to avoid truncation errors
       unsigned long currPointSize = FIXEDINT(
        (( MAKEFIXED(ppd->pfattrsCl->lMaxBaselineExt, 0 ) * 72) / vRes)+0x8000);
       ppd->fxClPointSize = MAKEFIXED( currPointSize, 0 );
     }
   }
#endif
#ifdef IC_WIN
   // Do a check for a canvas window (since Windows does not allow fonts
   // to be set for this window class)
   IEventResult er = fontWindow->sendEvent( IC_UM_REQUEST_CANVASFONT,
                                            IEventParameter1( 0 ),
                                            IEventParameter2( 0 ) );
   IFont* canvasFont = (IFont*)er.asUnsignedLong();

   // If so, then just copy the private data over from the canvas object
   if (canvasFont)
   {
     canvasFont->ppd->addRef();
     ppd->removeRef();
     ppd = canvasFont->ppd;
   }
   else
   {

     /*---------------------------------------------------------------*/
     /* Allocate a structure for storing the font information         */
     /*---------------------------------------------------------------*/
     ppd->plogfontCl = new LOGFONT;

     /*---------------------------------------------------------------*/
     /* Query the window for the font handle in use                   */
     /*---------------------------------------------------------------*/
     HFONT hFont = fontWindow->sendEvent( WM_GETFONT,
                                          IEventParameter1( 0 ),
                                          IEventParameter2( 0 ) );

     /*---------------------------------------------------------------*/
     /* If NULL, then system font is in use, so get the handle for it */
     /*---------------------------------------------------------------*/
     if (!hFont)
       hFont = ppd->systemFont;


     /*---------------------------------------------------------------*/
     /* Finally get the font information for this handle and then     */
     /*  create a new font with this information                      */
     /*---------------------------------------------------------------*/
     int rc = GetObject( hFont, sizeof(LOGFONT), ppd->plogfontCl );
     if (!rc)
       GetObject( ppd->systemFont, sizeof(LOGFONT),
                  ppd->plogfontCl );
     makeFontMetrics( hps );
     ITRACE_ALL( IString("Font handle is: ") +
                 IString( *(ppd->fpFontHandle) ));
   }
#endif
   fontWindow->releasePresSpace( hps );
}

/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
|                                                                              |
| Construct font based an input parameters.                                    |
------------------------------------------------------------------------------*/
IFont :: IFont         (const char*             pszFaceName,
                        unsigned long           ulPointSize,
                        bool                 useFixedFont,
                        bool                 useVectorFont,
                        const IPresSpaceHandle& presSpaceHandle)
         : ppd(new IFontPrivateData())
{
   IMODTRACE_DEVELOP("IFont-ctor(2)");

// pszFaceName could contain "qualified Name"...if so, parse into faceName and
// pointSize
IString qualName = pszFaceName;
IString faceName, pointSize;

if ( qualName.includes("IOC:") )
  {
  qualName >> "IOC:" >> faceName >> ":" >> pointSize;
  pszFaceName = faceName;
  ulPointSize = pointSize.asUnsigned();
  }

#ifdef IC_PM
   /* initialize font attributes structure */
   ppd->pfattrsCl = new FATTRS;
   memset(ppd->pfattrsCl, 0, sizeof(FATTRS));
   (ppd->pfattrsCl)->usRecordLength = sizeof(FATTRS);

// If setName(), setPointSize(), useNonPropOnly(), useBitmapOnly(),
// and useVectorOnly() are called from here we'll end up calling
// checkForMatch excesively.
// So just do what these functions do, but don't call fontChanged()
// and checkForMatch() until the end.

     strcpy((ppd->pfattrsCl)->szFacename, pszFaceName);

     ppd->fxClPointSize = MAKEFIXED(ulPointSize, 0);
     if(useVectorFont) {
       setCharHeight(ulPointSize, presSpaceHandle);
       setCharWidth(ulPointSize, presSpaceHandle);
     }

     if (useFixedFont)
        ppd->flClStyles = ppd->flClStyles | IC_NONPROPORTIONAL_FONTS_ONLY;
     else
        ppd->flClStyles = ppd->flClStyles & ~IC_NONPROPORTIONAL_FONTS_ONLY;

   if (useVectorFont)
      ppd->flClStyles = ppd->flClStyles | IC_VECTOR_FONTS_ONLY;
   else
      ppd->flClStyles = ppd->flClStyles & ~IC_VECTOR_FONTS_ONLY;

   if (!useVectorFont) {
      ppd->flClStyles = ppd->flClStyles | IC_BITMAP_FONTS_ONLY;
      mustUseVector(FALSE);
   } else {
      ppd->flClStyles = ppd->flClStyles & ~IC_BITMAP_FONTS_ONLY;
      mustUseVector(TRUE);
   } /* endif */

   checkForMatch(presSpaceHandle);

   // Insure that the fontmetrics information matches the font selected
   // by creating the font temporarily into the hps passed and then
   // querying for the accurate information (not the general info)
   if ( ppd->pfontmetCl )
   {
     delete ppd->pfontmetCl;
     ppd->pfontmetCl = 0;
   }
   makeFontMetrics( presSpaceHandle );
#endif
#ifdef IC_WIN
   /*------------------------------------------------------------------*/
   /* Create a new logfont structure, null it out and copy facename    */
   /*------------------------------------------------------------------*/
   ppd->plogfontCl = new LOGFONT;
   memset(ppd->plogfontCl, 0, sizeof(LOGFONT));
   strcpy((ppd->plogfontCl)->lfFaceName, pszFaceName);

   /*------------------------------------------------------------------*/
   /* Calculate height from point size, and set other values to default*/
   /*------------------------------------------------------------------*/
   (ppd->plogfontCl)->lfHeight =
                         CalcFontHeight( presSpaceHandle, ulPointSize );
   (ppd->plogfontCl)->lfWeight = FW_DONTCARE;
   (ppd->plogfontCl)->lfQuality = DEFAULT_QUALITY;
   (ppd->plogfontCl)->lfPitchAndFamily = FF_DONTCARE;

   /*------------------------------------------------------------------*/
   /* If fixed pitch specified, then set flag to denote this           */
   /*------------------------------------------------------------------*/
   if (useFixedFont)
     (ppd->plogfontCl)->lfPitchAndFamily |= FIXED_PITCH;

   /*------------------------------------------------------------------*/
   /* If vector font specified, then set flag to denote this           */
   /*------------------------------------------------------------------*/
   if (useVectorFont)
     (ppd->plogfontCl)->lfPitchAndFamily |= TMPF_TRUETYPE;

   /*------------------------------------------------------------------*/
   /* Finally, create the font from the logfont structure set-up       */
   /*------------------------------------------------------------------*/
   makeFontMetrics( presSpaceHandle );
   ITRACE_ALL( IString("Font handle is: ") +
               IString( *(ppd->fpFontHandle) ));
#endif
}

/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
| Construct font to one being used in Presentation Space                       |
------------------------------------------------------------------------------*/
IFont :: IFont(const IPresSpaceHandle& presSpace)
         : ppd(new IFontPrivateData())
{
   IMODTRACE_DEVELOP("IFont-ctor(3)");
   initialize(presSpace);
}

/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
| Copy constructor                                                             |
------------------------------------------------------------------------------*/
IFont :: IFont (const IFont& fntmCopy)
{
  IMODTRACE_DEVELOP("IFont-copy ctor");
  fntmCopy.ppd->addRef();
  ppd = fntmCopy.ppd;
}

/*------------------------------------------------------------------------------
| IFont::~IFont                                                                |
------------------------------------------------------------------------------*/
IFont :: ~IFont ()
{
   IMODTRACE_DEVELOP("IFont-dtor");
   ppd->removeRef();
   ppd = 0;
}

/*------------------------------------------------------------------------------
| IFont::operator=                                                             |
------------------------------------------------------------------------------*/
IFont& IFont::operator= (const IFont &aFont)
{
  if (this == &aFont )
    return *this;
  aFont.ppd->addRef();
  ppd->removeRef();
  ppd = aFont.ppd;
  return (*this);
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IFont::setSizeFromHps                                                        |
------------------------------------------------------------------------------*/
IFont& IFont :: setSizeFromHps (unsigned long ulSize,
                                const IPresSpaceHandle& presSpaceHandle)
{
   copyPrivate();
   ppd->fxClPointSize = MAKEFIXED(ulSize, 0);
   setWidthHeight(ulSize, presSpaceHandle);
   makeCharWidths(presSpaceHandle);
   return *this;
}
#endif // IC_PM

/*------------------------------------------------------------------------------
| IFont::makeFontMetrics                                                       |
------------------------------------------------------------------------------*/
IFont& IFont::makeFontMetrics( const IPresSpaceHandle& presSpaceHandle,
                               bool fUpdateCharSet )
{
   IMODTRACE_DEVELOP("IFont::makeFontMetrics");
#ifdef IC_PM
   if (!ppd->pfontmetCl) {
      IPresSpaceHandle hps;
      if(presSpaceHandle)
        hps = presSpaceHandle;
      else
        hps = WinGetPS(HWND_DESKTOP);
      beginUsingFont(hps);
      ppd->pfontmetCl = new FONTMETRICS;
      GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), ppd->pfontmetCl);
      endUsingFont(hps);
      if (presSpaceHandle == 0)
        WinReleasePS(hps);
   } /* endif */
#endif
#ifdef IC_WIN
   if (fUpdateCharSet)
     ppd->plogfontCl->lfCharSet = ppd->fcharSet;
   HFONT hFont = CreateFontIndirect( ppd->plogfontCl );
   ppd->fpFontHandle->removeRef();
   ppd->fpFontHandle = new IFontHandle( hFont, true );
   ppd->ptextmetricCl = new TEXTMETRIC;

   IPresSpaceHandle hps;
   if (presSpaceHandle == 0)
      hps = IGETPS(HWND_DESKTOP);
   else
      hps = presSpaceHandle;

   HFONT tempFont = (HFONT) SelectObject( hps, *(ppd->fpFontHandle) );
   GetTextMetrics( hps, ppd->ptextmetricCl );
   SelectObject( hps, tempFont );

   /*------------------------------------------------------------------*/
   /* Calculate and save the new point size for the font               */
   /*------------------------------------------------------------------*/
   ppd->fxClPointSize = CalcPointSize( hps, ppd->ptextmetricCl->tmHeight,
                                  ppd->ptextmetricCl->tmInternalLeading);

   if(presSpaceHandle == 0)
     IRELEASEPS(HWND_DESKTOP,hps);
#endif
   return *this;
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IFont::makeCharWidths                                                        |
------------------------------------------------------------------------------*/
IFont& IFont :: makeCharWidths (const IPresSpaceHandle& presSpaceHandle)
{
   IMODTRACE_DEVELOP("IFont::makeCharWidths");
   if (!ppd->plClCharWidths) {
      IPresSpaceHandle hps;
      if (presSpaceHandle)
        hps = presSpaceHandle;
      else
        hps = WinGetPS(HWND_DESKTOP);
      beginUsingFont(hps);
      ppd->plClCharWidths = new long[iCharArrayWidth];
      GpiQueryWidthTable(hps, 0, iCharArrayWidth, ppd->plClCharWidths);
      endUsingFont(hps);
      if (presSpaceHandle == 0)
        WinReleasePS(hps);
   } /* endif */
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::checkForMatch                                                         |
|                                                                              |
| Search for the closest match in all installed fonts.                         |
------------------------------------------------------------------------------*/
IFont& IFont :: checkForMatch (const IPresSpaceHandle& presSpaceHandle)
{

   unsigned long ulPtSize = FIXEDINT(ppd->fxClPointSize);
   long lNumFonts;
   long lReqFonts = 0;
   long lBitmapMatch = 0;
   long lNextBitmap = 0;
   long lVectorMatch = 0;
   long lHowClose = 99999;
   unsigned long
     Index = 0,
     BitmapIndex = 0,
     VectorIndex = 0,
     NextBitmapIndex = 0;

   IPresSpaceHandle hps;
   if (presSpaceHandle) {
      hps = presSpaceHandle;
   } else {
      hps = WinGetPS(HWND_DESKTOP);
   } /* endif */

   lNumFonts = GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE,
                  (PSZ)(ppd->pfattrsCl)->szFacename, &lReqFonts, 0L, 0);

   if (lNumFonts) {
      ITRACE_DEVELOP(IString(lNumFonts) + IString("Fonts found."));

      FONTMETRICS* pfm;

      pfm = new FONTMETRICS[lNumFonts];

      GpiQueryFonts(hps, QF_PUBLIC | QF_PRIVATE, (PSZ)(ppd->pfattrsCl)->szFacename,
         &lNumFonts, sizeof(FONTMETRICS), pfm);
      ITRACE_DEVELOP((IString(lNumFonts) + IString("Fonts found for ")) +
                      IString(ppd->pfattrsCl->szFacename) );
      long lHorzRes, lVertRes;
      HDC hdc = GpiQueryDevice(hps);
      DevQueryCaps(hdc, CAPS_HORIZONTAL_FONT_RES, 1L, &lHorzRes);
      DevQueryCaps(hdc, CAPS_VERTICAL_FONT_RES, 1L, &lVertRes);
      ITRACE_DEVELOP((IString("H&V resolution ") + IString(lHorzRes)) +
                      IString(lVertRes));

      for (Index = 0; Index < lNumFonts ; Index++)
      {
         // Skip font if it doesn't match the Proportional/Fixed specification
         // or the Vector/Bitmap specification
         if ( ((isNonPropOnly()) && (pfm[Index].fsType & FM_TYPE_FIXED))   ||
              ((!isNonPropOnly()) && (!(pfm[Index].fsType & FM_TYPE_FIXED))) ||
              ((isVectorOnly()) && (pfm[Index].fsDefn & FM_DEFN_OUTLINE))  ||
              ((isBitmapOnly()) && (!(pfm[Index].fsDefn & FM_DEFN_OUTLINE))) )
         {
            // Vector font doesn't need to check device resolution and point size
            if (pfm[Index].fsDefn & FM_DEFN_OUTLINE)
            {
               lVectorMatch = pfm[Index].lMatch;
               VectorIndex = Index;
            }
            else
              // Check bitmap for device resolution and pointSize
              if ((pfm[Index].sXDeviceRes == lHorzRes) &&
                  (pfm[Index].sYDeviceRes == lVertRes))
              {
                ITRACE_DEVELOP((IString(pfm[Index].sNominalPointSize) +
                                IString(" vs. ")) + IString(ulPtSize*10) );
                if (pfm[Index].sNominalPointSize == (ulPtSize*10))
                {
                   lBitmapMatch = pfm[Index].lMatch;
                   BitmapIndex = Index;
                }
                else
                {
                   long currentHowClose =
                       abs((ulPtSize*10) - pfm[Index].sNominalPointSize);
                   if (!lBitmapMatch &&
                       currentHowClose <= lHowClose)
                   {
                       lNextBitmap = pfm[Index].lMatch;
                       NextBitmapIndex = Index;
                       lHowClose = currentHowClose;
                   } /* endif */
                }
              }
         } /* endif for match on specification */

      } /* endfor */

      // We'll reset Index below to the position in pfm that's the best match.
      // That way we can save the FONTMETRICS.
      if (isBitmapOnly()) {
         if (!lBitmapMatch) {
            lBitmapMatch = lNextBitmap;
         } /* endif */
         (ppd->pfattrsCl)->lMatch = lBitmapMatch;
         ITRACE_DEVELOP("Exact bitmap match found.");
         (ppd->pfattrsCl)->fsFontUse = (ppd->pfattrsCl)->fsFontUse & ~FATTR_FONTUSE_OUTLINE;
         Index = BitmapIndex;
      } else {
         if (isVectorOnly() || (ppd->flClStyles & IC_MUSTUSEVECTORFONT)) {
            (ppd->pfattrsCl)->lMatch = lVectorMatch;
            ITRACE_DEVELOP("Exact vector match found.");
            (ppd->pfattrsCl)->fsFontUse = (ppd->pfattrsCl)->fsFontUse | FATTR_FONTUSE_OUTLINE;
            Index = VectorIndex;
         } else {
            if (lBitmapMatch) {
               (ppd->pfattrsCl)->lMatch = lBitmapMatch;
               (ppd->pfattrsCl)->fsFontUse = (ppd->pfattrsCl)->fsFontUse & ~FATTR_FONTUSE_OUTLINE;
               Index = BitmapIndex;
            } else {
               if (lVectorMatch) {
                  (ppd->pfattrsCl)->lMatch = lVectorMatch;
                  (ppd->pfattrsCl)->fsFontUse = (ppd->pfattrsCl)->fsFontUse | FATTR_FONTUSE_OUTLINE;
                  Index = VectorIndex;
               } else {
                  (ppd->pfattrsCl)->lMatch = lNextBitmap;
                  (ppd->pfattrsCl)->fsFontUse = (ppd->pfattrsCl)->fsFontUse & ~FATTR_FONTUSE_OUTLINE;
                  Index = NextBitmapIndex;
               } /* endif */
            } /* endif */
         } /* endif */
      } /* endif */

      // Ok, now let's save the FONTMETRICS
      delete ppd->pfontmetCl;
      ppd->pfontmetCl = new FONTMETRICS;
      memcpy(ppd->pfontmetCl, &pfm[Index], sizeof(FONTMETRICS));

      // Set some FATTRS values from the FONTMETRICS
      (ppd->pfattrsCl)->idRegistry = (ppd->pfontmetCl)->idRegistry;
      (ppd->pfattrsCl)->usCodePage = (ppd->pfontmetCl)->usCodePage;
      if (!(ppd->pfontmetCl->fsDefn & FM_DEFN_OUTLINE))
      {
         (ppd->pfattrsCl)->lMaxBaselineExt = (ppd->pfontmetCl)->lMaxBaselineExt;
         (ppd->pfattrsCl)->lAveCharWidth = (ppd->pfontmetCl)->lAveCharWidth;
      }
      (ppd->pfattrsCl)->fsFontUse |= FATTR_FONTUSE_NOMIX;

      delete [] pfm;
   } /* endif */

   if ((isNonPropOnly() || isBitmapOnly() || isVectorOnly())
         && !lBitmapMatch && !lVectorMatch) {
      if (strcmp((ppd->pfattrsCl)->szFacename, (char*)"Courier")) {
         strcpy((ppd->pfattrsCl)->szFacename, (char*)"Courier");
         ITRACE_DEVELOP("Checking for match again.");
         checkForMatch(hps);
      } /* endif */
   } /* endif */

   if(presSpaceHandle == 0)
     WinReleasePS(hps);

   return *this;
}
#endif // IC_PM


/*------------------------------------------------------------------------------
| IFont::mustUseVector                                                         |
------------------------------------------------------------------------------*/
IFont& IFont :: mustUseVector (bool fDoIt)
{
   if (isBitmapOnly()) {
      ppd->flClStyles = ppd->flClStyles & ~IC_MUSTUSEVECTORFONT;
      return *this;
   } /* endif */
   if (ppd->pntClAngle.x() != 1 || ppd->pntClAngle.y() != 0 ||
       ppd->pntClShear.x() != 0 || ppd->pntClShear.y() != 1 || isVectorOnly()) {
      ppd->flClStyles = ppd->flClStyles | IC_MUSTUSEVECTORFONT;
      return *this;
   } /* endif */

   if (fDoIt) {
      ppd->flClStyles = ppd->flClStyles | IC_MUSTUSEVECTORFONT;
   } else {
      ppd->flClStyles = ppd->flClStyles & ~IC_MUSTUSEVECTORFONT;
   } /* endif */
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::initialize                                                            |
------------------------------------------------------------------------------*/
IFont& IFont :: initialize(const IPresSpaceHandle& presSpaceHandle)
{
   IMODTRACE_DEVELOP("IFont::initialize");
   IPresSpaceHandle hps;
   if (presSpaceHandle == 0)
      hps = IGETPS(HWND_DESKTOP);
   else
      hps = presSpaceHandle;

#ifdef IC_PM
   ppd->pfattrsCl = new FATTRS;
   ppd->pfontmetCl = new FONTMETRICS;
   GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), ppd->pfontmetCl);

   // Set our internal flags for this font so that checkForMatch
   // will work properly.
   // Don't call the useNonPropOnly() type function since they'll
   // cause us to call checkForMatch() prematurely.

   if (ppd->pfontmetCl->fsType & FM_TYPE_FIXED)
     ppd->flClStyles = ppd->flClStyles | IC_NONPROPORTIONAL_FONTS_ONLY;

   if (ppd->pfontmetCl->fsDefn & FM_DEFN_OUTLINE)
     ppd->flClStyles = ppd->flClStyles | IC_VECTOR_FONTS_ONLY;
   else
     ppd->flClStyles = ppd->flClStyles | IC_BITMAP_FONTS_ONLY;

   /* initialize font attributes structure */
   (ppd->pfattrsCl)->usRecordLength = sizeof(FATTRS);
   (ppd->pfattrsCl)->fsSelection = (ppd->pfontmetCl)->fsSelection;
   (ppd->pfattrsCl)->lMatch = (ppd->pfontmetCl)->lMatch;
   (ppd->pfattrsCl)->idRegistry = (ppd->pfontmetCl)->idRegistry;
   (ppd->pfattrsCl)->usCodePage = (ppd->pfontmetCl)->usCodePage;
   (ppd->pfattrsCl)->fsType = 0;
   (ppd->pfattrsCl)->fsFontUse = FATTR_FONTUSE_NOMIX;
   strcpy((ppd->pfattrsCl)->szFacename, (ppd->pfontmetCl)->szFacename);

   long currPointSize;
   if (ppd->flClStyles & IC_VECTOR_FONTS_ONLY) {
     (ppd->pfattrsCl)->lMaxBaselineExt = 0;
     (ppd->pfattrsCl)->lAveCharWidth = 0;

     long lyFontResolution;
     SIZEF sizef;
     HDC hDC = GpiQueryDevice(hps);
     DevQueryCaps( hDC, CAPS_VERTICAL_FONT_RES,
                   (long)1, &lyFontResolution);
     GpiQueryCharBox (hps, &sizef);
     // Add 0.5 before doing integer conversion to avoid truncation errors
     currPointSize = FIXEDINT((((sizef.cy) * 72) / lyFontResolution)+0x8000);
     }
   else {
     (ppd->pfattrsCl)->lMaxBaselineExt = (ppd->pfontmetCl)->lMaxBaselineExt;
     (ppd->pfattrsCl)->lAveCharWidth = (ppd->pfontmetCl)->lAveCharWidth;
     currPointSize = (ppd->pfontmetCl)->sNominalPointSize/10;
     }

   setSizeFromHps(currPointSize, hps);
#endif
#ifdef IC_WIN
   /*------------------------------------------------------------------*/
   /* Allocate private data to store the logfont                       */
   /*------------------------------------------------------------------*/
   ppd->plogfontCl = new LOGFONT;

   /*------------------------------------------------------------------*/
   /* Use a temp font (stock) to use to get current selection with     */
   /*------------------------------------------------------------------*/
   HFONT hFont = (HFONT) SelectObject( hps, ppd->systemFont );

   /*------------------------------------------------------------------*/
   /* Select it into back into the hps and use the handle to retrieve  */
   /* the log font information for that font                           */
   /*------------------------------------------------------------------*/
   SelectObject( hps, hFont );
   GetObject( hFont, sizeof(LOGFONT), ppd->plogfontCl );

   /*------------------------------------------------------------------*/
   /* Construct a font using that information                          */
   /*------------------------------------------------------------------*/
   makeFontMetrics( hps );
#endif
   if(presSpaceHandle == 0)
     IRELEASEPS(HWND_DESKTOP,hps);

   return *this;
}


/*------------------------------------------------------------------------------
| IFont::fontChanged                                                           |
|                                                                              |
| Will cause a subsequent call to makeFontMetrics()                            |
------------------------------------------------------------------------------*/
IFont& IFont :: fontChanged ()
{
// IMODTRACE_DEVELOP("IFont::fontChanged");
#ifdef IC_PM
   delete [] ppd->plClCharWidths;
   ppd->plClCharWidths = 0;
   delete ppd->pfontmetCl;
   ppd->pfontmetCl = 0;
#endif
#ifdef IC_WIN
    delete ppd->ptextmetricCl;
    ppd->ptextmetricCl = 0;
#endif
   return *this;
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IFont::setWidthHeight                                                        |
------------------------------------------------------------------------------*/
IFont& IFont :: setWidthHeight   (unsigned long ulPointSize,
                                  const IPresSpaceHandle& presSpaceHandle)
{
   if (isBitmapOnly())
     return *this;

   long  devRes[2];
         /* Query the device context for the screen and then query */
         /* the resolution of the device for the device context.   */
   IPresSpaceHandle hps;
   if(presSpaceHandle)
      hps = presSpaceHandle;
   else
      hps = WinGetPS(HWND_DESKTOP);
   HDC hDC = GpiQueryDevice(hps);
   DevQueryCaps( hDC, CAPS_HORIZONTAL_FONT_RES,
                 (long)2, devRes);
   if (presSpaceHandle == 0)
     WinReleasePS(hps);

         /* Calculate the size of the character box, based on the  */
         /* point size selected and the resolution of the device.  */
         /* The size parameters are of type FIXED, NOT int.        */
         /*  NOTE: 1 point == 1/72 of an inch.                     */
   ppd->fxClWidth = (FIXED)((MAKEFIXED(ulPointSize, 0) / 72 ) *
                     devRes[0]);
   ppd->fxClHeight = (FIXED)(((MAKEFIXED(ulPointSize, 0)) / 72 ) *
                     devRes[1]);
   return *this;
}
#endif // IC_PM


/*------------------------------------------------------------------------------
| IFont::setFontFromDialog                                                     |
|                                                                              |
| Change the IFont to correspond to the one specified by the fontDlg.          |
------------------------------------------------------------------------------*/
IFont& IFont :: setFontFromDialog (IFontDialog& fontDlg,
                                   const IPresSpaceHandle& presSpaceHandle)
{
   IMODTRACE_DEVELOP("IFont::setFontFromDialog");
   copyPrivate();
#ifdef IC_PM


   delete ppd->pfattrsCl;

   ppd->pfattrsCl = new FATTRS;
   memcpy(ppd->pfattrsCl, &(fontDlg.fData->fAttrs), sizeof(FATTRS));

   if ( ppd->pfattrsCl->fsFontUse & FATTR_FONTUSE_OUTLINE )
   {
      ppd->flClStyles = ppd->flClStyles | IC_VECTOR_FONTS_ONLY;
      ppd->flClStyles = ppd->flClStyles & ~IC_BITMAP_FONTS_ONLY;
   }
   else
   {
      ppd->flClStyles = ppd->flClStyles & ~IC_VECTOR_FONTS_ONLY;
      ppd->flClStyles = ppd->flClStyles | IC_BITMAP_FONTS_ONLY;
   }


   fontChanged();
   setSizeFromHps(fontDlg.pointSize(), presSpaceHandle);
   checkForMatch(presSpaceHandle);
#endif
#ifdef IC_WIN
   delete ppd->plogfontCl;

   ppd->plogfontCl = new LOGFONT;
   memcpy(ppd->plogfontCl, fontDlg.fData->lpLogFont, sizeof(LOGFONT));

   fontChanged();
   makeFontMetrics( presSpaceHandle );
#endif
   return *this;
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IFont::copyWidthsArray                                                       |
|                                                                              |
| Copies the widths array into the specified buffer.                           |
| Used as a performance boost for calculations done for ICanvas layout.        |
------------------------------------------------------------------------------*/
IFont& IFont :: copyWidthsArray(char* buf, unsigned long length)
{
   IASSERTPARM(buf && length <= iCharArrayWidth);
   memcpy(buf, ppd->plClCharWidths, (length * sizeof(long)));

   return *this;
}

/*------------------------------------------------------------------------------
| IFont::fattrs                                                                |
|                                                                              |
| Return a const pointer to the OS/2 FATTRS struct.                            |
------------------------------------------------------------------------------*/
const struct _FATTRS* IFont :: fattrs () const
{
   return ppd->pfattrsCl;
}

/*------------------------------------------------------------------------------
| IFont::fontmetrics                                                           |
|                                                                              |
| Return a const pointer to the OS/2 FONTMETRICS struct.                       |
------------------------------------------------------------------------------*/
const struct _FONTMETRICS* IFont :: fontmetrics () const
{
   return ppd->pfontmetCl;
}
#endif  // IC_PM

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IFont::lpLogFont                                                             |
|                                                                              |
| Return a const pointer to the Windows LOGFONT struct.                        |
------------------------------------------------------------------------------*/
const LOGFONT* IFont :: logfont () const
{
   return ppd->plogfontCl;
}
#endif

/*------------------------------------------------------------------------------
| IFont::copyPrivate                                                           |
|                                                                              |
| Makes a new copy of the fonts private data, decrements the use count for     |
| the old private data and increments the count for the new private data.      |
------------------------------------------------------------------------------*/
IFont& IFont :: copyPrivate( )
{
  if ( ppd->refCount() > 1 )
  {
    IFontPrivateData* newppd = new IFontPrivateData( *ppd );

    // Decrement the ref count for the private data.
    // addRef does not need to be called for the new
    // ifont private data because it is already initialized
    // to 1.
    ppd->removeRef();

    ppd = newppd;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IFont::FaceNameCursor::FaceNameCursor                                        |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFont::FaceNameCursor::FaceNameCursor ( FontType fontType,
                                        const IPresSpaceHandle& presSpaceHandle)
                                      : index(0), count(0), facenames(0)
{
  IPresSpaceHandle hps(presSpaceHandle);
  HWND hwnd =
#ifdef IC_PM
              HWND_DESKTOP;
#endif
#ifdef IC_WIN
              GetDesktopWindow();
#endif
  if (!presSpaceHandle)
    hps = IGETPS( hwnd );

#ifdef IC_PM
  HDC hdc(GpiQueryDevice(hps));
#endif
#ifdef IC_WIN
  HDC hdc(hps);
#endif
  long hres,vres;
  IQUERYDEVICECAPS(hdc, CAPS_HORIZONTAL_FONT_RES, &hres);
  IQUERYDEVICECAPS(hdc, CAPS_VERTICAL_FONT_RES, &vres);

#ifdef IC_PM
  long numRequired(0);
  long fontCount( GpiQueryFonts( hps,
                                 QF_PRIVATE | QF_PUBLIC,
                                 0,
                                 &numRequired,
                                 (long)sizeof(FONTMETRICS),
                                 0 ));
  if ( fontCount )
  {
    PFONTMETRICS fontmetrics = new FONTMETRICS[fontCount];

    count = GpiQueryFonts( hps,
                           QF_PRIVATE | QF_PUBLIC,
                           0,
                           &fontCount,
                           (long)sizeof(FONTMETRICS),
                           fontmetrics );
    count = fontCount;
    int deviceCount;
    int i;
    for (i=0,deviceCount=0;i<count;i++)
    {
      if ( fontType == FaceNameCursor::both &&
         (((fontmetrics[i].sXDeviceRes == hres) &&
           (fontmetrics[i].sYDeviceRes == vres)) || (fontmetrics[i].fsDefn & FM_DEFN_OUTLINE)))
        deviceCount++;
      else if ( fontType == FaceNameCursor::bitmap &&
              ((fontmetrics[i].sXDeviceRes == hres) &&
               (fontmetrics[i].sYDeviceRes == vres)) &&
              !(fontmetrics[i].fsDefn & FM_DEFN_OUTLINE))
        deviceCount++;
      else if (fontType == FaceNameCursor::vector &&
               (fontmetrics[i].fsDefn & FM_DEFN_OUTLINE))
        deviceCount++;
    } /* endfor */
    facenames = new IString[deviceCount];
    for (i=0,deviceCount=0;i<count;i++)
    {
      if ( fontType == FaceNameCursor::both &&
         (((fontmetrics[i].sXDeviceRes == hres) &&
           (fontmetrics[i].sYDeviceRes == vres)) || (fontmetrics[i].fsDefn & FM_DEFN_OUTLINE)))
      {
        facenames[deviceCount] = fontmetrics[i].szFacename;
        deviceCount++;
      }
      else if (fontType == FaceNameCursor::bitmap &&
              ((fontmetrics[i].sXDeviceRes == hres) &&
               (fontmetrics[i].sYDeviceRes == vres)) &&
              !(fontmetrics[i].fsDefn & FM_DEFN_OUTLINE))
      {
        facenames[deviceCount] = fontmetrics[i].szFacename;
        deviceCount++;
      }
      else if (fontType == FaceNameCursor::vector &&
               (fontmetrics[i].fsDefn & FM_DEFN_OUTLINE))
      {
        facenames[deviceCount] = fontmetrics[i].szFacename;
        deviceCount++;
      }
    } /* endfor */
    delete [] fontmetrics;
    count = deviceCount;
  }
#endif
#ifdef IC_WIN
  FONTCOUNT fontCount = { 0, 0, 0, 0, 0, 0 };

  PFONTINFO pfontT;
  PFACEINFO pfaciFaces = NULL, pfaciT;

  /*------------------------------------------------------------------*/
  /* Enumerate the face names available                               */
  /*------------------------------------------------------------------*/
  EnumFontFamilies( hdc, NULL, (FONTENUMPROC)EnumAllFaces,
                    (LPARAM)&pfaciFaces);

  /*------------------------------------------------------------------*/
  /* If a specific type of font is to be searched for, set a flag     */
  /*  otherwise set both to indicate all fonts                        */
  /*------------------------------------------------------------------*/
  if (fontType == FaceNameCursor::bitmap)
    fontCount.dwType = RASTER_FONTTYPE;
  if (fontType == FaceNameCursor::vector)
    fontCount.dwType = TRUETYPE_FONTTYPE;
  if (fontType == FaceNameCursor::both)
    fontCount.dwType = RASTER_FONTTYPE | TRUETYPE_FONTTYPE;

  /*------------------------------------------------------------------*/
  /* Pass the device resolution in to check against                   */
  /*------------------------------------------------------------------*/
  fontCount.horzRes = hres;
  fontCount.vertRes = vres;

  /*------------------------------------------------------------------*/
  /* enumerate the fonts available for each face name                 */
  /*------------------------------------------------------------------*/
  for (pfaciT = pfaciFaces; pfaciT; pfaciT = pfaciT->pfaciNext)
  {
    EnumFontFamilies( hdc, pfaciT->szFaceName, (FONTENUMPROC)EnumAllFonts,
                      (LPARAM)&fontCount);
  }

  /*------------------------------------------------------------------*/
  /* Allocate the array of strings for the face names to list         */
  /*------------------------------------------------------------------*/
  facenames = new IString[fontCount.matchCount];

  /*------------------------------------------------------------------*/
  /* Free the face name list                                          */
  /*------------------------------------------------------------------*/
  while (pfaciFaces)
  {
    pfaciT = pfaciFaces->pfaciNext;
    delete pfaciFaces;
    pfaciFaces = pfaciT;
  }

  /*------------------------------------------------------------------*/
  /* Transfer the font name list over and then free each item         */
  /*------------------------------------------------------------------*/
  int deviceCount = 0;
  while (fontCount.pFontInfo)
  {
    facenames[deviceCount] = fontCount.pFontInfo->szFaceName;
    deviceCount++;
    pfontT = fontCount.pFontInfo->pfiNext;
    delete fontCount.pFontInfo;
    fontCount.pFontInfo = pfontT;
  }
  /*------------------------------------------------------------------*/
  /* Finally, save the total count of items to cursor over            */
  /*------------------------------------------------------------------*/
  count = deviceCount;
#endif

  if (!presSpaceHandle)
    IRELEASEPS( hwnd, hps );
}

/*------------------------------------------------------------------------------
| IFont::FacenameCursor::~FacenameCursor                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFont::FaceNameCursor::~FaceNameCursor ()
{
  if (count)
  {
    delete [] facenames;
  }
}

/*------------------------------------------------------------------------------
| IFont::FacenameCursor::setToFirst                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::FaceNameCursor::setToFirst()
{
  if (count)
  {
    index = 0;
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::FacenameCursor::setToNext                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::FaceNameCursor::setToNext()
{
  bool foundDuplicate;
  int j(index);
  index++;

  while (index < count)
  {
    foundDuplicate = false;
    for (int i=j;i>=0 && !foundDuplicate;i--)
    {
      if ( facenames[i] == facenames[index] )
        foundDuplicate = true;
    } /* endfor */
    if (!foundDuplicate)
      break;
    index++;
  } /* endwhile */

  if (count && index < count)
  {
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::FacenameCursor::setToPrevious                                         |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::FaceNameCursor::setToPrevious()
{

  bool foundDuplicate;
  int j(index);
  index--;

  while (index < count)
  {
    foundDuplicate = false;
    for (int i=j;i<count && !foundDuplicate;i++)
    {
      if ( facenames[i] == facenames[index] )
        foundDuplicate = true;
    } /* endfor */
    if (!foundDuplicate)
      break;
    index--;
  } /* endwhile */

  if (count && index < count)
  {
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::FacenameCursor::setToLast                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::FaceNameCursor::setToLast()
{

  if (count)
  {
    index = count - 1;
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::FaceNameCursor::isValid                                               |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::FaceNameCursor::isValid() const
{
  if (count && index < count)
  {
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::FaceNameCursor::invalidate                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
void IFont::FaceNameCursor::invalidate()
{
  index = count;
}

/*------------------------------------------------------------------------------
| IFont::faceNameAt                                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IString IFont::faceNameAt( const FaceNameCursor& faceNameCursor )
{
  return faceNameCursor.facenames[faceNameCursor.index];
}


/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::PointSizeCursor                                      |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFont::PointSizeCursor::PointSizeCursor ( const char* faceName,
                                        const IPresSpaceHandle& presSpaceHandle)
                                        : index(0), count(0), pointsizes(0)
{
  IPresSpaceHandle hps(presSpaceHandle);
  HWND hwnd =
#ifdef IC_PM
              HWND_DESKTOP;
#endif
#ifdef IC_WIN
              GetDesktopWindow();
#endif
  if (!presSpaceHandle)
    hps = IGETPS( hwnd );

#ifdef IC_PM
  HDC hdc(GpiQueryDevice(hps));
#endif
#ifdef IC_WIN
  HDC hdc(hps);
#endif
  long hres,vres;
  IQUERYDEVICECAPS(hdc, CAPS_HORIZONTAL_FONT_RES, &hres);
  IQUERYDEVICECAPS(hdc, CAPS_VERTICAL_FONT_RES, &vres);

#ifdef IC_PM
  long numRequired(0);
  long fontCount( GpiQueryFonts( hps,
                                 QF_PRIVATE | QF_PUBLIC,
                                 faceName,
                                 &numRequired,
                                 (long)sizeof(FONTMETRICS),
                                 0 ));
  if ( fontCount )
  {
    PFONTMETRICS fontmetrics = new FONTMETRICS[fontCount];

    count = GpiQueryFonts( hps,
                           QF_PRIVATE | QF_PUBLIC,
                           faceName,
                           &fontCount,
                           (long)sizeof(FONTMETRICS),
                           fontmetrics );
    count = fontCount;
    int deviceCount;
    int i;
    for (i=0,deviceCount=0;i<count;i++)
    {
      if ((fontmetrics[i].sXDeviceRes == hres &&
           fontmetrics[i].sYDeviceRes == vres ) ||
          (fontmetrics[i].fsDefn & FM_DEFN_OUTLINE))
        deviceCount++;
    } /* endfor */
    pointsizes = new long[deviceCount];
    for (i=0,deviceCount=0;i<count;i++)
    {
      if ((fontmetrics[i].sXDeviceRes == hres &&
           fontmetrics[i].sYDeviceRes == vres ) ||
          (fontmetrics[i].fsDefn & FM_DEFN_OUTLINE))
      {
        pointsizes[deviceCount] = fontmetrics[i].sNominalPointSize/10;
        deviceCount++;
      }
    } /* endfor */
    delete [] fontmetrics;
    count = deviceCount;
  }
#endif
#ifdef IC_WIN
  FONTCOUNT fontCount = { 0, 0, 0, 0, 0, 1 };
  PFONTINFO pfontT;

  /*------------------------------------------------------------------*/
  /* Compile a list only for bitmap fonts as vector can be any size   */
  /*------------------------------------------------------------------*/
  fontCount.dwType = RASTER_FONTTYPE;

  /*------------------------------------------------------------------*/
  /* Pass the device resolution in to check against                   */
  /*------------------------------------------------------------------*/
  fontCount.horzRes = hres;
  fontCount.vertRes = vres;

  /*------------------------------------------------------------------*/
  /* Enumerate the sizes available for the face name passed in        */
  /*------------------------------------------------------------------*/
  EnumFontFamilies( hdc, faceName, (FONTENUMPROC)EnumAllFonts,
                    (LPARAM)&fontCount);

  /*------------------------------------------------------------------*/
  /* Allocate the array of longs for the pointsizes to list           */
  /*------------------------------------------------------------------*/
  pointsizes = new long[fontCount.matchCount];

  /*------------------------------------------------------------------*/
  /* Transfer the pointsize list over and then free each item         */
  /*------------------------------------------------------------------*/
  int deviceCount = 0;
  while (fontCount.pFontInfo)
  {
    /*----------------------------------------------------------------*/
    /* Add in reverse order to since linked list is ordered last to   */
    /*  first since it was built that way                             */
    /*----------------------------------------------------------------*/
    deviceCount++;
    pointsizes[fontCount.matchCount-deviceCount] =
                                        fontCount.pFontInfo->iPointSize;
    pfontT = fontCount.pFontInfo->pfiNext;
    delete fontCount.pFontInfo;
    fontCount.pFontInfo = pfontT;
  }
  /*------------------------------------------------------------------*/
  /* Finally, save the total count of items to cursor over            */
  /*------------------------------------------------------------------*/
  count = deviceCount;
#endif

  if (!presSpaceHandle)
    IRELEASEPS( hwnd, hps );
}


/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::~PointSizeCursor                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFont::PointSizeCursor::~PointSizeCursor ()
{
  if (count)
  {
    delete [] pointsizes;
  }
}

/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::setToFirst                                           |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::PointSizeCursor::setToFirst()
{
  if (count)
  {
    index = 0;
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::setToNext                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::PointSizeCursor::setToNext()
{
  index++;

  if (count && index < count)
  {
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::setToPrevious                                        |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::PointSizeCursor::setToPrevious()
{
  index--;
  if (count && index < count)
  {
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::setToLast                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::PointSizeCursor::setToLast()
{

  if (count)
  {
    index = count - 1;
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::isValid                                              |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IFont::PointSizeCursor::isValid() const
{
  if (count && index < count)
  {
    return true;
  }
  return false;
}

/*------------------------------------------------------------------------------
| IFont::PointSizeCursor::invalidate                                           |
|                                                                              |
| Invalidate the cursor.                                                       |
------------------------------------------------------------------------------*/
void IFont::PointSizeCursor::invalidate()
{
  index = count;
}

/*------------------------------------------------------------------------------
| IFont::pointsizeAt                                                           |
|                                                                              |
| Return the point size for the argument cursor.                               |
------------------------------------------------------------------------------*/
long IFont::pointSizeAt( const PointSizeCursor& pointSizeCursor )
{
  return pointSizeCursor.pointsizes[pointSizeCursor.index];
}

#ifdef IC_WIN
/**********************************************************************/
/* Procedure: EnumAllFonts - Callback function for EnumFonts.         */
/*            Allocates a FONTINFO structure and saves all of the     */
/*            information passed into this function.                  */
/**********************************************************************/
int CALLBACK EnumAllFonts( LPENUMLOGFONT lpelf, LPNEWTEXTMETRIC lpntm,
                           int nFontType, LPARAM lpData )
{
  PFONTINFO pfiNew, pfiNav;
  BOOL found = FALSE;
  int iPointSize = 0;
  FONTCOUNT* pFontCount = (FONTCOUNT*)lpData;
  LPTEXTMETRIC lptm = (LPTEXTMETRIC)lpntm;

  /*------------------------------------------------------------------*/
  /* Add the font to our list if:                                     */
  /*   a) the request was for bitmap fonts and we have a bitmap font  */
  /*      and the device resolution matches that of our device        */
  /*   b) the request was for a vector font and this is not a bitmap  */
  /*      font (therefore it is either a vector or TrueType font)     */
  /*   c) the request was for both (a+b)                              */
  /*------------------------------------------------------------------*/
  if (((pFontCount->dwType & RASTER_FONTTYPE) &&
       (nFontType & RASTER_FONTTYPE) &&
       (pFontCount->horzRes == lptm->tmDigitizedAspectX) &&
       (pFontCount->vertRes == lptm->tmDigitizedAspectY)) ||
      ((pFontCount->dwType & TRUETYPE_FONTTYPE) &&
       !(nFontType & RASTER_FONTTYPE)))
  {
    /*----------------------------------------------------------------*/
    /* Check for duplicate listing.  If point size, then check sizes  */
    /*----------------------------------------------------------------*/
    if (pFontCount->bPointSizes)
    {
      /*--------------------------------------------------------------*/
      /* Calculate the point size for the font enumerated             */
      /*--------------------------------------------------------------*/
      iPointSize = ((1440 / (int)pFontCount->vertRes) *
               (int)(lptm->tmHeight - lptm->tmInternalLeading) + 10)/20;

      /*--------------------------------------------------------------*/
      /* Check if the list currently contains the pointsize found     */
      /*--------------------------------------------------------------*/
      pfiNav = pFontCount->pFontInfo;
      while (!found && (pfiNav))
      {
        /*------------------------------------------------------------*/
        /* If so, set flag to not add again, otherwise keep searching */
        /*------------------------------------------------------------*/
        if (pfiNav->iPointSize == iPointSize)
          found = TRUE;
        else
          pfiNav = pfiNav->pfiNext;
      }
    }
    /*----------------------------------------------------------------*/
    /* Otherwise check facenames for duplicates                       */
    /*----------------------------------------------------------------*/
    else
    {
      /*--------------------------------------------------------------*/
      /* Check if the list currently contains the facename found      */
      /*--------------------------------------------------------------*/
      pfiNav = pFontCount->pFontInfo;
      while (!found && (pfiNav))
      {
        /*------------------------------------------------------------*/
        /* If so, set flag to not add again, otherwise keep searching */
        /*------------------------------------------------------------*/
        if (!(lstrcmp(pfiNav->szFaceName,lpelf->elfLogFont.lfFaceName)))
          found = TRUE;
        else
          pfiNav = pfiNav->pfiNext;
      }
    }

    /*----------------------------------------------------------------*/
    /* Only add to list if not yet in the linked list                 */
    /*----------------------------------------------------------------*/
    if (!found)
    {
      /*--------------------------------------------------------------*/
      /* Allocate a FONTINFO from the heap.  If it fails, punt.       */
      /*--------------------------------------------------------------*/
      if (!(pfiNew = (PFONTINFO) new FONTINFO ))
        return (FALSE);

      /*--------------------------------------------------------------*/
      /* Initialize the FONTINFO structure that was allocated with    */
      /*  the facename passed, the point size calculated from the     */
      /*  font information and set pointer to next block to whole list*/
      /*--------------------------------------------------------------*/
      lstrcpy( pfiNew->szFaceName, lpelf->elfLogFont.lfFaceName );
      pfiNew->iPointSize = ((1440 / (int)pFontCount->vertRes) *
               (int)(lptm->tmHeight - lptm->tmInternalLeading) + 10)/20;
      pfiNew->pfiNext = pFontCount->pFontInfo;

      /*--------------------------------------------------------------*/
      /* Change the list head pointer to the newly allocated link     */
      /*--------------------------------------------------------------*/
      pFontCount->pFontInfo = pfiNew;
      pFontCount->matchCount++;
    }
  }
  return (TRUE);
}


/**********************************************************************/
/* Procedure: EnumAllFaces - Callback function for EnumFonts.         */
/*            Allocates a FONTINFO structure and saves all of the     */
/*            information passed into this function.                  */
/**********************************************************************/
int CALLBACK EnumAllFaces( LPENUMLOGFONT lpelf, LPNEWTEXTMETRIC lpntm,
                           int nFontType, LPARAM lpData )
{
  PFACEINFO pfaciNew;

  /*------------------------------------------------------------------*/
  /* Allocate a FACEINFO from the heap.  If it fails, punt.           */
  /*------------------------------------------------------------------*/
  if (!(pfaciNew = (PFACEINFO) new FACEINFO ))
    return (FALSE);

  /*------------------------------------------------------------------*/
  /* Initialize the FACEINFO                                          */
  /*------------------------------------------------------------------*/
  lstrcpy( pfaciNew->szFaceName, lpelf->elfLogFont.lfFaceName );
  pfaciNew->pfaciNext = *( (PFACEINFO FAR *)lpData );

  /*------------------------------------------------------------------*/
  /* Update the list head pointer                                     */
  /*------------------------------------------------------------------*/
  *((PFACEINFO FAR *) lpData) = pfaciNew;
  return (TRUE);
}
#endif // IC_WIN

#endif // IC_PMWIN

#ifdef IC_MOTIF
static void
ifontSetRCFontList( IWindow* piwParent, Widget rc, XmFontList fontList );

static void
ifontSetChildFontList( IWindow* pwin, Widget root, XmFontList fontList );

static void
ifontSetWidgetFontList(IWindow* piwParent,Widget current,XmFontList fontList);

/*------------------------------------------------------------------------------
| IFont::fontSet                                                               |
|                                                                              |
| Provide access to the private X-Windows _XFontSet structure.                 |
------------------------------------------------------------------------------*/
_XFontSet* IFont :: fontSet()
{
   return ppd->xFontSet();
}

_XmFontListRec* IFont :: mFontList()
{
   return ppd->xmFontList();
}

IFont::IFont(_XmFontListRec* fontList)
        : ppd(0)
{
    ppd = new IFontPrivateData(fontList);
}

/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
| Construct a font for the font being used in an IWindow                       |
------------------------------------------------------------------------------*/
IFont :: IFont (const IWindow* pwindow)
         : ppd(0)
{
    IMODTRACE_ALL("IFont-IWindow* ctor");

    if (pwindow == 0) {
        if (! gFontPrivateData) {
            // Create the default font data

            IPrimalLock lockinit;

            XmFontList fontList = 0;

            if (! gFontPrivateData) {
                Widget wh = IWindow::desktopWindow()->handle();
                // For CDE, the name is XmNfontList
                XtVaGetValues((Widget) wh, XmNfontList, &fontList, NULL);
                if (! fontList) {
                    // For fvwm, it's XmNtextFontList
                    XtVaGetValues((Widget) wh, XmNtextFontList, &fontList, NULL);
                }
                if (fontList)
                    gFontPrivateData = new IFontPrivateData(fontList);
                else
                    gFontPrivateData = new IFontPrivateData();

                adoptStaticObject(gFontPrivateData);
            }
        }
        ppd = new IFontPrivateData(*gFontPrivateData);
    } else {
        ppd = new IFontPrivateData(*(pwindow->font().ppd));
    }
}



/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
| Construct font based an input parameters.                                    |
------------------------------------------------------------------------------*/
IFont :: IFont         (const char*   pszFaceName,
                        unsigned long ulPointSize,
                        bool       useFixedFont,
                        bool       useVectorFont,
                        const IPresSpaceHandle& presSpaceHandle)
         : ppd(new IFontPrivateData(pszFaceName,
                                    ulPointSize,
                                    useFixedFont,
                                    useVectorFont,
                                    presSpaceHandle) )
{
   IMODTRACE_ALL("IFont-ctor with specified font");
}

/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
| Construct font to one being used in Presentation Space                       |
------------------------------------------------------------------------------*/
IFont :: IFont(const IPresSpaceHandle& presSpaceHandle)
         : ppd(new IFontPrivateData( presSpaceHandle) )
{
   IMODTRACE_ALL("IFont-IPresSpaceHandle ctor");
}


/*------------------------------------------------------------------------------
| IFont::IFont                                                                 |
| Copy constructor                                                             |
------------------------------------------------------------------------------*/
IFont :: IFont (const IFont& fntmCopy)
         : ppd(new IFontPrivateData(*(fntmCopy.ppd)))
{
  IMODTRACE_ALL("IFont-copy ctor");
}

/*------------------------------------------------------------------------------
| IFont::operator=                                                             |
------------------------------------------------------------------------------*/
IFont& IFont::operator=(const IFont& rhs)
{
  if (this == &rhs) return *this;

  delete ppd;
  ppd = new IFontPrivateData(*(rhs.ppd));
  return *this;
}

/*------------------------------------------------------------------------------
| IFont::~IFont                                                                |
------------------------------------------------------------------------------*/
IFont :: ~IFont ()
{
   IMODTRACE_ALL("IFont-dtor");
   delete ppd;
}


/*------------------------------------------------------------------------------
| IFont::setFontFromDialog                                                     |
| Change the IFont to correspond to the one specified by the fontDlg.          |
------------------------------------------------------------------------------*/
IFont& IFont :: setFontFromDialog (IFontPrivateData * fontData )
{
   IMODTRACE_DEVELOP("IFont::setFontFromDialog");

   // This function is basically an assignment operator, since we replace
   // the guts of the object with a copy provided by the dialog.
   *ppd = *fontData;

   return *this;
}

/*------------------------------------------------------------------------------
| ifontSetRCFontList                                                           |
| Sets the font list for row column children.                                  |
------------------------------------------------------------------------------*/
void ifontSetRCFontList( IWindow* piwParent, Widget rc, XmFontList fontList )
{
   unsigned   char  rcType = XmWORK_AREA;
   XtVaGetValues( rc,
                  XmNrowColumnType, &rcType,
                  NULL);

   Cardinal   numChildren = 0;
   WidgetList children = 0;
   XtVaGetValues(   rc,
                    XmNnumChildren, &numChildren,
                    XmNchildren, &children,
                    NULL );
   IWindow* piwChild=0;
   int i;
   for ( i = 0; i < numChildren; i++)
      {
      // d8175 - check for child being destroyed.
      if ( !children[i]->core.being_destroyed )
       {
        // d7294
        if ( piwParent )
          piwChild = IWindow::windowWithHandle( children[i] );
        // Menu related rowColumns have to be handled in a special
        // way to find all the logical children
        if ( (rcType == XmMENU_BAR)   ||
             (rcType == XmMENU_POPUP) ||
             (rcType == XmMENU_PULLDOWN) )
           {
           // Set the font for the individual buttons or labels
           ifontSetWidgetFontList( piwChild, children[i], fontList );

           if ( XmIsCascadeButtonGadget( children[i] ) ||
                XmIsCascadeButton( children[i] ) )          // 27912
              {
              Widget submenu = 0;
              XtVaGetValues( children[i],
                             XmNsubMenuId, &submenu,
                             NULL);
              if (  (submenu) && XmIsRowColumn(submenu) )
                 {
                 // d7294
                 IWindow* piwGrandChild=0;
                 // If menu shell is found start setting its resources.
                 // Otherwise start at the row column
                 Widget  menushell = XtParent(submenu);
                 if (menushell)
                  {
                    if ( piwChild )
                      piwGrandChild = IWindow::windowWithHandle( menushell );
                    ifontSetWidgetFontList( piwGrandChild, menushell, fontList );
                  }
                 else
                  {
                    if ( piwChild )
                      piwGrandChild = IWindow::windowWithHandle( submenu );
                    ifontSetWidgetFontList( piwGrandChild, submenu, fontList );
                  }
                 // d7294
                 if ( piwGrandChild && ( piwGrandChild != piwChild ) )
                   piwGrandChild->setLayoutDistorted( IWindow::fontChanged
                                                    | IWindow::minimumSizeChanged,
                                                      0);

                 }  // is rowColumn
              }  // if cascade button
           }  // if menu RowColumn
        else
           {
           // Non menu rowcolumn child
           ifontSetWidgetFontList( piwChild, children[i], fontList );
           }
        // d7294 - set layout distorted on a child IWindow.
        if ( piwChild && ( piwChild != piwParent ) )
          piwChild->setLayoutDistorted( IWindow::fontChanged
                                      | IWindow::minimumSizeChanged, 0);
       } // !being_destroyed
      }  // for
}


/*------------------------------------------------------------------------------
| ifontSetChildFontList                                                        |
| Sets the font resources of the children of root.  Motif apparently           |
| only uses the labelFontList, buttonFontList, or textFontList of              |
| parent widgets at creation time.  To get behavior we need we set both        |
| those resources and the font resources of the children.                      |
------------------------------------------------------------------------------*/
void ifontSetChildFontList( IWindow* piwParent,Widget root,XmFontList fontList )
{
   Cardinal   numChildren = 0;
   WidgetList children = 0;
   XtVaGetValues(   root,
                    XmNnumChildren, &numChildren,
                    XmNchildren, &children,
                    NULL );
   IWindow* piwChild=0;
   IFont fontParent;
   if (piwParent)
      fontParent = piwParent->font();
   int i;
   for ( i = 0; i < numChildren; i++)
      {
      // d8175 - check for child being destroyed.
      if ( !children[i]->core.being_destroyed )
       {
        if ( piwParent )
          piwChild = IWindow::windowWithHandle( children[i] );

        // If the child is an IWindow, call setFont.  This will prevent
        // us from overlaying the font of a window with an explicitly set
        // font.
        if (piwChild && (piwChild != piwParent))
           piwChild->setFont( fontParent );
        else
           ifontSetWidgetFontList( piwChild, children[i], fontList );
        // d7294 - set layout distorted on a child IWindow.
        if ( piwChild && ( piwChild != piwParent ) )
          piwChild->setLayoutDistorted( IWindow::fontChanged
                                      | IWindow::minimumSizeChanged, 0);
       }
      }
}

/*------------------------------------------------------------------------------
| ifontSetWidgetFontList                                                       |
| Sets the font resources of this widget.  If a composite widget,              |
| ifontSetChildFontList is called.                                             |
------------------------------------------------------------------------------*/
void ifontSetWidgetFontList( IWindow* pwin,Widget current,XmFontList fontList )
{
   if ( XtIsComposite( current ) )
      {
      if ( XmIsBulletinBoard( current ) )
         {
         XtVaSetValues(current,
                       XmNtextFontList,   fontList,
                       XmNlabelFontList,  fontList,
                       XmNbuttonFontList, fontList,
                       NULL);

         }  // if BulletinBoard
      else if ( XtIsVendorShell( current ) )
         {
         XtVaSetValues(current,
                       XmNdefaultFontList, 0,   // obsolete in 1.2
                       XmNtextFontList,   fontList,
                       XmNlabelFontList,  fontList,
                       XmNbuttonFontList, fontList,
                       NULL);

         }  // if VendorShell
      else if ( XmIsMenuShell( current ) )
         {
         XtVaSetValues(current,
                       XmNdefaultFontList, 0,   // obsolete in 1.2
                       XmNlabelFontList,  fontList,
                       XmNbuttonFontList, fontList,
                       NULL);
         }  // menushell
      else if ( XmIsContainer(current) )
         {
         // set the fontList for the Container
         XtVaSetValues(current,
                       XmNfontList, fontList,
                       NULL);

         // set the fontList for the iconGadgets
         ifontSetChildFontList( 0, current, fontList );

         // Now call private container method updateFont to render
         // the headings and text in the container, and to update the title font
         ICnrControlData::updateFont( IWindowHandle(current), fontList );

         }  // container
      else
         {
         XtVaSetValues(current,
                       XmNfontList, fontList,
                       NULL);
         }

      // Set font resources for non-container children. (Container already
      // dealt with.)  Rowcolumn is special case because of menus.
      if ( XmIsRowColumn(current ) )
         {
         ifontSetRCFontList( pwin, current, fontList );
         }
      else if (!XtIsSubclass(current,xmContainerWidgetClass))
         {
         propagationInProgress++;
         ifontSetChildFontList( pwin, current, fontList );
         propagationInProgress--;
         }
      }  // is composite
   else
      {
         XtVaSetValues(current,
                       XmNfontList, fontList,
                       NULL);
      }
}
#endif //IC_MOTIF


/*------------------------------------------------------------------------------
| IFont::setName                                                               |
| Set the faceName of the font.                                                |
------------------------------------------------------------------------------*/
IFont& IFont :: setName (const char* pszFaceName,
                         const IPresSpaceHandle& presSpaceHandle)
{
#ifdef IC_PMWIN
   copyPrivate();
#ifdef IC_PM
   strcpy((ppd->pfattrsCl)->szFacename, pszFaceName);
#else
   strcpy((ppd->plogfontCl)->lfFaceName, pszFaceName);
#endif
   fontChanged();
#ifdef IC_PM
   checkForMatch(presSpaceHandle);
#else
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
   if (ppd->faceName() != pszFaceName) {
     ppd->setFaceName( pszFaceName );
     ppd->checkForMatch(presSpaceHandle);
   }
#endif //IC_MOTIF
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::setPointSize                                                          |
------------------------------------------------------------------------------*/
IFont& IFont ::  setPointSize (unsigned long ulSize,
                               const IPresSpaceHandle& presSpaceHandle)
{
#ifdef IC_PMWIN
   copyPrivate();
#ifdef IC_PM
   ppd->fxClPointSize = MAKEFIXED(ulSize, 0);
   if (isBitmapOnly()) {
      fontChanged();
      checkForMatch(presSpaceHandle);
   } else {
      // As a performance boost for a vector font just need
      // to change the size and not try querying all fonts
      // for a closest match
      setCharHeight(ulSize, presSpaceHandle);
      setCharWidth(ulSize, presSpaceHandle);
   } /* endif */
#else
   (ppd->plogfontCl)->lfHeight = CalcFontHeight( presSpaceHandle, ulSize );
   (ppd->plogfontCl)->lfWidth = 0;
   fontChanged();
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
   if (ppd->pointSize() == ulSize)
      return *this;
   else
      {
      ppd->setPointSize (ulSize);
      ppd->checkForMatchSimple(presSpaceHandle);
      }
#endif //IC_MOTIF
   return *this;
}

/*------------------------------------------------------------------------------
| IFont::setUnderscore                                                         |
------------------------------------------------------------------------------*/
IFont& IFont :: setUnderscore(bool underscore)
{
    ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
    copyPrivate();
    if (underscore)
#ifdef IC_PM
       ppd->pfattrsCl->fsSelection |= FATTR_SEL_UNDERSCORE;
    else
       ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_UNDERSCORE;
#endif
#ifdef IC_WIN
       ppd->plogfontCl->lfUnderline = 0xFF;
    else
       ppd->plogfontCl->lfUnderline = 0;
    fontChanged();
    makeFontMetrics();
#endif
#endif //IC_PMWIN
    return *this;
}


/*------------------------------------------------------------------------------
| IFont::setStrikeout                                                          |
------------------------------------------------------------------------------*/
IFont& IFont :: setStrikeout(bool strikeout)
{
    ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
    copyPrivate();
    if (strikeout)
#ifdef IC_PM
       ppd->pfattrsCl->fsSelection |= FATTR_SEL_STRIKEOUT;
    else
       ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_STRIKEOUT;
#endif
#ifdef IC_WIN
       ppd->plogfontCl->lfStrikeOut = 0xFF;
    else
       ppd->plogfontCl->lfStrikeOut = 0;
    fontChanged();
    makeFontMetrics();
#endif
#endif //IC_PMWIN
    return *this;
}


/*------------------------------------------------------------------------------
| IFont::setOutline                                                            |
------------------------------------------------------------------------------*/
IFont& IFont :: setOutline(bool outline)
{
    ITRACE_WIN_NOP();
    ITRACE_MOTIF_NOP();
#ifdef IC_PM
    copyPrivate();
    if (outline)
       ppd->pfattrsCl->fsSelection |= FATTR_SEL_OUTLINE;
    else
       ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_OUTLINE;
#endif
    return *this;
}

/*------------------------------------------------------------------------------
| IFont::setBold                                                               |
------------------------------------------------------------------------------*/
IFont& IFont :: setBold(bool bold)
{
#ifdef IC_PMWIN
    copyPrivate();
    if (bold)
#ifdef IC_PM
       ppd->pfattrsCl->fsSelection |= FATTR_SEL_BOLD;
    else
       ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_BOLD;
#endif
#ifdef IC_WIN
       ppd->plogfontCl->lfWeight = FW_BOLD;
    else
       ppd->plogfontCl->lfWeight = FW_REGULAR;
    fontChanged();
    makeFontMetrics();
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
    if ( ( bold  && ppd->isBold()  )    ||
         ( !bold && !ppd->isBold() ) )
      return *this;

    ppd->setBold(bold);
    ppd->checkForMatchSimple();
#endif //IC_MOTIF
    return *this;
}

/*------------------------------------------------------------------------------
| IFont::setItalic                                                             |
------------------------------------------------------------------------------*/
IFont& IFont :: setItalic(bool italics)
{
#ifdef IC_PMWIN
    copyPrivate();
    if (italics)
#ifdef IC_PM
       ppd->pfattrsCl->fsSelection |= FATTR_SEL_ITALIC;
    else
       ppd->pfattrsCl->fsSelection &= ~FATTR_SEL_ITALIC;
#endif
#ifdef IC_WIN
       ppd->plogfontCl->lfItalic = 0xFF;
    else
       ppd->plogfontCl->lfItalic = 0;
    fontChanged();
    makeFontMetrics();
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
    ppd->setItalic( italics );
    ppd->checkForMatchSimple();
#endif //IC_MOTIF
    return *this;
}



/*------------------------------------------------------------------------------
| IFont::setAllEmphasis                                                        |
------------------------------------------------------------------------------*/
IFont& IFont :: setAllEmphasis(bool setOn)
{

#ifdef IC_PMWIN
    copyPrivate();
    if (setOn)
#ifdef IC_PM
       ppd->pfattrsCl->fsSelection = (FATTR_SEL_BOLD |
                                      FATTR_SEL_ITALIC |
                                      FATTR_SEL_UNDERSCORE |
                                      FATTR_SEL_STRIKEOUT |
                                      FATTR_SEL_OUTLINE);
#endif
#ifdef IC_WIN
    {
      ppd->plogfontCl->lfWeight = FW_BOLD;
      ppd->plogfontCl->lfItalic = 0xFF;
      ppd->plogfontCl->lfUnderline = 0xFF;
      ppd->plogfontCl->lfStrikeOut = 0xFF;
    }
#endif
    else
#ifdef IC_PM
       ppd->pfattrsCl->fsSelection = 0;
#endif
#ifdef IC_WIN
    {
      ppd->plogfontCl->lfWeight = FW_REGULAR;
      ppd->plogfontCl->lfItalic = 0;
      ppd->plogfontCl->lfUnderline = 0;
      ppd->plogfontCl->lfStrikeOut = 0;
    }
    fontChanged();
    makeFontMetrics();
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
    ppd->setBold( setOn );
    ppd->setItalic( setOn );
    ppd->checkForMatchSimple();
#endif //IC_MOTIF
    return *this;
}

/*------------------------------------------------------------------------------
| IFont::setDirection                                                          |
|                                                                              |
| This method only affects text drawn using beginUsingFont.                    |
------------------------------------------------------------------------------*/
IFont& IFont :: setDirection( Direction lFontDir )
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   copyPrivate();
   switch (lFontDir) {
      case leftToRight:
         ppd->lClFontDir = CHDIRN_LEFTRIGHT;
         break;
      case rightToLeft:
         ppd->lClFontDir = CHDIRN_RIGHTLEFT;
         break;
      case topToBottom:
         ppd->lClFontDir = CHDIRN_TOPBOTTOM;
         break;
      case bottomToTop:
         ppd->lClFontDir = CHDIRN_BOTTOMTOP;
         break;
      case defaultDir:
         ppd->lClFontDir = CHDIRN_DEFAULT;
         break;
   } /* endswitch */
#ifdef IC_WIN
   (ppd->plogfontCl)->lfEscapement = ppd->lClFontDir;
    fontChanged();
    makeFontMetrics();
#endif
#endif //IC_PMWIN
   return *this;
}

/*------------------------------------------------------------------------------
| IFont::charWidth                                                             |
------------------------------------------------------------------------------*/
unsigned long IFont :: charWidth (char c) const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeCharWidths();
   return ppd->plClCharWidths[c];
#endif
#ifdef IC_WIN
   HDC hdc = IGETPS(HWND_DESKTOP);
   int buffer = 0;

   HFONT tempFont = (HFONT) SelectObject( hdc, *(ppd->fpFontHandle) );
   GetCharWidth( hdc, c, c, &buffer );
   SelectObject( hdc, tempFont );

   IRELEASEPS(HWND_DESKTOP,hdc);
   return buffer;
#endif
#ifdef IC_MOTIF
  char str[2];
  str[0] = c;
  str[1] = '\0';
  return ppd->logicalSize(str).width();
#endif
}

/*------------------------------------------------------------------------------
| IFont::textWidth                                                             |
------------------------------------------------------------------------------*/
unsigned long  IFont :: textWidth (const char* pszText)  const
{
#ifdef IC_PM
   unsigned long ulWidth = 0;
   if (pszText)
  {
    IFont* const localThis = (IFont* const)this;
    IStaticText sttxtText(0,IWindow::desktopWindow()
                           ,IWindow::desktopWindow()
                           ,IRectangle(0,0,0,0));
    sttxtText.setFont(*this);
    IPresSpaceHandle hps;
    hps = sttxtText.presSpace();

	  POINTL aptl[4];
	  GpiQueryTextBox (hps, strlen (pszText), (char*)pszText, 4, &aptl[0]);
	  ulWidth = aptl[TXTBOX_TOPRIGHT].x - aptl[TXTBOX_BOTTOMLEFT].x;
    WinReleasePS(hps);
  } /* endif pszText*/
  return ulWidth;
#endif

#ifdef IC_WIN
   HDC hdc = IGETPS(HWND_DESKTOP);
   IString tempString(pszText);
   SIZE size;

   HFONT tempFont = (HFONT) SelectObject( hdc, *(ppd->fpFontHandle) );
   GetTextExtentPoint( hdc, tempString, tempString.length(), &size );
   SelectObject( hdc, tempFont );

   IRELEASEPS(HWND_DESKTOP,hdc);
   return size.cx;
#endif

#ifdef IC_MOTIF
 return ppd->logicalSize(pszText).width();
#endif
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| IFont::textWidth                                                             |
------------------------------------------------------------------------------*/
unsigned long  IFont :: textWidth (const char*    pszText,
                                   IBidiSettings& settings)  const
{
   IString tmpText(pszText);
#ifdef IC_PM
   LAYOUT_OBJECT layout = bidiCreateLayoutObject();
   unsigned long tilde = 0;
   bidiPrepareForDisplay(layout,tmpText,settings,tilde);
   bidiDeleteLayoutObject(layout);
#endif
   return textWidth(tmpText);
}
#endif

/*------------------------------------------------------------------------------
| IFont::minTextWidth                                                          |
------------------------------------------------------------------------------*/
unsigned long IFont :: minTextWidth     (const char* pszLine) const
{
   if (pszLine) {
      char* Temp = new char[strlen(pszLine)+1];
      strcpy(Temp, pszLine);
      char* Start = Temp;
      unsigned long ulWidth = 0;
      while (*Start) {
         while (*Start && isspace(*Start)) {
            Start++;
         } /* endwhile */
         char* End = Start;
         while (*End && !isspace(*End)) {
            End++;
         } /* endwhile */
         char SaveCh = *End;
         *End = 0;
         unsigned long ulChkWidth = textWidth(Start);
         if (ulChkWidth > ulWidth) {
            ulWidth = ulChkWidth;
         } /* endif */
         *End = SaveCh;
         Start = End;
      } /* endwhile */
      delete [] Temp;
      return ulWidth;
   } else {
      return 0;
   } /* endif */
}

/*------------------------------------------------------------------------------
| IFont::textLines                                                             |
------------------------------------------------------------------------------*/
unsigned long IFont :: textLines    (const char* pszLine,
                                     unsigned long ulLineWidth) const
{
// Investigate merging these implementations of textLines...Nothing platform
// dependent, just different algorithms

#ifdef IC_PMWIN
   unsigned long ulLines = 0;
   if (ulLineWidth  &&
       this->minTextWidth( pszLine ) <= ulLineWidth)
   {
      ulLines = 1;
      unsigned long ulWidth = 0;
      const char* LastSpace = 0;
      const char* Ptr = pszLine;
      while (*Ptr) {
         if (ulWidth + charWidth( *Ptr ) > ulLineWidth)
         {                   // More text than will fit.
            if (isspace( *Ptr ))
            {                // Space between lines.
               LastSpace = Ptr;
            }
            else if (LastSpace)
            {                // Back up to start of word.
               Ptr = LastSpace;
            }
            Ptr++;
            ulLines++;       // Need another line.
            ulWidth = 0;
         }
         else if (*Ptr == '\n')
         {                   // New line character (fits on this line).
            Ptr++;
            if (*Ptr)
            {                // More characters for another line.
               ulLines++;    // Need another line.
               ulWidth = 0;
            }
         }
         else
         {                   // Adding to current line.
            ulWidth += charWidth(*Ptr);
            if (isspace(*Ptr)) {
               LastSpace = Ptr;
            }
            Ptr++;
         }
      } /* endwhile */
   } /* endif */
   return ulLines;
#endif //IC_PMWIN
#ifdef IC_MOTIF
   unsigned long numBytes = strlen(pszLine);
   unsigned long ulLines = ulLineWidth ?1:0; // Even a null string takes one line.
   if ( ulLineWidth && numBytes )
    {
     unsigned long totalWidth = textWidth( pszLine );
     if ( totalWidth > ulLineWidth )  // We need more than one line.
      {
       //
       // This algorithm tries to minimize the number of calls to textWidth,
       // since they are relatively expensive.  At the same time, only textWidth
       // can be used to reliably calculate the length of a multibyte string,
       // from start to finish, including spaces.  Newline characters, and
       // all space characters recognized by isspace can be recognized on
       // single byte basis, since they all fall in the range x'00' to x'3f'.
       //
       // Here is the algorithm:
       //  - Guess at the average number of characters per line, based on
       //    the total text width and the number of bytes in the string.
       //  - For each line:
       //    - Calculate an estimated byte position (pos) for the end of the line,
       //      based on the above guess.
       //    - If pos and pos-1 are not spaces, move pos to the next space.
       //    - While the length from the start of the line (sol) to pos-1 is less
       //      than ulLineWidth, move pos to one byte beyond the end of the
       //      current word (if in a word) or to first byte of next word.
       //    - While the length from sol to pos-1 is greater than ulLineWidth,
       //      move pos back one space or one word.
       //  - While doing the above, watch out for \n and \0;
       //
       char* tempStr = new char[numBytes+1];
       strcpy(tempStr, pszLine);

       unsigned long avgNumBytesPerLine = 0;
       if ( totalWidth > 0 )
         avgNumBytesPerLine = (numBytes * ulLineWidth ) / totalWidth;

       unsigned long spaceWidth = textWidth( " " );

       unsigned long sol = 0;
       unsigned long pos = 0;
       unsigned long nextSep = sol + strcspn( &tempStr[sol], "\n" );
       //
       // For each line:
       //
       while ( tempStr[sol] != '\0' && ulLines > 0 )
        {
         //
         // Set pos to estimated end of line.  pos is always one byte beyond the
         // text we are measuring.
         //
         pos = pos + avgNumBytesPerLine;
         if ( pos > nextSep ) pos = nextSep;
         //
         // Move pos to the next space.  (Do not break a word in the middle.)
         //
         if ( pos == sol || !isspace( tempStr[pos-1] ) )
          {
           while ( pos < nextSep && !isspace( tempStr[pos] ) ) ++pos;
          }
         //
         // While the length from sol to pos-1 is < ulLineWidth, move pos
         // to either the first byte of the next word, or
         // to one byte beyond the end of the current word.
         //
         unsigned long sol2posWidth = 0;
         if ( pos > sol ) // There is something to measure.
          {
           char* tempEos = &tempStr[pos];
           char saveChar = *tempEos;
           *tempEos = '\0';
           sol2posWidth = textWidth( &tempStr[sol] );
           *tempEos = saveChar;
          }
         unsigned long prevWidth = sol2posWidth;
         unsigned long prevWidthPos = pos;
         while ( pos < nextSep && sol2posWidth < ulLineWidth )
          {
           prevWidth = sol2posWidth;
           prevWidthPos = pos;
           if ( isspace( tempStr[pos] ) ) // At a space, move to next word.
            {
             while ( ++pos < nextSep &&  isspace( tempStr[pos] ) ) ;
            }
           else // In a word, move to next space.
            {
             while ( ++pos < nextSep && !isspace( tempStr[pos] ) ) ;
            }
           char* tempEos = &tempStr[pos];
           char saveChar = *tempEos;
           *tempEos = '\0';
           sol2posWidth = textWidth( &tempStr[sol] );
           *tempEos = saveChar;
          }
         //
         // While the length from sol to pos-1 is > ulLineWidth, move pos
         // back one space or word.
         //
         while ( pos > sol && sol2posWidth > ulLineWidth )
          {
           --pos; // Last byte of previous text measured.
           //
           // Try to speed things up here.  Assume that a space following
           // a space has width spaceWidth.
           //
           if ( pos > sol
             && isspace( tempStr[pos]   )
             && isspace( tempStr[pos-1] )
               ) sol2posWidth -= spaceWidth;
           else
            {
             if ( pos > sol && !isspace( tempStr[pos] ) ) // In a word.
              {
               // Set pos to first byte of current word.
               while ( pos > sol && !isspace( tempStr[pos-1] ) ) --pos;
              }
             if ( pos > sol ) // There is something to measure.
              {
               if ( pos == prevWidthPos ) // We already know the width.
                 sol2posWidth = prevWidth;  // Avoid re-calculating it.
               else
                {
                 char* tempEos = &tempStr[pos];
                 char saveChar = *tempEos;
                 *tempEos = '\0';
                 sol2posWidth = textWidth( &tempStr[sol] );
                 *tempEos = saveChar;
                }
              }
            }
          }
         if ( tempStr[pos] == '\n' )
          {
           ++pos; // Skip over new line char.
           nextSep = pos + strcspn( &tempStr[pos], "\n" ); // Find next separator.
          }
         if ( tempStr[pos] != '\0' ) ++ulLines; // More text... need another line.
         if ( pos == sol ) ulLines = 0;  // This text will never fit in ulLineWidth.
         //
         // Set the next start of line.
         //
         sol = pos;
        } // end while
       delete [] tempStr;
      }
    }
   return ulLines;
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::setWindowFont                                                         |
|                                                                              |
| Used by IControl::setFont()
------------------------------------------------------------------------------*/
bool IFont :: setWindowFont(IWindow* pwindow) const
{
   IMODTRACE_DEVELOP("IFont::setWindowFont");
   bool result = True;
#ifdef IC_PM
   // Build string to pass on WinSetPresParam call
   IString presParm(pointSize());
   presParm += ".";
   presParm += name();

   if (isBold())
     presParm += ".Bold";

   if (isItalic())
     presParm += ".Italic";

   if (isOutline())
     presParm += ".Outline";
   if (isUnderscore())
     presParm += ".Underscore";
   if (isStrikeout())
     presParm += ".Strikeout";

   result = WinSetPresParam(pwindow->handle(),
                   PP_FONTNAMESIZE,
                   presParm.length()+1,
                   (char*)presParm);

   if (result == false)
     ITHROWGUIERROR("WinSetPresParam");
#endif
#ifdef IC_MOTIFWIN
	 pwindow->setFont(*this);
#endif
   return result;
}


/*------------------------------------------------------------------------------
| IFont::setCharWidth                                                          |
|                                                                              |
| Only affects chars draw after beginUsingFont                                 |
------------------------------------------------------------------------------*/
IFont& IFont :: setCharWidth     (unsigned long ulWidth,
                                  const IPresSpaceHandle& presSpaceHandle)
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   // Only works for Vector Fonts
   if (isBitmapOnly())
   {
     ITHROWLIBRARYERROR(IC_VECTOR_FONT_ONLY_FUNCTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   }
   copyPrivate();
#ifdef IC_PM
   long  lxFontResolution;
   HDC   hDC;
         /* Query the device context for the screen and then query */
         /* the resolution of the device for the device context.   */
   IPresSpaceHandle hps;
   if (presSpaceHandle)
     hps = presSpaceHandle;
   else
     hps = WinGetPS(HWND_DESKTOP);
   hDC = GpiQueryDevice(hps);
   DevQueryCaps( hDC, CAPS_HORIZONTAL_FONT_RES,
                 (long)1, &lxFontResolution);
         /* Calculate the size of the character box, based on the  */
         /* point size selected and the resolution of the device.  */
         /* The size parameters are of type FIXED, NOT int.        */
         /*  NOTE: 1 point == 1/72 of an inch.                     */
   ppd->fxClWidth = (FIXED)((MAKEFIXED(ulWidth, 0) / 72 ) *
                     lxFontResolution );
   delete [] ppd->plClCharWidths;
   ppd->plClCharWidths = 0;
   makeCharWidths(hps);

   if (presSpaceHandle == 0)
     WinReleasePS(hps);
#endif
#ifdef IC_WIN
   (ppd->plogfontCl)->lfWidth = ulWidth;

   fontChanged();
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
   return *this;
}

/*------------------------------------------------------------------------------
| IFont::setCharHeight                                                         |
|                                                                              |
| Only affects chars draw after beginUsingFont                                 |
------------------------------------------------------------------------------*/
IFont& IFont :: setCharHeight    (unsigned long ulHeight,
                                  const IPresSpaceHandle& presSpaceHandle)
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   // only valid for Vector fonts
   if (isBitmapOnly())
   {
     ITHROWLIBRARYERROR(IC_VECTOR_FONT_ONLY_FUNCTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   }
   copyPrivate();
#ifdef IC_PM
   long  lyFontResolution;
   HDC   hDC;
         /* Query the device context for the screen and then query */
         /* the resolution of the device for the device context.   */
   IPresSpaceHandle hps;
   if(presSpaceHandle)
     hps = presSpaceHandle;
   else
     hps = WinGetPS(HWND_DESKTOP);
   hDC = GpiQueryDevice(hps);
   DevQueryCaps( hDC, CAPS_VERTICAL_FONT_RES,
                 (long)1, &lyFontResolution);
         /* Calculate the size of the character box, based on the  */
         /* point size selected and the resolution of the device.  */
         /* The size parameters are of type FIXED, NOT int.        */
         /*  NOTE: 1 point == 1/72 of an inch.                     */
   ppd->fxClHeight = (FIXED)(((MAKEFIXED(ulHeight, 0)) / 72 ) *
                      lyFontResolution );

   if (presSpaceHandle == 0)
     WinReleasePS(hps);
#endif
#ifdef IC_WIN
   (ppd->plogfontCl)->lfHeight = ulHeight;

   fontChanged();
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::setCharSize                                                           |
------------------------------------------------------------------------------*/
IFont& IFont :: setCharSize (const ISize& size,
                             const IPresSpaceHandle& presSpaceHandle)
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   if (isBitmapOnly())
   {
     ITHROWLIBRARYERROR(IC_VECTOR_FONT_ONLY_FUNCTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   }
   copyPrivate();
#ifdef IC_PM
   setCharWidth(size.width(), presSpaceHandle);
   return setCharHeight(size.height(), presSpaceHandle);
#endif
#ifdef IC_WIN
   (ppd->plogfontCl)->lfHeight = size.height();
   (ppd->plogfontCl)->lfWidth  = size.width();

   fontChanged();
   makeFontMetrics( presSpaceHandle );
   return *this;
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return *this;
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IFont::setFontAngle                                                          |
------------------------------------------------------------------------------*/
IFont& IFont :: setFontAngle     (const IPoint& point,
                                  const IPresSpaceHandle& presSpaceHandle)
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   if (isBitmapOnly())
   {
     ITHROWLIBRARYERROR(IC_VECTOR_FONT_ONLY_FUNCTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   }
   copyPrivate();
   ppd->pntClAngle = point;
#ifdef IC_WIN
   double angle = 0.0;
   if (point.x() == 0)
   {
     if (point.y() > 0)
       angle = 900.0;
     else if (point.y() < 0)
       angle = -900.0;
   }
   else
   {
     angle = (double)DEGREETENTHS_TO_RADS *
                            atan((double)point.y() / (double)point.x());

     if (point.x() < 0)
       angle += 1800.0;
   }

   (ppd->plogfontCl)->lfOrientation = (long)angle;
   fontChanged();
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::setFontShear                                                          |
------------------------------------------------------------------------------*/
IFont& IFont :: setFontShear     (const IPoint& point,
                                  const IPresSpaceHandle& presSpaceHandle)
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PMWIN
   if (isBitmapOnly())
   {
     ITHROWLIBRARYERROR(IC_VECTOR_FONT_ONLY_FUNCTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   }
   copyPrivate();
   ppd->pntClShear = point;
#ifdef IC_WIN
   double angle = 0.0;
   if (point.x() == 0)
   {
     if (point.y() > 0)
       angle = 900.0;
     else if (point.y() < 0)
       angle = -900.0;
   }
   else
   {
     angle = (double)DEGREETENTHS_TO_RADS *
                            atan((double)point.y() / (double)point.x());

     if (point.x() < 0)
       angle += 1800.0;
   }

   (ppd->plogfontCl)->lfEscapement = (long)angle;
   fontChanged();
   makeFontMetrics();
#endif
#endif //IC_PMWIN
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::beginUsingFont                                                        |
------------------------------------------------------------------------------*/
IFont& IFont :: beginUsingFont (const IPresSpaceHandle& hps)
{
   IMODTRACE_DEVELOP("IFont::beginUsingFont");
#ifdef IC_PMWIN
#ifdef IC_PM
   if (!ppd->lClLCId) {
      ppd->lClLCId = 1;

      // Have to make sure our local Id is not already in use
      // First, get the total number of local Ids in use
      long NumLcId = GpiQueryNumberSetIds(hps);

      // Check if there are any in use
      if (NumLcId) {
         long Types[255];
         STR8 Names[255];
         long LcIds[255];

         // Get information about all local Ids
         bool RC = GpiQuerySetIds(hps, NumLcId, Types, Names, LcIds);
         if (!RC) {
            ITRACE_DEVELOP("Error getting ids.");
            ITHROWGUIERROR("GpiQuerySetIds");
         } /* endif */

         // Check all used Ids to see if ours is already used
         for (unsigned long LCNum = 0; LCNum < NumLcId ; LCNum++ ) {
            if (LcIds[LCNum] == ppd->lClLCId) {
               // Ours is already used, increment to next Id and restart search
               ppd->lClLCId++;
               LCNum = -1;

               // NewLCId must be in the range 1 to 254
               if (ppd->lClLCId > 254) {
                  // Error! No more free ids!
                  ITRACE_DEVELOP("Error no free ids.");
                  ITHROWGUIERROR("GpiQuerySetIds");
               } /* endif */
            } /* endif */
         } /* endfor */

      } /* endif */
   } /* endif */

   LONG match = GpiCreateLogFont(hps, 0, ppd->lClLCId, ppd->pfattrsCl);
   if (match==GPI_ERROR) {
      ITHROWGUIERROR("GpiCreateLogFont");
   } /* endif */
   if (!GpiSetCharSet(hps, ppd->lClLCId)) {
      ITHROWGUIERROR("GpiSetCharSet");
   }
          /* Check if the font found is an outline (vector) font      */
   if ((ppd->pfattrsCl)->fsFontUse & FATTR_FONTUSE_OUTLINE) {
            /* Set the Character mode so that the characters will     */
            /* be stretched properly.                                 */
      GpiSetCharMode(hps, CM_MODE2);

      SIZEF sizef;
      sizef.cx = ppd->fxClWidth;
      sizef.cy = ppd->fxClHeight;
            /* Set character box for the outline font to size above   */
      GpiSetCharBox( hps, &sizef );
      GRADIENTL gradlAngle;
      gradlAngle.x = ppd->pntClAngle.x();
      gradlAngle.y = ppd->pntClAngle.y();
      GpiSetCharAngle(hps, &gradlAngle);
      POINTL shearAngle = ppd->pntClShear.asPOINTL();
      GpiSetCharShear(hps, &shearAngle);
   } /* endif */
   else if ( ppd->pfontmetCl )
   {
      GpiSetCharMode(hps, CM_MODE2);
      SIZEF sizef = { MAKEFIXED(ppd->pfontmetCl->lEmInc,0),
                      MAKEFIXED(ppd->pfontmetCl->lEmHeight,0) };
      GpiSetCharBox( hps, &sizef );
   }
   GpiSetCharDirection(hps, ppd->lClFontDir);
#endif
#ifdef IC_WIN
   /*******************************************************************/
   /* Select our font into the device context for drawing operations, */
   /* saving the previous selected font to be restored on completion  */
   /*******************************************************************/
   HFONT prevFont = (HFONT) SelectObject( hps, *(ppd->fpFontHandle) );
   // If the font returned from SelectObject is the same as the font
   // being set into the device context, then another IFont reference
   // has already done a beginUsingFont.  In this case, increment the
   // inUseCount.  Otherwise, save the device context's previous font
   // handle.
   ppd->inUseCount++;
   if (prevFont != *(ppd->fpFontHandle))
   {
      ppd->prevFont = prevFont;
   }
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
   ppd->beginUsingFont( hps );
#endif //IC_MOTIF
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::endUsingFont                                                          |
------------------------------------------------------------------------------*/
IFont& IFont :: endUsingFont (const IPresSpaceHandle& hps)
{
   IMODTRACE_DEVELOP("IFont::endUsingFont");
#ifdef IC_PM
   GpiSetCharSet(hps, LCID_DEFAULT);
   if (ppd->lClLCId) {
      GpiDeleteSetId(hps, ppd->lClLCId);
      ppd->lClLCId = 0;
   } /* endif */
#endif
#ifdef IC_WIN
   /*******************************************************************/
   /* Reselect the original font back into the device context if      */
   /* the inUseCount is zero.  Otherwise, decrement the inUseCount.   */
   /*******************************************************************/
   ppd->inUseCount--;
   if (ppd->inUseCount == 0)
   {
      SelectObject( hps, ppd->prevFont );
   }
#endif
#ifdef IC_MOTIF
   ppd->endUsingFont( hps );
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IFont::isBitmap                                                              |
------------------------------------------------------------------------------*/
bool IFont :: isBitmap () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (((ppd->pfontmetCl)->fsDefn & FM_DEFN_OUTLINE) == 0);
#endif
#ifdef IC_WIN
   return (!((ppd->ptextmetricCl)->tmPitchAndFamily & TMPF_VECTOR));
#endif
#ifdef IC_MOTIF
   return !ppd->isVector();
#endif
}

/*------------------------------------------------------------------------------
| IFont::isFixed                                                               |
------------------------------------------------------------------------------*/
bool IFont :: isFixed () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (bool)((ppd->pfontmetCl)->fsType & FM_TYPE_FIXED);
#endif
#ifdef IC_WIN
   return (!((ppd->ptextmetricCl)->tmPitchAndFamily & TMPF_FIXED_PITCH));
#endif
#ifdef IC_MOTIF
   return ppd->isFixed();
#endif
}


/*------------------------------------------------------------------------------
| IFont::name                                                                  |
------------------------------------------------------------------------------*/
IString IFont :: name() const
{
#ifdef IC_PMWIN
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   IString strTemp((char*)((ppd->pfontmetCl)->szFacename));
#endif
#ifdef IC_WIN
   IString strTemp((char*)((ppd->plogfontCl)->lfFaceName));
#endif
   return strTemp;
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return ppd->faceName();
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::qualifiedName                                                                  |
------------------------------------------------------------------------------*/
IString IFont :: qualifiedName() const
{
  IString qualName = "IOC:";

#ifdef IC_PMWIN
  qualName += name() + ":" + IString(pointSize());
#endif //IC_PMWIN

#ifdef IC_MOTIF
  int fontListCount = ppd->numFontStructs();
  if ( fontListCount == 1 )
    qualName += ppd->xFontName();
  else
  {
    for (int i = 0; i < fontListCount; i++)
    {
      qualName += ppd->xFontName(i);
      qualName += ",";
    }
    qualName = qualName.stripTrailing(",");
  }
#endif //IC_MOTIF

  return qualName;
}


/*------------------------------------------------------------------------------
| IFont::useNonPropOnly                                                        |
------------------------------------------------------------------------------*/
IFont& IFont :: useNonPropOnly (bool nonProportionalOnly,
                                const IPresSpaceHandle& presSpaceHandle)
{
#ifdef IC_PMWIN
   copyPrivate();
   if (nonProportionalOnly) {
      ppd->flClStyles = ppd->flClStyles | IC_NONPROPORTIONAL_FONTS_ONLY;
      if (!isFixed()) {
         fontChanged();
      } /* endif */
   } else {
      ppd->flClStyles = ppd->flClStyles & ~IC_NONPROPORTIONAL_FONTS_ONLY;
      fontChanged();
   } /* endif */
#ifdef IC_PM
   checkForMatch(presSpaceHandle);
#else
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
   if ( ( nonProportionalOnly  && ppd->isFixed()  )   ||
        ( !nonProportionalOnly  && !ppd->isFixed() ) )
      return *this;

   ppd->setFixed( nonProportionalOnly );
   ppd->checkForMatchSimple(presSpaceHandle);
#endif //IC_MOTIF
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::useBitmapOnly                                                         |
------------------------------------------------------------------------------*/
IFont& IFont :: useBitmapOnly (bool bitmapOnly,
                              const IPresSpaceHandle& presSpaceHandle)
{
#ifdef IC_PMWIN
   copyPrivate();
   if (bitmapOnly) {
      ppd->flClStyles = ppd->flClStyles | IC_BITMAP_FONTS_ONLY;
      ppd->flClStyles = ppd->flClStyles & ~IC_VECTOR_FONTS_ONLY;
      mustUseVector(FALSE);
      if (!isBitmap()) {
         fontChanged();
      } /* endif */
   } else {
      ppd->flClStyles = ppd->flClStyles & ~IC_BITMAP_FONTS_ONLY;
      ppd->flClStyles = ppd->flClStyles | IC_VECTOR_FONTS_ONLY;
      fontChanged();
      mustUseVector(FALSE);
   } /* endif */
#ifdef IC_PM
   checkForMatch(presSpaceHandle);
#else
   makeFontMetrics( presSpaceHandle );
#endif
   return *this;
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return useVectorOnly(!bitmapOnly, presSpaceHandle);
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IFont::useVectorOnly                                                         |
------------------------------------------------------------------------------*/
IFont& IFont :: useVectorOnly (bool vectorOnly,
                               const IPresSpaceHandle& presSpaceHandle)
{
#ifdef IC_PMWIN
   copyPrivate();
   if (vectorOnly) {
      ppd->flClStyles = ppd->flClStyles & ~IC_BITMAP_FONTS_ONLY;
      ppd->flClStyles = ppd->flClStyles | IC_VECTOR_FONTS_ONLY;
      mustUseVector(TRUE);
      if (isBitmap()) {
         fontChanged();
      } /* endif */
   } else {
      ppd->flClStyles = ppd->flClStyles | IC_BITMAP_FONTS_ONLY;
      ppd->flClStyles = ppd->flClStyles & ~IC_VECTOR_FONTS_ONLY;
      fontChanged();
      mustUseVector(FALSE);
   } /* endif */
#ifdef IC_PM
   checkForMatch(presSpaceHandle);
#else
   makeFontMetrics( presSpaceHandle );
#endif
#endif //IC_PMWIN
#ifdef IC_MOTIF
   if ( ( vectorOnly && ppd->isVector()  )    ||
        ( !vectorOnly && !ppd->isVector() ) )
      return *this;

   ppd->setVector(vectorOnly);
   ppd->checkForMatch(presSpaceHandle);
#endif //IC_MOTIF
   return *this;
}


/*------------------------------------------------------------------------------
| IFont::isNonPropOnly                                                         |
------------------------------------------------------------------------------*/
bool IFont :: isNonPropOnly () const
{
#ifdef IC_PMWIN
   return (ppd->flClStyles & IC_NONPROPORTIONAL_FONTS_ONLY);
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return ppd->isFixed();
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IFont::isBitmapOnly                                                          |
------------------------------------------------------------------------------*/
bool IFont :: isBitmapOnly () const
{
#ifdef IC_PMWIN
   return (ppd->flClStyles & IC_BITMAP_FONTS_ONLY);
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return !ppd->isVector();
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IFont::isVectorOnly                                                          |
------------------------------------------------------------------------------*/
bool IFont :: isVectorOnly () const
{
#ifdef IC_PMWIN
   return (ppd->flClStyles & IC_VECTOR_FONTS_ONLY);
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return ppd->isVector();
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| IFont::pointSize                                                             |
------------------------------------------------------------------------------*/
unsigned long IFont :: pointSize () const
{
#ifdef IC_PM
   return FIXEDINT(ppd->fxClPointSize);
#endif
#ifdef IC_WIN
  return (ppd->fxClPointSize);
#endif
#ifdef IC_MOTIF
   return ppd->pointSize();
#endif
}


/*------------------------------------------------------------------------------
| IFont::isBold                                                                |
------------------------------------------------------------------------------*/
bool IFont :: isBold ( ) const
{
#ifdef IC_PM
   return (ppd->pfattrsCl->fsSelection & FATTR_SEL_BOLD);
#endif
#ifdef IC_WIN
   return ((ppd->ptextmetricCl)->tmWeight > FW_MEDIUM );
#endif
#ifdef IC_MOTIF
   return ppd->isBold();
#endif
}

/*------------------------------------------------------------------------------
| IFont::isItalic                                                              |
------------------------------------------------------------------------------*/
bool IFont :: isItalic ( ) const
{
#ifdef IC_PM
   return (ppd->pfattrsCl->fsSelection & FATTR_SEL_ITALIC);
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmItalic;
#endif
#ifdef IC_MOTIF
   return ppd->isItalic();
#endif
}

/*------------------------------------------------------------------------------
| IFont::isUnderscore                                                          |
------------------------------------------------------------------------------*/
bool IFont :: isUnderscore ( ) const
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PM
   return ppd->pfattrsCl->fsSelection & FATTR_SEL_UNDERSCORE ;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmUnderlined;
#endif
#ifdef IC_MOTIF
   return false;
#endif
}

/*------------------------------------------------------------------------------
| IFont::isStrikeout                                                           |
------------------------------------------------------------------------------*/
bool IFont :: isStrikeout ( ) const
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PM
   return ppd->pfattrsCl->fsSelection & FATTR_SEL_STRIKEOUT ;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmStruckOut;
#endif
#ifdef IC_MOTIF
   return false;
#endif
}

/*------------------------------------------------------------------------------
| IFont::isOutline                                                             |
------------------------------------------------------------------------------*/
bool IFont :: isOutline ( ) const
{
   ITRACE_MOTIF_NOP();
   ITRACE_WIN_NOP();
#ifdef IC_PM
   return ppd->pfattrsCl->fsSelection & FATTR_SEL_OUTLINE ;
#endif
#ifdef IC_MOTIFWIN
   return false;
#endif
}


/*------------------------------------------------------------------------------
| IFont::avgCharWidth                                                          |
------------------------------------------------------------------------------*/
unsigned long IFont :: avgCharWidth() const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (ppd->pfontmetCl)->lAveCharWidth;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmAveCharWidth;
#endif
#ifdef IC_MOTIF
   return ppd->avgCharWidth();
#endif
}


/*------------------------------------------------------------------------------
| IFont::maxCharHeight                                                         |
------------------------------------------------------------------------------*/
unsigned long  IFont :: maxCharHeight () const
{
#ifdef IC_PM
   return maxSize().height();
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmHeight;
#endif
#ifdef IC_MOTIF
   return ppd->maxCharSize().height();
#endif
}

/*------------------------------------------------------------------------------
| IFont::maxSize                                                               |
------------------------------------------------------------------------------*/
ISize IFont :: maxSize () const
{
#ifdef IC_PMWIN
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   ISize size(ppd->pfontmetCl->lMaxCharInc, ppd->pfontmetCl->lMaxBaselineExt);
#endif
#ifdef IC_WIN
   ISize size( (ppd->ptextmetricCl)->tmMaxCharWidth,
               (ppd->ptextmetricCl)->tmHeight );
#endif
   return size;
#endif //IC_PMWIN
#ifdef IC_MOTIF
   return ppd->maxCharSize();
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::maxUppercaseSize                                                      |
------------------------------------------------------------------------------*/
ISize IFont :: maxUppercaseSize( ) const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   ISize size(ppd->pfontmetCl->lEmInc, ppd->pfontmetCl->lMaxBaselineExt);
   return size;
#endif
#ifdef IC_WIN
   return maxSize();   // Set this the same as maxSize since we cannot query
                       // the max uppercase width (but assume it is largest)
#endif
#ifdef IC_MOTIF
   return ppd->maxCharSize();
#endif
}

#ifdef IC_MOTIFPM
/*------------------------------------------------------------------------------
| IFont::superscriptSize                                                       |
------------------------------------------------------------------------------*/
ISize IFont :: superscriptSize( ) const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   ISize size(ppd->pfontmetCl->lSuperscriptXSize,
              ppd->pfontmetCl->lSuperscriptYSize);
   return size;
#endif //IC_PM
#ifdef IC_MOTIF
   ISize ms = ppd->maxCharSize();
   return ISize(ms.width()  - ppd->getProp(XA_SUPERSCRIPT_X),
                ms.height() - ppd->getProp(XA_SUPERSCRIPT_Y) );
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::subscriptSize                                                         |
------------------------------------------------------------------------------*/
ISize IFont :: subscriptSize( ) const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   ISize size(ppd->pfontmetCl->lSubscriptXSize,
              ppd->pfontmetCl->lSubscriptYSize);
   return size;
#endif //IC_PM
#ifdef IC_MOTIF
   ISize ms = ppd->maxCharSize();
   return ISize(ms.width()  - ppd->getProp(XA_SUBSCRIPT_X),
                ms.height() - ppd->getProp(XA_SUBSCRIPT_Y) );
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::superscriptOffset                                                     |
------------------------------------------------------------------------------*/
ISize IFont :: superscriptOffset( ) const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   ISize size(ppd->pfontmetCl->lSuperscriptXOffset,
              ppd->pfontmetCl->lSuperscriptYOffset);
   return size;
#endif //IC_PM
#ifdef IC_MOTIF
   return ISize(ppd->getProp(XA_SUPERSCRIPT_X),
                ppd->getProp(XA_SUPERSCRIPT_Y) );
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::subscriptOffset                                                       |
------------------------------------------------------------------------------*/
ISize IFont :: subscriptOffset( ) const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   ISize size(ppd->pfontmetCl->lSubscriptXOffset,
              ppd->pfontmetCl->lSubscriptYOffset);
   return size;
#endif //IC_PM
#ifdef IC_MOTIF
   return ISize(ppd->getProp(XA_SUBSCRIPT_X),
                ppd->getProp(XA_SUBSCRIPT_Y) );
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| IFont::avgLowercase                                                          |
------------------------------------------------------------------------------*/
unsigned long IFont :: avgLowercase () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return ppd->pfontmetCl->lXHeight;
#endif //IC_PM
#ifdef IC_MOTIF
   return ppd->xHeight();
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| IFont::avgUppercase                                                          |
------------------------------------------------------------------------------*/
unsigned long IFont :: avgUppercase () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return ppd->pfontmetCl->lEmHeight;
#endif //IC_PM
#ifdef IC_MOTIF
   return ppd->capHeight();
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| IFont::maxLowercaseAscender                                                  |
------------------------------------------------------------------------------*/
unsigned long IFont :: maxLowercaseAscender () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return ppd->pfontmetCl->lLowerCaseAscent;
#endif //IC_PM
#ifdef IC_MOTIF
   // No data available on maxLowercase Ascender/Descender.
   // So use overall ascent and descent information.
   return ppd->maxAscentDescent().coord1();
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| IFont::maxLowercaseDescender                                                 |
------------------------------------------------------------------------------*/
unsigned long IFont :: maxLowercaseDescender () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return ppd->pfontmetCl->lLowerCaseDescent;
#endif //IC_PM
#ifdef IC_MOTIF
   return ppd->maxAscentDescent().coord2();
#endif //IC_MOTIF

}
#endif


/*------------------------------------------------------------------------------
| IFont::maxDescender                                                          |
------------------------------------------------------------------------------*/
unsigned long IFont :: maxDescender () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (ppd->pfontmetCl)->lMaxDescender;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmDescent;
#endif
#ifdef IC_MOTIF
   return ppd->maxAscentDescent().coord2();
#endif
}

/*------------------------------------------------------------------------------
| IFont::maxAscender                                                           |
------------------------------------------------------------------------------*/
unsigned long IFont :: maxAscender () const
{
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (ppd->pfontmetCl)->lMaxAscender;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmAscent;
#endif
#ifdef IC_MOTIF
   return ppd->maxAscentDescent().coord1();
#endif
}

/*------------------------------------------------------------------------------
| IFont::externalLeading                                                       |
------------------------------------------------------------------------------*/
unsigned long IFont :: externalLeading () const
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (ppd->pfontmetCl)->lExternalLeading;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmExternalLeading;
#endif
#ifdef IC_MOTIF
   // No value available. Since all fonts built into PM are designed with
   // external leading of zero, this seems safe to return here.
   return 0;
#endif
}

/*------------------------------------------------------------------------------
| IFont::internalLeading                                                       |
------------------------------------------------------------------------------*/
unsigned long IFont :: internalLeading () const
{
   ITRACE_MOTIF_NOP();
#ifdef IC_PM
   IFont* const localThis = (IFont* const)this;
   localThis->makeFontMetrics();
   return (ppd->pfontmetCl)->lInternalLeading;
#endif
#ifdef IC_WIN
   return (ppd->ptextmetricCl)->tmInternalLeading;
#endif
#ifdef IC_MOTIF
   // not available, ascent a good enough approximation

   //@@Rob.  Doesn't make sense if you read internalLeading documentation.
   //@@   return ppd->maxAscentDescent.coord1();
   return 0;
#endif
}


/*------------------------------------------------------------------------------
| IFont::styleSet                                                              |
------------------------------------------------------------------------------*/
ITextStyleSet IFont :: styleSet () const
{

  ITextStyleSet textStyles;

  textStyles.add( ITextBitmapStyle( isBitmap() ));
  textStyles.add( ITextBoldfaceStyle( isBold() ));
  textStyles.add( ITextItalicStyle( isItalic() ));
  textStyles.add( ITextOutlineStyle( isOutline() ));
  textStyles.add( ITextPointSizeStyle( pointSize() ));
  textStyles.add( ITextStrikethroughStyle( isStrikeout() ));
  textStyles.add( ITextTypefaceStyle( name() ));
  textStyles.add( ITextUnderlineStyle( isUnderscore() ));
  textStyles.add( ITextFixedPitchStyle( isFixed() ));

  return textStyles;
}

