// Revision: 81 1.10.2.1 source/ui/baseapp/ibidiset.cpp, nls, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: ibidiset.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions used for the    *
*   bidirectional support.                                                     *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

#include <ibase.hpp>

extern "C" {
#ifdef IC_MOTIF
  #define BIDI               // For macros in XmStrDefs.h
#endif
  #include <iwindefs.h>
#ifdef IC_PM
  #include <pmbidi.h>
#endif
}

#ifdef IC_MOTIF
  #include <iarglist.hpp>
#endif
#include <ibidiset.hpp>
#include <icconst.h>         // For IC_UM_TOOLBAR, IC_INVALIDENUMVALUE
#include <iexcept.hpp>
#include <iframe.hpp>
#include <igbidi.hpp>
#include <igrport.hpp>
#include <grafdevs.hpp>
#include <ireslib.hpp>
#include <istatics.hpp>
#include <iwindow.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IBIDISET_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_PM
typedef unsigned long
     (_System IWinSetLangInfo)( unsigned long,   // hwnd
                                unsigned long,   // effect
                                unsigned long,   // data
                                unsigned long,   // mask
                                unsigned long,   // flags
                                unsigned long ); // reserved

typedef unsigned long
     (_System IWinQueryLangInfo)( unsigned long,   // hwnd
                                  unsigned long,   // effect
                                  unsigned long,   // flags
                                  unsigned long ); // reserved

typedef unsigned long
     (_System IGpiQueryBidiAttr)( HPS ); // hps

typedef unsigned long
     (_System IGpiSetBidiAttr)( HPS,             // hps
                                unsigned long ); // attributes
#endif /* IC_PM */

#ifdef IC_PM
/*------------------------------------------------------------------------------
| class IBidiData                                                              |
|                                                                              |
| Private class used for implementation of bidi support.                       |
| Only a single instance of this class is needed per process, which is         |
| managed by IBidiSettingsData.                                                |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IBidiData {
public:
  IBidiData ( );
IDynamicLinkLibrary
 *fPmBidiDynLinkLib;
IWinSetLangInfo
 *fSetInfo;
IWinQueryLangInfo
 *fQueryInfo;
IGpiSetBidiAttr
 *fSetBidiAttr;
IGpiQueryBidiAttr
 *fQueryBidiAttr;
}; // IBidiData

#pragma pack(pop)
#pragma enum(pop)

IBidiData::IBidiData ( )
  : fPmBidiDynLinkLib( 0 ),
    fSetInfo( 0 ),
    fQueryInfo( 0 ),
    fSetBidiAttr( 0 ),
    fQueryBidiAttr( 0 )
{ }
#endif // IC_PM


/*------------------------------------------------------------------------------
| class IBidiSettingsData                                                      |
|                                                                              |
| Private data class for IBidiSettings.                                        |
------------------------------------------------------------------------------*/
#pragma enum(4)
#pragma pack(push,4)

class IBidiSettingsData {
public:
  IBidiSettingsData ( );
IBidiSettingsData
 &operator=         ( const IBidiSettingsData& data );
bool
  operator==        ( const IBidiSettingsData& data ) const;
#ifdef IC_PM
static IBidiData
 *fBidi;
#endif
IGBidiSettings  
  fGBidi;
#ifndef IC_PM
unsigned long
  fExplicitlySetFlags;
enum ESetFlags {
  kTextType             = 1,
  kTextOrientation      = 2,
  kWindowLayout         = 4,
  kNumeralDisplay       = 8,
  kSymmetricSwapping    = 16,
  kWordByWordReordering = 32,
  kTextShape            = 64
  };
#endif
}; // IBidiSettingsData

#pragma pack(pop)
#pragma enum(pop)

IBidiSettingsData::IBidiSettingsData ( )
  : fGBidi()
#ifndef IC_PM
  , fExplicitlySetFlags( 0 )
#endif
{ }

IBidiSettingsData&
  IBidiSettingsData::operator= ( const IBidiSettingsData& data )
{
  if ( this != &data )
  {  // Not copying self.
     this->fGBidi = data.fGBidi;
#ifndef IC_PM
     this->fExplicitlySetFlags = data.fExplicitlySetFlags;
#endif
  }
  return *this;
}

bool IBidiSettingsData::operator== ( const IBidiSettingsData& data ) const
{
  return ( this->fGBidi == data.fGBidi
#ifndef IC_PM
           &&
           this->fExplicitlySetFlags == data.fExplicitlySetFlags
#endif
         );
}

/*------------------------------------------------------------------------------
| Static data                                                                  |
------------------------------------------------------------------------------*/
#ifndef IC_PM
IBidiSettings
  IBidiSettings::fgApplicationDefaults;
#endif
#ifdef IC_PM
IBidiData
 *IBidiSettingsData::fBidi( 0 );
#endif

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
bool IBidiSettings::isBidiSupported ( )
{
  bool
    bidiSupported = IGBidiSettings::isBidiSupported();

#ifdef IC_PM
  if ( bidiSupported  &&  ( ! IBidiSettingsData::fBidi ) )
  {
     IBidiSettingsData::fBidi = new IBidiData;
     // Try to load the PMBIDI.DLL and get the address of its APIs.
     try
     {
        IBidiSettingsData::fBidi->fPmBidiDynLinkLib = new IDynamicLinkLibrary( "pmbidi" );
        IBidiSettingsData::fBidi->fSetInfo = (IWinSetLangInfo*)
                           IBidiSettingsData::fBidi->fPmBidiDynLinkLib->procAddress
                             ( "Win32SetLangInfo" );
        IBidiSettingsData::fBidi->fQueryInfo = (IWinQueryLangInfo*)
                           IBidiSettingsData::fBidi->fPmBidiDynLinkLib->procAddress
                             ( "Win32QueryLangInfo" );
        IBidiSettingsData::fBidi->fSetBidiAttr = (IGpiSetBidiAttr*)
                           IBidiSettingsData::fBidi->fPmBidiDynLinkLib->procAddress
                             ( "Gpi32SetBidiAttr" );
        IBidiSettingsData::fBidi->fQueryBidiAttr = (IGpiQueryBidiAttr*)
                           IBidiSettingsData::fBidi->fPmBidiDynLinkLib->procAddress
                             ( "Gpi32QueryBidiAttr" );
     }
     catch ( IException& exc )
     {
        if ( IBidiSettingsData::fBidi->fPmBidiDynLinkLib )
        {
           delete IBidiSettingsData::fBidi->fPmBidiDynLinkLib;
        }
        delete IBidiSettingsData::fBidi;
        IBidiSettingsData::fBidi = 0;
        bidiSupported = false;
        IRETHROW( exc );
     }
     adoptStaticObject( IBidiSettingsData::fBidi );
  }
#endif // IC_PM

  return bidiSupported;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IBidiSettings::setApplicationDefaults
                      ( const IBidiSettings& applicationValues )
{
  IASSERTSTATE( IBidiSettings::isBidiSupported() == true );

#ifdef IC_PM
  IBidiSettingsData::fBidi->fSetInfo( 0, LI_BD_PROCESS_ATTR,
                                      applicationValues.fBidiSettingsData
                                                 ->fGBidi.bidiFlags() |
                                        BDA_INIT | BDA_LEVEL,
                                        BDAM_INIT | BDAM_LEVEL |
                                        BDAM_TEXTTYPE | BDAM_TEXT_ORIENTATION |
                                        BDAM_WND_ORIENTATION | BDAM_NUMERALS |
                                        BDAM_SYM_SWAP | BDAM_WORD_BREAK |
                                        BDAM_TEXT_SHAPE,
                                      0, 0 );
#else
  fgApplicationDefaults = applicationValues;
#endif
}

IBidiSettings IBidiSettings::applicationDefaults ( )
{
  IASSERTSTATE( IBidiSettings::isBidiSupported() == true );

#ifdef IC_PM
  IBidiSettings
    applicationSettings;
  unsigned long flags =
      IBidiSettingsData::fBidi->fQueryInfo( 0, LI_BD_PROCESS_ATTR, 0, 0 );
  applicationSettings.fBidiSettingsData
                        ->fGBidi.setBidiFlags( flags );
  return applicationSettings;
#else
  return fgApplicationDefaults;
#endif
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::IBidiSettings ( const IWindow& window )
  : fBidiSettingsData( 0 )
{
  const IFrameWindow
   *frame = dynamic_cast< const IFrameWindow* >( &window );
  if ( frame )
  {
     this->initialize( frame->handleForChildCreation() );
  }
  else
  {
     this->initialize( window.handle() );
  }
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::IBidiSettings ( const IWindowHandle& windowHandle )
  : fBidiSettingsData( 0 )
{
  this->initialize( windowHandle );
}

/*------------------------------------------------------------------------------
|                                                                              |
| Note: Some contend that international settings should apply to each of the   |
|       items being drawn, rather than the graf port as a whole.               |
------------------------------------------------------------------------------*/
IBidiSettings::IBidiSettings ( const IGrafPort& port )
  : fBidiSettingsData( 0 )
{
  IGrafPort* nonConstPort = (IGrafPort*)&port;
  IPresSpaceHandle
    presSpace = nonConstPort->device()->deviceContext();
  this->initialize( presSpace );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::IBidiSettings ( const IPresSpaceHandle& presSpace )
  : fBidiSettingsData( 0 )
{
  this->initialize( presSpace );
}

// Note: IBidiSettings(const IGraphicContext&) moved to ographic\iuiobs.cpp
//       to remove dependency on old graphics.

/*------------------------------------------------------------------------------
| Default constructor (used for application-default functions only, for now).  |
------------------------------------------------------------------------------*/
IBidiSettings::IBidiSettings ( )
  : fBidiSettingsData( new IBidiSettingsData )
{
  // Don't assert for isBidiSupported here, because this will
  // cause IBidiSettings::fgApplicationDefaults to throw an
  // exception during static object initialization on OS/2 on a
  // non-bidi machine.
}

/*------------------------------------------------------------------------------
| Copy constructor.                                                            |
------------------------------------------------------------------------------*/
IBidiSettings::IBidiSettings ( const IBidiSettings& settings )
  : fBidiSettingsData( new IBidiSettingsData )
{
  *( this->fBidiSettingsData ) = *( settings.fBidiSettingsData );
}

/*------------------------------------------------------------------------------
| Assignment operator.                                                         |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::operator= ( const IBidiSettings& settings )
{
  *( this->fBidiSettingsData ) = *( settings.fBidiSettingsData );
  return *this;
}

/*------------------------------------------------------------------------------
| Comparison operator.                                                         |
------------------------------------------------------------------------------*/
bool IBidiSettings::operator== ( const IBidiSettings& settings ) const
{
  return ( *( this->fBidiSettingsData ) ==
                       *( settings.fBidiSettingsData ) );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::~IBidiSettings ( )
{
  if ( fBidiSettingsData )
  {
     delete fBidiSettingsData;
  }
}

/*------------------------------------------------------------------------------
| Initialization from an IWindowHandle.                                        |
------------------------------------------------------------------------------*/
void IBidiSettings::initialize ( const IWindowHandle& windowHandle )
{
  IASSERTSTATE( IBidiSettings::isBidiSupported() == true );

  fBidiSettingsData = new IBidiSettingsData;
#ifdef IC_PM
  unsigned long flags =
    fBidiSettingsData->fBidi->fQueryInfo( windowHandle.asUnsigned(),
                                          LI_BD_WND_ATTR, 0L, 0L);
  fBidiSettingsData->fGBidi.setBidiFlags( flags );
#endif
#ifdef IC_WIN
  // Check for RTL extended styles, and set flags in fBidiSettingsData->fData
  // accordingly.
  unsigned long
    style = GetWindowLong( windowHandle, GWL_EXSTYLE );
  ( *this )
    .setTextOrientation( ( style & WS_EX_RTLREADING ) ?
                            IBidiSettings::textRightToLeft :
                            IBidiSettings::textLeftToRight );
  ( *this )
    .setWindowLayout( ( style & WS_EX_RIGHT ) ?
                         IBidiSettings::layoutRightToLeft :
                         IBidiSettings::layoutLeftToRight );
  // We don't explicitly set other bidi attributes.  As a result,
  // they default as follows:
  //  - typeType             : IBidiSettings::implicit
  //  - typeShape            : IBidiSettings::displayShaped
  //  - numeralDisplay       : IBidiSettings::contextual
  //  - symmetricSwapping    : false
  //  - wordByWordReordering : false
  // (Are these all correct?  IC_NOTYET)
#endif // IC_WIN
#ifdef IC_MOTIF
  // Check for bidi resources, and set flags in fBidiSettingsData->fData
  // accordingly.
  // Below types are typedef'ed to unsigned char in Xm.h.
  XmTextMode
    textMode = 'z';
  XmStringDirection     // Use in place of XmLayoutDirection.
    stringDir = 'z';
  XmNssMode
    numeralsMode = 'z';
  XmCsdMode
    shapingMode = 'z';
  unsigned char
    symmetricSwap = 'z',
    wordBreak = 'z';

  // Issue the query.
  XtVaGetValues( (Widget) windowHandle,
                 XmNtextMode, &textMode,
                 XmNstringDirection, &stringDir,
                 XmNnssMode,  &numeralsMode,
                 XmNcsdMode,  &shapingMode,
                 XmNsymmetricSwap, &symmetricSwap,
                 XmNwordBreak, &wordBreak,
                 0 );
  // IC_NOTYET Need XmNtextCompose or XmNexpandTail?

  // Map to an IBidiSettings::BidiTextType value.
  if ( textMode == XmTEXT_MODE_VISUAL )
  {
     this->setTextType( IBidiSettings::visual );
  }
  else if ( textMode == XmTEXT_MODE_IMPLICIT )
  {
     this->setTextType( IBidiSettings::implicit );
  }
  else if ( textMode == XmTEXT_MODE_EXPLICIT )
  {
     this->setTextType( IBidiSettings::visual );  // Correct?  IC_NOTYET
  }
  // else default to implicit          Correct?  IC_NOTYET

  // Map to an IBidiSettings::BidiTextOrientation and BidiLayout value.
  if ( stringDir == XmSTRING_DIRECTION_L_TO_R )
  {
     bool
       explicitValue = true;
     if ( this != &fgApplicationDefaults )
     {  // Seeing this value doesn't guarantee that it was
        // explicitly set by anybody, since it seems to be a default
        // value.  For a child window to not inherit a default value
        // from its parent window (which prevents the child from
        // honoring bidi resources that may be set by the user), we
        // make a rather large assumption here.  Assume that the LTR
        // value is explicitly set only if the direction of the
        // application default value is explicitly set.
        if ( ! ( fgApplicationDefaults
                  .fBidiSettingsData->fExplicitlySetFlags &
                                        IBidiSettingsData::kWindowLayout ) )
        {
           explicitValue = false;
        }
     }
     if ( explicitValue )
     {
        this->setTextOrientation( IBidiSettings::textLeftToRight );
        this->setWindowLayout( IBidiSettings::layoutLeftToRight );
     }
  }
  else if ( stringDir == XmSTRING_DIRECTION_R_TO_L )
  {
     this->setTextOrientation( IBidiSettings::textRightToLeft );
     this->setWindowLayout( IBidiSettings::layoutRightToLeft );
  }
  else if ( stringDir == XmSTRING_DIRECTION_DEFAULT )
  {
     this->setTextOrientation( IBidiSettings::textContextual );  // Correct?  IC_NOTYET
     this->setWindowLayout( IBidiSettings::layoutLeftToRight );  // Correct?  IC_NOTYET
  }
  // else default to left-to-right

  // Map to an IBidiSettings::BidiNumeralType value.
  if ( numeralsMode == XmNSS_MODE_BILINGUAL )
  {
     this->setNumeralDisplay( IBidiSettings::contextual );
  }
  else if ( numeralsMode == XmNSS_MODE_PASSTHRU )
  {
     this->setNumeralDisplay( IBidiSettings::asStored );
  }
  else if ( numeralsMode == XmNSS_MODE_ARABIC )
  {
     this->setNumeralDisplay( IBidiSettings::arabic );
  }
  else if ( numeralsMode == XmNSS_MODE_HINDU )
  {
     this->setNumeralDisplay( IBidiSettings::national );
  }
  // else default to contextual        Correct?  IC_NOTYET

  // Map to an IBidiSettings::TextShape value.
  if ( shapingMode == XmCSD_MODE_AUTOMATIC )
  {
     this->setTextShape( IBidiSettings::displayShaped );
  }
  else if ( shapingMode == XmCSD_MODE_PASSTHRU )
  {
     this->setTextShape( IBidiSettings::saveShaped );  // Correct?  IC_NOTYET
  }
  else if ( shapingMode == XmCSD_MODE_INITIAL )
  {
     this->setTextShape( IBidiSettings::initialShape );
  }
  else if ( shapingMode == XmCSD_MODE_MIDDLE )
  {
     this->setTextShape( IBidiSettings::middleShape );
  }
  else if ( shapingMode == XmCSD_MODE_FINAL )
  {
     this->setTextShape( IBidiSettings::finalShape );
  }
  else if ( shapingMode == XmCSD_MODE_ISOLATED )
  {
     this->setTextShape( IBidiSettings::isolatedShape );
  }
  // else default to displayShaped     Correct?  IC_NOTYET
  // IC_NOTYET (anything map to nominalShape?)

  // Map to boolean values.
  if ( symmetricSwap == true )
  {
     this->enableSymmetricSwapping();
  }
  else if ( symmetricSwap == false )
  {
     this->disableSymmetricSwapping();
  }
  // else default to false             Correct?  IC_NOTYET

  if ( wordBreak == true )
  {
     this->enableWordByWordReordering();
  }
  else if ( wordBreak == false )
  {
     this->disableWordByWordReordering();
  }
  // else default to false             Correct?  IC_NOTYET
#endif // IC_MOTIF
}

/*------------------------------------------------------------------------------
| Initialization from an IPresSpaceHandle.                                     |
------------------------------------------------------------------------------*/
void IBidiSettings::initialize ( const IPresSpaceHandle& handle )
{
  ITRACE_WIN_NOP();
  ITRACE_MOTIF_NOP();

  IASSERTSTATE( IBidiSettings::isBidiSupported() == true );

  fBidiSettingsData = new IBidiSettingsData;
#ifdef IC_PM
  unsigned long flags =
    IBidiSettingsData::fBidi->fQueryBidiAttr( handle.asUnsigned() );
  fBidiSettingsData->fGBidi.setBidiFlags( flags );
#endif
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IBidiSettings::apply ( IWindow& window,
                            bool childInherit,
                            bool refresh ) const
{
  ITRACE_MOTIF_NOP();

  if ( window.isFrameWindow() )
  {
     this->applyToFrameWindow( window, childInherit, refresh );
  }
  else
  {
     IWindowHandle
       nonConstWindowHandle = window.handle();
     this->applyToWindow( nonConstWindowHandle, childInherit, refresh );
  }
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IBidiSettings::applyToFrameWindow ( IWindow& frameWindow,
                                         bool     childInherit,
                                         bool     refresh ) const
{
#if 0
  // Move the below code to IFrameWindow::setBidiSettings if it
  // is still needed?                              IC_NOTYET
#ifdef IC_PM
  //-----------------------------------------------------------------
  // There is a bug in PM that does not update frame extensions
  // when the BIDI attributes of the frame have changed (and the
  // child inherit flag is true).
  //
  // A fix for this problem is to send a private message to the frame
  // before we change the BIDI attributes.  This will enable the
  // window to decide if it should force the frame extensions to be
  // updated.  The following cases are covered:
  //
  // 1. IBidiSettings::apply (no child inherit)
  //    For this case we first send the private message to set a
  //    flag in the IFrameWindowData.  When IFrameHandler receives
  //    the bidi attribute change, it will not update frame
  //    extensions, but will turn off the IFrameWindowData flag.
  //
  // 2. IBidiSettings::apply (with child inherit)
  //    We do not send the private message.  When IFrameHandler
  //    receives the bidi attribute change, it will update all of
  //    its frame extensions.
  //
  // 3. IBidiSettings::apply (to parent of the frame)
  //    Same processing as item 2.
  //
  // 4. User update through Language Viewer (Alt+ScrollLock) popup
  //    Same processing as item 2.
  //-----------------------------------------------------------------
  if ( ! childInherit )
  {
     frameWindow.sendEvent( IC_UM_BIDI_CHANGE );
  }
#endif // IC_PM
#endif // 0

  //-----------------------------------------------------------------
  // Let applyToWindow do most of the work.
  //-----------------------------------------------------------------
  IWindowHandle
    nonConstWindowHandle = frameWindow.handle();
  this->applyToWindow( nonConstWindowHandle, childInherit, refresh );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IBidiSettings::apply ( const IWindowHandle& windowHandle,
                            bool childInherit,
                            bool refresh ) const
{
  ITRACE_MOTIF_NOP();

  IWindow
   *bidiWindow = IWindow::windowWithHandle( windowHandle );
  if ( bidiWindow  &&  bidiWindow->isFrameWindow() )
  {
     this->applyToFrameWindow( *bidiWindow, childInherit, refresh );
  }
  else
  {
     IWindowHandle
       nonConstWindowHandle = windowHandle;
     this->applyToWindow( nonConstWindowHandle, childInherit, refresh );
  }
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IBidiSettings::applyToWindow ( IWindowHandle& windowHandle,
                                    bool           childInherit,
                                    bool           refresh ) const
{
#ifdef IC_PMWIN
  // Notify the window that its bidi attributes are about to change.
  IWindow
   *bidiWindow = IWindow::windowWithHandle( windowHandle );
  if ( bidiWindow )
  {
     bidiWindow->setBidiSettings( *this, childInherit, refresh );
  }
#endif

#ifdef IC_PM
  unsigned long ulFlags = 0;
  if ( refresh )
  {
     ulFlags |= LIF_WND_REFRESH;
  }
  if ( childInherit )
  {
     ulFlags |= LIF_CHILD_INHERIT;
  }

  unsigned long
    flags = fBidiSettingsData->fGBidi.bidiFlags();
  IBidiSettingsData::fBidi->fSetInfo( windowHandle.asUnsigned(),
                                      LI_BD_WND_ATTR,
                                      flags |
                                        BDA_INIT | BDA_LEVEL,
                                        BDAM_INIT | BDAM_LEVEL |
                                        BDAM_TEXTTYPE | BDAM_TEXT_ORIENTATION |
                                        BDAM_WND_ORIENTATION | BDAM_NUMERALS |
                                        BDAM_SYM_SWAP | BDAM_WORD_BREAK |
                                        BDAM_TEXT_SHAPE,
                                      ulFlags, 0 );
#endif // IC_PM
#ifdef IC_WIN
  if ( refresh )
  {
     if ( bidiWindow )
     {
        bidiWindow->refresh();
     }
     // Add support for non-wrappered windows.   IC_NOTYET
  }

  if ( childInherit )
  {
     if ( bidiWindow )
     {
        // Propagate new bidi attributes to all child windows.
        IWindow::ChildCursor
          childCursor( *bidiWindow, false );
        for ( childCursor.setToFirst();
              childCursor.isValid();
              childCursor.setToNext() )
        {
           IWindowHandle
             childWindow( bidiWindow->childAt( childCursor ) );
           this->apply( childWindow, childInherit, refresh );
        }
     }
     // Add support for non-wrappered windows.   IC_NOTYET
  }
#endif // IC_WIN
#ifdef IC_MOTIF
  // Bidi resources cannot be set after a widget has been created.
#endif
}

/*------------------------------------------------------------------------------
|                                                                              |
| Note: Some contend that international settings should apply to each of the   |
|       items being drawn, rather than the graf port as a whole.               |
------------------------------------------------------------------------------*/
void IBidiSettings::apply ( IGrafPort& port ) const
{
  ITRACE_WIN_NOP();
  ITRACE_MOTIF_NOP();

#ifdef IC_PM
  this->apply( port.device()->deviceContext() );
#endif
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
void IBidiSettings::apply ( const IPresSpaceHandle& presSpace ) const
{
  ITRACE_WIN_NOP();
  ITRACE_MOTIF_NOP();

#ifdef IC_PM
  unsigned long
    flags = fBidiSettingsData->fGBidi.bidiFlags();
  IBidiSettingsData::fBidi->fSetBidiAttr( presSpace.asUnsigned(),
                                          flags );
#endif
}

// Note: apply(const IGraphicContext&) moved to ographic\iuiobs.cpp
//       to remove dependency upon on old graphics.

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::setTextType
                 ( IBidiSettings::BidiTextType textType )
{
  ITRACE_WIN_NOP();

  IGBidiSettings::EBidiTextType
    gTextType = IGBidiSettings::kVisual;
  switch ( textType )
  {
     case visual:
       // We have the correct value already.
       break;
     case implicit:
       gTextType = IGBidiSettings::kImplicit;
       break;
     default:
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }
  
  fBidiSettingsData->fGBidi.setTextType( gTextType );

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |= IBidiSettingsData::kTextType;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::setTextOrientation
                 ( IBidiSettings::BidiTextOrientation textOrientation )
{
  IGBidiSettings::EBidiTextOrientation
    gOrientation = IGBidiSettings::kTextLeftToRight;
  switch ( textOrientation )
  {
     case textLeftToRight:
       // We have the correct value already.
       break;
     case textRightToLeft:
       gOrientation = IGBidiSettings::kTextRightToLeft;
       break;
     case textContextual:
       gOrientation = IGBidiSettings::kTextContextual;
       break;
     default:
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }

  fBidiSettingsData->fGBidi.setTextOrientation( gOrientation );

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IBidiSettingsData::kTextOrientation;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::setWindowLayout
                 ( IBidiSettings::BidiLayout windowLayout )
{
  IGBidiSettings::EBidiLayout
    gLayout = IGBidiSettings::kLayoutLeftToRight;
  switch ( windowLayout )
  {
     case layoutLeftToRight:
       // We have the correct value already.
       break;
     case layoutRightToLeft:
       gLayout = IGBidiSettings::kLayoutRightToLeft;
       break;
     default:
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }

  fBidiSettingsData->fGBidi.setWindowLayout( gLayout );

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IBidiSettingsData::kWindowLayout;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::setNumeralDisplay
                 ( IBidiSettings::BidiNumeralType numeralDisplay )
{
  ITRACE_WIN_NOP();

  IGBidiSettings::EBidiNumeralType
    gNumeral = IGBidiSettings::kArabic;
  switch ( numeralDisplay )
  {
     case arabic:
       // We have the correct value already.
       break;
     case asStored:
       gNumeral = IGBidiSettings::kAsStored;
       break;
     case national:
       gNumeral = IGBidiSettings::kNational;
       break;
     case contextual:
       gNumeral = IGBidiSettings::kContextual;
       break;
     default:
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }

  fBidiSettingsData->fGBidi.setNumeralDisplay( gNumeral );

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IBidiSettingsData::kNumeralDisplay;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::enableSymmetricSwapping ( bool enable )
{
  ITRACE_WIN_NOP();

  fBidiSettingsData->fGBidi.enableSymmetricSwapping( enable );
#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IBidiSettingsData::kSymmetricSwapping;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::disableSymmetricSwapping ( )
{
  return this->enableSymmetricSwapping( false );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::enableWordByWordReordering ( bool enable )
{
  ITRACE_WIN_NOP();

  fBidiSettingsData->fGBidi.enableWordByWordReordering( enable );
#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |=
                        IBidiSettingsData::kWordByWordReordering;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::disableWordByWordReordering ( )
{
  return this->enableWordByWordReordering( false );
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings& IBidiSettings::setTextShape
                 ( IBidiSettings::BidiTextShape textShape )
{
  ITRACE_WIN_NOP();

  IGBidiSettings::EBidiTextShape  
    gTextShape = IGBidiSettings::kDisplayShaped;
  switch ( textShape )
  {
     case displayShaped:
       // We have the correct value already.
       break;
     case saveShaped:
       gTextShape = IGBidiSettings::kSaveShaped;
       break;
     case nominalShape:
       gTextShape = IGBidiSettings::kNominalShape;
       break;
     case initialShape:
       gTextShape = IGBidiSettings::kInitialShape;
       break;
     case middleShape:
       gTextShape = IGBidiSettings::kMiddleShape;
       break;
     case finalShape:
       gTextShape = IGBidiSettings::kFinalShape;
       break;
     case isolatedShape:
       gTextShape = IGBidiSettings::kIsolatedShape;
       break;
     default:
       ITHROWLIBRARYERROR( IC_INVALIDENUMVALUE,
                           IBaseErrorInfo::invalidParameter,
                           IException::recoverable );
       break;
  }

  fBidiSettingsData->fGBidi.setTextShape( gTextShape );

#ifndef IC_PM
  fBidiSettingsData->fExplicitlySetFlags |= IBidiSettingsData::kTextShape;
#endif

  return *this;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::BidiTextType IBidiSettings::textType ( ) const
{
  IBidiSettings::BidiTextType
    textType = implicit;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kTextType )
#endif
  {
     IGBidiSettings::EBidiTextType
       gTextType = fBidiSettingsData->fGBidi.textType();
     if ( gTextType == IGBidiSettings::kVisual )
     {
        textType = visual;
     }
  }
  return textType;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::BidiTextOrientation IBidiSettings::textOrientation ( ) const
{
  IBidiSettings::BidiTextOrientation
    orientation = textLeftToRight;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kTextOrientation )
#endif
  {
     IGBidiSettings::EBidiTextOrientation
       gOrientation = fBidiSettingsData->fGBidi.textOrientation();
     if ( gOrientation == IGBidiSettings::kTextRightToLeft )
     {
        orientation = textRightToLeft;
     }
     else if ( gOrientation == IGBidiSettings::kTextContextual )
     {
        orientation = textContextual;
     }
  }
  return orientation;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::BidiLayout IBidiSettings::windowLayout ( ) const
{
  IBidiSettings::BidiLayout
    layout = layoutLeftToRight;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kWindowLayout )
#endif
  {
     IGBidiSettings::EBidiLayout
       gLayout = fBidiSettingsData->fGBidi.windowLayout();
     if ( gLayout == IGBidiSettings::kLayoutRightToLeft )
     {
        layout = layoutRightToLeft;
     }
  }
  return layout;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::BidiNumeralType IBidiSettings::numeralDisplay ( ) const
{
  IBidiSettings::BidiNumeralType
    numeral = contextual;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kNumeralDisplay )
#endif
  {
     IGBidiSettings::EBidiNumeralType
       gNumeral = fBidiSettingsData->fGBidi.numeralDisplay();
     if ( gNumeral == IGBidiSettings::kAsStored )
     {
        numeral = asStored;
     }
     else if ( gNumeral == IGBidiSettings::kNational )
     {
        numeral = national;
     }
     else if ( gNumeral == IGBidiSettings::kArabic )
     {
        numeral = arabic;
     }
  }
  return numeral;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
bool IBidiSettings::isSymmetricSwappingEnabled ( ) const
{
  bool
    enabled = false;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kSymmetricSwapping )
#endif
  {
     if ( fBidiSettingsData->fGBidi.isSymmetricSwappingEnabled() )
     {
        enabled = true;
     }
  }
  return enabled;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
bool IBidiSettings::isWordByWordReorderingEnabled ( ) const
{
  bool
    enabled = false;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kWordByWordReordering )
#endif
  {
     if ( fBidiSettingsData->fGBidi.isWordByWordReorderingEnabled() )
     {
        enabled = true;
     }
  }
  return enabled;
}

/*------------------------------------------------------------------------------
|                                                                              |
------------------------------------------------------------------------------*/
IBidiSettings::BidiTextShape IBidiSettings::textShape ( ) const
{
  IBidiSettings::BidiTextShape
    textShape = displayShaped;
#ifndef IC_PM
  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kTextShape )
#endif
  {
     IGBidiSettings::EBidiTextShape
       gTextShape = fBidiSettingsData->fGBidi.textShape();
     switch ( gTextShape )
     {
        case ( IGBidiSettings::kIsolatedShape ):
          textShape = isolatedShape;
          break;
        case ( IGBidiSettings::kFinalShape ):
          textShape = finalShape;
          break;
        case ( IGBidiSettings::kMiddleShape ):
          textShape = middleShape;
          break;
        case ( IGBidiSettings::kInitialShape ):
          textShape = initialShape;
          break;
        case ( IGBidiSettings::kNominalShape ):
          textShape = nominalShape;
          break;
        case ( IGBidiSettings::kSaveShaped ):
          textShape = saveShaped;
          break;
        default:
          break;
     }
  }
  return textShape;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IBidiSettings::asArgList                                                     |
|                                                                              |
| Map the attributes of the bidi settings object to Motif resources that can   |
| be used when creating a window.                                              |
------------------------------------------------------------------------------*/
IArgList IBidiSettings::asArgList ( ) const
{
  IArgList
    argList;

  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kTextType )
  {
     // This and other data types are typedef'ed to unsigned char
     // in Xm.h.
     XmTextMode
       textMode = XmTEXT_MODE_IMPLICIT;
     if ( this->textType() == IBidiSettings::visual )
     {
        textMode = XmTEXT_MODE_VISUAL;
     }
     argList
      .setResource( XmNtextMode, (long) textMode );
  }

  if ( fBidiSettingsData->fExplicitlySetFlags &
                            ( IBidiSettingsData::kTextOrientation |
                              IBidiSettingsData::kWindowLayout ) )
  {
     XmStringDirection     // Use in place of XmLayoutDirection.
       stringDir = XmSTRING_DIRECTION_DEFAULT;
     IBidiSettings::BidiTextOrientation
       direction = this->textOrientation();
     IBidiSettings::BidiLayout
       layout = this->windowLayout();

     // Map both text orientation and layout direction to the
     // XmStringDirection resource.  If either is RTL, make the
     // resource RTL.
     if ( ( fBidiSettingsData->fExplicitlySetFlags &
                          IBidiSettingsData::kTextOrientation )  &&
          ( direction == IBidiSettings::textRightToLeft ) )
     {
        stringDir = XmSTRING_DIRECTION_R_TO_L;
     }
     else if ( ( fBidiSettingsData->fExplicitlySetFlags &
                             IBidiSettingsData::kWindowLayout )  &&
               ( layout == IBidiSettings::layoutRightToLeft ) )
     {
        stringDir = XmSTRING_DIRECTION_R_TO_L;
     }
     else if ( ( fBidiSettingsData->fExplicitlySetFlags &
                          IBidiSettingsData::kTextOrientation )  &&
               ( direction == IBidiSettings::textContextual ) )
     {
        stringDir = XmSTRING_DIRECTION_DEFAULT;
     }
     else if ( ( fBidiSettingsData->fExplicitlySetFlags &
                          IBidiSettingsData::kTextOrientation )  &&
               ( direction == IBidiSettings::textLeftToRight ) )
     {
        stringDir = XmSTRING_DIRECTION_L_TO_R;
     }
     else if ( ( fBidiSettingsData->fExplicitlySetFlags &
                             IBidiSettingsData::kWindowLayout )  &&
               ( layout == IBidiSettings::layoutLeftToRight ) )
     {
        stringDir = XmSTRING_DIRECTION_L_TO_R;
     }
     argList
      .setResource( XmNstringDirection, (long) stringDir );
  }

  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kNumeralDisplay )
  {
     XmNssMode
       numeralsMode = XmNSS_MODE_BILINGUAL;
     IBidiSettings::BidiNumeralType
       numberType = this->numeralDisplay();
     if ( numberType == IBidiSettings::asStored )
     {
        numeralsMode = XmNSS_MODE_PASSTHRU;
     }
     else if ( numberType == IBidiSettings::arabic )
     {
        numeralsMode = XmNSS_MODE_ARABIC;
     }
     else if ( numberType == IBidiSettings::national )
     {
        numeralsMode = XmNSS_MODE_HINDU;
     }
     argList
      .setResource( XmNnssMode, (long) numeralsMode );
  }

  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kSymmetricSwapping )
  {
     unsigned char
       symmetricSwap = this->isSymmetricSwappingEnabled();
     argList
      .setResource( XmNsymmetricSwap, (long) symmetricSwap );
  }

  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kWordByWordReordering )
  {
     unsigned char
       wordBreak = this->isWordByWordReorderingEnabled();
     argList
      .setResource( XmNwordBreak, (long) wordBreak );
  }

  if ( fBidiSettingsData->fExplicitlySetFlags &
                            IBidiSettingsData::kTextShape )
  {
     XmCsdMode
       shapingMode = XmCSD_MODE_AUTOMATIC;
     IBidiSettings::BidiTextShape
       shape = this->textShape();
     if ( shape == IBidiSettings::saveShaped )
     {
        shapingMode = XmCSD_MODE_PASSTHRU;
     }
     else if ( shape == IBidiSettings::initialShape )
     {
        shapingMode = XmCSD_MODE_INITIAL;
     }
     else if ( shape == IBidiSettings::middleShape )
     {
        shapingMode = XmCSD_MODE_MIDDLE;
     }
     else if ( shape == IBidiSettings::finalShape )
     {
        shapingMode = XmCSD_MODE_FINAL;
     }
     else if ( shape == IBidiSettings::isolatedShape )
     {
        shapingMode = XmCSD_MODE_ISOLATED;
     }
     argList
      .setResource( XmNcsdMode, (long) shapingMode );
  }

  return argList;
}
#endif // IC_MOTIF
