// Revision: 78 1.5.1.12 source/ui/baseapp/iapp.cpp, thread, ioc.v400, 001006 
/*----------------------------------------------------------------------------*/
/* FILE NAME: iapp.cpp                                                        */
/*                                                                            */
/* DESCRIPTION:                                                               */
/*   Implementation of the class(es):                                         */
/*     IApplication                                                           */
/*     ICurrentApplication                                                    */
/*                                                                            */
/*                                                                            */
/* COPYRIGHT:                                                                 */
/*   IBM Open Class Library                                                   */
/*   Licensed Materials - Property of IBM                                     */
/*                                                                            */
/*   5645-001                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.           */
/*                                                                            */
/*   US Government Users Restricted Rights - Use, duplication, or             */
/*   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.        */
/*                                                                            */
/*----------------------------------------------------------------------------*/
#include <ibase.hpp>

extern "C" {
#ifdef IC_PM
  #define INCL_DOSPROCESS         // For DosGetInfoBlocks.
  #include <iwindefs.h>
#endif
  #include <stdlib.h>
}

#include <iapp.hpp>
#include <ireslib.hpp>
#include <istatics.hpp>
#include <istring.hpp>
#include <ithread.hpp>
#include <itrace.hpp>
#include <icolmap.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IAPP_CPP_
  #include <ipagetun.h>
#endif


// Static objects.
ICurrentApplication
 *IApplication::pCurrent = 0;

static const char
#ifdef IC_PMWIN
 origResLibVar[]  = "ICLUI RESLIB",
#endif
 resLibVar[]      = "ICLUI_RESLIB",
#ifdef IC_PMWIN
 defaultLib[]     = OPENCLASS_RESOURCE_LIBRARY_NAME_NO_EXT(ocres, or);
#else
 defaultLib[]     = OPENCLASS_RESOURCE_LIBRARY_NAME_2(ocres, or);
#endif


/*------------------------------------------------------------------------------
| ICurrentApplication::ICurrentApplication                                     |
|                                                                              |
| Construct the current application for the current process.                   |
------------------------------------------------------------------------------*/
ICurrentApplication::ICurrentApplication ( )
  : IApplication ( 0 )
  , icluiLib     ( 0 )
  , userLib      ( 0 )
  , exeResource  ( 0 )
  , argCount     ( 0 )
  , argValue     ( 0 )
#ifdef IC_PM
  , pPIB         ( 0 )
#endif
  , pCurrentApplicationData ( 0 )
  , fColorMap    ( 0 )
{
  IMODTRACE_DEVELOP( "ICurrentApplication::ICurrentApplication" );

#ifdef IC_PM
  this->setId( this->pib().pib_ulpid );
#endif

#ifdef IC_WIN
  this->setId( INonGUIApplication::currentPID() );
#endif

#ifdef IC_MOTIF
  this->setId( getpid() );
#endif
}

/*------------------------------------------------------------------------------
| ICurrentApplication::~ICurrentApplication                                    |
|                                                                              |
| Destructor here for page tuning (defined here so the compiler will not       |
| statically generate it).                                                     |
------------------------------------------------------------------------------*/
ICurrentApplication::~ICurrentApplication ( )
{
  delete this->icluiLib;
  delete this->userLib;
  delete [] this->argValue;
  // Delete pCurrentApplicationData when it is defined.
  if ( fColorMap )
  {
     if ( *fColorMap == IColorMap::defaultColorMap() )
     {  // IColorMap still has a pointer to the memory we want to
        // delete. Force IColorMap to drop the pointer so it won't
        // later try to access deleted storage.
        IColorMap
         &newColorMap = IColorMap::defaultSystemColorMap();
        IColorMap::setDefaultColorMap( &newColorMap );
     }
     delete fColorMap;
  }
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| ICurrentApplication::pib                                                     |
|                                                                              |
| Return a pointer to the process information block for this process           |
| (ICurrentApplication caches it so it retrieves it only once).                |
------------------------------------------------------------------------------*/
struct pib_s& ICurrentApplication::pib ( )
{
  if ( ! this->pPIB )
  {
     PTIB ptib;
     DosGetInfoBlocks( &ptib, &this->pPIB );
  }

  return *this->pPIB;
}
#endif //IC_PM

/*------------------------------------------------------------------------------
| ICurrentApplication::resourceLibrary                                         |
|                                                                              |
| Return the resource library storing resources supplied by the Open Class     |
| library.                                                                     |
------------------------------------------------------------------------------*/
IResourceLibrary &ICurrentApplication::resourceLibrary ( ) const
{
  IMODTRACE_DEVELOP( "ICurrentApplication::resourceLibrary" );

  // This query precedes a call to ICurrentApplication::setResourceLibrary,
  // which explicitly identifies the name of the resource library.
  if ( !this->icluiLib  &&  !this->exeResource )
  {
     // Check if the ICLUI_RESLIB environment variable has been set.
     const char* pLibName = getenv( resLibVar );
     if ( ! pLibName )
     {
#ifdef IC_PMWIN
        // Check if the ICLUI RESLIB environment variable has been set.
        pLibName = getenv( origResLibVar );
        if ( ! pLibName )
        {  // Use the default name.
           pLibName = defaultLib;
        }
#else
        // Use the default name.
        pLibName = defaultLib;
#endif
     }

     // We didn't have the NLS path search in prior releases but that is really
     // how we should be looking for our library resources.  To handle both old
     // code that relied on the LIBPATH searching and new code that uses the NLS
     // path, we first try to find the resource library in the NLS path and
     // then look in the LIBPATH.
     ICurrentApplication
      *nonConstThis = (ICurrentApplication*) this;
     try
     {
        nonConstThis->setResourceLibrary( pLibName,
                                          IDynamicLinkLibrary::kNLSPathSearch );
     }
     catch(IException& )
     {
#ifdef IC_MOTIF
       try
       {
          nonConstThis->setResourceLibrary( pLibName,
                                            IDynamicLinkLibrary::kOSDefaultSearch );
       }
       catch( IException& )
       {
          // For Motif, we try one last time in /usr/vacpp/loc/C
          // for applications using a locale of C or Posix, or no
          // locale at all.
          nonConstThis->setResourceLibrary( IString( "/usr/vacpp/loc/C/" )
                                              + pLibName );
       }
#else
       nonConstThis->setResourceLibrary( pLibName,
                                         IDynamicLinkLibrary::kOSDefaultSearch);
#endif
     }
  }

  return *(this->icluiLib);
}

/*------------------------------------------------------------------------------
| ICurrentApplication::setResourceLibrary                                      |
|                                                                              |
| Set the resource library used to retrieve resources supplied by Open Class   |
| library.                                                                     |
------------------------------------------------------------------------------*/
ICurrentApplication
 &ICurrentApplication::setResourceLibrary
                             ( const char* resLibName,
                               IDynamicLinkLibrary::ESearchLocation searchLocation )
{
  IMODTRACE_DEVELOP( "ICurrentApplication::setResourceLibrary" );

  IResourceLibrary
   *oldLib = this->icluiLib;

  // Create a new resource library from the specified name.
  if ( resLibName )
  {
     this->icluiLib = new IDynamicLinkLibrary( resLibName, searchLocation );
     this->exeResource = false;
  }
  else
  {
     this->icluiLib = new IResourceLibrary;
     this->exeResource = true;
  }

  // Free the old resource library, if there was one.
  if ( oldLib )
  {
     delete oldLib;
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::userResourceLibrary                                     |
|                                                                              |
| Return the resource library storing application-defined resources.           |
------------------------------------------------------------------------------*/
IResourceLibrary &ICurrentApplication::userResourceLibrary ( ) const
{
  IMODTRACE_DEVELOP( "ICurrentApplication::userResourceLibrary" );

  // If no resource library has been specified, use the executable.
  if ( ! this->userLib )
  {
     ((ICurrentApplication*) this)->setUserResourceLibrary( 0 );
  }
  return *(this->userLib);
}

/*------------------------------------------------------------------------------
| ICurrentApplication::setUserResourceLibrary                                  |
|                                                                              |
| Set the resource library used to retrieve resources supplied by the          |
| application.                                                                 |
------------------------------------------------------------------------------*/
ICurrentApplication
 &ICurrentApplication::setUserResourceLibrary
                             ( const char* resLibName,
                               IDynamicLinkLibrary::ESearchLocation searchLocation )
{
  IMODTRACE_DEVELOP( "ICurrentApplication::setUserResourceLibrary" );

  IResourceLibrary
   *newLib = 0,
   *oldLib = this->userLib;

  // Attempt to load the new resource library.
  // Reset the userLib data member to point to the new resource library.
  if ( resLibName )
  {
     newLib = new IDynamicLinkLibrary( resLibName, searchLocation );
  }
  else
  {
     newLib = new IResourceLibrary;
  }
  this->userLib = newLib;

  // Free the old resource library, if there was one.
  if ( oldLib )
  {
     delete oldLib;
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::argc                                                    |
|                                                                              |
| Return the count of arguments previously set via setArgs.                    |
------------------------------------------------------------------------------*/
int ICurrentApplication::argc ( ) const
{
  return this->argCount;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::argv                                                    |
|                                                                              |
| argv - Return the array of arguments previously set via setArgs.             |
------------------------------------------------------------------------------*/
IString ICurrentApplication::argv ( int argNo ) const
{
  IString result;
  if ( argNo < argCount )
  {
     result = this->argValue[ argNo ];
  }
  return result;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::setArgs                                                 |
|                                                                              |
| Set the arguments for the application, to be returned by argc and argv.      |
------------------------------------------------------------------------------*/
ICurrentApplication &ICurrentApplication::setArgs ( int argc,
                                                    const char* const argv[] )
{
  // If we have previously-stored arguments, delete the old array.
  if ( this->argCount )
  {
     // Free the old array.
     delete [] this->argValue;
     this->argValue = 0;
  }

  if ( argc )
  {
     // Allocate a new array of pointers for the arguments.
     // For each element, allocate a buffer for an argument, initialize
     // it with the argument string, and store the address in the array.
     this->argValue = new IString[ argc ];
     this->argCount = 0;
     bool fInitializeGUI(false);
     while ( this->argCount < argc )
     {
        this->argValue[ this->argCount ] = IString(argv[ this->argCount ]);
        if ( this->argValue[this->argCount] == IString("-display") )
        {
          fInitializeGUI = true;
        }
        this->argCount++;
     }
     if (fInitializeGUI)
     {
       // if the display is set then we need to initialize the GUI so 2d graphics will
       // work correctly.
       ITRACE_DEVELOP("Initialize GUI from setArgs");
       if (!(IThread::current().isGUIInitialized()))
       {
          // Initialize GUI...
          IThread::current().initializeGUI();
       }
     }
  }

  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::run                                                     |
|                                                                              |
| Process presentation system messages on the current thread.                  |
------------------------------------------------------------------------------*/
ICurrentApplication &ICurrentApplication::run ( )
{
  IThread::current().processMsgs();
  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::exit                                                    |
|                                                                              |
| Exit the current thread, which will cause the application to exit if it is   |
| single-threaded.                                                             |
------------------------------------------------------------------------------*/
ICurrentApplication& ICurrentApplication::exit ( )
{
  INonGUIThread::current().exit( 0 );
  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentApplication::current                                                 |
|                                                                              |
| Return a pointer to the current application object.                          |
------------------------------------------------------------------------------*/
ICurrentApplication& IApplication::current ( )
{
  if ( ! IApplication::pCurrent )
  {
     IApplication::pCurrent = new ICurrentApplication;
     adoptStaticObject( IApplication::pCurrent );
  }

  return *IApplication::pCurrent;
}



/*------------------------------------------------------------------------------
| ICurrentApplication::colorMap                                                |
|                                                                              |
| Return the application color map.  If not set, returns the wev-safe color    |
| map.                                                                         |
------------------------------------------------------------------------------*/
IColorMap& ICurrentApplication::colorMap ( ) const
{
   if(fColorMap==0)
     return IColorMap::defaultColorMap();
   else
     return *fColorMap;
}


/*------------------------------------------------------------------------------
| ICurrentApplication::adoptColorMap                                           |
|                                                                              |
| Stores an application color map.                                             |
------------------------------------------------------------------------------*/
ICurrentApplication& ICurrentApplication::adoptColorMap ( IColorMap* colorMap )
{
   IColorMap::setDefaultColorMap( colorMap );
   if(fColorMap)
      delete fColorMap;
   fColorMap = colorMap;

   return *this;
}



/*------------------------------------------------------------------------------
| IApplication::IApplication                                                   |
|                                                                              |
| Constructor which initializes the process ID.                                |
------------------------------------------------------------------------------*/
IApplication::IApplication ( const IProcessId& id )
  : INonGUIApplication( id )
{
  IMODTRACE_DEVELOP( "IApplication::IApplication" );
}

/*------------------------------------------------------------------------------
| IApplication::~IApplication                                                  |
|                                                                              |
| Destructor here for page tuning (defined here so the compiler will not       |
| statically generate it).                                                     |
------------------------------------------------------------------------------*/
IApplication ::~IApplication ( )
{ }
