// Revision: 69 1.7.1.4 source/ui/baseapp/iaccelbd.cpp, userinput, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iaccelbd.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Implementation of the IAccelBld class.                                     *
*   By constructing a IAccelBld object the resChar string that is passed in    *
*   is parsed and the appropriate IAccelItems are added to the pMenu that      *
*   is passed in.                                                              *
*   IResourceLibrary::loadAccelTable() creates an IAccelBld object on the      *
*   stack to effect menu updating.                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   Licensed Materials - Property of IBM                                       *
*   (C) Copyright IBM Corporation 1997, 1997                                   *
*   All Rights Reserved                                                        *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#include <ilanglvl.hpp>

#include <X11/keysym.h>
#include <Xm/Xm.h>

#include <iaccelbd.hpp>
#include <icconst.h>
#include <iexcept.hpp>
#include <ikey.hpp>
#include <imstring.hpp>
#include <iresprv.hpp>
#include <ixconst.h>

// Segment definitions.
#ifdef IC_PAGETUNE
    #define _IACCELBD_CPP_
    #include <ipagetun.h>
#endif

// used in parsing accelerators in addItem
const IString iaccelKey("<Key>");
const IString iaccelPlus("+");
const IString iaccelV("V");
const IString iaccelK("K");
const IString iaccelCtl("Ctrl");
const IString iaccelSft("Shift");
const IString iaccelAt("Alt");
const IString iaccelMod1("Mod1");
const IString iaccelMod2("Mod2");
const IString iaccelMod3("Mod3");
const IString iaccelMod4("Mod4");
const IString iaccelMod5("Mod5");
const IString iaccelCtrl("CONTROL");
const IString iaccelShift("SHIFT");
const IString iaccelAlt("ALT");
const IString iaccelItem("ACCELITEM");
const IString iaccelEnd("END");

//
// Constructor merely calls getTokens() which makes everything happen.
//
IAccelBld :: IAccelBld(char* resChar,
                       IWindowHandle menu)
            : resString(resChar),
              menu(menu),
              state(looking),
              subState(startString),
              isShift(false),
              isCtrl(false),
              isAlt(false),
              isChar(false),
              altMask( Mod1Mask )
{
   // Use Mod1Mask for the Alt key modifier for now. We may remap
   // this later in ICurrentThreadData::getModifierMappings and
   // ICurrentThreadData::translateAccelerator.

   this->getTokens();
}

IAccelBld :: ~IAccelBld() { }

//
// getTokens - get all blank-delimited tokens from resString and call
//             parseIt() for each token
//
void IAccelBld :: getTokens(void)
{
   IString curToken;
   unsigned long resStringWords = resString.numWords();
   for(curPos = 1; curPos <= resStringWords; curPos++) {
         curToken = resString.word(curPos);
         parseIt(curToken);
   }
}

//
// parseIt - top level parser controlled by the state enum
//
void IAccelBld :: parseIt(IString& curTok)
{
   switch (state) {

     // looking for the next ACCELITEM
     case looking:
        if (curTok == iaccelItem) {
          state = accelitem;
          subState = startString;
        }
        break;

     // building up an ACCELITEM
     case accelitem:
        if (curTok == iaccelItem || curTok == iaccelEnd)
          accelitemDone(curTok);
        else
          subParseIt(curTok);
        break;
   }
}

//
// subParseIt - used to parse the info that follows the MENUITEM keyword.
//
void IAccelBld :: subParseIt(IString& curTok)
{
   switch (subState) {

     // expecting the start of a quoted string character or VK_ value
     case startString:
        if (curTok[1]  == '"') {
           accelString = curTok.subString(2,1);
           isChar = true;
        }
        else if ( (curTok[1]  == iaccelV) && (curTok[2] == iaccelK) )
           accelString = curTok.subString(4);

        accelString.insert(iaccelKey);
        subState = cmdId;
        break;

     // expecting the menu item id (numeric value)
     case cmdId:
        if (!curTok.isDigits())
          ITHROWGUIERROR("Expected numeric menu id and found: " + curTok);
        menuId = curTok.asUnsigned();
        subState = style;
        break;

     // expecting one or more modifier flags
     // currently only support CONTROL, SHIFT, and ALT
     // any other modifier is ignored
     case style:
        curTok.upperCase();
        if (curTok.includes(iaccelCtrl))
          isCtrl = true;
        if (curTok.includes(iaccelShift))
          isShift = true;
        if (curTok.includes(iaccelAlt))
          isAlt = true;
        break;
   }
}


//
// accelitemDone - called once we've parsed a complete ACCELITEM
//
void IAccelBld :: accelitemDone(IString& curTok)
{
   IString     accel;
   IString     altMod;
   IMString    accelText;

   Widget   itemHwnd = XtNameToWidget(menu,
      (char *)( (IString("*")) + (IString(menuId))));
   if (itemHwnd) {
      if (isShift)
         accelString.insert(iaccelSft);
      if (isCtrl) {
         if (isShift)
            accelString.insert(iaccelPlus);
         accelString.insert(iaccelCtl);
      }
      if (isAlt) {
         if ( (isShift) || (isCtrl) )
            accelString.insert(iaccelPlus);
         accelString.insert(iaccelAt);
      }
      accel = accelString;
      accel.change(iaccelPlus, " ");
      if ( (isCtrl) || (isShift) || (isAlt) )
         accelString.change(iaccelKey, iaccelPlus);
      else
         accelString.remove(1, iaccelKey.size());
      accelText = IMString(accelString);
      if (isAlt) {
        if (altMask == Mod1Mask)
          altMod.insert(iaccelMod1);
        if (altMask == Mod2Mask)
          altMod.insert(iaccelMod2);
        if (altMask == Mod3Mask)
          altMod.insert(iaccelMod3);
        if (altMask == Mod4Mask)
          altMod.insert(iaccelMod4);
        if (altMask == Mod5Mask)
          altMod.insert(iaccelMod5);
      }
      accel.change(iaccelAt, altMod);

      XtVaSetValues(itemHwnd,
                    XmNaccelerator, (char *)accel,
                    XmNacceleratorText, (XmString)accelText,
                    NULL);

   }
   if (curTok == iaccelItem) {
     state = accelitem;
     subState = startString;
   }
   else
     state = looking;

   clearItemData();
}

//
// clearItemData - reset data about item being built
//
void IAccelBld :: clearItemData(void)
{
   accelString = "";
   menuId = 0;
   isCtrl = false;
   isShift = false;
   isAlt = false;
   isChar = false;
}

IAccelTblHandle
  IAccelBld::rtacceltableToHandle ( const rtacceltable* resAccelTable )
{
  // Convert the rtacceltable defined in iresprv.hpp to an ACCELTABLE
  // defined in ixconst.h, and do other conversions that can only be
  // done at runtime.
  ACCELTABLE
   *destTable = 0;

  if ( resAccelTable )
  {  // Input data specified.
     unsigned short
       keyCount = resAccelTable->accelNumber;
     unsigned long
       destBufferSize = sizeof( ACCELTABLE ) +
                         ( sizeof(ACCEL) * ( keyCount ? keyCount - 1 : 0 ) );
     destTable = (ACCELTABLE*) new char[ destBufferSize ];
     destTable->accelNumber = keyCount;
     const accelerator
      *sourceItem = &( resAccelTable->accelerators[ 0 ] );
     ACCEL
      *destItem = destTable->accelerators;

     for ( int i = 1; i <= keyCount; i++ )
     {
        destItem->commandId = sourceItem->commandd;
        destItem->modifiers = 0;
        destItem->flags = 0;
   
        if ( sourceItem->options & 0x0001 )  // AF_CHAR
        {
           destItem->key = sourceItem->key;
           if ( destItem->key < 0x20  ||  destItem->key > 0x7e )
           {  // Ignore this case where character doesn't map to
              // a keysym (also ignore codepage).
              destItem->key = 0;
           }
        }
   
        if ( sourceItem->options & 0x0002 )  // AF_VIRTUALKEY
        {
           destItem->flags |= 0x0040;
           switch ( sourceItem->key )
           {  // Map the virtual key to an X keysym value.
              case 0x04:  // VK_BREAK
                destItem->key = IKey::kBreak;
                break;
              case 0x05:  // VK_BACKSPACE
                destItem->key = IKey::kBackSpace;
                break;
              case 0x06:  // VK_TAB
                destItem->key = IKey::kTab;
                break;
              case 0x07:  // VK_BACKTAB
                destItem->key = IKey::kTab;
                destItem->modifiers |= 0x01;  // Shift
                break;
              case 0x08:  // VK_NEWLINE
                destItem->key = IKey::kNewLine;
                break;
              case 0x09:  // VK_SHIFT
                destItem->key = IKey::kShift;
                break;
              case 0x0A:  // VK_CTRL
                destItem->key = IKey::kCtrl;
                break;
              case 0x0B:  // VK_ALT
                destItem->key = IKey::kAlt;
                break;
              case 0x0D:  // VK_PAUSE
                destItem->key = IKey::kPause;
                break;
              case 0x0E:  // VK_CAPSLOCK
                destItem->key = IKey::kCapsLock;
                break;
              case 0x0F:  // VK_ESC
                destItem->key = IKey::kEsc;
                break;
              case 0x10:  // VK_SPACE
                destItem->key = IKey::kSpace;
                break;
              case 0x11:  // VK_PAGEUP
                destItem->key = IKey::kPageUp;
                break;
              case 0x12:  // VK_PAGEDOWN
                destItem->key = IKey::kPageDown;
                break;
              case 0x13:  // VK_END
                destItem->key = IKey::kEnd;
                break;
              case 0x14:  // VK_HOME
                destItem->key = IKey::kHome;
                break;
              case 0x15:  // VK_LEFT
                destItem->key = IKey::kLeft;
                break;
              case 0x16:  // VK_UP
                destItem->key = IKey::kUp;
                break;
              case 0x17:  // VK_RIGHT
                destItem->key = IKey::kRight;
                break;
              case 0x18:  // VK_DOWN
                destItem->key = IKey::kDown;
                break;
              case 0x1A:  // VK_INSERT
                destItem->key = IKey::kInsert;
                break;
              case 0x1B:  // VK_DELETE
                destItem->key = IKey::kDelete;
                break;
              case 0x1C:  // VK_SCRLLOCK
                destItem->key = IKey::kScrollLock;
                break;
              case 0x1D:  // VK_NUMLOCK
                destItem->key = IKey::kNumLock;
                break;
              case 0x1E:  // VK_ENTER
                destItem->key = IKey::kEnter;
                break;
              case 0x20:  // VK_F1
                destItem->key = IKey::kF1;
                break;
              case 0x21:  // VK_F2
                destItem->key = IKey::kF2;
                break;
              case 0x22:  // VK_F3
                destItem->key = IKey::kF3;
                break;
              case 0x23:  // VK_F4
                destItem->key = IKey::kF4;
                break;
              case 0x24:  // VK_F5
                destItem->key = IKey::kF5;
                break;
              case 0x25:  // VK_F6
                destItem->key = IKey::kF6;
                break;
              case 0x26:  // VK_F7
                destItem->key = IKey::kF7;
                break;
              case 0x27:  // VK_F8
                destItem->key = IKey::kF8;
                break;
              case 0x28:  // VK_F9
                destItem->key = IKey::kF9;
                break;
              case 0x29:  // VK_F10
                destItem->key = IKey::kF10;
                break;
              case 0x2A:  // VK_F11
                destItem->key = IKey::kF11;
                break;
              case 0x2B:  // VK_F12
                destItem->key = IKey::kF12;
                break;
              case 0x2C:  // VK_F13
                destItem->key = IKey::kF13;
                break;
              case 0x2D:  // VK_F14
                destItem->key = IKey::kF14;
                break;
              case 0x2E:  // VK_F15
                destItem->key = IKey::kF15;
                break;
              case 0x2F:  // VK_F16
                destItem->key = IKey::kF16;
                break;
              case 0x30:  // VK_F17
                destItem->key = IKey::kF17;
                break;
              case 0x31:  // VK_F18
                destItem->key = IKey::kF18;
                break;
              case 0x32:  // VK_F19
                destItem->key = IKey::kF19;
                break;
              case 0x33:  // VK_F20
                destItem->key = IKey::kF20;
                break;
              case 0x34:  // VK_F21
                destItem->key = IKey::kF21;
                break;
              case 0x35:  // VK_F22
                destItem->key = IKey::kF22;
                break;
              case 0x36:  // VK_F23
                destItem->key = IKey::kF23;
                break;
              case 0x37:  // VK_F24
                destItem->key = IKey::kF24;
                break;
              default:
                destItem->key = sourceItem->key;
                    // Maybe it's a valid keysym?
                break;
           }
        }
   
        if ( sourceItem->options & 0x0004 )  // AF_SCANCODE
        {
           destItem->key = 0;
        }
        if ( sourceItem->options & 0x0008 )  // AF_SHIFT
        {
           destItem->modifiers |= 0x01;
        }
        if ( sourceItem->options & 0x0010 )  // AF_CONTROL
        {
           destItem->modifiers |= 0x04;
        }
        if ( sourceItem->options & 0x0020 )  // AF_ALT
        {
           destItem->modifiers |= 0x08;
           // Determine this flag at runtime?  IC_NOTYET
        }
        if ( sourceItem->options & 0x0040 ); // AF_LONEKEY
        if ( sourceItem->options & 0x0100 )  // AF_SYSCOMMAND
        {
           destItem->flags |= 0x0080;
        }
        if ( sourceItem->options & 0x0200 )  // AF_HELP
        {
           destItem->flags |= 0x0100;
           destItem->commandId = IC_ID_HELP;
        }
   
        sourceItem++;
        destItem++;
     }
  }

  return IAccelTblHandle( (IAccelTblHandle::Value) destTable );
}
