// Revision: 56 1.3.1.3 source/multmed/immwave.cpp, multimedia, ioc.v400, 001006  
/*******************************************************************************
* FILE NAME: immwave.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in immwave.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C"
   {
   #include <string.h>
   #define INCL_MCIOS2
   #define INCL_MMIOOS2
   #define IC_MULTIMEDIA
   #include <immdefs.h>
   }
#include <itrace.hpp>
#include <immwave.hpp>
#include <immexcpt.hpp>
#include <immowin.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMMWAVE_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IMMWaveAudio::IMMWaveAudio                                                   |
| This constructor creates a new device object.                                |
|                                                                              |
------------------------------------------------------------------------------*/
IMMWaveAudio::IMMWaveAudio( bool openNow,
                            unsigned long instance,
                            bool openShareable )
            : IMMConfigurableAudio("WAVEAUDIO", openNow,
                                   instance, openShareable)
#ifdef IC_WIN
              , fDeviceSettings(0)
#endif
{ IMODTRACE_DEVELOP("IMMWaveAudio::ctor"); }

/*------------------------------------------------------------------------------
| IMMWaveAudio::IMMWaveAudio                                                   |
| This constructor wrappers an existing device, and gives a derived class the  |
| ability to specify a desired alias (instead of ICLUI generating it).         |
------------------------------------------------------------------------------*/
IMMWaveAudio::IMMWaveAudio(unsigned long newDeviceId,
                           const IString &newAlias)
            : IMMConfigurableAudio(newDeviceId, newAlias)
#ifdef IC_WIN
              , fDeviceSettings(0)
#endif
{ IMODTRACE_DEVELOP("IMMWaveAudio::ctor #2"); }

/*------------------------------------------------------------------------------
| IMMWaveAudio::~IMMWaveAudio                                                  |
| Virtual destructor - nothing special needed as parent does cleanup.          |
|                                                                              |
------------------------------------------------------------------------------*/
IMMWaveAudio::~IMMWaveAudio()
{ IMODTRACE_DEVELOP("IMMWaveAudio::dtor"); }

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IMMWaveAudio::copyFromBuffer                                                 |
| Copies all of the audio data from the audio buffer to the current            |
| location.                                                                    |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::copyFromBuffer(const IMMAudioBuffer& audioBuffer,
                                           CallType call)
{
  IMODTRACE_DEVELOP("IMMWaveAudio::copyFromBuffer");

  IASSERTSTATE(isOpen());
  IString buf = audioBuffer.data();

  MCI_EDIT_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback   = HWND(deviceWindow().handle());
  parms.ulStructLen    = sizeof(parms);
  parms.pBuff          = PVOID((char*)buf);
  parms.ulBufLen       = audioBuffer.length();
  parms.pHeader        = PVOID((char*)(audioBuffer.headerData()));

  unsigned long flags = MCI_FROM_BUFFER;
  sendCommand(MCI_COPY, flags | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::copyToBuffer                                                   |
| Copies audio data into the audio buffer from the passed in begin             |
| position to the passed in end position.                                      |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::copyToBuffer(IMMAudioBuffer& audioBuffer,
                                         const IMMTime &from,
                                         const IMMTime &to,
                                         CallType call) const
{
  IMODTRACE_DEVELOP("IMMWaveAudio::copyToBuffer");

  IASSERTSTATE(isOpen());

  IString buf(0,audioBuffer.length());
  MCI_EDIT_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback   = HWND(deviceWindow().handle());
  parms.ulStructLen    = sizeof(parms);
  parms.pBuff          = PVOID((char*)buf);
  parms.ulBufLen       = audioBuffer.length();
  parms.pHeader        = PVOID((char*)(audioBuffer.headerData()));

  unsigned long flags = MCI_TO_BUFFER;
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }
  ((IMMWaveAudio*)this)->sendCommand(MCI_COPY, flags | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  audioBuffer.setData((char*)buf,audioBuffer.length());
  return *((IMMWaveAudio*)this);
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::cutToBuffer                                                    |
| Cuts audio data into the audio buffer from the passed in                     |
| start position to the passed in end position.                                |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::cutToBuffer(IMMAudioBuffer& audioBuffer,
                                        const IMMTime &from,
                                        const IMMTime &to,
                                        CallType call)
{
  IMODTRACE_DEVELOP("IMMWaveAudio::cutToBuffer");

  IASSERTSTATE(isOpen());

  IString buf(0,audioBuffer.length());
  MCI_EDIT_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback   = HWND(deviceWindow().handle());
  parms.ulStructLen    = sizeof(parms);
  parms.pBuff          = PVOID((char*)buf);
  parms.ulBufLen       = audioBuffer.length();
  parms.pHeader        = PVOID((char*)(audioBuffer.headerData()));

  unsigned long flags = MCI_TO_BUFFER;
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }

  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }

  sendCommand(MCI_CUT, flags | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  audioBuffer.setData((char*)buf,audioBuffer.length());
  return *this;
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::pasteFromBuffer                                                |
| Replaces the audio data with data from the audio buffer, from the passed     |
| in begin position to the passed in end position.                             |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::pasteFromBuffer(const IMMAudioBuffer& audioBuffer,
                                            const IMMTime &from,
                                            const IMMTime &to,
                                            CallType call)
{
  IMODTRACE_DEVELOP("IMMWaveAudio::pasteFromBuffer");

  IASSERTSTATE(isOpen());

  IString buf = audioBuffer.data();
  MCI_EDIT_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback   = HWND(deviceWindow().handle());
  parms.ulStructLen    = sizeof(parms);
  parms.pBuff          = PVOID((char*)buf);
  parms.ulBufLen       = audioBuffer.length();
  parms.pHeader        = PVOID((char*)(audioBuffer.headerData()));

  unsigned long flags = MCI_FROM_BUFFER;
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }
  sendCommand(MCI_PASTE, flags | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::pasteToBuffer                                                  |
| Pastes audio data into the passed in audio buffer.                           |
|                                                                              |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::pasteToBuffer(IMMAudioBuffer& audioBuffer,
                                          CallType call) const
{
  IMODTRACE_DEVELOP("IMMWaveAudio::pasteToBuffer");

  IASSERTSTATE(isOpen());

  IString buf(0,audioBuffer.length());
  MCI_EDIT_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback   = HWND(deviceWindow().handle());
  parms.ulStructLen    = sizeof(parms);
  parms.pBuff          = PVOID((char*)buf);
  parms.ulBufLen       = audioBuffer.length();
  parms.pHeader        = PVOID((char*)(audioBuffer.headerData()));

  unsigned long flags = MCI_TO_BUFFER;
  ((IMMWaveAudio*)this)->sendCommand(MCI_PASTE, flags | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  audioBuffer.setData((char*)buf,audioBuffer.length());
  return *((IMMWaveAudio*)this);
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::supportsWaveFormat                                             |
| Returns true if the specified format is supported.                           |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMWaveAudio::supportsWaveFormat(unsigned long bitsPerSample,
                                         unsigned long samplesPerSecond,
                                         unsigned long channels,
                                         IMMAudioBuffer::Format tag,
                                         CallType call) const
{
  IMODTRACE_DEVELOP("IMMWaveAudio::supportsWaveFormat");

  IASSERTSTATE(isOpen());

  MCI_WAVE_GETDEVCAPS_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback    = HWND(deviceWindow().handle());
  parms.ulItem          = MCI_GETDEVCAPS_WAVE_FORMAT;
  parms.ulBitsPerSample = bitsPerSample;
  parms.ulFormatTag     = (unsigned long)tag;
  parms.ulSamplesPerSec = samplesPerSecond;
  parms.ulChannels      = channels;
  ((IMMWaveAudio*)this)->sendCommand(MCI_GETDEVCAPS,
                  MCI_GETDEVCAPS_ITEM | MCI_GETDEVCAPS_EXTENDED |
                  call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return parms.ulReturn;
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::cutCopyBufferSize                                              |
| Returns the size the audio buffer needs to be to contain the audio           |
| data from the passed in beginning position to the passed in ending position. |
------------------------------------------------------------------------------*/
unsigned long IMMWaveAudio::cutCopyBufferSize(const IMMTime &from,
                                              const IMMTime &to) const
{
  IMODTRACE_DEVELOP("IMMWaveAudio::cutCopyBufferSize");

  IASSERTSTATE(isOpen());

  unsigned long rc = 0;
  MCI_EDIT_PARMS parms;
  char dummy;
  MMAUDIOHEADER header;

  memset(&parms, 0, sizeof(parms));
  memset(&header, 0, sizeof(header));
  parms.ulStructLen = sizeof(parms);

  unsigned long flags = MCI_TO_BUFFER | MCI_WAIT;
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }
  parms.pBuff    = &dummy;
  parms.ulBufLen = sizeof(dummy);
  parms.pHeader  = &header;
  ((IMMWaveAudio*)this)->sendCommand(MCI_COPY, flags, &parms);
  if (!lastError() || lastError() == MCIERR_INVALID_BUFFER)
     rc = parms.ulBufLen;
  else
     ITHROWMMERROR(lastError(),"sendCommand");
  return rc;
}
#endif

/*------------------------------------------------------------------------------
| IMMWaveAudio::isOpenStringValid                                              |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMWaveAudio::isOpenStringValid(const IString &deviceName) const
{
  IMODTRACE_DEVELOP("IMMWaveAudio::isOpenStringValid");

  IString tester(deviceName);
  //Open string should either be a device name valid for this object,
  //or a file name with an extension (verified by existence of a '.')
  //Eventually we should check for a valid extension for this device.
  return((tester.upperCase().indexOf("WAVEAUDIO") == 1 ? true : false) ||
         (tester.upperCase().indexOf(".") ? true : false)
        );
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMWaveAudio::saveDeviceSettings                                                |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::saveDeviceSettings()
{
  IMODTRACE_DEVELOP("IMMWaveAudio::saveDeviceSettings");
  Inherited::saveDeviceSettings();
  return *this;
}

/*------------------------------------------------------------------------------
| IMMWaveAudio::restoreDeviceSettings                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMWaveAudio& IMMWaveAudio::restoreDeviceSettings(bool newRecordMode)
{
  IMODTRACE_DEVELOP("IMMWaveAudio::restoreDeviceSettings");
  Inherited::restoreDeviceSettings();
  return *this;
}

#endif
