// Revision: 39 1.3.1.3 source/multmed/immrecrd.cpp, multimedia, ioc.v400, 001006  
/*******************************************************************************
* FILE NAME: immrecrd.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in immrecrd.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C"
   {
   #include <string.h>
   #define INCL_MCIOS2
   #define IC_MULTIMEDIA
   #include <immdefs.h>
   }
#include <itrace.hpp>
#include <immexcpt.hpp>
#include <immowin.hpp>
#include <immrecrd.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMMRECRD_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IMMRecordable::IMMRecordable                                                 |
| This constructor creates a new device object.                                |
|                                                                              |
------------------------------------------------------------------------------*/
IMMRecordable::IMMRecordable(const IString &baseDeviceName,
                             bool openNow,
                             unsigned long instance,
                             bool shareable)
             : IMMFileMedia(baseDeviceName, openNow, instance, shareable)
#ifdef IC_WIN
               , fDeviceSettings(0)
#endif
{ IMODTRACE_DEVELOP("IMMRecordable::ctor #1"); }

/*------------------------------------------------------------------------------
| IMMRecordable::IMMRecordable                                                 |
| This constructor wrappers an existing device, and gives a derived class the  |
| ability to specify a desired alias (instead of ICLUI generating it).         |
------------------------------------------------------------------------------*/
IMMRecordable::IMMRecordable(unsigned long newDeviceId,
                             const IString &newAlias)
             : IMMFileMedia(newDeviceId, newAlias)
#ifdef IC_WIN
               , fDeviceSettings(0)
#endif
{ IMODTRACE_DEVELOP("IMMRecordable::ctor #2"); }

/*------------------------------------------------------------------------------
| IMMRecordable::~IMMRecordable                                                |
| Destructor                                                                   |          |
|                                                                              |
------------------------------------------------------------------------------*/
IMMRecordable::~IMMRecordable()
{ IMODTRACE_DEVELOP("IMMRecordable::dtor"); }

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IMMRecordable::canRedo                                                       |
| Returns true if the device has any actions to be redone.                     |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMRecordable::canRedo() const
{
  IMODTRACE_DEVELOP("IMMRecordable::canRedo");

  IASSERTSTATE(isOpen());
  bool rc = itemStatus(MCI_STATUS_CAN_REDO);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMRecordable::canUndo                                                       |
| Returns true if the device has any actions to be redone.                     |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMRecordable::canUndo() const
{
  IMODTRACE_DEVELOP("IMMRecordable::canUndo");

  IASSERTSTATE(isOpen());
  bool rc = itemStatus(MCI_STATUS_CAN_UNDO);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMRecordable::redo                                                          |
| Re-does the record, cut, paste, or delete operation most recently undone by  |
| an undo.  The position is at the beginning of the file after a redo.         |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::redo(CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::redo");

  IASSERTSTATE(isOpen());
  MCI_GENERIC_PARMS parms;

  parms.hwndCallback = HWND(deviceWindow().handle());
  sendCommand(MCI_REDO, call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::undo                                                          |
| Un-does the record, cut, paste, or delete operation most recently undone by  |
| an undo.  The position is at the beginning of the file after a redo.         |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::undo(CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::undo");

  IASSERTSTATE(isOpen());
  MCI_GENERIC_PARMS parms;

  parms.hwndCallback = HWND(deviceWindow().handle());
  sendCommand(MCI_UNDO, call, &parms);
  if (lastError() && lastError()!=MCIERR_CANNOT_UNDO)
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

#endif

/*------------------------------------------------------------------------------
| IMMRecordable::copy                                                          |
| Copies data from the passed in start position to the passed in end position  |
| into the clipboard.                                                          |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::copy(const IMMTime &from,
                                   const IMMTime &to,
                                   CallType call) const
{
  IMODTRACE_DEVELOP("IMMRecordable::copy");

  IASSERTSTATE(isOpen());
#ifdef IC_PM
  MCI_EDIT_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_GENERIC_PARMS parms;
#endif
  unsigned long flags = call;

  memset(&parms, 0, sizeof(parms));

#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  ((IMMRecordable*)this)->sendCommand(MCI_COPY, flags, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *((IMMRecordable*)this);
}

/*------------------------------------------------------------------------------
| IMMRecordable::cut                                                           |
| Cuts data from the passed in start position to the passed in end position    |
|                                                                              |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::cut(const IMMTime &from,
                                  const IMMTime &to,
                                  CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::cut");

  IASSERTSTATE(isOpen());
#ifdef IC_PM
  MCI_EDIT_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_GENERIC_PARMS parms;
#endif
  unsigned long flags = call;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  sendCommand(MCI_CUT, flags, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::deleteSelection                                               |
| Deletes data from the passed in start position to the passed in end position.|
|                                                                              |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::deleteSelection(const IMMTime &from,
                                              const IMMTime &to,
                                              CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::deleteSelection");

  IASSERTSTATE(isOpen());
#ifdef IC_PM
  MCI_EDIT_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_GENERIC_PARMS parms;
#endif
  unsigned long flags = call;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  sendCommand(MCI_DELETE, flags, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::paste                                                         |
| Replaces the data from the passed in start position to the passed in end     |
| position with data from the clipboard.                                       |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::paste(const IMMTime &from,
                                    const IMMTime &to,
                                    bool convert,
                                    CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::paste");

  IASSERTSTATE(isOpen());
#ifdef IC_PM
  MCI_EDIT_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_GENERIC_PARMS parms;
#endif
  unsigned long flags = call;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());

  if (from.isValid())
  {
     flags |= MCI_FROM;
     parms.ulFrom = (unsigned long)from;
  }
  if (to.isValid())
  {
     flags |= MCI_TO;
     parms.ulTo = (unsigned long)to;
  }

  if (convert)
     flags |= MCI_CONVERT_FORMAT;
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  sendCommand(MCI_PASTE, flags, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::cueForRecording                                               |
| Cues the device for recording. Not required, but may reduce the delay        |
| associated with recording for some devices.                                  |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::cueForRecording(CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::cueForRecording");

  IASSERTSTATE(isOpen());
  MCI_GENERIC_PARMS parms;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  sendCommand(MCI_CUE,
#ifdef IC_PM
              MCI_CUE_INPUT |
#endif
#ifdef IC_WIN
              MCI_WAVE_INPUT |
#endif
              call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::record                                                        |
| Starts recording at the begin location and continues until it reaches the    |
| end location.                                                                |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::record(bool insert,
                                     const IMMTime &from,
                                     const IMMTime &to,
                                     bool resumeIfPaused,
                                     CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::record");
  ITRACE_DEVELOP("IMMRecordable::record, insert = "+IString(insert));
  ITRACE_DEVELOP("IMMRecordable::record, resumeIfPaused = "+IString(resumeIfPaused));

  IASSERTSTATE(isOpen());
  MCI_RECORD_PARMS parms;
  unsigned long    flags = 0;

  // Following does not take into account the case where a person
  // has pressed pause on an existing file and wishes to begin
  // recording at that point. This would result in PLAY resume,
  // not record resume. Issuing an MCI_RECORD on a device in paused
  // state will begin recording at the paused point, which is the
  // behavior we seek here.
  //if (resumeIfPaused && IMMDevice::paused == mode())
  //   resume(call);
  //else
  //{
     memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
     parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
     parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
     if (from.isValid())
     {
#ifdef IC_PM
        parms.ulFrom = (unsigned long)from;
#endif
#ifdef IC_WIN
        parms.dwFrom = (unsigned long)from;
#endif
        flags       |= MCI_FROM;
     }
     if (to.isValid())
     {
#ifdef IC_PM
        parms.ulTo = (unsigned long)to;
#endif
#ifdef IC_WIN
        parms.dwTo = (unsigned long)to;
#endif
        flags     |= MCI_TO;
     }
     flags |= (insert ? MCI_RECORD_INSERT : MCI_RECORD_OVERWRITE) | call;
     sendCommand(MCI_RECORD, flags, &parms);
  // Following line is the closing bracket for else clause commented out
  // above.
  //}
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::save                                                          |
| Saves the current file.                                                      |
|                                                                              |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::save(CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::save");

  IASSERTSTATE(isOpen());
  MCI_SAVE_PARMS parms;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  sendCommand(MCI_SAVE, call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::saveAs                                                        |
| Saves the current file with a new name, specified by the IString passed in.  |
|                                                                              |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::saveAs(const IString &filename, CallType call)
{
  IMODTRACE_DEVELOP("IMMRecordable::saveAs");

  IASSERTSTATE(isOpen());
  MCI_SAVE_PARMS parms;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.pszFileName  = (PSZ)filename;
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
  parms.lpfilename  = (PSZ)filename;
#endif
  sendCommand(MCI_SAVE, MCI_SAVE_FILE | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMRecordable::saveDeviceSettings                                            |
| Save class-specific settings for the current device, to allow MCI_LOAD       |
| emulation in Win32.                                                          |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::saveDeviceSettings()
{
  IMODTRACE_DEVELOP("IMMRecordable::saveDeviceSettings");
  Inherited::saveDeviceSettings();
  return *this;
}

/*------------------------------------------------------------------------------
| IMMRecordable::restoreDeviceSettings                                             |
| Restore class-specific settings for the current device, to allow MCI_LOAD    |
| emulation in Win32.                                                          |
------------------------------------------------------------------------------*/
IMMRecordable& IMMRecordable::restoreDeviceSettings(bool newRecordMode)
{
  IMODTRACE_DEVELOP("IMMRecordable::restoreDeviceSettings");
  Inherited::restoreDeviceSettings();
  return *this;
}

#endif
