// Revision: 33 1.4.1.3 source/multmed/immplayd.cpp, multimedia, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: immplayd.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in immplayd.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C"
   {
   #include <string.h>
   #define INCL_MCIOS2
   #define INCL_WINMESSAGEMGR
   #define IC_MULTIMEDIA
   #include <immdefs.h>
   }

#include <immplayd.hpp>
#include <immexcpt.hpp>
#include <ithread.hpp>
#include <immowin.hpp>
#include <itrace.hpp>
#include <icconst.h>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMMPLAYD_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IMMPlayableDevice::~IMMPlayableDevice                                        |
| Destructor                                                                   |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice::~IMMPlayableDevice()
{
   IMODTRACE_DEVELOP("IMMPlayableDevice::dtor");

#ifdef IC_WIN
   if (positionTimer.isStarted())
      positionTimer.stop() ;
#endif
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::cueForPlayback                                            |
| Cues the device for playback. Not required, but for some devices it          |
| might reduce the delay associated with the play function.                    |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::cueForPlayback(CallType call)
{
   IMODTRACE_DEVELOP("IMMPlayableDevice::cueForPlayback");

#ifdef IC_WIN
   // Win32 MCI devices that are not ready cannot handle this command.
   if ( IMMDevice::notReady == mode() )
      return *this;
#endif

   IASSERTSTATE(isOpen());

   MCI_GENERIC_PARMS parms;

   memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
   parms.hwndCallback = HWND( deviceWindow().handle());
#endif
#ifdef IC_WIN
   parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
   sendCommand(MCI_CUE,
#ifdef IC_PM
               MCI_CUE_OUTPUT |
#endif
               call, &parms);
   return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::play                                                      |
| Starts playing the device from the passed in start position                  |
| to the passed in end position.                                               |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::play(const IMMTime &from,
                                           const IMMTime &to,
                                           bool resumeIfPaused,
                                           CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::play");

#ifdef IC_WIN
  // Win32 MCI devices that are not ready cannot handle this command.
  if ( IMMDevice::notReady == mode() )
     return *this;
#endif

  IASSERTSTATE(isOpen());

  MCI_PLAY_PARMS parms;
  unsigned long flags = 0;

  if (resumeIfPaused && IMMDevice::paused == mode())
  {
     resume(true,call);
  }
  else
  {
     memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
     parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
     parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
     if (from.isValid())
     {
#ifdef IC_PM
        parms.ulFrom = (unsigned long)from;
#endif
#ifdef IC_WIN
        parms.dwFrom = (unsigned long)from;
#endif
        flags |= MCI_FROM;
     }
     if (to.isValid())
     {
#ifdef IC_PM
        parms.ulTo   = (unsigned long)to;
#endif
#ifdef IC_WIN
        parms.dwTo   = (unsigned long)to;
#endif
        flags |= MCI_TO;
     }

     sendCommand(MCI_PLAY,
                 flags | call,
                 &parms);
  }
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::length                                                    |
| Returns the total length in the current time format.                         |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMPlayableDevice::length(CallType call) const
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::length");

#ifdef IC_WIN
   // Win32 MCI devices that are not ready cannot handle this command.
   if ( IMMDevice::notReady == mode() )
      return 0;
#endif

  unsigned long result = itemStatus(MCI_STATUS_LENGTH, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return result;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::pause                                                     |
| Pauses the device if it is playing.                                          |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::pause( CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::pause");

#ifdef IC_WIN
  // Win32 MCI devices that are not ready cannot handle this command.
  if ( IMMDevice::notReady == mode() )
     return *this;
#endif

  IASSERTSTATE(isOpen());

  MCI_GENERIC_PARMS parms;

#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
  ITRACE_DEVELOP("IMMPlayableDevice::pause, mode = "+IString(mode()));
  // Win32 MCI barfs if a pause is issued to a device that is stopped
  // at end of file, so we detect and avoid this condition.
  if ( IMMDevice::playing != mode() )
     return *this;

  parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
  sendCommand(MCI_PAUSE, call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::position                                                  |
| Returns the current position in the current time format.                     |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMPlayableDevice::position(CallType call) const
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::position");

#ifdef IC_WIN
   // Win32 MCI devices that are not ready cannot handle this command.
   if ( IMMDevice::notReady == mode() )
      return 0;
#endif

  unsigned long result = itemStatus(MCI_STATUS_POSITION, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return result;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::resume                                                    |
| Resumes playback of the device from a paused state.                          |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::resume(bool resume ,
                                             CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::resume");

  IASSERTSTATE(isOpen());
  MCI_GENERIC_PARMS parms;

  if ( IMMDevice::paused == mode() && resume )
  {
#ifdef IC_PM
     parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
     parms.dwCallback = deviceWindow().handle().asUnsigned();
     call = nowait;
#endif
     sendCommand(MCI_RESUME, call, &parms);
     if (lastError())
        ITHROWMMERROR(lastError(),"sendCommand");
  }
  return *this;
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IMMPlayableDevice::removeCuePoint                                            |
| Removes the given cue point. A cue point is a location or time position in a |
| media device.                                                                |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::removeCuePoint(const IMMTime &time, CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::removeCuePoint");
  setCuePoint(time, false, call);
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::addCuePoint                                               |
| Sets a cue point at the specified location.                                  |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice&  IMMPlayableDevice::addCuePoint(const IMMTime &time, CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::addCuePoint");
  setCuePoint(time, true, call);
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::setCuePoint                                               |
| Sets or removes a cue point.                                                 |
|                                                                              |
------------------------------------------------------------------------------*/
void IMMPlayableDevice::setCuePoint(const IMMTime &time, bool on,
                                    CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::setCuePoint");

  IASSERTSTATE(isOpen());

  MCI_CUEPOINT_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.ulCuepoint   = (unsigned long)time;
  parms.usUserParm   = userParameter();
  sendCommand(MCI_SET_CUEPOINT,
              (on ? MCI_SET_CUEPOINT_ON : MCI_SET_CUEPOINT_OFF) | call,
              &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
}
#endif

/*------------------------------------------------------------------------------
| IMMPlayableDevice::stopPositionTracking                                      |
| Stops position tracking. Position tracking causes a position change event to |
| be generated every time the media position changes a given time amount.      |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::stopPositionTracking(CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::stopPostionTracking");

  setPositionAdvise(IMMTime(), false, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::startPositionTracking                                     |
| Start position tracking - generate a change event to every time the media    |
| position changes by the time amount passed in.                               |
------------------------------------------------------------------------------*/
IMMPlayableDevice&  IMMPlayableDevice::startPositionTracking(const IMMTime &time,
                                                             CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::startPostionTracking");

  setPositionAdvise(time, true, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::setPositionAdvise                                         |
| Start or stop position tracking.                                             |
|                                                                              |
------------------------------------------------------------------------------*/
void IMMPlayableDevice::setPositionAdvise(const IMMTime &time,
                                          bool on,
                                          CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::setPositionAdvise");

  IASSERTSTATE(isOpen());

#ifdef IC_PM
  MCI_POSITION_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback = HWND(deviceWindow().handle());
  if (on)
     parms.ulUnits   = (unsigned long)time;
  parms.usUserParm   = userParameter();
  sendCommand(MCI_SET_POSITION_ADVISE,
              (on ? MCI_SET_POSITION_ADVISE_ON :
              MCI_SET_POSITION_ADVISE_OFF) |
              call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
#endif
#ifdef IC_WIN
   // Win32 does not have MCI_SET_POSITION_ADVISE so we create our own timer
   // to do it.
   // NOTE: we return the position every <time interval> and NOT when
   //       the position has changed by <time interval>.
   if (positionTimer.isStarted())
      positionTimer.stop() ;

   if (on) {
      positionTimer.setInterval(MSECFROMMM((unsigned long) time )) ;
      positionTimer.start( new ITimerMemberFn0<IMMPlayableDevice>
                            ( *this, &IMMPlayableDevice::positionTimerChange ) ) ;
   }
#endif
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::stepFrame                                                 |
| Steps the play one or more time units forward or backward.                   |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::stepFrame(unsigned long frames,
                                                bool forward,
                                                CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::stepFrame");

#ifdef IC_WIN
  // Win32 MCI devices that are not ready cannot handle this command.
  if ( IMMDevice::notReady == mode() )
     return *this;
#endif

  IASSERTSTATE(isOpen());

#ifdef IC_PM
  MCI_STEP_PARMS parms;
  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.ulStep       = frames;
  sendCommand(MCI_STEP, MCI_STEP_FRAMES |
              (forward ? 0 : MCI_STEP_REVERSE) | call, &parms);
#endif
#ifdef IC_WIN
  //MCI_VD_STEP_PARMS parms;
  MCI_DGV_STEP_PARMS parms;
  memset(&parms, 0, sizeof(parms));
  parms.dwCallback = deviceWindow().handle().asUnsigned();
  parms.dwFrames     = frames;
  //sendCommand(MCI_STEP, MCI_VD_STEP_FRAMES |
  //            (forward ? 0 : MCI_VD_STEP_REVERSE) | call, &parms);
  sendCommand(MCI_STEP, MCI_DGV_STEP_FRAMES |
              (forward ? 0 : MCI_DGV_STEP_REVERSE) | call, &parms);
#endif
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::stop                                                      |
| Stops playback of the device.                                                |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::stop(CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::stop");

  if ( isOpen() ) {
     MCI_GENERIC_PARMS parms;

     memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
     parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
     // Win32 MCI devices that are not ready cannot handle this command.
     if ( IMMDevice::notReady == mode() )
        return *this;

     parms.dwCallback = deviceWindow().handle().asUnsigned();
#endif
     sendCommand(MCI_STOP, call, &parms);
     if (lastError())
        ITHROWMMERROR(lastError(),"sendCommand");
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::seek                                                      |
| Sets the current position of the device to the passed in position.           |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::seek(const IMMTime &to, CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::seek");

#ifdef IC_WIN
  // Win32 MCI devices that are not ready cannot handle this command.
  if ( IMMDevice::notReady == mode() )
     return *this;
#endif

  IASSERTSTATE(isOpen());

  MCI_SEEK_PARMS parms;

#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.ulTo         = (to.isValid() ? (unsigned long)to : 0);
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
  parms.dwTo         = (to.isValid() ? (unsigned long)to : 0);
#endif
  sendCommand(MCI_SEEK, MCI_TO | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::seekToEnd                                                 |
| Moves the current position to the end of the device's data.                  |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::seekToEnd(CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::seekToEnd");

#ifdef IC_WIN
  // Win32 MCI devices that are not ready cannot handle this command.
  if ( IMMDevice::notReady == mode() )
     return *this;
#endif

  IASSERTSTATE(isOpen());

  MCI_SEEK_PARMS parms;

#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.ulTo         = 0;
  sendCommand(MCI_SEEK, MCI_TO_END | call, &parms);
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
  parms.dwTo         = 0;
  sendCommand(MCI_SEEK, MCI_SEEK_TO_END | call, &parms);
#endif
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::seekToStart                                               |
| Moves the current position to the beginning of the device's data.            |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::seekToStart(CallType call)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::seekToStart");

#ifdef IC_WIN
  // Win32 MCI devices that are not ready cannot handle this command.
  if ( IMMDevice::notReady == mode() )
     return *this;
#endif

  IASSERTSTATE(isOpen());

  MCI_SEEK_PARMS parms;

#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.ulTo         = 0;
  sendCommand(MCI_SEEK, MCI_TO_START | call, &parms);
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
  parms.dwTo         = 0;
  sendCommand(MCI_SEEK, MCI_SEEK_TO_START | call, &parms);
#endif
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::IMMPlayableDevice                                         |
| This constructor creates a new device object.                                |
|                                                                              |
------------------------------------------------------------------------------*/
IMMPlayableDevice::IMMPlayableDevice(const IString &deviceOrFileName,
                                     bool openNow,
                                     unsigned long instance,
                                     bool shareable)
                 : IMMDevice(deviceOrFileName)
#ifdef IC_WIN
                   , positionTimer()
                   , fDeviceSettings(0)
#endif
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::ctor overload 1");

  if (openNow)
  {
     IString fullName(deviceOrFileName);
     if (instance)
        fullName += IString(instance).rightJustify(2, '0');
     openOnThread(fullName, shareable);
  }
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::IMMPlayableDevice                                         |
| This constructor wrappers an existing device, and gives a derived class the  |
| ability to specify a desired alias (instead of ICLUI generating it).         |
------------------------------------------------------------------------------*/
IMMPlayableDevice::IMMPlayableDevice(unsigned long newDeviceId,
                                     const IString &newAlias)
                 : IMMDevice(newDeviceId, newAlias)
#ifdef IC_WIN
                   , positionTimer()
                   , fDeviceSettings(0)
#endif
{ IMODTRACE_DEVELOP("IMMPlayableDevice::ctor overload 2"); }

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMPlayableDevice::positionTimerChange                                       |
| Generate a position changed event, emulating the OS/2 MCI behavior.          |
|                                                                              |
------------------------------------------------------------------------------*/
void IMMPlayableDevice::positionTimerChange()
{
   IMODTRACE_DEVELOP("IMMPlayableDevice::positionTimerChange");

   IMMTime::Format oldFormat = this->timeFormat() ;
   if (oldFormat != IMMTime::milliseconds)
      this->setTimeFormat(IMMTime::milliseconds) ;

   IEventParameter1 mParam(0, (unsigned short) this->deviceId()) ;
   IEventParameter2 lParam(MSECTOMM(this->position())) ;
   (this->deviceWindow()).postEvent(IC_UM_MM_MCI_POSITIONCHANGE,
                                    mParam,
                                    lParam) ;

   if (oldFormat != IMMTime::milliseconds)
      this->setTimeFormat(oldFormat) ;

}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::saveDeviceSettings                                        |
| Save class-specific settings for the current device, to allow MCI_LOAD       |
| emulation in Win32.                                                          |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::saveDeviceSettings()
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::saveDeviceSettings");
  Inherited::saveDeviceSettings();
  return *this;
}

/*------------------------------------------------------------------------------
| IMMPlayableDevice::restoreDeviceSettings                                     |
| Restore class-specific settings for the current device, to allow MCI_LOAD    |
| emulation in Win32.                                                          |
------------------------------------------------------------------------------*/
IMMPlayableDevice& IMMPlayableDevice::restoreDeviceSettings(bool newRecordMode)
{
  IMODTRACE_DEVELOP("IMMPlayableDevice::restoreDeviceSettings");
  Inherited::restoreDeviceSettings();
  return *this;
}

#endif
