// Revision: 26 1.3.1.3 source/multmed/immfilem.cpp, multimedia, ioc.v400, 001006  
/*******************************************************************************
* FILE NAME: immfilem.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in immfilem.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C"
   {
   #include <string.h>
   #define INCL_MCIOS2
   #define INCL_WINATOM
   #define INCL_WINMESSAGEMGR
   #define IC_MULTIMEDIA
   #include <immdefs.h>
   }

#include <immfilem.hpp>
#include <itrace.hpp>
#include <ithread.hpp>
#include <iexcept.hpp>
#include <immexcpt.hpp>
#include <immowin.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMMFILEM_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| IMMFileMedia::IMMFileMedia                                                   |
|                                                                              |
| Note that for Win32, compound devices (ones requiring a data file) cannot be |
| opened shareable, so we force shareable to false.                            |
------------------------------------------------------------------------------*/
IMMFileMedia::IMMFileMedia(const IString &deviceName,
                           bool openNow,
                           unsigned long instance,
                           bool shareable)
#ifdef IC_PM
            : IMMPlayableDevice(deviceName, openNow, instance, shareable),
#endif
#ifdef IC_WIN
            : IMMPlayableDevice(deviceName, openNow, instance, false),
              fDeviceSettings(0),
#endif
              fWriteable(true)
{ IMODTRACE_DEVELOP("IMMFileMedia::ctor #1"); }

/*------------------------------------------------------------------------------
| IMMFileMedia::IMMFileMedia                                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia::IMMFileMedia(unsigned long newDeviceId,
                           const IString &newAlias)
             : IMMPlayableDevice(newDeviceId, newAlias),
               fWriteable(true)
#ifdef IC_WIN
               , fDeviceSettings(0)
#endif
{ IMODTRACE_DEVELOP("IMMFileMedia::ctor #2"); }

/*------------------------------------------------------------------------------
| IMMFileMedia::~IMMFileMedia                                                  |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia::~IMMFileMedia()
{ IMODTRACE_DEVELOP("IMMFileMedia::dtor"); }

/*------------------------------------------------------------------------------
| IMMFileMedia::load                                                           |
|                                                                              |
| For windows, the device open is also performed here (WIN32 compound devices  |
| cannot be opened and loaded at separate times). If the existing device       |
| is found to be open already, it is first closed before the new open is       |
| performed. This emulates the OS2 MCI_LOAD function.                          |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::load(const IString &filename,
                                 bool readOnly, CallType call)
{
  IMODTRACE_DEVELOP("IMMFileMedia::load");

#ifdef IC_PM
  // Can't use for WIN32, since it is ok for device to not yet be open at
  // entry to this member function.
  IASSERTSTATE(isOpen());
  MCI_LOAD_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback   = HWND(deviceWindow().handle());
  parms.pszElementName = (PSZ)filename;
  sendCommand(MCI_LOAD,
              MCI_OPEN_ELEMENT | call | (readOnly ? MCI_READONLY : 0),
              &parms);
#endif
#ifdef IC_WIN
  // Perform the actual file opening here
  if ( isOpen() )
  {
     saveDeviceSettings();
     IMODTRACE_DEVELOP("IMMFileMedia::load, closing existing device to emulate MCI_LOAD");
     close();
  }

  // Perform an open under the covers to simulate the MCI_LOAD
  // Remember that shareable is always false for Win32 compound devices!
  open(filename, false);

  // Even for first time loads, it is possible that some device settings
  // may have been "pre-set". So we restore the state in all cases.
  restoreDeviceSettings();
#endif

  enableDataUpdate(!readOnly);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMFileMedia::isWriteable                                                    |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMFileMedia::isWriteable() const
{
  IMODTRACE_DEVELOP("IMMFileMedia::isWriteable");
  return fWriteable;
}

#if ( IC_OBSOLETE <= IC_OBSOLETE_1 )
/*------------------------------------------------------------------------------
| IMMFileMedia::isReadOnly                                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMFileMedia::isReadOnly() const
{
  IMODTRACE_DEVELOP("IMMFileMedia::isReadOnly");
  return !isWriteable();
}

/*------------------------------------------------------------------------------
| IMMFileMedia::setReadOnlyFlag                                                |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::setReadOnlyFlag(bool rdOnly)
{
  IMODTRACE_DEVELOP("IMMFileMedia::setReadOnlyFlag");
  enableDataUpdate(!rdOnly);
  return *this;
}
#endif

/*------------------------------------------------------------------------------
| IMMFileMedia::enableDataUpdate                                               |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::enableDataUpdate  ( bool update)
{
  IMODTRACE_DEVELOP("IMMFileMedia::enableDataUpdate");
  fWriteable = update;
  return *this;
}


/*------------------------------------------------------------------------------
| IMMFileMedia::loadOnThread                                                   |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::loadOnThread(const IString& filename,
                                         bool readOnly)
{
   IMODTRACE_DEVELOP("IMMFileMedia::loadOnThread");

#ifdef IC_WIN
   ITRACE_DEVELOP("IMMFileMedia::loadOnThread, temporary Win32 bypass--thread problems");
   load(filename, readOnly);
#endif /* IC_WIN */

#ifdef IC_PM
   QMSG qmsg;

   class LoadSyncFn : public IThreadFn
   {
      public:
         LoadSyncFn(IMMFileMedia& dev, const IString& file, bool readOnly)
            : device(dev),
              file(file),
              ro(readOnly)
            {;}
         void run() { device.load(file, ro, IMMDevice::nowait); }

      private:
         IMMFileMedia &device;
         const IString file;
         bool ro;
   };

   // Create our thread function object then construct an IThread
   // with the function object and auto initializing PM
   LoadSyncFn *loadSync = new LoadSyncFn(*this, filename, readOnly);
   IThread thread(loadSync, true);

   while (IGETMSG(IThread::current().anchorBlock(),
                    &qmsg,
                    (HWND)0,
                    0,
                    0))
   {
      IDISPATCHMSG(IThread::current().anchorBlock(), &qmsg);
      if ((HWND)deviceWindow().handle() == qmsg.hwnd)
      {
//#ifdef IC_PM
         IEvent evt(qmsg.hwnd, qmsg.msg, qmsg.mp1, qmsg.mp2);
//#endif
//#ifdef IC_WIN
//         IEvent evt(qmsg.hwnd, qmsg.message, (unsigned long)qmsg.wParam, (unsigned long)qmsg.lParam);
//#endif
         IMMNotifyEvent nEvt(evt);
         if (IMMNotifyEvent::load == nEvt.command())
         {
            setLastError(nEvt.errorId());
            break;
         }
      }
  }
#endif /* IC_PM */

  return *this;
}

/*------------------------------------------------------------------------------
| IMMFileMedia::filename                                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IString IMMFileMedia::filename(CallType call) const
{
  IMODTRACE_DEVELOP("IMMFileMedia::filename");

  IASSERTSTATE(isOpen());
  MCI_INFO_PARMS parms;
  IString        result;
  char           fileName[CCHMAXPATH + 1];

  memset(&parms, 0, sizeof(parms));
  memset(fileName, 0, sizeof(fileName));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.pszReturn    = (PSZ)fileName;
  parms.ulRetSize    = sizeof(fileName);
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned();
  parms.lpstrReturn    = (PSZ)fileName;
  parms.dwRetSize    = sizeof(fileName);
#endif
  ((IMMFileMedia*)this)->sendCommand(MCI_INFO, MCI_INFO_FILE |
                                     call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  else
     result = fileName;
  return result;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMFileMedia::open                                                           |
| This function forces shareable to false (as Win32 compound devices cannot be |
| opened shareable) before calling IMMDevice's play function.                  |
| If this override was not present, open calls to a compound device with       |
| shareable = true would generate an unnecessary exception.                    |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::open(const IString &fileOrDevice,
                                       bool shareable,
                                       CallType call)
{
   IMODTRACE_DEVELOP("IMMFileMedia::open #1");
   return (IMMFileMedia&) IMMDevice::open(fileOrDevice,
                                          false,        // force to false
                                          call) ;
} // open

/*------------------------------------------------------------------------------
| IMMFileMedia::openOnThread                                                   |
| This function forces shareable to false (as Win32 compound devices cannot be |
| opened shareable) before calling IMMDevice's play function.                  |
| If this override was not present, open calls to a compound device with       |
| shareable = true would generate an unncesessary exception.                   |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::openOnThread(const IString &fileOrDevice,
                                         bool shareable,
                                         CallType call)
{
   IMODTRACE_DEVELOP("IMMFileMedia::openOnThread #1");
   return (IMMFileMedia&) IMMDevice::openOnThread(fileOrDevice,
                                                  false,     // force to false
                                                  call) ;
} // openOnThread

/*------------------------------------------------------------------------------
| IMMFileMedia::open                                                           |
| This function forces shareable to false (as Win32 compound devices cannot be |
| opened shareable) before calling IMMDevice's play function.                  |
| If this override was not present, open calls to a compound device with       |
| shareable = true would generate an unncesessary exception.                   |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::open(unsigned long instanceNumber,
                                 bool shareable,
                                 CallType call)
{
   IMODTRACE_DEVELOP("IMMFileMedia::open #2");
   return (IMMFileMedia&) IMMDevice::open(instanceNumber,
                                          false,        // force to false
                                          call) ;
} // open

/*------------------------------------------------------------------------------
| IMMFileMedia::openOnThread                                                   |
| This function forces shareable to false (as Win32 compound devices cannot be |
| opened shareable) before calling IMMDevice's play function.                  |
| If this override was not present, open calls to a compound device with       |
| shareable = true would generate an unncesessary exception.                   |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::openOnThread(unsigned long instanceNumber,
                                         bool shareable)
{
   IMODTRACE_DEVELOP("IMMFileMedia::openOnThread #2");
   return (IMMFileMedia&) IMMDevice::openOnThread(instanceNumber,
                                                  false);    // force to false
} // openOnThread

/*------------------------------------------------------------------------------
| IMMFileMedia::saveDeviceSettings                                                |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::saveDeviceSettings()
{
  IMODTRACE_DEVELOP("IMMFileMedia::saveDeviceSettings");
  Inherited::saveDeviceSettings();
  return *this;
}

/*------------------------------------------------------------------------------
| IMMFileMedia::restoreDeviceSettings                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMFileMedia& IMMFileMedia::restoreDeviceSettings(bool newRecordMode)
{
  IMODTRACE_DEVELOP("IMMFileMedia::restoreDeviceSettings");
  Inherited::restoreDeviceSettings();
  return *this;
}

#endif
