// Revision: 18 1.5.1.6 source/multmed/immdigvd.cpp, multimedia, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: immdigvd.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in immdigvd.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C"
   {
   #include <string.h>
   #define INCL_MCIOS2
   #define IC_MULTIMEDIA
   #include <immdefs.h>
#ifdef IC_WIN
   #include <digitalv.h>
#endif
   }

#include <immdigvd.hpp>
#include <itrace.hpp>
#include <immtime.hpp>
#include <iexcept.hpp>
#include <immexcpt.hpp>
#include <immowin.hpp>
#include <icoordsy.hpp>
#include <iframe.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMMDIGVD_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_WIN
class IMMDVDeviceSettings
{
public:
  IMMDVDeviceSettings();
  ~IMMDVDeviceSettings();

  IRectangle
   myDestinationRect;
IWindowHandle
   myWindowHandle;
bool
   myDestinationRectTouched,
   myWindowHandleTouched,
   useDefaultWindow,
   useDefaultWindowTouched;
};

//JAD icl2_7809
IMMDVDeviceSettings::IMMDVDeviceSettings()
     : myWindowHandleTouched(false),
       useDefaultWindowTouched(false),
       myDestinationRectTouched(false)
     {
       IMODTRACE_DEVELOP("IMMDigitalVideo::IMMDVDeviceSettings CTOR") ;
     }

IMMDVDeviceSettings::~IMMDVDeviceSettings() {}
#endif

/*------------------------------------------------------------------------------
| IMMDigitalVideo::IMMDigitalVideo                                             |
| This constructor creates a new device object.                                |
|                                                                              |
| Note that for Win32, if a compound device (one requiring a data file) is     |
| opened without specifying a "device element name", the device will open, but |
| MCI_LOAD is not supported. Because of this, we defer the actual open until   |
| IMMFileMedia::load() is called by forcing openNow to false.                  |
------------------------------------------------------------------------------*/
IMMDigitalVideo::IMMDigitalVideo( bool openNow,
                                  unsigned long instance,
                                  bool openShareable)
#ifdef IC_PM
               : IMMConfigurableAudio("DIGITALVIDEO", openNow,
                                      instance, openShareable)
#endif
#ifdef IC_WIN
               : IMMConfigurableAudio("AVIVIDEO",  false,
                                      instance, openShareable)
                 , fDeviceSettings( new IMMDVDeviceSettings() )
#endif
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::IMMDigitalVideo(boolean, unsigned long, boolean)") ;
}


/*------------------------------------------------------------------------------
| IMMDigitalVideo::IMMDigitalVideo                                             |
| This constructor wrappers an existing device, and gives a derived class the  |
| ability to specify a desired alias (instead of ICLUI generating it).         |
------------------------------------------------------------------------------*/
IMMDigitalVideo::IMMDigitalVideo(unsigned long newDeviceId,
                                 const IString &newAlias)
               : IMMConfigurableAudio(newDeviceId, newAlias)
#ifdef IC_WIN
                 , fDeviceSettings( new IMMDVDeviceSettings() )
#endif
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::IMMDigitalVideo(boolean, unsigned long, boolean)") ;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::~IMMDigitalVideo                                            |
| Virtual destructor - nothing special needed as parent does cleanup.          |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo::~IMMDigitalVideo()
{
#ifdef IC_WIN           // defect: 29610
  if( fDeviceSettings )
  {
    delete fDeviceSettings;
    fDeviceSettings = 0;
  }
#endif                  // defect: 29610
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::supportsSizing                                              |
| Returns true if the device can independently stretch the horizontal          |
| and vertical dimensions of the image.                                        |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::supportsSizing(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::supportsSizing") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  bool rc = itemCapability(MCI_DGV_GETDEVCAPS_CAN_DISTORT, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
#endif
#ifdef IC_WIN
  return false ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::supportsReverse                                             |
| Returns true if the device can play in reverse.                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::supportsReverse(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::supportsReverse") ;
  IASSERTSTATE(isOpen());
  bool rc = itemCapability(MCI_DGV_GETDEVCAPS_CAN_REVERSE, call); // same call whether OS/2 or WIN
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::supportsStretchToFit                                        |
| Returns true if the device can stretch the video frames to fill the          |
| display rectangle.                                                           |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::supportsStretchToFit(CallType call) const
{
  IMODTRACE_DEVELOP("IMMdigitalVideo::supportsStrechToFit") ;
  IASSERTSTATE(isOpen());
  bool rc = itemCapability(MCI_DGV_GETDEVCAPS_CAN_STRETCH, call); // same call whether OS/2 or WIN
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::maximumSpeed                                                |
| Returns the device's maximum playback rate in the current speed format,      |
| either as a percentage or in frames-per-second.                              |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::maximumSpeed(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::maximumSpeed") ;
  IASSERTSTATE(isOpen());

  // in Win32, always comes back in framesPerSecond format
  IMMSpeed::Format speedFormat(IMMSpeed::framesPerSecond) ;

  // same call whether OS/2 or WIN
  unsigned long temp = itemCapability(MCI_DGV_GETDEVCAPS_MAXIMUM_RATE, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return IMMSpeed(speedFormat,temp);
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::minimumSpeed                                                |
| Returns the device's minimum playback rate in the current speed format,      |
| either as a percentage or in frames-per-second.                              |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::minimumSpeed(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::minimumSpeed") ;
  IASSERTSTATE(isOpen());

    // in Win32, always comes back in framesPerSecond format
    IMMSpeed::Format speedFormat(IMMSpeed::framesPerSecond) ;

    // same call whether OS/2 or WIN
  unsigned long temp = itemCapability(MCI_DGV_GETDEVCAPS_MINIMUM_RATE, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return IMMSpeed(speedFormat,temp);
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::normalSpeed                                                 |
| Returns the device's normal play rate, in the                                |
| current speed format, either as a percentage or in frames-per-second.        |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::normalSpeed(CallType call)  const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::normalSpeed") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  unsigned long temp = itemCapability(MCI_DGV_GETDEVCAPS_NORMAL_RATE, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return IMMSpeed(speedFormat(),temp);
#endif
#ifdef IC_WIN
  return fileNormalSpeed() ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::videoWidth                                                  |
| Returns the nominal (reasonable) width of the video for this machine.        |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::videoWidth(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::videoWidth") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  unsigned long rc = itemCapability(MCI_DGV_GETDEVCAPS_VIDEO_X_EXTENT, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
#endif
#ifdef IC_WIN
  return videoFileWidth(call) ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::videoHeight                                                 |
| Returns the nominal (reasonable) height of the video for this machine.       |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::videoHeight(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::videoHeight") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  unsigned long rc = itemCapability(MCI_DGV_GETDEVCAPS_VIDEO_Y_EXTENT, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
#endif
#ifdef IC_WIN
  return videoFileHeight(call) ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::maximumWindows                                              |
| Returns the maximum number of video windows allowed on this machine.         |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::maximumWindows(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::maximumWindows") ;
  IASSERTSTATE(isOpen());
  // same call whether OS/2 or WIN
  unsigned long rc = itemCapability(MCI_DGV_GETDEVCAPS_MAX_WINDOWS, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::record                                                      |
| Records from start to the end location.  If there was anything               |
| previously recorded it is deleted before the recording.                      |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::record(bool insert,
                                         const IMMTime &from,
                                         const IMMTime &to,
                                         bool resumeIfPaused,
                                         CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::record(insert,from,to,resumeIfPaused,calltype") ;
  IASSERTSTATE(isOpen());
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::record                                                      |
| Records from current to the end location.  If there was anything             |
| previously recorded it is deleted before the recording.                      |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::record(const IMMTime &to,
                                         bool resumeIfPaused,
                                         CallType call)
{
  // TTD
  // recording into streams ????
  // TTD

  IMODTRACE_DEVELOP("IMMDigitalVideo::record(to,resumeIfPaused,calltype") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  MCI_RECORD_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_DGV_RECORD_PARMS parms ;
#endif
  unsigned long    flags = 0;
  bool bToTimeValid = to.isValid() ;

  if (resumeIfPaused && IMMDevice::paused == mode()) {
     resume(call);
  }
  else {
     memset(&parms, 0, sizeof(parms));

#ifdef IC_PM
     parms.hwndCallback = HWND(deviceWindow().handle());
     if (bToTimeValid)
        parms.ulTo = (unsigned long)to;
#endif
#ifdef IC_WIN
     parms.dwCallback = deviceWindow().handle().asUnsigned() ;
     if (bToTimeValid)
        parms.dwTo = (unsigned long) to ;
#endif

     if (bToTimeValid)
        flags     |= MCI_TO;

     flags |=  call;

     sendCommand(MCI_RECORD, flags, &parms);
     if (lastError())
        ITHROWMMERROR(lastError(),"sendCommand");
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::fileNormalSpeed                                             |
| Returns the normal play rate of the currently loaded video file, in the      |
| current speed format, either as a percentage or in frames-per-second.        |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::fileNormalSpeed(CallType call) const
{
  // TTD
  // do i need to use _DGV_FRAME_RATE as well ??
  // TTD
  IMODTRACE_DEVELOP("IMMDigitalVideo::fileNormalSpeed") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  unsigned long temp = itemStatus(MCI_DGV_STATUS_NORMAL_RATE, 0, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return IMMSpeed(speedFormat(),temp);
#endif
#ifdef IC_WIN
  MCI_STATUS_PARMS mciStatusParms ;
  unsigned long ulMCIFlags = 0 ;

  // for Win32, we need to call sendCommand which is not const ! so, we ,
  // have to cast away the constness of IMMDigitalVideo for a bit
  IMMDigitalVideo * const nonConstThis = (IMMDigitalVideo * const) this ;

  memset(&mciStatusParms, 0, sizeof(mciStatusParms)) ;

  mciStatusParms.dwCallback = deviceWindow().handle().asUnsigned() ;
  mciStatusParms.dwItem = MCI_DGV_STATUS_FRAME_RATE ;
  ulMCIFlags = MCI_STATUS_ITEM | MCI_DGV_STATUS_NOMINAL | call ;

  nonConstThis->sendCommand(MCI_STATUS,
              ulMCIFlags,
              &mciStatusParms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");

  return IMMSpeed(IMMSpeed::framesPerSecond,        // in Win32, get back frames per seconds * 1000
                  (mciStatusParms.dwReturn)/1000);
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::supportsOverlayGraphics                                     |
| Returns true if the device supports the display by an application            |
| of overlay graphics in a video window.                                       |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::supportsOverlayGraphics(CallType call) const
{
  // TTD
  // check on this re video blaster ...
  // TTD
  IMODTRACE_DEVELOP("IMMDigitalVideo::supportsOverlayGraphics") ;
#ifdef IC_WIN
  return false ;
#endif
#ifdef IC_PM
  IASSERTSTATE(isOpen());
  bool rc = itemCapability(MCI_DGV_GETDEVCAPS_OVERLAY_GRAPHICS, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return rc;
#endif
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMDigitalVideo::play                                                        |
| This function is here because we need a way of resetting the play speed      |
| to nominal. Calls to playFast or playSlow may have set the speed to          |
| to some multiple of the normal frame rate. And that speed is NOT reset       |
| when the function ends.                                                      |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::play (const IMMTime& from,
                                        const IMMTime& to,
                                        bool resumeIfPaused,
                                        CallType call)
{
   IMODTRACE_DEVELOP("IMMDigitalVideo::play") ;

   // Win32 MCI devices that are not ready cannot handle this command.
   if ( IMMDevice::notReady == mode() )
      return *this;

   IASSERTSTATE(isOpen());

   setSpeed() ;

   return (IMMDigitalVideo&) IMMPlayableDevice::play(from,
                                                     to,
                                                     resumeIfPaused,
                                                     call) ;
} // play
#endif

/*------------------------------------------------------------------------------
| IMMDigitalVideo::playAt                                                      |
| Plays the video at the specified speed from the start                        |
| position to the end position.                                                |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::playAt(const IMMSpeed& speed,
                                         const IMMTime &from,
                                         const IMMTime &to,
                                         CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::playAt") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  return playDigital(from, to, call, MCI_DGV_PLAY_SPEED, speed.speed());
#endif
#ifdef IC_WIN
  // since we don't have a play at speed in WIN32, fake it out by calculating
  // the speed factor ratio then play
  unsigned long speedFactor=getSpeedFactor(speed) ;
  setSpeed(speedFactor) ;
  return playDigital(from, to, call) ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::playFast                                                    |
| Plays the video at fast speed from the start                                 |
| position to the end position.                                                |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::playFast(const IMMTime &from,
                                           const IMMTime &to,
                                           CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::playFast") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  return playDigital(from, to, call, MCI_DGV_PLAY_FAST);
#endif
#ifdef IC_WIN
  // since WIN32 does not have an MCI play-fast option, we'll fake it
  // out by setting the speed factor to 2X then play
  setSpeed(2000) ;
  return playDigital(from, to, call) ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::playScan                                                    |
| Plays frames when indexed; otherwise, plays the video as fast as             |
| possible from the start position to the end position with no audio.          |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::playScan(const IMMTime &from,
                                           const IMMTime &to,
                                           CallType call)
{
#ifdef IC_PM
  IMODTRACE_DEVELOP("IMMDigitalVideo::playScan") ;
  IASSERTSTATE(isOpen());
  return playDigital(from, to, call, MCI_DGV_PLAY_SCAN);
#endif
#ifdef IC_WIN
// no MCI way of finding out the max speed of a video file ...
// default it to play at 2X speed
  return playFast(from, to, call) ;
#endif
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMDigitalVideo::setSpeed                                                    |
| Win32 only. Sets the play speed as ratio of nominal frame rate.              |
| 1000= normal, 2000=2X normal, 500=1/2 normal                                 |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::setSpeed(const unsigned long speedRatio)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::setSpeed") ;
  IASSERTSTATE(isOpen());
  MCI_DGV_SET_PARMS parms ;
  parms.dwSpeed = speedRatio ;
  sendCommand(MCI_SET, MCI_DGV_SET_SPEED | IMMDevice::wait, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this ;

} // setSpeed

/*------------------------------------------------------------------------------
| IMMDigitalVideo::getSpeedFactor                                              |
| Win32 only. return the given speed as a ratio given                          |
| 1000= normal, 2000=2X normal, 500=1/2 normal                                 |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::getSpeedFactor(const IMMSpeed speed) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::getSpeedFactor") ;
  IASSERTSTATE(isOpen());
  return ( (speed.speed() / fileNormalSpeed().speed() ) * 1000 ) ;
} // getSpeedFactor
#endif

/*------------------------------------------------------------------------------
| IMMDigitalVideo::isPlayingForward                                            |
| Returns true if the play direction is forward or if the                      |
| device is not playing.                                                       |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::isPlayingForward(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::isPlayingForward") ;
  IASSERTSTATE(isOpen());
  bool rc = itemStatus(MCI_DGV_STATUS_FORWARD, 0,call); // same call whether OS/2 or WIN
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::playSlow                                                    |
| Plays the video at slow speed from the start                                 |
| position to the end position.                                                |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::playSlow(const IMMTime &from,
                                           const IMMTime &to,
                                           CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::playSlow") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  return playDigital(from, to, call, MCI_DGV_PLAY_SLOW);
#endif
#ifdef IC_WIN
  // not really support in Win32 MCI so we'll fake it by setting the
  // speed to 1/2 nominal rate first then play
  setSpeed(500) ;
  return playDigital(from, to, call) ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::setWindow                                                   |
| Use this function to supply your own window where the video is played.       |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::setWindow(const IWindowHandle& handle,
                                            CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::setWindow(IWindowHandle&, call)") ;

#ifdef IC_PM
  IASSERTSTATE(isOpen());

  MCI_VID_WINDOW_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_DGV_WINDOW_PARMS parms ;
#endif
  // JJB Need to try a window without cs_sizeredraw and cs_movenotify
  // styles, doc says need to have these styles, if so need to add to
  // the exception list and put comment in functions and class
  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.hwndDest     = handle;
  // TTD previously had _VID_WIN which I believe is a bug ...
  sendCommand(MCI_WINDOW, MCI_DGV_WINDOW_HWND | call, &parms);
#endif
#ifdef IC_WIN
  if ( isOpen()
       && ( deviceName() != "AVIVIDEO" || supportsRecord() ) )
  {
     parms.dwCallback = deviceWindow().handle().asUnsigned() ;
     parms.hWnd       = handle;
     sendCommand(MCI_WINDOW, MCI_DGV_WINDOW_HWND | call, &parms);
  }

  // Save setting for possible future reload
  ITRACE_DEVELOP("IMMDigitalVideo::setWindow, saving for MCI_LOAD emulation.");
  fDeviceSettings->myWindowHandle = handle;
  fDeviceSettings->myWindowHandleTouched = true;
  fDeviceSettings->useDefaultWindow = false;
#endif

  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::setWindow                                                   |
| Use this function to supply your own window where the video is played.       |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::setWindow(const IWindow& window,
                                            CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::setWindow(IWindow&, call)") ;
  setWindow(window.handle(),call);
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::useDefaultWindow                                            |
| Specifies that the digital video class creates and manages its own video     |
| window.                                                                      |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::useDefaultWindow(CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::useDefaultWindow") ;

#ifdef IC_PM
  IASSERTSTATE(isOpen());

  MCI_VID_WINDOW_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_DGV_WINDOW_PARMS parms ;
#endif

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  // TTD previously was _VID_ instead of _DGV_ bug ???
  sendCommand(MCI_WINDOW, MCI_DGV_WINDOW_DEFAULT | call, &parms);
#endif

#ifdef IC_WIN
  if ( isOpen()
       && ( deviceName() != "AVIVIDEO" || supportsRecord() ) )
  {
     parms.dwCallback = deviceWindow().handle().asUnsigned() ;
     parms.hWnd = MCI_DGV_WINDOW_DEFAULT;
     sendCommand(MCI_WINDOW, MCI_DGV_WINDOW_HWND | call, &parms);
  }

  // Save setting for possible future reload
  ITRACE_DEVELOP("IMMDigitalVideo::useDefaultWindow, saving for MCI_LOAD emulation.");
  fDeviceSettings->useDefaultWindow = true;
  fDeviceSettings->useDefaultWindowTouched = true;
#endif

  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::handle                                                      |
| Returns the window handle for the current video playback window.             |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowHandle IMMDigitalVideo::handle(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::handle") ;
  IASSERTSTATE(isOpen());
  IWindowHandle rc = (HWND) itemStatus(MCI_DGV_STATUS_HWND, 0,call); // same call whether OS/2 or WIN
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::videoFileWidth                                              |
| Returns the width of the video for the currently loaded video file.          |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::videoFileWidth(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::videoFileWidth") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  unsigned long rc = itemStatus(MCI_DGV_STATUS_VIDEO_X_EXTENT, 0,call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return rc;
#endif
#ifdef IC_WIN
  MCI_DGV_WHERE_PARMS parms ;
  unsigned long ulMCIFlags = 0 ;

  memset(&parms, 0, sizeof(parms));
  parms.dwCallback = deviceWindow().handle().asUnsigned() ;
  ulMCIFlags = MCI_DGV_WHERE_VIDEO |  call ;  // TTD not sure if source or video should be used .MCI_DGV_WHERE_MAX |
                                              // TTD thing is no device supports this right now
  ((IMMDigitalVideo*)this)->sendCommand(MCI_WHERE,
                  ulMCIFlags,
                  &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return parms.rc.right;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::videoFileHeight                                             |
| Returns the height of the video for the currently loaded video file.         |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::videoFileHeight(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::videoFileHeight") ;
  IASSERTSTATE(isOpen());
#ifdef IC_PM
  unsigned long rc = itemStatus(MCI_DGV_STATUS_VIDEO_Y_EXTENT,0, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
  return rc;
#endif
#ifdef IC_WIN
  MCI_DGV_WHERE_PARMS parms ;
  unsigned long ulMCIFlags = 0 ;

  memset(&parms, 0, sizeof(parms));
  parms.dwCallback = deviceWindow().handle().asUnsigned() ;
  ulMCIFlags = MCI_DGV_WHERE_VIDEO |  call ; // TTD  | MCI_DGV_WHERE_MAX |
  ((IMMDigitalVideo*)this)->sendCommand(MCI_WHERE,
                  ulMCIFlags,
                  &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return parms.rc.bottom ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::speed                                                       |
| Returns the currently set speed for the device.                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::speed(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::speed") ;
  IASSERTSTATE(isOpen());
  unsigned long temp = itemStatus(MCI_DGV_STATUS_SPEED, 0,call); // same call whether OS/2 or WIN
  if (lastError())
     ITHROWMMERROR(lastError(),"itemStatus");
#ifdef IC_PM
  return IMMSpeed(speedFormat(),temp);
#endif
#ifdef IC_WIN
  // Win32 returns the speed as a relative scale ( 500 for half speed, 1000 for normal, etc .. )
  // so we need to convert that into an absolute value calculated from the nominal speed
  return IMMSpeed(IMMSpeed::framesPerSecond,
                  (int) (fileNormalSpeed().speed() * (float) (temp/1000) )) ;
#endif
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::setDestination                                              |
| Sets the subrectangle within the video window where video is played.  This   |
| allows an application to move, grow, or shrink video inside the window.      |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::setDestination(const IRectangle& rect,
                                                 CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::setDestination") ;

#ifdef IC_PM
  IASSERTSTATE(isOpen());

  MCI_VID_RECT_PARMS parms;  // TTD was _DGV_ bug ???
#endif
#ifdef IC_WIN
  MCI_DGV_PUT_PARMS parms ;

  // Save setting for possible future reload, no matter what the device's state is.
  // Note that we save the rectangle BEFORE coordinate system conversion.
  ITRACE_DEVELOP("IMMDigitalVideo::setDestination, saving for MCI_LOAD emulation.");
  fDeviceSettings->myDestinationRect = rect;
  fDeviceSettings->myDestinationRectTouched = true;

  // This logic essentially translates to:
  //   If the device is not open, OR
  //   If the device is open, but it's device element is not loaded...
  //
  // If this is the case, we have nothing left to do, so return.
  if ( !isOpen() ||
       ( isOpen()
         && deviceName() == "AVIVIDEO"    // file not loaded
         && !supportsRecord() )           // not recordable
     )
     return *this;

  // Same applies if device is not yet open.
  if ( !isOpen() )
     return *this;
#endif
  unsigned long ulMCIFlags = 0 ;
  IRectangle windowRect(rect) ;

  if (ICoordinateSystem::isConversionNeeded()) {
     IWindow *pVideoWindow(IWindow::windowWithHandle(handle())) ;
     ISize refWinSize ;
     if (pVideoWindow) {
        if (pVideoWindow->isFrameWindow()) {
           IFrameWindow * pFrameWindow = (IFrameWindow *) pVideoWindow ;
           pVideoWindow = pFrameWindow->client() ;
        }
        refWinSize = pVideoWindow->size() ;
     }
     else {
        IWindow videoWindow(handle()) ;
        if (videoWindow.isFrameWindow()) {
           IFrameWindow *pFrameWindow = (IFrameWindow *) &videoWindow ;
           refWinSize = (pFrameWindow->client())->size() ;
        }
        else
           refWinSize = videoWindow.size() ;
     }
     windowRect = ICoordinateSystem::convertToNative(rect, refWinSize) ;
  }

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  ulMCIFlags         = MCI_DGV_PUT_DESTINATION | MCI_DGV_PUT_RECT | call ;
  parms.rc           = windowRect.asRECTL();
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned() ;
  ulMCIFlags       = MCI_DGV_RECT | MCI_DGV_PUT_DESTINATION | call ;
  parms.rc.left    = windowRect.minX() ;
  parms.rc.top     = windowRect.minY() ;
  parms.rc.right   = windowRect.width() ;
  parms.rc.bottom  = windowRect.height() ;
#endif

  sendCommand(MCI_PUT, ulMCIFlags, &parms);

  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::sourceRectangle                                             |
| Returns the subrectangle of the video from the currently loaded video        |
| file.                                                                        |
------------------------------------------------------------------------------*/
IRectangle IMMDigitalVideo::sourceRectangle(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::sourceRectangle") ;
  IASSERTSTATE(isOpen());

#ifdef IC_PM
  MCI_VID_RECT_PARMS parms;  // TTD was _DGV_ bug ???
#endif
#ifdef IC_WIN
  MCI_DGV_WHERE_PARMS parms ;
#endif
  unsigned long ulMCIFlags = 0 ;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  ulMCIFlags = MCI_DGV_WHERE_SOURCE | call ; // TTD was _VID_ bug ??
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned() ;
  ulMCIFlags = MCI_DGV_WHERE_VIDEO | call ; // TTD or maybe WHERE_SOURCE ???
#endif
  ((IMMDigitalVideo*)this)->sendCommand(MCI_WHERE,
                  ulMCIFlags,
                  &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");

#ifdef IC_PM
  IRectangle sourceRect(parms.rc) ;
#endif
#ifdef IC_WIN
  IRectangle sourceRect(IPair(parms.rc.left, parms.rc.top),
                      ISize(parms.rc.right, parms.rc.bottom) );
#endif

  if (ICoordinateSystem::isConversionNeeded()) {
     IWindow *pVideoWindow(IWindow::windowWithHandle(handle())) ;
     ISize refWinSize ;
     if (pVideoWindow) {
        if (pVideoWindow->isFrameWindow()) {
           IFrameWindow * pFrameWindow = (IFrameWindow *) pVideoWindow ;
           pVideoWindow = pFrameWindow->client() ;
        }
        refWinSize = pVideoWindow->size() ;
     }
     else {
        IWindow videoWindow(handle()) ;
        if (videoWindow.isFrameWindow()) {
           IFrameWindow *pFrameWindow = (IFrameWindow *) &videoWindow ;
           refWinSize = (pFrameWindow->client())->size() ;
        }
        else
           refWinSize = videoWindow.size() ;
     }
     sourceRect = ICoordinateSystem::convertToApplication(sourceRect, refWinSize) ;
  }

  return sourceRect ;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::destinationRectangle                                        |
| Returns the subrectangle within the video window where video is played.      |
|                                                                              |
------------------------------------------------------------------------------*/
IRectangle IMMDigitalVideo::destinationRectangle(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::destinationRectangle") ;

#ifdef IC_PM
  IASSERTSTATE(isOpen());

  MCI_VID_RECT_PARMS parms;  // TTD was _DGV_ bug ???
#endif
#ifdef IC_WIN
  MCI_DGV_WHERE_PARMS parms ;
  // This logic essentially translates to:
  // If the device is not open, OR
  // If the device is open, but it's device element is not loaded...
  // Note that we return the rectangle BEFORE coordinate system conversion,
  // since it was saved prior to conversion.
  if ( !isOpen() ||
       ( isOpen()
         && deviceName() == "AVIVIDEO"    // file not loaded
         && !supportsRecord() )           // not recordable
     )
  {
     // If the device's setting for destrect has been touched, use
     // this value. Otherwise return a reasonable size default rect
     ITRACE_DEVELOP("IMMDigitalVideo::destinationRectangle, emulating...");
     if (fDeviceSettings->myDestinationRectTouched)
        return fDeviceSettings->myDestinationRect;
     else
        return IRectangle(0,0,100,100);
  }
#endif
  unsigned long ulMCIFlags = 0 ;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned() ;
#endif
  ulMCIFlags = MCI_DGV_WHERE_DESTINATION | call ;  // TTD was _VID_ bug ??
  ((IMMDigitalVideo*)this)->sendCommand(MCI_WHERE,
                  ulMCIFlags,
                  &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");

#ifdef IC_PM
  IRectangle destRect(parms.rc) ;
#endif
#ifdef IC_WIN
  IRectangle destRect(IPair(parms.rc.left, parms.rc.top),
                      ISize(parms.rc.right, parms.rc.bottom) );
#endif

  if (ICoordinateSystem::isConversionNeeded()) {
     IWindow *pVideoWindow(IWindow::windowWithHandle(handle())) ;
     ISize refWinSize ;
     if (pVideoWindow) {
        if (pVideoWindow->isFrameWindow()) {
           IFrameWindow * pFrameWindow = (IFrameWindow *) pVideoWindow ;
           pVideoWindow = pFrameWindow->client() ;
        }
        refWinSize = pVideoWindow->size() ;
     }
     else {
        IWindow videoWindow(handle()) ;
        if (videoWindow.isFrameWindow()) {
           IFrameWindow *pFrameWindow = (IFrameWindow *) &videoWindow ;
           refWinSize = (pFrameWindow->client())->size() ;
        }
        else
           refWinSize = videoWindow.size() ;
     }
     destRect = ICoordinateSystem::convertToApplication(destRect, refWinSize) ;
  }

  return destRect ;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::videoFileName                                               |
| Returns the file name of the digital video file.                             |
|                                                                              |
------------------------------------------------------------------------------*/
IString IMMDigitalVideo::videoFileName() const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::videoFileName") ;
  IASSERTSTATE(isOpen());

#ifdef IC_PM
  MCI_INFO_PARMS parms;
#endif
#ifdef IC_WIN
  MCI_DGV_INFO_PARMS parms ;
#endif
  unsigned long ulMCIFlags ;
  char achResult[512 + 1];

  memset(&parms, 0, sizeof(parms));
  memset(achResult, 0, sizeof(achResult));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.pszReturn    = achResult;
  parms.ulRetSize    = 512;
  ulMCIFlags           = MCI_DGV_INFO_VIDEO_FILE | IMMDevice::wait ;
#endif
#ifdef IC_WIN
  parms.dwCallback  = deviceWindow().handle().asUnsigned() ;
  parms.lpstrReturn = achResult;
  parms.dwRetSize   = 512;
  ulMCIFlags          = MCI_INFO_FILE | IMMDevice::wait ;
#endif
  ((IMMDigitalVideo*)this)->sendCommand(MCI_INFO,
                  ulMCIFlags,
                  &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return IString(achResult);
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::playDigital                                                 |
| Allows the playing from a start to an end position with the specified        |
| flags and speed.                                                             |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::playDigital(const IMMTime &from,
                                              const IMMTime &to,
                                              CallType call,
                                              unsigned long flagsIn,
                                              unsigned long speed)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::playDigital") ;
  IASSERTSTATE(isOpen());

  MCI_DGV_PLAY_PARMS parms;
  unsigned long flags = flagsIn;

  memset(&parms, 0, sizeof(parms));
#ifdef IC_PM
  parms.hwndCallback = HWND(deviceWindow().handle());
#endif
#ifdef IC_WIN
  parms.dwCallback = deviceWindow().handle().asUnsigned() ;
#endif
  if (from.isValid())
  {
#ifdef IC_PM
     parms.ulFrom = (unsigned long)from;
#endif
#ifdef IC_WIN
     parms.dwFrom = (unsigned long)from;
#endif
     flags |= MCI_FROM;
  }
  if (to.isValid())
  {
#ifdef IC_PM
     parms.ulTo = (unsigned long)to;
#endif
#ifdef IC_WIN
     parms.dwTo = (unsigned long)to;
#endif
     flags |= MCI_TO;
  }

#ifdef IC_PM
  parms.ulSpeed = speed;
#endif

  sendCommand(MCI_PLAY, flags | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::isOpenStringValid                                           |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::isOpenStringValid(const IString &deviceName) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::isOpenStringValid") ;
  IString tester(deviceName);
  //Open string should either be a device name valid for this object,
  //or a file name with an extension (verified by existence of a '.')
  //Eventually we should check for a valid extension for this device.
#ifdef IC_PM
  return((tester.upperCase().indexOf("DIGITALVIDEO") == 1 ? true : false) ||
#endif
#ifdef IC_WIN
  return((tester.upperCase().indexOf("AVIVIDEO") == 1 ? true : false) ||
#endif
         (tester.upperCase().indexOf(".") ? true : false)
        );
}

#ifdef IC_PM
/*------------------------------------------------------------------------------
| IMMDigitalVideo::translateAudioFlag                                          |
| Maps the wave audio commands to the digital video command values.            |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMMDigitalVideo::translateAudioFlag(unsigned long flag) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::translateAudioFlag") ;
  unsigned long rc = 0;

  switch (flag)
  {
      case MCI_WAVE_SET_SAMPLESPERSEC:
         rc = MCI_DGV_SET_SAMPLESPERSEC;
         break;

      case MCI_WAVE_SET_AVGBYTESPERSEC:
         rc = MCI_DGV_SET_AVGBYTESPERSEC;
         break;

      case MCI_WAVE_SET_BLOCKALIGN:
         rc = MCI_DGV_SET_BLOCKALIGN;
         break;

      case MCI_WAVE_SET_FORMATTAG:
         rc = MCI_DGV_SET_FORMATTAG;
         break;

      case MCI_WAVE_SET_CHANNELS:
         rc = MCI_DGV_SET_CHANNELS;
         break;

      case MCI_WAVE_SET_BITSPERSAMPLE:
         rc = MCI_DGV_SET_BITSPERSAMPLE;
         break;

      case MCI_WAVE_STATUS_SAMPLESPERSEC:
         rc = MCI_DGV_STATUS_SAMPLESPERSEC;
         break;

      case MCI_WAVE_STATUS_AVGBYTESPERSEC:
         rc = MCI_DGV_STATUS_AVGBYTESPERSEC;
         break;

      case MCI_WAVE_STATUS_BLOCKALIGN:
         rc = MCI_DGV_STATUS_BLOCKALIGN;
         break;

      case MCI_WAVE_STATUS_FORMATTAG:
         rc = MCI_DGV_STATUS_FORMATTAG;
         break;

      case MCI_WAVE_STATUS_CHANNELS:
         rc = MCI_DGV_STATUS_CHANNELS;
         break;

      case MCI_WAVE_STATUS_BITSPERSAMPLE:
         rc = MCI_DGV_STATUS_BITSPERSAMPLE;
         break;
  }
  return rc;
}

// TTD
// need to calculate after getting status_nominal_frame_rate
// TTD
/*------------------------------------------------------------------------------
| IMMDigitalVideo::slowSpeed                                                   |
| Returns the device's slow playback rate in the current speed format,         |
| either as a percentage or in frames-per-second.                              |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::slowSpeed(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::slowSpeed") ;
  IASSERTSTATE(isOpen());
  unsigned long temp = itemCapability(MCI_DGV_GETDEVCAPS_SLOW_RATE, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return IMMSpeed(speedFormat(),temp);
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::fastSpeed                                                   |
| Returns the device's fast playback rate in the current speed format,         |
| either as a percentage or in frames-per-second.                              |
------------------------------------------------------------------------------*/
IMMSpeed IMMDigitalVideo::fastSpeed(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::fastSpeed") ;
  IASSERTSTATE(isOpen());
  unsigned long temp = itemCapability(MCI_DGV_GETDEVCAPS_FAST_RATE, call);
  if (lastError())
     ITHROWMMERROR(lastError(),"itemCapability");
  return IMMSpeed(speedFormat(),temp);
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::isMonitoringEnabled                                         |
| Returns true if monitoring of the incoming video signal is enabled.          |
|                                                                              |
------------------------------------------------------------------------------*/
bool IMMDigitalVideo::isMonitoringEnabled(CallType call) const
{
  // TTD
  // monitor means different things  in WIN. see the doc ???
  // don't have concept of monitor source or file.
  // no monitor window either ...
  // do we need to use _DGV_PARMS structure ?
  // TTD
  IMODTRACE_DEVELOP("IMMDigitalVideo::isMonitoringEnabled") ;
  IASSERTSTATE(isOpen());
  bool rc = itemStatus(MCI_STATUS_MONITOR, 0,call);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return rc;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::enableMonitoring                                            |
| Sets monitoring of the incoming video signal on or off.                      |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::enableMonitoring(bool enable, CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::enableMonitoring") ;
  IASSERTSTATE(isOpen());
  MCI_AMP_SET_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback     = HWND(deviceWindow().handle());
  parms.ulItem           = MCI_DGV_SET_MONITOR;
  sendCommand(MCI_SET,
              MCI_SET_ITEM | (enable ? MCI_SET_ON : MCI_SET_OFF) | call,
              &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::disableMonitoring                                           |
| Sets monitoring of the incoming video signal off.                            |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::disableMonitoring(CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::disableMonitoring") ;
  enableMonitoring(false,call);
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::refresh                                                     |
| Restores the video window by causing the video device to transfer a video    |
| image to the playback window (similar to a refresh of a normal window.       |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::refresh(const IRectangle& source,
                                          const IRectangle& destination,
                                          CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::refresh") ;
  // TTD
  // restore in WIN requires a file containing a bitmap ...
  // not supported for now
  // TTD
  IASSERTSTATE(isOpen());

  MCI_RESTORE_PARMS parms;
  unsigned long flags = 0;
  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback = HWND(deviceWindow().handle());
  if (IPair() != source.area())
  {
     flags |= MCI_RESTORE_SRC_RECT;
     parms.SrcRect = source.asRECTL();
  }

  if (IPair() != destination.area())
  {
     flags |= MCI_RESTORE_DEST_RECT;
     parms.DestRect = destination.asRECTL();
  }
  sendCommand(MCI_RESTORE,
              flags | call,
              &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::setMonitorWindow                                            |
| Use this function to supply your own window for video monitoring.            |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::setMonitorWindow(const IWindowHandle& handle,
                                                   CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::setMonitorWindow(IWindowHandle&,call)") ;
  IASSERTSTATE(isOpen());

  MCI_VID_WINDOW_PARMS parms;
  //JJB  Need to try a window without cs_sizeredraw and cs_movenotify
  //styles, doc says need to have these styles, if so need to add to
  //the exception list and put comment in functions and class
  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback = HWND(deviceWindow().handle());
  parms.hwndDest     = handle;
  sendCommand(MCI_WINDOW, MCI_VID_WINDOW_HWND | MCI_DGV_MONITOR | call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::setMonitorWindow                                            |
| Use this function to supply your own window for video monitoring.            |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::setMonitorWindow(const IWindow& window,
                                                   CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::setMonitorWindow(IWindow&,call)") ;
  IASSERTSTATE(isOpen());
  setMonitorWindow(window.handle(),call);
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::useDefaultMonitorWindow                                     |
| Specifies that the digital video class creates and manages its own video     |
| monitor window.                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::useDefaultMonitorWindow(CallType call)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::useDefaultMonitorWindow") ;
  IASSERTSTATE(isOpen());

  MCI_VID_WINDOW_PARMS parms;

  memset(&parms, 0, sizeof(parms));
  parms.hwndCallback = HWND(deviceWindow().handle());
  sendCommand(MCI_WINDOW, MCI_VID_WINDOW_DEFAULT | MCI_DGV_MONITOR |
              call, &parms);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::monitorHandle                                               |
| Returns the window handle for the current video monitor window.              |
|                                                                              |
------------------------------------------------------------------------------*/
IWindowHandle IMMDigitalVideo::monitorHandle(CallType call) const
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::monitorHandle") ;
  IASSERTSTATE(isOpen());
  IWindowHandle rc = itemStatus(MCI_DGV_STATUS_HWND_MONITOR, 0,call);
  if (lastError())
     ITHROWMMERROR(lastError(),"sendCommand");
  return rc;
}
#endif

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMMDigitalVideo::saveDeviceSettings                                                |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::saveDeviceSettings()
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::saveDeviceSettings");
  Inherited::saveDeviceSettings();
  return *this;
}

/*------------------------------------------------------------------------------
| IMMDigitalVideo::restoreDeviceSettings                                             |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IMMDigitalVideo& IMMDigitalVideo::restoreDeviceSettings(bool newRecordMode)
{
  IMODTRACE_DEVELOP("IMMDigitalVideo::restoreDeviceSettings");
  // Note that we cannot use the getter in this case, since we want
  // to recover the setting of the device as it was previously.
  // On a reload, the device will already be open, and the getter
  // will query the actual state of the device, which would be its default.
  // If the setting had been changed, we would miss it. If the setting had
  // not been previously changed (ie touched), we will leave it as is.
  if ( fDeviceSettings->useDefaultWindowTouched &&
       fDeviceSettings->useDefaultWindow )
  {
     useDefaultWindow();
  }
  else
  {
     if ( fDeviceSettings->myWindowHandleTouched )
        setWindow( fDeviceSettings->myWindowHandle );
  }

  if (fDeviceSettings->myDestinationRectTouched)
     setDestination( fDeviceSettings->myDestinationRect );

  Inherited::restoreDeviceSettings();

  return *this;
}

#endif
