// Revision: 29 1.9.2.1 source/dde/iddetsrv.cpp, dde, ioc.v400, 001006 
/*******************************************************************************
* FILE NAME: iddetsrv.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Definition of the class(es):                                               *
*     IDDETopicServer          - Defines a DDE Server on a single topic.       *
*     IDDEServerHotLinkItem    - Information about an item/format for which    *
*                                there are active hot links.                   *
*     IDDEServerHotLinkItemSet - Set of items having an active hot link.       *
*     IDDEServerHotLink        - Information about an active hot link.         *
*     IDDEServerHotLinkSet     - Set of active hot links.                      *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C"
{
   #define INCL_WINDDE
   #define INCL_DOSQUEUES
   #define INCL_DOSPROCESS
   #define INCL_DOSERRORS
   #define INCL_DOSSEMAPHORES
   #include <iwindefs.h>
   #include <string.h>
   #include <stdio.h>
   #ifdef IC_WIN
      #include <dde.h>            // for Win32 DDE definition
      #include <iplatfrm.hpp>   //mkb for debugging
   #endif
}

#include <ibase.hpp>

#ifdef IC_PMWIN

#include <iddetsrv.hpp>
#include <iddeinfo.hpp>
#include <itrace.hpp>
#include <ievent.hpp>
#include <ihandle.hpp>
#include <iexcept.hpp>
#include <iframe.hpp>
#include <ithread.hpp>
#include <iobjwin.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <iset2.h>
#include <ikeyset2.h>
#include <icconst.h>
#include <ireslock.hpp>
#ifdef IC_WIN
#include <iddecomm.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IDDETSRV_CPP_
  #include <ipagetun.h>
#endif

void    asDebugInfo( IEvent * foo );    //mkb for debugging
void    asDebugInfo( IEvent * foo )
{
   ITRACE_DEVELOP( "DBug: asDebugInfo(IEvent*) msg=" + IString(foo->eventId())
                 + ", wParam=" + IString(foo->parameter1().asUnsignedLong())
                 + ", lParam=" + IString(foo->parameter2().asUnsignedLong()));
}

/*------------------------------------------------------------------------------
| An object of this class is create in the dispatchEventFromQueue function.    |
| This guarantees that when we exit that secondary thread function, (either    |
| because an exception has been thrown, or because we issued a                 |
| DosCloseQueue in the IDDETopicServer dtor), that the DosWaitEvenSem in the   |
| IDDETopicServer dtor will be satisfied and the dtor will safely complete.    |
------------------------------------------------------------------------------*/
#ifdef IC_PM
#pragma pack(push,4)
class IDDEFreeSemaphore {
public:
  IDDEFreeSemaphore ( unsigned long semaphore )
       : ulClSem(semaphore) { }
 ~IDDEFreeSemaphore ( ) {
    DosPostEventSem(ulClSem); }
unsigned long
  ulClSem;
};  // IDDEFreeSemaphore
#pragma pack(pop)
#endif  /* IC_PM */

/*------------------------------------------------------------------------------
| This class holds information about an active hot link.                       |
------------------------------------------------------------------------------*/
#pragma pack(push,4)
class IDDEServerHotLink  {
public:
  IDDEServerHotLink ( IDDEServerHotLinkEvent& evt,
                      unsigned long           convId )
                     : fClPacing(evt.isPacingRequested()),
                       fClDataReq(evt.isDataRequested()),
                       fClUpdateOwed(false),
                       ulClConvId(convId)
                        { }
bool
  isPacingRequested ( ) const
    { return fClPacing; }
bool
  isDataRequested   ( ) const
    { return fClDataReq; }
bool
  isUpdateOwed ( ) const
    { return fClUpdateOwed; }
void
  setUpdateOwed ( bool outStandingAck = true)
    { fClUpdateOwed = outStandingAck; }
unsigned long
  conversationId ( ) const
    { return ulClConvId; }
private:
  IDDEServerHotLink ( const IDDEServerHotLink& aServerLink );
IDDEServerHotLink&
  operator=( const IDDEServerHotLink& aServerLink );
bool
  fClPacing,
  fClDataReq,
  fClUpdateOwed;
unsigned long
  ulClConvId;
}; // IDDEServerHotLink

/*------------------------------------------------------------------------------
| This class is a set that contains pointers to IDDEServerHotLink instances.   |
|                                                                              |
| An instance of this class is created by an IDDETopicServer instance to hold  |
| the set of active hot links on an item/format pair.                          |
------------------------------------------------------------------------------*/
class IDDEServerHotLinkSet : public IVPtrSet<IDDEServerHotLink*> {
public:
  IDDEServerHotLinkSet ( );
  ~IDDEServerHotLinkSet ( );
private:
  IDDEServerHotLinkSet ( const IDDEServerHotLinkSet& aHotLinkSet ) {}
IDDEServerHotLinkSet&
  operator=( const IDDEServerHotLinkSet& aHotLinkSet );
}; // IDDEServerHotLinkSet
#pragma pack(pop)

/*------------------------------------------------------------------------------
| IDdeleteElemSHLS                                                             |
------------------------------------------------------------------------------*/
static bool
  IDdeleteElemSHLS ( IDDEServerHotLink* const& hotLink,
                     void*                     anything )
{
   if (anything == 0 || *((unsigned long*)anything) ==hotLink->conversationId())
   {
      delete (IDDEServerHotLink*)hotLink;
      if ( anything )
      {
         *((unsigned long*)anything) = 0;
      }
      return true;
   }
   return false;
}

/*------------------------------------------------------------------------------
| This class holds information about an item and format that the server has    |
| an active hot link on.                                                       |
|                                                                              |
| Instances of this class are created to store information about each item     |
| and format pair that the server has an active hot link on.  Each instance    |
| also keeps a set of hot links which contain the specifics about the clients  |
| who initiated the hot links.                                                 |
------------------------------------------------------------------------------*/
#pragma pack(push,4)
class IDDEServerHotLinkItem  {
public:
  IDDEServerHotLinkItem  ( IDDEServerHotLinkEvent& evt );
  ~IDDEServerHotLinkItem ( );

IString
  item ( ) const
    { return strClItem; }
IString
  format ( ) const
    { return strClFormat; }
IDDEServerHotLinkSet
 &hotLinks ( ) const
    { return *pHLSetCl; }
private:
  IDDEServerHotLinkItem ( const IDDEServerHotLinkItem& aServerItem );
IDDEServerHotLinkItem&
  operator=( const IDDEServerHotLinkItem& aServerItem );
IString
  strClItem,
  strClFormat;
IDDEServerHotLinkSet
 *pHLSetCl;
}; // IDDEServerHotLinkItem

/*------------------------------------------------------------------------------
| This class is a set created using the ICLCC class library.  The set          |
| contains pointers to IDDEServerHotLinkItem instances.                        |
------------------------------------------------------------------------------*/
class IDDEServerHotLinkItemSet : public IVPtrSet<IDDEServerHotLinkItem*> {
public:
  IDDEServerHotLinkItemSet ( );
  ~IDDEServerHotLinkItemSet ( );
IPrivateResource
 &semaphor()
    { return priResCl; }
private:
  IDDEServerHotLinkItemSet ( const IDDEServerHotLinkItemSet& aServerItemSet ) {}
IDDEServerHotLinkItemSet&
  operator=( const IDDEServerHotLinkItemSet& aServerItemSet );
IPrivateResource
  priResCl;
}; // IDDEServerHotLinkItemSet

class IDDEServerConversationSet :
  public IKeySet<IDDEServerConversation*, unsigned long> {
public:
  IDDEServerConversationSet ( );
  ~IDDEServerConversationSet ( );
IPrivateResource
 &semaphor()
    { return priResCl; }
private:
  IDDEServerConversationSet ( const IDDEServerConversationSet& ) {}
  IDDEServerConversationSet& operator= ( const IDDEServerConversationSet& );
IPrivateResource
  priResCl;
}; // IDDEServerConversationSet
#pragma pack(pop)

/*------------------------------------------------------------------------------
| IDdeleteElemSCS                                                              |
|                                                                              |
| Terminate conversation with the client and delete the server conversation.   |
------------------------------------------------------------------------------*/
static bool
  IDdeleteElemSCS ( IDDEServerConversation* const& pdde,
                    void*                          pvoid )
{
#ifdef IC_PM
   bool bSuccess = WinDdePostMsg((HWND)pdde->conversationId(),
                               (HWND)((IDDETopicServer*)pvoid)->serverHandle(),
                               WM_DDE_TERMINATE,
                               0,
                               DDEPM_RETRY);
#endif  /* IC_PM */
#ifdef IC_WIN
   bool bSuccess = PostMessage( (HWND)pdde->conversationId(),
                                   WM_DDE_TERMINATE,
                                   (WPARAM)(HWND)((IDDETopicServer*)pvoid)->serverHandle(),
                                   0L );
#endif  /* IC_PM */
   if (!bSuccess)
      ITRACE_DEVELOP(
         "WM_DDE_TERMINATE failure on attempt to initate end of conversation.");
   delete (IDDEServerConversation*)pdde;
   return true;
}

/*------------------------------------------------------------------------------
| IDdeleteElemSCCS                                                             |
|                                                                              |
| Delete the terminated conversation. Client conversation did not send the     |
| terminate message back to server.                                            |
------------------------------------------------------------------------------*/
static bool
  IDdeleteElemSCCS ( IDDEServerConversation* const& pdde,
                     void*                          pvoid )
{
   ITRACE_DEVELOP(
      "Deleting terminated conversation, client never sent WM_DDE_TERMINATE");
   delete (IDDEServerConversation*)pdde;
   return true;
}

/*------------------------------------------------------------------------------
| IDdeleteElemSHLIS                                                            |
------------------------------------------------------------------------------*/
static bool
  IDdeleteElemSHLIS ( IDDEServerHotLinkItem* const& hotLinkItem,
                     void*                         anything )
{
   if ( anything == 0 || !(hotLinkItem->hotLinks().numberOfElements()) )
   {
      delete (IDDEServerHotLinkItem*)hotLinkItem;
      return true;
   }
   return false;
}

// define static data
static unsigned long IDDETopicServer__frmUseCntCl = 0;
static IFrameWindow* IDDETopicServer__pwndClFrame = 0;

static IDDEStatics IDDEStatics;

IPrivateResource* IDDEStatics :: pPriResCl = 0;

/*------------------------------------------------------------------------------
| IDDETopicServer::IDDETopicServer                                             |
|                                                                              |
| Constructor for DDE Topic Server. The server can use threads if desired by   |
| setting useEventThread true. If threads are used then events are dispatched  |
| via a queue so also create the queue.                                        |
------------------------------------------------------------------------------*/
IDDETopicServer :: IDDETopicServer ( const char*   applicationName,
                                     const char*   supportedTopic,
                                     IFrameWindow* owner,
                                     bool       useEventThread )
                    : strClApplication(applicationName),
                      strClTopic(supportedTopic),
                      fClPostMsgFail(false),
                      fClHdrActive(false),
                      pThreadCl(0)
#ifdef   IC_WIN
                    , hFinish(0)
#endif
{
   IMODTRACE_DEVELOP("DDETopServ::Ctor");
   IASSERTPARM(applicationName != 0 && supportedTopic != 0);
   IASSERTPARM(applicationName[0] != '\0' && supportedTopic[0] != '\0');
   //  create the frame window, an exception may be thrown here
   {
      IResourceLock frameLock(IDDEStatics::semaphor());
      if (IDDETopicServer__pwndClFrame == 0)
         IDDETopicServer__pwndClFrame =
             new IFrameWindow(IResourceId(999),
                              0,
                              owner,
                              IRectangle(),
                              IWindow::noStyle );
      // increment reference count
      IDDETopicServer__frmUseCntCl++;
   }

#ifdef IC_WIN
   useEventThread = false;  /* now in W32 platform, multithread doesn't work */
#endif

   if (useEventThread)
   {
#ifdef IC_PM
      PPIB ppib;
      PTIB ptib;
      unsigned long ulRc = DosCreateEventSem(0,
                                             &ulClSemaphore,
                                             0,
                                             0);
      if (ulRc) {
         {
            IResourceLock frameLock(IDDEStatics::semaphor());
            IDDETopicServer__frmUseCntCl--;
            if (!IDDETopicServer__frmUseCntCl)
            {
               delete IDDETopicServer__pwndClFrame;
               IDDETopicServer__pwndClFrame = 0;
            }
         }
         ITHROWSYSTEMERROR(ulRc,"DosCreateEventSem",
             IBaseErrorInfo::outOfSystemResource, IException::recoverable);
      }
      // ensure we have a unique queue handle by using PID
      DosGetInfoBlocks( &ptib, &ppib );
      IString tmp1((unsigned long)this);
      IString tmp2(ppib->pib_ulpid);
      IString strQueueName = IString("\\QUEUES\\") + tmp1 + IString("\\") +
         tmp2 + IString("\\.QUE");
      ITRACE_DEVELOP("Queue name is: " + strQueueName);
      ulRc = DosCreateQueue(&ulClQHandle,
                            0,
                           (char*)strQueueName);
      ITRACE_DEVELOP("DosCreateQueue rc =: " + IString(ulRc));
      if (ulRc) {
         {
            IResourceLock frameLock(IDDEStatics::semaphor());
            IDDETopicServer__frmUseCntCl--;
            if (!IDDETopicServer__frmUseCntCl)
            {
               delete IDDETopicServer__pwndClFrame;
               IDDETopicServer__pwndClFrame = 0;
            }
         }
         ITHROWSYSTEMERROR(ulRc,"DosCreateQueue",
             IBaseErrorInfo::outOfSystemResource, IException::recoverable);
      }
      IThreadMemberFn<IDDETopicServer> *myDispatcherFn = new
          IThreadMemberFn<IDDETopicServer>(*this,
                                           &IDDETopicServer::dispatchEventFromQueue);
      pThreadCl = new IThread(myDispatcherFn);
#endif  /* IC_PM */

#ifdef IC_WIN
   DWORD ulRc;

   hFinish = CreateEvent(NULL, TRUE, FALSE, NULL);

   if ( !CreatePipe(&hReadPipe, &hWritePipe, NULL, 0)) {
       ulRc = GetLastError();
       {
          IResourceLock frameLock(IDDEStatics::semaphor());
          IDDETopicServer__frmUseCntCl--;
          if (!IDDETopicServer__frmUseCntCl)
          {
             delete IDDETopicServer__pwndClFrame;
             IDDETopicServer__pwndClFrame = 0;
          }
       }
       ITHROWSYSTEMERROR(ulRc,"CreatPipe",
           IBaseErrorInfo::outOfSystemResource, IException::recoverable);
   }

      IThreadMemberFn<IDDETopicServer> *myDispatcherFn = new
          IThreadMemberFn<IDDETopicServer>(*this,
                                           &IDDETopicServer::dispatchEventFromQueue);
   pThreadCl = new IThread(myDispatcherFn, IThread::defaultAutoInitGUI());
#endif  /* IC_WIN */
   }

   handleEventsFor(IDDETopicServer__pwndClFrame);
   pwndClServer = new IObjectWindow(IObjectWindow::noStyle);
   handleEventsFor((IWindow*)pwndClServer);
   pConvSetCl = new IDDEServerConversationSet;
   pClsdConvSetCl = new IDDEServerConversationSet;
   pHLItemSetCl = new IDDEServerHotLinkItemSet;
#ifdef IC_PM
   pFormatSetCl = new IDDEFormatSet;
#endif
#ifdef IC_WIN
   pItemAtomSetCl = new IDDEItemAtomSet;
#endif
   wndhClServer = pwndClServer->handle();
}

/*------------------------------------------------------------------------------
| IDDETopicServer::~IDDETopicServer                                            |
|                                                                              |
| When destructing a topic server also remove all conversations, closed        |
| conversation, and hot links. If threads were used then close the queue and   |
| stop the thread.                                                             |
------------------------------------------------------------------------------*/
IDDETopicServer :: ~IDDETopicServer ( )
{
   IMODTRACE_DEVELOP("DDETopServ::Dtor");

   // end all conversations, remove & delete all elements in sets, and
   // delete sets
   if (pThreadCl)
   {
#ifdef IC_PM
   // wait for the secondary thread to finish
      unsigned long ulPostCount;
      unsigned long ulRc = DosCloseQueue(queueHandle());
      ITRACE_DEVELOP("DosCloseQueue rc =: " + IString(ulRc));
      ulRc = DosWaitEventSem(ulClSemaphore,
                             3000);
      if (ulRc == ERROR_TIMEOUT)
         pThreadCl->stop();
      delete pThreadCl;
#endif  /* IC_PM */

#ifdef IC_WIN
   // wait for the secondary thread to finish
      unsigned long ulRc;

      if ( !CloseHandle(hWritePipe))
        {
          ulRc = GetLastError();
          ITRACE_RUNTIME("CloseHandle rc =: " + IString(ulRc));
        }

      if ( !CloseHandle(hReadPipe))
        {
          ulRc = GetLastError();
          ITRACE_RUNTIME("CloseHandle rc =: " + IString(ulRc));
        }

      if (hFinish) {
         ulRc = WaitForSingleObject(hFinish, 3000);
         if (ulRc == WAIT_TIMEOUT)
            pThreadCl->stop();

         if ( !CloseHandle(hFinish))
           {
             ulRc = GetLastError();
             ITRACE_RUNTIME("CloseHandle rc =: " + IString(ulRc));
           }
      }
      delete pThreadCl;
#endif
   }
   conversations().removeAll(&IDdeleteElemSCS, this);
   delete &conversations();
   closedConversations().removeAll(&IDdeleteElemSCCS);
   delete &closedConversations();
   hotLinkItems().removeAll(&IDdeleteElemSHLIS);
   delete &hotLinkItems();
#ifdef IC_PM
   delete &formats();
#endif
#ifdef IC_WIN
   delete &itemAtoms();
#endif

   // remove handlers and delete windows
   stopHandlingEventsFor((IWindow*)pwndClServer);
   delete pwndClServer;
   stopHandlingEventsFor(IDDETopicServer__pwndClFrame);
   {
      // IResourceLock frameLock(IDDETopicServer__frameCnt);
      IResourceLock frameLock(IDDEStatics::semaphor());
      IDDETopicServer__frmUseCntCl--;
      if (!IDDETopicServer__frmUseCntCl)
      {
         delete IDDETopicServer__pwndClFrame;
         IDDETopicServer__pwndClFrame = 0;
      }
   }
}

/*------------------------------------------------------------------------------
| IDDETopicServer::beginConversation                                           |
|                                                                              |
| If the server has the clients handle then a conversation can be started      |
| using this method.                                                           |
------------------------------------------------------------------------------*/
IDDETopicServer& IDDETopicServer :: beginConversation (
                                       const IWindowHandle& clientHandle )
{
   IMODTRACE_DEVELOP("DDETopServ::beginConversation");
   IASSERTPARM(clientHandle.isValid());
   bool bFound = false;
   IDDEServerConversation* pConv;
   IDDEServerConversationSet::Cursor myCurs(closedConversations());
   // check closed q, remove old conv if present
   closedConversations().semaphor().lock();
#ifdef IC_PM
   bFound = closedConversations().locateElementWithKey(clientHandle, myCurs);
#endif
#ifdef IC_WIN
   bFound = closedConversations().locateElementWithKey((unsigned long)(void *)clientHandle, myCurs);
#endif
   if (bFound)
   {
      pConv = closedConversations().elementAt(myCurs);
      closedConversations().removeAt(myCurs);
      delete pConv;
      ITRACE_DEVELOP("Client initiating new conversation before responding to \
         close of previous conversation");
   }
   closedConversations().semaphor().unlock();
   // add conversation if not already in coversation with this client
#ifdef IC_PM
   pConv = new IDDEServerConversation(clientHandle);
#endif
#ifdef IC_WIN
   pConv = new IDDEServerConversation((unsigned long)(void *)clientHandle);
#endif
   conversations().semaphor().lock();  // serialize access to the set
   bFound = conversations().locateOrAddElementWithKey(pConv);
   conversations().semaphor().unlock();
   if (bFound)
   {
      delete pConv;
      ITHROWLIBRARYERROR(IC_DDE_SERVER_IN_CONVERSATION,
         IBaseErrorInfo::invalidRequest,IException::recoverable);
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::endConversation                                             |
|                                                                              |
| This method ends the conversation with client. All hot links are removed     |
| from the servers hot link set. If not in conversation with the client then   |
| an exception is thrown.                                                      |
------------------------------------------------------------------------------*/
IDDETopicServer& IDDETopicServer :: endConversation (
                                       unsigned long conversationId )
{
   IMODTRACE_DEVELOP("DDETopServ::endConversation");
   bool bFound = false;
   IDDEServerConversationSet::Cursor myCurs(conversations());
   conversations().semaphor().lock();  // serialize access to the set
   bFound = conversations().locateElementWithKey(conversationId, myCurs);
   conversations().semaphor().unlock();
   if(bFound)
      // delete all hotLinkItems
      removeLink(IString(""), IString(""), conversationId);
   IDDEServerConversation* pConv = 0;
   conversations().semaphor().lock();  // serialize access to the set
   bFound = conversations().locateElementWithKey(conversationId, myCurs);
   if (bFound)
   {
      pConv = conversations().elementAt(myCurs);
      conversations().removeAt(myCurs);
      conversations().semaphor().unlock();
#ifdef IC_PM
      bool bSuccess = WinDdePostMsg((HWND)pConv->conversationId(),
                                       (HWND)serverHandle(),
                                       WM_DDE_TERMINATE,
                                       0,
                                       DDEPM_RETRY);
#endif
#ifdef IC_WIN
      bool bSuccess = PostMessage( (HWND)pConv->conversationId(),
                                      WM_DDE_TERMINATE,
                                      (WPARAM)(HWND)serverHandle(),
                                      0L );
#endif
      if (!bSuccess || fClPostMsgFail)
      {
         delete pConv;
         ITRACE_DEVELOP("WM_DDE_TERMINATE failure on attempt to initate end \
            of conversation.");
      }
      else
      {
          closedConversations().semaphor().lock();
          closedConversations().add(pConv);
          closedConversations().semaphor().unlock();
      }
   }
   else
   {
      conversations().semaphor().unlock();
      ITHROWLIBRARYERROR(IC_DDE_SERVER_NOT_IN_CONVERSATION,
         IBaseErrorInfo::invalidRequest,IException::recoverable);
   }
   return *this;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::hotLinkUpdate                                               |
|                                                                              |
| Search the server hot link items set for a hot link on the item. If the      |
| item is found get the set of conversations that has a hot link on this       |
| item. The server hot link set will have the conversations that are           |
| interested in certain formats.                                               |
|                                                                              |
| Step through the server hot link set to find all of the formats that clients |
| have requested hot links on for this item. The data for this item is set in  |
| an IDDERequestDataEvent in the appropriate format by calling the method      |
| requestHotLinkData. A DDE struct is built using the IDDERequestDataEvent     |
| buffer for the data and sent to the client conversations that are interested |
| in this format.                                                              |
|                                                                              |
| Notes:                                                                       |
|   1) The client can specify that data is not sent with an update of a hot    |
|      link item, only a notification that the data has changed. If this is    |
|      the case and there are no other hot links on this item that require     |
|      data, then the requestHotLinkData method is not called and the          |
|      client(s) will only receive notification that the data has changed.     |
|      If at least one client has requested the data in the format being       |
|      checked, then the requestHotLinkData method is called for that format.  |
|      Only the clients that requested the data are sent the data, the others  |
|      are sent a notification that the data has changed.                      |
|                                                                              |
|   2) If the client has requested pacing, then an ack is requested from the   |
|      client in response to processing the data. When the ack is received by  |
|      the server then future updates can be sent. If the client has pacing    |
|      set and has received a previous hot link update to this item and has    |
|      not sent back an ack then the method setUpdateOwed for this client is   |
|      called. This will indicate that an update is owed to this client and    |
|      will be sent when the ack is finally returned. If there have been       |
|      multiple updates since the client returned their ack then only one      |
|      update is sent. (See handleAck for the sending of the owed update)      |
------------------------------------------------------------------------------*/
unsigned long IDDETopicServer :: hotLinkUpdate ( const char* item )
{
   IMODTRACE_DEVELOP("DDETopServ::hotLinkUpdate");
   IASSERTPARM(item != 0);
   IASSERTPARM(item[0] != '\0');

   bool bAckOutstanding = false;
   bool bLinkFound = false;
   bool bRequestData = false;
   unsigned long ulCount = 0;

#ifdef IC_PM
   IString inItem = IString::upperCase(IString(item));
   IString myItem;
   IString myFormat;
   // create a struct and request data event to use in calling back the server
   PDDESTRUCT pddest = (PDDESTRUCT)buildDDEStruct (item, 0, 0, 0, 0);
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_REQUEST,
                IEventParameter1((unsigned long)0),
                IEventParameter2((void*)pddest));
   IDDERequestDataEvent rdEvt(myEvt);

   // Find all hotLinkItems the update is for
   IDDEServerHotLinkItemSet::Cursor myCurs(hotLinkItems());
   IDDEServerHotLinkItem* pItem = 0;
   hotLinkItems().semaphor().lock();  // serialize access to the set
   forCursor(myCurs)
   {
      rdEvt.setData("");   // reset the data field
      pItem = hotLinkItems().elementAt(myCurs);
      myItem = IString::upperCase(pItem->item());
      if ( inItem == myItem )
      {
         bLinkFound = true;
         bRequestData = false;
         IDDEServerConversation* pConv1;
         IDDEServerHotLinkSet::Cursor myCurs1(pItem->hotLinks());
         IDDEServerHotLink* pLink = 0;
         conversations().semaphor().lock();  // serialize access to the set
         // see if we have any datalinks without outstanding acks
         for (myCurs1.setToFirst();
              myCurs1.isValid() && !bRequestData;
              myCurs1.setToNext())
         {
            pLink = pItem->hotLinks().elementAt(myCurs1);
            pConv1 = conversations().elementWithKey(pLink->conversationId());
            bAckOutstanding = pConv1->isAckOutstanding(pItem->item(),
               pItem->format());
            if (pLink->isDataRequested() && !bAckOutstanding)
               bRequestData = true;
         }
         conversations().semaphor().unlock();
         if (bRequestData)  // callback the server if data required
         {
            rdEvt.setFormat(pItem->format());
            this->requestHotLinkData(rdEvt);
            //  don't check data length, allow server to send null data
         }
         // Send the update to all clients linked to this item and format
         bool bSuccess;
         unsigned short usStatus;
         PDDESTRUCT pddes = 0;
         IDDEServerAcknowledgeEvent* pEvt = 0;
         conversations().semaphor().lock();  // serialize access to the set
         forCursor(myCurs1)
         {
            bSuccess = 0;
            pLink = pItem->hotLinks().elementAt(myCurs1);
            IDDEServerConversation* pConv = conversations().elementWithKey(
               pLink->conversationId());
            bAckOutstanding = pConv->isAckOutstanding(item, pItem->format());
            if (bAckOutstanding)
            {
               pLink->setUpdateOwed();
            }
            else                       // send the update
            {
               ulCount += 1;
               if ( pLink->isPacingRequested() ||
                    (rdEvt.status() & DDE_FACKREQ) )
                  usStatus = 0 | DDE_FACKREQ;
               else
                  usStatus = 0;
               try {
                  if ( pLink->isDataRequested() )
                  {
                     pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                    (const char*)pItem->format(),
                                                    usStatus,
                                                    (const char*)rdEvt.buffer(),
                                                    rdEvt.buffer().size() );
                  }
                  else
                  {
                     usStatus |=  DDE_FNODATA;
                     pddes = (PDDESTRUCT)buildDDEStruct (item,
                                                    (const char*)pItem->format(),
                                                    usStatus,
                                                    0,
                                                    0 );
                  }
               }
               catch (IException& excp)
               {
                   conversations().semaphor().unlock();
                   hotLinkItems().semaphor().unlock();
                   IRETHROW(excp);
               }
               if ( pLink->isPacingRequested() ||
                    (rdEvt.status() & DDE_FACKREQ) )
               {
                  // prebuild event for ack q as PM will free memory associated
                  // with pddes
                  IEvent myEvt1(serverHandle(),
                                (unsigned long)WM_DDE_ADVISE,
                                IEventParameter1(pLink->conversationId()),
                                IEventParameter2((void*)pddes));
                  pEvt = new IDDEServerAcknowledgeEvent(myEvt1);
               }
               bSuccess = WinDdePostMsg((HWND)pLink->conversationId(),
                                        (HWND)serverHandle(),
                                        WM_DDE_DATA,
                                        pddes,
                                        DDEPM_RETRY);
               if (!bSuccess)
               {
                  conversations().semaphor().unlock();
                  hotLinkItems().semaphor().unlock();
                  IDDEInfo__freeMemory(pddest);
                  ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,
                     IBaseErrorInfo::accessError, IException::recoverable,
                     "WinDdePostMsg (WM_DDE_DATA)");
               }
               if (pEvt)
               {
                  pConv->addEvent(pEvt);
                  pEvt = 0;
               }
            }
         } // end for
         conversations().semaphor().unlock();
      }  // end if (initem == myitem)
   } // end for
   hotLinkItems().semaphor().unlock();
   IDDEInfo__freeMemory(pddest);

   if (!bLinkFound)  // no links for this item
      ITHROWLIBRARYERROR1(IC_DDE_NO_HOTLINK,IBaseErrorInfo::invalidRequest,
                          IException::recoverable, "hotLinkUpdate()");

#endif  /* IC_PM */

#ifdef IC_WIN
   bool bAtomAdded      = false;
   IString hotLinkItem(item) ;
   IString inItem = IString::upperCase(hotLinkItem);
   IString myItem;
   IString myFormat;

   // create a struct and request data event to use in calling back the server
   ATOM hotLinkAtom = (ATOM) IDDEInfo__atomFromString( (const char *) hotLinkItem, &bAtomAdded );
   if (bAtomAdded)
   {
      itemAtoms().add( hotLinkAtom );
      ITRACE_DEVELOP("Added atom to item list.");
   }
   IEvent myEvt(serverHandle(),
                (unsigned long)WM_DDE_REQUEST,
                IEventParameter1((unsigned long)0),
                IEventParameter2((unsigned long)MAKELPARAM( CF_TEXT, hotLinkAtom)));
   IDDERequestDataEvent rdEvt(myEvt);

   // Find all hotLinkItems the update is for
   IDDEServerHotLinkItemSet::Cursor myCurs(hotLinkItems());
   IDDEServerHotLinkItem* pItem = 0;
   hotLinkItems().semaphor().lock();  // serialize access to the set
   forCursor( myCurs )
   {
      rdEvt.setData("");   // reset the data field
      pItem = hotLinkItems().elementAt(myCurs);
      myItem = IString::upperCase(pItem->item());
      if ( inItem == myItem )
      {
         bLinkFound = true;
         bRequestData = false;
         IDDEServerConversation* pConv1;
         IDDEServerHotLinkSet::Cursor myCurs1(pItem->hotLinks());
         IDDEServerHotLink* pLink = 0;
         conversations().semaphor().lock();  // serialize access to the set
         // see if we have any datalinks without outstanding acks
         for ( myCurs1.setToFirst();
               myCurs1.isValid() && !bRequestData;
               myCurs1.setToNext() )
         {
            pLink = pItem->hotLinks().elementAt(myCurs1);
            pConv1 = conversations().elementWithKey(pLink->conversationId());
            bAckOutstanding = pConv1->isAckOutstanding(pItem->item(),
                                                       pItem->format());
            if (pLink->isDataRequested() && !bAckOutstanding)
            {
               bRequestData = true;
            }
         }
         conversations().semaphor().unlock();
         if ( bRequestData )  // callback the server if data required
         {
            rdEvt.setFormat(pItem->format());
            this->requestHotLinkData(rdEvt);
            //  don't check data length, allow server to send null data
         }
         // Send the update to all clients linked to this item and format
         bool bSuccess;
         unsigned short usStatus;

         HGLOBAL hDdeData = 0;
         IDDEServerAcknowledgeEvent* pEvt = 0;
         conversations().semaphor().lock();  // serialize access to the set
         forCursor( myCurs1 )
         {
            bSuccess = 0;
            pLink = pItem->hotLinks().elementAt(myCurs1);
            IDDEServerConversation* pConv = conversations().elementWithKey(
                                            pLink->conversationId());
            bAckOutstanding = pConv->isAckOutstanding(item, pItem->format());
            if (bAckOutstanding)
            {
               pLink->setUpdateOwed();
            }
            else                       // send the update
            {
               ulCount += 1;
               if ( pLink->isPacingRequested() ||
                    (rdEvt.status() & IDDEInfo::acknowledgeRequested) )
               {
                  usStatus = 0 | IDDEInfo::acknowledgeRequested;
               }
               else
               {
                  usStatus = 0;
               }

               try {
                  if ( pLink->isDataRequested() )
                  {
                     hDdeData = buildDDEStruct( IDDEData,
                                                (const char*)pItem->format(),
                                                usStatus,
                                                (const char*)rdEvt.buffer(),
                                                rdEvt.buffer().size() );
                  }
                  else
                  {
                     // DEFECT 28265
                     // we used to build a hDdeData pointer via the
                     // buildDDEStruct() api.
                     // This was WRONG.  For a warm DDE notify, then we
                     // should pass a NULL hDdeData parameter
                     hDdeData = NULL;           // 28265
                  }
               }
               catch (IException& excp)
               {
                   conversations().semaphor().unlock();
                   hotLinkItems().semaphor().unlock();
                   IRETHROW(excp);
               }

               if ( pLink->isPacingRequested() ||
                    (rdEvt.status() & IDDEInfo::acknowledgeRequested) )
               {
                  // prebuild event for ack q as system will free memory associated
                  // with hDdeData
                  HGLOBAL hDdeAdvise = buildDDEStruct( IDDEAdvise,
                                                       (const char*)pItem->format(),
                                                       usStatus,
                                                       0,
                                                       0 );
                  IEvent myEvt1( serverHandle(),
                                 (unsigned long)WM_DDE_ADVISE,
                                 IEventParameter1(pLink->conversationId()),
                                 IEventParameter2((unsigned long)PackDDElParam(WM_DDE_ADVISE, (UINT)hDdeAdvise, hotLinkAtom)) );
                  pEvt = new IDDEServerAcknowledgeEvent(myEvt1);
                  IDDEInfo__freeMemory( hDdeAdvise );
               }

               bSuccess = PostMessage( (HWND)pLink->conversationId(),
                                       WM_DDE_DATA,
                                       (WPARAM)(HWND)serverHandle(),
                                       PackDDElParam( WM_DDE_DATA,
                                                      (UINT)hDdeData, hotLinkAtom ) );

               if (!bSuccess)
               {
                  conversations().semaphor().unlock();
                  hotLinkItems().semaphor().unlock();
                  IDDEInfo__freeMemory( hDdeData );
                  ITHROWLIBRARYERROR1(IC_DDE_WINDDEPOSTMESSAGE_USER,
                              IBaseErrorInfo::accessError, IException::recoverable,
                              "PostMessage (WM_DDE_DATA)");
               }

               if (pEvt)
               {
                  pConv->addEvent(pEvt);
                  pEvt = 0;
               }

            }
         } // end for
         conversations().semaphor().unlock();
      }  // end if (initem == myitem)
   } // end for
   hotLinkItems().semaphor().unlock();

   if (!bLinkFound)  // no links for this item
      ITHROWLIBRARYERROR1(IC_DDE_NO_HOTLINK,IBaseErrorInfo::invalidRequest,
                          IException::recoverable, "hotLinkUpdate()");
#endif  /* IC_WIN */

   return ulCount;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::pokeData                                                    |
|                                                                              |
| This is the callback for the client trying to poke data to the server. The   |
| default action is to return false, which sends a negative ack back to the    |
| client.                                                                      |
------------------------------------------------------------------------------*/
bool IDDETopicServer :: pokeData ( unsigned long conversationId,
                                      IDDEPokeEvent& event)
{
   IMODTRACE_DEVELOP("DDETopServ::pokeData");
   return false;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::beginHotLink                                                |
|                                                                              |
| This is the callback for the client trying to start a hot link. The default  |
| action is to return false, which sends a negative ack back to the client.    |
------------------------------------------------------------------------------*/
bool IDDETopicServer :: beginHotLink ( unsigned long conversationId,
                                          IDDEServerHotLinkEvent& event)
{
   IMODTRACE_DEVELOP("DDETopServ::beginHotLink");
   return false;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::executeCommands                                             |
|                                                                              |
| This is the callback for the client commands sent to the server. The default |
| action is to return false which sends a negative ack back to the client.     |
------------------------------------------------------------------------------*/
bool IDDETopicServer :: executeCommands ( unsigned long conversationId,
                                             IDDEExecuteEvent& event)
{
   IMODTRACE_DEVELOP("DDETopServ::executeCommands");
   return false;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::acceptConversation                                          |
|                                                                              |
| This is the callback for the client trying to begin a conversation with the  |
| server. The default action is to return true, which sends a positive ack     |
| back to the client.                                                          |
------------------------------------------------------------------------------*/
bool IDDETopicServer :: acceptConversation ( unsigned long conversationId,
                                                IDDEBeginEvent& event)
{
   IMODTRACE_DEVELOP("DDETopServ::acceptConversation");
   return true;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::requestHotLinkData                                          |
|                                                                              |
| Callback for rendering hot link data in the appropriate format.              |
------------------------------------------------------------------------------*/
void IDDETopicServer :: requestHotLinkData ( IDDERequestDataEvent& event )
{
   IMODTRACE_DEVELOP("DDETopServ::requestHotLinkData");
}

/*------------------------------------------------------------------------------
| IDDETopicServer::hotLinkEnded                                                |
|                                                                              |
| Informs the receiver that the client has requested an end to the hot link.   |
------------------------------------------------------------------------------*/
void IDDETopicServer :: hotLinkEnded ( unsigned long conversationId,
                                       IDDEEvent& event)
{
   IMODTRACE_DEVELOP("DDETopServ::hotLinkEnded");
}

/*------------------------------------------------------------------------------
| IDDETopicServer::acknowledged                                                |
------------------------------------------------------------------------------*/
void IDDETopicServer :: acknowledged ( unsigned long conversationId,
                                       IDDEServerAcknowledgeEvent& event)
{
   IMODTRACE_DEVELOP("DDETopServ::ack");
}

/*------------------------------------------------------------------------------
| IDDETopicServer::conversationEnded                                           |
------------------------------------------------------------------------------*/
void IDDETopicServer :: conversationEnded ( unsigned long conversationId,
                                            IDDEEndEvent& event )
{
   IMODTRACE_DEVELOP("DDETopServ::conversationEnded");
}

/*------------------------------------------------------------------------------
| IDDETopicServer::conversationCount                                           |
|                                                                              |
| Returns the number of conversations that the server is engaged in.           |
------------------------------------------------------------------------------*/
unsigned long IDDETopicServer :: conversationCount ( ) const
{
   return conversations().numberOfElements();
}

/*------------------------------------------------------------------------------
| IDDETopicServer::hotLinkCount                                                |
|                                                                              |
| Returns the number of hot links that have been established. This number is   |
| determined by adding up the number of hot links on each item that is in the  |
| hot link item set.                                                           |
------------------------------------------------------------------------------*/
unsigned long IDDETopicServer :: hotLinkCount ( ) const
{
   unsigned long ulCount = 0;
   IDDEServerHotLinkItemSet::Cursor myCurs(*pHLItemSetCl);
   IDDEServerHotLinkItem* pItem = 0;
   hotLinkItems().semaphor().lock();
   forCursor(myCurs)
   {
      pItem = (*pHLItemSetCl).elementAt(myCurs);
      ulCount += pItem->hotLinks().numberOfElements();
   } // end for
   hotLinkItems().semaphor().unlock();

   return ulCount;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::dispatchHandlerEvent                                        |
|                                                                              |
| Overloaded virtual dispatch function. Dispatched the DDE events to the       |
| handlers.  If threads are used for dispatching the handlers then the event   |
| is written to a queue.                                                       |
|                                                                              |
| Note:                                                                        |
|   For Windows 3.1 thread is not supported, the event queue is disabled.      |
|                                                                              |
| Note:                                                                        |
|   if a WM_DDE message is received while one is being serviced then           |
|   throw an exception that DDE synchronization has been violated.             |
|   This situation can occur when useEventThread is set false (single          |
|   threaded) on the class constructor and the receiver's handle... method     |
|   puts up a dialog box.                                                      |
------------------------------------------------------------------------------*/
bool IDDETopicServer :: dispatchHandlerEvent ( IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::dispatchHandlerEvent");
::asDebugInfo( &evt );
   if (!pThreadCl)
   {
      switch (evt.eventId())
      {
         case WM_DDE_ACK:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleAck(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }
         case WM_DDE_ADVISE:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleAdvise(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }

         case WM_DDE_EXECUTE:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleExecute(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }

         case WM_DDE_INITIATE:
         {
            handleInitiate(evt);
            evt.setResult(IEventResult(1));
            return false;  // ensure that other topic servers see this message
         }

         case WM_DDE_POKE:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handlePoke(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }

         case WM_DDE_REQUEST:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleRequest(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }

         case WM_DDE_TERMINATE:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleTerminate(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }

         case WM_DDE_UNADVISE:
         {
            if (!fClHdrActive) {
               fClHdrActive = true;
               handleUnadvise(evt);
            }
            else
               ITHROWLIBRARYERROR(IC_DDE_SYNCHRONIZATION,
                  IBaseErrorInfo::accessError, IException::unrecoverable);
            fClHdrActive = false;
            break;
         }

         default:
         {
            return false;
         }

      } /* endswitch */
   }
#ifdef IC_PM
   else      // dispatch the event on a separate thread
   {
      switch (evt.eventId())
      {
         case WM_DDE_ACK:
         case WM_DDE_ADVISE:
         case WM_DDE_EXECUTE:
         case WM_DDE_POKE:
         case WM_DDE_REQUEST:
         case WM_DDE_TERMINATE:
         case WM_DDE_UNADVISE:
         {
            unsigned long ulRc = 0;
            unsigned long ulSize(sizeof(IEvent));
            IEvent* pEvt = new IEvent(evt);
            ulRc = DosWriteQueue(queueHandle(),
                                 (unsigned long)0,
                                 ulSize,
                                 (void*)pEvt,
                                 (unsigned long)0);
            ITRACE_DEVELOP("DosWriteQueue rc =: " + IString(ulRc));
            if (ulRc) { // no queue memory
               delete pEvt;
               ITHROWSYSTEMERROR(ulRc,"DosWriteQueue",
                  IBaseErrorInfo::outOfSystemResource, IException::recoverable);
            }
            break;
         }

         case WM_DDE_INITIATE:
         {
            handleInitiate(evt);
            evt.setResult(IEventResult(1));
            return false;  // ensure that other topic servers see this message
         }

         default:
         {
            return false;
         }

      } /* endswitch */
   }
#endif  /* IC_PM */

#ifdef IC_WIN
   else      // dispatch the event on a separate thread
   {
      switch (evt.eventId())
      {

         case WM_DDE_ACK:
         case WM_DDE_ADVISE:
         case WM_DDE_EXECUTE:
         case WM_DDE_POKE:
         case WM_DDE_REQUEST:
         case WM_DDE_TERMINATE:
         case WM_DDE_UNADVISE:
         {
            unsigned long ulRc;
            unsigned long ulSize;
            IEvent* pEvt = new IEvent(evt);

            if ( !WriteFile(hWritePipe, &pEvt, sizeof(IEvent*), &ulSize, NULL)
               || ulSize != sizeof(IEvent*))
              {
                delete pEvt;
                ulRc = GetLastError();
                ITHROWSYSTEMERROR(ulRc,"WriteFile",
                    IBaseErrorInfo::outOfSystemResource, IException::recoverable);
              }
            break;
         }

         case WM_DDE_INITIATE:
         {
            handleInitiate(evt);
            evt.setResult(IEventResult(1));
            return false;  // ensure that other topic servers see this message
         }

         default:
         {
            return false;
         }

      } /* endswitch */
   }
#endif  /* IC_WIN */

   return true;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleAck                                                   |
|                                                                              |
| Called by dispatcher to handle acks.                                         |
|                                                                              |
| Notes:                                                                       |
|   1) If the ack is to a hot link update check to see if the client is owed   |
|      an update on a hot link due to having pacing set. If the client is      |
|      owed a hot link update, and data is requested then get the data from    |
|      the topic server by calling the method requestHotLinkData. The data     |
|      will be posted to the client.                                           |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleAck ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleAck");
   IDDEServerAcknowledgeEvent dEvt(evt);
#ifdef IC_WIN
   // save the shared memory handle for use in the call to __freeMemory later
   // in some timing cases, the LPARAM might become invalid
   HGLOBAL sharedMemoryHandle = //mkb was IDDEInfo__loWord
      (HGLOBAL)IDDEInfo__hiWord(WM_DDE_ACK, evt.parameter2().asUnsignedLong()) ;
   // Windows ack event has to manually load the event's private data.
   dEvt.loadStatus();
   dEvt.loadItem();
#endif
   bool bFound,
           bSuccess = 0;
   IDDEServerConversation* pConv;
   IDDEServerConversationSet::Cursor myCurs(conversations());
   conversations().semaphor().lock();  // serialize access to the set
   bFound = conversations().locateElementWithKey(
      (unsigned long)evt.parameter1(), myCurs);

#ifdef IC_PM
   if (bFound)
   {
      pConv = conversations().elementAt(myCurs);
      IString inItem = IString::upperCase(dEvt.item());
      IString inFormat = IString::upperCase(dEvt.format());
      IString myItem;
      IString myFormat;
      // set the default status to positive ack
      unsigned short usStatus = 0 | DDE_FACK;
      // remove all ack events from Q up to one that matches
      IDDEServerAcknowledgeEvent* pEvt;
      unsigned long ulCount = pConv->eventCount();
      for(unsigned long ulIndex = 0;
          (bSuccess == 0) && (ulIndex < ulCount);
          ulIndex++)
      {
         pEvt = pConv->removeEvent();
         myItem = IString::upperCase(pEvt->item());
         myFormat = IString::upperCase(pEvt->format());
         if ( ( myItem == inItem ) &&
              ( myFormat == inFormat || inFormat == "") )
         {
            bSuccess = 1;
            // copy the status field from the ack
            pEvt->setStatus(dEvt.status());
         }
         else
            pEvt->setStatus(usStatus);
         // callback the server and delete the event
         this->acknowledged( (unsigned long)evt.parameter1(), *pEvt);
         if (pEvt->isAckToHotLinkUpdate())
         {
            //  see if we owe this client data
            IDDEServerAcknowledgeEvent* pEvt1 = 0;
            IDDEServerHotLinkItemSet::Cursor myCurs1(hotLinkItems());
            IDDEServerHotLinkItem* pItem = 0;
            IString hotItem;
            IString hotFormat;
            bFound = false;
            hotLinkItems().semaphor().lock();  // serialize access to the set
            for(myCurs1.setToFirst();
                myCurs1.isValid() && bFound == false;
                myCurs1.setToNext())
            {
               pItem = hotLinkItems().elementAt(myCurs1);
               hotItem = IString::upperCase(pItem->item());
               hotFormat = IString::upperCase(pItem->format());
               if ( hotItem == myItem && hotFormat == myFormat )
               {
                  IDDEServerHotLinkSet::Cursor myCurs2(pItem->hotLinks());
                  IDDEServerHotLink* pLink = 0;
                  for(myCurs2.setToFirst();
                      myCurs2.isValid() && bFound == false;
                      myCurs2.setToNext())
                  {
                     pLink = pItem->hotLinks().elementAt(myCurs2);
                     if (pLink->conversationId() ==
                              (unsigned long)evt.parameter1())
                     {
                        bFound = true;
                        if (pLink->isUpdateOwed())
                        {
                           pLink->setUpdateOwed(false);
                           PDDESTRUCT pddes, pddes1;
                           unsigned short usStatus = 0;
                           if (pLink->isDataRequested())
                           {
                              try {
                                  pddes1 =
                                      (PDDESTRUCT)buildDDEStruct(pItem->item(),
                                                      pItem->format(), 0, 0, 0);
                              }
                              catch (IException& excp)
                              {
                                  conversations().semaphor().unlock();
                                  hotLinkItems().semaphor().unlock();
                                  IRETHROW(excp);
                              }
                              IEvent myEvt(serverHandle(),
                                           (unsigned long)WM_DDE_ADVISE,
                                           IEventParameter1((unsigned long)0),
                                           IEventParameter2((void*)pddes1));
                              IDDERequestDataEvent rdEvt(myEvt);
                              IDDEInfo__freeMemory(pddes1);
                              this->requestHotLinkData(rdEvt);
                              if ( pLink->isPacingRequested() ||
                                   (rdEvt.status() & DDE_FACKREQ) )
                                 usStatus |= DDE_FACKREQ;
                              try {
                                  pddes = (PDDESTRUCT)buildDDEStruct (
                                                (const char*)pItem->item(),
                                                (const char*)pItem->format(),
                                                usStatus,
                                                (const char*)rdEvt.buffer(),
                                                rdEvt.buffer().size() );
                              }
                              catch (IException& excp)
                              {
                                  conversations().semaphor().unlock();
                                  hotLinkItems().semaphor().unlock();
                                  IRETHROW(excp);
                              }
                           }
                           else
                           {
                              if ( pLink->isPacingRequested() )
                                 usStatus |= DDE_FACKREQ;
                              usStatus |= DDE_FNODATA;
                              try {
                                  pddes = (PDDESTRUCT)buildDDEStruct (
                                                (const char*)pItem->item(),
                                                (const char*)pItem->format(),
                                                usStatus,
                                                0,
                                                0);
                              }
                              catch (IException& excp)
                              {
                                  conversations().semaphor().unlock();
                                  hotLinkItems().semaphor().unlock();
                                  IRETHROW(excp);
                              }
                           }
                           if (usStatus & DDE_FACKREQ)
                           {
        // prebuild event for ack q as PM will free memory associated with pddes
                              IEvent myEvt1(serverHandle(),
                                            (unsigned long)WM_DDE_ADVISE,
                                            IEventParameter1(
                                               pLink->conversationId()),
                                            IEventParameter2((void*)pddes));
                              pEvt1 = new IDDEServerAcknowledgeEvent(myEvt1);
                           }
                           bool bSuccess1 = WinDdePostMsg(
                                                 (HWND)pLink->conversationId(),
                                                 (HWND)serverHandle(),
                                                 WM_DDE_DATA,
                                                 pddes,
                                                 DDEPM_RETRY);
                           if (!bSuccess1)
                           {
                              fClPostMsgFail = true;
                              IEvent myEvt2(serverHandle(),
                                            // create dummy event for callback
                                            (unsigned long)WM_DDE_TERMINATE,
                                            IEventParameter1(
                                                (unsigned long)serverHandle()),
                                            IEventParameter2((void*)0));
                              IDDEEndEvent endEvt(myEvt2, IDDEEndEvent::error);
                              bool bEnded = true;
                              try {
                                 this->endConversation(
                                   (unsigned long)evt.parameter1());
                              }
                              catch (IInvalidRequest& excReq) {
                                 bEnded = false;
                              }
                              if (bEnded)
                                 this->conversationEnded(
                                   (unsigned long)evt.parameter1(),endEvt);
                              fClPostMsgFail = false;
                           }
                           if ( usStatus & DDE_FACKREQ )
                              pConv->addEvent(pEvt1);
                        }  // end if data is owed
                     }
                  }  // end for
               }  // end if
            }  // end for
            hotLinkItems().semaphor().unlock();
         }  // end check to see if client is owed data
         delete pEvt;
      }  // end for
      conversations().semaphor().unlock();
      if(bSuccess == 0)
         ITRACE_DEVELOP("WM_DDE_ACK discarded, no matching outstanding ack");
   }
   else
   {
      conversations().semaphor().unlock();
      ITRACE_DEVELOP("WM_DDE_ACK discarded, no matching conversation");
   }

   IDDEInfo__freeMemory( (_DDESTRUCT*)(void*)evt.parameter2());
#endif  /* IC_PM */

#ifdef IC_WIN
   if (bFound)
   {
      pConv = conversations().elementAt(myCurs);
      IString inItem = IString::upperCase(dEvt.item());
      IString inFormat = IString::upperCase(dEvt.format());
      IString myItem;
      IString myFormat;

      // set the default status to positive ack
      unsigned short usStatus = 0 | IDDEInfo::acknowledge;

      // remove all ack events from Q up to one that matches
      IDDEServerAcknowledgeEvent* pEvt;
      unsigned long ulCount = pConv->eventCount();
      for(unsigned long ulIndex = 0;
          (bSuccess == 0) && (ulIndex < ulCount);
          ulIndex++)
      {
         pEvt = pConv->removeEvent();
         myItem = IString::upperCase(pEvt->item());
         myFormat = IString::upperCase(pEvt->format());
         if ( ( myItem == inItem ) &&
              ( myFormat == inFormat || inFormat == "") )
         {
            bSuccess = 1;
            // copy the status field from the ack
            pEvt->setStatus(dEvt.status());
         }
         else
            pEvt->setStatus(usStatus);

         // callback the server and delete the event
         this->acknowledged( (unsigned long)evt.parameter1(), *pEvt);
         if (pEvt->isAckToHotLinkUpdate())
         {
            //  see if we owe this client data
            IDDEServerAcknowledgeEvent* pEvt1 = 0;
            IDDEServerHotLinkItemSet::Cursor myCurs1(hotLinkItems());
            IDDEServerHotLinkItem* pItem = 0;
            IString hotItem;
            IString hotFormat;
            bFound = false;
            hotLinkItems().semaphor().lock();  // serialize access to the set
            for(myCurs1.setToFirst();
                myCurs1.isValid() && bFound == false;
                myCurs1.setToNext())
            {
               pItem = hotLinkItems().elementAt(myCurs1);
               hotItem = IString::upperCase(pItem->item());
               hotFormat = IString::upperCase(pItem->format());
               if ( hotItem == myItem && hotFormat == myFormat )
               {
                  IDDEServerHotLinkSet::Cursor myCurs2(pItem->hotLinks());
                  IDDEServerHotLink* pLink = 0;
                  for(myCurs2.setToFirst();
                      myCurs2.isValid() && bFound == false;
                      myCurs2.setToNext())
                  {
                     pLink = pItem->hotLinks().elementAt(myCurs2);
                     if (pLink->conversationId() ==
                              (unsigned long)evt.parameter1())
                     {
                        bFound = true;
                        if (pLink->isUpdateOwed())
                        {
                           pLink->setUpdateOwed(false);
                           HGLOBAL hDdeSMem;
                           unsigned short usStatus = 0;
                           bool bAtomAdded;
                           ATOM aItem = IDDEInfo__atomFromString( pItem->item(),
                                                                  &bAtomAdded );
                           if (bAtomAdded)
                           {
                              itemAtoms().add( aItem );
                              ITRACE_DEVELOP("Added atom to item list.");
                           }
                           if (pLink->isDataRequested())
                           {
                              HGLOBAL hDdeAdvise;
                              try {
                                 hDdeAdvise = buildDDEStruct( IDDEAdvise,
                                                      (const char*)pItem->format(),
                                                      0,
                                                      0, 0 );
                              }
                              catch (IException& excp)
                              {
                                  conversations().semaphor().unlock();
                                  hotLinkItems().semaphor().unlock();
                                  IRETHROW(excp);
                              }
                              IEvent myEvt(serverHandle(),
                                           (unsigned long)WM_DDE_ADVISE,
                                           IEventParameter1((unsigned long)0),
                                           IEventParameter2((unsigned long)
                                                            PackDDElParam( WM_DDE_ADVISE,
                                                                           (UINT)hDdeAdvise,
                                                                           aItem ) ));
                              IDDERequestDataEvent rdEvt( myEvt );
                              IDDEInfo__freeMemory( hDdeAdvise );

                              // call server callback method
                              this->requestHotLinkData( rdEvt );
                              if ( pLink->isPacingRequested() ||
                                   (rdEvt.status() & IDDEInfo::acknowledgeRequested) )
                              {
                                 usStatus |= IDDEInfo::acknowledgeRequested;
                              }

                              // make sure the client will free the data
                              usStatus |= IDDEInfo::releaseData;
                              try {
                                 hDdeSMem = buildDDEStruct( IDDEData,
                                                    (const char*)pItem->format(),
                                                    usStatus,
                                                    (const char*)rdEvt.buffer(),
                                                    rdEvt.buffer().size() );
                              }
                              catch (IException& excp)
                              {
                                  conversations().semaphor().unlock();
                                  hotLinkItems().semaphor().unlock();
                                  IRETHROW(excp);
                              }
                           }
                           else
                           {
                              if ( pLink->isPacingRequested() )
                                 usStatus |= IDDEInfo::acknowledgeRequested;

                              usStatus |= IDDEInfo::noData;

                              // DEFECT 28265
                              // we used to build a hDdeData pointer via the
                              // 1uildDDEStruct() api.
                              // This was WRONG.  For a warm DDE notify, then
                              // we should pass a NULL hDdeData parameter
                              hDdeSMem = 0;                         // 28265
                           }

                           if ( usStatus & IDDEInfo::acknowledgeRequested )
                           {
                  // prebuild event for ack q as system will free memory associated
                  // with hDdeData
                              HGLOBAL hDdeAdvise;
                              try {
                                 hDdeAdvise = buildDDEStruct( IDDEAdvise,
                                                      (const char*)pItem->format(),
                                                      usStatus,
                                                      0, 0 );
                              }
                              catch (IException& excp)
                              {
                                  conversations().semaphor().unlock();
                                  hotLinkItems().semaphor().unlock();
                                  IRETHROW(excp);
                              }
                              IEvent myEvt1( serverHandle(),
                                             (unsigned long)WM_DDE_ADVISE,
                                             IEventParameter1(
                                                       pLink->conversationId()),
                                             IEventParameter2((unsigned long)
                                                       PackDDElParam( WM_DDE_ADVISE,
                                                                      (UINT)hDdeAdvise,
                                                                      aItem ) ));
                              pEvt1 = new IDDEServerAcknowledgeEvent(myEvt1);
                              IDDEInfo__freeMemory( hDdeAdvise );
                           }

                           bool bSuccess1 = PostMessage(
                                                 (HWND)pLink->conversationId(),
                                                 WM_DDE_DATA,
                                                 (WPARAM)(HWND)serverHandle(),
                                                 PackDDElParam( WM_DDE_DATA,
                                                                (UINT)hDdeSMem, aItem ) );
                           if (!bSuccess1)
                           {
                              fClPostMsgFail = true;
                              IDDEInfo__freeMemory( hDdeSMem );
                              IEvent myEvt2(serverHandle(),
                                            // create dummy event for callback
                                            (unsigned long)WM_DDE_TERMINATE,
                                            IEventParameter1(
                                                (unsigned long)(void *)serverHandle()),
                                            IEventParameter2((void*)0));
                              IDDEEndEvent endEvt(myEvt2, IDDEEndEvent::error);
                              bool bEnded = true;
                              try {
                                 this->endConversation(
                                            (unsigned long)evt.parameter1());
                              }
                              catch (IInvalidRequest& excReq) {
                                 bEnded = false;
                              }

                              if (bEnded)
                                 this->conversationEnded(
                                   (unsigned long)evt.parameter1(), endEvt);
                              fClPostMsgFail = false;
                           }

                           if ( usStatus & IDDEInfo::acknowledgeRequested )
                              pConv->addEvent(pEvt1);
                        }  // end if data is owed
                     }
                  }
               }
            }
            hotLinkItems().semaphor().unlock();
         }  // end check to see if client is owed data

         delete pEvt;
      }  // end for

      conversations().semaphor().unlock();
      if(bSuccess == 0)
         ITRACE_DEVELOP("WM_DDE_ACK discarded, no matching outstanding ack");
   }
   else
   {
      conversations().semaphor().unlock();
      ITRACE_DEVELOP("WM_DDE_ACK discarded, no matching conversation");
   }

   // NOTE - don't know why -ve ack is not handled in here
   //
   // This class is defaulted to let the client free the data memory once
   //     client received the WM_DDE_DATA msg. So hData will not be free in
   //     here.

   // free the DDE structure accompany with this event

   IDDEInfo__freeMemory( sharedMemoryHandle );
//   IDDEInfo__freeMemory( (HGLOBAL)IDDEInfo__loWord(WM_DDE_ACK, (long)(void*)evt.parameter2()) );

   FreeDDElParam( WM_DDE_ACK, evt.parameter2().asUnsignedLong());
#endif  /* IC_WIN */

}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleAdvise                                                |
|                                                                              |
| Called by the dispatcher when the client requests a hot link.                |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleAdvise ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleAdvise");
   IDDEServerHotLinkEvent hlEvt(evt);
#ifdef IC_WIN
   HGLOBAL sharedMemoryHandle =
      (HGLOBAL)IDDEInfo__loWord( WM_DDE_ADVISE,
                                 evt.parameter2().asUnsignedLong() ) ;
#endif
   bool bFound = 0;
   bool bSuccess = 1;
   bool bLinkFound = 0;
   conversations().semaphor().lock();
   bFound = conversations().containsElementWithKey(
               (unsigned long)evt.parameter1());
   conversations().semaphor().unlock();
   if (bFound)
   {
      IDDEServerHotLinkItemSet::Cursor myCurs(hotLinkItems());
      IDDEServerHotLinkItem* pItem = 0;
      IDDEServerHotLink* pLink = 0;
      IString inItem = IString::upperCase(hlEvt.item());
      IString inFormat = IString::upperCase(hlEvt.format());
      IString myItem;
      IString myFormat;
      hotLinkItems().semaphor().lock();  // serialize access to the set
      // look to see if the hot link is already established.
      for(myCurs.setToFirst();
          myCurs.isValid() && bLinkFound == 0;
          myCurs.setToNext())
      {
         pItem = hotLinkItems().elementAt(myCurs);
         myItem = IString::upperCase(pItem->item());
         myFormat = IString::upperCase(pItem->format());
         if ( ( myItem == inItem ) && ( myFormat == inFormat ) )
         {
            // already have a link on this item
            IDDEServerHotLinkSet::Cursor myCurs1(pItem->hotLinks());
            bLinkFound = 1;
            for(myCurs1.setToFirst();
                myCurs1.isValid() && bSuccess == 1;
                myCurs1.setToNext())
            {
               pLink = pItem->hotLinks().elementAt(myCurs1);
               if (pLink->conversationId() == (unsigned long)evt.parameter1())
                  bSuccess = 0;
            } // end for
         }
      } // end for
      hotLinkItems().semaphor().unlock();


      if (bSuccess)
         // no active hotlink for the client on this item and format,
         // callback the server
         bSuccess = this->beginHotLink( (unsigned long)evt.parameter1(), hlEvt);
      unsigned short usStatus = hlEvt.status();

#ifdef IC_PM
      if (bSuccess)
      {
         usStatus |=  DDE_FACK;
         // ensure none of the negative flags are on
         usStatus &= ~(DDE_FBUSY | DDE_NOTPROCESSED);
      }
      else
         usStatus &= ~DDE_FACK;                       // ensure ack is negative
      PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (hlEvt.item(),
                                                     hlEvt.format(),
                                                     usStatus,
                                                     0,
                                                     0);
      bool bSuccess1 = WinDdePostMsg((HWND)evt.parameter1(),
                                        (HWND)serverHandle(),
                                        WM_DDE_ACK,
                                        pddes,
                                        DDEPM_RETRY);
      if (!bSuccess1)
      {                      // don't throw exception as client will also crash
         fClPostMsgFail = true;
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }
      conversations().semaphor().lock();
      bool bConvFound = conversations().containsElementWithKey(
                                         (unsigned long)evt.parameter1());
      conversations().semaphor().unlock();
      if (bSuccess && bConvFound) // server has accepted, and ack was successful
      {
    //  add the hot link information to the sets
         hotLinkItems().semaphor().lock();
         pLink = new IDDEServerHotLink(hlEvt, (unsigned long)evt.parameter1());
         if (!bLinkFound)  // need a new hotLinkItem
         {
            pItem = new IDDEServerHotLinkItem(hlEvt);
            hotLinkItems().add(pItem);
         }
         pItem->hotLinks().add(pLink);
         hotLinkItems().semaphor().unlock();
      }
   }
   else
      ITRACE_DEVELOP("WM_DDE_ADVISE discarded, no matching conversation");
   IDDEInfo__freeMemory( (_DDESTRUCT*)(void*)evt.parameter2());
#endif  /* IC_PM */

#ifdef IC_WIN
      if (bSuccess)     // begin hot link is OK
      {
         usStatus |=  IDDEInfo::acknowledge;
         // ensure none of the negative flags are on
         usStatus &= ~(IDDEInfo::busy | IDDEInfo::notProcessed);
      }
      else
      {
         usStatus &= ~IDDEInfo::acknowledge;         // ensure ack is negative
      }

      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

      bool bAtomAdded;
      ATOM aItem = IDDEInfo__atomFromString( hlEvt.item(), &bAtomAdded );
      if (bAtomAdded)
      {
         itemAtoms().add( aItem );
         ITRACE_DEVELOP("Added atom to item list.");
      }
      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK first" );
      bool bSuccess1 = PostMessage( (HWND)evt.parameter1(),
                                       WM_DDE_ACK,
                                       (WPARAM)(HWND)serverHandle(),
                                       PackDDElParam( WM_DDE_ACK,
                                                      ackBuffer, aItem ) );
      if (!bSuccess1)
      {                      // don't throw exception as client will also crash
         fClPostMsgFail = true;
//       IDDEInfo__freeMemory( hDdeAck );
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)(void *)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)(void *)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }

      conversations().semaphor().lock();
      bool bConvFound = conversations().containsElementWithKey(
                                         (unsigned long)evt.parameter1());
      conversations().semaphor().unlock();
      if (bSuccess && bConvFound) // server has accepted, and ack was successful
      {
         // delete the advise struct if ack post back is +ve
         if ( usStatus & IDDEInfo::acknowledge )
         {
            IDDEInfo__freeMemory( sharedMemoryHandle) ;
//            IDDEInfo__freeMemory( (HGLOBAL)IDDEInfo__loWord( WM_DDE_ADVISE,
//                                  evt.parameter2().asUnsignedLong() ) );
         }

         //  add the hot link information to the sets
         hotLinkItems().semaphor().lock();
         pLink = new IDDEServerHotLink(hlEvt, (unsigned long)evt.parameter1());
         if (!bLinkFound)  // need a new hotLinkItem
         {
            pItem = new IDDEServerHotLinkItem(hlEvt);
            hotLinkItems().add(pItem);
         }
         pItem->hotLinks().add(pLink);
         hotLinkItems().semaphor().unlock();
      }
   }
   else
   {
      ITRACE_DEVELOP("WM_DDE_ADVISE discarded, no matching conversation");

      // send a -ve ack back

      unsigned short usStatus = hlEvt.status();
      usStatus &= ~IDDEInfo::acknowledge;
      UINT uOpt;
      UINT aItem;
      UnpackDDElParam( WM_DDE_ADVISE, evt.parameter2(), &uOpt, &aItem );
      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;
      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK second" );
      if ( !( PostMessage( (HWND)evt.parameter1(),
                           WM_DDE_ACK,
                           (WPARAM)(HWND)serverHandle(),
                           PackDDElParam( WM_DDE_ACK, ackBuffer, aItem ) ) ) )
      {
//          IDDEInfo__freeMemory( hDdeAck );
      }
   }
   FreeDDElParam( WM_DDE_ADVISE, evt.parameter2().asUnsignedLong());
#endif  /* IC_WIN */

}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleExecute                                               |
|                                                                              |
| This method is called from the dispatcher in response to a clients request   |
| to execute commands.                                                         |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleExecute ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleExecute");
   IDDEExecuteEvent dEvt(evt);
   bool bFound, bSuccess;
   conversations().semaphor().lock();
   bFound = conversations().containsElementWithKey(
               (unsigned long)evt.parameter1());
   conversations().semaphor().unlock();

#ifdef IC_PM
   if (bFound)
   {
      bSuccess = this->executeCommands( (unsigned long)evt.parameter1(), dEvt);
      unsigned short usStatus = dEvt.status();
      if (bSuccess)
      {
         usStatus |=  DDE_FACK;
         // ensure none of the negative flags are on
         usStatus &= ~(DDE_FBUSY | DDE_NOTPROCESSED);
      }
      else
         usStatus &= ~DDE_FACK;                       // ensure ack is negative
      PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (0,
                                                  0,
                                                  usStatus,
                                                  (const char*)dEvt.commands(),
                                                  (dEvt.commands()).size());
      bSuccess = WinDdePostMsg((HWND)evt.parameter1(),
                               (HWND)serverHandle(),
                               WM_DDE_ACK,
                               pddes,
                               DDEPM_RETRY);
      if (!bSuccess)
      {                      // don't throw exception as client will also crash
         fClPostMsgFail = true;
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }
   }
   else
      ITRACE_DEVELOP("WM_DDE_EXECUTE discarded, no matching conversation");

   IDDEInfo__freeMemory( (_DDESTRUCT*)(void*)evt.parameter2());
#endif  /* IC_PM */

#ifdef IC_WIN
   if (bFound)
   {
      // call server callback
      bSuccess = this->executeCommands( (unsigned long)evt.parameter1(), dEvt);
      unsigned short usStatus = dEvt.status();

      if (bSuccess)
      {
         usStatus |=  IDDEInfo::acknowledge;
         // ensure none of the negative flags are on
         usStatus &= ~(IDDEInfo::busy | IDDEInfo::notProcessed);
      }
      else
         usStatus &= ~IDDEInfo::acknowledge;      // ensure ack is negative

      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;
      UINT  hCommand = evt.parameter2().asUnsignedLong();
      LPARAM  lParam = PackDDElParam( WM_DDE_ACK, ackBuffer, hCommand );
      ITRACE_DEVELOP( "DBug: Sending Ack for Execute, hCommand=" + IString(hCommand)
                    + ", ackBuffer=" + IString(ackBuffer)
                    + ", lParam=" + IString((unsigned long)lParam));

#ifdef  DEBUG
      if ( IPlatform::isWinNT())        //different technique on w95
      {         // from Microsoft phone call (nt3.51 implementation)
         struct Foo { UINT low; UINT high; };
         struct Foo *pFoo = (struct myFoo *)GlobalLock( (HGLOBAL)lParam );
         if ( pFoo != NULL )
         {      //SET STORAGE ALTERATION BREAKPOINT on struct Foo
            ITRACE_DEVELOP( "DBug: Packed low=" + IString(pFoo->low)
                          + ", high=" + IString(pFoo->high));
            GlobalUnlock( (HGLOBAL)lParam );
         }
         else
            ITRACE_DEVELOP( "DBug: Couldn't lock the packed handle" );
      }
#endif

      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK third" );
      bSuccess = PostMessage( (HWND)evt.parameter1(),
                              WM_DDE_ACK,
                              (WPARAM)(HWND)serverHandle(),
                              lParam );

      if (!bSuccess)
      {                   // don't throw exception as client will also crash
         fClPostMsgFail = true;
//         IDDEInfo__freeMemory( hDdeAck );
         // create dummy event for callback
         IEvent myEvt( serverHandle(),
                       (unsigned long)WM_DDE_TERMINATE,
                       IEventParameter1((unsigned long)(void *)serverHandle()),
                       IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;

         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }

         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }
   }
   else   // return -ve ack
   {
      ITRACE_DEVELOP("WM_DDE_EXECUTE discarded, no matching conversation");

      unsigned short usStatus = dEvt.status();
      usStatus &= ~IDDEInfo::acknowledge;

      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

      UINT uCmd, uLo;
      UnpackDDElParam( WM_DDE_EXECUTE, evt.parameter2(), &uLo, &uCmd );
      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK fourth" );
      if ( !( PostMessage( (HWND)evt.parameter1(),
                           WM_DDE_ACK,
                           (WPARAM)(HWND)serverHandle(),
                           PackDDElParam( WM_DDE_ACK, ackBuffer, uCmd ) ) ) )
//                            PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, uCmd ) ) ) )
      {
//         IDDEInfo__freeMemory( hDdeAck );
      }
   }
#endif  /* IC_WIN */

}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleInitiate                                              |
|                                                                              |
| The dispatcher calls this method when a client tries to initiate a           |
| conversation with this server.                                               |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleInitiate ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleInitiate");
   if ( (unsigned long)evt.parameter1() )     // insure client hwnd is not zero
   {
      ITRACE_DEVELOP("Client hwnd is not zero");
      // insure we aren't already in conversation with this client
      conversations().semaphor().lock();
      bool bConvFound = conversations().containsElementWithKey(
                                    (unsigned long)evt.parameter1());
      conversations().semaphor().unlock();
      if (!bConvFound)
      {
         ITRACE_DEVELOP("Not in conversation with this client");
         IDDEServerConversation* pConv;
     // check to see if application and topic match (including wildcards)
         IDDEBeginEvent initevt(evt);
         IString strApp = IString::upperCase(application());
         IString strTopic = IString::upperCase(topic());
         IString strInApp = strApp;
         IString strInTopic = strTopic;
         if ( initevt.application().size() != 0 )
            strInApp = IString::upperCase(initevt.application());
         if ( initevt.topic().size() != 0 )
            strInTopic = IString::upperCase(initevt.topic());
         if ( (strApp == strInApp) && (strTopic == strInTopic) )
         {
       // check closed q, remove old conv if present
            IDDEServerConversationSet::Cursor myCurs(closedConversations());
            closedConversations().semaphor().lock();
            bool bFound = closedConversations().locateElementWithKey(
               (unsigned long)evt.parameter1(), myCurs);
            if (bFound)
            {
               pConv = closedConversations().elementAt(myCurs);
               closedConversations().removeAt(myCurs);
               delete pConv;
               ITRACE_DEVELOP("Client initiating new conversation before \
                  responding to close of previous conversation");
            }
            closedConversations().semaphor().unlock();
     // callback server
            ITRACE_DEVELOP("Topic and application match, callback the Server");
            bool bAccept = this->acceptConversation(
                                   (unsigned long)evt.parameter1(), initevt);

#ifdef IC_PM
            if (bAccept)      // server has accepted respond to client
            {
               ITRACE_DEVELOP("Server has accepted Conversation");
               CONVCONTEXT contxt = {0,0,0,0,0,0};   //32bit
               contxt.cb = sizeof(CONVCONTEXT);
               if (initevt.isCaseSensitive())
                  contxt.fsContext = DDECTXT_CASESENSITIVE;
            // fill in application and topic if necessary for broadcasts
               if ( initevt.application().size() == 0 )
                  strInApp = application();
               else strInApp = initevt.application();
               if ( initevt.topic().size() == 0 )
                  strInTopic = topic();
               else strInTopic = initevt.topic();
               MRESULT mr = WinDdeRespond((HWND)evt.parameter1(),
                                         (HWND)serverHandle(),
                                         (PSZ)strInApp,
                                         (PSZ)strInTopic,
                                         &contxt);
#endif  /* IC_PM */
#ifdef IC_WIN
            if (bAccept)      // server has accepted respond to client
            {
               ITRACE_DEVELOP("Server has accepted Conversation");

               ATOM aApp,
                    aTopic;

               // fill in application and topic
               if ( initevt.application().size() == 0 )
                  aApp = GlobalAddAtom( application() );
               else
                  aApp = GlobalAddAtom( initevt.application() );

               if ( initevt.topic().size() == 0 )
                  aTopic = GlobalAddAtom( topic() );
               else
                  aTopic = GlobalAddAtom( initevt.topic() );

               LRESULT rc = SendMessage( (HWND)evt.parameter1(),
                                         WM_DDE_ACK,
                                         (WPARAM)(HWND)serverHandle(),
                                         MAKELPARAM(aApp, aTopic ) );
//                                          PackDDElParam( WM_DDE_ACK, aApp, aTopic ) );
#endif  /* IC_WIN */

               IDDEServerConversation* pConv =
                   new IDDEServerConversation((unsigned long)evt.parameter1());
               conversations().semaphor().lock();
               conversations().add(pConv);
               conversations().semaphor().unlock();
            }
         }
      }
   }
}

/*------------------------------------------------------------------------------
| IDDETopicServer::handlePoke                                                  |
|                                                                              |
| This method is called by the dispatcher when the client tries to poke data   |
| to the server.                                                               |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handlePoke ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handlePoke");
   IDDEPokeEvent dEvt(evt);
   bool bFound, bSuccess;
   conversations().semaphor().lock();
   bFound = conversations().containsElementWithKey(
      (unsigned long)evt.parameter1());
   conversations().semaphor().unlock();

#ifdef IC_PM
   if (bFound)
   {
      bSuccess = this->pokeData( (unsigned long)evt.parameter1(), dEvt);
      unsigned short usStatus = dEvt.status();
      if (bSuccess)
      {
         usStatus |=  DDE_FACK;
         // ensure none of the negative flags are on
         usStatus &= ~(DDE_FBUSY | DDE_NOTPROCESSED);
      }
      else
         usStatus &= ~DDE_FACK;                       // ensure ack is negative
      PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (
                                                  (const char*)dEvt.item(),
                                                  (const char*)dEvt.format(),
                                                  usStatus,
                                                  (const char*)dEvt.pokedData(),
                                                  (dEvt.pokedData()).size());
      bSuccess = WinDdePostMsg((HWND)evt.parameter1(),
                               (HWND)serverHandle(),
                               WM_DDE_ACK,
                               pddes,
                               DDEPM_RETRY);
      if (!bSuccess)
      {                       // don't throw exception as client will also crash
         fClPostMsgFail = true;
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }
   }
   else
      ITRACE_DEVELOP("WM_DDE_POKE discarded, no matching conversation");

   IDDEInfo__freeMemory( (_DDESTRUCT*)(void*)evt.parameter2());
#endif  /* IC_PM */

#ifdef IC_WIN
   if (bFound)
   {
      // callback for poke data
      unsigned short usStatus = dEvt.status();
      bSuccess = this->pokeData( (unsigned long)evt.parameter1(), dEvt);

      if (bSuccess)
      {
         usStatus |=  IDDEInfo::acknowledge;
         // ensure none of the negative flags are on
         usStatus &= ~(IDDEInfo::busy | IDDEInfo::notProcessed);
      }
      else
         usStatus &= ~IDDEInfo::acknowledge;      // ensure ack is negative

      // reuse the item atom for ack
      bool bAtomAdded(false) ;
      ATOM aItem = (ATOM) IDDEInfo__atomFromString(dEvt.item(), &bAtomAdded) ;

      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK fifth" );
      bSuccess = PostMessage( (HWND)evt.parameter1(),
                              WM_DDE_ACK,
                              (WPARAM)(HWND)serverHandle(),
                              PackDDElParam( WM_DDE_ACK, ackBuffer, aItem ) );
//                               PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, aItem ) );
      if (!bSuccess)
      {                       // don't throw exception as client will also crash
         fClPostMsgFail = true;
//          IDDEInfo__freeMemory( hDdeAck );
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)(void *)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }

      // free up the data memory if not needed anymore
      if ( (dEvt.status() & IDDEInfo::releaseData) &&
           (usStatus & IDDEInfo::acknowledge) != 0 )
      {
         HGLOBAL pData = (HGLOBAL) IDDEInfo__loWord(WM_DDE_POKE, dEvt.parameter2().asUnsignedLong()) ;
         ITRACE_DEVELOP("freeing poke data " + IString((unsigned long)pData) ) ;
         IDDEInfo__freeMemory( pData );
      }
   }
   else
   {
      ITRACE_DEVELOP("WM_DDE_POKE discarded, no matching conversation");

      unsigned short usStatus = dEvt.status();
      usStatus &= ~IDDEInfo::acknowledge;
      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;
      bool bAtomAdded(false) ;
      ATOM uItem = (ATOM) IDDEInfo__atomFromString(dEvt.item(), &bAtomAdded) ;

      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK sixth" );
      if ( !( PostMessage( (HWND)evt.parameter1(),
                           WM_DDE_ACK,
                           (WPARAM)(HWND)serverHandle(),
                           PackDDElParam( WM_DDE_ACK, ackBuffer, uItem ) ) ) )
//                            PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, uItem ) ) ) )
      {
//          IDDEInfo__freeMemory( hDdeAck );
      }
   }

   FreeDDElParam( WM_DDE_ADVISE, evt.parameter2().asUnsignedLong());
#endif  /* IC_WIN */

}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleRequest                                               |
|                                                                              |
| This method is called by the dispatcher when the client conversation         |
| requests data from the server.                                               |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleRequest ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleRequest");
   IDDERequestDataEvent rdevt(evt);
   bool bFound, bSuccess, bEnded;
   IDDEServerConversation* pConv;
   IDDEServerConversationSet::Cursor myCurs(conversations());

#ifdef IC_PM
   IEvent myEvt1(serverHandle(),        // create dummy event for callback
                 (unsigned long)WM_DDE_TERMINATE,
                 IEventParameter1((unsigned long)serverHandle()),
                 IEventParameter2((void*)0));
   IDDEEndEvent endEvt(myEvt1, IDDEEndEvent::error);
   conversations().semaphor().lock();
   bFound = conversations().locateElementWithKey(
               (unsigned long)evt.parameter1(), myCurs);
   conversations().semaphor().unlock();
   if (bFound)
   {
      PDDESTRUCT pddes;
      unsigned short usStatus = 0;
      bSuccess = this->requestData( (unsigned long)evt.parameter1(), rdevt);
      if (bSuccess)          // send the Data
      {
         ITRACE_DEVELOP("Server has provided data");
         usStatus = DDE_FRESPONSE;
         if (rdevt.status() & DDE_FACKREQ)
            // request that client return an ack.
            usStatus |= DDE_FACKREQ;
         pddes = (PDDESTRUCT)buildDDEStruct ((const char*)rdevt.item(),
                                             (const char*)rdevt.format(),
                                             usStatus,
                                             (const char*)rdevt.buffer(),
                                             (rdevt.buffer()).size() );
         // build the event for adding to the ack q early as pddes will be
         // freed by PM
         IEvent myEvt(serverHandle(),
                      (unsigned long)WM_DDE_REQUEST,
                       IEventParameter1(evt.parameter1()),
                       IEventParameter2((void*)pddes));
         IDDEServerAcknowledgeEvent* pEvt =
            new IDDEServerAcknowledgeEvent(myEvt);

         bSuccess = WinDdePostMsg((HWND)evt.parameter1(),
                                  (HWND)serverHandle(),
                                  WM_DDE_DATA,
                                  pddes,
                                  DDEPM_RETRY);
         if (bSuccess)
         {
            //   add to ack Q if necessary
            conversations().semaphor().lock();
            if ( (rdevt.status() & DDE_FACKREQ) &&
                  conversations().locateElementWithKey(
                     (unsigned long)evt.parameter1(), myCurs) )
               conversations().elementAt(myCurs)->addEvent(pEvt);
            else
               delete pEvt;
            conversations().semaphor().unlock();
         }
         else
         {                   // don't throw exception as client will also crash
            // set flag that posting a message has failed
            fClPostMsgFail = true;
            bEnded = true;
            try {
               this->endConversation((unsigned long)evt.parameter1());
            }
            catch (IInvalidRequest& excReq) {
               bEnded = false;
            }
            if (bEnded)
               this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
            fClPostMsgFail = false;
         }
      }
      else          // send a Nack, server has not provided data
      {
         ITRACE_DEVELOP("Server has NOT provided data");
         usStatus = rdevt.status();
         usStatus &= ~DDE_FACK;                       // ensure ack is negative
         pddes = (PDDESTRUCT)buildDDEStruct ((const char*)rdevt.item(),
                                             (const char*)rdevt.format(),
                                             usStatus,
                                             0,
                                             0);
         bSuccess = WinDdePostMsg((HWND)evt.parameter1(),
                                  (HWND)serverHandle(),
                                  WM_DDE_ACK,
                                  pddes,
                                  DDEPM_RETRY);
         if (!bSuccess)
         {                   // don't throw exception as client will also crash
            fClPostMsgFail = true;
            bEnded = true;
            try {
               this->endConversation((unsigned long)evt.parameter1());
            }
            catch (IInvalidRequest& excReq) {
               bEnded = false;
            }
            if (bEnded)
               this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
            fClPostMsgFail = false;
         }
      }  // end of else for sending NACK
   }
   else
      ITRACE_DEVELOP("WM_DDE_REQUEST discarded, no matching conversation");
   IDDEInfo__freeMemory( (_DDESTRUCT*)(void*)evt.parameter2());
#endif  /* IC_PM */

#ifdef IC_WIN
   IEvent myEvt1(serverHandle(),        // create dummy event for callback
                 (unsigned long)WM_DDE_TERMINATE,
                 IEventParameter1((unsigned long)(void *)serverHandle()),
                 IEventParameter2((void*)0));
   IDDEEndEvent endEvt(myEvt1, IDDEEndEvent::error);

   conversations().semaphor().lock();
   bFound = conversations().locateElementWithKey(
               (unsigned long)evt.parameter1(), myCurs);
   conversations().semaphor().unlock();
   ITRACE_DEVELOP( "DBug: bFound=" + IString(bFound));

   if (bFound)
   {
//mkb      UINT uFormat( LOWORD(evt.parameter2()));
//mkb      UnpackDDElParam( WM_DDE_REQUEST, evt.parameter2(), &uFormat, &aItem );
      UINT   aItem( HIWORD(evt.parameter2()));
      unsigned short usStatus = 0;

      // call the server callback of request data
      bSuccess = this->requestData( (unsigned long)evt.parameter1(), rdevt);

      if (bSuccess)          // send the Data
      {
         ITRACE_DEVELOP("Server has provided data");
         usStatus = IDDEInfo::responseFromReq | IDDEInfo::releaseData;

         if ( rdevt.status() & IDDEInfo::acknowledgeRequested )
         {
            // server callback requests client return an ack.
            usStatus |= IDDEInfo::acknowledgeRequested;
         }

         HGLOBAL hDdeData = buildDDEStruct( IDDEData,
                                            (const char*)rdevt.format(),
                                            usStatus,
                                            (const char*)rdevt.buffer(),
                                            (rdevt.buffer()).size() );

         // build the event for adding to the ack q early as pddes will be
         // freed by system
//mkb         IEvent myEvt( serverHandle(), (unsigned long)WM_DDE_REQUEST, IEventParameter1(evt.parameter1()), IEventParameter2((unsigned long)MAKELPARAM( uFormat, aItem)) );
         IDDEServerAcknowledgeEvent* pEvt =
//mkb                     new IDDEServerAcknowledgeEvent(myEvt);
                          new IDDEServerAcknowledgeEvent(evt);

         LPARAM lParam( PackDDElParam( WM_DDE_DATA, (UINT)hDdeData, aItem ));
         ITRACE_DEVELOP( "DBug: post data, lParam=" + IString((unsigned long)lParam)
             + ", hDdeData=" + IString((unsigned long)hDdeData)
             + ", aItem=" + IString((unsigned long)aItem));
         bSuccess = PostMessage( (HWND)evt.parameter1(),
                                 WM_DDE_DATA,
                                 (WPARAM)(HWND)serverHandle(),
                                 lParam );
         if (bSuccess)
         {
            // add to ack Q if necessary
            conversations().semaphor().lock();
            if ( (rdevt.status() & IDDEInfo::acknowledgeRequested) &&
                  conversations().locateElementWithKey(
                     (unsigned long)evt.parameter1(), myCurs) )
               {
               ITRACE_DEVELOP( "DBug: ACK from client is requested." );
               conversations().elementAt(myCurs)->addEvent(pEvt);
               }
            else
               {
               ITRACE_DEVELOP( "DBug: NO response from client expected." );
               delete pEvt;
               }
            conversations().semaphor().unlock();
         }
         else
         {                 // don't throw exception as client will also crash
            // set flag that posting a message has failed
            fClPostMsgFail = true;
            IDDEInfo__freeMemory( hDdeData );
            bEnded = true;
            try {
               this->endConversation((unsigned long)evt.parameter1());
            }
            catch (IInvalidRequest& excReq) {
               bEnded = false;
            }
            if (bEnded)
               this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
            fClPostMsgFail = false;
         }
      }
      else          // send a -ve ack, server has not provided data
      {
         ITRACE_DEVELOP("Server has NOT provided data");
         usStatus = rdevt.status();
         usStatus &= ~IDDEInfo::acknowledge;

         UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

         ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK seventh" );
         bSuccess = PostMessage( (HWND)evt.parameter1(),
                                 WM_DDE_ACK,
                                 (WPARAM)(HWND)serverHandle(),
                                 PackDDElParam( WM_DDE_ACK, ackBuffer, aItem ));
//                                  PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, aItem ) );
         if (!bSuccess)
         {                   // don't throw exception as client will also crash
            fClPostMsgFail = true;
//             IDDEInfo__freeMemory( hDdeAck );
            bEnded = true;
            try {
               this->endConversation((unsigned long)evt.parameter1());
            }
            catch (IInvalidRequest& excReq) {
               bEnded = false;
            }
            if (bEnded)
               this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
            fClPostMsgFail = false;
         }
      }  // end of else for sending -ve ack
   }
   else
   {
      ITRACE_DEVELOP("WM_DDE_REQUEST discarded, no matching conversation");

      unsigned short usStatus = 0;
      usStatus &= ~IDDEInfo::acknowledge;
      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

      UINT uFormat, uItem;
      UnpackDDElParam( WM_DDE_REQUEST, evt.parameter2(), &uFormat, &uItem );
      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK eighth" );
      if ( !( PostMessage( (HWND)evt.parameter1(),
                           WM_DDE_ACK,
                           (WPARAM)(HWND)serverHandle(),
                           PackDDElParam( WM_DDE_ACK, ackBuffer, uItem ) ) ) )
//                            PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, uItem ) ) ) )
      {
//          IDDEInfo__freeMemory( hDdeAck );
      }
   }
#endif  /* IC_WIN */
}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleTerminate                                             |
|                                                                              |
| Handler for terminating a conversation with the client. If the client        |
| initiated the terminate then the server must respond with a WM_DDE_TERMINATE |
| message. If the server initiated the termination then we are here because    |
| the client responded with the WM_DDE_TERMINATE message and no response is    |
| required.                                                                    |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleTerminate ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleTerminate");
   IDDEServerConversationSet::Cursor myCurs(conversations());
   IDDEServerConversation* pConv = 0;
   bool bFound = false;
#ifdef IC_PM
   IEvent myEvt(serverHandle(),        // create dummy event for callback
                (unsigned long)WM_DDE_TERMINATE,
                IEventParameter1((unsigned long)serverHandle()),
                IEventParameter2((void*)0));
#endif
#ifdef IC_WIN
   IEvent myEvt( serverHandle(),        // create dummy event for callback
                 (unsigned long)WM_DDE_TERMINATE,
                 IEventParameter1((unsigned long)(void *)serverHandle()),
                 IEventParameter2((void*)0));
#endif
   //remove all hot links first even if we don't have a conversation to avoid a
   // deadlock
   removeLink(IString(""), IString(""), (unsigned long)evt.parameter1());
   conversations().semaphor().lock();  // serialize access to the set
   // look to see if we are in conversation with this client.
   bFound = conversations().locateElementWithKey(
               (unsigned long)evt.parameter1(), myCurs);
   if (bFound)
   {
      pConv = conversations().elementAt(myCurs);
      conversations().removeAt(myCurs);
      conversations().semaphor().unlock();
#ifdef IC_PM
      bool bSuccess = WinDdePostMsg((HWND)evt.parameter1(),
                                       (HWND)serverHandle(),
                                       WM_DDE_TERMINATE,
                                       0,
                                       DDEPM_RETRY);
#endif
#ifdef IC_WIN
      bool bSuccess = PostMessage( (HWND)evt.parameter1(),
                                      WM_DDE_TERMINATE,
                                      (WPARAM)(HWND)serverHandle(),
                                      0L );
#endif
      if (!bSuccess)
         ITRACE_DEVELOP("WM_DDE_TERMINATE failure on attempt to acknowledge \
            end of conversation.");
      delete pConv;
      IDDEEndEvent endEvt(myEvt, IDDEEndEvent::client);
      this->conversationEnded( (unsigned long)evt.parameter1(), endEvt);
   }
   else
   {
// check closed q, see if we initiated close and remove old conv if present
      conversations().semaphor().unlock();
      ITRACE_DEVELOP(
         "Conversation for client ID not found on terminate request");
      IDDEServerConversationSet::Cursor myCurs1(closedConversations());
      closedConversations().semaphor().lock();
      bFound = closedConversations().locateElementWithKey(
         (unsigned long)evt.parameter1(), myCurs1);
      if (bFound)
      {
         pConv = closedConversations().elementAt(myCurs1);
         closedConversations().removeAt(myCurs1);
         delete pConv;
         closedConversations().semaphor().unlock();
         IDDEEndEvent endEvt1(myEvt, IDDEEndEvent::server);
         this->conversationEnded( (unsigned long)evt.parameter1(), endEvt1);
      }
      else {
         ITRACE_DEVELOP(
            "Unknown client sent terminate, no matching conversation");
         closedConversations().semaphor().unlock();
      }
   }
}

/*------------------------------------------------------------------------------
| IDDETopicServer::handleUnadvise                                              |
|                                                                              |
| Handler for the clients request to end a hot link.                           |
------------------------------------------------------------------------------*/
void IDDETopicServer :: handleUnadvise ( const IEvent& evt )
{
   IMODTRACE_DEVELOP("DDETopServ::handleUnadvise");
   IDDEEvent dEvt(evt);
   bool bFound = 0;
   bool bSuccess = 0;
   bool bSuccess1 = 0;
   // Make sure that we are in conversation with this client.
   conversations().semaphor().lock();
   bFound = conversations().containsElementWithKey(
      (unsigned long)evt.parameter1());
   conversations().semaphor().unlock();

#ifdef IC_PM
   if (bFound)
   {
      unsigned short usStatus = 0;
      // Remove the hot link from the server's hot link set.
      bSuccess = removeLink(dEvt.item(), dEvt.format(),
         (unsigned long)evt.parameter1());
      if (bSuccess)
      {
         // send a positive ack back to the client
         usStatus |= DDE_FACK;
         this->hotLinkEnded( (unsigned long)evt.parameter1(), dEvt);
      }
      PDDESTRUCT pddes = (PDDESTRUCT)buildDDEStruct (dEvt.item(),
                                                     dEvt.format(),
                                                     usStatus,
                                                     0,
                                                     0);
      bool bSuccess1 = WinDdePostMsg((HWND)evt.parameter1(),
                                        (HWND)serverHandle(),
                                        WM_DDE_ACK,
                                        pddes,
                                        DDEPM_RETRY);
      if (!bSuccess1)
      {  // don't throw exception as client will also crash
         // if sending ack to client failed terminate conversation.
         fClPostMsgFail = true;
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }
   }
   else
      ITRACE_DEVELOP("WM_DDE_UNADVISE discarded, no matching conversation");
   IDDEInfo__freeMemory( (_DDESTRUCT*)(void*)evt.parameter2());
#endif   /* IC_PM */

#ifdef IC_WIN
   if (bFound)
   {
      UINT uLo;
      UINT aItem;
      UnpackDDElParam( WM_DDE_UNADVISE, evt.parameter2(), &uLo, &aItem );

      unsigned short usStatus = 0;
      // Remove the hot link from the server's hot link set.
      bSuccess = removeLink( dEvt.item(), dEvt.format(),
                             (unsigned long)evt.parameter1() );
      if (bSuccess)
      {
         // send a positive ack back to the client
         usStatus |= IDDEInfo::acknowledge;
         // call the server callback
         this->hotLinkEnded( (unsigned long)evt.parameter1(), dEvt );
      }

      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK ninth" );
      bool bSuccess1 = PostMessage( (HWND)evt.parameter1(),
                                       WM_DDE_ACK,
                                       (WPARAM)(HWND)serverHandle(),
                                       PackDDElParam( WM_DDE_ACK, ackBuffer, aItem ) );
//                                        PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, aItem ) );
      if (!bSuccess1)
      {  // don't throw exception as client will also crash
         // if sending ack to client failed terminate conversation.
         fClPostMsgFail = true;
//          IDDEInfo__freeMemory( hDdeAck );
         IEvent myEvt(serverHandle(),        // create dummy event for callback
                      (unsigned long)WM_DDE_TERMINATE,
                      IEventParameter1((unsigned long)(void *)serverHandle()),
                      IEventParameter2((void*)0));
         IDDEEndEvent endEvt(myEvt, IDDEEndEvent::error);
         bool bEnded = true;
         try {
            this->endConversation((unsigned long)evt.parameter1());
         }
         catch (IInvalidRequest& excReq) {
            bEnded = false;
         }
         if (bEnded)
            this->conversationEnded((unsigned long)evt.parameter1(),endEvt);
         fClPostMsgFail = false;
      }
   }
   else
   {
      ITRACE_DEVELOP("WM_DDE_UNADVISE discarded, no matching conversation");

      unsigned short usStatus = 0;
      usStatus &= ~IDDEInfo::acknowledge;
      UINT ackBuffer = IDDEInfo__buildDDEAck(usStatus) ;

      UINT uLo, uItem;
      UnpackDDElParam( WM_DDE_UNADVISE, evt.parameter2(), &uLo, &uItem );

      ITRACE_DEVELOP( "DBug: postMessage WM_DDE_ACK tenth" );
      if ( !( PostMessage( (HWND)evt.parameter1(),
                           WM_DDE_ACK,
                           (WPARAM)(HWND)serverHandle(),
                           PackDDElParam( WM_DDE_ACK, ackBuffer, uItem ) ) ) )
//                            PackDDElParam( WM_DDE_ACK, (UINT)hDdeAck, uItem ) ) ) )
      {
//          IDDEInfo__freeMemory( hDdeAck );
      }
   }
#endif   /* IC_WIN */

}

/*------------------------------------------------------------------------------
| IDDETopicServer::removeLink                                                  |
|                                                                              |
| Remove hot links registered by the client in conversation. If the item name  |
| is an empty string then remove all hot links. If the format is an empty      |
| string then remove all hot links for that item. If both the item and format  |
| are specified only remove that hot link from the server's hot link set.      |
------------------------------------------------------------------------------*/
bool IDDETopicServer :: removeLink ( IString item,
                                        IString format,
                                        unsigned long conversationId )
{
   IMODTRACE_DEVELOP("DDETopServ::removeLink");
   // flag to indicate a link was found and removed, used for return value
   bool bFound = 0;
   // flag to stop iterating the IDDEServerHotLinkItemSet
   bool bFinished = 0;
   bool bMatch = 0;
   // flag to indicate an IDDEServerHotLinkItem with matching item and
   // format was found
   unsigned long tmpId = conversationId;
   IString inItem = IString::upperCase(item);
   IString inFormat = IString::upperCase(format);
   IString myItem;
   IString myFormat;
#pragma enum(4)
   enum checkLevel {all, itemonly, none};
     // all      - check both item and format
     // itemonly - check only the item, can be of any format
     // none     - any hot link matches
#pragma enum(pop)
   checkLevel chkLvl;
   if (item == "")
      chkLvl = none;
   else if (format == "")
      chkLvl = itemonly;
   else chkLvl = all;
// Find the hot link(s)
   hotLinkItems().semaphor().lock();  // serialize access to the set
   IDDEServerHotLinkItemSet::Cursor myCurs(hotLinkItems());
   IDDEServerHotLinkItem* pItem = 0;
   for(myCurs.setToFirst();
       (myCurs.isValid()) && (bFinished == false);
       bMatch = false)
   {
      pItem = hotLinkItems().elementAt(myCurs);
      myItem = IString::upperCase(pItem->item());
      myFormat = IString::upperCase(pItem->format());
      switch (chkLvl)
      {
         case none:
         {
            // all hot links match
            bMatch = true;
            break;
         }
         case itemonly:
         {
            // all items that match are removed regardless
            // of format.
            if (inItem == myItem)
               bMatch = true;
            break;
         }
         case all:
         {
            if ( (inItem == myItem) && (inFormat == myFormat) )
            {
               bMatch = true;
               // Since we are matching both the format and item
               // we know that there can only be one match and
               // we are finished.
               bFinished = true;
            }
            break;
         }
      } /* endswitch */
      if (bMatch)  // item and format match
         pItem->hotLinks().removeAll(&IDdeleteElemSHLS, &tmpId);
      if (!tmpId)
      {
         tmpId = conversationId;
         bFound = 1;
      }
      myCurs.setToNext();
   } // end for
//  iterate item's list and delete those with 0 elements
   hotLinkItems().removeAll(&IDdeleteElemSHLIS, &tmpId);
   hotLinkItems().semaphor().unlock();
   return bFound;
}

/*------------------------------------------------------------------------------
| IDDETopicServer::dispatchEventFromQueue                                      |
|                                                                              |
| If useEventThread was enabled then the events are pulled of the queue and    |
| are dispatched.                                                              |
------------------------------------------------------------------------------*/
void IDDETopicServer :: dispatchEventFromQueue ( )
{
   IMODTRACE_DEVELOP("DDETopServ::dsptchEvtFromQ");

#ifdef IC_PM
   HAB hab = WinInitialize(0);
   REQUESTDATA reqData;
   unsigned long ulSize;
   unsigned long ulRc;
   void* pv;
   IEvent* pEvt;
   BYTE byt;
   IDDEFreeSemaphore safeDtor(ulClSemaphore);
   while (true) {
      ulRc = DosReadQueue(queueHandle(),
                          &reqData,
                          &ulSize,
                          &pv,
                          (unsigned long)0,
                          (unsigned long)0,
                          &byt,
                          (HEV)0);
      if ( ulRc == ERROR_QUE_INVALID_HANDLE )
         break;
      pEvt = (IEvent*)(pv);
      switch (pEvt->eventId())
      {
         case WM_DDE_ACK:
         {
            handleAck(*pEvt);
            break;
         }
         case WM_DDE_ADVISE:
         {
            handleAdvise(*pEvt);
            break;
         }
         case WM_DDE_EXECUTE:
         {
            handleExecute(*pEvt);
            break;
         }
         case WM_DDE_POKE:
         {
            handlePoke(*pEvt);
            break;
         }
         case WM_DDE_REQUEST:
         {
            handleRequest(*pEvt);
            break;
         }
         case WM_DDE_TERMINATE:
         {
            handleTerminate(*pEvt);
            break;
         }
         case WM_DDE_UNADVISE:
         {
            handleUnadvise(*pEvt);
            break;
         }

      } /* endswitch */
      delete pEvt;
   } /* endwhile */
#endif  /* IC_PM */

#ifdef IC_WIN
   unsigned long ulSize;
   unsigned long ulRc = 0;
   IEvent* pEvt;

   while (true) {
      if ( !ReadFile(hReadPipe, &pEvt, sizeof(IEvent*), &ulSize, NULL)
         || ulSize != sizeof(IEvent*))
        { ulRc = GetLastError();
          if ( ulRc == ERROR_BROKEN_PIPE )
            break;
        }
::asDebugInfo( pEvt );

      switch (pEvt->eventId())
      {
         case WM_DDE_ACK:
         {
            handleAck(*pEvt);
            break;
         }
         case WM_DDE_ADVISE:
         {
            handleAdvise(*pEvt);
            break;
         }
         case WM_DDE_EXECUTE:
         {
            handleExecute(*pEvt);
            break;
         }
         case WM_DDE_POKE:
         {
            handlePoke(*pEvt);
            break;
         }
         case WM_DDE_REQUEST:
         {
            handleRequest(*pEvt);
            break;
         }
         case WM_DDE_TERMINATE:
         {
            handleTerminate(*pEvt);
            break;
         }
         case WM_DDE_UNADVISE:
         {
            handleUnadvise(*pEvt);
            break;
         }
      } /* endswitch */
      delete pEvt;
      pEvt = 0;
   } /* endwhile */
   if (hFinish)
      SetEvent(hFinish);
#endif  /* IC_WIN */
}

/*------------------------------------------------------------------------------
| IDDETopicServer::buildDDEStruct                                              |
------------------------------------------------------------------------------*/
#ifdef IC_PM
void* IDDETopicServer :: buildDDEStruct ( const char*    itemName,
                                          const char*    dataFormat,
                                          unsigned short status,
                                          const void*    xferData,
                                          unsigned long  dataLength)
{
   bool bAtomAdded = false;
   void* pStruct = IDDEInfo__buildDDEStruct( itemName,
                                             dataFormat,
                                             status,
                                             xferData,
                                             dataLength,
                                             &bAtomAdded );
   if (bAtomAdded)
   {
      formats().add(((PDDESTRUCT)pStruct)->usFormat);
      ITRACE_DEVELOP("Added atom to format list.");
   }
   return pStruct;
}
#endif  /* IC_PM */

#ifdef IC_WIN
void* IDDETopicServer :: buildDDEStruct ( int            structType,
                                          const char*    dataFormat,
                                          unsigned short status,
                                          const void*    xferData,
                                          unsigned long  dataLength )
{
   bool bAtomAdded = false;
   void* pStruct = IDDEInfo__buildDDEStruct( structType,
                                             dataFormat,
                                             status,
                                             xferData,
                                             dataLength,
                                             &bAtomAdded );
   return pStruct;
}
#endif  /* IC_WIN */

/*------------------------------------------------------------------------------
| IDDEServerHotLinkSet::IDDEServerHotLinkSet                                   |
------------------------------------------------------------------------------*/
IDDEServerHotLinkSet :: IDDEServerHotLinkSet ( )
        : IVPtrSet<IDDEServerHotLink*>(10)
{ }

/*------------------------------------------------------------------------------
| IDDEServerHotLinkSet::~IDDEServerHotLinkSet                                  |
------------------------------------------------------------------------------*/
IDDEServerHotLinkSet :: ~IDDEServerHotLinkSet ( )

{ }

/*------------------------------------------------------------------------------
| IDDEServerHotLinkItem::IDDEServerHotLinkItem                                 |
------------------------------------------------------------------------------*/
IDDEServerHotLinkItem :: IDDEServerHotLinkItem  ( IDDEServerHotLinkEvent& evt )
                 : strClItem(evt.item()),
                   strClFormat(evt.format())

{
   pHLSetCl = new IDDEServerHotLinkSet();
}

IDDEServerHotLinkItem :: ~IDDEServerHotLinkItem ( )
{
   // remove all elements of hotlinkset and delete
   hotLinks().removeAll(&IDdeleteElemSHLS);
   delete &hotLinks();
}

/*------------------------------------------------------------------------------
| IDDEServerHotLinkItemSet::IDDEServerHotLinkItemSet                           |
------------------------------------------------------------------------------*/
IDDEServerHotLinkItemSet :: IDDEServerHotLinkItemSet ( )
              : IVPtrSet<IDDEServerHotLinkItem*>(5)
{ }

/*------------------------------------------------------------------------------
| IDDEServerHotLinkItemSet::~IDDEServerHotLinkItemSet                          |
------------------------------------------------------------------------------*/
IDDEServerHotLinkItemSet :: ~IDDEServerHotLinkItemSet ( )
{ }

/*------------------------------------------------------------------------------
| IDDEServerConversationSet::IDDEServerConversationSet                         |
------------------------------------------------------------------------------*/
IDDEServerConversationSet :: IDDEServerConversationSet ( )
           : IKeySet<IDDEServerConversation*, unsigned long>(5)
{ }

/*------------------------------------------------------------------------------
| IDDEServerConversationSet::~IDDEServerConversationSet                        |
------------------------------------------------------------------------------*/
IDDEServerConversationSet :: ~IDDEServerConversationSet ( )
{ }

#endif /* IC_PMWIN  */
