#ifndef _ISTATICS_HPP_
#define _ISTATICS_HPP_

// <prologue>
// <copyright>
/*****************************************************************************
* FILE NAME: istatics.hpp                                                    *
*                                                                            *
* COPYRIGHT:                                                                 *
*   IBM Open Class Library                                                   *
*   (C) Copyright International Business Machines Corporation 1992, 1999     *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.       *
*   US Government Users Restricted Rights - Use, duplication, or disclosure  *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                   *
*                                                                            *
*****************************************************************************/
// </copyright>
//
// Revision: 43 1.9 source/core/utils/istatics.hpp, utilities, ioc.v400, 001006 
//
// </prologue>
// <header>

// <include-files>
#include <ilanglvl.hpp>
#include <ireslock.hpp>
// </include-files>

// <forward-declarations>
class IMRefCounted;
// </forward-declarations>

// <pragma namemangling/enum/pack/align>
#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif /* __IBMCPP__ */

#pragma enum(4)

#if defined(__TOS_WIN__) || defined(__TOS_OS2__) || defined(__TOS_OS400__)
  #pragma pack(push,4)
#endif
#if defined(__TOS_AIX__)
  #pragma options align=power
#endif
// </pragma namemangling/enum/pack/align>

// <defines>
// </defines>

// <class-declarations>

// ---------------------------------------------------------------------------
// IStaticObject provides the virtual destructor for the derived template
// helper class.
// ---------------------------------------------------------------------------

class IStaticObject
{
public:
    virtual ~IStaticObject() {}
};

// ---------------------------------------------------------------------------
// IStaticObjectHelper is a template class derived from IStaticObject.
// The sole reason for it being a template is to let the compiler
// know what type of static object it needs to destroy.
// ---------------------------------------------------------------------------

template<class T>
class IStaticObjectHelper : public IStaticObject
{
public:
    IStaticObjectHelper(T* data) : fObject(data) {}
    virtual ~IStaticObjectHelper() { delete fObject; }
private:
    T* fObject;
};

// ---------------------------------------------------------------------------
// IStaticObjectArrayHelper is a template class derived from IStaticObject.
// The only difference of this class from IStaticObjectHelper is that this
// class calls the vector delete.
// ---------------------------------------------------------------------------

template<class T>
class IStaticObjectArrayHelper : public IStaticObject
{
public:
    IStaticObjectArrayHelper(T* data) : fObject(data) {}
    virtual ~IStaticObjectArrayHelper() { delete [] fObject; }
private:
    T* fObject;
};

// ---------------------------------------------------------------------------
// IStaticObjectManager is the main storage manager.
//
// This is how the whole thing works:
//
// IStaticObjectManager maintains an array of IStaticObject pointers
// (initialized through the adopt() method). When DLL terminates, this
// class goes through the array and delete each IStaticObject in REVERSE
// order.
//
// When an IStaticObject is deleted, the virtual destructor of the
// templatized IStaticObjectHelper object will get invoked. This in turn
// delete the static object in a type-safe manner.
// ---------------------------------------------------------------------------

class IStaticObjectManager
{
public:
    static void adopt(IStaticObject* obj);
    static void manageRefCountOf ( IMRefCounted* obj );
    ~IStaticObjectManager();
    IStaticObjectManager();
private:
    void adoptObject(IStaticObject* obj);
    void prepareStorage();
    IPrivateResource fLock;
    IStaticObject** fData;
    unsigned long fCount;
    unsigned long fMaxCount;
};

// ---------------------------------------------------------------------------
// Functions for clients to use:
//   adoptStaticObject
//      manages deletion of the specified object at program termination
//   adoptStaticObjectArray
//      manages deletion of the specified array object at program
//      termination
//   IStaticObjectManager::manageRefCountOf
//      manages decrementing a reference count of the specified
//      reference-counted object at program termination
//
// Note: If specialized clean up processing is needed for some class,
//       the client can create a new class derived from IStaticObject
//       which performs that clean-up in its destructor, and pass an
//       object of that class to IStaticObjectManager::adopt. This is
//       essentially what all of the above functions do.
// ---------------------------------------------------------------------------
template<class T> inline void adoptStaticObject(T* obj);
template<class T> inline void adoptStaticObjectArray(T* obj);

// </class-declarations>

// <pragma align/pack/enum/namemangling restore>
#if defined(__TOS_WIN__) || defined(__TOS_OS2__) || defined(__TOS_OS400__)
  #pragma pack(pop)
#endif
#if defined(__TOS_AIX__)
  #pragma options align=reset
#endif

#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling(pop)
#endif /* __IBMCPP__ */
// </pragma align/pack/enum/namemangling restore>

// <inline-functions>
#include <istatics.inl>
// </inline-functions>

// </header>
#endif // _ISTATICS_HPP_
