// <copyright>
/*****************************************************************************
* FILE NAME: istatics.cpp                                                    *
*                                                                            *
* COPYRIGHT:                                                                 *
*   IBM Open Class Library                                                   *
*   (C) Copyright International Business Machines Corporation 1992, 1999     *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.       *
*   US Government Users Restricted Rights - Use, duplication, or disclosure  *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                   *
*                                                                            *
*****************************************************************************/
// </copyright>
//
// Revision: 42 1.4.1.3 source/core/utils/istatics.cpp, utilities, ioc.v400, 001006 
//

// <include-files>
#include <istatics.hpp>
#include <imrefcnt.hpp>
// </include-files>

#pragma priority(-2147482524)
static IStaticObjectManager gStaticObjectManager;

IStaticObjectManager::IStaticObjectManager()
        : fData(0), fCount(0), fMaxCount(32)
{}

IStaticObjectManager::~IStaticObjectManager()
{
    IResourceLock ctrLock(fLock);
    // Delete static objects in reverse order
    // Optimization, avoid arithmetic within the loop
    for (--fCount; fCount != (unsigned long)(-1); --fCount)
        delete fData[fCount];
    delete [] fData;
}

void
IStaticObjectManager::prepareStorage()
{
    if (fData == 0) {
        fData = new IStaticObject*[fMaxCount];
    } else {
        if (fCount >= fMaxCount) {
            fMaxCount = fMaxCount * 2;
            IStaticObject** tmpData = new IStaticObject*[fMaxCount];
            memcpy(tmpData, fData, fCount * sizeof(IStaticObject *));
            IStaticObject** oldData = fData;
            fData = tmpData;
            delete [] oldData;
        }
    }
}

void
IStaticObjectManager::adoptObject(IStaticObject* obj)
{
    IResourceLock ctrLock(fLock);
    prepareStorage();
    fData[fCount++] = obj;
}

void
IStaticObjectManager::adopt(IStaticObject* obj)
{
    gStaticObjectManager.adoptObject(obj);
}

// ---------------------------------------------------------------------------
// IStaticRefCountRemover        
// This is a helper class for managing the deletion of reference-counted
// static objects through their reference-counting mechanism.
// ---------------------------------------------------------------------------

class IStaticRefCountRemover : public IStaticObject
{
public:
    IStaticRefCountRemover ( IMRefCounted* staticObject );
    virtual ~IStaticRefCountRemover ( );
private:
    IMRefCounted* fStaticObject;
};

IStaticRefCountRemover::IStaticRefCountRemover
                           ( IMRefCounted* staticObject )
  : fStaticObject( staticObject )
{ }

IStaticRefCountRemover::~IStaticRefCountRemover ( )
{
    if ( fStaticObject )
    {
       fStaticObject->removeRef();
    }
}
void IStaticObjectManager::manageRefCountOf ( IMRefCounted* obj )
{
    // Allow IStaticObjectManager to manage the deletion of
    // reference-counted objects through their reference-counting
    // mechanism, rather than directly calling operator delete.
    // This avoids inadvertantly deleting an object that another
    // piece of code is still using.
    IStaticRefCountRemover
       *refCountRemover = new IStaticRefCountRemover( obj );
    gStaticObjectManager.adoptObject( refCountRemover );
}
