// Revision: 77 1.61.2.2 source/core/text/itext/itxtstor.cpp, text, ioc.v400, 001006 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//      ITxtStor.cpp    (ITextStorage.C)
//
//      Primary author: Richard Gillam
//
//      Function: Abstract base class for the reference-counted storage vehicle for IText.
//
/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/

//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#include "itxtstor.hpp"
#include "istxstor.hpp"
#include "igtxstor.hpp"
#include "itext.hpp"
#include "istylsto.hpp"
#include "itrancod.hpp"
#include "istrmmod.hpp"
#include "igstream.hpp"
#include <iexcept.hpp>
#include <istring.hpp>
#include <itrace.hpp>
#include "iunicomp.hpp"
#include <istatics.hpp>

#ifdef IC_PAGETUNE
        #define _ITXTSTOR_CPP
        #include <ipagetun.h>
#endif

// These are defined in itext.cpp
extern ITranscoder* gDefaultTranscoder;
extern bool gInitDefaultTranscoderDone;
extern void gInitDefaultTranscoder();

//========================================================================================
// Utility functions
//========================================================================================

void
writeCompressedLong( unsigned long      inValue,
                     IDataStream&       outStream,
                     size_t&            outBytesWritten )
{
  typedef unsigned char Byte;

  size_t        bytesWritten = 0;


  if( inValue < 0x80 )
        {
          (Byte) (inValue & 0xFF) >>= outStream;
          bytesWritten = 1;
        }
  else if( inValue < 0x4000 )
        {
          Byte bytes[ 2 ] = { (Byte) (0x80 | (inValue >> 7)),
                                                  (Byte) (inValue & 0x7F) };
          outStream.writeBytes( (const char*) bytes, 2 );
          bytesWritten = 2;
          }
  else if( inValue < 0x40000000 )
        {
          Byte bytes[ 4 ] = { (Byte) (0x80 | (inValue >> 23)),
                                                  (Byte) (0x80 | ( (inValue >> 16) & 0x7F)),
                                                  (Byte) ( (inValue >> 8) & 0xFF),
                                                  (Byte) (inValue & 0xFF) };
          outStream.writeBytes( (const char*) bytes, 4 );
          bytesWritten = 4;
        }
  else
        {
          // value out of range.
        }

  outBytesWritten = bytesWritten;
  return;
}

unsigned long
readCompressedLong( IDataStream&        inStream,
                    size_t&             outBytesRead )
{
  typedef unsigned char Byte;

  Byte                  byte1           = 0x00;
  Byte                  byte2           = 0x00;
  Byte                  byte3           = 0x00;
  Byte                  byte4           = 0x00;
  unsigned long         result          = 0L;


  byte1 <<= inStream;

  if( byte1 & 0x80 )
        {
          byte1 &= 0x7F;
          byte2 <<= inStream;

          if( byte2 & 0x80 )
                {
                  Byte bytes[2];
                  inStream.readBytes( (char*) bytes, 2 );

                  byte2         &= 0x7F;
                  byte3         = bytes[0];
                  byte4         = bytes[1];
                  result        = (byte1 << 23) | (byte2 << 16) | (byte3 << 8) | byte4;
                }
          else
                result = (byte1 << 7) | byte2;
        }
  else
        result = byte1;

  return result;
}


//========================================================================================
// CLASS ITextStorage:  The abstract base class for the reference-counted storage vehicle
// for IText.  Currently, we declare three concrete subclasses of ITextStorage:
// ISimpleTextStorage, which is used for short strings and individual nodes of long
// strings; IFancyTextStorage, which is used for long strings (it manages a group of
// ISimpleTextStorages); and IStreamInTextStorage, which is used to read either of the
// other two types in from a stream.  Currently, this is a closed-ended set of
// subclasses, but we intend to expose ITextStorage as part of our subclassing API in
// the future.
//========================================================================================

ISynchronized<ITextStorage*> ITextStorage::fgEmptyTextStorage;

const long kInvalidHash = 0;
const long kMaxHashCount = 32;  // maximum number of characters to consider when computing a
                                                                // hash value

//ISynchronized<ITranscoder*> gDefaultTranscoder;    ---> see itext.cpp
IStreamModule   gITextStreamModule("itext");
//StreamableDefinitionsMacro(ITextStorage, gITextStreamModule);


ITextStorage::ITextStorage()
: IMRefCounted(),
  IMStreamable(),
  fLength(0),
  fCapacity(0),
  f8BitCharCache(0),
  fHash(kInvalidHash),
  fStyles(0)
{
}

ITextStorage::ITextStorage(const ITextStorage&  that)
: IMRefCounted(),
  IMStreamable(),
  fLength(0),
  fCapacity(0),
  f8BitCharCache(0),
  fHash(kInvalidHash),
  fStyles(0)
{
        // DO NOT COPY-CONSTRUCT!
}

ITextStorage::~ITextStorage()
{
        delete f8BitCharCache;
        delete fStyles;
}

ITextStorage&
ITextStorage::operator=(const ITextStorage&     that)
{
        return *this;
        // do not assign!
}

ITextStorage*
ITextStorage::createNew(length_type     minCapacity)
{
        // The createNew() and drop() functions implement a "poor man's free store".
        // The syntax of IText means that many operations that could be done directly
        // on a particular IText are instead done on a freshly-allocated temporary and
        // then assigned back to the original IText (consder the difference between
        // "fred.assign(barney, 3, 5)" and "fred = barney.substr(3, 5)" or
        // "fred = IText(barney, 3, 5)", for example).  This causes an extra new-delete
        // cycle.  To cut down on these, we always keep around one ITextStorage that
        // otherwise would have been deleted and reuse it the next time we need a new
        // one.

        // clients call createNew() where they otherwise would have done "new ITextStorage".
        // If gExtraTextStorage (our cache) points to an existing ITextStorage (of sufficient
        // capacity), we clear out the cache (so no one else tries to reuse this ITextStorage,
        // and so the next drop() will fill it again), zero out its length, and return it.
        // Otherwise, we go ahead and create a brand-new one.
        // [NOTE:  Because of the way reference-counting has to work, this function
        // is required to always return an ITextStorage with a reference count of 1.]

        ITextStorage*           returnVal;

        if (minCapacity < 1500)
                returnVal = new (minCapacity) ISimpleTextStorage(minCapacity);
        else
                returnVal = new (minCapacity) IGapTextStorage(minCapacity);

        IASSERTSTATE(returnVal != 0)
                // if allocation fails, no exception is thrown, the routine just returns 0.  So we check for that
                // and throw an exception ourselves if need be

        returnVal->addRef();
        return returnVal;
}

void
ITextStorage::initFrom( ITextStorage*   that,
                        offset_type     pos1,
                        length_type     length1,
                        offset_type     pos2,
                        length_type     length2,
                        length_type     gapLength,
                        bool            willDropOriginal)
{
        // this function should only be called on a freshly-allocated ITextStorage
        IASSERTSTATE(fLength == 0)

        handleInsertGap(0, length1 + length2 + gapLength);
        if (length1 != 0)
                copyIn(0, that, pos1, length1);
        if (length2 != 0)
                copyIn(0 + length1 + gapLength, that, pos2, length2);

        if (that->isStyled())
                initStylesFrom(that, willDropOriginal);
}

ITextStorage*
ITextStorage::replace(  offset_type     thisPos,
                        length_type     thisLength,
                        ITextStorage*   that,
                        offset_type     thatPos,
                        length_type     thatLength)
{
        length_type     oldLength = length();   // cache for performance
        length_type     thatOldLength = that->length();
        length_type     newLength = oldLength - thisLength + thatLength;
        ITextStorage*   returnVal;

        IASSERTPARM(thisPos <= oldLength && thatPos <= thatOldLength)
        IASSERTSTATE(thatLength <= thisLength || (newLength >= oldLength && newLength <= maxSize()))
                        // %##$%^&$@%# unsigned arithmetic...

        if (thisPos == 0 && thatPos == 0 && thisLength == oldLength && thatLength == thatOldLength) {
                returnVal = that;
                returnVal->addRef();
                drop();
        }

        // create a new ITextStorage if necessary and do all necessary insertion, deletion, or
        // copying to put it in the right state
        else {
                that->addRef(); // if that==this, we need to bump its ref count or makeCopy will
                                                // cause "that" to be gone by the time we try to call copyIn()
                if (mustCopy(newLength, that))
                        returnVal = makeCopy(thisPos, thisLength, oldLength, newLength);
                else
                        returnVal = modifyInPlace(thisPos, thisLength, thatLength);

                // if we're copying in a destination string, do that
                if (thatLength != 0)
                        returnVal->copyIn(thisPos, that, thatPos, thatLength);

                // if the source ITextStorage is styled, synchronize our style runs with the changes to
                // the characters and copy in the styles from the source ITextStorage
                if (that->isStyled())
                        returnVal->styles()->replaceStyles(thisPos, thisLength, that->styles(), thatPos, thatLength,
                                                                oldLength);

                // if we're already styled, but the source ITextStorage isn't, synchronize our style runs
                // with the changes to the characters and insert a new empty style run if necessary
                else if (returnVal->isStyled())
                        returnVal->styles()->replaceStyles(thisPos, thisLength, thatLength);

                that->drop();
        }

        return returnVal;
}

ITextStorage*
ITextStorage::replace(  offset_type     thisPos,
                        length_type     thisLength,     // must be valid (can't be npos)
                        const UniChar*  that,
                        length_type     thatLength,     // # of chars to copy from "that"
                        length_type     gapLength,      // # of extra chars to leave room for
                        bool            fill,           // if true, fill extra space with 1st char of "that"
                        bool            propagateStyles)        // if true, new characters take on styles of character preceding them
{
        length_type     oldLength = length();   // cache for performance
        length_type     newLength = oldLength - thisLength + thatLength + gapLength;
        ITextStorage*   returnVal;

        IASSERTPARM(thisPos <= oldLength)
        IASSERTSTATE(thatLength <= thisLength || (newLength >= oldLength && newLength <= maxSize()))
                        // %##$%^&$@%# unsigned arithmetic...

        // create a new ITextStorage if necessary and do all necessary insertion, deletion, or
        // copying to put it in the right state
        if (mustCopy(newLength))
                returnVal = makeCopy(thisPos, thisLength, oldLength, newLength);
        else
                returnVal = modifyInPlace(thisPos, thisLength, thatLength + gapLength);

        // if we're copying in a destination string, do that
        if (thatLength != 0)
                returnVal->handleCopyIn(thisPos, that, thatLength);

        // if we're doing a fill, do that
        if (fill)
                returnVal->handleFill(thisPos + thatLength, *that, gapLength);

        if (returnVal->isStyled()) {
                if (thisLength == 0 && propagateStyles)
                        returnVal->styles()->lengthenRunAt(thisPos, thatLength + gapLength);
                else
                        returnVal->styles()->replaceStyles(thisPos, thisLength, thatLength + gapLength);
        }

        return returnVal;
}

ITextStorage*
ITextStorage::replace(  offset_type     thisPos,
                        length_type     thisLength,
                        const char*     that,
                        length_type     thatLength)
{
        UniChar         smallBuffer[256];
        UniChar*        bigBuffer = 0;

        if (thatLength <= 256) {
                const unsigned char*    from = (const unsigned char*)that;
                const unsigned char*    fromEnd = from + thatLength;
                UniChar*        to = smallBuffer;

                while (from != fromEnd)
                        *to++ = *from++;

                return replace(thisPos, thisLength, smallBuffer, thatLength, 0, false);
        }
        else {
                bigBuffer = new UniChar[thatLength];

                ITextStorage*                   returnVal;
                const unsigned char*    from = (const unsigned char*)that;
                const unsigned char*    fromEnd = from + thatLength;
                UniChar*        to = bigBuffer;

                while (from != fromEnd)
                        *to++ = *from++;

                returnVal = replace(thisPos, thisLength, bigBuffer, thatLength, 0, false);

                delete [] bigBuffer;
                return returnVal;
        }
}

ITextStorage*
ITextStorage::atPut( offset_type        pos,
                     UniChar            newChar)
{
        // NOTE:  This function is here solely to provide one extra error check.
        // It's legal to pass the past-the-end offset (i.e., length()) to replace()
        // and have it do an append.  It is not legal to do that with at_put(), which
        // should always replace an existing character and never add a new character.
        // So we do the extra check here before calling replace().  [We added an atPut
        // to ITextStorage rather than just adding the extra error check to IText::at_put()
        // because we want to inline all of IText and didn't want that check in all the
        // inline expansions of the function.]

        IASSERTPARM(pos < length())
        return replace(pos, 1, &newChar, 1, 0, false);
}

IString
ITextStorage::asIString()
{
        // create the default transcoder to do the conversion
        if (!gInitDefaultTranscoderDone) {
           gInitDefaultTranscoder();
        }

        if (f8BitCharCache == 0) {
           const UniChar*  from = c_str();
           const UniChar*  fromEnd = from + length();
           const UniChar*  fromNext = NULL;

           length_type stringLength;
           if (gDefaultTranscoder) {
              stringLength = gDefaultTranscoder->byteBufferSize(from, fromEnd);
           }
           else {
              stringLength = length();
           }
           char*  bigBuffer = new char[stringLength + 1];
           memset(bigBuffer, 0x00, stringLength + 1);

           char*  to = bigBuffer;
           char*  toNext = NULL;

           ITranscoder::result result;

           if (gDefaultTranscoder) {
              result = gDefaultTranscoder->fromUnicode(from, fromEnd, fromNext,
                                                       to, to + stringLength, toNext);
           }

           // if the transcoder wasn't created, or if it failed, use the old way
           if (!gDefaultTranscoder || result != codecvt_base::ok) {
              while (from != fromEnd) {
                 if (*from <= 0xff)
                    *to++ = (char)(*from++);
                 else {
                    *to++ = 0x1a;   // ASCII substitute character
                    ++from;
                 }
              }
           }

           f8BitCharCache = new IString(bigBuffer); // NOT a memory leak because it is deleted in ~ITextStorage()
           delete [] bigBuffer;
        }
        return *f8BitCharCache;
}

ITextStorage*
ITextStorage::cloneIfNecessary()
{
        if (count() == 1)
                return this;
        else
                return makeCopy();
}

long
ITextStorage::hash()
{
        if (fHash == kInvalidHash) {
                const UniChar*  curChar;
                const UniChar*  chunk;
                offset_type             chunkPos = 0;
                length_type             chunkLength = 0;
                long                    value = 0;
                long                    c;
                length_type             count = length();
                length_type             chunkCount;

                if (count > kMaxHashCount)
                        count = kMaxHashCount;  // consider only kMaxHashCount chars

                while (count != 0) {
                        chunk = storageChunk(chunkPos + chunkLength, chunkPos, chunkLength);
                        curChar = chunk;
                        chunkCount = chunkLength;

                        while (count != 0 && chunkCount != 0) {
                                c = *curChar++;
                                value = ((value << (c & 0x0f)) ^ (c << 8)) + (c ^ value);
                                count--;
                                chunkCount--;
                        }
                }
                if (value == kInvalidHash)
                        fHash = 1;
                else
                        fHash = value;
        }
        return fHash;
}

bool
ITextStorage::isStyled() const
{
        return fStyles != 0;
}

IStyleStorage*
ITextStorage::styles()
{
        if (fStyles == 0)
                fStyles = new IStyleStorage(this);
        return fStyles;
}

const IStyleStorage*
ITextStorage::styles() const
{
        return fStyles;
}

void
ITextStorage::initStylesFrom( ITextStorage*   that,
                              bool            canAdoptStyles)
{
        if (isStyled())
                removeAllStyles();
        if (canAdoptStyles && that->count() == 1) {
                fStyles = that->fStyles;
                fStyles->setTextStorage(this);
                that->fStyles = 0;
        }
        else
                fStyles = new IStyleStorage(*(that->fStyles), this);
}

void
ITextStorage::removeAllStyles()
{
        delete fStyles;
        fStyles = 0;
}

ITextStorage *
ITextStorage::emptyTextStorage()
{
    ITextStorage* theEmptyStorage = ITextStorage::fgEmptyTextStorage;

    // check to see if fgEmptyTextStorage has been initialized yet. If it hasn't,
    // create a new ITextStorage and point fgEmptyTextStorage to it.
    if (theEmptyStorage == 0) {
        theEmptyStorage = createNew(0);
        theEmptyStorage = synchronizedInit(ITextStorage::fgEmptyTextStorage,
                                           theEmptyStorage);
        if (theEmptyStorage == ITextStorage::fgEmptyTextStorage) {
            theEmptyStorage->asIString();   // this call prevents some lazy-evaluate problems by ensuring that our
            // default transcoder is set up and that gEmptyTextStorage's f8BitCharCache
            // field is filled in
            IStaticObjectManager::manageRefCountOf( theEmptyStorage );
        } else {
            delete theEmptyStorage;
        }
    }
    return theEmptyStorage;
}

void ITextStorage::printDebugInfo() const
{
    ITRACE_RUNTIME(IString("\nLength = ") + IString(length()));
    ITRACE_RUNTIME(IString("Capacity = ") + IString(capacity()));
}

void
ITextStorage::copyIn( offset_type     thisPos,
                      ITextStorage*   that,
                      offset_type     thatPos,
                      length_type     numChars)
{
        if (numChars == 0)
                return;

        const UniChar*  chunk;
        offset_type     chunkPos;
        length_type     chunkLength;
        length_type     charsToCopy;

        chunk = that->storageChunk(thatPos, chunkPos, chunkLength);
        chunk += thatPos - chunkPos;
        chunkLength -= thatPos - chunkPos;
        chunkPos = thatPos;
        while (numChars != 0) {
                charsToCopy = chunkLength;
                if (charsToCopy > numChars)
                        charsToCopy = numChars;
                handleCopyIn(thisPos, chunk, charsToCopy);
                numChars -= charsToCopy;
                thisPos += charsToCopy;
                if (numChars != 0)
                        chunk = that->storageChunk(chunkPos + chunkLength, chunkPos, chunkLength);
        }
}

bool
ITextStorage::mustCopy(length_type      newLength) const
{
        return newLength >= capacity() || count() > 1;
}

bool
ITextStorage::mustCopy( length_type                     newLength,
                                                const ITextStorage*     that) const
{
        return newLength >= capacity() || count() > 1 || this == that;
}

ITextStorage*
ITextStorage::makeCopy(length_type      newCapacity)
{
        length_type     newLength = length();
        if (newCapacity == 0)
                newCapacity = newLength;

        ITextStorage*   returnVal = createNew(newCapacity);

        returnVal->initFrom(this, 0, newLength, 0, 0, 0, true);

        drop();

        return returnVal;
}

ITextStorage*
ITextStorage::makeCopy( offset_type     gapPos,
                        length_type     oldGapLength,
                        length_type     oldTotalLength,
                        length_type     newTotalLength)
{
        ITextStorage*   returnVal = createNew(newTotalLength);

        length_type     charsToCopy = oldTotalLength - (gapPos + oldGapLength);

        returnVal->initFrom(this, 0, gapPos, oldTotalLength - charsToCopy, charsToCopy,
                                                oldGapLength + (newTotalLength - oldTotalLength), true);

        drop();

        return returnVal;
}

ITextStorage*
ITextStorage::modifyInPlace( offset_type     gapPos,
                             length_type     oldGapLength,
                             length_type     newGapLength)
{
        // dump the has value and the IString/char* cache
        delete f8BitCharCache;
        f8BitCharCache = 0;
        fHash = kInvalidHash;

        if (newGapLength < oldGapLength)
                handleDeleteChars(gapPos + newGapLength, oldGapLength - newGapLength);
        else if (newGapLength > oldGapLength)
                handleInsertGap(gapPos + oldGapLength, newGapLength - oldGapLength);

        return this;
}

void
ITextStorage::writeToStream(IDataStream&        toWhere) const
{
  typedef unsigned char Byte;

  IStreamOutFrame       frame(toWhere);

  offset_type           nextOffset              = 0;
  offset_type           currentOffset           = 0;
  length_type           currentLength           = 0;
  length_type           totalLength             = length();

  size_t                byteBufferLength        = 0;
  size_t                unicharsCompressed      = 0;
  size_t                bytesInCompressedBuffer = 0;
  size_t                foo                     = 0;

  const UniChar         *currentChunk           = 0;
  Byte                  *byteBuffer             = 0;

  const short           maxBufferSize           = 4096;

  IUniCharCompressor    theCompressor;


  try
        {
          // get a pointer to the first block of raw UniChars
          currentChunk = storageChunk( nextOffset, currentOffset, currentLength );

          // calculate maximum buffer size - worst case compression is 3/2 - and allocate
          byteBufferLength =  (size_t) ( (currentLength * 1.5) + 1 );

          byteBuffer = new Byte [ byteBufferLength ];
          if( byteBuffer == 0 )
                throw IException( "Memory allocation failed", 0, IException::recoverable );

          // compress the first chunk of UniChar data
          theCompressor.compress( currentChunk, currentLength, unicharsCompressed,
                                                          byteBuffer, byteBufferLength, bytesInCompressedBuffer );

          // fewer than 64 bytes in compressed stream (and this is the only block in
          // the IText), use type I header
          if( bytesInCompressedBuffer < 0x40 && currentLength == length())
                {
                  Byte header = (Byte) ( (bytesInCompressedBuffer << 1) | 0x01);

                  // write special header byte and then compressed data
                  header >>= toWhere;
                  toWhere.writeBytes( ( const char*) byteBuffer, bytesInCompressedBuffer );
                }

          // more than 64 bytes in compressed stream, use type II header
          else
                {
                  // write number of UniChars in uncompressed IText, length of current compressed
                  // chunk, and then compressed data
                  writeCompressedLong( (unsigned long) (totalLength << 1), toWhere, foo );
                  writeCompressedLong( (unsigned long) bytesInCompressedBuffer, toWhere, foo );
                  toWhere.writeBytes( (const char*) byteBuffer, bytesInCompressedBuffer );

                  // since this isn't the only block, reallocate the buffer to a large value
                  // use the new, larger buffer unless we need to allocate even more space
                  delete [] byteBuffer;
                  byteBuffer = 0;

                  byteBuffer = new Byte [ maxBufferSize ];
                  if( byteBuffer == 0 )
                        throw IException( "Memory allocation failed", 0, IException::recoverable );
                  byteBufferLength = maxBufferSize;

                  nextOffset += currentLength;

                  while( nextOffset < totalLength )
                        {
                          // get a pointer to the next block of raw UniChars
                          currentChunk = storageChunk( nextOffset, currentOffset, currentLength );

                          // calculate the maximum size of the compressed buffer, and if it's larger than
                          // the currently allocated buffer, allocate a new one
                          size_t        maxSize = (size_t) ( (currentLength * 1.5) + 1 );

                          if( maxSize > byteBufferLength )
                                {
                                  delete [] byteBuffer;
                                  byteBuffer = 0;

                                  byteBuffer = new Byte [ maxSize ];
                                  if( byteBuffer == 0 )
                                        throw IException( "Memory allocation failed", 0, IException::recoverable );
                                  byteBufferLength = maxSize;
                                }

                          // compress the UniChar data
                          theCompressor.compress( currentChunk, currentLength, unicharsCompressed,
                                                                          byteBuffer, byteBufferLength, bytesInCompressedBuffer );

                          // write length of current compressed chunk, and then write the compressed data
                          writeCompressedLong( (unsigned long) bytesInCompressedBuffer, toWhere, foo );
                          toWhere.writeBytes( (const char*) byteBuffer, bytesInCompressedBuffer );

                          nextOffset += currentLength;
                        }
                }

          delete [] byteBuffer;

          // now write out the styling information
          if( isStyled() )
                fStyles->writeToStream(toWhere);
        }       // try

  catch(...)
        {
          delete [] byteBuffer;
          throw;
        }
}



void ITextStorage::readFromStream(IDataStream&  fromWhere)
{
  typedef unsigned char Byte;

  IStreamInFrame        frame(fromWhere);

  UniChar               *unicharBuffer       = 0;
  Byte                  *byteBuffer          = 0;

  size_t                byteBufferLength     = 0;
  size_t                numBytesDecompressed = 0;
  size_t                unicharBufferLength  = 0;
  size_t                numUnicharsInBuffer  = 0;
  size_t                foo;

  unsigned long         header               = 0;

  IUniCharCompressor    theCompressor;


  try
        {
          // read the header from the stream
          header = readCompressedLong( fromWhere, foo );

          // this is a type I header
          if( header & 0x01 )
                {
                  Byte dataSize = (Byte) (header >> 1);

                  byteBufferLength              = dataSize;
                  // worst case (best case) is two times the length of compressed data
                  unicharBufferLength   = byteBufferLength * 2;

                  byteBuffer                    = new Byte              [ byteBufferLength ];
                  unicharBuffer                 = new UniChar   [ unicharBufferLength ];

                  if( byteBuffer == 0 || unicharBuffer == 0 )   // allocation failed
                        throw IException( "Memory allocation failed", 0, IException::recoverable );

                  // read the compressed bytes from the stream, and decompress
                  fromWhere.readBytes( (char*) byteBuffer, dataSize );
                  theCompressor.decompress( byteBuffer, dataSize, numBytesDecompressed,
                                                                        unicharBuffer, unicharBufferLength, numUnicharsInBuffer );

                  // fill the IText with the decompressed UniChars
                  // (use resize() and handleCopyIn() to get around the copy-on-write logic
                  // in replace().)
                  IASSERTSTATE(numUnicharsInBuffer <= capacity())
                  handleInsertGap(0, numUnicharsInBuffer);
                  handleCopyIn( 0, unicharBuffer, numUnicharsInBuffer );
                }

          // this is a type II header
          else
                {
                  unsigned long         unicharsInDecompressed          = (header >> 1);
                  unsigned long         unicharsToGo                            = unicharsInDecompressed;
                  unsigned long         bytesInNextBlock                        = 0;
                  size_t                copyPos                                         = 0;

                  IASSERTSTATE(unicharsInDecompressed <= capacity())
                  while( unicharsToGo != 0 )
                        {
                          // read in the number of bytes in the next compressed block
                          bytesInNextBlock = readCompressedLong( fromWhere, foo );

                          // if the current byte buffer isn't large enough to hold compressed data,
                          // allocate a bigger one
                          if( bytesInNextBlock > byteBufferLength )
                                {
                                  byteBufferLength              = (size_t) bytesInNextBlock;
                                  // worst case (best) is two times the length of compressed data
                                  unicharBufferLength   = byteBufferLength * 2;

                                  // the first time through we are calling vector delete without
                                  // a previous call to new, but this is OK because the pointers were
                                  // zeroed intially
                                  delete [] byteBuffer;
                                  delete [] unicharBuffer;
                                  byteBuffer    = 0;
                                  unicharBuffer = 0;

                                  byteBuffer                    = new Byte              [ byteBufferLength ];
                                  unicharBuffer                 = new UniChar   [ unicharBufferLength ];

                                  if( byteBuffer == 0 || unicharBuffer == 0 )   // allocation failed
                                        throw IException( "Memory allocation failed", 0, IException::recoverable );
                                }

                          // read the compressed bytes from the stream, and then decompress
                          fromWhere.readBytes( (char*) byteBuffer, (size_t) bytesInNextBlock );
                          theCompressor.decompress( byteBuffer, (size_t) bytesInNextBlock, numBytesDecompressed,
                                                                                unicharBuffer, unicharBufferLength, numUnicharsInBuffer );

                          // if this is the first time through, replace old text and reserve space for the rest
                          // otherwise just copy in the UniChars
                          if( copyPos == 0 )
                                {
                                  // use resize() and handleCopyIn() to get around the copy-on-write logic
                                  // in replace().
                                  handleInsertGap(0, (length_type)unicharsToGo);
                                  handleCopyIn( 0, unicharBuffer, numUnicharsInBuffer );
                                }
                          else
                                handleCopyIn( copyPos, unicharBuffer, numUnicharsInBuffer );

                          copyPos               += numUnicharsInBuffer;
                          unicharsToGo  -= numUnicharsInBuffer;
                        }       // while
                }

          delete [] byteBuffer;
          delete [] unicharBuffer;

          // if there were styles, then create an IStyleStorage and read them in off the stream
          if( ! frame.atEnd() )
                styles()->readFromStream(fromWhere);
        }

  catch(...)
        {
          delete [] byteBuffer;
          delete [] unicharBuffer;
          throw;
        }
}

void
ITextStorage::fillInIteratorFields( IFastTextIterator& iterator,
                                    UniChar*           theChar,
                                    UniChar*           gapStart,
                                    length_type        gapLength)
{
        iterator = IFastTextIterator(theChar, gapStart, gapLength);
}

