// Revision: 64 1.11.2.1 source/core/text/intl/transcod/posix/psxtrans.cpp, text, ioc.v400, 001006 
/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
//
//  File: psxtrans.cpp
//  Function: POSIX implementation of ITranscoder::CreateTranscoder()
//  Author: Chen-Lieh Huang
//  Date created: 05/08/1996
//
#ifndef __errno_h
#include <errno.h>
#endif
#ifndef __string_h
#include <string.h>
#endif
#ifndef __langinfo_h
#include <langinfo.h>
#endif
#ifndef __locale_h
#include <locale.h>
#endif
#ifndef _ICONV_H
#include <iconv.h>
#endif

#ifndef IC_IGENEXC
#include <igenexc.hpp>
#endif
#ifndef IC_ITRANCOD
#include <itrancod.hpp>
#endif
#ifndef IC_ISO88591
#include "iso88591.hpp"
#endif
#ifndef IC_UTF8TRAN
#include "utf8tran.hpp"
#endif

#ifndef IC_TRANSCOD
#include "transcod.h"
#endif

#ifdef IC_PAGETUNE
        #define _PSXTRANS_CPP
        #include <ipagetun.h>
#endif


class IPosixTranscoder : public ITranscoder {
public:
        //      Standard member functions
        IPosixTranscoder();
        IPosixTranscoder(const IText& charSet, EMappingProximity proximity);
        IPosixTranscoder(const IPosixTranscoder& source);
        IPosixTranscoder& operator=(const IPosixTranscoder& right);
        virtual                         ~IPosixTranscoder();

        // Overrides
        length_type             maximumBytesPerUniChar() const;
        length_type             maximumUniCharsPerByte() const;

        length_type             byteBufferSize(const IText& uniText) const;
        length_type             byteBufferSize(const UniChar* from, const UniChar* from_end) const;
        length_type             uniCharBufferSize(const IString& text) const;
        length_type             uniCharBufferSize(const char* from, const char* from_end) const;

        static const UniChar*   portableCSet(const char* posixCSet);
                // used by ITranscoder::characterSet

protected:
        //      called by toUnicode
        result          doToUnicode(const char* from, const char* from_end,
                                                        const char*& from_next,
                                                        UniChar* to, UniChar* to_limit, UniChar*& to_next);

        //      called by fromUnicode
        result          doFromUnicode(const UniChar* from, const UniChar* from_end,
                                                        const UniChar*& from_next,
                                                        char* to, char* to_limit, char*& to_next);

        // Sets the maximum number of bytes generated by converting a UniChar.
        void                    setMaximumBytesPerUniChar(length_type bytes);

        // Sets the maximum number of UniChars generated by converting
        // a multibyte char.
        void                    setMaximumUniCharsPerByte(length_type bytes);

private:
        const char*     posixCSet(const IText& portableCSet, EMappingProximity proximity) const;

        void                            OpenToConvertor();
        void                            OpenFromConvertor();

        length_type fMaximumBytesPerUniChar;
        length_type fMaximumUniCharsPerByte;

        char                            fCodeSet[256];
        iconv_t                 fToConvertor;
        iconv_t                 fFromConvertor;
        bool                            fToIsOpen;
        bool                            fFromIsOpen;
};


typedef struct PortableToPosixCodeSetMapping {
        char* portableCSet;     // portable name of code set
        char* posixCSet;                // Posix codeset name to create transcoder
        int             proximity;
} PortableToPosixCodeSetMappingType;

// The pre-sorted mapping table that maps the portable name of
// a character set to a corresponding Posix codeset name.
static const PortableToPosixCodeSetMappingType portableCSetMap[] =
{       {"CNS-11643.1986",      "IBM-eucTW",    ITranscoder::kSupersetMapping},
        {"EUC",                                                 "IBM-eucJP",    ITranscoder::kSupersetMapping},
        {"GB-2312",                                     "IBM-eucCN",    ITranscoder::kSupersetMapping},
        {"IBM-437",                                     "IBM-437",              ITranscoder::kExactMapping},
        {"IBM-850",                                     "IBM-850",              ITranscoder::kExactMapping},
        {"IBM-950",                                     "IBM-950",              ITranscoder::kExactMapping},
        {"ISO-8859-1",                  "ISO88591",             ITranscoder::kExactMapping},
        {"ISO-8859-2",                  "ISO88592",             ITranscoder::kExactMapping},
        {"ISO-8859-3",                  "ISO88593",             ITranscoder::kExactMapping},
        {"ISO-8859-4",                  "ISO88594",             ITranscoder::kExactMapping},
        {"ISO-8859-5",                  "ISO88595",             ITranscoder::kExactMapping},
        {"ISO-8859-6",                  "ISO88596",             ITranscoder::kExactMapping},
        {"ISO-8859-7",                  "ISO88597",             ITranscoder::kExactMapping},
        {"ISO-8859-8",                  "ISO88598",             ITranscoder::kExactMapping},
        {"ISO-8859-9",                  "ISO88599",             ITranscoder::kExactMapping},
        {"KSC-5601",                            "IBM-eucKR",    ITranscoder::kSupersetMapping},
        {"Shift-JIS",                           "IBM-932",              ITranscoder::kSupersetMapping},
        {"US-ASCII",                            "ISO88591",             ITranscoder::kExactMapping},
        {"UTF-8",                                               "UTF-8",                        ITranscoder::kExactMapping}
};

static const int numPortableCSets = sizeof(portableCSetMap)/sizeof(PortableToPosixCodeSetMappingType);


typedef struct PosixToPortableCodeSetMapping {
        char*           posixCSet;
        UniChar portableCSet[20];
} PosixToPortableCodeSetMappingType;

// The pre-sorted mapping table that maps a Posix codeset name
// to the corresponding portable name of a character set.
static const PosixToPortableCodeSetMappingType posixCSetMap[] =
{       {"IBM-437",             {'I','B','M','-','4','3','7',0x0000}},
        {"IBM-850",             {'I','B','M','-','8','5','0',0x0000}},
        {"IBM-932",             {'S','h','i','f','t','-','J','I','S',0x0000}},
        {"IBM-950",             {'I','B','M','-','9','5','0',0x0000}},
        {"IBM-eucCN",   {'G','B','-','2','3','1','2',0x0000}},
        {"IBM-eucJP",   {'E','U','C',0x0000}},
        {"IBM-eucKR",   {'K','S','C','-','5','6','0','1',0x0000}},
        {"IBM-eucTW",   {'C','N','S','-','1','1','6','4','3','.','1','9','8','6',0x0000}},
        {"ISO88591",    {'I','S','O','-','8','8','5','9','-','1',0x0000}},
        {"ISO88592",    {'I','S','O','-','8','8','5','9','-','2',0x0000}},
        {"ISO88593",    {'I','S','O','-','8','8','5','9','-','3',0x0000}},
        {"ISO88594",    {'I','S','O','-','8','8','5','9','-','4',0x0000}},
        {"ISO88595",    {'I','S','O','-','8','8','5','9','-','5',0x0000}},
        {"ISO88596",    {'I','S','O','-','8','8','5','9','-','6',0x0000}},
        {"ISO88597",    {'I','S','O','-','8','8','5','9','-','7',0x0000}},
        {"ISO88598",    {'I','S','O','-','8','8','5','9','-','8',0x0000}},
        {"ISO88599",    {'I','S','O','-','8','8','5','9','-','9',0x0000}}
};

static const int numPosixCSets = sizeof(posixCSetMap)/sizeof(PosixToPortableCodeSetMappingType);

static const UniChar iso88591[]={'I','S','O','-','8','8','5','9','-','1',0x0000};
static const UniChar usAscii[]={'U','S','-','A','S','C','I','I',0x0000};
static const UniChar utf8[]={'U','T','F','-','8',0x0000};


//
//      Platform-specific APIs implementation.
//
ILineBreakConverter::ELineBreakConvention
ILineBreakConverter::hostConvention()
{
#ifdef __OS2__
    return ILineBreakConverter::kCRLF;
#else
        return ILineBreakConverter::kLF;
#endif
}


const IText
ICharacterSetIterator::operator*()
{
        if (fCurCharSetIdx >= numPortableCSets)
                return IText(); // pass-the-end
        else return (IText(portableCSetMap[fCurCharSetIdx].portableCSet));
}


ICharacterSetIterator::operator bool() const
{
        if (fCurCharSetIdx >= numPortableCSets)
                return false;   // pass-the-end
        else return true;
}


const IText
ITranscoder::characterSet(const ILocaleKey& key)
{
        IText   posixID = key.posixID();
        char* tmpStr;
                char oldLocale[20];
                tmpStr = setlocale(LC_ALL, 0);
                if (tmpStr != 0)
                        strcpy(oldLocale, tmpStr);

        if (NULL == (tmpStr=setlocale(LC_ALL, posixID)))
        {
                IMessageText text(kTranscoderNotInstalled, kMessageFile);
                IObjectNotFound exc(text);
                ITHROW(exc);
        }

        char* posixCSet = nl_langinfo(CODESET);

                setlocale(LC_ALL, tmpStr);

        return IText(IPosixTranscoder::portableCSet(posixCSet));
}


//  Posix implementation of ITranscoder::createTranscoder().
//
ITranscoder *
ITranscoder::createTranscoder(const IText& charSet, EMappingProximity proximity)
{
        // If charSet is ISO-8859-1, handle the conversion with
        // fast IISO8859_1Transcoder.
        if (charSet == IText(iso88591) || charSet == IText(usAscii))
                return new IISO8859_1Transcoder();
        else if (charSet == IText(utf8))
                return new UTF8Transcoder();
        else if (charSet.empty())
                return new IPosixTranscoder();
        else
                return new IPosixTranscoder(charSet, proximity);
}


ITranscoder *
ITranscoder::createTranscoder()
{
        char* posixCSet = nl_langinfo(CODESET);

        if (strcmp(posixCSet, "ISO8859-1") == 0)
                return new IISO8859_1Transcoder();
        else
                return new IPosixTranscoder();    // use host conversion API
}


//
//      Posix Transcoder implementation
//
const char*
IPosixTranscoder::posixCSet(const IText& portableCSet, EMappingProximity proximity) const
{
        // Convert portableCSet from IText to char* algoritmically, which
        // turns IText into UniChar, and then converts from UniChar
        // to char* using the ISO-8859-1 transcoding algorithm.
        length_type     length = portableCSet.length();
        size_t  dummy1, dummy2;

        const UniChar* uniCharSet = portableCSet.storage_chunk(0, dummy1, dummy2);

        char csName[256];       // portableCSet name size never greater than 256.
        for (int i=0; i<length; i++)
                csName[i] = (unsigned char)uniCharSet[i];
        csName[length] = '\0';

        // Binary search for the posixCSet using csName.
        int     low = 0, mid = 0;
        int     high = numPortableCSets - 1;

        while (low <= high) {
                mid = (low + high) / 2;
                if (strcmp(csName, portableCSetMap[mid].portableCSet) < 0)
                        high = mid - 1;
                else if (strcmp(csName, portableCSetMap[mid].portableCSet) > 0)
                        low = mid + 1;
                else if (proximity >= portableCSetMap[mid].proximity) // match found!
                        return portableCSetMap[mid].posixCSet;
                else return NULL;       // charset matched, but proximity was not met!
                }
        return NULL;
}


const UniChar*
IPosixTranscoder::portableCSet(const char* posixCSet)
{
        iconv_t convertor = iconv_open("UCS-2", posixCSet);
        if (convertor == (iconv_t)-1)
        {
                IMessageText text(kTranscoderNotInstalled, kMessageFile);
                IObjectNotFound exc(text);
                ITHROW(exc);
        }
        else iconv_close(convertor);

        // Binary search for the mapping entry for codePage specified.
        int     low = 0, mid = 0;
        int     high = numPosixCSets - 1;

        while (low <= high) {
                mid = (low + high) / 2;
                if (strcmp(posixCSet, posixCSetMap[mid].posixCSet) < 0)
                        high = mid - 1;
                else if (strcmp(posixCSet, posixCSetMap[mid].posixCSet) > 0)
                        low = mid + 1;
                else return posixCSetMap[mid].portableCSet;     // found match!
                }
        return NULL;
}


IPosixTranscoder::IPosixTranscoder()
        : fMaximumBytesPerUniChar(5),
                fMaximumUniCharsPerByte(1),
                fToIsOpen(false),
                fFromIsOpen(false)
{
        strcpy(fCodeSet, nl_langinfo(CODESET));

        // Search the name of the character set based on
        // the default code page fCurrentCP, and set the
        // Character Encoding using this name.
        const UniChar *cSet;
        if ((cSet=portableCSet(fCodeSet)) != NULL)
                setCharacterEncoding(IText(cSet));
        else setCharacterEncoding(IText());
}


IPosixTranscoder::IPosixTranscoder(const IText& charSet, EMappingProximity proximity)
        : fMaximumBytesPerUniChar(5),
                fMaximumUniCharsPerByte(1),
                fToIsOpen(false),
                fFromIsOpen(false)
{
        // Map charSet to Posix codeset name. If the specified charSet
        // can't be mapped, that means the charSet specified is not
        // supported, and kNoAdequateTranscoder is thrown.
        const char* posixName;
        if ((posixName=posixCSet(charSet, proximity)) == NULL)
        {
                IMessageText text(kNoAdequateTranscoder, kMessageFile);
                IObjectNotFound exc(text);
                ITHROW(exc);
        }

        // The Posix codeset is supported. Check if it is installed.
        // If it isn't, kTranscoderNotInstalled exception is thrown.
        iconv_t convertor = iconv_open("UCS-2", posixName);
        if (convertor == (iconv_t)-1)
        {
                IMessageText text(kTranscoderNotInstalled, kMessageFile);
                IObjectNotFound exc(text);
                ITHROW(exc);
        }
        else iconv_close(convertor);

        strcpy (fCodeSet, posixName);
        setCharacterEncoding(charSet);
}


IPosixTranscoder::~IPosixTranscoder()
{
        if (fToIsOpen)
                iconv_close(fToConvertor);
        if (fFromIsOpen)
                iconv_close(fFromConvertor);
}


IPosixTranscoder::IPosixTranscoder(const IPosixTranscoder& source)
        : ITranscoder(source),
                fMaximumBytesPerUniChar(source.fMaximumBytesPerUniChar),
                fMaximumUniCharsPerByte(source.fMaximumUniCharsPerByte),
                fToIsOpen(false),
                fFromIsOpen(false)
{
        strcpy (fCodeSet, source.fCodeSet);
}


IPosixTranscoder &
IPosixTranscoder::operator=(const IPosixTranscoder& right)
{
        if (this != &right)
        {
                if (fToIsOpen) {
                        iconv_close(fToConvertor);
                        fToIsOpen = false;
                        }
                if (fFromIsOpen) {
                        iconv_close(fFromConvertor);
                        fFromIsOpen = false;
                        }
                ITranscoder::operator=(right);
                fMaximumBytesPerUniChar = right.fMaximumBytesPerUniChar;
                fMaximumUniCharsPerByte = right.fMaximumUniCharsPerByte;
                strcpy (fCodeSet, right.fCodeSet);
        }
        return *this;
}


ITranscoder::length_type
IPosixTranscoder::maximumBytesPerUniChar() const
{
        return fMaximumBytesPerUniChar;
}


ITranscoder::length_type
IPosixTranscoder::maximumUniCharsPerByte() const
{
        return fMaximumUniCharsPerByte;
}


void
IPosixTranscoder::setMaximumBytesPerUniChar(length_type bytes)
{
        fMaximumBytesPerUniChar = bytes;
}


void
IPosixTranscoder::setMaximumUniCharsPerByte(length_type bytes)
{
        fMaximumUniCharsPerByte = bytes;
}


ITranscoder::length_type
IPosixTranscoder::byteBufferSize(const IText& uniText) const
{
        return uniText.length() * fMaximumBytesPerUniChar;
}


ITranscoder::length_type
IPosixTranscoder::byteBufferSize(
        const UniChar* from, const UniChar* from_end) const
{
        if (from < from_end)
                return (from_end - from) * fMaximumBytesPerUniChar;
        else
                return (from - from_end) * fMaximumBytesPerUniChar;
}


ITranscoder::length_type
IPosixTranscoder::uniCharBufferSize(const IString& text) const
{
        return text.length() * fMaximumUniCharsPerByte;
}


ITranscoder::length_type
IPosixTranscoder::uniCharBufferSize(
        const char* from, const char* from_end) const
{
        if (from < from_end)
                return (from_end - from) * fMaximumUniCharsPerByte;
        else
                return (from - from_end) * fMaximumUniCharsPerByte;
}


// to Unicode (UCS-2) from char
void
IPosixTranscoder::OpenToConvertor()
{
        if (!fToIsOpen)
        {
                const char* fromCode = fCodeSet;
                fToConvertor = iconv_open("UCS-2", fromCode);
                if (fToConvertor == (iconv_t)-1)
                {
                        IMessageText text(kTranscoderNotInstalled, kMessageFile);
                        IObjectNotFound exc(text);
                        ITHROW(exc);
                }
                fToIsOpen = true;
        }
}


// from Unicode (UCS-2) to char
void
IPosixTranscoder::OpenFromConvertor()
{
        if (!fFromIsOpen)
        {
                const char* toCode = fCodeSet;
                fFromConvertor = iconv_open(toCode, "UCS-2");
                if (fFromConvertor == (iconv_t)-1)
                {
                        IMessageText text(kTranscoderNotInstalled, kMessageFile);
                        IObjectNotFound exc(text);
                        ITHROW(exc);
                }
                fFromIsOpen = true;
        }
}


ITranscoder::result
IPosixTranscoder::doToUnicode(
        const char* from, const char* from_end, const char*& from_next,
        UniChar* to, UniChar* to_limit, UniChar*& to_next)
{
        // Initial conditions checking
    if (from == NULL || to == NULL ||
        from > from_end || to > to_limit ||
        from == (char*)to)
    {
        IInvalidParameter       exc(
            IMessageText(kConvBufRangeError, kMessageFile),
            kConvBufRangeError,
            IException::recoverable);
        ITHROW(exc);
    }

    OpenToConvertor();

    UniChar conversionBuffer[1024] = {0}; // fix-size conversion buffer
    size_t inBytesLeft = from_end - from;
    size_t outBytesLeft = sizeof(conversionBuffer);
    size_t missing = 0;
    const char *inBuf = from;
    char *outBuf = (char *)conversionBuffer;
    long lengthConverted, lengthToCopy;

    from_next = from;
    to_next = to;

    // iconv() doesn't stops conversion when it sees an unmapped char.
    while (inBytesLeft > 0 && missing == 0) {
        const char *preInBuf = inBuf;
        size_t preInBytesLeft = inBytesLeft;

        missing = iconv(fToConvertor, &inBuf, &inBytesLeft,
                        &outBuf, &outBytesLeft);
        if (missing != 0)
        {
            // Output buffer overflow, which is fine. Continue conversion.
            switch (errno)
            {
                case E2BIG:
                    missing = 0;
                    break;
                case EILSEQ: {
                    IInvalidRequest exc(
                        //providing a message file name defined by user
                        IMessageText(kInvalidInputCharacter, IMessageText::messageFile()),
                        kInvalidInputCharacter,
                        IException::recoverable);
                    ITHROW(exc);
                }
                case EINVAL: {
                    IInvalidRequest exc(
                        //providing a message file name defined by user
                        IMessageText(kIncompleteCharOrSequence, IMessageText::messageFile()),
                        kIncompleteCharOrSequence,
                        IException::recoverable);
                    ITHROW(exc);
                }
            }
        }

        from_next = inBuf;

        lengthConverted = ((sizeof(conversionBuffer)-outBytesLeft)/
                           sizeof(UniChar));

        if ((signed long)((to_next + lengthConverted) - to_limit) > 0)
            lengthToCopy = to_limit - to_next;
        else
            lengthToCopy = lengthConverted;

        // Copy out the fix buffer conversionBuffer.
        UniChar* up = 0;

        if (missing > 0) // There were missing characters it can't convert
        {
            // There are unmapped chars, handle based on unmappedBehavior.
            ITranscoder::EUnmappedBehavior ub = unmappedBehavior();
            switch (ub)
            {
                case ITranscoder::kOmit:
                    for (up = conversionBuffer; lengthToCopy > 0;
                         lengthToCopy--)
                        // omit the FFFD which is converted from
                        // an unmapped char.
                        if (*up == 0xFFFD)
                            up++;
                        else
                            *to_next++ = *up++;
                    break;
                case ITranscoder::kUseSub:
                    // iconv unisub ==> uniCharSubstitute()
                    for (up = conversionBuffer; lengthToCopy > 0;
                         lengthToCopy--)
                        // Needed if iconv unisub is not 0xFFFD,
                        // Otherwise, remove this code.
                        if (*up == 0xFFFD) {
                            *to_next++ = uniCharSubstitute();
                            up++;
                        } else
                            *to_next++ = *up++;
                    break;
                case ITranscoder::kStop:
                    for (up = conversionBuffer; lengthToCopy > 0;
                         lengthToCopy--)
                        if (*up == 0xFFFD) {
                            // Re-calculate where to stop the input buffer
                            char *newOutBuf = (char *)conversionBuffer;
                            size_t newOutBytesLeft = (up - conversionBuffer) *
                                sizeof(UniChar);
                            iconv(fToConvertor, &preInBuf, &preInBytesLeft,
                                  &newOutBuf, &newOutBytesLeft);
                            from_next = preInBuf;
                            return codecvt_base::partial;
                        }
                        else
                            *to_next++ = *up++;
                    break;
            }
        } else // There is no unmapped char, simply copy out.
            for (up = conversionBuffer; lengthToCopy > 0; lengthToCopy--)
                *to_next++ = *up++;

        if (to_next == to_limit) // we are done with required conversion
            break;

        // Resets outBuf to beginning of conversionBuffer,
        // and outBytesLeft to full size of conversionBuffer.
        outBuf = (char *)conversionBuffer;
        outBytesLeft = sizeof(conversionBuffer);
    }
    
    if (missing)
        return codecvt_base::error;
    else if (from_next < from_end)
        return codecvt_base::partial;
    else if (from_next == from)
        return codecvt_base::noconv;
    else return codecvt_base::ok;
}


ITranscoder::result
IPosixTranscoder::doFromUnicode(
                const UniChar* from, const UniChar* from_end, const UniChar*& from_next,
                char* to, char* to_limit, char*& to_next)
{
    // Initial conditions checking
    if (from == NULL || to == NULL ||
        from > from_end || to > to_limit ||
        (const char*)from == to)
    {
        IInvalidParameter       exc(
            IMessageText(kConvBufRangeError, kMessageFile),
            kConvBufRangeError,
            IException::recoverable);
        ITHROW(exc);
    }

    char conversionBuffer[1024];    // fix-size conversion buffer
    
    OpenFromConvertor();
    
    size_t inBytesLeft = (from_end - from) * sizeof(UniChar);
    size_t outBytesLeft = sizeof(conversionBuffer);
    long missing = 0;
    const char *inBuf = (const char*)from;
    char *outBuf = (char *) conversionBuffer;
    long lengthConverted, lengthToCopy;

    from_next = from;
    to_next = to;

    while (inBytesLeft != 0 && missing == 0)
    {
        const char *preInBuf = inBuf;
        size_t preInBytesLeft = inBytesLeft;

        missing = iconv(fFromConvertor, &inBuf, &inBytesLeft,
                        (char**)&outBuf, &outBytesLeft);

        if (missing == -1)
        {
            // Output buffer overflow, which is fine. Continue conversion.
            switch (errno)
            {
                case E2BIG:
                    missing = 0;
                    break;
                case EILSEQ: {
                    IInvalidRequest exc(
                        //providing a message file name defined by user
                        IMessageText(kInvalidInputCharacter, IMessageText::messageFile()),
                        kInvalidInputCharacter,
                        IException::recoverable);
                    ITHROW(exc);
                }
                case EINVAL: {
                    IInvalidRequest exc(
                        //providing a message file name defined by user
                        IMessageText(kIncompleteCharOrSequence, IMessageText::messageFile()),
                        kIncompleteCharOrSequence,
                        IException::recoverable);
                    ITHROW(exc);
                }
            }
        }

        lengthConverted = sizeof(conversionBuffer) - outBytesLeft;

        if ((signed long)((to_next + lengthConverted) - to_limit) > 0)
            lengthToCopy = to_limit - to_next;
        else
            lengthToCopy = lengthConverted;

        from_next = (const UniChar*)inBuf;

        // Copy out the fix buffer conversionBuffer.
        char* p = 0;

        if (missing > 0) // There were missing characters it can't convert
        {
            // There are unmapped chars, handle based on unmappedBehavior.
            ITranscoder::EUnmappedBehavior ub = unmappedBehavior();
            switch (ub)
            {
                case ITranscoder::kOmit:
                    for (p = conversionBuffer; lengthToCopy > 0;
                         lengthToCopy--)
                        // omit DEL which is converted from an unmapped char.
                        if (*p == 0x7F)
                            p++;
                        else
                            *to_next++ = *p++;
                    break;
                case ITranscoder::kUseSub: // DEL ==> charSubstitute()
                    for (p = conversionBuffer; lengthToCopy > 0;
                         lengthToCopy--)
                        if (*p == 0x7F) {
                            *to_next++ = charSubstitute();
                            p++;
                        } else
                            *to_next++ = *p++;
                    break;
                case ITranscoder::kStop:
                    for (p = conversionBuffer; lengthToCopy > 0;
                         lengthToCopy--)
                        if (*p == 0x7F) {
                            // Re-calculate where to stop the input buffer
                            char *newOutBuf = (char *)conversionBuffer;
                            size_t newOutBytesLeft = p - conversionBuffer;
                            iconv(fFromConvertor, &preInBuf, &preInBytesLeft,
                                  &newOutBuf, &newOutBytesLeft);
                            from_next = (const UniChar *)preInBuf;
                            return codecvt_base::partial;
                        }
                        else
                            *to_next++ = *p++;
                    break;
            }
        }
        else // There is no unmapped char, simply copy out.
            for (p = conversionBuffer; lengthToCopy > 0; lengthToCopy--)
                *to_next++ = *p++;

        if (to_next == to_limit) // we are done with required conversion
            break;

        // Resets outBuf to beginning of conversionBuffer,
        // and outBytesLeft to full size of conversionBuffer.
        outBuf = conversionBuffer;
        outBytesLeft = sizeof(conversionBuffer);
    }
    
    if (missing)
        return codecvt_base::error;
    else if (from_next < from_end)
        return codecvt_base::partial;
    else if (from_next == from)
        return codecvt_base::noconv;
    else return codecvt_base::ok;
}
