/**
 * ITextBoundaryData wraps up the state machine tables needed for ISimpleTextBoundary.
 *
 * @package	Text and International
 * @category	Text Scanning
 *
 * @copyright
 *		IBM Open Class Library
 *		(C) Copyright International Business Machines Corporation 1997
 *		(C) Copyright Taligent, Inc.,  1997
 *		Licensed Material - Program-Property of IBM - All Rights Reserved.
 *		US Government Users Restricted Rights - Use, duplication, or disclosure
 *		restricted by GSA ADP Schedule Contract with IBM Corp.
 */
 
#ifndef _ITXTBDAT_
#define _ITXTBDAT_

#include <iprimtyp.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

class IWordBreakTable;
class IUnicodeClassMapping;
class ISpecialMapping;
 
#pragma enum(4)
#pragma pack(push,4)

/**
 * This class wraps up the data tables needed for SimpleTextBoundary.
 * It is statically instantiated for each type of text boundary.  This
 * class is not designed to be subclassed.
 */
class ITextBoundaryData {
public:
	~ITextBoundaryData(); // Do not subclass

	// Fast inline accessors
	const IWordBreakTable* forward() const;
	const IWordBreakTable* backward() const;
	const IUnicodeClassMapping* map() const;
 
	static const ITextBoundaryData kCharacterBreakData;
	static const ITextBoundaryData kWordBreakData;
	static const ITextBoundaryData kLineBreakData;
	static const ITextBoundaryData kSentenceBreakData;

	typedef unsigned char Node;
	typedef unsigned char Type;

private:
	static const Node kSI;

private:
	// Character data
	enum CharacterMapping
	{
		// These enum values must occur in this order; do not
		// modify unless you know what you are doing!  The forward
		// and backward data tables are indexed by these enums.
		kAccent_diacritic,
		kBaseForm,
		kBaseCR,
		kBaseLF,
		kCharacterCol_count
	};

	static const Node				kCharacterForwardData[];
	static const int				kCharacterForwardData_length;
	static IWordBreakTable*			kCharacterForward;
	static const Node				kCharacterBackwardData[];
	static const int				kCharacterBackwardData_length;
	static IWordBreakTable*			kCharacterBackward;
	static const Type				kCharacterRawMapping[];
	static const int				kCharacterRawMapping_length;
	static const ISpecialMapping	kCharacterExceptionChar[];
	static const int				kCharacterExceptionChar_length;
	static const bool				kCharacterExceptionFlags[];
	static IUnicodeClassMapping*	kCharacterMap;

private:
	// Word data
	enum WordMapping
	{
		// These enum values must occur in this order; do not
		// modify unless you know what you are doing!  The forward
		// and backward data tables are indexed by these enums.
		kBreak,
		kLetter,
		kNumber,
		kMidLetter,
		kMidLetNum,
		kPreNum,
		kPostNum,
		kMidNum,
		kPreMidNum,
		kBlank,
		kLF,
		kKata,
		kHira,
		kKanji,
		kDiacrit,
		kCR,
		kWordCol_count
	};

	static const Node				kWordForwardData[];
	static const int				kWordForwardData_length;
	static IWordBreakTable*			kWordForward;
	static const Node				kWordBackwardData[];
	static const int				kWordBackwardData_length;
	static IWordBreakTable*			kWordBackward;
	static const Type				kWordRawMapping[];
	static const int				kWordRawMapping_length;
	static const ISpecialMapping	kWordExceptionChar[];
	static const int				kWordExceptionChar_length;
	static const bool				kWordExceptionFlags[];
	static IUnicodeClassMapping*	kWordMap;

private:
	// Sentence data
	enum SentenceMapping
	{
		// These enum values must occur in this order; do not
		// modify unless you know what you are doing!  The forward
		// and backward data tables are indexed by these enums.
		kOther,
		kSpace,
		kTerminator,
		kAmbiguousTerm,
		kOpenBracket,
		kCloseBracket,
		kCJK,
		kParagraphBreak,
		kLowerCase,
		kUpperCase,
		kSNumber,
		kQuote,
		kSentCR,
		kSentenceCol_count_forward,
		kSentenceCol_count_backward		= kSentenceCol_count_forward
	};

	static const Node				kSentenceForwardData[];
	static const int				kSentenceForwardData_length;
	static IWordBreakTable*			kSentenceForward;
	static const Node				kSentenceBackwardData[];
	static const int				kSentenceBackwardData_length;
	static IWordBreakTable*			kSentenceBackward;
	static const Type				kSentenceRawMapping[];
	static const int				kSentenceRawMapping_length;
	static const ISpecialMapping	kSentenceExceptionChar[];
	static const int				kSentenceExceptionChar_length;
	static const bool				kSentenceExceptionFlags[];
	static IUnicodeClassMapping*	kSentenceMap;

private:
	// Line data
	enum LineMapping
	{
		// These enum values must occur in this order; do not
		// modify unless you know what you are doing!  The forward
		// and backward data tables are indexed by these enums.
		kLineBreak,
		kLineBlank,
		kLineCR,
		kLineNonBlank,
		kLineOp,
		kLineNami,
		kLineKata,
		kLineSmlKata,
		kLineSmlKtsu,
		kLineChoon,
		kLineHira,
		kLineSmlHira,
		kLineSmlHtsu,
		kLineHiraDitto,
		kLineDiacrit,
		kLineKanji,
		kLineKanjiDitto,
		kLinePreJwrd,
		kLinePostJwrd,
		kLineCol_count
	};

	static const Node				kLineForwardData[];
	static const int				kLineForwardData_length;
	static IWordBreakTable*			kLineForward;
	static const Node				kLineBackwardData[];
	static const int				kLineBackwardData_length;
	static IWordBreakTable*			kLineBackward;
	static const Type				kLineRawMapping[];
	static const int				kLineRawMapping_length;
	static const ISpecialMapping	kLineExceptionChar[];
	static const int				kLineExceptionChar_length;
	static const bool				kLineExceptionFlags[];
	static IUnicodeClassMapping*	kLineMap;

private:
	static const Node			kStop;
	static const Node			kSI_Stop;

private:
	/**
	 * Copy constructor and assignment operator provided to make
	 * compiler happy only. DO NOT CALL.
	 */
	ITextBoundaryData(const ITextBoundaryData&) {}
	ITextBoundaryData& operator=(const ITextBoundaryData&) { return *this; }
	ITextBoundaryData() {} // Do not subclass
	ITextBoundaryData(const IWordBreakTable* forward,
					 const IWordBreakTable* backward,
					 const IUnicodeClassMapping* map)
					 : fForward(forward), fBackward(backward), fMap(map) {}
		
	const IWordBreakTable*		fForward;
	const IWordBreakTable*		fBackward;
	const IUnicodeClassMapping*	fMap;
};

#pragma pack(pop)
#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

inline const IWordBreakTable* ITextBoundaryData::forward() const
{
	return fForward;
}

inline const IWordBreakTable* ITextBoundaryData::backward() const
{
	return fBackward;
}

inline const IUnicodeClassMapping* ITextBoundaryData::map() const
{
	return fMap;
}

#endif // _ITXTBDAT_
