// Revision: 21 1.15.1.8 source/core/testfw/ittxtbuf.cpp, testfw, ioc.v400, 001006 
/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright Taligent, Inc.,  1996                                                 *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
#include <ibase.hpp>
#ifdef IC_MOTIF
extern "C" 
{
#include <pthread.h>
}
#endif

#include <itiertxt.hpp>
#include <ittxtbuf.hpp>
#include <stdio.h>
#include <assert.h>
// Segment definitions
#ifdef IC_PAGETUNE
    #define _ITTXTBUF_CPP_
    #include <ipagetun.h>
#endif

#ifdef IC_MOTIF
// defect 29176
int __page_size;
int __page_size_K;
int __page_sizeX16;
int __page_sizeX24;
int __page_sizeM1;
pthread_mutex_t _mutex_global_np;
#endif

// Debugging flag for the gnarly indent/tier stuff
bool gDebugIndent = false;

//-----------------------------------------------------------------------------
// ITieredTextBuffer: PUBLIC Special member functions
//-----------------------------------------------------------------------------

ITieredTextBuffer::ITieredTextBuffer()
{
    fEchoTier = fTierStack[0] = ITieredTextBuffer::kNormal;
    fTierStackTop = 0;
    fIndentPending = true;
    fIndentLevel = 0;
    flogFile = NULL;
    fdoLogging = false;
}

ITieredTextBuffer::ITieredTextBuffer(const ITieredTextBuffer& rhs)
{
    flogFile = NULL;
    *this = rhs;    // Use operator= to do the work.
}

ITieredTextBuffer& ITieredTextBuffer::operator=(const ITieredTextBuffer& rhs)
{
    if (this != &rhs) 
    {
        fEchoTier = rhs.fEchoTier;
        fTierStackTop = rhs.fTierStackTop;
        for (short j=0; j<kTierStackDepth; j++) fTierStack[j] = rhs.fTierStack[j];
        fIndentPending = rhs.fIndentPending;
        fIndentLevel = rhs.fIndentLevel;
        fdoLogging = rhs.fdoLogging;
        if (flogFile) fclose(flogFile);
        flogFile = rhs.flogFile;
    }
    return *this;
}


ITieredTextBuffer::~ITieredTextBuffer()
{
    if (flogFile)
        fclose(flogFile);
}


//-----------------------------------------------------------------------------
// ITieredTextBuffer: PUBLIC C++ ostream-style formatted output
//-----------------------------------------------------------------------------

//
//  All of the operator<< methods do some formatting of the text, then call
//  indentAndPrintText.
//

ITieredTextBuffer& ITieredTextBuffer::operator<<(const long value)
{
    indentAndPrintText(IString(value));
    return *this;
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const int value)
{
    return operator<<((long)value);
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const short value)
{
    return operator<<((long)value);
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const signed char value)
{
    return operator<<((long)value);
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const unsigned long value)
{
    indentAndPrintText(IString(value));
    return *this;
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const unsigned int value)
{
    return operator<<((unsigned long)value);
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const unsigned short value)
{
    return operator<<((unsigned long)value);
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const unsigned char value)
{
    return operator<<((unsigned long)value);
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const long double value)
{
    // Arbitrary value
    const size_t kBufferSize = 1024;

    char buffer[kBufferSize];
    sprintf(buffer, "%Lg", value);
    indentAndPrintText(IString(buffer));
    return *this;
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const double value)
{
    indentAndPrintText(IString(value));
    return *this;
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const float value)
{
    indentAndPrintText(IString(value));
    return *this;
}


ITieredTextBuffer& ITieredTextBuffer::operator<<(const char c)
{
    indentAndPrintText(IString(c));
    return *this;
}

ITieredTextBuffer& ITieredTextBuffer::operator<<(const char* str)
{
    indentAndPrintText(IString(str));
    return *this;
}

/*
ITieredTextBuffer& ITieredTextBuffer::operator<<(const IString& text)
{
    indentAndPrintText(text);
    return *this;
}
*/
/*
ITieredTextBuffer& ITieredTextBuffer::operator<<(const void* p)
{
    indentAndPrintText(IString((long)p));
    return *this;
}
*/
ITieredTextBuffer& ITieredTextBuffer::operator<<(endlfn fn)
{
    return (*fn)(*this);
}

//-----------------------------------------------------------------------------
// TTieredTextBuffer: PUBLIC Indentation methods for ostream-stype output
//-----------------------------------------------------------------------------

unsigned short ITieredTextBuffer::setIndent(unsigned short indent)
//
//  setIndent changes the indent level. This should probably be changed to a
//  relative method which takes a +/- value to alter the indentation by. The
//  routine is weird because it handles the tiered indents. The rule is this:
//  When indenting to the left, simply move the indent level back. When going
//  to the right, fill in new indents with the current tier.
//
{
    unsigned short oldIndent = fIndentLevel;
    if (indent > kMaxIndent) indent = kMaxIndent;
    fIndentLevel = indent;

    // Update the fIndentTier array if indenting to the right (only)
    // Careful! Loaded with potential off-by-one errors!
    if (indent > oldIndent) 
    {
        for (indent = oldIndent + 1; indent <= fIndentLevel; indent++)
            fIndentTier[indent - 1] = tier();
    }

    return oldIndent;
}

unsigned short ITieredTextBuffer::setRelativeIndent(short indent)
{
    unsigned short oldIndent = fIndentLevel;
    unsigned short newIndent;
    
    if (oldIndent + indent < 0)
        newIndent = 0;
    else 
        newIndent = oldIndent + indent;
    
    return setIndent(newIndent);
}


unsigned short ITieredTextBuffer::indent() const
{
    return fIndentLevel;
}


//------------------------------------------------------------------------------
// TTieredTextBuffer: PRIVATE Methods
//------------------------------------------------------------------------------

ITieredTextBuffer::ETier Min(ITieredTextBuffer::ETier a, ITieredTextBuffer::ETier b)
{
    if (b > a) return a;
    else return b;
}

inline ITieredTextBuffer::ETier Max(ITieredTextBuffer::ETier a, ITieredTextBuffer::ETier b)
{
    if (a > b) return a;
    else return b;
}

void ITieredTextBuffer::indentAndPrintText(const IString& theText)
//
//  This is the internal method through which all operator<< variants go. It
//  looks for newline ('\n') characters, and adds indentation as appropriate
//  after them.
//
{
    unsigned i, start;
    unsigned length;
    ITieredText tieredText; // Scratch
    static const IString kOneIndent("    ");
    
    i = 0;
    length = theText.length();
    
    // We go once through this outer while loop for each segment of the text.
    // A segment consists of all characters up to an including a CR. If there
    // is no CR in the text, there is only one segment. For example:
    //        ___\n        _____                    2 segments
    //        \n            _____                    2 segments
    //        ___\n        ___\n                    2 segments
    //        ___\n        ___\n        _____        3 segments
    while (i < length) {
        start = i; // Remember the start of the segment
    
        // At the start of each segment we check for pending indentation. If
        // we need to indent we insert blanks at the START of the segment and
        // advance both the i index variable and the length.
        
        // The indent tier should be adjusted as follows:
        //   indentTier = max(indentTier, min(textTier1, textTier2,...))
        // where textTier1, textTier2,... are the tiers of the text items on the
        // line being indented. However, in practice this means delaying the
        // output of text until the entire line is received, which is bad. As a
        // compromise, the code just looks at the first text item, that is, it
        // does:
        //   indentTier = max(indentTier, textTier1)
        // which works as long as textTier1 == min(textTier1, textTier2,...).
        
        if (fIndentPending) 
        {
            ITieredTextBuffer::ETier textTier = tier(); // 'textTier1'

            tieredText = ITieredText(kOneIndent);
            for (short indent = 0; indent < fIndentLevel; indent++) 
            {
                ITieredTextBuffer::ETier theTier = Max(fIndentTier[indent], textTier);
                tieredText.setTier(theTier);
                printText(tieredText);
            }

            fIndentPending = false;
        }
        
        // after inserting indentation we look for the end of the segment.
        while (i<length && theText[i+1] != '\n') i++;
        
        // If there is a CR at the end of this segment then we setup a pending
        // indent before the next character.
        if (i<length) { // if characterAt(i) == CR
            fIndentPending = true; // Do an indent before the next character
            i++; // Advance past the CR
        }
        
        // Output the segment
        if (start < i) 
        {
            tieredText = theText.subString(start+1, i-start);
            tieredText.setTier(tier());
            printText(tieredText);
        }
    }
}

void ITieredTextBuffer::printText(const ITieredText& text)
{
    if (text.tier() <= fEchoTier) 
    {
        fwrite((const char*)text, 1, text.length(), stdout);
        fflush(stdout);
        if (fdoLogging)
            fprintf(flogFile, "%s", (const char *)text);
    }
}

//------------------------------------------------------------------------------
// TTieredTextBuffer: PUBLIC Echo Configuration
//------------------------------------------------------------------------------

void ITieredTextBuffer::setEchoTier(ITieredTextBuffer::ETier echoTier)
{
    fEchoTier = echoTier;
}

ITieredTextBuffer::ETier ITieredTextBuffer::echoTier() const
{
    return fEchoTier;
}

//-----------------------------------------------------------------------------
// ITieredTextBuffer: PUBLIC Tier methods for ostream-stype output
//-----------------------------------------------------------------------------

ITieredTextBuffer::ETier ITieredTextBuffer::setTier(ITieredTextBuffer::ETier theTier)
{
    ITieredTextBuffer::ETier oldTier = tier();
    if (theTier < ITieredTextBuffer::kTop) theTier = ITieredTextBuffer::kTop;
    if (theTier > ITieredTextBuffer::kBottom) theTier = ITieredTextBuffer::kBottom;
    fTierStack[fTierStackTop] = theTier;
    return oldTier;
}

ITieredTextBuffer::ETier ITieredTextBuffer::tier() const
{
    return fTierStack[fTierStackTop];
}

void ITieredTextBuffer::pushTier(ITieredTextBuffer::ETier newTier)
{
    if (fTierStackTop < (kTierStackDepth-1))
    {
        fTierStackTop++;

        // Setup for setTier
        fTierStack[fTierStackTop] = fTierStack[fTierStackTop-1];

        // MUST call setTier to work right!
        setTier(newTier); 
    }
}

ITieredTextBuffer::ETier ITieredTextBuffer::popTier()
{
    if (fTierStackTop > 0) 
    {
        fTierStackTop--;
        ITieredTextBuffer::ETier newTier = fTierStack[fTierStackTop];
        // Setup for setTier
        fTierStack[fTierStackTop] = fTierStack[fTierStackTop+1];

        // MUST call setTier to work right!
        return setTier(newTier); 
    }
    else return tier();
}

void ITieredTextBuffer::openLog(const char* logFileName)
{
    fdoLogging = true;
    flogFile = fopen(logFileName, "a+");
    if (!flogFile)
    {
        printf("can't open log file\n");
        fdoLogging = false;
    }
}

// friend function. Override "ostream& endl(ostream&)"
ITieredTextBuffer& endl(ITieredTextBuffer& stream)
{
    stream.indentAndPrintText(IString("\n"));
    return stream;
}
