// IBM Open Class Library
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// (C) Copyright IBM Corporation, 1992, 1997 All Rights Reserved
//
// Revision: 76 1.15.1.3 source/core/filesys/os2/os2vol.cpp, filesystem, ioc.v400, 001006 
//os2vol.cpp
//created by Pravin Singhal


#define      INCL_DOSMISC
#define      INCL_DOSERRORS
#define      INCL_DOSFILEMGR
#define      INCL_DOSDEVICES
#define      INCL_DOSDEVIOCTL

#include <os2.h>
#include "os2flsys.hpp"
#include "os2vol.hpp"
#include "os2dir.hpp"
#include "iexcfile.hpp"
#include <igenexc.hpp>
#include <itext.hpp>

# include <itmstamp.hpp>

#ifdef IC_PAGETUNE
    #define _OS2VOL_CPP_
    #include <ipagetun.h>
#endif

IOS2VolumeRecord::IOS2VolumeRecord(const IOS2VolumeRecord& copy)
 : fPath("")

{
 //Disallowed
}

IOS2VolumeRecord&
IOS2VolumeRecord::operator= (const IOS2VolumeRecord& copy)
{
   //Disallowed
   return *this;
}

IOS2VolumeRecord::~IOS2VolumeRecord()
{ }


IOS2VolumeRecord::IOS2VolumeRecord(const IFileName& path)
 : fPath(path)
{ }


bool
IOS2VolumeRecord::isEqual ( const IFSRecord * other) const
{
   IFSRecordLock lock1(this);
   IFSRecordLock lock2(other);
   return  (fPath == ((IOS2VolumeRecord *)other)->fPath);

}

bool
IOS2VolumeRecord::testConnection() const
{
   IFSRecordLock lock(this);
   ULONG attr;
   try
      {
             IOS2PathName path(fPath);
             attr = IOS2FileSystem::getFileAttributes(path);
      }
   catch (const IObjectNotFound& )
      {
             return false;
      }
   return (attr & FILE_DIRECTORY);
}

IFileName
IOS2VolumeRecord::name() const
{
   IFSRecordLock lock(this);
   return (fPath.getVolumeName());
}


//
IFileSystemEntity
IOS2VolumeRecord::copySelf(IDirectory &to, const IFileName& name) const
{
   IOS2PathName newPath = to.name() + name;
   {
      IFSRecordLock lock(this);
      IOS2FileSystem::createDirectoryDuplicate(fPath, newPath);
   }
   return new IOS2VolumeRecord(newPath);
}

IHostPathName
IOS2VolumeRecord::hostPath() const
{
   IFSRecordLock lock(this);
   return IHostPathName(fPath);

}

ITimeStamp
IOS2VolumeRecord::modificationTime() const
{
   IFSRecordLock lock(this);
    return IOS2FileSystem::modificationTime(fPath);
}

void
IOS2VolumeRecord::setModificationTime(const ITimeStamp& modificationTime)
{
   IFSRecordLock lock(this);
    IOS2FileSystem::setModificationTime(fPath, modificationTime);
}

IFileSize
IOS2VolumeRecord::freeSpace() const
{
      ULONG driveNum = fPath.getDriveNumber();
      FSALLOCATE fsDeviceInfo;

      APIRET returnValue = DosQueryFSInfo(driveNum, FSIL_ALLOC, &fsDeviceInfo,sizeof(fsDeviceInfo));
      if (returnValue != NO_ERROR)
         IOS2FileSystem::throwError(returnValue, "DosQueryFSInfo");

      IFileSize spaceAvail = fsDeviceInfo.cUnitAvail * fsDeviceInfo.cSectorUnit * fsDeviceInfo.cbSector;

      return spaceAvail;
}

IFileSize
IOS2VolumeRecord::totalSpace() const
{
      ULONG driveNum = fPath.getDriveNumber();
      FSALLOCATE fsDeviceInfo;

      APIRET returnValue = DosQueryFSInfo(driveNum, FSIL_ALLOC, &fsDeviceInfo,sizeof(fsDeviceInfo));
      if (returnValue != NO_ERROR)
         IOS2FileSystem::throwError(returnValue, "DosQueryFSInfo");

      IFileSize totalAvail = fsDeviceInfo.cUnit * fsDeviceInfo.cSectorUnit * fsDeviceInfo.cbSector;

      return totalAvail;

}

bool
IOS2VolumeRecord::local() const
{
      IFSRecordLock lock(this);
      BYTE queryBuffer[sizeof(FSQBUFFER2) + 3 * CCHMAXPATH] = {0};
      ULONG  bufferLen = sizeof(queryBuffer);
      PFSQBUFFER2 pQueryBuffer = (PFSQBUFFER2 )queryBuffer;

      IText driveName = fPath.getDriveName();
      APIRET returnValue  = DosQueryFSAttach(driveName, 0,
                                                  FSAIL_QUERYNAME, pQueryBuffer,&bufferLen);
      if (returnValue != NO_ERROR)
         IOS2FileSystem::throwError(returnValue, "DosQueryFSAttach");

      return (pQueryBuffer->iType & FSAT_LOCALDRV);

}


IVolume::EState
IOS2VolumeRecord::removable() const
{

   typedef struct
   {
      UCHAR commandInfo;
      UCHAR driveUnit;
   } BlockDeviceParam;

   BlockDeviceParam param;
   param.commandInfo = 0;
   param.driveUnit = fPath.getDriveNumber() -1; //drive units A = 0, B = 1 c = 2 etc..
   BYTE data;
   ULONG dataLen = sizeof(data);
   ULONG paramLen = sizeof(param);
      APIRET resultValue = DosDevIOCtl( -1,
                                 IOCTL_DISK,
                                 DSK_BLOCKREMOVABLE,
                                 &param,
                                 paramLen,
                                 &paramLen,
                                 &data,
                                 dataLen,
                                 &dataLen);
     if (resultValue != NO_ERROR)
          IOS2FileSystem::throwError(resultValue, "DosDevIOCtl::DSK_BLOCKREMOVABLE");

      if  (data == 0)
         return IVolume::kTrue;
      else
         return IVolume::kFalse;
}

IVolume::EState
IOS2VolumeRecord::remote() const
{
      if (local())
             return IVolume::kFalse;
      else
             return IVolume::kTrue;
}


IDirectory
IOS2VolumeRecord::parent() const
{
   return IDirectory();
}


IDirectory
IOS2VolumeRecord::topmostParent() const
{
   return IDirectory();
}



IVolume
IOS2VolumeRecord::volume() const
{
   return IVolume((IFSRecord *)this);
}


IFile
IOS2VolumeRecord::createFile(const IFileName& name,IFileSize initialFileSize)
{
      IFSRecordLock lock(this);
      return IOS2FileSystem::createFile(fPath + name, initialFileSize);
}

IDirectory
IOS2VolumeRecord::createDirectory(const IFileName& name)
{
      IFSRecordLock lock(this);
      return IOS2FileSystem::createDirectory(fPath + name);
}

IFSIterator *
IOS2VolumeRecord::createIterator() const
{
      IFSRecordLock lock(this);
      return new IOS2DirectoryIterator(fPath);
}


IFileSystemEntity
IOS2VolumeRecord::lookUpByName(const IFileName& theName) const
{
      IFSRecordLock lock(this);
      return IOS2FileSystem::getFileSystemEntity(fPath + theName);
}


void
IOS2VolumeRecord::setCurrent() const
{
   IFSRecordLock lock(this);
   IOS2FileSystem::setCurrentDirectory(fPath);
}


IOS2VolumeIterator::IOS2VolumeIterator()
   : fDriveStrings( IOS2FileSystem::getValidDriveStrings()),
     fCurrentDrive( fDriveStrings)
{ }

IOS2VolumeIterator::~IOS2VolumeIterator()
{
      delete [] fDriveStrings;
}


IOS2VolumeIterator::IOS2VolumeIterator(const IOS2VolumeIterator& other)
{
      fDriveStrings = new char[ 26 * 3 + 1];
      memcpy(fDriveStrings, other.fDriveStrings, 26*3 + 1);
      fCurrentDrive = fDriveStrings + (other.fCurrentDrive - other.fDriveStrings);
}

IFileSystemEntity
IOS2VolumeIterator::first()
{
      fCurrentDrive = fDriveStrings;
      return next();
}

IFileSystemEntity
IOS2VolumeIterator::next()
{
   IFSRecord* result = NULL;

   while (*fCurrentDrive && !result)
   {
      IFileName path = fCurrentDrive;

      while (*fCurrentDrive++)
      {  }

      try
      {
         //check if the drive is still valid and online.
         IOS2FileSystem::getVolumeInformation(path);
         result = new IOS2VolumeRecord(path);
      }
      catch (const IVolumeOffline& volExcept)
      {
             // do nothing...go to the next drive..
      }
      catch (const IDeviceError& deviceExcept)
      {
         IDeviceError   devExcpt( deviceExcept );
         ITHROW( devExcpt );
      }
      catch (const IEntityInUse& entityExcept)
      {
         IEntityInUse   entExcpt( entityExcept );
         ITHROW( entExcpt );
      }
   }
   return IFileSystemEntity(result);
}

