/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 74 1.14.2.1 source/core/filesys/os2/os2path.cpp, filesystem, ioc.v400, 001006 
//os2path.cpp
//Pravin Singhal

//#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <memory.h>

#include "os2path.hpp"
#include <itext.hpp>
//#include <stdio.h>

#ifdef IC_PAGETUNE
    #define _OS2PATH_CPP_
    #include <ipagetun.h>
#endif

IOS2PathName::IOS2PathName()
 : fPathName("")
{ }

IOS2PathName::IOS2PathName(const IOS2PathName& copy)
 : fPathName(copy.fPathName)
{  }


IOS2PathName::IOS2PathName(const IText& other)
  : fPathName(other)
{
  // take care of trailing slash which is required for a volume name
  // and not for other pathnames;
  if (fPathName.length() > 0)
      fixPathName();
}


IOS2PathName::~IOS2PathName()
{ }

//return the driveName of this pathname if full path
IText
IOS2PathName::getDriveName() const
{

      char driveName[3];
      driveName[0] = '\0';
      driveName[1] = ':';
      driveName[2] = '\0';
      if ((fPathName.length() >=2) && (fPathName[(size_t)1] == kOS2DriveSeperator))
         driveName[(size_t)0] = fPathName[(size_t)0]; //toupper(fPathName[0]);

      return IText(driveName);

}

const char*
IOS2PathName::findLastSeperator() const
{

      //point to the last char
      const char* found = 0;
      const char* pathName = fPathName;
      const char* lastChar = pathName + fPathName.length();
      while (( --lastChar >= pathName) && !found)
            if (*lastChar == kOS2PathSeperator)
               found = ++lastChar;

      return found;
}

IText
IOS2PathName::getParentName() const
{
      IText result;
      char parentName[255];
      const char* fullName = fPathName;
      const char* leafName = findLastSeperator();
      if (leafName > fullName)
      {
           int parentLen = (leafName - 1) - fullName;
           memcpy(parentName, fullName, parentLen);
           parentName[parentLen] = '\0';
           result = parentName;
      }
      return result;

}

IText
IOS2PathName::getLeafName() const
{
      const char* leaf = findLastSeperator();
      return IText(leaf);
}


bool
IOS2PathName::getUNCName(IText& uncName)  const
{
      bool retVal = false;
      const char* currentPos = fPathName;
      char name[255] = { 0 };
      char* pathName = &name[0];
      if  ((fPathName[(size_t)0] == kOS2PathSeperator) && (fPathName[(size_t)1] == kOS2PathSeperator))
      {
             *pathName++ = *currentPos++;
             *pathName++ = *currentPos++;
             int seperatorCount = 0;

             while ((seperatorCount < 2) && (*currentPos))
             {
                    *pathName++ = *currentPos++;
                    if (*currentPos == kOS2PathSeperator)
                           seperatorCount++;

             }
             *pathName = '\0';
             retVal = true;
             uncName = name;
       }
      return retVal;
}


IText
IOS2PathName::getVolumeName() const
{
      IText name;
      char volName[4];
      volName[0] = '\0';volName[2] = '\\';volName[1] = ':';
      volName[3] = '\0';
  //  printf("IOS2PathName::getVolumeName::fPathName: %s \n",(const char*)fPathName);
      if (fPathName[(size_t)1] == ':')
      {
         volName[0] = fPathName[(size_t)0];
         return IText(volName);
      }
      else
      {
             if (getUNCName(name))
                return name;
      }
      return IText(volName);
}

// returns the driveNumber if full path else returns -1;
int
IOS2PathName::getDriveNumber() const
{
  //  char driveName[3];
  //  ::strcpy(driveName,getDriveName());
      int driveNo = -1;
      IText driveName = getDriveName();
      if (driveName[(size_t)0])
             driveNo = toupper(driveName[(size_t)0]) - 'A' + (size_t)1; // change to add, this was subtracting one...ajd-8/2/1996

      return driveNo;
}

IText
IOS2PathName::getDriveName(ULONG driveNumber) const
{
      IText result;
      if ((driveNumber > 0) && (driveNumber <=26))
      {
             char   driveName[3];
             driveName[1] = ':'; driveName[2] = '\0';
             driveName[0] = (int)driveNumber + 'A' - 1;
             result = driveName;
      }
      return result;

}

//returns true if the pathname is in UNC format
bool
IOS2PathName::isUNCName() const
{
     const char* path = fPathName;
     if ((path[0] == kOS2PathSeperator) && (path[1] == kOS2PathSeperator))
        return true;
     return false;
}

// changes the driveletter of the path.
// does nothing if the pathname is in UNC format.
void
IOS2PathName::updateDrive(ULONG driveNumber)
{

      if ((!isUNCName()) && ((driveNumber > 0) && (driveNumber <= 26)))
      {
             char driveName[4];
              const char* path = fPathName;
             driveName[1] = ':';
             driveName[2] = '\0';
             driveName[3] = '\0';
              // null or relative path
              driveName[0] = (int)driveNumber + 'A' - 1;
             if ((!path[0]) || ((path[0] != kOS2PathSeperator) &&
                                      (path[1] != kOS2DriveSeperator)))
              {
                      driveName[2] = '\\';
                      fPathName = IText(driveName) + fPathName;
              }
              else
                 // full path \tmp ...change to drive:\tmp
              if (path[0] == kOS2PathSeperator)
                 fPathName = IText(driveName) + fPathName;
              else
                 // a:\tmp change to drive:\tmp
             if (fPathName[(size_t)1] == kOS2DriveSeperator)
                    fPathName[(size_t)0] = (int) driveNumber + 'A' - 1;
      }
}


IOS2PathName
IOS2PathName::appendLeafName(const IText& leafName)
{
       ULONG len = fPathName.length();

       if (leafName.length() > 0)
       {
             if (fPathName[(size_t)(len-1)] != kOS2PathSeperator)
                   fPathName  = fPathName + "\\";
             fPathName = fPathName + leafName;
       }

       return *this;
}

IOS2PathName
IOS2PathName::changeLeafName(const IText& newLeafName)
{
          if (!(isVolumeName()))
          {
                  //printf("IOS2PathName::changeLeafName::fPathName %s \n",(const char*)fPathName);
                  const char* fullName = fPathName;
                  const char* leafName = findLastSeperator();
                  fPathName.replace((leafName - fullName), strlen(leafName),
                                                        newLeafName);
                  //printf("IOS2PathName::changeLeafName::fPathName %s \n",(const char*)fPathName);
          }
          return *this;
}

bool
IOS2PathName::isVolumeName() const
{
      bool result = false;
      if (fPathName.length() > 0)
      {
             if (fPathName.length() <=3)
             {
                    if (fPathName[(size_t)1] == kOS2DriveSeperator)
                           result = true;
             }
             else
             {
                    if ((fPathName[(size_t)0] == kOS2PathSeperator) && (fPathName[(size_t)1] == kOS2PathSeperator))
                       result = true;
             }
      }
      return result;

}

// appends a trailing slash if it is a volume name.
// if not a volume name then removes trailing slashes if there is any.
// expects a non NULL fPathName;
void
IOS2PathName::fixPathName()
{
      size_t nameLen = fPathName.length();
      if (isVolumeName())
      {
         if (nameLen <=2)
             fPathName += "\\";
      }
      else
      {
         if (fPathName[nameLen -1] == kOS2PathSeperator)
           fPathName.erase(nameLen-1,1);
      }
}


IOS2PathName&
IOS2PathName::operator+=(const IOS2PathName& other)
{
      appendLeafName(other);
      return *this;
}

IOS2PathName&
IOS2PathName::operator+=(const IText& other)
{
      appendLeafName(other);
      return *this;
}

// compares two pathnames ignoring the case.
bool
IOS2PathName::isEqual(const IOS2PathName& other) const
{
      size_t thisLen = fPathName.length();
      if (other.fPathName.length() != thisLen)
           return false;

      const char* thisPath = fPathName;
      const char* otherPath = other.fPathName;
      bool result = true;

      for(int i = 0; i < thisLen; i++)
      {
         if (toupper(thisPath[i]) != toupper(otherPath[i]))
         {
             result = false;
             break;
         }
      }
      return result;
}

bool
IOS2PathName::operator==(const IOS2PathName& other) const
{

      return isEqual(other);
}

IOS2PathName::operator IText() const
{
      return fPathName;
}

IOS2PathName
operator+ (const IOS2PathName& first, const IOS2PathName& second)
{
      IOS2PathName result = first;
      return result += second;
}


IOS2PathName
operator+ (const IOS2PathName& first, const IText& second)
{
      IOS2PathName result = first;
      return result += second;
}

