// IBM Open Class Library
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// (C) Copyright IBM Corporation, 1992, 1997 All Rights Reserved
//
#include <iiseqdil.h>

#ifdef IC_PAGETUNE
  #define _IISEQDIL_CPP_
  #include <ipagetun.h>
#endif


#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (nocls, nocnd, nocns, nocnv, noext, nognr, novft)
#endif

#if defined (__SOM_ENABLED__) && defined (__RRBC_LIB__)
#pragma SOMDefine (IASequenceAsDilTableOps)
#endif

// -------------
// IDilTableImpl
// -------------

// public members

inline void*
IDilTableImpl::
ElementAtIndex (INumber index) const
{
    size_t offset = 4;
    if (ivUnused)
        offset = ivUnused->offset;
    if (offset != 4 && offset != 8 && offset != 12 && offset != 16) {
        offset = 4;
    }
    return (void *)(((size_t)((Node*)NodeAtIndex (index))->ivPtr) + offset);
}

inline void*&
IDilTableImpl::
PtrAtIndex (INumber index) const
{ return ((Node*)NodeAtIndex (index))->ivPtr;
}

inline void
IDilTableImpl::
GetUnused (INumber index)
{ IASSERT (index > 0 && index <= ivAllocatedElements)
  IASSERT (ivUnused && ivUnused->unused != 0)

  PtrAtIndex (index) = ivUnused->unused;
  ivUnused->unused = *((void**)(ivUnused->unused) + 1);
}

inline void
IDilTableImpl::
PutUnused (INumber index)
{ IASSERT (index > 0 && index <= ivAllocatedElements)
  IASSERT (PtrAtIndex (index) != 0)

  *((void**)PtrAtIndex (index) + 1) = ivUnused->unused;
  ivUnused->unused = PtrAtIndex (index);
  PtrAtIndex (index) = 0;
}

inline void
IDilTableImpl::
RemoveAtIndex (INumber index)
{ IASSERT (index > 0 && index <= ivAllocatedElements)

  PtrAtIndex (index) = 0;
}

// -----------------------------
// ISequenceAsDilTableCursorImpl
// -----------------------------

// public members

ISequenceAsDilTableCursorImpl::
ISequenceAsDilTableCursorImpl (IACollectionImpl const& collection)
: ICursorImpl (collection), ivIndex (0)
{
}

ISequenceAsDilTableCursorImpl::
ISequenceAsDilTableCursorImpl
  (ISequenceAsDilTableCursorImpl const& cursor)
: ICursorImpl (cursor), ivIndex (cursor.ivIndex)
{
}

ISequenceAsDilTableCursorImpl::
~ISequenceAsDilTableCursorImpl ()
{
}

ICursorImpl*
ISequenceAsDilTableCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
ISequenceAsDilTableCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

#if ! defined (__INO_MNGCURSOR__)
// --------------------------------
// ISequenceAsDilTableMngCursorImpl
// --------------------------------

INumber const
ISequenceAsDilTableMngCursorImpl::kIsInBetween = (INumber) -1;

// public members

ISequenceAsDilTableMngCursorImpl::
ISequenceAsDilTableMngCursorImpl
  (ISequenceAsDilTableImpl const& collection)
: ISequenceAsDilTableCursorImpl (collection),
  ivMngCollection ((ISequenceAsDilTableImpl*)&collection)
{ ivNext = ivMngCollection->ivMngCursors;
  ivMngCollection->ivMngCursors = this;
}

ISequenceAsDilTableMngCursorImpl::
~ISequenceAsDilTableMngCursorImpl ()
{ Self*& first = ivMngCollection->ivMngCursors;
  if (this == first) {
    first = ivNext;
  } else {
    // find previous cursor
    for (Self* cursor = first;
         cursor->ivNext != this;
         cursor = cursor->ivNext);

    cursor->ivNext = ivNext;
  }
}

ICursorImpl*
ISequenceAsDilTableMngCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
ISequenceAsDilTableMngCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

bool
ISequenceAsDilTableMngCursorImpl::
IsInBetween () const
{ return ivMngCollection->IsInBetween (*this);
}

bool
ISequenceAsDilTableMngCursorImpl::
IsMngCursor () const
{ return true;
}

void
ISequenceAsDilTableMngCursorImpl::
operator= (ISequenceAsDilTableMngCursorImpl const& cursor)
{ Inherited::operator= (cursor);
  ivNextIndex     = cursor.ivNextIndex;
  ivPreviousIndex = cursor.ivPreviousIndex;
  if (ivMngCollection != cursor.ivMngCollection) {

    // detach 'this' from current list of mngcursors
    Self*& first = ivMngCollection->ivMngCursors;
    if (this == first)
      first = ivNext;
    else {
      // find previous cursor
      for (Self* cursor = first;
           cursor->ivNext != this;
           cursor = cursor->ivNext);

      cursor->ivNext = ivNext;
    }

    // insert 'this' in new list of mngcursors
    ivMngCollection = cursor.ivMngCollection;
    ivNext = (cursor.ivMngCollection)->ivMngCursors;
    ivMngCollection->ivMngCursors = this;
  }
}
#endif

// ------------------------
// ISequenceAsDilTableImpl
// ------------------------

char const*
ISequenceAsDilTableImpl::cvClassName = "SequenceAsDilTable";

// inline private members

inline ISequenceAsDilTableImpl::Ops&
ISequenceAsDilTableImpl::
OpsOf (ISequenceAsDilTableImpl const& collection)
{ return (Ops&) IACollectionImpl::OpsOf (collection);
}

inline void*
ISequenceAsDilTableImpl::
ElementOf (void const* node) const
{ return (OpsOf (*this).ivContainsDTSObjects ?
    (void*)*(void const**)node :
    (void*) (void const**)node);

}

inline INumber&
ISequenceAsDilTableImpl::
IndexOf (ICursorImpl const& cursor)
{ return ((Cursor&)cursor).ivIndex;
}

inline bool
ISequenceAsDilTableImpl::
IsInBetween (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return (((Cursor const&)cursor).ivIndex == MngCursor::kIsInBetween);
#else
  return false;
#endif
}

inline bool
ISequenceAsDilTableImpl::
IsRemoved (INumber index) const
{ return (ivTable.PtrAtIndex (index) == 0);
}

inline INumber&
ISequenceAsDilTableImpl::
NextIndexOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivNextIndex;
#else
  return ((Cursor&)cursor).ivIndex; // just a dummy
#endif
}

inline INumber&
ISequenceAsDilTableImpl::
PreviousIndexOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivPreviousIndex;
#else
  return ((Cursor&)cursor).ivIndex;  // just a dummy
#endif
}

// inline protected members

inline void
ISequenceAsDilTableImpl::
Assign (void* element1, void const* element2) const
{ OpsOf (*this).Assign (element1, element2);
}

inline void*
ISequenceAsDilTableImpl::
CreateNodes (INumber n, size_t& size) const
{
 return OpsOf (*this).CreateNodes
     (n, size, (size_t &)((ivTable.ivUnused)->offset));
}

inline void
ISequenceAsDilTableImpl::
ConstructNode (void* node, void const* element) const
{ OpsOf (*this).ConstructNode (node, element);
}

#if ! defined (__INO_STREAMING__)
inline void
ISequenceAsDilTableImpl::
ConstructNodeWithStreamer (void* node, IDataStream& fromWhere) const
{ OpsOf (*this).ConstructNodeWithStreamer (node, ivStreamer, fromWhere);
}
#endif

inline void
ISequenceAsDilTableImpl::
DeleteNodes (void* nodes) const
{ OpsOf (*this).DeleteNodes (nodes);
}

inline void
ISequenceAsDilTableImpl::
DestroyNode (void* node) const
{ OpsOf (*this).DestroyNode (node);
}

// inline public members

#define inline

inline bool
ISequenceAsDilTableImpl::
Add (void const* element, ICursorImpl& cursor)
{ Self::AddAsLast (element, cursor);
  return true;
}

inline void*
ISequenceAsDilTableImpl::
Any () const
{ return ElementOf (ivTable.ElementAtIndex (ivTable.ivFirst));
}

inline char const*
ISequenceAsDilTableImpl::
ClassName () const
{ return cvClassName;
}

inline void*
ISequenceAsDilTableImpl::
ElementAt (ICursorImpl const& cursor) const
{ return ElementOf (ivTable.ElementAtIndex (IndexOf (cursor)));
}

inline void*
ISequenceAsDilTableImpl::
First () const
{ return ElementOf (ivTable.ElementAtIndex (ivTable.ivFirst));
}

inline bool
ISequenceAsDilTableImpl::
IsBounded () const
{ return false;
}

inline bool
ISequenceAsDilTableImpl::
IsEmpty () const
{ return (ivTable.ivNumberOfElements == 0);
}

inline bool
ISequenceAsDilTableImpl::
IsFirstAt (ICursorImpl const& cursor) const
{ return (IndexOf (cursor) == ivTable.ivFirst);
}

inline bool
ISequenceAsDilTableImpl::
IsFull () const
{ return false;
}

inline bool
ISequenceAsDilTableImpl::
IsLastAt (ICursorImpl const& cursor) const
{ return (IndexOf (cursor) == ivTable.ivLast);
}

inline void*
ISequenceAsDilTableImpl::
Last () const
{ return ElementOf (ivTable.ElementAtIndex (ivTable.ivLast));
}

inline INumber
ISequenceAsDilTableImpl::
NumberOfElements () const
{ return ivTable.ivNumberOfElements;
}

inline IPosition
ISequenceAsDilTableImpl::
PositionAt (ICursorImpl const& cursor) const
{ return PositionAtIndex (IndexOf (cursor));
}

inline void
ISequenceAsDilTableImpl::
RemoveFirst ()
{ RemoveAtIndex (ivTable.ivFirst);
}

inline void
ISequenceAsDilTableImpl::
RemoveLast ()
{ RemoveAtIndex (ivTable.ivLast);
}

inline void
ISequenceAsDilTableImpl::
ReplaceAt (ICursorImpl const& cursor, void const* element)
{ Assign (Self::ElementAt (cursor), element);
}

inline bool
ISequenceAsDilTableImpl::
SetToNext (ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    IndexOf (cursor) = NextIndexOf (cursor);
    return (IndexOf (cursor) != 0);
  }
  else
#endif
    return SetToNextIndex (IndexOf (cursor));
}

inline bool
ISequenceAsDilTableImpl::
SetToPrevious (ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    IndexOf (cursor) = PreviousIndexOf (cursor);
    return (IndexOf (cursor) != 0);
  }
  else
#endif
    return SetToPreviousIndex (IndexOf (cursor));
}

#undef inline

// public members

ISequenceAsDilTableImpl::
ISequenceAsDilTableImpl (Ops& ops, INumber numberOfElements)
: IASequenceImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivTable ()
{ if (numberOfElements == 0)
    numberOfElements++;
  Self::Initialize (numberOfElements);
}

ISequenceAsDilTableImpl::
ISequenceAsDilTableImpl
  (Ops& ops, ISequenceAsDilTableImpl const& collection)
: IASequenceImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivTable ()
{ Self::Initialize (collection.ivTable.ivAllocatedElements);
  Self::AddAllFrom (collection);
#if ! defined (__INO_STREAMING__)
  ivStreamer = collection.ivStreamer;
#endif
}

ISequenceAsDilTableImpl::
~ISequenceAsDilTableImpl ()
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    cursor->Invalidate();
  }
#endif
  Self::RemoveAll ();
  DeleteNodes (ivTable.ivNodes);
  delete ivTable.ivUnused;
}

void
ISequenceAsDilTableImpl::
AddAllFrom (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ()) {
    Self const& seq =  (Self const&)collection;
    if (! ivTable.CanAdd (seq.ivTable.ivNumberOfElements))
      GrowBy (seq.ivTable.ivNumberOfElements);
    CopyFrom (seq);
  }
  else
    Inherited::AddAllFrom (collection);
}

void
ISequenceAsDilTableImpl::
AddAsFirst (void const* element, ICursorImpl& cursor)
{ if (Self::IsEmpty ())
    Self::AddAsLast (element, cursor);
  else {
    Self::SetToFirst (cursor);
    Self::AddAsPrevious (element, cursor);
  }
}

void
ISequenceAsDilTableImpl::
AddAsLast (void const* element, ICursorImpl& cursor)
{ if (! ivTable.CanAdd (1))
    GrowBy (1);

  INumber index = ivTable.ivLast;
  ivTable.IncMod (index);
  if (index == ivTable.ivFirst && ! Self::IsEmpty ())
    MakeFree (index);
  ivTable.GetUnused (index);
  ConstructNode (ivTable.PtrAtIndex (index), element);
  IndexOf (cursor) = index;
  ivTable.ivNumberOfElements++;
  ivTable.ivLast = index;
}

#if ! defined (__INO_STREAMING__)
void
ISequenceAsDilTableImpl::
AddAsLastFromStreamer (IDataStream& fromWhere, ICursorImpl& cursor)
{ INumber index = ivTable.ivLast;
  ivTable.IncMod (index);
  if (index == ivTable.ivFirst && ! Self::IsEmpty ())
    MakeFree (index);
  ivTable.GetUnused (index);
  ConstructNodeWithStreamer (ivTable.PtrAtIndex(index), fromWhere);
  IndexOf (cursor) = index;
  ivTable.ivNumberOfElements++;
  ivTable.ivLast = index;
}
#endif

void
ISequenceAsDilTableImpl::
AddAsNext (void const* element, ICursorImpl& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    if (IndexOf (cursor) = NextIndexOf (cursor))
      Self::AddAsPrevious (element, cursor);
    else
      Self::AddAsLast (element, cursor);
  }
  else
#endif
  { if (IndexOf (cursor) == ivTable.ivLast)
      Self::AddAsLast (element, cursor);
    else {
      Self::SetToNext (cursor);
      Self::AddAsPrevious (element, cursor);
    }
  }
}

void
ISequenceAsDilTableImpl::
AddAsPrevious (void const* element, ICursorImpl& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    if ((IndexOf (cursor) = PreviousIndexOf (cursor)) == 0)
      Self::SetToFirst (cursor);
    else if (! Self::SetToNext (cursor)) {
      Self::AddAsLast (element, cursor);
      return;
    }
  }
#endif

  INumber index = IndexOf (cursor); // no reference 'cause of mng cursor
  if (! ivTable.CanAdd (1)) {
    index = ivTable.Pos (index);
    GrowBy (1);
  }

  bool isFirst = (index == ivTable.ivFirst);
  MakeFree (index);
  if (isFirst)
    ivTable.ivFirst = index;
  ivTable.GetUnused (index);
  ConstructNode (ivTable.PtrAtIndex (index), element);
  IndexOf (cursor) = index;
  ivTable.ivNumberOfElements++;
}

void
ISequenceAsDilTableImpl::
AddAtPosition (IPosition position,
               void const* element,
               ICursorImpl& cursor)
{ if (position == Self::NumberOfElements () + 1)
    Self::AddAsLast (element, cursor);
  else {
    Self::SetToPosition (position, cursor);
    Self::AddAsPrevious (element, cursor);
  }
}

bool
ISequenceAsDilTableImpl::
AllElementsDo (IApplFunc applFunc, void* addArg)
{ INumber index = ivTable.ivFirst;
  for (INumber i = 0; i < Self::NumberOfElements (); i++) {
    if (! (*applFunc) (ElementOf (ivTable.ElementAtIndex (index)),
                       addArg))
      return false;
    SetToNextIndex (index);
  }
  return true;
}

bool
ISequenceAsDilTableImpl::
CheckCursor (ICursorImpl const& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor))
    return true;
  else
#endif
  {
    INumber const& index = IndexOf (cursor);
    if (Self::IsEmpty () || IsRemoved (index))
      return false;
    else if (ivTable.ivFirst <= ivTable.ivLast)
      return ivTable.ivFirst <= index && index <= ivTable.ivLast;
    else
      return (ivTable.ivFirst <= index &&
                index <= ivTable.ivAllocatedElements) ||
              index <= ivTable.ivLast;
  }
}

IACollectionImpl*
ISequenceAsDilTableImpl::
Clone () const
{ return (IACollectionImpl*) CheckPointer (OpsOf (*this).Clone ());
}

void
ISequenceAsDilTableImpl::
Copy (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ())
    operator= ((Self const&)collection);
  else
    Inherited::Copy (collection);
}

ISequenceAsDilTableImpl::Cursor*
ISequenceAsDilTableImpl::
CreateCursor () const
{ return (Cursor*) CheckPointer (new Cursor (*this));
}

ISequenceAsDilTableImpl::MngCursor*
ISequenceAsDilTableImpl::
CreateMngCursor () const
{ return (MngCursor*) CheckPointer (new MngCursor (*this));
}

void*
ISequenceAsDilTableImpl::
ElementAtPosition (IPosition position) const
{ Self::SetToPosition (position, CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

bool
ISequenceAsDilTableImpl::
IsConsistent () const
{ if (Self::IsEmpty ()) {
    INumber last = ivTable.ivLast;
    return (ivTable.IncMod (last) == ivTable.ivFirst);
  } else {
    if (IsRemoved (ivTable.ivFirst) || IsRemoved (ivTable.ivLast))
      return false;
    INumber numberOfElements = 0;
    for (INumber index = ivTable.ivFirst; ; ivTable.IncMod (index)) {
      if (! IsRemoved (index)) {
        numberOfElements++;
      }
      if (index == ivTable.ivLast) {
        break;
      }
    }
    return (numberOfElements == Self::NumberOfElements ());
  }
}

INumber
ISequenceAsDilTableImpl::
RemoveAll ()
{ INumber n = Self::NumberOfElements ();
  INumber index = ivTable.ivFirst;
  for (INumber i = ivTable.ivNumberOfElements; i != 0; i--) {
    DestroyNode (ivTable.PtrAtIndex (index));
    ivTable.PutUnused (index);
    SetToNextIndex (index);
  }

#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    if (cursor->IsValid ()) {
      IndexOf (*cursor)         = MngCursor::kIsInBetween;
      NextIndexOf (*cursor)     = 0;
      PreviousIndexOf (*cursor) = 0;
    }
  }
#endif

  ivTable.ivFirst = 1;
  ivTable.ivLast = ivTable.ivAllocatedElements;
  ivTable.ivNumberOfElements = 0;
  return n;
}

INumber
ISequenceAsDilTableImpl::
RemoveAll (IPredFunc predFunc, void* addArg)
{ INumber removed = 0;
  if (! Self::IsEmpty ()) {
    INumber index = ivTable.ivFirst;
    for (;;) {
      bool isLast = (index == ivTable.ivLast);
      if ((*predFunc) (ElementOf (ivTable.ElementAtIndex (index)),
                       addArg)) {
        RemoveAtIndex (index);
        removed++;
      }
      if (isLast) {
        break;
      }
      SetToNextIndex (index);
    }
  }
  return removed;
}

void
ISequenceAsDilTableImpl::
RemoveAt (ICursorImpl& cursor)
{ RemoveAtIndex (IndexOf (cursor));
  if (! IsInBetween (cursor))
    cursor.Invalidate ();
}

void
ISequenceAsDilTableImpl::
RemoveAtPosition (IPosition position)
{ Self::SetToPosition (position, CursorOf (*this));
  RemoveAtIndex (IndexOf (CursorOf (*this)));
}

void
ISequenceAsDilTableImpl::
ReplaceAtPosition (IPosition position, void const* element)
{ Self::SetToPosition (position, CursorOf (*this));
  ReplaceAt (CursorOf (*this), element);
}

void
ISequenceAsDilTableImpl::
Reverse ()
{ if (Self::NumberOfElements () > 1) {
    INumber count = Self::NumberOfElements () / 2;
    INumber forward = ivTable.ivFirst,
            backward = ivTable.ivLast;
    if (ivTable.CanAdd (1)) {
      if (forward == backward)
        MakeFree (forward);
      else
        ivTable.DecMod (forward);
      ivTable.ivFirst = forward;
      while (count--) {
        ivTable.PtrAtIndex (forward) = ivTable.PtrAtIndex (backward);
#if ! defined (__INO_MNGCURSOR__)
        UpdateMngCursorsForSwap (forward, backward);
#endif
        SetToNextIndex (forward);
        ivTable.PtrAtIndex (backward) = ivTable.PtrAtIndex (forward);
#if ! defined (__INO_MNGCURSOR__)
        UpdateMngCursorsForSwap (forward, backward);
#endif
        SetToPreviousIndex (backward);
      }
      ivTable.RemoveAtIndex (forward);
    }
    else {
      void* swap;
      while (count--) {
#if ! defined (__INO_MNGCURSOR__)
        UpdateMngCursorsForSwap (forward, backward);
#endif
        swap = ivTable.PtrAtIndex (forward);
        ivTable.PtrAtIndex (forward) = ivTable.PtrAtIndex (backward);
        ivTable.PtrAtIndex (backward) = swap;
        SetToNextIndex (forward);
        SetToPreviousIndex (backward);
      }
    }
  }
#if ! defined (__INO_MNGCURSOR__)
  else {
    // to make all 'in between' mng cursors in valid
    UpdateMngCursorsForSwap (0, 0);
  }
#endif
}

bool
ISequenceAsDilTableImpl::
SetToFirst (ICursorImpl& cursor) const
{ if (! Self::IsEmpty ())
    IndexOf (cursor) = ivTable.ivFirst;
  else
    IndexOf (cursor) = 0;

  return (IndexOf (cursor) != 0);
}

bool
ISequenceAsDilTableImpl::
SetToLast (ICursorImpl& cursor) const
{ if (! Self::IsEmpty ())
    IndexOf (cursor) = ivTable.ivLast;
  else
    IndexOf (cursor) = 0;

  return (IndexOf (cursor) != 0);
}

void
ISequenceAsDilTableImpl::
SetToPosition (IPosition position, ICursorImpl& cursor) const
{ IndexOf (cursor) = ivTable.ivFirst;
  for (IPosition i = 1; i < position; i++)
    SetToNextIndex (IndexOf (cursor));
}

void
ISequenceAsDilTableImpl::
Sort (ICompFunc compFunc, void* addArg)
{ INumber left, last, newLast;
  INumber index;
  void* swap;

  if (Self::NumberOfElements () > 1) {
    last = ivTable.ivLast;
    while (last != ivTable.ivFirst) {
      index = ivTable.ivFirst;
      left = index;
      while (left != last) {
        SetToNextIndex (index);
        if ((*compFunc) (ElementOf (ivTable.ElementAtIndex (index)),
                         ElementOf (ivTable.ElementAtIndex (left)),
                         addArg) < 0) {
#if ! defined (__INO_MNGCURSOR__)
          UpdateMngCursorsForSwap (index, left);
#endif
          swap = ivTable.PtrAtIndex (index);
          ivTable.PtrAtIndex (index) = ivTable.PtrAtIndex (left);
          ivTable.PtrAtIndex (left) = swap;
        }
        newLast = left;
        left = index;
      }
      last = newLast;
    }
  }
}

void
ISequenceAsDilTableImpl::
operator= (ISequenceAsDilTableImpl const& collection)
{ if (this != &collection) {
    Self::RemoveAll ();
    if (! ivTable.CanAdd (collection.ivTable.ivNumberOfElements))
      GrowBy (collection.ivTable.ivNumberOfElements);
    CopyFrom (collection);
#if ! defined (__INO_STREAMING__)
    ivStreamer = collection.ivStreamer;
#endif
  }
}

#if ! defined (__INO_STREAMING__)
void
ISequenceAsDilTableImpl::
StreamIn (IDataStream& fromWhere)
{ ReadVersion (fromWhere);
  Self::RemoveAll ();
  INumber n;
  n <<= fromWhere;
  GrowBy (n);
  while (n--) {
    Self::AddAsLastFromStreamer (fromWhere, CursorOf (*this));
  }
}

void
ISequenceAsDilTableImpl::
StreamOut (IDataStream& toWhere) const
{ WriteVersion (toWhere);
  Self::NumberOfElements () >>= toWhere;
  for (Self::SetToFirst (CursorOf (*this));
       CursorOf (*this).IsValid ();
       Self::SetToNext (CursorOf (*this))) {
    ivStreamer->streamOut (toWhere, Self::ElementAt (CursorOf (*this)));
  }
}

void
ISequenceAsDilTableImpl::
AdoptStreamer (IACollectionStreamer *streamer)
{ if (streamer != ivStreamer.PtrOf())
    ivStreamer = IStreamerPtr(streamer);
}

IStreamerPtr&
ISequenceAsDilTableImpl::
StreamerOf (IACollectionImpl const& collection)
{ return ivStreamer;
}
#endif

// protected members

void
ISequenceAsDilTableImpl::
CopyFrom (ISequenceAsDilTableImpl const& collection)
{ IDilTableImpl const& table = collection.ivTable;
  if (table.ivNumberOfElements != 0) {
    INumber fromIndex, toIndex;
    if (ivTable.ivNumberOfElements != 0 &&
        (ivTable.ivFirst > ivTable.ivLast ?
          (ivTable.ivFirst - ivTable.ivLast - 1) :
          (ivTable.ivFirst + ivTable.ivAllocatedElements
                           - ivTable.ivLast - 1))
         < table.ivNumberOfElements) {
      toIndex = fromIndex = ivTable.ivFirst;
      for (;;) {
        if (fromIndex != toIndex) {
#if ! defined (__INO_MNGCURSOR__)
          UpdateMngCursorsForCopyFrom (fromIndex, toIndex);
#endif
          ivTable.PtrAtIndex (toIndex) = ivTable.PtrAtIndex (fromIndex);
        }
        if (! SetToNextIndex (fromIndex))
          break;
        ivTable.IncMod (toIndex);
      }
      ivTable.ivLast = toIndex;
    }
    toIndex = ivTable.ivLast;
    fromIndex = table.ivFirst;
    for (;;) {
      ivTable.IncMod (toIndex);
      ivTable.GetUnused (toIndex);
      ConstructNode (ivTable.PtrAtIndex (toIndex),
                     ElementOf (collection.ivTable.ElementAtIndex
                                 (fromIndex)));
      if (fromIndex == table.ivLast)
        break;
      do
        table.IncMod (fromIndex);
      while (collection.ivTable.PtrAtIndex (fromIndex) == 0);
    }
    ivTable.ivNumberOfElements += table.ivNumberOfElements;
    ivTable.ivLast = toIndex;
  }
}

void
ISequenceAsDilTableImpl::
GrowBy (INumber n)
{
#if ! defined (__INO_MNGCURSOR__)
  UpdateMngCursorsForGrowBy ();
#endif

  n += ivTable.ivNumberOfElements;

  IDilTableImpl table = ivTable;

  INumber newAllocatedElements = ivTable.ivAllocatedElements;
  while (newAllocatedElements < n)
    newAllocatedElements *= 2;

  Initialize (newAllocatedElements);
  INumber& toIndex   = ivTable.ivLast;
  INumber  fromIndex = table.ivFirst;
  for (INumber i = table.ivNumberOfElements; i != 0; i--) {
    ivTable.IncMod (toIndex);
    ivTable.GetUnused (toIndex);
    ConstructNode (ivTable.PtrAtIndex (toIndex),
                   ElementOf (table.ElementAtIndex (fromIndex)));
    DestroyNode (table.PtrAtIndex (fromIndex));
    do {
      table.IncMod (fromIndex);
    }
    while (table.PtrAtIndex (fromIndex) == 0);
  }
  DeleteNodes (table.ivNodes);
  ivTable.ivNumberOfElements = table.ivNumberOfElements;
}

void
ISequenceAsDilTableImpl::
Initialize (INumber n)
{
  ivTable.ivUnused = new IDilTableImpl::UnusedOffset;  
  typedef  ISequenceAsDilTableNodeImpl<void *> Node;
  Node tmpNode((void *)0);
  (ivTable.ivUnused)->offset = (size_t)(&(tmpNode.ivElement)) - (size_t)(&tmpNode);
  ivTable.ivNodes = CreateNodes (n, ivTable.ivNodeSize);

  ivTable.ivFirst = 1;
  ivTable.ivLast = ivTable.ivAllocatedElements = n;
  ivTable.ivNumberOfElements = 0;

  while (--n) {
    *((void**)ivTable.NodeAtIndex (n) + 1) =
              ivTable.NodeAtIndex (n + 1);
  }
  (ivTable.ivUnused)->unused = ivTable.NodeAtIndex (ivTable.ivFirst);
}

void
ISequenceAsDilTableImpl::
MakeFree (INumber& index)
{ INumber leftFree = index;
  while (leftFree > 1 && ! IsRemoved (leftFree)) {
    if (leftFree == ivTable.ivFirst && leftFree - 1 != ivTable.ivLast)
      ivTable.RemoveAtIndex (leftFree - 1);
    leftFree--;
  }
  if (index - leftFree <= 1 && IsRemoved (leftFree)) {
    index = leftFree;
    return;
  }
  INumber rightFree = index;
  while (rightFree < ivTable.ivAllocatedElements &&
         ! IsRemoved (rightFree)) {
    if (rightFree == ivTable.ivLast && rightFree + 1 != ivTable.ivFirst)
      ivTable.RemoveAtIndex (rightFree + 1);
    rightFree++;
  }
  if (! IsRemoved (leftFree) ||
      (index - leftFree > rightFree - index && IsRemoved (rightFree))) {
    for (INumber i = rightFree,
                 j = rightFree - (index + 1) + 1; j != 0; i--, j--) {
      ivTable.PtrAtIndex (i) = ivTable.PtrAtIndex (i - 1);
#if ! defined (__INO_MNGCURSOR__)
      UpdateMngCursorsForCopyFrom (i - 1, i);
#endif
    }
    if (index <= ivTable.ivFirst && ivTable.ivFirst < rightFree)
      ivTable.ivFirst++;
    if (index <= ivTable.ivLast && ivTable.ivLast < rightFree)
      ivTable.ivLast++;
  }
  else {
    for (INumber i = leftFree, j = (index - 1) - (leftFree + 1) + 1;
         j != 0; i++, j--) {
      ivTable.PtrAtIndex (i) = ivTable.PtrAtIndex (i + 1);
#if ! defined (__INO_MNGCURSOR__)
      UpdateMngCursorsForCopyFrom (i + 1, i);
#endif
    }
    if (leftFree < ivTable.ivFirst && ivTable.ivFirst < index)
      ivTable.ivFirst--;
    if (leftFree < ivTable.ivLast && ivTable.ivLast < index)
      ivTable.ivLast--;
    index--;
  }
}

IPosition
ISequenceAsDilTableImpl::
PositionAtIndex (INumber index) const
{ IPosition position = 1;
  for (INumber currentIndex = ivTable.ivFirst;
       currentIndex != index;
       SetToNextIndex (currentIndex), position++);
  return position;
}

void
ISequenceAsDilTableImpl::
RemoveAtIndex (INumber index)
{
#if ! defined (__INO_MNGCURSOR__)
  UpdateMngCursorsForRemoveOf (index);
#endif
  DestroyNode (ivTable.PtrAtIndex (index));
  ivTable.PutUnused (index);
  if (--ivTable.ivNumberOfElements != 0) {
    if (index == ivTable.ivFirst)
      SetToNextIndex (ivTable.ivFirst);
    else if (index == ivTable.ivLast)
      SetToPreviousIndex (ivTable.ivLast);
  }
  else
    ivTable.DecMod (ivTable.ivLast);
}

void
ISequenceAsDilTableImpl::
SetInBetween (MngCursor& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (! IsInBetween (cursor)) {
    PreviousIndexOf (cursor) = NextIndexOf (cursor) = IndexOf (cursor);
    SetToPreviousIndex (PreviousIndexOf (cursor));
    SetToNextIndex (NextIndexOf (cursor));

    IndexOf (cursor) = MngCursor::kIsInBetween;
  }
#endif
}

bool
ISequenceAsDilTableImpl::
SetToNextIndex (INumber& index) const
{ if (index != ivTable.ivLast) {
    do {
      ivTable.IncMod (index);
    }
    while (IsRemoved (index));
  }
  else
    index = 0;

  return (index != 0);
}

bool
ISequenceAsDilTableImpl::
SetToPreviousIndex (INumber& index) const
{ if (index != ivTable.ivFirst) {
    do {
      ivTable.DecMod (index);
    }
    while (IsRemoved (index));
  }
  else
    index = 0;

  return (index != 0);
}

#if defined (__INO_MNGCURSOR__)
#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (nopar, nouse)
#endif
#endif

void
ISequenceAsDilTableImpl::
UpdateMngCursorsForCopyFrom (INumber fromIndex, INumber toIndex) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (fromIndex != toIndex) {
    forMngCursors (cursor) {
      if (IsInBetween (*cursor)) {
        if (PreviousIndexOf (*cursor) == fromIndex)
          PreviousIndexOf (*cursor) = toIndex;
        else if (NextIndexOf (*cursor) == fromIndex)
          NextIndexOf (*cursor) = toIndex;
      }
      else {
        if (IndexOf (*cursor) == fromIndex)
          IndexOf (*cursor) = toIndex;
      }
    }
  }
#endif
}

void
ISequenceAsDilTableImpl::
UpdateMngCursorsForGrowBy () const
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    if (IsInBetween (*cursor)) {
      if (PreviousIndexOf (*cursor) != 0) {
        PreviousIndexOf (*cursor)
          = PositionAtIndex (PreviousIndexOf (*cursor));
      }
      if (NextIndexOf (*cursor) != 0) {
        NextIndexOf (*cursor)
          = PositionAtIndex (NextIndexOf (*cursor));
      }
    }
    else if (IndexOf (*cursor) != 0) {
      IndexOf (*cursor) = PositionAtIndex (IndexOf (*cursor));
    }
  }
#endif
}

void
ISequenceAsDilTableImpl::
UpdateMngCursorsForRemoveOf (INumber index) const
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    if (IsInBetween (*cursor)) {
      if (PreviousIndexOf (*cursor) == index)
        SetToPreviousIndex (PreviousIndexOf (*cursor));
      else if (NextIndexOf (*cursor) == index)
        SetToNextIndex (NextIndexOf (*cursor));
    }
    else if (IndexOf (*cursor) == index) {
      SetInBetween (*cursor);
    }
  }
#endif
}

void
ISequenceAsDilTableImpl::
UpdateMngCursorsForSwap (INumber index1, INumber index2) const
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    if (IsInBetween (*cursor))
      cursor->Invalidate ();
    else {
      if (IndexOf (*cursor) == index1)
        IndexOf (*cursor) = index2;
      else if (IndexOf (*cursor) == index2)
        IndexOf (*cursor) = index1;
    }
  }
#endif
}

#if defined (__INO_MNGCURSOR__)
#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (restore)
#endif
#endif

#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (restore)
#endif
