// IBM Open Class Library
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// (C) Copyright IBM Corporation, 1992, 1997 All Rights Reserved
//
#include <iikssbst.h>
#include <iiexc.h>

#ifdef IC_PAGETUNE
  #define _IIKSSBST_CPP_
  #include <ipagetun.h>
#endif


#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (nocls, nocnd, nocns, nocnv, noext, nognr, novft)
#endif

#if defined (__SOM_ENABLED__) && defined (__RRBC_LIB__)
#pragma SOMDefine (IAKeySortedSetAsBstTreeOps)
#endif

// --------------------------------
// IKeySortedSetAsBstTreeCursorImpl
// --------------------------------

// public members

IKeySortedSetAsBstTreeCursorImpl::
IKeySortedSetAsBstTreeCursorImpl (IACollectionImpl const& collection)
: ICursorImpl (collection), ivNode (0)
{
}

IKeySortedSetAsBstTreeCursorImpl::
IKeySortedSetAsBstTreeCursorImpl
  (IKeySortedSetAsBstTreeCursorImpl const& cursor)
: ICursorImpl (cursor), ivNode (cursor.ivNode)
{
}

IKeySortedSetAsBstTreeCursorImpl::
~IKeySortedSetAsBstTreeCursorImpl ()
{
}

ICursorImpl*
IKeySortedSetAsBstTreeCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
IKeySortedSetAsBstTreeCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

#if ! defined (__INO_MNGCURSOR__)
// -----------------------------------
// IKeySortedSetAsBstTreeMngCursorImpl
// -----------------------------------

IAKeySortedSetAsBstTreeNodeImpl const*const
IKeySortedSetAsBstTreeMngCursorImpl::kIsInBetween =
   (IAKeySortedSetAsBstTreeNodeImpl*)
   &IKeySortedSetAsBstTreeMngCursorImpl::kIsInBetween;

// public members

IKeySortedSetAsBstTreeMngCursorImpl::
IKeySortedSetAsBstTreeMngCursorImpl
  (IKeySortedSetAsBstTreeImpl const& collection)
: IKeySortedSetAsBstTreeCursorImpl (collection),
  ivMngCollection ((IKeySortedSetAsBstTreeImpl*)&collection)
{ ivNext = ivMngCollection->ivMngCursors->cursor;
  ivMngCollection->ivMngCursors->cursor = this;
}

IKeySortedSetAsBstTreeMngCursorImpl::
~IKeySortedSetAsBstTreeMngCursorImpl ()
{ Self*& first = ivMngCollection->ivMngCursors->cursor;
  if (this == first)
    first = ivNext;
  else {
    // find previous cursor
    for (Self* cursor = first;
         cursor->ivNext != this;
         cursor = cursor->ivNext);

    cursor->ivNext = ivNext;
  }
}

ICursorImpl*
IKeySortedSetAsBstTreeMngCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
IKeySortedSetAsBstTreeMngCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

bool
IKeySortedSetAsBstTreeMngCursorImpl::
IsInBetween () const
{ return ivMngCollection->IsInBetween (*this);
}

bool
IKeySortedSetAsBstTreeMngCursorImpl::
IsMngCursor () const
{ return true;
}

void
IKeySortedSetAsBstTreeMngCursorImpl::
operator= (IKeySortedSetAsBstTreeMngCursorImpl const& cursor)
{ Inherited::operator= (cursor);
  ivNextNode     = cursor.ivNextNode;
  ivPreviousNode = cursor.ivPreviousNode;
  if (ivMngCollection != cursor.ivMngCollection) {

    // detach 'this' from current list of mngcursors
    Self*& first = ivMngCollection->ivMngCursors->cursor;
    if (this == first)
      first = ivNext;
    else {
      // find previous cursor
      for (Self* cursor = first;
           cursor->ivNext != this;
           cursor = cursor->ivNext);

      cursor->ivNext = ivNext;
    }

    // insert 'this' in new list of mngcursors
    ivMngCollection = cursor.ivMngCollection;
    ivNext = (cursor.ivMngCollection)->ivMngCursors->cursor;
    ivMngCollection->ivMngCursors->cursor = this;
  }
}
#endif

// ---------------------------
// IKeySortedSetAsBstTreeImpl
// ---------------------------

char const*
IKeySortedSetAsBstTreeImpl::cvClassName = "KeySortedSetAsBstTree";

// inline private members

inline IKeySortedSetAsBstTreeImpl::Ops&
IKeySortedSetAsBstTreeImpl::
OpsOf (IKeySortedSetAsBstTreeImpl const& collection)
{ return (Ops&) Inherited::OpsOf (collection);
}

inline void*
IKeySortedSetAsBstTreeImpl::
ElementOf (LNode const* lnode) const
{
  size_t offset = 4;
  if (ivMngCursors)
      offset = ivMngCursors->offset;
  if (offset != 4 && offset != 8 && offset != 12 && offset != 16) {
      offset = 4;
  }
  size_t *elementPtr = (size_t *)((size_t)lnode + offset);
  size_t element = *elementPtr;  
  return (OpsOf (*this).ivContainsDTSObjects ?
          (void *)element :
          (void *)elementPtr);
}

inline bool
IKeySortedSetAsBstTreeImpl::
IsInBetween (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((Cursor const&)cursor).ivNode == MngCursor::kIsInBetween;
#else
  return false;
#endif
}

inline IKeySortedSetAsBstTreeImpl::LNode*
IKeySortedSetAsBstTreeImpl::
MaximumLeaf (Node* node)
{ IASSERT (node != 0);

  while (! node->ivIsLeaf) {
    node = ((INode*)node)->ivPtrs [((INode*)node)->ivm - 1];
  }

  return (LNode*)node;
}

inline IKeySortedSetAsBstTreeImpl::LNode*
IKeySortedSetAsBstTreeImpl::
MinimumLeaf (Node* node)
{ IASSERT (node != 0);

  while (! node->ivIsLeaf) {
    node = ((INode*)node)->ivPtrs [0];
  }

  return (LNode*)node;
}

inline IKeySortedSetAsBstTreeImpl::Node*&
IKeySortedSetAsBstTreeImpl::
NextNodeOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivNextNode;
#else
  return ((Cursor&)cursor).ivNode; // just a dummy
#endif
}

inline IKeySortedSetAsBstTreeImpl::Node*&
IKeySortedSetAsBstTreeImpl::
NodeOf (ICursorImpl const& cursor)
{ return ((Cursor&)cursor).ivNode;
}

inline IKeySortedSetAsBstTreeImpl::Node*&
IKeySortedSetAsBstTreeImpl::
PreviousNodeOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivPreviousNode;
#else
  return ((Cursor&)cursor).ivNode; // just a dummy
#endif
}

// inline protected members

inline void
IKeySortedSetAsBstTreeImpl::
Assign (void* element1, void const* element2) const
{ OpsOf (*this).Assign (element1, element2);
}

long
IKeySortedSetAsBstTreeImpl::
CompareKeys (void const* arg1, void const* arg2, IArgType argType) const
{ return OpsOf (*this).CompareKeys (arg1, arg2, argType);
}

inline long
IKeySortedSetAsBstTreeImpl::
CompareKeys (void const* key, INumber index,
             void const* arg, IArgType argType) const
{ return OpsOf (*this).CompareKeys (key, index, arg, argType);
}

inline void
IKeySortedSetAsBstTreeImpl::
CopyKey (void* to, INumber toIndex,
         void const* from, INumber fromIndex) const
{ OpsOf (*this).CopyKey (to, toIndex, from, fromIndex);
}

inline void*
IKeySortedSetAsBstTreeImpl::
CreateINode (void const* inode) const
{ return OpsOf (*this).CreateINode (inode);
}

inline void*
IKeySortedSetAsBstTreeImpl::
CreateKey () const
{ return OpsOf (*this).CreateKey ();
}

inline void*
IKeySortedSetAsBstTreeImpl::
CreateLNode (void const* element) const
{ return OpsOf (*this).CreateLNode (element, ivMngCursors->offset);
}

#if ! defined (__INO_STREAMING__)
inline void*
IKeySortedSetAsBstTreeImpl::
CreateLNodeWithStreamer (IDataStream& stream) const
{ return OpsOf (*this).CreateLNodeWithStreamer (ivStreamer, stream);
}
#endif

inline void
IKeySortedSetAsBstTreeImpl::
DeleteINode (void* inode) const
{ OpsOf (*this).DeleteINode (inode);
}

inline void
IKeySortedSetAsBstTreeImpl::
DeleteKey (void* key) const
{ OpsOf (*this).DeleteKey (key);
}

inline void
IKeySortedSetAsBstTreeImpl::
DeleteLNode (void* lnode) const
{ OpsOf (*this).DeleteLNode (lnode);
}

// inline public members

#define inline

inline bool
IKeySortedSetAsBstTreeImpl::
AllElementsDo (IApplFunc applFunc, void* addArg)
{ if (ivRoot != 0)
    return AllElementsDoSubtree (applFunc, addArg, ivRoot);
  else
    return true;
}

inline void*
IKeySortedSetAsBstTreeImpl::
Any () const
{ return First ();
}

inline bool
IKeySortedSetAsBstTreeImpl::
CheckReplacement (ICursorImpl const& cursor, void const* element) const
{ return (CompareKeys (Self::ElementAt (cursor),
                       element, kElementElement) == 0);
}

inline char const*
IKeySortedSetAsBstTreeImpl::
ClassName () const
{ return cvClassName;
}

inline bool
IKeySortedSetAsBstTreeImpl::
ContainsElementWithKey (void const* key) const
{ return Self::LocateElementWithKey (key, CursorOf (*this));
}

inline void*
IKeySortedSetAsBstTreeImpl::
ElementAt (ICursorImpl const& cursor) const
{ Node const* node = NodeOf (cursor);
  IASSERT (node->ivIsLeaf);
  return ElementOf ((LNode const*)node);
}

inline void*
IKeySortedSetAsBstTreeImpl::
ElementAtPosition (IPosition position) const
{ Self::SetToPosition (position, CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

inline void*
IKeySortedSetAsBstTreeImpl::
First () const
{ Self::SetToFirst (CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

inline bool
IKeySortedSetAsBstTreeImpl::
IsBounded () const
{ return false;
}

inline bool
IKeySortedSetAsBstTreeImpl::
IsEmpty () const
{ return (ivRoot == 0);
}

inline bool
IKeySortedSetAsBstTreeImpl::
IsFirstAt (ICursorImpl const& cursor) const
{ Self::SetToFirst (CursorOf (*this));
  return (NodeOf (cursor) == NodeOf (CursorOf (*this)));
}

inline bool
IKeySortedSetAsBstTreeImpl::
IsFull () const
{ return false;
}

inline bool
IKeySortedSetAsBstTreeImpl::
IsLastAt (ICursorImpl const& cursor) const
{ Self::SetToLast (CursorOf (*this));
  return (NodeOf (cursor) == NodeOf (CursorOf (*this)));
}

inline void*
IKeySortedSetAsBstTreeImpl::
Key (void const* element) const
{ return OpsOf (*this).Key (element);
}

inline void*
IKeySortedSetAsBstTreeImpl::
Last () const
{ Self::SetToLast (CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

inline bool
IKeySortedSetAsBstTreeImpl::
LocateOrAddElementWithKey (void const* element, ICursorImpl& cursor)
{ return ! Self::Add (element, cursor);
}

inline INumber
IKeySortedSetAsBstTreeImpl::
NumberOfElements () const
{ return ivNumberOfElements;
}

inline void
IKeySortedSetAsBstTreeImpl::
RemoveAtPosition (IPosition position)
{ Self::SetToPosition (position, CursorOf (*this));
  Self::RemoveAt (CursorOf (*this));
}

inline void
IKeySortedSetAsBstTreeImpl::
RemoveFirst ()
{ Self::SetToFirst (CursorOf (*this));
  Self::RemoveAt (CursorOf (*this));
}

inline void
IKeySortedSetAsBstTreeImpl::
RemoveLast ()
{ Self::SetToLast (CursorOf (*this));
  Self::RemoveAt (CursorOf (*this));
}

inline void
IKeySortedSetAsBstTreeImpl::
ReplaceAt (ICursorImpl const& cursor, void const* element)
{ Node const* node = NodeOf (cursor);
  IASSERT (node->ivIsLeaf);
  Assign (ElementOf ((LNode const*)node), element);
}

#undef inline

// public members

IKeySortedSetAsBstTreeImpl::
IKeySortedSetAsBstTreeImpl (Ops& ops, INumber)
: IAKeySortedSetImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivTreeOrder (ops),
  ivRoot (0), ivHeight (0), ivNumberOfElements (0)
{
  ivMngCursors = new MngOffsetCursor;
  ivMngCursors->cursor = 0;  
  typedef IKeySortedSetAsBstTreeLNodeImpl<long> TypedLNode;
  TypedLNode tmpNode(0);
  ivMngCursors->offset = (size_t)(&(tmpNode.ivElement)) - (size_t)(&tmpNode);
}

IKeySortedSetAsBstTreeImpl::
IKeySortedSetAsBstTreeImpl
  (Ops& ops, IKeySortedSetAsBstTreeImpl const& collection)
: IAKeySortedSetImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivTreeOrder (collection.ivTreeOrder),
  ivRoot (0), ivHeight (0), ivNumberOfElements (0)
{
  ivMngCursors = new MngOffsetCursor;
  ivMngCursors->cursor = 0;  
  ivMngCursors->offset = (collection.ivMngCursors)->offset;
  operator= (collection);
}

IKeySortedSetAsBstTreeImpl::
~IKeySortedSetAsBstTreeImpl ()
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    cursor->Invalidate ();
  }
#endif
  Self::RemoveAll ();
  delete ivMngCursors;
}

bool
IKeySortedSetAsBstTreeImpl::
Add (void const* element, ICursorImpl& cursor)
{ bool hasBeenAdded;
 
  if (Self::IsEmpty ()) {
    NodeOf (cursor) = ivRoot = (Node*) CreateLNode (element);
    ivHeight = 1;
    ivNumberOfElements = 1;
    hasBeenAdded = true;
  }
  else {
    Node* node = 0;
    void* key = CheckPointer (CreateKey ());

    ITRY {
      hasBeenAdded = AddSubtree (ivRoot, element, node, key, cursor);
      if (hasBeenAdded)
        ivNumberOfElements++;
    }
    IENDTRY

    ICATCH (IOutOfCollectionMemoryImpl exc) {
      if (node != 0)
        ivNumberOfElements -= DeleteSubtree (node);
      DeleteKey (key);
      IRETHROW (exc);
    }
    IENDCATCH

    if (node != 0) {
      //
      // the root has splitted, create a new root
      //
      INode* inode;

      ITRY {
        inode = (INode*) CreateINode ();
      }
      IENDTRY

      ICATCH (IOutOfCollectionMemoryImpl exc) {
        ivNumberOfElements -= DeleteSubtree (node);
        DeleteKey (key);
        IRETHROW (exc);
      }
      IENDCATCH

      if (ivRoot->ivIsLeaf &&
          CompareKeys
            (ElementOf ((LNode*)ivRoot), key, kElementKey) > 0) {
          //
          // added key is smaller than already present key
          //
          CopyKey (key, 0, Key (ElementOf ((LNode*)ivRoot)), 0);
          CopyKey (inode->ivKeys, 0, key, 0);
          inode->ivPtrs [0] = node;
          inode->ivPtrs [1] = ivRoot;
      }
      else {
        //
        // root is an interior node or
        // added key is the largest in new node
        // this means that the new node is always larger
        // than the key added
        //
        CopyKey (inode->ivKeys, 0, key, 0);
        inode->ivPtrs [0] = ivRoot;
        inode->ivPtrs [1] = node;
      }

      inode->ivm = 2;
      ivRoot = inode;
      ivHeight++;
    }
    DeleteKey (key);
  }

  return hasBeenAdded;
}

void
IKeySortedSetAsBstTreeImpl::
AddAllFrom (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ()) {
    Self const& kss = (Self const&)collection;
    if (kss.ivRoot != 0) {
      AddAllFromSubtree (kss.ivRoot);
    }
  }
  else
    Inherited::AddAllFrom (collection);
}

bool
IKeySortedSetAsBstTreeImpl::
AddOrReplaceElementWithKey (void const* element, ICursorImpl& cursor)
{ bool hasBeenAdded = Self::Add (element, cursor);

  if (! hasBeenAdded)
    Self::ReplaceAt (cursor, element);

  return hasBeenAdded;
}

bool
IKeySortedSetAsBstTreeImpl::
CheckCursor (ICursorImpl const& cursor) const
{ if (IsInBetween (cursor) || (NodeOf (cursor) == 0))
    return true;
  else if (ivRoot != 0)
    return CheckCursorSubtree (cursor, ivRoot);
  else
    return false;
}

IACollectionImpl*
IKeySortedSetAsBstTreeImpl::
Clone () const
{ return (IACollectionImpl*) CheckPointer (OpsOf (*this).Clone ());
}

bool
IKeySortedSetAsBstTreeImpl::
ContainsAllKeysFrom (IACollectionImpl const& collection) const
{ if (Self::ClassName () == collection.ClassName ()) {
    bool result = true;
    if (this != &collection) {
      Self const& kss = (Self const&)collection;
      if (kss.ivRoot != 0)
        result = ContainsAllKeysFromSubtree (kss.ivRoot);
    }
    return result;
  }
  else
    return Inherited::ContainsAllKeysFrom (collection);
}

void
IKeySortedSetAsBstTreeImpl::
Copy (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ())
    operator= ((Self const&)collection);
  else
    Inherited::Copy (collection);
}

IKeySortedSetAsBstTreeImpl::Cursor*
IKeySortedSetAsBstTreeImpl::
CreateCursor () const
{ return (Cursor*) CheckPointer (new Cursor (*this));
}

IKeySortedSetAsBstTreeImpl::MngCursor*
IKeySortedSetAsBstTreeImpl::
CreateMngCursor () const
{ return (MngCursor*) CheckPointer (new MngCursor (*this));
}

void*
IKeySortedSetAsBstTreeImpl::
ElementWithKey (void const* key) const
{ ICursorImpl& cursor = CursorOf (*this);
  ICHECK_ContainsKey (ClassName (), "ElementWithKey")
  return Self::ElementAt (cursor);
}

bool
IKeySortedSetAsBstTreeImpl::
IsConsistent () const
{ bool result = true;

  if (ivRoot != 0)
    result = IsConsistentSubtree (ivRoot);

  return result;
}

bool
IKeySortedSetAsBstTreeImpl::
LocateElementWithKey (void const* key, ICursorImpl& cursor) const
{ bool result = false;

  if (ivRoot == 0)
    cursor.Invalidate ();
  else {
    result = LocateElementWithKeySubtree (key, cursor, ivRoot);
  }

  return result;
}

bool
IKeySortedSetAsBstTreeImpl::
LocateNextElementWithKey (void const* key, ICursorImpl& cursor) const
{ bool result = false;

  if (CompareKeys (Self::ElementAt (cursor), key, kElementKey) >= 0)
    cursor.Invalidate ();
  else
    result = Self::LocateElementWithKey (key, cursor);

  return result;
}

IPosition
IKeySortedSetAsBstTreeImpl::
PositionAt (ICursorImpl const& cursor) const
{ IPosition position = 1;

  for (Self::SetToFirst (CursorOf (*this));
       NodeOf (CursorOf (*this)) != NodeOf (cursor);
       Self::SetToNext (CursorOf (*this)), position++);

  return position;
}

INumber
IKeySortedSetAsBstTreeImpl::
RemoveAll ()
{ INumber n = Self::NumberOfElements ();
  if (! Self::IsEmpty ()) {
    DeleteSubtree (ivRoot);
    ivRoot = 0;
    ivHeight = 0;
    ivNumberOfElements = 0;
  }

#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    if (cursor->IsValid ()) {
      NodeOf (*cursor)         = (Node*) MngCursor::kIsInBetween;
      NextNodeOf (*cursor)     = 0;
      PreviousNodeOf (*cursor) = 0;
    }
  }
#endif

  return n;
}

INumber
IKeySortedSetAsBstTreeImpl::
RemoveAll (IPredFunc predFunc, void* addArg)
{ Cursor current (*this), next (*this);
  INumber removed = 0;

  for (Self::SetToFirst (current), Self::SetToFirst (next);
       current.IsValid ();
       current = next) {
    Self::SetToNext (next);
    if ((*predFunc) (Self::ElementAt (current), addArg)) {
      Self::RemoveAt (current);
      removed++;
    }
  }

  return removed;
}

void
IKeySortedSetAsBstTreeImpl::
RemoveAt (ICursorImpl& cursor)
{ Cursor backupCursor = (Cursor&)cursor;

#if ! defined (__INO_MNGCURSOR__)
  Node* removedNode = NodeOf (cursor);
  forMngCursors (cursor2) {
    if (IsInBetween (*cursor2)) {
      if (removedNode == NextNodeOf (*cursor2)) {
        NodeOf (*cursor2) = (Node*)removedNode;
        SetToNext (*cursor2);
        NextNodeOf(*cursor2) = NodeOf (*cursor2);
        NodeOf (*cursor2) = (Node*) MngCursor::kIsInBetween;
      }
      else if (removedNode == PreviousNodeOf (*cursor2)) {
        NodeOf (*cursor2) = (Node*)removedNode;
        SetToPrevious (*cursor2);
        PreviousNodeOf(*cursor2) = NodeOf (*cursor2);
        NodeOf (*cursor2) = (Node*) MngCursor::kIsInBetween;
      }
    }
    else if (removedNode == NodeOf (*cursor2)) {
      SetInBetween (*cursor2);
    }
  }
  NodeOf (backupCursor) = removedNode;
#endif

  if (ivRoot->ivIsLeaf) {
    IASSERT (NodeOf (backupCursor) == ivRoot);

    DeleteLNode ((LNode*)ivRoot);
    ivRoot = 0;
    ivHeight = 0;
  }
  else {
    //
    // the root points to an interior node
    //
    INode* root = (INode*)ivRoot;

    RemoveSubtree (root, backupCursor);

    if (root->ivm < 2) {
      //
      // root has become obsolete
      // root must have at least two children
      //
      Node* tmpNode = root->ivPtrs [0];
      DeleteINode (root);
      ivHeight--;
      ivRoot = tmpNode;
    }
  }
  ivNumberOfElements--;

  if (! IsInBetween (cursor))
    cursor.Invalidate ();
}

bool
IKeySortedSetAsBstTreeImpl::
RemoveElementWithKey (void const* key)
{ bool isContained =
    Self::LocateElementWithKey (key, CursorOf (*this));

  if (isContained)
    Self::RemoveAt (CursorOf (*this));

  return isContained;
}

bool
IKeySortedSetAsBstTreeImpl::
ReplaceElementWithKey (void const* element, ICursorImpl& cursor)
{ bool isContained =
    Self::LocateElementWithKey (Key (element), cursor);

  if (isContained)
    Self::ReplaceAt (cursor, element);

  return isContained;
}

bool
IKeySortedSetAsBstTreeImpl::
SetToFirst (ICursorImpl& cursor) const
{ bool hasFirst = ! Self::IsEmpty ();

  if (! hasFirst)
    cursor.Invalidate ();
  else
    NodeOf (cursor) = MinimumLeaf (ivRoot);

  return hasFirst;
}

bool
IKeySortedSetAsBstTreeImpl::
SetToLast (ICursorImpl& cursor) const
{ bool hasLast = ! Self::IsEmpty ();

  if (! hasLast)
    cursor.Invalidate ();
  else
    NodeOf (cursor) = MaximumLeaf (ivRoot);

  return hasLast;
}

bool
IKeySortedSetAsBstTreeImpl::
SetToNext (ICursorImpl& cursor) const
{ bool hasNext = false;

#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    hasNext = ((NodeOf (cursor) = NextNodeOf (cursor)) != 0);
  }
  else
#endif
  { if (ivRoot->ivIsLeaf) {
      cursor.Invalidate ();
    }
    else {
      hasNext = SetToNextSubtree (cursor, (INode const*)ivRoot);
      if (! hasNext) {
        cursor.Invalidate ();
      }
    }
  }

  return hasNext;
}

void
IKeySortedSetAsBstTreeImpl::
SetToPosition (IPosition position, ICursorImpl& cursor) const
{ if (position - 1 < ivNumberOfElements - position) {
    //
    // better to step forward
    //
    Self::SetToFirst (cursor);
    IASSERT (cursor.IsValid ());
    for (IPosition i = 1; i < position; i++) {
      Self::SetToNext (cursor);
      IASSERT (cursor.IsValid ());
    }
  }
  else {
    //
    // better to step backward
    //
    Self::SetToLast (cursor);
    IASSERT (cursor.IsValid ());
    for (IPosition i = ivNumberOfElements; i > position; i--) {
      Self::SetToPrevious (cursor);
      IASSERT (cursor.IsValid ());
    }
  }
}

bool
IKeySortedSetAsBstTreeImpl::
SetToPrevious (ICursorImpl& cursor) const
{ bool hasPrevious = false;

#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    hasPrevious = ((NodeOf (cursor) = PreviousNodeOf (cursor)) != 0);
  }
  else
#endif
  { if (ivRoot->ivIsLeaf) {
      cursor.Invalidate ();
    }
    else {
      hasPrevious = SetToPreviousSubtree (cursor, (INode const*)ivRoot);
      if (! hasPrevious) {
        cursor.Invalidate ();
      }
    }
  }

  return hasPrevious;
}

void
IKeySortedSetAsBstTreeImpl::
operator= (IKeySortedSetAsBstTreeImpl const& collection)
{ if (this != &collection) {
    Self::RemoveAll ();
    ivMngCursors->offset = (collection.ivMngCursors)->offset;
    if (collection.ivRoot != 0) {
      ivRoot = CopySubtree (collection.ivRoot);
      ivHeight = collection.ivHeight;
      ivNumberOfElements = collection.ivNumberOfElements;
    }
#if ! defined (__INO_STREAMING__)
    ivStreamer = collection.ivStreamer;
#endif
  }
}

#if ! defined (__INO_STREAMING__)
void
IKeySortedSetAsBstTreeImpl::
StreamIn (IDataStream& fromWhere)
{ ReadVersion (fromWhere);
  Self::RemoveAll ();
  INumber n;
  n <<= fromWhere;
  while (n--) {
    Node *node = (Node*) CreateLNodeWithStreamer (fromWhere);
    Self::Add (ElementOf((LNode*)node), CursorOf (*this));
    delete node;
  }
}

void
IKeySortedSetAsBstTreeImpl::
StreamOut (IDataStream& toWhere) const
{ WriteVersion (toWhere);
  Self::NumberOfElements () >>= toWhere;
  for (Self::SetToFirst (CursorOf (*this));
       CursorOf (*this).IsValid ();
       Self::SetToNext (CursorOf (*this))) {
    ivStreamer->streamOut (toWhere, Self::ElementAt (CursorOf (*this)));
  }
}

void
IKeySortedSetAsBstTreeImpl::
AdoptStreamer (IACollectionStreamer *streamer)
{ if (streamer != ivStreamer.PtrOf())
    ivStreamer = IStreamerPtr(streamer);
}

IStreamerPtr&
IKeySortedSetAsBstTreeImpl::
StreamerOf (IACollectionImpl const& collection)
{ return ivStreamer;
}
#endif

// private members

bool
IKeySortedSetAsBstTreeImpl::
AddSubtree (Node* page, void const* element, Node*& node,
            void* key, ICursorImpl& cursor)
{ bool hasBeenAdded = false;

  if (page->ivIsLeaf) {
    LNode* lpage = (LNode*)page;

    if (CompareKeys
          (ElementOf (lpage), element, kElementElement) == 0) {
      //
      // element already in tree
      //
      NodeOf (cursor) = lpage;
      node = 0;
    }
    else {
      NodeOf (cursor) = node = (Node*) CreateLNode (element);
      CopyKey (key, 0, Key (element), 0);
      hasBeenAdded = true;
    }
  }
  else {
    INode* ipage = (INode*)page;

    INumber index = BinarySearchInPageWithElement (ipage, element);
    hasBeenAdded =
      AddSubtree (ipage->ivPtrs [index], element, node, key, cursor);

    if (node != 0) {
      //
      // node has to be inserted
      //
      INumber i;
      void* tmpKey = CheckPointer (CreateKey ());

      if (ipage->ivm < ivTreeOrder) {
        //
        // node can be inserted in this page, move elements
        //
        for (i = ipage->ivm - 1; i >= index + 1; i--) {
          ipage->ivPtrs [i + 1] = ipage->ivPtrs [i];
          CopyKey (ipage->ivKeys, i, ipage->ivKeys, i - 1);
        }

        //
        // room available in key[index] and ptr[index+1]
        // if index is 0 keys have to be compared
        //
        if (index == 0 || ipage->ivPtrs [index]->ivIsLeaf)
          CopyKey (tmpKey, 0,
            Key (ElementOf (MinimumLeaf (ipage->ivPtrs [index]))), 0);

        if ((index == 0 || ipage->ivPtrs [index]->ivIsLeaf) &&
            CompareKeys (key, tmpKey, kKeyKey) < 0) {
          ipage->ivPtrs [index + 1] = ipage->ivPtrs[index];
          ipage->ivPtrs [index] = node;
          CopyKey (ipage->ivKeys, index, tmpKey, 0);
        }
        else {
          ipage->ivPtrs [index + 1] = node;
          CopyKey (ipage->ivKeys, index, key, 0);
        }

        ipage->ivm += 1;
        node = 0;
      }
      else {
        //
        // overflow
        // redistribute elements between current and new page
        // determine a new key value
        // first create temp storage
        // this storage is allocated as an internal node
        // using its overloaded new operator
        // note that this internal node provides storage for one
        // extra pointer and one extra key
        // we also need some temp storage for a key and for a node
        // pointer. This is because we want to insert a key and a
        // pointer and we already have a full page (inPage)
        // Just allocating an interior node as temporary storage
        // is therefore insufficient.
        // The key storage is still available as tmpKey
        // So only create a tmp node ptr
        //
        INode* inode;

        ITRY {
          inode = (INode*) CreateINode ();
        }
        IENDTRY

        ICATCH (IOutOfCollectionMemoryImpl exc) {
          DeleteKey (tmpKey);
          IRETHROW (exc);
        }
        IENDCATCH

        INode* tmpStore;

        ITRY {
          tmpStore = (INode*) CreateINode ();
        }
        IENDTRY

        ICATCH (IOutOfCollectionMemoryImpl exc) {
          DeleteKey (tmpKey);
          DeleteINode (inode);
          IRETHROW (exc);
        }
        IENDCATCH

        Node* tmpNode;

        //
        // put the first part of ipage in the temporary storage
        //
        for (i = 0; i < index; i++) {
          CopyKey (tmpStore->ivKeys, i, ipage->ivKeys, i);
          tmpStore->ivPtrs [i] = ipage->ivPtrs [i];
        }

        if (index == 0 || ipage->ivPtrs [index]->ivIsLeaf)
          CopyKey (tmpKey, 0,
            Key (ElementOf (MinimumLeaf (ipage->ivPtrs [index]))), 0);

        if ((index == 0 || ipage->ivPtrs [index]->ivIsLeaf) &&
            CompareKeys (key, tmpKey, kKeyKey) < 0) {
          tmpStore->ivPtrs [index] = node;
          if (index == ivTreeOrder - 1) {
            //
            // remember node and key, they don't fit in tmpstore
            //
            tmpNode = ipage->ivPtrs [index];
          }
          else {
            tmpStore->ivPtrs [index + 1] = ipage->ivPtrs [index];
            CopyKey (tmpStore->ivKeys, index, tmpKey, 0);
          }
        }
        else {
          tmpStore->ivPtrs [index] = ipage->ivPtrs [index];
          if (index == ivTreeOrder - 1) {
            //
            // remember node and key, they don't fit in tmpstore
            //
            tmpNode = node;
            CopyKey (tmpKey, 0, key, 0);
          }
          else {
            tmpStore->ivPtrs [index + 1] = node;
            CopyKey (tmpStore->ivKeys, index, key, 0);
          }
        }

        //
        // put the other part of inPage in the temporary storage
        //
        for (i = index + 1; i < ivTreeOrder - 1; i++) {
          CopyKey (tmpStore->ivKeys, i, ipage->ivKeys, i-1);
          tmpStore->ivPtrs [i + 1] = ipage->ivPtrs [i];
        }

        //
        // Store the last key and the last pointer of inPage
        // in the temp pointer and the temp key, because
        // they do not fit in tmpStore if this is not already done
        //
        if (index != ivTreeOrder - 1) {
          tmpNode = ipage->ivPtrs [ivTreeOrder - 1];
          CopyKey (tmpKey, 0, ipage->ivKeys, ivTreeOrder - 2);
        }

        //
        // now determine new key, and split
        //
        INumber keyindex = ((ivTreeOrder + 1) / 2) - 1;

        //
        // Put the first half of the temp storage back in inPage
        //
        for (i = 0 ; i < keyindex; i++) {
          CopyKey (ipage->ivKeys, i, tmpStore->ivKeys, i);
          ipage->ivPtrs [i] = tmpStore->ivPtrs [i];
        }
        ipage->ivPtrs [keyindex] = tmpStore->ivPtrs [keyindex];
        ipage->ivm = keyindex + 1;

        //
        // Store the key that will be used in the previous level
        // of recursion to insert in the parent inode
        //
        CopyKey (key, 0, tmpStore->ivKeys, keyindex);

        //
        // Put the latter half of the temp storage in inode
        //
        for (i = keyindex + 1; i < ivTreeOrder - 1; i++) {
          CopyKey (inode->ivKeys, i - keyindex - 1,
                   tmpStore->ivKeys, i);
          inode->ivPtrs [i - keyindex - 1] = tmpStore->ivPtrs [i];
        }
        CopyKey (inode->ivKeys, ivTreeOrder - keyindex - 2, tmpKey, 0);
        inode->ivPtrs [ivTreeOrder - keyindex - 2] =
          tmpStore->ivPtrs [ivTreeOrder - 1];
        inode->ivPtrs [ivTreeOrder - keyindex - 1] = tmpNode;
        inode->ivm = ivTreeOrder - keyindex;

        node = inode;

        //
        // remove the temporary storage
        //
        DeleteINode (tmpStore);
      }
      //
      // remove temporary key
      //
      DeleteKey (tmpKey);
    }
  }

  return hasBeenAdded;
}

void
IKeySortedSetAsBstTreeImpl::
AddAllFromSubtree (Node const* current)
{ IASSERT (current != 0);

  if (current->ivIsLeaf) {
    Self::Add (ElementOf ((LNode const*) current), CursorOf (*this));
  }
  else {
    INode const* inode = (INode const*)current;

    for (INumber i = 0; i < inode->ivm; i++) {
      AddAllFromSubtree (inode->ivPtrs [i]);
    }
  }
}

bool
IKeySortedSetAsBstTreeImpl::
AllElementsDoSubtree (IApplFunc applFunc, void* addArg, Node* current)
{ IASSERT (current != 0);

  bool result = true;

  if (current->ivIsLeaf) {
    result = (*applFunc) (ElementOf ((LNode const*)current), addArg);
  }
  else {
    INode* inode = (INode*)current;
    for (INumber i = 0; i < inode->ivm && result; i++) {
      result = AllElementsDoSubtree
        (applFunc, addArg, inode->ivPtrs [i]);
    }
  }

  return result;
}

INumber
IKeySortedSetAsBstTreeImpl::
BinarySearchInPageWithKey (INode const* inode, void const* key) const
{ INumber startIndex = 0,
          between,
          endIndex = inode->ivm - 1;

  long comp;
  while (startIndex < endIndex) {
    between = (startIndex + endIndex) / 2;
    comp = CompareKeys (inode->ivKeys, between, key, kKey);
    if (comp < 0) {
      startIndex = between + 1;
    }
    else if (comp > 0) {
      endIndex = between;
    }
    else {
      return between + 1;
    }
  }

  return endIndex;
}

INumber
IKeySortedSetAsBstTreeImpl::
BinarySearchInPageWithElement
  (INode const* inode, void const* element) const
{ INumber startIndex = 0,
          between,
          endIndex = inode->ivm - 1;

  long comp = 0;
  while (startIndex < endIndex) {
    between = (startIndex + endIndex) / 2;
    comp = CompareKeys (inode->ivKeys, between, element, kElement);
    if (comp < 0) {
      startIndex = between + 1;
    }
    else if (comp > 0) {
      endIndex = between;
    }
    else {
      return between + 1;
    }
  }

  return endIndex;
}

bool
IKeySortedSetAsBstTreeImpl::
CheckCursorSubtree
  (ICursorImpl const& cursor, Node const* current) const
{ if (current->ivIsLeaf) {
    return (NodeOf (cursor) == current);
  }
  else {
    INode const* inode = (INode const*)current;

    for (INumber i = 0; i < inode->ivm; i++) {
      if (CheckCursorSubtree (cursor, inode->ivPtrs [i]))
        return true;
    }
    return false;
  }
}

bool
IKeySortedSetAsBstTreeImpl::
ContainsAllKeysFromSubtree (Node const* current) const
{ bool result = true;

  if (current->ivIsLeaf) {
    result = Self::LocateElementWithKey
      (Key (ElementOf ((LNode const*)current)), CursorOf (*this));
  }
  else {
    INode const* inode = (INode const*)current;

    for (INumber i = 0; i < inode->ivm && result; i++) {
      result = ContainsAllKeysFromSubtree (inode->ivPtrs [i]);
    }
  }

  return result;
}

IKeySortedSetAsBstTreeImpl::Node*
IKeySortedSetAsBstTreeImpl::
CopySubtree (Node const* node) const
{ IASSERT (node != 0);

  Node* newNode;

  if (node->ivIsLeaf) {
    newNode = (INode*) CreateLNode (ElementOf ((LNode const*)node));
  }
  else {
    INode const* inode = (INode const*)node;
    INode* newINode = (INode*) CreateINode (inode);

    for (INumber i = 0; i < inode->ivm; i++) {
      ITRY {
        newINode->ivPtrs [i] = CopySubtree (inode->ivPtrs [i]);
      }
      IENDTRY

      ICATCH (IOutOfCollectionMemoryImpl exc) {
        newINode->ivm = i;
        DeleteSubtree (newINode);
        IRETHROW (exc);
      }
      IENDCATCH
    }

    newNode = newINode;
  }

  return newNode;
}

INumber
IKeySortedSetAsBstTreeImpl::
DeleteSubtree (Node* current) const
{ IASSERT (current != 0);

  INumber removed = 0;

  if (current->ivIsLeaf) {
    DeleteLNode (current);
    removed = 1;
  }
  else {
    INode* inode = (INode*)current;

    removed = 0;
    for (INumber i = 0; i < inode->ivm; i++) {
      removed += DeleteSubtree (inode->ivPtrs [i]);
    }

    DeleteINode (inode);
  }

  return removed;
}

bool
IKeySortedSetAsBstTreeImpl::
IsConsistentSubtree (Node const* current) const
{ bool result = true;

  if (! current->ivIsLeaf) {
    INode const* inode = (INode const*)current;
    for (INumber i = 0; result && i < inode->ivm; i++)
      if (inode->ivPtrs [i] != 0)
        result = IsConsistentSubtree (inode->ivPtrs [i]);
      else
        result = false;
  }

  return result;
}

void
IKeySortedSetAsBstTreeImpl::
JoinINodes (INode* leftINode, INode* rightINode, INode* parentINode,
            INumber indexOfLeftINode)
{ CopyKey (leftINode->ivKeys, leftINode->ivm - 1,
           parentINode->ivKeys, indexOfLeftINode);
  leftINode->ivPtrs [leftINode->ivm] = rightINode->ivPtrs [0];
  leftINode->ivm += 1;

  INumber i;
  for (i = 0; i < rightINode->ivm - 1; i++) {
    CopyKey (leftINode->ivKeys, leftINode->ivm - 1,
             rightINode->ivKeys, i);
    leftINode->ivPtrs [leftINode->ivm] = rightINode->ivPtrs [i + 1];
    leftINode->ivm++;
  }

  //
  // adjust parent node
  //
  for (i = indexOfLeftINode; i < parentINode->ivm - 2; i++) {
    CopyKey (parentINode->ivKeys, i,
             parentINode->ivKeys, i + 1);
    parentINode->ivPtrs [i + 1] = parentINode->ivPtrs [i + 2];
  }
  parentINode->ivm--;
}

bool
IKeySortedSetAsBstTreeImpl::
LocateElementWithKeySubtree
(void const* key, ICursorImpl& cursor, Node* node) const
{ bool isContained;

  if (node->ivIsLeaf) {
    LNode* lnode = (LNode*)node;

    isContained =
      (CompareKeys (ElementOf (lnode), key, kElementKey) == 0);
    if (isContained) {
      NodeOf (cursor) = lnode;
    }
    else {
      cursor.Invalidate ();
    }
  }
  else {
    INode* inode = (INode*)node;

    INumber index = BinarySearchInPageWithKey (inode, key);
    isContained =
      LocateElementWithKeySubtree (key, cursor, inode->ivPtrs [index]);
  }

  return isContained;
}

void
IKeySortedSetAsBstTreeImpl::
RemoveSubtree (INode* inode, ICursorImpl& cursor)
{ INumber index =
    BinarySearchInPageWithElement (inode, Self::ElementAt (cursor));

  INumber i;
  if (inode->ivPtrs [index] == NodeOf (cursor)) {
    DeleteLNode ((LNode*)inode->ivPtrs [index]);

    //
    // shift elements to the left
    //
    for (i = index; i < inode->ivm - 1; i++) {
      inode->ivPtrs [i] = inode->ivPtrs [i + 1];
    }

    for (i = index; i < inode->ivm - 2; i++) {
      CopyKey (inode->ivKeys, i, inode->ivKeys, i + 1);
    }

    inode->ivm--;
  }
  else {
    //
    // node is at level two or higher
    //

    IASSERT (! inode->ivPtrs [index]->ivIsLeaf);

    INode* child = (INode*)inode->ivPtrs [index];
    RemoveSubtree (child, cursor);

    if (child->ivm < (ivTreeOrder + 1) / 2) {
      //
      // child is underflown
      //
      INode* sibling;

      if (index != 0) {
        //
        // it has a left sibling
        //
        sibling = (INode*)inode->ivPtrs [index - 1];

        if (sibling->ivm == (ivTreeOrder + 1) / 2) {
          //
          // child can be moved into sibling
          //
          JoinINodes (sibling, child, inode, index - 1);
          DeleteINode (child);
        }
        else {
          //
          // sibling has > [m/2] sons, move some elements of sibling
          // in child, determine the number of movements
          //
          INumber noMoves =
            ((sibling->ivm + child->ivm) / 2) - child->ivm;
          IASSERT (noMoves != 0);

          //
          // make room in child
          //
          for (i = child->ivm - 1; i != 0; i--) {
            CopyKey (child->ivKeys, i + noMoves - 1,
                     child->ivKeys, i - 1);
            child->ivPtrs [i + noMoves] = child->ivPtrs [i];
          }
          child->ivPtrs [noMoves] = child->ivPtrs [0];

          //
          // adjust parent
          //
          CopyKey (child->ivKeys, noMoves - 1,
                   inode->ivKeys, index - 1);
          CopyKey (inode->ivKeys, index - 1,
                   sibling->ivKeys, sibling->ivm - noMoves - 1);

          //
          // now move elements of sibling to child
          //
          child->ivPtrs [0] = sibling->ivPtrs [sibling->ivm - noMoves];
          for (i = 1; i < noMoves; i++) {
            child->ivPtrs [i] =
              sibling->ivPtrs [sibling->ivm - noMoves + i];
            CopyKey (child->ivKeys, i - 1,
                     sibling->ivKeys, sibling->ivm - noMoves + i - 1);
          }

          child->ivm += noMoves;
          sibling->ivm -= noMoves;
        }
      }
      else {
        //
        // we must look at the right sibling
        //
        sibling = (INode*)inode->ivPtrs [index + 1];
        if (sibling->ivm == (ivTreeOrder + 1) / 2) {
          //
          // move sibling into child
          //
          JoinINodes (child, sibling, inode, index);
          DeleteINode (sibling);
        }
        else {
          //
          // sibling has > [m/2] sons, move some elements of sibling
          // in child, determine the number of movements
          //
          INumber noMoves;

          noMoves = ((sibling->ivm + child->ivm) / 2) - child->ivm;
          IASSERT (noMoves > 0);

          //
          // move elements of sibling to child
          //
          for (i = 0; i < noMoves - 1; i++) {
            CopyKey (child->ivKeys, child->ivm + i,
                     sibling->ivKeys, i);
            child->ivPtrs[child->ivm + i] = sibling->ivPtrs [i];
          }
          child->ivPtrs [child->ivm + noMoves - 1] =
            sibling->ivPtrs [noMoves - 1];

          //
          // adjust parent
          //
          CopyKey (child->ivKeys, child->ivm - 1,
                   inode->ivKeys, index);
          CopyKey (inode->ivKeys, index,
                   sibling->ivKeys, noMoves - 1);

          //
          // rearrange sibling node
          //
          for (i = 0; i < sibling->ivm - noMoves - 1; i++) {
            CopyKey (sibling->ivKeys, i, sibling->ivKeys, i + noMoves);
            sibling->ivPtrs [i] = sibling->ivPtrs [i + noMoves];
          }
          sibling->ivPtrs [sibling->ivm - noMoves - 1] =
            sibling->ivPtrs [sibling->ivm - 1];

          child->ivm += noMoves;
          sibling->ivm -= noMoves;
        }
      }
    }
  }
}

void
IKeySortedSetAsBstTreeImpl::
SetInBetween (MngCursor& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (! IsInBetween (cursor)) {
    Node *node = NodeOf (cursor);
    SetToPrevious (cursor);
    PreviousNodeOf (cursor) = NodeOf (cursor);

    NodeOf (cursor) = node;
    SetToNext (cursor);
    NextNodeOf (cursor) = NodeOf (cursor);

    NodeOf (cursor) = (Node*) MngCursor::kIsInBetween;
   }
#endif
}


bool
IKeySortedSetAsBstTreeImpl::
SetToNextSubtree (ICursorImpl& cursor, INode const* inode) const
{ IASSERT (! inode->ivIsLeaf);

  bool hasNext = false;

  INumber index =
    BinarySearchInPageWithElement (inode, Self::ElementAt (cursor));

  if (inode->ivPtrs [index] == NodeOf (cursor) ||
      ! SetToNextSubtree
          (cursor, (INode const*)inode->ivPtrs [index])) {
    //
    // try to let cursor point to first element of next inode
    //
    if (index < inode->ivm - 1) {
      //
      // set to next lnode
      //
      NodeOf (cursor) = MinimumLeaf (inode->ivPtrs [index + 1]);
      hasNext = true;
    }
  }
  else {
    //
    // recursive call to SetToNextSubtree must have returned true in
    // order to come here, propagate this return value!
    //
    hasNext = true;
  }

  return hasNext;
}

bool
IKeySortedSetAsBstTreeImpl::
SetToPreviousSubtree (ICursorImpl& cursor, INode const* inode) const
{ IASSERT (! inode->ivIsLeaf);

  bool hasPrevious = false;

  INumber index = BinarySearchInPageWithElement
    (inode, Self::ElementAt (cursor));

  if (inode->ivPtrs [index] == NodeOf (cursor) ||
      ! SetToPreviousSubtree
          (cursor, (INode const*) inode->ivPtrs [index])) {
    //
    // try to let cursor point to previous element of inode
    //
    if (index != 0) {
      //
      // set to previous lnode
      //
      NodeOf (cursor) = MaximumLeaf (inode->ivPtrs [index - 1]);
      hasPrevious = true;
    }
  }
  else {
    //
    // recursive call to SetToPreviousSubtree must have returned true in
    // order ro come here, propagate this return value!
    //
    hasPrevious = true;
  }

  return hasPrevious;
}

#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (restore)
#endif
