// IBM Open Class Library
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// (C) Copyright IBM Corporation, 1992, 1997 All Rights Reserved
//
#include <iikssavl.h>
#include <iiexc.h>

#ifdef IC_PAGETUNE
  #define _IIKSSAVL_CPP_
  #include <ipagetun.h>
#endif

const short kAvlLeft  = -1;
const short kAvlRight =  1;

#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (nocls, nocnd, nocns, nocnv, noext, nognr, novft)
#endif

#if defined (__SOM_ENABLED__) && defined (__RRBC_LIB__)
#pragma SOMDefine (IAKeySortedSetAsAvlTreeOps)
#endif

// -------------------------------
// IAKeySortedSetAsAvlTreeNodeImpl
// -------------------------------

inline IAKeySortedSetAsAvlTreeNodeImpl*&
IAKeySortedSetAsAvlTreeNodeImpl::
Link (Direction d)
{ IASSERT (d != kAvlNone);
  return d == kAvlLeft ? ivLeft : ivRight;
}

inline void
IAKeySortedSetAsAvlTreeNodeImpl::
Parent (IAKeySortedSetAsAvlTreeNodeImpl*& parent, Direction& d)
{ parent = ivParent;
  if (parent != 0)
    d = (parent->ivLeft == this ? kAvlLeft : kAvlRight);
}

inline void
IAKeySortedSetAsAvlTreeNodeImpl::
ConnectChild (Direction d, IAKeySortedSetAsAvlTreeNodeImpl* node)
{ IASSERT (node != 0);
  Link (d) = node;
  node->ivParent = this;
}

inline void
IAKeySortedSetAsAvlTreeNodeImpl::
ConnectChild0 (Direction d, IAKeySortedSetAsAvlTreeNodeImpl* node)
{ Link (d) = node;
  if (node != 0)
    node->ivParent = this;
}

// --------------------------------
// IKeySortedSetAsAvlTreeCursorImpl
// --------------------------------

// public members

IKeySortedSetAsAvlTreeCursorImpl::
IKeySortedSetAsAvlTreeCursorImpl (IACollectionImpl const& collection)
: ICursorImpl (collection), ivNode (0)
{
}

IKeySortedSetAsAvlTreeCursorImpl::
IKeySortedSetAsAvlTreeCursorImpl
  (IKeySortedSetAsAvlTreeCursorImpl const& cursor)
: ICursorImpl (cursor), ivNode (cursor.ivNode)
{
}

IKeySortedSetAsAvlTreeCursorImpl::
~IKeySortedSetAsAvlTreeCursorImpl ()
{
}

ICursorImpl*
IKeySortedSetAsAvlTreeCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
IKeySortedSetAsAvlTreeCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

#if ! defined (__INO_MNGCURSOR__)
// -----------------------------------
// IKeySortedSetAsAvlTreeMngCursorImpl
// -----------------------------------

IAKeySortedSetAsAvlTreeNodeImpl const*const
IKeySortedSetAsAvlTreeMngCursorImpl::kIsInBetween =
   (IAKeySortedSetAsAvlTreeNodeImpl*)
    &IKeySortedSetAsAvlTreeMngCursorImpl::kIsInBetween;

// public members

IKeySortedSetAsAvlTreeMngCursorImpl::
IKeySortedSetAsAvlTreeMngCursorImpl
  (IKeySortedSetAsAvlTreeImpl const& collection)
: IKeySortedSetAsAvlTreeCursorImpl (collection),
  ivMngCollection ((IKeySortedSetAsAvlTreeImpl*)&collection)
{ ivNext = ivMngCollection->ivMngCursors;
  ivMngCollection->ivMngCursors = this;
}

IKeySortedSetAsAvlTreeMngCursorImpl::
~IKeySortedSetAsAvlTreeMngCursorImpl ()
{ Self*& first = ivMngCollection->ivMngCursors;
  if (this == first)
    first = ivNext;
  else {
    // find previous cursor
    for (Self* cursor = first;
         cursor->ivNext != this;
         cursor = cursor->ivNext);

    cursor->ivNext = ivNext;
  }
}

ICursorImpl*
IKeySortedSetAsAvlTreeMngCursorImpl::
Clone () const
{ return (Self*) CheckPointer (new Self (*this));
}

void
IKeySortedSetAsAvlTreeMngCursorImpl::
Copy (ICursorImpl const& cursor)
{ operator= ((Self const&)cursor);
}

bool
IKeySortedSetAsAvlTreeMngCursorImpl::
IsInBetween () const
{ return ivMngCollection->IsInBetween (*this);
}

bool
IKeySortedSetAsAvlTreeMngCursorImpl::
IsMngCursor () const
{ return true;
}

void
IKeySortedSetAsAvlTreeMngCursorImpl::
operator= (IKeySortedSetAsAvlTreeMngCursorImpl const& cursor)
{ Inherited::operator= (cursor);
  ivNextNode     = cursor.ivNextNode;
  ivPreviousNode = cursor.ivPreviousNode;
  if (ivMngCollection != cursor.ivMngCollection) {

    // detach 'this' from current list of mngcursors
    Self*& first = ivMngCollection->ivMngCursors;
    if (this == first)
      first = ivNext;
    else {
      // find previous cursor
      for (Self* cursor = first;
           cursor->ivNext != this;
           cursor = cursor->ivNext);

      cursor->ivNext = ivNext;
    }

    // insert 'this' in new list of mngcursors
    ivMngCollection = cursor.ivMngCollection;
    ivNext = (cursor.ivMngCollection)->ivMngCursors;
    ivMngCollection->ivMngCursors = this;
  }
}
#endif

// --------------------------
// IKeySortedSetAsAvlTreeImpl
// --------------------------

char const*
IKeySortedSetAsAvlTreeImpl::cvClassName = "KeySortedSetAsAvlTree";

// inline private members

inline IKeySortedSetAsAvlTreeImpl::Ops&
IKeySortedSetAsAvlTreeImpl::
OpsOf (IKeySortedSetAsAvlTreeImpl const& collection)
{ return (Ops&) Inherited::OpsOf (collection);
}

inline void*
IKeySortedSetAsAvlTreeImpl::
ElementOf (Node const* node) const
{ typedef IKeySortedSetAsAvlTreeNodeImpl <long> TypedNode;
  return (OpsOf (*this).ivContainsDTSObjects ?
    (void*) ((TypedNode const*)node)->ivElement :
    (void*)&((TypedNode const*)node)->ivElement);
}

inline bool
IKeySortedSetAsAvlTreeImpl::
IsInBetween (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return (((MngCursor const&)cursor).ivNode == MngCursor::kIsInBetween);
#else
  return false;
#endif
}

inline IKeySortedSetAsAvlTreeImpl::Node*&
IKeySortedSetAsAvlTreeImpl::
NextNodeOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivNextNode;
#else
  return ((Cursor&)cursor).ivNode; // just a dummy
#endif
}

inline IKeySortedSetAsAvlTreeImpl::Node*&
IKeySortedSetAsAvlTreeImpl::
NodeOf (ICursorImpl const& cursor)
{ return ((Cursor&)cursor).ivNode;
}

inline IKeySortedSetAsAvlTreeImpl::Node*&
IKeySortedSetAsAvlTreeImpl::
PreviousNodeOf (ICursorImpl const& cursor)
{
#if ! defined (__INO_MNGCURSOR__)
  return ((MngCursor&)cursor).ivPreviousNode;
#else
  return ((Cursor&)cursor).ivNode; // just a dummy
#endif
}

inline Direction avlDirection (long comp)
{ IASSERT (comp != 0);
  return comp < 0 ? kAvlRight : kAvlLeft;
}

// inline protected members

inline void
IKeySortedSetAsAvlTreeImpl::
Assign (void* element1, void const* element2) const
{ OpsOf (*this).Assign (element1, element2);
}

long
IKeySortedSetAsAvlTreeImpl::
CompareKeys (void const* arg1, void const* arg2, IArgType argType) const
{ return OpsOf (*this).CompareKeys (arg1, arg2, argType);
}

inline void*
IKeySortedSetAsAvlTreeImpl::
CreateNode (void const* element) const
{ return OpsOf (*this).CreateNode (element);
}

#if ! defined (__INO_STREAMING__)
inline void*
IKeySortedSetAsAvlTreeImpl::
CreateNodeWithStreamer (IDataStream& stream) const
{ return OpsOf (*this).CreateNodeWithStreamer (ivStreamer, stream);
}
#endif

inline void
IKeySortedSetAsAvlTreeImpl::
DeleteNode (void* node) const
{ OpsOf (*this).DeleteNode (node);
}

// inline public members

#define inline

inline void*
IKeySortedSetAsAvlTreeImpl::
Any () const
{ return ElementOf (ivRoot);
}

inline bool
IKeySortedSetAsAvlTreeImpl::
CheckReplacement (ICursorImpl const& cursor, void const* element) const
{ return (CompareKeys (Self::ElementAt (cursor),
                       element, kElementElement) == 0);
}

inline char const*
IKeySortedSetAsAvlTreeImpl::
ClassName () const
{ return cvClassName;
}

inline bool
IKeySortedSetAsAvlTreeImpl::
ContainsElementWithKey (void const* key) const
{ return Self::LocateElementWithKey (key, CursorOf (*this));
}

inline void*
IKeySortedSetAsAvlTreeImpl::
ElementAt (ICursorImpl const& cursor) const
{ return ElementOf (NodeOf (cursor));
}

inline bool
IKeySortedSetAsAvlTreeImpl::
IsBounded () const
{ return false;
}

inline bool
IKeySortedSetAsAvlTreeImpl::
IsEmpty () const
{ return (ivNumberOfElements == 0);
}

inline bool
IKeySortedSetAsAvlTreeImpl::
IsFirstAt (ICursorImpl const& cursor) const
{ Self::SetToFirst (CursorOf (*this));
  return (NodeOf (cursor) == NodeOf (CursorOf (*this)));
}

inline bool
IKeySortedSetAsAvlTreeImpl::
IsFull () const
{ return false;
}

inline bool
IKeySortedSetAsAvlTreeImpl::
IsLastAt (ICursorImpl const& cursor) const
{ Self::SetToLast (CursorOf (*this));
  return (NodeOf (cursor) == NodeOf (CursorOf (*this)));
}

inline void*
IKeySortedSetAsAvlTreeImpl::
Key (void const* key) const
{ return OpsOf (*this).Key (key);
}

inline bool
IKeySortedSetAsAvlTreeImpl::
LocateOrAddElementWithKey (void const* element, ICursorImpl& cursor)
{ return ! Self::Add (element, cursor);
}

inline INumber
IKeySortedSetAsAvlTreeImpl::
NumberOfElements () const
{ return ivNumberOfElements;
}

inline void
IKeySortedSetAsAvlTreeImpl::
ReplaceAt (ICursorImpl const& cursor, void const* element)
{ Assign (Self::ElementAt (cursor), element);
}

#undef inline

// public members

IKeySortedSetAsAvlTreeImpl::
IKeySortedSetAsAvlTreeImpl (Ops& ops, INumber)
: IAKeySortedSetImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivRoot (0), ivNumberOfElements (0)
{
}

IKeySortedSetAsAvlTreeImpl::
IKeySortedSetAsAvlTreeImpl
  (Ops& ops, IKeySortedSetAsAvlTreeImpl const& collection)
: IAKeySortedSetImpl ((Inherited::Ops&)ops), ivMngCursors (0),
  ivRoot (0), ivNumberOfElements (0)
{ operator= (collection);
}

IKeySortedSetAsAvlTreeImpl::
~IKeySortedSetAsAvlTreeImpl ()
{
#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    cursor->Invalidate ();
  }
#endif
  Self::RemoveAll ();
}

bool
IKeySortedSetAsAvlTreeImpl::
Add (void const* element, ICursorImpl& cursor)
{ Node*& node = NodeOf (cursor);
  if (Self::IsEmpty ()) {
    node = ivRoot = (Node*) CreateNode (element);
    ivNumberOfElements++;
    return true;
  }

  Node *S = ivRoot,
       *P = ivRoot,
       *Q, *R;

  Direction d;

  void* key = Key (element);

  for (;;) {
    // P will move down the tree.
    // S will point to the place where rebalancing may be necessary.

    long comp = CompareKeys (ElementOf (P), key, kElementKey);

    if (comp == 0) {
      node = P;
      return false;
    }

    d = avlDirection (comp);
    Q = P->Link (d);
    if (Q == 0) {
      node = Q = (Node*) CreateNode (element);
      P->ConnectChild (d, Q);
      ivNumberOfElements++;
      break;
    }

    P = Q;

    if (P->ivDeeperSubtree != kAvlNone)
      S = P;
  } // end for

  // Q is newly inserted node
  // P->Link (d) == Q
  // S == min N : (N == ivRoot || ! N is balanced) && N is ancestor of Q
  // where 'min' means 'nearest to Q'

  for (R = Q;; R = P, R->Parent (P, d)) {
    // P->Link (d) == R
    // P ancestor of Q && S ancestor of or equal to P.
    // P == S || P is balanced

    if (P == S)
      break;

    // P is balanced
    P->ivDeeperSubtree = d;
  }

  // R->ivParent == S
  // S->Link (d) has become higher by 1

  if (S->ivDeeperSubtree == kAvlNone)
    S->ivDeeperSubtree = d;
  else if (S->ivDeeperSubtree == Direction (-d))
    S->ivDeeperSubtree = kAvlNone;
  else {
    // S->ivDeeperSubtree == d.
    // Rotate

    if (R->ivDeeperSubtree == d) {
      // assume d == kAvlRight
      // Single rotation: (U, V may be 0)
      //         S(N)                                      R(N)
      //
      //  U(N-2)       R(N)          ----->        S(N-1)        W(N-1)
      //
      //         V(N-2)    W(N-1)            U(N-2)      V(N-2)
      //                         ...
      //                          Q

      ReplaceNode (S, R);
      S->ConnectChild0 (d, R->Link (Direction (-d)));
      R->ConnectChild (Direction (-d), S);

      S->ivDeeperSubtree = kAvlNone;
      R->ivDeeperSubtree = kAvlNone;
    }
    else {
      // assume d == kAvlRight
      // Double rotation: (U, W, X, Y may be 0)
      //         S(N)                                P(N)
      //
      //   U(N-2)     R(N)                  S(N-1)           R(M-1)
      //
      //        P(N-1)    W(N-2) -----> U(N-2) X(N-2/3) Y(N-3/2) W(N-2)
      //
      //     X(N-2/3) Y(N-3/2)
      //            ...
      //             Q

      P = R->Link (Direction (-d));
      ReplaceNode (S, P);
      R->ConnectChild0 (Direction (-d), P->Link (d));
      S->ConnectChild0 (d, P->Link (Direction (-d)));
      P->ConnectChild (d, R);
      P->ConnectChild (Direction (-d), S);

      if (P->ivDeeperSubtree == d)
        S->ivDeeperSubtree = Direction (-d);
      else
        S->ivDeeperSubtree = kAvlNone;

      if (P->ivDeeperSubtree == Direction (-d))
        R->ivDeeperSubtree = d;
      else
        R->ivDeeperSubtree = kAvlNone;

      P->ivDeeperSubtree = kAvlNone;
    } // single of double
  }

  //--------------------------------------------------
  // DEFECT 9119 :
  // update the cursor to the Newly Added position...
  // otherwise, a notifyOfAdd will reflect position
  // at which insert was Attempted only. (not always
  // the same thing for sorted sets).
  //--------------------------------------------------
  // NOTE : the cursor manipulation below only works
  //  if a cursor OTHER than the defaultCursor is
  //  used, because NodeOf() uses the default itself.
  //--------------------------------------------------
  /* Node* pNodeTest = 0;
  Node* pNodeNew = node;
  bool bFound = false;

  ICursorImpl* pLocalCursor = cursor.Clone();
  cursor.SetToFirst();
  pLocalCursor->SetToFirst();
  do {
      pNodeTest = NodeOf(*pLocalCursor);
      if ( pNodeNew == pNodeTest ) {
         bFound = true;
         break;
      }
      cursor.SetToNext();
  } while ( pLocalCursor->SetToNext() );
  delete pLocalCursor;
  */
  //--------------------------------------------------

  return true;
}

#if ! defined (__INO_STREAMING__)
void
IKeySortedSetAsAvlTreeImpl::
AddFromStream (IDataStream& fromWhere, ICursorImpl& cursor)
// same algorithm as in Add()...
{ Node*& node = NodeOf (cursor);
  Node  *strn = (Node*) CreateNodeWithStreamer (fromWhere);
  if (Self::IsEmpty ()) {
    node = ivRoot = strn;
    ivNumberOfElements++;
    return;
  }

  Node *S = ivRoot,
       *P = ivRoot,
       *Q, *R;

  Direction d;

  void* key = Key (&((StreamerNode*)strn)->ivElement);

  for (;;) {
    long comp = CompareKeys (ElementOf (P), key, kElementKey);

    if (comp == 0) {
      node = P;
      DeleteNode (strn);
      // change exception...
      bool alwaysFalse = false;
      ICHECK (alwaysFalse, IKeyAlreadyExistsExceptionImpl,
             "IKeySortedSetAsAvlTreeImpl", "ElementExists...");
    }

    d = avlDirection (comp);
    Q = P->Link (d);
    if (Q == 0) {
      node = Q = strn;
      P->ConnectChild (d, Q);
      ivNumberOfElements++;
      break;
    }

    P = Q;

    if (P->ivDeeperSubtree != kAvlNone)
      S = P;
  } // end for

  for (R = Q;; R = P, R->Parent (P, d)) {
    if (P == S)
      break;

    P->ivDeeperSubtree = d;
  }

  if (S->ivDeeperSubtree == kAvlNone)
    S->ivDeeperSubtree = d;
  else if (S->ivDeeperSubtree == Direction (-d))
    S->ivDeeperSubtree = kAvlNone;
  else {

    if (R->ivDeeperSubtree == d) {
      ReplaceNode (S, R);
      S->ConnectChild0 (d, R->Link (Direction (-d)));
      R->ConnectChild (Direction (-d), S);

      S->ivDeeperSubtree = kAvlNone;
      R->ivDeeperSubtree = kAvlNone;
    }
    else {
      P = R->Link (Direction (-d));
      ReplaceNode (S, P);
      R->ConnectChild0 (Direction (-d), P->Link (d));
      S->ConnectChild0 (d, P->Link (Direction (-d)));
      P->ConnectChild (d, R);
      P->ConnectChild (Direction (-d), S);

      if (P->ivDeeperSubtree == d)
        S->ivDeeperSubtree = Direction (-d);
      else
        S->ivDeeperSubtree = kAvlNone;

      if (P->ivDeeperSubtree == Direction (-d))
        R->ivDeeperSubtree = d;
      else
        R->ivDeeperSubtree = kAvlNone;

      P->ivDeeperSubtree = kAvlNone;
    } // single of double
  }
}
#endif

void
IKeySortedSetAsAvlTreeImpl::
AddAllFrom (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ()) {
    Self const& kss = (Self const&)collection;
    if (kss.ivRoot != 0)
      AddAllFromSubtree (kss.ivRoot);
  }
  else
     Inherited::AddAllFrom (collection);
}

bool
IKeySortedSetAsAvlTreeImpl::
AddOrReplaceElementWithKey (void const* element, ICursorImpl& cursor)
{ bool hasBeenAdded = Self::Add (element, cursor);

  if (! hasBeenAdded)
    Assign (Self::ElementAt (cursor), element);

  return hasBeenAdded;
}

bool
IKeySortedSetAsAvlTreeImpl::
AllElementsDo (IApplFunc applFunc, void* addArg)
{ if (ivRoot != 0)
    return AllElementsDoSubtree (applFunc, addArg, ivRoot);
  else
    return true;
}

bool
IKeySortedSetAsAvlTreeImpl::
CheckCursor (ICursorImpl const& cursor) const
{ if (cursor.IsInBetween () || (NodeOf (cursor) == 0))
    return true;
  else if (ivRoot != 0)
    return CheckNode (NodeOf (cursor), ivRoot);
  else
    return false;
}

IACollectionImpl*
IKeySortedSetAsAvlTreeImpl::
Clone () const
{ return (IACollectionImpl*) CheckPointer (OpsOf (*this).Clone ());
}

bool
IKeySortedSetAsAvlTreeImpl::
ContainsAllKeysFrom (IACollectionImpl const& collection) const
{ if (Self::ClassName () == collection.ClassName ()) {
    Self const& kss = (Self const&)collection;
    if (kss.ivRoot != 0)
      return ContainsAllKeysFromSubtree (kss.ivRoot);
    else
      return true;
  }
  else
    return Inherited::ContainsAllKeysFrom (collection);
}

void
IKeySortedSetAsAvlTreeImpl::
Copy (IACollectionImpl const& collection)
{ if (Self::ClassName () == collection.ClassName ())
    operator= ((Self const&)collection);
  else
    Inherited::Copy (collection);
}

IKeySortedSetAsAvlTreeImpl::Cursor*
IKeySortedSetAsAvlTreeImpl::
CreateCursor () const
{ return (Cursor*) CheckPointer (new Cursor (*this));
}

IKeySortedSetAsAvlTreeImpl::MngCursor*
IKeySortedSetAsAvlTreeImpl::
CreateMngCursor () const
{ return (MngCursor*) CheckPointer (new MngCursor (*this));
}

void*
IKeySortedSetAsAvlTreeImpl::
ElementAtPosition (IPosition position) const
{ Self::SetToPosition (position, CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

void*
IKeySortedSetAsAvlTreeImpl::
ElementWithKey (void const* key) const
{ Cursor cursor (*this);
  ICHECK_ContainsKey (ClassName (), "ElementWithKey")
  return Self::ElementAt (cursor);
}

void*
IKeySortedSetAsAvlTreeImpl::
First () const
{ Self::SetToFirst (CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}


bool
IKeySortedSetAsAvlTreeImpl::
IsConsistent () const
{ bool result = true;

  if (ivRoot != 0 && ivRoot->ivParent != 0)
    result = false;
  else {
    int dummy;
    INumber numberOfElements;
    if (! IsConsistentSubtree (ivRoot, dummy, numberOfElements))
      result = false;
    else if (numberOfElements != Self::NumberOfElements ())
      result = false;
  }

  return result;
}

void*
IKeySortedSetAsAvlTreeImpl::
Last () const
{ Self::SetToLast (CursorOf (*this));
  return Self::ElementAt (CursorOf (*this));
}

bool
IKeySortedSetAsAvlTreeImpl::
LocateElementWithKey (void const* key, ICursorImpl& cursor) const
{ Node*& node = NodeOf (cursor) = ivRoot;
  bool result = false;

  while (node != 0) {
    long comp = CompareKeys (ElementOf (node), key, kElementKey);

    if (comp == 0) {
      result = true;
      break;
    }
    else
      node = node->Link (avlDirection (comp));
  }

  return result;
}

bool
IKeySortedSetAsAvlTreeImpl::
LocateNextElementWithKey (void const* key, ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    NodeOf (cursor) = (PreviousNodeOf (cursor) ?
                       PreviousNodeOf(cursor) : NextNodeOf(cursor));
    if (NodeOf (cursor) == 0)
      return false;
  }
#endif

  if (CompareKeys (ElementAt (cursor), key, kElementKey) >= 0) {
    cursor.Invalidate ();
    return false;
  }
  else
    return Self::LocateElementWithKey (key, cursor);
}

IPosition
IKeySortedSetAsAvlTreeImpl::
PositionAt (ICursorImpl const& cursor) const
{ IPosition position = 1;

  for (Self::SetToFirst (CursorOf (*this));
       NodeOf (CursorOf (*this)) != NodeOf (cursor);
       position++, Self::SetToNext (CursorOf (*this)));

  return position;
}

INumber
IKeySortedSetAsAvlTreeImpl::
RemoveAll ()
{ INumber n = Self::NumberOfElements ();
  if (! Self::IsEmpty ()) {
    DeleteSubtree (ivRoot);
    ivRoot = 0;
    ivNumberOfElements = 0;
  }

#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor) {
    if (cursor->IsValid ()) {
      NodeOf (*cursor)         = (Node*) MngCursor::kIsInBetween;
      NextNodeOf (*cursor)     = 0;
      PreviousNodeOf (*cursor) = 0;
    }
  }
#endif

  return n;
}


INumber
IKeySortedSetAsAvlTreeImpl::
RemoveAll (IPredFunc predFunc, void* addArg)
{ Cursor current (*this), next (*this);
  INumber removed = 0;

  for (Self::SetToFirst (current), Self::SetToFirst (next);
       current.IsValid ();
       current = next) {
    Self::SetToNext (next);
    if (predFunc (Self::ElementAt (current), addArg)) {
      Self::RemoveAt (current);
      removed++;
    }
  }

  return removed;
}

void
IKeySortedSetAsAvlTreeImpl::
RemoveAt (ICursorImpl& cursor)
{ Node* node = NodeOf (cursor);

#if ! defined (__INO_MNGCURSOR__)
  forMngCursors (cursor2) {
    if (IsInBetween (*cursor2)) {
      if (node == NextNodeOf (*cursor2)) {
        NodeOf (*cursor2) = node;
        SetToNext (*cursor2);
        NextNodeOf (*cursor2) = NodeOf (*cursor2);
        NodeOf (*cursor2) = (Node*) MngCursor::kIsInBetween;
      }
      else if (node == PreviousNodeOf (*cursor2)) {
        NodeOf (*cursor2) = node;
        SetToPrevious (*cursor2);
        PreviousNodeOf (*cursor2) = NodeOf (*cursor2);
        NodeOf (*cursor2) = (Node*) MngCursor::kIsInBetween;
      }
    }
    else if (node == NodeOf (*cursor2)) {
      SetInBetween (*cursor2);
    }
  }
#endif

  bool changeDepth;
  Node* P;
  Direction d, dP;
  node->Parent (P, d);

  DisconnectAndRemove (node, changeDepth);

  for (node = P; changeDepth && node != 0; node = P, d = dP) {
    node->Parent (P, dP);
    Balance (node, d, changeDepth);
  }

  ivNumberOfElements--;

  if (! IsInBetween (cursor))
    cursor.Invalidate ();
}

void
IKeySortedSetAsAvlTreeImpl::
RemoveAtPosition (IPosition position)
{ Self::SetToPosition (position, CursorOf (*this));
  Self::RemoveAt (CursorOf (*this));
}

bool
IKeySortedSetAsAvlTreeImpl::
RemoveElementWithKey (void const* key)
{ bool result = false;

  if (LocateElementWithKey (key, CursorOf (*this))) {
    Self::RemoveAt (CursorOf (*this));
    result = true;
  }

  return result;
}

void
IKeySortedSetAsAvlTreeImpl::
RemoveFirst ()
{ Self::SetToFirst (CursorOf (*this));
  Self::RemoveAt (CursorOf (*this));
}

void
IKeySortedSetAsAvlTreeImpl::
RemoveLast ()
{ Self::SetToLast (CursorOf (*this));
  Self::RemoveAt (CursorOf (*this));
}

void
IKeySortedSetAsAvlTreeImpl::
ReplaceAtPosition (IPosition position, void const* element)
{ Self::SetToPosition (position, CursorOf (*this));
  Self::ReplaceAt (CursorOf (*this), element);
}

bool
IKeySortedSetAsAvlTreeImpl::
ReplaceElementWithKey (void const* element, ICursorImpl& cursor)
{ bool isContained =
    Self::LocateElementWithKey (Key (element), cursor);

  if (isContained)
    Self::ReplaceAt (cursor, element);

  return isContained;
}

bool
IKeySortedSetAsAvlTreeImpl::
SetToFirst (ICursorImpl& cursor) const
{ Node*& node = NodeOf (cursor) = ivRoot;
  if (node == 0)
    return false;
  else {
    while (node->ivLeft != 0)
      node = node->ivLeft;
    return true;
  }
}

bool
IKeySortedSetAsAvlTreeImpl::
SetToLast (ICursorImpl& cursor) const
{ Node*& node = NodeOf (cursor) = ivRoot;
  if (node == 0)
    return false;
  else {
    while (node->ivRight != 0)
      node = node->ivRight;
    return true;
  }
}

bool
IKeySortedSetAsAvlTreeImpl::
SetToNext (ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
    NodeOf (cursor) = NextNodeOf (cursor);
  }
  else
#endif
  { Node*& node = NodeOf (cursor);

    // if the given node has a right subtree, set the node to the
    // leftmost child of this subtree; otherwise find the next ancestor
    // of the given node for which the given node is in its left subtree

    if (node->ivRight != 0) {
      node = node->ivRight;
      while (node->ivLeft != 0)
        node = node->ivLeft;
    }
    else {
      while (node->ivParent != 0 && node->ivParent->ivLeft != node)
        node = node->ivParent;
      node = node->ivParent;
    }
  }

  return (NodeOf (cursor) != 0);
}

void
IKeySortedSetAsAvlTreeImpl::
SetToPosition (IPosition position, ICursorImpl& cursor) const
{ // When the position is past the half of the tree, the node
  // is moved backwards, otherwise it is moved forward.

  IASSERT (position >= 1);
  IASSERT (position <= Self::NumberOfElements ());

  if (position < (Self::NumberOfElements () / 2)) {
    Self::SetToFirst (cursor);
    while (--position != 0)
      Self::SetToNext (cursor);
  }
  else {
    position = Self::NumberOfElements () - position + 1;
    Self::SetToLast (cursor);
    while (--position != 0)
      Self::SetToPrevious (cursor);
  }
}

bool
IKeySortedSetAsAvlTreeImpl::
SetToPrevious (ICursorImpl& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (IsInBetween (cursor)) {
     NodeOf (cursor) = PreviousNodeOf (cursor);
  }
  else
#endif
  { Node*& node = NodeOf (cursor);

    // mirror to SetToNext ()

    if (node->ivLeft != 0) {
      node = node->ivLeft;
      while (node->ivRight != 0)
        node = node->ivRight;
    }
    else {
      while (node->ivParent != 0 && node->ivParent->ivRight != node)
        node = node->ivParent;
      node = node->ivParent;
    }
  }

  return (NodeOf (cursor) != 0);
}

void
IKeySortedSetAsAvlTreeImpl::
operator= (IKeySortedSetAsAvlTreeImpl const& collection)
{ if (this != &collection) {
    Self::RemoveAll ();
    if (collection.ivRoot != 0) {
      ivRoot = CopySubtree (collection.ivRoot);
      ivNumberOfElements = collection.ivNumberOfElements;
      ivRoot->ivParent = 0;
    }
#if ! defined (__INO_STREAMING__)
    ivStreamer = collection.ivStreamer;
#endif
  }
}

#if ! defined (__INO_STREAMING__)
void
IKeySortedSetAsAvlTreeImpl::
StreamIn (IDataStream& fromWhere)
{ ReadVersion (fromWhere);
  Self::RemoveAll ();
  INumber n;
  n <<= fromWhere;
  while (n--) {
    Self::AddFromStream (fromWhere, CursorOf (*this));
  }
}

void
IKeySortedSetAsAvlTreeImpl::
StreamOut (IDataStream& toWhere) const
{ WriteVersion (toWhere);
  Self::NumberOfElements () >>= toWhere;
  for (Self::SetToFirst (CursorOf (*this));
       CursorOf (*this).IsValid ();
       Self::SetToNext (CursorOf (*this))) {
    ivStreamer->streamOut (toWhere, Self::ElementAt (CursorOf (*this)));
  }
}

void
IKeySortedSetAsAvlTreeImpl::
AdoptStreamer (IACollectionStreamer *streamer)
{ if (streamer != ivStreamer.PtrOf())
    ivStreamer = IStreamerPtr(streamer);
}

IStreamerPtr&
IKeySortedSetAsAvlTreeImpl::
StreamerOf (IACollectionImpl const& collection)
{ return ivStreamer;
}
#endif

// private members

void
IKeySortedSetAsAvlTreeImpl::
AddAllFromSubtree (Node const* current)
{ Self::Add (ElementOf (current), CursorOf (*this));

  if (current->ivLeft != 0)
    AddAllFromSubtree (current->ivLeft);

  if (current->ivRight != 0)
    AddAllFromSubtree (current->ivRight);
}

bool
IKeySortedSetAsAvlTreeImpl::
AllElementsDoSubtree (IApplFunc applFunc,
                      void* addArg, Node* current) const
{ IASSERT (current != 0);

  bool result = true;

  if (current->ivLeft != 0)
    result = AllElementsDoSubtree (applFunc, addArg, current->ivLeft);

  result = result && applFunc (ElementOf (current), addArg);

  if (result && current->ivRight != 0)
    result = AllElementsDoSubtree (applFunc, addArg, current->ivRight);

  return result;
}

void
IKeySortedSetAsAvlTreeImpl::
Balance (Node* P, Direction d, bool &changeDepth)
{
  // (re)balance node P for which the depth of the subtree in
  // direction d has decreased by 1;
  // changeDepth is set to False if the depth of n does not change.

  IASSERT (P != 0);

  Node *Q, *R;

  if (P->ivDeeperSubtree == d) {

    P->ivDeeperSubtree = kAvlNone;
  }
  else if (P->ivDeeperSubtree == kAvlNone) {

    P->ivDeeperSubtree = Direction (-d);
    changeDepth = false;
  }
  else {
    // P->ivDeeperSubtree == Direction (-d)
    // Rebalance

    Q = P->Link (Direction (-d));
    if (Q->ivDeeperSubtree != d) {
      // assume d == kAvlLeft
      // Single rotation: (U, V, W may be 0)
      //         P(N)                                 Q(N/N-1)
      //
      //  U(N-3)      Q(N-1)         ----->     P(N-1/2)        W(N-2)
      //
      //         V(N-2/3)  W(N-2)          U(N-3)      V(N-2/3)

      ReplaceNode (P, Q);
      P->ConnectChild0 (Direction (-d), Q->Link (d));
      Q->ConnectChild (d, P);
      if (Q->ivDeeperSubtree == kAvlNone) {
        P->ivDeeperSubtree = Direction (-d);
        Q->ivDeeperSubtree = d;
        changeDepth = false;
      }
      else {
        P->ivDeeperSubtree = kAvlNone;
        Q->ivDeeperSubtree = kAvlNone;
      }
    }
    else {
      // assume d == kAvlLeft
      // Double rotation: (U, W, X, Y may be 0)
      //        P(N)                                R(N-1)
      //
      //   U(N-3)   Q(N-1)                  P(N-2)            Q(M-2)
      //
      //      R(N-2)     W(N-3) ----->  U(N-3) X(N-3/4)  Y(N-4/3) W(N-3)
      //
      //  X(N-3/4) Y(N-4/3)

      R = Q->Link (d);

      ReplaceNode (P, R);
      Q->ConnectChild0 (d, R->Link (Direction (-d)));
      P->ConnectChild0 (Direction (-d), R->Link (d));
      R->ConnectChild (Direction (-d), Q);
      R->ConnectChild (d, P);

      if (R->ivDeeperSubtree == Direction (-d))
          P->ivDeeperSubtree = d;
      else
          P->ivDeeperSubtree = kAvlNone;

      if (R->ivDeeperSubtree == d)
          Q->ivDeeperSubtree = Direction (-d);
      else
          Q->ivDeeperSubtree = kAvlNone;

      R->ivDeeperSubtree = kAvlNone;
    } // single or double rotate
  } // P->ivDeeperSubtree == Direction (-d)
}

bool
IKeySortedSetAsAvlTreeImpl::
CheckNode (Node const* node, Node* current) const
{ bool found = (node == current);

  if (! found && current->ivLeft != 0)
    found = CheckNode (node, current->ivLeft);

  if (! found && current->ivRight != 0)
    found = CheckNode (node, current->ivRight);

  return found;
}

bool
IKeySortedSetAsAvlTreeImpl::
ContainsAllKeysFromSubtree (Node const* current) const
{ // NOTE: current is not in the 'this' tree!
  bool result =
    Self::ContainsElementWithKey (Key (ElementOf (current)));

  if (result && current->ivLeft != 0)
    result = ContainsAllKeysFromSubtree (current->ivLeft);

  if (result && current->ivRight != 0)
    result = ContainsAllKeysFromSubtree (current->ivRight);

  return result;
}

IKeySortedSetAsAvlTreeImpl::Node*
IKeySortedSetAsAvlTreeImpl::
CopySubtree (Node const* current) const
{ IASSERT (current != 0);

  Node* newNode = (Node*) CreateNode (ElementOf (current));
  newNode->ivDeeperSubtree = current->ivDeeperSubtree;

  ITRY {
    if (current->ivLeft != 0) {
      newNode->ivLeft = CopySubtree (current->ivLeft);
      newNode->ivLeft->ivParent = newNode;
    }
    if (current->ivRight != 0) {
      newNode->ivRight = CopySubtree (current->ivRight);
      newNode->ivRight->ivParent = newNode;
    }
  }
  IENDTRY
  ICATCH (IOutOfMemory exc) {
    DeleteSubtree (newNode);
    IRETHROW (exc);
  }
  IENDCATCH

  return newNode;
}

void
IKeySortedSetAsAvlTreeImpl::
DeleteSubtree (Node* current) const
{ IASSERT (current != 0);

  // By testing nulliness of the tree to count BEFORE calling this
  // methods, about half of the function calls is saved.
  // (An optimal tree of depth n calls for the leaves (with null
  // child pointers). In our approach, these calls are saved.

  if (current->ivLeft != 0)
    DeleteSubtree (current->ivLeft);

  if (current->ivRight != 0)
    DeleteSubtree (current->ivRight);

  DeleteNode (current);
}

void
IKeySortedSetAsAvlTreeImpl::
DisconnectAndRemove (Node* P, bool& changeDepth)
{
  // remove P and rebalance subtrees
  // changeDepth tells whether the depth of the replaced tree has
  // changed

  if (P->ivRight == 0) {
    ReplaceNode (P, P->ivLeft);
    changeDepth = true;
  }
  else if (P->ivLeft == 0) {
    ReplaceNode (P, P->ivRight);
    changeDepth = true;
  }
  else {

    // Find the node R which is in-order just to the
    // left of the node P which is to be removed.
    // That is, go left and then right until 0 encountered.
    //
    // A, ..., B, W may be 0
    //
    //          P                                  R
    //
    //    A         V                        A           V
    //     .                                  .
    //      .                                  .
    //   ..  B              ----->          ..  B
    //
    //    ..   R                             ..  W
    //
    //        W
    //

    Node* R = P->ivLeft;
    while (R->ivRight != 0)
      R = R->ivRight;

    Node* B;
    if (R != P->ivLeft) {
      B = R->ivParent;
      B->ConnectChild0 (kAvlRight, R->ivLeft);
      R->ConnectChild (kAvlLeft, P->ivLeft);
    }
    else {
    // there is no A, ..., B, and W remains the left child of R
      B = 0;
    }

    ReplaceNode (P, R);
    R->ConnectChild (kAvlRight, P->ivRight);
    R->ivDeeperSubtree = P->ivDeeperSubtree;
    // possibly diminished depth of R->ivLeft is considered later

    // rebalance A, ..., B
    changeDepth = true;
    Node* BParent;
    for (changeDepth = true;
         changeDepth && B != 0 && B != R;
         B = BParent) {
      BParent = B->ivParent;
      Balance (B, kAvlRight, changeDepth);
    }

    if (changeDepth)
      Balance (R, kAvlLeft, changeDepth);
  }

  DeleteNode (P);
}

bool
IKeySortedSetAsAvlTreeImpl::
IsConsistentSubtree
  (Node const* node, int& depth, INumber& nrOfElem) const
{ bool result = true;

  if (node == 0) {
    depth = 0;
    nrOfElem = 0;
    result = true;
  }
  else {
    if (node->ivLeft != 0 && node->ivLeft->ivParent != node)
      result = false;
    else if (node->ivRight != 0 && node->ivRight->ivParent != node)
      result = false;
    else if (! IsConsistentSubtree (node->ivLeft, depth, nrOfElem))
      result = false;
    else {
      int d;
      INumber nr;
      if (! IsConsistentSubtree (node->ivRight, d, nr))
        result = false;
      else if (node->ivDeeperSubtree != Direction (d - depth))
        result = false;
      else {
        nrOfElem += nr + 1;
        if (d > depth)
          depth = d;
        depth++;
      }
    }
  }

  return result;
}

void
IKeySortedSetAsAvlTreeImpl::
ReplaceNode (Node* oldNode, Node* newNode)
{ Node* parent = oldNode->ivParent;

  if (newNode != 0)
    newNode->ivParent = parent;

  if (parent == 0)
    ivRoot = newNode;
  else if (oldNode == parent->ivRight)
    parent->ivRight = newNode;
  else
    parent->ivLeft = newNode;
}

void
IKeySortedSetAsAvlTreeImpl::
SetInBetween (MngCursor& cursor) const
{
#if ! defined (__INO_MNGCURSOR__)
  if (! IsInBetween (cursor)) {
    Node *node = NodeOf (cursor);
    SetToPrevious (cursor);
    PreviousNodeOf (cursor) = NodeOf (cursor);

    NodeOf (cursor) = node;
    SetToNext (cursor);
    NextNodeOf (cursor) = NodeOf (cursor);

    NodeOf (cursor) = (Node*) MngCursor::kIsInBetween;
  }
#endif
}

#if defined (__OS2__) || defined (__OS400__) || defined (__WINDOWS__)
#pragma info (restore)
#endif
